// apt_options.cc
//
//  Copyright 2000 Daniel Burrows

#include "apt_options.h"
#include "apt_config_widgets.h"
#include "pkg_columnizer.h"
#include "ui.h"

#include "aptitude.h"

#include <vscreen/vs_button.h>
#include <vscreen/vs_center.h>
#include <vscreen/vs_frame.h>
#include <vscreen/vs_label.h>
#include <vscreen/vs_table.h>

#include <vscreen/config/keybindings.h>
#include <vscreen/config/colors.h>

#include <generic/apt.h>

#include <sigc++/slot.h>
#include <sigc++/object_slot.h>
#include <sigc++/bind.h>

using namespace std;

struct option_item
{
  enum {OPTION_BOOL, OPTION_STRING, OPTION_END} type;
  const char *description;
  const char *option_name;

  union
  {
    bool b_default;
    const char *s_default;
  };

  option_item()
    :type(OPTION_END), description(NULL), option_name(NULL)
  {
  }

  option_item(const char *_description, const char *_option_name,
	      bool def)
    :type(OPTION_BOOL), description(_description), option_name(_option_name),
     b_default(def)
  {
  }

  option_item(const char *_description, const char *_option_name,
	      const char *def)
    :type(OPTION_STRING), description(_description), option_name(_option_name),
     s_default(def)
  {
  }
};

option_item ui_options[]={
  option_item(N_("Display some available commands at the top of the screen"),
	      PACKAGE "::UI::HelpBar", true),
  option_item(N_("Hide the menubar when it is not being used"),
	      PACKAGE "::UI::Menubar-Autohide", false),
  option_item(N_("Use a minibuffer-style prompt when possible"),
	      PACKAGE "::UI::Minibuf-Prompts", false),
  option_item(N_("Show partial search results (incremental search)"),
	      PACKAGE "::UI::Incremental-Search", true),
  option_item(N_("Closing the last view exits the program"),
	      PACKAGE "::UI::Exit-On-Last-Close", true),
  option_item(N_("Prompt for confirmation at exit"),
	      PACKAGE "::UI::Prompt-On-Exit", true),
  option_item(N_("Pause after downloading files"),
	      PACKAGE "::UI::Pause-After-Download", true),
  option_item(N_("Use a 'status-line' download indicator for all downloads"),
	      PACKAGE "::UI::Minibuf-Download-Bar", false),
  option_item(N_("Display the extended description area by default"),
	      PACKAGE "::UI::Description-Visible-By-Default", true),
  option_item(N_("Advance to the next item after changing the state of a package"),
	      PACKAGE "::UI::Advance-On-Action", false),
  option_item(N_("Automatically show why packages are broken"),
	      PACKAGE "::UI::Auto-Show-Reasons", true),
  option_item(N_("The default grouping method for package views"),
	      PACKAGE "::UI::Default-Grouping", default_grpstr),
  option_item(N_("The default display-limit for package views"),
	      PACKAGE "::Pkg-Display-Limit", ""),
  option_item(N_("The display format for package views"),
	      PACKAGE "::UI::Package-Display-Format",
	      pkg_item::pkg_columnizer::default_pkgdisplay),
  option_item(N_("The display format for the status line"),
	      PACKAGE "::UI::Package-Status-Format",
	      default_pkgstatusdisplay),
  option_item(N_("The display format for the header line"),
	      PACKAGE "::UI::Package-Header-Format",
	      default_pkgheaderdisplay),
  //option_item(N_("Use new (idempotent) package command behavior"),
  //	      PACKAGE "::UI::New-Package-Commands", true),
  option_item()
};

option_item misc_options[]={
  option_item(N_("Automatically upgrade installed packages"),
	      PACKAGE "::Auto-Upgrade", false),
  option_item(N_("Remove obsolete package files after downloading new package lists"),
	      PACKAGE "::AutoClean-After-Update", false),
  option_item(N_("URL to use to download changelogs"),
	      PACKAGE "::Changelog-URL-Template",
	      "http://cgi.debian.org/cgi-bin/get-changelog?package=%s"),
  option_item(N_("Display a preview of what will be done before doing it"),
	      PACKAGE "::Display-Planned-Action",
	      true),
  option_item(N_("Forget which packages are \"new\" whenever the package lists are updated"),
	      PACKAGE "::Forget-New-On-Update",
	      false),
  option_item(N_("Forget which packages are \"new\" whenever packages are installed or removed"),
	      PACKAGE "::Forget-New-On-Install",
	      false),
  option_item(N_("Warn when attempting to perform a privileged action as a non-root user"),
	      PACKAGE "::Warn-Not-Root",
	      true),
  option_item(N_("File to log actions into"),
	      PACKAGE "::Log",
	      "/var/log/aptitude"),
  option_item()
};

option_item dependency_options[]={
  option_item(N_("Automatically resolve dependencies of a package when it is selected"),
	      PACKAGE "::Auto-Install", true),
  option_item(N_("Automatically fix broken packages before installing or removing"),
	      PACKAGE "::Auto-Fix-Broken", true),
  option_item(N_("Install Recommended packages automatically"),
	      PACKAGE "::Recommends-Important", true),
  option_item(N_("Install Suggested packages automatically"),
	      PACKAGE "::Suggests-Important", false),
  option_item(N_("Remove unused packages automatically"),
	      PACKAGE "::Delete-Unused", true),
  option_item(N_("Automatically remove unused packages matching this filter"),
	      PACKAGE "::Delete-Unused-Pattern", ""),
  option_item()
};

// Commits all the entries of an options dialog, then saves the
// present settings and destroys the dialog.
class dialog_manager:public SigC::Object
{
public:
  dialog_manager(vscreen_widget *_dialog):dialog(_dialog) {}

  void add_widget(apt_config_widget *w) {widgets.push_back(w);}

  void commit()
  {
    for(vector<apt_config_widget *>::iterator i=widgets.begin();
	i!=widgets.end(); ++i)
      (*i)->commit();

    apt_dumpcfg(PACKAGE);

    destroy();
  }

  void destroy()
  {
    dialog->destroy();

    delete this;
  }

private:
  vector<apt_config_widget *> widgets;

  vscreen_widget *dialog;
};

static vscreen_widget *realize_options_dialog(option_item *items)
{
  vs_center *center=new vs_center;

  // Create general infrastructure (FIXME: make a plugin dialog widget thingy
  // that does this?
  vs_table *table=new vs_table;
  table->set_colsep(1);
  table->set_bg(get_color("DefaultWidgetBackground")|A_REVERSE);

  vs_button *okbutton=new vs_button(_("Ok"));
  vs_button *cancelbutton=new vs_button(_("Cancel"));
  okbutton->set_bg(get_color("DefaultWidgetBackground")|A_REVERSE);
  cancelbutton->set_bg(get_color("DefaultWidgetBackground")|A_REVERSE);

  table->connect_key("Confirm", &global_bindings, okbutton->pressed.slot());
  table->connect_key("Cancel", &global_bindings, cancelbutton->pressed.slot());

  dialog_manager *manager=new dialog_manager(center);

  okbutton->pressed.connect(slot(*manager, &dialog_manager::commit));
  cancelbutton->pressed.connect(slot(*manager, &dialog_manager::destroy));

  // FIXME: tables should support left/center/right-alignment..
  vs_center *cok=new vs_center(okbutton);
  vs_center *ccancel=new vs_center(cancelbutton);

  int row=0;

  while(items->type!=option_item::OPTION_END)
    {
      vs_label *l=NULL;

      switch(items->type)
	{
	case option_item::OPTION_BOOL:
	  {
	    apt_bool_widget *w=new apt_bool_widget(_(items->description),
						   items->option_name,
						   items->b_default);

	    w->set_bg(get_color("DefaultWidgetBackground")|A_REVERSE);

	    table->add_widget(w, row, 0, 1, 2, true, false);

	    manager->add_widget(w);
	  }
	  break;
	case option_item::OPTION_STRING:
	  {
	    apt_string_widget *w=new apt_string_widget(items->option_name,
						       items->s_default);

	    l=new vs_label(_(items->description),
			   get_color("DefaultWidgetBackground")|A_REVERSE);

	    table->add_widget(l, row, 0, 1, 1, false, false);
	    table->add_widget(w, row, 1, 1, 1, true, false);

	    manager->add_widget(w);
	  }
	  break;
	default:
	  assert(0);  // Someone probably goofed in setting the tables up
	}

      items++;
      row++;
    }

  table->add_widget(cok, row, 0, 1, 1, false, false);
  table->add_widget(ccancel, row, 1, 1, 1, false, false);

  vs_frame *frame=new vs_frame(table);
  frame->set_bg(get_color("DefaultWidgetBackground")|A_REVERSE);

  center->add_widget(frame);

  return center;
}

vscreen_widget *make_ui_options_dialog()
{
  return realize_options_dialog(ui_options);
}

vscreen_widget *make_misc_options_dialog()
{
  return realize_options_dialog(misc_options);
}

vscreen_widget *make_dependency_options_dialog()
{
  return realize_options_dialog(dependency_options);
}
