// cmdline.cc
//
//  Copyright 2002 Daniel Burrows
//
// Analogous to apt-get.  Indeed, large sections of code are borrowed
// from apt-get.
//
// FIXME: duplicated init stanzas in each function should be collected.
//
// FIXME: do the GainLock calls create a potential (albiet not overly
//       disastrous) race condition?
//
// TODO: add an "installtask" command.

#include <termios.h>
#include <sys/ioctl.h>
#include <sys/stat.h>

#include <map>
#include <set>
#include <algorithm>

#include <sigc++/object_slot.h>

#include <apt-pkg/algorithms.h>
#include <apt-pkg/acquire.h>
#include <apt-pkg/acquire-item.h>
#include <apt-pkg/clean.h>
#include <apt-pkg/dpkgpm.h>
#include <apt-pkg/error.h>
#include <apt-pkg/pkgsystem.h>
#include <apt-pkg/sourcelist.h>
#include <apt-pkg/srcrecords.h>
#include <apt-pkg/strutl.h>
#include <apt-pkg/version.h>

#include <generic/apt.h>
#include <generic/aptcache.h>
#include <generic/acqprogress.h>
#include <generic/config_signal.h>
#include <generic/infer_reason.h>
#include <generic/matchers.h>
#include <generic/pkg_acqfile.h>
#include <generic/tasks.h>

#include <vscreen/config/column_definition.h>
#include <vscreen/columnify.h>
#include <vscreen/fragment.h>
#include <vscreen/vscreen.h>

#include <sys/signal.h>

#include "cmdline.h"
#include "desc_parse.h"
#include "download.h"
#include "download_manager.h"
#include "load_sortpolicy.h"
#include "pkg_columnizer.h"
#include "pkg_sortpolicy.h"
#include "ui.h"

using namespace std;

unsigned int screen_width=80;

void update_screen_width(int foo=0)
{
  // Ripped from apt-get, which ripped it from GNU ls
  winsize ws;

  if (ioctl(1, TIOCGWINSZ, &ws) != -1 && ws.ws_col >= 5)
    screen_width = ws.ws_col - 1;
}

static inline bool operator<(const pkgCache::PkgIterator &a, const pkgCache::PkgIterator &b)
{
  return a->ID<b->ID;
}

static ostream &operator<<(ostream &out, const fragment_contents &contents)
{
  for(fragment_contents::const_iterator i=contents.begin();
      i!=contents.end(); ++i)
    {
      string s;
      // Drop the attributes for now.
      for(chstring::const_iterator j=i->begin(); j!=i->end(); ++j)
	s.push_back((*j)&A_CHARTEXT);

      out << s << endl;
    }

  return out;
}

bool pkg_byname_compare(const pkgCache::PkgIterator &a, const pkgCache::PkgIterator &b)
{
  return strcmp(a.Name(), b.Name())<0;
}

// Save typing.
typedef vector<pkgCache::PkgIterator> pkgvector;
typedef vector<pkgCache::PrvIterator> prvvector;
typedef set<pkgCache::PkgIterator> pkgset;
typedef vector<string> strvector;

// Hack around stupidity in apt-get.
struct fetchinfo
{
  double FetchBytes, FetchPBytes, DebBytes;
};

bool get_fetchinfo(fetchinfo &f)
{
  download_manager m;
  pkgAcquire fetcher(&m);
  pkgSourceList l;
  if(!l.ReadMainList())
    return _error->Error(_("Couldn't read list of sources"));

  pkgDPkgPM pm(*apt_cache_file);
  pm.GetArchives(&fetcher, &l, apt_package_records);

  f.FetchBytes=fetcher.FetchNeeded();
  f.FetchPBytes=fetcher.PartialPresent();
  f.DebBytes=fetcher.TotalNeeded();

  return true;
}

void cmdline_show_stringlist(strvector &items)
{
  int loc=2;

  printf("  ");

  for(strvector::iterator i=items.begin(); i!=items.end(); ++i)
    {
      if(loc+i->size()>screen_width-5)
	{
	  printf("\n  ");
	  loc=2;
	}

      printf("%s ", i->c_str());
      loc+=i->size()+1;
    }

  printf("\n");
}

void cmdline_show_pkglist(pkgvector &items)
{
  strvector tmp;

  for(pkgvector::iterator i=items.begin(); i!=items.end(); ++i)
    tmp.push_back(i->Name());

  cmdline_show_stringlist(tmp);
}

// Helper for below.

static string reason_string_list(set<reason> &reasons)
{
  set<reason>::iterator prev=reasons.end();
  string s;

  bool first=true;
  for(set<reason>::iterator why=reasons.begin();
      why!=reasons.end(); prev=why++)
    {
      // Filter duplicates.
      if(prev!=reasons.end() &&
	 prev->pkg==why->pkg && prev->dep->Type==why->dep->Type)
	continue;

      if(!first)
	s+=", ";
      else
	{
	  s+=" (";
	  first=false;
	}

      s+=const_cast<pkgCache::DepIterator &>(why->dep).DepType()[0];
      s+=": ";
      s+=why->pkg.Name();
    }
  if(!first)
    s+=")";

  return s;
}

/** Prints a description of a list of packages, with annotations
 *  reflecting how or why they will be changed.
 *
 *  Tries to infer the dependencies that caused a package to be installed,
 *  removed, or held.
 *
 *  \param items the set of items to examine
 *  \param showvers if true, display version numbers as appropriate
 *  \param showsize if true, display the change in each package's size
 */
void cmdline_show_instinfo(pkgvector &items,
			   bool showvers,
			   bool showdeps,
			   bool showsize)
{
  sort(items.begin(), items.end(), pkg_byname_compare);
  strvector output;

  for(pkgvector::iterator i=items.begin(); i!=items.end(); ++i)
    {
      string s=i->Name();

      pkgDepCache::StateCache &state=(*apt_cache_file)[*i];
      //aptitudeDepCache::aptitude_state &extstate=(*apt_cache_file)->get_ext_state(*i);
      pkgCache::VerIterator instver=state.InstVerIter(*apt_cache_file);

      // Display version numbers.
      if(showvers)
	{
	  pkgCache::VerIterator cur=i->CurrentVer();
	  pkgCache::VerIterator cand=state.CandidateVerIter(*apt_cache_file);

	  // Display x -> y for upgraded, held, and downgraded packages.
	  if( (state.Status==1 || state.Downgrade()) &&
	      i->CurrentVer()!=cand)
	    {
	      s+=" [";
	      s+=cur.VerStr();
	      s+=" -> ";
	      s+=cand.VerStr();
	      s+="]";
	    }
	  else
	    {
	      s+=" [";
	      s+=cand.VerStr();
	      s+="]";
	    }
	}

      // Show the change in size between the versions.
      if(showsize)
	{
	  int dsize=(instver.end()?0:instver->InstalledSize)
	    -(i->CurrentVer().end()?0:i->CurrentVer()->InstalledSize);

	  if(dsize>0)
	    s+=" <+"+SizeToStr(dsize)+"B>";
	  else if(dsize<0)
	    s+=" <-"+SizeToStr(dsize)+"B>";
	}

      if(showdeps)
	{
	  set<reason> reasons;
	  infer_reason(*i, reasons);
	  s+=reason_string_list(reasons);
	}

      output.push_back(s);
    }

  cmdline_show_stringlist(output);
}

// Shows broken dependencies for a single package

static void show_broken_deps(pkgCache::PkgIterator pkg)
{
  unsigned int indent=strlen(pkg.Name())+3;
  bool is_first_dep=true;
  pkgCache::VerIterator ver=(*apt_cache_file)[pkg].InstVerIter(*apt_cache_file);

  printf("  %s:", pkg.Name());
  for(pkgCache::DepIterator dep=ver.DependsList(); !dep.end(); ++dep)
    {
      pkgCache::DepIterator first=dep, prev=dep;

      while(dep->CompareOp & pkgCache::Dep::Or)
	++dep;

      // Yep, it's broken.
      if(dep.IsCritical() &&
	 !((*apt_cache_file)[dep]&pkgDepCache::DepGInstall))
	{		  
	  bool is_first_of_or=true;
	  // Iterate over the OR group, print out the information.

	  do
	    {
	      if(!is_first_dep)
		for(unsigned int i=0; i<indent; ++i)
		  printf(" ");

	      is_first_dep=false;

	      if(!is_first_of_or)
		for(unsigned int i=0; i<strlen(dep.DepType())+3; ++i)
		  printf(" ");
	      else
		printf(" %s: ", first.DepType());

	      is_first_of_or=false;

	      if(first.TargetVer())
		printf("%s (%s %s)", first.TargetPkg().Name(), first.CompType(), first.TargetVer());
	      else
		printf("%s", first.TargetPkg().Name());

	      // FIXME: handle virtual packages sanely.
	      pkgCache::PkgIterator target=first.TargetPkg();
	      // Don't skip real packages which are provided.
	      if(!target.VersionList().end())
		{
		  printf(" ");

		  pkgCache::VerIterator ver=(*apt_cache_file)[target].InstVerIter(*apt_cache_file);

		  if(!ver.end()) // ok, it's installable.
		    {
		      if((*apt_cache_file)[target].Install())
			printf(_("but %s is to be installed."),
			       ver.VerStr());
		      else if((*apt_cache_file)[target].Upgradable())
			printf(_("but %s is installed and it is kept back."),
			       target.CurrentVer().VerStr());
		      else
			printf(_("but %s is installed."),
			       target.CurrentVer().VerStr());
		    }
		  else
		    printf(_("but it is not installable"));
		}
	      else
		// FIXME: do something sensible here!
		printf(_(" which is a virtual package."));

	      if(first!=dep)
		printf(_(" or"));

	      printf("\n");

	      prev=first;
	      ++first;
	    } while(prev!=dep);
	}
    }
}

const char *cmdline_action_descriptions[num_pkg_action_states]={
  N_("The following packages are BROKEN:"),
  N_("The following packages are unused and will be REMOVED:"),
  N_("The following packages have been automatically kept back:"),
  N_("The following NEW packages will be automatically installed:"),
  N_("The following packages will be automatically REMOVED:"),
  N_("The following packages will be DOWNGRADED:"),
  N_("The following packages have been kept back:"),
  N_("The following packages will be REINSTALLED:"),
  N_("The following NEW packages will be installed:"),
  N_("The following packages will be REMOVED:"),
  N_("The following packages will be upgraded:"),
};

// Probably something like cin.getline() would work, but I don't trust that
// for interactive use.
static string prompt_string()
{
  string rval;
  char buf[1024];
  cin.getline(buf, 1023);
  rval+=buf;

  while(!cin)
    {
      cin.getline(buf, 1023);
      rval+=buf;
    }

  return rval;
}

#if 0
// A little like PromptYn in apt-get.
static bool prompt_yn()
{
  char buf[1024];

  cin.getline(buf, 1023);

  bool rval=(buf[0]=='y' || buf[0]=='Y' || buf[0]=='\0');

  while(!cin)
    cin.getline(buf, 1023);

  return rval;
}
#endif

// Checks for broken essential packages.  Returns false if the user
// doesn't want to continue.
static bool prompt_essential()
{
  pkgvector todelete, whatsbroken;
  bool ok=true;

  for(pkgCache::PkgIterator pkg=(*apt_cache_file)->PkgBegin();
      !pkg.end(); ++pkg)
    {
      if( (pkg->Flags & pkgCache::Flag::Essential) ||
	  (pkg->Flags & pkgCache::Flag::Important))
	{
	  // Eek!
	  if((*apt_cache_file)[pkg].Delete())
	    todelete.push_back(pkg);

	  if((*apt_cache_file)[pkg].InstBroken())
	    whatsbroken.push_back(pkg);
	}
    }

  if(!todelete.empty())
    {
      ok=false;

      printf(_("The following ESSENTIAL packages will be REMOVED!\n"));
      cmdline_show_pkglist(todelete);
      printf("\n");
    }

  if(!whatsbroken.empty())
    {
      ok=false;

      printf(_("The following ESSENTIAL packages will be BROKEN by this action:\n"));

      for(pkgvector::iterator i=whatsbroken.begin(); i!=whatsbroken.end(); ++i)
	show_broken_deps(*i);

      printf("\n");
    }

  if(!ok)
    {
      printf(_("WARNING: Performing this action will probably cause your system to break!\n         Do NOT continue unless you know EXACTLY what you are doing!\n"));

      string prompt=_("I am aware that this is a very bad idea");
      char buf[1024];

      printf(_("To continue, type the phrase \"%s\":\n"), prompt.c_str());
      cin.getline(buf, 1023);
      bool rval=(prompt==buf);

      while(!cin)
	cin.getline(buf, 1023);

      return rval;
    }

  return true;
}

/** Displays a preview of the stuff to be done -- like apt-get, it collects
 *  all the "stuff to install" in one place.
 *
 *  The arguments and return value are for when this is used for a targeted
 *  install/remove; it can figure out whether any packages not requested by the
 *  user are being installed/removed (eg, because of sticky states) and
 *  tell the caller to pause for confirmation.
 */
bool cmdline_show_preview(bool as_upgrade, pkgset &to_install,
			  pkgset &to_hold, pkgset &to_remove,
			  bool showvers, bool showdeps, bool showsize)
{
  bool all_empty=true;

  pkgvector lists[num_pkg_action_states];
  pkgvector extra_install, extra_remove;
  unsigned long Upgrade=0, Downgrade=0, Install=0, ReInstall=0;

  for(pkgCache::PkgIterator pkg=(*apt_cache_file)->PkgBegin();
      !pkg.end(); ++pkg)
    {
      if((*apt_cache_file)[pkg].NewInstall())
	++Install;
      else if((*apt_cache_file)[pkg].Upgrade())
	++Upgrade;
      else if((*apt_cache_file)[pkg].Downgrade())
	++Downgrade;
      else if(!(*apt_cache_file)[pkg].Delete() &&
	      ((*apt_cache_file)[pkg].iFlags & pkgDepCache::ReInstall))
	++ReInstall;

      pkg_action_state state=find_pkg_state(pkg);

      switch(state)
	{
	case pkg_auto_install:
	case pkg_install:
	case pkg_upgrade:
	  if(to_install.find(pkg)==to_install.end())
	    extra_install.push_back(pkg);
	  break;
	case pkg_auto_remove:
	case pkg_unused_remove:
	case pkg_remove:
	  if(to_remove.find(pkg)==to_remove.end())
	    extra_remove.push_back(pkg);
	  break;
	default:
	  break;
	}

      if(state!=pkg_unchanged &&
	 !((state==pkg_auto_hold || state==pkg_hold) && !as_upgrade))
	lists[state].push_back(pkg);

      if(state==pkg_auto_install)
	lists[pkg_install].push_back(pkg);
      else if(state==pkg_auto_remove)
	lists[pkg_remove].push_back(pkg);
    }

  for(int i=0; i<num_pkg_action_states; ++i)
    {
      if(!lists[i].empty())
	{
	  all_empty=false;

	  printf("%s\n", _(cmdline_action_descriptions[i]));
	  if(i==pkg_auto_install || i==pkg_auto_remove || i==pkg_unused_remove ||
	     i==pkg_auto_hold || i==pkg_broken)
	    cmdline_show_instinfo(lists[i],
				  showvers, showdeps, showsize);
	  else
	    cmdline_show_instinfo(lists[i],
				  showvers, false, showsize);
	}
    }

  if(all_empty)
    printf(_("No packages will be installed, upgraded, or removed.\n"));

  printf(_("%lu packages upgraded, %lu newly installed, "),
	 Upgrade, Install);

  if(ReInstall!=0)
    printf(_("%lu reinstalled, "), ReInstall);
  if(Downgrade!=0)
    printf(_("%lu downgraded, "), Downgrade);

  printf(_("%lu to remove and %lu not upgraded.\n"),
	 (*apt_cache_file)->DelCount(),(*apt_cache_file)->KeepCount());

  fetchinfo f;
  if(get_fetchinfo(f))
    {
      if(f.DebBytes!=f.FetchBytes)
	printf(_("Need to get %sB/%sB of archives. "),
	       SizeToStr(f.FetchBytes).c_str(), SizeToStr(f.DebBytes).c_str());
      else
	printf(_("Need to get %sB of archives. "),
	       SizeToStr(f.DebBytes).c_str());
    }
  else
    _error->DumpErrors();

  if((*apt_cache_file)->UsrSize() >=0)
    printf(_("After unpacking %sB will be used.\n"),
	   SizeToStr((*apt_cache_file)->UsrSize()).c_str());
  else
    printf(_("After unpacking %sB will be freed.\n"),
	   SizeToStr(-(*apt_cache_file)->UsrSize()).c_str());

  // If I return directly below, g++ complains about control reaching the
  // end of a non-void function!
  bool rval;

  rval=((as_upgrade && !lists[pkg_upgrade].empty()) ||
	!(extra_install.empty() && extra_remove.empty()));

  return rval;
}

// Shows a list of all broken packages together with their dependencies.
// Similar to and based on the equivalent routine in apt-get.
//
// Returns false if some packages are broken.
//
// FIXME: translatability
static bool show_broken()
{
  pkgvector broken;
  for(pkgCache::PkgIterator i=(*apt_cache_file)->PkgBegin();
      !i.end(); ++i)
    {
      if((*apt_cache_file)[i].InstBroken())
	broken.push_back(i);
    }

  if(!broken.empty())
    {
      // Use apt-get's familiar error message,
      printf(_("Some packages had unmet dependencies.  This may mean that you have\nrequested an impossible situation or if you are using the unstable\ndistribution that some required packages have not yet been created\nor been moved out of Incoming.\n\n"));
      printf(_("The following packages have unmet dependencies:\n"));

      for(pkgvector::iterator pkg=broken.begin(); pkg!=broken.end(); ++pkg)
	show_broken_deps(*pkg);
      return false;
    }

  return true;
}

static void dl_complete(download_manager &manager)
{
}

download_manager *gen_cmdline_download_progress()
{
  download_manager *m=new download_manager;

  update_screen_width();
  // FIXME: put this in a shared init function.
  signal(SIGWINCH, update_screen_width);

  // FIXME: LEAK!!!!  (since the command-line stuff will only use this once
  // and then terminate, and killing this leak is a major pain, I'm
  // letting it by for now, but this MUST BE FIXED EVENTUALLY!)
  AcqTextStatus *acqprogress=new AcqTextStatus(screen_width, 0);

  m->MediaChange_sig.connect(slot(*acqprogress, &AcqTextStatus::MediaChange));
  m->IMSHit_sig.connect(slot(*acqprogress, &AcqTextStatus::IMSHit));
  m->Fetch_sig.connect(slot(*acqprogress, &AcqTextStatus::Fetch));
  m->Done_sig.connect(slot(*acqprogress, &AcqTextStatus::Done));
  m->Fail_sig.connect(slot(*acqprogress, &AcqTextStatus::Fail));
  m->Pulse_sig.connect(slot(*acqprogress, &AcqTextStatus::Pulse));
  m->Start_sig.connect(slot(*acqprogress, &AcqTextStatus::Start));
  m->Stop_sig.connect(slot(*acqprogress, &AcqTextStatus::Stop));
  m->Complete_sig.connect(SigC::slot(dl_complete));

  // FIXME: Maybe the AcqTextStatus should be deleted on Complete?
  //m->Complete_sig.connect(slot(acqprogress, &AcqTextStatus::Complete));

  return m;
}

int cmdline_update(int argc, char *argv[])
{
  _error->DumpErrors();

  if(argc!=1)
    {
      fprintf(stderr, _("E: The update command takes no arguments\n"));
      return -1;
    }

  OpTextProgress progress;

  apt_init(&progress, false);

  if(_error->PendingError())
    {
      _error->DumpErrors();
      return -1;
    }

  int rval=do_pkglist_update(&progress, true)?0:-1;

  if(_error->PendingError())
    rval=-1;

  _error->DumpErrors();

  return rval;
}

int cmdline_clean(int argc, char *argv[], bool simulate)
{
  _error->DumpErrors();

  OpTextProgress progress;

  apt_init(&progress, false);

  if(_error->PendingError())
    {
      _error->DumpErrors();
      return -1;
    }

  // In case we aren't root.
  if(!simulate)
    apt_cache_file->GainLock();
  else
    apt_cache_file->ReleaseLock();

  if(_error->PendingError())
    {
      _error->DumpErrors();
      return -1;
    }

  if(aptcfg)
    {
      if(simulate)
	printf(_("Del %s* %spartial/*\n"),
	       aptcfg->FindDir("Dir::Cache::archives").c_str(),
	       aptcfg->FindDir("Dir::Cache::archives").c_str());
      else
	{
	  pkgAcquire fetcher;
	  fetcher.Clean(aptcfg->FindDir("Dir::Cache::archives"));
	  fetcher.Clean(aptcfg->FindDir("Dir::Cache::archives")+"partial/");
	}
    }

  int rval=_error->PendingError()?0:-1;

  _error->DumpErrors();

  return rval;
}

// Shamelessly stolen from apt-get:
class LogCleaner : public pkgArchiveCleaner
{
  bool simulate;

protected:
  virtual void Erase(const char *File,string Pkg,string Ver,struct stat &St) 
  {
    cout << "Del " << Pkg << " " << Ver << " [" << SizeToStr(St.st_size) << "B]" << endl;
      
    if (!simulate)
      unlink(File);      
  };
public:
  LogCleaner(bool _simulate):simulate(_simulate) { }
};

int cmdline_autoclean(int argc, char *argv[], bool simulate)
{
  _error->DumpErrors();

  OpTextProgress progress;

  apt_init(&progress, false);

  if(_error->PendingError())
    {
      _error->DumpErrors();
      return -1;
    }

  // In case we aren't root.
  if(!simulate)
    apt_cache_file->GainLock();
  else
    apt_cache_file->ReleaseLock();

  if(_error->PendingError())
    {
      _error->DumpErrors();
      return -1;
    }

  LogCleaner cleaner(simulate);
  int rval=0;
  if(!(cleaner.Go(aptcfg->FindDir("Dir::Cache::archives"), *apt_cache_file) &&
       cleaner.Go(aptcfg->FindDir("Dir::Cache::archives")+"partial/",
		  *apt_cache_file)) ||
     _error->PendingError())
    rval=-1;

  _error->DumpErrors();
  return rval;
}

int cmdline_forget_new(int argc, char *argv[],
		       const char *status_fname, bool simulate)
{
  _error->DumpErrors();

  OpTextProgress progress;

  apt_init(&progress, false, status_fname);

  if(_error->PendingError())
    {
      _error->DumpErrors();
      return -1;
    }

  // In case we aren't root.
  if(!simulate)
    apt_cache_file->GainLock();
  else
    apt_cache_file->ReleaseLock();

  if(_error->PendingError())
    {
      _error->DumpErrors();
      return -1;
    }

  if(simulate)
    printf(_("Would forget what packages are new\n"));
  else
    {
      (*apt_cache_file)->forget_new(NULL);

      (*apt_cache_file)->save_selection_list(progress);

      if(_error->PendingError())
	{
	  _error->DumpErrors();

	  return -1;
	}
    }

  return 0;
}

// compare...uses the candidate version of each package.
class compare
{
  pkg_sortpolicy *s;
public:
  compare(pkg_sortpolicy *_s):s(_s) {}

  bool operator()(pkgCache::PkgIterator a, pkgCache::PkgIterator b)
  {
    pkgCache::VerIterator av=(*apt_cache_file)[a].CandidateVerIter(*apt_cache_file);
    pkgCache::VerIterator bv=(*apt_cache_file)[b].CandidateVerIter(*apt_cache_file);

    return s->compare(a, av, b, bv)<0;
  }
};

// FIXME: apt-cache does lots of tricks to make this fast.  Should I?
int cmdline_search(int argc, char *argv[], const char *status_fname,
		   string display_format, string width, std::string sort)
{
  pkg_item::pkg_columnizer::setup_columns();

  pkg_sortpolicy *s=parse_sortpolicy(sort);

  if(!s)
    {
      _error->DumpErrors();
      return -1;
    }

  compare comp(s);

  _error->DumpErrors();

  update_screen_width();

  if(!width.empty())
    {
      unsigned long tmp=screen_width;
      StrToNum(width.c_str(), tmp, width.size());
      screen_width=tmp;
    }

  column_definition_list *columns=parse_columns(display_format,
						pkg_item::pkg_columnizer::parse_column_type,
						pkg_item::pkg_columnizer::defaults);

  if(!columns)
    {
      _error->DumpErrors();
      return -1;
    }

  if(argc<=1)
    {
      fprintf(stderr, _("search: You must provide at least one search term\n"));
      delete columns;
      return -1;
    }

  OpProgress progress;

  apt_init(&progress, true, status_fname);

  if(_error->PendingError())
    {
      _error->DumpErrors();
      delete columns;
      return -1;
    }

  std::vector<pkg_matcher *> matchers;

  for(int i=1; i<argc; ++i)
    {
      pkg_matcher *m=parse_pattern(argv[i]);
      if(!m)
	{
	  while(!matchers.empty())
	    {
	      delete matchers.back();
	      matchers.pop_back();
	    }

	  _error->DumpErrors();

	  delete columns;
	  return -1;
	}

      matchers.push_back(m);
    }

  vector<pkgCache::PkgIterator> output;
  for(pkgCache::PkgIterator pkg=(*apt_cache_file)->PkgBegin();
      !pkg.end(); ++pkg)
    {
      for(std::vector<pkg_matcher *>::iterator m=matchers.begin();
	  m!=matchers.end(); ++m)
	if((*m)->matches(pkg, (*apt_cache_file)[pkg].CandidateVerIter(*apt_cache_file)))
	  output.push_back(pkg);
    }

  std::sort(output.begin(), output.end(), comp);

  for(vector<pkgCache::PkgIterator>::iterator i=output.begin();
      i!=output.end(); ++i)
    printf("%s\n",
	   pkg_item::pkg_columnizer(*i,
				    i->VersionList(),
				    *columns,
				    0).layout_columns(screen_width).c_str());

  delete columns;

  return 0;
}

enum cmdline_version_source {cmdline_version_cand,
			     cmdline_version_archive,
			     cmdline_version_version};

// Finds a candidate version for the package using the given source.
pkgCache::VerIterator cmdline_find_ver(pkgCache::PkgIterator pkg,
				       cmdline_version_source source,
				       string archive="",
				       string version="")
{
  switch(source)
    {
    case cmdline_version_cand:
      return (*apt_cache_file)[pkg].CandidateVerIter(*apt_cache_file);
    case cmdline_version_archive:
      for(pkgCache::VerIterator ver=pkg.VersionList(); !ver.end(); ++ver)
	for(pkgCache::VerFileIterator verfile=ver.FileList();
	    !verfile.end(); ++verfile)
	  {
	    pkgCache::PkgFileIterator pkgfile=verfile.File();
	    if(pkgfile.Archive() && archive==pkgfile.Archive())
	      return ver;
	  }

      printf(_("Unable to find an archive \"%s\" for the package \"%s\"\n"),
	     archive.c_str(),
	     pkg.Name());

      return pkgCache::VerIterator(*apt_cache_file, 0);
    case cmdline_version_version:
      for(pkgCache::VerIterator ver=pkg.VersionList(); !ver.end(); ++ver)
	if(version==ver.VerStr())
	  return ver;

      printf(_("Unable to find a version \"%s\" for the package \"%s\"\n"),
	     version.c_str(),
	     pkg.Name());

      return pkgCache::VerIterator(*apt_cache_file, 0);
    default:
      printf(_("Internal error: invalid value %i passed to cmdline_find_ver!\n"),
	     source);
      return pkg.VersionList();
    }
}

// An action on a single package.

enum cmdline_pkgaction_type
  {cmdline_install, cmdline_remove, cmdline_purge, cmdline_hold,
   cmdline_unhold, cmdline_markauto, cmdline_unmarkauto,
   cmdline_forbid_version, cmdline_reinstall};

bool cmdline_applyaction(cmdline_pkgaction_type action,
			 pkgProblemResolver &fixer, pkgCache::PkgIterator pkg,
			 pkgset &to_install, pkgset &to_hold,
			 pkgset &to_remove, pkgset &to_purge,
			 cmdline_version_source source=cmdline_version_cand,
			 string archive="", string version="")
{
  // Handle virtual packages.
  if(!pkg.ProvidesList().end())
    {
      if(pkg.VersionList().end())
	{
	  for(pkgCache::PrvIterator prv=pkg.ProvidesList(); !prv.end(); ++prv)
	    {
	      if(prv.OwnerPkg().CurrentVer()==prv.OwnerVer())
		{
		  printf(_("Note: \"%s\", providing the virtual package\n      \"%s\", is already installed.\n"),
			 prv.OwnerPkg().Name(), pkg.Name());
		  return true;
		}
	      else if((*apt_cache_file)[prv.OwnerPkg()].InstVerIter(*apt_cache_file)==prv.OwnerVer())
		{
		  printf(_("Note: \"%s\", providing the virtual package\n      \"%s\", is already going to be installed.\n"),
			 prv.OwnerPkg().Name(), pkg.Name());
		  return true;
		}
	    }

	  // See if there's only one possible package to install.
	  pkgvector cands;

	  for(pkgCache::PrvIterator prv=pkg.ProvidesList();
	      !prv.end(); ++prv)
	    {
	      if((*apt_cache_file)[prv.OwnerPkg()].CandidateVerIter(*apt_cache_file)==prv.OwnerVer())
		cands.push_back(prv.OwnerPkg());
	    }

	  if(cands.size()==0)
	    {
	      printf(_("\"%s\" exists in the package database, but it is not a\nreal package and no package provides it.\n"),
		     pkg.Name());
	      return false;
	    }
	  else if(cands.size()>1)
	    {
	      printf(_("\"%s\" is a virtual package provided by:\n"),
		     pkg.Name());
	      cmdline_show_pkglist(cands);
	      printf(_("You must choose one to install.\n"));
	      return false;
	    }
	  else if(cands.size()==1)
	      {
		printf(_("Note: selecting \"%s\" instead of the\n      virtual package \"%s\"\n"),
		       cands[0].Name(), pkg.Name());
		pkg=cands[0];
	      }
 	}
    }

  pkgCache::VerIterator ver=pkg.CurrentVer();
  if(action==cmdline_install)
    ver=cmdline_find_ver(pkg, source, archive, version);

  switch(action)
    {
    case cmdline_install:
      if(pkg.CurrentVer()!=ver || pkg->CurrentState!=pkgCache::State::Installed)
	to_install.insert(pkg);
      else if((*apt_cache_file)[pkg].Keep())
	printf(_("%s is already installed at the requested version (%s)\n"),
	       pkg.Name(),
	       ver.VerStr());
      break;
    case cmdline_reinstall:
      if(pkg.CurrentVer().end())
	printf(_("%s is not currently installed, so it will not be reinstalled.\n"), pkg.Name());

      break;
    case cmdline_remove:
      if(!pkg.CurrentVer().end())
	to_remove.insert(pkg);
      else if((*apt_cache_file)[pkg].Keep())
	printf(_("Package %s is not installed, so it will not be removed\n"), pkg.Name());
      break;
    case cmdline_purge:
      if(!pkg.CurrentVer().end() || pkg->CurrentState!=pkgCache::State::ConfigFiles)
	to_purge.insert(pkg);
      else if((*apt_cache_file)[pkg].Keep())
	printf(_("Package %s is not installed, so it will not be removed\n"), pkg.Name());
      break;
    case cmdline_hold:
      to_hold.insert(pkg);
      break;
    case cmdline_unhold:
      to_hold.erase(pkg);
      break;
    case cmdline_forbid_version:
      if(!version.empty())
	if(pkg.CurrentVer().end())
	  printf(_("Package %s is not installed, cannot forbid an upgrade\n"), pkg.Name());
	else if((*apt_cache_file)[pkg].Upgradable())
	  printf(_("Package %s is not upgradable, cannot forbid an upgrade\n"), pkg.Name());
    default:
      break;
    }

  switch(action)
    {
    case cmdline_install:
    case cmdline_reinstall:
      if(action==cmdline_reinstall && pkg.CurrentVer().end())
	break;

      (*apt_cache_file)->set_candidate_version(ver, NULL);
      (*apt_cache_file)->mark_install(pkg, true, action == cmdline_reinstall, NULL);
      fixer.Clear(pkg);
      fixer.Protect(pkg);
      break;
    case cmdline_remove:
      (*apt_cache_file)->mark_delete(pkg, false, false, NULL);
      fixer.Clear(pkg);
      fixer.Protect(pkg);
      fixer.Remove(pkg);
      break;
    case cmdline_purge:
      (*apt_cache_file)->mark_delete(pkg, true, false, NULL);
      fixer.Clear(pkg);
      fixer.Protect(pkg);
      fixer.Remove(pkg);
      break;
    case cmdline_hold:
      (*apt_cache_file)->mark_keep(pkg, false, true, NULL);
      fixer.Clear(pkg);
      fixer.Protect(pkg);
      break;
    case cmdline_unhold:
      fixer.Clear(pkg);
      if(pkg->CurrentState==pkgCache::State::Installed)
	{
	  (*apt_cache_file)->mark_install(pkg, true, false, NULL);
	  fixer.Protect(pkg);
	}
      else
	(*apt_cache_file)->mark_keep(pkg, false, false, NULL);
      break;
    case cmdline_markauto:
      (*apt_cache_file)->mark_auto_installed(pkg, true, NULL);
      break;
    case cmdline_unmarkauto:
      (*apt_cache_file)->mark_auto_installed(pkg, false, NULL);
      break;
    case cmdline_forbid_version:
      if(!version.empty())
	(*apt_cache_file)->forbid_upgrade(pkg, version, NULL);
      else
	{
	  pkgCache::VerIterator curver=pkg.CurrentVer();
	  pkgCache::VerIterator candver=(*apt_cache_file)[pkg].CandidateVerIter(*apt_cache_file);
	  if(!curver.end() && !candver.end() && curver!=candver)
	    (*apt_cache_file)->forbid_upgrade(pkg, candver.VerStr(), NULL);
	}
      break;
    default:
      fprintf(stderr, "Internal error: impossible pkgaction type\n");
      abort();
    }

  return true;
}

static bool cmdline_applyaction(string s,
				cmdline_pkgaction_type action,
				pkgProblemResolver &fixer,
				pkgset &to_install, pkgset &to_hold,
				pkgset &to_remove, pkgset &to_purge)
{
  bool rval=true;

  cmdline_version_source source=cmdline_version_cand;

  string archive;
  string version;

  // Handle task installation.  Won't work if tasksel isn't installed.
  if(task_list->find(s)!=task_list->end())
    {
      task t=(*task_list)[s];

      printf(_("Note: selecting the task \"%s: %s\" for installation\n"),
	     s.c_str(), t.shortdesc.c_str());

      for(pkgCache::PkgIterator pkg=(*apt_cache_file)->PkgBegin();
	  !pkg.end(); ++pkg)
	{
	  std::list<std::string> *tasks=get_tasks(pkg);

	  for(std::list<std::string>::iterator i=tasks->begin();
	      i!=tasks->end(); ++i)
	    if(*i==s)
	      rval=cmdline_applyaction(action, fixer, pkg,
				       to_install, to_hold, to_remove, to_purge,
				       source, archive, version) && rval;
	}

      // break out.
      return rval;
    }

  if(s.find('/')!=s.npos)
    {
      source=cmdline_version_archive;
      // Use the last one.
      string::size_type loc=s.rfind('/');

      archive=string(s, loc+1);
      s.erase(loc);
    }

  if(s.find('=')!=s.npos)
    {
      if(source==cmdline_version_archive)
	{
	  printf(_("You cannot specify both an archive and a version for a package\n"));
	  return false;
	}

      source=cmdline_version_version;
      string::size_type loc=s.rfind('=');

      version=string(s, loc+1);
      s.erase(loc);
    }

  // This is harmless for other commands, but it won't make sense.
  if(source==cmdline_version_version && action!=cmdline_install && action!=cmdline_forbid_version)
    {
      printf(_("You can only specify a package version with an 'install' command.\n"));
      return false;
    }

  if(source==cmdline_version_archive && action!=cmdline_install)
    {
      printf(_("You can only specify a package archive with an 'install' command.\n"));
      return false;
    }

  if(s.find('~')==s.npos)
    {
      pkgCache::PkgIterator pkg=(*apt_cache_file)->FindPkg(s.c_str());
      if(pkg.end())
	{
	  // Maybe they misspelled the package name?
	  pkgvector possible;
	  pkg_matcher *m=parse_pattern(s);

	  if(!m)
	    {
	      _error->Error("Badly formed pattern %s", s.c_str());
	      _error->DumpErrors();

	      return false;
	    }

	  for(pkgCache::PkgIterator j=(*apt_cache_file)->PkgBegin();
	      !j.end(); ++j)
	    {
	      if(m->matches(j, j.CurrentVer()))
		possible.push_back(j);
	    }

	  delete m;

	  if(!possible.empty())
	    {
	      // Don't overwhelm the user.
	      if(possible.size()>40)
		printf(_("Couldn't find package \"%s\", and more than 40\npackages contain \"%s\" in their name.\n"), s.c_str(), s.c_str());
	      else
		{
		  printf(_("Couldn't find package \"%s\".  However, the following\npackages contain \"%s\" in their name:\n"), s.c_str(), s.c_str());
		  cmdline_show_pkglist(possible);
		}
	    }
	  else
	    {
	      m=parse_pattern("~d"+s);

	      if(!m)
		{
		  _error->DumpErrors();
		  return false;
		}

	      for(pkgCache::PkgIterator j=(*apt_cache_file)->PkgBegin();
		  !j.end(); ++j)
		{
		  if(m->matches(j, (*apt_cache_file)[j].InstVerIter(*apt_cache_file)))
		    possible.push_back(j);
		}

	      delete m;

	      if(possible.empty())
		printf(_("Couldn't find any package whose name or description matched \"%s\"\n"), s.c_str());
	      else if(possible.size()>40)
		printf(_("Couldn't find any package matching \"%s\", and more than 40\npackages contain \"%s\" in their description.\n"), s.c_str(), s.c_str());
	      else
		{
		  printf(_("Couldn't find any package matching \"%s\".  However, the following\npackages contain \"%s\" in their description:\n"), s.c_str(), s.c_str());
		  cmdline_show_pkglist(possible);
		}
	    }

	  return false;
	}

      rval=cmdline_applyaction(action, fixer, pkg,
			       to_install, to_hold, to_remove, to_purge,
			       source, archive, version);
    }
  else
    {
      pkg_matcher *m=parse_pattern(s.c_str());
      if(!m)
	{
	  _error->DumpErrors();
	  return false;
	}

      for(pkgCache::PkgIterator pkg=(*apt_cache_file)->PkgBegin();
	  !pkg.end(); ++pkg)
	if(m->matches(pkg, (*apt_cache_file)[pkg].InstVerIter(**apt_cache_file)))
	  rval=cmdline_applyaction(action, fixer, pkg,
				   to_install, to_hold, to_remove, to_purge,
				   source, archive, version) && rval;

      delete m;
    }

  return rval;
}

// Parses a list of actions and does them, running a fixer afterwards.
// Aborts after a partial action if something bad happens.
static void cmdline_parse_action(string s,
				 pkgset &to_install, pkgset &to_hold,
				 pkgset &to_remove, pkgset &to_purge)
{
  string::size_type loc=0;
  pkgProblemResolver fixer(*apt_cache_file);

  while(loc<s.size() && isspace(s[loc]))
    ++loc;

  if(loc==s.size())
    return;

  cmdline_pkgaction_type action;

  switch(s[loc])
    {
    case '_':
      action=cmdline_purge;
      break;
    case '-':
      action=cmdline_remove;
      break;
    case '=':
      action=cmdline_hold;
      break;
    case '+':
      action=cmdline_install;
      break;
    default:
      printf(_("Bad action character '%c'\n"), s[0]);
      return;
    }

  ++loc;

  while(loc<s.size())
    {
      while(loc<s.size() && isspace(s[loc]))
	++loc;

      if(loc<s.size())
	switch(s[loc])
	  {
	  case '_':
	    action=cmdline_purge;
	    ++loc;
	    break;
	  case '-':
	    action=cmdline_remove;
	    ++loc;
	    break;
	  case '=':
	    action=cmdline_hold;
	    ++loc;
	    break;
	  case '+':
	    action=cmdline_install;
	    ++loc;
	    break;
	  default:
	    {
	      string pkgname;
	      while(loc<s.size() && !isspace(s[loc]))
		pkgname+=s[loc++];

	      if(!cmdline_applyaction(pkgname, action, fixer,
				      to_install, to_hold,
				      to_remove, to_purge))
		return;
	    }
	  }
    }

  // Er, what do we do here?
  if(!(*apt_cache_file)->try_fix_broken(fixer, NULL))
    _error->DumpErrors();
}

static bool cmdline_do_prompt(bool as_upgrade,
			      pkgset &to_install,
			      pkgset &to_hold,
			      pkgset &to_remove,
			      pkgset &to_purge,
			      bool showvers,
			      bool showdeps,
			      bool showsize,
			      bool always_prompt,
			      int verbose,
			      bool assume_yes)
{
  bool cont=false;
  bool rval=true;
  bool first=true;

  while(!cont)
    {
      if(!cmdline_show_preview(true, to_install, to_hold, to_remove,
			       showvers, showdeps, showsize) &&
	 first &&
	 !always_prompt)
	cont=true;
      else if(assume_yes)
	cont=true;
      else
	{
	  bool valid_response=false;

	  while(!valid_response)
	    {
	      valid_response=true;
	      printf(_("Do you want to continue? [Y/n/?] "));
	      fflush(stdout);

	      string response=prompt_string();
	      string::size_type loc=0;

	      while(loc<response.size() && isspace(response[loc]))
		++loc;

	      if(loc==response.size())
		response='y';

	      switch(toupper(response[0]))
		{
		case 'Y':
		  rval=true;
		  cont=true;
		  break;
		case 'N':
		  rval=false;
		  cont=true;
		  break;
		case 'D':
		  showdeps=!showdeps;
		  if(showdeps)
		    printf(_("\nDependency information will be shown.\n\n"));
		  else
		    printf(_("\nDependency information will not be shown.\n\n"));
		  break;
		case 'V':
		  showvers=!showvers;

		  if(showvers)
		    printf(_("\nVersions will be shown.\n\n"));
		  else
		    printf(_("\nVersions will not be shown.\n\n"));
		  break;
		case 'S':
		  showsize=!showsize;
		  if(showsize)
		    printf(_("\nSize changes will be shown.\n\n"));
		  else
		    printf(_("\nSize changes will not be shown.\n\n"));
		  break;
		case '+':
		case '-':
		case '=':
		case '_':
		  cmdline_parse_action(response, to_install, to_hold,
				       to_remove, to_purge);
		  break;
		case 'E':
		  // FIXME: exit(0)ing from here is a hack.
		  ui_init();
		  file_quit.connect(SigC::slot(vscreen_exitmain));
		  do_package_run_or_show_preview();
		  ui_main();
		  exit(0);
		case '?':
		  valid_response=false;
		  printf(_("Commands:\n  y: continue with the installation\n  n: abort and quit\n  d: toggle display of dependency information\n  s: toggle display of changes in package sizes\n  v: toggle display of versions\n  e: enter the full visual interface\n\n  You may also specify modifications to the set of package actions.\n  To do so, type an action character followed by one or more package names\nor patterns.  The action will be applied to all packages. (you may specify\nadditional actions; each will be applied to all packages that follow it)\n\nActions:\n  + : Install\n  - : Remove\n  _ : Purge\n  = : Hold\n"));
		  break;
		default:
		  printf(_("Invalid response.  Please enter a valid command or '?' for help.\n"));
		  valid_response=false;
		  break;
		}
	    }
	}

      if(!prompt_essential())
	{
	  rval=false;
	  cont=true;
	}

      first=false;
    }

  return rval;
}

int do_simulate(bool as_upgrade,
		pkgset &to_install, pkgset &to_hold, pkgset &to_remove,
		pkgset &to_purge,
		bool showvers, bool showdeps, bool showsize,
		bool always_prompt, int verbose, bool assume_yes)
{
  if(!cmdline_do_prompt(as_upgrade,
			to_install, to_hold, to_remove, to_purge,
			showvers, showdeps, showsize, always_prompt, verbose,
			assume_yes))
    {
      printf(_("Abort.\n"));
      return 0;
    }

  if(verbose==0)
    {
      printf(_("Would download/install/remove packages.\n"));
      return 0;
    }

  pkgSimulate PM(*apt_cache_file);
  pkgPackageManager::OrderResult Res=PM.DoInstall();

  if(Res==pkgPackageManager::Failed)
    return -1;
  else if(Res!=pkgPackageManager::Completed)
    {
      _error->Error(_("Internal Error, Ordering didn't finish"));
      return -1;
    }
  else
    return 0;
}

// TODO: add an option to only update the selection states, not do a
//     command run.
//
// TODO: add an option to obey sticky states even if it wasn't
//      explicitly requested.
//
// TODO: perhaps when trying to find a list of possible candidates for
// installation, we should use a formatted display?
int cmdline_pkgaction(int argc, char *argv[],
		      const char *status_fname, bool simulate,
		      bool assume_yes, bool download_only, bool fix_broken,
		      bool showvers, bool showdeps, bool showsize,
		      bool always_prompt, int verbose)
{
  _error->DumpErrors();

  cmdline_pkgaction_type default_action=cmdline_install;
  bool dist_upgrade=false;

  // Parse the action.
  if(!strcasecmp(argv[0], "install"))
    default_action=cmdline_install;
  else if(!strcasecmp(argv[0], "reinstall"))
    default_action=cmdline_reinstall;
  else if(!strcasecmp(argv[0], "dist-upgrade"))
    {
      default_action=cmdline_install;
      dist_upgrade=true;
    }
  else if(!strcasecmp(argv[0], "remove"))
    default_action=cmdline_remove;
  else if(!strcasecmp(argv[0], "purge"))
    default_action=cmdline_purge;
  else if(!strcasecmp(argv[0], "hold"))
    default_action=cmdline_hold;
  else if(!strcasecmp(argv[0], "unhold"))
    default_action=cmdline_unhold;
  else if(!strcasecmp(argv[0], "markauto"))
    default_action=cmdline_markauto;
  else if(!strcasecmp(argv[0], "unmarkauto"))
    default_action=cmdline_unmarkauto;
  else if(!strcasecmp(argv[0], "forbid-version"))
    default_action=cmdline_forbid_version;
  else
    {
      // Should never happen.
      _error->Error(_("Invalid operation %s"), argv[0]);
      _error->DumpErrors();
      return -1;
    }

  OpTextProgress progress;

  aptcfg->Set(PACKAGE "::Auto-Upgrade", "false");

  // If there are no arguments and the default action is "install", act
  // on stickies.
  //
  // This way, "aptitude install" will just perform any pending
  // installations.
  apt_init(&progress, (argc==1 && default_action==cmdline_install &&
		       !dist_upgrade), status_fname);

  if(_error->PendingError())
    {
      _error->DumpErrors();
      return -1;
    }

  // In case we aren't root.
  if(!simulate)
    apt_cache_file->GainLock();
  else
    apt_cache_file->ReleaseLock();

  if(_error->PendingError())
    {
      _error->DumpErrors();
      return -1;
    }

  pkgset to_install, to_hold, to_remove, to_purge;

  if(dist_upgrade)
    {
      // Build to_install to avoid a big printout
      for(pkgCache::PkgIterator i=(*apt_cache_file)->PkgBegin(); !i.end(); ++i)
	{
	  pkgDepCache::StateCache &state=(*apt_cache_file)[i];

	  if(!i.CurrentVer().end() &&
	     state.Upgradable() && !(*apt_cache_file)->is_held(i))
	    to_install.insert(i);

	}

      (*apt_cache_file)->mark_all_upgradable(true, NULL);
    }
  else if(argc==1 && default_action==cmdline_install)
    {
      // FIXME: Build to_install to avoid a big printout
      for(pkgCache::PkgIterator i=(*apt_cache_file)->PkgBegin(); !i.end(); ++i)
	{

	}
    }

  // TODO: look for filenames and call dpkg directly if that's the case.

  pkgProblemResolver fixer(*apt_cache_file);
  // Used when doing only a local optimization (ie, we aren't trying to
  // do a dist-upgrade)

  (*apt_cache_file)->begin_action_group();
  // Mark packages 'n stuff.
  for(int i=1; i<argc; ++i)
    {
      cmdline_pkgaction_type action=default_action;
      int tmp=strlen(argv[i])-1;

      // HACK: disable interpreting of escapes if it's an existing
      //      package name.
      if((*apt_cache_file)->FindPkg(argv[i]).end())
	switch(argv[i][tmp])
	  {
	  case '-':
	    action=cmdline_remove;
	    argv[i][tmp]=0;
	    break;
	  case '=':
	    action=cmdline_hold;
	    argv[i][tmp]=0;
	    break;
	  case '+':
	    action=cmdline_install;
	    argv[i][tmp]=0;
	    break;
	  case '_':
	    action=cmdline_purge;
	    argv[i][tmp]=0;
	    break;
	  }

      cmdline_applyaction(argv[i], action, fixer,
			  to_install, to_hold, to_remove, to_purge);
    }
  (*apt_cache_file)->end_action_group(NULL);

  if(dist_upgrade ||
     (fix_broken && argc==1 && default_action==cmdline_install))
    {
      if(!(*apt_cache_file)->try_fix_broken(NULL))
	{
	  _error->DumpErrors();
	  return -1;
	}
    }
  else
    {
      if(!(*apt_cache_file)->try_fix_broken(fixer, NULL))
	_error->Error("Unable to resolve some dependencies!");
    }

  _error->DumpErrors();

  if(!show_broken())
    return -1;

  if(simulate)
    return do_simulate(dist_upgrade, to_install, to_hold, to_remove, to_purge,
		       showvers, showdeps, showsize,
		       always_prompt, verbose, assume_yes);
  else
    {
      if(!cmdline_do_prompt(dist_upgrade,
			    to_install, to_hold, to_remove, to_purge,
			    showvers, showdeps, showsize,
			    always_prompt, verbose, assume_yes))
	{
	  printf(_("Abort.\n"));
	  return 0;
	}

      int rval=do_install_run(&progress, true, download_only)?0:-1;

      if(_error->PendingError())
	rval=-1;

      _error->DumpErrors();

      return rval;
    }
}

int cmdline_upgrade(int argc, char *argv[],
		    const char *status_fname, bool simulate,
		    bool assume_yes, bool download_only,
		    bool showvers, bool showdeps, bool showsize,
		    bool always_prompt, int verbose)
{
  pkgset to_install, to_hold, to_remove, to_purge;

  _error->DumpErrors();

  OpTextProgress progress;
  apt_init(&progress, false, status_fname);

  if(_error->PendingError())
    {
      _error->DumpErrors();
      return -1;
    }

  if(!simulate)
    apt_cache_file->GainLock();
  else
    apt_cache_file->ReleaseLock();

  if(_error->PendingError())
    {
      _error->DumpErrors();
      return -1;
    }

  // Build to_install to avoid a big printout
  for(pkgCache::PkgIterator i=(*apt_cache_file)->PkgBegin(); !i.end(); ++i)
    {
      pkgDepCache::StateCache &state=(*apt_cache_file)[i];

      if(!i.CurrentVer().end() &&
	 state.Upgradable() && !(*apt_cache_file)->is_held(i))
	to_install.insert(i);
    }

  if(!(*apt_cache_file)->all_upgrade(NULL))
    {
      _error->DumpErrors();
      return -1;
    }

  if(!show_broken())
    return -1;

  if(simulate)
    return do_simulate(true, to_install, to_hold, to_remove, to_purge,
		       showvers, showdeps, showsize,
		       always_prompt, verbose, assume_yes);
  else
    {
      if(!cmdline_do_prompt(true, to_install, to_hold, to_remove,
			    to_purge, showvers, showdeps, showsize,
			    always_prompt, verbose, assume_yes))
	{
	  printf(_("Abort.\n"));
	  return 0;
	}

      int rval=do_install_run(&progress, true, download_only)?0:-1;

      if(_error->PendingError())
	rval=-1;

      _error->DumpErrors();

      return rval;
    }
}

// Download stuff to the current directory
int cmdline_download(int argc, char *argv[])
{
  if(argc<=1)
    {
      printf(_("download: you must specify at least one package to download\n"));
      return -1;
    }

  update_screen_width();

  _error->DumpErrors();

  OpTextProgress progress;
  apt_init(&progress, false);

  pkgSourceList list;
  if(!list.ReadMainList())
    {
      _error->Error(_("Couldn't read source list"));

      _error->DumpErrors();
      return -1;
    }
  pkgAcquire fetcher(gen_cmdline_download_progress());
  string filenames[(*apt_cache_file)->Head().PackageCount];

  for(int i=1; i<argc; ++i)
    // FIXME: use the same logic as pkgaction here.
    if(!strchr(argv[i], '~'))
      {
	pkgCache::PkgIterator pkg=(*apt_cache_file)->FindPkg(argv[i]);
	if(!pkg.end())
	  {
	    if((*apt_cache_file)[pkg].CandidateVerIter(*apt_cache_file).end())
	      _error->Error("No candidate version for %s", pkg.Name());
	    else
	      get_archive(&fetcher, &list, apt_package_records,
			  (*apt_cache_file)[pkg].CandidateVerIter(*apt_cache_file),
			  ".", filenames[pkg->ID]);
	  }
      }

  if(fetcher.Run()!=pkgAcquire::Continue)
    // We failed or were cancelled
    {
      _error->DumpErrors();
      return -1;
    }

  return 0;
}

static fragment *dep_lst_frag(pkgCache::DepIterator dep,
			      string title, pkgCache::Dep::DepType T)
{
  vector<fragment *> fragments;

  while(!dep.end())
    {
      pkgCache::DepIterator start, end;
      dep.GlobOr(start, end);

      if(start->Type==T)
	do
	  {
	    fragment *verfrag;

	    if(start.TargetVer())
	      verfrag=fragf(" (%s %s)", start.CompType(), start.TargetVer());
	    else
	      verfrag=fragf("");

	    fragments.push_back(fragf("%s%F",
				      start.TargetPkg().Name(),
				      verfrag));

	    if(start==end)
	      break;
	    ++start;
	  } while(1);
    }

  if(fragments.size()==0)
    return fragf("");
  else
    return fragf("%s: %F",
		 title.c_str(),
		 indentbox(0, title.size()+2,
			   flowbox(join_fragments(fragments, ", "))));
}

static fragment *prv_lst_frag(pkgCache::PrvIterator prv,
			      bool reverse,
			      const std::string &title)
{
  vector<fragment *> fragments;

  for( ; !prv.end(); ++prv)
    {
      string name=reverse?prv.OwnerPkg().Name():prv.ParentPkg().Name();

      fragments.push_back(text_fragment(name));
    }

  if(fragments.size()==0)
    return fragf("");
  else
    return fragf("%s: %F",
		 title.c_str(),
		 indentbox(0, title.size()+2,
			   flowbox(join_fragments(fragments, ", "))));
}

static fragment *archive_lst_frag(pkgCache::VerFileIterator vf,
				  const std::string &title)
{
  vector<fragment *> fragments;

  for( ; !vf.end(); ++vf)
    fragments.push_back(text_fragment(vf.File().Archive()));

  if(fragments.size()==0)
    return fragf("");
  else
    return fragf("%s: %F",
		 title.c_str(),
		 indentbox(0, title.size()+2,
			   flowbox(join_fragments(fragments, ", "))));
}

// Shows only information about a package
static void show_package(pkgCache::PkgIterator pkg, int verbose)
{
  vector<fragment *> fragments;

  fragments.push_back(fragf("%s%s%n", _("Package: "), pkg.Name()));
  fragments.push_back(prv_lst_frag(pkg.ProvidesList(), true, _("Provided by")));

  fragment *f=sequence_fragment(fragments);

  cout << f->layout(screen_width);

  delete f;
}

static fragment *version_file_fragment(pkgCache::VerIterator ver,
				       pkgCache::VerFileIterator vf,
				       int verbose)
{
  vector<fragment *> fragments;

  pkgCache::PkgIterator pkg=ver.ParentPkg();
  pkgRecords::Parser &rec=apt_package_records->Lookup(vf);


  fragments.push_back(fragf("%s%s%n", _("Package: "), pkg.Name()));
  fragments.push_back(fragf("%s%s%n", _("Version: "), ver.VerStr()));
  fragments.push_back(fragf("%s%s%n", _("Priority: "),
			    ver.PriorityType()?ver.PriorityType():_("N/A")));
  fragments.push_back(fragf("%s%s%n", _("Section: "),
			    ver.Section()?ver.Section():_("N/A")));
  fragments.push_back(fragf("%s%s%n", _("Maintainer: "),
			    rec.Maintainer().c_str()));

  fragments.push_back(fragf("%s%s%n", _("Uncompressed Size: "),
			    SizeToStr(ver->InstalledSize).c_str()));
  if(verbose>0)
    {
      fragments.push_back(fragf("%s%s%n", _("Architecture: "),
				ver.Arch()?ver.Arch():_("N/A")));
      fragments.push_back(fragf("%s%s%n", _("Compressed Size: "),
				SizeToStr(ver->Size).c_str()));
      fragments.push_back(fragf("%s%s%n", _("Filename: "),
				rec.FileName().c_str()));
      fragments.push_back(fragf("%s%s%n", _("MD5sum: "),
				rec.MD5Hash().c_str()));

      if(verbose<2) // Show all archives in a list.
	fragments.push_back(archive_lst_frag(ver.FileList(), _("Archive")));
      else
	fragments.push_back(fragf("%s: %s%n", _("Archive"), vf.File().Archive()));
    }

  fragments.push_back(dep_lst_frag(ver.DependsList(),
				   _("Depends"), pkgCache::Dep::Depends));
  fragments.push_back(dep_lst_frag(ver.DependsList(),
				   _("PreDepends"), pkgCache::Dep::PreDepends));
  fragments.push_back(dep_lst_frag(ver.DependsList(),
				   _("Recommends"), pkgCache::Dep::Recommends));
  fragments.push_back(dep_lst_frag(ver.DependsList(),
				   _("Suggests"), pkgCache::Dep::Suggests));
  fragments.push_back(dep_lst_frag(ver.DependsList(),
				   _("Conflicts"), pkgCache::Dep::Conflicts));
  fragments.push_back(dep_lst_frag(ver.DependsList(),
				   _("Replaces"), pkgCache::Dep::Replaces));
  fragments.push_back(dep_lst_frag(ver.DependsList(),
				   _("Obsoletes"), pkgCache::Dep::Obsoletes));

  fragments.push_back(prv_lst_frag(ver.ProvidesList(), false, _("Provides")));
  fragments.push_back(prv_lst_frag(ver.ParentPkg().ProvidesList(), true, _("Provided by")));

  fragments.push_back(fragf("%s%s%n",
			    _("Description: "), rec.ShortDesc().c_str()));
  fragments.push_back(indentbox(1, 1, make_desc_fragment(rec.LongDesc())));

  return sequence_fragment(fragments);
}

static void show_version(pkgCache::VerIterator ver, int verbose)
{
  if(ver.FileList().end())
    {
      fragment *f=version_file_fragment(ver, ver.FileList(), verbose);

      cout << f->layout(screen_width);

      delete f;
    }
  else
    {
      for(pkgCache::VerFileIterator vf=ver.FileList(); !vf.end(); ++vf)
	{
	  fragment *f=version_file_fragment(ver, vf, verbose);

	  cout << f->layout(screen_width) << endl;

	  delete f;

	  // If verbose<2, only show the first file.
	  if(verbose<2)
	    break;
	}
    }
}

int cmdline_show(int argc, char *argv[], int verbose)
{
  update_screen_width();

  _error->DumpErrors();

  OpProgress progress;
  apt_init(&progress, false);

  if(_error->PendingError())
    {
      _error->DumpErrors();
      return -1;
    }

  for(int i=1; i<argc; ++i)
    {
      pkgCache::PkgIterator pkg=(*apt_cache_file)->FindPkg(argv[i]);
      if(!pkg.end())
	{
	  if(!pkg.VersionList().end())
	    for(pkgCache::VerIterator V=pkg.VersionList(); !V.end(); ++V)
	      {
		show_version(V, verbose);
		if(verbose==0)
		  break;
	      }
	  else
	    show_package(pkg, verbose);
	}
      else
	{
	  pkg_matcher *m=parse_pattern(argv[i]);

	  if(!m)
	    {
	      _error->Error("Unable to parse pattern %s", argv[i]);
	      _error->DumpErrors();
	      return -1;
	    }

	  for(pkgCache::PkgIterator P=(*apt_cache_file)->PkgBegin();
	      !P.end(); ++P)
	    {
	      for(pkgCache::VerIterator V=P.VersionList(); !V.end(); ++V)
		if(m->matches(P, V))
		  {
		    show_version(V, verbose);
		    if(verbose==0)
		      break;
		  }
	    }

	  delete m;
	}
    }

  return 0;
}

int cmdline_moo(int argc, char *argv[], int verbose)
{
  switch(verbose)
    {
    case 0:
      printf(_("There are no Easter Eggs in this program.\n"));
      break;
    case 1:
      printf(_("There really are no Easter Eggs in this program.\n"));
      break;
    case 2:
      printf(_("Didn't I already tell you that there are no Easter Eggs in this program?\n"));
      break;
    case 3:
      printf(_("Stop it!\n"));
      break;
    case 4:
      printf(_("Okay, okay, if I give you an Easter Egg, will you go away?\n"));
      break;
    case 5:
      printf(_("All right, you win.\n"));
      printf("\n");
      printf("                               /----\\\n");
      printf("                       -------/      \\\n");
      printf("                      /               \\\n");
      printf("                     /                |\n");
      printf("   -----------------/                  --------\\\n");
      printf("   ----------------------------------------------\n");
      printf("\n");
      printf(_("Happy?"));
      break;

    case 6:
      printf(_("What is it?  It's an elephant being eaten by a snake, of course.\nAre you dense?\n"));
      break;

    default:
      printf(_("Go away, I'm trying to think.\n"));
      break;
    }

  return 0;
}
