/* cmd-whereis-archive.c
 *
 * vim:smartindent ts=8:sts=2:sta:et:ai:shiftwidth=2
 ****************************************************************
 * Copyright (C) 2003 Tom Lord
 *
 * See the file "COPYING" for further information about
 * the copyright and warranty status of this work.
 */


#include "config-options.h"
#include "po/gettext.h"
#include "hackerlab/bugs/exception.h"
#include "hackerlab/cmd/main.h"
#include "libfsutils/ensure-dir.h"
#include "libarch/my.h"
#include "libarch/namespace.h"
#include "libarch/archives.h"
#include "libarch/pfs.h"
#include "libarch/project-tree.h"
#include "commands/whereis-archive.h"
#include "commands/version.h"



static t_uchar * usage = N_("[options] archive");

#define OPTS(OP) \
  OP (opt_help_msg, "h", "help", 0, \
      N_("Display a help message and exit.")) \
  OP (opt_long_help, "H", 0, 0, \
      N_("Display a verbose help message and exit.")) \
  OP (opt_version, "V", "version", 0, \
      N_("Display a release identifier string\n" \
      "and exit.")) \
  OP (opt_all_locations , NULL, "all-locations", 0, \
      N_("Show all known locations for this archive name"))

t_uchar arch_cmd_whereis_archive_help[] = N_("print an archive location registration\n"
                                           "Print the registered location of an archive.\n"
                                           "\n"
                                           "Usually the archive must have been previously registered with\n"
                                           "\"baz register-archive\".\n"
                                           "\n"
                                           "As a special exception, the the archive is not registered, but\n"
                                           "is the name of the archive rooted at the location given with\n"
                                           "the option -R (--root) or in the environment variable ARCHROOT\n"
                                           "then print that root directory.\n");

enum options
{
  OPTS (OPT_ENUM)
};

static struct opt_desc opts[] =
{
  OPTS (OPT_DESC)
    {-1, 0, 0, 0, 0}
};



int
arch_cmd_whereis_archive (t_uchar * program_name, int argc, char * argv[])
{
  int o;
  struct opt_parsed * option;
  t_uchar * which_archive;
  int all_locations = 0;
  ar_archive_location locations;
  
  safe_buffer_fd (1, 0, O_WRONLY, 0);

  option = 0;

  while (1)
    {
      o = opt_standard (lim_use_must_malloc, &option, opts, &argc, argv, program_name, usage, libarch_version_string, arch_cmd_whereis_archive_help, opt_help_msg, opt_long_help, opt_version);
      if (o == opt_none)
        break;
      switch (o)
        {
        default:
          safe_printfmt (2, "unhandled option `%s'\n", option->opt_string);
          panic ("internal error parsing arguments");

        usage_error:
          opt_usage (2, argv[0], program_name, usage, 1);
          exit (1);

          /* bogus_arg: */
          safe_printfmt (2, "ill-formed argument for `%s' (`%s')\n", option->opt_string, option->arg_string);
          goto usage_error;
	case opt_all_locations:
	  all_locations = 1;
          break;
        }
    }

  if (argc == 1)
    {
      arch_project_tree_t * tree = arch_project_tree_new (talloc_context, ".");
      which_archive = str_save (0, tree->archive);
      arch_project_tree_delete (tree);
    }
  else if (argc == 2)
    {
      which_archive = str_save (0, argv[1]);
    }
  else
    goto usage_error;

  if (!arch_valid_archive_name (which_archive))
    {
      safe_printfmt (2, "%s: invalid archive name (%s)\n",
                     argv[0], which_archive);
      exit (2);
    }

  locations = arch_archive_locations (which_archive);
  if (all_locations)
    {
      int index;
      ar_for_each (locations, index)
        {
          t_uchar *temp_location = unescape_location(locations[index]->url);
          safe_printfmt (1, "%s\n", temp_location);
          lim_free (0, temp_location);
        }
    }
  else if (ar_size_archive_location (locations))
    {
      t_uchar *temp_location = unescape_location(locations[0]->url);
      safe_printfmt (1, "%s\n", temp_location);
      lim_free (0, temp_location);
    }
  if (!ar_size_archive_location(locations))
      exit (1);
  ar_free_archive_location (&locations);
  lim_free (0, which_archive);
  return 0;
}




/* tag: Tom Lord Sun May 18 21:37:02 2003 (whereis-archive.c)
 */
