/*
 *
 *  BlueZ - Bluetooth protocol stack for Linux
 *
 *  Copyright (C) 2005-2007  Marcel Holtmann <marcel@holtmann.org>
 *  Copyright (C) 2006-2007  Bastien Nocera <hadess@hadess.net>
 *
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <errno.h>
#include <stdlib.h>
#include <string.h>

#include <dbus/dbus-glib.h>

#include <glib/gi18n.h>

#include <gtk/gtk.h>

#include "general.h"
#include "service.h"
#include "adapter.h"

static void delete_callback(GtkWidget *window, GdkEvent *event,
						gpointer user_data)
{
	gtk_widget_destroy(GTK_WIDGET(window));

	gtk_main_quit();
}

static void close_callback(GtkWidget *button, gpointer user_data)
{
	GtkWidget *window = user_data;

	gtk_widget_destroy(GTK_WIDGET(window));

	gtk_main_quit();
}

static void create_window(GtkWidget *notebook)
{
	GtkWidget *window;
	GtkWidget *widget;
	GtkWidget *vbox;
	GtkWidget *buttonbox;
	GtkWidget *button;

	window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
	gtk_window_set_title(GTK_WINDOW(window), _("Bluetooth Preferences"));
	gtk_window_set_icon_name(GTK_WINDOW(window), "stock_bluetooth");
	gtk_window_set_position(GTK_WINDOW(window), GTK_WIN_POS_CENTER);
	g_signal_connect(G_OBJECT(window), "delete-event",
					G_CALLBACK(delete_callback), NULL);

	vbox = gtk_vbox_new(FALSE, 12);
	gtk_container_set_border_width(GTK_CONTAINER(vbox), 12);
	gtk_container_add(GTK_CONTAINER(window), vbox);

	gtk_box_pack_start(GTK_BOX(vbox), notebook, TRUE, TRUE, 0);

	buttonbox = gtk_hbutton_box_new();
	gtk_button_box_set_layout(GTK_BUTTON_BOX(buttonbox), GTK_BUTTONBOX_END);
	gtk_box_pack_start(GTK_BOX(vbox), buttonbox, FALSE, FALSE, 0);

	button = gtk_button_new_from_stock(GTK_STOCK_CLOSE);
	gtk_container_add(GTK_CONTAINER(buttonbox), button);
	g_signal_connect(G_OBJECT(button), "clicked",
					G_CALLBACK(close_callback), window);

#if 0
	button = gtk_button_new_from_stock(GTK_STOCK_HELP);
	gtk_button_box_set_child_secondary(GTK_BUTTON_BOX(buttonbox),
								button, TRUE);
	gtk_container_add(GTK_CONTAINER(buttonbox), button);
#endif

	widget = create_service();
	if (widget != NULL) {
		gtk_notebook_append_page(GTK_NOTEBOOK(notebook), widget, NULL);
		gtk_notebook_set_tab_label_text(GTK_NOTEBOOK(notebook),
							widget, _("Services"));
	}

	widget = create_general();
	gtk_notebook_append_page(GTK_NOTEBOOK(notebook), widget, NULL);
	gtk_notebook_set_tab_label_text(GTK_NOTEBOOK(notebook),
						widget, _("General"));

	gtk_widget_show_all(window);
}

static void name_owner_changed(DBusGProxy *object, const char *name,
			const char *prev, const char *new, gpointer user_data)
{
	if (!strcmp(name, "org.bluez") && *new == '\0') {
		disable_adapter();
		disable_service();
	}
}

static DBusGProxy *setup_manager(DBusGConnection *conn)
{
	DBusGProxy *proxy;

	proxy = dbus_g_proxy_new_for_name(conn, DBUS_SERVICE_DBUS,
					DBUS_PATH_DBUS, DBUS_INTERFACE_DBUS);

	dbus_g_proxy_add_signal(proxy, "NameOwnerChanged",
		G_TYPE_STRING, G_TYPE_STRING, G_TYPE_STRING, G_TYPE_INVALID);

	dbus_g_proxy_connect_signal(proxy, "NameOwnerChanged",
				G_CALLBACK(name_owner_changed), NULL, NULL);

	setup_service(conn);

	setup_adapter(conn);

	return proxy;
}

int main(int argc, char *argv[])
{
	GtkWidget *notebook;
	DBusGConnection *conn;
	DBusGProxy *manager;
	GError *error = NULL;

	bindtextdomain(GETTEXT_PACKAGE, LOCALEDIR);
	bind_textdomain_codeset(GETTEXT_PACKAGE, "UTF-8");
	textdomain(GETTEXT_PACKAGE);

	gtk_init(&argc, &argv);

	conn = dbus_g_bus_get(DBUS_BUS_SYSTEM, &error);
	if (error != NULL) {
		g_printerr("Connecting to system bus failed: %s\n",
							error->message);
		g_error_free(error);
		exit(EXIT_FAILURE);
	}

	setup_general();

	notebook = gtk_notebook_new();

	assign_adapter(notebook);

	manager = setup_manager(conn);

	create_window(notebook);

	gtk_main();

	cleanup_adapter();

	cleanup_service();

	cleanup_general();

	g_object_unref(manager);

	dbus_g_connection_unref(conn);

	return 0;
}
