/*
 *
 *  BlueZ - Bluetooth protocol stack for Linux
 *
 *  Copyright (C) 2005-2006  Marcel Holtmann <marcel@holtmann.org>
 *
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 */

#include <dbus/dbus-glib.h>

#include "private.h"
#include "manager.h"
#include "adapter.h"

#define BLUETOOTH_MANAGER_GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE((obj), \
				BLUETOOTH_TYPE_MANAGER, BluetoothManagerPrivate))

typedef struct _BluetoothManagerPrivate BluetoothManagerPrivate;

struct _BluetoothManagerPrivate {
	DBusGProxy *proxy;

	GList *adapters;
};

G_DEFINE_TYPE(BluetoothManager, bluetooth_manager, G_TYPE_OBJECT)

static void update_adapter_list(BluetoothManager *self, DBusGProxy *object)
{
	BluetoothAdapter *adapter;
	GError *error = NULL;
	const gchar **array = NULL;

	dbus_g_proxy_call(object, "ListAdapters", &error,
			G_TYPE_INVALID, G_TYPE_STRV, &array, G_TYPE_INVALID);

	if (error != NULL) {
		g_printerr("Getting adapter list failed: %s\n",
							error->message);
		g_error_free(error);
	} else {
		while (*array) {
			//g_printf("%s\n", *array);
			array++;
		}
	}
}

static void adapter_added(DBusGProxy *object,
				const char *path, gpointer user_data)
{
	BluetoothManager *manager = user_data;

	update_adapter_list(manager, object);
}

static void adapter_removed(DBusGProxy *object,
				const char *path, gpointer user_data)
{
	BluetoothManager *manager = user_data;

	update_adapter_list(manager, object);
}

static void bluetooth_manager_init(BluetoothManager *self)
{
	BluetoothManagerPrivate *priv = BLUETOOTH_MANAGER_GET_PRIVATE(self);
	DBusGConnection *conn;
	DBusGProxy *object;

	conn = bluetooth_get_connection();

	object = dbus_g_proxy_new_for_name(conn, "org.bluez",
					"/org/bluez", "org.bluez.Manager");
	if (!object) {
		g_printerr("Creation of manager proxy failed\n");
		return;
	}

	priv->proxy = object;

	dbus_g_proxy_add_signal(object, "AdapterAdded",
					G_TYPE_STRING, G_TYPE_INVALID);

	dbus_g_proxy_connect_signal(object, "AdapterAdded",
				G_CALLBACK(adapter_added), self, NULL);

	dbus_g_proxy_add_signal(object, "AdapterRemoved",
					G_TYPE_STRING, G_TYPE_INVALID);

	dbus_g_proxy_connect_signal(object, "AdapterRemoved",
				G_CALLBACK(adapter_removed), self, NULL);

	update_adapter_list(self, object);
}

static void bluetooth_manager_finalize(GObject *object)
{
	//BluetoothManagerPrivate *priv = BLUETOOTH_MANAGER_GET_PRIVATE(object);
}

static void bluetooth_manager_set_property(GObject *object, guint prop_id,
					const GValue *value, GParamSpec *pspec)
{
	switch (prop_id) {
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
		break;
	}
}
static void bluetooth_manager_get_property(GObject *object, guint prop_id,
					GValue *value, GParamSpec *pspec)
{
	switch (prop_id) {
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
		break;
	}
}

static void bluetooth_manager_class_init(BluetoothManagerClass *klass)
{
	g_type_class_add_private(klass, sizeof(BluetoothManagerPrivate));

	G_OBJECT_CLASS(klass)->finalize = bluetooth_manager_finalize;

	G_OBJECT_CLASS(klass)->set_property = bluetooth_manager_set_property;
	G_OBJECT_CLASS(klass)->get_property = bluetooth_manager_get_property;
}

BluetoothManager *bluetooth_manager_new(void)
{
	return g_object_new(BLUETOOTH_TYPE_MANAGER, NULL);
}
