/*
 * Copyright 2009 Canonical Ltd.
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of either or both of the following licenses:
 *
 * 1) the GNU Lesser General Public License version 3, as published by the
 * Free Software Foundation; and/or
 * 2) the GNU Lesser General Public License version 2.1, as published by
 * the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranties of
 * MERCHANTABILITY, SATISFACTORY QUALITY or FITNESS FOR A PARTICULAR
 * PURPOSE.  See the applicable version of the GNU Lesser General Public
 * License for more details.
 *
 * You should have received a copy of both the GNU Lesser General Public
 * License version 3 and version 2.1 along with this program.  If not, see
 * <http://www.gnu.org/licenses/>
 *
 * Authored by: Neil Jagdish Patel <neil.patel@canonical.com>
 *
 */
/** 
 * SECTION:ctk-effect-cache
 * @short_description: A cache for actors.
 * @include: ctk-effect-cache.h
 *
 * #CtkEffectCache is an implementation of #CtkEffect that caches the rendering of the #CtkActor
 * it's attached to.
 */

#if HAVE_CONFIG_H
#include <config.h>
#endif

#include <math.h>
#include <GL/glew.h>
#include <GL/glxew.h>

#include "ctk-gfx-private.h"
#include "ctk-arb-asm-private.h"
#include "ctk-actor.h"
#include "ctk-effect-context.h"
#include "ctk-effect-cache.h"
#include "ctk-private.h"
#include "ctk-actor.h"

G_DEFINE_TYPE (CtkEffectCache, ctk_effect_cache, CTK_TYPE_EFFECT);

enum
{
  PROP_0,
  CACHE_FACTOR,
  CACHE_COLOR,
};

#define GET_PRIVATE(o) \
  (G_TYPE_INSTANCE_GET_PRIVATE ((o), CTK_TYPE_EFFECT_CACHE, CtkEffectCachePrivate))

struct _CtkEffectCachePrivate
{
  /* opengl texture id use to cache the effect. The texture is updated when the actor is damaged */
  guint cached_effect_texture;
  gboolean cached_init;
  
  gboolean update_texture_cache;
  gboolean invalidate_texture_cache;
};


/* Globals */

/* Forwards */
static void ctk_effect_cache_paint (CtkEffect          *effect,
                                   CtkEffectPaintFunc func,
                                   gboolean           is_last_effect);

static void ctk_effect_cache_paint_dummy (CtkEffect          *effect,
                                         CtkEffectPaintFunc  func,
                                         gboolean            is_last_effect);

static void ctk_effect_cache_set_property (GObject* gobject,
                                            guint prop,
                                            const GValue* value,
                                            GParamSpec* spec);

static void ctk_effect_cache_get_property (GObject* gobject,
                                            guint prop,
                                            GValue* value,
                                            GParamSpec* spec);
/*
 * GObject stuff
 */

static void
ctk_effect_cache_finalize (GObject *object)
{
  CtkEffectCachePrivate *priv = GET_PRIVATE (CTK_EFFECT_CACHE (object));
  g_return_if_fail (CTK_IS_EFFECT_CACHE (object));

  g_return_if_fail (priv);


  if (priv && priv->cached_effect_texture)
    {
      CHECKGL (glDeleteTextures (1, &priv->cached_effect_texture));
    }
  
  G_OBJECT_CLASS (ctk_effect_cache_parent_class)->finalize (object);
}

static void
ctk_effect_cache_class_init (CtkEffectCacheClass *klass)
{
  CtkEffectClass *eff_class = CTK_EFFECT_CLASS (klass);

  GObjectClass*   obj_class = G_OBJECT_CLASS (klass);

  if (ctk_has_capability (CTK_CAPABILITY_VERTEX_PROGRAM)
      && ctk_has_capability (CTK_CAPABILITY_FRAGMENT_PROGRAM)
      && ctk_asm_shaders_compiled_and_ready ())
    eff_class->paint = ctk_effect_cache_paint;
  else
    {
      if (!ctk_has_capability (CTK_CAPABILITY_FBO) || !ctk_has_capability (CTK_CAPABILITY_VERTEX_PROGRAM) || !ctk_has_capability (CTK_CAPABILITY_FRAGMENT_PROGRAM))
        {
          g_message ("WARNING: CtkEffectCache cannot work without FBO and assembly shader program capabiltities");
        }
      else
        {
          if (!ctk_asm_shaders_compiled_and_ready ())
            g_message ("WARNING: Shaders programs are not ready");
        }
      eff_class->paint = ctk_effect_cache_paint_dummy;
    }

  obj_class->finalize     = ctk_effect_cache_finalize;
  obj_class->set_property = ctk_effect_cache_set_property;
  obj_class->get_property = ctk_effect_cache_get_property;

  
  g_type_class_add_private (obj_class, sizeof (CtkEffectCachePrivate));
}

static void
ctk_effect_cache_init (CtkEffectCache *self)
{
  CtkEffectCachePrivate *priv;
  priv = GET_PRIVATE (self);

  priv->cached_effect_texture = 0;
  CHECKGL (glGenTextures (1, &priv->cached_effect_texture));
  CHECKGL (glActiveTextureARB(GL_TEXTURE0) );
  CHECKGL (glBindTexture (GL_TEXTURE_2D, priv->cached_effect_texture));
  CHECKGL (glTexImage2D (GL_TEXTURE_2D, 0, GL_RGBA8, 1, 1, 0, GL_RGBA, GL_UNSIGNED_BYTE, NULL));
  CHECKGL (glTexParameteri (GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR));
  CHECKGL (glTexParameteri (GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR));
  CHECKGL (glTexParameteri (GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP));
  CHECKGL (glTexParameteri (GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP));

  priv->cached_init = FALSE;
  priv->update_texture_cache = FALSE;
  priv->invalidate_texture_cache = TRUE;
}

static void ctk_effect_cache_set_property (GObject* gobject,
                                            guint prop,
                                            const GValue* value,
                                            GParamSpec* spec)
{
  CtkEffectCache*        cache = NULL;
  CtkEffectCachePrivate* priv = NULL;
 
  cache = CTK_EFFECT_CACHE (gobject);
  priv = GET_PRIVATE (cache);
 
  switch (prop)
  {
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (gobject, prop, spec);
    break;
  }
}

static void ctk_effect_cache_get_property (GObject* gobject,
                                            guint prop,
                                            GValue* value,
                                            GParamSpec* spec)
{
  CtkEffectCache*        cache = NULL;
  CtkEffectCachePrivate* priv = NULL;
 
  cache = CTK_EFFECT_CACHE (gobject);
  priv = GET_PRIVATE (cache);
 
  switch (prop)
  {
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (gobject, prop, spec);
    break;
  }
}

/*
 * Private Methods
 */
static void
ctk_effect_cache_paint_dummy (CtkEffect          *effect,
                             CtkEffectPaintFunc  paint_func,
                             gboolean            is_last_effect)
{
  ClutterActor *actor = ctk_effect_get_actor (effect);

  paint_func (actor);
}

static void
ctk_effect_cache_paint (CtkEffect         *effect,
                       CtkEffectPaintFunc paint_func,
                       gboolean           is_last_effect)
{
  ClutterActor*               actor        = NULL;
  CtkEffectContext*           fxctx        = NULL;
  ClutterActor*               stage        = NULL;
  CtkRenderTarget*            top_rt       = NULL;
  CtkRenderTarget*            rt0          = NULL;
  CtkRenderTarget*            rt1          = NULL;
  CtkEffectCachePrivate*      priv         = NULL;
  gfloat                      stage_width  = 0.0f;
  gfloat                      stage_height = 0.0f;
  gfloat actor_screen_width, actor_screen_height;
  gfloat actor_screen_x, actor_screen_y;
  
  ClutterVertex vtx[4];
  //gboolean dirty_effect_cache = TRUE;
  
  actor = ctk_effect_get_actor (effect);
  g_return_if_fail (CLUTTER_IS_ACTOR (actor));

  priv = GET_PRIVATE (CTK_EFFECT_CACHE (effect));
  //dirty_effect_cache = ctk_effect_is_effect_cache_dirty (CTK_EFFECT (effect));
  
  /* before rendering an effect, render everything that has been cahched so far */
  cogl_flush ();
   
  /* get effect-context to which this actor belongs */
  fxctx = ctk_effect_context_get_default_for_actor (actor);

  top_rt = ctk_effect_context_peek_render_target (fxctx);
  
  stage = clutter_actor_get_stage (actor);
  clutter_actor_get_size (CLUTTER_ACTOR (stage), &stage_width, &stage_height);

  ctk_get_actor_screen_position (CTK_ACTOR(actor),
      &actor_screen_x,
      &actor_screen_y,
      &actor_screen_width,
      &actor_screen_height, vtx);

  int BorderMargin = 10;
  
  actor_screen_y = ceilf (actor_screen_y);
  actor_screen_width = ceilf(actor_screen_width) + BorderMargin;
  actor_screen_height = ceilf(actor_screen_height);
  
  
  
  if (/*(priv->invalidate_texture_cache == TRUE) &&*/ priv->update_texture_cache)
    {
      /* reserve 2 render targets */
      rt0 = ctk_effect_context_grab_render_target (fxctx);
      rt1 = ctk_effect_context_grab_render_target (fxctx);
      
      // make sure the render target has a color and depth buffer
      ctk_render_target_resize (rt0, stage_width, stage_height);
      
      ctk_effect_context_push_render_target (fxctx, rt0);
      
      ctk_render_target_bind (rt0);

      /* clear render target and prepare the surface */
      {
        CHECKGL (glViewport (0, 0, stage_width, stage_height));
        CHECKGL (glScissor (0, 0, stage_width, stage_height));
        CHECKGL (glDisable (GL_SCISSOR_TEST));
        CHECKGL (glDisable (GL_ALPHA_TEST));
        CHECKGL (glDisable (GL_STENCIL_TEST));

        /* make sure we are not using a shader program */
        CHECKGL (glClearDepth (1.0f));
        CHECKGL (glClearStencil (0));
        CHECKGL (glClearColor (0.0f, 0.0f, 0.0f, 0.0f));
        CHECKGL (glColorMask (GL_TRUE, GL_TRUE, GL_TRUE, GL_TRUE));
    
        /* clear the FBO content */
        CHECKGL (glClear (GL_COLOR_BUFFER_BIT|GL_DEPTH_BUFFER_BIT|GL_STENCIL_BUFFER_BIT));
      }

      /* Paint the actor and its children. If ctk_actor_set_effects_painting has been set to true
        for this actor, it will draw itself without any effect (avoiding an infinite loop). */
      paint_func (actor);
      cogl_flush ();

      // The actor has been rendered.
      
      if (rt0 != ctk_effect_context_pop_render_target(fxctx))
        {
          /* This is to make sure that all the right push and pop have been while drawing
          the children of this actor.*/
          g_warning("%s() at line %s: Invalid push/pop detected", G_STRFUNC, G_STRLOC);
        }
      
      /* Copy from a stage sized render target to an actor sized render target */
      {
        float u0 = actor_screen_x/(float)ctk_render_target_get_width(rt0);
        float v1 = ((float)ctk_render_target_get_height(rt0) - actor_screen_y)/(float)ctk_render_target_get_height(rt0);
        float u1 = (actor_screen_x+actor_screen_width)/(float)ctk_render_target_get_width(rt0);
        float v0 = ((float)ctk_render_target_get_height(rt0) - (actor_screen_y + actor_screen_height))/(float)ctk_render_target_get_height(rt0);
      
        // Make the rt a little bit bigger than the actor size.
        ctk_render_target_resize(rt1, actor_screen_width, actor_screen_height);
        
        ctk_render_target_bind(rt1);
        CHECKGL( glClear(GL_COLOR_BUFFER_BIT|GL_DEPTH_BUFFER_BIT|GL_STENCIL_BUFFER_BIT) );

        ctk_copy_rendertarget_to_rendertarget_asm (
          rt0,
          u0,
          v0,
          u1,
          v1,
          rt1, 0, 0, actor_screen_width, actor_screen_height);
      }

      /* Save the result in the cache texture */
      {
        ctk_copy_render_target_to_cached_texture_asm(fxctx, rt1, priv->cached_effect_texture);
        ctk_effect_set_invalidate_effect_cache (CTK_EFFECT (effect), FALSE);
      }

      glEnable(GL_BLEND);
      
      priv->update_texture_cache = FALSE;
      priv->invalidate_texture_cache = FALSE;
      
      /* Now draw into the top render target on the stack */
      if(top_rt)
        {
          ctk_render_target_bind(top_rt);
          CHECKGL( glViewport(0, 0, ctk_render_target_get_width(top_rt), ctk_render_target_get_height(top_rt)) );
          CHECKGL( glScissor(0, 0, ctk_render_target_get_width(top_rt), ctk_render_target_get_height(top_rt)) );
        }
      else
        {
          /* There is no render target on the stack. Set back the regular frame buffer */
          ctk_render_target_unbind(); /* unbind whatever render target is binded */
          CHECKGL( glViewport(0, 0, stage_width, stage_height) );
          CHECKGL( glScissor(0, 0, stage_width, stage_height) );
        }
      
      // Paint the newly updated cache texture
      ctk_render_quad_asm (
        priv->cached_effect_texture,
        actor_screen_width,
        actor_screen_height,
        g_shTexture_asm,
        stage_width,
        stage_height,
        actor_screen_x,
        actor_screen_y,
        actor_screen_width,
        actor_screen_height,
        1.0f,
        1.0f,
        1.0f,
        1.0f);
        
    } /* end damaged */
  else if (priv->invalidate_texture_cache == TRUE)
    {
      /* Paint the actor and its children. If ctk_actor_set_effects_painting has been set to true
      for this actor, it will draw itself without any effect (avoiding an infinite loop). */
      paint_func (actor);
      cogl_flush ();
    }
  else
    {
      // reuse the cache texture
      ctk_render_quad_asm (
        priv->cached_effect_texture,
        actor_screen_width,
        actor_screen_height,
        g_shTexture_asm,
        stage_width,
        stage_height,
        actor_screen_x,
        actor_screen_y,
        actor_screen_width,
        actor_screen_height,
        1.0f,
        1.0f,
        1.0f,
        1.0f);
    }


  /* leave and make sure the top render target on the stack is binded */
  if (top_rt)
    ctk_render_target_bind (top_rt);
  else
    ctk_render_target_unbind ();

  /* We cannot test ctk_actor_get_damaged anymore as it may have been changed during the last call to paint_func */
  if (rt0)
    {
      ctk_effect_context_release_render_target (fxctx, rt0);
    }
  if (rt1)
    {
      ctk_effect_context_release_render_target (fxctx, rt1);
    }
}

/*
 * Public Methods
 */

/**
 * ctk_effect_cache_new
 *
 * Returns: A new #CtkEffectCache
 **/
CtkEffect *
ctk_effect_cache_new (void)
{
  return g_object_new (CTK_TYPE_EFFECT_CACHE,
                       NULL);
}

/**
 * ctk_effect_cache_invalidate_texture_cache:
 * @self: a #CtkEffect
 *
 * Set a flags to indicate to that the cached texture is invalid and should not be used. 
 * Actors should be drawn normally.
 *
 **/
void
ctk_effect_cache_invalidate_texture_cache (CtkEffectCache *self)
{
  g_return_if_fail (CTK_IS_EFFECT_CACHE (self));

  GET_PRIVATE(self)->invalidate_texture_cache = TRUE;
  
  ClutterActor* actor = ctk_effect_get_actor (CTK_EFFECT (self));
  if (CLUTTER_IS_ACTOR (actor))
    clutter_actor_queue_redraw (actor);
}

/**
 * ctk_effect_cache_update_texture_cache:
 * @self: a #CtkEffect
 *
 * Set a flags to indicate to initiate the caching into a texture and reuse that texture
 * from now on until ctk_effect_set_invalidate_effect_cache with argument FALSE.
 *
 **/
void
ctk_effect_cache_update_texture_cache (CtkEffectCache *self)
{
  g_return_if_fail (CTK_IS_EFFECT_CACHE (self));

  GET_PRIVATE(self)->update_texture_cache = TRUE;
  
  ClutterActor* actor = ctk_effect_get_actor (CTK_EFFECT (self));
  if (CLUTTER_IS_ACTOR (actor))
    clutter_actor_queue_redraw (actor);
}

