/*
 * Copyright (C) 2009 Canonical Ltd
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Authored by Gord Allott <gord.allott@canonical.com>
 *
 */

#include <glib.h>
#include <stdlib.h>
#include <unistd.h>
#include <clutk/clutk.h>

static void test_menu (void);
static void test_menuitem (void);
static void test_menuseperator (void);
static void test_menu_get_num_items (void);
static void test_menucheck (void);
static void test_menuradio (void);

void
test_menu_create_suite (void)
{
#define DOMAIN "/Menu"

  g_test_add_func (DOMAIN"/Menu", test_menu);
  g_test_add_func (DOMAIN"/MenuItem", test_menuitem);
  g_test_add_func (DOMAIN"/MenuSeperator", test_menuseperator);
  g_test_add_func (DOMAIN"/MenuCheckMenuItem", test_menucheck);
  g_test_add_func (DOMAIN"/MenuRadioMenuItem", test_menuradio);
  g_test_add_func (DOMAIN"/MenuGetNumItems", test_menu_get_num_items);
}

static void
test_menu (void)
{
  ClutterActor *stage;
  CtkMenu      *menu;

  stage = clutter_stage_get_default ();

  menu = ctk_menu_new ();

  clutter_container_add (CLUTTER_CONTAINER (stage), CLUTTER_ACTOR (menu), NULL);
  clutter_actor_show_all (stage);

  clutter_actor_hide (stage);

  g_assert (CTK_IS_MENU (menu));

  clutter_container_remove_actor (CLUTTER_CONTAINER (stage),
                                  CLUTTER_ACTOR (menu));
}

static void
test_menuitem (void)
{
  ClutterActor *stage;
  CtkMenu      *menu;
  CtkMenuItem  *menuitem;

  stage = clutter_stage_get_default ();

  menu = ctk_menu_new ();
  clutter_container_add (CLUTTER_CONTAINER (stage), CLUTTER_ACTOR (menu), NULL);
  clutter_actor_show_all (stage);

  clutter_actor_hide (stage);

  menuitem = ctk_menu_item_new_with_label ("Hello World!");
  ctk_menu_append (menu, CLUTTER_ACTOR (menuitem), TRUE);

  g_assert (CTK_IS_MENU (clutter_actor_get_parent (CLUTTER_ACTOR (menuitem))));
  g_assert (!g_strcmp0(ctk_menu_item_get_label (menuitem), "Hello World!"));

  clutter_container_remove_actor (CLUTTER_CONTAINER (stage),
                                  CLUTTER_ACTOR (menu));
}

static void
test_menuseperator (void)
{
  ClutterActor      *stage;
  CtkMenu           *menu;
  CtkMenuSeperator  *menuseperator;

  stage = clutter_stage_get_default ();

  menu = ctk_menu_new ();
  g_object_ref_sink (menu);
  clutter_container_add (CLUTTER_CONTAINER (stage), CLUTTER_ACTOR (menu), NULL);
  clutter_actor_show_all (stage);

  clutter_actor_hide (stage);

  menuseperator = ctk_menu_seperator_new ();
  ctk_menu_append (menu, CLUTTER_ACTOR (menuseperator), FALSE);

  g_assert (CTK_IS_MENU (clutter_actor_get_parent (CLUTTER_ACTOR (menuseperator))));
  g_assert (CTK_IS_MENU_SEPERATOR (menuseperator));

  clutter_container_remove_actor (CLUTTER_CONTAINER (stage),
                                  CLUTTER_ACTOR (menu));

  ctk_menu_remove_all (menu);
  g_object_unref (menu);
  g_object_unref (menuseperator);
}

static void
test_menucheck (void)
{
  ClutterActor      *stage;
  CtkMenu           *menu;
  CtkCheckMenuItem  *menuitem;

  stage = clutter_stage_get_default ();

  menu = ctk_menu_new ();
  clutter_container_add (CLUTTER_CONTAINER (stage), CLUTTER_ACTOR (menu), NULL);
  clutter_actor_show_all (stage);

  clutter_actor_hide (stage);

  menuitem = ctk_check_menu_item_new_with_label ("foobar!");
  ctk_menu_append (menu, CLUTTER_ACTOR (menuitem), FALSE);

  g_assert (CTK_IS_MENU (clutter_actor_get_parent (CLUTTER_ACTOR (menuitem))));
  g_assert (CTK_IS_CHECK_MENU_ITEM (menuitem));

  ctk_menu_remove_all (menu);
  clutter_container_remove_actor (CLUTTER_CONTAINER (stage),
                                  CLUTTER_ACTOR (menu));
}

static void
test_menuradio (void)
{
  ClutterActor      *stage;
  CtkMenu           *menu;
  CtkRadioMenuItem  *menuitem;
  CtkRadioMenuItem  *menuitem_b;

  stage = clutter_stage_get_default ();

  menu = ctk_menu_new ();
  clutter_container_add (CLUTTER_CONTAINER (stage), CLUTTER_ACTOR (menu), NULL);
  clutter_actor_show_all (stage);

  clutter_actor_hide (stage);

  menuitem = ctk_radio_menu_item_new_with_label (NULL, "foobar1");
  menuitem_b = ctk_radio_menu_item_new_with_label (ctk_radio_menu_item_get_group (menuitem), "foobar2");

  ctk_menu_append (menu, CLUTTER_ACTOR (menuitem), FALSE);
  ctk_menu_append (menu, CLUTTER_ACTOR (menuitem_b), FALSE);

  g_assert (CTK_IS_MENU (clutter_actor_get_parent (CLUTTER_ACTOR (menuitem))));
  g_assert (CTK_IS_CHECK_MENU_ITEM (menuitem));

  ctk_check_menu_item_set_active (CTK_CHECK_MENU_ITEM (menuitem), TRUE);
  g_assert (ctk_check_menu_item_get_active (CTK_CHECK_MENU_ITEM (menuitem)) == TRUE);
  g_assert (ctk_check_menu_item_get_active (CTK_CHECK_MENU_ITEM (menuitem_b)) == FALSE);

  ctk_check_menu_item_set_active (CTK_CHECK_MENU_ITEM (menuitem_b), TRUE);
  g_assert (ctk_check_menu_item_get_active (CTK_CHECK_MENU_ITEM (menuitem)) == FALSE);
  g_assert (ctk_check_menu_item_get_active (CTK_CHECK_MENU_ITEM (menuitem_b)) == TRUE);

  ctk_menu_remove_all (menu);
  clutter_container_remove_actor (CLUTTER_CONTAINER (stage),
                                  CLUTTER_ACTOR (menu));
}


static void
test_menu_get_num_items (void)
{
  ClutterActor *stage;
  CtkMenu      *menu;
  CtkMenuItem  *menuitem;

  stage = clutter_stage_get_default ();

  menu = ctk_menu_new ();
  clutter_container_add (CLUTTER_CONTAINER (stage), CLUTTER_ACTOR (menu), NULL);
  clutter_actor_show_all (stage);

  clutter_actor_hide (stage);

  g_assert_cmpint (ctk_menu_get_num_items (menu), ==, 0);

  menuitem = ctk_menu_item_new_with_label ("item 1");
  ctk_menu_append (menu, CLUTTER_ACTOR (menuitem), TRUE);

  g_assert_cmpint (ctk_menu_get_num_items (menu), ==, 1);

  clutter_container_remove_actor (CLUTTER_CONTAINER (stage),
                                  CLUTTER_ACTOR (menu));
}

