# Copyright 2009 Canonical Ltd.
#
# This file is part of desktopcouch.
#
#  desktopcouch is free software: you can redistribute it and/or modify
# it under the terms of the GNU Lesser General Public License version 3
# as published by the Free Software Foundation.
#
# desktopcouch is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with desktopcouch.  If not, see <http://www.gnu.org/licenses/>.
"Desktop Couch helper files"

from __future__ import with_statement
import os, re
from desktopcouch.start_local_couchdb import process_is_couchdb, read_pidfile

def find_pid(start_if_not_running=True):
    # Work out whether CouchDB is running by looking at its pid file
    pid = read_pidfile()
    if not process_is_couchdb(pid) and start_if_not_running:
        # start CouchDB by running the startup script
        print "Desktop CouchDB is not running; starting it.",
        from desktopcouch import start_local_couchdb
        pid = start_local_couchdb.start_couchdb()
        # now load the design documents, because it's started
        start_local_couchdb.update_design_documents()

        if not process_is_couchdb(pid):
            raise RuntimeError("desktop-couch not started")

    return pid

def find_port__linux(pid=None):
    if pid is None:
        pid = find_pid()

    proc_dir = "/proc/%s" % (pid,)

    # enumerate the process' file descriptors
    fd_dir = os.path.join(proc_dir, 'fd')
    try:
        fd_paths = [os.readlink(os.path.join(fd_dir, fd))
                    for fd in os.listdir(fd_dir)]
    except OSError:
        raise RuntimeError("Unable to find file descriptors in /proc")

    # identify socket fds
    socket_matches = [re.match('socket:\\[([0-9]+)\\]', p) for p in fd_paths]
    # extract their inode numbers
    socket_inodes = [m.group(1) for m in socket_matches if m is not None]

    # construct a subexpression which matches any one of these inodes
    inode_subexp = "|".join(map(re.escape, socket_inodes))
    # construct regexp to match /proc/net/tcp entries which are listening
    # sockets having one of the given inode numbers
    listening_regexp = re.compile(r'''
        \s*\d+:\s*                # sl
        [0-9A-F]{8}:              # local_address part 1
        ([0-9A-F]{4})\s+          # local_address part 2
        00000000:0000\s+          # rem_address
        0A\s+                     # st (0A = listening)
        [0-9A-F]{8}:              # tx_queue
        [0-9A-F]{8}\s+            # rx_queue
        [0-9A-F]{2}:              # tr
        [0-9A-F]{8}\s+            # tm->when
        [0-9A-F]{8}\s*            # retrnsmt
        \d+\s+\d+\s+              # uid, timeout
        (?:%s)\s+                 # inode
        ''' % (inode_subexp,), re.VERBOSE)

    # extract the TCP port from the first matching line in /proc/$pid/net/tcp
    port = None
    with open(os.path.join(proc_dir, 'net', 'tcp')) as tcp_file:
        for line in tcp_file:
            match = listening_regexp.match(line)
            if match is not None:
                port = str(int(match.group(1), 16))
                break
    if port is None:
        raise RuntimeError("Unable to find listening port")

    return port



import platform
os_name = platform.system()
try:
    find_port = {
        "Linux": find_port__linux
        } [os_name]
except KeyError:
    raise NotImplementedError("os %r is not yet supported" % (os_name,))
