# Copyright 2009 Canonical Ltd.
#
# This file is part of desktopcouch.
#
#  desktopcouch is free software: you can redistribute it and/or modify
# it under the terms of the GNU Lesser General Public License version 3
# as published by the Free Software Foundation.
#
# desktopcouch is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with desktopcouch.  If not, see <http://www.gnu.org/licenses/>.
#
# Authors: Eric Casteleijn <eric.casteleijn@canonical.com>

"""Tests for the RecordDict object on which the Contacts API is built."""

from testtools import TestCase
import doctest

# pylint does not like relative imports from containing packages
# pylint: disable-msg=F0401
from desktopcouch.records.server import CouchDatabase
from desktopcouch.records.record import (Record, RecordDict, MergeableList,
    record_factory, IllegalKeyException, validate, NoRecordTypeSpecified)
import desktopcouch.tests as test_environment


class TestRecords(TestCase):
    """Test the record functionality"""

    def setUp(self):
        """Test setup."""
        super(TestRecords, self).setUp()
        self.dict = {
            "a": "A",
            "b": "B",
            "subfield": {
                    "field11s": "value11s",
                    "field12s": "value12s"},
            "subfield_uuid": {
                    "e47455fb-da05-481e-a2c7-88f14d5cc163": {
                        "field11": "value11",
                        "field12": "value12",},
                    "d6d2c23b-279c-45c8-afb2-ec84ee7c81c3": {
                        "field21": "value21",
                        "field22": "value22"}},
            "application_annotations": {
                "my_awesome_app": {"foo": "bar", "some_setting": "set_to"}},
            "record_type": "http://fnord.org/smorgasbord",
        }
        self.record = Record(self.dict)

    def test_revision(self):
        self.assertEquals(self.record.record_revision, None)
        def set_rev(rec): rec.record_revision = "1"
        self.assertRaises(AttributeError, set_rev, self.record)

        ctx = test_environment.test_context
        db = CouchDatabase('testing', create=True, ctx=ctx)
        record_id = db.put_record(self.record)

        db.get_record(record_id)
        self.assertNotEquals(self.record.record_revision, None)

        first = self.record.record_revision

        record_id = db.put_record(self.record)
        db.get_record(record_id)
        second = self.record.record_revision

        self.assertTrue(first < second)

        db.delete_record(record_id)

    def test_delitem(self):
        def f(r):
            del r["_id"]
        self.assertRaises(KeyError, f, self.record)

        del self.record["a"]

    def test_iter(self):
        self.assertEquals(sorted(list(iter(self.record))),
            ['a', 'b', 'record_type', 'subfield', 'subfield_uuid'])

    def test_setitem_internal(self):
        def f(r):
            r["_id"] = "new!"
        self.assertRaises(IllegalKeyException, f, self.record)

    def test_no_record_type(self):
        self.assertRaises(NoRecordTypeSpecified, Record, {})

    def test_get_item(self):
        "Does a RecordDict basically wrap a dict properly?"
        self.assertEqual(self.dict["a"], self.record["a"])
        self.assertRaises(KeyError,
                          self.record.__getitem__, "application_annotations")

    def test_get(self):
        "Does a RecordDict get() work?"
        self.assertEqual(self.dict["a"], self.record.get("a"))
        self.assertEqual(None, self.record.get("application_annotations"))

    def test_keys(self):
        """Test getting the keys."""
        self.assertEqual(
            ['a', 'b', 'record_type', 'subfield', 'subfield_uuid'],
            sorted(self.record.keys()))
        self.assertIn("a", self.record)
        self.assertTrue(self.record.has_key("a"))
        self.assertFalse(self.record.has_key("f"))
        self.assertNotIn("_id", self.record)  # is internal.  play dumb.

    def test_application_annotations(self):
        """Test getting application specific data."""
        my_app_data = self.record.application_annotations["my_awesome_app"]
        self.assertEqual(
            'bar',
            my_app_data['foo'])
        my_app_data["foo"] = "baz"
        self.assertEqual(
            {'my_awesome_app': {'foo': 'baz', "some_setting": "set_to"}},
            self.record._data["application_annotations"])
        self.assertEqual(
            'baz',
            my_app_data['foo'])

    def test_loads_dict_subdict(self):
        "Are subdicts supported?"
        self.assertEqual(2, len(self.record["subfield"]))
        subfield_single = self.record["subfield"]
        self.assertEqual(
            subfield_single["field11s"],
            self.dict["subfield"]["field11s"])

    def test_loads_dict_multi_subdict(self):
        "Are subdicts with multiple entries supported?"
        self.assertEqual(2, len(self.record["subfield_uuid"]))

    def test_mergeable_list_index(self):
        """Test the subset of list behavior we'll be supporting"""
        self.assertEqual("value21", self.record["subfield_uuid"][0]["field21"])
        self.assertEqual("value22", self.record["subfield_uuid"][0]["field22"])
        self.assertEqual("value11", self.record["subfield_uuid"][-1]["field11"])
        self.assertEqual("value12", self.record["subfield_uuid"][-1]["field12"])

    def test_mergeable_list_del(self):
        """Test deletion of uuid keys."""
        del self.record["subfield_uuid"][0]
        self.assertEqual(1, len(self.record["subfield_uuid"]))
        self.assertEqual("value11", self.record["subfield_uuid"][0]["field11"])
        self.assertEqual("value12", self.record["subfield_uuid"][0]["field12"])

    def test_mergeable_list_set_value_in_list_item(self):
        """Test assignment by index."""
        self.record["subfield_uuid"][0]["field12"] = "new exciting value"
        self.assertEqual(
            "new exciting value", self.record["subfield_uuid"][0]["field12"])

    def test_mergeable_list_append(self):
        """Test append."""
        self.record["subfield_uuid"].append(
            {"field31": "value31", "field32": "value32"})
        self.assertEqual(3, len(self.record["subfield_uuid"]))
        self.assertEqual("value32", self.record["subfield_uuid"][-1]["field32"])

    def test_mergeable_list_append_record_dict(self):
        """Test appending a RecordDict value."""
        value = RecordDict({
            "field31": "value31",
            "field32": "value32"})
        self.record["subfield_uuid"].append(value)
        self.assertEqual(3, len(self.record["subfield_uuid"]))
        self.assertEqual("value32", self.record["subfield_uuid"][-1]["field32"])
        
    def test_mergeable_list_remove(self):
        """Test remove a normal value"""
        value = "string"
        self.record["subfield_uuid"].append(value)
        self.record["subfield_uuid"].remove(value)
        self.assertFalse(value in self.record["subfield_uuid"])
        
    def test_mergeable_list_remove_record_dict(self):
        """Test remove a RecordDict value"""
        value = RecordDict({
            "field31": "value31",
            "field32": "value32"})
        self.record["subfield_uuid"].append(value)
        self.record["subfield_uuid"].remove(value)
        self.assertFalse(value in self.record["subfield_uuid"])

    def test_mergeable_list_remove_list(self):
        """Test list removal"""
        value = [1, 2, 3, 4, 5]
        self.record["subfield_uuid"].append(value)
        self.record["subfield_uuid"].remove(value)
        self.assertFalse(value in self.record["subfield_uuid"])
        
    def test_mergeable_list_remove_tuple(self):
        """Test tuple removal"""
        value = (1, 2, 3, 4, 5)
        self.record["subfield_uuid"].append(value)
        self.record["subfield_uuid"].remove(value)
        self.assertFalse(value in self.record["subfield_uuid"])
        
    def test_mergeable_list_remove_missing(self):
        """Test missing data rmeoval"""
        self.assertRaises(ValueError, self.record["subfield_uuid"].remove, 
            "missing_data")
    def test_mergeable_list_remove_last(self):
        """Test that exception is raised when removing last item."""
        self.record["subfield_uuid"] = [1]
        self.assertRaises(ValueError, self.record["subfield_uuid"].remove, 1)
        
    def test_mergeable_list_pop_correct_index(self):
        """Test the pop method when working with a correct index."""
        value = [1, 2, 3, 4, 5]
        self.record["subfield_uuid"] = value
        # test with negative index
        poped_value = self.record["subfield_uuid"].pop(-2)
        self.assertEqual(value[-2], poped_value)
        # test with positive index
        poped_value = self.record["subfield_uuid"].pop(1)
        self.assertEqual(value[1], poped_value)
        
    def test_mergeable_list_pop_wrong_index(self):
        """Test pop when index is out or range."""
        value = [1, 2, 3, 4, 5]
        self.record["subfield_uuid"] = value
        self.assertRaises(IndexError, self.record["subfield_uuid"].pop, 
            len(value)*2)
    
    def test_mergeable_list_pop_last(self):
        """Test that exception is raised when poping last item"""
        self.record["subfield_uuid"] = [1]
        self.assertRaises(ValueError, self.record["subfield_uuid"].pop, 0)
        
    def test_tuple(self):
        """Test assigning tuples to a key results in mergeable lists."""
        rec = Record({'record_type': 'http://fnord.org/smorgasbord'})
        rec['key'] = (1, 2, 3, 4)
        self.assert_(isinstance(rec['key'], MergeableList))
        self.assertEqual([1, 2, 3, 4], [value for value in rec['key']])

    def test_list(self):
        """Test assigning lists to a key results in mergeable lists."""
        rec = Record({'record_type': 'http://fnord.org/smorgasbord'})
        rec['key'] = [1, 2, 3, 4]
        self.assert_(isinstance(rec['key'], MergeableList))
        self.assertEqual([1, 2, 3, 4], [value for value in rec['key']])

    def test_dictionary_access_to_mergeable_list(self):
        """Test that appropriate errors are raised."""
        keys = self.record["subfield_uuid"]._data.keys()
        self.assertRaises(
            TypeError,
            self.record["subfield_uuid"].__getitem__, keys[0])
        self.assertRaises(
            TypeError,
            self.record["subfield_uuid"].__setitem__, keys[0], 'stuff')

    def test_validate(self):
        """Test that validation of mixed keys raises appropriate
        error."""
        self.assertRaises(
            IllegalKeyException,
            validate,
            {'foo': 1, 'bar': 2, 'd6d2c23b-279c-45c8-afb2-ec84ee7c81c3': 3})
        self.assertRaises(
            IllegalKeyException,
            validate,
            {'baz':
             {'foo': 1, 'bar': 2, 'd6d2c23b-279c-45c8-afb2-ec84ee7c81c3': 3},
             'qux': 5})

    def test_uuid_like_keys(self):
        """Test that appropriate errors are raised."""
        keys = self.record["subfield_uuid"]._data.keys()
        self.assertRaises(
            IllegalKeyException,
            self.record["subfield"].__setitem__, keys[0], 'stuff')

    def test_record_type(self):
        """Test the record_type property."""
        self.assertEqual('http://fnord.org/smorgasbord',
          self.record.record_type)

    def test_run_doctests(self):
        ctx = test_environment.test_context
        globs = { "db": CouchDatabase('testing', create=True, ctx=ctx) }
        results = doctest.testfile('../doc/records.txt', globs=globs)
        self.assertEqual(0, results.failed)

    def test_record_id(self):
        data = {"_id":"recordid"}
        record = Record(data, record_type="url")
        self.assertEqual(data["_id"], record.record_id)
        data = {}
        record_id = "recordid"
        record = Record(data, record_type="url", record_id=record_id)
        self.assertEqual(record_id, record.record_id)
        data = {"_id":"differentid"}
        self.assertRaises(ValueError,
            Record, data, record_id=record_id, record_type="url")


class TestRecordFactory(TestCase):
    """Test Record/Mergeable List factories."""

    def setUp(self):
        """Test setup."""
        super(TestRecordFactory, self).setUp()
        self.dict = {
            "a": "A",
            "b": "B",
            "subfield": {
                "field11s": "value11s",
                "field12s": "value12s",},
            "subfield_uuid":
                [
                {"field11": "value11",
                 "field12": "value12"},
                {"field21": "value21",
                 "field22": "value22"}],
            "record_type": "http://fnord.org/smorgasbord",
        }

    def test_build(self):
        """Test RecordDict/MergeableList factory method."""
        record = record_factory(self.dict)
        self.assertEquals('A', record._data['a'])
        self.assertEquals(
            'value12s', record._data['subfield']['field12s'])
