# Copyright 2009 Canonical Ltd.
#
# This file is part of desktopcouch.
#
# desktopcouch is free software: you can redistribute it and/or modify
# it under the terms of the GNU Lesser General Public License version 3
# as published by the Free Software Foundation.
#
# desktopcouch is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with desktopcouch.  If not, see <http://www.gnu.org/licenses/>.
#
# Author: Manuel de la Pena <manuel.delapena@canonical.com>
"""Test keyring module."""
import base64
from mocker import MockerTestCase

from desktopcouch.application.platform.windows.keyring import Keyring


class TestKeyring(MockerTestCase):
    """Test the class that manages the keyring on gnome."""

    def setUp(self):
        super(TestKeyring, self).setUp()
        self._random_string_fn = self.mocker.mock()
        self._registry = self.mocker.mock()
        self._crypto = self.mocker.mock()
        self._keyring = Keyring(make_random_string_fn=self._random_string_fn,
            registry=self._registry, crypto=self._crypto)
        self._username = 'username'
        self._password = 'password'
        self._consumer_key = 'consumer_key'
        self._consumer_secret = 'consumer_secret'
        self._token = 'token'
        self._token_secret = 'token_secret'
        self._current_user = 'HKCU'

    def test_get_user_name_password_no_key(self):
        """Test to get the value when the key is not present."""
        # try to open the key and thorw an exception
        # pylint: disable=W0104
        self._registry.HKEY_CURRENT_USER
        # pylint: enable=W0104
        self.mocker.result(0)
        # pylint: disable=W0104
        self._registry.KEY_ALL_ACCESS
        # pylint: enable=W0104
        self.mocker.result(0)
        self._registry.OpenKey(0, 'Canonical', 0, 0)
        self.mocker.throw(Exception('No key'))
        self._random_string_fn(10)
        self.mocker.result(self._username)
        self._random_string_fn(10)
        self.mocker.result(self._password)
        # open the key
        # pylint: disable=W0104
        self._registry.HKEY_CURRENT_USER
        # pylint: enable=W0104
        self.mocker.result(self._current_user)
        self._registry.CreateKey(self._current_user,
            'Canonical')
        self.mocker.result(self._registry)
        self._registry.CreateKey(self._registry, 'Keyrings')
        self.mocker.result(self._registry)
        self._registry.CreateKey(self._registry, 'Default')
        self.mocker.result(self._registry)
        self._registry.CreateKey(self._registry, 'Desktopcouch')
        self.mocker.result(self._registry)
        # store the key
        self._crypto.CryptProtectData(
            ':'.join([self._username, self._password]),
            u'basic', None, None, None, 0)
        self.mocker.result('secret')
        # pylint: disable=W0104
        self._registry.REG_SZ
        # pylint: enable=W0104
        self.mocker.result(0)
        self._registry.SetValueEx(self._registry, 'basic', 0,
            0, base64.b64encode('secret'))
        self.mocker.replay()
        username, password = self._keyring.get_user_name_password()
        self.assertEqual(username, self._username)
        self.assertEqual(password, self._password)

    def test_get_user_name_password_no_value(self):
        """Test to get the value when it is no present in the key."""
        # pylint: disable=W0104
        self._registry.HKEY_CURRENT_USER
        self.mocker.result(0)
        self._registry.KEY_ALL_ACCESS
        self.mocker.result(0)
        self._registry.OpenKey(0, 'Canonical', 0, 0)
        self.mocker.result(self._registry)
        self._registry.OpenKey(self._registry, 'Keyrings', 0, 0)
        self.mocker.result(self._registry)
        self._registry.OpenKey(self._registry, 'Default', 0, 0)
        self.mocker.result(self._registry)
        self._registry.OpenKey(self._registry, 'Desktopcouch', 0, 0)
        self.mocker.result(self._registry)
        # pylint: enable=W0104
        self._registry.EnumValue(self._registry, 0)
        self.mocker.throw(Exception('No values'))
        self._random_string_fn(10)
        self.mocker.result(self._username)
        self._random_string_fn(10)
        self.mocker.result(self._password)
        # open the key
        # pylint: disable=W0104
        self._registry.HKEY_CURRENT_USER
        # pylint: enable=W0104
        self.mocker.result(self._current_user)
        self._registry.CreateKey(self._current_user,
            'Canonical')
        self.mocker.result(self._registry)
        self._registry.CreateKey(self._registry, 'Keyrings')
        self.mocker.result(self._registry)
        self._registry.CreateKey(self._registry, 'Default')
        self.mocker.result(self._registry)
        self._registry.CreateKey(self._registry, 'Desktopcouch')
        self.mocker.result(self._registry)
        # store the key
        self._crypto.CryptProtectData(
            ':'.join([self._username, self._password]),
            u'basic', None, None, None, 0)
        self.mocker.result('secret')
        # pylint: disable=W0104
        self._registry.REG_SZ
        # pylint: enable=W0104
        self.mocker.result(0)
        self._registry.SetValueEx(self._registry, 'basic', 0,
            0, base64.b64encode('secret'))
        self.mocker.replay()
        username, password = self._keyring.get_user_name_password()
        self.assertEqual(username, self._username)
        self.assertEqual(password, self._password)

    def test_get_user_name_password(self):
        """Test get the data when it is present."""
        # pylint: disable=W0104
        self._registry.HKEY_CURRENT_USER
        self.mocker.result(0)
        self._registry.KEY_ALL_ACCESS
        self.mocker.result(0)
        self._registry.OpenKey(0, 'Canonical', 0, 0)
        self.mocker.result(self._registry)
        self._registry.OpenKey(self._registry, 'Keyrings', 0, 0)
        self.mocker.result(self._registry)
        self._registry.OpenKey(self._registry, 'Default', 0, 0)
        self.mocker.result(self._registry)
        self._registry.OpenKey(self._registry, 'Desktopcouch', 0, 0)
        self.mocker.result(self._registry)
        # pylint: enable=W0104
        self._registry.EnumValue(self._registry, 0)
        self.mocker.result(['basic'])
        # pylint: disable=W0104
        self._registry.HKEY_CURRENT_USER
        # pylint: enable=W0104
        self.mocker.result(self._current_user)
        self._registry.CreateKey(self._current_user,
            'Canonical')
        self.mocker.result(self._registry)
        self._registry.CreateKey(self._registry, 'Keyrings')
        self.mocker.result(self._registry)
        self._registry.CreateKey(self._registry, 'Default')
        self.mocker.result(self._registry)
        self._registry.CreateKey(self._registry, 'Desktopcouch')
        self.mocker.result(self._registry)
        # get the data from the key
        self._registry.QueryValueEx(self._registry, 'basic')
        self.mocker.result([base64.b64encode('secret')])
        self._crypto.CryptUnprotectData('secret', None, None, None, 0)
        self.mocker.result([0, ':'.join([self._username, self._password])])
        self.mocker.replay()
        username, password = self._keyring.get_user_name_password()
        self.assertEqual(username, self._username)
        self.assertEqual(password, self._password)

    def test_get_oauth_data(self):
        """Test get the oauth data."""
        secret_data = ':'.join([self._consumer_key,
                                self._consumer_secret,
                                self._token,
                                self._token_secret])
        self._random_string_fn(10)
        self.mocker.result(self._consumer_key)
        self._random_string_fn(10)
        self.mocker.result(self._consumer_secret)
        self._random_string_fn(10)
        self.mocker.result(self._token)
        self._random_string_fn(10)
        self.mocker.result(self._token_secret)
        # open the key
        # pylint: disable=W0104
        self._registry.HKEY_CURRENT_USER
        # pylint: enable=W0104
        self.mocker.result(self._current_user)
        self._registry.CreateKey(self._current_user,
            'Canonical')
        self.mocker.result(self._registry)
        self._registry.CreateKey(self._registry, 'Keyrings')
        self.mocker.result(self._registry)
        self._registry.CreateKey(self._registry, 'Default')
        self.mocker.result(self._registry)
        self._registry.CreateKey(self._registry, 'Desktopcouch')
        self.mocker.result(self._registry)
        # store the key
        self._crypto.CryptProtectData(secret_data,
            'oauth', None, None, None, 0)
        self.mocker.result('secret')
        # pylint: disable=W0104
        self._registry.REG_SZ
        # pylint: enable=W0104
        self.mocker.result(0)
        self._registry.SetValueEx(self._registry, 'oauth', 0, 0,
            base64.b64encode('secret'))
        self.mocker.replay()
        consumer_key, consumer_secret, token, token_secret = \
            self._keyring.get_oauth_data()
        self.assertEqual(self._consumer_key, consumer_key)
        self.assertEqual(self._consumer_secret, consumer_secret)
        self.assertEqual(self._token, token)
        self.assertEqual(self._token_secret, token_secret)
