/* $Id$ */

/***
  UEC component listener
  - Listens to announcements of new _eucalyptus._tcp services
  - Calls corresponding registration action based on type:
    <ACTIONS_DIR>/<type> <name> <ipaddress_to_register>

  This file is based on an example that is part of avahi, which is copyright:
  Lennart Poettering <lennart (at) poettering (dot) de>
  Trent Lloyd <lathiat@bur.st>
  Sebastien Estienne <sebastien.estienne@gmail.com>
  Jakub Stachowski
  James Willcox <snorp@snorp.net>
  Collabora Ltd.
  Modifications for eucalyptus are copyright 2009 Canonical Ltd.

  avahi is free software; you can redistribute it and/or modify it
  under the terms of the GNU Lesser General Public License as
  published by the Free Software Foundation; either version 2.1 of the
  License, or (at your option) any later version.
 
  avahi is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
  or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General
  Public License for more details.
 
  You should have received a copy of the GNU Lesser General Public
  License along with avahi; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
  USA.
***/

#define _GNU_SOURCE 1

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <assert.h>
#include <stdlib.h>
#include <time.h>
#include <libgen.h>
#include <errno.h>
#include <unistd.h>
#include <string.h>
#include <signal.h>

#include <avahi-client/client.h>
#include <avahi-client/lookup.h>

#include <avahi-common/simple-watch.h>
#include <avahi-common/malloc.h>
#include <avahi-common/error.h>

#define ACTIONS_DIR "/usr/share/eucalyptus/registration"
#define AVAHI_EUCALYPTUS "_eucalyptus._tcp"

static AvahiSimplePoll *simple_poll = NULL;

char *xasprintf (const char *fmt, ...)
{
    va_list ap;
    char *result;

    va_start(ap, fmt);
    if (vasprintf(&result, fmt, ap) < 0) {
        if (errno == ENOMEM) {
            fprintf(stderr, "out of memory\n");
            exit(1);
        }
        result = NULL;
    }
    va_end(ap);

    return result;
}

/* Returns first value in TXT that matches key 
   Returned pointer must be freed by calling function */
char *get_txt_contents(AvahiStringList *txt, char *key) {
    AvahiStringList *list;
    char *itemkey, *itemvalue;

    list = avahi_string_list_find(txt, key);
    if (!list || avahi_string_list_get_pair(list, &itemkey, &itemvalue, NULL) != 0) {
        fprintf(stderr, "Failed to get %s from service TXT\n",key);
        return NULL;
   }
    avahi_free(itemkey);
    return itemvalue;
}

/* Called whenever a service has been resolved successfully or timed out */
static void resolve_callback(
    AvahiServiceResolver *r,
    AVAHI_GCC_UNUSED AvahiIfIndex interface,
    AVAHI_GCC_UNUSED AvahiProtocol protocol,
    AvahiResolverEvent event,
    const char *name,
    const char *type,
    const char *domain,
    const char *host_name,
    const AvahiAddress *address,
    uint16_t port,
    AvahiStringList *txt,
    AvahiLookupResultFlags flags,
    AVAHI_GCC_UNUSED void* userdata) {

    assert(r);
    if (protocol == AVAHI_PROTO_INET) { /* Ignore IPv6 broadcasts, Eucalyptus doesn't current support it */
        switch (event) {
            case AVAHI_RESOLVER_FAILURE:
                fprintf(stderr, "(Resolver) Failed to resolve service '%s' of type '%s' in domain '%s': %s\n", name, type, domain, avahi_strerror(avahi_client_errno(avahi_service_resolver_get_client(r))));
                break;

            case AVAHI_RESOLVER_FOUND: {
                char *ipaddr, *nodetype;

                /* Get IP address */
                ipaddr = get_txt_contents(txt, "ipaddr");
                /* Fallback to avahi IP resolution */
                if (! ipaddr) {
                    ipaddr = avahi_malloc0(AVAHI_ADDRESS_STR_MAX);
                    avahi_address_snprint(ipaddr, AVAHI_ADDRESS_STR_MAX, address);
                }
                /* Get component type */
                nodetype = get_txt_contents(txt, "type");

                if (! nodetype || ! ipaddr) {
                    fprintf(stderr, "Missing items in TXT record ! Skipping.\n");
                } else if (strchr(nodetype, '/')) {
                    fprintf(stderr, "nodetype \"%s\" contains forbidden character '/'\n", nodetype);
                } else {
                  char *action_command = xasprintf("%s/%s", ACTIONS_DIR, nodetype);
                  if (!access(action_command, F_OK)) {
                    /* Call corresponding action */
                    pid_t child = 0;
                    fprintf(stderr, "Running %s \"%s\" \"%s\"\n", action_command, name, ipaddr);
                    child = fork();
                    if (child < 0) {
                        perror("fork");
                        exit(1);
                    } else if (child == 0) {
                        /* child */
                        char *action_args[4];
                        action_args[0] = action_command;
                        action_args[1] = (char *) name; /* execvp won't actually modify this */
                        action_args[2] = ipaddr;
                        action_args[3] = NULL;
                        execvp(action_command, action_args);
                        _exit(255); /* shouldn't be reached */
                    }
                  }
                }
                avahi_free(ipaddr);
                avahi_free(nodetype);
            }
        }
    }
    avahi_service_resolver_free(r);
}

/* Called whenever a new services becomes available on the LAN or is removed from the LAN */
static void browse_callback(
    AvahiServiceBrowser *b,
    AvahiIfIndex interface,
    AvahiProtocol protocol,
    AvahiBrowserEvent event,
    const char *name,
    const char *type,
    const char *domain,
    AVAHI_GCC_UNUSED AvahiLookupResultFlags flags,
    void* userdata) {
    
    AvahiClient *c = userdata;
    assert(b);

    switch (event) {

        case AVAHI_BROWSER_FAILURE:
            
            fprintf(stderr, "(Browser) %s\n", avahi_strerror(avahi_client_errno(avahi_service_browser_get_client(b))));
            avahi_simple_poll_quit(simple_poll);
            return;

        case AVAHI_BROWSER_NEW:
            fprintf(stderr, "(Browser) NEW: service '%s' of type '%s' in domain '%s'\n", name, type, domain);

            if (!(avahi_service_resolver_new(c, interface, protocol, name, type, domain, AVAHI_PROTO_UNSPEC, 0, resolve_callback, c)))
                fprintf(stderr, "Failed to resolve service '%s': %s\n", name, avahi_strerror(avahi_client_errno(c)));
            
            break;

    }
}

/* Called whenever the client or server state changes */
static void client_callback(AvahiClient *c, AvahiClientState state, AVAHI_GCC_UNUSED void * userdata) {
    assert(c);

    if (state == AVAHI_CLIENT_FAILURE) {
        fprintf(stderr, "Server connection failure: %s\n", avahi_strerror(avahi_client_errno(c)));
        avahi_simple_poll_quit(simple_poll);
    }
}

/* SIGCHLD handler */
static void sigchld_hdl (int sig) {
    while (waitpid(-1, NULL, WNOHANG) > 0) {
    }
}

int main(AVAHI_GCC_UNUSED int argc, AVAHI_GCC_UNUSED char*argv[]) {
    AvahiClient *client = NULL;
    AvahiServiceBrowser *sb = NULL;
    int error;
    int ret = 1;

    /* Register a SIGCHLD handler */
    struct sigaction act;
    memset (&act, 0, sizeof(act));
    act.sa_handler = sigchld_hdl;
    if (sigaction(SIGCHLD, &act, 0)) {
        fprintf(stderr, "Failed to register the SIGCHLD handler.\n");
        exit(1);
    }

    /* Allocate main loop object */
    if (!(simple_poll = avahi_simple_poll_new())) {
        fprintf(stderr, "Failed to create simple poll object.\n");
        goto fail;
    }

    /* Allocate a new client */
    client = avahi_client_new(avahi_simple_poll_get(simple_poll), 0, client_callback, NULL, &error);

    /* Check wether creating the client object succeeded */
    if (!client) {
        fprintf(stderr, "Failed to create client: %s\n", avahi_strerror(error));
        goto fail;
    }
    
    /* Create the service browser */
    if (!(sb = avahi_service_browser_new(client, AVAHI_IF_UNSPEC, AVAHI_PROTO_UNSPEC, AVAHI_EUCALYPTUS, NULL, 0, browse_callback, client))) {
        fprintf(stderr, "Failed to create service browser: %s\n", avahi_strerror(avahi_client_errno(client)));
        goto fail;
    }

    /* Run the main loop */
    avahi_simple_poll_loop(simple_poll);
    
    ret = 0;
    
fail:
    
    /* Cleanup things */
    if (sb)
        avahi_service_browser_free(sb);
    
    if (client)
        avahi_client_free(client);

    if (simple_poll)
        avahi_simple_poll_free(simple_poll);

    return ret;
}

