/*
 * hostname	This file contains an implementation of the command
 *		that maintains the host name and the domain name. It
 *		is also used to show the FQDN and the IP-Addresses.
 *
 * Usage:	hostname [-d|-f|-s|-a|-i|-y]
 *		hostname [-h|-V]
 *		hostname [-b] {name|-F file}
 *		dnsdomainname
 *
 * Authors:	Peter Tobias <tobias@et-inf.fho-emden.de>
 *		Bernd Eckenfels
 *		Graham Wilson <graham@debian.org>
 *
 *		This program is free software; you can redistribute it
 *		and/or  modify it under  the terms of  the GNU General
 *		Public  License as  published  by  the  Free  Software
 *		Foundation;  either  version 2 of the License, or  (at
 *		your option) any later version.
 *
 *		The localdomain and localhost functions are copyright
 *		(C) 1996 Free Software Foundation, Inc. and were
 *		written by Miles Bader <miles@gnu.ai.mit.edu> and
 *		Marcus Brinkmann <brinkmd@debian.org>.
 */

#include <sys/stat.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>
#include <getopt.h>
#include <string.h>
#include <netdb.h>
#include <errno.h>
#include <ctype.h>
#include <err.h>

#define VERSION "2.95"

enum type_t { DEFAULT, DNS, FQDN, SHORT, ALIAS, IP, NIS };

/*
 * Return the name of the localdomain. This is just a wrapper for
 * getdomainname, which takes care of allocating a big enough buffer, and
 * caches the result after the first call (so the result should be copied
 * before modification). If something goes wrong, 0 is returned, and errno set.
 */
char *
localdomain()
{
	char *buf = 0;
	size_t buf_len = 0;
	int myerror = 0;

	do {
		errno = 0;

		if (buf) {
			buf_len += buf_len;
			if ((buf = realloc (buf, buf_len)) == NULL)
				err(1, NULL);
		} else {
			buf_len = 128;        /* Initial guess */
			if ((buf = malloc (buf_len)) == NULL)
				err(1, NULL);
		}
	} while (((myerror = getdomainname(buf, buf_len)) == 0 && !memchr (buf, '\0', buf_len))
	        || errno == ENAMETOOLONG);

	/* getdomainname failed, abort. */
	if (myerror)
		err(1, NULL);

	return buf;
}

/*
 * Return the name of the localhost. This is just a wrapper for gethostname,
 * which takes care of allocating a big enough buffer, and caches the result
 * after the first call (so the result should be copied before modification).
 * If something goes wrong, 0 is returned, and errno set.
 */
char *
localhost()
{
	char *buf = 0;
	size_t buf_len = 0;
	int myerror = 0;

	do {
		errno = 0;

		if (buf) {
			buf_len += buf_len;
			if ((buf = realloc (buf, buf_len)) == NULL)
				err(1, NULL);
		} else {
			buf_len = 128;        /* Initial guess */
			if ((buf = malloc (buf_len)) == NULL)
				err(1, NULL);
		}
	} while (((myerror = gethostname(buf, buf_len)) == 0 && !memchr (buf, '\0', buf_len))
	        || errno == ENAMETOOLONG);

	/* gethostname failed, abort. */
	if (myerror)
		err(1, NULL);

	return buf;
}

void
usage(FILE *stream)
{
	fprintf(stream,
		"Usage: hostname [-v] [-b] {hostname|-F file} set host name (from file)\n"
		"       domainname [-v] {nisdomain|-F file}   set NIS domain name (from file)\n"
		"       hostname [-v] [-d|-f|-s|-a|-i|-y]     display formated name\n"
		"       hostname [-v]                         display host name\n\n"
		"       hostname -V|--version|-h|--help       print info and exit\n\n"
		"    dnsdomainname=hostname -d, {yp,nis,}domainname=hostname -y\n\n"
		"    -s, --short           short host name\n"
		"    -a, --alias           alias names\n"
		"    -i, --ip-address      addresses for the host name\n"
		"    -f, --fqdn, --long    long host name (FQDN)\n"
		"    -d, --domain          DNS domain name\n"
		"    -y, --yp, --nis       NIS/YP domain name\n"
		"    -b, --boot            set default hostname if none available\n"
		"    -F, --file            read host name or NIS domain name from given file\n\n"
		"   This command can get or set the host name or the NIS domain name. You can\n"
		"   also get the DNS domain or the FQDN (fully qualified domain name).\n"
		"   Unless you are using bind or NIS for host lookups you can change the\n"
		"   FQDN (Fully Qualified Domain Name) and the DNS domain name (which is\n"
		"   part of the FQDN) in the /etc/hosts file.\n");
	exit(-1);
}

/*
 * Check the format of a user-specified hostname. Uses the rules from RFC 1035,
 * section 2.3.1.
 */
int
check_name(char *name)
{
	int i, len = strlen(name);

	/* Handle leading and trailing hyphen now. */
	if (!len || !isalnum(name[0]) || !isalnum(name[len-1]))
		return 0;

	for (i = 0; i < len; i++) {
		if (!isalnum(name[i]) && name[i] != '-' && name[i] != '.')
			return 0;
		if (name[i] == '-' && (name[i - 1] == '.' || name[i + 1] == '.'))
			return 0;
		if (name[i] == '.' && name[i - 1] == '.')
			return 0;
	}

	return 1;
}

void
set_name(enum type_t type, char *name)
{
	switch (type) {
	case DEFAULT:
		if (!check_name(name))
			warnx("the specified hostname is invalid");

		if (sethostname(name, strlen(name))) {
			if (errno == EPERM)
				errx(1, "you must be root to change the host name");
			else if (errno == EINVAL)
				errx(1, "name too long");
		}
		break;

	case NIS:
		if (setdomainname(name, strlen(name))) {
			if (errno == EPERM)
				errx(1, "you must be root to change the domain name");
			else if (errno == EINVAL)
				errx(1, "name too long");
		}
		break;

	default:
		/*
		 * Only the host name and the domain name can be set using this
		 * command.
		 */
		usage(stderr);
	}
}

void
show_name(enum type_t type)
{
	struct hostent *hp;
	char *p;
	int i;

	/* Handle a few cases specially. */
	if (type == DEFAULT) {
		printf("%s\n", localhost());
		return;
	} else if (type == NIS) {
		printf("%s\n", localdomain());
		return;
	}

	/* Otherwise, resolve the host name to get more information. */
	if ((hp = gethostbyname(localhost())) == NULL)
		errx(1, "%s", hstrerror(h_errno));

	p = strchr(hp->h_name, '.');

	switch (type) {
	case ALIAS:
		for (i = 0; hp->h_aliases[i]; i++) {
			if (i > 0)
				printf(" ");
			printf("%s ", hp->h_aliases[i]);
		}
		printf("\n");
		break;

	case IP: {
		char buf[INET6_ADDRSTRLEN];

		for (i = 0; hp->h_addr_list[i]; i++) {
			if (i > 0)
				printf(" ");
			printf("%s", inet_ntop(hp->h_addrtype,
			       hp->h_addr_list[i], buf, sizeof(buf)));
		}
		printf("\n");
		break;
	}

	case DNS:
		if (p != NULL)
			printf("%s\n", ++p);
		break;

	case FQDN:
		printf("%s\n", hp->h_name);
		break;

	case SHORT:
		if (p != NULL)
			*p = '\0';
		printf("%s\n", hp->h_name);
		break;

	default:
		break;
	}
}

char *
read_file(char *filename, int boot)
{
	struct stat st;
  	FILE *fp;
	char *buf;

	if ((fp = fopen(filename, "r")) == NULL) {
		if (boot)
			return NULL;
		else
			err(1, NULL);
	}

	if (fstat(fileno(fp), &st) == -1
	   || (buf = (char *) malloc(st.st_size + 1)) == NULL)
		err(1, NULL);

	while (fgets(buf, st.st_size + 1, fp) != NULL) {
		char *p;

		if (buf[0] == '\n' || buf[0] == '#')
			continue;

		if ((p = strchr(buf, '\n')) != NULL)
			*p = '\0';
		break;
	}

	fclose(fp);
	return buf;
}

int
main(int argc, char **argv)
{
	char *progname, *file = NULL, *name = NULL;
	enum type_t type = DEFAULT;
	int boot = 0;
	int o;

	static const struct option long_options[] =
	{
		{"domain", no_argument, 0, 'd'},
		{"boot", no_argument, 0, 'b'},
		{"file", required_argument, 0, 'F'},
		{"fqdn", no_argument, 0, 'f'},
		{"help", no_argument, 0, 'h'},
		{"long", no_argument, 0, 'f'},
		{"short", no_argument, 0, 's'},
		{"version", no_argument, 0, 'V'},
		{"verbose", no_argument, 0, 'v'},
		{"alias", no_argument, 0, 'a'},
		{"ip-address", no_argument, 0, 'i'},
		{"nis", no_argument, 0, 'y'},
		{"yp", no_argument, 0, 'y'},
		{0, 0, 0, 0}
	};

	/* If called as 'dnsdomainname', by default show the DNS domain name. */
	progname = (rindex(argv[0], '/')) ? rindex(argv[0], '/') + 1 : argv[0];
	if (!strcmp(progname, "dnsdomainname"))
		type = DNS;

	while((o = getopt_long(argc, argv, "adfbF:h?isVvy", long_options, NULL)) != -1)
		switch (o) {
		case 'd':
			type = DNS;
			break;
		case 'a':
			type = ALIAS;
			break;
		case 'f':
			type = FQDN;
			break;
		case 'i':
			type = IP;
			break;
		case 's':
			type = SHORT;
			break;
		case 'y':
			type = NIS;
			break;
		case 'b':
			boot = 1;
			break;
		case 'F':
			file = optarg;
			break;
		case 'v':
			/* Does not do anything. */
			break;
		case 'V':
			printf("hostname %s\n", VERSION);
			return 0;
		case '?':
		case 'h':
			usage(stdout);
			break;
		default:
			usage(stderr);
		}

	/*
	 * Hostname may be read from a file, it's ok for this file to not
	 * exist or be empty if -b is given in which case we default to
	 * "localhost" if there is not one already set.
	 */
	if (file) {
		name = read_file(file, boot);
		if (boot && (name == NULL || name[0] == '\0')) {
			free(name);

			name = localhost();
			if (name[0] == '\0' || !strcmp(name,"(none)"))
				strcpy(name, "localhost");
		}
	}

	/* Otherwise the hostname is the first non-option argument. */
	if (optind < argc) {
		/*
		 * It is an error to specify a host name as an argument, and to
		 * be read from a file.
		 */
		if (name)
			usage(stderr);

		if ((name = strdup(argv[optind])) == NULL)
			err(1, NULL);
		optind++;
	}

	/* Check for any remaining arguments. */
	if (optind < argc)
		usage(stderr);

	if (name) {
		set_name(type, name);
		free(name);
	} else
		show_name(type);

	return 0;
}
