/*
 * Line6 PODxt Pro USB driver - 0.5
 *
 * Copyright (C) 2004, 2005 Markus Grabner (grabner@icg.tu-graz.ac.at)
 *
 *	This program is free software; you can redistribute it and/or
 *	modify it under the terms of the GNU General Public License as
 *	published by the Free Software Foundation, version 2.
 *
 */

#ifndef AUDIO_H
#define AUDIO_H


#include <linux/config.h>

#include <sound/driver.h>
#include <sound/core.h>
#include <sound/pcm.h>

#include "config.h"


#define PODXTPRO_ISO_BUFFERS 8  /* number of URBs */
#define PODXTPRO_ISO_PACKETS 2  /* number of USB frames per URB */

#define PODXTPRO_ISO_INTERVAL 1  /* in a "full speed" device (such as the PodXT Pro) this means 1ms */

/*
	PODxt Pro transmits and receives audio data in a sequence of 15 frames of 234
	bytes and one frame of 240 bytes. Each URB contains two USB frames.
*/
#define PODXTPRO_ISO_PACKET_SIZE_NORMAL 234
#define PODXTPRO_ISO_PACKET_SIZE_LONG   240

#define PODXTPRO_ISO_PACKET_SIZE_MAX    252  /* this should be queried dynamically from the USB interface! */

#define PODXTPRO_BYTES_PER_FRAME 6  /* 24bit audio (stereo) */
#define PODXTPRO_FRAMES_PER_PERIOD ((PODXTPRO_ISO_PACKET_SIZE_NORMAL * 15 + PODXTPRO_ISO_PACKET_SIZE_LONG) / PODXTPRO_BYTES_PER_FRAME)

/*
	Extract the messaging device from the substream instance
*/
#define s2m(s) (((snd_podxtpro_pcm_t *)snd_pcm_substream_chip(s))->podxtpro->msgdev)


struct device;
struct usb_interface;

struct snd_podxtpro_pcm
{
	/**
		 Pointer back to the PODxt Pro driver data structure.
	*/
	struct usb_podxtpro *podxtpro;

	/**
		 ALSA pcm stream
	*/
	snd_pcm_t *pcm;

	/**
		 URBs for audio playback.
	*/
	struct urb *urb_audio_out[PODXTPRO_ISO_BUFFERS];

	/**
		 URBs for audio capture.
	*/
	struct urb *urb_audio_in[PODXTPRO_ISO_BUFFERS];

	/**
		 Temporary buffer to hold data when playback buffer wraps.
	*/
	unsigned char *wrap_out;

	/**
		 Temporary buffer for capture.
		 Since the packet size is not known in advance, this buffer is large enough
		 to store maximum size packets.
	*/
	unsigned char *buffer_in;

	/**
		 Free frame position in the playback buffer.
	*/
	snd_pcm_uframes_t pos_out;

	/**
		 Count processed bytes for playback.
		 This is modulo period size (to determine when a period is finished).
	*/
	unsigned bytes_out;

	/**
		 Playback period size in bytes
	*/
	unsigned period_out;

	/**
		 Processed frame position in the playback buffer.
		 The contents of the output ring buffer have been consumed by the USB
		 subsystem (i.e., sent to the USB device) up to this position.
	*/
	snd_pcm_uframes_t pos_out_done;

	/**
		 Count processed bytes for capture.
		 This is modulo period size (to determine when a period is finished).
	*/
	unsigned bytes_in;

	/**
		 Capture period size in bytes
	*/
	unsigned period_in;

	/**
		 Processed frame position in the capture buffer.
		 The contents of the output ring buffer have been consumed by the USB
		 subsystem (i.e., sent to the USB device) up to this position.
	*/
	snd_pcm_uframes_t pos_in_done;

	/**
		 Bit mask of active playback URBs.
	*/
	unsigned long active_urb_out;

	/**
		 Bit mask of active capture URBs.
	*/
	unsigned long active_urb_in;

	/**
		 Bit mask of playback URBs currently being unlinked.
	*/
	unsigned long unlink_urb_out;

	/**
		 Bit mask of capture URBs currently being unlinked.
	*/
	unsigned long unlink_urb_in;

	/**
		 Spin lock to protect updates of the playback buffer positions (not
		 contents!)
	*/
	spinlock_t lock_audio_out;

	/**
		 Spin lock to protect updates of the capture buffer positions (not
		 contents!)
	*/
	spinlock_t lock_audio_in;

	/**
		 Spin lock to protect trigger.
	*/
	spinlock_t lock_trigger;

	/**
		 PCM playback volume (left and right).
	*/
	int volume[2];

	/**
		 Flags to indicate running playback and capture streams.
	*/
	char running_playback, running_capture;

	/**
		 Flags to indicate pause status.
	*/
	char pause_playback;
};  

typedef struct snd_podxtpro_pcm snd_podxtpro_pcm_t;


extern void podxtpro_cleanup_audio(struct usb_podxtpro *);
extern int podxtpro_init_audio(struct usb_podxtpro *);


#endif
