#ifndef BCM43xx_LO_H_
#define BCM43xx_LO_H_

#include "bcm43xx_phy.h"

struct bcm43xx_wldev;

/* Local Oscillator control value-pair. */
struct bcm43xx_loctl {
	/* Control values. */
	s8 v0;
	s8 v1;
	/* "Used by hardware" flag. */
	u8 used:1;
};

/* TX Power LO Control Array.
 * Value-pairs to adjust the LocalOscillator are stored
 * in this structure.
 * There are two different set of values. One for "Flag is Set"
 * and one for "Flag is Unset".
 * By "Flag" the flag in struct bcm43xx_rfatt is meant.
 * The Value arrays are two-dimensional. The first index
 * is the baseband attenuation and the second index
 * is the radio attenuation.
 * Use bcm43xx_get_loctl() to retrieve a value from the lists.
 */
struct bcm43xx_txpower_lo_control {
#define BCM43xx_NR_BB	9
#define BCM43xx_NR_RF	16
	struct bcm43xx_loctl flagged[ BCM43xx_NR_BB ][ BCM43xx_NR_RF ];
	struct bcm43xx_loctl unflagged[ BCM43xx_NR_BB ][ BCM43xx_NR_RF ];

	/* Lists of valid RF and BB attenuation values for this device. */
	struct bcm43xx_rfatt_list rfatt_list;
	struct bcm43xx_bbatt_list bbatt_list;

	/* Current RF and BB attenuation and LO control values. */
	struct bcm43xx_rfatt rfatt;
	struct bcm43xx_bbatt bbatt;

	u16 txctl1;
	//TODO rename: This is the "Minimum TX Bias". Move to PHY struct?
	u16 txctl2;

	/* GPHY LO is measured. */
	u8 lo_measured:1;
	/* One or more loctl in the flagged or unflagged arrays
	 * are used (see "used" flag of struct bcm43xx_loctl)
	 */
	u8 some_loctl_is_used:1;

	/* Saved device PowerVector */
	u64 power_vector;
};


/* Measure the BPHY Local Oscillator. */
void bcm43xx_lo_b_measure(struct bcm43xx_wldev *dev);
/* Measure the BPHY/GPHY Local Oscillator. */
void bcm43xx_lo_g_measure(struct bcm43xx_wldev *dev);

/* Adjust the Local Oscillator to the saved attenuation
 * and txctl values.
 */
void bcm43xx_lo_adjust(struct bcm43xx_wldev *dev);
/* Adjust to specific values. */
void bcm43xx_lo_adjust_to(struct bcm43xx_wldev *dev,
			  u16 rfatt, u16 bbatt, u16 txctl1);

/* Returns the bcm43xx_loctl corresponding to the current
 * attenuation values.
 */
struct bcm43xx_loctl * bcm43xx_loctl_current(struct bcm43xx_wldev *dev);
/* Mark all possible bcm43xx_loctl as "unused" */
void bcm43xx_loctl_mark_all_unused(struct bcm43xx_wldev *dev);
/* Mark the bcm43xx_loctl corresponding to the current
 * attenuation values as used.
 */
void bcm43xx_loctl_mark_cur_used(struct bcm43xx_wldev *dev);

/* Get a reference to a LO Control value pair in the
 * TX Power LO Control Array.
 */
struct bcm43xx_loctl * bcm43xx_get_loctl(struct bcm43xx_wldev *dev,
					 const struct bcm43xx_rfatt *rfatt,
					 const struct bcm43xx_bbatt *bbatt);

#endif /* BCM43xx_LO_H_ */
