/******************************************************************************

  Copyright(c) 2003 - 2006 Intel Corporation. All rights reserved.

  Portions of this file are derived from the ipw3945 project, as well
  as portions of the ieee80211 subsystem header files.

  802.11 status code portion of this file from ethereal-0.10.6:
    Copyright 2000, Axis Communications AB
    Ethereal - Network traffic analyzer
    By Gerald Combs <gerald@ethereal.com>
    Copyright 1998 Gerald Combs

  This program is free software; you can redistribute it and/or modify it
  under the terms of version 2 of the GNU General Public License as
  published by the Free Software Foundation.

  This program is distributed in the hope that it will be useful, but WITHOUT
  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
  more details.

  You should have received a copy of the GNU General Public License along with
  this program; if not, write to the Free Software Foundation, Inc.,
  51 Franklin Street, Fifth Floor, Boston, MA 02110, USA

  The full GNU General Public License is included in this distribution in the
  file called LICENSE.

  Contact Information:
  James P. Ketrenos <ipw2100-admin@linux.intel.com>
  Intel Corporation, 5200 N.E. Elam Young Parkway, Hillsboro, OR 97124-6497

******************************************************************************/

/*
 * Theory of operation:
 *
 * The primary NIC mode of operation is configured within the RXON command.
 * As such, there are two instances of RXON data -- staging_rxon and
 * active_rxon.  The former is used to configure the device, the later to
 * reflect the actual currently active configuration.
 *
 * After configuring staging_rxon, it is activated by calling
 * ipw_commit_rxon(). The ipw_commit_rxon() will ensure the correct sequence
 * of commands are invoked on the hardware to result in Tx being possible
 * (if the channel is appropriate for Tx)
 *
 * About the threading and lock model of the driver...
 *
 * There are three paths of execution through the driver.
 *
 * 1.  ioctl based (wireless extensions, netdev, etc.)
 * 2.  interrupt based
 * 3.  scheduled work queue items
 *
 * As soon as an interrupt comes in, it schedules a tasklet.  That tasklet,
 * when run, does any HW checks, pulls any data from the read queue,
 * and schedules other layers to do the actual work.
 *
 * NOTE: This driver is a work in progress.  It is currently structured
 * in a way that we can add support for the next wireless adapter
 * available from Intel.
 *
 * Our current focus is as follows:
 * 1.  Get the driver working w/ mac80211 in STA(client) mode.
 * 2.  Clean up the dual-driver code from function redirection to
 *     if/else switches where the code detlas are minimal.
 * 3.  Evaluate the amount of delta between the hardware versions and
 *     possibly split iwlwifi into multiple drivers for each piece of
 *     HW, sharing as much base code as possible (would result in
 *     multiple driver files on disk vs. a single module that works
 *     with multiple hardware types)
 * 4.  Document the uCode regulatory enforcement
 * 5.  Document the rate scaling capabilities of the 3945
 * 6.  Provide an initial rate scaling algorithm tailored for the
 *     statistic data exposed by the 3945.
 */
#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/version.h>
#include <linux/init.h>
#include <linux/pci.h>
#include <linux/dma-mapping.h>
#include <linux/delay.h>
#include <linux/skbuff.h>
#include <linux/netdevice.h>
#include <linux/wireless.h>
#include <linux/firmware.h>
#include <net/iw_handler.h>

//#define BIT(x) (1 << (x))
#include <net/mac80211.h>
#include <net/ieee80211.h>

//#include <net/mac80211_mgmt.h>
//#include <net/mac80211_common.h>
//#include <net/mac80211_shared.h>
#include <../net/mac80211/ieee80211_rate.h>

#include <linux/if_arp.h>
#include <asm/div64.h>

#include "iwlwifi.h"

#define IWLWIFI_VERSION "0.0.11"

#define DRV_DESCRIPTION	"Intel(R) Wireless Link driver for Linux"
#define DRV_COPYRIGHT	"Copyright(c) 2003-2006 Intel Corporation"
#define DRV_VERSION     IWLWIFI_VERSION

MODULE_DESCRIPTION(DRV_DESCRIPTION);
MODULE_VERSION(DRV_VERSION);
MODULE_AUTHOR(DRV_COPYRIGHT);
MODULE_LICENSE("GPL");

/*MAC80211  move this to mac80211 files */
#define IEEE80211_FC(type, stype) cpu_to_le16((type << 2) | (stype << 4))

#define IEEE80211_CHAN_W_RADAR_DETECT 0x00000010

#define IEEE80211_24GHZ_BAND     (1<<0)
#define IEEE80211_52GHZ_BAND     (1<<1)

#define IEEE80211_CCK_RATE_1MB                  0x02
#define IEEE80211_CCK_RATE_2MB                  0x04
#define IEEE80211_CCK_RATE_5MB                  0x0B
#define IEEE80211_CCK_RATE_11MB                 0x16
#define IEEE80211_OFDM_RATE_6MB                 0x0C
#define IEEE80211_OFDM_RATE_9MB                 0x12
#define IEEE80211_OFDM_RATE_12MB                0x18
#define IEEE80211_OFDM_RATE_18MB                0x24
#define IEEE80211_OFDM_RATE_24MB                0x30
#define IEEE80211_OFDM_RATE_36MB                0x48
#define IEEE80211_OFDM_RATE_48MB                0x60
#define IEEE80211_OFDM_RATE_54MB                0x6C
#define IEEE80211_BASIC_RATE_MASK               0x80

#define IEEE80211_CCK_RATE_1MB_MASK             (1<<0)
#define IEEE80211_CCK_RATE_2MB_MASK             (1<<1)
#define IEEE80211_CCK_RATE_5MB_MASK             (1<<2)
#define IEEE80211_CCK_RATE_11MB_MASK            (1<<3)
#define IEEE80211_OFDM_RATE_6MB_MASK            (1<<4)
#define IEEE80211_OFDM_RATE_9MB_MASK            (1<<5)
#define IEEE80211_OFDM_RATE_12MB_MASK           (1<<6)
#define IEEE80211_OFDM_RATE_18MB_MASK           (1<<7)
#define IEEE80211_OFDM_RATE_24MB_MASK           (1<<8)
#define IEEE80211_OFDM_RATE_36MB_MASK           (1<<9)
#define IEEE80211_OFDM_RATE_48MB_MASK           (1<<10)
#define IEEE80211_OFDM_RATE_54MB_MASK           (1<<11)

#define IEEE80211_CCK_RATES_MASK                0x0000000F
#define IEEE80211_CCK_BASIC_RATES_MASK  (IEEE80211_CCK_RATE_1MB_MASK | \
        IEEE80211_CCK_RATE_2MB_MASK)
#define IEEE80211_CCK_DEFAULT_RATES_MASK        (IEEE80211_CCK_BASIC_RATES_MASK | \
        IEEE80211_CCK_RATE_5MB_MASK | \
        IEEE80211_CCK_RATE_11MB_MASK)

#define IEEE80211_OFDM_RATES_MASK               0x00000FF0
#define IEEE80211_OFDM_BASIC_RATES_MASK (IEEE80211_OFDM_RATE_6MB_MASK | \
        IEEE80211_OFDM_RATE_12MB_MASK | \
        IEEE80211_OFDM_RATE_24MB_MASK)
#define IEEE80211_OFDM_DEFAULT_RATES_MASK       (IEEE80211_OFDM_BASIC_RATES_MASK | \
        IEEE80211_OFDM_RATE_9MB_MASK  | \
        IEEE80211_OFDM_RATE_18MB_MASK | \
        IEEE80211_OFDM_RATE_36MB_MASK | \
        IEEE80211_OFDM_RATE_48MB_MASK | \
        IEEE80211_OFDM_RATE_54MB_MASK)
#define IEEE80211_DEFAULT_RATES_MASK (IEEE80211_OFDM_DEFAULT_RATES_MASK | \
                                IEEE80211_CCK_DEFAULT_RATES_MASK)

#define IEEE80211_NUM_OFDM_RATES            8
#define IEEE80211_NUM_CCK_RATES             4
#define IEEE80211_OFDM_SHIFT_MASK_A         4

/*MAC80211 end */

static u32 ipw_debug_level;

static int param_debug = 0;
static int param_channel = 0;
static int param_mode = 0;
static int param_auto_create = 1;	/* def: create new adhoc network if needed */
static int param_led = 1;	/* def: use LEDs */
static int param_disable = 0;	/* def: enable radio */
static int param_antenna = 0;	/* def: 0 = both antennas (use diversity) */
static int param_hwcrypto = 0;	/* def: using software encryption */
static int param_qos_enable = 1;

static const char ipw_modes[] = {
	'a', 'b', 'g', '?'
};

#define LD_TIME_LINK_ON 300

static int ipw_power_init_handle(struct ipw_priv *priv);

static int ipw_queue_tx_hcmd(struct ipw_priv *priv, struct ipw_host_cmd *cmd);

static int ipw_init_rate_scaling(struct ipw_priv *priv);

static int ipw_rate_scale_rxon_handle(struct ipw_priv *priv, s32 sta_id);
static u8 ipw_find_station(struct ipw_priv *priv, u8 * bssid);
static void ipw_clear_stations_table(struct ipw_priv *priv);

static void ipw_link_up(struct ipw_priv *priv);

static int ipw_wx_set_powermode(struct ipw_priv *priv, int mode);

#define SLP PMC_TCMD_FLAG_DRIVER_ALLOW_SLEEP_MSK

#define MSEC_TO_USEC 1024

/* default power management (not Tx power) table values */
static struct ipw_power_vec_entry range_0[IPW_POWER_AC] = {	// for tim  0-10
	{{0, 0 * MSEC_TO_USEC, 0 * MSEC_TO_USEC, {0, 0, 0, 0, 0}}, 0},
	{{SLP, 200 * MSEC_TO_USEC, 500 * MSEC_TO_USEC,
	  {1, 2, 3, 4, 4}}, 0},
	{{SLP, 200 * MSEC_TO_USEC, 300 * MSEC_TO_USEC,
	  {2, 4, 6, 7, 7}}, 0},
	{{SLP, 50 * MSEC_TO_USEC, 100 * MSEC_TO_USEC,
	  {2, 6, 9, 9, 10}}, 0},
	{{SLP, 50 * MSEC_TO_USEC, 25 * MSEC_TO_USEC,
	  {2, 7, 9, 9, 10}}, 1},
	{{SLP, 25 * MSEC_TO_USEC, 25 * MSEC_TO_USEC,
	  {4, 7, 10, 10, 10}}, 1}
};

static struct ipw_power_vec_entry range_1[IPW_POWER_AC] = {	// for tim > 10
	{{0, 0 * MSEC_TO_USEC, 0 * MSEC_TO_USEC, {0, 0, 0, 0, 0}}, 0},
	{{SLP, 200 * MSEC_TO_USEC, 500 * MSEC_TO_USEC,
	  {1, 2, 3, 4, 0xFF}}, 0},
	{{SLP, 200 * MSEC_TO_USEC, 300 * MSEC_TO_USEC,
	  {2, 4, 6, 7, 0xFF}}, 0},
	{{SLP, 50 * MSEC_TO_USEC, 100 * MSEC_TO_USEC,
	  {2, 6, 9, 9, 0xFF}}, 0},
	{{SLP, 50 * MSEC_TO_USEC, 25 * MSEC_TO_USEC,
	  {2, 7, 9, 9, 0xFF}}, 0},
	{{SLP, 25 * MSEC_TO_USEC, 25 * MSEC_TO_USEC,
	  {4, 7, 10, 10, 0xFF}}, 0}
};

/************************************************/
static int ipw_queue_tx_reclaim(struct ipw_priv *priv, int qindex, int index);
static int ipw_queue_reset(struct ipw_priv *priv);

static void ipw_tx_queue_free(struct ipw_priv *);

static int ipw_stop_tx_queue(struct ipw_priv *priv);

static struct ipw_rx_queue *ipw_rx_queue_alloc(struct ipw_priv *);
static void ipw_rx_queue_free(struct ipw_priv *, struct ipw_rx_queue *);
static void ipw_rx_queue_replenish(struct ipw_priv *priv);

static int ipw_up(struct ipw_priv *);
static void ipw_bg_up(struct work_struct *work);
static void ipw_down(struct ipw_priv *);
static void ipw_bg_down(struct work_struct *work);

static int ipw_card_show_info(struct ipw_priv *priv);
static void ipw_bg_alive_start(struct work_struct *work);
static int ipw_send_card_state(struct ipw_priv *priv, u32 flags, u8 meta_flag);

static void ipw_link_down(struct ipw_priv *priv);
static u8 ipw_add_station(struct ipw_priv *priv, u8 * bssid,
			  int is_ap, u8 flags);
static u8 ipw_remove_station(struct ipw_priv *priv, u8 * bssid, int is_ap);

static int snprint_line(char *buf, size_t count,
			const u8 * data, u32 len, u32 ofs)
{
	int out, i, j, l;
	char c;

	out = snprintf(buf, count, "%08X", ofs);

	for (l = 0, i = 0; i < 2; i++) {
		out += snprintf(buf + out, count - out, " ");
		for (j = 0; j < 8 && l < len; j++, l++)
			out +=
			    snprintf(buf + out, count - out, "%02X ",
				     data[(i * 8 + j)]);
		for (; j < 8; j++)
			out += snprintf(buf + out, count - out, "   ");
	}
	out += snprintf(buf + out, count - out, " ");
	for (l = 0, i = 0; i < 2; i++) {
		out += snprintf(buf + out, count - out, " ");
		for (j = 0; j < 8 && l < len; j++, l++) {
			c = data[(i * 8 + j)];
			if (!isascii(c) || !isprint(c))
				c = '.';

			out += snprintf(buf + out, count - out, "%c", c);
		}

		for (; j < 8; j++)
			out += snprintf(buf + out, count - out, " ");
	}

	return out;
}

static void printk_buf(int level, const void *p, u32 len)
{
	const u8 *data = p;
	char line[81];
	u32 ofs = 0;
	if (!(ipw_debug_level & level))
		return;

	while (len) {
		snprint_line(line, sizeof(line), &data[ofs],
			     min(len, 16U), ofs);
		printk(KERN_DEBUG "%s\n", line);
		ofs += 16;
		len -= min(len, 16U);
	}
}

#define IGNORE_RETURN(x) if (x);else;

/*MAC80211  support function */
static inline int ieee80211_is_empty_essid(const char *essid, int essid_len)
{
	/* Single white space is for Linksys APs */
	if (essid_len == 1 && essid[0] == ' ')
		return 1;

	/* Otherwise, if the entire essid is 0, we assume it is hidden */
	while (essid_len) {
		essid_len--;
		if (essid[essid_len] != '\0')
			return 0;
	}

	return 1;
}

static const char *iwl_escape_essid(const char *essid, u8 essid_len)
{
	static char escaped[IW_ESSID_MAX_SIZE * 2 + 1];
	const char *s = essid;
	char *d = escaped;

	if (ieee80211_is_empty_essid(essid, essid_len)) {
		memcpy(escaped, "<hidden>", sizeof("<hidden>"));
		return escaped;
	}

	essid_len = min(essid_len, (u8) IW_ESSID_MAX_SIZE);
	while (essid_len--) {
		if (*s == '\0') {
			*d++ = '\\';
			*d++ = '0';
			s++;
		} else {
			*d++ = *s++;
		}
	}
	*d = '\0';
	return escaped;
}

static struct ieee80211_hw_mode *ipw_get_hw_mode(struct ipw_priv *priv,
						 int mode)
{
	struct ieee80211_hw_mode *hw_mode;
	struct ieee80211_local *local = hw_to_local(priv->ieee);

	list_for_each_entry(hw_mode, &local->modes_list, list) {
		if (hw_mode->mode == mode)
			return hw_mode;
	}

	return NULL;
}

static inline int check_bits(unsigned long field, unsigned long mask)
{
	return ((field & mask) == mask) ? 1 : 0;
}

/*MAC80211-END*/

#define IPW_RATE_SCALE_MAX_WINDOW 62
/*
 * IO, register, and NIC memory access functions
 *
 * NOTE on naming convention and macro usage for these
 *
 * A single _ prefix before a an access function means that no state
 * check or debug information is printed when that function is called.
 *
 * A double __ prefix before an access function means that state is checked
 * (in the case of *restricted calls) and the current line number is printed
 * in addition to any other debug output.
 *
 * The non-prefixed name is the #define that maps the caller into a
 * #define that provides the caller's __LINE__ to the double prefix version.
 *
 * If you wish to call the function without any debug or state checking,
 * you should use the single _ prefix version (as is used by dependent IO
 * routines, for example _ipw_read_restricted calls the non-check version of
 * _ipw_read32.
 *
 */

#define _ipw_write32(ipw, ofs, val) writel((val), (ipw)->hw_base + (ofs))
static void ipw_write32(struct ipw_priv *ipw, u32 ofs, u32 val)
{
	IPW_DEBUG_IO("%s %d: write_direct32(0x%08X, 0x%08X)\n", __FILE__,
		     __LINE__, (u32) (ofs), (u32) (val));
	_ipw_write32(ipw, ofs, val);
}

#define _ipw_read32(ipw, ofs) readl((ipw)->hw_base + (ofs))
static inline u32 __ipw_read32(char *f, u32 l, struct ipw_priv *ipw, u32 ofs)
{
	IPW_DEBUG_IO("%s %d: read_direct32(0x%08X)\n", f, l, (u32) (ofs));
	return _ipw_read32(ipw, ofs);
}

static u32 ipw_read32(struct ipw_priv *ipw, u32 ofs)
{
	return __ipw_read32(__FILE__, __LINE__, ipw, ofs);
}

static int ipw_poll_bit(struct ipw_priv *priv, u32 addr,
			u32 bits, u32 mask, int timeout)
{
	int i = 0;

	do {
		if ((_ipw_read32(priv, addr) & mask) == (bits & mask))
			return i;
		mdelay(10);
		i += 10;
	} while (i < timeout);

	return -ETIMEDOUT;
}

static void ipw_set_bit(struct ipw_priv *priv, u32 reg, u32 mask)
{
	_ipw_write32(priv, reg, ipw_read32(priv, reg) | mask);
}

static void ipw_clear_bit(struct ipw_priv *priv, u32 reg, u32 mask)
{
	_ipw_write32(priv, reg, ipw_read32(priv, reg) & ~mask);
}

static inline int _ipw_grab_restricted_access(struct ipw_priv *priv)
{
	if (priv->is_3945) {
		int rc;
		ipw_set_bit(priv, CSR_GP_CNTRL,
			    CSR_GP_CNTRL_REG_FLAG_MAC_ACCESS_REQ);
		rc = ipw_poll_bit(priv, CSR_GP_CNTRL,
				  CSR_GP_CNTRL_REG_VAL_MAC_ACCESS_EN,
				  (CSR_GP_CNTRL_REG_FLAG_MAC_CLOCK_READY |
				   CSR_GP_CNTRL_REG_FLAG_GOING_TO_SLEEP), 50);
		if (rc < 0) {
			IPW_ERROR("MAC is in deep sleep!\n");
			return -EIO;
		}
	}

	priv->status |= STATUS_RESTRICTED;

	return 0;
}

static inline int __ipw_grab_restricted_access(u32 line, struct ipw_priv *priv)
{
	if (priv->status & STATUS_RESTRICTED)
		IPW_ERROR
		    ("Grabbing access while already held at line %d.\n", line);

	return _ipw_grab_restricted_access(priv);
}

static int ipw_grab_restricted_access(struct ipw_priv *priv)
{
	return __ipw_grab_restricted_access(__LINE__, priv);
}

static inline void _ipw_release_restricted_access(struct ipw_priv
						  *priv)
{
	if (priv->is_3945)
		ipw_clear_bit(priv, CSR_GP_CNTRL,
			      CSR_GP_CNTRL_REG_FLAG_MAC_ACCESS_REQ);

	priv->status &= ~STATUS_RESTRICTED;
}

static inline void __ipw_release_restricted_access(u32 line, struct ipw_priv
						   *priv)
{
	if (!(priv->status & STATUS_RESTRICTED))
		IPW_ERROR
		    ("Release unheld restricted access at line %d.\n", line);

	_ipw_release_restricted_access(priv);
}

static void ipw_release_restricted_access(struct ipw_priv *priv)
{
	__ipw_release_restricted_access(__LINE__, priv);
}

static inline u32 _ipw_read_restricted(struct ipw_priv *priv, u32 reg)
{
	u32 val;
	val = _ipw_read32(priv, reg);
	return val;
}

static u32 __ipw_read_restricted(u32 line, struct ipw_priv *priv, u32 reg)
{
	u32 value;

	if (!(priv->status & STATUS_RESTRICTED))
		IPW_ERROR("Unrestricted access from line %d\n", line);

	value = _ipw_read_restricted(priv, reg);

	IPW_DEBUG_IO(" reg = 0x%4X : value = 0x%4x \n", reg, value);

	return value;
}

static u32 ipw_read_restricted(struct ipw_priv *priv, u32 reg)
{
	return __ipw_read_restricted(__LINE__, priv, reg);
}

static void inline _ipw_write_restricted(struct ipw_priv *priv,
					 u32 reg, u32 value)
{
	_ipw_write32(priv, reg, value);
}
static void __ipw_write_restricted(u32 line,
				   struct ipw_priv *priv, u32 reg, u32 value)
{
	if (!(priv->status & STATUS_RESTRICTED))
		IPW_ERROR("Unrestricted access from line %d\n", line);
	_ipw_write_restricted(priv, reg, value);
}

static void ipw_write_restricted(struct ipw_priv *priv, u32 reg, u32 value)
{
	__ipw_write_restricted(__LINE__, priv, reg, value);
}

static void ipw_write_buffer_restricted(struct ipw_priv *priv,
					u32 reg, u32 len, u32 * values)
{
	u32 count = sizeof(u32);
	if ((priv != NULL) && (values != NULL)) {
		for (; 0 < len; len -= count, reg += count, values++)
			_ipw_write_restricted(priv, reg, *values);
	}
}

static int ipw_poll_restricted_bit(struct ipw_priv *priv,
				   u32 addr, u32 mask, int timeout)
{
	int i = 0;

	do {
		if ((_ipw_read_restricted(priv, addr) & mask) == mask)
			return i;
		mdelay(10);
		i += 10;
	} while (i < timeout);

	return -ETIMEDOUT;
}

static inline u32 _ipw_read_restricted_reg(struct ipw_priv *priv, u32 reg)
{
	_ipw_write_restricted(priv, HBUS_TARG_PRPH_RADDR, reg | (3 << 24));
	return _ipw_read_restricted(priv, HBUS_TARG_PRPH_RDAT);
}
static inline u32 __ipw_read_restricted_reg(u32 line,
					    struct ipw_priv *priv, u32 reg)
{
	if (!(priv->status & STATUS_RESTRICTED))
		IPW_ERROR("Unrestricted access from line %d\n", line);
	return _ipw_read_restricted_reg(priv, reg);
}

static inline void _ipw_write_restricted_reg(struct ipw_priv *priv,
					     u32 addr, u32 val)
{
	_ipw_write_restricted(priv, HBUS_TARG_PRPH_WADDR,
			      ((addr & 0x0000FFFF) | (3 << 24)));
	_ipw_write_restricted(priv, HBUS_TARG_PRPH_WDAT, val);
}
static inline void __ipw_write_restricted_reg(u32 line,
					      struct ipw_priv *priv,
					      u32 addr, u32 val)
{
	if (!(priv->status & STATUS_RESTRICTED))
		IPW_ERROR("Unrestricted access from line %d\n", line);
	_ipw_write_restricted_reg(priv, addr, val);
}

static void ipw_write_restricted_reg(struct ipw_priv *priv, u32 addr, u32 val)
{
	__ipw_write_restricted_reg(__LINE__, priv, addr, val);
}

#define _ipw_set_bits_restricted_reg(priv, reg, mask) \
	_ipw_write_restricted_reg(priv, reg, \
				  (_ipw_read_restricted_reg(priv, reg) | mask))
static inline void __ipw_set_bits_restricted_reg(u32 line, struct ipw_priv
						 *priv, u32 reg, u32 mask)
{
	if (!(priv->status & STATUS_RESTRICTED))
		IPW_ERROR("Unrestricted access from line %d\n", line);
	_ipw_set_bits_restricted_reg(priv, reg, mask);
}

static void ipw_set_bits_restricted_reg(struct ipw_priv
					*priv, u32 reg, u32 mask)
{
	__ipw_set_bits_restricted_reg(__LINE__, priv, reg, mask);
}

#define _ipw_set_bits_mask_restricted_reg(priv, reg, bits, mask) \
        _ipw_write_restricted_reg( \
            priv, reg, ((_ipw_read_restricted_reg(priv, reg) & mask) | bits))
static void inline __ipw_set_bits_mask_restricted_reg(u32 line, struct ipw_priv
						      *priv, u32 reg,
						      u32 bits, u32 mask)
{
	if (!(priv->status & STATUS_RESTRICTED))
		IPW_ERROR("Unrestricted access from line %d\n", line);
	_ipw_set_bits_mask_restricted_reg(priv, reg, bits, mask);
}

static void ipw_set_bits_mask_restricted_reg(struct ipw_priv *priv,
					     u32 reg, u32 bits, u32 mask)
{
	__ipw_set_bits_mask_restricted_reg(__LINE__, priv, reg, bits, mask);
}

static void ipw_clear_bits_restricted_reg(struct ipw_priv
					  *priv, u32 reg, u32 mask)
{
	u32 val = _ipw_read_restricted_reg(priv, reg);
	_ipw_write_restricted_reg(priv, reg, (val & ~mask));
}

static u32 ipw_read_restricted_mem(struct ipw_priv *priv, u32 addr)
{
	ipw_write_restricted(priv, HBUS_TARG_MEM_RADDR, addr);
	return ipw_read_restricted(priv, HBUS_TARG_MEM_RDAT);
}

static void ipw_write_restricted_mem(struct ipw_priv *priv, u32 addr, u32 val)
{
	ipw_write_restricted(priv, HBUS_TARG_MEM_WADDR, addr);
	ipw_write_restricted(priv, HBUS_TARG_MEM_WDAT, val);
}

static void ipw_enable_interrupts(struct ipw_priv *priv)
{
	if (priv->status & STATUS_INT_ENABLED)
		return;
	priv->status |= STATUS_INT_ENABLED;
	ipw_write32(priv, CSR_INT_MASK, CSR_INI_SET_MASK);
}

static inline void ipw_disable_interrupts(struct ipw_priv *priv)
{
	if (!(priv->status & STATUS_INT_ENABLED))
		return;
	priv->status &= ~STATUS_INT_ENABLED;
	ipw_write32(priv, CSR_INT_MASK, 0x00000000);
	ipw_write32(priv, CSR_INT, CSR_INI_SET_MASK);
	ipw_write32(priv, CSR_FH_INT_STATUS, 0xff);
	ipw_write32(priv, CSR_FH_INT_STATUS, 0x00070000);
}

/************************* END *******************************/

static const char *desc_lookup(int i)
{
	switch (i) {
	case 1:
		return "FAIL";
	case 2:
		return "BAD_PARAM";
	case 3:
		return "BAD_CHECKSUM";
	case 4:
		return "NMI_INTERRUPT";
	case 5:
		return "SYSASSERT";
	case 6:
		return "FATAL_ERROR";
	}

	return "UNKNOWN";
}

#define ERROR_START_OFFSET  (1 * sizeof(u32))
#define ERROR_ELEM_SIZE     (7 * sizeof(u32))
static void ipw_dump_nic_error_log(struct ipw_priv *priv)
{
	u32 desc, time, blnk, blink2, ilnk, ilink2, idata, i, count, base;
	int rc;

	base = priv->card_alive.error_event_table_ptr;

	if (!VALID_RTC_DATA_ADDR(base)) {
		IPW_ERROR("Not valid error log pointer 0x%08X\n", base);
		return;
	}

	rc = ipw_grab_restricted_access(priv);
	if (rc) {
		IPW_WARNING("Can not read from adapter at this time.\n");
		return;
	}

	count = ipw_read_restricted_mem(priv, base);

	if (ERROR_START_OFFSET <= count * ERROR_ELEM_SIZE) {
		IPW_ERROR("Start IPW Error Log Dump:\n");
		IPW_ERROR("Status: 0x%08X, Config: %08X count: %d\n",
			  priv->status, priv->config, count);
	}

	IPW_ERROR("Desc       Time       asrtPC     const      "
		  "ilink1     nmiPC      Line\n");
	for (i = ERROR_START_OFFSET;
	     i < (count * ERROR_ELEM_SIZE) + ERROR_START_OFFSET;
	     i += ERROR_ELEM_SIZE) {
		desc = ipw_read_restricted_mem(priv, base + i);
		time =
		    ipw_read_restricted_mem(priv, base + i + 1 * sizeof(u32));
		blnk =
		    ipw_read_restricted_mem(priv, base + i + 2 * sizeof(u32));
		blink2 =
		    ipw_read_restricted_mem(priv, base + i + 3 * sizeof(u32));
		ilnk =
		    ipw_read_restricted_mem(priv, base + i + 4 * sizeof(u32));
		ilink2 =
		    ipw_read_restricted_mem(priv, base + i + 5 * sizeof(u32));
		idata =
		    ipw_read_restricted_mem(priv, base + i + 6 * sizeof(u32));

		IPW_ERROR
		    ("%-8s (#%d) 0x%08X 0x%08X 0x%08X 0x%08X 0x%08X %u\n",
		     desc_lookup(desc), desc, time, blnk, blink2,
		     ilnk, ilink2, idata);
	}

	ipw_release_restricted_access(priv);

}

#define IPW_EVT_DISABLE (0)	/* 1 = enable the ipw_disable_events() function */
#define IPW_EVT_DISABLE_SIZE (1532/32)
/* Disable selected events in uCode event log, by writing "1"s into "disable"
 *   bitmap in SRAM.  Bit position corresponds to Event # (id/type).
 *   Default values of 0 enable uCode events to be logged.
 * Use for only special debugging.  This function is just a placeholder as-is,
 *   you'll need to provide the special bits! ...
 *   ... and set IPW_EVT_DISABLE to 1. */
static void ipw_disable_events(struct ipw_priv *priv)
{
	int rc;
	int i;
	u32 base;		/* SRAM address of event log header */
	u32 disable_ptr;	/* SRAM address of event-disable bitmap array */
	u32 array_size;		/* # of u32 entries in array */
	u32 evt_disable[IPW_EVT_DISABLE_SIZE] = {
		0x00000000,	/*   31 -    0  Event id numbers */
		0x00000000,	/*   63 -   32 */
		0x00000000,	/*   95 -   64 */
		0x00000000,	/*  127 -   96 */
		0x00000000,	/*  159 -  128 */
		0x00000000,	/*  191 -  160 */
		0x00000000,	/*  223 -  192 */
		0x00000000,	/*  255 -  224 */
		0x00000000,	/*  287 -  256 */
		0x00000000,	/*  319 -  288 */
		0x00000000,	/*  351 -  320 */
		0x00000000,	/*  383 -  352 */
		0x00000000,	/*  415 -  384 */
		0x00000000,	/*  447 -  416 */
		0x00000000,	/*  479 -  448 */
		0x00000000,	/*  511 -  480 */
		0x00000000,	/*  543 -  512 */
		0x00000000,	/*  575 -  544 */
		0x00000000,	/*  607 -  576 */
		0x00000000,	/*  639 -  608 */
		0x00000000,	/*  671 -  640 */
		0x00000000,	/*  703 -  672 */
		0x00000000,	/*  735 -  704 */
		0x00000000,	/*  767 -  736 */
		0x00000000,	/*  799 -  768 */
		0x00000000,	/*  831 -  800 */
		0x00000000,	/*  863 -  832 */
		0x00000000,	/*  895 -  864 */
		0x00000000,	/*  927 -  896 */
		0x00000000,	/*  959 -  928 */
		0x00000000,	/*  991 -  960 */
		0x00000000,	/* 1023 -  992 */
		0x00000000,	/* 1055 - 1024 */
		0x00000000,	/* 1087 - 1056 */
		0x00000000,	/* 1119 - 1088 */
		0x00000000,	/* 1151 - 1120 */
		0x00000000,	/* 1183 - 1152 */
		0x00000000,	/* 1215 - 1184 */
		0x00000000,	/* 1247 - 1216 */
		0x00000000,	/* 1279 - 1248 */
		0x00000000,	/* 1311 - 1280 */
		0x00000000,	/* 1343 - 1312 */
		0x00000000,	/* 1375 - 1344 */
		0x00000000,	/* 1407 - 1376 */
		0x00000000,	/* 1439 - 1408 */
		0x00000000,	/* 1471 - 1440 */
		0x00000000,	/* 1503 - 1472 */
	};

	base = priv->card_alive.log_event_table_ptr;
	if (!VALID_RTC_DATA_ADDR(base)) {
		IPW_ERROR("Invalid event log pointer 0x%08X\n", base);
		return;
	}

	rc = ipw_grab_restricted_access(priv);
	if (rc) {
		IPW_WARNING("Can not read from adapter at this time.\n");
		return;
	}

	disable_ptr = ipw_read_restricted_mem(priv, base + (4 * sizeof(u32)));
	array_size = ipw_read_restricted_mem(priv, base + (5 * sizeof(u32)));
	ipw_release_restricted_access(priv);

	if (IPW_EVT_DISABLE && (array_size == IPW_EVT_DISABLE_SIZE)) {
		IPW_DEBUG_INFO("Disabling selected uCode log events at 0x%x\n",
			       disable_ptr);
		rc = ipw_grab_restricted_access(priv);
		for (i = 0; i < IPW_EVT_DISABLE_SIZE; i++) {
			ipw_write_restricted_mem(priv,
						 disable_ptr +
						 (i * sizeof(u32)),
						 evt_disable[i]);
		}
		ipw_release_restricted_access(priv);
	} else {
		IPW_DEBUG_INFO("Selected uCode log events may be disabled\n");
		IPW_DEBUG_INFO("  by writing \"1\"s into disable bitmap\n");
		IPW_DEBUG_INFO("  in SRAM at 0x%x, size %d u32s\n",
			       disable_ptr, array_size);
	}

}

#define EVENT_START_OFFSET (6 * sizeof(u32))
/* Must be called with ipw_grab_restricted_access() already obtained! */
static void ipw_print_event_log(struct ipw_priv *priv, u32 start_idx,
				u32 num_events, u32 mode)
{
	u32 i;
	u32 base;		/* SRAM byte address of event log header */
	u32 event_size;		/* 2 u32s, or 3 u32s if timestamp recorded */
	u32 ptr;		/* SRAM byte address of log data */
	u32 ev, time, data;	/* event log data */

	if (num_events == 0)
		return;

	base = priv->card_alive.log_event_table_ptr;

	if (mode == 0)
		event_size = 2 * sizeof(u32);
	else
		event_size = 3 * sizeof(u32);

	ptr = base + EVENT_START_OFFSET + (start_idx * event_size);

	/* "time" is actually "data" for mode 0 (no timestamp).
	 * place event id # at far right for easier visual parsing. */
	for (i = 0; i < num_events; i++) {
		ev = ipw_read_restricted_mem(priv, ptr);
		ptr += sizeof(u32);
		time = ipw_read_restricted_mem(priv, ptr);
		ptr += sizeof(u32);
		if (mode == 0) {
			IPW_ERROR("0x%08x\t%04u\n", time, ev);	/* data, ev */
		} else {
			data = ipw_read_restricted_mem(priv, ptr);
			ptr += sizeof(u32);
			IPW_ERROR("%010u\t0x%08x\t%04u\n", time, data, ev);
		}
	}
}

static void ipw_dump_nic_event_log(struct ipw_priv *priv)
{
	int rc;
	u32 base;		/* SRAM byte address of event log header */
	u32 capacity;		/* event log capacity in # entries */
	u32 mode;		/* 0 - no timestamp, 1 - timestamp recorded */
	u32 num_wraps;		/* # times uCode wrapped to top of log */
	u32 next_entry;		/* index of next entry to be written by uCode */
	u32 size;		/* # entries that we'll print */

	base = priv->card_alive.log_event_table_ptr;
	if (!VALID_RTC_DATA_ADDR(base)) {
		IPW_ERROR("Invalid event log pointer 0x%08X\n", base);
		return;
	}

	rc = ipw_grab_restricted_access(priv);
	if (rc) {
		IPW_WARNING("Can not read from adapter at this time.\n");
		return;
	}

	/* event log header */
	capacity = ipw_read_restricted_mem(priv, base);
	mode = ipw_read_restricted_mem(priv, base + (1 * sizeof(u32)));
	num_wraps = ipw_read_restricted_mem(priv, base + (2 * sizeof(u32)));
	next_entry = ipw_read_restricted_mem(priv, base + (3 * sizeof(u32)));

	size = num_wraps ? capacity : next_entry;

	/* bail out if nothing in log */
	if (size == 0) {
		ipw_release_restricted_access(priv);
		return;
	}

	IPW_ERROR("Start IPW Event Log Dump: display count %d, wraps %d\n",
		  size, num_wraps);

	/* if uCode has wrapped back to top of log, start at the oldest entry,
	 *    i.e the next one that uCode would fill. */
	if (num_wraps) {
		ipw_print_event_log(priv, next_entry,
				    capacity - next_entry, mode);
	}

	/* (then/else) start at top of log */
	ipw_print_event_log(priv, 0, next_entry, mode);

	ipw_release_restricted_access(priv);
}

static inline int ipw_is_ready(struct ipw_priv *priv)
{
	/* The adapter is 'ready' if READY and GEO_CONFIGURED bits are
	 * set but EXIT_PENDING is not */
	return ((priv->status & (STATUS_READY |
				 STATUS_GEO_CONFIGURED |
				 STATUS_EXIT_PENDING)) ==
		(STATUS_READY | STATUS_GEO_CONFIGURED)) ? 1 : 0;
}

static inline int ipw_is_associated(struct ipw_priv *priv)
{
	return (priv->active_rxon.filter_flags & RXON_FILTER_ASSOC_MSK) ? 1 : 0;
}

static inline int ipw_is_alive(struct ipw_priv *priv)
{
	return (priv->status & STATUS_ALIVE) ? 1 : 0;
}

static inline int ipw_is_init(struct ipw_priv *priv)
{
	return (priv->status & STATUS_INIT) ? 1 : 0;
}

static inline int ipw_is_ready_rf(struct ipw_priv *priv)
{

	if (priv->status & STATUS_RF_KILL_MASK)
		return 0;

	return ipw_is_ready(priv);
}

#define IPW_RX_HDR(x) ((struct ipw_rx_frame_hdr *)(\
                       x->u.rx_frame.stats.payload + \
                       x->u.rx_frame.stats.mib_count))
#define IPW_RX_END(x) ((struct ipw_rx_frame_end *)(\
                       IPW_RX_HDR(x)->payload + \
                       le16_to_cpu(IPW_RX_HDR(x)->len)))
#define IPW_RX_STATS(x) (&x->u.rx_frame.stats)
#define IPW_RX_DATA(x) (IPW_RX_HDR(x)->payload)

#define IPW_CMD(x) case x : return #x
#define IPW_CMD3945(x) case REPLY_ ## x : return #x

static const char *get_cmd_string(u8 cmd)
{
	switch (cmd) {
		IPW_CMD(SCAN_START_NOTIFICATION);
		IPW_CMD(SCAN_RESULTS_NOTIFICATION);
		IPW_CMD(SCAN_COMPLETE_NOTIFICATION);
		IPW_CMD(STATISTICS_NOTIFICATION);
		IPW_CMD3945(ALIVE);
		IPW_CMD3945(ERROR);
		IPW_CMD3945(RXON_ASSOC);
		IPW_CMD3945(RXON);
		IPW_CMD3945(QOS_PARAM);
		IPW_CMD3945(RXON_TIMING);
		IPW_CMD3945(ADD_STA);
		IPW_CMD3945(RX);
		IPW_CMD3945(TX);
		IPW_CMD3945(BCON);
		IPW_CMD3945(RATE_SCALE);
		IPW_CMD3945(LEDS_CMD);
		IPW_CMD3945(SCAN_ABORT_CMD);
		IPW_CMD3945(TX_BEACON);
		IPW_CMD3945(BT_CONFIG);
		IPW_CMD3945(SCAN_CMD);
		IPW_CMD3945(TX_PWR_TABLE_CMD);
		IPW_CMD3945(STATISTICS_CMD);
		IPW_CMD3945(CARD_STATE_CMD);
		IPW_CMD3945(TX_LINK_QUALITY_CMD);
	case POWER_TABLE_CMD:
		return "POWER_TABLE_CMD";
	default:
		return "UNKNOWN";

	}
}

#define HOST_COMPLETE_TIMEOUT (HZ / 2)

static inline int is_cmd_sync(struct ipw_host_cmd *cmd)
{
	return !(cmd->meta.flags & CMD_ASYNC);
}

static inline int is_cmd_small(struct ipw_host_cmd *cmd)
{
	return !(cmd->meta.flags & CMD_SIZE_HUGE);
}

static inline int cmd_needs_lock(struct ipw_host_cmd *cmd)
{
	return !(cmd->meta.flags & CMD_NO_LOCK);
}

/*
  low level function to send 3945 command.

  The caller can control if the command is async/sync.
  sync    : 1 do sync call and sleep until notified the command is complete
  resp    : pointer to command response.
  do_lock : 1 use spin lock, 0 assume the user has lock.
            we can not have do_lock=0 and sync = 1.
  is_huge_cmd : used for scan since this command is very large, we use
                special buffer to copy the command.
*/
//static int ipw_send_cmd(struct ipw_priv *priv, struct ipw_host_cmd *cmd)
 //   __attribute__ ((warn_unused_result));
static int ipw_send_cmd(struct ipw_priv *priv, struct ipw_host_cmd *cmd)
{
	int rc;
	unsigned long flags = 0;

	/* If this is an asynchronous command, and we are in a shutdown
	 * process then don't let it start */
	if (!is_cmd_sync(cmd) && (priv->status & STATUS_EXIT_PENDING))
		return -EBUSY;

	/*
	 * The following BUG_ONs are meant to catch programming API misuse
	 * and not run-time failures due to timing, resource constraint, etc.
	 */

	/* A command can not be asynchronous AND expect an SKB to be set */
	BUG_ON((cmd->meta.flags & CMD_ASYNC)
	       && (cmd->meta.flags & CMD_WANT_SKB));

	/* The skb/callback union must be NULL if an SKB is requested */
	BUG_ON(cmd->meta.u.skb && (cmd->meta.flags & CMD_WANT_SKB));

	/* A command can not be synchronous AND have a callback set */
	BUG_ON(is_cmd_sync(cmd) && cmd->meta.u.callback);

	/* An asynchronous command MUST have a callback */
	BUG_ON((cmd->meta.flags & CMD_ASYNC)
	       && !cmd->meta.u.callback);

	/* A command can not be synchronous AND not use locks */
	BUG_ON(is_cmd_sync(cmd) && (cmd->meta.flags & CMD_NO_LOCK));

	if (cmd_needs_lock(cmd))
		spin_lock_irqsave(&priv->lock, flags);

	if (is_cmd_sync(cmd) && (priv->status & STATUS_HCMD_ACTIVE)) {
		IPW_ERROR("Error sending %s: "
			  "Already sending a host command\n",
			  get_cmd_string(cmd->id));
		if (cmd_needs_lock(cmd))
			spin_unlock_irqrestore(&priv->lock, flags);
		return -EBUSY;
	}

	if (is_cmd_sync(cmd))
		priv->status |= STATUS_HCMD_ACTIVE;

	/* When the SKB is provided in the tasklet, it needs
	 * a backpointer to the originating caller so it can
	 * actually copy the skb there */
	if (cmd->meta.flags & CMD_WANT_SKB)
		cmd->meta.u.source = &cmd->meta;

	cmd->meta.len = cmd->len;

	rc = ipw_queue_tx_hcmd(priv, cmd);
	if (rc) {
		if (is_cmd_sync(cmd))
			priv->status &= ~STATUS_HCMD_ACTIVE;
		if (cmd_needs_lock(cmd))
			spin_unlock_irqrestore(&priv->lock, flags);

		IPW_ERROR("Error sending %s: "
			  "ipw_queue_tx_hcmd failed: %d\n",
			  get_cmd_string(cmd->id), rc);

		return -ENOSPC;
	}
	if (cmd_needs_lock(cmd))
		spin_unlock_irqrestore(&priv->lock, flags);

	if (is_cmd_sync(cmd)) {
		rc = wait_event_interruptible_timeout(priv->
						      wait_command_queue,
						      !(priv->
							status &
							STATUS_HCMD_ACTIVE),
						      HOST_COMPLETE_TIMEOUT);
		if (rc == 0) {
			if (cmd_needs_lock(cmd))
				spin_lock_irqsave(&priv->lock, flags);

			if (priv->status & STATUS_HCMD_ACTIVE) {
				IPW_ERROR("Error sending %s: "
					  "time out after %dms.\n",
					  get_cmd_string(cmd->id),
					  jiffies_to_msecs
					  (HOST_COMPLETE_TIMEOUT));
				priv->status &= ~STATUS_HCMD_ACTIVE;
				if ((cmd->meta.flags & CMD_WANT_SKB)
				    && cmd->meta.u.skb) {
					dev_kfree_skb_any(cmd->meta.u.skb);
					cmd->meta.u.skb = NULL;
				}

				if (cmd_needs_lock(cmd))
					spin_unlock_irqrestore(&priv->
							       lock, flags);
				return -ETIMEDOUT;
			}

			if (cmd_needs_lock(cmd))
				spin_unlock_irqrestore(&priv->lock, flags);
		}
	}

	if (priv->status & STATUS_RF_KILL_HW) {
		if ((cmd->meta.flags & CMD_WANT_SKB)
		    && cmd->meta.u.skb) {
			dev_kfree_skb_any(cmd->meta.u.skb);
			cmd->meta.u.skb = NULL;
		}

		IPW_DEBUG_INFO("Command %s aborted: RF KILL Switch\n",
			       get_cmd_string(cmd->id));

		return -ECANCELED;
	}

	if (priv->status & STATUS_FW_ERROR) {
		if ((cmd->meta.flags & CMD_WANT_SKB)
		    && cmd->meta.u.skb) {
			dev_kfree_skb_any(cmd->meta.u.skb);
			cmd->meta.u.skb = NULL;
		}

		IPW_DEBUG_INFO("Command %s failed: FW Error\n",
			       get_cmd_string(cmd->id));

		return -EIO;
	}

	if ((cmd->meta.flags & CMD_WANT_SKB) && !cmd->meta.u.skb) {
		IPW_ERROR("Error: Response NULL in '%s'\n",
			  get_cmd_string(cmd->id));
		return -EIO;
	}

	return 0;
}

static int ipw_send_cmd_pdu(struct ipw_priv *priv, u8 id, u16 len, void *data)
{
	struct ipw_host_cmd cmd = {
		.id = id,
		.len = len,
		.data = data,
	};

	return ipw_send_cmd(priv, &cmd);
}

static int ipw_send_cmd_u32(struct ipw_priv *priv, u8 id, u32 val)
    __attribute__ ((warn_unused_result));
static int ipw_send_cmd_u32(struct ipw_priv *priv, u8 id, u32 val)
{
	struct ipw_host_cmd cmd = {
		.id = id,
		.len = sizeof(val),
		.data = &val,
	};

	return ipw_send_cmd(priv, &cmd);
}

static struct ipw_link_blink link_led_table[] = {
	[IPW_LED_LINK_RADIOOFF] = {1000, 0, 0},
	[IPW_LED_LINK_UNASSOCIATED] = {10000, 200, 2},
	[IPW_LED_LINK_SCANNING] = {1000, 60, 10},
	[IPW_LED_LINK_ASSOCIATED] = {1000, 0, 1},
	[IPW_LED_LINK_ROAMING] = {1000, 10, 100},
};

static void ipw_update_link_led(struct ipw_priv *priv)
{
	int state;
	struct ipw_led_cmd led_cmd = {
		.id = IPW_LED_LINK,
	};

	/* If we are in RF KILL then we can't send the LED
	 * command, so cache that the LED is in the
	 * RADIOOFF state so we'll turn it back on when
	 * we come back from RF KILL. */
	if (priv->status & STATUS_RF_KILL_MASK) {
		IPW_DEBUG_LED("Not sending LINK LED off cmd due to RF KILL.\n");
		priv->led_state = IPW_LED_LINK_RADIOOFF;
		return;
	}

	if (priv->status & STATUS_IN_SUSPEND) {
		IPW_DEBUG_LED("Not sending LINK LED off cmd due to SUSPEND.\n");
		priv->led_state = IPW_LED_LINK_RADIOOFF;
		return;
	}

	if ((priv->config & CFG_NO_LED) ||
	    (priv->status & STATUS_EXIT_PENDING) ||
	    !(priv->status & STATUS_READY))
		state = IPW_LED_LINK_RADIOOFF;
	else if (priv->status & STATUS_SCANNING)
		state = IPW_LED_LINK_SCANNING;
	else if (ipw_is_associated(priv))
		state = IPW_LED_LINK_ASSOCIATED;
	else
		state = IPW_LED_LINK_UNASSOCIATED;

	if (state == priv->led_state)
		return;

	led_cmd.interval = link_led_table[state].interval;
	led_cmd.on = link_led_table[state].on;
	led_cmd.off = link_led_table[state].off;

	priv->led_state = state;

	switch (state) {
	case IPW_LED_LINK_RADIOOFF:
		IPW_DEBUG_LED("Link state: RADIO OFF\n");
		break;
	case IPW_LED_LINK_ROAMING:
		IPW_DEBUG_LED("Link state: ROAMING\n");
		break;
	case IPW_LED_LINK_ASSOCIATED:
		IPW_DEBUG_LED("Link state: ASSOCIATED\n");
		break;
	case IPW_LED_LINK_UNASSOCIATED:
		IPW_DEBUG_LED("Link state: UNASSOCIATED\n");
		break;
	case IPW_LED_LINK_SCANNING:
		IPW_DEBUG_LED("Link state: SCANNING\n");
		break;
	default:
		IPW_DEBUG_LED("Link state: UNKNOWN\n");
		break;
	}

	IPW_DEBUG_LED("On: %d, Off: %d, Interval: %d\n",
		      led_cmd.on, led_cmd.off, led_cmd.interval);
/*	IGNORE_RETURN(ipw_send_cmd_pdu(priv, REPLY_LEDS_CMD,
				       sizeof(struct ipw_led_cmd), &led_cmd));
*/
}

static struct ipw_activity_blink activity_led_table[] = {
	{300, 25, 25},
	{200, 40, 40},
	{100, 55, 55},
	{70, 65, 65},
	{50, 75, 75},
	{20, 85, 85},
	{10, 95, 95},
	{5, 110, 110},
	{1, 130, 130},
	{0, 167, 167},
};

/*
  set to correct blink rate. set to solid blink we can not find correct
  rate value or the blink value exceed the blink threshold
*/
static void get_led_blink_rate(struct ipw_priv *priv,
			       struct ipw_activity_blink *blink)
{
	/* Adjust to Mbs throughput table */
	u32 bit_count = (priv->led_packets * 10) >> 17;
	u32 index = 0;

	/* If < 1mbs then just quick blink over long duration to
	 * indicate "some" activity */
	if (!bit_count) {
		blink->on = 10;
		blink->off = 200;
		return;
	}

	while ((bit_count <= activity_led_table[index].throughput) &&
	       index < ARRAY_SIZE(activity_led_table))
		index++;

	if (index == ARRAY_SIZE(activity_led_table)) {
		blink->on = 1;	/* turn on */
		blink->off = 0;	/* never turn off */
		return;
	}

	blink->on = activity_led_table[index].on;
	blink->off = activity_led_table[index].off;
}

#define IPW_ACTIVITY_PERIOD msecs_to_jiffies(100)

static void ipw_update_activity_led(struct ipw_priv *priv)
{
	static struct ipw_activity_blink last_blink = { 0, 0, 0 };
	struct ipw_activity_blink blink;
	struct ipw_led_cmd led_cmd = {
		.id = IPW_LED_ACTIVITY,
		.interval = IPW_LED_INTERVAL,
	};

	/* If configured to not use LEDs or LEDs are disabled,
	 * then we don't toggle a activity led */
	if (priv->config & CFG_NO_LED || (priv->status & STATUS_EXIT_PENDING)) {
		blink.on = blink.off = 0;
	} else {
		IPW_DEBUG_LED("total Tx/Rx bytes = %lu\n", priv->led_packets);
		get_led_blink_rate(priv, &blink);
		priv->led_packets = 0;
	}

	if (last_blink.on != blink.on || last_blink.off != blink.off) {
		last_blink = blink;
		IPW_DEBUG_LED
		    ("Blink rate is %d On, %d ms Off, at %d interval.\n",
		     blink.on, blink.off, led_cmd.interval);

		led_cmd.off = blink.off;
		led_cmd.on = blink.on;
/*
		IGNORE_RETURN(ipw_send_cmd_pdu(priv, REPLY_LEDS_CMD,
					       sizeof(struct
						      ipw_led_cmd), &led_cmd));
*/
	}
}

static void ipw_setup_activity_timer(struct ipw_priv *priv)
{
	if (priv->activity_timer_active)
		return;

	priv->activity_timer_active = 1;
	queue_delayed_work(priv->workqueue, &priv->activity_timer,
			   IPW_ACTIVITY_PERIOD);
}

static void ipw_bg_activity_timer(struct work_struct *work)
{
	struct ipw_priv *priv =
	    container_of(work, struct ipw_priv, activity_timer.work);

	if (priv->status & STATUS_EXIT_PENDING)
		return;

	mutex_lock(&priv->mutex);

	ipw_update_activity_led(priv);

	/* If we haven't Tx/Rx any packets, then don't bother
	 * running this timer any more until we do one of those things */
	if (!priv->led_packets)
		priv->activity_timer_active = 0;
	else
		queue_delayed_work(priv->workqueue,
				   &priv->activity_timer, IPW_ACTIVITY_PERIOD);

	mutex_unlock(&priv->mutex);
}

static void ipw_update_tech_led(struct ipw_priv *priv)
{
}

#define IPW_EEPROM_LED_FROM_EEPROM     	0x80
#define IPW_EEPROM_LED_MODE             0x03
#define IPW_EEPROM_LED_SAVE		0x04

static int ipw_block_until_driver_ready(struct ipw_priv *priv)
{
	if (!ipw_is_alive(priv))
		return -EAGAIN;

	return 0;
}

/*
 * The following adds a new attribute to the sysfs representation
 * of this device driver (i.e. a new file in /sys/bus/pci/drivers/ipw/)
 * used for controlling the debug level.
 *
 * See the level definitions in ipw for details.
 */
static ssize_t show_debug_level(struct device_driver *d, char *buf)
{
	return sprintf(buf, "0x%08X\n", ipw_debug_level);
}
static ssize_t store_debug_level(struct device_driver *d,
				 const char *buf, size_t count)
{
	char *p = (char *)buf;
	u32 val;

	val = simple_strtoul(p, &p, 0);
	if (p == buf)
		printk(KERN_INFO DRV_NAME
		       ": %s is not in hex or decimal form.\n", buf);
	else
		ipw_debug_level = val;

	return strnlen(buf, count);
}

static DRIVER_ATTR(debug_level, S_IWUSR | S_IRUGO,
		   show_debug_level, store_debug_level);

static int ipw_send_statistics_request(struct ipw_priv *priv)
{
	return ipw_send_cmd_u32(priv, REPLY_STATISTICS_CMD, 0);
}

static void ipw_clear_free_frames(struct ipw_priv *priv)
{
	struct list_head *element;

	IPW_DEBUG_INFO("%d frames on pre-allocated heap on clear.\n",
		       priv->frames_count);

	while (!list_empty(&priv->free_frames)) {
		element = priv->free_frames.next;
		list_del(element);
		kfree(list_entry(element, struct ipw_frame, list));
		priv->frames_count--;
	}

	if (priv->frames_count) {
		IPW_WARNING
		    ("%d frames still in use.  Did we lose one?\n",
		     priv->frames_count);
		priv->frames_count = 0;
	}
}

#define FREE_FRAME_THRESHOLD 5
static struct ipw_frame *ipw_get_free_frame(struct ipw_priv *priv)
{
	struct ipw_frame *frame;
	struct list_head *element;
	if (list_empty(&priv->free_frames)) {
		frame = kzalloc(sizeof(*frame), GFP_ATOMIC);
		if (!frame) {
			IPW_ERROR("Could not allocate frame!\n");
			return NULL;
		}

		priv->frames_count++;

		if (priv->frames_count >= FREE_FRAME_THRESHOLD) {
			IPW_WARNING("%d frames allocated.  "
				    "Are we losing them?\n",
				    priv->frames_count);
		}

		return frame;
	}

	element = priv->free_frames.next;
	list_del(element);
	return list_entry(element, struct ipw_frame, list);
}

static void ipw_free_frame(struct ipw_priv *priv, struct ipw_frame *frame)
{
	memset(frame, 0, sizeof(*frame));
	list_add(&frame->list, &priv->free_frames);
}

static int ipw_get_temperature(struct ipw_priv *priv)
{
	return ipw_read32(priv, CSR_UCODE_DRV_GP2);
}

static ssize_t show_temperature(struct device *d,
				struct device_attribute *attr, char *buf)
{
	struct ipw_priv *priv = (struct ipw_priv *)d->driver_data;
	int rc;

	rc = ipw_block_until_driver_ready(priv);
	if (rc)
		return rc;

	return sprintf(buf, "%d\n", ipw_get_temperature(priv));
}

static DEVICE_ATTR(temperature, S_IRUGO, show_temperature, NULL);

static int reg_set_txpower(struct ipw_priv *priv, s8 power);

static ssize_t show_tx_power(struct device *d,
			     struct device_attribute *attr, char *buf)
{
	struct ipw_priv *priv = (struct ipw_priv *)d->driver_data;
	return sprintf(buf, "%d\n", priv->user_txpower_limit);
}

static ssize_t store_tx_power(struct device *d,
			      struct device_attribute *attr,
			      const char *buf, size_t count)
{
	struct ipw_priv *priv = (struct ipw_priv *)d->driver_data;
	char *p = (char *)buf;
	u32 val;

	val = simple_strtoul(p, &p, 10);
	if (p == buf)
		printk(KERN_INFO DRV_NAME
		       ": %s is not in decimal form.\n", buf);
	else
		reg_set_txpower(priv, val);

	return count;
}

static DEVICE_ATTR(tx_power, S_IWUSR | S_IRUGO, show_tx_power, store_tx_power);

static ssize_t store_retry_rate(struct device *d,
				struct device_attribute *attr,
				const char *buf, size_t count)
{
	struct ipw_priv *priv = dev_get_drvdata(d);

	priv->retry_rate = simple_strtoul(buf, NULL, 0);
	if (priv->retry_rate <= 0)
		priv->retry_rate = 1;

	return count;
}

static ssize_t show_retry_rate(struct device *d,
			       struct device_attribute *attr, char *buf)
{
	struct ipw_priv *priv = dev_get_drvdata(d);
	return sprintf(buf, "%d", priv->retry_rate);
}

static DEVICE_ATTR(retry_rate, S_IWUSR | S_IRUSR, show_retry_rate,
		   store_retry_rate);

static ssize_t store_power_level(struct device *d,
				 struct device_attribute *attr,
				 const char *buf, size_t count)
{
	struct ipw_priv *priv = dev_get_drvdata(d);
	int mode;

	mode = simple_strtoul(buf, NULL, 0);
	mutex_lock(&priv->mutex);
	ipw_wx_set_powermode(priv, mode);
	mutex_unlock(&priv->mutex);
	return count;
}

#define MAX_WX_STRING 80

/* Values are in microsecond */
static const s32 timeout_duration[] = {
	350000,
	250000,
	75000, 37000, 25000,
};
static const s32 period_duration[] = {
	400000,
	700000,
	1000000,
	1000000,
	1000000
};

static ssize_t show_power_level(struct device *d,
				struct device_attribute *attr, char *buf)
{
	struct ipw_priv *priv = dev_get_drvdata(d);
	int level = IPW_POWER_LEVEL(priv->power_mode);
	char *p = buf;

	p += sprintf(p, "Power save level: %d ", level);
	switch (level) {
	case IPW_POWER_MODE_CAM:
	case IPW_POWER_AC:
		p += sprintf(p, "(AC)");
		break;
	case IPW_POWER_BATTERY:
		p += sprintf(p, "(BATTERY)");
		break;
	default:
		p += sprintf(p,
			     "(Timeout %dms, Period %dms)",
			     timeout_duration[level - 1] / 1000,
			     period_duration[level - 1] / 1000);
	}

	if (!(priv->power_mode & IPW_POWER_ENABLED))
		p += sprintf(p, " OFF\n");
	else
		p += sprintf(p, " \n");

	return (p - buf + 1);

}

static DEVICE_ATTR(power_level, S_IWUSR | S_IRUSR, show_power_level,
		   store_power_level);

static ssize_t show_channels(struct device *d,
			     struct device_attribute *attr, char *buf)
{
	struct ipw_priv *priv = dev_get_drvdata(d);
	int len = 0, i;
	struct ieee80211_channel *channels = NULL;
	struct ieee80211_hw_mode *hw_mode = NULL;
	int count = 0;

	if (!ipw_is_ready(priv))
		return -EAGAIN;

	hw_mode = ipw_get_hw_mode(priv, MODE_IEEE80211G);
	if (!hw_mode)
		hw_mode = ipw_get_hw_mode(priv, MODE_IEEE80211B);
	if (hw_mode) {
		channels = hw_mode->channels;
		count = hw_mode->num_channels;
	}

	len +=
	    sprintf(&buf[len],
		    "Displaying %d channels in 2.4Ghz band "
		    "(802.11bg):\n", count);

	for (i = 0; i < count; i++) {
		len += sprintf(&buf[len], "%d: %ddBm: BSS%s%s, %s.\n",
			       channels[i].chan,
			       channels[i].power_level,
			       channels[i].
			       flag & IEEE80211_CHAN_W_RADAR_DETECT ?
			       " (IEEE 802.11h required)" : "",
			       (!(channels[i].flag & IEEE80211_CHAN_W_IBSS)
				|| (channels[i].
				    flag &
				    IEEE80211_CHAN_W_RADAR_DETECT)) ? "" :
			       ", IBSS",
			       channels[i].
			       flag & IEEE80211_CHAN_W_ACTIVE_SCAN ?
			       "active/passive" : "passive only");
	}

	hw_mode = ipw_get_hw_mode(priv, MODE_IEEE80211A);
	if (hw_mode) {
		channels = hw_mode->channels;
		count = hw_mode->num_channels;
	} else {
		channels = NULL;
		count = 0;
	}

	len +=
	    sprintf(&buf[len],
		    "Displaying %d channels in 5.2Ghz band "
		    "(802.11a):\n", count);

	for (i = 0; i < count; i++) {
		len += sprintf(&buf[len], "%d: %ddBm: BSS%s%s, %s.\n",
			       channels[i].chan,
			       channels[i].power_level,
			       channels[i].
			       flag & IEEE80211_CHAN_W_RADAR_DETECT ?
			       " (IEEE 802.11h required)" : "",
			       (!(channels[i].flag & IEEE80211_CHAN_W_IBSS)
				|| (channels[i].
				    flag &
				    IEEE80211_CHAN_W_RADAR_DETECT)) ? "" :
			       ", IBSS",
			       channels[i].
			       flag & IEEE80211_CHAN_W_ACTIVE_SCAN ?
			       "active/passive" : "passive only");
	}

	return len;
}

static DEVICE_ATTR(channels, S_IRUSR, show_channels, NULL);

static ssize_t show_statistics(struct device *d,
			       struct device_attribute *attr, char *buf)
{
	struct ipw_priv *priv = dev_get_drvdata(d);
	u32 size = priv->hw_setting.statistics_size;
	u32 len = 0, ofs = 0;
	u8 *data = (u8 *) & priv->statistics.payload[0];
	int rc = 0;

	if (!ipw_is_alive(priv))
		return -EAGAIN;

	mutex_lock(&priv->mutex);
	rc = ipw_send_statistics_request(priv);
	mutex_unlock(&priv->mutex);

	if (rc) {
		len = sprintf(buf,
			      "Error sending statistics request: 0x%08X\n", rc);
		return len;
	}

	while (size && (PAGE_SIZE - len)) {
		len +=
		    snprint_line(&buf[len], PAGE_SIZE - len,
				 &data[ofs], min(size, 16U), ofs);
		if (PAGE_SIZE - len)
			buf[len++] = '\n';

		ofs += 16;
		size -= min(size, 16U);
	}

	return len;
}

static DEVICE_ATTR(statistics, S_IRUGO, show_statistics, NULL);

static ssize_t show_roam(struct device *d,
			 struct device_attribute *attr, char *buf)
{
	struct ipw_priv *priv = dev_get_drvdata(d);

	if (!ipw_is_alive(priv))
		return -EAGAIN;

	return sprintf(buf, "%d\n", (priv->config & CFG_NO_ROAMING) ? 0 : 1);
}

static ssize_t store_roam(struct device *d,
			  struct device_attribute *attr,
			  const char *buf, size_t count)
{
	struct ipw_priv *priv = dev_get_drvdata(d);

	switch (buf[0]) {
	case '1':
		priv->config &= ~CFG_NO_ROAMING;
		break;
	case '0':
		priv->config |= CFG_NO_ROAMING;
		break;
	}
	return count;
}

static DEVICE_ATTR(roam, S_IWUSR | S_IRUGO, show_roam, store_roam);

static ssize_t show_antenna(struct device *d,
			    struct device_attribute *attr, char *buf)
{
	struct ipw_priv *priv = dev_get_drvdata(d);

	if (!ipw_is_alive(priv))
		return -EAGAIN;

	return sprintf(buf, "%d\n", priv->antenna);
}

static ssize_t store_antenna(struct device *d,
			     struct device_attribute *attr,
			     const char *buf, size_t count)
{
	int ant;
	struct ipw_priv *priv = dev_get_drvdata(d);

	if (count == 0)
		return 0;

	if (sscanf(buf, "%1i", &ant) != 1) {
		IPW_DEBUG_INFO("not in hex or decimal form.\n");
		return count;
	}

	if ((ant >= 0) && (ant <= 2)) {
		IPW_DEBUG_INFO("Setting antenna select to %d.\n", ant);
		priv->antenna = ant;
	} else {
		IPW_DEBUG_INFO("Bad antenna select value %d.\n", ant);
	}

	return count;
}

static DEVICE_ATTR(antenna, S_IWUSR | S_IRUGO, show_antenna, store_antenna);

static ssize_t show_led(struct device *d,
			struct device_attribute *attr, char *buf)
{
	struct ipw_priv *priv = dev_get_drvdata(d);

	if (!ipw_is_alive(priv))
		return -EAGAIN;

	return sprintf(buf, "%d\n", (priv->config & CFG_NO_LED) ? 0 : 1);
}

static ssize_t store_led(struct device *d,
			 struct device_attribute *attr,
			 const char *buf, size_t count)
{
	struct ipw_priv *priv = dev_get_drvdata(d);

	if (count == 0)
		return 0;

	if (*buf == '0') {
		IPW_DEBUG_LED("Disabling LED control.\n");
		priv->config |= CFG_NO_LED;
	} else {
		IPW_DEBUG_LED("Enabling LED control.\n");
		priv->config &= ~CFG_NO_LED;
	}

	ipw_update_link_led(priv);
	ipw_update_activity_led(priv);
	ipw_update_tech_led(priv);

	return count;
}

static DEVICE_ATTR(led, S_IWUSR | S_IRUGO, show_led, store_led);

static ssize_t show_status(struct device *d,
			   struct device_attribute *attr, char *buf)
{
	struct ipw_priv *priv = (struct ipw_priv *)d->driver_data;
	if (!ipw_is_alive(priv))
		return -EAGAIN;
	return sprintf(buf, "0x%08x\n", (int)priv->status);
}

static DEVICE_ATTR(status, S_IRUGO, show_status, NULL);

static ssize_t show_cfg(struct device *d,
			struct device_attribute *attr, char *buf)
{
	struct ipw_priv *priv = (struct ipw_priv *)d->driver_data;

	if (!ipw_is_alive(priv))
		return -EAGAIN;

	return sprintf(buf, "0x%08x\n", (int)priv->config);
}

static DEVICE_ATTR(cfg, S_IRUGO, show_cfg, NULL);

static ssize_t dump_error_log(struct device *d,
			      struct device_attribute *attr,
			      const char *buf, size_t count)
{
	char *p = (char *)buf;

	if (p[0] == '1')
		ipw_dump_nic_error_log((struct ipw_priv *)d->driver_data);

	return strnlen(buf, count);
}

static DEVICE_ATTR(dump_errors, S_IWUSR, NULL, dump_error_log);

static ssize_t dump_event_log(struct device *d,
			      struct device_attribute *attr,
			      const char *buf, size_t count)
{
	char *p = (char *)buf;

	if (p[0] == '1')
		ipw_dump_nic_event_log((struct ipw_priv *)d->driver_data);

	return strnlen(buf, count);
}

static DEVICE_ATTR(dump_events, S_IWUSR, NULL, dump_event_log);

static ssize_t show_rf_kill(struct device *d,
			    struct device_attribute *attr, char *buf)
{
	/* 0 - RF kill not enabled
	   1 - SW based RF kill active (sysfs)
	   2 - HW based RF kill active
	   3 - Both HW and SW based RF kill active */
	struct ipw_priv *priv = (struct ipw_priv *)d->driver_data;
	int val = ((priv->status & STATUS_RF_KILL_SW) ? 0x1 : 0x0) |
	    ((priv->status & STATUS_RF_KILL_HW) ? 0x2 : 0x0);

	if (!ipw_is_alive(priv))
		return -EAGAIN;

	return sprintf(buf, "%i\n", val);
}

static int ipw_scan_schedule(struct ipw_priv *priv, unsigned long ms)
{
	if (priv->status & STATUS_SCAN_ABORTING) {
		IPW_DEBUG_SCAN
		    ("Scan abort in progress.  Deferring scan " "request.\n");
		priv->status |= STATUS_SCAN_PENDING;
		return 0;
	}

	queue_delayed_work(priv->workqueue,
			   &priv->request_scan, msecs_to_jiffies(ms));

	return 0;
}

static int ipw_scan_initiate(struct ipw_priv *priv, unsigned long ms)
{
	if (priv->status & STATUS_SCANNING) {
		IPW_DEBUG_SCAN("Scan already in progress.\n");
		return 0;
	}

	if (priv->status & STATUS_EXIT_PENDING) {
		IPW_DEBUG_SCAN("Aborting scan due to device shutdown\n");
		priv->status |= STATUS_SCAN_PENDING;
		return 0;
	}

	if (priv->status & STATUS_SCAN_ABORTING) {
		IPW_DEBUG_SCAN("Scan request while abort pending.  Queuing.\n");
		priv->status |= STATUS_SCAN_PENDING;
		return 0;
	}

	if (priv->status & STATUS_RF_KILL_MASK) {
		IPW_DEBUG_HC("Aborting scan due to RF Kill activation\n");
		priv->status |= STATUS_SCAN_PENDING;
		return 0;
	}

	if (!(priv->status & STATUS_READY)) {
		IPW_DEBUG_SCAN("Scan request while uninitialized.  Queuing.\n");
		priv->status |= STATUS_SCAN_PENDING;
		return 0;
	}

	IPW_DEBUG_INFO("Setting scan to on\n");
	priv->scan_bands = 2;
	priv->status |= STATUS_SCANNING;
	priv->scan_start = jiffies;
	priv->scan_pass_start = priv->scan_start;

	return ipw_scan_schedule(priv, ms);
}

static int ipw_scan_cancel(struct ipw_priv *priv)
{
	if (priv->status & STATUS_SCAN_PENDING) {
		IPW_DEBUG_SCAN("Canceling pending scan request.\n");
		priv->status &= ~STATUS_SCAN_PENDING;
		cancel_delayed_work(&priv->request_scan);
	}

	if (priv->status & STATUS_SCANNING) {
		if (!(priv->status & STATUS_SCAN_ABORTING)) {
			IPW_DEBUG_SCAN("Queuing scan abort.\n");
			priv->status |= STATUS_SCAN_ABORTING;
			queue_work(priv->workqueue, &priv->abort_scan);
		} else {
			IPW_DEBUG_SCAN("Scan abort already in progress.\n");
		}
	}

	return 0;
}

static inline unsigned long elapsed_jiffies(unsigned long start,
					    unsigned long end)
{
	if (end > start)
		return end - start;

	return end + (MAX_JIFFY_OFFSET - start);
}

static int ipw_scan_completed(struct ipw_priv *priv, int success)
{
	/* The HW is no longer scanning */
	priv->status &= ~STATUS_SCAN_HW;

	/* The scan completion notification came in, so kill that timer... */
	cancel_delayed_work(&priv->scan_check);

	IPW_DEBUG_INFO("Scan pass on %sGhz took %dms\n",
		       (priv->scan_bands == 2) ? "2.4" : "5.2",
		       jiffies_to_msecs(elapsed_jiffies
					(priv->scan_pass_start, jiffies)));

	/* Remove this scanned band from the list
	 * of pending bands to scan */
	priv->scan_bands--;

	/* If a request to abort was given, or the scan did not succeed
	 * then we reset the scan state machine and terminate,
	 * re-queuing another scan if one has been requested */
	if (priv->status & STATUS_SCAN_ABORTING) {
		IPW_DEBUG_INFO("Aborted scan completed.\n");
		priv->status &= ~STATUS_SCAN_ABORTING;
	} else {
		/* If there are more bands on this scan pass reschedule */
		if (priv->scan_bands > 0)
			goto reschedule;
	}

	IPW_DEBUG_INFO("Setting scan to off\n");

	priv->one_direct_scan = 0;
	priv->status &= ~STATUS_SCANNING;

	IPW_DEBUG_INFO("Scan took %dms\n",
		       jiffies_to_msecs(elapsed_jiffies
					(priv->scan_start, jiffies)));

	queue_work(priv->workqueue, &priv->update_link_led);

	if (priv->status & STATUS_SCAN_PENDING)
		ipw_scan_initiate(priv, 0);

	return 0;

      reschedule:
	priv->scan_pass_start = jiffies;
	ipw_scan_schedule(priv, 0);

	return 0;
}

static void ipw_radio_kill_sw(struct ipw_priv *priv, int disable_radio)
{
	if ((disable_radio ? 1 : 0) ==
	    ((priv->status & STATUS_RF_KILL_SW) ? 1 : 0))
		return;

	IPW_DEBUG_RF_KILL("Manual SW RF KILL set to: RADIO %s\n",
			  disable_radio ? "OFF" : "ON");

	if (disable_radio) {
		ipw_update_link_led(priv);
		ipw_scan_cancel(priv);
		ipw_write32(priv, CSR_UCODE_DRV_GP1_SET,
			    CSR_UCODE_SW_BIT_RFKILL);
		ipw_send_card_state(priv, CARD_STATE_CMD_DISABLE, 0);
		priv->status |= STATUS_RF_KILL_SW;
		ipw_write32(priv, CSR_UCODE_DRV_GP1_SET,
			    CSR_UCODE_DRV_GP1_BIT_CMD_BLOCKED);
		return;
	}

	ipw_write32(priv, CSR_UCODE_DRV_GP1_CLR, CSR_UCODE_SW_BIT_RFKILL);

	priv->status &= ~STATUS_RF_KILL_SW;

	if (priv->status & STATUS_RF_KILL_HW) {
		IPW_DEBUG_RF_KILL("Can not turn radio back on - "
				  "disabled by HW switch\n");
		return;
	}

	queue_work(priv->workqueue, &priv->down);

	return;
}

static ssize_t store_rf_kill(struct device *d,
			     struct device_attribute *attr,
			     const char *buf, size_t count)
{
	struct ipw_priv *priv = (struct ipw_priv *)d->driver_data;

	mutex_lock(&priv->mutex);
	ipw_radio_kill_sw(priv, buf[0] == '1');
	mutex_unlock(&priv->mutex);

	return count;
}

static DEVICE_ATTR(rf_kill, S_IWUSR | S_IRUGO, show_rf_kill, store_rf_kill);

static void ipw_print_rx_config_cmd(struct ipw_rxon_cmd *rxon)
{
	IPW_DEBUG_RADIO("RX CONFIG:\n");
	printk_buf(IPW_DL_RADIO, (u8 *) rxon, sizeof(*rxon));
	IPW_DEBUG_RADIO("u16 channel: 0x%x\n", rxon->channel);
	IPW_DEBUG_RADIO("u32 flags: 0x%08X " BIT_FMT32 "\n",
			rxon->flags, BIT_ARG32(rxon->flags));
	IPW_DEBUG_RADIO("u32 filter_flags: 0x%08x " BIT_FMT32 "\n",
			rxon->filter_flags, BIT_ARG32(rxon->filter_flags));
	IPW_DEBUG_RADIO("u8 dev_type: 0x%x\n", rxon->dev_type);
	IPW_DEBUG_RADIO("u8 ofdm_basic_rates: 0x%02x " BIT_FMT8 "\n",
			rxon->ofdm_basic_rates,
			BIT_ARG8(rxon->ofdm_basic_rates));
	IPW_DEBUG_RADIO("u8 cck_basic_rates: 0x%02x " BIT_FMT8 "\n",
			rxon->cck_basic_rates, BIT_ARG8(rxon->cck_basic_rates));
	IPW_DEBUG_RADIO("u8[6] node_addr: " MAC_FMT "\n",
			MAC_ARG(rxon->node_addr));
	IPW_DEBUG_RADIO("u8[6] bssid_addr: " MAC_FMT "\n",
			MAC_ARG(rxon->bssid_addr));
	IPW_DEBUG_RADIO("u16 assoc_id: 0x%x\n", rxon->assoc_id);
}

static void ipw_irq_handle_error(struct ipw_priv *priv)
{
	/* Set the FW error flag -- cleared on ipw_down */
	priv->status |= STATUS_FW_ERROR;

	/* Cancel currently queued command. */
	priv->status &= ~STATUS_HCMD_ACTIVE;

	if (ipw_debug_level & IPW_DL_FW_ERRORS) {
		ipw_dump_nic_error_log(priv);
		ipw_dump_nic_event_log(priv);
		ipw_print_rx_config_cmd(&priv->active_rxon);
	}

	wake_up_interruptible(&priv->wait_command_queue);

	/* Keep the restart process from trying to send host
	 * commands by clearing the INIT status bit */
	priv->status &= ~STATUS_READY;
	if (!(priv->status & STATUS_EXIT_PENDING)) {
		IPW_DEBUG(IPW_DL_INFO | IPW_DL_FW_ERRORS,
			  "Restarting adapter due to uCode error.\n");
		queue_work(priv->workqueue, &priv->down);
	}
}

static inline void ipw_rx_queue_update_write_ptr(struct ipw_priv *priv,
						 struct ipw_rx_queue *q)
{
	priv->hw_setting.rx_queue_update_wr_ptr(priv, q);
}

static int ipw_tx_queue_update_write_ptr(struct ipw_priv *priv,
					 struct ipw_tx_queue *txq, int tx_id)
{
	u32 reg = 0;
	int rc = 0;

	if (txq->need_update == 0)
		return rc;

	if (priv->status & STATUS_POWER_PMI) {
		reg = ipw_read32(priv, CSR_UCODE_DRV_GP1);

		if (reg & CSR_UCODE_DRV_GP1_BIT_MAC_SLEEP) {
			ipw_set_bit(priv, CSR_GP_CNTRL,
				    CSR_GP_CNTRL_REG_FLAG_MAC_ACCESS_REQ);
			return rc;
		}

		rc = ipw_grab_restricted_access(priv);
		if (rc)
			return rc;
		ipw_write_restricted(priv, HBUS_TARG_WRPTR,
				     txq->q.first_empty | (tx_id << 8));
		ipw_release_restricted_access(priv);
	} else {
		ipw_write32(priv, HBUS_TARG_WRPTR,
			    txq->q.first_empty | (tx_id << 8));
	}

	txq->need_update = 0;

	return rc;
}

static void ipw_rx_handle(struct ipw_priv *priv);

static void ipw_irq_tasklet(struct ipw_priv *priv)
{
	u32 inta, inta_mask, handled = 0;
	unsigned long flags;

	spin_lock_irqsave(&priv->lock, flags);

	inta = ipw_read32(priv, CSR_INT);
	inta_mask = ipw_read32(priv, CSR_INT_MASK);
	ipw_write32(priv, CSR_INT, inta);
	inta &= (CSR_INI_SET_MASK & inta_mask);

	/* Add any cached INTA values that need to be handled */
	inta |= priv->isr_inta;

	if (inta & BIT_INT_ERR) {
		IPW_ERROR("Microcode HW error detected.  Restarting.\n");

		/* tell the device to stop sending interrupts */
		ipw_disable_interrupts(priv);

		ipw_irq_handle_error(priv);

		handled |= BIT_INT_ERR;

		spin_unlock_irqrestore(&priv->lock, flags);

		return;
	}

	if (inta & BIT_INT_SWERROR) {
		IPW_ERROR("Microcode SW error detected.  Restarting 0x%X.\n",
			  inta);
		ipw_irq_handle_error(priv);
		handled |= BIT_INT_SWERROR;
	}

	if (inta & BIT_INT_WAKEUP) {
		IPW_DEBUG_ISR("Wakeup interrupt\n");
		ipw_rx_queue_update_write_ptr(priv, priv->rxq);
		ipw_tx_queue_update_write_ptr(priv, &priv->txq[0], 0);
		ipw_tx_queue_update_write_ptr(priv, &priv->txq[1], 1);
		ipw_tx_queue_update_write_ptr(priv, &priv->txq[2], 2);
		ipw_tx_queue_update_write_ptr(priv, &priv->txq[3], 3);
		ipw_tx_queue_update_write_ptr(priv, &priv->txq[4], 4);
		ipw_tx_queue_update_write_ptr(priv, &priv->txq[5], 5);

		handled |= BIT_INT_WAKEUP;
	}

	if (inta & BIT_INT_ALIVE) {
		IPW_DEBUG_ISR("Alive interrupt\n");
		handled |= BIT_INT_ALIVE;
	}

	/* handle all the justifications for the interrupt */
	if (inta & BIT_INT_RX) {
		IPW_DEBUG_ISR("Rx interrupt\n");
		ipw_rx_handle(priv);
		handled |= BIT_INT_RX;
	}

	if (inta & BIT_INT_TX) {
		IPW_DEBUG_ISR("Command completed.\n");
		ipw_write32(priv, CSR_FH_INT_STATUS, (1 << 6));
		if (!ipw_grab_restricted_access(priv)) {
			ipw_write_restricted(priv,
					     FH_TCSR_CREDIT
					     (ALM_FH_SRVC_CHNL), 0x0);
			ipw_release_restricted_access(priv);
		}

		handled |= BIT_INT_TX;
	}

	if (handled != inta) {
		IPW_ERROR("Unhandled INTA bits 0x%08x\n", inta & ~handled);
	}

	/* enable all interrupts */
	ipw_enable_interrupts(priv);

	spin_unlock_irqrestore(&priv->lock, flags);
}

/****************** 3945ABG-FUNCTIONS *********************/

static int ipw_send_rxon_assoc(struct ipw_priv *priv)
{
	int rc = 0;
	struct ipw_rx_packet *res = NULL;
	struct ipw_rxon_assoc_cmd rxon_assoc;
	struct ipw_host_cmd cmd = {
		.id = REPLY_RXON_ASSOC,
		.len = sizeof(struct ipw_rxon_assoc_cmd),
		.meta.flags = CMD_WANT_SKB,
		.data = &rxon_assoc,
	};

	rxon_assoc.flags = priv->staging_rxon.flags;
	rxon_assoc.filter_flags = priv->staging_rxon.filter_flags;
	rxon_assoc.ofdm_basic_rates = priv->staging_rxon.ofdm_basic_rates;
	rxon_assoc.cck_basic_rates = priv->staging_rxon.cck_basic_rates;
	rxon_assoc.reserved = 0;

	rc = ipw_send_cmd(priv, &cmd);
	if (rc)
		return rc;

	res = (struct ipw_rx_packet *)cmd.meta.u.skb->data;
	if (res->hdr.flags & 0x40) {
		IPW_ERROR("Bad return from REPLY_RXON_ASSOC command\n");
		rc = -EIO;
	}

	dev_kfree_skb_any(cmd.meta.u.skb);

	return rc;
}

static int ipw_add_sta_sync_callback(struct ipw_priv *priv,
				     struct ipw_cmd *cmd, struct sk_buff *skb)
{
	struct ipw_rx_packet *res = NULL;

	if (!skb) {
		IPW_ERROR("Error: Response NULL in " "REPLY_ADD_STA.\n");
		return 1;
	}

	res = (struct ipw_rx_packet *)skb->data;
	if (res->hdr.flags & 0x40) {
		IPW_ERROR("Bad return from REPLY_ADD_STA (0x%08X)\n",
			  res->hdr.flags);
		return 1;
	}

	switch (res->u.add_sta.status) {
	case ADD_STA_SUCCESS_MSK:
		break;
	default:
		break;
	}

	return 1;		/* We didn't cache the SKB; let the caller free it */
}

static int ipw_send_add_station(struct ipw_priv *priv,
				struct ipw_addsta_cmd *sta, u8 flags)
{
	struct ipw_rx_packet *res = NULL;
	int rc = 0;
	struct ipw_host_cmd cmd = {
		.id = REPLY_ADD_STA,
		.len = priv->hw_setting.add_station_size,
		.meta.flags = flags,
		.data = sta,
	};

	if (flags & CMD_ASYNC)
		cmd.meta.u.callback = ipw_add_sta_sync_callback;
	else
		cmd.meta.flags |= CMD_WANT_SKB;

	rc = ipw_send_cmd(priv, &cmd);

	if (rc || (flags & CMD_ASYNC))
		return rc;

	res = (struct ipw_rx_packet *)cmd.meta.u.skb->data;
	if (res->hdr.flags & 0x40) {
		IPW_ERROR("Bad return from REPLY_ADD_STA (0x%08X)\n",
			  res->hdr.flags);
		rc = -EIO;
	}

	if (rc == 0) {
		switch (res->u.add_sta.status) {
		case ADD_STA_SUCCESS_MSK:
			IPW_DEBUG_INFO("REPLY_ADD_STA PASSED\n");
			break;
		default:
			rc = -EIO;
			IPW_WARNING("REPLY_ADD_STA failed\n");
			break;
		}
	}
	dev_kfree_skb_any(cmd.meta.u.skb);

	return rc;
}

static int ipw_card_state_sync_callback(struct ipw_priv *priv,
					struct ipw_cmd *cmd,
					struct sk_buff *skb)
{

	return 1;
}

/*
 * CARD_STATE_CMD
 *
 * Use: Sets the internal card state to enable, disable, or halt
 *
 * When in the 'enable' state the card operates as normal.
 * When in the 'disable' state, the card enters into a low power mode.
 * When in the 'halt' state, the card is shut down and must be fully
 * restarted to come back on.
 */
static int ipw_send_card_state(struct ipw_priv *priv, u32 flags, u8 meta_flag)
{
	struct ipw_host_cmd cmd = {
		.id = REPLY_CARD_STATE_CMD,
		.len = sizeof(u32),
		.data = &flags,
		.meta.flags = meta_flag,
	};

	if (meta_flag & CMD_ASYNC)
		cmd.meta.u.callback = ipw_card_state_sync_callback;

	return ipw_send_cmd(priv, &cmd);
}

/**
* rate  duration
**/
#define MMAC_SCHED_2_4GHZ_SIFS               10
#define MMAC_SCHED_5_2GHZ_SIFS               16
#define MMAC_SCHED_CCK_PHY_OVERHEAD_SHORT    96
#define MMAC_SCHED_CCK_PHY_OVERHEAD_LONG    192
#define MMAC_SCHED_OFDM_SYMBOL_TIME           4
#define MMAC_SCHED_OFDM_PHY_OVERHEAD         20
#define MMAC_SCHED_UCODE_OVERHEAD             8

/* NOTE:
 * Do not reorder the rate_table_info list --
 *
 * OFDM rates must come before CCK rates in order for routines like
 * ipw_send_txpower_cmd to be compatible with the uCode
 */
static struct ipw_rate_info rate_table_info[] = {
/*  OFDM rate info   */
	{13, 6 * 2, 0, 24, 44, 52, 44, 228},	/*   6mbps */
	{15, 9 * 2, 1, 36, 36, 44, 36, 160},	/*   9mbps */
	{5, 12 * 2, 2, 48, 32, 36, 32, 124},	/*  12mbps */
	{7, 18 * 2, 3, 72, 28, 32, 28, 92},	/*  18mbps */
	{9, 24 * 2, 4, 96, 28, 32, 28, 72},	/*  24mbps */
	{11, 36 * 2, 5, 144, 24, 28, 24, 56},	/*  36mbps */
	{1, 48 * 2, 6, 192, 24, 24, 24, 48},	/*  48mbps */
	{3, 54 * 2, 7, 216, 24, 24, 24, 44},	/*  54mbps */
/*  CCK rate info   */
	{10, 2, 8, 0, 112, 160, 112, 1216},	/*   1mbps */
	{20, 4, 9, 0, 56, 80, 56, 608},	/*   2mbps */
	{55, 11, 10, 0, 21, 29, 21, 222},	/* 5.5mbps */
	{110, 22, 11, 0, 11, 15, 11, 111},	/*  11mbps */
};

static int ipw_rate_plcp2index(u8 x)
{
	int i;
	for (i = 0; i < ARRAY_SIZE(rate_table_info); i++) {
		if (rate_table_info[i].rate_plcp == x)
			return i;
	}
	return -1;
}

static int ipw_rate_scale2index(int x)
{
	int i;
	for (i = 0; i < ARRAY_SIZE(rate_table_info); i++) {
		if (rate_table_info[i].rate_scale_index == x)
			return i;
	}
	return -1;
}

static u8 ipw_rate_scale2plcp(int x)
{
	int i;
	for (i = 0; i < ARRAY_SIZE(rate_table_info); i++) {
		if (rate_table_info[i].rate_scale_index == x)
			return rate_table_info[i].rate_plcp;
	}
	return IPW_INVALID_RATE;
}

static int ipw_rate_plcp2scale(u8 x)
{
	int i;
	for (i = 0; i < ARRAY_SIZE(rate_table_info); i++) {
		if (rate_table_info[i].rate_plcp == x)
			return rate_table_info[i].rate_scale_index;
	}
	return -1;
}

#if 0
static int ipw_rate_ieee2index(u8 x)
{
	int i;
	for (i = 0; i < ARRAY_SIZE(rate_table_info); i++) {
		if (rate_table_info[i].rate_ieee == x)
			return i;
	}
	return -1;
}
#endif

static u8 ipw_rate_index2ieee(int x)
{

	if (x < ARRAY_SIZE(rate_table_info))
		return rate_table_info[x].rate_ieee;

	return IPW_INVALID_RATE;
}

static u8 ipw_rate_index2plcp(int x)
{

	if (x < ARRAY_SIZE(rate_table_info))
		return rate_table_info[x].rate_plcp;

	return IPW_INVALID_RATE;
}

static u8 ipw_rate_get_lowest_plcp(int rate_mask)
{
	int start, i;

	if (rate_mask & 0xF)
		start = 0;
	else if (rate_mask & 0xF0)
		start = 4;
	else
		start = 8;

	for (i = start; i < 12; i++) {
		if ((rate_mask & (1 << i)) != 0)
			return rate_table_info[i].rate_plcp;
	}
	return IPW_INVALID_RATE;
}

/**
* RXON functions
**/
static u8 BROADCAST_ADDR[ETH_ALEN] = { 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF };

#if 0
/*
  validate rxon command before issuing the command to the card.
*/
static int ipw_check_rx_config_cmd(struct ipw_rxon_cmd *rxon)
{
	int error = 0;
	int counter = 1;

	if (rxon->flags & RXON_FLG_BAND_24G_MSK) {
		error |= (rxon->flags & RXON_FLG_TGJ_NARROW_BAND_MSK);
		error |= (rxon->flags & RXON_FLG_RADAR_DETECT_MSK);
		if (error)
			IPW_WARNING("check 24G fields %d | %d\n",
				    counter++, error);
	} else {
		error |= ((rxon->flags & RXON_FLG_SHORT_SLOT_MSK) !=
			  RXON_FLG_SHORT_SLOT_MSK);
		if (error)
			IPW_WARNING("check 52 fields %d | %d\n",
				    counter++, error);
		error |= (rxon->flags & RXON_FLG_CCK_MSK);
		if (error)
			IPW_WARNING("check 52 CCK %d | %d\n", counter++, error);

	}
	error |= (rxon->node_addr[0] | rxon->bssid_addr[0]) & 0x1;
	if (error)
		IPW_WARNING("check mac addr %d | %d\n", counter++, error);

	// make sure basic rates 6Mbps and 1Mbps are supported
	error |= (((rxon->ofdm_basic_rates & R_6M_MSK) == 0) &&
		  ((rxon->cck_basic_rates & R_1M_MSK) == 0));

	if (error)
		IPW_WARNING("check basic rate %d | %d\n", counter++, error);
	error |= (rxon->assoc_id > 2007);
	if (error)
		IPW_WARNING("check assoc id %d | %d\n", counter++, error);

	error |=
	    ((rxon->
	      flags & (RXON_FLG_CCK_MSK | RXON_FLG_SHORT_SLOT_MSK)) ==
	     (RXON_FLG_CCK_MSK | RXON_FLG_SHORT_SLOT_MSK));

	if (error)
		IPW_WARNING("check CCK and short slot %d | %d\n",
			    counter++, error);
	error |= ((rxon->flags & (RXON_FLG_CCK_MSK | RXON_FLG_AUTO_DETECT_MSK))
		  == (RXON_FLG_CCK_MSK | RXON_FLG_AUTO_DETECT_MSK));

	if (error)
		IPW_WARNING("check CCK & auto detect %d | %d\n",
			    counter++, error);
	error |=
	    ((rxon->
	      flags & (RXON_FLG_AUTO_DETECT_MSK |
		       RXON_FLG_TGG_PROTECT_MSK)) == RXON_FLG_TGG_PROTECT_MSK);

	if (error)
		IPW_WARNING("check TGG %d | %d\n", counter++, error);
	if (rxon->flags & RXON_FLG_BAND_24G_MSK)
		error |= ((rxon->channel < MIN_B_CHANNELS) ||
			  (rxon->channel > MAX_B_CHANNELS));
	else
		error |= ((rxon->channel < MIN_A_CHANNELS) ||
			  (rxon->channel > MAX_A_CHANNELS));
	if (error)
		IPW_WARNING("check channel number channel %d counter %d | %d\n",
			    rxon->channel, counter++, error);
	if ((rxon->flags & RXON_FLG_DIS_DIV_MSK))
		error |=
		    ((rxon->
		      flags & (RXON_FLG_ANT_B_MSK | RXON_FLG_ANT_A_MSK)) == 0);

	if (error)
		IPW_WARNING("check antenna %d %d\n", counter++, error);
	if (error)
		IPW_WARNING("Tuning to channel %d\n", rxon->channel);
	if (error) {
		IPW_ERROR
		    ("Error not a valid ipw_rxon_assoc_cmd field values\n");
		return -1;
	}

	return 0;
}
#endif

/* Get antenna flags for RxOn command. */
static int ipw_get_antenna_flags(struct ipw_priv *priv)
{
	switch (priv->antenna) {
	case 0:		/* "diversity", NIC selects best antenna by itself */
		return 0;

	case 1:		/* force Main antenna */
		if (priv->eeprom.antenna_switch_type)
			return RXON_FLG_DIS_DIV_MSK | RXON_FLG_ANT_B_MSK;
		return RXON_FLG_DIS_DIV_MSK | RXON_FLG_ANT_A_MSK;

	case 2:		/* force Aux antenna */
		if (priv->eeprom.antenna_switch_type)
			return RXON_FLG_DIS_DIV_MSK | RXON_FLG_ANT_A_MSK;
		return RXON_FLG_DIS_DIV_MSK | RXON_FLG_ANT_B_MSK;
	}

	/* bad antenna selector value */
	IPW_ERROR("Bad antenna selector value (0x%x)\n", priv->antenna);
	return 0;		/* "diversity" is default if error */
}

static inline int is_channel_valid(const struct ipw_channel_info *ch_info)
{
	if (ch_info == NULL)
		return 0;
	return (ch_info->flags & IPW_CHANNEL_VALID) ? 1 : 0;
}

static inline int is_channel_narrow(const struct ipw_channel_info *ch_info)
{
	return (ch_info->flags & IPW_CHANNEL_NARROW) ? 1 : 0;
}

static inline int is_channel_radar(const struct ipw_channel_info *ch_info)
{
	return (ch_info->flags & IPW_CHANNEL_RADAR) ? 1 : 0;
}

static const struct ipw_channel_info *find_channel(struct ipw_priv *priv,
						   u8 channel)
{
	int i;

	for (i = 0; i < priv->channel_count; i++) {
		if (priv->channel_info[i].channel == channel)
			return &priv->channel_info[i];
	}

	return NULL;
}

static inline u8 is_channel_a_band(const struct ipw_channel_info *ch_info)
{
	return (ch_info->phymode == MODE_IEEE80211A) ? 1 : 0;
}

static inline u8 is_channel_bg_band(const struct ipw_channel_info *ch_info)
{
	return (ch_info->phymode == MODE_IEEE80211B ||
		ch_info->phymode == MODE_IEEE80211G) ? 1 : 0;
}

static inline int is_channel_passive(const struct ipw_channel_info *ch)
{
	return (!(ch->flags & IPW_CHANNEL_ACTIVE)) ? 1 : 0;
}

/*
 * return index delta into power gain settings table
 */
static int reg_adjust_power_by_temp(int new_reading, int old_reading)
{
	return (new_reading - old_reading) * (-11) / 100;
}

/* Keep temperature in sane range */
static inline int reg_temp_out_of_range(int temperature)
{
	return (((temperature < -260) || (temperature > 25)) ? 1 : 0);
}

/* get current temperature by reading from NIC via driver. */
static int reg_txpower_get_temperature(struct ipw_priv *priv)
{
	int temperature;

	temperature = ipw_get_temperature(priv);

	/* driver's okay range is -260 to +25.
	 *   human readable okay range is 0 to +285 */
	IPW_DEBUG_INFO("Temperature: %d\n", temperature + 260);

	/* handle insane temp reading */
	if (reg_temp_out_of_range(temperature)) {
		IPW_ERROR("Error bad temperature value  %d\n", temperature);

		/* if really really hot(?),
		 *   substitute the 3rd band/group's temp measured at factory */
		if (priv->last_temperature > 100)
			temperature = priv->eeprom.groups[2].temperature;
		else		/* else use most recent "sane" value from driver */
			temperature = priv->last_temperature;
	}

	return temperature;	/* raw, not "human readable" */
}

/* Adjust Txpower only if temperature variance is greater than threshold.
 *
 * Both are lower than older versions' 9 degrees */
#define IPW_TEMPERATURE_LIMIT_TIMER   6

/* reads temperature and determines if new calibration is needed.
 * records new temperature in tx_mgr->curr_temperature.
 * replaces tx_mgr->last_temperature *only* if calib needed
 *    (assumes caller will actually do the calibration!). */
static int is_temp_calib_needed(struct ipw_priv *priv)
{
	int temp_diff;

	priv->curr_temperature = reg_txpower_get_temperature(priv);
	temp_diff = priv->curr_temperature - priv->last_temperature;

	/* get absolute value */
	if (temp_diff < 0) {
		IPW_DEBUG_POWER("Getting cooler, delta %d,\n", temp_diff);
		temp_diff = -temp_diff;
	} else if (temp_diff == 0)
		IPW_DEBUG_POWER("Same temp,\n");
	else
		IPW_DEBUG_POWER("Getting warmer, delta %d,\n", temp_diff);

	/* if we don't need calibration, *don't* update last_temperature */
	if (temp_diff < IPW_TEMPERATURE_LIMIT_TIMER) {
		IPW_DEBUG_POWER("Timed thermal calib not needed\n");
		return 0;
	}

	IPW_DEBUG_POWER("Timed thermal calib needed\n");

	/* assume that caller will actually do calib ...
	 *   update the "last temperature" value */
	priv->last_temperature = priv->curr_temperature;
	return 1;
}

#define IPW_MAX_GAIN_ENTRIES 78
#define IPW_CCK_FROM_OFDM_POWER_DIFF  -5
#define IPW_CCK_FROM_OFDM_INDEX_DIFF (10)

/* radio and DSP power table, each step is 1/2 dB.
 * 1st number is for RF analog gain, 2nd number is for DSP pre-DAC gain. */
static struct ipw_tx_power power_gain_table[2][IPW_MAX_GAIN_ENTRIES] = {
	{
	 {251, 127},		/* 2.4 GHz, highest power */
	 {251, 127},
	 {251, 127},
	 {251, 127},
	 {251, 125},
	 {251, 110},
	 {251, 105},
	 {251, 98},
	 {187, 125},
	 {187, 115},
	 {187, 108},
	 {187, 99},
	 {243, 119},
	 {243, 111},
	 {243, 105},
	 {243, 97},
	 {243, 92},
	 {211, 106},
	 {211, 100},
	 {179, 120},
	 {179, 113},
	 {179, 107},
	 {147, 125},
	 {147, 119},
	 {147, 112},
	 {147, 106},
	 {147, 101},
	 {147, 97},
	 {147, 91},
	 {115, 107},
	 {235, 121},
	 {235, 115},
	 {235, 109},
	 {203, 127},
	 {203, 121},
	 {203, 115},
	 {203, 108},
	 {203, 102},
	 {203, 96},
	 {203, 92},
	 {171, 110},
	 {171, 104},
	 {171, 98},
	 {139, 116},
	 {227, 125},
	 {227, 119},
	 {227, 113},
	 {227, 107},
	 {227, 101},
	 {227, 96},
	 {195, 113},
	 {195, 106},
	 {195, 102},
	 {195, 95},
	 {163, 113},
	 {163, 106},
	 {163, 102},
	 {163, 95},
	 {131, 113},
	 {131, 106},
	 {131, 102},
	 {131, 95},
	 {99, 113},
	 {99, 106},
	 {99, 102},
	 {99, 95},
	 {67, 113},
	 {67, 106},
	 {67, 102},
	 {67, 95},
	 {35, 113},
	 {35, 106},
	 {35, 102},
	 {35, 95},
	 {3, 113},
	 {3, 106},
	 {3, 102},
	 {3, 95}},		/* 2.4 GHz, lowest power */
	{
	 {251, 127},		/* 5.x GHz, highest power */
	 {251, 120},
	 {251, 114},
	 {219, 119},
	 {219, 101},
	 {187, 113},
	 {187, 102},
	 {155, 114},
	 {155, 103},
	 {123, 117},
	 {123, 107},
	 {123, 99},
	 {123, 92},
	 {91, 108},
	 {59, 125},
	 {59, 118},
	 {59, 109},
	 {59, 102},
	 {59, 96},
	 {59, 90},
	 {27, 104},
	 {27, 98},
	 {27, 92},
	 {115, 118},
	 {115, 111},
	 {115, 104},
	 {83, 126},
	 {83, 121},
	 {83, 113},
	 {83, 105},
	 {83, 99},
	 {51, 118},
	 {51, 111},
	 {51, 104},
	 {51, 98},
	 {19, 116},
	 {19, 109},
	 {19, 102},
	 {19, 98},
	 {19, 93},
	 {171, 113},
	 {171, 107},
	 {171, 99},
	 {139, 120},
	 {139, 113},
	 {139, 107},
	 {139, 99},
	 {107, 120},
	 {107, 113},
	 {107, 107},
	 {107, 99},
	 {75, 120},
	 {75, 113},
	 {75, 107},
	 {75, 99},
	 {43, 120},
	 {43, 113},
	 {43, 107},
	 {43, 99},
	 {11, 120},
	 {11, 113},
	 {11, 107},
	 {11, 99},
	 {131, 107},
	 {131, 99},
	 {99, 120},
	 {99, 113},
	 {99, 107},
	 {99, 99},
	 {67, 120},
	 {67, 113},
	 {67, 107},
	 {67, 99},
	 {35, 120},
	 {35, 113},
	 {35, 107},
	 {35, 99},
	 {3, 120}}		/* 5.x GHz, lowest power */
};

/* keep gain index within range of the gain table */
static inline u8 reg_fix_power_index(int index)
{
	if (index < 0)
		return 0;
	if (index >= IPW_MAX_GAIN_ENTRIES)
		return IPW_MAX_GAIN_ENTRIES - 1;
	return (u8) index;
}

/* Kick off thermal recalibration check every 60 seconds */
#define REG_RECALIB_PERIOD (60)

/*
 * Set (in our channel info database) the direct scan Tx power for 1 Mbit (CCK)
 *    or 6 Mbit (OFDM) rates.
 */
static void reg_set_scan_power(struct ipw_priv *priv, u32 scan_tbl_index,
			       s32 rate_index, const s8 * clip_pwrs,
			       struct ipw_channel_info *ch_info, int band_index)
{
	struct ipw_scan_power_info *scan_power_info;
	s8 power;
	u8 power_index;

	scan_power_info = &ch_info->scan_pwr_info[scan_tbl_index];

	/* use this channel group's 6Mbit clipping/saturation pwr,
	 *   but cap at regulatory scan power restriction (set during init
	 *   based on eeprom channel data) for this channel.  */
	power = min(ch_info->scan_power, clip_pwrs[RATE_SCALE_6M_INDEX]);

	/* further limit to user's max power preference.
	 * FIXME:  Other spectrum management power limitations do not
	 *   seem to apply?? */
	power = min(power, priv->user_txpower_limit);
	scan_power_info->requested_power = power;

	/* find difference between new scan *power* and current "normal"
	 *   Tx *power* for 6Mb.  Use this difference (x2) to adjust the
	 *   current "normal" temperature-compensated Tx power *index* for
	 *   this rate (1Mb or 6Mb) to yield new temp-compensated scan power
	 *   *index*. */
	power_index = ch_info->power_info[rate_index].power_table_index
	    - (power - ch_info->power_info
	       [RATE_SCALE_6M_INDEX].requested_power) * 2;

#if 0
	IPW_DEBUG_POWER("chnl %d scan power index %d\n",
			ch_info->channel, power_index);
#endif

	/* store reference index that we use when adjusting *all* scan
	 *   powers.  So we can accommodate user (all channel) or spectrum
	 *   management (single channel) power changes "between" temperature
	 *   feedback compensation procedures.
	 * don't force fit this reference index into gain table; it may be a
	 *   negative number.  This will help avoid errors when we're at
	 *   the lower bounds (highest gains, for warmest temperatures)
	 *   of the table. */

	/* don't exceed table bounds for "real" setting */
	power_index = reg_fix_power_index(power_index);

	scan_power_info->power_table_index = power_index;
	scan_power_info->tpc.tx_gain =
	    power_gain_table[band_index][power_index].tx_gain;
	scan_power_info->tpc.dsp_atten =
	    power_gain_table[band_index][power_index].dsp_atten;
}

static struct ipw_channel_info *ipw_get_channel_info(struct ipw_priv *priv,
						     int phymode, int channel)
{
	int i;

	switch (phymode) {
	case MODE_IEEE80211A:
		for (i = 14; i < priv->channel_count; i++) {
			if (priv->channel_info[i].channel == channel)
				return &priv->channel_info[i];
		}
		break;

	case MODE_IEEE80211B:
	case MODE_IEEE80211G:
		if (channel >= 1 && channel <= 14)
			return &priv->channel_info[channel - 1];
		break;

	}

	return NULL;
}

/* fill in Tx Power command with gain settings for all rates for the current
 * channel, using values from channel info struct, and send to NIC */
static int ipw_reg_send_txpower(struct ipw_priv *priv)
{
	int rate_idx;
	struct ipw_channel_info *ch_info = NULL;
	struct ipw_txpowertable_cmd txpower = {
		.channel = priv->active_conf.channel,
	};

	txpower.band = (priv->active_conf.phymode == MODE_IEEE80211A) ? 0 : 1;
	ch_info = ipw_get_channel_info(priv,
				       priv->active_conf.phymode,
				       priv->active_conf.channel);
	if (!ch_info) {
		IPW_ERROR
		    ("Failed to get channel info for channel %d [%d]\n",
		     priv->active_conf.channel, priv->active_conf.phymode);
		return -EINVAL;
	}

	if (!is_channel_valid(ch_info)) {
		IPW_DEBUG_POWER("Not calling TX_PWR_TABLE_CMD on "
				"non-Tx channel.\n");
		return 0;
	}

	/* fill cmd with power settings for all rates for current channel */
	for (rate_idx = 0; rate_idx < IPW_MAX_RATES; rate_idx++) {
		txpower.power[rate_idx].tpc = ch_info->power_info[rate_idx].tpc;
		txpower.power[rate_idx].rate = ipw_rate_index2plcp(rate_idx);

		IPW_DEBUG_POWER("ch %d:%d rf %d dsp %3d rate code 0x%02x\n",
				txpower.channel,
				txpower.band,
				txpower.power[rate_idx].tpc.tx_gain,
				txpower.power[rate_idx].tpc.dsp_atten,
				txpower.power[rate_idx].rate);
	}

	return ipw_send_cmd_pdu(priv, REPLY_TX_PWR_TABLE_CMD,
				sizeof(struct ipw_txpowertable_cmd), &txpower);

}

/* Replace requested_power and base_power_index ch_info fields for one channel.
 * Called if user or spectrum management changes power preferences.
 * Takes into account h/w and modulation limitations (clip power).
 *
 * This does *not* send anything to NIC, just sets up ch_info for one channel.
 *
 * NOTE:reg_compensate_for_temperature_dif() *must* be run after this to
 *	properly fill out the scan powers, and actual h/w gain settings,
 *	and send changes to NIC
 */
static int reg_set_new_power(struct ipw_priv *priv,
			     struct ipw_channel_info *ch_info)
{
	struct ipw_channel_power_info *power_info;
	int power_changed = 0;
	int i;
	const s8 *clip_pwrs;
	int power;

	/* Get this chnlgrp's rate-to-max/clip-powers table */
	clip_pwrs = priv->clip_groups[ch_info->group_index].clip_powers;

	/* Get this channel's rate-to-current-power settings table */
	power_info = ch_info->power_info;

	/* update OFDM Txpower settings */
	for (i = 0; i < IPW_OFDM_RATES; i++, ++power_info) {
		int delta_idx;

		/* limit new power to be no more than h/w capability */
		power = min(ch_info->curr_txpow, clip_pwrs[i]);
		if (power == power_info->requested_power)
			continue;

		/* find difference between old and new requested powers,
		 *    update base (non-temp-compensated) power index */
		delta_idx = (power - power_info->requested_power) * 2;
		power_info->base_power_index -= delta_idx;

		/* save new requested power value */
		power_info->requested_power = power;

		power_changed = 1;
	}

	/* update CCK Txpower settings, based on OFDM 12M setting ...
	 *    ... all CCK power settings for a given channel are the *same*. */
	if (power_changed) {
		power =
		    ch_info->power_info[RATE_SCALE_12M_INDEX].
		    requested_power + IPW_CCK_FROM_OFDM_POWER_DIFF;

		/* do all CCK rates' ipw_channel_power_info structures */
		for (i = IPW_OFDM_RATES; i < IPW_MAX_RATES; i++) {
			power_info->requested_power = power;
			power_info->base_power_index =
			    ch_info->power_info[RATE_SCALE_12M_INDEX].
			    base_power_index + IPW_CCK_FROM_OFDM_INDEX_DIFF;
			++power_info;
		}
	}

	return 0;
}

/* selects and returns new power limit for channel,
 *   which may be less (but not more) than requested,
 *   based strictly on regulatory (eeprom and spectrum mgt) limitations
 *   (no consideration for h/w clipping limitations). */
static int reg_get_channel_txpower_limit(struct ipw_channel_info *ch_info)
{
	s8 max_power = 0;

#if 0
	/* if we're using TGd limits, use lower of TGd or EEPROM */
	if (ch_info->tgd_data.max_power != 0)
		max_power = min(ch_info->tgd_data.max_power,
				ch_info->eeprom.max_power_avg);

	/* else just use EEPROM limits */
	else
#endif
		max_power = ch_info->eeprom.max_power_avg;
	return min(max_power, ch_info->max_power_avg);
}

/* Compensate txpower settings of *all* channels for temperature.
 * This only accounts for the difference between current temperature
 *   and the factory calibration temperatures, and bases the new settings
 *   on the channel's base_power_index.
 * If RxOn is "associated", this sends the new Txpower to NIC! */
static int reg_txpower_compensate_for_temperature_dif(struct ipw_priv *priv)
{
	struct ipw_channel_info *ch_info = NULL;
	int delta_index;
	const s8 *clip_pwrs;	/* array of h/w max power levels for each rate */
	u8 a_band;
	u8 rate_index;
	u8 scan_tbl_index;
	u8 i;
	int ref_temp;
	int temperature = priv->curr_temperature;

	/* set up new Tx power info for each and every channel, 2.4 and 5.x */
	for (i = 0; i < priv->channel_count; i++) {
		ch_info = &priv->channel_info[i];
		a_band = is_channel_a_band(ch_info);

		/* Get this chnlgrp's factory calibration temperature */
		ref_temp = priv->eeprom.groups[ch_info->group_index].
		    temperature;

		/* get power index adjustment based on curr and factory
		 * temps */
		delta_index = reg_adjust_power_by_temp(temperature, ref_temp);

		/* set tx power value for all rates, OFDM and CCK */
		for (rate_index = 0; rate_index < IPW_MAX_RATES; rate_index++) {
			int power_idx =
			    ch_info->power_info[rate_index].base_power_index;

			/* temperature compensate */
			power_idx += delta_index;

			/* stay within table range */
			power_idx = reg_fix_power_index(power_idx);
			ch_info->power_info[rate_index].
			    power_table_index = (u8) power_idx;
			ch_info->power_info[rate_index].tpc =
			    power_gain_table[a_band][power_idx];
		}

		/* Get this chnlgrp's rate-to-max/clip-powers table */
		clip_pwrs = priv->clip_groups[ch_info->group_index].clip_powers;

		/* set scan tx power, 1Mbit for CCK, 6Mbit for OFDM */
		for (scan_tbl_index = 0;
		     scan_tbl_index < IPW_NUM_SCAN_RATES; scan_tbl_index++) {
			s32 actual_index = (scan_tbl_index == 0) ?
			    RATE_SCALE_1M_INDEX : RATE_SCALE_6M_INDEX;
			reg_set_scan_power(priv, scan_tbl_index,
					   actual_index, clip_pwrs,
					   ch_info, a_band);
		}
	}

	/* send Txpower command for current channel to ucode */
	return ipw_reg_send_txpower(priv);
}

static int reg_set_txpower(struct ipw_priv *priv, s8 power)
{
	struct ipw_channel_info *ch_info;
	s8 max_power;
	u8 channel;
	u8 a_band;
	u8 i;

	if (priv->user_txpower_limit == power) {
		IPW_DEBUG_POWER("Requested Tx power same as current "
				"limit: %ddBm.\n", power);
		return 0;
	}

	IPW_DEBUG_POWER("Setting upper limit clamp to %ddBm.\n", power);
	priv->user_txpower_limit = power;

	/* set up new Tx powers for each and every channel, 2.4 and 5.x */

	for (i = 0; i < priv->channel_count; i++) {
		ch_info = &priv->channel_info[i];
		a_band = is_channel_a_band(ch_info);
		channel = ch_info->channel;

		/* find minimum power of all user and regulatory constraints
		 *    (does not consider h/w clipping limitations) */
		max_power = reg_get_channel_txpower_limit(ch_info);
		max_power = min(power, max_power);
		if (max_power != ch_info->curr_txpow) {
			ch_info->curr_txpow = max_power;

			/* this considers the h/w clipping limitations */
			reg_set_new_power(priv, ch_info);
		}
	}

	/* update txpower settings for all channels,
	 *   send to NIC if associated. */
	is_temp_calib_needed(priv);
	reg_txpower_compensate_for_temperature_dif(priv);

	return 0;
}

/* called whenever it's time to check our temperature.
 * -- reset periodic timer
 * -- see if temp has changed enough to warrant re-calibration ... if so:
 *     -- correct coeffs for temp (can reset temp timer)
 *     -- save this temp as "last",
 *     -- send new set of gain settings to NIC
 * NOTE:  This should continue working, even when we're not associated,
 *   so we can keep our internal table of scan powers current. */
static void reg_txpower_periodic(struct ipw_priv *priv)
{
	/* This will kick in the "brute force"
	 *   reg_txpower_compensate_for_temperature_dif() below */
	if (!is_temp_calib_needed(priv))
		goto reschedule;

	/* Set up a new set of temp-adjusted TxPowers, send to NIC.
	 * This is based *only* on current temperature,
	 * ignoring any previous power measurements */
	reg_txpower_compensate_for_temperature_dif(priv);

      reschedule:
	queue_delayed_work(priv->workqueue,
			   &priv->thermal_periodic, REG_RECALIB_PERIOD * HZ);
}

static void ipw_bg_reg_txpower_periodic(struct work_struct *work)
{
	struct ipw_priv *priv =
	    container_of(work, struct ipw_priv, thermal_periodic.work);

	if (priv->status & STATUS_EXIT_PENDING)
		return;

	mutex_lock(&priv->mutex);
	reg_txpower_periodic(priv);
	mutex_unlock(&priv->mutex);
}

/* find the *channel-group* index (0-4) for the channel.
 *  ... used when initializing channel-info structs.
 * NOTE:  These channel groups do *NOT* match the bands above!
 *   These channel groups are based on factory-tested channels;
 *   on A-band, EEPROM's "group frequency" entries represent the top channel
 *   in each group 1-4.  Group 5 All B/G channels are in group 0.  */
static u16 reg_get_chnl_grp_index(struct ipw_priv *priv,
				  const struct ipw_channel_info *ch_info)
{
	struct ipw_eeprom_txpower_group *ch_grp = &priv->eeprom.groups[0];
	u8 group;
	u16 group_index = 0;	/* based on factory calib frequencies */
	u8 grp_channel;

	/* Find the group index for the channel ... don't use index 1(?) */
	if (is_channel_a_band(ch_info)) {
		for (group = 1; group < 5; group++) {
			grp_channel = ch_grp[group].group_channel;
			if (ch_info->channel <= grp_channel) {
				group_index = group;
				break;
			}
		}
		/* group 4 has a few channels *above* its factory cal freq */
		if (group == 5)
			group_index = 4;
	} else
		group_index = 0;	/* 2.4 GHz, group 0 */

	IPW_DEBUG_POWER("Chnl %d mapped to grp %d\n", ch_info->channel,
			group_index);
	return group_index;
}

/*
 * Interpolate to get nominal (i.e. at factory calibration temperature) index
 *   into radio/DSP gain settings table for requested power.
 */
static int reg_get_matched_power_index(struct ipw_priv *priv,
				       s8 requested_power,
				       s32 setting_index, s32 * new_index)
{
	const struct ipw_eeprom_txpower_group *chnl_grp = NULL;
	s32 index0, index1;
	s32 rPower = 2 * requested_power;
	s32 i;
	const struct ipw_eeprom_txpower_sample *samples;
	s32 gains0, gains1;
	s32 res;
	s32 denominator;

	chnl_grp = &priv->eeprom.groups[setting_index];
	samples = chnl_grp->samples;
	for (i = 0; i < 5; i++) {
		if (rPower == samples[i].power) {
			*new_index = samples[i].gain_index;
			return 0;
		}
	}

	if (rPower > samples[1].power) {
		index0 = 0;
		index1 = 1;
	} else if (rPower > samples[2].power) {
		index0 = 1;
		index1 = 2;
	} else if (rPower > samples[3].power) {
		index0 = 2;
		index1 = 3;
	} else {
		index0 = 3;
		index1 = 4;
	}

	denominator = (s32) samples[index1].power - (s32) samples[index0].power;
	if (denominator == 0)
		return -EINVAL;
	gains0 = (s32) samples[index0].gain_index * (1 << 19);
	gains1 = (s32) samples[index1].gain_index * (1 << 19);
	res = gains0 + (gains1 - gains0) *
	    ((s32) rPower - (s32) samples[index0].power) / denominator +
	    (1 << 18);
	*new_index = res >> 19;
	return 0;
}

static void reg_init_channel_groups(struct ipw_priv *priv)
{
	u32 i;
	s32 rate_index;
	const struct ipw_eeprom_txpower_group *group;

	IPW_DEBUG_POWER("Initializing factory calib info from EEPROM\n");

	for (i = 0; i < IPW_NUM_TX_CALIB_GROUPS; i++) {
		s8 *clip_pwrs;	/* table of power levels for each rate */
		s8 satur_pwr;	/* saturation power for each chnl group */
		group = &priv->eeprom.groups[i];

		/* sanity check on factory saturation power value */
		if (group->saturation_power < 40) {
			IPW_WARNING("Error: saturation power is %d, "
				    "less than minimum expected 40\n",
				    group->saturation_power);
			return;
		}

		/*
		 * Derive requested power levels for each rate, based on
		 *   hardware capabilities (saturation power for band).
		 * Basic value is 3dB down from saturation, with further
		 *   power reductions for highest 3 data rates.  These
		 *   backoffs provide headroom for high rate modulation
		 *   power peaks, without too much distortion (clipping).
		 */
		/* we'll fill in this array with h/w max power levels */
		clip_pwrs = (s8 *) priv->clip_groups[i].clip_powers;

		/* divide factory saturation power by 2 to find -3dB level */
		satur_pwr = (s8) (group->saturation_power >> 1);

		/* fill in channel group's nominal powers for each rate */
		for (rate_index = 0;
		     rate_index < IPW_MAX_RATES; rate_index++, clip_pwrs++) {
			switch (rate_index) {
			case RATE_SCALE_36M_INDEX:
				if (i == 0)	/* B/G */
					*clip_pwrs = satur_pwr;
				else	/* A */
					*clip_pwrs = satur_pwr - 5;
				break;
			case RATE_SCALE_48M_INDEX:
				if (i == 0)
					*clip_pwrs = satur_pwr - 7;
				else
					*clip_pwrs = satur_pwr - 10;
				break;
			case RATE_SCALE_54M_INDEX:
				if (i == 0)
					*clip_pwrs = satur_pwr - 9;
				else
					*clip_pwrs = satur_pwr - 12;
				break;
			default:
				*clip_pwrs = satur_pwr;
				break;
			}
		}
	}
}

/*
 * Second pass (during init) to set up priv->channel_info
 *
 * Set up Tx-power settings in our channel info database for each VALID
 * (for this geo/SKU) channel, at all Tx data rates, based on eeprom values
 * and current temperature.
 *
 * Since this is based on current temperature (at init time), these values may
 * not be valid for very long, but it gives us a starting/default point,
 * and allows us to active (i.e. using Tx) scan.
 *
 * This does *not* write values to NIC, just sets up our internal table.
 */
static int reg_txpower_set_from_eeprom(struct ipw_priv *priv)
{
	struct ipw_channel_info *ch_info = NULL;
	struct ipw_channel_power_info *pwr_info;
	int delta_index;
	u8 rate_index;
	u8 scan_tbl_index;
	const s8 *clip_pwrs;	/* array of power levels for each rate */
	u8 gain, dsp_atten;
	s8 power;
	u8 pwr_index, base_pwr_index, a_band;
	u8 i;
	int temperature;

	/* save temperature reference,
	 *   so we can determine next time to calibrate */
	temperature = reg_txpower_get_temperature(priv);
	priv->last_temperature = temperature;

	reg_init_channel_groups(priv);

	/* initialize Tx power info for each and every channel, 2.4 and 5.x */
	for (i = 0, ch_info = priv->channel_info; i < priv->channel_count;
	     i++, ch_info++) {
		a_band = is_channel_a_band(ch_info);
		if (!is_channel_valid(ch_info))
			continue;

		/* find this channel's channel group (*not* "band") index */
		ch_info->group_index = reg_get_chnl_grp_index(priv, ch_info);

		/* Get this chnlgrp's rate->max/clip-powers table */
		clip_pwrs = priv->clip_groups[ch_info->group_index].clip_powers;

		/* calculate power index *adjustment* value according to
		 *   diff between current temperature and factory temperature */
		delta_index = reg_adjust_power_by_temp(temperature,
						       priv->eeprom.
						       groups[ch_info->
							      group_index].
						       temperature);
		IPW_DEBUG_POWER("Delta index for channel %d: %d [%d]\n",
				ch_info->channel, delta_index,
				temperature + 260);

		/* set tx power value for all OFDM rates */
		for (rate_index = 0; rate_index < IPW_OFDM_RATES; rate_index++) {
			s32 power_idx;
			s32 old_power_idx;
			int rc = 0;

			/* use channel group's clip-power table,
			 *   but don't exceed channel's max power */
			s8 power = min(ch_info->max_power_avg,
				       clip_pwrs[rate_index]);

			pwr_info = &ch_info->power_info[rate_index];

			/* get base (i.e. at factory-measured temperature)
			 *    power table index for this rate's power */
			rc = reg_get_matched_power_index(priv, power,
							 ch_info->group_index,
							 &power_idx);
			if (rc)
				return rc;
			pwr_info->base_power_index = (u8) power_idx;

			/* temperature compensate */
			power_idx += delta_index;
			old_power_idx = power_idx;

			/* stay within range of gain table */
			power_idx = reg_fix_power_index(power_idx);

			/* fill 1 OFDM rate's ipw_channel_power_info struct */
			pwr_info->requested_power = power;
			pwr_info->power_table_index = (u8) power_idx;
			pwr_info->tpc.tx_gain =
			    power_gain_table[a_band][power_idx].tx_gain;
			pwr_info->tpc.dsp_atten =
			    power_gain_table[a_band][power_idx].dsp_atten;
		}

		/* set tx power for CCK rates, based on OFDM 12 Mbit settings */
		pwr_info = &ch_info->power_info[RATE_SCALE_12M_INDEX];
		power = pwr_info->requested_power
		    + IPW_CCK_FROM_OFDM_POWER_DIFF;
		pwr_index = pwr_info->power_table_index
		    + IPW_CCK_FROM_OFDM_INDEX_DIFF;
		base_pwr_index = pwr_info->base_power_index
		    + IPW_CCK_FROM_OFDM_INDEX_DIFF;

		/* stay within table range */
		pwr_index = reg_fix_power_index(pwr_index);
		gain = power_gain_table[a_band][pwr_index].tx_gain;
		dsp_atten = power_gain_table[a_band][pwr_index].dsp_atten;

		/* fill each CCK rate's ipw_channel_power_info structure
		 * NOTE:  All CCK-rate Txpwrs are the same for a given chnl!
		 * NOTE:  CCK rates start at end of OFDM rates! */
		for (rate_index = IPW_OFDM_RATES;
		     rate_index < IPW_MAX_RATES; rate_index++) {
			pwr_info = &ch_info->power_info[rate_index];
			pwr_info->requested_power = power;
			pwr_info->power_table_index = pwr_index;
			pwr_info->base_power_index = base_pwr_index;
			pwr_info->tpc.tx_gain = gain;
			pwr_info->tpc.dsp_atten = dsp_atten;
		}

		/* set scan tx power, 1Mbit for CCK, 6Mbit for OFDM */
		for (scan_tbl_index = 0;
		     scan_tbl_index < IPW_NUM_SCAN_RATES; scan_tbl_index++) {
			s32 actual_index = (scan_tbl_index == 0) ?
			    RATE_SCALE_1M_INDEX : RATE_SCALE_6M_INDEX;
			reg_set_scan_power(priv, scan_tbl_index,
					   actual_index, clip_pwrs,
					   ch_info, a_band);
		}
	}

	return 0;
}

/*
  add AP station into station table. there is only one AP
  station with id=0
*/
static int ipw_rxon_add_station(struct ipw_priv *priv, u8 * addr, int is_ap)
{
	/* Remove this station if it happens to already exist */
	ipw_remove_station(priv, addr, is_ap);

	return ipw_add_station(priv, addr, is_ap, 0);
}

static int ipw_send_bt_config(struct ipw_priv *priv)
{
	struct ipw_bt_cmd bt_cmd = {
		.flags = 3,
		.leadTime = 0xAA,
		.maxKill = 1,
		.killAckMask = 0,
		.killCTSMask = 0,
	};

	return ipw_send_cmd_pdu(priv, REPLY_BT_CONFIG,
				sizeof(struct ipw_bt_cmd), &bt_cmd);
}

static void ipw_set_flags_for_channel(struct ipw_priv *priv,
				      const struct ipw_channel_info *ch_info)
{
	if (is_channel_a_band(ch_info)) {
		priv->staging_rxon.flags &=
		    ~(RXON_FLG_BAND_24G_MSK | RXON_FLG_AUTO_DETECT_MSK
		      | RXON_FLG_CCK_MSK);
		priv->staging_rxon.flags |= RXON_FLG_SHORT_SLOT_MSK;
	} else {
		priv->staging_rxon.flags &= ~RXON_FLG_SHORT_SLOT_MSK;
		priv->staging_rxon.flags |= RXON_FLG_BAND_24G_MSK;
		priv->staging_rxon.flags |= RXON_FLG_AUTO_DETECT_MSK;
		priv->staging_rxon.flags &= ~RXON_FLG_CCK_MSK;
	}
}

/*
  initialize rxon structure with default values from eeprom
*/
static void ipw_connection_init_rx_config(struct ipw_priv *priv)
{
	const struct ipw_channel_info *ch_info;

	memset(&priv->staging_rxon, 0, sizeof(priv->staging_rxon));

	switch (priv->iw_mode) {
	case IEEE80211_IF_TYPE_MGMT:
		priv->staging_rxon.dev_type = RXON_DEV_TYPE_AP;
		break;

	case IEEE80211_IF_TYPE_STA:
		priv->staging_rxon.dev_type = RXON_DEV_TYPE_ESS;
		priv->staging_rxon.filter_flags = RXON_FILTER_ACCEPT_GRP_MSK;
		break;

	case IEEE80211_IF_TYPE_IBSS:
		priv->staging_rxon.dev_type = RXON_DEV_TYPE_IBSS;
		priv->staging_rxon.flags = RXON_FLG_SHORT_PREAMBLE_MSK;
		priv->staging_rxon.filter_flags = 0;
		break;

	case IEEE80211_IF_TYPE_MNTR:
		priv->staging_rxon.dev_type = RXON_DEV_TYPE_SNIFFER;
		priv->staging_rxon.filter_flags = RXON_FILTER_PROMISC_MSK |
		    RXON_FILTER_CTL2HOST_MSK | RXON_FILTER_ACCEPT_GRP_MSK;
		break;
	}

	if ((priv->config & CFG_PREAMBLE_LONG) ||
	    !hw_to_local(priv->ieee)->short_preamble)
		priv->staging_rxon.flags &= ~RXON_FLG_SHORT_PREAMBLE_MSK;

	ch_info = find_channel(priv, priv->active_conf.channel);
	if (ch_info == NULL)
		ch_info = &priv->channel_info[0];

	priv->staging_rxon.channel = ch_info->channel;
	priv->active_conf.channel = ch_info->channel;

	ipw_set_flags_for_channel(priv, ch_info);
	if (is_channel_a_band(ch_info))
		priv->active_conf.phymode = MODE_IEEE80211A;
	else
		priv->active_conf.phymode = MODE_IEEE80211G;

	priv->staging_rxon.ofdm_basic_rates =
	    R_6M_MSK | R_24M_MSK | R_36M_MSK | R_48M_MSK | R_54M_MSK |
	    R_9M_MSK | R_12M_MSK | R_18M_MSK;

	priv->staging_rxon.cck_basic_rates =
	    R_5_5M_MSK | R_1M_MSK | R_11M_MSK | R_2M_MSK;
}

/***************** END ***********************************/

#define IPW_SCAN_CHECK_WATCHDOG (7 * HZ)

static void ipw_bg_scan_check(struct work_struct *work)
{
	struct ipw_priv *priv =
	    container_of(work, struct ipw_priv, scan_check.work);

	if (priv->status & STATUS_EXIT_PENDING)
		return;

	mutex_lock(&priv->mutex);
	if (priv->status & (STATUS_SCANNING | STATUS_SCAN_ABORTING)) {
		IPW_DEBUG(IPW_DL_INFO | IPW_DL_SCAN,
			  "Scan completion watchdog resetting "
			  "adapter (%dms).\n",
			  jiffies_to_msecs(IPW_SCAN_CHECK_WATCHDOG));
		if (!(priv->status & STATUS_EXIT_PENDING))
			ipw_down(priv);
	}
	mutex_unlock(&priv->mutex);
}

static int ipw_send_scan_abort(struct ipw_priv *priv)
{
	int rc = 0;
	struct ipw_rx_packet *res;
	struct ipw_host_cmd cmd = {
		.id = REPLY_SCAN_ABORT_CMD,
		.meta.flags = CMD_WANT_SKB,
	};

	/* If there isn't a scan actively going on in the hardware
	 * then we are in between scan bands and not actually
	 * actively scanning, so don't send the abort command */
	if (!(priv->status & STATUS_SCAN_HW)) {
		priv->status &= ~STATUS_SCAN_ABORTING;
		return 0;
	}

	rc = ipw_send_cmd(priv, &cmd);
	if (rc) {
		priv->status &= ~STATUS_SCAN_ABORTING;
		return rc;
	}

	res = (struct ipw_rx_packet *)cmd.meta.u.skb->data;
	if (res->u.status != CAN_ABORT_STATUS) {
		/* The scan abort will return 1 for success or
		 * 2 for "failure".  A failure condition can be
		 * due to simply not being in an active scan which
		 * can occur if we send the scan abort before we
		 * the microcode has notified us that a scan is
		 * completed. */
		IPW_DEBUG_INFO("SCAN_ABORT returned %d.\n", res->u.status);
		priv->status &= ~(STATUS_SCAN_ABORTING | STATUS_SCAN_HW);
	}

	dev_kfree_skb_any(cmd.meta.u.skb);

	return rc;
}

#define MAX_UCODE_BEACON_INTERVAL	1024
#define INTEL_CONN_LISTEN_INTERVAL	0xA

#if 0
static u16 ipw_adjust_beacon_interval(u16 beacon_val)
{
	u16 new_val = 0;
	u16 beacon_factor = 0;

	beacon_factor =
	    (beacon_val +
	     MAX_UCODE_BEACON_INTERVAL) / MAX_UCODE_BEACON_INTERVAL;
	new_val = beacon_val / beacon_factor;

	return new_val;
}
#endif

static struct ieee80211_conf *ieee80211_get_hw_conf(struct ieee80211_hw *hw)
{
	return &hw->conf;
}

#if 0
static int ipw_setup_rxon_timing(struct ipw_priv *priv)
{
	int rc = 0;
	u64 interval_tm_unit;
	u64 tsf, result;
	unsigned long flags;
	struct ieee80211_conf *conf = NULL;

	conf = ieee80211_get_hw_conf(priv->ieee);

	/*MAC80211 we need to get beacon timestamp from upper stack
	   for now we set to 0 TODO */
	spin_lock_irqsave(&priv->lock, flags);
	priv->rxon_timing.timestamp.dw[1] = priv->timestamp1;
	priv->rxon_timing.timestamp.dw[0] = priv->timestamp0;

	priv->rxon_timing.listenInterval = INTEL_CONN_LISTEN_INTERVAL;

	tsf = priv->timestamp1;
	tsf = ((tsf << 32) | priv->timestamp0);

	spin_unlock_irqrestore(&priv->lock, flags);

	if (priv->iw_mode == IEEE80211_IF_TYPE_STA) {
		if (conf->beacon_int == 0) {
			priv->rxon_timing.beaconInterval = 100;
			priv->rxon_timing.beaconTimerInitVal = 102400;
		} else {
			priv->rxon_timing.beaconInterval = conf->beacon_int;
			priv->rxon_timing.beaconInterval =
			    ipw_adjust_beacon_interval(priv->
						       rxon_timing.
						       beaconInterval);
		}

		priv->rxon_timing.atimWindow = 0;
	} else {
		priv->rxon_timing.beaconInterval =
		    ipw_adjust_beacon_interval(conf->beacon_int);
		/*MAC80211 we need to get atim_window from upper stack
		   for now we set to 0 TODO */
		priv->rxon_timing.atimWindow = 0;
	}

	interval_tm_unit = (priv->rxon_timing.beaconInterval * 1024);
	result = do_div(tsf, interval_tm_unit);
	priv->rxon_timing.beaconTimerInitVal =
	    (u32) ((u64) interval_tm_unit - result);

	IPW_DEBUG_ASSOC
	    ("beacon interval %d beacon timer %d beacon tim %d\n",
	     priv->rxon_timing.beaconInterval,
	     priv->rxon_timing.beaconTimerInitVal,
	     priv->rxon_timing.atimWindow);
	return rc;
}
#endif

/*
  fill in the supported rate in IE field
  return : set the bit for each supported rate insert in ie
*/
static u16 ipw_supported_rate_to_ie(u8 * ie,
				    u16 supported_rate,
				    u16 basic_rate, int max_count)
{
	u16 ret_rates = 0, bit;
	int i;
	u8 *rates;

	rates = &(ie[1]);

	for (bit = 1, i = 0; i < IPW_MAX_RATES; i++, bit <<= 1) {
		if (bit & supported_rate) {
			ret_rates |= bit;
			rates[*ie] = ipw_rate_index2ieee(i) |
			    ((bit & basic_rate) ? 0x80 : 0x00);
			*ie = *ie + 1;
			if (*ie >= max_count)
				break;
		}
	}

	return ret_rates;
}

#define IEEE80211_ERP_PRESENT                  (0x01)
#define IEEE80211_ERP_USE_PROTECTION           (0x02)
#define IEEE80211_ERP_BARKER_PREAMBLE_MODE     (0x04)

union ht_cap_info {
	struct {
		u16 advancedCodingCapability:1;
		u16 supportedChannelWidthSet:1;
		u16 mimoPowerSaveMode:2;
		u16 greenField:1;
		u16 shortGI20:1;
		u16 shortGI40:1;
		u16 TxStbc:1;
		u16 RxStbc:1;
		u16 beamForming:1;
		u16 delayedBA:1;
		u16 maximalAMsduSize:1;
		u16 cckModeAt40MHz:1;
		u16 psmpSupport:1;
		u16 stbcControlFrameSupport:1;
		u16 lSigTxOpProtectionSupport:1;
	};
	u16 val;
} __attribute__ ((packed));

union ht_param_info {
	struct {
		u8 maxRxAMpduFactor:2;
		u8 mpduDensity:3;
		u8 reserved:3;
	};
	u8 val;
} __attribute__ ((packed));

#define CFG_HT_RX_AMPDU_FACTOR_DEF  (0x3)
#define HT_IE_MAX_AMSDU_SIZE_4K     (0)
#define CFG_HT_MPDU_DENSITY_2USEC   (0x5)
#define CFG_HT_MPDU_DENSITY_DEF CFG_HT_MPDU_DENSITY_2USEC

/*
  fill in all required fields and ie for probe request frame
*/
static int ipw_fill_probe_req(struct ipw_priv *priv,
			      struct ieee80211_mgmt *frame,
			      int left, int is_direct)
{
	int len = 0;
	u8 *pos = NULL;
	u16 ret_rates;

	/* Make sure there is enough space for the probe request,
	 * two mandatory IEs and the data */
	left -= 24;
	if (left < 0)
		return 0;
	len += 24;

	frame->frame_control = IEEE80211_STYPE_PROBE_REQ;
	memcpy(frame->da, BROADCAST_ADDR, ETH_ALEN);
	memcpy(frame->sa, priv->mac_addr, ETH_ALEN);
	memcpy(frame->bssid, BROADCAST_ADDR, ETH_ALEN);
	frame->seq_ctrl = 0;

	/* fill in our indirect SSID IE */
	/* ...next IE... */

	left -= 2;
	if (left < 0)
		return 0;
	len += 2;
	pos = &(frame->u.probe_req.variable[0]);
	*pos++ = WLAN_EID_SSID;
	*pos++ = 0;

	/* fill in our direct SSID IE... */
	if (is_direct) {
		/* ...next IE... */
		left -= 2 + priv->essid_len;
		if (left < 0)
			return 0;
		/* ... fill it in... */
		*pos++ = WLAN_EID_SSID;
		*pos++ = priv->essid_len;
		memcpy(pos, priv->essid, priv->essid_len);
		pos += priv->essid_len;
		len += 2 + priv->essid_len;
	}

	/* fill in supported rate */
	/* ...next IE... */
	left -= 2;
	if (left < 0)
		return 0;
	/* ... fill it in... */
	*pos++ = WLAN_EID_SUPP_RATES;
	*pos = 0;
	ret_rates =
	    ipw_supported_rate_to_ie(pos, priv->active_rate,
				     priv->active_rate_basic, left);
	len += 2 + *pos;
	pos += (*pos) + 1;
	ret_rates = ~ret_rates & priv->active_rate;

	if (ret_rates == 0)
		goto fill_end;

	/* fill in supported extended rate */
	/* ...next IE... */
	left -= 2;
	if (left < 0)
		return 0;
	/* ... fill it in... */
	*pos++ = WLAN_EID_EXT_SUPP_RATES;
	*pos = 0;
	ipw_supported_rate_to_ie(pos, ret_rates, priv->active_rate_basic, left);
	if (*pos > 0)
		len += 2 + *pos;
      fill_end:
	return len;
}

static int ipw_fill_beacon_frame(struct ipw_priv *priv,
				 struct ieee80211_hdr *hdr, u8 * dest, int left)
{

	if ((priv->iw_mode != IEEE80211_IF_TYPE_IBSS) || !priv->ibss_beacon)
		return 0;

	if (priv->ibss_beacon->len > left)
		return 0;

	memcpy(hdr, priv->ibss_beacon->data, priv->ibss_beacon->len);

	return priv->ibss_beacon->len;
}

static int ipw_send_beacon_cmd(struct ipw_priv *priv)
{
	struct ipw_frame *frame;
	int frame_size, rc;
	u16 rate;

	frame = ipw_get_free_frame(priv);

	if (!frame) {
		IPW_ERROR
		    ("Could not obtain free frame buffer for beacon "
		     "command.\n");
		return -ENOMEM;
	}

	if (!(priv->active_rxon.flags & RXON_FLG_BAND_24G_MSK)) {
		rate =
		    ipw_rate_get_lowest_plcp(priv->active_rate_basic & 0xFF0);

		if (rate <= 0)
			rate = R_6M;
	} else {
		rate = ipw_rate_get_lowest_plcp(priv->active_rate_basic & 0xF);

		if (rate <= 0)
			rate = R_1M;
	}

	frame_size = priv->hw_setting.get_beacon_cmd(priv, frame, rate);

	rc = ipw_send_cmd_pdu(priv, REPLY_TX_BEACON, frame_size,
			      &frame->u.cmd[0]);

	ipw_free_frame(priv, frame);

	return rc;
}

#define BEACON_JIFFIES(x) msecs_to_jiffies(x->beacon_int)

#if 0
static void ipw_set_rxon_conf(struct ipw_priv *priv, u8 beacon)
{
	struct ipw_rxon_cmd *rxon = &priv->rxon;
	u16 cap = 0;
	u16 epr = 0;

	if (beacon) {
		struct ieee80211_local *local = priv->net_dev->ieee80211_ptr;
		cap = priv->assoc_capability;
		epr = local->cts_protect_erp_frames;
	}
	memcpy(rxon->bssid_addr, priv->bssid, ETH_ALEN);
	rxon->channel = priv->active_conf.channel;

	rxon->flags =
	    (RXON_FLG_DIS_DIV_MSK | RXON_FLG_ANT_A_MSK | RXON_FLG_ANT_B_MSK);

	rxon->filter_flags = RXON_FILTER_ACCEPT_GRP_MSK;	// todoG is this needed

	if (priv->active_conf.phymode == MODE_IEEE80211A)
		rxon->flags |= RXON_FLG_SHORT_SLOT_MSK;
	else
		rxon->flags |=
		    (RXON_FLG_BAND_24G_MSK | RXON_FLG_AUTO_DETECT_MSK);

	switch (priv->iw_mode) {
	case IEEE80211_IF_TYPE_IBSS:
		rxon->filter_flags |= RXON_FILTER_BCON_AWARE_MSK;
		break;

	case IEEE80211_IF_TYPE_MNTR:
		rxon->dev_type = RXON_DEV_TYPE_SNIFFER;
		rxon->filter_flags |= (RXON_FILTER_PROMISC_MSK |
				       RXON_FILTER_CTL2HOST_MSK |
				       RXON_FILTER_ACCEPT_GRP_MSK);
		break;

	case IEEE80211_IF_TYPE_STA:
	default:
		rxon->dev_type = RXON_DEV_TYPE_ESS;
		break;
	}

	if (priv->active_conf.phymode == MODE_IEEE80211A) {
		priv->rxon.flags &=
		    ~(RXON_FLG_BAND_24G_MSK | RXON_FLG_AUTO_DETECT_MSK
		      | RXON_FLG_CCK_MSK);
		priv->rxon.flags |= RXON_FLG_SHORT_SLOT_MSK;
	} else {
		priv->rxon.flags &= ~RXON_FLG_SHORT_SLOT_MSK;
		priv->rxon.flags |= RXON_FLG_BAND_24G_MSK;
		priv->rxon.flags |= RXON_FLG_AUTO_DETECT_MSK;
		priv->rxon.flags &= ~RXON_FLG_CCK_MSK;
	}
	if ((cap & WLAN_CAPABILITY_SHORT_PREAMBLE))
		priv->rxon.flags |= RXON_FLG_SHORT_PREAMBLE_MSK;
	else
		priv->rxon.flags &= ~RXON_FLG_SHORT_PREAMBLE_MSK;

	if ((priv->rxon.flags & RXON_FLG_BAND_24G_MSK)) {
		if (cap & WLAN_CAPABILITY_SHORT_SLOT_TIME)
			priv->rxon.flags |= RXON_FLG_SHORT_SLOT_MSK;
		else
			priv->rxon.flags &= ~RXON_FLG_SHORT_SLOT_MSK;

		if (priv->iw_mode == IEEE80211_IF_TYPE_IBSS)
			priv->rxon.flags &= ~RXON_FLG_SHORT_SLOT_MSK;

		if (epr)
			priv->rxon.flags |= RXON_FLG_TGG_PROTECT_MSK;
		else
			priv->rxon.flags &= ~RXON_FLG_TGG_PROTECT_MSK;
	}

	priv->rxon.cck_basic_rates =
	    ((priv->active_rate_basic & 0xF) | R_1M_MSK);
	priv->rxon.ofdm_basic_rates =
	    ((priv->active_rate_basic >> 4) | R_6M_MSK);

	if ((priv->active_rate_basic & 0xF) == 0)
		priv->rxon.cck_basic_rates =
		    R_1M_MSK | R_2M_MSK | R_5_5M_MSK | R_11M_MSK;
	if (priv->active_rate_basic >> 4 == 0)
		priv->rxon.ofdm_basic_rates = R_6M_MSK | R_12M_MSK | R_24M_MSK;
}
#endif

static void ipw_set_supported_rates_mask(struct ipw_priv *priv, int rates_mask)
{

	priv->active_rate = rates_mask & 0xffff;
	priv->active_rate_basic = (rates_mask >> 16) & 0xffff;
}

/* The ipw_eeprom_band definitions below provide the mapping from the
 * EEPROM contents to the specific channel number supported for each
 * band.
 *
 * For example, ipw_priv->eeprom.band_3_channels[4] from the band_3
 * definition below maps to physical channel 42 in the 5.2Ghz spectrum.
 * The specific geography and calibration information for that channel
 * is contained in the eeprom map itself.
 *
 * During init, we copy the eeprom information and channel map
 * information into priv->channel_info_24/52 and priv->channel_map_24/52
 *
 * channel_map_24/52 provides the index in the channel_info array for a
 * given channel.  We have to have two separate maps as there is channel
 * overlap with the 2.4Ghz and 5.2Ghz spectrum as seen in band_1 and
 * band_2
 *
 * A value of 0xff stored in the channel_map indicates that the channel
 * is not supported by the hardware at all.
 *
 * A value of 0xfe in the channel_map indicates that the channel is not
 * valid for Tx with the current hardware.  This means that
 * while the system can tune and receive on a given channel, it may not
 * be able to associate or transmit any frames on that
 * channel.  There is no corresponding channel information for that
 * entry.
 *
 */

/* 2.4 GHz */
static u8 ipw_eeprom_band_1[] = {
	1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14
};

/* 5.2 Ghz bands */
static u8 ipw_eeprom_band_2[] = {
	183, 184, 185, 187, 188, 189, 192, 196, 7, 8, 11, 12, 16
};

static u8 ipw_eeprom_band_3[] = {	/* 5205-5320MHz */
	34, 36, 38, 40, 42, 44, 46, 48, 52, 56, 60, 64
};

static u8 ipw_eeprom_band_4[] = {	/* 5500-5700MHz */
	100, 104, 108, 112, 116, 120, 124, 128, 132, 136, 140
};

static u8 ipw_eeprom_band_5[] = {	/* 5725-5825MHz */
	145, 149, 153, 157, 161, 165
};

static void ipw_init_band_reference(struct ipw_priv *priv, int band,
				    int *eeprom_ch_count,
				    const struct ipw_eeprom_channel
				    **eeprom_ch_info,
				    const u8 ** eeprom_ch_index)
{
	switch (band) {
	case 1:		/* 2.4Ghz band */
		*eeprom_ch_count = ARRAY_SIZE(ipw_eeprom_band_1);
		*eeprom_ch_info = priv->eeprom.band_1_channels;
		*eeprom_ch_index = ipw_eeprom_band_1;
		break;
	case 2:		/* 5.2Ghz band */
		*eeprom_ch_count = ARRAY_SIZE(ipw_eeprom_band_2);
		*eeprom_ch_info = priv->eeprom.band_2_channels;
		*eeprom_ch_index = ipw_eeprom_band_2;
		break;
	case 3:		/* 5.2Ghz band */
		*eeprom_ch_count = ARRAY_SIZE(ipw_eeprom_band_3);
		*eeprom_ch_info = priv->eeprom.band_3_channels;
		*eeprom_ch_index = ipw_eeprom_band_3;
		break;
	case 4:		/* 5.2Ghz band */
		*eeprom_ch_count = ARRAY_SIZE(ipw_eeprom_band_4);
		*eeprom_ch_info = priv->eeprom.band_4_channels;
		*eeprom_ch_index = ipw_eeprom_band_4;
		break;
	case 5:		/* 5.2Ghz band */
		*eeprom_ch_count = ARRAY_SIZE(ipw_eeprom_band_5);
		*eeprom_ch_info = priv->eeprom.band_5_channels;
		*eeprom_ch_index = ipw_eeprom_band_5;
		break;
	default:
		BUG();
		return;
	}
}

#define IPW_INVALID_CHANNEL                   0xFF
#define IPW_INVALID_TX_CHANNEL                0xFE
#define CHECK_AND_PRINT(x) ((eeprom_ch_info[c].flags & IPW_CHANNEL_##x) ? # x " " : "")

static int ipw_init_channel_map(struct ipw_priv *priv)
{
	int eeprom_ch_count = 0;
	const u8 *eeprom_ch_index = NULL;
	const struct ipw_eeprom_channel *eeprom_ch_info = NULL;
	int b, c;
	struct ipw_channel_info *ch_info = priv->channel_info;
	if (priv->eeprom.version < 0x2f) {
		IPW_WARNING("Unsupported EEPROM version: 0x%04X\n",
			    priv->eeprom.version);
		return -EINVAL;
	}
	IPW_DEBUG_INFO("Initializing regulatory info from EEPROM\n");

	priv->channel_count =
	    ARRAY_SIZE(ipw_eeprom_band_1) +
	    ARRAY_SIZE(ipw_eeprom_band_2) +
	    ARRAY_SIZE(ipw_eeprom_band_3) +
	    ARRAY_SIZE(ipw_eeprom_band_4) + ARRAY_SIZE(ipw_eeprom_band_5);

	IPW_DEBUG_INFO("Parsing data for %d channels.\n", priv->channel_count);

	kfree(priv->channel_info);
	priv->channel_info = kzalloc(sizeof(struct ipw_channel_info) *
				     priv->channel_count, GFP_KERNEL);
	if (!priv->channel_info)
		return -ENOMEM;

	ch_info = priv->channel_info;

	/* Loop through the 5 EEPROM bands adding them in order to the
	 * channel map we maintain (that contains additional information than
	 * what just in the EEPROM) */
	for (b = 1; b <= 5; b++) {

		ipw_init_band_reference(priv, b, &eeprom_ch_count,
					&eeprom_ch_info, &eeprom_ch_index);

		/* Loop through each band adding each of the channels */
		for (c = 0; c < eeprom_ch_count; c++) {
			ch_info->channel = eeprom_ch_index[c];
			ch_info->phymode = (b == 1) ? MODE_IEEE80211B :
			    MODE_IEEE80211A;

			/* permanently store EEPROM's channel regulatory flags
			 *   and max power in channel info database. */
			ch_info->eeprom = eeprom_ch_info[c];

			/* Copy the run-time flags so they are there even on
			 * invalid channels */
			ch_info->flags = eeprom_ch_info[c].flags;

			if (!(is_channel_valid(ch_info))) {
				IPW_DEBUG_INFO("Ch. %d [%sGhz] - No Tx\n",
					       ch_info->channel,
					       is_channel_a_band(ch_info) ?
					       "5.2" : "2.4");
				ch_info++;
				continue;
			}

			/* Initialize regulatory-based run-time data */
			ch_info->max_power_avg = ch_info->curr_txpow =
			    eeprom_ch_info[c].max_power_avg;
			ch_info->scan_power = eeprom_ch_info[c].max_power_avg;
			ch_info->min_power = 0;

			if (is_channel_passive(ch_info) ||
			    is_channel_radar(ch_info)) {
				ch_info->tx_locked = 1;
				ch_info->rx_unlock = 0;
			}

			IPW_DEBUG_INFO("Ch. %d [%sGhz] %s%s%s%s%s%s(" BIT_FMT8
				       " %ddBm): Ad-Hoc %ssupported\n",
				       ch_info->channel,
				       is_channel_a_band(ch_info) ?
				       "5.2" : "2.4",
				       CHECK_AND_PRINT(IBSS),
				       CHECK_AND_PRINT(ACTIVE),
				       CHECK_AND_PRINT(RADAR),
				       CHECK_AND_PRINT(WIDE),
				       CHECK_AND_PRINT(NARROW),
				       CHECK_AND_PRINT(DFS),
				       BIT_ARG8(eeprom_ch_info[c].flags),
				       eeprom_ch_info[c].
				       max_power_avg,
				       ((eeprom_ch_info[c].
					 flags & IPW_CHANNEL_IBSS)
					&& !(eeprom_ch_info[c].
					     flags & IPW_CHANNEL_RADAR))
				       ? "" : "not ");

			/* Set the user_txpower_limit to the highest power
			 * supported by any channel */
			if (eeprom_ch_info[c].max_power_avg >
			    priv->user_txpower_limit)
				priv->user_txpower_limit =
				    eeprom_ch_info[c].max_power_avg;

			ch_info++;
		}
	}

	reg_txpower_set_from_eeprom(priv);

	return 0;
}

static int ipw_send_power_mode(struct ipw_priv *priv, u32 mode)
{
	u32 final_mode = mode;
	int rc = 0;
	unsigned long flags;

	/* If on battery, set to 3, if AC set to CAM, else user
	 * level */
	switch (mode) {
	case IPW_POWER_BATTERY:
		final_mode = IPW_POWER_INDEX_3;
		break;
	case IPW_POWER_AC:
		final_mode = IPW_POWER_MODE_CAM;
		break;
	default:
		final_mode = mode;
		break;
	}

	rc = priv->hw_setting.send_power_mode(priv, final_mode);

	spin_lock_irqsave(&priv->lock, flags);

	if (final_mode == IPW_POWER_MODE_CAM) {
		priv->status &= ~STATUS_POWER_PMI;
	} else {
		priv->status |= STATUS_POWER_PMI;
	}

	spin_unlock_irqrestore(&priv->lock, flags);
	return rc;
}

static struct ieee80211_hw_mode *ipw_get_current_hw(struct ipw_priv *priv)
{
	struct ieee80211_hw_mode *hw_mode;
	struct ieee80211_local *local = hw_to_local(priv->ieee);

	list_for_each_entry(hw_mode, &local->modes_list, list)
	    if (hw_mode->mode == priv->active_conf.phymode)
		return hw_mode;

	return NULL;
}

static void ipw_set_supported_rates(struct ipw_priv *priv)
{
	struct ieee80211_hw_mode *hw = NULL;
	int index, i;
	struct ieee80211_rate *rate;

	priv->active_rate = 0;
	priv->active_rate_basic = 0;

	hw = ipw_get_current_hw(priv);
	if (!hw || !hw->rates)
		return;

	for (i = 0; i < hw->num_rates; i++) {
		rate = &(hw->rates[i]);
		index = ipw_rate_plcp2index(rate->val);
		if ((index != -1) && (rate->flags & IEEE80211_RATE_SUPPORTED)) {
			priv->active_rate |= (1 << index);
			if (rate->flags & IEEE80211_RATE_BASIC)
				priv->active_rate_basic |= (1 << index);
		}
	}
}

static int ipw_set_rate(struct ipw_priv *priv)
{
	ipw_set_supported_rates(priv);
	priv->staging_rxon.cck_basic_rates =
	    ((priv->active_rate_basic & 0xF) | R_1M_MSK);
	priv->staging_rxon.ofdm_basic_rates =
	    ((priv->active_rate_basic >> 4) | R_6M_MSK);

	if ((priv->active_rate_basic & 0xF) == 0)
		priv->staging_rxon.cck_basic_rates =
		    R_1M_MSK | R_2M_MSK | R_5_5M_MSK | R_11M_MSK;
	if (priv->active_rate_basic >> 4 == 0)
		priv->staging_rxon.ofdm_basic_rates =
		    R_6M_MSK | R_12M_MSK | R_24M_MSK;

	return 0;
}

static void
ipw_init_hw_rates(struct ipw_priv *priv, struct ieee80211_rate *rates)
{
	/*
	 * Rates initialization.
	 */
	rates[0].rate = 10;
	rates[0].val = RATE_SCALE_1M_PLCP;
	rates[0].flags = IEEE80211_RATE_CCK;
	rates[0].val2 = RATE_SCALE_1M_PLCP;
	rates[0].min_rssi_ack = 0;
	rates[0].min_rssi_ack_delta = 0;

	rates[1].rate = 20;
	rates[1].val = RATE_SCALE_2M_PLCP;
	rates[1].flags = IEEE80211_RATE_CCK_2;
	rates[1].val2 = RATE_SCALE_2M_PLCP;
	rates[1].min_rssi_ack = 0;
	rates[1].min_rssi_ack_delta = 0;

	rates[2].rate = 55;
	rates[2].val = RATE_SCALE_5_5M_PLCP;
	rates[2].flags = IEEE80211_RATE_CCK_2;
	rates[2].val2 = RATE_SCALE_5_5M_PLCP;
	rates[2].min_rssi_ack = 0;
	rates[2].min_rssi_ack_delta = 0;

	rates[3].rate = 110;
	rates[3].val = RATE_SCALE_11M_PLCP;
	rates[3].flags = IEEE80211_RATE_CCK_2;
	rates[3].val2 = RATE_SCALE_11M_PLCP;
	rates[3].min_rssi_ack = 0;
	rates[3].min_rssi_ack_delta = 0;

	rates[4].rate = 60;
	rates[4].val = RATE_SCALE_6M_PLCP;
	rates[4].flags = IEEE80211_RATE_OFDM;
	rates[4].val2 = RATE_SCALE_6M_PLCP;
	rates[4].min_rssi_ack = 0;
	rates[4].min_rssi_ack_delta = 0;

	rates[5].rate = 90;
	rates[5].val = RATE_SCALE_9M_PLCP;
	rates[5].flags = IEEE80211_RATE_OFDM;
	rates[5].val2 = RATE_SCALE_9M_PLCP;
	rates[5].min_rssi_ack = 0;
	rates[5].min_rssi_ack_delta = 0;

	rates[6].rate = 120;
	rates[6].val = RATE_SCALE_12M_PLCP;
	rates[6].flags = IEEE80211_RATE_OFDM;
	rates[6].val2 = RATE_SCALE_12M_PLCP;
	rates[6].min_rssi_ack = 0;
	rates[6].min_rssi_ack_delta = 0;

	rates[7].rate = 180;
	rates[7].val = RATE_SCALE_18M_PLCP;
	rates[7].flags = IEEE80211_RATE_OFDM;
	rates[7].val2 = RATE_SCALE_18M_PLCP;
	rates[7].min_rssi_ack = 0;
	rates[7].min_rssi_ack_delta = 0;

	rates[8].rate = 240;
	rates[8].val = RATE_SCALE_24M_PLCP;
	rates[8].flags = IEEE80211_RATE_OFDM;
	rates[8].val2 = RATE_SCALE_24M_PLCP;
	rates[8].min_rssi_ack = 0;
	rates[8].min_rssi_ack_delta = 0;

	rates[9].rate = 360;
	rates[9].val = RATE_SCALE_36M_PLCP;
	rates[9].flags = IEEE80211_RATE_OFDM;
	rates[9].val2 = RATE_SCALE_36M_PLCP;
	rates[9].min_rssi_ack = 0;
	rates[9].min_rssi_ack_delta = 0;

	rates[10].rate = 480;
	rates[10].val = RATE_SCALE_48M_PLCP;
	rates[10].flags = IEEE80211_RATE_OFDM;
	rates[10].val2 = RATE_SCALE_48M_PLCP;
	rates[10].min_rssi_ack = 0;
	rates[10].min_rssi_ack_delta = 0;

	rates[11].rate = 540;
	rates[11].val = RATE_SCALE_54M_PLCP;
	rates[11].flags = IEEE80211_RATE_OFDM;
	rates[11].val2 = RATE_SCALE_54M_PLCP;
	rates[11].min_rssi_ack = 0;
	rates[11].min_rssi_ack_delta = 0;

	rates[12].rate = 600;
	rates[12].val = RATE_SCALE_54M_PLCP;
	rates[12].flags = IEEE80211_RATE_OFDM;
	rates[12].val2 = RATE_SCALE_54M_PLCP;
	rates[12].min_rssi_ack = 0;
	rates[12].min_rssi_ack_delta = 0;

	priv->hw_setting.init_hw_rates(priv, rates);
}

/* TODO mac80211 will rewrite flag attr of the channel, this function will
   reset it.
*/
static void ipw_reset_channel_flag(struct ipw_priv *priv)
{
	int i;
	struct ieee80211_channel *chan;
	struct ieee80211_hw_mode *hw_mode;
	struct ieee80211_local *local = hw_to_local(priv->ieee);

	list_for_each_entry(hw_mode, &local->modes_list, list) {
		for (i = 0; i < hw_mode->num_channels; i++) {
			chan = &(hw_mode->channels[i]);
			chan->flag = chan->val;
		}
	}
}

/* Initialize 80211's geo/channel info based from eeprom */
static void ipw_init_geos(struct ipw_priv *priv)
{
	struct ieee80211_local *local = hw_to_local(priv->ieee);
	struct ipw_channel_info *ch;
	struct ieee80211_hw_mode *modes;
	struct ieee80211_channel *channels;
	struct ieee80211_channel *geo_ch;
	struct ieee80211_rate *rates;
	int i = 0;
	enum {
		A = 0,
		B = 1,
		G = 2,
	};

	if (!list_empty(&local->modes_list))
		return;

	modes = kzalloc(sizeof(struct ieee80211_hw_mode) * 3, GFP_ATOMIC);
	if (!modes)
		return;

	channels = kzalloc(sizeof(struct ieee80211_channel) *
			   priv->channel_count, GFP_ATOMIC);
	if (!channels) {
		kfree(modes);
		return;
	}

	rates = kzalloc((sizeof(struct ieee80211_rate) * (IPW_MAX_RATES + 1)),
			GFP_ATOMIC);
	if (!rates) {
		kfree(modes);
		kfree(channels);
		return;
	}

	/* 0 = 802.11a
	 * 1 = 802.11b
	 * 2 = 802.11g
	 */

	/* 5.2Ghz channels start after the 2.4Ghz channels */
	modes[A].mode = MODE_IEEE80211A;
	modes[A].channels = &channels[ARRAY_SIZE(ipw_eeprom_band_1)];
	modes[A].rates = &rates[4];
	modes[A].num_rates = 8;	/* just OFDM */
	modes[A].num_channels = 0;

	modes[B].mode = MODE_IEEE80211B;
	modes[B].channels = channels;
	modes[B].rates = rates;
	modes[B].num_rates = 4;	/* just CCK */
	modes[B].num_channels = 0;

	modes[G].mode = MODE_IEEE80211G;
	modes[G].channels = channels;
	modes[G].rates = rates;
	modes[G].num_rates = 12;	/* OFDM & CCK */
	modes[G].num_channels = 0;

	priv->ieee_channels = channels;
	priv->ieee_rates = rates;

	ipw_init_hw_rates(priv, rates);

	for (i = 0, geo_ch = channels; i < priv->channel_count; i++) {
		ch = &priv->channel_info[i];

		if (!is_channel_valid(ch)) {
			IPW_WARNING
			    ("Channel %d [%sGhz] is Tx only -- skipping.\n",
			     ch->channel,
			     is_channel_a_band(ch) ? "5.2" : "2.4");
			continue;
		}

		if (is_channel_a_band(ch)) {
			if (ch->channel < IEEE80211_52GHZ_MIN_CHANNEL ||
			    ch->channel > IEEE80211_52GHZ_MAX_CHANNEL) {
				IPW_WARNING
				    ("Channel %d [5.2Ghz] not supported by "
				     "mac80211.\n", ch->channel);
				continue;
			}

			geo_ch = &modes[A].channels[modes[A].num_channels++];
		} else {
			if (ch->channel < IEEE80211_24GHZ_MIN_CHANNEL ||
			    ch->channel > IEEE80211_24GHZ_MAX_CHANNEL) {
				IPW_WARNING
				    ("Channel %d [2.4Ghz] not supported by "
				     "mac80211.\n", ch->channel);
				continue;
			}

			geo_ch = &modes[B].channels[modes[B].num_channels++];
			modes[G].num_channels++;
		}

		geo_ch->freq = ieee80211chan2mhz((ch->channel));
		geo_ch->chan = ch->channel;
		geo_ch->val = ch->channel;
		geo_ch->power_level = ch->max_power_avg;
		geo_ch->antenna_max = 0xff;

		if (is_channel_valid(ch)) {
			geo_ch->flag = IEEE80211_CHAN_W_SCAN;
			if (!(ch->flags & IPW_CHANNEL_IBSS))
				geo_ch->flag |= IEEE80211_CHAN_W_IBSS;

			if (ch->flags & IPW_CHANNEL_ACTIVE)
				geo_ch->flag |= IEEE80211_CHAN_W_ACTIVE_SCAN;

			if (ch->flags & IPW_CHANNEL_RADAR)
				geo_ch->flag |= IEEE80211_CHAN_W_RADAR_DETECT;

			if (ch->max_power_avg > priv->max_channel_txpower_limit)
				priv->max_channel_txpower_limit =
				    ch->max_power_avg;
		}

		geo_ch->val = ch->flags;

	}

	if ((modes[A].num_channels == 0) && priv->is_abg) {
		printk(KERN_INFO DRV_NAME
		       ": Incorrectly detected BG card as ABG.  Please send "
		       "your PCI ID 0x%04X:0x%04X to maintainer.\n",
		       priv->pci_dev->device, priv->pci_dev->subsystem_device);
		priv->is_abg = 0;
	}

	if ((priv->config & CFG_STATIC_CHANNEL) &&
	    !ipw_get_channel_info(priv, priv->active_conf.phymode,
				  priv->active_conf.channel)) {
		IPW_WARNING("Invalid channel configured. Resetting to ANY.\n");
		if (priv->active_conf.phymode == MODE_IEEE80211A)
			priv->active_conf.channel = 34;
		else
			priv->active_conf.channel = 1;
		priv->config &= ~CFG_STATIC_CHANNEL;
	}

	printk(KERN_INFO DRV_NAME
	       ": Tunable channels: %d 802.11bg, %d 802.11a channels\n",
	       modes[G].num_channels, modes[A].num_channels);

	if (modes[A].num_channels)
		ieee80211_register_hwmode(priv->ieee, &modes[A]);
	if (modes[B].num_channels)
		ieee80211_register_hwmode(priv->ieee, &modes[B]);
	if (modes[G].num_channels)
		ieee80211_register_hwmode(priv->ieee, &modes[G]);

	priv->status |= STATUS_GEO_CONFIGURED;
}

static int ipw_commit_rxon(struct ipw_priv *priv);

static void ipw_bg_post_associate(struct work_struct *work)
{
	struct ipw_priv *priv =
	    container_of(work, struct ipw_priv, post_associate);
	struct ieee80211_conf *conf = NULL;

	if (priv->status & STATUS_EXIT_PENDING)
		return;

	mutex_lock(&priv->mutex);

	conf = ieee80211_get_hw_conf(priv->ieee);

	memset(&priv->rxon_timing, 0, sizeof(struct ipw_rxon_time_cmd));
/*	ipw_setup_rxon_timing(priv);
	ipw_send_cmd_pdu(priv, REPLY_RXON_TIMING,
			 sizeof(priv->rxon_timing), &priv->rxon_timing);
*/
	priv->staging_rxon.assoc_id = priv->assoc_id;
	ipw_commit_rxon(priv);

	switch (priv->iw_mode) {
	case IEEE80211_IF_TYPE_STA:
		if (ipw_rxon_add_station(priv, priv->bssid, 1) ==
		    IPW_INVALID_STATION) {
			IPW_WARNING("Could not add STA " MAC_FMT "\n",
				    MAC_ARG(priv->bssid));
			break;
		}

		ipw_rate_scale_rxon_handle(priv, AP_ID);

		break;

	case IEEE80211_IF_TYPE_IBSS:

		/* clear out the station table */
		ipw_clear_stations_table(priv);

		ipw_rxon_add_station(priv, BROADCAST_ADDR, 0);
		ipw_rxon_add_station(priv, priv->bssid, 0);
		ipw_rate_scale_rxon_handle(priv, STA_ID);
		ipw_send_beacon_cmd(priv);

		break;
	}

	ipw_link_up(priv);

	mutex_unlock(&priv->mutex);
}

static void eeprom_parse_mac(struct ipw_priv *priv, u8 * mac)
{
	memcpy(mac, priv->eeprom.mac_address, 6);
}

/*
 * Read EEPROM contents
 */
static int ipw_eeprom_init(struct ipw_priv *priv)
{
	u16 *e = (u16 *) & priv->eeprom;
	u32 r;
	int to;
	u32 gp = ipw_read32(priv, CSR_EEPROM_GP);
	u16 sz = sizeof(priv->eeprom);
	int rc;
	u16 addr;

	if (sizeof(priv->eeprom) != 1024) {
		IPW_ERROR("EEPROM structure size incorrect!\n");
		return -EINVAL;
	}

	if ((gp & 0x00000007) == 0x00000000) {
		IPW_ERROR("EEPROM not found, EEPROM_GP=0x%08x", gp);
		return -ENOENT;
	}

	ipw_clear_bit(priv, CSR_EEPROM_GP, 0x00000180);
	for (addr = 0, r = 0; addr < sz; addr += 2) {
		ipw_write32(priv, CSR_EEPROM_REG, addr << 1);
		ipw_clear_bit(priv, CSR_EEPROM_REG, 0x00000002);
		rc = ipw_grab_restricted_access(priv);
		if (rc)
			return rc;

		for (to = 0; to < 10; to++) {
			r = ipw_read_restricted(priv, CSR_EEPROM_REG);
			if (r & 1)
				break;
			udelay(5);
		}

		ipw_release_restricted_access(priv);

		if (!(r & 1)) {
			IPW_ERROR("Time out reading EEPROM[%d]", addr);
			return -ETIMEDOUT;
		}

		e[addr / 2] = r >> 16;
	}

	return 0;
}

/************************** END ***********************/

/************************** RX-FUNCTIONS ****************************/
/*
 * Rx theory of operation
 *
 * The host allocates 32 DMA target addresses and passes the host address
 * to the firmware at register IPW_RFDS_TABLE_LOWER + N * RFD_SIZE where N is
 * 0 to 31
 *
 * Rx Queue Indexes
 * The host/firmware share two index registers for managing the Rx buffers.
 *
 * The READ index maps to the first position that the firmware may be writing
 * to -- the driver can read up to (but not including) this position and get
 * good data.
 * The READ index is managed by the firmware once the card is enabled.
 *
 * The WRITE index maps to the last position the driver has read from -- the
 * position preceding WRITE is the last slot the firmware can place a packet.
 *
 * The queue is empty (no good data) if WRITE = READ - 1, and is full if
 * WRITE = READ.
 *
 * During initialization the host sets up the READ queue position to the first
 * INDEX position, and WRITE to the last (READ - 1 wrapped)
 *
 * When the firmware places a packet in a buffer it will advance the READ index
 * and fire the RX interrupt.  The driver can then query the READ index and
 * process as many packets as possible, moving the WRITE index forward as it
 * resets the Rx queue buffers with new memory.
 *
 * The management in the driver is as follows:
 * + A list of pre-allocated SKBs is stored in ipw->rxq->rx_free.  When
 *   ipw->rxq->free_count drops to or below RX_LOW_WATERMARK, work is scheduled
 *   to replenish the ipw->rxq->rx_free.
 * + In ipw_rx_queue_replenish (scheduled) if 'processed' != 'read' then the
 *   ipw->rxq is replenished and the READ INDEX is updated (updating the
 *   'processed' and 'read' driver indexes as well)
 * + A received packet is processed and handed to the kernel network stack,
 *   detached from the ipw->rxq.  The driver 'processed' index is updated.
 * + The Host/Firmware ipw->rxq is replenished at tasklet time from the rx_free
 *   list. If there are no allocated buffers in ipw->rxq->rx_free, the READ
 *   INDEX is not incremented and ipw->status(RX_STALLED) is set.  If there
 *   were enough free buffers and RX_STALLED is set it is cleared.
 *
 *
 * Driver sequence:
 *
 * ipw_rx_queue_alloc()       Allocates rx_free
 * ipw_rx_queue_replenish()   Replenishes rx_free list from rx_used, and calls
 *                            ipw_rx_queue_restock
 * ipw_rx_queue_restock()     Moves available buffers from rx_free into Rx
 *                            queue, updates firmware pointers, and updates
 *                            the WRITE index.  If insufficient rx_free buffers
 *                            are available, schedules ipw_rx_queue_replenish
 *
 * -- enable interrupts --
 * ISR - ipw_rx()             Detach ipw_rx_mem_buffers from pool up to the
 *                            READ INDEX, detaching the SKB from the pool.
 *                            Moves the packet buffer from queue to rx_used.
 *                            Calls ipw_rx_queue_restock to refill any empty
 *                            slots.
 * ...
 *
 */

static inline int ipw_rx_queue_space(struct ipw_rx_queue *q)
{
	int s = q->read - q->write;
	if (s <= 0)
		s += RX_QUEUE_SIZE;
	s -= 2;			// keep some buffer to not confuse full and empty queue
	if (s < 0)
		s = 0;
	return s;
}

static inline u32 ipw_dma_addr2rbd_ptr(struct ipw_priv *priv,
				       dma_addr_t dma_addr)
{
	if (priv->is_3945)
		return (u32) dma_addr;
	else
		return (u32) (dma_addr >> 8);
}

/*
 * If there are slots in the RX queue that  need to be restocked,
 * and we have free pre-allocated buffers, fill the ranks as much
 * as we can pulling from rx_free.
 *
 * This moves the 'write' index forward to catch up with 'processed', and
 * also updates the memory address in the firmware to reference the new
 * target buffer.
 */
static int ipw_rx_queue_restock(struct ipw_priv *priv)
{
	struct ipw_rx_queue *rxq = priv->rxq;
	struct list_head *element;
	struct ipw_rx_mem_buffer *rxb;
	unsigned long flags;
	int write;
	int counter = 0;

	spin_lock_irqsave(&rxq->lock, flags);
	write = rxq->write & ~0x7;
	while ((ipw_rx_queue_space(rxq) > 0) && (rxq->free_count)) {
		element = rxq->rx_free.next;
		rxb = list_entry(element, struct ipw_rx_mem_buffer, list);
		list_del(element);
		((u32 *) rxq->bd)[rxq->write] =
		    ipw_dma_addr2rbd_ptr(priv, rxb->dma_addr);
		rxq->queue[rxq->write] = rxb;
		rxq->write = (rxq->write + 1) % RX_QUEUE_SIZE;
		rxq->free_count--;
		counter++;
	}
	spin_unlock_irqrestore(&rxq->lock, flags);
	/* If the pre-allocated buffer pool is dropping low, schedule to
	 * refill it */
	if (rxq->free_count <= RX_LOW_WATERMARK) {
		queue_work(priv->workqueue, &priv->rx_replenish);
	}

	counter = ipw_rx_queue_space(rxq);
	/* If we've added more space for the firmware to place data, tell it */
	if ((write != (rxq->write & ~0x7))
	    || (abs(rxq->write - rxq->read) > 7)) {
		spin_lock_irqsave(&rxq->lock, flags);
		rxq->need_update = 1;
		spin_unlock_irqrestore(&rxq->lock, flags);
		ipw_rx_queue_update_write_ptr(priv, rxq);
	}

	return 0;
}

/*
 * Move all used packet from rx_used to rx_free, allocating a new SKB for each.
 * Also restock the Rx queue via ipw_rx_queue_restock.
 *
 * This is called as a scheduled work item (except for during initialization)
 */
static void ipw_rx_queue_replenish(struct ipw_priv *priv)
{
	struct ipw_rx_queue *rxq = priv->rxq;
	struct list_head *element;
	struct ipw_rx_mem_buffer *rxb;
	unsigned long flags;
	spin_lock_irqsave(&rxq->lock, flags);
	while (!list_empty(&rxq->rx_used)) {
		element = rxq->rx_used.next;
		rxb = list_entry(element, struct ipw_rx_mem_buffer, list);
		rxb->skb =
		    alloc_skb(priv->hw_setting.rx_buffer_size,
			      GFP_DMA | __GFP_NOWARN | GFP_ATOMIC);
		if (!rxb->skb) {
			printk(KERN_CRIT
			       "%s: Can not allocate SKB buffers.\n",
			       priv->net_dev->name);
			/* We don't reschedule replenish work here -- we will
			 * call the restock method and if it still needs
			 * more buffers it will schedule replenish */
			break;
		}
		list_del(element);
		rxb->dma_addr =
		    pci_map_single(priv->pci_dev, rxb->skb->data,
				   priv->hw_setting.rx_buffer_size,
				   PCI_DMA_FROMDEVICE);
		list_add_tail(&rxb->list, &rxq->rx_free);
		rxq->free_count++;
	}
	spin_unlock_irqrestore(&rxq->lock, flags);

	spin_lock_irqsave(&priv->lock, flags);
	ipw_rx_queue_restock(priv);
	spin_unlock_irqrestore(&priv->lock, flags);
}

static void ipw_bg_rx_queue_replenish(struct work_struct *work)
{
	struct ipw_priv *priv =
	    container_of(work, struct ipw_priv, rx_replenish);

	if (priv->status & STATUS_EXIT_PENDING)
		return;

	mutex_lock(&priv->mutex);
	ipw_rx_queue_replenish(priv);
	mutex_unlock(&priv->mutex);
}

/* Assumes that the skb field of the buffers in 'pool' is kept accurate.
 * If an SKB has been detached, the POOL needs to have it's SKB set to NULL
 * This free routine walks the list of POOL entries and if SKB is set to
 * non NULL it is unmapped and freed
 */
static void ipw_rx_queue_free(struct ipw_priv *priv, struct ipw_rx_queue *rxq)
{
	int i;
	if (!rxq)
		return;
	for (i = 0; i < RX_QUEUE_SIZE + RX_FREE_BUFFERS; i++) {
		if (rxq->pool[i].skb != NULL) {
			pci_unmap_single(priv->pci_dev,
					 rxq->pool[i].dma_addr,
					 priv->hw_setting.rx_buffer_size,
					 PCI_DMA_FROMDEVICE);
			dev_kfree_skb(rxq->pool[i].skb);
		}
	}

	pci_free_consistent(priv->pci_dev, 4 * RX_QUEUE_SIZE, rxq->bd,
			    rxq->dma_addr);
	kfree(rxq);
}

static int ipw_rxq_stop(struct ipw_priv *priv)
{

	return priv->hw_setting.rxq_stop(priv);
}

static int ipw_rx_init(struct ipw_priv *priv, struct ipw_rx_queue *rxq)
{

	return priv->hw_setting.rx_init(priv, rxq);
}

static struct ipw_rx_queue *ipw_rx_queue_alloc(struct ipw_priv *priv)
{
	struct ipw_rx_queue *rxq;
	struct pci_dev *dev = priv->pci_dev;
	int i;
	rxq = (struct ipw_rx_queue *)kzalloc(sizeof(*rxq), GFP_ATOMIC);
	memset(rxq, 0, sizeof(*rxq));

	spin_lock_init(&rxq->lock);
	INIT_LIST_HEAD(&rxq->rx_free);
	INIT_LIST_HEAD(&rxq->rx_used);
	rxq->bd = pci_alloc_consistent(dev, 4 * RX_QUEUE_SIZE, &rxq->dma_addr);
	/* Fill the rx_used queue with _all_ of the Rx buffers */
	for (i = 0; i < RX_FREE_BUFFERS + RX_QUEUE_SIZE; i++)
		list_add_tail(&rxq->pool[i].list, &rxq->rx_used);
	/* Set us so that we have processed and used all buffers, but have
	 * not restocked the Rx queue with fresh buffers */
	rxq->read = rxq->write = 0;
	rxq->free_count = 0;
	rxq->need_update = 0;
	return rxq;
}

static inline void ipw_rx_queue_reset(struct ipw_priv *priv,
				      struct ipw_rx_queue *rxq)
{
	unsigned long flags;
	int i;
	spin_lock_irqsave(&rxq->lock, flags);
	INIT_LIST_HEAD(&rxq->rx_free);
	INIT_LIST_HEAD(&rxq->rx_used);
	/* Fill the rx_used queue with _all_ of the Rx buffers */
	for (i = 0; i < RX_FREE_BUFFERS + RX_QUEUE_SIZE; i++) {
		/* In the reset function, these buffers may have been allocated
		 * to an SKB, so we need to unmap and free potential storage */
		if (rxq->pool[i].skb != NULL) {
			pci_unmap_single(priv->pci_dev,
					 rxq->pool[i].dma_addr,
					 priv->hw_setting.rx_buffer_size,
					 PCI_DMA_FROMDEVICE);
			dev_kfree_skb(rxq->pool[i].skb);
			rxq->pool[i].skb = NULL;
		}
		list_add_tail(&rxq->pool[i].list, &rxq->rx_used);
	}

	/* Set us so that we have processed and used all buffers, but have
	 * not restocked the Rx queue with fresh buffers */
	rxq->read = rxq->write = 0;
	rxq->free_count = 0;
	spin_unlock_irqrestore(&rxq->lock, flags);
}

/************************ NIC-FUNCTIONS *******************/
/* These functions load the firmware and micro code for the operation of
 * the ipw hardware.  It assumes the buffer has all the bits for the
 * image and the caller is handling the memory allocation and clean up.
 */

static int ipw_nic_stop_master(struct ipw_priv *priv)
{
	return priv->hw_setting.nic_stop_master(priv);
}

static void ipw_nic_start(struct ipw_priv *priv)
{
	unsigned long flags;

	spin_lock_irqsave(&priv->lock, flags);
	ipw_clear_bit(priv, CSR_RESET,
		      CSR_RESET_REG_FLAG_MASTER_DISABLED |
		      CSR_RESET_REG_FLAG_STOP_MASTER |
		      CSR_RESET_REG_FLAG_NEVO_RESET);
	spin_unlock_irqrestore(&priv->lock, flags);
}

static int ipw3945_nic_set_pwr_src(struct ipw_priv *priv, int pwr_max)
{
	int rc = 0;
	unsigned long flags;

	//return 0;
	spin_lock_irqsave(&priv->lock, flags);
	rc = ipw_grab_restricted_access(priv);
	if (rc) {
		spin_unlock_irqrestore(&priv->lock, flags);
		return rc;
	}

	if (!pwr_max) {
		u32 val;
		rc = pci_read_config_dword(priv->pci_dev, 0x0C8, &val);
		if (val & PCI_CFG_PMC_PME_FROM_D3COLD_SUPPORT) {
			ipw_set_bits_mask_restricted_reg(priv,
							 ALM_APMG_PS_CTL,
							 APMG_PS_CTRL_REG_VAL_POWER_SRC_VAUX,
							 ~APMG_PS_CTRL_REG_MSK_POWER_SRC);
			ipw_release_restricted_access(priv);

			ipw_poll_bit(priv, CSR_GPIO_IN,
				     CSR_GPIO_IN_VAL_VAUX_PWR_SRC,
				     CSR_GPIO_IN_BIT_AUX_POWER, 5000);
		} else
			ipw_release_restricted_access(priv);

	} else {
		ipw_set_bits_mask_restricted_reg(priv,
						 ALM_APMG_PS_CTL,
						 APMG_PS_CTRL_REG_VAL_POWER_SRC_VMAIN,
						 ~APMG_PS_CTRL_REG_MSK_POWER_SRC);

		ipw_release_restricted_access(priv);
		ipw_poll_bit(priv, CSR_GPIO_IN, CSR_GPIO_IN_VAL_VMAIN_PWR_SRC, CSR_GPIO_IN_BIT_AUX_POWER, 5000);	//uS
	}
	spin_unlock_irqrestore(&priv->lock, flags);

	return rc;
}

static int ipw_rate_scale_init_handle(struct ipw_priv *priv, s32 window_size);

static int ipw_nic_init(struct ipw_priv *priv)
{
	u8 rev_id;
	int rc;
	unsigned long flags;

	ipw_power_init_handle(priv);
	ipw_rate_scale_init_handle(priv, IPW_RATE_SCALE_MAX_WINDOW);

	spin_lock_irqsave(&priv->lock, flags);
	ipw_set_bit(priv, CSR_ANA_PLL_CFG, (1 << 24));
	ipw_set_bit(priv, CSR_GIO_CHICKEN_BITS,
		    CSR_GIO_CHICKEN_BITS_REG_BIT_L1A_NO_L0S_RX);

	ipw_set_bit(priv, CSR_GP_CNTRL, CSR_GP_CNTRL_REG_FLAG_INIT_DONE);
	rc = ipw_poll_bit(priv, CSR_GP_CNTRL,
			  CSR_GP_CNTRL_REG_FLAG_MAC_CLOCK_READY,
			  CSR_GP_CNTRL_REG_FLAG_MAC_CLOCK_READY, 25000);
	if (rc < 0) {
		spin_unlock_irqrestore(&priv->lock, flags);
		IPW_DEBUG_INFO("Failed to init the card\n");
		return rc;
	}

	rc = ipw_grab_restricted_access(priv);
	if (rc) {
		spin_unlock_irqrestore(&priv->lock, flags);
		return rc;
	}
	ipw_write_restricted_reg(priv, ALM_APMG_CLK_EN,
				 APMG_CLK_REG_VAL_DMA_CLK_RQT |
				 APMG_CLK_REG_VAL_BSM_CLK_RQT);
	udelay(20);
	ipw_set_bits_restricted_reg(priv, ALM_APMG_PCIDEV_STT,
				    APMG_DEV_STATE_REG_VAL_L1_ACTIVE_DISABLE);
	ipw_release_restricted_access(priv);
	spin_unlock_irqrestore(&priv->lock, flags);

	/* Determine HW type */
	rc = pci_read_config_byte(priv->pci_dev, PCI_REVISION_ID, &rev_id);
	if (rc)
		return rc;
	IPW_DEBUG_INFO("HW Revision ID = 0x%X\n", rev_id);

	ipw3945_nic_set_pwr_src(priv, 1);
	spin_lock_irqsave(&priv->lock, flags);

	if (rev_id & PCI_CFG_REV_ID_BIT_RTP)
		IPW_DEBUG_INFO("RTP type \n");
	else if (rev_id & PCI_CFG_REV_ID_BIT_BASIC_SKU) {
		IPW_DEBUG_INFO("ALM-MB type\n");
		ipw_set_bit(priv, CSR_HW_IF_CONFIG_REG,
			    CSR_HW_IF_CONFIG_REG_BIT_ALMAGOR_MB);
	} else {
		IPW_DEBUG_INFO("ALM-MM type\n");
		ipw_set_bit(priv, CSR_HW_IF_CONFIG_REG,
			    CSR_HW_IF_CONFIG_REG_BIT_ALMAGOR_MM);
	}

	spin_unlock_irqrestore(&priv->lock, flags);

	/* Initialize the EEPROM */
	rc = ipw_eeprom_init(priv);
	if (rc)
		return rc;

	spin_lock_irqsave(&priv->lock, flags);
	if (EEPROM_SKU_CAP_OP_MODE_MRC == priv->eeprom.sku_cap) {
		IPW_DEBUG_INFO("SKU OP mode is mrc\n");
		ipw_set_bit(priv, CSR_HW_IF_CONFIG_REG,
			    CSR_HW_IF_CONFIG_REG_BIT_SKU_MRC);
	} else {
		IPW_DEBUG_INFO("SKU OP mode is basic\n");
	}

	if ((priv->eeprom.board_revision & 0xF0) == 0xD0) {
		IPW_DEBUG_INFO("3945ABG revision is 0x%X\n",
			       priv->eeprom.board_revision);
		ipw_set_bit(priv, CSR_HW_IF_CONFIG_REG,
			    CSR_HW_IF_CONFIG_REG_BIT_BOARD_TYPE);
	} else {
		IPW_DEBUG_INFO("3945ABG revision is 0x%X\n",
			       priv->eeprom.board_revision);
		ipw_clear_bit(priv, CSR_HW_IF_CONFIG_REG,
			      CSR_HW_IF_CONFIG_REG_BIT_BOARD_TYPE);
	}

	if (priv->eeprom.almgor_m_version <= 1) {
		ipw_set_bit(priv, CSR_HW_IF_CONFIG_REG,
			    CSR_HW_IF_CONFIG_REG_BITS_SILICON_TYPE_A);
		IPW_DEBUG_INFO("Card M type A version is 0x%X\n",
			       priv->eeprom.almgor_m_version);
	} else {
		IPW_DEBUG_INFO("Card M type B version is 0x%X\n",
			       priv->eeprom.almgor_m_version);
		ipw_set_bit(priv, CSR_HW_IF_CONFIG_REG,
			    CSR_HW_IF_CONFIG_REG_BITS_SILICON_TYPE_B);
	}
	spin_unlock_irqrestore(&priv->lock, flags);

	if (priv->eeprom.sku_cap & EEPROM_SKU_CAP_SW_RF_KILL_ENABLE)
		priv->capability |= CAP_RF_SW_KILL;
	else
		priv->capability &= ~CAP_RF_SW_KILL;

	if (priv->eeprom.sku_cap & EEPROM_SKU_CAP_HW_RF_KILL_ENABLE)
		priv->capability |= CAP_RF_HW_KILL;
	else
		priv->capability &= ~CAP_RF_HW_KILL;

	switch (priv->capability & (CAP_RF_HW_KILL | CAP_RF_SW_KILL)) {
	case CAP_RF_HW_KILL:
		IPW_DEBUG_RF_KILL("HW RF KILL supported in EEPROM.\n");
		break;
	case CAP_RF_SW_KILL:
		IPW_DEBUG_RF_KILL("SW RF KILL supported in EEPROM.\n");
		break;
	case (CAP_RF_HW_KILL | CAP_RF_SW_KILL):
		IPW_DEBUG_RF_KILL("HW & HW RF KILL supported in EEPROM.\n");
		break;
	default:
		IPW_DEBUG_RF_KILL("NO RF KILL supported in EEPROM.\n");
		break;
	}

	/* Allocate the RX queue, or reset if it is already allocated */
	if (!priv->rxq)
		priv->rxq = ipw_rx_queue_alloc(priv);
	else
		ipw_rx_queue_reset(priv, priv->rxq);

	if (!priv->rxq) {
		IPW_ERROR("Unable to initialize Rx queue\n");
		return -ENOMEM;
	}
	ipw_rx_queue_replenish(priv);

	ipw_rx_init(priv, priv->rxq);

	spin_lock_irqsave(&priv->lock, flags);

/*
 * Look at using this instead :::
	priv->rxq->need_update = 1;
	ipw_rx_queue_update_write_ptr(priv, priv->rxq);
*/

	rc = ipw_grab_restricted_access(priv);
	if (rc) {
		spin_unlock_irqrestore(&priv->lock, flags);
		return rc;
	}
	ipw_write_restricted(priv, FH_RCSR_WPTR(0), priv->rxq->write & ~7);
	ipw_release_restricted_access(priv);

	spin_unlock_irqrestore(&priv->lock, flags);

	rc = ipw_queue_reset(priv);
	if (rc)
		return rc;

	priv->status |= STATUS_INIT;

	return 0;
}

/* Call this function from process context, it will sleeps */
static int ipw_nic_reset(struct ipw_priv *priv)
{
	int rc = 0;
	unsigned long flags;

	ipw_nic_stop_master(priv);

	spin_lock_irqsave(&priv->lock, flags);
	ipw_set_bit(priv, CSR_RESET, CSR_RESET_REG_FLAG_SW_RESET);

	rc = ipw_poll_bit(priv, CSR_GP_CNTRL,
			  CSR_GP_CNTRL_REG_FLAG_MAC_CLOCK_READY,
			  CSR_GP_CNTRL_REG_FLAG_MAC_CLOCK_READY, 25000);

	rc = ipw_grab_restricted_access(priv);
	if (rc) {
		spin_unlock_irqrestore(&priv->lock, flags);
		return rc;
	}

	ipw_write_restricted_reg(priv, APMG_CLK_CTRL_REG,
				 APMG_CLK_REG_VAL_BSM_CLK_RQT);

	udelay(10);

	ipw_set_bit(priv, CSR_GP_CNTRL, CSR_GP_CNTRL_REG_FLAG_INIT_DONE);

	ipw_write_restricted_reg(priv, ALM_APMG_LARC_INT_MSK, 0x0);
	ipw_write_restricted_reg(priv, ALM_APMG_LARC_INT, 0xFFFFFFFF);

	/* enable DMA */
	ipw_write_restricted_reg(priv, ALM_APMG_CLK_EN,
				 APMG_CLK_REG_VAL_DMA_CLK_RQT |
				 APMG_CLK_REG_VAL_BSM_CLK_RQT);
	udelay(10);

	ipw_set_bits_restricted_reg(priv, ALM_APMG_PS_CTL,
				    APMG_PS_CTRL_REG_VAL_ALM_R_RESET_REQ);
	udelay(5);
	ipw_clear_bits_restricted_reg(priv, ALM_APMG_PS_CTL,
				      APMG_PS_CTRL_REG_VAL_ALM_R_RESET_REQ);
	ipw_release_restricted_access(priv);

	/* Clear the 'host command active' bit... */
	priv->status &= ~STATUS_HCMD_ACTIVE;

	wake_up_interruptible(&priv->wait_command_queue);
	spin_unlock_irqrestore(&priv->lock, flags);

	return rc;
}

/********************* UCODE-DOWNLOAD-FUNCTIONS ***********************/

static int attach_buffer_to_tfd_frame(struct ipw_priv *priv,
				      void *tfd, dma_addr_t addr, u16 len)
{

	return priv->hw_setting.attach_buffer_to_tfd_frame(tfd, addr, len);
}

/* this header = beginning of ucode file */
struct ipw_ucode {
	u32 ver;
	u32 inst_size;		// size of runtime instructions
	u32 data_size;		// size of runtime data
	u32 boot_size;		// size of bootstrap instructions
	u32 boot_data_size;	// size of bootstrap data
	u8 data[0];		// data appears in same order as "size" elements
};

/* verify bootstrap uCode image in card vs. host */
static int ipw_verify_bootstrap(struct ipw_priv *priv)
{
	u32 *image;
	u32 len, val;
	int rc1;
	int rc2 = 0;
	u32 errcnt;

	len = priv->ucode_boot_data.len;
	image = (u32 *) priv->ucode_boot_data.v_addr;

	IPW_DEBUG_INFO("bootstrap data image size is %u\n", len);

	rc1 = ipw_grab_restricted_access(priv);
	if (rc1)
		return rc1;

	/* read from card's data memory to verify */
	ipw_write_restricted(priv, HBUS_TARG_MEM_RADDR, RTC_DATA_LOWER_BOUND);

	for (errcnt = 0; len > 0; len -= sizeof(u32), image++) {
		/* read data comes through single port, auto-incr addr */
		val = ipw_read_restricted(priv, HBUS_TARG_MEM_RDAT);
		if (val != *image) {
			IPW_ERROR("bootstrap DATA section is invalid at offset "
				  "0x%x\n", priv->ucode_boot_data.len - len);
			rc2 = -EIO;
			errcnt++;
			if (errcnt >= 20)
				break;
		}
	}

	ipw_release_restricted_access(priv);

	if (!errcnt)
		IPW_DEBUG_INFO("bootstrap image in DATA memory is good\n");

	/* check instruction image */
	len = priv->ucode_boot.len;
	image = (u32 *) priv->ucode_boot.v_addr;

	IPW_DEBUG_INFO("bootstrap instruction image size is %u\n", len);

	rc1 = ipw_grab_restricted_access(priv);
	if (rc1)
		return rc1;

	/* read from card's instruction memory to verify */
	ipw_write_restricted(priv, HBUS_TARG_MEM_RADDR, RTC_INST_LOWER_BOUND);

	for (errcnt = 0; len > 0; len -= sizeof(u32), image++) {
		/* read data comes through single port, auto-incr addr */
		val = ipw_read_restricted(priv, HBUS_TARG_MEM_RDAT);
		if (val != *image) {
			IPW_ERROR("bootstrap INST section is invalid at offset "
				  "0x%x\n", priv->ucode_boot.len - len);
			rc2 = -EIO;
			errcnt++;
			if (errcnt >= 20)
				break;
		}
	}

	ipw_release_restricted_access(priv);

	if (!errcnt)
		IPW_DEBUG_INFO
		    ("bootstrap image in INSTRUCTION memory is good\n");

	return rc2;
}

/* verify runtime uCode image in card vs. host */
static int ipw_verify_ucode(struct ipw_priv *priv)
{
	u32 *image;
	u32 len, val;
	int rc = 0;
	u32 errcnt;

	/* Since data memory has already been modified by running uCode,
	 * we can't really verify the data image, but we'll show its size. */
	IPW_DEBUG_INFO("ucode data image size is %u\n", priv->ucode_data.len);

	/* read from instruction memory to verify instruction image */
	image = priv->ucode_code.v_addr;
	len = priv->ucode_code.len;

	IPW_DEBUG_INFO("ucode inst image size is %u\n", len);

	rc = ipw_grab_restricted_access(priv);
	if (rc)
		return rc;

	ipw_write_restricted(priv, HBUS_TARG_MEM_RADDR, RTC_INST_LOWER_BOUND);

	errcnt = 0;
	for (; len > 0; len -= sizeof(u32), image++) {
		/* read data comes through single port, auto-incr addr */
		val = ipw_read_restricted(priv, HBUS_TARG_MEM_RDAT);
		if (val != *image) {
			IPW_ERROR("uCode INST section is invalid at "
				  "offset 0x%x, is 0x%x, s/b 0x%x\n",
				  priv->ucode_code.len - len, val, *image);
			rc = -EIO;
			errcnt++;
			if (errcnt >= 20)
				break;
		}
	}

	ipw_release_restricted_access(priv);

	if (!errcnt)
		IPW_DEBUG_INFO
		    ("runtime ucode image in INSTRUCTION memory is good\n");

	return rc;
}

/* load uCode image via 3945's Transmit Frame Descriptor DMA channel */
static int ipw_load_ucode(struct ipw_priv *priv,
			  struct fw_image_desc *desc,
			  u32 mem_size, dma_addr_t dst_addr)
{
	dma_addr_t phy_addr = 0;
	u32 len = 0;
	u32 count = 0;
	u32 pad;
	struct tfd_frame tfd;
	u32 tx_config = 0;
	int rc;

	memset(&tfd, 0, sizeof(struct tfd_frame));

	phy_addr = desc->p_addr;
	len = desc->len;

	if (mem_size < len) {
		IPW_ERROR("invalid image size, too big %d %d\n", mem_size, len);
		return -EINVAL;
	}

	while (len > 0) {
		if (ALM_TB_MAX_BYTES_COUNT < len) {
			attach_buffer_to_tfd_frame(priv, &tfd, phy_addr,
						   ALM_TB_MAX_BYTES_COUNT);
			len -= ALM_TB_MAX_BYTES_COUNT;
			phy_addr += ALM_TB_MAX_BYTES_COUNT;
		} else {
			attach_buffer_to_tfd_frame(priv, &tfd, phy_addr, len);
			break;
		}
	}

	pad = U32_PAD(len);
	count = TFD_CTL_COUNT_GET(tfd.control_flags);
	tfd.control_flags = TFD_CTL_COUNT_SET(count) | TFD_CTL_PAD_SET(pad);

	rc = ipw_grab_restricted_access(priv);
	if (rc)
		return rc;

	ipw_write_restricted(priv, FH_TCSR_CONFIG(ALM_FH_SRVC_CHNL),
			     ALM_FH_TCSR_TX_CONFIG_REG_VAL_DMA_CHNL_PAUSE);
	ipw_write_buffer_restricted(priv,
				    ALM_FH_TFDB_CHNL_BUF_CTRL_REG
				    (ALM_FH_SRVC_CHNL),
				    sizeof(struct tfd_frame), (u32 *) & tfd);
	ipw_write_restricted(priv, HBUS_TARG_MEM_WADDR, dst_addr);
	ipw_write_restricted(priv, FH_TCSR_CREDIT(ALM_FH_SRVC_CHNL),
			     0x000FFFFF);
	ipw_write_restricted(priv,
			     FH_TCSR_BUFF_STTS(ALM_FH_SRVC_CHNL),
			     ALM_FH_TCSR_CHNL_TX_BUF_STS_REG_VAL_TFDB_VALID
			     | ALM_FH_TCSR_CHNL_TX_BUF_STS_REG_BIT_TFDB_WPTR);

	tx_config = ALM_FH_TCSR_TX_CONFIG_REG_VAL_DMA_CHNL_ENABLE |
	    ALM_FH_TCSR_TX_CONFIG_REG_VAL_DMA_CREDIT_DISABLE_VAL |
	    ALM_FH_TCSR_TX_CONFIG_REG_VAL_MSG_MODE_DRIVER;

	ipw_write_restricted(priv, FH_TCSR_CONFIG(ALM_FH_SRVC_CHNL), tx_config);

	rc = ipw_poll_restricted_bit(priv, FH_TSSR_TX_STATUS,
				     ALM_FH_TSSR_TX_STATUS_REG_MSK_CHNL_IDLE
				     (ALM_FH_SRVC_CHNL), 1000);
	if (rc < 0) {
		IPW_WARNING("3945ABG card ucode DOWNLOAD FAILED \n");
		goto done;
	}

	rc = 0;

	IPW_DEBUG_INFO("3945ABG card ucode download is good \n");

	ipw_write_restricted(priv, FH_TCSR_CREDIT(ALM_FH_SRVC_CHNL), 0x0);

      done:
	ipw_release_restricted_access(priv);
	return rc;
}

static void ipw_dealloc_ucode_pci(struct ipw_priv *priv)
{
	if (priv->ucode_code.v_addr != NULL)
		pci_free_consistent(priv->pci_dev,
				    priv->ucode_code.len,
				    priv->ucode_code.v_addr,
				    priv->ucode_code.p_addr);
	if (priv->ucode_data.v_addr != NULL)
		pci_free_consistent(priv->pci_dev,
				    priv->ucode_data.len,
				    priv->ucode_data.v_addr,
				    priv->ucode_data.p_addr);
	if (priv->ucode_boot.v_addr != NULL)
		pci_free_consistent(priv->pci_dev,
				    priv->ucode_boot.len,
				    priv->ucode_boot.v_addr,
				    priv->ucode_boot.p_addr);
	if (priv->ucode_boot_data.v_addr != NULL)
		pci_free_consistent(priv->pci_dev,
				    priv->ucode_boot_data.len,
				    priv->ucode_boot_data.v_addr,
				    priv->ucode_boot_data.p_addr);
	priv->ucode_code.v_addr = NULL;
	priv->ucode_data.v_addr = NULL;
	priv->ucode_boot.v_addr = NULL;
	priv->ucode_boot_data.v_addr = NULL;
}

/* Read uCode images from disk file.
 * Copy into buffers for card to fetch via bus-mastering */
static int ipw_read_ucode(struct ipw_priv *priv)
{
	struct ipw_ucode *ucode;
	int rc = 0;
	const struct firmware *ucode_raw;
	const char *name = "iwlwifi-3945.ucode";	/* firmware file name */
	u8 *src;
	size_t len;

	/* ask firmware_class module to get the boot firmware off disk.
	 * request_firmware() is synchronous call, file is in memory on return.
	 * TODO:  Would it be more polite to use asynchronous
	 *        request_firmware_nowait()??  If so, put request back into
	 *        ipw_pci_probe(), and rest of this function would serve as
	 *        the callback for request_firmware_nowait().  Also need to
	 *        make sure everything waits for this callback to complete! */
	rc = request_firmware(&ucode_raw, name, &priv->pci_dev->dev);
	if (rc < 0) {
		IPW_ERROR("%s firmware file req failed: Reason %d\n", name, rc);
		goto error;
	}

	IPW_DEBUG_INFO("Got firmware '%s' file (%zd bytes) from disk\n",
		       name, ucode_raw->size);

	/* make sure that we got at least our header! */
	if (ucode_raw->size < sizeof(*ucode)) {
		rc = -EINVAL;
		goto err_release;
	}

	/* data from ucode file:  header followed by uCode images */
	ucode = (void *)ucode_raw->data;

	IPW_DEBUG_INFO("f/w package hdr ucode version = 0x%x\n", ucode->ver);
	IPW_DEBUG_INFO("f/w package hdr runtime inst size = %u\n",
		       ucode->inst_size);
	IPW_DEBUG_INFO("f/w package hdr runtime data size = %u\n",
		       ucode->data_size);
	IPW_DEBUG_INFO("f/w package hdr boot inst size = %u\n",
		       ucode->boot_size);
	IPW_DEBUG_INFO("f/w package hdr boot data size = %u\n",
		       ucode->boot_data_size);

	/* verify size of file vs. image size info in file's header */
	if (ucode_raw->size < sizeof(*ucode) +
	    ucode->inst_size + ucode->data_size +
	    ucode->boot_size + ucode->boot_data_size) {
		IPW_DEBUG_INFO("uCode file size %d too small\n",
			       (int)ucode_raw->size);
		rc = -EINVAL;
		goto err_release;
	}

	/* verify that uCode images will fit in card's SRAM */
	if (ucode->inst_size > ALM_RTC_INST_SIZE) {
		IPW_DEBUG_INFO("uCode instr len %d too large to fit in card\n",
			       (int)ucode->inst_size);
		rc = -EINVAL;
		goto err_release;
	}

	if (ucode->data_size > ALM_RTC_DATA_SIZE) {
		IPW_DEBUG_INFO("uCode data len %d too large to fit in card\n",
			       (int)ucode->data_size);
		rc = -EINVAL;
		goto err_release;
	}

	if (ucode->boot_size > ALM_RTC_INST_SIZE) {
		IPW_DEBUG_INFO
		    ("uCode boot instr len %d too large to fit in card\n",
		     (int)ucode->boot_size);
		rc = -EINVAL;
		goto err_release;
	}

	if (ucode->boot_data_size > ALM_RTC_DATA_SIZE) {
		IPW_DEBUG_INFO
		    ("uCode boot data len %d too large to fit in card\n",
		     (int)ucode->boot_data_size);
		rc = -EINVAL;
		goto err_release;
	}

	/* allocate ucode buffers for card's bus-master loading */
	priv->ucode_code.len = ucode->inst_size;
	priv->ucode_code.v_addr =
	    pci_alloc_consistent(priv->pci_dev,
				 priv->ucode_code.len,
				 &(priv->ucode_code.p_addr));

	priv->ucode_data.len = ucode->data_size;
	priv->ucode_data.v_addr =
	    pci_alloc_consistent(priv->pci_dev,
				 priv->ucode_data.len,
				 &(priv->ucode_data.p_addr));

	priv->ucode_boot.len = ucode->boot_size;
	priv->ucode_boot.v_addr =
	    pci_alloc_consistent(priv->pci_dev,
				 priv->ucode_boot.len,
				 &(priv->ucode_boot.p_addr));

	priv->ucode_boot_data.len = ucode->boot_data_size;
	priv->ucode_boot_data.v_addr =
	    pci_alloc_consistent(priv->pci_dev,
				 priv->ucode_boot_data.len,
				 &(priv->ucode_boot_data.p_addr));

	if (!priv->ucode_code.v_addr || !priv->ucode_data.v_addr
	    || !priv->ucode_boot.v_addr || !priv->ucode_boot_data.v_addr) {
		IPW_ERROR("failed to allocate pci memory\n");
		rc = -ENOMEM;
		goto err_pci_alloc;
	}

	/* Copy images into buffers for card's bus-master reads ... */

	/* runtime instructions (first block of data in file) */
	src = &ucode->data[0];
	len = priv->ucode_code.len;
	IPW_DEBUG_INFO("Copying (but not loading) uCode instr len %d\n",
		       (int)len);
	memcpy(priv->ucode_code.v_addr, src, len);

	/* runtime data (2nd block) */
	src = &ucode->data[ucode->inst_size];
	len = priv->ucode_data.len;
	IPW_DEBUG_INFO("Copying (but not loading) uCode data len %d\n",
		       (int)len);
	memcpy(priv->ucode_data.v_addr, src, len);

	/* bootstrap instructions (3rd block) */
	src = &ucode->data[ucode->inst_size + ucode->data_size];
	len = priv->ucode_boot.len;
	IPW_DEBUG_INFO("Copying (but not loading) boot instr len %d\n",
		       (int)len);
	memcpy(priv->ucode_boot.v_addr, src, len);

	/* bootstrap data (4th block) */
	src = &ucode->data[ucode->inst_size + ucode->data_size
			   + ucode->boot_size];
	len = priv->ucode_boot_data.len;
	IPW_DEBUG_INFO("Copying (but not loading) boot data len %d\n",
		       (int)len);
	memcpy(priv->ucode_boot_data.v_addr, src, len);

	release_firmware(ucode_raw);
	return 0;

      err_pci_alloc:
	ipw_dealloc_ucode_pci(priv);

      err_release:
	release_firmware(ucode_raw);

      error:
	return rc;
}

/* Load bootstrap uCode and prepare to start it */
static int ipw_setup_bootstrap(struct ipw_priv *priv)
{
	int rc = 0;

	/* Load bootstrap uCode data into card via card's TFD DMA channel */
	rc = ipw_load_ucode(priv, &(priv->ucode_boot_data),
			    ALM_RTC_DATA_SIZE, RTC_DATA_LOWER_BOUND);
	if (rc)
		goto error;

	/* Load bootstrap uCode instructions, same way */
	rc = ipw_load_ucode(priv, &(priv->ucode_boot),
			    ALM_RTC_INST_SIZE, RTC_INST_LOWER_BOUND);
	if (rc)
		goto error;

	/* verify bootstrap in-place in DATA and INSTRUCTION SRAM */
	ipw_verify_bootstrap(priv);

	/* tell bootstrap uCode where to find the runtime uCode in host DRAM */
	rc = ipw_grab_restricted_access(priv);
	if (rc)
		goto error;

	ipw_write_restricted_reg(priv, BSM_DRAM_INST_PTR_REG,
				 priv->ucode_code.p_addr);
	ipw_write_restricted_reg(priv, BSM_DRAM_DATA_PTR_REG,
				 priv->ucode_data.p_addr);
	ipw_write_restricted_reg(priv, BSM_DRAM_INST_BYTECOUNT_REG,
				 priv->ucode_code.len);
	ipw_write_restricted_reg(priv, BSM_DRAM_DATA_BYTECOUNT_REG,
				 priv->ucode_data.len);
	ipw_release_restricted_access(priv);

	return 0;

      error:
	return rc;
}

/*************** DMA-QUEUE-GENERAL-FUNCTIONS  *****/
/**
 * DMA services
 *
 * Theory of operation
 *
 * A queue is a circular buffers with 'Read' and 'Write' pointers.
 * 2 empty entries always kept in the buffer to protect from overflow.
 *
 * For Tx queue, there are low mark and high mark limits. If, after queuing
 * the packet for Tx, free space become < low mark, Tx queue stopped. When
 * reclaiming packets (on 'tx done IRQ), if free space become > high mark,
 * Tx queue resumed.
 *
 * The IPW operates with six queues, one receive queue in the device's
 * sram, one transmit queue for sending commands to the device firmware,
 * and four transmit queues for data.
 *
 * The four transmit queues allow for performing quality of service (qos)
 * transmissions as per the 802.11 protocol.  Currently Linux does not
 * provide a mechanism to the user for utilizing prioritized queues, so
 * we only utilize the first data transmit queue (queue1).
 */

/**
 * Driver allocates buffers of this size for Rx
 */

static inline int ipw_queue_space(const struct ipw_queue *q)
{
	int s = q->last_used - q->first_empty;
	if (q->last_used > q->first_empty)
		s -= q->n_bd;

	if (s <= 0)
		s += q->n_window;
	s -= 2;			/* keep some reserve to not confuse empty and full situations */
	if (s < 0)
		s = 0;
	return s;
}

static inline int ipw_queue_inc_wrap(int index, int n_bd)
{
	return (++index == n_bd) ? 0 : index;
}

static inline int x2_queue_used(const struct ipw_queue *q, int i)
{
	return q->first_empty > q->last_used ?
	    (i >= q->last_used && i < q->first_empty) :
	    !(i < q->last_used && i >= q->first_empty);
}

static inline u8 get_next_cmd_index(struct ipw_queue *q, u32 index, int is_huge)
{
	if (is_huge)
		return q->n_window;

	return (u8) (index % q->n_window);
}

/***************** END **************************/

/********************* TX-QUEUE-FUNCTIONS ******************/
static int ipw_stop_tx_queue(struct ipw_priv *priv)
{
	int rc = 0;
	rc = priv->hw_setting.stop_tx_queue(priv);
	ipw_tx_queue_free(priv);
	return rc;
}

/**
 * Initialize common DMA queue structure
 *
 * @param q                queue to init
 * @param count            Number of BD's to allocate. Should be power of 2
 * @param read_register    Address for 'read' register
 *                         (not offset within BAR, full address)
 * @param write_register   Address for 'write' register
 *                         (not offset within BAR, full address)
 * @param base_register    Address for 'base' register
 *                         (not offset within BAR, full address)
 * @param size             Address for 'size' register
 *                         (not offset within BAR, full address)
 */

static int ipw_tx_reset(struct ipw_priv *priv)
{

	return priv->hw_setting.tx_reset(priv);
}

static int ipw_queue_init(struct ipw_priv *priv, struct ipw_queue *q,
			  int count, int size, u32 id)
{
	q->n_bd = count;
	q->n_window = size;
	q->id = id;

	q->low_mark = q->n_window / 4;
	if (q->low_mark < 4)
		q->low_mark = 4;

	q->high_mark = q->n_window / 8;
	if (q->high_mark < 2)
		q->high_mark = 2;

	q->first_empty = q->last_used = 0;

	priv->hw_setting.queue_init(priv, q, count, size, id);
	return 0;
}

static int ipw_queue_tx_init(struct ipw_priv *priv,
			     struct ipw_tx_queue *q, int count, u32 id)
{
	struct pci_dev *dev = priv->pci_dev;
	int len;
	int rc = 0;

	/* allocate command space + one big command for scan since scan
	 * command is very huge the system will not have two scan at the
	 * same time */
	len = (sizeof(struct ipw_cmd) * count) + IPW_MAX_SCAN_SIZE;
	q->cmd = pci_alloc_consistent(dev, len, &q->dma_addr_cmd);
	if (!q->cmd)
		return -ENOMEM;

	rc = priv->hw_setting.queue_tx_init(priv, q, count, id);
	if (rc) {
		pci_free_consistent(dev, len, q->cmd, q->dma_addr_cmd);

		return -ENOMEM;
	}

	q->need_update = 0;
	ipw_queue_init(priv, &q->q, TFD_QUEUE_SIZE_MAX, count, id);
	return 0;
}

static void ipw_txstatus_to_ieee(struct ipw_priv *priv,
				 struct ipw_tx_info *tx_sta)
{

	tx_sta->status.ack_signal = 0;
	tx_sta->status.excessive_retries = 0;
	tx_sta->status.queue_length = 0;
	tx_sta->status.queue_number = 0;

	if (in_interrupt())
		ieee80211_tx_status_irqsafe(priv->ieee,
					    tx_sta->skb[0], &(tx_sta->status));
	else
		ieee80211_tx_status(priv->ieee,
				    tx_sta->skb[0], &(tx_sta->status));

	tx_sta->skb[0] = NULL;

}

/**
 * Free one TFD, those at index [txq->q.last_used].
 * Do NOT advance any indexes
 */
static void ipw_queue_tx_free_tfd(struct ipw_priv *priv,
				  struct ipw_tx_queue *txq)
{

	priv->hw_setting.queue_tx_free_tfd(priv, txq);
}

/**
 * Deallocate DMA queue.
 *
 * Empty queue by removing and destroying all BD's.
 * Free all buffers.
 *
 * @param dev
 * @param q
 */
static void ipw_queue_tx_free(struct ipw_priv *priv, struct ipw_tx_queue *txq)
{
	struct ipw_queue *q = &txq->q;
	struct pci_dev *dev = priv->pci_dev;
	int len;

	if (q->n_bd == 0)
		return;

	/* first, empty all BD's */
	for (; q->first_empty != q->last_used;
	     q->last_used = ipw_queue_inc_wrap(q->last_used, q->n_bd)) {
		ipw_queue_tx_free_tfd(priv, txq);
	}

	len = (sizeof(txq->cmd[0]) * q->n_window) + IPW_MAX_SCAN_SIZE;
	pci_free_consistent(dev, len, txq->cmd, txq->dma_addr_cmd);

	/* free buffers belonging to queue itself */
	priv->hw_setting.queue_tx_free(priv, txq);

	if (txq->txb) {
		kfree(txq->txb);
		txq->txb = NULL;
	}

	/* 0 fill whole structure */
	memset(txq, 0, sizeof(*txq));
}

/**
 * Destroy all DMA queues and structures
 *
 * @param priv
 */
static void ipw_tx_queue_free(struct ipw_priv *priv)
{
	int i;

	/* Tx queues */
	for (i = 0; i < priv->hw_setting.max_queue_number; i++)
		ipw_queue_tx_free(priv, &priv->txq[i]);
}

/**
 * Destroys all DMA structures and initialize them again
 *
 * @param priv
 * @return error code
 */
static int ipw_queue_reset(struct ipw_priv *priv)
{
	int rc = 0;
	int i, num_slots;

	ipw_tx_queue_free(priv);

	/* Tx CMD queue */
	ipw_tx_reset(priv);

	/* Tx queue(s) */
	for (i = 0; i < priv->hw_setting.max_queue_number; i++) {
		num_slots =
		    (i ==
		     priv->hw_setting.
		     cmd_queue_no) ? TFD_CMD_SLOTS : TFD_TX_CMD_SLOTS;
		rc = ipw_queue_tx_init(priv, &priv->txq[i], num_slots, i);
		if (rc) {
			IPW_ERROR("Tx %d queue init failed\n", i);
			goto error;
		}
	}

	return rc;

      error:
	ipw_tx_queue_free(priv);
	return rc;
}

/**************************************************************/

static inline int ipw_is_broadcast_ether_addr(const u8 * addr)
{
	return (addr[0] & addr[1] & addr[2] & addr[3] & addr[4] &
		addr[5]) == 0xff;
}

static u8 ipw_remove_station(struct ipw_priv *priv, u8 * bssid, int is_ap)
{
	int index = IPW_INVALID_STATION;
	int i;
	unsigned long flags;

	spin_lock_irqsave(&priv->sta_lock, flags);
	if (is_ap) {
		index = AP_ID;
		if ((priv->stations[index].used))
			priv->stations[index].used = 0;
	} else if (ipw_is_broadcast_ether_addr(bssid)) {
		index = priv->hw_setting.broadcast_id;
		if ((priv->stations[index].used))
			priv->stations[index].used = 0;
	} else {
		for (i = STA_ID; i < (priv->num_stations + STA_ID); i++) {
			if ((priv->stations[i].used)
			    &&
			    (!compare_ether_addr
			     (priv->stations[i].sta.sta.MACAddr, bssid))) {
				index = i;
				priv->stations[index].used = 0;
				break;
			}
		}
	}

	if (index != IPW_INVALID_STATION) {
		if (priv->num_stations > 0)
			priv->num_stations--;
		IPW_DEBUG_ASSOC("Removing STA ID %d: " MAC_FMT "\n",
				index, MAC_ARG(bssid));

	}

	spin_unlock_irqrestore(&priv->sta_lock, flags);
	return 0;
}

static void ipw_delete_stations_table(struct ipw_priv *priv, u8 just_ibss)
{
	unsigned long flags;

	spin_lock_irqsave(&priv->sta_lock, flags);
	priv->num_stations = 0;
	memset(priv->stations, 0,
	       priv->hw_setting.number_of_stations *
	       sizeof(struct ipw_station_entry));
	spin_unlock_irqrestore(&priv->sta_lock, flags);
}

static void ipw_clear_stations_table(struct ipw_priv *priv)
{
	unsigned long flags;

	spin_lock_irqsave(&priv->sta_lock, flags);

	priv->num_stations = 0;
	memset(priv->stations, 0,
	       priv->hw_setting.number_of_stations *
	       sizeof(struct ipw_station_entry));
	spin_unlock_irqrestore(&priv->sta_lock, flags);
}

static u8 ipw_add_station(struct ipw_priv *priv, u8 * bssid,
			  int is_ap, u8 flags)
{
	int i = priv->hw_setting.number_of_stations;
	int index = IPW_INVALID_STATION;
	unsigned long flags_spin;

	spin_lock_irqsave(&priv->sta_lock, flags_spin);
	if (is_ap) {
		index = AP_ID;
		if ((priv->stations[index].used) &&
		    (!compare_ether_addr
		     (priv->stations[index].sta.sta.MACAddr, bssid)))
			goto done;
	} else if (ipw_is_broadcast_ether_addr(bssid)) {
		index = priv->hw_setting.broadcast_id;
		if ((priv->stations[index].used) &&
		    (!compare_ether_addr(priv->stations[index].sta.sta.MACAddr,
					 bssid)))
			goto done;
	} else {
		for (i = STA_ID; i < (priv->num_stations + STA_ID); i++) {
			if ((priv->stations[i].used)
			    &&
			    (!compare_ether_addr
			     (priv->stations[i].sta.sta.MACAddr, bssid))) {
				goto done;
			}

			if ((priv->stations[i].used == 0) &&
			    (index == IPW_INVALID_STATION))
				index = i;
		}
	}

	if (index != IPW_INVALID_STATION)
		i = index;

	if (i == priv->hw_setting.number_of_stations) {
		index = IPW_INVALID_STATION;
		goto done;
	}

	IPW_DEBUG_ASSOC("Adding STA ID %d: " MAC_FMT "\n", i, MAC_ARG(bssid));

	priv->stations[i].used = 1;
	priv->stations[i].current_rate.s.rate = R_1M;
	memset(&priv->stations[i].sta, 0, sizeof(struct ipw_addsta_cmd));
	memcpy(priv->stations[i].sta.sta.MACAddr, bssid, ETH_ALEN);
	priv->stations[i].sta.ctrlAddModify = 0;
	priv->stations[i].sta.sta.staID = i;
	priv->stations[i].sta.station_flags = 0;

	//todoG do we need this
//      priv->stations[i].sta.tid_disable_tx = 0xffff;  /* all TID's disabled */
	if (priv->active_conf.phymode == MODE_IEEE80211A)
		priv->stations[i].sta.tx_rate.rate_n_flags = R_6M;
	else
		priv->stations[i].sta.tx_rate.rate_n_flags = R_1M |
		    priv->hw_setting.cck_flag;

	priv->stations[i].sta.tx_rate.rate_n_flags |= RATE_MCS_ANT_B_MSK;
	priv->stations[i].sta.tx_rate.rate_n_flags &= ~RATE_MCS_ANT_A_MSK;

	priv->stations[i].sta.station_flags |= STA_MODIFY_TX_RATE_MSK;

	priv->stations[i].current_rate.rate_n_flags =
	    priv->stations[i].sta.tx_rate.rate_n_flags;

	priv->num_stations++;
	spin_unlock_irqrestore(&priv->sta_lock, flags_spin);
	ipw_send_add_station(priv, &priv->stations[i].sta, flags);
	return i;
      done:
	spin_unlock_irqrestore(&priv->sta_lock, flags_spin);
	return index;

}

static u8 ipw_find_station(struct ipw_priv *priv, u8 * bssid)
{
	int i;
	int start = 0;
	int ret = IPW_INVALID_STATION;
	unsigned long flags;

	if (priv->iw_mode == IEEE80211_IF_TYPE_IBSS)
		start = STA_ID;

	if (ipw_is_broadcast_ether_addr(bssid))
		return priv->hw_setting.broadcast_id;

	spin_lock_irqsave(&priv->sta_lock, flags);
	for (i = start; i < (start + priv->num_stations); i++)
		if ((priv->stations[i].used) &&
		    (!compare_ether_addr
		     (priv->stations[i].sta.sta.MACAddr, bssid))) {
			ret = i;
			goto out;
		}

	IPW_DEBUG_ASSOC("can not find STA " MAC_FMT " (total %d)\n",
			MAC_ARG(bssid), priv->num_stations);
      out:
	spin_unlock_irqrestore(&priv->sta_lock, flags);
	return ret;
}

static u8 ipw_sync_station(struct ipw_priv *priv, int sta_id,
			   u16 tx_rate, u8 flags)
{

	if (sta_id != IPW_INVALID_STATION) {
		unsigned long flags_spin;

		spin_lock_irqsave(&priv->sta_lock, flags_spin);

		priv->stations[sta_id].sta.sta.modify_mask =
		    STA_CONTROL_MODIFY_MSK;
		priv->stations[sta_id].sta.tx_rate.rate_n_flags = tx_rate;
		priv->stations[sta_id].current_rate.rate_n_flags = tx_rate;
		priv->stations[sta_id].sta.ctrlAddModify =
		    STA_CONTROL_MODIFY_MSK;

		spin_unlock_irqrestore(&priv->sta_lock, flags_spin);

		ipw_send_add_station(priv, &priv->stations[sta_id].sta, flags);
		IPW_DEBUG_RATE("SCALE sync station %d to rate %d\n",
			       sta_id, tx_rate);
		return sta_id;
	}

	return IPW_INVALID_STATION;
}

static void ipw_bg_scan_completed(struct work_struct *work)
{
	struct ipw_priv *priv =
	    container_of(work, struct ipw_priv, scan_completed);

	if (priv->status & STATUS_EXIT_PENDING)
		return;

	IPW_DEBUG(IPW_DL_INFO | IPW_DL_SCAN, "SCAN complete scan\n");
	ieee80211_scan_completed(priv->ieee);

	if (!priv->only_active_channel)
		return;

	mutex_lock(&priv->mutex);
	mutex_unlock(&priv->mutex);
}

struct ipw_status_code {
	u16 status;
	const char *reason;
};

static void inline average_init(struct average *avg)
{
	memset(avg, 0, sizeof(*avg));
}

static void inline average_add(struct average *avg, s16 val)
{
	avg->sum -= avg->entries[avg->pos];
	avg->sum += val;
	avg->entries[avg->pos++] = val;
	if (unlikely(avg->pos == AVG_ENTRIES)) {
		avg->init = 1;
		avg->pos = 0;
	}
}

static s16 inline average_value(struct average *avg)
{
	if (!unlikely(avg->init)) {
		if (avg->pos)
			return avg->sum / avg->pos;
		return 0;
	}

	return avg->sum / AVG_ENTRIES;
}

static void ipw_reset_stats(struct ipw_priv *priv)
{
	priv->quality = 0;
	average_init(&priv->average_missed_beacons);
	average_init(&priv->average_rssi);
	average_init(&priv->average_noise);
	priv->last_rate = 0;
	priv->last_missed_beacons = 0;
	priv->last_rx_packets = 0;
	priv->last_tx_packets = 0;
	priv->last_tx_failures = 0;
	/* Firmware managed, reset only when NIC is restarted, so we have to
	 * normalize on the current value */

	/* TODO -- obtain the RX error and TX failure stats from HW */

	/* Driver managed, reset with each association */
	priv->missed_beacons = 0;
	priv->tx_packets = 0;
	priv->rx_packets = 0;
}

static inline u32 ipw_get_max_rate(struct ipw_priv *priv)
{
	u32 i = 0x80000000;
	u32 mask = priv->rates_mask;
	struct ieee80211_conf *conf = NULL;

	conf = ieee80211_get_hw_conf(priv->ieee);

	/* If currently associated in B mode, restrict the maximum
	 * rate match to B rates */
	if (priv->active_conf.phymode == MODE_IEEE80211B)
		mask &= IEEE80211_CCK_RATES_MASK;
	/* TODO: Verify that the rate is supported by the current rates
	 * list. */
	while (i && !(mask & i))
		i >>= 1;
	switch (i) {
	case IEEE80211_CCK_RATE_1MB_MASK:
		return 1000000;
	case IEEE80211_CCK_RATE_2MB_MASK:
		return 2000000;
	case IEEE80211_CCK_RATE_5MB_MASK:
		return 5500000;
	case IEEE80211_OFDM_RATE_6MB_MASK:
		return 6000000;
	case IEEE80211_OFDM_RATE_9MB_MASK:
		return 9000000;
	case IEEE80211_CCK_RATE_11MB_MASK:
		return 11000000;
	case IEEE80211_OFDM_RATE_12MB_MASK:
		return 12000000;
	case IEEE80211_OFDM_RATE_18MB_MASK:
		return 18000000;
	case IEEE80211_OFDM_RATE_24MB_MASK:
		return 24000000;
	case IEEE80211_OFDM_RATE_36MB_MASK:
		return 36000000;
	case IEEE80211_OFDM_RATE_48MB_MASK:
		return 48000000;
	case IEEE80211_OFDM_RATE_54MB_MASK:
		return 54000000;
	}

	return 54000000;
}

static u32 ipw_get_current_rate(struct ipw_priv *priv)
{
	u32 rate;
	unsigned long flags;

	spin_lock_irqsave(&priv->sta_lock, flags);
	if (priv->iw_mode == IEEE80211_IF_TYPE_IBSS)
		rate = priv->stations[STA_ID].current_rate.s.rate;
	else
		rate = priv->stations[AP_ID].current_rate.s.rate;
	spin_unlock_irqrestore(&priv->sta_lock, flags);

	switch (rate) {
	case IPW_TX_RATE_1MB:
		return 1000000;
	case IPW_TX_RATE_2MB:
		return 2000000;
	case IPW_TX_RATE_5MB:
		return 5500000;
	case IPW_TX_RATE_6MB:
		return 6000000;
	case IPW_TX_RATE_9MB:
		return 9000000;
	case IPW_TX_RATE_11MB:
		return 11000000;
	case IPW_TX_RATE_12MB:
		return 12000000;
	case IPW_TX_RATE_18MB:
		return 18000000;
	case IPW_TX_RATE_24MB:
		return 24000000;
	case IPW_TX_RATE_36MB:
		return 36000000;
	case IPW_TX_RATE_48MB:
		return 48000000;
	case IPW_TX_RATE_54MB:
		return 54000000;
	}

	return 0;
}

#define IPW_STATS_INTERVAL (2 * HZ)
static void ipw_gather_stats(struct ipw_priv *priv)
{
	u32 rx_err = 0, rx_err_delta, rx_packets_delta;
	u32 tx_failures = 0, tx_failures_delta, tx_packets_delta;
	u32 missed_beacons_percent, missed_beacons_delta;
	u32 quality = 0;
	s16 rssi;
	u32 beacon_quality, signal_quality, tx_quality, rx_quality,
	    rate_quality;
	u32 max_rate;
	struct ieee80211_conf *conf = NULL;

	conf = ieee80211_get_hw_conf(priv->ieee);

	/* TODO -- gather missed beacon stats */
	missed_beacons_delta = priv->missed_beacons - priv->last_missed_beacons;
	priv->last_missed_beacons = priv->missed_beacons;
	if (conf->beacon_int) {
		missed_beacons_percent =
		    missed_beacons_delta * (HZ *
					    conf->beacon_int) /
		    (IPW_STATS_INTERVAL * 10);
	} else {
		missed_beacons_percent = 0;
	}
	average_add(&priv->average_missed_beacons, missed_beacons_percent);

	/* TODO -- gather RX ERROR stats */

	rx_err_delta = rx_err - priv->last_rx_err;
	priv->last_rx_err = rx_err;

	/* TODO -- gather TX failure stats */

	tx_failures_delta = tx_failures - priv->last_tx_failures;
	priv->last_tx_failures = tx_failures;
	rx_packets_delta = priv->rx_packets - priv->last_rx_packets;
	priv->last_rx_packets = priv->rx_packets;
	tx_packets_delta = priv->tx_packets - priv->last_tx_packets;
	priv->last_tx_packets = priv->tx_packets;
	/* Calculate quality based on the following:
	 *
	 * Missed beacon: 100% = 0, 0% = 70% missed
	 * Rate: 60% = 1Mbs, 100% = Max
	 * Rx and Tx errors represent a straight % of total Rx/Tx
	 * RSSI: 100% = > -50,  0% = < -80
	 * Rx errors: 100% = 0, 0% = 50% missed
	 *
	 * The lowest computed quality is used.
	 *
	 */
#define BEACON_THRESHOLD 5
	beacon_quality = 100 - missed_beacons_percent;
	if (beacon_quality < BEACON_THRESHOLD)
		beacon_quality = 0;
	else
		beacon_quality =
		    (beacon_quality - BEACON_THRESHOLD) * 100 / (100 -
								 BEACON_THRESHOLD);
	IPW_DEBUG_STATS("Missed beacon: %3d%% (%d%%)\n",
			beacon_quality, missed_beacons_percent);
	priv->last_rate = ipw_get_current_rate(priv);
	max_rate = ipw_get_max_rate(priv);
	rate_quality = priv->last_rate * 40 / max_rate + 60;
	IPW_DEBUG_STATS("Rate quality : %3d%% (%dMbs)\n",
			rate_quality, priv->last_rate / 1000000);
	if (rx_packets_delta > 100 && rx_packets_delta + rx_err_delta)
		rx_quality = 100 - (rx_err_delta * 100) /
		    (rx_packets_delta + rx_err_delta);
	else
		rx_quality = 100;
	IPW_DEBUG_STATS
	    ("Rx quality   : %3d%% (%u errors, %u packets)\n",
	     rx_quality, rx_err_delta, rx_packets_delta);
	if (tx_packets_delta > 100 && tx_packets_delta + tx_failures_delta)
		tx_quality =
		    100 -
		    (tx_failures_delta * 100) / (tx_packets_delta +
						 tx_failures_delta);
	else
		tx_quality = 100;
	IPW_DEBUG_STATS
	    ("Tx quality   : %3d%% (%u errors, %u packets)\n",
	     tx_quality, tx_failures_delta, tx_packets_delta);
	rssi = average_value(&priv->average_rssi);
	signal_quality =
	    (100 *
	     (priv->perfect_rssi - priv->worst_rssi) *
	     (priv->perfect_rssi - priv->worst_rssi) -
	     (priv->perfect_rssi - rssi) *
	     (15 *
	      (priv->perfect_rssi - priv->worst_rssi) +
	      62 * (priv->perfect_rssi -
		    rssi))) / ((priv->perfect_rssi -
				priv->worst_rssi) *
			       (priv->perfect_rssi - priv->worst_rssi));
	if (signal_quality > 100)
		signal_quality = 100;
	else if (signal_quality < 1)
		signal_quality = 0;
	IPW_DEBUG_STATS("Signal level : %3d%% (%d dBm)\n",
			signal_quality, rssi);
	quality = min(beacon_quality,
		      min(rate_quality,
			  min(tx_quality, min(rx_quality, signal_quality))));
	if (quality == beacon_quality)
		IPW_DEBUG_STATS
		    ("Quality (%d%%): Clamped to missed beacons.\n", quality);
	if (quality == rate_quality)
		IPW_DEBUG_STATS
		    ("Quality (%d%%): Clamped to rate quality.\n", quality);
	if (quality == tx_quality)
		IPW_DEBUG_STATS
		    ("Quality (%d%%): Clamped to Tx quality.\n", quality);
	if (quality == rx_quality)
		IPW_DEBUG_STATS
		    ("Quality (%d%%): Clamped to Rx quality.\n", quality);
	if (quality == signal_quality)
		IPW_DEBUG_STATS
		    ("Quality (%d%%): Clamped to signal quality.\n", quality);
	priv->quality = quality;

	if (!priv->is_3945)
		if (time_after(jiffies, priv->last_statistics_time + (45 * HZ)))
			ipw_send_statistics_request(priv);

	queue_delayed_work(priv->workqueue, &priv->gather_stats,
			   IPW_STATS_INTERVAL);
}

static void ipw_bg_gather_stats(struct work_struct *work)
{
	struct ipw_priv *priv =
	    container_of(work, struct ipw_priv, gather_stats.work);

	if (priv->status & STATUS_EXIT_PENDING)
		return;

	mutex_lock(&priv->mutex);
	ipw_gather_stats(priv);
	mutex_unlock(&priv->mutex);
}

static int ipw_queue_tx_hcmd(struct ipw_priv *priv, struct ipw_host_cmd *cmd)
{
	struct ipw_tx_queue *txq = &priv->txq[priv->hw_setting.cmd_queue_no];
	struct ipw_queue *q = &txq->q;
	u8 *tfd;
	u32 *control_flags;
	struct ipw_cmd *out_cmd;
	u32 idx = 0;
	u16 fix_size = (u16) (cmd->meta.len + sizeof(out_cmd->hdr));
	dma_addr_t phys_addr;
	u8 fifo = priv->hw_setting.cmd_queue_no;
	int rc;
	int pad;
	u16 count;

	/* If any of the command structures end up being larger than
	 * the TFD_MAX_PAYLOAD_SIZE, and it sent as a 'small' command then
	 * we will need to increase the size of the TFD entries */
	BUG_ON((fix_size > TFD_MAX_PAYLOAD_SIZE)
	       && is_cmd_small(cmd));
	if (ipw_queue_space(q) < (is_cmd_sync(cmd) ? 1 : 2)) {
		IPW_ERROR("No space for Tx\n");
		return -ENOSPC;
	}
	tfd = &txq->bd[q->first_empty * q->element_size];
	memset(tfd, 0, q->element_size);

	control_flags = (u32 *) tfd;

	idx = get_next_cmd_index(q, q->first_empty,
				 cmd->meta.flags & CMD_SIZE_HUGE);
	out_cmd = &txq->cmd[idx];

	out_cmd->hdr.cmd = cmd->id;
	memcpy(&out_cmd->meta, &cmd->meta, sizeof(cmd->meta));
	memcpy(&out_cmd->cmd.payload, cmd->data, cmd->meta.len);

	/* At this point, the out_cmd now has all of the incoming cmd
	 * information */

	out_cmd->hdr.flags = 0;
	out_cmd->hdr.sequence = FIFO_TO_SEQ(fifo) |
	    INDEX_TO_SEQ(q->first_empty);
	if (out_cmd->meta.flags & CMD_SIZE_HUGE)
		out_cmd->hdr.sequence |= SEQ_HUGE_FRAME;

	phys_addr = txq->dma_addr_cmd + sizeof(txq->cmd[0]) * idx +
	    offsetof(struct ipw_cmd, hdr);

	attach_buffer_to_tfd_frame(priv, tfd, phys_addr, fix_size);

	if (priv->is_3945) {
		pad = U32_PAD(out_cmd->meta.len);
		count = TFD_CTL_COUNT_GET(*control_flags);
		*control_flags = TFD_CTL_COUNT_SET(count) |
		    TFD_CTL_PAD_SET(pad);
	}

	if ((out_cmd->hdr.cmd != 0x23 &&
	     out_cmd->hdr.cmd != 0x24 && out_cmd->hdr.cmd != 0x22)) {
		IPW_DEBUG_HC("Sending command %s (#%x), seq: 0x%04X, "
			     "%d bytes at %d[%d]:%d\n",
			     get_cmd_string(out_cmd->hdr.cmd),
			     out_cmd->hdr.cmd, out_cmd->hdr.sequence,
			     fix_size, q->first_empty, idx, fifo);
		printk_buf(IPW_DL_HOST_COMMAND, cmd->data, cmd->len);
	}

	txq->need_update = 1;
	rc = priv->hw_setting.tx_queue_update_wr_ptr(priv, txq,
						     priv->hw_setting.
						     cmd_queue_no, 0);
	q->first_empty = ipw_queue_inc_wrap(q->first_empty, q->n_bd);
	ipw_tx_queue_update_write_ptr(priv, txq, priv->hw_setting.cmd_queue_no);

	if (rc)
		return rc;

	return 0;
}

/* For active scan, listen ACTIVE_DWELL_TIME (msec) on each channel after
 * sending probe req.  This should be set long enough to hear probe responses
 * from more than one AP.  */
#define IPW_ACTIVE_DWELL_TIME_24    (20)	/* all times in msec */
#define IPW_ACTIVE_DWELL_TIME_52    (10)

/* For faster active scanning, scan will move to the next channel if fewer than
 * PLCP_QUIET_THRESH packets are heard on this channel within
 * ACTIVE_QUIET_TIME after sending probe request.  This shortens the dwell
 * time if it's a quiet channel (nothing responded to our probe, and there's
 * no other traffic).
 * Disable "quiet" feature by setting PLCP_QUIET_THRESH to 0. */
#define IPW_PLCP_QUIET_THRESH       (1)	/* packets */
#define IPW_ACTIVE_QUIET_TIME       (5)	/* msec */

/* For passive scan, listen PASSIVE_DWELL_TIME (msec) on each channel.
 * Must be set longer than active dwell time.
 * For the most reliable scan, set > AP beacon interval (typically 100msec). */
#define IPW_PASSIVE_DWELL_TIME_24   (20)	/* all times in msec */
#define IPW_PASSIVE_DWELL_TIME_52   (10)
#define IPW_PASSIVE_DWELL_BASE      (100)
#define IPW_CHANNEL_TUNE_TIME       5
static void ipw_abort_scan(struct ipw_priv *priv)
{
	priv->status &= ~STATUS_SCAN_PENDING;
	priv->status |= STATUS_SCAN_ABORTING;
	ipw_send_scan_abort(priv);
}

static inline u16 ipw_get_active_dwell_time(struct ipw_priv *priv, int phymode)
{
	if (phymode == MODE_IEEE80211A)
		return IPW_ACTIVE_DWELL_TIME_52;
	else
		return IPW_ACTIVE_DWELL_TIME_24;
}

static inline u16 ipw_get_passive_dwell_time(struct ipw_priv *priv, int phymode)
{
	u16 active = ipw_get_active_dwell_time(priv, phymode);
	u16 passive = (phymode != MODE_IEEE80211A) ?
	    IPW_PASSIVE_DWELL_BASE + IPW_PASSIVE_DWELL_TIME_24 :
	    IPW_PASSIVE_DWELL_BASE + IPW_PASSIVE_DWELL_TIME_52;

	if (ipw_is_associated(priv)) {
		struct ieee80211_conf *conf = NULL;

		conf = ieee80211_get_hw_conf(priv->ieee);

		/* If we're associated, we clamp the maximum passive
		 * dwell time to be 98% of the beacon interval (minus
		 * 2 * channel tune time) */
		passive = conf->beacon_int;
		if (passive > IPW_PASSIVE_DWELL_BASE)
			passive = IPW_PASSIVE_DWELL_BASE;
		passive = (passive * 98) / 100 - IPW_CHANNEL_TUNE_TIME * 2;
	}

	if (passive <= active)
		passive = active + 1;

	return passive;
}

static int ipw_get_channels_for_scan(struct ipw_priv *priv, int phymode,
				     u8 is_active, u8 direct_mask,
				     struct ipw_scan_channel *scan_ch)
{
	const struct ieee80211_channel *channels = NULL;
	const struct ieee80211_hw_mode *hw_mode;
	struct ipw_channel_info *ch_info;
	u16 passive_dwell = 0;
	u16 active_dwell = 0;
	int added, i;

	hw_mode = ipw_get_hw_mode(priv, phymode);
	if (!hw_mode)
		return 0;

	channels = hw_mode->channels;

	active_dwell = ipw_get_active_dwell_time(priv, phymode);
	passive_dwell = ipw_get_passive_dwell_time(priv, phymode);

	for (i = 0, added = 0; i < hw_mode->num_channels; i++) {
		if (channels[i].chan == priv->active_conf.channel) {
			if (ipw_is_associated(priv)) {
				IPW_DEBUG_SCAN
				    ("Skipping current channel %d\n",
				     priv->active_conf.channel);
				continue;
			}
		} else if (priv->only_active_channel)
			continue;

		scan_ch->channel = channels[i].chan;

		ch_info = ipw_get_channel_info(priv, phymode, scan_ch->channel);
		if (!is_channel_valid(ch_info)) {
			IPW_DEBUG_SCAN("Channel %d is INVALID for this SKU.\n",
				       scan_ch->channel);
			continue;
		}

		if (!is_active || is_channel_passive(ch_info) ||
		    !(channels[i].flag & IEEE80211_CHAN_W_ACTIVE_SCAN))
			scan_ch->type = 0;	/* passive */
		else
			scan_ch->type = 1;	/* active */

		if (scan_ch->type & 1)
			scan_ch->type |= (direct_mask << 1);

		if (is_channel_narrow(ch_info))
			scan_ch->type |= (1 << 7);

		scan_ch->active_dwell = active_dwell;
		scan_ch->passive_dwell = passive_dwell;

		/* Set power levels to defaults */
		scan_ch->tpc.dsp_atten = 110;	/*scan_pwr_info->tpc.dsp_atten; */

		/*scan_pwr_info->tpc.tx_gain; */
		if (phymode == MODE_IEEE80211A) {
			scan_ch->tpc.tx_gain = ((1 << 5) | (3 << 3)) | 3;
		} else {
			scan_ch->tpc.tx_gain = ((1 << 5) | (5 << 3));
			/* NOTE: if we were doing 6Mb OFDM for scans we'd use
			 * power level
			 scan_ch->tpc.tx_gain = ((1<<5) | (2 << 3)) | 3;
			 */
		}

		IPW_DEBUG_SCAN("Scanning %d [%s %d]\n",
			       scan_ch->channel,
			       (scan_ch->type & 1) ? "ACTIVE" : "PASSIVE",
			       (scan_ch->
				type & 1) ? active_dwell : passive_dwell);

		scan_ch++;
		added++;
	}

	IPW_DEBUG_SCAN("total channels to scan %d \n", added);
	return added;
}

static void ipw_bg_request_scan(struct work_struct *work)
{

	struct ipw_host_cmd cmd = {
		.id = REPLY_SCAN_CMD,
		.len = sizeof(struct ipw_scan_cmd),
		.meta.flags = CMD_SIZE_HUGE,
	};
	struct ipw_priv *priv =
	    container_of(work, struct ipw_priv, request_scan.work);
	int rc = 0;
	struct ipw_scan_cmd *scan;
	struct ieee80211_hw_mode *hw_mode = NULL;
	struct ieee80211_conf *conf = NULL;
	u8 direct_mask;
	int phymode;

	conf = ieee80211_get_hw_conf(priv->ieee);

	if (!ipw_is_ready(priv)) {
		IPW_WARNING("request scan called when driver not ready.\n");
		return;
	}

	mutex_lock(&priv->mutex);

	/* This should never be called or scheduled if there is currently
	 * a scan active in the hardware. */
	if (priv->status & STATUS_SCAN_HW) {
		IPW_DEBUG_INFO
		    ("Multiple concurrent scan requests in parallel. "
		     "Ignoring second request.\n");
		rc = -EIO;
		goto done;
	}

	if (priv->status & STATUS_EXIT_PENDING) {
		IPW_DEBUG_SCAN("Aborting scan due to device shutdown\n");
		priv->status |= STATUS_SCAN_PENDING;
		goto done;
	}

	if (priv->status & STATUS_SCAN_ABORTING) {
		IPW_DEBUG_HC("Scan request while abort pending.  Queuing.\n");
		priv->status |= STATUS_SCAN_PENDING;
		goto done;
	}

	if (priv->status & STATUS_RF_KILL_MASK) {
		IPW_DEBUG_HC("Aborting scan due to RF Kill activation\n");
		priv->status |= STATUS_SCAN_PENDING;
		goto done;
	}

	if (!(priv->status & STATUS_READY)) {
		IPW_DEBUG_HC("Scan request while uninitialized.  Queuing.\n");
		priv->status |= STATUS_SCAN_PENDING;
		goto done;
	}

	if (!priv->scan_bands) {
		IPW_DEBUG_HC("Aborting scan due to no requested bands.\n");
		goto done;
	}

	if (!priv->scan) {
		priv->scan = kmalloc(sizeof(struct ipw_scan_cmd) +
				     IPW_MAX_SCAN_SIZE, GFP_ATOMIC);
		if (!priv->scan) {
			rc = -ENOMEM;
			goto done;
		}
	}
	scan = priv->scan;
	memset(scan, 0, sizeof(struct ipw_scan_cmd) + IPW_MAX_SCAN_SIZE);

	scan->quiet_plcp_th = IPW_PLCP_QUIET_THRESH;
	scan->quiet_time = IPW_ACTIVE_QUIET_TIME;

	if (ipw_is_associated(priv)) {
		u16 interval = conf->beacon_int;
		u32 extra;

		IPW_DEBUG_INFO("Scanning while associated...\n");
		scan->suspend_time = 100;
		scan->max_out_time = 600 * 1024;
		if (interval) {
			/*
			 * suspend time format:
			 *  0-19: beacon interval in usec (time before exec.)
			 * 20-23: 0
			 * 24-31: number of beacons (suspend between channels)
			 */

			extra = (scan->suspend_time / interval) << 24;
			scan->suspend_time = 0xFF0FFFFF & (extra |
							   ((scan->
							     suspend_time
							     % interval)
							    * 1024));
		}
	}

	/* We should add the ability for user to lock to PASSIVE ONLY */
	if (priv->one_direct_scan) {
		IPW_DEBUG_SCAN
		    ("Kicking off one direct scan for '%s'\n",
		     iwl_escape_essid(priv->direct_ssid,
				      priv->direct_ssid_len));
		scan->direct_scan[0].id = WLAN_EID_SSID;
		scan->direct_scan[0].len = priv->direct_ssid_len;
		memcpy(scan->direct_scan[0].ssid,
		       priv->direct_ssid, priv->direct_ssid_len);
		direct_mask = 1;
	} else if (!ipw_is_associated(priv)) {
		scan->direct_scan[0].id = WLAN_EID_SSID;
		scan->direct_scan[0].len = priv->essid_len;
		memcpy(scan->direct_scan[0].ssid, priv->essid, priv->essid_len);
		direct_mask = 1;
	} else {
		direct_mask = 0;
	}

	/* We don't build a direct scan probe request; the uCode will do
	 * that based on the direct_mask added to each channel entry */
	scan->tx_cmd.len = ipw_fill_probe_req(priv,
					      (struct ieee80211_mgmt *)scan->
					      data,
					      IPW_MAX_SCAN_SIZE - sizeof(scan),
					      0);
	scan->tx_cmd.tx_flags = TX_CMD_FLG_SEQ_CTL_MSK;
	scan->tx_cmd.sta_id = priv->hw_setting.broadcast_id;
	scan->tx_cmd.u.life_time = TX_CMD_LIFE_TIME_INFINITE;

	/* flags + rate selection */

	switch (priv->scan_bands) {
	case 2:
		scan->flags = RXON_FLG_BAND_24G_MSK | RXON_FLG_AUTO_DETECT_MSK;
		scan->tx_cmd.rate = R_1M;
		scan->good_CRC_th = 0;
		hw_mode = ipw_get_hw_mode(priv, MODE_IEEE80211G);
		phymode = MODE_IEEE80211G;
		break;

	case 1:
		scan->tx_cmd.rate = R_6M;
		scan->good_CRC_th = IPW_GOOD_CRC_TH;
		hw_mode = ipw_get_hw_mode(priv, MODE_IEEE80211A);
		phymode = MODE_IEEE80211A;
		break;

	default:
		IPW_WARNING("Invalid scan band count\n");
		goto done;
	}

	if (!hw_mode) {
		IPW_WARNING("Could not obtain hw_mode in scan.  Aborting.\n");
		goto done;
	}

	scan->flags |= ipw_get_antenna_flags(priv);

	if (priv->iw_mode == IEEE80211_IF_TYPE_MNTR)
		scan->filter_flags = RXON_FILTER_PROMISC_MSK;

	if (direct_mask)
		IPW_DEBUG_SCAN
		    ("Initiating direct scan for %s.\n",
		     iwl_escape_essid(priv->essid, priv->essid_len));
	else
		IPW_DEBUG_SCAN("Initiating indirect scan.\n");

	scan->channel_count =
	    ipw_get_channels_for_scan(priv, phymode, 1 /* active */ ,
				      direct_mask,
				      (void *)&scan->data[scan->tx_cmd.len]);

	cmd.len += scan->tx_cmd.len +
	    scan->channel_count * sizeof(struct ipw_scan_channel);
	cmd.data = scan;
	scan->len = cmd.len;

	priv->status |= STATUS_SCAN_HW;

	rc = ipw_send_cmd(priv, &cmd);
	if (rc)
		goto done;

	queue_delayed_work(priv->workqueue, &priv->scan_check,
			   IPW_SCAN_CHECK_WATCHDOG);

	priv->status &= ~STATUS_SCAN_PENDING;

	goto done;

      done:
	if (!rc)
		ipw_update_link_led(priv);

	mutex_unlock(&priv->mutex);
}

static void ipw_bg_abort_scan(struct work_struct *work)
{
	struct ipw_priv *priv = container_of(work, struct ipw_priv, abort_scan);

	if (priv->status & STATUS_EXIT_PENDING)
		return;

	mutex_lock(&priv->mutex);
	if (priv->status & STATUS_READY)
		ipw_abort_scan(priv);
	mutex_unlock(&priv->mutex);
}

#if 0
/* QoS -- untested and not fully plumbed */
static int ipw_send_qos_params_command(struct ipw_priv *priv,
				       struct ipw_qosparam_cmd *qos)
{
	return 0;
}
#endif

static void ipw_set_default_qos(struct ipw_priv *priv)
{
	u16 cw_min = 15;
	u16 cw_max = 1023;
	u8 aifs = 2;
	u16 burst_time = 0;
	u8 is_legacy = 0;
	unsigned long flags;
	int i;

	spin_lock_irqsave(&priv->lock, flags);
	priv->qos_data.qos_active = 0;
	if (priv->iw_mode == IEEE80211_IF_TYPE_IBSS) {
		if (priv->qos_data.qos_enable)
			priv->qos_data.qos_active = IPW_QOS_WMM;
		if (!(priv->active_rate & 0xfff0)) {
			cw_min = 31;
			is_legacy = 1;
		}
	} else {

		if (!(priv->active_rxon.flags & RXON_FLG_SHORT_SLOT_MSK)) {
			cw_min = 31;
			is_legacy = 1;
		}
	}

	if (priv->qos_data.qos_active)
		aifs = 3;

	priv->qos_data.def_qos_parm.ac[0].dot11CWmin = cw_min;
	priv->qos_data.def_qos_parm.ac[0].dot11CWmax = cw_max;
	priv->qos_data.def_qos_parm.ac[0].dot11AIFSN = aifs;
	priv->qos_data.def_qos_parm.ac[0].edca_txop = burst_time;
	priv->qos_data.def_qos_parm.ac[0].reserved1 = 0;

	if (priv->qos_data.qos_active) {
		i = 1;
		priv->qos_data.def_qos_parm.ac[i].dot11CWmin = cw_min;
		priv->qos_data.def_qos_parm.ac[i].dot11CWmax = cw_max;
		priv->qos_data.def_qos_parm.ac[i].dot11AIFSN = 7;
		priv->qos_data.def_qos_parm.ac[i].edca_txop = burst_time;
		priv->qos_data.def_qos_parm.ac[i].reserved1 = 0;

		i = 2;
		priv->qos_data.def_qos_parm.ac[i].dot11CWmin =
		    (cw_min + 1) / 2 - 1;
		priv->qos_data.def_qos_parm.ac[i].dot11CWmax = cw_max;
		priv->qos_data.def_qos_parm.ac[i].dot11AIFSN = 2;
		if (is_legacy)
			priv->qos_data.def_qos_parm.ac[i].edca_txop = 6016;
		else
			priv->qos_data.def_qos_parm.ac[i].edca_txop = 3008;
		priv->qos_data.def_qos_parm.ac[i].reserved1 = 0;

		i = 3;
		priv->qos_data.def_qos_parm.ac[i].dot11CWmin =
		    (cw_min + 1) / 4 - 1;
		priv->qos_data.def_qos_parm.ac[i].dot11CWmax =
		    (cw_max + 1) / 2 - 1;
		priv->qos_data.def_qos_parm.ac[i].dot11AIFSN = 2;
		priv->qos_data.def_qos_parm.ac[i].reserved1 = 0;
		if (is_legacy)
			priv->qos_data.def_qos_parm.ac[i].edca_txop = 3264;
		else
			priv->qos_data.def_qos_parm.ac[i].edca_txop = 1504;

	} else {
		for (i = 1; i < 4; i++) {
			priv->qos_data.def_qos_parm.ac[i].dot11CWmin = cw_min;
			priv->qos_data.def_qos_parm.ac[i].dot11CWmax = cw_max;
			priv->qos_data.def_qos_parm.ac[i].dot11AIFSN = aifs;
			priv->qos_data.def_qos_parm.ac[i].edca_txop =
			    burst_time;
			priv->qos_data.def_qos_parm.ac[i].reserved1 = 0;
		}
	}
	IPW_DEBUG_QOS("set QoS to default \n");

	spin_unlock_irqrestore(&priv->lock, flags);
}

static void ipw_qos_activate(struct ipw_priv *priv, u8 force)
{
	unsigned long flags;

	if (priv->status & STATUS_EXIT_PENDING)
		return;

	if (!priv->qos_data.qos_enable)
		return;

	spin_lock_irqsave(&priv->lock, flags);
	priv->qos_data.def_qos_parm.qos_flags = 0;

	if (priv->qos_data.qos_cap.q_AP.queue_request &&
	    !priv->qos_data.qos_cap.q_AP.txop_request)
		priv->qos_data.def_qos_parm.qos_flags |=
		    QOS_PARAM_FLG_TXOP_TYPE_MSK;

	if (priv->qos_data.qos_active)
		priv->qos_data.def_qos_parm.qos_flags |=
		    QOS_PARAM_FLG_UPDATE_EDCA_MSK;

	spin_unlock_irqrestore(&priv->lock, flags);

	if (force || ipw_is_associated(priv)) {
		IPW_DEBUG_QOS("send QoS cmd with Qos active %d \n",
			      priv->qos_data.qos_active);
/*		ipw_send_cmd_pdu(priv, REPLY_QOS_PARAM,
				 sizeof(struct ipw_qosparam_cmd),
				 &(priv->qos_data.def_qos_parm));
*/ }
}

static void ipw_bg_qos_activate(struct work_struct *work)
{
	struct ipw_priv *priv =
	    container_of(work, struct ipw_priv, qos_activate);

	mutex_lock(&priv->mutex);
	ipw_qos_activate(priv, 0);
	mutex_unlock(&priv->mutex);
}

static void ipw_bg_report_work(struct work_struct *work)
{
	struct ipw_priv *priv =
	    container_of(work, struct ipw_priv, report_work);

	if (priv->status & STATUS_EXIT_PENDING)
		return;

	mutex_lock(&priv->mutex);
	mutex_unlock(&priv->mutex);
}

/***************************************************************/

static int ipw_associate(struct ipw_priv *priv)
{
#if 0
	struct ipw_priv *priv =
	    container_of(work, struct ipw_priv, associate.work);
	int rc = 0;
	struct ipw_rxon_cmd *rxon = &priv->rxon;

	ipw_update_link_led(priv);

	if (ipw_is_associated(priv)) {
		IPW_DEBUG_ASSOC
		    ("Not attempting association (already in progress)\n");
		return 0;
	}

	if (!ipw_is_alive(priv)) {
		IPW_DEBUG_ASSOC("Not attempting association (not "
				"initialized)\n");
		return 0;
	}

	if (priv->status & STATUS_SCANNING) {
		IPW_DEBUG_ASSOC("Not attempting association (scanning)\n");
		return 0;
	}

	if (priv->status & STATUS_EXIT_PENDING) {
		IPW_DEBUG_ASSOC
		    ("Not attempting association driver unloading\n");
		return 0;
	}

	ipw_scan_cancel(priv);

	priv->status &= ~STATUS_SECURITY_UPDATED;

	ipw_set_rxon_conf(priv, 1);
	ipw_commit_rxon(priv);

	if (rc)
		return rc;

	if (rxon->dev_type == RXON_DEV_TYPE_SNIFFER)
		return 0;

	if (priv->iw_mode == IEEE80211_IF_TYPE_IBSS)
		priv->assoc_id = 0;

	ipw_init_rate_scaling(priv);
	if (!rc)
		ipw_send_statistics_request(priv);

	return rc;
#endif
	return 0;
}

#define IPW_FORCE_TXPOWR_CALIB  (180 * HZ)

static void ipw_set_decrypted_flag(struct ipw_priv *priv,
				   struct sk_buff *skb,
				   u32 decrypt_res,
				   struct ieee80211_rx_status *stats)
{
	u16 fc =
	    le16_to_cpu(((struct ieee80211_hdr *)skb->data)->frame_control);

	if (priv->active_rxon.filter_flags & RXON_FILTER_DIS_DECRYPT_MSK)
		return;

	if (!(fc & IEEE80211_FCTL_PROTECTED))
		return;

	IPW_DEBUG_RX("decrypt_res:0x%x\n", decrypt_res);
	switch (decrypt_res & RX_RES_STATUS_SEC_TYPE_MSK) {
	case RX_RES_STATUS_SEC_TYPE_TKIP:
		if ((decrypt_res & RX_RES_STATUS_DECRYPT_TYPE_MSK) ==
		    RX_RES_STATUS_BAD_ICV_MIC)
			stats->flag |= RX_FLAG_MMIC_ERROR;
	case RX_RES_STATUS_SEC_TYPE_WEP:
	case RX_RES_STATUS_SEC_TYPE_CCMP:
		if ((decrypt_res & RX_RES_STATUS_DECRYPT_TYPE_MSK) ==
		    RX_RES_STATUS_DECRYPT_OK) {
			IPW_DEBUG_RX("hw decrypt successfully!!!\n");
			stats->flag |= RX_FLAG_DECRYPTED;
		}
		break;

	default:
		break;
	}
}

static void ipw_handle_data_packet_monitor(struct ipw_priv *priv, struct ipw_rx_mem_buffer
					   *rxb, void *data, short len,
					   struct ieee80211_rx_status *stats);

static void ipw_handle_data_packet(struct ipw_priv *priv, int is_data,
				   struct ipw_rx_mem_buffer *rxb,
				   struct ieee80211_rx_status *stats)
{
	struct ieee80211_hdr *hdr;
	struct ipw_rx_packet *pkt = (struct ipw_rx_packet *)rxb->skb->data;
	struct ipw_rx_frame_hdr *rx_hdr = IPW_RX_HDR(pkt);
	struct ipw_rx_frame_end *rx_end = IPW_RX_END(pkt);
	short len = le16_to_cpu(rx_hdr->len);

	/* We received data from the HW, so stop the watchdog */
	ieee80211_netif_oper(priv->ieee, NETIF_UPDATE_TX_START);
	if (unlikely((len + IPW_RX_FRAME_SIZE) > skb_tailroom(rxb->skb))) {
		priv->wstats.discard.misc++;
		IPW_DEBUG_DROP("Corruption detected! Oh no!\n");
		return;
	}

	/* We only process data packets if the interface is open */
	if (unlikely
	    (!priv->netdev_registered
	     || (is_data && !netif_running(priv->net_dev)))) {
		priv->wstats.discard.misc++;
		IPW_DEBUG_DROP("Dropping packet while interface is not up.\n");
		return;
	}
	if (priv->iw_mode == IEEE80211_IF_TYPE_MNTR) {
		if (param_hwcrypto)
			ipw_set_decrypted_flag(priv, rxb->skb, rx_end->status,
					       stats);
		ipw_handle_data_packet_monitor(priv, rxb, IPW_RX_DATA(pkt), len,
					       stats);
		return;
	}

	stats->flag = 0;
	skb_reserve(rxb->skb, (void *)rx_hdr->payload - (void *)pkt);
	/* Set the size of the skb to the size of the frame */
	skb_put(rxb->skb, le16_to_cpu(rx_hdr->len));

	hdr = (void *)rxb->skb->data;
	priv->rx_bytes += rxb->skb->len -
	    ieee80211_get_hdrlen(le16_to_cpu(hdr->frame_control));

	if (param_hwcrypto)
		ipw_set_decrypted_flag(priv, rxb->skb, rx_end->status, stats);

	ieee80211_rx_irqsafe(priv->ieee, rxb->skb, stats);
	rxb->skb = NULL;
	priv->led_packets += rx_hdr->len;
	ipw_setup_activity_timer(priv);
}

#define IPW_PACKET_RETRY_TIME HZ

static inline int is_duplicate_packet(struct ipw_priv *priv, struct ieee80211_hdr
				      *header)
{
	u16 sc = le16_to_cpu(header->seq_ctrl);
	u16 seq = WLAN_GET_SEQ_SEQ(sc);
	u16 frag = WLAN_GET_SEQ_FRAG(sc);
	u16 *last_seq, *last_frag;
	unsigned long *last_time;

	switch (priv->iw_mode) {
	case IEEE80211_IF_TYPE_IBSS:
		{
			struct list_head *p;
			struct ipw_ibss_seq *entry = NULL;
			u8 *mac = header->addr2;
			int index = mac[5] % IPW_IBSS_MAC_HASH_SIZE;

			__list_for_each(p, &priv->ibss_mac_hash[index]) {
				entry =
				    list_entry(p, struct ipw_ibss_seq, list);
				if (!compare_ether_addr(entry->mac, mac))
					break;
			}
			if (p == &priv->ibss_mac_hash[index]) {
				entry = kzalloc(sizeof(*entry), GFP_ATOMIC);
				if (!entry) {
					IPW_ERROR
					    ("Cannot malloc new mac entry\n");
					return 0;
				}
				memcpy(entry->mac, mac, ETH_ALEN);
				entry->seq_num = seq;
				entry->frag_num = frag;
				entry->packet_time = jiffies;
				list_add(&entry->list,
					 &priv->ibss_mac_hash[index]);
				return 0;
			}
			last_seq = &entry->seq_num;
			last_frag = &entry->frag_num;
			last_time = &entry->packet_time;
			break;
		}
	case IEEE80211_IF_TYPE_STA:
		last_seq = &priv->last_seq_num;
		last_frag = &priv->last_frag_num;
		last_time = &priv->last_packet_time;
		break;
	default:
		return 0;
	}
	if ((*last_seq == seq) &&
	    time_after(*last_time + IPW_PACKET_RETRY_TIME, jiffies)) {
		if (*last_frag == frag)
			goto drop;
		if (*last_frag + 1 != frag)
			/* out-of-order fragment */
			goto drop;
	} else
		*last_seq = seq;

	*last_frag = frag;
	*last_time = jiffies;
	return 0;

      drop:
	/* Comment this line now since we observed the card receives
	 * duplicate packets but the FCTL_RETRY bit is not set in the
	 * IBSS mode with fragmentation enabled.
	 BUG_ON(!(le16_to_cpu(header->frame_control) & IEEE80211_FCTL_RETRY)); */
	return 1;
}

enum ieee80211_radiotap_type {
	IEEE80211_RADIOTAP_TSFT = 0,
	IEEE80211_RADIOTAP_FLAGS = 1,
	IEEE80211_RADIOTAP_RATE = 2,
	IEEE80211_RADIOTAP_CHANNEL = 3,
	IEEE80211_RADIOTAP_FHSS = 4,
	IEEE80211_RADIOTAP_DBM_ANTSIGNAL = 5,
	IEEE80211_RADIOTAP_DBM_ANTNOISE = 6,
	IEEE80211_RADIOTAP_LOCK_QUALITY = 7,
	IEEE80211_RADIOTAP_TX_ATTENUATION = 8,
	IEEE80211_RADIOTAP_DB_TX_ATTENUATION = 9,
	IEEE80211_RADIOTAP_DBM_TX_POWER = 10,
	IEEE80211_RADIOTAP_ANTENNA = 11,
	IEEE80211_RADIOTAP_DB_ANTSIGNAL = 12,
	IEEE80211_RADIOTAP_DB_ANTNOISE = 13,
	IEEE80211_RADIOTAP_EXT = 31,
};

/* Channel flags. */
#define IEEE80211_CHAN_TURBO    0x0010	/* Turbo channel */
#define IEEE80211_CHAN_CCK      0x0020	/* CCK channel */
#define IEEE80211_CHAN_OFDM     0x0040	/* OFDM channel */
#define IEEE80211_CHAN_2GHZ     0x0080	/* 2 GHz spectrum channel. */
#define IEEE80211_CHAN_5GHZ     0x0100	/* 5 GHz spectrum channel */
#define IEEE80211_CHAN_PASSIVE  0x0200	/* Only passive scan allowed */
#define IEEE80211_CHAN_DYN      0x0400	/* Dynamic CCK-OFDM channel */
#define IEEE80211_CHAN_GFSK     0x0800	/* GFSK channel (FHSS PHY) */

/* For IEEE80211_RADIOTAP_FLAGS */
#define IEEE80211_RADIOTAP_F_CFP        0x01	/* sent/received
						 * during CFP
						 */
#define IEEE80211_RADIOTAP_F_SHORTPRE   0x02	/* sent/received
						 * with short
						 * preamble
						 */
#define IEEE80211_RADIOTAP_F_WEP        0x04	/* sent/received
						 * with WEP encryption
						 */
#define IEEE80211_RADIOTAP_F_FRAG       0x08	/* sent/received
						 * with fragmentation
						 */
#define IEEE80211_RADIOTAP_F_FCS        0x10	/* frame includes FCS */
#define IEEE80211_RADIOTAP_F_DATAPAD    0x20	/* frame has padding between
						 * 802.11 header and payload
						 * (to 32-bit boundary)
						 */
#define PKTHDR_RADIOTAP_VERSION                0

static void ipw_handle_data_packet_monitor(struct ipw_priv *priv, struct ipw_rx_mem_buffer
					   *rxb, void *data, short len,
					   struct ieee80211_rx_status *stats)
{
	struct ipw_rt_hdr *ipw_rt;

	/* First cache any information we need before we overwrite
	 * the information provided in the skb from the hardware */
	s8 signal = stats->ssi;
	s8 noise = 0;
	u16 phy_flags = stats->flag;
	u16 channel = cpu_to_le16(stats->channel);
	u8 rate = stats->rate;
	u64 tsf = stats->mactime;

	stats->flag = 0;
	/* We received data from the HW, so stop the watchdog */
	if (len > priv->hw_setting.rx_buffer_size - sizeof(struct ipw_rt_hdr)) {
		/* FIXME: Should alloc bigger skb instead */
		priv->wstats.discard.misc++;
		IPW_DEBUG_DROP("Dropping too large packet in monitor\n");
		return;
	}

	/* copy the frame data to write after where the radiotap header goes */
	ipw_rt = (void *)rxb->skb->data;
	memmove(ipw_rt->payload, data, len);

	/* Zero the radiotap static buffer  ...  We only need to zero the bytes
	 * NOT part of our real header, saves a little time.
	 *
	 * No longer necessary since we fill in all our data.  Purge before
	 * merging patch officially.
	 * memset(rxb->skb->data + sizeof(struct ipw_rt_hdr), 0,
	 *        IEEE80211_RADIOTAP_HDRLEN - sizeof(struct ipw_rt_hdr));
	 */

	ipw_rt->rt_hdr.it_version = PKTHDR_RADIOTAP_VERSION;
	ipw_rt->rt_hdr.it_pad = 0;	/* always good to zero */
	ipw_rt->rt_hdr.it_len = sizeof(*ipw_rt);	/* total header+data */

	/* Set the size of the skb to the size of the frame */
	skb_put(rxb->skb, ipw_rt->rt_hdr.it_len + len);

	/* Big bitfield of all the fields we provide in radiotap */
	ipw_rt->rt_hdr.it_present =
	    ((1 << IEEE80211_RADIOTAP_TSFT) |
	     (1 << IEEE80211_RADIOTAP_FLAGS) |
	     (1 << IEEE80211_RADIOTAP_RATE) |
	     (1 << IEEE80211_RADIOTAP_CHANNEL) |
	     (1 << IEEE80211_RADIOTAP_DBM_ANTSIGNAL) |
	     (1 << IEEE80211_RADIOTAP_DBM_ANTNOISE) |
	     (1 << IEEE80211_RADIOTAP_ANTENNA));

	/* Zero the flags, we'll add to them as we go */
	ipw_rt->rt_flags = 0;

	ipw_rt->rt_tsf = tsf;

	/* Convert to DBM */
	ipw_rt->rt_dbmsignal = signal;
	ipw_rt->rt_dbmnoise = noise;

	/* Convert the channel data and set the flags */
	ipw_rt->rt_channel = channel;
	if (!(phy_flags & RX_RES_PHY_FLAGS_BAND_24_MSK)) {
		ipw_rt->rt_chbitmask =
		    cpu_to_le16((IEEE80211_CHAN_OFDM | IEEE80211_CHAN_5GHZ));
	} else if (phy_flags & RX_RES_PHY_FLAGS_MOD_CCK_MSK) {
		ipw_rt->rt_chbitmask =
		    cpu_to_le16((IEEE80211_CHAN_CCK | IEEE80211_CHAN_2GHZ));
	} else {		/* 802.11g */
		ipw_rt->rt_chbitmask =
		    cpu_to_le16((IEEE80211_CHAN_OFDM | IEEE80211_CHAN_2GHZ));
	}

	/* set the rate in multiples of 500k/s */
	switch (rate) {
	case IPW_TX_RATE_1MB:
		ipw_rt->rt_rate = 2;
		break;
	case IPW_TX_RATE_2MB:
		ipw_rt->rt_rate = 4;
		break;
	case IPW_TX_RATE_5MB:
		ipw_rt->rt_rate = 10;
		break;
	case IPW_TX_RATE_6MB:
		ipw_rt->rt_rate = 12;
		break;
	case IPW_TX_RATE_9MB:
		ipw_rt->rt_rate = 18;
		break;
	case IPW_TX_RATE_11MB:
		ipw_rt->rt_rate = 22;
		break;
	case IPW_TX_RATE_12MB:
		ipw_rt->rt_rate = 24;
		break;
	case IPW_TX_RATE_18MB:
		ipw_rt->rt_rate = 36;
		break;
	case IPW_TX_RATE_24MB:
		ipw_rt->rt_rate = 48;
		break;
	case IPW_TX_RATE_36MB:
		ipw_rt->rt_rate = 72;
		break;
	case IPW_TX_RATE_48MB:
		ipw_rt->rt_rate = 96;
		break;
	case IPW_TX_RATE_54MB:
		ipw_rt->rt_rate = 108;
		break;
	default:
		ipw_rt->rt_rate = 0;
		break;
	}

	/* antenna number */
	ipw_rt->rt_antenna = (phy_flags & RX_RES_PHY_FLAGS_ANTENNA_MSK) >> 4;

	/* set the preamble flag if we have it */
	if (phy_flags & RX_RES_PHY_FLAGS_SHORT_PREAMBLE_MSK)
		ipw_rt->rt_flags |= IEEE80211_RADIOTAP_F_SHORTPRE;

	IPW_DEBUG_RX("Rx packet of %d bytes.\n", rxb->skb->len);

	ieee80211_rx_irqsafe(priv->ieee, rxb->skb, stats);
	rxb->skb = NULL;
}

#define ieee80211_is_probe_response(fc) \
   ((WLAN_FC_GET_TYPE(fc) == IEEE80211_FTYPE_MGMT) && \
    ( WLAN_FC_GET_STYPE(fc) == IEEE80211_STYPE_PROBE_RESP ))

#define ieee80211_is_management(fc) \
   (WLAN_FC_GET_TYPE(fc) == IEEE80211_FTYPE_MGMT)

#define ieee80211_is_control(fc) \
   (WLAN_FC_GET_TYPE(fc) == IEEE80211_FTYPE_CTL)

#define ieee80211_is_data(fc) \
   (WLAN_FC_GET_TYPE(fc) == IEEE80211_FTYPE_DATA)

#define ieee80211_is_assoc_request(fc) \
   (WLAN_FC_GET_STYPE(fc) == IEEE80211__STYPE_ASSOC_REQ)

#define ieee80211_is_reassoc_request(fc) \
   (WLAN_FC_GET_STYPE(fc) == IEEE80211__STYPE_REASSOC_REQ)

static int is_network_packet(struct ipw_priv *priv,
			     struct ieee80211_hdr *header)
{
	/* Filter incoming packets to determine if they are targeted toward
	 * this network, discarding packets coming from ourselves */
	switch (priv->iw_mode) {
	case IEEE80211_IF_TYPE_IBSS:	/* Header: Dest. | Source    | BSSID */
		/* packets from our adapter are dropped (echo) */
		if (!compare_ether_addr(header->addr2, priv->net_dev->dev_addr))
			return 0;
		/* {broad,multi}cast packets to our IBSS go through */
		if (ipw_is_broadcast_ether_addr(header->addr1) ||
		    is_multicast_ether_addr(header->addr1))
			return !compare_ether_addr(header->addr3, priv->bssid);
		/* packets to our adapter go through */
		return !compare_ether_addr(header->addr1,
					   priv->net_dev->dev_addr);
	case IEEE80211_IF_TYPE_STA:	/* Header: Dest. | AP{BSSID} | Source */
		/* packets from our adapter are dropped (echo) */
		if (!compare_ether_addr(header->addr3, priv->net_dev->dev_addr))
			return 0;
		/* {broad,multi}cast packets to our BSS go through */
		if (ipw_is_broadcast_ether_addr(header->addr1) ||
		    is_multicast_ether_addr(header->addr1))
			return !compare_ether_addr(header->addr2, priv->bssid);
		/* packets to our adapter go through */
		return !compare_ether_addr(header->addr1,
					   priv->net_dev->dev_addr);
	}

	return 1;
}

/*
 * Reclaim Tx queue entries no more used by NIC.
 *
 * When FW advances 'R' index, all entries between old and
 * new 'R' index need to be reclaimed. As result, some free space
 * forms. If there is enough free space (> low mark), wake Tx queue.
 */
static int ipw_queue_tx_reclaim(struct ipw_priv *priv, int fifo, int index)
{
	struct ipw_tx_queue *txq = &priv->txq[fifo];
	struct ipw_queue *q = &txq->q;
	u8 is_next = 0;
	int used;
	if ((index >= q->n_bd) || (x2_queue_used(q, index) == 0)) {
		IPW_ERROR
		    ("Read index for DMA queue (%d) is out of range [0-%d) %d %d\n",
		     index, q->n_bd, q->first_empty, q->last_used);
		goto done;
	}
	index = ipw_queue_inc_wrap(index, q->n_bd);
	for (; q->last_used != index;
	     q->last_used = ipw_queue_inc_wrap(q->last_used, q->n_bd)) {
		if (is_next) {
			IPW_WARNING("command skipped\n");
			queue_work(priv->workqueue, &priv->down);
		}
		if (fifo != priv->hw_setting.cmd_queue_no) {
			ipw_txstatus_to_ieee(priv,
					     &(txq->txb[txq->q.last_used]));
			ipw_queue_tx_free_tfd(priv, txq);
			priv->tx_packets++;
		}

		is_next = 1;
	}
      done:
	if (ipw_queue_space(q) > q->low_mark && (fifo >= 0)
	    && (fifo != priv->hw_setting.cmd_queue_no)
	    && priv->netdev_registered && netif_running(priv->net_dev))
		ieee80211_netif_oper(priv->ieee, NETIF_WAKE);
	used = q->first_empty - q->last_used;
	if (used < 0)
		used += q->n_window;
	return used;
}

static void ipw_tx_complete(struct ipw_priv *priv,
			    struct ipw_rx_mem_buffer *rxb)
{
	struct ipw_rx_packet *pkt = (struct ipw_rx_packet *)rxb->skb->data;
	int fifo = SEQ_TO_FIFO(pkt->hdr.sequence);
	int index = SEQ_TO_INDEX(pkt->hdr.sequence);
	int is_huge = (pkt->hdr.sequence & SEQ_HUGE_FRAME);
	int cmd_index;
	struct ipw_cmd *cmd;
	if (fifo > MAX_REAL_TX_QUEUE_NUM)
		return;
	if (fifo != priv->hw_setting.cmd_queue_no) {
		ipw_queue_tx_reclaim(priv, fifo, index);
		return;
	}

	cmd_index =
	    get_next_cmd_index(&priv->txq[priv->hw_setting.cmd_queue_no].q,
			       index, is_huge);
	cmd = &priv->txq[priv->hw_setting.cmd_queue_no].cmd[cmd_index];
	/* Input error checking is done when commands are added to queue. */
	if (cmd->meta.flags & CMD_WANT_SKB) {
		cmd->meta.u.source->u.skb = rxb->skb;
		rxb->skb = NULL;
	} else if (cmd->meta.u.callback &&
		   !cmd->meta.u.callback(priv, cmd, rxb->skb))
		rxb->skb = NULL;

	ipw_queue_tx_reclaim(priv, fifo, index);

	/* is_cmd_sync(cmd) works with ipw_host_cmd... here we only have ipw_cmd */
	if (!(cmd->meta.flags & CMD_ASYNC)) {
		priv->status &= ~STATUS_HCMD_ACTIVE;
		wake_up_interruptible(&priv->wait_command_queue);
	}
}

/* hack this function to show different aspects of received frames,
 * including selective frame dumps.
 * group100 parameter selects whether to show 1 out of 100 good frames. */
static void ipw_report_frame(struct ipw_priv *priv,
			     struct ipw_rx_packet *pkt,
			     struct ieee80211_hdr *header, int group100)
{
	u32 to_us = 0;
	u32 print_summary = 0;
	u32 print_dump = 0;	/* set to 1 to dump all frames' contents */
	u32 hundred = 0;
	u32 dataframe = 0;

	/* these are declared without "=" to avoid compiler warnings if we
	 *   don't use them in the debug messages below */
	u16 frame_ctl;
	u16 seq_ctl;
	u16 channel;
	u16 phy_flags;
	u8 rate_sym;
	u16 length;
	u16 status;
	u16 bcn_tmr;
	u32 tsf_low;
	u64 tsf;
	u8 rssi;
	u8 agc;
	u16 sig_avg;
	u16 noise_diff;

	struct ipw_rx_frame_stats *rx_stats = IPW_RX_STATS(pkt);
	struct ipw_rx_frame_hdr *rx_hdr = IPW_RX_HDR(pkt);
	struct ipw_rx_frame_end *rx_end = IPW_RX_END(pkt);
	u8 *data = IPW_RX_DATA(pkt);

	/* MAC header */
	frame_ctl = le16_to_cpu(header->frame_control);
	seq_ctl = le16_to_cpu(header->seq_ctrl);

	/* metadata */
	channel = le16_to_cpu(rx_hdr->channel);
	phy_flags = le16_to_cpu(rx_hdr->phy_flags);
	rate_sym = rx_hdr->rate;
	length = le16_to_cpu(rx_hdr->len);

	/* end-of-frame status and timestamp */
	status = le16_to_cpu(rx_end->status);
	bcn_tmr = le32_to_cpu(rx_end->beaconTimeStamp);
	tsf_low = (u32) (rx_end->timestamp & 0x0ffffffff);
	tsf = le64_to_cpu(rx_end->timestamp);

	/* signal statistics */
	rssi = rx_stats->rssi;
	agc = rx_stats->agc;
	sig_avg = le16_to_cpu(rx_stats->sig_avg);
	noise_diff = le16_to_cpu(rx_stats->noise_diff);

	to_us = !compare_ether_addr(header->addr1, priv->net_dev->dev_addr);

	/* if data frame is to us and all is good,
	 *   (optionally) print summary for only 1 out of every 100 */
	if (to_us && (frame_ctl & ~0x4000) == 0x0208) {
		dataframe = 1;
		if (!group100) {
			print_summary = 1;	/* print each frame */
		} else if (priv->framecnt_to_us < 100) {
			priv->framecnt_to_us++;
			print_summary = 0;
		} else {
			priv->framecnt_to_us = 0;
			print_summary = 1;
			hundred = 1;
		}
	} else {
		print_summary = 1;	/* print summary for all other frames */
	}

	if (print_summary) {
		char *title;
		u32 rate;

		if (hundred)
			title = "100Frames";
		else if (frame_ctl & 0x0800)
			title = "Retry";
		else if ((frame_ctl & 0x00FF) == 0x10)
			title = "AscRsp";
		else if ((frame_ctl & 0x00FF) == 0x30)
			title = "RasRsp";
		else if ((frame_ctl & 0x00FF) == 0x50) {
			title = "PrbRsp";
			print_dump = 1;	/* dump frame contents */
		} else if ((frame_ctl & 0x00FF) == 0x80) {
			title = "Beacon";
			print_dump = 1;	/* dump frame contents */
		} else if ((frame_ctl & 0x00FF) == 0x90)
			title = "ATIM";
		else if ((frame_ctl & 0x00FF) == 0xB0)
			title = "Auth";
		else
			title = "Frame";

		switch (rate_sym) {
		case IPW_TX_RATE_1MB:
			rate = 1;
			break;
		case IPW_TX_RATE_2MB:
			rate = 2;
			break;
		case IPW_TX_RATE_5MB:
			rate = 5;
			break;
		case IPW_TX_RATE_6MB:
			rate = 6;
			break;
		case IPW_TX_RATE_9MB:
			rate = 9;
			break;
		case IPW_TX_RATE_11MB:
			rate = 11;
			break;
		case IPW_TX_RATE_12MB:
			rate = 12;
			break;
		case IPW_TX_RATE_18MB:
			rate = 18;
			break;
		case IPW_TX_RATE_24MB:
			rate = 24;
			break;
		case IPW_TX_RATE_36MB:
			rate = 36;
			break;
		case IPW_TX_RATE_48MB:
			rate = 48;
			break;
		case IPW_TX_RATE_54MB:
			rate = 54;
			break;
		default:
			rate = 0;
		}

		/* print frame summary.
		 * MAC addresses show just the last byte (for brevity),
		 *    but you can hack it to show more, if you'd like to. */
		if (dataframe) {
			IPW_DEBUG_RX
			    ("%s: mhd=0x%04x, dst=0x%02x, len=%u, rssi=%d, chnl=%d, rate=%u, \n",
			     title, frame_ctl, header->addr1[5],
			     length, rssi, channel, rate);
		} else {
			/* src/dst addresses assume managed mode */
			IPW_DEBUG_RX
			    ("%s: 0x%04x, dst=0x%02x, src=0x%02x, rssi=%u, tim=%lu usec, phy=0x%02x, chnl=%d\n",
			     title, frame_ctl, header->addr1[5],
			     header->addr3[5], rssi,
			     tsf_low - priv->scan_start_tsf,
			     phy_flags, channel);
		}
	}
	if (print_dump)
		printk_buf(IPW_DL_RX, data, length);
}

static void ipw_handle_reply_rx(struct ipw_priv *priv,
				struct ipw_rx_mem_buffer *rxb)
{
	struct ipw_rx_packet *pkt = (void *)rxb->skb->data;
	struct ipw_rx_frame_stats *rx_stats = IPW_RX_STATS(pkt);
	struct ipw_rx_frame_hdr *rx_hdr = IPW_RX_HDR(pkt);
	struct ipw_rx_frame_end *rx_end = IPW_RX_END(pkt);
	struct ieee80211_hdr *header;
	struct ieee80211_rx_status stats = {
		.mactime = rx_end->beaconTimeStamp,
		.freq =
		    (rx_hdr->
		     phy_flags & RX_RES_PHY_FLAGS_BAND_24_MSK) ?
		    IEEE80211_24GHZ_BAND : IEEE80211_52GHZ_BAND,
		.channel = rx_hdr->channel,
		.phymode =
		    (rx_hdr->
		     phy_flags & RX_RES_PHY_FLAGS_BAND_24_MSK) ?
		    MODE_IEEE80211G : MODE_IEEE80211A,
		.ssi = rx_stats->rssi - IPW_RSSI_OFFSET,
		.antenna = 0,
		.rate = rx_hdr->rate,
		.flag = rx_hdr->phy_flags,
	};

	u8 network_packet;
	if ((unlikely(rx_stats->mib_count > 20))) {
		IPW_DEBUG_DROP
		    ("dsp size out of range [0,20]: "
		     "%d/n", rx_stats->mib_count);
		priv->wstats.discard.misc++;
		return;
	}

	if (!(rx_end->status & RX_RES_STATUS_NO_CRC32_ERROR)
	    || !(rx_end->status & RX_RES_STATUS_NO_RXE_OVERFLOW)) {
		IPW_DEBUG_RX("Bad CRC or FIFO: 0x%08X.\n", rx_end->status);
		priv->wstats.discard.misc++;
		return;
	}

	if (priv->iw_mode == IEEE80211_IF_TYPE_MNTR) {
		ipw_handle_data_packet(priv, 1, rxb, &stats);
		return;
	}

	stats.freq = ieee80211chan2mhz((stats.channel));
	stats.flag = 0;

	/* can be covered by ipw_report_frame() in most cases */
//      IPW_DEBUG_RX("RX status: 0x%08X\n", rx_end->status);

	priv->rx_packets++;

	header = (struct ieee80211_hdr *)IPW_RX_DATA(pkt);

	network_packet = is_network_packet(priv, header);

	if (ipw_debug_level & IPW_DL_STATS && net_ratelimit())
		IPW_DEBUG_STATS
		    ("[%c] %d RSSI: %d Signal: %u, Noise: %u, Rate: %u\n",
		     network_packet ? '*' : ' ',
		     stats.channel, stats.ssi, stats.ssi,
		     stats.ssi, stats.rate);

	if (network_packet) {
		if (rx_stats->noise_diff) {
			average_add(&priv->average_noise,
				    le16_to_cpu(rx_stats->noise_diff));
		}
	}

	if (network_packet) {
		average_add(&priv->average_rssi, stats.ssi);

		priv->last_rx_rssi = stats.ssi;

		priv->last_rx_jiffies = jiffies;
		priv->last_beacon_time = rx_end->beaconTimeStamp;
		priv->last_tsf = rx_end->timestamp;
	}
	if (ipw_debug_level & (IPW_DL_RX))
		/* Set "1" to report good data frames in groups of 100 */
		ipw_report_frame(priv, pkt, header, 1);

	switch (WLAN_FC_GET_TYPE(le16_to_cpu(header->frame_control))) {
	case IEEE80211_FTYPE_MGMT:
		switch (WLAN_FC_GET_STYPE(le16_to_cpu(header->frame_control))) {
		case IEEE80211_STYPE_PROBE_RESP:
		case IEEE80211_STYPE_BEACON:{
				if ((((priv->iw_mode == IEEE80211_IF_TYPE_STA)
				      && !compare_ether_addr(header->addr2,
							     priv->bssid))
				     ||
				     ((priv->iw_mode == IEEE80211_IF_TYPE_IBSS)
				      && !compare_ether_addr(header->addr3,
							     priv->bssid)))) {
					struct ieee80211_mgmt *mgmt =
					    (struct ieee80211_mgmt *)header;
					u32 *pos;
					pos =
					    (u32 *) & mgmt->u.beacon.timestamp;
					priv->timestamp0 = le64_to_cpu(pos[0]);
					priv->timestamp1 = le32_to_cpu(pos[1]);
					priv->assoc_capability =
					    le16_to_cpu(mgmt->u.beacon.
							capab_info);
/*
  if ((priv->status & STATUS_ASSOCIATING)
  && (priv->iw_mode ==
  IEEE80211_IF_TYPE_STA)
  && (priv->assoc_id != 0x0)) {
  queue_work(priv->workqueue,
  &priv->
  associate_timeout);

  }
*/

				}

				break;
			}

		case IEEE80211_STYPE_ACTION:
			/* TODO: Parse 802.11h frames for CSA... */
			break;

			/*MAC80211 there is no callback function from upper
			   stack to inform us when associated status. this
			   work around to sniff assoc_resp management frame
			   and finish the association process for 3945 */
		case IEEE80211_STYPE_ASSOC_RESP:
		case IEEE80211_STYPE_REASSOC_RESP:{
				struct ieee80211_mgmt *mgnt =
				    (struct ieee80211_mgmt *)header;
				priv->assoc_id = (~((1 << 15) | (1 << 14))
						  & mgnt->u.assoc_resp.aid);
				priv->assoc_capability =
				    le16_to_cpu(mgnt->u.assoc_resp.capab_info);
				queue_work(priv->workqueue,
					   &priv->post_associate);
				break;
			}

		case IEEE80211_STYPE_PROBE_REQ:{
				if (priv->iw_mode == IEEE80211_IF_TYPE_IBSS)
					IPW_DEBUG_DROP
					    ("Dropping (non network): " MAC_FMT
					     ", " MAC_FMT ", " MAC_FMT "\n",
					     MAC_ARG(header->addr1),
					     MAC_ARG(header->addr2),
					     MAC_ARG(header->addr3));
				return;
			}
		}
		ipw_handle_data_packet(priv, 0, rxb, &stats);
		break;

	case IEEE80211_FTYPE_CTL:
		break;

	case IEEE80211_FTYPE_DATA:
		if (unlikely(is_duplicate_packet(priv, header)))
			IPW_DEBUG_DROP("Dropping (dup): " MAC_FMT ", "
				       MAC_FMT ", " MAC_FMT "\n",
				       MAC_ARG(header->addr1),
				       MAC_ARG(header->addr2),
				       MAC_ARG(header->addr3));
		else
			ipw_handle_data_packet(priv, 1, rxb, &stats);
		break;
	}
}

#define TX_STATUS_ENTRY(x) case TX_STATUS_FAIL_ ## x: return #x

static const char *get_tx_fail_reason(u32 status)
{
	switch (status & TX_STATUS_MSK) {
	case TX_STATUS_SUCCESS:
		return "SUCCESS";
		TX_STATUS_ENTRY(SHORT_LIMIT);
		TX_STATUS_ENTRY(LONG_LIMIT);
		TX_STATUS_ENTRY(FIFO_UNDERRUN);
		TX_STATUS_ENTRY(MGMNT_ABORT);
		TX_STATUS_ENTRY(NEXT_FRAG);
		TX_STATUS_ENTRY(LIFE_EXPIRE);
		TX_STATUS_ENTRY(DEST_PS);
		TX_STATUS_ENTRY(ABORTED);
		TX_STATUS_ENTRY(BT_RETRY);
		TX_STATUS_ENTRY(STA_INVALID);
		TX_STATUS_ENTRY(FRAG_DROPPED);
		TX_STATUS_ENTRY(TID_DISABLE);
		TX_STATUS_ENTRY(FRAME_FLUSHED);
		TX_STATUS_ENTRY(INSUFFICIENT_CF_POLL);
		TX_STATUS_ENTRY(TX_LOCKED);
		TX_STATUS_ENTRY(NO_BEACON_ON_RADAR);
	}

	return "UNKNOWN";
}

static void ipw_handle_reply_tx(struct ipw_priv *priv, void *data, u16 sequence)
{
	int fifo = SEQ_TO_FIFO(sequence);
	int index = SEQ_TO_INDEX(sequence);
	struct ipw_tx_queue *txq = &priv->txq[fifo];
	struct ieee80211_tx_status *status;
	struct ipw_tx_resp *resp = (struct ipw_tx_resp *)data;

	if ((index >= txq->q.n_bd) || (x2_queue_used(&txq->q, index) == 0)) {
		IPW_ERROR("Read index for DMA queue (%d) "
			  "is out of range [0-%d) %d %d\n",
			  index, txq->q.n_bd, txq->q.first_empty,
			  txq->q.last_used);
		return;
	}

	status = &(txq->txb[txq->q.last_used].status);
	status->flags = ((resp->status & 0xFF) == 0x1) ?
	    IEEE80211_TX_STATUS_ACK : 0;
	status->retry_count = resp->failure_frame;
	status->queue_number = resp->status;
	status->queue_length = resp->bt_kill_count;
	status->queue_length |= resp->failure_rts;
	status->control.tx_rate = resp->rate;

	IPW_DEBUG_TX("Tx fifo %d Status %s (0x%08x) plcp rate %d retries %d\n",
		     fifo, get_tx_fail_reason(resp->status),
		     resp->status, resp->rate, resp->failure_frame);

	if (check_bits(resp->status, TX_ABORT_REQUIRED_MSK)) {
		IPW_ERROR("TODO:  Implement Tx ABORT REQUIRED!!!\n");
	}

	return;
}

/*
 * Main entry function for receiving a packet with 80211 headers.  This
 * should be called when ever the FW has notified us that there is a new
 * skb in the receive queue.
 */
static void ipw_rx_handle(struct ipw_priv *priv)
{
	struct ipw_rx_mem_buffer *rxb;
	struct ipw_rx_packet *pkt;
	u32 r, i;
	int pkt_from_hardware;

	r = priv->hw_setting.get_rx_read(priv);
	i = priv->rxq->read;

	while (i != r) {
		rxb = priv->rxq->queue[i];
		BUG_ON(rxb == NULL);
		priv->rxq->queue[i] = NULL;
		pci_dma_sync_single_for_cpu(priv->pci_dev,
					    rxb->dma_addr,
					    priv->hw_setting.rx_buffer_size,
					    PCI_DMA_FROMDEVICE);
		pkt = (struct ipw_rx_packet *)rxb->skb->data;

		/* If this frame wasn't received then it is a response from
		 * a host request */
		pkt_from_hardware = !(pkt->hdr.sequence & SEQ_RX_FRAME);

		/* Don't report replies covered by debug messages below ...
		 * switch statement for readability ... compiler may optimize.
		 * Hack at will to see/not-see what you want in logs. */
		switch (pkt->hdr.cmd) {
		case SCAN_START_NOTIFICATION:
		case SCAN_RESULTS_NOTIFICATION:
		case SCAN_COMPLETE_NOTIFICATION:
		case REPLY_STATISTICS_CMD:
		case STATISTICS_NOTIFICATION:
		case REPLY_RX:
		case REPLY_ALIVE:
		case REPLY_ADD_STA:
		case REPLY_ERROR:
			break;
		default:
			IPW_DEBUG_RX
			    ("Received %s command (#%x), seq:0x%04X, "
			     "flags=0x%02X, len = %d\n",
			     get_cmd_string(pkt->hdr.cmd),
			     pkt->hdr.cmd, pkt->hdr.sequence,
			     pkt->hdr.flags, le16_to_cpu(pkt->len));
		}
		switch (pkt->hdr.cmd) {
		case REPLY_RX:	/* 802.11 frame */
			ipw_handle_reply_rx(priv, rxb);
			break;

		case REPLY_ALIVE:{
				memcpy(&priv->card_alive,
				       &pkt->u.alive_frame,
				       sizeof(struct ipw_alive_resp));

				ipw_disable_events(priv);

				IPW_DEBUG_INFO
				    ("Alive ucode status 0x%08X revision "
				     "0x%01X 0x%01X\n",
				     priv->card_alive.is_valid,
				     priv->card_alive.ver_type,
				     priv->card_alive.ver_subtype);
				/* We delay the ALIVE response by 5ms to
				 * give the HW RF Kill time to activate... */
				if (priv->card_alive.is_valid == UCODE_VALID_OK)
					queue_delayed_work(priv->
							   workqueue,
							   &priv->
							   alive_start,
							   msecs_to_jiffies(5));
				else
					IPW_WARNING
					    ("uCode did not respond OK.\n");
				break;
			}

		case REPLY_ADD_STA:{
				IPW_DEBUG_RX
				    ("Received REPLY_ADD_STA: 0x%02X\n",
				     pkt->u.status);
				break;
			}

		case REPLY_ERROR:{
				u32 err_type = pkt->u.err_resp.enumErrorType;
				u8 cmd_id = pkt->u.err_resp.currentCmdID;
				u16 seq = pkt->u.err_resp.erroneousCmdSeqNum;
				u32 info = pkt->u.err_resp.error_info;
				IPW_ERROR("Error Reply type 0x%08X "
					  "cmd %s (0x%02X) "
					  "seq 0x%04X info 0x%08X\n",
					  err_type,
					  get_cmd_string(cmd_id),
					  cmd_id, seq, info);
				break;
			}
		case REPLY_TX:
			ipw_handle_reply_tx(priv, &pkt->u.raw[0],
					    pkt->hdr.sequence);
			break;

		case CHANNEL_SWITCH_NOTIFICATION:{
				struct ipw_csa_notification *csa =
				    &(pkt->u.csa_notif);
				IPW_DEBUG_11H
				    ("CSA notif: channel %d, status %d\n",
				     csa->channel, csa->status);
				priv->active_conf.channel = csa->channel;
				break;
			}

		case SPECTRUM_MEASURE_NOTIFICATION:{
				struct ipw_spectrum_notification
				*report = &(pkt->u.spectrum_notif);

				if (!report->state) {
					IPW_DEBUG(IPW_DL_11H |
						  IPW_DL_INFO,
						  "Spectrum Measure Notification: "
						  "Start\n");
					break;
				}

				memcpy(&priv->measure_report, report,
				       sizeof(*report));
				queue_work(priv->workqueue, &priv->report_work);
				break;
			}

		case QUIET_NOTIFICATION:
			IPW_DEBUG_INFO("UNHANDLED - Quiet Notification.\n");
			break;

		case MEASURE_ABORT_NOTIFICATION:
			IPW_DEBUG_INFO
			    ("UNHANDLED - Measure Abort Notification.\n");
			break;

		case RADAR_NOTIFICATION:
			IPW_DEBUG_INFO("UNHANDLED - Radar Notification.\n");
			break;

		case PM_SLEEP_NOTIFICATION:{
				struct ipw_sleep_notification *sleep =
				    &(pkt->u.sleep_notif);
				IPW_DEBUG_RX
				    ("sleep mode: %d, src: %d\n",
				     sleep->pm_sleep_mode,
				     sleep->pm_wakeup_src);
				break;
			}

		case PM_DEBUG_STATISTIC_NOTIFIC:
			IPW_DEBUG_RADIO
			    ("Dumping %d bytes of unhandled "
			     "notification for %s:\n",
			     le16_to_cpu(pkt->len),
			     get_cmd_string(pkt->hdr.cmd));
			printk_buf(IPW_DL_RADIO, pkt->u.raw,
				   le16_to_cpu(pkt->len));
			break;

		case BEACON_NOTIFICATION:{
				struct BeaconNtfSpecifics *beacon =
				    &(pkt->u.beacon_status);
				IPW_DEBUG_INFO
				    ("beacon status %x retries %d iss %d "
				     "tsf %d %d rate %d\n",
				     beacon->bconNotifHdr.status,
				     beacon->bconNotifHdr.
				     failure_frame,
				     beacon->ibssMgrStatus,
				     beacon->highTSF, beacon->lowTSF,
				     beacon->bconNotifHdr.rate);

			}
			break;

		case REPLY_STATISTICS_CMD:
		case STATISTICS_NOTIFICATION:
			IPW_DEBUG_RX
			    ("Statistics notification received (%d vs %d).\n",
			     priv->hw_setting.statistics_size, pkt->len);

			memcpy(&priv->statistics.payload[0], pkt->u.raw,
			       priv->hw_setting.statistics_size);

			priv->last_statistics_time = jiffies;
			break;

		case WHO_IS_AWAKE_NOTIFICATION:
			IPW_DEBUG_RX("Notification from the card \n");
			break;

		case REPLY_SCAN_CMD:{
				struct ipw_scanreq_notification *notif
				    =
				    (struct ipw_scanreq_notification
				     *)pkt->u.raw;
				IPW_DEBUG_RX
				    ("Scan request status = 0x%x\n",
				     notif->status);
				break;

			}

		case SCAN_START_NOTIFICATION:{
				struct ipw_scanstart_notification
				*notif =
				    (struct ipw_scanstart_notification
				     *)pkt->u.raw;
				priv->scan_start_tsf = notif->tsf_low;
				IPW_DEBUG_SCAN("Scan start: "
					       "%d [802.11%s] "
					       "(TSF: 0x%08X:%08X) - %d (beacon timer %u)\n",
					       notif->channel,
					       notif->
					       band ? "bg" : "a",
					       notif->tsf_high,
					       notif->tsf_low,
					       notif->status,
					       notif->beacon_timer);
				break;
			}

		case SCAN_RESULTS_NOTIFICATION:{
				struct ipw_scanresults_notification
				*notif = (struct ipw_scanresults_notification *)
				    pkt->u.raw;

				IPW_DEBUG_SCAN("Scan ch.res: "
					       "%d [802.11%s] "
					       "(TSF: 0x%08X:%08X) - %d "
					       "elapsed=%lu usec (%dms since last)\n",
					       notif->channel,
					       notif->
					       band ? "bg" : "a",
					       notif->tsf_high,
					       notif->tsf_low,
					       notif->statistics[0],
					       notif->tsf_low -
					       priv->scan_start_tsf,
					       jiffies_to_msecs
					       (elapsed_jiffies
						(priv->
						 last_scan_jiffies, jiffies)));
				priv->last_scan_jiffies = jiffies;
				break;
			}

		case SCAN_COMPLETE_NOTIFICATION:{
				struct ipw_scancomplete_notification
				*scan_notif =
				    (struct ipw_scancomplete_notification *)
				    pkt->u.raw;
				IPW_DEBUG_SCAN
				    ("Scan complete: %d channels "
				     "(TSF 0x%08X:%08X) - %d\n",
				     scan_notif->scanned_channels,
				     scan_notif->tsf_low,
				     scan_notif->tsf_high, scan_notif->status);

				ipw_scan_completed(priv,
						   scan_notif->status == 1);
				break;
			}

		case CARD_STATE_NOTIFICATION:{
				u32 flags =
				    le32_to_cpu(pkt->u.card_state_notif.flags);
				u32 status = priv->status;
				IPW_DEBUG_RF_KILL
				    ("Card state received: "
				     "HW:%s SW:%s\n",
				     (flags & HW_CARD_DISABLED) ?
				     "Off" : "On",
				     (flags & SW_CARD_DISABLED) ? "Off" : "On");

				if (flags & HW_CARD_DISABLED) {
					ipw_write32(priv,
						    CSR_UCODE_DRV_GP1_SET,
						    CSR_UCODE_DRV_GP1_BIT_CMD_BLOCKED);

					priv->status |= STATUS_RF_KILL_HW;
				} else
					priv->status &= ~STATUS_RF_KILL_HW;

				if (flags & SW_CARD_DISABLED)
					priv->status |= STATUS_RF_KILL_SW;
				else
					priv->status &= ~STATUS_RF_KILL_SW;

				ipw_scan_cancel(priv);

				if (((status & STATUS_RF_KILL_HW) !=
				     (priv->status & STATUS_RF_KILL_HW))
				    || ((status & STATUS_RF_KILL_SW)
					!= (priv->status & STATUS_RF_KILL_SW))) {

					queue_work(priv->workqueue,
						   &priv->rf_kill);
				} else
					wake_up_interruptible(&priv->
							      wait_command_queue);

				break;
			}
		default:
			priv->hw_setting.rx_cmd(priv, rxb, pkt->hdr.cmd);
			break;
		}

		if (pkt_from_hardware) {
			/* Invoke any callbacks, transfer the skb to
			 * caller, and fire off the (possibly) blocking
			 * ipw_send_cmd() via as we reclaim the queue... */
			if (rxb && rxb->skb)
				ipw_tx_complete(priv, rxb);
			else
				IPW_WARNING("Claim null rxb?\n");
		}

		/* For now we just don't re-use anything.  We can tweak this
		 * later to try and re-use notification packets and SKBs that
		 * fail to Rx correctly */
		if (rxb->skb != NULL) {
			dev_kfree_skb_any(rxb->skb);
			rxb->skb = NULL;
		}

		pci_unmap_single(priv->pci_dev, rxb->dma_addr,
				 priv->hw_setting.rx_buffer_size,
				 PCI_DMA_FROMDEVICE);
		list_add_tail(&rxb->list, &priv->rxq->rx_used);
		i = (i + 1) % RX_QUEUE_SIZE;
	}

	/* Backtrack one entry */
	priv->rxq->read = i;
	ipw_rx_queue_restock(priv);
}

/*
 * This file defines the Wireless Extension handlers.  It does not
 * define any methods of hardware manipulation and relies on the
 * functions defined in ipw_main to provide the HW interaction.
 *
 */

static int ipw_set_channel(struct ipw_priv *priv, struct ieee80211_conf *conf)
{
	int rc = 0;

	unsigned long flags;
	struct ipw_channel_info *ch;

	if (conf->channel == 0)
		return 0;

	spin_lock_irqsave(&priv->lock, flags);

	priv->config |= CFG_STATIC_CHANNEL;
	if (priv->active_conf.channel == conf->channel &&
	    priv->active_conf.phymode == conf->phymode) {
		IPW_DEBUG_INFO
		    ("Request to set channel to current value (%d [%d])\n",
		     conf->channel, conf->phymode);
		rc = 0;
		goto exit;
	}

	ch = ipw_get_channel_info(priv, conf->phymode, conf->channel);
	if (!ch) {
		IPW_DEBUG_INFO("Could not set channel to %d [%d]\n",
			       conf->channel, conf->phymode);
		rc = -EINVAL;
		goto exit;
	}

	IPW_DEBUG_INFO("Setting channel to %d [%d]\n",
		       conf->channel, conf->phymode);
	priv->active_conf.channel = ch->channel;
	priv->active_conf.phymode = ch->phymode;
	priv->curr_channel_flag = ch->flags;

      exit:
	spin_unlock_irqrestore(&priv->lock, flags);
	return rc;
}

static int ipw_set_mode(struct ipw_priv *priv, int mode)
{
	unsigned long now = jiffies;

	if (!ipw_is_ready_rf(priv))
		return -EAGAIN;

	cancel_delayed_work(&priv->scan_check);
	cancel_delayed_work(&priv->gather_stats);
	priv->status &= ~STATUS_SCAN_PENDING;
	if (priv->status & STATUS_SCANNING)
		ipw_abort_scan(priv);

	while (!time_after(jiffies, now + msecs_to_jiffies(100)) &&
	       priv->status & STATUS_SCANNING)
		msleep(1);

	if (priv->status & STATUS_SCANNING) {
		IPW_DEBUG_SCAN("Scan still in progress after %dms\n", 100);
	}
//      ipw_disassociate(priv);

	priv->iw_mode = mode;

	if (priv->iw_mode == IEEE80211_IF_TYPE_MNTR)
		priv->net_dev->type = ARPHRD_IEEE80211_RADIOTAP;

	ipw_connection_init_rx_config(priv);
	memcpy(priv->staging_rxon.node_addr, priv->net_dev->dev_addr, ETH_ALEN);

	// switching to managed mode, delete all ibss stations
	if (mode == IEEE80211_IF_TYPE_STA)
		ipw_delete_stations_table(priv, 1);

	ipw_commit_rxon(priv);

	return 0;
}

/* RTS threshold here is total size [2347] minus 4 FCS bytes
 * Per spec:
 * a value of 0 means RTS on all data/management packets
 * a value > max MSDU size means no RTS
 * else RTS for data/management frames where MPDU is larger
 * than RTS value.  */
#define DEFAULT_RTS_THRESHOLD     2347U
#define MIN_RTS_THRESHOLD         0U
#define MAX_RTS_THRESHOLD         2347U
#define MAX_MSDU_SIZE		  2304U
#define MAX_MPDU_SIZE		  2346U
#define DEFAULT_BEACON_INTERVAL   100U
#define	DEFAULT_SHORT_RETRY_LIMIT 7U
#define	DEFAULT_LONG_RETRY_LIMIT  4U

static int ipw_set_power(struct ipw_priv *priv, struct ieee80211_conf *conf)
{
	int err;
	int disabled = 0;

//      disabled = !conf->power_management_enable;

	//TODO for now just return;
	return 0;

	if (!ipw_is_ready(priv))
		return -EAGAIN;

	if (disabled) {
		priv->power_mode = IPW_POWER_MODE_CAM;
		err = ipw_send_power_mode(priv, IPW_POWER_MODE_CAM);
		if (err) {
			IPW_DEBUG_WX("failed setting power mode.\n");
			return err;
		}
		IPW_DEBUG_WX("SET Power Management Mode -> off\n");
		return 0;
	}

	/* If the user hasn't specified a power management mode yet, default
	 * to BATTERY */
	if ((IPW_POWER_LEVEL(priv->power_mode) == IPW_POWER_AC) ||
	    (IPW_POWER_LEVEL(priv->power_mode) == IPW_POWER_MODE_CAM))
		priv->power_mode = IPW_POWER_MODE_CAM;
	else
		priv->power_mode = IPW_POWER_ENABLED | priv->power_mode;
	err = ipw_send_power_mode(priv, IPW_POWER_LEVEL(priv->power_mode));
	if (err) {
		IPW_DEBUG_WX("failed setting power mode.\n");
		return err;
	}

	IPW_DEBUG_WX("SET Power Management Mode -> 0x%02X\n", priv->power_mode);
	return 0;
}

/* MAC80211 fix this with mac80211 stack */

static int ipw_wx_set_powermode(struct ipw_priv *priv, int mode)
{
	int err;
	int old_power_mode = priv->power_mode;

	if (!ipw_is_ready(priv))
		return -EAGAIN;

	if ((mode < 1) || (mode > IPW_POWER_LIMIT)
	    || (mode == IPW_POWER_AC)) {
		mode = IPW_POWER_AC;
		priv->power_mode = mode;
	} else {
		priv->power_mode = IPW_POWER_ENABLED | mode;
	}

	if (old_power_mode != priv->power_mode) {
		err = ipw_send_power_mode(priv, mode);
		if (err) {
			IPW_DEBUG_WX("failed setting power mode.\n");
			return err;
		}
	}
	return 0;
}

/* net device stuff */

static int d_open(struct ieee80211_hw *hw)
{
	struct ipw_priv *priv = hw->priv;

	IPW_DEBUG_MAC80211("enter %p\n", hw);
	IPW_DEBUG_INFO("dev->open %p\n", priv->net_dev);
	/* we should be verifying the device is ready to be opened */
	mutex_lock(&priv->mutex);

	if (priv->net_dev && !(priv->status & STATUS_RF_KILL_MASK)) {
		ieee80211_netif_oper(priv->ieee, NETIF_ATTACH);
		ieee80211_netif_oper(priv->ieee, NETIF_START);
		ieee80211_netif_oper(priv->ieee, NETIF_WAKE);
		netif_carrier_on(priv->net_dev);

	}
	mutex_unlock(&priv->mutex);
	IPW_DEBUG_MAC80211("leave %p\n", hw);
	return 0;
}

static int d_stop(struct ieee80211_hw *hw)
{
	IPW_DEBUG_MAC80211("enter %p\n", hw);
	IPW_DEBUG_INFO("dev->close\n");
	//netif_stop_queue(dev);
	IPW_DEBUG_MAC80211("leave %p\n", hw);
	flush_scheduled_work();
	return 0;
}

/*************** RATE-SCALING functions ***********************/
#if 0

static struct ipw_tpt_entry ipw_rssi_rates_a[] = {
	{-60, RATE_SCALE_54M_INDEX},
	{-64, RATE_SCALE_48M_INDEX},
	{-72, RATE_SCALE_36M_INDEX},
	{-80, RATE_SCALE_24M_INDEX},
	{-84, RATE_SCALE_18M_INDEX},
	{-85, RATE_SCALE_12M_INDEX},
	{-87, RATE_SCALE_9M_INDEX},
	{-89, RATE_SCALE_6M_INDEX}
};

static struct ipw_tpt_entry ipw_rssi_rates_b[] = {
	{-86, RATE_SCALE_11M_INDEX},
	{-88, RATE_SCALE_5_5M_INDEX},
	{-90, RATE_SCALE_2M_INDEX},
	{-92, RATE_SCALE_1M_INDEX}

};

static struct ipw_tpt_entry ipw_rssi_rates_g[] = {
	{-60, RATE_SCALE_54M_INDEX},
	{-64, RATE_SCALE_48M_INDEX},
	{-68, RATE_SCALE_36M_INDEX},
	{-80, RATE_SCALE_24M_INDEX},
	{-84, RATE_SCALE_18M_INDEX},
	{-85, RATE_SCALE_12M_INDEX},
	{-86, RATE_SCALE_11M_INDEX},
	{-88, RATE_SCALE_5_5M_INDEX},
	{-90, RATE_SCALE_2M_INDEX},
	{-92, RATE_SCALE_1M_INDEX}
};
#endif

#if 0
static int ipw_get_rate_by_rssi(s32 rssi, int phymode)
{
	u32 index = 0;
	u32 table_size = 0;
	struct ipw_tpt_entry *tpt_table = NULL;

	if ((rssi < IPW_MIN_RSSI_VAL) || (rssi > IPW_MAX_RSSI_VAL))
		rssi = IPW_MIN_RSSI_VAL;

	switch (phymode) {
	case MODE_IEEE80211G:
		tpt_table = ipw_rssi_rates_g;
		table_size = ARRAY_SIZE(ipw_rssi_rates_g);
		break;

	case MODE_IEEE80211B:
		tpt_table = ipw_rssi_rates_b;
		table_size = ARRAY_SIZE(ipw_rssi_rates_b);
		break;

	case MODE_IEEE80211A:
	default:
		tpt_table = ipw_rssi_rates_a;
		table_size = ARRAY_SIZE(ipw_rssi_rates_a);
		break;
	}

	for (index = 0;
	     (index < table_size) && (rssi < tpt_table[index].min_rssi);
	     index++) ;

	index = min(index, (table_size - 1));

	return tpt_table[index].rate_scale_index;
}
#endif

#if 0
static int ipw_init_rate_scaling(struct ipw_priv *priv)
{
	int rc, i;
	struct ipw_rate_scaling_cmd_specifics rate_cmd = {
		.reserved = {0, 0, 0},
	};
	struct ipw_rate_scaling_info *table = rate_cmd.table;

	for (i = 0; i < IPW_MAX_RATES; i++)
		table[i].try_cnt = priv->retry_rate;

	if (priv->active_conf.phymode == MODE_IEEE80211A) {
		IPW_DEBUG_RATE("Select A mode rate scale\n");

		table[RATE_SCALE_6M_INDEX].next_rate_index =
		    RATE_SCALE_6M_INDEX;
		table[RATE_SCALE_9M_INDEX].next_rate_index =
		    RATE_SCALE_6M_INDEX;
		table[RATE_SCALE_12M_INDEX].next_rate_index =
		    RATE_SCALE_9M_INDEX;
		table[RATE_SCALE_18M_INDEX].next_rate_index =
		    RATE_SCALE_12M_INDEX;
		table[RATE_SCALE_24M_INDEX].next_rate_index =
		    RATE_SCALE_18M_INDEX;
		table[RATE_SCALE_36M_INDEX].next_rate_index =
		    RATE_SCALE_24M_INDEX;
		table[RATE_SCALE_48M_INDEX].next_rate_index =
		    RATE_SCALE_36M_INDEX;
		table[RATE_SCALE_54M_INDEX].next_rate_index =
		    RATE_SCALE_48M_INDEX;

		/* If one of the following CCK rates is used,
		 * have it fall back to the 6M OFDM rate */
		for (i = RATE_SCALE_1M_INDEX; i <= RATE_SCALE_11M_INDEX; i++)
			table[i].next_rate_index = RATE_SCALE_6M_INDEX;

	} else {
		/* CCK rates... */
		table[RATE_SCALE_1M_INDEX].next_rate_index =
		    RATE_SCALE_1M_INDEX;
		table[RATE_SCALE_2M_INDEX].next_rate_index =
		    RATE_SCALE_1M_INDEX;
		table[RATE_SCALE_5_5M_INDEX].next_rate_index =
		    RATE_SCALE_2M_INDEX;
		table[RATE_SCALE_11M_INDEX].next_rate_index =
		    RATE_SCALE_5_5M_INDEX;

		if (priv->active_conf.phymode == MODE_IEEE80211B) {
			IPW_DEBUG_RATE("Select B mode rate scale\n");

			/* If an OFDM rate is used, have it fall back to the
			 * 1M CCK rates */
			for (i = RATE_SCALE_6M_INDEX; i <= RATE_SCALE_54M_INDEX;
			     i++)
				table[i].next_rate_index = RATE_SCALE_1M_INDEX;
		} else {
			IPW_DEBUG_RATE("Select G mode rate scale\n");

			table[RATE_SCALE_1M_INDEX].next_rate_index =
			    RATE_SCALE_1M_INDEX;
			table[RATE_SCALE_2M_INDEX].next_rate_index =
			    RATE_SCALE_1M_INDEX;
			table[RATE_SCALE_5_5M_INDEX].next_rate_index =
			    RATE_SCALE_2M_INDEX;
			table[RATE_SCALE_6M_INDEX].next_rate_index =
			    RATE_SCALE_5_5M_INDEX;
			table[RATE_SCALE_9M_INDEX].next_rate_index =
			    RATE_SCALE_6M_INDEX;
			table[RATE_SCALE_11M_INDEX].next_rate_index =
			    RATE_SCALE_9M_INDEX;
			table[RATE_SCALE_12M_INDEX].next_rate_index =
			    RATE_SCALE_11M_INDEX;
			table[RATE_SCALE_18M_INDEX].next_rate_index =
			    RATE_SCALE_12M_INDEX;
			table[RATE_SCALE_24M_INDEX].next_rate_index =
			    RATE_SCALE_18M_INDEX;
			table[RATE_SCALE_36M_INDEX].next_rate_index =
			    RATE_SCALE_24M_INDEX;
			table[RATE_SCALE_48M_INDEX].next_rate_index =
			    RATE_SCALE_36M_INDEX;
			table[RATE_SCALE_54M_INDEX].next_rate_index =
			    RATE_SCALE_48M_INDEX;
		}
	}

	/* Update the rate scaling for control frame Tx */
	rate_cmd.table_id = 0;

	rc = ipw_send_cmd_pdu(priv, REPLY_RATE_SCALE, sizeof(rate_cmd),
			      &rate_cmd);
	if (rc)
		return rc;

	/* Update the rate scaling for data frame Tx */
	rate_cmd.table_id = 1;
	return ipw_send_cmd_pdu(priv, REPLY_RATE_SCALE, sizeof(rate_cmd),
				&rate_cmd);
}
#endif

#define IPW_RATE_SCALE_AVG_PACKETS      300
#define IPW_RATE_FLUSH_MAX              5000	// msec
#define IPW_RATE_FLUSH_MIN              50	// msec

#define IPW_TX_MEASURE_LEN 100

/*
  TX command functions
*/
/*
 * get the available rate. if management frame or broadcast frame only return
 * basic available rates.
 */
static void ipw_get_supported_rates(struct ipw_priv *priv,
				    struct ieee80211_hdr *hdr,
				    u16 * data_rate, u16 * ctrl_rate)
{
	*data_rate = priv->active_rate;
	*ctrl_rate = priv->active_rate;

	if (hdr && (is_multicast_ether_addr(hdr->addr1))
	    && (priv->active_rate_basic)) {
		*data_rate = priv->active_rate_basic;
		*ctrl_rate = priv->active_rate_basic;
	}
}

/*
  handle build REPLY_TX command notification. it is responsible for rates fields.
*/
static int ipw_build_tx_cmd_rate(struct ipw_priv *priv,
				 struct ipw_cmd *cmd,
				 struct ieee80211_tx_control *ctrl,
				 struct ieee80211_hdr *hdr,
				 int sta_id, int tx_id)
{
	u16 rate_mask, ctrl_rate;
	unsigned long flags;
	int rate = ctrl->tx_rate;
	u8 rts_retry_limit = 0;
	u8 data_retry_limit = 0;
	u32 tx_flags;

	tx_flags = cmd->cmd.tx.tx_flags;

	ipw_get_supported_rates(priv, hdr, &rate_mask, &ctrl_rate);
	rate = ctrl->tx_rate;

	if ((rate & 0xFF) == IPW_INVALID_RATE) {
		IPW_ERROR("ERROR: No TX rate available.\n");
		return -1;
	}

	IPW_DEBUG_RATE
	    ("Tx sta %d ew plcp rate %X old %d rate mask %x\n",
	     sta_id, rate,
	     priv->stations[sta_id].current_rate.rate_n_flags, rate_mask);

	spin_lock_irqsave(&priv->sta_lock, flags);

	priv->stations[sta_id].current_rate.rate_n_flags = rate;

	if ((priv->iw_mode == IEEE80211_IF_TYPE_IBSS) &&
	    (sta_id != priv->hw_setting.broadcast_id) &&
	    (sta_id != MULTICAST_ID))
		priv->stations[STA_ID].current_rate.rate_n_flags = rate;

	spin_unlock_irqrestore(&priv->sta_lock, flags);

	if (tx_id >= priv->hw_setting.cmd_queue_no)
		rts_retry_limit = 3;
	else
		rts_retry_limit = 7;

	if (ieee80211_is_probe_response(hdr->frame_control)) {
		data_retry_limit = 3;
		if (data_retry_limit < rts_retry_limit)
			rts_retry_limit = data_retry_limit;
	} else
		data_retry_limit = IPW_DEFAULT_TX_RETRY;

	if (priv->data_retry_limit != -1)
		data_retry_limit = priv->data_retry_limit;

	if (WLAN_FC_GET_TYPE(hdr->frame_control) == IEEE80211_FTYPE_MGMT) {
		switch (WLAN_FC_GET_STYPE(hdr->frame_control)) {
		case IEEE80211_STYPE_AUTH:
		case IEEE80211_STYPE_DEAUTH:
		case IEEE80211_STYPE_ASSOC_REQ:
		case IEEE80211_STYPE_REASSOC_REQ:
			if (tx_flags & TX_CMD_FLG_RTS_MSK) {
				tx_flags &= ~TX_CMD_FLG_RTS_MSK;
				tx_flags |= TX_CMD_FLG_CTS_MSK;
			}
			break;
		default:
			break;
		}
	}

	cmd->cmd.tx.rts_retry_limit = rts_retry_limit;
	cmd->cmd.tx.data_retry_limit = data_retry_limit;
	cmd->cmd.tx.rate = rate;
	cmd->cmd.tx.tx_flags = tx_flags;

	/* OFDM */
	cmd->cmd.tx.supp_rates[0] = rate_mask >> 4;

	/* CCK */
	cmd->cmd.tx.supp_rates[1] = rate_mask & 0xF;

	return 0;
}

/*
  handle build REPLY_TX command notification.
*/
static int ipw_build_tx_cmd_basic(struct ipw_priv *priv,
				  struct ipw_cmd *cmd,
				  struct ieee80211_tx_control *ctrl,
				  struct ieee80211_hdr *hdr,
				  int is_unicast, u8 std_id, int tx_id)
{
	u32 tx_flags;
	u16 fc = le16_to_cpu(hdr->frame_control);

	tx_flags = cmd->cmd.tx.tx_flags;

	cmd->cmd.tx.u.life_time = 0xFFFFFFFF;
	if (!ctrl->flags & IEEE80211_TXCTL_NO_ACK) {
		tx_flags |= TX_CMD_FLG_ACK_MSK;
		if (WLAN_FC_GET_TYPE(hdr->frame_control) ==
		    IEEE80211_FTYPE_MGMT) {
			tx_flags |= TX_CMD_FLG_SEQ_CTL_MSK;
		}
		if (ieee80211_is_probe_response(hdr->frame_control)) {
			if ((hdr->seq_ctrl & 0x000F) == 0) {
				tx_flags |= TX_CMD_FLG_TSF_MSK;
			}
		}
	} else {
		tx_flags &= (~TX_CMD_FLG_ACK_MSK);
		tx_flags |= TX_CMD_FLG_SEQ_CTL_MSK;
	}

	if (std_id == IPW_INVALID_STATION)
		return -1;

	cmd->cmd.tx.sta_id = std_id;
	if (ieee80211_get_morefrag(hdr))
		tx_flags |= TX_CMD_FLG_MORE_FRAG_MSK;
	if (ieee80211_get_hdrlen(fc) == 26) {
		u8 *qc =
		    (u8 *) hdr + ieee80211_get_hdrlen(fc) - QOS_CONTROL_LEN;
		cmd->cmd.tx.tid_tspec = (u8) (qc[0] & 0x0F);
	} else
		tx_flags |= TX_CMD_FLG_SEQ_CTL_MSK;

	if (ctrl->flags & IEEE80211_TXCTL_USE_RTS_CTS) {
		tx_flags |= TX_CMD_FLG_RTS_MSK;
		tx_flags &= ~TX_CMD_FLG_CTS_MSK;
	} else if (ctrl->flags & IEEE80211_TXCTL_USE_CTS_PROTECT) {
		tx_flags &= ~TX_CMD_FLG_RTS_MSK;
		tx_flags |= TX_CMD_FLG_CTS_MSK;
	}

	if ((tx_flags & TX_CMD_FLG_RTS_MSK) || (tx_flags & TX_CMD_FLG_CTS_MSK))
		tx_flags |= TX_CMD_FLG_FULL_TXOP_PROT_MSK;

	tx_flags &= ~(TX_CMD_FLG_ANT_SEL_MSK);
	if (WLAN_FC_GET_TYPE(hdr->frame_control) == IEEE80211_FTYPE_MGMT) {
		if (((WLAN_FC_GET_STYPE(hdr->frame_control)) ==
		     IEEE80211_STYPE_ASSOC_REQ)
		    || ((WLAN_FC_GET_STYPE(hdr->frame_control)) ==
			IEEE80211_STYPE_REASSOC_REQ)) {
			cmd->cmd.tx.u2.pm_frame_timeout = 3;
		} else {
			cmd->cmd.tx.u2.pm_frame_timeout = 2;
		}
	} else
		cmd->cmd.tx.u2.pm_frame_timeout = 0;

	cmd->cmd.tx.driver_txop = 0;

	cmd->cmd.tx.tx_flags = tx_flags;

	cmd->cmd.tx.next_frame_len = 0;

	return 0;
}

static inline void ipw_build_tx_cmd_hwcrypto(struct ipw_priv *priv, struct ieee80211_tx_control
					     *ctl, struct ipw_cmd *cmd,
					     struct sk_buff *skb_frag,
					     int last_frag)
{
	struct ipw_hw_key *keyinfo = &priv->stations[ctl->key_idx].keyinfo;

	switch (keyinfo->alg) {
	case ALG_CCMP:
		cmd->cmd.tx.sec_ctl = TX_CMD_SEC_CCM;

		cmd->cmd.tx.hdr[0].frame_control |= IEEE80211_FCTL_PROTECTED;
		/* XXX: ACK flag must be set for CCMP even if it
		 * is a multicast/broadcast packet, because CCMP
		 * group communication encrypted by GTK is
		 * actually done by the AP. */
		cmd->cmd.tx.tx_flags |= TX_CMD_FLG_ACK_MSK;
		memcpy(cmd->cmd.tx.key, keyinfo->key, keyinfo->keylen);
		IPW_DEBUG_TX("tx_cmd with aes  hwcrypto\n");
		break;
	case ALG_TKIP:
		cmd->cmd.tx.sec_ctl = TX_CMD_SEC_TKIP;

		if (last_frag)
			memcpy(cmd->cmd.tx.tkip_mic.byte, skb_frag->tail - 8,
			       8);
		else
			memset(cmd->cmd.tx.tkip_mic.byte, 0, 8);

		cmd->cmd.tx.hdr[0].frame_control |= IEEE80211_FCTL_PROTECTED;
		/* XXX: ACK flag must be set for CCMP even if it
		 * is a multicast/broadcast packet, because CCMP
		 * group communication encrypted by GTK is
		 * actually done by the AP. */
		cmd->cmd.tx.tx_flags |= TX_CMD_FLG_ACK_MSK;
		break;
	case ALG_WEP:
		cmd->cmd.tx.sec_ctl = 1 |	/* WEP */
		    (ctl->key_idx & 0x3) << 6;

		if (keyinfo->keylen == 13)
			cmd->cmd.tx.sec_ctl |= (1 << 3);	/* 128-bit */

		memcpy(&cmd->cmd.tx.key[3], keyinfo->key, keyinfo->keylen);

		cmd->cmd.tx.hdr[0].frame_control |= IEEE80211_FCTL_PROTECTED;

		IPW_DEBUG_TX("Configuring packet for WEP encryption "
			     "with key %d\n", ctl->key_idx);
		break;

	case ALG_NONE:
		IPW_DEBUG_TX("Tx packet in the clear "
			     "(encrypt requested).\n");
		break;

	default:
		printk(KERN_ERR "Unknown encode alg %d\n", keyinfo->alg);
		break;
	}

}

static int ipw_get_sta_id(struct ipw_priv *priv, struct ieee80211_hdr *hdr)
{
	int sta_id;
	int unicast = !ipw_is_broadcast_ether_addr(hdr->addr1) &&
	    !is_multicast_ether_addr(hdr->addr1);

	/* If this frame is not unicast, is not data, or is a probe response
	 * then we use the broadcast station id */
	if ((WLAN_FC_GET_TYPE(hdr->frame_control) != IEEE80211_FTYPE_DATA) ||
	    !unicast || ieee80211_is_probe_response(hdr->frame_control))
		return ipw_find_station(priv, BROADCAST_ADDR);

	/* If this frame is part of a BSS network (we're a station), then
	 * we use the AP's station id */
	if (priv->iw_mode == IEEE80211_IF_TYPE_STA)
		return AP_ID;

	/* If this frame is part of a IBSS network, then we use the
	 * target specific station id */
	if (priv->iw_mode == IEEE80211_IF_TYPE_IBSS) {
		sta_id = ipw_find_station(priv, hdr->addr1);
		if (sta_id != IPW_INVALID_STATION)
			return sta_id;

		sta_id = ipw_add_station(priv, hdr->addr1, 0,
					 (CMD_ASYNC | CMD_NO_LOCK));

		if (sta_id != IPW_INVALID_STATION)
			return sta_id;

		IPW_DEBUG_DROP("Station " MAC_FMT " not in station map. "
			       "Defaulting to broadcast...\n",
			       MAC_ARG(hdr->addr1));
		printk_buf(IPW_DL_DROP, (u8 *) hdr, sizeof(*hdr));
		return ipw_find_station(priv, BROADCAST_ADDR);
	}

	/* Otherwise we default to the broadcast station id */
	return ipw_find_station(priv, BROADCAST_ADDR);

}

/*
  start REPLY_TX command process
*/
static int ipw_tx_skb(struct ipw_priv *priv,
		      struct sk_buff *skb, struct ieee80211_tx_control *ctl)
{
	struct ieee80211_hdr *hdr = (struct ieee80211_hdr *)skb->data;
	u8 *tfd;
	u32 *control_flags;
	int tx_id = ctl->queue + priv->hw_setting.start_cmd_queue;
	struct ipw_tx_queue *txq = &priv->txq[tx_id];
	struct ipw_queue *q = &txq->q;
	dma_addr_t phys_addr, scratch_phys;
	struct ipw_cmd *out_cmd = NULL;
	u16 len, idx;
	u8 id, hdr_len, unicast;
	u8 sta_id;
	u16 seq_number;
	int rc;
	u16 fc;

	if (priv->status & STATUS_RF_KILL_MASK)
		goto drop;

	unicast = !ipw_is_broadcast_ether_addr(hdr->addr1) &&
	    !is_multicast_ether_addr(hdr->addr1);
	id = 0;

	fc = le16_to_cpu(hdr->frame_control);

	if ((priv->status & STATUS_SCANNING) &&
	    (WLAN_FC_GET_TYPE(fc) == IEEE80211_FTYPE_MGMT) &&
	    (WLAN_FC_GET_STYPE(fc) == IEEE80211_STYPE_PROBE_REQ))
		goto drop;

	hdr_len = ieee80211_get_hdrlen(fc);
	sta_id = ipw_get_sta_id(priv, hdr);
	if (sta_id == IPW_INVALID_STATION)
		goto drop;

	IPW_DEBUG_RATE("station Id %d\n", sta_id);
	seq_number = priv->stations[sta_id].tid[tx_id].seq_number;
	priv->stations[sta_id].tid[tx_id].seq_number++;

	tfd = (u8 *) (&txq->bd[q->first_empty * q->element_size]);
	memset(tfd, 0, q->element_size);
	control_flags = (u32 *) tfd;
	idx = get_next_cmd_index(q, q->first_empty, 0);

	memset(&(txq->txb[q->first_empty]), 0, sizeof(struct ipw_tx_info));
	txq->txb[q->first_empty].skb[0] = skb;
	memcpy(&(txq->txb[q->first_empty].status.control),
	       ctl, sizeof(struct ieee80211_tx_control));
	out_cmd = &txq->cmd[idx];
	memset(&out_cmd->hdr, 0, sizeof(out_cmd->hdr));
	memset(&out_cmd->cmd.tx, 0, sizeof(out_cmd->cmd.tx));
	out_cmd->hdr.cmd = REPLY_TX;
	out_cmd->hdr.sequence = FIFO_TO_SEQ(tx_id) |
	    INDEX_TO_SEQ(q->first_empty);
	/* copy frags header */
	memcpy(out_cmd->cmd.tx.hdr, hdr, hdr_len);

	//hdr = (struct ieee80211_hdr *)out_cmd->cmd.tx.hdr;
	len = priv->hw_setting.tx_cmd_len +
	    sizeof(struct ipw_cmd_header) + hdr_len;

	len = (len + 3) & ~3;

	phys_addr =
	    txq->dma_addr_cmd + sizeof(struct ipw_cmd) * idx +
	    offsetof(struct ipw_cmd, hdr);

	scratch_phys = phys_addr;

	attach_buffer_to_tfd_frame(priv, tfd, phys_addr, cpu_to_le16(len));

	if (ctl->key_idx != -1)
		ipw_build_tx_cmd_hwcrypto(priv, ctl, out_cmd, skb, 0);

	phys_addr = cpu_to_le32(pci_map_single(priv->pci_dev,
					       skb->
					       data + hdr_len,
					       skb->
					       len - hdr_len,
					       PCI_DMA_TODEVICE));
	len = skb->len - hdr_len;
	attach_buffer_to_tfd_frame(priv, tfd, phys_addr, cpu_to_le16(len));

	out_cmd->cmd.tx.len = skb->len;
	if (priv->is_3945)
		*control_flags = TFD_CTL_COUNT_SET(2) |
		    TFD_CTL_PAD_SET(U32_PAD(len));

	//todoG need this for burst mode later on
	if (ipw_build_tx_cmd_basic(priv, out_cmd,
				   ctl, hdr, unicast, sta_id, tx_id)) {
		IPW_ERROR("tx build cmd basic failed.\n");
		goto drop;
	}

	if (ipw_build_tx_cmd_rate(priv, out_cmd, ctl, hdr, sta_id, tx_id)) {
		IPW_ERROR("tx cmd rate scale  failed.\n");
		goto drop;
	}

	IPW_DEBUG_TX("Tx rate %d (%02X:%02X)\n",
		     out_cmd->cmd.tx.rate,
		     out_cmd->cmd.tx.supp_rates[0],
		     out_cmd->cmd.tx.supp_rates[1]);

	len = out_cmd->cmd.tx.len;
	out_cmd->cmd.tx.tx_flags |= TX_CMD_FLG_SEQ_CTL_MSK;

	printk_buf(IPW_DL_TX, out_cmd->cmd.payload, sizeof(out_cmd->cmd.tx));

	printk_buf(IPW_DL_TX, (u8 *) out_cmd->cmd.tx.hdr,
		   ieee80211_get_hdrlen(out_cmd->cmd.tx.hdr->frame_control));

	priv->hw_setting.tx_cmd(priv, out_cmd, sta_id, scratch_phys,
				hdr, hdr_len, ctl);

	txq->need_update = 1;

	rc = priv->hw_setting.tx_queue_update_wr_ptr(priv, txq, tx_id, len);
	q->first_empty = ipw_queue_inc_wrap(q->first_empty, q->n_bd);
	rc = ipw_tx_queue_update_write_ptr(priv, txq, tx_id);

	if (rc)
		return rc;
	if ((ipw_queue_space(q) < q->high_mark)
	    && priv->netdev_registered)
		ieee80211_stop_queue(priv->ieee, ctl->queue);

	return 0;

      drop:
	IPW_DEBUG_DROP("Silently dropping Tx packet.\n");
	return -1;
}

static int d_tx(struct ieee80211_hw *hw,
		struct sk_buff *skb, struct ieee80211_tx_control *ctl)
{
	struct ipw_priv *priv = hw->priv;
	unsigned long flags = 0;
	int rc = 0;
	IPW_DEBUG_MAC80211("enter\n");
	if (priv->iw_mode == IEEE80211_IF_TYPE_MNTR) {
		IPW_DEBUG_MAC80211("leave\n");
		return -1;
	}

	IPW_DEBUG_TX("dev->xmit(%d bytes) at rate %d\n", skb->len,
		     ctl->tx_rate);

	spin_lock_irqsave(&priv->lock, flags);
	rc = ipw_tx_skb(priv, skb, ctl);
	priv->led_packets += skb->len;
	ipw_setup_activity_timer(priv);
	spin_unlock_irqrestore(&priv->lock, flags);

	IPW_DEBUG_MAC80211("leave\n");
	return 0;
}

//TODO dead code
#ifdef IPW_DISABLE_CODE
static int ipw_net_set_mac_address(struct net_device *dev, void *p)
{
	struct ipw_priv *priv = ieee80211_dev_hw_data(dev);
	struct sockaddr *addr = p;
	if (!is_valid_ether_addr(addr->sa_data))
		return -EADDRNOTAVAIL;
	mutex_lock(&priv->mutex);
	priv->config |= CFG_CUSTOM_MAC;
	memcpy(priv->mac_addr, addr->sa_data, ETH_ALEN);
	printk(KERN_INFO "%s: Setting MAC to " MAC_FMT "\n",
	       priv->net_dev->name, MAC_ARG(priv->mac_addr));
	IPW_DEBUG_INFO("Restarting adapter to set new MAC.\n");
	queue_work(priv->workqueue, &priv->down);
	mutex_unlock(&priv->mutex);
	return 0;
}
#endif

static irqreturn_t ipw_isr(int irq, void *data)
{
	struct ipw_priv *priv = data;
	u32 inta, inta_mask;
	if (!priv)
		return IRQ_NONE;

	spin_lock(&priv->lock);
	if (!(priv->status & STATUS_INT_ENABLED)) {
		/* Shared IRQ */
		goto none;
	}

	inta = ipw_read32(priv, CSR_INT);
	inta_mask = ipw_read32(priv, CSR_INT_MASK);
	if (inta == 0xFFFFFFFF) {
		/* Hardware disappeared */
		IPW_WARNING("IRQ INTA == 0xFFFFFFFF\n");
		goto none;
	}

	if (!(inta & (CSR_INI_SET_MASK & inta_mask))) {
		if (inta)
			ipw_write32(priv, CSR_INT, inta);
		/* Shared interrupt */
		goto none;
	}

	/* tell the device to stop sending interrupts */

	IPW_DEBUG_ISR
	    ("interrupt received 0x%08x masked 0x%08x card mask 0x%08x\n",
	     inta, inta_mask, CSR_INI_SET_MASK);

	priv->status &= ~STATUS_INT_ENABLED;
	ipw_write32(priv, CSR_INT_MASK, 0x00000000);
	/* ack current interrupts */
	ipw_write32(priv, CSR_INT, inta);
	inta &= (CSR_INI_SET_MASK & inta_mask);
	/* Cache INTA value for our tasklet */
	priv->isr_inta = inta;
	tasklet_schedule(&priv->irq_tasklet);
	spin_unlock(&priv->lock);

	return IRQ_HANDLED;

      none:
	spin_unlock(&priv->lock);
	return IRQ_NONE;
}

static void ipw_bg_rf_kill(struct work_struct *work)
{
	struct ipw_priv *priv = container_of(work, struct ipw_priv, rf_kill);

	wake_up_interruptible(&priv->wait_command_queue);

	if (priv->status & STATUS_EXIT_PENDING)
		return;

	mutex_lock(&priv->mutex);

	if (!(priv->status & STATUS_RF_KILL_MASK)) {
		IPW_DEBUG(IPW_DL_INFO | IPW_DL_RF_KILL,
			  "HW RF Kill no longer active, restarting "
			  "device\n");
		if (!(priv->status & STATUS_EXIT_PENDING))
			ipw_down(priv);
	} else {
		priv->led_state = IPW_LED_LINK_RADIOOFF;

		if (!(priv->status & STATUS_RF_KILL_HW))
			IPW_DEBUG_RF_KILL
			    ("Can not turn radio back on - "
			     "disabled by SW switch\n");
		else
			IPW_WARNING
			    ("Radio Frequency Kill Switch is On:\n"
			     "Kill switch must be turned off for "
			     "wireless networking to work.\n");
	}
	mutex_unlock(&priv->mutex);
}

static void ipw_link_up(struct ipw_priv *priv)
{
	BUG_ON(!priv->netdev_registered);

	/* Reset ieee stats */

	/* We don't reset the net_device_stats (ieee->stats) on
	 * re-association */

	priv->last_seq_num = -1;
	priv->last_frag_num = -1;
	priv->last_packet_time = 0;

	if (ieee80211_netif_oper(priv->ieee, NETIF_IS_STOPPED)) {
		IPW_DEBUG_NOTIF("waking queue\n");
		ieee80211_netif_oper(priv->ieee, NETIF_WAKE);
	} else {
		IPW_DEBUG_NOTIF("starting queue\n");
		ieee80211_netif_oper(priv->ieee, NETIF_START);
	}

	ipw_scan_cancel(priv);
	ipw_reset_stats(priv);

	/* Ensure the rate is updated immediately */
	priv->last_rate = ipw_get_current_rate(priv);
	ipw_gather_stats(priv);

	ipw_update_link_led(priv);

	if (priv->config & CFG_BACKGROUND_SCAN)
		ipw_scan_initiate(priv, 1000);
}

static void ipw_bg_update_link_led(struct work_struct *work)
{
	struct ipw_priv *priv =
	    container_of(work, struct ipw_priv, update_link_led);
	if (priv->status & STATUS_EXIT_PENDING)
		return;

	mutex_lock(&priv->mutex);
	ipw_update_link_led(priv);
	mutex_unlock(&priv->mutex);
}
static void ipw_bg_link_up(struct work_struct *work)
{
	struct ipw_priv *priv = container_of(work, struct ipw_priv, link_up);
	if (priv->status & STATUS_EXIT_PENDING)
		return;

	mutex_lock(&priv->mutex);
	ipw_link_up(priv);
	mutex_unlock(&priv->mutex);
}
static void ipw_link_down(struct ipw_priv *priv)
{
	/* ipw_link_down should never be called if we aren't
	 * registered with netdev. */
	BUG_ON(!priv->netdev_registered);

	priv->led_packets = 0;
	ieee80211_stop_queues(priv->ieee);
	/* Cancel any queued work ... */
	cancel_delayed_work(&priv->scan_check);
	cancel_delayed_work(&priv->gather_stats);
	ipw_scan_cancel(priv);
	ipw_reset_stats(priv);
	/* Queue up another scan... */

	if (priv->status & STATUS_EXIT_PENDING)
		return;

}
static void ipw_bg_link_down(struct work_struct *work)
{
	struct ipw_priv *priv = container_of(work, struct ipw_priv, link_down);
	if (priv->status & STATUS_EXIT_PENDING)
		return;

	mutex_lock(&priv->mutex);
	ipw_link_down(priv);
	mutex_unlock(&priv->mutex);
}

static void ipw_bg_resume_work(struct work_struct *work)
{
	struct ipw_priv *priv =
	    container_of(work, struct ipw_priv, resume_work.work);
	unsigned long flags;

	mutex_lock(&priv->mutex);

	/* The following it a temporary work around due to the
	 * suspend / resume not fully initializing the NIC correctly.
	 * Without all of the following, resume will not attempt to take
	 * down the NIC (it shouldn't really need to) and will just try
	 * and bring the NIC back up.  However that fails during the
	 * ucode verification process.  This then causes ipw_down to be
	 * called *after* ipw_nic_init() has succeeded -- which
	 * then lets the next init sequence succeed.  So, we've
	 * replicated all of that NIC init code here... */

	ipw_write32(priv, CSR_INT, 0xFFFFFFFF);

	ipw_nic_init(priv);

	ipw_write32(priv, CSR_UCODE_DRV_GP1_CLR, CSR_UCODE_SW_BIT_RFKILL);
	ipw_write32(priv, CSR_UCODE_DRV_GP1_CLR,
		    CSR_UCODE_DRV_GP1_BIT_CMD_BLOCKED);
	ipw_write32(priv, CSR_INT, 0xFFFFFFFF);
	ipw_write32(priv, CSR_UCODE_DRV_GP1_CLR, CSR_UCODE_SW_BIT_RFKILL);
	ipw_write32(priv, CSR_UCODE_DRV_GP1_CLR, CSR_UCODE_SW_BIT_RFKILL);

	/* tell the device to stop sending interrupts */
	ipw_disable_interrupts(priv);

	spin_lock_irqsave(&priv->lock, flags);
	ipw_clear_bit(priv, CSR_GP_CNTRL, CSR_GP_CNTRL_REG_FLAG_MAC_ACCESS_REQ);
	spin_unlock_irqrestore(&priv->lock, flags);

	spin_lock_irqsave(&priv->lock, flags);
	if (!ipw_grab_restricted_access(priv)) {
		ipw_write_restricted_reg(priv, ALM_APMG_CLK_DIS,
					 APMG_CLK_REG_VAL_DMA_CLK_RQT);
		ipw_release_restricted_access(priv);
	}
	spin_unlock_irqrestore(&priv->lock, flags);

	udelay(5);

	ipw_nic_reset(priv);

	/* Bring the device back up */
	priv->status &= ~STATUS_IN_SUSPEND;

	mutex_unlock(&priv->mutex);
}

#if 0
static void ipw_associate_timeout(struct work_struct *work)
{
	struct ipw_priv *priv =
	    container_of(work, struct ipw_priv, associate_timeout.work);

	mutex_lock(&priv->mutex);

	if ((priv->iw_mode == IEEE80211_IF_TYPE_IBSS) &&
	    (priv->assoc_id == 0xFFFD)) {
		unsigned long flags;

//              ipw_disassociate(priv);
		ipw_qos_activate(priv, 1);
		ipw_associate(priv);
		spin_lock_irqsave(&priv->lock, flags);
		priv->assoc_id = 0xFFFF;
		spin_unlock_irqrestore(&priv->lock, flags);
		udelay(50);
		spin_lock_irqsave(&priv->lock, flags);
		priv->assoc_id = 0x0;
		spin_unlock_irqrestore(&priv->lock, flags);
		ipw_post_associate(priv, priv->assoc_id);
		mutex_unlock(&priv->mutex);
		return;
	} else if (priv->iw_mode == IEEE80211_IF_TYPE_IBSS) {
		mutex_unlock(&priv->mutex);
		return;
	}

	if (priv->assoc_id) {
		ipw_post_associate(priv, priv->assoc_id);
		ipw_qos_activate(priv, 0);
	}

	IPW_DEBUG_ASSOC("Associate to a network with aid: %d\n",
			priv->assoc_id);
	mutex_unlock(&priv->mutex);
}
#endif

static void ipw_bg_rate_scale_flush(unsigned long data);

static void ipw_setup_deferred_work(struct ipw_priv *priv)
{
	priv->workqueue = create_workqueue(DRV_NAME);

	init_waitqueue_head(&priv->wait_command_queue);

	INIT_WORK(&priv->scan_completed, ipw_bg_scan_completed);
	INIT_WORK(&priv->rx_replenish, ipw_bg_rx_queue_replenish);
	INIT_WORK(&priv->rf_kill, ipw_bg_rf_kill);
	INIT_WORK(&priv->up, ipw_bg_up);
	INIT_WORK(&priv->down, ipw_bg_down);
	INIT_WORK(&priv->abort_scan, ipw_bg_abort_scan);
	INIT_WORK(&priv->link_up, ipw_bg_link_up);
	INIT_WORK(&priv->update_link_led, ipw_bg_update_link_led);
	INIT_WORK(&priv->link_down, ipw_bg_link_down);
//      INIT_WORK(&priv->associate_timeout, ipw_associate_timeout);

	INIT_DELAYED_WORK(&priv->alive_start, ipw_bg_alive_start);
	INIT_DELAYED_WORK(&priv->scan_check, ipw_bg_scan_check);
	INIT_DELAYED_WORK(&priv->request_scan, ipw_bg_request_scan);
	INIT_DELAYED_WORK(&priv->gather_stats, ipw_bg_gather_stats);
	INIT_DELAYED_WORK(&priv->activity_timer, ipw_bg_activity_timer);
	INIT_DELAYED_WORK(&priv->thermal_periodic, ipw_bg_reg_txpower_periodic);
	INIT_DELAYED_WORK(&priv->resume_work, ipw_bg_resume_work);

	/* QoS */
	INIT_WORK(&priv->qos_activate, ipw_bg_qos_activate);

	/* 802.11h */
	INIT_WORK(&priv->report_work, ipw_bg_report_work);

	INIT_WORK(&priv->post_associate, ipw_bg_post_associate);

	init_timer(&priv->rate_scale_flush);
	priv->rate_scale_flush.data = (unsigned long)priv;
	priv->rate_scale_flush.function = &ipw_bg_rate_scale_flush;

	tasklet_init(&priv->irq_tasklet, (void (*)(unsigned long))
		     ipw_irq_tasklet, (unsigned long)priv);
}

static void ipw_cancel_deferred_work(struct ipw_priv *priv)
{
	cancel_delayed_work(&priv->alive_start);
	cancel_delayed_work(&priv->request_scan);
	cancel_delayed_work(&priv->gather_stats);
	cancel_delayed_work(&priv->scan_check);
	cancel_delayed_work(&priv->activity_timer);
	cancel_delayed_work(&priv->thermal_periodic);
	cancel_delayed_work(&priv->resume_work);
	del_timer_sync(&priv->rate_scale_flush);
}

/*
  Power management (not Tx power!) functions
*/
#define PCI_LINK_CTRL      0x0F0
static int ipw_power_init_handle(struct ipw_priv *priv)
{
	int rc = 0, i;
	struct ipw_power_mgr *pow_data;
	int size = sizeof(struct ipw_power_vec_entry) * IPW_POWER_AC;
	u16 pci_pm;

	IPW_DEBUG_POWER("Initialize power \n");

	pow_data = &(priv->power_data);

	memset(pow_data, 0, sizeof(*pow_data));

	pow_data->active_index = IPW_POWER_RANGE_0;
	pow_data->dtim_val = 0xffff;

	memcpy(&pow_data->pwr_range_0[0], &range_0[0], size);
	memcpy(&pow_data->pwr_range_1[0], &range_1[0], size);

	rc = pci_read_config_word(priv->pci_dev, PCI_LINK_CTRL, &pci_pm);
	if (rc != 0)
		return 0;
	else {
		struct ipw_powertable_cmd *cmd;

		IPW_DEBUG_POWER("adjust power command flags\n");

		for (i = 0; i < IPW_POWER_AC; i++) {
			cmd = &pow_data->pwr_range_0[i].cmd;

			if (pci_pm & 0x1)
				cmd->flags &= ~0x8;
			else
				cmd->flags |= 0x8;
		}
	}
	return rc;
}

static int ipw_update_power_cmd(struct ipw_priv *priv,
				struct ipw_powertable_cmd *cmd, u32 mode)
{
	int rc = 0, i;
	u8 skip = 0;
	u32 max_sleep = 0;
	struct ipw_power_vec_entry *range;
	u8 period = 0;
	struct ipw_power_mgr *pow_data;

	if ((mode < IPW_POWER_MODE_CAM) || (mode > IPW_POWER_INDEX_5)) {
		IPW_DEBUG_POWER("Error invalid power mode \n");
		return -1;
	}
	pow_data = &(priv->power_data);

	if (pow_data->active_index == IPW_POWER_RANGE_0)
		range = &pow_data->pwr_range_0[0];
	else
		range = &pow_data->pwr_range_1[1];

	memcpy(cmd, &range[mode].cmd, sizeof(struct ipw_powertable_cmd));

#ifdef IPW_MAC80211_DISABLE
	if (priv->assoc_network != NULL) {
		unsigned long flags;

		period = priv->assoc_network->tim.tim_period;
	}
#endif				/*IPW_MAC80211_DISABLE */
	skip = range[mode].no_dtim;

	if (period == 0) {
		period = 1;
		skip = 0;
	}

	if (skip == 0) {
		max_sleep = period;
		cmd->flags &= ~PMC_TCMD_FLAG_SLEEP_OVER_DTIM_MSK;
	} else {
		max_sleep =
		    (cmd->
		     SleepInterval[PMC_TCMD_SLEEP_INTRVL_TABLE_SIZE -
				   1] / period) * period;
		cmd->flags |= PMC_TCMD_FLAG_SLEEP_OVER_DTIM_MSK;
	}

	for (i = 0; i < PMC_TCMD_SLEEP_INTRVL_TABLE_SIZE; i++) {
		if (cmd->SleepInterval[i] > max_sleep)
			cmd->SleepInterval[i] = max_sleep;
	}

	IPW_DEBUG_POWER("Flags value = 0x%08X\n", cmd->flags);
	IPW_DEBUG_POWER("Tx timeout = %u\n", cmd->TxDataTimeout);
	IPW_DEBUG_POWER("Rx timeout = %u\n", cmd->RxDataTimeout);
	IPW_DEBUG_POWER
	    ("Sleep interval vector = { %d , %d , %d , %d , %d }\n",
	     cmd->SleepInterval[0], cmd->SleepInterval[1],
	     cmd->SleepInterval[2], cmd->SleepInterval[3],
	     cmd->SleepInterval[4]);

	return rc;
}

/************* CARD-FUNCTION ************/
/* Print card information like version number other eeprom values */
static int ipw_card_show_info(struct ipw_priv *priv)
{
	IPW_DEBUG_INFO("3945ABG HW Version %u.%u.%u\n",
		       ((priv->eeprom.board_revision >> 8) & 0x0F),
		       ((priv->eeprom.board_revision >> 8) >> 4),
		       (priv->eeprom.board_revision & 0x00FF));

	IPW_DEBUG_INFO("3945ABG PBA Number %.*s\n",
		       (int)sizeof(priv->eeprom.board_pba_number),
		       priv->eeprom.board_pba_number);

	IPW_DEBUG_INFO("EEPROM_ANTENNA_SWITCH_TYPE is 0x%02X\n",
		       priv->eeprom.antenna_switch_type);

	return 0;
}

/*
  Call all registered function for card RXON status
  this function called after we call REPLY_RXON command.
*/
/*
  called at ipw_pci_remove to clean the driver data
*/
static int ipw_card_remove_notify(struct ipw_priv *priv)
{
	int rc = 0;
	return rc;
}

/*
 * called after we received REPLY_ALIVE notification.
 * this function starts the calibration then start the process
 * of transferring the card to receiving state
 */
static void ipw_bg_alive_start(struct work_struct *work)
{
	struct ipw_priv *priv =
	    container_of(work, struct ipw_priv, alive_start.work);
	int rc = 0;
	int thermal_spin = 0;

	if (priv->status & STATUS_EXIT_PENDING)
		return;

	mutex_lock(&priv->mutex);
	if (priv->card_alive.is_valid != 1) {
		/* We had an error bringing up the hardware, so take it
		 * all the way back down so we can try again */
		IPW_DEBUG_INFO("Alive failed.\n");
		ipw_down(priv);
		mutex_unlock(&priv->mutex);
		return;
	}

	/* bootstrap uCode has loaded runtime uCode ... verify inst image */
	if (ipw_verify_ucode(priv)) {
		/* Runtime instruction load was bad;
		 * take it all the way back down so we can try again */
		IPW_DEBUG_INFO("Bad runtime uCode load.\n");
		ipw_down(priv);
		mutex_unlock(&priv->mutex);
		return;
	}

	/* After the ALIVE response, we can processed host commands */
	priv->status |= STATUS_ALIVE;

	IPW_DEBUG_INFO("Alive received.\n");

	ipw_clear_stations_table(priv);

	if (!(priv->status & STATUS_RF_KILL_MASK)) {
		/* if rfkill is not on, then
		 * wait for thermal sensor in adapter to kick in */
		while (ipw_get_temperature(priv) == 0) {
			thermal_spin++;
			udelay(10);
		}
		if (thermal_spin)
			IPW_DEBUG_INFO("Thermal calibration took %dus\n",
				       thermal_spin * 10);
	}

	rc = ipw_init_channel_map(priv);
	if (rc) {
		IPW_ERROR("initializing regulatory failed: %d\n", rc);
		mutex_unlock(&priv->mutex);
		return;
	}

	ipw_init_geos(priv);

	if (!priv->netdev_registered) {
		mutex_unlock(&priv->mutex);
		ieee80211_rate_control_register(&priv->rate_control);

		rc = ieee80211_register_hw(priv->ieee);
		if (rc) {
			IPW_ERROR("Failed to register network "
				  "device (error %d)\n", rc);
			return;
		}

		mutex_lock(&priv->mutex);
		priv->netdev_registered = 1;

		ipw_reset_channel_flag(priv);
	}

	memcpy(priv->net_dev->dev_addr, priv->mac_addr, ETH_ALEN);

	priv->rates_mask = IEEE80211_DEFAULT_RATES_MASK;
	ipw_set_supported_rates_mask(priv, priv->rates_mask);

	ipw_set_rate(priv);

	ipw_send_power_mode(priv, IPW_POWER_LEVEL(priv->power_mode));

/*
 * ipw_qos_activate(priv, NULL);
 */
	ipw_send_power_mode(priv, IPW_POWER_LEVEL(priv->power_mode));

	/* Initialize our rx_config data */
	ipw_connection_init_rx_config(priv);
	memcpy(priv->staging_rxon.node_addr, priv->net_dev->dev_addr, ETH_ALEN);

	/* Configure BT coexistence */
	ipw_send_bt_config(priv);

	/* Configure the adapter for unassociated operation */
	ipw_commit_rxon(priv);

	/* Add the broadcast address so we can send probe requests */
	ipw_rxon_add_station(priv, BROADCAST_ADDR, 0);
	ipw_init_rate_scaling(priv);

	/* At this point, the NIC is initialized and operational */
	priv->notif_missed_beacons = 0;
	priv->status |= STATUS_READY;

	ipw_update_link_led(priv);

	reg_txpower_periodic(priv);

	mutex_unlock(&priv->mutex);
}

#define MAX_HW_RESTARTS 5
static int ipw_up(struct ipw_priv *priv)
{
	int rc, i;

	if (priv->status & STATUS_EXIT_PENDING) {
		IPW_WARNING("Exit pending will not bring the NIC up\n");
		return -EIO;
	}

	if (priv->status & STATUS_RF_KILL_SW) {
		IPW_WARNING("Radio disabled by module parameter.\n");
		return 0;
	} else if (priv->status & STATUS_RF_KILL_HW)
		return 0;

	ipw_write32(priv, CSR_INT, 0xFFFFFFFF);

	rc = ipw_nic_init(priv);
	if (rc) {
		IPW_ERROR("Unable to int nic\n");
		return rc;
	}

	ipw_write32(priv, CSR_UCODE_DRV_GP1_CLR, CSR_UCODE_SW_BIT_RFKILL);
	ipw_write32(priv, CSR_UCODE_DRV_GP1_CLR,
		    CSR_UCODE_DRV_GP1_BIT_CMD_BLOCKED);
	ipw_write32(priv, CSR_INT, 0xFFFFFFFF);

	ipw_enable_interrupts(priv);

	ipw_write32(priv, CSR_UCODE_DRV_GP1_CLR, CSR_UCODE_SW_BIT_RFKILL);
	ipw_write32(priv, CSR_UCODE_DRV_GP1_CLR, CSR_UCODE_SW_BIT_RFKILL);

	for (i = 0; i < MAX_HW_RESTARTS; i++) {

		ipw_clear_stations_table(priv);

		rc = ipw_setup_bootstrap(priv);
		if (rc) {
			IPW_ERROR("Unable to set up bootstrap uCode: %d\n", rc);
			continue;
		}

		/* start card; bootstrap will load runtime ucode */
		ipw_nic_start(priv);

		ipw_card_show_info(priv);

		if (!(priv->config & CFG_CUSTOM_MAC)) {
			eeprom_parse_mac(priv, priv->mac_addr);
			IPW_DEBUG_INFO("MAC address: " MAC_FMT "\n",
				       MAC_ARG(priv->mac_addr));
		}

		memcpy(priv->net_dev->dev_addr, priv->mac_addr, ETH_ALEN);
		memcpy(priv->ieee->wiphy->perm_addr, priv->mac_addr, ETH_ALEN);

		return 0;
	}

	priv->status |= STATUS_EXIT_PENDING;
	ipw_down(priv);

	/* tried to restart and config the device for as long as our
	 * patience could withstand */
	IPW_ERROR("Unable to initialize device after %d attempts.\n", i);
	return -EIO;
}

static void ipw_bg_up(struct work_struct *work)
{
	struct ipw_priv *priv = container_of(work, struct ipw_priv, up);

	if (priv->status & STATUS_EXIT_PENDING)
		return;

	mutex_lock(&priv->mutex);
	ipw_up(priv);
	mutex_unlock(&priv->mutex);
}

static void ipw_down(struct ipw_priv *priv)
{
	unsigned long flags;
	int exit_pending = priv->status & STATUS_EXIT_PENDING;
	struct ieee80211_conf *conf = NULL;

	IPW_DEBUG_INFO("ipw going down \n");

	conf = ieee80211_get_hw_conf(priv->ieee);

	priv->status |= STATUS_EXIT_PENDING;

	/* If we are coming down due to a microcode error, then
	 * don't bother trying to do anything that results in sending
	 * host commands... */
	if (!(priv->status & STATUS_FW_ERROR) && ipw_is_alive(priv)) {

		ipw_update_link_led(priv);
		ipw_update_activity_led(priv);
		ipw_update_tech_led(priv);
	}

	ipw_clear_stations_table(priv);

	/* Cancel any pending scheduled work */
	ipw_cancel_deferred_work(priv);

	/* Unblock any waiting calls */
	wake_up_interruptible_all(&priv->wait_command_queue);

	/* Wipe out the EXIT_PENDING status bit if we are not actually
	 * exiting the module */
	if (!exit_pending)
		priv->status &= ~STATUS_EXIT_PENDING;

	/* tell the device to stop sending interrupts */
	ipw_write32(priv, CSR_RESET, CSR_RESET_REG_FLAG_NEVO_RESET);
	ipw_disable_interrupts(priv);

	if (priv->netdev_registered) {
		netif_carrier_off(priv->net_dev);
		ieee80211_stop_queues(priv->ieee);
	}

	/* If we have not previously called ipw_init() then
	 * clear all bits but the RF Kill and SUSPEND bits and return */
	if (!ipw_is_init(priv)) {
		priv->status &= (STATUS_RF_KILL_MASK | STATUS_IN_SUSPEND);
		goto exit;
	}

	/* ...otherwise clear out all the status bits but the RF Kill and
	 * SUSPEND bits and continue taking the NIC down. */
	priv->status &= (STATUS_RF_KILL_MASK | STATUS_IN_SUSPEND);

	spin_lock_irqsave(&priv->lock, flags);
	ipw_clear_bit(priv, CSR_GP_CNTRL, CSR_GP_CNTRL_REG_FLAG_MAC_ACCESS_REQ);
	spin_unlock_irqrestore(&priv->lock, flags);

	ipw_stop_tx_queue(priv);
	ipw_rxq_stop(priv);

	spin_lock_irqsave(&priv->lock, flags);
	if (!ipw_grab_restricted_access(priv)) {
		ipw_write_restricted_reg(priv, ALM_APMG_CLK_DIS,
					 APMG_CLK_REG_VAL_DMA_CLK_RQT);
		ipw_release_restricted_access(priv);
	}
	spin_unlock_irqrestore(&priv->lock, flags);

	udelay(5);

	ipw_nic_stop_master(priv);

	spin_lock_irqsave(&priv->lock, flags);
	ipw_set_bit(priv, CSR_RESET, CSR_RESET_REG_FLAG_SW_RESET);
	spin_unlock_irqrestore(&priv->lock, flags);

	ipw_nic_reset(priv);

      exit:
	memset(&priv->card_alive, 0, sizeof(struct ipw_alive_resp));

	if (priv->ibss_beacon)
		dev_kfree_skb(priv->ibss_beacon);
	priv->ibss_beacon = NULL;

	if (priv->scan) {
		kfree(priv->scan);
		priv->scan = NULL;
	}

	/* clear out any free frames */
	ipw_clear_free_frames(priv);

}

static void ipw_bg_down(struct work_struct *work)
{
	struct ipw_priv *priv = container_of(work, struct ipw_priv, down);

	if (priv->status & STATUS_EXIT_PENDING)
		return;

	mutex_lock(&priv->mutex);
	ipw_down(priv);
	mutex_unlock(&priv->mutex);
}

static struct pci_device_id card_ids[] __devinitdata = {
	{0x8086, 0x4222, PCI_ANY_ID, PCI_ANY_ID, 0, 0, 0},
	{0x8086, 0x4227, PCI_ANY_ID, PCI_ANY_ID, 0, 0, 0},
	{0}
};

//MODULE_DEVICE_TABLE(pci, card_ids);
static struct attribute *ipw_sysfs_entries[] = {
	&dev_attr_tx_power.attr,
	&dev_attr_rf_kill.attr,
	&dev_attr_status.attr,
	&dev_attr_cfg.attr,
	&dev_attr_dump_errors.attr,
	&dev_attr_dump_events.attr,
	&dev_attr_roam.attr,
	&dev_attr_led.attr,
	&dev_attr_antenna.attr,
	&dev_attr_statistics.attr,
	&dev_attr_temperature.attr,
	&dev_attr_channels.attr, &dev_attr_retry_rate.attr,
	&dev_attr_power_level.attr,
	NULL
};

static int d_add_interface(struct ieee80211_hw *hw,
			   struct ieee80211_if_init_conf *conf)
{
	struct ipw_priv *priv = hw->priv;
	struct ieee80211_conf *conf_mode = NULL;
	unsigned long flags;

	IPW_DEBUG_MAC80211("enter\n");
	IPW_DEBUG_WX("call from mac80211: add_interface %d MAC "
		     MAC_FMT "\n", conf->if_id, MAC_ARG(conf->mac_addr));

	if (in_interrupt()) {
		IPW_DEBUG_WX("call inside interrupt NOT-ALLOWED\n");
		IPW_DEBUG_MAC80211("leave\n");
		return 0;
	}

	spin_lock_irqsave(&priv->lock, flags);
	priv->interface_id = conf->if_id;

	conf_mode = ieee80211_get_hw_conf(hw);

	if (conf_mode)
		priv->active_conf.phymode = conf_mode->phymode;

	spin_unlock_irqrestore(&priv->lock, flags);

	if (conf_mode)
		ipw_set_channel(priv, conf_mode);

	mutex_lock(&priv->mutex);
	ipw_set_mode(priv, conf->type);
	mutex_unlock(&priv->mutex);

	IPW_DEBUG_MAC80211("leave\n");
	return 0;
}

static void d_remove_interface(struct ieee80211_hw *hw,
			       struct ieee80211_if_init_conf *conf)
{
	struct ipw_priv *priv = hw->priv;

	IPW_DEBUG_MAC80211("enter\n");
	IPW_DEBUG_WX("call from mac80211: remove interface\n");
	if (in_interrupt()) {
		IPW_DEBUG_WX("call inside interrupt NOT-ALLOWED\n");
		IPW_DEBUG_MAC80211("leave\n");
		return;
	}
	mutex_lock(&priv->mutex);
	priv->interface_id = 0;
	memset(priv->bssid, 0, ETH_ALEN);
	memset(priv->essid, 0, IW_ESSID_MAX_SIZE);
	priv->essid_len = 0;
	mutex_unlock(&priv->mutex);

	IPW_DEBUG_MAC80211("leave\n");

}

static int d_config_interface(struct ieee80211_hw *hw,
			      int if_id, struct ieee80211_if_conf *conf)
{
	struct ipw_priv *priv = hw->priv;
	unsigned long flags;
	struct ieee80211_conf *conf_mode = NULL;

	IPW_DEBUG_MAC80211("enter\n");

	spin_lock_irqsave(&priv->lock, flags);

	if (conf->bssid) {
		IPW_DEBUG_WX("call from mac80211: config interface id %d:"
			     "bssid: " MAC_FMT "\n", if_id,
			     MAC_ARG(conf->bssid));
		memcpy(priv->staging_rxon.bssid_addr, conf->bssid, ETH_ALEN);
		if ((conf->bssid[0] + conf->bssid[1] + conf->bssid[2] +
		     conf->bssid[3] + conf->bssid[4] + conf->bssid[5]) == 0)
			priv->staging_rxon.filter_flags &=
			    ~RXON_FILTER_ASSOC_MSK;
		else
			priv->staging_rxon.filter_flags |=
			    RXON_FILTER_ASSOC_MSK;

		if (compare_ether_addr(priv->bssid, conf->bssid)) {
			memcpy(priv->bssid, conf->bssid, ETH_ALEN);
			priv->timestamp1 = 0;
			priv->timestamp0 = 0;
		}
	}

	if (check_bits(priv->staging_rxon.filter_flags,
		       RXON_FILTER_ASSOC_MSK) !=
	    check_bits(priv->active_rxon.filter_flags, RXON_FILTER_ASSOC_MSK))
		IPW_DEBUG_MAC80211("assoc change to %sassociated\n",
				   check_bits(priv->staging_rxon.filter_flags,
					      RXON_FILTER_ASSOC_MSK) ?
				   "un" : "");

	if (!conf->ssid_len) {
		priv->config &= ~CFG_STATIC_ESSID;
		memset(priv->essid, 0, IW_ESSID_MAX_SIZE);
	} else {
		priv->config |= CFG_STATIC_ESSID;
		memcpy(priv->essid, conf->ssid, conf->ssid_len);
	}

	priv->essid_len = conf->ssid_len;

	conf_mode = ieee80211_get_hw_conf(hw);

	if (conf_mode)
		priv->active_conf.phymode = conf_mode->phymode;

	spin_unlock_irqrestore(&priv->lock, flags);

	if (conf_mode)
		ipw_set_channel(priv, conf_mode);

	if (in_interrupt()) {
		IPW_WARNING("TODO: Schedule background commit..\n");
		IPW_DEBUG_MAC80211("leave\n");
		return 0;
	}

	ipw_commit_rxon(priv);
	IPW_DEBUG_MAC80211("leave\n");

	return 0;
}
static void d_set_multicast_list(struct ieee80211_hw *hw,
				 unsigned short netflags, int mc_count)
{
	IPW_DEBUG_MAC80211("enter\n");
	IPW_DEBUG_WX("call from mac80211: set multicast list\n");
	IPW_DEBUG_MAC80211("leave\n");
}

static int d_reset(struct ieee80211_hw *hw)
{
	IPW_DEBUG_MAC80211("enter\n");
	IPW_DEBUG_WX("call from mac80211: reset nic\n");
	IPW_DEBUG_MAC80211("leave\n");
	return 0;
}

/* commit_flags
 * examine staging_rxon vs. active_rxon
 * if no change from active_rxon:assoc and staging_rxon:assoc then
 * rxon_assoc.
 */
static inline int tune_required(struct ipw_priv *priv)
{
	if (compare_ether_addr(priv->staging_rxon.bssid_addr,
			       priv->active_rxon.bssid_addr))
		return 1;

	if ((priv->staging_rxon.dev_type != priv->active_rxon.dev_type) ||
	    (priv->staging_rxon.channel != priv->active_rxon.channel))
		return 1;

	/* Check if we are not switching bands */
	if (check_bits(priv->staging_rxon.flags, RXON_FLG_BAND_24G_MSK) !=
	    check_bits(priv->active_rxon.flags, RXON_FLG_BAND_24G_MSK))
		return 1;

	/* Check if we are switching association toggle */
	if (check_bits(priv->staging_rxon.filter_flags,
		       RXON_FILTER_ASSOC_MSK) &&
	    check_bits(priv->active_rxon.filter_flags, RXON_FILTER_ASSOC_MSK))
		return 0;

	return 1;
}

static int ipw_commit_rxon(struct ipw_priv *priv)
{
	int rc = 0;

	if (!ipw_is_alive(priv))
		return -1;

	/* always get timestamp with Rx frame */
	priv->staging_rxon.flags |= RXON_FLG_TSF2HOST_MSK;

	/* select antenna */
	priv->staging_rxon.flags &=
	    ~(RXON_FLG_DIS_DIV_MSK | RXON_FLG_ANT_SEL_MSK);
	priv->staging_rxon.flags |= ipw_get_antenna_flags(priv);

	/* If we don't need to retune, we can use ipw_rxon_assoc_cmd which
	 * is used to reconfigure filter and other flags for the current
	 * radio configuration.
	 *
	 * If we need to tune, we need to request the regulatory
	 * daemon to tune and configure the radio via ipw_send_rx_config. */
	if (!tune_required(priv))
		rc = ipw_send_rxon_assoc(priv);
	else {
		/* Sending the RXON command clears out the station table,
		 * so we must clear out our cached table values so we will
		 * re-add stations to the uCode for TX */
		ipw_clear_stations_table(priv);

		/* If we are currently associated and the new config requires
		 * a tune *and* the new config wants the associated mask enabled,
		 * we must clear the associated from the active configuration
		 * before we apply the new config */
		if (ipw_is_associated(priv) &&
		    (priv->staging_rxon.filter_flags & RXON_FILTER_ASSOC_MSK)) {
			IPW_DEBUG_INFO("Toggling associated bit on current "
				       "RXON\n");
			priv->active_rxon.filter_flags &=
			    ~RXON_FILTER_ASSOC_MSK;
			rc = ipw_send_cmd_pdu(priv, REPLY_RXON,
					      sizeof(struct ipw_rxon_cmd),
					      &priv->active_rxon);

			/* If the mask clearing failed then we set
			 * active_config back to what it was previously */
			if (!rc)
				priv->active_rxon.filter_flags |=
				    RXON_FILTER_ASSOC_MSK;

		}

		if (!rc)
			rc = ipw_send_cmd_pdu(priv, REPLY_RXON,
					      sizeof(struct ipw_rxon_cmd),
					      &priv->staging_rxon);
		if (!rc)
			rc = ipw_reg_send_txpower(priv);

		/* Add the broadcast address so we can send broadcast frames */
		if (!rc) {
			if (ipw_rxon_add_station(priv, BROADCAST_ADDR, 0) ==
			    IPW_INVALID_STATION)
				rc = -EIO;
		}
	}

	if (rc)
		IPW_ERROR("Error setting configuration.  Reload driver.\n");
	else
		memcpy(&priv->active_rxon, &priv->staging_rxon,
		       sizeof(priv->active_rxon));

	return rc;
}

static int d_config(struct ieee80211_hw *hw, struct ieee80211_conf *conf)
{
	struct ipw_priv *priv = hw->priv;
	unsigned long flags;
	const struct ipw_channel_info *ch_info;

	IPW_DEBUG_MAC80211("enter\n");

	if (!ipw_is_ready(priv)) {
		IPW_DEBUG_MAC80211("leave - not ready\n");
		return -EIO;
	}

/*	if (local->sta_scanning) {
		IPW_DEBUG_MAC80211("leave - scanning\n");
		return 0;
	}
*/
	IPW_DEBUG_WX("call from mac80211: config hw rxon values\n");

	spin_lock_irqsave(&priv->lock, flags);

	memcpy(&priv->active_conf, conf, sizeof(priv->active_conf));
	printk_buf(IPW_DL_MAC80211, conf, sizeof(*conf));

	ch_info = ipw_get_channel_info(priv, conf->phymode, conf->channel);
	if (!is_channel_valid(ch_info)) {
		IPW_DEBUG_SCAN("Channel %d [%d] is INVALID for this SKU.\n",
			       conf->channel, conf->phymode);
		spin_unlock_irqrestore(&priv->lock, flags);
		return -EINVAL;
	}

	priv->staging_rxon.channel = conf->channel;
	ipw_set_flags_for_channel(priv, ch_info);

	spin_unlock_irqrestore(&priv->lock, flags);

	if (check_bits(conf->flags, IEEE80211_CONF_SHORT_SLOT_TIME) !=
	    check_bits(priv->active_conf.flags,
		       IEEE80211_CONF_SHORT_SLOT_TIME)) {
		/* Long / short preamble change */
		IPW_WARNING("TODO: Look into long/short preamble change "
			    "handling.\n");
	}

	spin_lock_irqsave(&priv->lock, flags);
	priv->only_active_channel = 0;
	spin_unlock_irqrestore(&priv->lock, flags);

	mutex_lock(&priv->mutex);

	ipw_radio_kill_sw(priv, !conf->radio_enabled);

	if (priv->status & STATUS_RF_KILL_MASK) {
		mutex_unlock(&priv->mutex);
		IPW_DEBUG_MAC80211("leave - rf kill\n");
		return -EIO;
	}

	ipw_set_power(priv, conf);

	ipw_set_rate(priv);
	ipw_commit_rxon(priv);

//      reg_set_txpower(priv, conf->power_level);

	mutex_unlock(&priv->mutex);

	IPW_DEBUG_MAC80211("leave\n");
	return 0;
}

#if 0
static int d_hw_scan(struct ieee80211_hw *hw, u8 * ssid, size_t len)
{
	unsigned long flags;
	struct ipw_priv *priv = hw->priv;

	IPW_DEBUG_MAC80211("enter\n");
	spin_lock_irqsave(&priv->lock, flags);

	IPW_DEBUG_SCAN("scan requested\n");
	priv->only_active_channel = 0;
	if (len) {
		IPW_DEBUG_SCAN("direct scan for '%s'\n",
			       iwl_escape_essid(ssid, len));

		priv->one_direct_scan = 1;
		priv->direct_ssid_len = (u8)
		    min((u8) len, (u8) IW_ESSID_MAX_SIZE);
		memcpy(priv->direct_ssid, ssid, priv->direct_ssid_len);
	}

	ipw_scan_initiate(priv, 0);

	spin_unlock_irqrestore(&priv->lock, flags);

	IPW_DEBUG_MAC80211("leave\n");
	return 0;
}
#endif

static int ipw_update_sta_key_info(struct ipw_priv *priv,
				   struct ieee80211_key_conf *keyconf,
				   u8 sta_id)
{
	unsigned long flags;
	u16 key_flags = 0;

	switch (keyconf->alg) {
	case ALG_CCMP:
		key_flags |= STA_KEY_FLG_CCMP;
		key_flags |= keyconf->keyidx << STA_KEY_FLG_KEYID_POS;
		key_flags &= ~STA_KEY_FLG_INVALID;
		break;
	case ALG_TKIP:
	case ALG_WEP:
		return -EINVAL;
	default:
		return -EINVAL;
	}
	spin_lock_irqsave(&priv->sta_lock, flags);
	priv->stations[sta_id].keyinfo.alg = keyconf->alg;
	priv->stations[sta_id].keyinfo.keylen = keyconf->keylen;
	memcpy(priv->stations[sta_id].keyinfo.key, keyconf->key,
	       keyconf->keylen);

	memcpy(priv->stations[sta_id].sta.key.key, keyconf->key,
	       keyconf->keylen);
	priv->stations[sta_id].sta.key.key_flags = key_flags;
	priv->stations[sta_id].sta.sta.modify_mask |= STA_MODIFY_KEY_MASK;
	priv->stations[sta_id].sta.ctrlAddModify |= STA_CONTROL_MODIFY_MSK;

	spin_unlock_irqrestore(&priv->sta_lock, flags);

	printk("hwcrypto: modify ucode station key info\n");
	ipw_send_add_station(priv, &priv->stations[sta_id].sta, 0);
	return 0;
}

static int ipw_set_rxon_hwcrypto(struct ipw_priv *priv, int hw_decrypt)
{
	struct ipw_rxon_cmd *rxon = &priv->staging_rxon;

	if (hw_decrypt)
		rxon->filter_flags &= ~RXON_FILTER_DIS_DECRYPT_MSK;
	else
		rxon->filter_flags |= RXON_FILTER_DIS_DECRYPT_MSK;

	return 0;
}
static int d_set_key(struct ieee80211_hw *hw,
		     set_key_cmd cmd,
		     u8 * addr, struct ieee80211_key_conf *key, int aid)
{
	struct ipw_priv *priv = hw->priv;
	//struct ieee80211_conf *conf = ieee80211_get_hw_conf(hw);
	int rc = 0;
	u8 sta_id;

	IPW_DEBUG_MAC80211("enter\n");
	IPW_DEBUG_WX("call from mac80211: set security key\n");

	if (!param_hwcrypto) {
		printk("hwcrypto disabled!\n");
		IPW_DEBUG_MAC80211("leave - hwcrypto disabled\n");
		return -EOPNOTSUPP;
	}

	sta_id = ipw_find_station(priv, addr);
	if (sta_id == IPW_INVALID_STATION) {
		IPW_DEBUG_WX("Station " MAC_FMT
			     " not in station map.\n", MAC_ARG(addr));
		IPW_DEBUG_MAC80211("leave - invalid station\n");
		return -EINVAL;
	}

	mutex_lock(&priv->mutex);

	switch (cmd) {
	case SET_KEY:
		if (!(rc = ipw_update_sta_key_info(priv, key, sta_id))) {
			ipw_set_rxon_hwcrypto(priv, 1);
			ipw_commit_rxon(priv);
			key->flags &= (u32)
			    (~IEEE80211_KEY_FORCE_SW_ENCRYPT);
			key->hw_key_idx = sta_id;
			//TODO do we need below
			/*
			   conf->sw_encrypt = 0;
			   conf->sw_decrypt = 0;
			 */
			IPW_DEBUG_WX("set_key success, using hwcrypto\n");
			printk("set_key success, using hwcrypto\n");
		}
		break;
	case DISABLE_KEY:
	case REMOVE_ALL_KEYS:
	default:
		rc = -EINVAL;
	}

	mutex_unlock(&priv->mutex);
	IPW_DEBUG_MAC80211("leave\n");
	return rc;
}

static int d_conf_tx(struct ieee80211_hw *hw, int queue, const struct ieee80211_tx_queue_params
		     *params)
{
	struct ipw_priv *priv = hw->priv;
	unsigned long flags;
	int i = 0;

	IPW_DEBUG_MAC80211("enter\n");
	IPW_DEBUG_WX("call from mac80211: config queue for QoS\n");

	if (!ipw_is_ready_rf(priv)) {
		IPW_DEBUG_MAC80211("leave - not ready\n");
		return -EIO;
	}

	if (queue > AC_NUM) {
		IPW_DEBUG_MAC80211("leave\n");
		return 0;
	}

	if (!priv->qos_data.qos_enable) {
		priv->qos_data.qos_active = 0;
		return 0;
	}
	switch (queue) {
	case 0:
		i = 3;
		break;
	case 1:
		i = 2;
		break;
	case 2:
		i = 1;
		break;
	case 3:
		i = 0;
		break;
	}

	spin_lock_irqsave(&priv->lock, flags);

	priv->qos_data.def_qos_parm.ac[i].dot11CWmin = (u16)
	    params->cw_min;
	priv->qos_data.def_qos_parm.ac[i].dot11CWmax = (u16)
	    params->cw_max;
	priv->qos_data.def_qos_parm.ac[i].dot11AIFSN = (u8)
	    params->aifs;
	priv->qos_data.def_qos_parm.ac[i].edca_txop = (u16)
	    (params->burst_time * 100);

	priv->qos_data.def_qos_parm.ac[i].reserved1 = 0;
	priv->qos_data.qos_active = IPW_QOS_WMM;

	spin_unlock_irqrestore(&priv->lock, flags);
	if (in_interrupt()) {
		if (i == AC_NUM - 1) {
			queue_work(priv->workqueue, &priv->qos_activate);
			IPW_DEBUG_QOS("New QoS data schedule Qos Cmd\n");
		}
	}

	if (i == AC_NUM - 1) {
		ipw_qos_activate(priv, 0);
		IPW_DEBUG_QOS("New QoS data schedule Qos Cmd\n");
	}
	IPW_DEBUG_MAC80211("leave\n");
	return 0;
}

static int d_get_tx_stats(struct ieee80211_hw *hw,
			  struct ieee80211_tx_queue_stats *stats)
{
	struct ipw_priv *priv = hw->priv;
	int i, avail;
	struct ipw_tx_queue *txq;
	struct ipw_queue *q;
	unsigned long flags;
	IPW_DEBUG_MAC80211("enter\n");

	if (!ipw_is_ready_rf(priv)) {
		IPW_DEBUG_MAC80211("leave - not ready\n");
		return -EIO;
	}

	spin_lock_irqsave(&priv->lock, flags);

	for (i = 0; i < AC_NUM; i++) {
		txq = &priv->txq[i];
		q = &txq->q;
		avail = ipw_queue_space(q);

		stats->data[i].len = q->n_window - avail;
		stats->data[i].limit = q->n_window - q->high_mark;
		stats->data[i].count = q->n_window;

	}
	spin_unlock_irqrestore(&priv->lock, flags);

	IPW_DEBUG_MAC80211("leave\n");
	return 0;
}

static int d_get_stats(struct ieee80211_hw *hw,
		       struct ieee80211_low_level_stats *stats)
{
	struct ipw_priv *priv = hw->priv;

	IPW_DEBUG_MAC80211("enter\n");
	IPW_DEBUG_WX("call from mac80211: get stats\n");

	if (!ipw_is_ready_rf(priv)) {
		IPW_DEBUG_MAC80211("leave - not ready\n");
		return -EIO;
	}
	if (in_interrupt()) {
		IPW_DEBUG_WX("call inside interrupt NOT-ALLOWED\n");
		IPW_DEBUG_MAC80211("leave - in interrupt\n");
		return 0;
	}

	mutex_lock(&priv->mutex);

	mutex_unlock(&priv->mutex);
	IPW_DEBUG_MAC80211("leave\n");
	return 0;
}

static u64 d_get_tsf(struct ieee80211_hw *hw)
{
	struct ipw_priv *priv = hw->priv;

	IPW_DEBUG_MAC80211("enter\n");
	IPW_DEBUG_WX("call from mac80211: get tsf value\n");

	if (!ipw_is_ready_rf(priv))
		return -EIO;

	if (in_interrupt()) {
		IPW_DEBUG_WX("call inside interrupt NOT-ALLOWED\n");
		IPW_DEBUG_MAC80211("leave - in interrupt\n");
		return 0;
	}

	IPW_DEBUG_MAC80211("leave\n");
	return 0;
}

static int
d_beacon_update(struct ieee80211_hw *hw,
		struct sk_buff *skb, struct ieee80211_tx_control *control)
{
	struct ipw_priv *priv = hw->priv;
	unsigned long flags;
	struct ieee80211_conf *conf = NULL;

	if (!ipw_is_ready_rf(priv))
		return -EIO;

	conf = ieee80211_get_hw_conf(hw);

	IPW_DEBUG_WX("call from mac80211: update beacon for "
		     "channel %d\n", conf->channel);

	if (priv->iw_mode != IEEE80211_IF_TYPE_IBSS)
		return -EIO;

	spin_lock_irqsave(&priv->lock, flags);

	if (priv->ibss_beacon)
		dev_kfree_skb(priv->ibss_beacon);

	priv->ibss_beacon = skb;

	priv->active_conf.phymode = conf->phymode;

	priv->assoc_id = 0xFFFD;

	queue_work(priv->workqueue, &priv->associate_timeout);

	spin_unlock_irqrestore(&priv->lock, flags);
	return 0;
}

static void d_reset_tsf(struct ieee80211_hw *hw)
{
	struct ipw_priv *priv = hw->priv;
	unsigned long flags;
	int counter = 6;

	IPW_DEBUG_WX("call from mac80211: reset tsf\n");
	if (!priv)
		return;

	spin_lock_irqsave(&priv->lock, flags);
	priv->assoc_id = 0;
	priv->assoc_capability = 0;

	/* new association get rid of ibss beacon skb */
	if (priv->ibss_beacon)
		dev_kfree_skb(priv->ibss_beacon);

	priv->ibss_beacon = NULL;

	if ((priv->iw_mode == IEEE80211_IF_TYPE_STA)) {
		priv->timestamp1 = 0;
		priv->timestamp0 = 0;
	}

	spin_unlock_irqrestore(&priv->lock, flags);

	ipw_set_default_qos(priv);

	if (!ipw_is_ready_rf(priv))
		return;

	if (in_interrupt()) {
		IPW_DEBUG_WX("call inside interrupt NOT-ALLOWED\n");
		return;
	}

	mutex_lock(&priv->mutex);

	priv->only_active_channel = 0;

	while ((priv->status & STATUS_SCANNING) && counter--)
		schedule_timeout_interruptible(HZ);

//      ipw_disassociate(priv);

	ipw_set_rate(priv);
	ipw_qos_activate(priv, 1);

	/* for IBSS we will handle association in
	   beacon_update callback from ieee stack */
	if ((priv->iw_mode == IEEE80211_IF_TYPE_STA)) {

		ipw_associate(priv);

		IPW_DEBUG(IPW_DL_STATE, "associating: "
			  "'%s' " MAC_FMT " channel id %d\n",
			  iwl_escape_essid(priv->essid, priv->essid_len),
			  MAC_ARG(priv->bssid), priv->active_conf.channel);
	}

	mutex_unlock(&priv->mutex);

}

static struct attribute_group ipw_attribute_group = {
	.name = NULL,		/* put in device directory */
	.attrs = ipw_sysfs_entries,
};

static struct ieee80211_ops iwlwifi_hw_ops = {
	.tx = d_tx,
	.open = d_open,
	.stop = d_stop,
	.add_interface = d_add_interface,
	.remove_interface = d_remove_interface,
	.reset = d_reset,

	/* Changes to HW config within the config handler will result in
	 * retunes of the NIC, which requires at Tx power command be sent */
	.config = d_config,

	/* Changes to the run-time interface config impact the associated state */
	.config_interface = d_config_interface,

	.set_multicast_list = d_set_multicast_list,
	.set_key = d_set_key,
	.get_stats = d_get_stats,
	.get_tx_stats = d_get_tx_stats,
	.conf_tx = d_conf_tx,
	.get_tsf = d_get_tsf,
	.reset_tsf = d_reset_tsf,
	.beacon_update = d_beacon_update,
//      .hw_scan = d_hw_scan,
};

static int ipw3945_nic_stop_master(struct ipw_priv *);
static int ipw3945_stop_tx_queue(struct ipw_priv *);
static int ipw3945_tx_reset(struct ipw_priv *);
static int ipw3945_queue_init(struct ipw_priv *, struct ipw_queue *,
			      int, int, u32);
static int ipw3945_queue_tx_free_tfd(struct ipw_priv *, struct ipw_tx_queue *);
static int ipw3945_tx_queue_update_wr_ptr(struct ipw_priv *,
					  struct ipw_tx_queue *, int, u16);
static int ipw3945_queue_tx_init(struct ipw_priv *,
				 struct ipw_tx_queue *, int, u32);
static int ipw3945_queue_tx_free(struct ipw_priv *, struct ipw_tx_queue *);
static int ipw3945_rx_queue_update_wr_ptr(struct ipw_priv *,
					  struct ipw_rx_queue *);
static int ipw3945_rxq_stop(struct ipw_priv *);
static int ipw3945_rx_init(struct ipw_priv *, struct ipw_rx_queue *);
static int ipw3945_attach_buffer_to_tfd_frame(void *, dma_addr_t, u16);
static int ipw3945_get_rx_read(struct ipw_priv *);
static int ipw3945_tx_cmd(struct ipw_priv *, struct ipw_cmd *, u8, dma_addr_t,
			  struct ieee80211_hdr *, u8,
			  struct ieee80211_tx_control *);
static int ipw3945_rx_cmd(struct ipw_priv *, struct ipw_rx_mem_buffer *, int);
static int ipw3945_get_beacon_cmd(struct ipw_priv *, struct ipw_frame *, u16);
static int ipw3945_send_rxon_assoc(struct ipw_priv *);
static int ipw3945_send_power_mode(struct ipw_priv *, u32);
static int ipw3945_init_hw_rates(struct ipw_priv *, struct ieee80211_rate *);

/*************** RATE-SCALING functions ***********************/

#define NUM_RATES 12

static s8 ipw_lower_rate_g[NUM_RATES] = {
	RATE_SCALE_5_5M_INDEX, RATE_SCALE_5_5M_INDEX,
	RATE_SCALE_11M_INDEX, RATE_SCALE_12M_INDEX,
	RATE_SCALE_18M_INDEX, RATE_SCALE_24M_INDEX,
	RATE_SCALE_36M_INDEX, RATE_SCALE_48M_INDEX,
	255, RATE_SCALE_1M_INDEX, RATE_SCALE_2M_INDEX,
	RATE_SCALE_5_5M_INDEX,

};

static s8 ipw_higher_rate_g[NUM_RATES] = {
	RATE_SCALE_11M_INDEX, RATE_SCALE_11M_INDEX,
	RATE_SCALE_18M_INDEX,
	RATE_SCALE_24M_INDEX, RATE_SCALE_36M_INDEX,
	RATE_SCALE_48M_INDEX,
	RATE_SCALE_54M_INDEX, 255, RATE_SCALE_2M_INDEX,
	RATE_SCALE_5_5M_INDEX, RATE_SCALE_11M_INDEX,
	RATE_SCALE_12M_INDEX
};

static s32 ipw_expected_tpt_g[NUM_RATES] = {
	0, 0, 76, 104, 130, 168, 191, 202, 7, 13, 35, 58
};

static s32 ipw_expected_tpt_g_prot[NUM_RATES] = {
	0, 0, 0, 80, 93, 113, 123, 125, 7, 13, 35, 58
};

static s32 ipw_expected_tpt_a[NUM_RATES] = {
	40, 57, 72, 98, 121, 154, 177, 186, 0, 0, 0, 0
};

static s32 ipw_expected_tpt_b[NUM_RATES] = {
	0, 0, 0, 0, 0, 0, 0, 0, 7, 13, 35, 58
};

static s8 ipw_lower_rate_g_prot[NUM_RATES] = {
	RATE_SCALE_5_5M_INDEX, RATE_SCALE_5_5M_INDEX,
	RATE_SCALE_11M_INDEX,
	RATE_SCALE_11M_INDEX, RATE_SCALE_18M_INDEX,
	RATE_SCALE_24M_INDEX,
	RATE_SCALE_36M_INDEX, RATE_SCALE_48M_INDEX, 255,
	RATE_SCALE_1M_INDEX, RATE_SCALE_2M_INDEX,
	RATE_SCALE_5_5M_INDEX,
};

static s8 ipw_higher_rate_g_prot[NUM_RATES] = {
	RATE_SCALE_11M_INDEX, RATE_SCALE_11M_INDEX,
	RATE_SCALE_18M_INDEX,
	RATE_SCALE_24M_INDEX, RATE_SCALE_36M_INDEX,
	RATE_SCALE_48M_INDEX,
	RATE_SCALE_54M_INDEX, 255, RATE_SCALE_2M_INDEX,
	RATE_SCALE_5_5M_INDEX, RATE_SCALE_11M_INDEX,
	RATE_SCALE_18M_INDEX
};

static struct ipw_tpt_entry ipw_tpt_table_a[] = {
	{-60, 22000, 0, 0, RATE_SCALE_54M_INDEX},
	{-64, 20000, 0, 0, RATE_SCALE_48M_INDEX},
	{-72, 18000, 0, 0, RATE_SCALE_36M_INDEX},
	{-80, 16000, 0, 0, RATE_SCALE_24M_INDEX},
	{-84, 12000, 0, 0, RATE_SCALE_18M_INDEX},
	{-85, 8000, 0, 0, RATE_SCALE_12M_INDEX},
	{-87, 7000, 0, 0, RATE_SCALE_9M_INDEX},
	{-89, 5000, 0, 0, RATE_SCALE_6M_INDEX}
};

static struct ipw_tpt_entry ipw_tpt_table_b[] = {
	{-86, 6000, 0, 0, RATE_SCALE_11M_INDEX},
	{-88, 3000, 0, 0, RATE_SCALE_5_5M_INDEX},
	{-90, 1000, 0, 0, RATE_SCALE_2M_INDEX},
	{-92, 800, 0, 0, RATE_SCALE_1M_INDEX}

};

static struct ipw_tpt_entry ipw_tpt_table_g[] = {
	{-60, 22000, 12000, 14000, RATE_SCALE_54M_INDEX},
	{-64, 20000, 11000, 13000, RATE_SCALE_48M_INDEX},
	{-68, 18000, 10000, 14000, RATE_SCALE_36M_INDEX},
	{-80, 16000, 9000, 11000, RATE_SCALE_24M_INDEX},
	{-84, 12000, 7000, 10000, RATE_SCALE_18M_INDEX},
	{-85, 8000, 5000, 8000, RATE_SCALE_12M_INDEX},
	{-86, 6000, 6000, 6000, RATE_SCALE_11M_INDEX},
	{-88, 3000, 3000, 3000, RATE_SCALE_5_5M_INDEX},
	{-90, 1000, 1000, 1000, RATE_SCALE_2M_INDEX},
	{-92, 800, 800, 800, RATE_SCALE_1M_INDEX}
};

static struct ipw_tpt_entry *ipw_get_tpt_by_rssi(s32 rssi, u8 mode)
{
	u32 index = 0;
	u32 table_size = 0;
	struct ipw_tpt_entry *tpt_table = NULL;

	if ((rssi < IPW_MIN_RSSI_VAL) || (rssi > IPW_MAX_RSSI_VAL))
		rssi = IPW_MIN_RSSI_VAL;

	switch (mode) {
	case MODE_IEEE80211G:
		tpt_table = ipw_tpt_table_g;
		table_size = ARRAY_SIZE(ipw_tpt_table_g);
		break;

	case MODE_IEEE80211B:
		tpt_table = ipw_tpt_table_b;
		table_size = ARRAY_SIZE(ipw_tpt_table_b);
		break;

	case MODE_IEEE80211A:
		tpt_table = ipw_tpt_table_a;
		table_size = ARRAY_SIZE(ipw_tpt_table_a);
		break;

	default:
		BUG();
		return NULL;
	}

	while ((index < table_size)
	       && (rssi < tpt_table[index].min_rssi))
		index++;

	index = min(index, (table_size - 1));

	return &tpt_table[index];
}

static int ipw_init_rate_scaling(struct ipw_priv *priv)
{
	int rc;
	struct ipw_rate_scaling_cmd_specifics *cmd;
	struct ipw_rate_scaling_info *table;
	unsigned long flags;
	int i;

	cmd = &priv->lq_mngr.scale_rate_cmd;
	table = &cmd->table[0];

	spin_lock_irqsave(&priv->lq_mngr.lock, flags);

	priv->lq_mngr.flush_time = IPW_RATE_SCALE_FLUSH;
	priv->lq_mngr.stamp_last = jiffies;
	priv->lq_mngr.tx_packets = 0;

	switch (priv->active_conf.phymode) {
	case MODE_IEEE80211A:
		IPW_DEBUG_RATE("Select A mode rate scale\n");

		table[RATE_SCALE_6M_INDEX].try_cnt = priv->retry_rate;
		table[RATE_SCALE_6M_INDEX].next_rate_index =
		    RATE_SCALE_6M_INDEX;
		table[RATE_SCALE_9M_INDEX].try_cnt = priv->retry_rate;
		table[RATE_SCALE_9M_INDEX].next_rate_index =
		    RATE_SCALE_6M_INDEX;
		table[RATE_SCALE_12M_INDEX].try_cnt = priv->retry_rate;
		table[RATE_SCALE_12M_INDEX].next_rate_index =
		    RATE_SCALE_9M_INDEX;
		table[RATE_SCALE_18M_INDEX].try_cnt = priv->retry_rate;
		table[RATE_SCALE_18M_INDEX].next_rate_index =
		    RATE_SCALE_12M_INDEX;
		table[RATE_SCALE_24M_INDEX].try_cnt = priv->retry_rate;
		table[RATE_SCALE_24M_INDEX].next_rate_index =
		    RATE_SCALE_18M_INDEX;
		table[RATE_SCALE_36M_INDEX].try_cnt = priv->retry_rate;
		table[RATE_SCALE_36M_INDEX].next_rate_index =
		    RATE_SCALE_24M_INDEX;
		table[RATE_SCALE_48M_INDEX].try_cnt = priv->retry_rate;
		table[RATE_SCALE_48M_INDEX].next_rate_index =
		    RATE_SCALE_36M_INDEX;
		table[RATE_SCALE_54M_INDEX].try_cnt = priv->retry_rate;
		table[RATE_SCALE_54M_INDEX].next_rate_index =
		    RATE_SCALE_48M_INDEX;

		/* If one of the following CCK rates is used,
		 * have it fall back to an above OFDM rate */
		table[RATE_SCALE_1M_INDEX].try_cnt = priv->retry_rate;
		table[RATE_SCALE_1M_INDEX].next_rate_index =
		    RATE_SCALE_6M_INDEX;
		table[RATE_SCALE_2M_INDEX].try_cnt = priv->retry_rate;
		table[RATE_SCALE_2M_INDEX].next_rate_index =
		    RATE_SCALE_6M_INDEX;
		table[RATE_SCALE_5_5M_INDEX].try_cnt = priv->retry_rate;
		table[RATE_SCALE_5_5M_INDEX].next_rate_index =
		    RATE_SCALE_6M_INDEX;
		table[RATE_SCALE_11M_INDEX].try_cnt = priv->retry_rate;
		table[RATE_SCALE_11M_INDEX].next_rate_index =
		    RATE_SCALE_6M_INDEX;

		for (i = NUM_RATES; i < IPW_MAX_RATES; i++) {
			table[i].try_cnt = priv->retry_rate;
			table[i].next_rate_index = RATE_SCALE_6M_INDEX;
		}
		break;
	case MODE_IEEE80211B:
		IPW_DEBUG_RATE("Select B mode rate scale\n");

		/* If one of the following OFDM rates is used,
		 * have it fall back to the CCK rates at the end */
		table[RATE_SCALE_6M_INDEX].try_cnt = priv->retry_rate;
		table[RATE_SCALE_6M_INDEX].next_rate_index =
		    RATE_SCALE_1M_INDEX;
		table[RATE_SCALE_9M_INDEX].try_cnt = priv->retry_rate;
		table[RATE_SCALE_9M_INDEX].next_rate_index =
		    RATE_SCALE_1M_INDEX;
		table[RATE_SCALE_12M_INDEX].try_cnt = priv->retry_rate;
		table[RATE_SCALE_12M_INDEX].next_rate_index =
		    RATE_SCALE_1M_INDEX;
		table[RATE_SCALE_18M_INDEX].try_cnt = priv->retry_rate;
		table[RATE_SCALE_18M_INDEX].next_rate_index =
		    RATE_SCALE_1M_INDEX;
		table[RATE_SCALE_24M_INDEX].try_cnt = priv->retry_rate;
		table[RATE_SCALE_24M_INDEX].next_rate_index =
		    RATE_SCALE_1M_INDEX;
		table[RATE_SCALE_36M_INDEX].try_cnt = priv->retry_rate;
		table[RATE_SCALE_36M_INDEX].next_rate_index =
		    RATE_SCALE_1M_INDEX;
		table[RATE_SCALE_48M_INDEX].try_cnt = priv->retry_rate;
		table[RATE_SCALE_48M_INDEX].next_rate_index =
		    RATE_SCALE_1M_INDEX;
		table[RATE_SCALE_54M_INDEX].try_cnt = priv->retry_rate;
		table[RATE_SCALE_54M_INDEX].next_rate_index =
		    RATE_SCALE_1M_INDEX;

		/* CCK rates... */
		table[RATE_SCALE_1M_INDEX].try_cnt = priv->retry_rate;
		table[RATE_SCALE_1M_INDEX].next_rate_index =
		    RATE_SCALE_1M_INDEX;
		table[RATE_SCALE_2M_INDEX].try_cnt = priv->retry_rate;
		table[RATE_SCALE_2M_INDEX].next_rate_index =
		    RATE_SCALE_1M_INDEX;
		table[RATE_SCALE_5_5M_INDEX].try_cnt = priv->retry_rate;
		table[RATE_SCALE_5_5M_INDEX].next_rate_index =
		    RATE_SCALE_2M_INDEX;
		table[RATE_SCALE_11M_INDEX].try_cnt = priv->retry_rate;
		table[RATE_SCALE_11M_INDEX].next_rate_index =
		    RATE_SCALE_5_5M_INDEX;
		for (i = NUM_RATES; i < IPW_MAX_RATES; i++) {
			table[i].try_cnt = priv->retry_rate;
			table[i].next_rate_index = RATE_SCALE_1M_INDEX;
		}
		break;
	case MODE_IEEE80211G:
		IPW_DEBUG_RATE("Select G mode rate scale\n");

		table[RATE_SCALE_6M_INDEX].try_cnt = priv->retry_rate;
		table[RATE_SCALE_6M_INDEX].next_rate_index =
		    RATE_SCALE_2M_INDEX;
		table[RATE_SCALE_9M_INDEX].try_cnt = priv->retry_rate;
		table[RATE_SCALE_9M_INDEX].next_rate_index =
		    RATE_SCALE_6M_INDEX;
		table[RATE_SCALE_12M_INDEX].try_cnt = priv->retry_rate;
		table[RATE_SCALE_12M_INDEX].next_rate_index =
		    RATE_SCALE_9M_INDEX;
		table[RATE_SCALE_18M_INDEX].try_cnt = priv->retry_rate;
		table[RATE_SCALE_18M_INDEX].next_rate_index =
		    RATE_SCALE_12M_INDEX;
		table[RATE_SCALE_24M_INDEX].try_cnt = priv->retry_rate;
		table[RATE_SCALE_24M_INDEX].next_rate_index =
		    RATE_SCALE_18M_INDEX;
		table[RATE_SCALE_36M_INDEX].try_cnt = priv->retry_rate;
		table[RATE_SCALE_36M_INDEX].next_rate_index =
		    RATE_SCALE_24M_INDEX;
		table[RATE_SCALE_48M_INDEX].try_cnt = priv->retry_rate;
		table[RATE_SCALE_48M_INDEX].next_rate_index =
		    RATE_SCALE_36M_INDEX;
		table[RATE_SCALE_54M_INDEX].try_cnt = priv->retry_rate;
		table[RATE_SCALE_54M_INDEX].next_rate_index =
		    RATE_SCALE_48M_INDEX;
		table[RATE_SCALE_1M_INDEX].try_cnt = priv->retry_rate;
		table[RATE_SCALE_1M_INDEX].next_rate_index =
		    RATE_SCALE_1M_INDEX;
		table[RATE_SCALE_2M_INDEX].try_cnt = priv->retry_rate;
		table[RATE_SCALE_2M_INDEX].next_rate_index =
		    RATE_SCALE_1M_INDEX;
		table[RATE_SCALE_5_5M_INDEX].try_cnt = priv->retry_rate;
		table[RATE_SCALE_5_5M_INDEX].next_rate_index =
		    RATE_SCALE_2M_INDEX;
		table[RATE_SCALE_11M_INDEX].try_cnt = priv->retry_rate;
		table[RATE_SCALE_11M_INDEX].next_rate_index =
		    RATE_SCALE_5_5M_INDEX;
		for (i = NUM_RATES; i < IPW_MAX_RATES; i++) {
			table[i].try_cnt = priv->retry_rate;
			table[i].next_rate_index = RATE_SCALE_6M_INDEX;

		}
		break;
	}

	spin_unlock_irqrestore(&priv->lq_mngr.lock, flags);

	/* Update the rate scaling for control frame Tx */
	cmd->table_id = 0;
	rc = ipw_send_cmd_pdu(priv, REPLY_RATE_SCALE, sizeof(*cmd), cmd);
	if (rc)
		return rc;

	/* Update the rate scaling for data frame Tx */
	cmd->table_id = 1;
	rc = ipw_send_cmd_pdu(priv, REPLY_RATE_SCALE, sizeof(*cmd), cmd);

	return rc;
}

static int ipw_rate_scale_clear_window(struct ipw_rate_scale_data
				       *window)
{

	window->data = 0;
	window->success_counter = 0;
	window->success_ratio = IPW_INVALID_VALUE;
	window->counter = 0;
	window->average_tpt = IPW_INVALID_VALUE;
	window->stamp = 0;
	return 0;
}

static int ipw_rate_scale_flush_old(struct ipw_priv *priv,
				    struct ipw_rate_scale_data *ctl,
				    u32 flush_interval)
{
	int rc = 0;
	s32 rate_index = 0;
	unsigned long flags;
	struct ipw_rate_scale_data *window;

	for (rate_index = 0; rate_index < NUM_RATES; rate_index++) {
		window = &(ctl[rate_index]);

		spin_lock_irqsave(&priv->lq_mngr.lock, flags);
		if (window->counter > 0) {
			if (time_after(jiffies, window->stamp + flush_interval)) {
				IPW_DEBUG_RATE
				    ("flushed samples of rate index %d\n",
				     rate_index);
				ipw_rate_scale_clear_window(window);
			} else
				rc = 1;

		}
		spin_unlock_irqrestore(&priv->lq_mngr.lock, flags);
	}

	return rc;
}

#define IPW_RATE_SCALE_AVG_PACKETS      300
#define IPW_RATE_FLUSH_MAX              5000	// msec
#define IPW_RATE_FLUSH_MIN              50	// msec

static void ipw_bg_rate_scale_flush(unsigned long data)
{
	struct ipw_priv *priv = (struct ipw_priv *)data;
	int rc = 0, i;
	unsigned long flags;
	u32 tx_ave, duration;
	struct ipw_rate_scale_data *ctl;
	struct ipw_rate_scale_priv *scale_priv;
	struct sta_info *sta;
	struct ieee80211_local *local = wdev_priv(priv->net_dev->ieee80211_ptr);

	if (priv->status & STATUS_EXIT_PENDING)
		return;

	if (!ipw_is_associated(priv))
		return;

	if (priv->iw_mode != IEEE80211_IF_TYPE_IBSS) {

		sta = sta_info_get(local, priv->bssid);
		if (sta && sta->rate_ctrl_priv) {
			scale_priv = (struct ipw_rate_scale_priv *)
			    sta->rate_ctrl_priv;
			ctl = (struct ipw_rate_scale_data *)
			    &(scale_priv->win[0]);
			rc = ipw_rate_scale_flush_old(priv,
						      ctl,
						      IPW_RATE_SCALE_WIN_FLUSH);
			sta_info_put(sta);
		} else if (sta)
			sta_info_put(sta);
	} else {
		struct ipw_station_entry *stations;
		int count = 0;

		stations = &(priv->stations[0]);
		for (i = STA_ID; i <
		     (priv->hw_setting.number_of_stations - 1); i++) {
			if ((stations[i].used)) {
				count++;
				sta = sta_info_get(local,
						   stations[i].sta.sta.MACAddr);

				if (sta && sta->rate_ctrl_priv) {
					scale_priv =
					    (struct ipw_rate_scale_priv *)
					    sta->rate_ctrl_priv;

					ctl = (struct ipw_rate_scale_data *)
					    &(scale_priv->win[0]);
					rc |=
					    ipw_rate_scale_flush_old(priv,
								     ctl,
								     IPW_RATE_SCALE_WIN_FLUSH);

					sta_info_put(sta);
				} else if (sta)
					sta_info_put(sta);

			}
			if (count >= (priv->num_stations - 1))
				break;
		}
	}
	spin_lock_irqsave(&priv->lq_mngr.lock, flags);

	tx_ave = (priv->tx_packets - priv->lq_mngr.tx_packets)
	    + 1;

	priv->lq_mngr.tx_packets = priv->tx_packets + 1;

	if (rc && !(priv->status & STATUS_EXIT_PENDING)) {

		duration = priv->lq_mngr.flush_time * 1000 / HZ;

		IPW_DEBUG_RATE("Packets: %d duration: %d msec\n",
			       tx_ave, duration);
		if (duration == 0)
			tx_ave = 0;
		else
			tx_ave = (tx_ave * 1000) / duration;

		if (tx_ave == 0)
			duration = IPW_RATE_FLUSH_MAX;
		else {
			duration = (IPW_RATE_SCALE_AVG_PACKETS) / tx_ave;

			if (duration > IPW_RATE_FLUSH_MAX)
				duration = IPW_RATE_FLUSH_MAX;
			else if (duration < IPW_RATE_FLUSH_MIN)
				duration = IPW_RATE_FLUSH_MIN;
		}
		priv->lq_mngr.flush_time = duration * HZ / 1000;

		IPW_DEBUG_RATE("new flush period: %d msec ave %d\n",
			       duration, tx_ave);

		mod_timer(&priv->rate_scale_flush, jiffies +
			  priv->lq_mngr.flush_time);

		priv->lq_mngr.stamp = jiffies;
	}

	priv->lq_mngr.stamp_last = jiffies;

	spin_unlock_irqrestore(&priv->lq_mngr.lock, flags);
}

static int ipw_rate_scale_init_handle(struct ipw_priv *priv, s32 window_size)
{
	int rc = 0;
	unsigned long flags;
	struct ipw_rate_scaling_cmd_specifics *cmd;
	struct ipw_rate_scaling_info *table;

	cmd = &priv->lq_mngr.scale_rate_cmd;
	memset(cmd, 0, sizeof(*cmd));
	table = &cmd->table[0];

	IPW_DEBUG_RATE("initialize rate scale window to %d\n", window_size);

	if ((window_size > IPW_RATE_SCALE_MAX_WINDOW)
	    || (window_size < 0))
		window_size = IPW_RATE_SCALE_MAX_WINDOW;

	spin_lock_irqsave(&priv->lq_mngr.lock, flags);

	priv->lq_mngr.expected_tpt = NULL;
	priv->lq_mngr.next_higher_rate = NULL;
	priv->lq_mngr.next_lower_rate = NULL;

	priv->lq_mngr.stamp = jiffies;
	priv->lq_mngr.stamp_last = jiffies;
	priv->lq_mngr.flush_time = IPW_RATE_SCALE_FLUSH;
	priv->lq_mngr.tx_packets = 0;

	priv->lq_mngr.max_window_size = window_size;

	table[RATE_SCALE_6M_INDEX].tx_rate = R_6M;
	table[RATE_SCALE_6M_INDEX].try_cnt = priv->retry_rate;
	table[RATE_SCALE_6M_INDEX].next_rate_index = RATE_SCALE_11M_INDEX;

	table[RATE_SCALE_9M_INDEX].tx_rate = R_9M;
	table[RATE_SCALE_9M_INDEX].try_cnt = priv->retry_rate;
	table[RATE_SCALE_9M_INDEX].next_rate_index = RATE_SCALE_6M_INDEX;

	table[RATE_SCALE_12M_INDEX].tx_rate = R_12M;
	table[RATE_SCALE_12M_INDEX].try_cnt = priv->retry_rate;
	table[RATE_SCALE_12M_INDEX].next_rate_index = RATE_SCALE_9M_INDEX;

	table[RATE_SCALE_18M_INDEX].tx_rate = R_18M;
	table[RATE_SCALE_18M_INDEX].try_cnt = priv->retry_rate;
	table[RATE_SCALE_18M_INDEX].next_rate_index = RATE_SCALE_12M_INDEX;

	table[RATE_SCALE_24M_INDEX].tx_rate = R_24M;
	table[RATE_SCALE_24M_INDEX].try_cnt = priv->retry_rate;
	table[RATE_SCALE_24M_INDEX].next_rate_index = RATE_SCALE_18M_INDEX;

	table[RATE_SCALE_36M_INDEX].tx_rate = R_36M;
	table[RATE_SCALE_36M_INDEX].try_cnt = priv->retry_rate;
	table[RATE_SCALE_36M_INDEX].next_rate_index = RATE_SCALE_24M_INDEX;

	table[RATE_SCALE_48M_INDEX].tx_rate = R_48M;
	table[RATE_SCALE_48M_INDEX].try_cnt = priv->retry_rate;
	table[RATE_SCALE_48M_INDEX].next_rate_index = RATE_SCALE_36M_INDEX;

	table[RATE_SCALE_54M_INDEX].tx_rate = R_54M;
	table[RATE_SCALE_54M_INDEX].try_cnt = priv->retry_rate;
	table[RATE_SCALE_54M_INDEX].next_rate_index = RATE_SCALE_48M_INDEX;

	table[RATE_SCALE_1M_INDEX].tx_rate = R_1M;
	table[RATE_SCALE_1M_INDEX].try_cnt = priv->retry_rate;
	table[RATE_SCALE_1M_INDEX].next_rate_index = RATE_SCALE_1M_INDEX;

	table[RATE_SCALE_2M_INDEX].tx_rate = R_2M;
	table[RATE_SCALE_2M_INDEX].try_cnt = priv->retry_rate;
	table[RATE_SCALE_2M_INDEX].next_rate_index = RATE_SCALE_1M_INDEX;

	table[RATE_SCALE_5_5M_INDEX].tx_rate = R_5_5M;
	table[RATE_SCALE_5_5M_INDEX].try_cnt = priv->retry_rate;
	table[RATE_SCALE_5_5M_INDEX].next_rate_index = RATE_SCALE_2M_INDEX;

	table[RATE_SCALE_11M_INDEX].tx_rate = R_11M;
	table[RATE_SCALE_11M_INDEX].try_cnt = priv->retry_rate;
	table[RATE_SCALE_11M_INDEX].next_rate_index = RATE_SCALE_5_5M_INDEX;

	spin_unlock_irqrestore(&priv->lq_mngr.lock, flags);

	return rc;
}

static int ipw_collect_tx_data(struct ipw_priv *priv,
			       struct ipw_rate_scale_data *windows,
			       int scale_index, u32 status)
{
	int rc = 0;
	struct ipw_rate_scale_data *window = NULL;
	unsigned long flags;
	u64 mask;

	if (scale_index < 0)
		return -1;

	if ((priv->lq_mngr.max_window_size < 0) ||
	    (priv->lq_mngr.max_window_size > IPW_RATE_SCALE_MAX_WINDOW))
		return -1;

	if (scale_index >= IPW_MAX_RATES)
		return -1;

	spin_lock_irqsave(&priv->lq_mngr.lock, flags);

	window = &(windows[scale_index]);
	if (window->counter >= priv->lq_mngr.max_window_size) {

		window->counter = priv->lq_mngr.max_window_size;
		window->counter = window->counter - 1;
		mask = (1 << (priv->lq_mngr.max_window_size - 1));
		if ((window->data & mask) != 0) {
			window->data &= ~mask;
			window->success_counter = window->success_counter - 1;
		}
	}

	window->counter = window->counter + 1;
	mask = window->data;
	window->data = (mask << 1);
	if (status != 0) {
		window->success_counter = window->success_counter + 1;
		window->data |= 0x1;
	}

	if (window->counter > 0) {
		window->success_ratio = 128 * (100 * window->success_counter)
		    / window->counter;
	} else
		window->success_ratio = IPW_INVALID_VALUE;

	window->stamp = jiffies;

	spin_unlock_irqrestore(&priv->lq_mngr.lock, flags);

	return rc;
}

static int ipw_rate_scale_rxon_handle(struct ipw_priv *priv, s32 sta_id)
{
	int rc = 0;
	int i;
	s32 rssi = 0;
	struct ipw_tpt_entry *entry = NULL;
	unsigned long flags;
	struct ieee80211_conf *conf = NULL;

	conf = ieee80211_get_hw_conf(priv->ieee);

	if (!ipw_is_associated(priv))
		return 0;

	spin_lock_irqsave(&priv->lq_mngr.lock, flags);

	priv->lq_mngr.next_lower_rate = ipw_lower_rate_g;
	priv->lq_mngr.next_higher_rate = ipw_higher_rate_g;

	switch (priv->active_conf.phymode) {
	case MODE_IEEE80211G:
		if (priv->active_rxon.flags & RXON_FLG_TGG_PROTECT_MSK) {
			priv->lq_mngr.expected_tpt = ipw_expected_tpt_g_prot;
			priv->lq_mngr.next_higher_rate = ipw_higher_rate_g_prot;
			priv->lq_mngr.next_lower_rate = ipw_lower_rate_g_prot;
		} else {
			priv->lq_mngr.expected_tpt = ipw_expected_tpt_g;
			priv->lq_mngr.next_lower_rate = ipw_lower_rate_g;
			priv->lq_mngr.next_higher_rate = ipw_higher_rate_g;
		}
		break;

	case MODE_IEEE80211B:
		priv->lq_mngr.expected_tpt = ipw_expected_tpt_b;
		break;

	case MODE_IEEE80211A:
		priv->lq_mngr.expected_tpt = ipw_expected_tpt_a;
		break;
	default:
		printk(KERN_WARNING "Invalid mode in rxon_handle\n");
		return -EINVAL;
	}

	rssi = priv->last_rx_rssi;
	if (rssi == 0)
		rssi = IPW_MIN_RSSI_VAL;

	IPW_DEBUG_INFO("Network RSSI: %d\n", rssi);

	IPW_WARNING("Network RSSI: %d\n", rssi);
	entry = ipw_get_tpt_by_rssi(rssi, priv->active_conf.phymode);
	if (entry)
		i = entry->rate_scale_index;
	else if (priv->active_conf.phymode == MODE_IEEE80211A)
		i = RATE_SCALE_6M_INDEX;
	else
		i = RATE_SCALE_1M_INDEX;

	priv->stations[sta_id].current_rate.s.rate = ipw_rate_scale2plcp(i);

	ipw_sync_station(priv, sta_id,
			 priv->stations[sta_id].current_rate.rate_n_flags,
			 CMD_ASYNC | CMD_NO_LOCK);

	IPW_DEBUG_RATE
	    ("for rssi %d assign rate scale index %d plcp %x\n", rssi,
	     i, priv->stations[sta_id].sta.tx_rate.rate_n_flags);

	spin_unlock_irqrestore(&priv->lq_mngr.lock, flags);

	return rc;
}

static int ipw_convert_to_rate_index(struct ipw_priv *priv, int rate)
{
	if (priv->active_conf.phymode == MODE_IEEE80211A)
		rate += 4;

	if (rate < 0)
		rate = 0;

	if (rate > IPW_MAX_RATES)
		rate = IPW_MAX_RATES;

	return rate;
}

static void ipw_rate_scale_rate_init(void *priv_rate, void *priv_sta,
				     struct ieee80211_local *local,
				     struct sta_info *sta)
{
	int i;
	struct ipw_priv *priv = (struct ipw_priv *)priv_rate;

	sta->txrate = 4;
	IPW_DEBUG_RATE("rate scale global init\n");
	/* TODO: what is a good starting rate for STA? About middle? Maybe not
	 * the lowest or the highest rate.. Could consider using RSSI from
	 * previous packets? Need to have IEEE 802.1X auth succeed immediately
	 * after assoc.. */

	if (priv && (priv->iw_mode == IEEE80211_IF_TYPE_IBSS)) {
		u8 sta_id;

		sta_id = ipw_add_station(priv, sta->addr,
					 0, (CMD_ASYNC | CMD_NO_LOCK));
	}

	for (i = 0; i < local->num_curr_rates; i++) {
		if ((sta->supp_rates & BIT(i)) &&
		    (local->curr_rates[i].flags & IEEE80211_RATE_SUPPORTED))
			sta->txrate = i;
	}
	sta->last_txrate = ipw_convert_to_rate_index(priv, sta->txrate);
#ifdef CONFIG_IWLWIFI_HT
	memcpy(&priv->current_assoc_ht, &sta->ht_info,
	       sizeof(struct sta_ht_info));

	IPW_WARNING("XXXY is HT network %d \n", sta->ht_info.is_ht);
#endif
	if (priv && (priv->iw_mode != IEEE80211_IF_TYPE_STA))
		queue_work(priv->workqueue, &priv->associate_timeout);

}

static void *ipw_rate_scale_alloc(struct ieee80211_local *local)
{
	IPW_DEBUG_RATE("allocate memory for global rate scale\n");
	IPW_WARNING("XXXY start rate scale\n");
	return local->hw.priv;
}

static void ipw_rate_scale_free(void *priv)
{
	IPW_DEBUG_RATE("free global rate scale pointer\n");
}

static void *ipw_rate_scale_alloc_sta(void *priv, gfp_t gfp)
{
	struct ipw_rate_scale_priv *rctrl;
	int i;

	IPW_DEBUG_RATE("create station rate scale window\n");
	rctrl = kzalloc(sizeof(struct ipw_rate_scale_priv), gfp);
	if (rctrl == NULL) {
		return NULL;
	}

	for (i = 0; i < IPW_MAX_RATES; i++)
		ipw_rate_scale_clear_window(&(rctrl->win[i]));

	return rctrl;
}

static void ipw_rate_scale_free_sta(void *priv, void *priv_sta)
{
	struct ipw_rate_scale_priv *rctrl = priv_sta;

	IPW_DEBUG_RATE("remove station rate scale window\n");
	kfree(rctrl);
}

static void ipw_rate_scale_clear(void *priv)
{
	IPW_DEBUG_RATE("rate scale clear\n");
}

static void ipw_rate_scale_tx_resp_handle(void *priv_rate,
					  struct net_device *dev,
					  struct sk_buff *skb,
					  struct ieee80211_tx_status *tx_resp)
{
	int rc = 0, i;
	u8 retries, current_count;
	int scale_rate_index, first_index, last_index;
	struct ipw_rate_scaling_info *table;
	unsigned long flags;
	struct sta_info *sta;
	struct ieee80211_hdr *hdr = (struct ieee80211_hdr *)skb->data;
	struct ipw_priv *priv = (struct ipw_priv *)priv_rate;
	struct ieee80211_local *local = wdev_priv(dev->ieee80211_ptr);
	struct ipw_rate_scale_data *window = NULL;
	struct ipw_rate_scale_priv *scale_priv;

	IPW_DEBUG_RATE("getting frame ack response, update rate "
		       "scale window\n");

	if (!ipw_is_associated(priv))
		return;

	retries = tx_resp->retry_count;

	first_index = ipw_rate_plcp2scale(tx_resp->control.tx_rate);

	if ((first_index < 0) || (first_index >= IPW_MAX_RATES)) {
		return;
	}

	sta = sta_info_get(local, hdr->addr1);

	if (!sta || !sta->rate_ctrl_priv) {
		if (sta)
			sta_info_put(sta);
		return;
	}

	scale_priv = (struct ipw_rate_scale_priv *)
	    sta->rate_ctrl_priv;

	window = (struct ipw_rate_scale_data *)
	    &(scale_priv->win[0]);

	scale_rate_index = first_index;
	last_index = first_index;

	table = &priv->lq_mngr.scale_rate_cmd.table[0];
	if (table[scale_rate_index].try_cnt == 0) {
		scale_rate_index = table[scale_rate_index].next_rate_index;
		last_index = scale_rate_index;
	}

	while (retries > 0) {
		if (table[scale_rate_index].try_cnt > retries) {
			current_count = retries;
			last_index = scale_rate_index;
		} else {
			current_count = table[scale_rate_index].try_cnt;
			last_index = table[scale_rate_index].next_rate_index;
		}

		for (i = 0; i < current_count; i++)
			rc = ipw_collect_tx_data(priv, window,
						 scale_rate_index, 0);

		retries -= current_count;

		scale_rate_index = table[scale_rate_index].next_rate_index;
	}

	if (tx_resp->flags & IEEE80211_TX_STATUS_ACK)
		rc = ipw_collect_tx_data(priv, window, last_index, 1);
	else
		rc = ipw_collect_tx_data(priv, window, last_index, 0);

	sta_info_put(sta);

	spin_lock_irqsave(&priv->lq_mngr.lock, flags);
	if (time_after(jiffies, priv->lq_mngr.stamp + priv->lq_mngr.flush_time)) {
		mod_timer(&priv->rate_scale_flush,
			  jiffies + IPW_RATE_SCALE_WIN_FLUSH);
		priv->lq_mngr.stamp = jiffies;

	}
	spin_unlock_irqrestore(&priv->lq_mngr.lock, flags);

	return;
}

static int ipw_convert_to_phy_index(struct ipw_priv *priv, int rate)
{
	if (priv->active_conf.phymode == MODE_IEEE80211A) {
		rate -= 4;

		if (rate < 0)
			rate = 0;
	}

	return rate;
}

static struct ieee80211_rate *ipw_get_lowest_rate(struct ieee80211_local
						  *local)
{
	int i;

	for (i = 0; i < local->num_curr_rates; i++) {
		struct ieee80211_rate *rate = &local->curr_rates[i];

		if (rate->flags & IEEE80211_RATE_SUPPORTED)
			return rate;
	}

	return &local->curr_rates[0];
}

static int ipw_get_adjacent_rate(struct ipw_priv *priv,
				 s32 scale_index,
				 u32 supported_rate_mask, s32 * low, s32 * high)
{
	int rc = 0;
	s32 index, find_index;
	u16 msk;
	u16 rate_mask = (u16) supported_rate_mask;

	*low = IPW_INVALID_RATE;
	*high = IPW_INVALID_RATE;

	rate_mask = (u16) ((supported_rate_mask >> 4) |
			   ((supported_rate_mask & 0xf) << 8));

	index = scale_index;

	if (priv->active_conf.phymode == MODE_IEEE80211A) {
		find_index = index - 1;
		while ((find_index >= 0) && (find_index < IPW_MAX_RATES)) {
			msk = (1 << find_index);
			if (msk & rate_mask) {
				*low = find_index;
				break;
			}
			find_index--;
		}

		find_index = index + 1;
		while ((find_index >= 0) && (find_index < IPW_MAX_RATES)) {
			msk = (1 << find_index);
			if (msk & rate_mask) {
				*high = find_index;
				break;
			}
			find_index++;
		}
		return rc;
	}

	if (priv->lq_mngr.next_lower_rate != NULL) {
		*low = scale_index;

		find_index = scale_index;

		while (find_index != IPW_INVALID_RATE) {
			*low = priv->lq_mngr.next_lower_rate[*low];

			if (*low == IPW_INVALID_RATE)
				break;

			find_index = *low;

			msk = (1 << find_index);
			if (msk & rate_mask) {
				break;
			}
		}
	}

	if (priv->lq_mngr.next_higher_rate != NULL) {
		*high = scale_index;

		find_index = scale_index;

		while (find_index != IPW_INVALID_RATE) {
			*high = priv->lq_mngr.next_higher_rate[*high];

			if (*high == IPW_INVALID_RATE)
				break;
			find_index = *high;

			msk = (1 << find_index);
			if (msk & rate_mask) {
				break;
			}
		}
	}

	return rc;
}

/*
  find the highest available rate for REPLY_TX command.
*/
static struct ieee80211_rate *ipw_get_tx_rate(void *priv_rate,
					      struct net_device *dev,
					      struct sk_buff *skb,
					      struct rate_control_extra
					      *extra)
{

	int low = IPW_INVALID_RATE;
	int high = IPW_INVALID_RATE;
	int index, rc = 0;
	int i;
	u8 rate;
	struct ipw_rate_scale_priv *scale_priv;
	struct ipw_rate_scale_data *window = NULL;
	int current_tpt = IPW_INVALID_VALUE;
	int low_tpt = IPW_INVALID_VALUE;
	int high_tpt = IPW_INVALID_VALUE;
	u32 fail_count;
	s8 scale_action = 0;
	unsigned long flags;
	struct ieee80211_local *local = wdev_priv(dev->ieee80211_ptr);
	struct ieee80211_hdr *hdr = (struct ieee80211_hdr *)skb->data;
	struct sta_info *sta;
	u16 fc, rate_mask, ctrl_rate;
	struct ipw_priv *priv = (struct ipw_priv *)priv_rate;

	IPW_DEBUG_RATE("rate scale calculate new rate for skb\n");

	memset(extra, 0, sizeof(*extra));

	fc = le16_to_cpu(hdr->frame_control);
	if (WLAN_FC_GET_TYPE(fc) != IEEE80211_FTYPE_DATA
	    || (hdr->addr1[0] & 0x01)) {
		/* Send management frames and broadcast/multicast data using
		 * lowest rate. */
		/* TODO: this could probably be improved.. */
		return ipw_get_lowest_rate(local);
	}

	sta = sta_info_get(local, hdr->addr1);

	if (!sta || !sta->rate_ctrl_priv) {
		if (sta)
			sta_info_put(sta);
		return NULL;
	}

	i = ipw_convert_to_rate_index(priv, sta->txrate);

	rate = ipw_rate_index2plcp(i);

	ipw_get_supported_rates(priv, hdr, &rate_mask, &ctrl_rate);

	//mask with station rate restriction
	if (priv->active_conf.phymode == MODE_IEEE80211A)
		rate_mask &= (sta->supp_rates << 4);
	else
		rate_mask &= sta->supp_rates;

	if ((i >= 0) && ((1 << i) & rate_mask))
		goto done;

	/* get the lowest available rate */
	rate = IPW_INVALID_RATE;
	for (i = 0; i < 12; i++) {
		if ((1 << i) & rate_mask) {
			rate = ipw_rate_index2plcp(i);
			if (rate != IPW_INVALID_RATE)
				break;
		}
	}
      done:
	if (rate == IPW_INVALID_RATE) {
		sta_info_put(sta);
		return ipw_get_lowest_rate(local);
	}

	index = ipw_rate_plcp2scale(rate);

	spin_lock_irqsave(&priv->lq_mngr.lock, flags);

	scale_priv = (struct ipw_rate_scale_priv *)
	    sta->rate_ctrl_priv;

	window = &(scale_priv->win[index]);

	fail_count = window->counter - window->success_counter;

	if (((fail_count <= IPW_RATE_SCALE_MIN_FAILURE_TH) &&
	     (window->success_counter < IPW_RATE_SCALE_MIN_SUCCESS_TH))
	    || (priv->lq_mngr.expected_tpt == NULL)) {

		window->average_tpt = IPW_INVALID_VALUE;
		spin_unlock_irqrestore(&priv->lq_mngr.lock, flags);
		goto out;

	} else {
		window->average_tpt = ((window->success_ratio *
					priv->lq_mngr.
					expected_tpt[index] + 64) / 128);
	}

	rc = ipw_get_adjacent_rate(priv, index, rate_mask, &low, &high);

	if (rc) {
		spin_unlock_irqrestore(&priv->lq_mngr.lock, flags);
		goto out;

	}

	current_tpt = window->average_tpt;

	if (low != IPW_INVALID_RATE) {
		low_tpt = scale_priv->win[low].average_tpt;
	}

	if (high != IPW_INVALID_RATE)
		high_tpt = scale_priv->win[high].average_tpt;

	spin_unlock_irqrestore(&priv->lq_mngr.lock, flags);

	scale_action = 1;

	if ((window->success_ratio < IPW_RATE_SCALE_DECREASE_TH) ||
	    (current_tpt == 0)) {
		IPW_DEBUG_RATE("decrease rate because of low success_ratio\n");
		scale_action = -1;
	} else if ((low_tpt == IPW_INVALID_VALUE) &&
		   (high_tpt == IPW_INVALID_VALUE)) {
		scale_action = 1;
	} else if ((low_tpt != IPW_INVALID_VALUE) &&
		   (high_tpt != IPW_INVALID_VALUE)
		   && (low_tpt < current_tpt)
		   && (high_tpt < current_tpt)) {

		scale_action = 0;
	} else {
		if (high_tpt != IPW_INVALID_VALUE) {
			if (high_tpt > current_tpt)
				scale_action = 1;
			else {
				IPW_DEBUG_RATE
				    ("decrease rate because of high tpt\n");
				scale_action = -1;
			}
		} else if (low_tpt != IPW_INVALID_VALUE) {
			if (low_tpt > current_tpt) {
				IPW_DEBUG_RATE
				    ("decrease rate because of low tpt\n");
				scale_action = -1;
			} else
				scale_action = 1;
		}
	}

	if ((window->success_ratio > IPW_RATE_SCALE_HIGH_TH) ||
	    (current_tpt > (128 * window->average_tpt))) {
		scale_action = 0;
	}

	switch (scale_action) {
	case -1:
		if (low != IPW_INVALID_RATE)
			index = low;
		break;

	case 1:
		if (high != IPW_INVALID_RATE)
			index = high;

		break;

	case 0:
	default:
		break;
	}

	IPW_DEBUG_RATE
	    ("choose rate scale index %d action %d low %d high %d\n",
	     index, scale_action, low, high);

      out:

	i = ipw_rate_scale2index(index);
	sta->last_txrate = ipw_convert_to_phy_index(priv, i);
	sta->txrate = sta->last_txrate;
	sta_info_put(sta);

	return &priv->ieee_rates[i];
}

static char *rate_scale_name = "iwlwifi rate-scale";
static void ipw_set_rate_scale_handlers(struct ipw_priv *priv)
{
	priv->rate_control.module = NULL;
	priv->rate_control.name = rate_scale_name;
	priv->rate_control.tx_status = &ipw_rate_scale_tx_resp_handle;
	priv->rate_control.get_rate = &ipw_get_tx_rate;
	priv->rate_control.rate_init = ipw_rate_scale_rate_init;
	priv->rate_control.clear = ipw_rate_scale_clear;
	priv->rate_control.alloc = ipw_rate_scale_alloc;
	priv->rate_control.free = ipw_rate_scale_free;
	priv->rate_control.alloc_sta = ipw_rate_scale_alloc_sta;
	priv->rate_control.free_sta = ipw_rate_scale_free_sta;
}

static int ipw_pci_probe(struct pci_dev *pdev, const struct pci_device_id *ent)
{
	int err = 0;
	struct net_device *net_dev;
	void __iomem *base;
	u32 length;
	u32 pci_id;
	struct ipw_priv *priv;
	struct ieee80211_hw *ieee;
	int i;
	struct ieee80211_local *local;

	ieee = ieee80211_alloc_hw(sizeof(struct ipw_priv), &iwlwifi_hw_ops);
	if (ieee == NULL) {
		IPW_ERROR("Can not allocate network device\n");
		err = -ENOMEM;
		goto out;
	}
	SET_IEEE80211_DEV(ieee, &pdev->dev);
	ieee->max_rssi = 60;

	local = hw_to_local(ieee);

	net_dev = local->mdev;

	IPW_DEBUG_INFO("*** LOAD DRIVER ***\n");
	priv = ieee->priv;
	priv->net_dev = net_dev;
	priv->ieee = ieee;

	priv->pci_dev = pdev;
	priv->rxq = NULL;
	priv->antenna = param_antenna;
	ipw_debug_level = param_debug;
	priv->retry_rate = 1;

	priv->ibss_beacon = NULL;
	priv->last_scan = 0;

	ieee = priv->ieee;

	local->scan_flags |= IEEE80211_SCAN_EXTRA_INFO;
	local->short_preamble = 1;

	ieee->flags = IEEE80211_HW_WEP_INCLUDE_IV;
	ieee->queues = 4;

	spin_lock_init(&priv->lock);
	spin_lock_init(&priv->power_data.lock);
	spin_lock_init(&priv->sta_lock);
	spin_lock_init(&priv->lq_mngr.lock);

	for (i = 0; i < IPW_IBSS_MAC_HASH_SIZE; i++)
		INIT_LIST_HEAD(&priv->ibss_mac_hash[i]);

	INIT_LIST_HEAD(&priv->free_frames);

	mutex_init(&priv->mutex);
	if (pci_enable_device(pdev)) {
		err = -ENODEV;
		goto out_ieee80211_free_hw;
	}

	pci_set_master(pdev);

	ipw_clear_stations_table(priv);

	memset(&(priv->txq[0]), 0,
	       sizeof(struct ipw_tx_queue) * IPW_MAX_NUM_QUEUES);
	memset(&priv->card_alive, 0, sizeof(struct ipw_alive_resp));
	priv->data_retry_limit = -1;
	priv->ieee_channels = NULL;
	priv->ieee_rates = NULL;

	err = pci_set_dma_mask(pdev, DMA_32BIT_MASK);
	if (!err)
		err = pci_set_consistent_dma_mask(pdev, DMA_32BIT_MASK);
	if (err) {
		printk(KERN_WARNING DRV_NAME ": No suitable DMA available.\n");
		goto out_pci_disable_device;
	}

	pci_set_drvdata(pdev, priv);
	err = pci_request_regions(pdev, DRV_NAME);
	if (err)
		goto out_pci_disable_device;
	/* We disable the RETRY_TIMEOUT register (0x41) to keep
	 * PCI Tx retries from interfering with C3 CPU state */
	pci_write_config_byte(pdev, 0x41, 0x00);

	length = pci_resource_len(pdev, 0);
	priv->hw_len = length;
	base = ioremap_nocache(pci_resource_start(pdev, 0), length);
	if (!base) {
		err = -ENODEV;
		goto out_pci_release_regions;
	}

	priv->hw_base = base;
	IPW_DEBUG_INFO("pci_resource_len = 0x%08x\n", length);
	IPW_DEBUG_INFO("pci_resource_base = %p\n", base);

	/* Initialize module parameter values here */

	if (!param_led)
		priv->config |= CFG_NO_LED;
	if (param_auto_create)
		priv->config |= CFG_ADHOC_CREATE;
	else
		IPW_DEBUG_INFO("Auto adhoc creation disabled.\n");
	if (param_disable) {
		priv->status |= STATUS_RF_KILL_SW;
		IPW_DEBUG_INFO("Radio disabled.\n");
	}

	/*************************************/
	switch (param_mode) {
	case 1:
		priv->iw_mode = IEEE80211_IF_TYPE_IBSS;
		break;
	case 2:
		priv->iw_mode = IEEE80211_IF_TYPE_MNTR;
		break;
	default:
	case 0:
		priv->iw_mode = IEEE80211_IF_TYPE_STA;
		break;
	}

	priv->freq_band = IEEE80211_24GHZ_BAND;

	pci_id =
	    (priv->pci_dev->device << 16) | priv->pci_dev->subsystem_device;

	priv->is_3945 = 1;

	switch (pci_id) {
	case 0x42221005:	/* 0x4222 0x8086 0x1005 is BG SKU */
	case 0x42221034:	/* 0x4222 0x8086 0x1034 is BG SKU */
	case 0x42271014:	/* 0x4227 0x8086 0x1014 is BG SKU */
	case 0x42221044:	/* 0x4222 0x8086 0x1044 is BG SKU */
		priv->is_abg = 0;
		break;

	default:		/* Rest are assumed ABG SKU -- if this is not the
				 * case then the card will get the wrong 'Detected'
				 * line in the kernel log however the code that
				 * initializes the GEO table will detect no A-band
				 * channels and remove the is_abg mask. */
		priv->freq_band |= IEEE80211_52GHZ_BAND;
		priv->is_abg = 1;
		break;
	}

	printk(KERN_INFO DRV_NAME
	       ": Detected Intel PRO/Wireless 3945%s Network Connection\n",
	       priv->is_abg ? "ABG" : "BG");

	if (param_channel != 0) {
		priv->config |= CFG_STATIC_CHANNEL;
		priv->active_conf.channel = param_channel;
		IPW_DEBUG_INFO("Bind to static channel %d\n", param_channel);
		/* TODO: Validate that provided channel is in range */
	} else
		priv->active_conf.channel = 1;

	ipw_set_rate_scale_handlers(priv);

	ipw_setup_deferred_work(priv);

	priv->rates_mask = IEEE80211_DEFAULT_RATES_MASK;
	priv->missed_beacon_threshold = IPW_MB_DISASSOCIATE_THRESHOLD_DEFAULT;
	priv->rts_threshold = DEFAULT_RTS_THRESHOLD;
	/* If power management is turned on, default to AC mode */
	priv->power_mode = IPW_POWER_AC;
	err = request_irq(pdev->irq, ipw_isr, SA_SHIRQ, DRV_NAME, priv);
	if (err) {
		IPW_ERROR("Error allocating IRQ %d\n", pdev->irq);
		goto out_destroy_workqueue;
	}

	SET_MODULE_OWNER(net_dev);
	SET_NETDEV_DEV(net_dev, &pdev->dev);
	mutex_lock(&priv->mutex);

	priv->perfect_rssi = -20;
	priv->worst_rssi = -95;
	memset(&priv->qos_data, 0, sizeof(struct ipw_qos_info));
	if (param_qos_enable)
		priv->qos_data.qos_enable = 1;
	priv->qos_data.qos_active = 0;
	priv->qos_data.qos_cap.val = 0;

	net_dev->irq = pdev->irq;
	net_dev->base_addr = (unsigned long)priv->hw_base;
	net_dev->mem_start = pci_resource_start(pdev, 0);
	net_dev->mem_end = net_dev->mem_start + pci_resource_len(pdev, 0) - 1;

	err = sysfs_create_group(&pdev->dev.kobj, &ipw_attribute_group);
	if (err) {
		IPW_ERROR("failed to create sysfs device attributes\n");
		mutex_unlock(&priv->mutex);
		goto out_release_irq;
	}

	/* fetch uCode file from disk, copy to bus-master buffers */
	err = ipw_read_ucode(priv);
	if (err) {
		IPW_ERROR("Could not read microcode from disk: %d\n", err);
		goto out_remove_sysfs;
	}

	priv->hw_setting.irq_tasklet = NULL;

	priv->hw_setting.shared_virt =
	    pci_alloc_consistent(priv->pci_dev,
				 sizeof(struct ipw_shared_t),
				 &priv->hw_setting.shared_phys);

	if (!priv->hw_setting.shared_virt) {
		IPW_ERROR("failed to allocate pci memory\n");
		mutex_unlock(&priv->mutex);
		goto out_pci_alloc;
	}

	priv->hw_setting.eeprom_size = sizeof(struct ipw_eeprom);
	priv->hw_setting.cmd_queue_no = IPW3945_CMD_QUEUE_NUM;
	priv->hw_setting.number_of_stations = IPW3945_NUM_OF_STATIONS;
	priv->hw_setting.broadcast_id = IPW3945_BROADCAST_ID;
	priv->hw_setting.max_num_rate = IPW_MAX_RATES;
	priv->hw_setting.max_queue_number = TFD_QUEUE_MAX;
	priv->hw_setting.ac_queue_count = AC_NUM;
	priv->hw_setting.rx_buffer_size = IPW3945_RX_BUF_SIZE;
	priv->hw_setting.max_inst_size = ALM_RTC_INST_SIZE;
	priv->hw_setting.max_data_size = ALM_RTC_DATA_SIZE;
	priv->hw_setting.start_cmd_queue = 0;
	priv->hw_setting.tx_cmd_len = sizeof(struct ipw_tx_cmd);
	priv->hw_setting.statistics_size = sizeof(struct ipw_notif_statistics);
	priv->hw_setting.rate_scale_size =
	    sizeof(struct ipw_rate_scaling_cmd_specifics);
	priv->hw_setting.add_station_size = sizeof(struct ipw_addsta_cmd);
	priv->hw_setting.max_rxq_size = RX_QUEUE_SIZE;
	priv->hw_setting.max_rxq_log = RX_QUEUE_SIZE_LOG;
	priv->hw_setting.cck_flag = 0;

	priv->hw_setting.nic_stop_master = ipw3945_nic_stop_master;
//      priv->hw_setting.nic_start = ipw3945_nic_start;
//      priv->hw_setting.nic_init = ipw3945_nic_init;
//      priv->hw_setting.nic_reset = ipw3945_nic_reset;
	priv->hw_setting.stop_tx_queue = ipw3945_stop_tx_queue;
	priv->hw_setting.tx_reset = ipw3945_tx_reset;
	priv->hw_setting.queue_init = ipw3945_queue_init;
	priv->hw_setting.queue_tx_free_tfd = ipw3945_queue_tx_free_tfd;
	priv->hw_setting.tx_queue_update_wr_ptr =
	    ipw3945_tx_queue_update_wr_ptr;

	priv->hw_setting.queue_tx_init = ipw3945_queue_tx_init;
	priv->hw_setting.queue_tx_free = ipw3945_queue_tx_free;
	priv->hw_setting.rx_queue_update_wr_ptr =
	    ipw3945_rx_queue_update_wr_ptr;
	priv->hw_setting.rxq_stop = ipw3945_rxq_stop;
	priv->hw_setting.rx_init = ipw3945_rx_init;
	priv->hw_setting.get_rx_read = ipw3945_get_rx_read;
	priv->hw_setting.attach_buffer_to_tfd_frame =
	    ipw3945_attach_buffer_to_tfd_frame;

//      priv->hw_setting.alive_notify = ipw3945_alive_notify;
//      priv->hw_setting.init_rate_scale = ipw3945_init_rate_scale;

	priv->hw_setting.rx_cmd = ipw3945_rx_cmd;
//      priv->hw_setting.handle_reply_tx = ipw3945_handle_reply_tx;
	priv->hw_setting.get_beacon_cmd = ipw3945_get_beacon_cmd;
	priv->hw_setting.tx_cmd = ipw3945_tx_cmd;
	priv->hw_setting.init_hw_rates = ipw3945_init_hw_rates;
	priv->hw_setting.send_rxon_assoc = ipw3945_send_rxon_assoc;
	priv->hw_setting.send_power_mode = ipw3945_send_power_mode;

	mutex_unlock(&priv->mutex);

	IPW_DEBUG_INFO("Queue up transitioning to the UP state.\n");
	queue_work(priv->workqueue, &priv->up);

	return 0;

      out_pci_alloc:
	ipw_dealloc_ucode_pci(priv);
	if (priv->hw_setting.shared_virt)
		pci_free_consistent(priv->pci_dev,
				    sizeof(struct ipw_shared_t),
				    priv->hw_setting.shared_virt,
				    priv->hw_setting.shared_phys);
	priv->hw_setting.shared_virt = NULL;

      out_remove_sysfs:
	sysfs_remove_group(&pdev->dev.kobj, &ipw_attribute_group);

      out_release_irq:
	free_irq(pdev->irq, priv);

      out_destroy_workqueue:
	destroy_workqueue(priv->workqueue);
	priv->workqueue = NULL;

      out_pci_release_regions:
	pci_release_regions(pdev);

      out_pci_disable_device:
	pci_disable_device(pdev);
	pci_set_drvdata(pdev, NULL);

      out_ieee80211_free_hw:
	ieee80211_free_hw(priv->ieee);

      out:
	return err;
}

static void ipw_pci_remove(struct pci_dev *pdev)
{
	struct ipw_priv *priv = pci_get_drvdata(pdev);
	struct list_head *p, *q;
	int i;

	if (!priv)
		return;

	IPW_DEBUG_INFO("*** UNLOAD DRIVER ***\n");

	mutex_lock(&priv->mutex);

	priv->status |= STATUS_EXIT_PENDING;

	ipw_down(priv);

	mutex_unlock(&priv->mutex);

	destroy_workqueue(priv->workqueue);
	priv->workqueue = NULL;

	/* Free MAC hash list for ADHOC */
	for (i = 0; i < IPW_IBSS_MAC_HASH_SIZE; i++) {
		list_for_each_safe(p, q, &priv->ibss_mac_hash[i]) {
			list_del(p);
			kfree(list_entry(p, struct ipw_ibss_seq, list));
		}
	}

	sysfs_remove_group(&pdev->dev.kobj, &ipw_attribute_group);

	ipw_dealloc_ucode_pci(priv);

	if (priv->rxq)
		ipw_rx_queue_free(priv, priv->rxq);
	ipw_tx_queue_free(priv);
	ipw_card_remove_notify(priv);

	ipw_delete_stations_table(priv, 0);

	if (priv->netdev_registered) {
		ieee80211_rate_control_unregister(&priv->rate_control);

		ieee80211_unregister_hw(priv->ieee);
	}

	free_irq(pdev->irq, priv);
	iounmap(priv->hw_base);
	pci_release_regions(pdev);
	pci_disable_device(pdev);
	pci_set_drvdata(pdev, NULL);

	kfree(priv->ieee_channels);
	kfree(priv->ieee_rates);
	priv->ieee_channels = NULL;
	priv->ieee_rates = NULL;

	if (priv->ibss_beacon)
		dev_kfree_skb(priv->ibss_beacon);

	ieee80211_free_hw(priv->ieee);
}

#ifdef CONFIG_PM
static int ipw_pci_suspend(struct pci_dev *pdev, pm_message_t state)
{
	struct ipw_priv *priv = pci_get_drvdata(pdev);

	mutex_lock(&priv->mutex);

	priv->status |= STATUS_IN_SUSPEND;

	/* Take down the device; powers it off, etc. */
	ipw_down(priv);

	/* Remove the PRESENT state of the device */
	if (priv->netdev_registered)
		ieee80211_netif_oper(priv->ieee, NETIF_DETACH);
	pci_save_state(pdev);
	pci_disable_device(pdev);
	pci_set_power_state(pdev, PCI_D3hot);

	mutex_unlock(&priv->mutex);

	return 0;
}
static int ipw_pci_resume(struct pci_dev *pdev)
{
	struct ipw_priv *priv = pci_get_drvdata(pdev);
	int err;

	printk(KERN_INFO "Coming out of suspend...\n");

	mutex_lock(&priv->mutex);

	pci_set_power_state(pdev, PCI_D0);
	err = pci_enable_device(pdev);
	pci_restore_state(pdev);

	/*
	 * Suspend/Resume resets the PCI configuration space, so we have to
	 * re-disable the RETRY_TIMEOUT register (0x41) to keep PCI Tx retries
	 * from interfering with C3 CPU state. pci_restore_state won't help
	 * here since it only restores the first 64 bytes pci config header.
	 */
	pci_write_config_byte(pdev, 0x41, 0x00);

	/* Set the device back into the PRESENT state; this will also wake
	 * the queue of needed */
	if (priv->netdev_registered)
		ieee80211_netif_oper(priv->ieee, NETIF_ATTACH);

	queue_delayed_work(priv->workqueue, &priv->resume_work, 3 * HZ);

	mutex_unlock(&priv->mutex);

	return 0;
}
#endif
/* driver initialization stuff */ static struct pci_driver ipw_driver = {
	.name = DRV_NAME,
	.id_table = card_ids,
	.probe = ipw_pci_probe,
	.remove = __devexit_p(ipw_pci_remove),
#ifdef CONFIG_PM
	.suspend = ipw_pci_suspend,.resume = ipw_pci_resume,
#endif
};
static int __init ipw_init(void)
{

	int ret;
	printk(KERN_INFO DRV_NAME ": " DRV_DESCRIPTION ", " DRV_VERSION "\n");
	printk(KERN_INFO DRV_NAME ": " DRV_COPYRIGHT "\n");
	ret = pci_register_driver(&ipw_driver);
	if (ret) {
		IPW_ERROR("Unable to initialize PCI module\n");
		return ret;
	}

	ret = driver_create_file(&ipw_driver.driver, &driver_attr_debug_level);
	if (ret) {
		IPW_ERROR("Unable to create driver sysfs file\n");
		pci_unregister_driver(&ipw_driver);
		return ret;
	}

	return ret;
}

static void __exit ipw_exit(void)
{
	driver_remove_file(&ipw_driver.driver, &driver_attr_debug_level);
	pci_unregister_driver(&ipw_driver);
}

module_param_named(antenna, param_antenna, int, 0444);
MODULE_PARM_DESC(antenna, "select antenna (1=Main, 2=Aux, default 0 [both])");
module_param_named(disable, param_disable, int, 0444);
MODULE_PARM_DESC(disable, "manually disable the radio (default 0 [radio on])");
module_param_named(auto_create, param_auto_create, int, 0444);
MODULE_PARM_DESC(auto_create, "auto create adhoc network (default 1 on)");
module_param_named(led, param_led, int, 0444);
MODULE_PARM_DESC(led, "enable led control (default 1 on)\n");
module_param_named(hwcrypto, param_hwcrypto, int, 0444);
MODULE_PARM_DESC(hwcrypto,
		 "using hardware crypto engine (default 0 [software])\n");
module_param_named(debug, param_debug, int, 0444);
MODULE_PARM_DESC(debug, "debug output mask");
module_param_named(channel, param_channel, int, 0444);
MODULE_PARM_DESC(channel, "channel to limit associate to (default 0 [ANY])");

/* QoS */
module_param_named(qos_enable, param_qos_enable, int, 0444);
MODULE_PARM_DESC(qos_enable, "enable all QoS functionality");

module_param_named(mode, param_qos_enable, int, 0444);
MODULE_PARM_DESC(mode, "network mode (0=BSS,1=IBSS,2=Monitor)");
module_exit(ipw_exit);
module_init(ipw_init);

/*** 3945 specific callbacks  ***/

/* These functions load the firmware and micro code for the operation of
 * the ipw hardware.  It assumes the buffer has all the bits for the
 * image and the caller is handling the memory allocation and clean up.
 */

static int ipw3945_nic_stop_master(struct ipw_priv *priv)
{
	int rc = 0;
	u32 reg_val;
	unsigned long flags;

	spin_lock_irqsave(&priv->lock, flags);

	/* set stop master bit */
	ipw_set_bit(priv, CSR_RESET, CSR_RESET_REG_FLAG_STOP_MASTER);

	reg_val = ipw_read32(priv, CSR_GP_CNTRL);

	if (CSR_GP_CNTRL_REG_FLAG_MAC_POWER_SAVE ==
	    (reg_val & CSR_GP_CNTRL_REG_MSK_POWER_SAVE_TYPE)) {
		IPW_DEBUG_INFO
		    ("Card in power save, master is already stopped\n");
	} else {
		rc = ipw_poll_bit(priv,
				  CSR_RESET,
				  CSR_RESET_REG_FLAG_MASTER_DISABLED,
				  CSR_RESET_REG_FLAG_MASTER_DISABLED, 100);
		if (rc < 0) {
			spin_unlock_irqrestore(&priv->lock, flags);
			return rc;
		}
	}

	spin_unlock_irqrestore(&priv->lock, flags);
	IPW_DEBUG_INFO("stop master\n");

	return rc;
}

static int ipw3945_stop_tx_queue(struct ipw_priv *priv)
{
	int queueId;
	int rc;
	unsigned long flags;

	spin_lock_irqsave(&priv->lock, flags);
	rc = ipw_grab_restricted_access(priv);
	if (rc) {
		spin_unlock_irqrestore(&priv->lock, flags);
		return rc;
	}

	/* stop SCD */
	ipw_write_restricted_reg(priv, SCD_MODE_REG, 0);

	/* reset TFD queues */
	for (queueId = TFD_QUEUE_MIN; queueId < TFD_QUEUE_MAX; queueId++) {
		ipw_write_restricted(priv, FH_TCSR_CONFIG(queueId), 0x0);
		ipw_poll_restricted_bit(priv, FH_TSSR_TX_STATUS,
					ALM_FH_TSSR_TX_STATUS_REG_MSK_CHNL_IDLE
					(queueId), 1000);
	}

	ipw_release_restricted_access(priv);
	spin_unlock_irqrestore(&priv->lock, flags);

	return 0;
}

static int ipw3945_tx_reset(struct ipw_priv *priv)
{
	int rc;
	unsigned long flags;

	spin_lock_irqsave(&priv->lock, flags);
	rc = ipw_grab_restricted_access(priv);
	if (rc) {
		spin_unlock_irqrestore(&priv->lock, flags);
		return rc;
	}

	ipw_write_restricted_reg(priv, SCD_MODE_REG, 0x2);	// bypass mode
	ipw_write_restricted_reg(priv, SCD_ARASTAT_REG, 0x01);	// RA 0 is active
	ipw_write_restricted_reg(priv, SCD_TXFACT_REG, 0x3f);	// all 6 fifo are active
	ipw_write_restricted_reg(priv, SCD_SBYP_MODE_1_REG, 0x010000);
	ipw_write_restricted_reg(priv, SCD_SBYP_MODE_2_REG, 0x030002);
	ipw_write_restricted_reg(priv, SCD_TXF4MF_REG, 0x000004);
	ipw_write_restricted_reg(priv, SCD_TXF5MF_REG, 0x000005);

	ipw_write_restricted(priv, FH_TSSR_CBB_BASE,
			     priv->hw_setting.shared_phys);

	ipw_write_restricted(priv,
			     FH_TSSR_MSG_CONFIG,
			     ALM_FH_TSSR_TX_MSG_CONFIG_REG_VAL_SNOOP_RD_TXPD_ON
			     |
			     ALM_FH_TSSR_TX_MSG_CONFIG_REG_VAL_ORDER_RD_TXPD_ON
			     |
			     ALM_FH_TSSR_TX_MSG_CONFIG_REG_VAL_MAX_FRAG_SIZE_128B
			     |
			     ALM_FH_TSSR_TX_MSG_CONFIG_REG_VAL_SNOOP_RD_TFD_ON
			     |
			     ALM_FH_TSSR_TX_MSG_CONFIG_REG_VAL_ORDER_RD_CBB_ON
			     |
			     ALM_FH_TSSR_TX_MSG_CONFIG_REG_VAL_ORDER_RSP_WAIT_TH
			     | ALM_FH_TSSR_TX_MSG_CONFIG_REG_VAL_RSP_WAIT_TH);

	ipw_release_restricted_access(priv);

	spin_unlock_irqrestore(&priv->lock, flags);

	return 0;
}

static int ipw3945_queue_init(struct ipw_priv *priv, struct ipw_queue *q,
			      int count, int size, u32 id)
{
	int rc;
	unsigned long flags;
	struct ipw_shared_t *shared_data =
	    (struct ipw_shared_t *)priv->hw_setting.shared_virt;

	shared_data->tx_base_ptr[id] = (u32) q->dma_addr;

	q->element_size = sizeof(struct tfd_frame);

	spin_lock_irqsave(&priv->lock, flags);
	rc = ipw_grab_restricted_access(priv);
	if (rc) {
		spin_unlock_irqrestore(&priv->lock, flags);
		return rc;
	}
	ipw_write_restricted(priv, FH_CBCC_CTRL(id), 0);
	ipw_write_restricted(priv, FH_CBCC_BASE(id), 0);

	ipw_write_restricted(priv, FH_TCSR_CONFIG(id),
			     ALM_FH_TCSR_TX_CONFIG_REG_VAL_CIRQ_RTC_NOINT
			     |
			     ALM_FH_TCSR_TX_CONFIG_REG_VAL_MSG_MODE_TXF
			     |
			     ALM_FH_TCSR_TX_CONFIG_REG_VAL_CIRQ_HOST_IFTFD
			     |
			     ALM_FH_TCSR_TX_CONFIG_REG_VAL_DMA_CREDIT_ENABLE_VAL
			     | ALM_FH_TCSR_TX_CONFIG_REG_VAL_DMA_CHNL_ENABLE);
	ipw_release_restricted_access(priv);

	ipw_read32(priv, FH_TSSR_CBB_BASE);	/* fake read to flush all prev. writes */

	spin_unlock_irqrestore(&priv->lock, flags);
	return 0;
}

/**
 * Free one TFD, those at index [txq->q.last_used].
 * Do NOT advance any indexes
 */
static int ipw3945_queue_tx_free_tfd(struct ipw_priv *priv,
				     struct ipw_tx_queue *txq)
{
	struct tfd_frame *bd_tmp = (struct tfd_frame *)&txq->bd[0];
	struct tfd_frame *bd = &bd_tmp[txq->q.last_used];
	struct pci_dev *dev = priv->pci_dev;
	int i;
	int counter = 0;
	/* classify bd */
	if (txq->q.id == priv->hw_setting.cmd_queue_no)
		/* nothing to cleanup after for host commands */
		return 0;

	/* sanity check */
	counter = TFD_CTL_COUNT_GET(le32_to_cpu(bd->control_flags));
	if (counter > NUM_TFD_CHUNKS) {
		IPW_ERROR("Too many chunks: %i\n", counter);
		/** @todo issue fatal error, it is quite serious situation */
		return 0;
	}

	/* unmap chunks if any */

	for (i = 1; i < counter; i++) {
		pci_unmap_single(dev, le32_to_cpu(bd->pa[i].addr),
				 le16_to_cpu(bd->pa[i].len), PCI_DMA_TODEVICE);
		if (txq->txb[txq->q.last_used].skb[0]) {
			struct sk_buff *skb = txq->txb[txq->q.last_used].skb[0];
			priv->tx_bytes += skb->len -
			    ieee80211_get_hdrlen_from_skb(skb);

			/*do we still own skb, then released */
			if (txq->txb[txq->q.last_used].skb[0]) {
				dev_kfree_skb(skb);
				txq->txb[txq->q.last_used].skb[0] = NULL;
			}
		}
	}
	return 0;
}

static int ipw3945_tx_queue_update_wr_ptr(struct ipw_priv *priv,
					  struct ipw_tx_queue *txq,
					  int tx_id, u16 byte_cnt)
{

	return 0;
}

static int ipw3945_queue_tx_init(struct ipw_priv *priv,
				 struct ipw_tx_queue *q, int count, u32 id)
{
	struct pci_dev *dev = priv->pci_dev;

	if (id != priv->hw_setting.cmd_queue_no) {
		q->txb = kmalloc(sizeof(q->txb[0]) *
				 TFD_QUEUE_SIZE_MAX, GFP_ATOMIC);
		if (!q->txb) {
			IPW_ERROR("kmalloc for auxiliary BD "
				  "structures failed\n");
			return -ENOMEM;
		}
	} else
		q->txb = NULL;

	q->bd = (u8 *)
	    pci_alloc_consistent(dev,
				 sizeof(struct tfd_frame) *
				 TFD_QUEUE_SIZE_MAX, &q->q.dma_addr);

	q->q.element_size = sizeof(struct tfd_frame);
	if (!q->bd) {
		IPW_ERROR("pci_alloc_consistent(%zd) failed\n",
			  sizeof(q->bd[0]) * count);
		if (q->txb) {
			kfree(q->txb);
			q->txb = NULL;
		}
		return -ENOMEM;
	}

	return 0;
}

static int ipw3945_queue_tx_free(struct ipw_priv *priv,
				 struct ipw_tx_queue *txq)
{
	struct ipw_queue *q = &txq->q;
	struct pci_dev *dev = priv->pci_dev;

	if (q->n_bd == 0)
		return 0;

	/* free buffers belonging to queue itself */
	pci_free_consistent(dev, sizeof(struct tfd_frame) * q->n_bd,
			    txq->bd, q->dma_addr);
	return 0;
}

static int ipw3945_rx_queue_update_wr_ptr(struct ipw_priv *priv,
					  struct ipw_rx_queue *q)
{
	u32 reg = 0;
	int rc = 0;
	unsigned long flags;

	spin_lock_irqsave(&q->lock, flags);

	if (q->need_update == 0)
		goto exit_unlock;

	if (priv->status & STATUS_POWER_PMI) {
		reg = ipw_read32(priv, CSR_UCODE_DRV_GP1);

		if (reg & CSR_UCODE_DRV_GP1_BIT_MAC_SLEEP) {
			ipw_set_bit(priv, CSR_GP_CNTRL,
				    CSR_GP_CNTRL_REG_FLAG_MAC_ACCESS_REQ);
			goto exit_unlock;
		}

		rc = ipw_grab_restricted_access(priv);
		if (rc)
			goto exit_unlock;

		ipw_write_restricted(priv, FH_RCSR_WPTR(0), q->write & ~0x7);
		ipw_release_restricted_access(priv);
	} else {
		ipw_write32(priv, FH_RCSR_WPTR(0), q->write & ~0x7);
	}

	q->need_update = 0;

      exit_unlock:
	spin_unlock_irqrestore(&q->lock, flags);
	return rc;
}

static int ipw3945_rxq_stop(struct ipw_priv *priv)
{
	int rc;
	unsigned long flags;

	spin_lock_irqsave(&priv->lock, flags);
	rc = ipw_grab_restricted_access(priv);
	if (rc) {
		spin_unlock_irqrestore(&priv->lock, flags);
		return rc;
	}

	ipw_write_restricted(priv, FH_RCSR_CONFIG(0), 0);
	rc = ipw_poll_restricted_bit(priv, FH_RSSR_STATUS, (1 << 24), 1000);
	if (rc < 0)
		IPW_ERROR("Can't stop Rx DMA.\n");

	ipw_release_restricted_access(priv);
	spin_unlock_irqrestore(&priv->lock, flags);

	return 0;
}

static int ipw3945_rx_init(struct ipw_priv *priv, struct ipw_rx_queue *rxq)
{
	int rc;
	unsigned long flags;

	spin_lock_irqsave(&priv->lock, flags);
	rc = ipw_grab_restricted_access(priv);
	if (rc) {
		spin_unlock_irqrestore(&priv->lock, flags);
		return rc;
	}

	ipw_write_restricted(priv, FH_RCSR_RBD_BASE(0), rxq->dma_addr);
	ipw_write_restricted(priv, FH_RCSR_RPTR_ADDR(0),
			     priv->hw_setting.shared_phys +
			     offsetof(struct ipw_shared_t, rx_read_ptr[0]));
	ipw_write_restricted(priv, FH_RCSR_WPTR(0), 0);
	ipw_write_restricted(priv, FH_RCSR_CONFIG(0),
			     ALM_FH_RCSR_RX_CONFIG_REG_VAL_DMA_CHNL_EN_ENABLE
			     |
			     ALM_FH_RCSR_RX_CONFIG_REG_VAL_RDRBD_EN_ENABLE
			     |
			     ALM_FH_RCSR_RX_CONFIG_REG_BIT_WR_STTS_EN
			     |
			     ALM_FH_RCSR_RX_CONFIG_REG_VAL_MAX_FRAG_SIZE_128
			     | (RX_QUEUE_SIZE_LOG <<
				ALM_FH_RCSR_RX_CONFIG_REG_POS_RBDC_SIZE)
			     |
			     ALM_FH_RCSR_RX_CONFIG_REG_VAL_IRQ_DEST_INT_HOST
			     | (1 << ALM_FH_RCSR_RX_CONFIG_REG_POS_IRQ_RBTH)
			     | ALM_FH_RCSR_RX_CONFIG_REG_VAL_MSG_MODE_FH);

	/* fake read to flush all prev I/O */
	ipw_read_restricted(priv, FH_RSSR_CTRL);

	ipw_release_restricted_access(priv);

	spin_unlock_irqrestore(&priv->lock, flags);

	return 0;
}

static int ipw3945_attach_buffer_to_tfd_frame(void *ptr,
					      dma_addr_t addr, u16 len)
{
	int count = 0;
	u32 pad;
	struct tfd_frame *tfd = (struct tfd_frame *)ptr;

	count = TFD_CTL_COUNT_GET(tfd->control_flags);
	pad = TFD_CTL_PAD_GET(tfd->control_flags);

	if ((count >= NUM_TFD_CHUNKS) || (count < 0)) {
		IPW_ERROR("Error can not send more than %d chunks\n",
			  NUM_TFD_CHUNKS);
		return -EINVAL;
	}

	tfd->pa[count].addr = (u32) addr;
	tfd->pa[count].len = len;

	count++;

	tfd->control_flags = TFD_CTL_COUNT_SET(count) | TFD_CTL_PAD_SET(pad);

	return 0;
}

static int ipw3945_get_rx_read(struct ipw_priv *priv)
{
	struct ipw_shared_t *shared_data =
	    (struct ipw_shared_t *)priv->hw_setting.shared_virt;

	return shared_data->rx_read_ptr[0];
}

int ipw3945_tx_cmd(struct ipw_priv *priv, struct ipw_cmd *out_cmd,
		   u8 sta_id, dma_addr_t scratch_phys,
		   struct ieee80211_hdr *hdr, u8 hdr_len,
		   struct ieee80211_tx_control *ctrl)
{
	out_cmd->cmd.tx.tx_flags &= ~TX_CMD_FLG_ANT_A_MSK;
	out_cmd->cmd.tx.tx_flags &= ~TX_CMD_FLG_ANT_B_MSK;
	return 0;
}

int ipw3945_rx_cmd(struct ipw_priv *priv,
		   struct ipw_rx_mem_buffer *rxb, int cmd)
{
	return 0;
}

int ipw3945_get_beacon_cmd(struct ipw_priv *priv,
			   struct ipw_frame *frame, u16 rate)
{
	struct ipw_tx_beacon_cmd *tx_beacon_cmd;
	int frame_size;

	tx_beacon_cmd = (struct ipw_tx_beacon_cmd *)&frame->u;
	memset(tx_beacon_cmd, 0, sizeof(*tx_beacon_cmd));

	tx_beacon_cmd->tx.sta_id = priv->hw_setting.broadcast_id;
	tx_beacon_cmd->tx.u.life_time = 0xFFFFFFFF;

	frame_size = ipw_fill_beacon_frame(priv,
					   tx_beacon_cmd->frame,
					   BROADCAST_ADDR,
					   sizeof(frame->u) -
					   sizeof(*tx_beacon_cmd));

	tx_beacon_cmd->tx.len = frame_size;

	tx_beacon_cmd->tx.rate = rate;
	tx_beacon_cmd->tx.tx_flags = (TX_CMD_FLG_SEQ_CTL_MSK |
				      TX_CMD_FLG_TSF_MSK);

	tx_beacon_cmd->tx.supp_rates[0] = priv->active_rate >> 4;
	tx_beacon_cmd->tx.supp_rates[1] = priv->active_rate & 0xF;

	return (sizeof(struct ipw_tx_beacon_cmd) + frame_size);
}

static int ipw3945_send_rxon_assoc(struct ipw_priv *priv)
{
	int rc = 0;
	struct ipw_rx_packet *res = NULL;
	struct ipw_rxon_assoc_cmd rxon_assoc;
	struct ipw_host_cmd cmd = {
		.id = REPLY_RXON_ASSOC,
		.len = sizeof(struct ipw_rxon_assoc_cmd),
		.meta.flags = CMD_WANT_SKB,
		.data = &rxon_assoc,
	};

	rxon_assoc.flags = priv->staging_rxon.flags;
	rxon_assoc.filter_flags = priv->staging_rxon.filter_flags;
	rxon_assoc.ofdm_basic_rates = priv->staging_rxon.ofdm_basic_rates;
	rxon_assoc.cck_basic_rates = priv->staging_rxon.cck_basic_rates;
	rxon_assoc.reserved = 0;

	rc = ipw_send_cmd(priv, &cmd);
	if (rc) {
		priv->active_rxon.flags = priv->staging_rxon.flags;
		priv->active_rxon.filter_flags =
		    priv->staging_rxon.filter_flags;
		priv->active_rxon.ofdm_basic_rates =
		    priv->staging_rxon.ofdm_basic_rates;
		priv->active_rxon.cck_basic_rates =
		    priv->staging_rxon.cck_basic_rates;
		return rc;
	}

	res = (struct ipw_rx_packet *)cmd.meta.u.skb->data;
	if (res->hdr.flags & 0x40) {
		IPW_ERROR("Bad return from REPLY_RXON_ASSOC command\n");
		rc = -EIO;
	}

	dev_kfree_skb_any(cmd.meta.u.skb);

	return rc;
}

int ipw3945_send_power_mode(struct ipw_priv *priv, u32 mode)
{
	int rc = 0;
	struct ipw_powertable_cmd cmd;

	ipw_update_power_cmd(priv, &cmd, mode);

	rc = ipw_send_cmd_pdu(priv, POWER_TABLE_CMD, sizeof(cmd), &cmd);
	return rc;
}

int ipw3945_init_hw_rates(struct ipw_priv *priv, struct ieee80211_rate *rates)
{

	return 0;
}
