/* -*- Mode: C; indent-tabs-mode: nil; c-basic-offset: 2; tab-width: 2 -*- */
/*
  A menu that should be close to the user, it's the user's status,
  their friends.  All about them.  It's a user-focused-menu.

  Copyright 2009 Canonical Ltd.

  Authors:
  Ted Gould <ted@canonical.com>
  Cody Russell <cody.russell@canonical.com>

  This program is free software: you can redistribute it and/or modify it
  under the terms of the GNU General Public License version 3, as published
  by the Free Software Foundation.

  This program is distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranties of
  MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
  PURPOSE.  See the GNU General Public License for more details.

  You should have received a copy of the GNU General Public License along
  with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#include <glib.h>
#include <glib-object.h>
#include <glib/gi18n.h>
#include <gtk/gtk.h>
#include <gio/gio.h>
#include <libdbusmenu-glib/menuitem.h>
#include <libdbusmenu-gtk/menu.h>

#include <libindicator/indicator.h>
#include <libindicator/indicator-object.h>
#include <libindicator/indicator-service-manager.h>
#include <libindicator/indicator-image-helper.h>
#include <libido/idoentrymenuitem.h>

#include "about-me-menu-item.h"

#include "dbus-shared-names.h"
#include "gen-me-service.xml.h"

#define INDICATOR_ME_TYPE            (indicator_me_get_type ())
#define INDICATOR_ME(obj)            (G_TYPE_CHECK_INSTANCE_CAST ((obj), INDICATOR_ME_TYPE, IndicatorMe))
#define INDICATOR_ME_CLASS(klass)    (G_TYPE_CHECK_CLASS_CAST ((klass), INDICATOR_ME_TYPE, IndicatorMeClass))
#define IS_INDICATOR_ME(obj)         (G_TYPE_CHECK_INSTANCE_TYPE ((obj), INDICATOR_ME_TYPE))
#define IS_INDICATOR_ME_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), INDICATOR_ME_TYPE))
#define INDICATOR_ME_GET_CLASS(obj)  (G_TYPE_INSTANCE_GET_CLASS ((obj), INDICATOR_ME_TYPE, IndicatorMeClass))

#define DEFAULT_ICON   "user-offline"

typedef struct _IndicatorMe      IndicatorMe;
typedef struct _IndicatorMeClass IndicatorMeClass;

struct _IndicatorMeClass {
	IndicatorObjectClass parent_class;
};

struct _IndicatorMe {
	IndicatorObject parent;
	IndicatorServiceManager * service;
};

GType indicator_me_get_type (void);

/* Indicator stuff */
INDICATOR_SET_VERSION
INDICATOR_SET_TYPE(INDICATOR_ME_TYPE)

/* Globals */
static GtkImage * status_image = NULL;
static GtkLabel *label = NULL;
static GDBusProxy * status_proxy = NULL;

static GCancellable * status_proxy_cancel = NULL;

static IdoEntryMenuItem *ido_entry = NULL;

static const gchar *status_desc = NULL;

/* Prototypes */
static GtkLabel * get_label (IndicatorObject * io);
static GtkImage * get_icon (IndicatorObject * io);
static GtkMenu * get_menu (IndicatorObject * io);
static const gchar * get_accessible_desc (IndicatorObject * io);

static void indicator_me_class_init (IndicatorMeClass *klass);
static void indicator_me_init       (IndicatorMe *self);
static void indicator_me_dispose    (GObject *object);
static void indicator_me_finalize   (GObject *object);
static void connection_changed      (IndicatorServiceManager * sm, gboolean connected, gpointer userdata);
static void receive_signal (GDBusProxy * proxy, gchar * sender_name, gchar * signal_name, GVariant * parameters, gpointer user_data);

static gboolean new_entry_item (DbusmenuMenuitem * newitem,
                                DbusmenuMenuitem * parent,
                                DbusmenuClient   * client,
				gpointer user_data);
static gboolean new_about_me_item (DbusmenuMenuitem * newitem,
                                DbusmenuMenuitem * parent,
                                DbusmenuClient   * client,
				gpointer user_data);
static void     entry_activate_cb (GtkEntry *entry, DbusmenuMenuitem *mi);
static void     entry_prop_change_cb (DbusmenuMenuitem *mi, gchar *prop, GVariant *value, GtkEntry *entry);
static gboolean entry_hint_is_shown (GtkWidget *widget);
static void status_proxy_cb (GObject * object, GAsyncResult * res, gpointer user_data);
static void status_icon_changed (IndicatorMe * self, gchar * icon);
static void status_icon_cb (GObject * object, GAsyncResult * res, gpointer user_data);
static void username_changed (IndicatorMe * self, gchar * username);
static void username_cb (GObject * object, GAsyncResult * res, gpointer user_data);

G_DEFINE_TYPE (IndicatorMe, indicator_me, INDICATOR_OBJECT_TYPE);

static void
indicator_me_class_init (IndicatorMeClass *klass)
{
	GObjectClass *object_class = G_OBJECT_CLASS (klass);

	object_class->dispose = indicator_me_dispose;
	object_class->finalize = indicator_me_finalize;

	IndicatorObjectClass * io_class = INDICATOR_OBJECT_CLASS(klass);
	io_class->get_label = get_label;
	io_class->get_image = get_icon;
	io_class->get_menu = get_menu;
	io_class->get_accessible_desc = get_accessible_desc;

	return;
}

static void
indicator_me_init (IndicatorMe *self)
{

	/* Init variables */
	self->service = NULL;

	/* Do stuff with them */
	self->service = indicator_service_manager_new_version(INDICATOR_ME_DBUS_NAME, INDICATOR_ME_DBUS_VERSION);
	g_signal_connect(G_OBJECT(self->service), INDICATOR_SERVICE_MANAGER_SIGNAL_CONNECTION_CHANGE, G_CALLBACK(connection_changed), self);

	return;
}

static void
indicator_me_dispose (GObject *object)
{
        IndicatorMe * self = INDICATOR_ME(object);

	if (self->service != NULL) {
		g_object_unref(G_OBJECT(self->service));
		self->service = NULL;
	}

	if (status_proxy != NULL) {
		g_object_unref(status_proxy);
		status_proxy = NULL;
	}

	if (status_proxy_cancel != NULL) {
		g_cancellable_cancel(status_proxy_cancel);
		g_object_unref(status_proxy_cancel);
		status_proxy_cancel = NULL;
	}

	G_OBJECT_CLASS (indicator_me_parent_class)->dispose (object);

	return;
}

static void
indicator_me_finalize (GObject *object)
{
	G_OBJECT_CLASS (indicator_me_parent_class)->finalize (object);
	return;
}

static void
username_cb (GObject * object, GAsyncResult * res, gpointer user_data)
{
	GError * error = NULL;
	gchar * username = NULL;
	GVariant * result;

	IndicatorMe * self = INDICATOR_ME (user_data);
	g_return_if_fail(self != NULL);

	result = g_dbus_proxy_call_finish(status_proxy, res, &error);
	g_return_if_fail(status_proxy != NULL);

	if (error != NULL) {
		g_warning("Could not get the username %s", error->message);
		g_error_free(error);
		return;
	}

	g_variant_get(result, "(&s)", &username);

	if (username != NULL) {
		username_changed (self, username);
	}

}

static void
username_changed (IndicatorMe * self, gchar * username)
{
  if (label == NULL) {
    label = GTK_LABEL(gtk_label_new(NULL));
    if (label == NULL) return;
  }

  if (username != NULL && username[0] != '\0') {
    g_debug ("Updating username label");
    gtk_label_set_text (label, username);
    gtk_widget_show(GTK_WIDGET(label));
  } else {
    gtk_widget_hide(GTK_WIDGET(label));
  }
}

static GtkLabel *
get_label (IndicatorObject * io)
{
	if (label == NULL) {
		/* Create the label if it doesn't exist already */
		username_changed (NULL, NULL);
	}
	return label;
}

static GtkImage *
get_icon (IndicatorObject * io)
{
	if (status_image == NULL) {
		/* Will create the status icon if it doesn't exist already */
		status_icon_changed(NULL, DEFAULT_ICON);
	}
	return status_image;
}

static const gchar *
get_accessible_desc (IndicatorObject * io)
{
	if (label == NULL) {
		/* Create the label if it doesn't exist already */
		username_changed (NULL, NULL);
	}
	return g_strdup_printf("%s (%s)", gtk_label_get_text(label), status_desc);
}

static void
status_icon_cb (GObject * object, GAsyncResult * res, gpointer user_data)
{
	GError * error = NULL;
	gchar * icon = NULL;
	GVariant * result;

	IndicatorMe * self = INDICATOR_ME (user_data);
	g_return_if_fail(self != NULL);

	result = g_dbus_proxy_call_finish(status_proxy, res, &error);
	g_return_if_fail(status_proxy != NULL);

	if (error != NULL) {
		g_warning("Could not get the status icon %s", error->message);
		g_error_free(error);
		return;
	}

	g_variant_get(result, "(&s)", &icon);

	if (icon != NULL) {
		status_icon_changed (self, icon);
	}

}

static void
status_icon_changed (IndicatorMe * self, gchar * icon)
{
	g_return_if_fail(icon != NULL);
	g_return_if_fail(icon[0] != '\0');

	if (status_image == NULL) {
		status_image = indicator_image_helper (DEFAULT_ICON "-panel");
		gtk_widget_show(GTK_WIDGET(status_image));
	}

	if (g_strcmp0(icon, "user-away") == 0) {
		/* Translators: user status "Away" */
		status_desc = _("Away");
	} else if (g_strcmp0(icon, "user-invisible") == 0) {
		/* Translators: user status "Invisible" */
		status_desc = _("Invisible");
	} else if (g_strcmp0(icon, "user-busy") == 0) {
		/* Translators: user status "Busy" */
		status_desc = _("Busy");
	} else if (g_strcmp0(icon, "user-available") == 0) {
		/* Translators: user status "Available" */
		status_desc = _("Available");
	} else if (g_strcmp0(icon, "user-indeterminate") == 0) {
		/* Translators: user status "Unknown" */
		status_desc = _("Unknown");
	} else if (g_strcmp0(icon, "user-offline") == 0) {
		/* Translators: user status "Offline" */
		status_desc = _("Offline");
	}

	gchar *panel_icon = g_strconcat (icon, "-panel", NULL);
	indicator_image_helper_update (status_image, panel_icon);
	g_free (panel_icon);

	return;
}

/* Receives all signals from the service, routed to the appropriate functions */
static void
receive_signal (GDBusProxy * proxy, gchar * sender_name, gchar * signal_name,
                GVariant * parameters, gpointer user_data)
{
        IndicatorMe * self = INDICATOR_ME(user_data);

        if (g_strcmp0(signal_name, "StatusIconsChanged") == 0) {
                gchar * icon;
                g_variant_get (parameters, "(&s)", &icon);
                status_icon_changed (self, icon);
        }
        else if (g_strcmp0(signal_name, "UserChanged") == 0) {
                gchar * username;
                g_variant_get (parameters, "(&s)", &username);
                username_changed(self, username);
        }
	return;
}

static void
item_destroyed_cb (GtkObject *item,
                   gpointer   user_data)
{
  g_signal_handlers_disconnect_by_func (user_data,
                                        G_CALLBACK (entry_activate_cb),
                                        item);

  g_signal_handlers_disconnect_by_func (user_data,
                                        G_CALLBACK (entry_prop_change_cb),
                                        item);
}

static void
connection_changed (IndicatorServiceManager * self, gboolean connected, gpointer userdata)
{
	if (connected) {
		if (status_proxy == NULL) {
			if (status_proxy_cancel == NULL) {
				status_proxy_cancel = g_cancellable_new();
			}
			g_dbus_proxy_new_for_bus (G_BUS_TYPE_SESSION,
						G_DBUS_PROXY_FLAGS_NONE,
						NULL,
						INDICATOR_ME_DBUS_NAME,
						INDICATOR_ME_SERVICE_DBUS_OBJECT,
						INDICATOR_ME_SERVICE_DBUS_INTERFACE,
						status_proxy_cancel,
						status_proxy_cb,
						userdata);

		}
	} else {
		DbusmenuMenuitem *mi = g_object_get_data (G_OBJECT (ido_entry),
                                              "dbusmenuitem");

		g_signal_handlers_disconnect_by_func (ido_entry,
                                          G_CALLBACK (entry_activate_cb),
                                          mi);

		g_signal_handlers_disconnect_by_func (ido_entry,
                                          G_CALLBACK (entry_prop_change_cb),
                                          mi);

		/* If we're disconnecting, go back to offline */
		status_icon_changed(NULL, DEFAULT_ICON);

		g_object_unref (status_proxy);
		status_proxy = NULL;

		if (ido_entry != NULL) {
			gtk_widget_destroy (GTK_WIDGET (ido_entry));
			ido_entry = NULL;
		}
	}

	return;
}

/* Callback from trying to create the proxy for the service, this
   could include starting the service.  Sometimes it'll fail and
   we'll try to start that dang service again! */
static void
status_proxy_cb (GObject * object, GAsyncResult * res, gpointer user_data)
{
	GError * error = NULL;

	g_return_if_fail(object != NULL);

	GDBusProxy * proxy = g_dbus_proxy_new_for_bus_finish(res, &error);

	if (status_proxy_cancel != NULL) {
		g_object_unref(status_proxy_cancel);
		status_proxy_cancel = NULL;
	}

	if (error != NULL) {
		g_error("Could not grab DBus proxy for %s: %s", INDICATOR_ME_DBUS_NAME, error->message);
		g_error_free(error);
		return;
	}

	IndicatorMe * self = INDICATOR_ME (user_data);

	/* Okay, we're good to grab the proxy at this point, we're
	sure that it's ours. */
	status_proxy = proxy;

	g_signal_connect(status_proxy, "g-signal", G_CALLBACK(receive_signal), self);

	/* Query to get the username */
	g_debug("Get the username");
	g_dbus_proxy_call(status_proxy, "PrettyUserName", NULL,
		G_DBUS_CALL_FLAGS_NONE, -1, NULL,
		(GAsyncReadyCallback) username_cb, self);

	/* Query to get the status icon */
	g_debug("Get the status icon");
	g_dbus_proxy_call(status_proxy, "StatusIcons", NULL,
		G_DBUS_CALL_FLAGS_NONE, -1, NULL,
		(GAsyncReadyCallback) status_icon_cb, self);

	return;
}

static void
entry_hint_set_shown (GtkWidget *widget, gboolean flag)
{
  g_object_set_data (G_OBJECT (widget),
                     DBUSMENU_ENTRY_MENUITEM_PROP_HINT "_shown",
                     GUINT_TO_POINTER (flag));
}


static void
entry_set_style (GtkEntry *entry, GtkStateType state)
{
    GtkStyle *style = gtk_widget_get_style (GTK_WIDGET (entry));
    GdkColor *color = &style->text[state];
    gtk_widget_modify_text (GTK_WIDGET (entry), GTK_STATE_NORMAL, color);
    gtk_widget_queue_draw (GTK_WIDGET (entry));
}

/* unconditionnaly show the hint in the entry */
static void
entry_hint_show_hint (GtkEntry *entry)
{
  g_return_if_fail (GTK_IS_ENTRY (entry));

  const gchar *hint = g_object_get_data (G_OBJECT (entry),
                                         DBUSMENU_ENTRY_MENUITEM_PROP_HINT);
  gtk_entry_set_text (entry, hint);
  entry_set_style (entry, GTK_STATE_INSENSITIVE);
  entry_hint_set_shown (GTK_WIDGET (entry), TRUE);
}

static void
entry_maybe_show_hint (GtkEntry *entry)
{
  g_return_if_fail (GTK_IS_ENTRY (entry));

  const gchar *hint = g_object_get_data (G_OBJECT (entry),
                                         DBUSMENU_ENTRY_MENUITEM_PROP_HINT);

  /* enforce style when typing a message */
  if (GTK_WIDGET_HAS_FOCUS (entry)) {
    entry_set_style (entry, GTK_STATE_NORMAL);
    entry_hint_set_shown (GTK_WIDGET (entry), FALSE);

    return;
  }

  if (gtk_entry_get_text_length (entry) > 0) {

    entry_set_style (entry, GTK_STATE_NORMAL);
    entry_hint_set_shown (GTK_WIDGET (entry), FALSE);

  } else {
    g_debug ("%s, nothing in the entry or not focused, so setting the hint to: %s", __func__, hint);

    entry_hint_show_hint (entry);
  }
}

static void
entry_set_hint (GtkEntry *entry, const char *hint)
{
    g_return_if_fail (GTK_IS_ENTRY (entry));

    g_debug ("entry hint: %s", hint);
		g_object_set_data_full (G_OBJECT (entry), DBUSMENU_ENTRY_MENUITEM_PROP_HINT,
                            g_strdup (hint), (GDestroyNotify) g_free);
    entry_maybe_show_hint (entry);
}

static gboolean
entry_hint_is_shown (GtkWidget *widget)
{
  gboolean shown = GPOINTER_TO_UINT
    (g_object_get_data (G_OBJECT (widget),
                        DBUSMENU_ENTRY_MENUITEM_PROP_HINT "_shown"));

  return shown;
}

static gboolean
entry_focus_grab_cb (GtkWidget *widget, GdkEventFocus *event)
{
  GtkEntry *entry = GTK_ENTRY (widget);
  GtkSettings *settings = gtk_widget_get_settings (GTK_WIDGET (entry));
  gboolean select_on_focus;

  g_debug ("%s", __func__);

  if (entry_hint_is_shown (GTK_WIDGET (entry))) {
    /* override select-on-focus */
    g_object_get (settings, "gtk-entry-select-on-focus", &select_on_focus, NULL);
    g_object_set (settings, "gtk-entry-select-on-focus", FALSE, NULL);
    gtk_entry_set_text (entry, "");
    g_object_set (settings, "gtk-entry-select-on-focus", select_on_focus, NULL);
  }

  entry_hint_set_shown (GTK_WIDGET (entry), FALSE);
  entry_set_style (entry, GTK_STATE_NORMAL);

  return FALSE;
}

static gboolean
entry_focus_out_cb (GtkWidget *widget, GdkEventFocus *event)
{
  g_return_val_if_fail (GTK_IS_ENTRY (widget), FALSE);

  GtkEntry *entry = GTK_ENTRY (widget);

  g_debug ("%s", __func__);

  if (! entry_hint_is_shown (GTK_WIDGET (entry))) {
    if (gtk_entry_get_text_length (entry) == 0) {
      /* show the hint unconditionnaly, as the focus change
         may not have propagated already and entry_maybe_show_hint
         could get confused */
      entry_hint_show_hint (entry);
    }
  }

  return FALSE;
}

static void
entry_prop_change_cb (DbusmenuMenuitem *mi, gchar *prop, GVariant *value, GtkEntry *entry)
{
  g_return_if_fail (GTK_IS_ENTRY (entry));

	if (g_strcmp0 (prop, DBUSMENU_ENTRY_MENUITEM_PROP_TEXT) == 0) {
		gtk_entry_set_text (entry, g_variant_get_string (value, NULL));
    if (gtk_entry_get_text_length (entry) == 0)
      entry_maybe_show_hint (entry);
	}
	if (g_strcmp0 (prop, DBUSMENU_ENTRY_MENUITEM_PROP_HINT) == 0) {
    entry_set_hint (entry, g_variant_get_string (value, NULL));
        entry_hint_show_hint (entry); /* visual update */
	}
}

static void
entry_activate_cb (GtkEntry *entry, DbusmenuMenuitem *mi)
{
  g_return_if_fail (GTK_IS_ENTRY (entry));
  g_return_if_fail (DBUSMENU_IS_MENUITEM (mi));

	GVariant * value = g_variant_new_string (gtk_entry_get_text (entry));
	
	g_debug ("user typed: %s", g_variant_get_string (value, NULL));

	dbusmenu_menuitem_handle_event (mi, "send", value, gtk_get_current_event_time());
}

static gboolean
new_entry_item (DbusmenuMenuitem * newitem,
                DbusmenuMenuitem * parent,
                DbusmenuClient   * client,
                gpointer user_data)
{
	g_return_val_if_fail(DBUSMENU_IS_MENUITEM(newitem), FALSE);
	g_return_val_if_fail(DBUSMENU_IS_GTKCLIENT(client), FALSE);
	/* Note: not checking parent, it's reasonable for it to be NULL */

	IdoEntryMenuItem *ido = IDO_ENTRY_MENU_ITEM (ido_entry_menu_item_new ());
	GtkEntry *entry = GTK_ENTRY(ido_entry_menu_item_get_entry (ido));
	if (dbusmenu_menuitem_property_get (newitem, DBUSMENU_ENTRY_MENUITEM_PROP_TEXT) != NULL)
		gtk_entry_set_text(entry, dbusmenu_menuitem_property_get(newitem, DBUSMENU_ENTRY_MENUITEM_PROP_TEXT));
	if (dbusmenu_menuitem_property_get (newitem, DBUSMENU_ENTRY_MENUITEM_PROP_HINT) != NULL) {
    entry_set_hint (entry, dbusmenu_menuitem_property_get (newitem, DBUSMENU_ENTRY_MENUITEM_PROP_HINT));
	}

  g_object_set_data (G_OBJECT (ido),
                     "dbusmenuitem",
                     newitem);

	gtk_entry_set_width_chars (entry, 23); /* set some nice aspect ratio for the menu */
  gtk_entry_set_max_length (entry, 140); /* enforce current gwibber limit */

  /* override select-on-focus */
  GtkSettings *settings = gtk_widget_get_settings (GTK_WIDGET (entry));
  g_object_set (settings, "gtk-entry-select-on-focus", FALSE, NULL);

  ido_entry = ido;

#if 0
  /* this is dangerous: it leaves the signal connected even if the dbusmenu
     object is disposed, for example if the service quits
  */
  g_signal_connect (ido,
                    "notify::parent", G_CALLBACK (entry_parent_changed),
                    NULL);
#endif

	dbusmenu_gtkclient_newitem_base(DBUSMENU_GTKCLIENT(client), newitem, GTK_MENU_ITEM(ido), parent);

	/* disconnect the activate signal that newitem_base connected with the wrong
	   widget, ie menuitem, and re-connect it with the /entry/ instead */
	gulong signal_id = g_signal_handler_find (GTK_MENU_ITEM (ido), G_SIGNAL_MATCH_DATA, 0, 0, 0, 0, parent);
  if (signal_id > 0)
    g_signal_handler_disconnect(GTK_MENU_ITEM (ido), signal_id);

	g_signal_connect (DBUSMENU_MENUITEM (newitem), DBUSMENU_MENUITEM_SIGNAL_PROPERTY_CHANGED, G_CALLBACK (entry_prop_change_cb), entry);

	g_signal_connect (GTK_ENTRY (entry), "activate", G_CALLBACK (entry_activate_cb), newitem);

  g_signal_connect (entry,
                    "grab-focus", G_CALLBACK (entry_focus_grab_cb),
                    NULL);

  g_signal_connect (entry,
                    "focus-out-event", G_CALLBACK (entry_focus_out_cb),
                    NULL);

  g_signal_connect (newitem,
                    "destroy",
                    G_CALLBACK (item_destroyed_cb),
                    entry);

	return TRUE;
}

/* Whenever we have a property change on a DbusmenuMenuitem
   we need to be responsive to that. */
static void
about_me_prop_change_cb (DbusmenuMenuitem * mi, gchar * prop, GVariant * value, AboutMeMenuItem *item)
{
  g_return_if_fail (ABOUT_IS_ME_MENU_ITEM (item));

	if (!g_strcmp0(prop, DBUSMENU_ABOUT_ME_MENUITEM_PROP_ICON)) {
    /* reload the avatar icon */
    about_me_menu_item_load_avatar (item, g_variant_get_string(value, NULL));
  } else if (!g_strcmp0(prop, DBUSMENU_MENUITEM_PROP_VISIBLE)) {
    /* normal, ignore */
	} else {
		g_warning("Indicator Item property '%s' unknown", prop);
	}

	return;
}

static gboolean
new_about_me_item (DbusmenuMenuitem * newitem,
                   DbusmenuMenuitem * parent,
                   DbusmenuClient * client,
                   gpointer user_data)
{
	g_return_val_if_fail(DBUSMENU_IS_MENUITEM(newitem), FALSE);
	g_return_val_if_fail(DBUSMENU_IS_GTKCLIENT(client), FALSE);
	/* Note: not checking parent, it's reasonable for it to be NULL */

	const gchar *name = dbusmenu_menuitem_property_get (newitem, DBUSMENU_ABOUT_ME_MENUITEM_PROP_NAME);
	AboutMeMenuItem *about = ABOUT_ME_MENU_ITEM (about_me_menu_item_new (name));
  if (about != NULL) {
    dbusmenu_gtkclient_newitem_base(DBUSMENU_GTKCLIENT(client), newitem, GTK_MENU_ITEM(about), parent);
    const gchar *avatar = dbusmenu_menuitem_property_get (newitem, DBUSMENU_ABOUT_ME_MENUITEM_PROP_ICON);
    about_me_menu_item_load_avatar (about, avatar);
    g_signal_connect(G_OBJECT(newitem), DBUSMENU_MENUITEM_SIGNAL_PROPERTY_CHANGED, G_CALLBACK(about_me_prop_change_cb), about);
  }

	return TRUE;
}

/* Builds the dbusmenu for the service. */
static GtkMenu *
get_menu (IndicatorObject * io)
{
	DbusmenuGtkMenu *menu = dbusmenu_gtkmenu_new(INDICATOR_ME_DBUS_NAME, INDICATOR_ME_DBUS_OBJECT);
	DbusmenuGtkClient * client = dbusmenu_gtkmenu_get_client(menu);

	dbusmenu_client_add_type_handler(DBUSMENU_CLIENT(client), DBUSMENU_ENTRY_MENUITEM_TYPE, new_entry_item);
	dbusmenu_client_add_type_handler(DBUSMENU_CLIENT(client), DBUSMENU_ABOUT_ME_MENUITEM_TYPE, new_about_me_item);

	return GTK_MENU (menu);
}
