
/****************************************************************************
 *
 * All portions copyright their respective authors.  All rights reserved.
 *
 * This file is part of IVMan (ivm).
 *
 * This file may be distributed under the terms of the Q Public License
 * as defined by Troll Tech AS of Norway and appearing in the file
 * LICENSE.QPL included in the packaging of this file.
 * 
 * See http://www.troll.no/qpl for QPL licensing information.
 *
 * $Id: IvmConfigProperties.c,v 1.16 2005/05/12 01:42:34 ro_han Exp $
 *****************************************************************************/

#include <string.h>
#include <glib.h>
#include <libxml/tree.h>
#include <libxml/parser.h>
#include <libxml/xpath.h>
#include <libxml/xpathInternals.h>


#include "IvmConfigProperties.h"
#include "IvmConfigActions.h"
#include "IvmConfigCommon.h"

#include <libhal.h>
LibHalContext *hal_ctx;

#include <dbus/dbus.h>

#include <assert.h>

IvmConfigProperties * parseIvmConfigProperties(char const * const path, char const * udi,
                                                char const * const property)
{
    DBusError dbus_error;
    dbus_error_init(&dbus_error);

    char *currentProperty = NULL;
    gboolean mountable = ivm_device_is_mountable(udi);

    LIBXML_TEST_VERSION IvmConfigProperties * ret = NULL;

    ret = (IvmConfigProperties *) malloc(sizeof(IvmConfigProperties));

    if (ret == NULL)
    {
        DEBUG("Malloc of ret failed");
        return NULL;
    }

    memset(ret, 0, sizeof(IvmConfigProperties));

    ret->checkOnInit = FALSE;
    ret->exec = NULL;
    int num_exec = 0;

    xmlDocPtr doc;
    xmlNodePtr cur;

    doc = xmlParseFile(path);
    if (doc == NULL)
    {
        DEBUG("Document not parsed successfully.");
        return NULL;
    }

    cur = xmlDocGetRootElement(doc);
    if (cur == NULL)
    {
        DEBUG("Empty document");
        xmlFreeDoc(doc);
        return NULL;
    }
    if (xmlStrcmp(cur->name, (const xmlChar *) "PropertiesConfig"))
    {
        DEBUG("Document of the wrong type, root node != PropertiesConfig");
        xmlFreeDoc(doc);
        return NULL;
    }
    cur = cur->children;
    while (cur != NULL)
    {
        //DEBUG("At XML node %s",cur->name);
        if (!xmlStrcmp(cur->name, (const xmlChar *) "Match"))
        {
            /* Check if our device matches - if so, continue iterating
               through siblings, otherwise jump back up to parent */

            gboolean matches = FALSE;

            xmlChar *name = xmlGetProp(cur, "name");
            xmlChar *value = xmlGetProp(cur, "value");

            if (!xmlStrcmp(name, (const xmlChar *) "*"))
                matches = TRUE;

            else if (!xmlStrcmp(name, (const xmlChar *) "ivm.mountable"))
            {
                if (((!xmlStrcmp(value, (const xmlChar *) "true"))
                     && mountable)
                    || ((!xmlStrcmp(value, (const xmlChar *) "false"))
                        && !mountable))
                    matches = TRUE;
            }

            else if (!xmlStrncmp(name, (const xmlChar *) "hal.", 4))
            {
                matches = hal_xml_property_matches(name, value, udi);
            }
            else
                DEBUG("Invalid match name: %s", (char *) name);

            if (matches == TRUE && cur->children != NULL)
                cur = cur->children;
            else if (cur->next != NULL)
                cur = cur->next;
            else
                cur = cur->parent->next;

            xmlFree(value);
            xmlFree(name);
        }

        else if (!xmlStrcmp(cur->name, (const xmlChar *) "Option"))
        {

            xmlChar *name = xmlGetProp(cur, "name");
            xmlChar *value = xmlGetProp(cur, "value");

            if (!xmlStrcmp(name, (xmlChar *) "checkOnInit"))
            {
                if (!xmlStrcmp(value, (xmlChar *) "true"))
                    ret->checkOnInit = TRUE;
                else
                    ret->checkOnInit = FALSE;
            }

            else
                DEBUG("Invalid Option name: %s", (char *) name);

            xmlFree(name);
            xmlFree(value);

            if (cur->next != NULL)
                cur = cur->next;
            else
                cur = cur->parent->next;
        }


        else if (!xmlStrcmp(cur->name, (const xmlChar *) "Property"))
        {
            gboolean matches = FALSE;

            xmlChar *name = xmlGetProp(cur, "name");

            if (!xmlStrncmp(name, (const xmlChar *) "hal.", 4))
            {
                int length = xmlStrlen(name) - 4;

                xmlChar *confProperty = xmlStrsub(name, 4, length);

                if (!xmlStrcmp(confProperty, (const xmlChar *) property))
                {
                    matches = TRUE;
                    currentProperty = (char *)strdup((char *)name);
                }

            } else
                DEBUG("Invalid Property name: %s", (char *) name);

            if (matches == TRUE && cur->children != NULL)
                cur = cur->children;
            else
            {
                if (currentProperty != NULL)
                    free(currentProperty);
                if (cur->next != NULL)
                    cur = cur->next;
                else
                    cur = cur->parent->next;
            }

            xmlFree(name);
        }

        else if (!xmlStrcmp(cur->name, (const xmlChar *) "Action"))
        {
            gboolean matches = FALSE;

            xmlChar *value = xmlGetProp(cur, "value");

            if (!xmlStrcmp(value, (const xmlChar *) "*"))
                matches = TRUE;

            else
                matches = hal_xml_property_matches(currentProperty, value, udi);

            if (matches)
            {
                num_exec++;

                ret->exec = (char **) realloc(ret->exec,
                                              (num_exec + 1) * sizeof(char *));
                ret->exec[num_exec - 1] = (char *)xmlGetProp(cur, "exec");
                ret->exec[num_exec] = NULL;
            }

            if (cur->next != NULL)
                cur = cur->next;
            else
            {
                if (currentProperty != NULL)
                    free(currentProperty);
                cur = cur->parent->next;
            }

            xmlFree(value);
        }


        else if (cur->next != NULL)
            cur = cur->next;
        else
            cur = cur->parent->next;

    }
    xmlFreeDoc(doc);

    xmlCleanupParser();

    ivm_check_dbus_error(&dbus_error);

    return ret;
}








IvmConfigProperties * IvmConfigPropertiesAll(char const * const path, char const * udi)
{
    assert(path != NULL);
    assert(udi != NULL);

    DBusError dbus_error;
    dbus_error_init(&dbus_error);

    char *currentProperty = NULL;
    gboolean mountable = ivm_device_is_mountable(udi);

    LIBXML_TEST_VERSION IvmConfigProperties * ret = NULL;

    ret = (IvmConfigProperties *) malloc(sizeof(IvmConfigProperties));

    if (ret == NULL)
    {
        DEBUG("Malloc of ret failed");
        return NULL;
    }

    memset(ret, 0, sizeof(IvmConfigProperties));

    ret->checkOnInit = FALSE;
    ret->exec = NULL;
    int num_exec = 0;

    xmlDocPtr doc;
    xmlNodePtr cur;

    doc = xmlParseFile(path);
    if (doc == NULL)
    {
        DEBUG("Document not parsed successfully.");
        return NULL;
    }

    cur = xmlDocGetRootElement(doc);
    if (cur == NULL)
    {
        DEBUG("Empty document");
        xmlFreeDoc(doc);
        return NULL;
    }
    if (xmlStrcmp(cur->name, (const xmlChar *) "PropertiesConfig"))
    {
        DEBUG("Document of the wrong type, root node != PropertiesConfig");
        xmlFreeDoc(doc);
        return NULL;
    }
    cur = cur->children;
    while (cur != NULL)
    {
        // DEBUG("At XML node %s",cur->name);
        if (!xmlStrcmp(cur->name, (const xmlChar *) "Match"))
        {
            /* Check if our device matches - if so, continue iterating
               through siblings, otherwise jump back up to parent */

            gboolean matches = FALSE;

            xmlChar *name = xmlGetProp(cur, "name");

            xmlChar *value = xmlGetProp(cur, "value");

            if (!xmlStrcmp(name, (const xmlChar *) "*"))
                matches = TRUE;


            else if (!xmlStrcmp(name, (const xmlChar *) "ivm.mountable"))
                matches = mountable;

            else if (!xmlStrncmp(name, (const xmlChar *) "hal.", 4))
                matches = hal_xml_property_matches(name, value, udi);
            else
                DEBUG("Invalid match name: %s", (char *) name);

            if (matches == TRUE && cur->children != NULL)
                cur = cur->children;
            else if (cur->next != NULL)
                cur = cur->next;
            else
                cur = cur->parent->next;

            xmlFree(value);
            xmlFree(name);
        }

        else if (!xmlStrcmp(cur->name, (const xmlChar *) "Option"))
        {
            xmlChar *name = xmlGetProp(cur, "name");

            xmlChar *value = xmlGetProp(cur, "value");

            if (!xmlStrcmp(name, (xmlChar *) "checkOnInit"))
            {
                if (!xmlStrcmp(value, (xmlChar *) "true"))
                    ret->checkOnInit = TRUE;
                else
                    ret->checkOnInit = FALSE;
            }

            else
                DEBUG("Invalid Option name: %s", (char *) name);

            xmlFree(name);
            xmlFree(value);

            if (cur->next != NULL)
                cur = cur->next;
            else
                cur = cur->parent->next;
        }


        else if (!xmlStrcmp(cur->name, (const xmlChar *) "Property"))
        {
            gboolean matches = FALSE;

            xmlChar *name = xmlGetProp(cur, "name");

            if (!xmlStrncmp(name, (const xmlChar *) "hal.", 4))
                currentProperty = (char *)strdup((char *)name);

            else
                DEBUG("Invalid Property name: %s", (char *) name);

            if (matches == TRUE && cur->children != NULL)
                cur = cur->children;
            else
            {
                if (currentProperty != NULL)
                    free(currentProperty);
                if (cur->next != NULL)
                    cur = cur->next;
                else
                    cur = cur->parent->next;
            }

            xmlFree(name);
        }

        else if (!xmlStrcmp(cur->name, (const xmlChar *) "Action"))
        {

            gboolean matches = FALSE;

            xmlChar *value = xmlGetProp(cur, "value");

            if (!xmlStrcmp(value, (const xmlChar *) "*"))
                matches = TRUE;

            else
                matches = hal_xml_property_matches(currentProperty, value, udi);

            if (matches)
            {
                num_exec++;

                ret->exec = (char **) realloc(ret->exec,
                                              (num_exec +
                                               1) * sizeof(char *));
                ret->exec[num_exec - 1] = (char *)xmlGetProp(cur, "exec");;
                ret->exec[num_exec] = NULL;
            }

            if (cur->next != NULL)
                cur = cur->next;
            else
            {
                if (currentProperty != NULL)
                    free(currentProperty);
                cur = cur->parent->next;
            }

            xmlFree(value);
        }


        else if (cur->next != NULL)
            cur = cur->next;
        else
            cur = cur->parent->next;

    }
    xmlFreeDoc(doc);

    xmlCleanupParser();

    ivm_check_dbus_error(&dbus_error);

    return ret;
}
