/*
 *
 *  KBluetooth4 - KDE Bluetooth Framework
 *
 *  Copyright (C) 2008  Tom Patzig <tpatzig@suse.de>
 *
 *  This file is part of kbluetooth4.
 *
 *  kbluetooth4 is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  kbluetooth4 is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with kbluetooth4; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
*/


#include "trayicon.h"

KBlueTray::KBlueTray(QObject* parent) : QObject(parent)
{
	 modes << "Off" << "Discoverable" << "Connectable" ;
	 adapter = 0;
	 authagent = 0;
	 passagent = 0;
	 server = 0;
	 serversession = 0;
	 
	 setParent(kapp);

	 tray = new QSystemTrayIcon(this);
	 tray->setIcon(KIcon("preferences-system-bluetooth"));

	 mainmenu = new QMenu();

//	 kbluelockMenu = mainmenu->addMenu(KIcon("system-lock-screen"),i18n("KBlueLock"));
	 kbluelockMenu = mainmenu->addMenu(KIcon("preferences-desktop-user-password"),i18n("KBlueLock"));
	 

	 lockEnableAction = kbluelockMenu->addAction(KIcon("system-lock-screen"),i18n("Enable"));
	 lockEnableAction->setCheckable(true);
	 lockEnableAction->setChecked(false);
	 connect(lockEnableAction, SIGNAL(triggered(bool)), this, SLOT(enableLock()));

	 lockConfigureAction = kbluelockMenu->addAction(KIcon("configure"), i18n("Configure..."));
	 connect(lockConfigureAction,SIGNAL(triggered(bool)), this, SLOT(slotConfigureKBlueLock()));
	 lockConfigureAction->setEnabled(false);

	 sendToAction = mainmenu->addAction(KIcon("text-directory"), i18n("Send File"));
	 connect(sendToAction, SIGNAL(triggered(bool)), this, SLOT(sendFile()));

	 wizardAction = mainmenu->addAction(KIcon("preferences-system-bluetooth"), i18n("Device Manager"));
	 connect(wizardAction, SIGNAL(triggered(bool)), this, SLOT(showManager()));
	
	 settingsMenu = mainmenu->addMenu(KIcon("preferences-system"), i18n("Settings"));
	
	 adapterAction = settingsMenu->addAction(KIcon("configure"), i18n("Bluetooth Adapters"));
	 connect(adapterAction, SIGNAL(triggered(bool)), this, SLOT(showAdapterInterface()));
	 
	 serverAction = settingsMenu->addAction(KIcon("network-server"), i18n("Obex server"));
	 serverAction->setCheckable(true);
	 serverAction->setChecked(false);
	 connect(serverAction, SIGNAL(triggered(bool)), this, SLOT(enableServer()));

	 aboutAction = mainmenu->addAction(KIcon("folder-remote"),i18n("About"));
	 connect(aboutAction, SIGNAL(triggered(bool)), this, SLOT(showAboutInfo()));

	 quitAction = mainmenu->addAction(KIcon("application-exit"),i18n("Quit"));
	 connect(quitAction, SIGNAL(triggered(bool)), this, SLOT(slotQuitApp(bool)));
	 
	 tray->setContextMenu(mainmenu);

	 Solid::Control::BluetoothManager &man = Solid::Control::BluetoothManager::self();
         dbus = new QDBusConnection("dbus");

	 connect(&man,SIGNAL(interfaceAdded(const QString&)),this,SLOT(adapterAdded(const QString&)));
	 connect(&man,SIGNAL(interfaceRemoved(const QString&)),this,SLOT(adapterRemoved(const QString&)));
	 connect(&man,SIGNAL(defaultInterfaceChanged(const QString&)),this,SLOT(defaultAdapterChanged(const QString&)));

	 config = new KConfig("kbluetoothrc");
	 confGroup = new KConfigGroup(config,"General");
	 
	 QDir dir(QDir::homePath());
	 dir.mkdir(".kbluetooth4");

/*
	 if (confGroup->exists()) {
		avahiClient->receiver->setTargetPath(confGroup->readEntry("Path",avahiClient->receiver->getTargetPath()));
		avahiClient->setUserName(confGroup->readEntry("Nick",avahiClient->getUserName()));
	 }
*/

	//tray->setVisible(true);
	adapterConfig = 0;
	session = 0;
	progress = 0;
	devSelector = 0;
	lockSelector = 0;
	kblueLockEnabled = false;
	
	if ( man.bluetoothInterfaces().size() > 0 ) {
		onlineMode(man);
	} else {
		offlineMode();
	}
	
}

KBlueTray::~KBlueTray()
{

	if (adapter) {
		if ( adapter->isPeriodicDiscoveryActive() ) 
			adapter->stopPeriodicDiscovery();
		delete adapter;
	}

	if (session)
		delete session;
	if (progress)
		delete progress;
	if (devSelector)
//		delete devSelector;
	if (lockSelector)
//		delete lockSelector;

	if(server) {
		server->stop();
		server->close();
		delete server;
	}
	
	if(serversession) {
		serversession->cancel();
		serversession->disconnect();
		delete serversession;
	}
	
	delete mainmenu;
	delete confGroup;
	delete config;
	delete tray;
	
	qDebug() << "bye bye";
}

void KBlueTray::onlineMode(Solid::Control::BluetoothManager& man)
{
	kDebug() << "online Mode";
	
	secure = man.security();
	authagent = new AuthAgent(this);
	secure->setAuthorizationAgent(authagent);
	passagent = new PasskeyAgent(this);
	secure->setPasskeyAgent(passagent);

	online = true;
	tray->setVisible(true);
	
	adapter = new Solid::Control::BluetoothInterface(man.defaultInterface());
	connect(adapter,SIGNAL(nameChanged(const QString&)),this,SLOT(slotNameChanged(const QString&)));
	connect(adapter,SIGNAL(modeChanged(Solid::Control::BluetoothInterface::Mode)),this,SLOT(slotModeChanged(Solid::Control::BluetoothInterface::Mode)));
	connect(adapter,SIGNAL(remoteDeviceConnected(const QString&)),this,SLOT(slotRemoteDeviceConnected(const QString&)));
	connect(adapter,SIGNAL(remoteDeviceDisconnected(const QString&)),this,SLOT(slotRemoteDeviceDisconnected(const QString&)));
	
	updateTooltip();
	
	sendToAction->setEnabled(true);
	wizardAction->setEnabled(true);
	adapterAction->setEnabled(true);
	kbluelockMenu->setEnabled(true);
	
	server = new ObexServer(this, "00:00:00:00:00:00", "opp", true);
	connect(server, SIGNAL(serverReady()), this, SLOT(slotServerReady()));
	connect(server, SIGNAL(started()), this, SLOT(slotServerStarted()));
	connect(server, SIGNAL(stopped()), this, SLOT(slotServerStopped()));
	connect(server, SIGNAL(closed()), this, SLOT(slotServerClosed()));
	connect(server, SIGNAL(sessionCreated(const QString&)), this, SLOT(slotServerSessionCreated(const QString&)));
	connect(server, SIGNAL(sessionRemoved(const QString&)), this, SLOT(slotServerSessionRemoved(const QString&)));
	connect(server, SIGNAL(errorOccured(const QString&, const QString&)), this, SLOT(slotServerErrorOccured(const QString&, const QString&)));

}

void KBlueTray::offlineMode()
{
	kDebug() << "offline Mode";
	online = false;

	sendToAction->setEnabled(false);
	wizardAction->setEnabled(false);
	adapterAction->setEnabled(false);
	kbluelockMenu->setEnabled(false);
	
	//tray->setToolTip("No Bluetooth Adapter");
	tray->setVisible(false);
		
	if (authagent) {
		delete authagent;
		authagent = 0;
	}
	if (passagent) {
		delete passagent;
		passagent = 0;
	}
	if (adapter) {
		disconnect(adapter,0,0,0);
		delete adapter;
		adapter = 0;
	}
	
	if(server) {
		server->close();
		delete server;
		server = 0;
	}
	
	if(serversession) {
		serversession->cancel();
		serversession->disconnect();
		delete serversession;
		serversession = 0;
	}
}

void KBlueTray::updateTooltip() {
	
	if(!adapter || !server) return;
	
	toolTipInfo = 	"<b>Name:</b> " + adapter->name() + "<br />" \
			+	"<b>Address:</b> " + adapter->address() + "<br />" \
			+      	"<b>Mode:</b> " + modes.at(adapter->mode()) + "<br />" \
			+	"<b>KBlueLock:</b> ";

	if (!kblueLockEnabled)
		toolTipInfo += "Disabled"; 
	else
		toolTipInfo += ("Enabled for " + adapter->getRemoteName(lockDevice.split('/').at(4)));
		
	toolTipInfo += "<br /><b>Receiving Files:</b> ";
	if(server) {
		toolTipInfo += ((server->isStarted()) ? "Enabled" : "Disabled");
	} else {
		toolTipInfo += "Disabled";
	}
	
	tray->setToolTip(toolTipInfo);
}

void KBlueTray::adapterAdded(const QString& ubi)
{
	 kDebug() << "adapterAdded: " << ubi;

	 if ( !online ) {
	 	kDebug() << "new default adapter UBI: " << ubi;
		onlineMode(Solid::Control::BluetoothManager::self());
	 }
}

void KBlueTray::adapterRemoved(const QString& ubi)
{
	 kDebug() << "adapterRemoved: " << ubi;
	 Solid::Control::BluetoothManager &man = Solid::Control::BluetoothManager::self();
	 
	 if (man.bluetoothInterfaces().size() == 0) {
		offlineMode();
		kDebug() << "default Adapter removed";
	 }
}

void KBlueTray::defaultAdapterChanged(const QString& ubi)
{
	kDebug() << "default Adapter changed " << ubi;
	defaultAdapterUBI = ubi;

}

void KBlueTray::slotQuitApp(bool /*set*/)
{
        kapp->quit();
}

void KBlueTray::showAboutInfo()
{
        KAboutData aboutData("kbluetooth4", 0, ki18n("KBluetooth4"), "0.2",
                           ki18n("KDE4 Bluetooth Framework"), KAboutData::License_GPL_V2);
        aboutData.addAuthor(ki18n("Tom Patzig"), KLocalizedString(), "tpatzig@suse.de", "http://www.kde.org/");
	aboutData.addCredit(ki18n("Arjan Topolovec"), ki18n(""), "arjan.top@gmail.com", "");
        aboutData.setProgramLogo(KIcon("preferences-system-bluetooth"));

        KAboutApplicationDialog about(&aboutData, mainmenu);
//        about.setIcon( KIcon( "preferences-system-bluetooth" ) );
        about.exec();
}

void KBlueTray::showAdapterInterface()
{
	if (!adapterConfig)
		adapterConfig = new AdapterConfig(this);

	adapterConfig->initialize();

}

void KBlueTray::showWizard()
{
	BTWizard wiz;
	wiz.exec();
}

void KBlueTray::showManager()
{
//	DeviceMan manager;
	kDebug() << "Starting Device Manager";
	KProcess process;
	process.setProgram("kbluetooth4-devicemanager");
	process.startDetached();

	
//	manager.exec();
}

//  ################# SEND FILE ################################

void KBlueTray::sendFile()
{
	QString fileName = KFileDialog::getOpenFileName(KUrl("./"),"*",mainmenu, tr("Select File"));
        if (!fileName.isEmpty()) {
		fileToSend = fileName;
		devSelector = new DeviceSel(this,QString("computer,phone").split(','));
	 	devSelector->hideExtraCheckBox();
		devSelector->setInfoLabel(i18n("Selecting a device starts the Filetransfer."));
		connect(devSelector,SIGNAL(deviceSelected(const QString&,const QString&)),this,SLOT(slotSendFile(const QString&,const QString&)));

        }


}

void KBlueTray::slotSendFile(const QString& mac,const QString& /* ubi */ )
{
	session = new ObexSession(this,mac,"ftp");
	connect(session,SIGNAL(connected()),this,SLOT(obexSessionReady()));

}

void KBlueTray::obexSessionReady()
{
	kDebug() << "is connected: " << session->isConnected();

	session->sendFile(fileToSend);
	
	connect(session,SIGNAL(transferStarted(const QString&,const QString&,qulonglong)),this,SLOT(slotFileSendStarted(const QString,const QString&,qulonglong)));
	connect(session,SIGNAL(transferProgress(qulonglong)),this,SLOT(slotTransferProgress(qulonglong)));
	connect(session,SIGNAL(transferCompleted()),this,SLOT(slotFileSendFinished()));
	connect(session, SIGNAL(errorOccurred(const QString&, const QString&)), this, SLOT(slotServerErrorOccured(const QString&, const QString&)));

}

void KBlueTray::slotFileSendStarted(const QString& filename ,const QString& path ,qulonglong size)
{
	totalFileSize = size;

	kDebug() << "file send started: " << path << filename;
	kDebug() << "size: " << size;

	progress = new QProgressDialog(i18n("Sending ") + filename,i18n("Cancel"),0,100);
	progress->setAutoClose(true);
	progress->setWindowIcon(KIcon("preferences-system-bluetooth"));
	progress->show();
	connect(progress,SIGNAL(canceled()),this,SLOT(slotFileSendCanceled()));
	
}

void KBlueTray::slotTransferProgress(qulonglong val)
{
	kDebug() << "Transfered: " << val;

		
	qulonglong prog = ( ( val * 100 ) / totalFileSize) ;

	kDebug() << "Progress: " << prog;

	if(progress)
		  progress->setValue(prog);

}

void KBlueTray::slotFileSendFinished()
{
	kDebug() << "Transfer completed";

	session->disconnect();
	session->close();

	delete session;
	session = 0;
	

}

void KBlueTray::slotFileSendCanceled()
{
	session->cancel();
	session->disconnect();
	session->close();

	delete session;
	delete devSelector;

}

// ######################################### KBlueLock #####################################################

void KBlueTray::enableLock()
{
	if (lockEnableAction->isChecked() ) {
		if (lockDevice.isEmpty()) {
	 	 	kblueUnlockEnabled = true;
			lockSelector = new DeviceSel(this,QString("computer,phone").split(','));
			lockSelector->setOkButtonText(i18n("Select"));
			lockSelector->setInfoLabel(i18n("Selecting a device enables KBlueLock (a balloon message in the tray icon pops up when your target device is found and KBlueLock is ready).\nIf the device disappears your screen gets locked."));

			connect(lockSelector,SIGNAL(deviceSelected(const QString&,const QString&)),this,SLOT(enableBlueLock(const QString&,const QString&)));
			connect(lockSelector,SIGNAL(selectorCancelled()),this,SLOT(slotBlueSelectorCancelled()));
			connect(lockSelector,SIGNAL(unlockChanged(bool)),this,SLOT(slotUnlockScreenChanged(bool)));
		} else 
			enableBlueLock("",lockDevice);

	} else {
		disableBlueLock();
	}
	updateTooltip();
}


void KBlueTray::enableBlueLock(QString  mac ,const QString& ubi)
{
	disconnect(lockSelector,0,0,0);
	disconnect(adapter,SIGNAL(remoteDeviceDisappeared(const QString)),0,0);

	lockDevice = ubi;
	if (!lockDevice.isEmpty()) {

		if (mac.isEmpty())
			mac = ubi.split('/').at(4);

		lockConfigureAction->setEnabled(true);

		kDebug() << "KBlueLock enabled";
		kDebug() << "lockdevice: " << lockDevice;
		if (! adapter->isPeriodicDiscoveryActive()) {
			adapter->startPeriodicDiscovery();
			kDebug() << "Periodic Discovery Started";
			connect(adapter,SIGNAL(remoteDeviceFound(const QString&,int,int)),this,SLOT(slotTargetDeviceFound(const QString&,int,int)));
			tray->showMessage("KBluetooth","kbluelock is looking for the target device...",
	 	 	 	 	 	 	 	 QSystemTrayIcon::Information,30000);
		}

		connect(adapter,SIGNAL(remoteDeviceDisappeared(const QString)),this,SLOT(slotValidateLockDev(const QString&)));
		
		kblueLockEnabled = false;

		
	} else {
		lockEnableAction->setChecked(false);
	}
}

void KBlueTray::disableBlueLock()
{

	kDebug() << "KBlueLock disabled";
	kDebug() << "lockdevice: " << lockDevice;
	
	updateTooltip();
	kblueLockEnabled = false;
	
	lockConfigureAction->setEnabled(false);

	if ( adapter->isPeriodicDiscoveryActive() ) {
		adapter->stopPeriodicDiscovery();
		kDebug() << "Periodic Discovery Stopped";

	}
	disconnect(adapter,SIGNAL(remoteDeviceDisappeared(const QString)),this,SLOT(slotValidateLockDev(const QString&)));
	disconnect(adapter,SIGNAL(remoteDeviceFound(const QString&,int,int)),this,SLOT(slotTargetDeviceFound(const QString&,int,int)));

}

void KBlueTray::slotValidateLockDev(const QString& ubi)
{
	if (ubi == lockDevice) {
		
		kDebug() << "KBlueLock armed -- Lock Device disappeared";
		QDBusConnection dbusconn = dbus->connectToBus(QDBusConnection::SessionBus, "dbus");
		QString service = "org.freedesktop.ScreenSaver";
	        QString path = "/ScreenSaver";
        	QString method = "Lock";
        	QString iface = "org.freedesktop.ScreenSaver";

        	QDBusInterface* interface = new QDBusInterface(service,path,iface,dbusconn);
        	interface->call(method);

	 	 	 // log the Away times ;)
		QFile file(QDir::homePath() + "/kbluelock.log");
	 	if (!file.open(QIODevice::Append | QIODevice::Text))
         		kDebug() << "cant open logfile";
	 	else {
			QTextStream out(&file);
	 	 	awaytime = QDateTime::currentDateTime();
			out << "Away from Computer: " << awaytime.toString() << "\n";
			file.flush();	 	 	
			file.close();
	 	}
		

		connect(adapter,SIGNAL(remoteDeviceFound(const QString&,int,int)),this,SLOT(slotTargetDeviceBack(const QString&,int,int)));

	}
}


void KBlueTray::slotConfigureKBlueLock()
{
		if ( adapter->isPeriodicDiscoveryActive() ) 
			adapter->stopPeriodicDiscovery();
			kDebug() << "Periodic Discovery Stopped";
			
		kblueLockEnabled = false;

		lockSelector = new DeviceSel(this,QString("computer,phone").split(','));
		lockSelector->setOkButtonText("Select");
		lockSelector->setInfoLabel(i18n("Selecting a device enables KBlueLock (a balloon message in the tray icon pops up when your target device is found and KBlueLock is ready).\nIf the device disappears your screen gets locked."));
		connect(lockSelector,SIGNAL(deviceSelected(const QString&,const QString&)),this,SLOT(enableBlueLock(const QString&,const QString&)));
		connect(lockSelector,SIGNAL(selectorCancelled()),this,SLOT(slotBlueSelectorCancelled()));
		connect(lockSelector,SIGNAL(unlockChanged(bool)),this,SLOT(slotUnlockScreenChanged(bool)));

}


void KBlueTray::slotBlueSelectorCancelled()
{

	disconnect(adapter,SIGNAL(remoteDeviceDisappeared(const QString)),this,SLOT(slotValidateLockDev(const QString&)));
	disconnect(adapter,SIGNAL(remoteDeviceFound(const QString&,int,int)),this,SLOT(slotTargetDeviceFound(const QString&,int,int)));
	disconnect(lockSelector,0,0,0);
	kblueLockEnabled = false;
	enableBlueLock("",lockDevice);

}
void KBlueTray::slotTargetDeviceFound(const QString& ubi,int /*dclass*/,int /*rssi*/)
{
	if ((!kblueLockEnabled) && (ubi == lockDevice)) {
		kDebug() << "Target device found - Kbluelock ready";
		tray->showMessage("KBluetooth","kbluelock is enabled now");
		kblueLockEnabled = true;
		disconnect(adapter,SIGNAL(remoteDeviceFound(const QString&,int,int)),this,SLOT(slotTargetDeviceFound(const QString&,int,int)));
		updateTooltip();
	}

}

void KBlueTray::slotTargetDeviceBack(const QString& ubi,int /*dclass*/,int /*rssi*/)
{
	if ( (kblueLockEnabled && kblueUnlockEnabled ) && (ubi == lockDevice)) {
		kDebug() << "Target device is back. Disabling Screensaver.";

		QDBusConnection dbusconn = dbus->connectToBus(QDBusConnection::SessionBus, "dbus");
		QString service = "org.freedesktop.ScreenSaver";
	        QString path = "/ScreenSaver";
       		QString method = "SetActive";
       		QString iface = "org.freedesktop.ScreenSaver";

       		QDBusInterface* interface = new QDBusInterface(service,path,iface,dbusconn);
       		interface->call(method,false);

			 // log the Away times ;)
		QFile file(QDir::homePath() + "/.kbluetooth4/kbluelock.log");
		if (!file.open(QIODevice::Append | QIODevice::Text))
         	 	 kDebug() << "cant open logfile";
	 	else {
			QTextStream out(&file);

			out << "Back on Computer: " << QDateTime::currentDateTime ().toString() << "\n";
	 	 	out << "---------------------------------------------\n";
	 	    	out << ">>> Pause: " << awaytime.secsTo(QDateTime::currentDateTime())/60 << " min ";
	 	    	out << awaytime.secsTo(QDateTime::currentDateTime())%60 << " sec\n\n";
			file.flush();	 	 	
			file.close();
	 	}
	
		disconnect(adapter,SIGNAL(remoteDeviceFound(const QString&,int,int)),this,SLOT(slotTargetDeviceBack(const QString&,int,int)));
	}
}

//################# external SIGNALS ####################################

void KBlueTray::slotUnlockScreenChanged(bool val)
{
	kDebug() << "slotUnlockScreenChanged " << val;
	kblueUnlockEnabled = val;
}


void KBlueTray::slotModeChanged(Solid::Control::BluetoothInterface::Mode mode)
{
	  kDebug() << "Adapter mode changed: " << modes.at(mode);
	  updateTooltip();
}

void KBlueTray::slotNameChanged(const QString& name)
{
	  kDebug() << "Adapter name changed: " << name;
	  updateTooltip();
}

void KBlueTray::slotRemoteDeviceConnected(const QString& address)
{
	kDebug() << "Connection to " << address << "established";
	tray->showMessage("KBluetooth",i18n("Connected to ") + adapter->getRemoteName(address));
}

void KBlueTray::slotRemoteDeviceDisconnected(const QString& address)
{
	kDebug() << "Disconnected from " << address << "lost";
	tray->showMessage("KBluetooth",i18n("Disconnected from ") + adapter->getRemoteName(address));
}

//################### RECEIVE FILES #############################################

void KBlueTray::slotServerReady() {
	if(!server->isStarted())
		server->start(QDir::homePath()+"/.kbluetooth4", true, true);
}

void KBlueTray::slotServerErrorOccured(const QString& error_name, const QString& error_message) {
	kDebug() << error_name << ": " << error_message;
	KMessageBox::error(0, error_name, error_message);
}

void KBlueTray::enableServer() {
	if(serverAction->isChecked()) {
		slotServerReady();
	} else {
		server->stop();
	}
}

void KBlueTray::slotServerStarted() {
	if(!server->isStarted()) {
	      kDebug() << "obex server not started";
	      server->close();
	} else {
	      kDebug() << "obex server started";
	      serverAction->setChecked(true);
	}
	updateTooltip();
}

void KBlueTray::slotServerStopped() {
	serverAction->setChecked(false);
	updateTooltip();
	kDebug() << "obex server stopped";
}

void KBlueTray::slotServerClosed() {
	kDebug() << "obex server closed";
	serverAction->setChecked(false);
	updateTooltip();
	delete server;
	server = 0;
}

void KBlueTray::slotServerSessionCreated(const QString& path) {
	serversession = new ObexServerSession(this, path);
	connect(serversession, SIGNAL(disconnected()), this, SLOT(slotServerTransferDisconnected()));
	
	/*if(receiveFiles == 4) {
		serversession->disconnect();
		return;
	}*/
	
	/*KDirSelectDialog dlg(KUrl("."), false, mainmenu);
	dlg.exec();
	
	kDebug() << dlg.url().url();*/
	
	connect(serversession, SIGNAL(transferStarted(const QString&, const QString&, qulonglong)), this, SLOT(slotServerSessionTransferStarted(const QString&, const QString&, qulonglong)));
	connect(serversession, SIGNAL(transferProgress(qulonglong)), this, SLOT(slotTransferProgress(qulonglong)));
	connect(serversession, SIGNAL(transferCompleted()), this, SLOT(slotServerSessionTransferCompleted()));
	connect(serversession, SIGNAL(errorOccurred(const QString&, const QString&)), this, SLOT(slotServerErrorOccured(const QString&, const QString&)));
}

void KBlueTray::slotServerSessionRemoved(const QString& /*path*/) {
	if(progress) {
		 disconnect(progress, 0,0,0);
		 delete progress;
		 progress = 0;
	}
	if(serversession) {
		serversession->disconnect();
		delete serversession;
		serversession = 0;
	}
}

void KBlueTray::slotServerSessionTransferStarted(const QString& filename, const QString& local_path, qulonglong total_bytes) {
	
	/*int receiveFiles = KMessageBox::questionYesNo(mainmenu, "Receive files over bluetooth?", "Incoming files over Bluetooth");
	
	if(receiveFiles == 4) {
		serversession->reject();
		serversession->disconnect();
		return;
	}*/
	
	//serversession->accept();
	
	kDebug() << local_path;
	kDebug() << total_bytes;
	
	totalFileSize = total_bytes;
	
	progress = new QProgressDialog(i18n("Receiving ") + filename, i18n("Cancel"), 0, 100);
	progress->setAutoClose(true);
	progress->setWindowIcon(KIcon("preferences-system-bluetooth"));
	progress->window()->setWindowTitle(i18n("Receiving ") + filename);
	progress->show();
	connect(progress, SIGNAL(canceled()), this, SLOT(slotFileReceiveCancelled()));
}

void KBlueTray::slotFileReceiveCancelled() {
	serversession->cancel();
	serversession->disconnect();
}

void KBlueTray::slotServerSessionTransferCompleted() {
	/*KDirSelectDialog dlg(KUrl("."), false, 0);
	dlg.exec();
	
	kDebug() << dlg.url().url();*/
	
	delete progress;
	progress = 0;
	
}

void KBlueTray::slotServerTransferDisconnected() {
	if(progress) {
		 delete progress;
		 progress = 0;
	}
}

