/*
    This file is part of kio_obex.

    Copyright (c) 2003 Mathias Froehlich <Mathias.Froehlich@web.de>

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public License
    along with this library; see the file COPYING.LIB.  If not, write to
    the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
    Boston, MA 02111-1307, USA.
*/

#include <qstring.h>
#include <qlayout.h>
#include <qtabwidget.h>
#include <qpushbutton.h>
#include <qwidget.h>
#include <qlabel.h>
#include <qlistbox.h>
#include <qspinbox.h>
#include <qwidgetstack.h>
#include <krestrictedline.h>
#include <klocale.h>

#include "irdadevicewidget.h"
#include "bluetoothdevicewidget.h"
#include "serialdevicewidget.h"
#include "ipdevicewidget.h"

#include "edit.h"

Edit::Edit( QWidget* parent, const char* name, bool modal, WFlags /*fl*/ )
  : KDialogBase( parent, name , modal, i18n( "OBEX Device Configuration" ), KDialogBase::Ok|KDialogBase::Cancel, KDialogBase::Ok, true )
{
  int margin = 11;
  int space = 6;
  QWidget *page = new QWidget( this );
  setMainWidget(page);
  mTopLayout = new QGridLayout( page, 2, 3, margin, space, "mTopLayout");

  mTabWidget = new QTabWidget( page, "TabWidget" );

  // Prepare page 0 of the wizard.
  mGeneralTab = new QWidget( mTabWidget, "GeneralTab" );
  mGeneralTabLayout = new QGridLayout( mGeneralTab, 2, 2, margin, space, "mGeneralTabLayout");

  mAliasInputText = new QLabel( mGeneralTab, "mAliasInputText" );
  mAliasInputText->setText( i18n( "Device alias:" ) );
  mGeneralTabLayout->addWidget( mAliasInputText, 0, 0 );

  mAliasInput = new KRestrictedLine( mGeneralTab, "mAliasInput" );
  mAliasInput->setValidChars( "0123456789abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ_-" );
  mGeneralTabLayout->addWidget( mAliasInput, 0, 1 );

  mTransportInputText = new QLabel( mGeneralTab, "mTransportInputText" );
  mTransportInputText->setText( i18n( "Connection transport:" ) );
  mGeneralTabLayout->addWidget( mTransportInputText, 1, 0 );

  mTransportInputBox = new QListBox( mGeneralTab, "mTransportInputBox" );
  mTransportInputBox->insertItem( i18n( "Bluetooth" ), 0 );
#if !defined(HAVE_QOBEX_BLUETOOTH)
  mTransportInputBox->item( 0 )->setSelectable( false );
#endif
  mTransportInputBox->insertItem( i18n( "IrDA" ), 1 );
#if !defined(HAVE_QOBEX_IRDA)
  mTransportInputBox->item( 1 )->setSelectable( false );
#endif
  mTransportInputBox->insertItem( i18n( "Serial Line - Ericsson mobile protocol" ) );
  mTransportInputBox->insertItem( i18n( "Serial Line - Siemens mobile protocol" ) );
  mTransportInputBox->insertItem( i18n( "Serial Line" ) );
  mTransportInputBox->insertItem( i18n( "TCP/IP" ) );
  mGeneralTabLayout->addWidget( mTransportInputBox, 1, 1 );

  mTabWidget->insertTab( mGeneralTab, i18n( "General" ) );


  // Prepare page 2 of the wizard
  mWidgetStack = new QWidgetStack( mTabWidget, "mWidgetStack" );

  mSerial = new SerialDeviceWidget( mWidgetStack, "SerialTransportPage" );
  mWidgetStack->addWidget( mSerial, SerialConfigurationPage );


  // Prepare page 2 for bluetooth transports
  mBluetooth = new BluetoothDeviceWidget( mWidgetStack, "BluetoothTransportPage" );
  mWidgetStack->addWidget( mBluetooth, BluetoothConfigurationPage );

  // Prepare page 2 for IrDA transports
  mIrDA = new IrDADeviceWidget( mWidgetStack, "IrDATransportPage" );
  mWidgetStack->addWidget( mIrDA, IrDAConfigurationPage );

  // Prepare page 2 of the wizard for ip connections
  mIp = new IpDeviceWidget( mWidgetStack, "IpTransportPage" );
  mWidgetStack->addWidget( mIp, IpConfigurationPage );

  mTabWidget->insertTab( mWidgetStack, i18n( "Parameters" ) );



  mAdvancedTab = new QWidget( mTabWidget, "AdvancedParameters" );
  mAdvancedTabLayout = new QGridLayout( mAdvancedTab, 2, 2, margin, space, "mAdvancedTabLayout");

  mDisconnectTimeoutText = new QLabel( mAdvancedTab, "mDisconnectTimeoutText" );
  mDisconnectTimeoutText->setText( i18n( "Timeout for disconnecting an idle client:" ) );
  mAdvancedTabLayout->addWidget( mDisconnectTimeoutText, 0, 0 );

  mDisconnectTimeoutInput = new QSpinBox( mAdvancedTab, "mDisconnectTimeoutInput" );
  mDisconnectTimeoutInput->setMinValue( 0 );
  mAdvancedTabLayout->addWidget( mDisconnectTimeoutInput, 0, 1 );

  mOverridePermsText = new QLabel( mAdvancedTab, "mOverridePermsText" );
  mOverridePermsText->setText( i18n( "Permissions to add to the ones sent by the device:" ) );
  mAdvancedTabLayout->addWidget( mOverridePermsText, 1, 0 );

  mOverridePermsInput = new KRestrictedLine( mAdvancedTab, "mOverridePermsInput" );
  mOverridePermsInput->setValidChars( "01234567" );
  mAdvancedTabLayout->addWidget( mOverridePermsInput, 1, 1 );

  mTabWidget->insertTab( mAdvancedTab, i18n( "Advanced" ) );

  mTopLayout->addMultiCellWidget( mTabWidget, 0, 0, 0, 2 );


  // signals and slots connections
  connect( mTransportInputBox, SIGNAL(selectionChanged()),
	   SLOT(slotTransportSelectionChanged()) );
//   connect( mTransportInputBox, SIGNAL(highlighted(int)),
// 	   SLOT(slotTransportHighlighted(int)) );

  connect( mAliasInput, SIGNAL(textChanged(const QString&)),
	   SLOT(slotCheckConfig()) );
  connect( mSerial, SIGNAL(configurationReady(bool)),
	   SLOT(slotCheckConfig()) );
  connect( mIrDA, SIGNAL(configurationReady(bool)),
	   SLOT(slotCheckConfig()) );
  connect( mBluetooth, SIGNAL(configurationReady(bool)),
	   SLOT(slotCheckConfig()) );
  connect( mIp, SIGNAL(configurationReady(bool)),
	   SLOT(slotCheckConfig()) );
}

Edit::~Edit()
{
}

QString Edit::name() const
{
  return mAliasInput->text();
}

QString Edit::transport() const
{
  switch ( mTransportInputBox->index( mTransportInputBox->selectedItem() ) ) {
  case 0:
    return "bluetooth";
  case 1:
    return "irda";
  case 2:
    return "serialericsson";
  case 3:
    return "serialsiemens";
  case 4:
    return "serial";
  case 5:
    return "ip";
  }
  return QString::null;
}

QString Edit::device() const
{
  return mSerial->device();
}

QString Edit::peer() const
{
  if ( transport() == "irda" )
    return mIrDA->address();
  else if ( transport() == "bluetooth" )
    return mBluetooth->address();
  else
    return mIp->peer();
}

int Edit::overridePerms() const
{
  return (mOverridePermsInput->text()).toInt( 0, 8 );
}

int Edit::ipPort() const
{
  return mIp->ipPort();
}

int Edit::speed() const
{
  return mSerial->speed();
}

int Edit::disconnectTimeout() const
{
  return mDisconnectTimeoutInput->value();
}

void Edit::setName( const QString& s )
{
  return mAliasInput->setText( s );
}

void Edit::setTransport( const QString& t )
{
  if ( t == "bluetooth" ) {
    mTransportInputBox->setSelected( 0, true );
  }
  else if ( t == "irda" ) {
    mTransportInputBox->setSelected( 1, true );
  }
  else if ( t =="serialericsson" ) {
    mTransportInputBox->setSelected( 2, true );
  }
  else if ( t == "serialsiemens" ) {
    mTransportInputBox->setSelected( 3, true );
  }
  else if ( t == "serial" ) {
    mTransportInputBox->setSelected( 4, true );
  }
  else if ( t == "ip" ) {
    mTransportInputBox->setSelected( 5, true );
  }
}

void Edit::setDevice( const QString& d )
{
  mSerial->setDevice( d );
}

void Edit::setPeer( const QString& p )
{
  mIrDA->setAddress( p );
  mBluetooth->setAddress( p );
  mIp->setPeer( p );
}

void Edit::setOverridePerms( int op )
{
  mOverridePermsInput->setText( QString::number( op, 8 ) );
}

void Edit::setIpPort( int p )
{
  mIp->setIpPort( p );
}

void Edit::setSpeed( int s )
{
  mSerial->setSpeed( s );
}

void Edit::setDisconnectTimeout( int to )
{
  return mDisconnectTimeoutInput->setValue( to );
}

void Edit::slotCheckConfig()
{
  QString t = transport();
  bool ok = !name().isEmpty();

  if ( t =="serialericsson" || t == "serialsiemens" || t == "serial" ) {
    ok = ok && 0 < speed() && !device().isEmpty();
  }
  else if ( t == "ip" ) {
    ok = ok && 0 < ipPort() && !peer().isEmpty();
  }
  enableButtonOK( ok );
}

void Edit::slotTransportSelectionChanged() {
  switch ( mTransportInputBox->index( mTransportInputBox->selectedItem() ) ) {
  case 0:
    mWidgetStack->raiseWidget( int(BluetoothConfigurationPage) );
    break;
  case 1:
    mWidgetStack->raiseWidget( int(IrDAConfigurationPage) );
    break;
  case 2:
  case 4:
    mWidgetStack->raiseWidget( int(SerialConfigurationPage) );
    mSerial->setAvailableSpeeds( SerialDeviceWidget::AllSpeeds );
    break;
  case 3:
    mWidgetStack->raiseWidget( int(SerialConfigurationPage) );
    mSerial->setAvailableSpeeds( SerialDeviceWidget::BFBSpeeds );
    break;
  case 5:
    mWidgetStack->raiseWidget( int(IpConfigurationPage) );
    break;
  }
  slotCheckConfig();
}

#include "edit.moc"
