/***************************************************************************
 *
 * knetworkmanager-vpn.cpp - A NetworkManager frontend for KDE 
 *
 * Copyright (C) 2006 Novell, Inc.
 *
 * Author: Timo Hoenig <thoenig@suse.de>, <thoenig@nouse.net>
 *         Helmut Schaa <hschaa@suse.de>, <helmut.schaa@gmx.de>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 **************************************************************************/

#define SERVICE_DIR "/etc/NetworkManager/VPN"
#define GCONF_PRE   "/.gconf"
#define VPN_PATH    "/system/networking/vpn_connections"

#include <stdlib.h>
#include <kconfig.h>
#include <qdom.h> 
#include <qdir.h>
#include <klocale.h>
#include <kdebug.h>
#include <kstddirs.h>
#include <kprocess.h>
#include <kconfig.h>
#include <kplugininfo.h>

#include "knetworkmanager-vpn.h"
#include "knetworkmanager-vpn_dbus.h"
#include "knetworkmanager-storage.h"
#include "knetworkmanager-vpnplugin.h"
#include "knetworkmanager-vpnconnectionsdialog.h"
#include "knetworkmanager-vpnauthenticationdialog.h"

/*
 * class VPNConnection
 *
 */
void
VPNConnection::setConfigGroup(const QString& cfgGrp)
{
	_cfgGrp   = new KConfigGroup(KGlobal::config(), cfgGrp);
	_readonly = false;
}

QString
VPNConnection::getName () const
{
	return _name;
}

void
VPNConnection::setName (const QString & name)
{
	_name = name;
}

VPNService*
VPNConnection::getVPNService() const
{
	return _vpnService;
}

QString
VPNConnection::getService () const
{
	if (_vpnService)
		return _vpnService->getService();
	else
		return QString::null;
}

QString
VPNConnection::getServiceName () const
{
	if (_vpnService)
		return _vpnService->getName();
	else
		return QString::null;
}

void
VPNConnection::setServiceName(const QString& service)
{
	_vpnService  = _vpn->getVPNService(service);
}

QString
VPNConnection::getUser () const
{
	return _user;
}

void
VPNConnection::setUser (const QString & user)
{
	_user = user;
}

QStringList
VPNConnection::getRoutes () const
{
	return _routes;
}

void
VPNConnection::addRoute (const QString & route)
{
	_routes.append (route);
}

void
VPNConnection::setRoutes(const QStringList& routes)
{
	_routes = routes;
}

QStringList
VPNConnection::getData () const
{
	return _data;
}

void
VPNConnection::addData (const QString & data)
{
	_data.append (data);
}

void
VPNConnection::setData(const QStringList & data)
{
	// overwrite all data
	_data = data;
}

QStringList
VPNConnection::getPasswords () const
{
	return _passwords;
}

void
VPNConnection::addPasswords (const QStringList & passwords, bool storePasswordsPermanent, bool storePasswordsSession)
{
	_passwords = passwords;
	if (_storePasswordsPermanent = storePasswordsPermanent)
	{
		// store passwords in KWallet
		KNetworkManagerStorage* storage = KNetworkManagerStorage::getInstance();
		QMap<QString, QString> map;
		map.insert("passwords", _passwords.join(" "));
		storage->storeCredentials(_name, map);
	}
	// should the pwd stored for this session?
	_storePasswordsSession = storePasswordsSession;
}

NMVPNActStage
VPNConnection::getActivationStage () const
{
	return _activationStage;
}

void
VPNConnection::setActivationStage (NMVPNActStage activationStage)
{
	_activationStage = activationStage;
	emit activationStateChanged ();
}

void
VPNConnection::setVPNConnectionFailure(QString& member, QString& err_msg)
{
	emit connectionFailure(member, err_msg);
}

QString
VPNConnection::getAuthHelper () const
{
	return _vpnService->getAuthHelper();
}

void
VPNConnection::setAuthHelper (const QString & authHelper)
{
	_authHelper = authHelper;
}

bool
VPNConnection::isValid ()
{
	bool status = true;
	
	/* _routes is allowed to be empty for some VPN service */
	if (_name == QString::null || _user == QString::null || _data.empty () || _vpnService == NULL)
		status = false;

	return status;
}

bool
VPNConnection::isReadonly()
{
	return _readonly;
}

bool
VPNConnection::isConfirmedByNM()
{
	return _confirmedByNM;
}

void
VPNConnection::setConfirmedByNM(bool confirmed)
{
	_confirmedByNM = confirmed;
}

VPNConnection::VPNConnection (VPN* parent, const char * name ) : QObject( parent, name )
{
	_vpn             	= parent;
	_name            	= QString::null;
	_user            	= getenv ("USER");
	_authHelper      	= QString::null;
	_activationStage 	= NM_VPN_ACT_STAGE_UNKNOWN;
	_dirty           	= true;
	_deleted         	= false;
	_vpnService      	= NULL;
	_storePasswordsPermanent = false;
	_storePasswordsSession = false;
	_cfgGrp          	= NULL;
	_readonly        	= true;
	_confirmedByNM    = false;
}

VPNConnection::VPNConnection (const QString& cfgGrp, VPN* parent, const char * name ) : QObject( parent, name )
{
	_vpn             	= parent;
	_name            	= QString::null;
	_user            	= getenv ("USER");
	_authHelper      	= QString::null;
	_activationStage 	= NM_VPN_ACT_STAGE_UNKNOWN;
	_dirty           	= true;
	_deleted         	= false;
	_vpnService      	= NULL;
	_storePasswordsPermanent = false;
	_storePasswordsSession = false;
	_readonly        	= false;
	_confirmedByNM    = false;

	// our ConfigGroup
	_cfgGrp          	= new KConfigGroup(KGlobal::config(), cfgGrp);

	// read attribs from config
	_name            	= _cfgGrp->readEntry("name");
	_vpnService      	= _vpn->getVPNService(_cfgGrp->readEntry("service_name"));
	_routes          	= _cfgGrp->readPropertyEntry("routes", QVariant::StringList).toStringList();
	_data            	= _cfgGrp->readPropertyEntry("vpn_data", QVariant::StringList).toStringList();
}


VPNConnection::~VPNConnection ()
{
	this->save();
	if (_cfgGrp)
		delete _cfgGrp;
}

bool VPNConnection::save()
{
	if(!_cfgGrp)
		return false;

	if (_dirty)
	{
		// write attribs to configfile
		_cfgGrp->writeEntry("name", _name);
		_cfgGrp->writeEntry("service_name", _vpnService->getName());
		_cfgGrp->writeEntry("routes", _routes);
		_cfgGrp->writeEntry("vpn_data", _data);
		//_cfgGrp->sync();
	}
	if (_deleted)
	{
		_cfgGrp->deleteGroup();
		//_cfgGrp->sync();
	}
	return true;
}

void VPNConnection::remove()
{
	_deleted = true;
}

bool VPNConnection::hasPasswordsStored()
{
	// do we have some passwords in kwallet?
	KNetworkManagerStorage* storage = KNetworkManagerStorage::getInstance();
	_storePasswordsPermanent = storage->hasCredentialsStored(_name);

	// check if we have the passwords already
	return _storePasswordsPermanent || _storePasswordsSession;
}

void
VPNConnection::slotCredentialsLoaded (QString id, QMap<QString, QString> map, bool canceled)
{
	// credentials got loaded
	if ( !map.isEmpty() )
	{
		_passwords = QStringList::split(" ", map["passwords"], TRUE);
		_storePasswordsPermanent = true;
	}
	emit passwordsRestored(canceled, _passwords, _storePasswordsPermanent, _storePasswordsSession);
}

void
VPNConnection::restorePasswords()
{
	if (_storePasswordsSession && !_passwords.empty())
	{
		// we have the passwords here -> reply
		emit passwordsRestored(false, _passwords, _storePasswordsPermanent, _storePasswordsSession);
	}
	else
	{
		// no passwords here -> ask the wallet and wait for answer
		CredentialsRequest* req = KNetworkManagerStorage::getInstance()->credentialsAsync(_name);
		connect(req, SIGNAL(credentialsLoaded(QString, QMap<QString, QString>, bool)), this, SLOT(slotCredentialsLoaded (QString, QMap<QString, QString>, bool)));
		req->loadCredentials();
	}
}

VPNConnection::VPNConnection (const VPNConnection & vpnConnection) : QObject( vpnConnection.parent(), vpnConnection.name())
{
	_name = vpnConnection.getName ();
	_user = vpnConnection.getUser();
	_routes = vpnConnection.getRoutes ();
	_data = vpnConnection.getData ();
	_passwords = vpnConnection.getPasswords ();
	_activationStage = vpnConnection.getActivationStage ();
	_authHelper = vpnConnection.getAuthHelper ();
}

void
VPNConnection::receiveAuthenticationData (KProcess* /* authHelper */, char* buffer, int len)
{
	QStringList passwords = QStringList::split (QString ("\n"), QString::fromLatin1 (buffer, len), false);
	this->addPasswords (passwords);
}

void
VPNConnection::authHelperExited (KProcess* authHelper)
{
	if (authHelper->exitStatus() != 0)
		this->setActivationStage(NM_VPN_ACT_STAGE_CANCELED);
	else
		VPNDBus::activateVPNConnection (this);
}

void
VPNConnection::authHelperExited (bool cancel, QStringList& passwords)
{
	this->authHelperExited(cancel, passwords, false, false);
}

void
VPNConnection::authHelperExited (bool cancel, QStringList& passwords, bool storePasswordsPermanent, bool storePasswordsSession)
{
	if (cancel)
		// lets fake a CANCELD Message, so the Statuswindow gets closed
		this->setActivationStage(NM_VPN_ACT_STAGE_CANCELED);
	else
	{
		this->addPasswords(passwords, storePasswordsPermanent, storePasswordsSession);
		VPNDBus::activateVPNConnection ( this );
	}
}

/*
 *  class VPNService
 *
 */

VPNService::VPNService(const QString& serviceName, const QString& service, VPN* parent, const char* name)
	: QObject(parent, name)
{
	_name = serviceName;
	_service = service;
	_vpn = parent;
	_vpnPlugin = NULL;
	_useInternalAuthentication = false;

	// query if a plugin for this vpn service is available
	PluginManager* plugMan = _vpn->getCtx()->getPluginManager();
	if (plugMan)
	{
		QStringList list = plugMan->getPluginList("KNetworkManager/VPNPlugin", "X-NetworkManager-Services", serviceName);
		if (list.size() > 0)
		{
			// get the first VPN Plugin	handling our VPNService
			VPNPlugin* vpnPlugin = dynamic_cast<VPNPlugin*>( plugMan->getPlugin(list.first()) );
			if (vpnPlugin)
			{
				kdDebug() << k_funcinfo << i18n("Using VPN plugin '%1' for service '%2'").arg(list.first()).arg(serviceName) << endl;
				_vpnPlugin = vpnPlugin;
				_useInternalAuthentication = true;
			}
		}
	}

	// do we have a authentication dialog for this service?
	if (!_useInternalAuthentication)
	{
		// we do not have an own auth dialog -> try to get the gnome dialog
		_extAuthDialog = readAuthHelper();
		kdDebug() << i18n("VPN: service %1 uses external authentication dialog (%2)").arg(serviceName).arg(_extAuthDialog) << endl;
	}
}

VPNService::~VPNService()
{

}

QString VPNService::getIcon()
{
	if (!_vpnPlugin.isNull())
	{
		PluginManager* plugMan = _vpn->getCtx()->getPluginManager();
		if (plugMan)
		{
			const KPluginInfo* info = plugMan->getPluginInfo(_vpnPlugin);
			if (info)
			{
				QString icon = info->icon();
				if (!icon.isEmpty())
					return icon;
			}
		}
	}
	return "encrypted";
}

VPNPlugin* VPNService::getVPNPlugin()
{
	return _vpnPlugin;
}

QString VPNService::getService() const
{
	return _service;
}

QString VPNService::getDisplayName() const
{
	const KPluginInfo* info = NULL;
	PluginManager* plugMan = _vpn->getCtx()->getPluginManager();
	if (_vpnPlugin && plugMan)
		if ( (info = plugMan->getPluginInfo(_vpnPlugin)) )
			if (!info->name().isEmpty())
				return info->name();
	return _name;
}

QString VPNService::getName() const
{
	return _name;
}

bool VPNService::hasInternalAuthentication()
{
	return _useInternalAuthentication;
}

QString VPNService::getAuthHelper()
{
	return _extAuthDialog;
}

QString VPNService::readAuthHelper()
{
	// get the auth helper
	QDir serviceDir(SERVICE_DIR, QString::null, QDir::Name|QDir::IgnoreCase, QDir::Files);
	QStringList services = serviceDir.entryList ().grep (".name", true);
	QString retVal = QString::null;

	for (QStringList::Iterator i = services.begin (); i != services.end (); ++i) {
		QString service = SERVICE_DIR + QString ("/") + *i;
		KConfig kconfig (service, true, true, "config");
		kconfig.setGroup ("VPN Connection");
		if (kconfig.readEntry ("service", QString::null) == _service) {
			kconfig.setGroup ("GNOME");
			QString authDialog = kconfig.readEntry ("auth-dialog", QString::null);
			if ( !authDialog.isEmpty() ) {
				retVal = authDialog;
				break;
			} else {
				printf ("Warning: No authentication helper for service \"%s\" found.\n", service.ascii ());
			}
		}
	}
	return retVal;
}

/*
 *  class VPN
 *
 */

bool
VPN::hasGnomeVPNConfig()
{
	return !_gnomeVPNApplet.isEmpty();
}

void
VPN::startGnomeVPNConfig()
{
	if (!_gnomeVPNApplet.isEmpty())
	{
		KProcess* configureVPNHelper = new KProcess ();

		*configureVPNHelper << _gnomeVPNApplet;

		// trigger the vpn module to reread the available connections 
		connect (configureVPNHelper, SIGNAL (processExited      (KProcess*)),
		         this,               SLOT   (updateVPNConnections(KProcess*)));

		configureVPNHelper->start (KProcess::NotifyOnExit);
	}
}

void
VPN::vpnActivationStateChanged()
{
	// look if at least one VPN Connection is active
	for (VPNList::ConstIterator it = _vpnList->begin(); it != _vpnList->end(); ++it)
	{
		if ((*it)->getActivationStage() == NM_VPN_ACT_STAGE_ACTIVATED)
		{
			emit vpnConnectionStateChanged(true);
			return;
		}
	}

	emit vpnConnectionStateChanged(false);
}

void
VPN::activateVPNConnection (VPNConnection* vpnConnection)
{
	if (!vpnConnection) return;

	if (vpnConnection->getVPNService()->hasInternalAuthentication())
	{
		// lets ask the plugin whether we have to show the widget or not
		bool waitForPwds = true;
		VPNPlugin* plugin = vpnConnection->getVPNService()->getVPNPlugin();
		if (plugin)
		{	
			VPNAuthenticationWidget* auth = plugin->CreateAuthenticationWidget();
			if (auth)
			{
				auth->setVPNData(vpnConnection->getRoutes(), vpnConnection->getData());
				waitForPwds = auth->needsUserInteraction();
				if (!waitForPwds)
				{
					QStringList helper = auth->getPasswords();
					vpnConnection->authHelperExited(false, helper, false, false);
				}
				delete auth;
			}
		}

		if (waitForPwds)
		{
			// does the connection has its passwords stored (in session or permanent)
			if (vpnConnection->hasPasswordsStored())
			{
        // disconnect every slot from the signal to avoid multiple calls of authHelperExited
				disconnect(vpnConnection, SIGNAL(passwordsRestored(bool, QStringList&, bool, bool)), 0, 0);
        connect(vpnConnection, SIGNAL(passwordsRestored(bool, QStringList&, bool, bool)), vpnConnection, SLOT(authHelperExited(bool, QStringList&, bool, bool)));
				vpnConnection->restorePasswords();
			}
			else
			{
				// ok, we have to show the auth dialog so the user can type in his pwds	
				VPNAuthenticationDialog* dlg = new VPNAuthenticationDialog(vpnConnection);
				connect(dlg, SIGNAL(done(bool, QStringList&, bool, bool)), vpnConnection, SLOT(authHelperExited(bool, QStringList&, bool, bool)));
				dlg->show();
			}
		}
	}
	else
	{
		// no internal authentication dialog -> use the external one
		printf("External AuthHelper: %s\n", vpnConnection->getAuthHelper().ascii());
		KProcess*             authHelper = new KProcess ();

		*authHelper << vpnConnection->getAuthHelper ()           \
	              << "-n" << vpnConnection->getName ()         \
	              << "-s" << vpnConnection->getService ()  \
	              << "-r";

		connect (authHelper, SIGNAL (receivedStdout             (KProcess*, char*, int)),
			 vpnConnection,    SLOT   (receiveAuthenticationData (KProcess*, char*, int)));

		connect (authHelper, SIGNAL (processExited    (KProcess*)),
			 vpnConnection,    SLOT   (authHelperExited (KProcess*)));

		authHelper->start (KProcess::NotifyOnExit, KProcess::Stdout);
	}
}

void
VPN::updateVPNConnections ()
{
	// notify NM about the changed connections
	for (VPNList::iterator i = _vpnList->begin (); i != _vpnList->end (); ++i) {
		VPNDBus::updateVPNConnection (*i);
	}
}

void
VPN::updateVPNConnections (KProcess* /* p */)
{
	for (VPNList::iterator i = _vpnList->begin (); i != _vpnList->end (); ++i) {
		delete (*i);
	}
	
	_vpnList->clear ();
	getConnections  ();

	this->updateVPNConnections();
}

void
VPN::disconnectVPNConnection (void)
{
	VPNDBus::disconnectVPNConnection ();
}

bool
VPN::isAvailable (void)
{
	return _available;
}

QString
VPN::getAuthHelper(const QString & serviceName)
{
	// get the auth helper
	QDir* serviceDir = new QDir (SERVICE_DIR, QString::null, QDir::Name|QDir::IgnoreCase, QDir::Files);
	QStringList services = serviceDir->entryList ().grep (".name", true);

	for (QStringList::Iterator i = services.begin (); i != services.end (); ++i) {
		QString service = SERVICE_DIR + QString ("/") + *i;
		KConfig* kconfig = new KConfig (service, true, true, "config");
		kconfig->setGroup ("VPN Connection");
		if (kconfig->readEntry ("service", QString::null) == serviceName) {
			kconfig->setGroup ("GNOME");
			QString authDialog = kconfig->readEntry ("auth-dialog", QString::null);
			if ( !authDialog.isEmpty() ) {
				return authDialog;
			} else {
				printf ("Warning: No authentication helper for service \"%s\" found.\n", service.ascii ());
			}
		}
		delete kconfig;
	}
	return QString();
}

void
VPN::addConnection (const QString & cfgGrp)
{
	VPNConnection* vpnConnection = new VPNConnection ( cfgGrp, this, "vpnconnection");

	if (!vpnConnection->getName () || !vpnConnection->getServiceName ())
		return;

	if (vpnConnection->isValid ())
	{
		_vpnList->append (vpnConnection);
		connect(vpnConnection, SIGNAL(activationStateChanged()), this, SLOT(vpnActivationStateChanged()));
	}
	else
		delete vpnConnection;
}

bool
VPN::appendVPNConnection(VPNConnection* vpnConnection)
{
	if (vpnConnection->isValid())
	{
		_vpnList->append(vpnConnection);
		connect(vpnConnection, SIGNAL(activationStateChanged()), this, SLOT(vpnActivationStateChanged()));
		return true;
	}
	return false;
}

void
VPN::getGConfConnection (const QString & connection)
{
	VPNConnection* vpnConnection = new VPNConnection ( this, "vpnconnection" );
	QString        name;
	QString        serviceName;
	QStringList    routes;
	QStringList    data;

	QDomDocument doc ("vpnconnection");
	QFile file (_confPath + "/" + connection +  "/%gconf.xml");
	if (file.open (IO_ReadOnly) == false)
		return;
	if (doc.setContent (&file) == false) {
		file.close ();
		return;
	}
	file.close ();

	QDomElement docElem = doc.documentElement ();
	QDomNode n = docElem.firstChild ();
	while (n.isNull () == false) {
		QDomElement e = n.toElement ();
		if (e.isNull () == false) {
			if (e.hasAttribute ("name")) {
				QString nameValue = e.attribute ("name", QString::null);
				if (nameValue == "name") {
					vpnConnection->setName (e.text () );
				} else if (nameValue == "service_name") {
					vpnConnection->setServiceName (e.text ());
				} else if (nameValue == "routes") {
					QDomNode m = n.firstChild ();
					while (m.isNull () == false) {
						QDomElement f = m.toElement ();
						if (f.isNull () == false)
							vpnConnection->addRoute (f.text ());
						m = m.nextSibling ();
					}
				} else if (nameValue == "vpn_data") {
					QDomNode m = n.firstChild ();
					while (m.isNull () == false) {
						QDomElement f = m.toElement ();
						if (f.isNull () == false)
							vpnConnection->addData (f.text ());
						m = m.nextSibling ();
					}
				}
			}
		}
		n = n.nextSibling ();
	}

	if (!vpnConnection->getName () || !vpnConnection->getServiceName () || vpnConnection->getService()==NULL)
	{
		delete vpnConnection;
		return;
	}

	// we want this conection only if we have none with the same name
	if (this->getVPNConnection(vpnConnection->getName()) != NULL)
	{
		delete vpnConnection;
		return;
	}

	QDir* serviceDir = new QDir (SERVICE_DIR, QString::null, QDir::Name|QDir::IgnoreCase, QDir::Files);
	QStringList services = serviceDir->entryList ().grep (".name", true);

	for (QStringList::Iterator i = services.begin (); i != services.end (); ++i) {
		QString service = SERVICE_DIR + QString ("/") + *i;
		KConfig* kconfig = new KConfig (service, true, true, "config");
		kconfig->setGroup ("VPN Connection");
		if (kconfig->readEntry ("service", QString::null) == vpnConnection->getServiceName ()) {
			kconfig->setGroup ("GNOME");
			QString helper = kconfig->readEntry ("auth-dialog", QString::null);
			if ( !helper.isEmpty() ) {
				vpnConnection->setAuthHelper (helper);
			} else {
				printf ("Warning: No authentication helper for service \"%s\" found.\n", vpnConnection->getServiceName ().ascii ());
			}
		}
		delete kconfig;
	}

	if (vpnConnection->isValid ())
	{
		_vpnList->append (vpnConnection);
		connect(vpnConnection, SIGNAL(activationStateChanged()), this, SLOT(vpnActivationStateChanged()));
	}
	else
		delete vpnConnection;
}

QStringList
VPN::listGConfConnections()
{
	QStringList connections;
	QDir confDir(_confPath);
	confDir.setFilter (QDir::Dirs);

	QStringList entryList = confDir.entryList ();
	for (QStringList::Iterator it = entryList.begin (); it != entryList.end (); ++it) {
		if ((*it) != "." && (*it) != "..")
			if (getVPNConnection(*it)	== NULL)
				connections.append (*it);
	}
	return connections;
}

VPNConnection*
VPN::newVPNConnection ()
{
	KNetworkManagerStorage* storage = KNetworkManagerStorage::getInstance();
	QString group = storage->vpnConnectionNewGroup();
	return new VPNConnection(group, this, "vpnconnection");
}

bool
VPN::importVPNConnection(const QString& conn_name)
{
	VPNConnection* conn = getVPNConnection(conn_name);
	if (conn)
	{
		KNetworkManagerStorage* storage = KNetworkManagerStorage::getInstance();
		QString group = storage->vpnConnectionNewGroup();
		conn->setConfigGroup(group);
		conn->save();
		return true;
	}
	return false;
}

void
VPN::getConnections (void)
{
	KNetworkManagerStorage* storage = KNetworkManagerStorage::getInstance();
	QStringList connections = storage->vpnConnectionGroups();
	for(QStringList::Iterator it = connections.begin(); it != connections.end(); ++it)
	{
		addConnection (*it);
	}

	// import all gconf connections for which we do not have an plugin
	QDir confDir(_confPath);
	confDir.setFilter (QDir::Dirs);

	QStringList entryList = confDir.entryList ();
	for (QStringList::Iterator it = entryList.begin (); it != entryList.end (); ++it) {
		if ((*it) != "." && (*it) != "..")
			getGConfConnection(*it);
	}

	emit vpnConnectionsUpdated();
}

void
VPN::receiveKeyringData (KProcess* /* keyringDaemon */, char* buffer, int len)
{
	QStringList env = QStringList::split (QString ("\n"), QString::fromLatin1 (buffer, len), false);

	for (QStringList::Iterator it = env.begin (); it != env.end (); ++it) {
		QString* item = &(*it);
		if (item->startsWith ("GNOME_KEYRING_SOCKET")) {
			setenv ("GNOME_KEYRING_SOCKET", item->section ('=', 1, 1).ascii(), 1);
		}
	}
}


void
VPN::initKeyring ()
{
	QString keyringSocket = getenv ("GNOME_KEYRING_SOCKET");

	if (!keyringSocket) {
		KProcess* keyringDaemon = new KProcess ();
		*keyringDaemon << "gnome-keyring-daemon";

		connect (keyringDaemon, SIGNAL (receivedStdout     (KProcess*, char*, int)),
			 this,          SLOT   (receiveKeyringData (KProcess*, char*, int)));
		
		keyringDaemon->start (KProcess::NotifyOnExit, KProcess::Stdout);
	}
}

bool
VPN::getServices ()
{
	bool status = false;
	
	QDir serviceDir(SERVICE_DIR, QString::null, QDir::Name|QDir::IgnoreCase, QDir::Files);
	QStringList services = serviceDir.entryList ().grep (".name", true);

	if (services.count () > 0)
	{	
		status = true;
		// read in all available Services
		_vpnServiceList = new VPNServiceList();
		for (QStringList::Iterator i = services.begin (); i != services.end (); ++i) {
			QString service = SERVICE_DIR + QString ("/") + *i;
			KConfig* kconfig = new KConfig (service, true, true, "config");
			kconfig->setGroup ("VPN Connection");
			// create new VPNService Object
			_vpnServiceList->push_back(new VPNService(kconfig->readEntry ("name", QString::null), kconfig->readEntry ("service", QString::null), this));
			delete kconfig;
		}
	}

	return status;
}

VPNService*
VPN::getVPNService(const QString & name)
{
	for (VPNServiceList::Iterator i = _vpnServiceList->begin (); i != _vpnServiceList->end (); ++i) {
		if ((*i)->getName() == name || (*i)->getService() == name)
			return (*i);
	}
	return NULL;
}

QStringList
VPN::getVPNServices ()
{
	QStringList retval;
	for (VPNServiceList::Iterator i = _vpnServiceList->begin (); i != _vpnServiceList->end (); ++i) {
		retval.push_back((*i)->getName());
	}
	return retval;
}

void
VPN::deleteVPNConnection (const QString & name)
{
	VPNConnection* vpnConnection = NULL;
	for (VPNList::iterator i = _vpnList->begin (); i != _vpnList->end (); ++i) {
		if ((*i)->getName () == name) {
			// remove the configuration
			vpnConnection = *i;
			vpnConnection->remove();
			// remove our reference
			_vpnList->erase(i);
			delete vpnConnection;
			vpnConnection = NULL;
			// no more to do
			break;
		}
	}
}

VPNConnection*
VPN::getVPNConnection (const QString & name)
{
	VPNConnection* vpnConnection = NULL;
	for (VPNList::iterator i = _vpnList->begin (); i != _vpnList->end (); ++i) {
		if ((*i)->getName () == name) {
			vpnConnection = *i;
			goto out;
		}
	}
out:
	return vpnConnection;
}

bool
VPN::isActive (void)
{
	bool status = false;
	
	for (VPNList::iterator i = _vpnList->begin (); i != _vpnList->end (); ++i) {
		if ((*i)->getActivationStage () == NM_VPN_ACT_STAGE_ACTIVATED) {
			status = true;
			goto out;
		}
	}

out:
	return status;
}

VPNList*
VPN::getVPNList (void)
{
	return _vpnList;
}

void
VPN::push (KNetworkManager* ctx)
{
	_ctx = ctx;
	VPNDBus::push (ctx);
}

KNetworkManager* VPN::getCtx()
{
	return _ctx;
}

VPN::VPN ( KNetworkManager* parent, const char * name )
	: QObject( parent, name ), _vpnServiceList(NULL), _vpnList(NULL)
{
	_ctx = parent;
	VPNDBus::push(_ctx);
	_confPath  = QDir::homeDirPath() + GCONF_PRE + VPN_PATH;
	_available = getServices();

	if (_available == true) {
		initKeyring ();
		_vpnList = new VPNList ();
		getConnections ();
	}
	// find the gnome vpn configuration applet
	_gnomeVPNApplet = KStandardDirs::findExe("nm-vpn-properties");
}

VPN::~VPN ()
{
	// delete connectionslist
	if (_vpnList) {
		for (VPNList::iterator i = _vpnList->begin (); i != _vpnList->end (); ++i) {
			delete (*i);
		}
		_vpnList->clear ();
		delete _vpnList;
		_vpnList = NULL;
	}

	if (_vpnServiceList) {
		for (VPNServiceList::iterator i = _vpnServiceList->begin (); i != _vpnServiceList->end (); ++i) {
			delete (*i);
		}
		_vpnServiceList->clear ();
		delete _vpnServiceList;
		_vpnServiceList = NULL;
	}
}

#include "knetworkmanager-vpn.moc"
