/** -*- C++ -*-
    @file cache/package.h
    @author Peter Rockai <me@mornfall.net>
*/

#include <sigc++/sigc++.h>
#include <sigc++/object_slot.h>
#include <wibble/mixin.h>
#include <wibble/fallback.h>
#include <wibble/range.h>
#include <ept/forward.h>
#include <ept/cache/entity.h>
#include <ept/cache/tag.h>

#ifndef EPT_CACHE_PACKAGE
#define EPT_CACHE_PACKAGE

namespace ept {
namespace t {
namespace cache {

template< typename C, typename Package >
struct PackageEditor
{
    PackageEditor( Package p, typename C::Aggregator &pkgs )
        : m_package( p ), m_aggregator( pkgs )
    {}

    PackageEditor &self() { return *this; }
    Package &package() {
        return m_package;
    }

    typename C::Aggregator &aggregator() { return m_aggregator; }

    void markInstall() {
        package().checkValid();
        aggregator().state().markInstall( package() );
    }

    void markRemove() {
        package().checkValid();
        aggregator().state().markRemove( package() );
    }

    void markKeep() {
        package().checkValid();
        aggregator().state().markKeep( package() );
    }

    void markPurge() {
        package().checkValid();
        aggregator().state().markPurge( package() );
    }

    void markReInstall() {
        package().checkValid();
        aggregator().state().markReInstall( package() );
    }
protected:
    Package m_package;
    typename C::Aggregator &m_aggregator;
};

template< typename C >
struct Package : InstallableMixin< Package< C > >, EntityMixin< Package< C > > {
    typedef typename C::Version Version;
    typedef typename C::Relation Relation;
    typedef typename InstallableMixin< Package >::State State;
    typedef typename C::Tag Tag;
    typedef PackageEditor< C, Package > Editor;

    const typename C::PackagePointer &pointer() const { return m_pointer; }
    typename C::PackagePointer &pointer() { return m_pointer; }
    const typename C::Aggregator &aggregator() const { return pointer().aggregator(); }

    Package( typename C::PackagePointer p ) : m_pointer( p ) {}
    Package() {}

    Editor editor( typename C::Aggregator &p ) {
        return Editor( *this, p );
    }

    Version anyVersion() const {
        Version v = this->candidateVersion();
        if ( !v.valid() && this->hasVersion() )
            v = this->versions().head();
        return v;
    }

    bool valid() const { return pointer().valid(); }
    long id() const { return pointer().id(); }
    long ondiskId() const { return aggregator().index().ondiskId( *this ); }

    typedef wibble::Fallback< std::string > FallbackString;

    std::string name( FallbackString d = wibble::Error() ) const {
        return this->valid() ? this->aggregator().index().packageName( *this ) : d.get();
    }

    std::string section( FallbackString d = wibble::Error() ) const {
        return this->valid() ? aggregator().index().packageSection( *this ) : d.get();
    }

    std::string shortDescription( FallbackString d = wibble::Error() ) const {
        return this->hasVersion() ? this->anyVersion().shortDescription( d ) : d.get();
    }

    std::string longDescription( FallbackString d = wibble::Error() ) const {
        return this->hasVersion() ? this->anyVersion().longDescription( d ) : d.get();
    }

    std::string md5sum( FallbackString d = wibble::Error() ) const {
        return this->hasVersion() ? this->anyVersion().md5sum( d ) : d.get();
    }

    std::string fileName( FallbackString d = wibble::Error() ) const {
        return this->hasVersion() ? this->anyVersion().fileName( d ) : d.get();
    }

    std::string source( FallbackString d = wibble::Error() ) const {
        return this->hasVersion() ? this->anyVersion().source( d ) : d.get();
    }

    std::string maintainer( FallbackString d = wibble::Error() ) const {
        return this->hasVersion() ? this->anyVersion().maintainer( d ) : d.get();
    }

    std::string architecture( FallbackString d = wibble::Error() ) const {
        return this->hasVersion() ? this->anyVersion().architecture( d ) : d.get();
    }

    typename Tag::Set tags( wibble::Fallback< typename Tag::Set > d = wibble::Error() ) const {
        return this->valid() ? aggregator().tagMap().getTagsOfItem( *this ) : d.get();
    }

    State state( wibble::Fallback< State > d = wibble::Error() ) const {
        return this->valid() ? aggregator().state().packageState( *this ) : d.get();
    }

    wibble::Range< Version > versions() const {
        return this->aggregator().index().versions( *this );
    }

#ifndef SWIG_I
    wibble::Range< Relation > depends() const {
	return this->anyVersion().relations();
    }
#endif

    Version versionByString( const std::string& ver ) const {
        wibble::Range< Version > r = this->versions();
        while ( !r.empty() && r.head().versionString() != ver )
            r = r.tail();
        if ( r.head().versionString() == ver )
            return r.head();
        throw wibble::exception::Consistency(
            "No such version " + ver + " for " + name( std::string( "unknown" ) ) );
    }

    bool hasVersion() const { if ( !valid() ) return false; return !versions().empty(); }

    Version candidateVersion() const
    {
        if ( this->valid() )
            return this->aggregator().state().candidateVersion( *this );
        return Version();
    }

    Version installedVersion() const
    {
        if ( this->valid() )
            return this->aggregator().index().currentVersion( *this );
        return Version();
    }

protected:
    typename C::PackagePointer m_pointer;
};

}
}
}

#endif
