/* GNOME DB library
 * Copyright (C) 1999-2002 The GNOME Foundation.
 *
 * AUTHORS:
 * 	Rodrigo Moya <rodrigo@gnome-db.org>
 *
 * This Library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This Library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this Library; see the file COPYING.LIB.  If not,
 * write to the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <string.h>
#include <libgnomedb/gnome-db-model.h>
#include <libgnomedb/gnome-db-util.h>
#include <gtk/gtkcellrenderertext.h>
#include <gtk/gtkcellrenderertoggle.h>
#include <gtk/gtktreestore.h>
#include <gtk/gtktreeview.h>
#include <gtk/gtktreeviewcolumn.h>
#include "libgnomedb-private.h"

#define DATA_MODEL_INFO_TYPE data_model_info_get_type ()

/*
 * Private functions
 */

static void
data_model_info_free (DataModelInfo *info)
{
	g_object_unref (G_OBJECT (info->model));
	g_free (info);
}

static DataModelInfo *
data_model_info_copy (DataModelInfo *src)
{
	DataModelInfo *info;

	g_return_val_if_fail (src != NULL, NULL);

	info = g_new0 (DataModelInfo, 1);
	info->row = src->row;
	info->model = src->model;
	if (info->model)
		g_object_ref (G_OBJECT (info->model));

	return info;
}

static GType
data_model_info_get_type (void)
{
	static GType type = 0;

	if (!type) {
		type = g_boxed_type_register_static (
			"GNOME_DB_DataModelInfo",
			(GBoxedCopyFunc) data_model_info_copy,
			(GBoxedFreeFunc) data_model_info_free);
	}
	return type;
}

static void
column_clicked_cb (GtkTreeViewColumn *column, gpointer user_data)
{
	//gtk_tree_view_set_search_column (GTK_TREE_VIEW (user_data),
	//				 gtk_tree_view_column_get_sort_column_id (column));
}

static gint
sort_values (GtkTreeModel *model, GtkTreeIter *itera, GtkTreeIter *iterb, gpointer user_data)
{
	DataModelInfo *infoa, *infob;
	const GdaValue *valuea, *valueb;

	gtk_tree_model_get (model, itera, 0, &infoa, -1);
	gtk_tree_model_get (model, iterb, 0, &infob, -1);
	if (!infoa || !infob)
		return 0;

	valuea = gda_data_model_get_value_at (infoa->model,
					      GPOINTER_TO_INT (user_data),
					      infoa->row);
	valueb = gda_data_model_get_value_at (infob->model,
					      GPOINTER_TO_INT (user_data),
					      infob->row);
	
	return gda_value_compare (valuea, valueb);
}

/*
 * Callbacks
 */

static void
default_value_set_func (GtkTreeViewColumn *tree_column,
			GtkCellRenderer *cell,
			GtkTreeModel *model,
			GtkTreeIter *iter,
			gpointer user_data)
{
	DataModelInfo *info = NULL;
	GdaValue *value;
	gchar *txt;

	gtk_tree_model_get (model, iter, 0, &info, -1);
	if (!info)
		return;

	value = (const GdaValue *) gda_data_model_get_value_at (info->model,
					     GPOINTER_TO_INT (user_data),
					     info->row);
	if (!value)
		return;

	txt = gda_value_stringify (value);
	if (gda_value_isa (value, GDA_VALUE_TYPE_BIGINT)
	    || gda_value_isa (value, GDA_VALUE_TYPE_DOUBLE)
	    || gda_value_isa (value, GDA_VALUE_TYPE_INTEGER)
	    || gda_value_isa (value, GDA_VALUE_TYPE_NUMERIC)
	    || gda_value_isa (value, GDA_VALUE_TYPE_SINGLE)
	    || gda_value_isa (value, GDA_VALUE_TYPE_SMALLINT)
	    || gda_value_isa (value, GDA_VALUE_TYPE_TINYINT)) {
		/* FIXME: get color for numbers from configuration */
		g_object_set (G_OBJECT (cell), "text", txt,
			      "foreground", "blue",
			      "xalign", 1.0,
			      "yalign", 0.0, NULL);
	}
	else
		g_object_set (G_OBJECT (cell), "text", txt, "yalign", 0.0, NULL);

	g_free (txt);
}

static void
toggle_value_set_func (GtkTreeViewColumn *tree_column,
		       GtkCellRenderer *cell,
		       GtkTreeModel *model,
		       GtkTreeIter *iter,
		       gpointer user_data)
{
	DataModelInfo *info = NULL;
	const GdaValue *value;

	gtk_tree_model_get (model, iter, 0, &info, -1);
	if (!info)
		return;

	value = gda_data_model_get_value_at (info->model,
					     GPOINTER_TO_INT (user_data),
					     info->row);
	if (!value)
		return;

	gtk_cell_renderer_toggle_set_active (GTK_CELL_RENDERER_TOGGLE (cell),
					     gda_value_get_boolean ((GdaValue *) value));
	g_object_set (G_OBJECT (cell), "xalign", 0.0, "yalign", 0.0, NULL);
}

static gchar *
double_underscores (const gchar *str)
{
	gchar **arr;
	gchar *ret;

	arr = g_strsplit (str, "_", 0);
	ret = g_strjoinv ("__", arr);
	g_strfreev (arr);

	return ret;
}

static void
add_tree_model_row (GtkTreeStore *tree_model, GdaDataModel *model,
		    GtkTreeIter *parent, gint row)
{
	DataModelInfo info;
	GtkTreeIter iter;
	const GdaValue *value;

	info.row = row;
	info.model = model;
	gtk_tree_store_append (tree_model, &iter, parent);
	gtk_tree_store_set (tree_model, &iter, 0, &info, -1);

	/* if column 0 of this row contains a GdaValueList,
	   then, add a child row */
	value = gda_data_model_get_value_at (model, 0, row);
	if (value && value->type == GDA_VALUE_TYPE_LIST) {
	}
}

static void
model_row_inserted_cb (GdaDataModel *model, gint row, gpointer user_data)
{
	GtkTreeView *tree_view = user_data;

	add_tree_model_row (gtk_tree_view_get_model (tree_view), model, NULL, row);
}

static void
model_row_updated_cb (GdaDataModel *model, gint row, gpointer user_data)
{
	GtkTreeIter iter;
	GtkTreePath *path;
	GtkTreeView *tree_view = user_data;

	gtk_tree_model_iter_nth_child (gtk_tree_view_get_model (tree_view), &iter, NULL, row);
	path = gtk_tree_path_new ();
	gtk_tree_path_append_index (path, row);
	gtk_tree_model_row_changed (gtk_tree_view_get_model (tree_view), path, &iter);

	gtk_tree_path_free (path);
}

static void
model_row_removed_cb (GdaDataModel *model, gint row, gpointer user_data)
{
	GtkTreePath *path;
	GtkTreeView *tree_view = user_data;

	path = gtk_tree_path_new ();
	gtk_tree_path_append_index (path, row);
	gtk_tree_model_row_deleted (gtk_tree_view_get_model (tree_view), path);

	gtk_tree_path_free (path);
}

static gboolean
tree_view_equal_func (GtkTreeModel *model,
		      gint column,
		      const gchar *key,
		      GtkTreeIter *iter,
		      gpointer user_data)
{
	DataModelInfo *info = NULL;
	const GdaValue *value;
	gint key_len;
	gboolean retval = TRUE;
	gchar *normalized_string, *normalized_key, *tmp, *case_ns, *case_nk;

	gtk_tree_model_get (model, iter, 0, &info, -1);
	if (!info)
		return TRUE;

	value = gda_data_model_get_value_at (info->model, column, info->row);
	if (!value)
		return TRUE;

	/* normalize strings for UTF8 */
	tmp = gda_value_stringify (value);
	normalized_string = g_utf8_normalize (tmp, -1, G_NORMALIZE_ALL);
	normalized_key = g_utf8_normalize (key, -1, G_NORMALIZE_ALL);
	case_ns = g_utf8_casefold (normalized_string, -1);
	case_nk = g_utf8_casefold (normalized_key, -1);

	g_free (tmp);

	key_len = strlen (case_nk);
	if (!strncmp (case_nk, case_ns, key_len))
		retval = FALSE;

	g_free (normalized_key);
	g_free (normalized_string);
	g_free (case_nk);
	g_free (case_ns);
                                                                                           
	return retval;
}

/**
 * gnome_db_model_to_gtk_tree_view
 * @model: a #GdaDataModel object.
 *
 * Convert a #GdaDataModel into a GtkTreeView widget, filled in
 * with the corresponding GtkTreeModel.
 *
 * Returns: a pointer to the newly created object.
 */
GtkTreeView *
gnome_db_model_to_gtk_tree_view (GdaDataModel *model)
{
	gint i;
	gint row_count;
	gint col_count;
	GtkTreeView *tree_view;
	GtkTreeStore *tree_model = NULL;

	tree_model = gtk_tree_store_new (1, DATA_MODEL_INFO_TYPE);
	row_count = gda_data_model_get_n_rows (model);
	for (i = 0; i < row_count; i++) {
		add_tree_model_row (tree_model, model, NULL, i);
	}

	/* create the tree view widget */
	tree_view = GTK_TREE_VIEW (gnome_db_new_tree_view_widget (
					   GTK_TREE_MODEL (tree_model)));
	g_object_unref (G_OBJECT (tree_model));

	col_count = gda_data_model_get_n_columns (model);
	if (col_count > 0) {
		//gtk_tree_view_set_search_equal_func (GTK_TREE_VIEW (tree_view),
		//				     (GtkTreeViewSearchEqualFunc) tree_view_equal_func,
		//				     tree_view, NULL);
		//gtk_tree_view_set_search_column (GTK_TREE_VIEW (tree_view), 0);
	}

	for (i = 0; i < col_count; i++) {
		GtkTreeViewColumn *column;
		GtkCellRenderer *renderer;
		GdaFieldAttributes *attrs;
		gchar *title;
		gboolean alloc_title;

		alloc_title = FALSE;
		title = (gchar *) gda_data_model_get_column_title (model, i);
		if (title && strchr (title, '_')) {
			title = double_underscores (title);
			alloc_title = TRUE;
		}
		attrs = gda_data_model_describe_column (model, i);
		if (attrs && attrs->gda_type == GDA_VALUE_TYPE_BOOLEAN) {
						renderer = gtk_cell_renderer_toggle_new ();
			gtk_tree_view_insert_column_with_data_func (
				GTK_TREE_VIEW (tree_view),
				i,
				title,
				renderer,
				toggle_value_set_func,
				GINT_TO_POINTER (i),
				NULL);
		}
		else {
			renderer = gtk_cell_renderer_text_new ();
			gtk_tree_view_insert_column_with_data_func (
				GTK_TREE_VIEW (tree_view),
				i,
				title,
				renderer,
				default_value_set_func,
				GINT_TO_POINTER (i),
				NULL);
		}

		if (alloc_title)
			g_free (title);

		column = gtk_tree_view_get_column (GTK_TREE_VIEW (tree_view), i);
		gtk_tree_view_column_set_sort_column_id (column, i);
		gtk_tree_view_column_set_resizable (column, TRUE);
		gtk_tree_view_column_set_clickable (column, TRUE);
		g_signal_connect (G_OBJECT (column), "clicked",
				  G_CALLBACK (column_clicked_cb), tree_view);

		gtk_tree_sortable_set_sort_func (GTK_TREE_SORTABLE (tree_model),
						 i, sort_values,
						 GINT_TO_POINTER (i),
						 NULL);
		if (attrs)
			gda_field_attributes_free (attrs);
	}

	gtk_tree_view_set_headers_clickable (GTK_TREE_VIEW (tree_view), TRUE);

	/* connect to data model's signals */
	g_signal_connect (G_OBJECT (model), "row_inserted", G_CALLBACK (model_row_inserted_cb), tree_view);
	g_signal_connect (G_OBJECT (model), "row_updated", G_CALLBACK (model_row_updated_cb), tree_view);
	g_signal_connect (G_OBJECT (model), "row_removed", G_CALLBACK (model_row_removed_cb), tree_view);

	return tree_view;
}
