/*
 * Copyright 2010, 2011 Canonical, Ltd.
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of either or both of the following licenses:
 *
 * 1) the GNU Lesser General Public License version 3, as published by the
 * Free Software Foundation; and/or
 * 2) the GNU Lesser General Public License version 2.1, as published by
 * the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranties of
 * MERCHANTABILITY, SATISFACTORY QUALITY or FITNESS FOR A PARTICULAR
 * PURPOSE.  See the applicable version of the GNU Lesser General Public
 * License for more details.
 *
 * You should have received a copy of both the GNU Lesser General Public
 * License version 3 and version 2.1 along with this program.  If not, see
 * <http://www.gnu.org/licenses/>
 *
 * Authors:
 *    Cody Russell <crussell@canonical.com>
 */

#include "gripgesturemanager.h"

#include <gdk/gdkx.h>
#include <geis/geis.h>
#include "gripinputdevice.h"

typedef struct _GripGestureRegistration GripGestureRegistration;
typedef struct _GripGestureBinding      GripGestureBinding;
typedef struct _GripRegistrationRequest GripRegistrationRequest;

typedef GPtrArray GripDevices;

struct Registrations {
  GripGestureRegistration *touchscreen;
  GripGestureRegistration *touchpad;
  GripGestureRegistration *independent;
};

struct _GripGestureManagerPrivate
{
  GHashTable  *registered_windows;
  GList       *requests;
  GripDevices *devices;
};

/*
 * GripGestureBinding:
 * @type:
 * @widget:
 * @touches:
 * @callback:
 * @data:
 * @destroy:
 *
 * The mapping between a GEIS subscription and a GTK widget.  Sort of.
 */
struct _GripGestureBinding
{
  GripGestureType      type;
  GtkWidget           *widget;
  guint                touches;
  GripGestureCallback  callback;
  gpointer             data;
  GDestroyNotify       destroy;
};

/*
 * GripGestureRegistration:
 * @window: the GTK window
 * @bindings: a list of #GripGestureBinding
 * @gesture_list: a list the names of GEIS gestures beging subscribed to 
 * @instance:
 * @iochannel:
 *
 * A collection of all gesture subscriptions for all widgets contained within a
 * top-level GTK window.
 */
struct _GripGestureRegistration
{
  GtkWindow         *window;
  GList             *bindings;
  GPtrArray         *gesture_list;
  GeisInstance       instance;
  GIOChannel        *iochannel;
};

struct _GripRegistrationRequest
{
  GripGestureManager *manager;
  GtkWidget          *widget;
  GripGestureType     gesture_type;
  GripDeviceType      device_type;
  gint                touch_points;
  GripGestureCallback callback;
  gpointer            user_data;
  GDestroyNotify      destroy;
};

static const gchar *geis_gesture_types[] = {
  GEIS_GESTURE_TYPE_DRAG1,
  GEIS_GESTURE_TYPE_DRAG2,
  GEIS_GESTURE_TYPE_DRAG3,
  GEIS_GESTURE_TYPE_DRAG4,
  GEIS_GESTURE_TYPE_DRAG5,
  GEIS_GESTURE_TYPE_PINCH1,
  GEIS_GESTURE_TYPE_PINCH2,
  GEIS_GESTURE_TYPE_PINCH3,
  GEIS_GESTURE_TYPE_PINCH4,
  GEIS_GESTURE_TYPE_PINCH5,
  GEIS_GESTURE_TYPE_ROTATE1,
  GEIS_GESTURE_TYPE_ROTATE2,
  GEIS_GESTURE_TYPE_ROTATE3,
  GEIS_GESTURE_TYPE_ROTATE4,
  GEIS_GESTURE_TYPE_ROTATE5,
  GEIS_GESTURE_TYPE_TAP1,
  GEIS_GESTURE_TYPE_TAP2,
  GEIS_GESTURE_TYPE_TAP3,
  GEIS_GESTURE_TYPE_TAP4,
  GEIS_GESTURE_TYPE_TAP5,
};

enum {
  DEVICE_AVAILABLE,
  DEVICE_UNAVAILABLE,
  SIGNAL_LAST
};

static guint signals[SIGNAL_LAST];

static void gesture_added (void              *cookie,
                           GeisGestureType    gesture_type,
                           GeisGestureId      gesture_id,
                           GeisSize           attr_count,
                           GeisGestureAttr   *attrs);

static void gesture_removed (void              *cookie,
                             GeisGestureType    gesture_type,
                             GeisGestureId      gesture_id,
                             GeisSize           attr_count,
                             GeisGestureAttr   *attrs);

static void gesture_start (void              *cookie,
                           GeisGestureType    gesture_type,
                           GeisGestureId      gesture_id,
                           GeisSize           attr_count,
                           GeisGestureAttr   *attrs);

static void gesture_update (void              *cookie,
                            GeisGestureType    gesture_type,
                            GeisGestureId      gesture_id,
                            GeisSize           attr_count,
                            GeisGestureAttr   *attrs);

static void gesture_finish (void              *cookie,
                            GeisGestureType    gesture_type,
                            GeisGestureId      gesture_id,
                            GeisSize           attr_count,
                            GeisGestureAttr   *attrs);

static void window_mapped_cb (GtkWidget       *widget,
                              GdkEvent        *event,
                              gpointer         user_data);

static GeisGestureFuncs gesture_funcs = {
  gesture_added,
  gesture_removed,
  gesture_start,
  gesture_update,
  gesture_finish
};


static void
device_added (void *cookie, GeisInputDeviceId id G_GNUC_UNUSED, void *attrs)
{
  GripGestureManager *gesture_manager = (GripGestureManager *) cookie;
  GripInputDevice *input_device = g_object_new (GRIP_TYPE_INPUT_DEVICE,
                                                "device-attrs", attrs,
                                                NULL);

  g_ptr_array_add(gesture_manager->priv->devices, input_device);
  g_signal_emit (gesture_manager, signals[DEVICE_AVAILABLE], 0, input_device);
}


static void
device_removed (void *cookie, GeisInputDeviceId id, void *attrs G_GNUC_UNUSED)
{
  guint i;
  GripGestureManager *gesture_manager = (GripGestureManager *) cookie;
  GripDevices *devices = gesture_manager->priv->devices;

  for (i = 0; i < devices->len; ++i)
  {
    GripInputDevice *input_device = g_ptr_array_index (devices, i);
    if (id == grip_input_device_get_id (input_device))
    {
      g_signal_emit (gesture_manager, signals[DEVICE_UNAVAILABLE], 0,
                     input_device);
      g_ptr_array_remove_index (devices, i);
      break;
    }
  }
}


static GeisInputFuncs input_funcs = {
  device_added,
  NULL,
  device_removed,
};


static void
grip_get_devices (GripGestureManager *gesture_manager)
{
  GeisXcbWinInfo xcb_win_info = {
    .display_name = NULL,
    .screenp = NULL,
    .window_id = 0,
  };
  GeisWinInfo win_info = {
    GEIS_XCB_FULL_WINDOW,
    &xcb_win_info,
  };
  GeisInstance instance;
  GeisStatus status;

  status = geis_init (&win_info, &instance);
  if (status != GEIS_STATUS_SUCCESS)
  {
    g_warning ("failed to determine device types\n");
    return;
  }

  status = geis_input_devices (instance, &input_funcs, gesture_manager);
  if (status != GEIS_STATUS_SUCCESS)
  {
    g_warning ("failed to determine device types\n");
    return;
  }

  status = geis_event_dispatch (instance);
  if (status != GEIS_STATUS_SUCCESS)
  {
    g_warning ("failed to determine device types\n");
    return;
  }

  geis_finish (instance);
}

G_DEFINE_TYPE (GripGestureManager, grip_gesture_manager, G_TYPE_OBJECT)

#define GRIP_GESTURE_MANAGER_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), GRIP_TYPE_GESTURE_MANAGER, GripGestureManagerPrivate))

static void
grip_gesture_manager_dispose (GObject *object G_GNUC_UNUSED)
{
}


static void
free_registration(GripGestureRegistration *reg)
{
  GList *tmp = NULL;

  if (!reg)
    return;

  geis_unsubscribe (reg->instance,
      (GeisGestureType*)reg->gesture_list->pdata);

  /* We don't need to free the values in the GPtrArray. */
  g_ptr_array_free (reg->gesture_list, TRUE);

  for (tmp = reg->bindings; tmp != NULL; tmp = tmp->next)
    {
      g_free (tmp->data);
    }

  g_list_free (reg->bindings);
}

/* Helper function needed to make things work with GTK. */

static void
free_registration_adapter (gpointer key G_GNUC_UNUSED,
                           gpointer value,
                           gpointer user_data G_GNUC_UNUSED)
{
  struct Registrations *regs = (struct Registrations *)value;
  free_registration(regs->touchscreen);
  free_registration(regs->touchpad);
  free_registration(regs->independent);
}

static void
grip_gesture_manager_finalize (GObject *object)
{
  GripGestureManagerPrivate *priv = GRIP_GESTURE_MANAGER_GET_PRIVATE (object);

  if (priv->registered_windows != NULL)
    {
      g_hash_table_foreach (priv->registered_windows,
			    free_registration_adapter,
			    NULL);

      g_hash_table_unref (priv->registered_windows);
      priv->registered_windows = NULL;

      g_ptr_array_foreach (priv->devices, (GFunc) g_object_unref, NULL);
      g_ptr_array_unref (priv->devices);
    }
}

static GObject *
grip_gesture_manager_constructor (GType                  type,
                                  guint                  n_params,
                                  GObjectConstructParam *params)
{
  static GObject *self = NULL;
  
  if (self == NULL)
  {
    self = G_OBJECT_CLASS (grip_gesture_manager_parent_class)->
                              constructor (type, n_params, params);
    g_object_add_weak_pointer (self, (gpointer) &self);
    grip_get_devices (GRIP_GESTURE_MANAGER (self));
  }

  return g_object_ref (self);
}

static void
grip_gesture_manager_class_init (GripGestureManagerClass *class)
{
  GObjectClass     *gobject_class;

  gobject_class = G_OBJECT_CLASS (class);

  grip_gesture_manager_parent_class = g_type_class_peek_parent (class);

  g_type_class_add_private (gobject_class, sizeof (GripGestureManagerPrivate));

  gobject_class->constructor = grip_gesture_manager_constructor;
  gobject_class->dispose     = grip_gesture_manager_dispose;
  gobject_class->finalize    = grip_gesture_manager_finalize;

  /**
   * GripGestureManager::device-available:
   * @gesture_manager: the #GripGestureManager sending the signal
   * @input_device: the new #GripInputDevice just added.
   *
   * Signals the availability of a new gesture-capable input device.
   */
  signals[DEVICE_AVAILABLE] = g_signal_new ("device-available",
                                            GRIP_TYPE_GESTURE_MANAGER,
                                            G_SIGNAL_RUN_LAST,
                                            0,
                                            NULL, NULL,
                                            g_cclosure_marshal_VOID__OBJECT,
                                            G_TYPE_NONE,
                                            1,
                                            GRIP_TYPE_INPUT_DEVICE);

  /**
   * GripGestureManager::device-unavailable:
   * @gesture_manager: the #GripGestureManager sending the signal
   * @input_device: the new #GripInputDevice just added.
   *
   * Signals the unavailability of a gesture-capable input device.
   */
  signals[DEVICE_UNAVAILABLE] = g_signal_new ("device-unavailable",
                                              GRIP_TYPE_GESTURE_MANAGER,
                                              G_SIGNAL_RUN_LAST,
                                              0,
                                              NULL, NULL,
                                              g_cclosure_marshal_VOID__OBJECT,
                                              G_TYPE_NONE,
                                              1,
                                              GRIP_TYPE_INPUT_DEVICE);

}


/*
 * device_id_to_input_device:
 * @gesture_manager: the #GripGestureManager
 * @device_id: the device ID
 *
 * Finds a known #GripInputDevice given a GEIS device identifier.
 *
 * Returns: a pointer to a #GripInputDevice or %NULL if not found.
 */
static GripInputDevice *
device_id_to_input_device (GripGestureManager *gesture_manager,
                           GeisInputDeviceId   device_id)
{
  guint j;
  GripInputDevice *input_device = NULL;

  for (j = 0; j < gesture_manager->priv->devices->len; ++j)
  {
    GripInputDevice *device;

    device = g_ptr_array_index (gesture_manager->priv->devices, j);
    if (grip_input_device_get_id (device) == device_id)
    {
      input_device = device;
      break;
    }
  }
  return input_device;
}

/*
 * registration_for_input_device:
 * @registrations: A collection of #GripGestureRegistration
 * @input_device: A pointer to a #GripInputDevice
 *
 * Determines which #GripGestureRegistration to use dependeng on properties of
 * the #GripInputDevice.
 *
 * Returns: a pointer to a #GripGestureRegistration or %NULL if not found.
 */
static GripGestureRegistration *
registration_for_input_device (struct Registrations *registrations,
                               GripInputDevice      *input_device)
{
  GripDeviceType device_type = grip_get_device_type(input_device);
  if (device_type == GRIP_DEVICE_TOUCHSCREEN)
  {
    return registrations->touchscreen;
  }
  else if (GRIP_DEVICE_TOUCHPAD)
  {
    return registrations->touchpad;
  }
  else if (GRIP_DEVICE_INDEPENDENT)
  {
    return registrations->independent;
  }
  return NULL;
}


static gint
pinch_gesture_handle_properties (GripEventGesturePinch *event,
                                 GeisSize               attr_count,
                                 GeisGestureAttr       *attrs)
{
  guint i = 0;
  gint touches = 0;

  for (i = 0; i < attr_count; ++i)
    {
      if (g_strcmp0 (attrs[i].name, GEIS_GESTURE_ATTRIBUTE_TOUCHES) == 0 &&
          attrs[i].type == GEIS_ATTR_TYPE_INTEGER)
        {
          touches = attrs[i].integer_val;
        }
      if (g_strcmp0 (attrs[i].name, GEIS_GESTURE_ATTRIBUTE_TIMESTAMP) == 0 &&
          attrs[i].type == GEIS_ATTR_TYPE_INTEGER)
        {
          event->timestamp = attrs[i].integer_val;
        }
      else if (g_strcmp0 (attrs[i].name, GEIS_GESTURE_ATTRIBUTE_FOCUS_X) == 0 &&
               attrs[i].type == GEIS_ATTR_TYPE_FLOAT)
        {
          event->focus_x = attrs[i].float_val;
        }
      else if (g_strcmp0 (attrs[i].name, GEIS_GESTURE_ATTRIBUTE_FOCUS_Y) == 0 &&
               attrs[i].type == GEIS_ATTR_TYPE_FLOAT)
        {
          event->focus_y = attrs[i].float_val;
        }
      else if (g_strcmp0 (attrs[i].name, GEIS_GESTURE_ATTRIBUTE_RADIUS_DELTA) == 0 &&
               attrs[i].type == GEIS_ATTR_TYPE_FLOAT)
        {
          event->radius_delta = attrs[i].float_val;
        }
      else if (g_strcmp0 (attrs[i].name, GEIS_GESTURE_ATTRIBUTE_RADIAL_VELOCITY) == 0 &&
               attrs[i].type == GEIS_ATTR_TYPE_FLOAT)
        {
          event->radial_velocity = attrs[i].float_val;
        }
      else if (g_strcmp0 (attrs[i].name, GEIS_GESTURE_ATTRIBUTE_RADIUS) == 0 &&
               attrs[i].type == GEIS_ATTR_TYPE_FLOAT)
        {
          event->radius = attrs[i].float_val;
        }
      else if (g_strcmp0 (attrs[i].name, GEIS_GESTURE_ATTRIBUTE_POSITION_X) == 0 &&
               attrs[i].type == GEIS_ATTR_TYPE_FLOAT)
        {
          event->position_x = attrs[i].float_val;
        }
      else if (g_strcmp0 (attrs[i].name, GEIS_GESTURE_ATTRIBUTE_POSITION_Y) == 0 &&
               attrs[i].type == GEIS_ATTR_TYPE_FLOAT)
        {
          event->position_y = attrs[i].float_val;
        }
    }

  return touches;
}

static gint
drag_gesture_handle_properties (GripEventGestureDrag *event,
                                GeisSize              attr_count,
                                GeisGestureAttr      *attrs)
{
  guint i;
  gint touches = 0;

  for (i = 0; i < attr_count; ++i)
    {
      if (g_strcmp0 (attrs[i].name, GEIS_GESTURE_ATTRIBUTE_TOUCHES) == 0 &&
          attrs[i].type == GEIS_ATTR_TYPE_INTEGER)
        {
          touches = attrs[i].integer_val;
        }
      if (g_strcmp0 (attrs[i].name, GEIS_GESTURE_ATTRIBUTE_TIMESTAMP) == 0 &&
          attrs[i].type == GEIS_ATTR_TYPE_INTEGER)
        {
          event->timestamp = attrs[i].integer_val;
        }
      else if (g_strcmp0 (attrs[i].name, GEIS_GESTURE_ATTRIBUTE_FOCUS_X) == 0 &&
               attrs[i].type == GEIS_ATTR_TYPE_FLOAT)
        {
          event->focus_x = attrs[i].float_val;
        }
      else if (g_strcmp0 (attrs[i].name, GEIS_GESTURE_ATTRIBUTE_FOCUS_Y) == 0 &&
               attrs[i].type == GEIS_ATTR_TYPE_FLOAT)
        {
          event->focus_y = attrs[i].float_val;
        }
      else if (g_strcmp0 (attrs[i].name, GEIS_GESTURE_ATTRIBUTE_DELTA_X) == 0 &&
               attrs[i].type == GEIS_ATTR_TYPE_FLOAT)
        {
          event->delta_x = attrs[i].float_val;
        }
      else if (g_strcmp0 (attrs[i].name, GEIS_GESTURE_ATTRIBUTE_DELTA_Y) == 0 &&
               attrs[i].type == GEIS_ATTR_TYPE_FLOAT)
        {
          event->delta_y = attrs[i].float_val;
        }
      else if (g_strcmp0 (attrs[i].name, GEIS_GESTURE_ATTRIBUTE_VELOCITY_X) == 0 &&
               attrs[i].type == GEIS_ATTR_TYPE_FLOAT)
        {
          event->velocity_x = attrs[i].float_val;
        }
      else if (g_strcmp0 (attrs[i].name, GEIS_GESTURE_ATTRIBUTE_VELOCITY_Y) == 0 &&
               attrs[i].type == GEIS_ATTR_TYPE_FLOAT)
        {
          event->velocity_y = attrs[i].float_val;
        }
      else if (g_strcmp0 (attrs[i].name, GEIS_GESTURE_ATTRIBUTE_POSITION_X) == 0 &&
               attrs[i].type == GEIS_ATTR_TYPE_FLOAT)
        {
          event->position_x = attrs[i].float_val;
        }
      else if (g_strcmp0 (attrs[i].name, GEIS_GESTURE_ATTRIBUTE_POSITION_Y) == 0 &&
               attrs[i].type == GEIS_ATTR_TYPE_FLOAT)
        {
          event->position_y = attrs[i].float_val;
        }
    }

  return touches;
}

static gint
rotate_gesture_handle_properties (GripEventGestureRotate *event,
                                  GeisSize                attr_count,
                                  GeisGestureAttr        *attrs)
{
  guint i;
  gint touches = 0;

  for (i = 0; i < attr_count; ++i)
    {
      if (g_strcmp0 (attrs[i].name, GEIS_GESTURE_ATTRIBUTE_TOUCHES) == 0 &&
          attrs[i].type == GEIS_ATTR_TYPE_INTEGER)
        {
          touches = attrs[i].integer_val;
        }
      if (g_strcmp0 (attrs[i].name, GEIS_GESTURE_ATTRIBUTE_TIMESTAMP) == 0 &&
          attrs[i].type == GEIS_ATTR_TYPE_INTEGER)
        {
          event->timestamp = attrs[i].integer_val;
        }
      else if (g_strcmp0 (attrs[i].name, GEIS_GESTURE_ATTRIBUTE_FOCUS_X) == 0 &&
               attrs[i].type == GEIS_ATTR_TYPE_FLOAT)
        {
          event->focus_x = attrs[i].float_val;
        }
      else if (g_strcmp0 (attrs[i].name, GEIS_GESTURE_ATTRIBUTE_FOCUS_Y) == 0 &&
               attrs[i].type == GEIS_ATTR_TYPE_FLOAT)
        {
          event->focus_y = attrs[i].float_val;
        }
      else if (g_strcmp0 (attrs[i].name, GEIS_GESTURE_ATTRIBUTE_ANGLE_DELTA) == 0 &&
               attrs[i].type == GEIS_ATTR_TYPE_FLOAT)
        {
          event->angle_delta = attrs[i].float_val;
        }
      else if (g_strcmp0 (attrs[i].name, GEIS_GESTURE_ATTRIBUTE_ANGULAR_VELOCITY) == 0 &&
               attrs[i].type == GEIS_ATTR_TYPE_FLOAT)
        {
          event->angular_velocity = attrs[i].float_val;
        }
      else if (g_strcmp0 (attrs[i].name, GEIS_GESTURE_ATTRIBUTE_ANGLE) == 0 &&
               attrs[i].type == GEIS_ATTR_TYPE_FLOAT)
        {
          event->angle = attrs[i].float_val;
        }
      else if (g_strcmp0 (attrs[i].name, GEIS_GESTURE_ATTRIBUTE_POSITION_X) == 0 &&
               attrs[i].type == GEIS_ATTR_TYPE_FLOAT)
        {
          event->position_x = attrs[i].float_val;
        }
      else if (g_strcmp0 (attrs[i].name, GEIS_GESTURE_ATTRIBUTE_POSITION_Y) == 0 &&
               attrs[i].type == GEIS_ATTR_TYPE_FLOAT)
        {
          event->position_y = attrs[i].float_val;
        }
    }

  return touches;
}

static gint
tap_gesture_handle_properties (GripEventGestureTap *event,
			       GeisSize             attr_count,
			       GeisGestureAttr     *attrs)
{
  guint i;
  gint touches = 0;

  for (i = 0; i < attr_count; ++i)
    {
      if (g_strcmp0 (attrs[i].name, GEIS_GESTURE_ATTRIBUTE_TOUCHES) == 0 &&
	  attrs[i].type == GEIS_ATTR_TYPE_INTEGER)
	{
	  touches = attrs[i].integer_val;
	}
      else if (g_strcmp0 (attrs[i].name, GEIS_GESTURE_ATTRIBUTE_TIMESTAMP) == 0 &&
          attrs[i].type == GEIS_ATTR_TYPE_INTEGER)
        {
          event->timestamp = attrs[i].integer_val;
        }
      else if (g_strcmp0 (attrs[i].name, GEIS_GESTURE_ATTRIBUTE_FOCUS_X) == 0 &&
               attrs[i].type == GEIS_ATTR_TYPE_FLOAT)
        {
          event->focus_x = attrs[i].float_val;
        }
      else if (g_strcmp0 (attrs[i].name, GEIS_GESTURE_ATTRIBUTE_FOCUS_Y) == 0 &&
               attrs[i].type == GEIS_ATTR_TYPE_FLOAT)
        {
          event->focus_y = attrs[i].float_val;
        }
      else if (g_strcmp0 (attrs[i].name, GEIS_GESTURE_ATTRIBUTE_POSITION_X) == 0 &&
               attrs[i].type == GEIS_ATTR_TYPE_FLOAT)
        {
          event->position_x = attrs[i].float_val;
        }
      else if (g_strcmp0 (attrs[i].name, GEIS_GESTURE_ATTRIBUTE_POSITION_Y) == 0 &&
               attrs[i].type == GEIS_ATTR_TYPE_FLOAT)
        {
          event->position_y = attrs[i].float_val;
        }
      else if (g_strcmp0 (attrs[i].name, GEIS_GESTURE_ATTRIBUTE_TAP_TIME) == 0 &&
               attrs[i].type == GEIS_ATTR_TYPE_INTEGER)
        {
          event->tap_time = attrs[i].float_val;
        }
    }

  return touches;
}

static void
gesture_added (void              *cookie G_GNUC_UNUSED,
               GeisGestureType    gesture_type G_GNUC_UNUSED,
               GeisGestureId      gesture_id G_GNUC_UNUSED,
               GeisSize           attr_count G_GNUC_UNUSED,
               GeisGestureAttr   *attrs G_GNUC_UNUSED)
{
}

static void
gesture_removed (void              *cookie G_GNUC_UNUSED,
                 GeisGestureType    gesture_type G_GNUC_UNUSED,
                 GeisGestureId      gesture_id G_GNUC_UNUSED,
                 GeisSize           attr_count G_GNUC_UNUSED,
                 GeisGestureAttr   *attrs G_GNUC_UNUSED)
{
}

static gboolean
matches_widget (GtkWidget *widget,
                GdkWindow *window,
                gint       x,
                gint       y)
{
  GtkAllocation alloc;
  gint ax, ay;

  gtk_widget_get_allocation (widget, &alloc);
  gdk_window_get_root_coords (window, alloc.x, alloc.y, &ax, &ay);

  return (x >= ax && x < ax + alloc.width && y >= ay && y < ay + alloc.height);
}


/*
 * process_gesture:
 * @cookie:
 * @type:
 * @id:
 * @attr_count:
 * @attrs:
 * @time_type:
 *
 * The generic gesture event dispatch engine.
 */
static void
process_gesture (void              *cookie,
                 GeisGestureType    type,
                 GeisGestureId      id,
                 GeisSize           attr_count,
                 GeisGestureAttr   *attrs,
                 GripTimeType       time_type)
{
  GripGestureManager *manager = (GripGestureManager *) cookie;
  GripInputDevice *input_device = NULL;
  GtkWindow *gtk_window = NULL;
  struct Registrations *registrations = NULL;
  GripGestureRegistration *reg = NULL;
  GList *l = NULL;
  GeisSize i;

  for (i = 0; i < attr_count; ++i)
  {
    if (0 == g_strcmp0(attrs[i].name, GEIS_GESTURE_ATTRIBUTE_DEVICE_ID))
    {
      input_device = device_id_to_input_device(manager, attrs[i].integer_val);
    }
    else if (0 == g_strcmp0(attrs[i].name, GEIS_GESTURE_ATTRIBUTE_EVENT_WINDOW_ID))
    {
      GHashTableIter iter;
      GtkWidget *key;
      struct Registrations *value;
      guint window_id = attrs[i].integer_val;

      g_hash_table_iter_init (&iter, manager->priv->registered_windows);
      while (g_hash_table_iter_next(&iter, (gpointer *)&key, (gpointer *)&value)) 
      {
        if (GDK_WINDOW_XID (gtk_widget_get_window (key)) == window_id)
        {
          gtk_window = (GtkWindow *)key;
          break;
        }
      }
    }
  }
  g_return_if_fail(input_device != NULL);
  g_return_if_fail(gtk_window != NULL);

  registrations = g_hash_table_lookup (manager->priv->registered_windows, gtk_window);
  g_return_if_fail(registrations != NULL);
  reg = registration_for_input_device (registrations, input_device);
  g_return_if_fail(reg != NULL);

  for (l = reg->bindings; l != NULL; l = l->next)
    {
      GripGestureBinding *binding = (GripGestureBinding *)l->data;

      if (binding->type == type)
        {
          GripGestureEvent *event = grip_gesture_event_new (type);

          if (type == GRIP_GESTURE_DRAG)
            {
              GripEventGestureDrag *drag = (GripEventGestureDrag *)event;

              drag->type    = type;
              drag->id      = id;
              drag->input_device = input_device;
              drag->fingers = drag_gesture_handle_properties (drag,
                  attr_count,
                  attrs);

              if (drag->fingers == (gint)binding->touches)
                {
                  if (matches_widget (binding->widget,
                                      gtk_widget_get_window(GTK_WIDGET (reg->window)),
                                      (gint)drag->focus_x,
                                      (gint)drag->focus_y))
                    {
                      binding->callback (binding->widget,
                                         time_type,
                                         event,
                                         binding->data);
                    }
                }
            }
          else if (type == GRIP_GESTURE_PINCH)
            {
              GripEventGesturePinch *pinch = (GripEventGesturePinch *)event;

              pinch->type    = type;
              pinch->id      = id;
              pinch->input_device = input_device;
              pinch->fingers = pinch_gesture_handle_properties (pinch,
                  attr_count,
                  attrs);

              if (pinch->fingers == binding->touches)
                {
                  if (matches_widget (binding->widget,
                                      gtk_widget_get_window(GTK_WIDGET (reg->window)),
                                      (gint)pinch->focus_x,
                                      (gint)pinch->focus_y))
                    {
                      binding->callback (binding->widget,
                                         time_type,
                                         event,
                                         binding->data);
                    }
                }
            }
          else if (type == GRIP_GESTURE_ROTATE)
            {
              GripEventGestureRotate *rotate = (GripEventGestureRotate *)event;

              rotate->type    = type;
              rotate->id      = id;
              rotate->input_device = input_device;
              rotate->fingers = rotate_gesture_handle_properties (rotate,
                  attr_count,
                  attrs);

              if (rotate->fingers == binding->touches)
                {
                  if (matches_widget (binding->widget,
                                      gtk_widget_get_window(GTK_WIDGET (reg->window)),
                                      (gint)rotate->focus_x,
                                      (gint)rotate->focus_y))
                    {
                      binding->callback (binding->widget,
                                         time_type,
                                         event,
                                         binding->data);
                    }
                }
            }
          else if (type == GRIP_GESTURE_TAP)
            {
              GripEventGestureTap *tap = (GripEventGestureTap *)event;

              tap->type    = type;
              tap->id      = id;
              tap->input_device = input_device;
              tap->fingers = tap_gesture_handle_properties (tap,
                  attr_count,
                  attrs);

              if (tap->fingers == binding->touches)
                {
                  if (matches_widget (binding->widget,
                                      gtk_widget_get_window(GTK_WIDGET (reg->window)),
                                      (gint)tap->focus_x,
                                      (gint)tap->focus_y))
                    {
                      binding->callback (binding->widget,
                                         time_type,
                                         event,
                                         binding->data);
                    }
                }
            }

          grip_gesture_event_free (event);
        }
    }
}

static void
gesture_start (void              *cookie,
               GeisGestureType    type,
               GeisGestureId      id,
               GeisSize           attr_count,
               GeisGestureAttr   *attrs)
{
  process_gesture(cookie, type, id, attr_count, attrs, GRIP_TIME_START);
}

static void
gesture_update (void              *cookie,
                GeisGestureType    type,
                GeisGestureId      id,
                GeisSize           attr_count,
                GeisGestureAttr   *attrs)
{
  process_gesture(cookie, type, id, attr_count, attrs, GRIP_TIME_UPDATE);
}

static void
gesture_finish (void              *cookie,
                GeisGestureType    type,
                GeisGestureId      id,
                GeisSize           attr_count,
                GeisGestureAttr   *attrs)
{
  process_gesture(cookie, type, id, attr_count, attrs, GRIP_TIME_END);
}

static void
grip_gesture_manager_init (GripGestureManager *item)
{
  GripGestureManagerPrivate *priv;

  priv = item->priv = GRIP_GESTURE_MANAGER_GET_PRIVATE (item);

  priv->registered_windows = g_hash_table_new (g_direct_hash, g_direct_equal);

  priv->devices = g_ptr_array_new ();
}

static gboolean
io_callback (GIOChannel   *source G_GNUC_UNUSED,
             GIOCondition  condition G_GNUC_UNUSED,
             gpointer      data)
{
  GripGestureRegistration *reg = (GripGestureRegistration *)data;

  geis_event_dispatch (reg->instance);

  return TRUE;
}

static void
destroy_registration(GripGestureRegistration *reg)
{
  GList *list;

  for (list = reg->bindings; list != NULL; list = list->next)
    {
      GripGestureBinding *binding = (GripGestureBinding *)list->data;

      if (binding->destroy)
        {
          GDestroyNotify d = binding->destroy;

          d (binding->data);
        }

      g_free (binding);
    }

  g_list_free (reg->bindings);
  g_io_channel_shutdown (reg->iochannel, TRUE, NULL);
  geis_finish (reg->instance);
}

static void
window_destroyed_cb (GtkWidget *object,
                     gpointer   user_data)
{
  GripGestureManager *manager = (GripGestureManager *)user_data;
  GripGestureManagerPrivate *priv = manager->priv;
  struct Registrations *reg = g_hash_table_lookup (priv->registered_windows, object);

  if (!reg)
    return;

  destroy_registration(reg->touchpad);
  destroy_registration(reg->touchscreen);
  destroy_registration(reg->independent);

  g_hash_table_remove (priv->registered_windows, object);
  g_free (reg);
}

static const gchar *
grip_type_to_geis_type (GripGestureType gesture_type,
			gint            touch_points)
{
  /* grail taps begin at 15, so let's convert that into something we
   * can index in geis_gesture_types. */
  int t = gesture_type == 15 ? 3 : gesture_type;

  return geis_gesture_types[(t * 5 + touch_points) - 1];
}


/* Public API */
GripGestureManager *
grip_gesture_manager_get (void)
{
  return g_object_new (GRIP_TYPE_GESTURE_MANAGER, NULL);
}


static void
grip_devices_for_type (GripDeviceType type, GArray *selection,
                       GripDevices *devices)
{
  guint i;

  for (i = 0; i < devices->len; ++i)
  {
    GripInputDevice *input_device = g_ptr_array_index (devices, i);
    GeisInputDeviceId id = grip_input_device_get_id (input_device);
    GripDeviceType device_type= grip_get_device_type(input_device);

    if ((type & GRIP_DEVICE_TOUCHSCREEN) && device_type == GRIP_DEVICE_TOUCHSCREEN)
    {
      g_array_append_val (selection, id);
    }
    if ((type & GRIP_DEVICE_TOUCHPAD) && device_type == GRIP_DEVICE_TOUCHPAD)
    {
      g_array_append_val (selection, id);
    }
    if ((type & GRIP_DEVICE_INDEPENDENT) && device_type == GRIP_DEVICE_INDEPENDENT)
    {
      g_array_append_val (selection, id);
    }
  }
}

/*
 * new_window_registration:
 * @manager: a GripGestureManager
 * @toplevel: a toplevel #GtkWindow
 *
 * Constructs a new #GripGestureRegistration for a #GtkWidget.
 */
static GripGestureRegistration *
new_window_registration(GripGestureManager *manager,
                        GtkWidget          *toplevel)
{
  GripGestureRegistration *reg;
  GeisInstance  instance;
  GIOChannel   *iochannel;
  gint fd = -1;
  GeisXcbWinInfo xcb_win_info = {
    .display_name = NULL,
    .screenp      = NULL,
    .window_id    = GDK_WINDOW_XID (gtk_widget_get_window(toplevel))
  };
  GeisWinInfo win_info = {
    GEIS_XCB_FULL_WINDOW,
    &xcb_win_info
  };

  if (geis_init (&win_info, &instance) != GEIS_STATUS_SUCCESS)
    {
      g_warning ("Failed to initialize gesture manager.");
      return NULL;
    }

  if (geis_configuration_supported (instance,
                                    GEIS_CONFIG_UNIX_FD) != GEIS_STATUS_SUCCESS)
    {
      g_warning ("Gesture manager does not support UNIX fd.");
      return NULL;
    }

  if (geis_configuration_get_value (instance,
                                    GEIS_CONFIG_UNIX_FD,
                                    &fd) != GEIS_STATUS_SUCCESS)
    {
      g_error ("Gesture manager failed to obtain UNIX fd.");
      return NULL;
    }

  reg = g_new0 (GripGestureRegistration, 1);

  reg->window   = GTK_WINDOW (toplevel);
  reg->instance = instance;

  g_signal_connect (toplevel,
                    "destroy",
                    G_CALLBACK (window_destroyed_cb),
                    manager);

  iochannel = g_io_channel_unix_new (fd);
  g_io_add_watch (iochannel,
                  G_IO_IN,
                  io_callback,
                  reg);

  reg->iochannel = iochannel;

  reg->gesture_list = g_ptr_array_new ();

  return reg;
}

static void
bind_registration(GripGestureManager *manager,
    GripGestureRegistration *reg,
    GtkWidget               *widget,
    GripGestureType         gesture_type,
    GripDeviceType          device_type,
    gint                    touch_points,
    GripGestureCallback     callback,
    gpointer                user_data,
    GDestroyNotify          destroy)
{
  GripGestureManagerPrivate *priv;
  GripGestureBinding *binding;
  GArray *devices;

  priv = manager->priv;

  if (reg->gesture_list->len)
    g_ptr_array_remove_index (reg->gesture_list,
                              reg->gesture_list->len - 1);

  devices = g_array_new (TRUE, FALSE, sizeof(GeisInputDeviceId));
  grip_devices_for_type(device_type, devices, priv->devices);
  if (devices->len == 0) {
      g_array_free(devices, TRUE);
      return;
  }

  g_ptr_array_add (reg->gesture_list,
                   (gchar *)grip_type_to_geis_type (gesture_type, touch_points));
  g_ptr_array_add (reg->gesture_list,
                   NULL);

  geis_subscribe (reg->instance,
                  (GeisInputDeviceId *)(void *)devices->data,
                  (const char**)reg->gesture_list->pdata,
                  &gesture_funcs,
                  manager);

  g_array_unref (devices);

  /* XXX - check for duplicates in reg->bindings first */
  binding = g_new0 (GripGestureBinding, 1);

  binding->type     = gesture_type;
  binding->widget   = widget;
  binding->touches  = touch_points;
  binding->callback = callback;
  binding->data     = user_data;
  binding->destroy  = destroy;

  reg->bindings = g_list_append (reg->bindings, binding);
}

/*
 * register_internal:
 * @manager: the @GripGestureManager
 * @widget: the GtkWidget to be bound
 * @gesture_type: the type of gesture to subscribe for
 * @device_type: the type of input device to subscribe for
 * @touch_points: 
 * @callback: the callback to be invoked on gesture events
 * @user_data: the callbackl context to be passed back on gesture events
 * @destroy:
 *
 * Binds or rebinds a gesture subscription for a widget.
 */
static void
register_internal (GripGestureManager *manager,
                   GtkWidget          *widget,
                   GripGestureType     gesture_type,
                   GripDeviceType      device_type,
                   gint                touch_points,
                   GripGestureCallback callback,
                   gpointer            user_data,
                   GDestroyNotify      destroy)
{
  GripGestureManagerPrivate *priv;
  GtkWidget *toplevel;
  struct Registrations *registrations;

  g_return_if_fail (GRIP_IS_GESTURE_MANAGER (manager));
  g_return_if_fail (GTK_IS_WIDGET (widget));

  toplevel = gtk_widget_get_toplevel (widget);

  g_return_if_fail (GTK_IS_WINDOW (toplevel));

  priv = manager->priv;

  if (!(registrations = g_hash_table_lookup (priv->registered_windows, toplevel)))
    {
      registrations = g_new(struct Registrations, 1);
      registrations->touchscreen = new_window_registration(manager, toplevel);
      registrations->touchpad = new_window_registration(manager, toplevel);
      registrations->independent = new_window_registration(manager, toplevel);

      if (registrations->touchscreen == NULL ||
          registrations->touchpad == NULL ||
          registrations->independent == NULL)
        return;
    }
  else
    {
      if (device_type & GRIP_DEVICE_TOUCHSCREEN)
        geis_unsubscribe (registrations->touchscreen->instance,
            (GeisGestureType*)registrations->touchscreen->gesture_list->pdata);
      if (device_type & GRIP_DEVICE_TOUCHPAD)
        geis_unsubscribe (registrations->touchpad->instance,
            (GeisGestureType*)registrations->touchpad->gesture_list->pdata);
      if (device_type & GRIP_DEVICE_INDEPENDENT)
        geis_unsubscribe (registrations->independent->instance,
            (GeisGestureType*)registrations->independent->gesture_list->pdata);
    }

  if (device_type & GRIP_DEVICE_TOUCHSCREEN)
    bind_registration(manager,
        registrations->touchscreen, widget, gesture_type, GRIP_DEVICE_TOUCHSCREEN, touch_points,
        callback, user_data, destroy);

  if (device_type & GRIP_DEVICE_TOUCHPAD)
    bind_registration(manager,
        registrations->touchpad, widget, gesture_type, GRIP_DEVICE_TOUCHPAD, touch_points,
        callback, user_data, destroy);

  if (device_type & GRIP_DEVICE_INDEPENDENT)
    bind_registration(manager,
        registrations->independent, widget, gesture_type, GRIP_DEVICE_INDEPENDENT, touch_points,
        callback, user_data, destroy);
  g_hash_table_insert (priv->registered_windows,
                       toplevel,
                       registrations);
}

static void
toplevel_notify_cb (GtkWidget    *widget,
                    GParamSpec   *pspec,
                    gpointer      user_data)
{
  if (pspec->name == g_intern_static_string ("parent"))
    {
      GtkWidget *toplevel = gtk_widget_get_toplevel (widget);
      GripRegistrationRequest *req = (GripRegistrationRequest *)user_data;

      if (GTK_IS_WINDOW (toplevel))
        {
          g_signal_handlers_disconnect_by_func (widget,
                                                G_CALLBACK (toplevel_notify_cb),
                                                user_data);

          if (gtk_widget_get_mapped (GTK_WIDGET (toplevel)))
            {
              register_internal (req->manager,
                                 req->widget,
                                 req->gesture_type,
                                 req->device_type,
                                 req->touch_points,
                                 req->callback,
                                 req->user_data,
                                 req->destroy);

              g_free (req);
            }
          else
            {
              GripGestureManagerPrivate *priv = req->manager->priv;

              priv->requests = g_list_append (priv->requests, req);

              g_signal_connect (toplevel,
                                "map-event",
                                G_CALLBACK (window_mapped_cb),
                                req->manager);
            }
        }
      else
        {
          g_signal_connect (G_OBJECT (toplevel),
                            "notify",
                            G_CALLBACK (toplevel_notify_cb),
                            user_data);
        }
    }
}

/*
 * register_widget:
 *
 * Registers a specific widget for a specific gesture made by a specific input
 * device type.
 *
 * If the widget's containing window is a valid window, the widget is registered
 * right away, otherwise the registration is deferred until the widget actually
 * has a top-level window.
 */
static void
register_widget (GripGestureManager *manager,
                 GtkWidget          *widget,
                 GripGestureType     gesture_type,
                 GripDeviceType      device_type,
                 gint                touch_points,
                 GripGestureCallback callback,
                 gpointer            user_data,
                 GDestroyNotify      destroy)
{
  GtkWidget *toplevel = gtk_widget_get_toplevel (widget);

  if (GTK_IS_WINDOW (toplevel))
    {
      register_internal (manager,
                         widget,
                         gesture_type,
                         device_type,
                         touch_points,
                         callback,
                         user_data,
                         destroy);
    }
  else
    {
      GripRegistrationRequest *req = g_new0 (GripRegistrationRequest, 1);

      req->manager      = manager;
      req->widget       = widget;
      req->gesture_type = gesture_type;
      req->device_type  = device_type;
      req->touch_points = touch_points;
      req->callback     = callback;
      req->user_data    = user_data;
      req->destroy      = destroy;

      g_signal_connect (toplevel,
                        "notify",
                        G_CALLBACK (toplevel_notify_cb),
                        req);
    }
}

static void
window_mapped_cb (GtkWidget   *widget,
                  GdkEvent    *event G_GNUC_UNUSED,
                  gpointer     user_data)
{
  GripGestureManager *manager = (GripGestureManager *)user_data;
  GripGestureManagerPrivate *priv = manager->priv;
  GList *tmp;

  for (tmp = priv->requests; tmp != NULL; tmp = tmp->next)
    {
      GripRegistrationRequest *req = tmp->data;

      register_widget (req->manager,
                       req->widget,
                       req->gesture_type,
                       req->device_type,
                       req->touch_points,
                       req->callback,
                       req->user_data,
                       req->destroy);

      g_free (req);
    }

  g_list_free (priv->requests);
  priv->requests = NULL;

  g_signal_handlers_disconnect_by_func (widget,
                                        window_mapped_cb,
                                        user_data);
}

/**
 * grip_gesture_manager_register_window:
 * @manager: A #GripGestureManager instance.
 * @widget: A #GtkWidget to register the gesture event for.
 * @gesture_type: The type of gesture event to register.
 * @device_type: The type of the device use to create the gesture.
 * @touch_points: Number of touch points for this gesture.
 * @callback: Called when a gesture starts, updates, or ends.
 * @user_data: User data
 * @destroy: Destroy callback for user data.
 *
 * Registers a widget to receive gesture events.
 *
 * The callback parameters provided will be called by the
 * #GripGestureManager whenever the user initiates a gesture
 * on the specified window.
 */
void
grip_gesture_manager_register_window (GripGestureManager  *manager,
                                      GtkWidget           *widget,
                                      GripGestureType      gesture_type,
                                      GripDeviceType       device_type,
                                      gint                 touch_points,
                                      GripGestureCallback  callback,
                                      gpointer             user_data,
                                      GDestroyNotify       destroy)
{
  GtkWidget *toplevel;

  g_return_if_fail (GRIP_IS_GESTURE_MANAGER (manager));
  g_return_if_fail (GTK_IS_WIDGET (widget));

  toplevel = gtk_widget_get_toplevel (widget);

  if (GTK_IS_WINDOW (toplevel))
    {
      if (gtk_widget_get_mapped (GTK_WIDGET (toplevel)))
        {
          register_internal (manager,
                             widget,
                             gesture_type,
                             device_type,
                             touch_points,
                             callback,
                             user_data,
                             destroy);
        }
      else
        {
          GripRegistrationRequest *req = g_new0 (GripRegistrationRequest, 1);
          GripGestureManagerPrivate *priv = manager->priv;

          req->manager      = manager;
          req->widget       = widget;
          req->gesture_type = gesture_type;
          req->device_type  = device_type;
          req->touch_points = touch_points;
          req->callback     = callback;
          req->user_data    = user_data;
          req->destroy      = destroy;

          priv->requests = g_list_append (priv->requests, req);

          g_signal_connect (toplevel,
                            "map-event",
                            G_CALLBACK (window_mapped_cb),
                            manager);
        }
    }
  else
    {
      GripRegistrationRequest *req = g_new0 (GripRegistrationRequest, 1);

      req->manager      = manager;
      req->widget       = widget;
      req->gesture_type = gesture_type;
      req->device_type  = device_type;
      req->touch_points = touch_points;
      req->callback     = callback;
      req->user_data    = user_data;
      req->destroy      = destroy;

      g_signal_connect (toplevel,
                        "notify",
                        G_CALLBACK (toplevel_notify_cb),
                        req);
    }
}

static void
shutdown_registration(GripGestureRegistration *reg)
{
  free_registration(reg);
  geis_finish(reg->instance);
  reg->instance = NULL;
  g_free(reg);
}

void
grip_gesture_manager_unregister_window (GripGestureManager  *manager,
                                        GtkWidget           *toplevel)
{
  GripGestureManagerPrivate *priv = manager->priv;
  struct Registrations *registrations;

  g_return_if_fail (GRIP_IS_GESTURE_MANAGER (manager));
  g_return_if_fail (GTK_IS_WINDOW (toplevel));

  /* Currently only allow unsubscribing after the window has been shown. */
  g_return_if_fail (gtk_widget_get_mapped (GTK_WIDGET (toplevel)));
  registrations = g_hash_table_lookup (priv->registered_windows, toplevel);
  if (!registrations) {
      return;
  }

  shutdown_registration(registrations->touchscreen);
  shutdown_registration(registrations->touchpad);
  shutdown_registration(registrations->independent);
  g_free(registrations);
  g_hash_table_remove(priv->registered_windows, toplevel);

}

GType
grip_gesture_event_get_type (void)
{
  static GType type = 0;

  if (type == 0)
    {
      type = g_boxed_type_register_static (g_intern_static_string ("GripGestureEvent"),
					   (GBoxedCopyFunc)grip_gesture_event_copy,
					   (GBoxedFreeFunc)grip_gesture_event_free);
    }

  return type;
}

/**
 * grip_gesture_event_new:
 * @gesture_type: the type of the gesture
 *
 * Creates a new Grip gesture event.
 *
 * Returns: a new #GripGestureEvent
 */
GripGestureEvent *
grip_gesture_event_new (GripGestureType gesture_type)
{
  GripGestureEvent *event = g_slice_new0 (GripGestureEvent);

  event->any.type = gesture_type;

  return event;
}

/**
 * grip_gesture_event_copy:
 * @event: an existing #GripGestureEvent
 *
 * Creates a new #GripGestureEvent instance using a deep copy of and existing
 * event.
 *
 * Returns: a new #GripGestureEvent
 */
GripGestureEvent *
grip_gesture_event_copy (const GripGestureEvent *event)
{
  GripGestureEvent *new_event;

  g_return_val_if_fail (event != NULL, NULL);

  new_event = grip_gesture_event_new (event->type);
  *new_event = *event;

  return new_event;
}

/**
 * grip_gesture_event_free:
 * @event: a #GripGestureEvent
 *
 * Frees the resources allocated for a #GripGestureEvent.
 */
void
grip_gesture_event_free (GripGestureEvent *event)
{
  g_return_if_fail (event != NULL);

  g_slice_free (GripGestureEvent, event);
}
