/*
A library to allow applictions to provide simple indications of
information to be displayed to users of the application through the
interface shell.

Copyright 2009 Canonical Ltd.

Authors:
    Ted Gould <ted@canonical.com>

This program is free software: you can redistribute it and/or modify it 
under the terms of either or both of the following licenses:

1) the GNU Lesser General Public License version 3, as published by the 
Free Software Foundation; and/or
2) the GNU Lesser General Public License version 2.1, as published by 
the Free Software Foundation.

This program is distributed in the hope that it will be useful, but 
WITHOUT ANY WARRANTY; without even the implied warranties of 
MERCHANTABILITY, SATISFACTORY QUALITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the applicable version of the GNU Lesser General Public 
License for more details.

You should have received a copy of both the GNU Lesser General Public 
License version 3 and version 2.1 along with this program.  If not, see 
<http://www.gnu.org/licenses/>
*/

#include <stdlib.h>
#include <gio/gio.h>
#include "listener.h"
#include "interests-priv.h"
#include "dbus-shared.h"
#include "gen-indicate-interface.xml.h"
#include "gen-indicate-listener.xml.h"
#include "indicate-marshal.h"

/* Errors */
enum {
	LAST_ERROR
};

/* Signals */
enum {
	INDICATOR_ADDED,
	INDICATOR_REMOVED,
	INDICATOR_MODIFIED,
	SERVER_ADDED,
	SERVER_REMOVED,
	SERVER_COUNT_CHANGED,
	INDICATOR_SERVERS_REPORT,
	LAST_SIGNAL
};

static guint signals[LAST_SIGNAL] = { 0 };

#include "listener-private.h"

typedef struct {
	GDBusProxy * proxy;
	GDBusProxy * property_proxy;
	GDBusConnection * connection;
	gchar * name;
	gchar * path;
	gchar * type;
	IndicateListener * listener;
	GHashTable * indicators;
	guint introspect_level;
	gboolean hidden;
	guint dbus_listener_sub;

	IndicateListenerServer server;
} proxy_t;

static gint
proxy_t_equal (gconstpointer pa, gconstpointer pb)
{
	proxy_t * a = (proxy_t *)pa; proxy_t * b = (proxy_t *)pb;

	if (a->connection == b->connection) {
		return g_strcmp0(a->name, b->name);
	} else {
		/* we're only using this for equal, not sorting */
		return 1;
	}
}

typedef struct {
	GDBusConnection * bus;
	gchar * name;
	gchar * path;
} proxy_todo_t;

G_DEFINE_TYPE (IndicateListener, indicate_listener, G_TYPE_OBJECT);

/* Prototypes */
static void indicate_listener_finalize (GObject * obj);
static void indicate_listener_dispose (GObject * obj);
static void proxy_struct_destroy (gpointer data);
static void todo_list_add (const gchar * name, const gchar * path, IndicateListener * listener);
static gboolean todo_idle (gpointer data);
static void set_max_indicators_cb (GObject * object, GAsyncResult * res, gpointer user_data);
static void get_type_initial_cb (IndicateListener * listener, IndicateListenerServer * server, const gchar * type, gpointer data);
static void get_type_cb (IndicateListener * listener, IndicateListenerServer * server, const gchar * type, gpointer data);
static void proxy_indicator_added_legacy (guint id, gchar * type, proxy_t * proxyt);
static void proxy_indicator_added (guint id, proxy_t * proxyt);
static void proxy_indicator_removed_legacy (guint id, gchar * type, proxy_t * proxyt);
static void proxy_indicator_removed (guint id, proxy_t * proxyt);
static void proxy_indicator_modified (guint id, const gchar * property, proxy_t * proxyt);
static void proxy_server_count_changed (guint count, proxy_t * proxyt);
static void proxy_get_indicator_list (GObject * object, GAsyncResult * res, gpointer data);
static void proxy_destroyed (GObject * proxy, gpointer user_data);
static void bus_method_call (GDBusConnection * connection, const gchar * sender, const gchar * path, const gchar * interface, const gchar * method, GVariant * params, GDBusMethodInvocation * invocation, gpointer user_data);
static void bus_get_cb (GObject * object, GAsyncResult * res, gpointer user_data);
static void bus_filter_show_server (GDBusConnection * connection, const gchar * sender, const gchar * object, const gchar * interface, const gchar * signal, GVariant * params, gpointer user_data);
static void prox_ready_cb (GObject * object, GAsyncResult * res, gpointer user_data);
static void proxy_signal_cb (GDBusProxy * proxy, gchar * sender, gchar * signal, GVariant * params, gpointer user_data);

/* Bus Stuff */
static GDBusNodeInfo *            bus_indicate_node_info = NULL;
static GDBusInterfaceInfo *       bus_indicate_interface_info = NULL;
static GDBusNodeInfo *            bus_listener_node_info = NULL;
static GDBusInterfaceInfo *       bus_listener_interface_info = NULL;
static const GDBusInterfaceVTable bus_interface_table = {
	method_call:    bus_method_call,
	get_property:   NULL, /* No properties that can be get */
	set_property:   NULL  /* No properties that can be set */
};


/* Code */
static void
indicate_listener_class_init (IndicateListenerClass * class)
{
	/* g_debug("Listener Class Initialized"); */
	GObjectClass * gobj;
	gobj = G_OBJECT_CLASS(class);

	g_type_class_add_private (class, sizeof (IndicateListenerPrivate));

	gobj->dispose = indicate_listener_dispose;
	gobj->finalize = indicate_listener_finalize;

	signals[INDICATOR_ADDED] = g_signal_new(INDICATE_LISTENER_SIGNAL_INDICATOR_ADDED,
	                                        G_TYPE_FROM_CLASS (class),
	                                        G_SIGNAL_RUN_LAST,
	                                        G_STRUCT_OFFSET (IndicateListenerClass, indicator_added),
	                                        NULL, NULL,
	                                        _indicate_marshal_VOID__POINTER_POINTER,
	                                        G_TYPE_NONE, 2, INDICATE_TYPE_LISTENER_SERVER, INDICATE_TYPE_LISTENER_INDICATOR);
	signals[INDICATOR_REMOVED] = g_signal_new(INDICATE_LISTENER_SIGNAL_INDICATOR_REMOVED,
	                                        G_TYPE_FROM_CLASS (class),
	                                        G_SIGNAL_RUN_LAST,
	                                        G_STRUCT_OFFSET (IndicateListenerClass, indicator_removed),
	                                        NULL, NULL,
	                                        _indicate_marshal_VOID__POINTER_POINTER,
	                                        G_TYPE_NONE, 2, INDICATE_TYPE_LISTENER_SERVER, INDICATE_TYPE_LISTENER_INDICATOR);
	signals[INDICATOR_MODIFIED] = g_signal_new(INDICATE_LISTENER_SIGNAL_INDICATOR_MODIFIED,
	                                        G_TYPE_FROM_CLASS (class),
	                                        G_SIGNAL_RUN_LAST,
	                                        G_STRUCT_OFFSET (IndicateListenerClass, indicator_modified),
	                                        NULL, NULL,
	                                        _indicate_marshal_VOID__POINTER_POINTER_STRING,
	                                        G_TYPE_NONE, 3, INDICATE_TYPE_LISTENER_SERVER, INDICATE_TYPE_LISTENER_INDICATOR, G_TYPE_STRING);
	signals[SERVER_ADDED] = g_signal_new(INDICATE_LISTENER_SIGNAL_SERVER_ADDED,
	                                        G_TYPE_FROM_CLASS (class),
	                                        G_SIGNAL_RUN_LAST,
	                                        G_STRUCT_OFFSET (IndicateListenerClass, server_added),
	                                        NULL, NULL,
	                                        _indicate_marshal_VOID__POINTER_STRING,
	                                        G_TYPE_NONE, 2, INDICATE_TYPE_LISTENER_SERVER, G_TYPE_STRING);
	signals[SERVER_REMOVED] = g_signal_new(INDICATE_LISTENER_SIGNAL_SERVER_REMOVED,
	                                        G_TYPE_FROM_CLASS (class),
	                                        G_SIGNAL_RUN_LAST,
	                                        G_STRUCT_OFFSET (IndicateListenerClass, server_removed),
	                                        NULL, NULL,
	                                        _indicate_marshal_VOID__POINTER_STRING,
	                                        G_TYPE_NONE, 2, INDICATE_TYPE_LISTENER_SERVER, G_TYPE_STRING);
	signals[SERVER_COUNT_CHANGED] = g_signal_new(INDICATE_LISTENER_SIGNAL_SERVER_COUNT_CHANGED,
	                                        G_TYPE_FROM_CLASS (class),
	                                        G_SIGNAL_RUN_LAST,
	                                        G_STRUCT_OFFSET (IndicateListenerClass, server_count_changed),
	                                        NULL, NULL,
	                                        _indicate_marshal_VOID__POINTER_UINT,
	                                        G_TYPE_NONE, 2, INDICATE_TYPE_LISTENER_SERVER, G_TYPE_UINT);
	signals[INDICATOR_SERVERS_REPORT] = g_signal_new("indicator-servers-report",
	                                        G_TYPE_FROM_CLASS (class),
	                                        G_SIGNAL_RUN_LAST,
	                                        G_STRUCT_OFFSET (IndicateListenerClass, indicator_servers_report),
	                                        NULL, NULL,
	                                        g_cclosure_marshal_VOID__VOID,
	                                        G_TYPE_NONE, 0, G_TYPE_NONE);

	/* DBus interfaces */
	if (bus_indicate_node_info == NULL) {
		GError * error = NULL;

		bus_indicate_node_info = g_dbus_node_info_new_for_xml(_indicate_interface, &error);
		if (error != NULL) {
			g_error("Unable to parse Indicate Interface description: %s", error->message);
			g_error_free(error);
		}
	}

	if (bus_indicate_interface_info == NULL) {
		bus_indicate_interface_info = g_dbus_node_info_lookup_interface(bus_indicate_node_info, INDICATE_DBUS_IFACE);

		if (bus_indicate_interface_info == NULL) {
			g_error("Unable to find interface '" INDICATE_DBUS_IFACE "'");
		}
	}

	if (bus_listener_node_info == NULL) {
		GError * error = NULL;

		bus_listener_node_info = g_dbus_node_info_new_for_xml(_indicate_listener, &error);
		if (error != NULL) {
			g_error("Unable to parse Listener Interface description: %s", error->message);
			g_error_free(error);
		}
	}

	if (bus_listener_interface_info == NULL) {
		bus_listener_interface_info = g_dbus_node_info_lookup_interface(bus_listener_node_info, INDICATE_LISTENER_DBUS_IFACE);

		if (bus_listener_interface_info == NULL) {
			g_error("Unable to find interface '" INDICATE_LISTENER_DBUS_IFACE "'");
		}
	}

	return;
}

static void
indicate_listener_init (IndicateListener * listener)
{
	/* g_debug("Listener Object Initialized"); */
	IndicateListenerPrivate * priv = INDICATE_LISTENER_GET_PRIVATE(listener);

	/* Get the buses */
	g_bus_get(G_BUS_TYPE_SESSION,
	          NULL, /* TODO: Cancelable */
	          bus_get_cb,
	          listener);

	/* Initialize Data structures */
	priv->proxies = NULL;

	/* TODO: Look at some common scenarios and find out how to make this sized */
	priv->proxy_todo = g_array_new(FALSE, TRUE, sizeof(proxy_todo_t));
	priv->todo_idle = 0;

	priv->max_indicators = -1;

	return;
}

/* Remove references to objects */
static void
indicate_listener_dispose (GObject * obj)
{
	IndicateListener * listener = INDICATE_LISTENER(obj);
	IndicateListenerPrivate * priv = INDICATE_LISTENER_GET_PRIVATE(listener);

	if (priv->signal_subscription != 0) {
		g_dbus_connection_signal_unsubscribe(priv->session_bus, priv->signal_subscription);
		priv->signal_subscription = 0;
	}

	if (priv->object_registration != 0) {
		g_dbus_connection_signal_unsubscribe(priv->session_bus, priv->object_registration);
		priv->object_registration = 0;
	}

	G_OBJECT_CLASS (indicate_listener_parent_class)->dispose (obj);
	return;
}

/* Free memory */
static void
indicate_listener_finalize (GObject * obj)
{
	IndicateListener * listener = INDICATE_LISTENER(obj);
	IndicateListenerPrivate * priv = INDICATE_LISTENER_GET_PRIVATE(listener);

	if (priv->todo_idle != 0) {
		g_idle_remove_by_data(obj);
	}

	/* Hack: proxy_struct_destroy() lacks a user_data parameter, but since the
	 * caller is responsible for handling params on the stack, it works
	 */
	g_list_foreach(priv->proxies, (GFunc)proxy_struct_destroy, NULL);
	g_list_free(priv->proxies);

	G_OBJECT_CLASS (indicate_listener_parent_class)->finalize (obj);
	return;
}

/* Respond to a name change related to the proxy, usually mean that
   it's falling off of the bus. */
static void
proxy_name_owner_changed (GDBusConnection * connection, const gchar * sender_name, const gchar * object_path, const gchar * interface_name, const gchar * signal_name, GVariant * parameters, gpointer user_data)
{
	const gchar * new_name;
	g_variant_get(parameters, "(&s&s&s)", NULL, NULL, &new_name);

	if (new_name == NULL || new_name[0] == 0) {
		proxy_t * proxyt = (proxy_t *)user_data;

		g_object_unref(proxyt->proxy);
		proxyt->proxy = NULL;

		proxy_destroyed(NULL, user_data);
	}

	return;
}

/* Response to connecting to the session bus */
static void
bus_get_cb (GObject * object, GAsyncResult * res, gpointer user_data)
{
	GError * error = NULL;
	GDBusConnection * connection = g_bus_get_finish(res, &error);

	if (error != NULL) {
		g_error("Unable to get session bus: %s", error->message);
		g_error_free(error);
		return;
	}

	IndicateListenerPrivate * priv = INDICATE_LISTENER_GET_PRIVATE(user_data);
	priv->session_bus = connection;

	/* Export this object onto the bus */
	priv->object_registration = g_dbus_connection_register_object(priv->session_bus,
	                                                              INDICATE_LISTENER_OBJ_PATH,
	                                                              bus_listener_interface_info,
	                                                              &bus_interface_table,
	                                                              user_data,
	                                                              NULL,
	                                                              &error);

	if (error != NULL) {
		g_error("Unable to register listener on the bus: %s", error->message);
		g_error_free(error);
		return;
	}

	/* Watch for servers starting up */
	priv->signal_subscription = g_dbus_connection_signal_subscribe(priv->session_bus,
	                                                               NULL, /* sender */
	                                                               INDICATE_DBUS_IFACE,
	                                                               "ServerShow",
	                                                               NULL, /* path */
	                                                               NULL, /* arg0 */
	                                                               G_DBUS_SIGNAL_FLAGS_NONE,
	                                                               bus_filter_show_server,
	                                                               user_data,
	                                                               NULL); /* destroy notify */

	/* Tell that we're here! */
	g_dbus_connection_emit_signal(priv->session_bus,
	                              NULL, /* dest */
	                              INDICATE_LISTENER_OBJ_PATH,
	                              INDICATE_LISTENER_DBUS_IFACE,
	                              "IndicatorServersReport",
	                              NULL, /* params */
	                              &error);

	if (error != NULL) {
		g_warning("Unable to send the 'IndicatorServersReport' signal: %s", error->message);
		g_error_free(error);
		return;
	}

	return;
}

/* Handle a metho call from dbus */
static void
bus_method_call (GDBusConnection * connection, const gchar * sender, const gchar * path, const gchar * interface, const gchar * method, GVariant * params, GDBusMethodInvocation * invocation, gpointer user_data)
{

	/* TODO: We should do this, in the future though */
}

/**
	indicate_listener_new:

	Creates a new Listener object.  Does not set this to the default
	listener object.

	Return value: A new listener object.
*/
IndicateListener *
indicate_listener_new (void)
{
	IndicateListener * listener;
	listener = g_object_new(INDICATE_TYPE_LISTENER, NULL);
	return listener;
}

/* The pointer to the default listener object */
static IndicateListener * default_indicate_listener = NULL;

/**
	indcate_listener_ref_default:

	Looks for the default listener, and if it doesn't exist it'll
	allocate a new one and mark it the default.  This is what most
	programs should use rather than #indicate_listener_new because
	it is rare that anyone would need more than a single listener.
	This function does increase the ref count on the object so anyone
	calling it should unref thier instance of the default listener.

	Return value: The default instance of #IndicateListener for the
		program.
*/
IndicateListener *
indicate_listener_ref_default (void)
{
	if (default_indicate_listener != NULL) {
		g_object_ref(default_indicate_listener);
	} else {
		default_indicate_listener = g_object_new(INDICATE_TYPE_LISTENER, NULL);
		g_object_add_weak_pointer(G_OBJECT(default_indicate_listener),
		                          (gpointer *)&default_indicate_listener);
	}

	return default_indicate_listener;
}

/* A small filter function that notices when someone sends
   a ServerShow signal and creates an entry for us to investigate
   them more in an idle loop lookup. */
static void
bus_filter_show_server (GDBusConnection * connection, const gchar * sender, const gchar * object, const gchar * interface, const gchar * signal, GVariant * params, gpointer user_data)
{
	if (g_strcmp0(signal, "ServerShow") != 0) {
		return;
	}

	todo_list_add(sender, object, INDICATE_LISTENER(user_data));

	return;
}

/* A hashtable for each function to look at all of the indicators
   on a proxy_t object and signal their destruction */
static void
proxy_struct_destroy_indicators (gpointer key, gpointer value, gpointer data)
{
	proxy_t * proxy_data = (proxy_t *)data;

	if (value) {
		g_signal_emit(proxy_data->listener, signals[INDICATOR_REMOVED], 0, &proxy_data->server, GUINT_TO_POINTER(key), TRUE);
	}
	return;
}

/* Cleans up a proxy_t struct after allocation.  It signals that
   all of the indicators are going away and the server itself. */
static void
proxy_struct_destroy (gpointer data)
{
	proxy_t * proxy_data = data;

	if (proxy_data->indicators != NULL) {
		g_hash_table_foreach(proxy_data->indicators,
							 proxy_struct_destroy_indicators,
							 proxy_data);
		g_hash_table_destroy(proxy_data->indicators);

		g_signal_emit(proxy_data->listener, signals[SERVER_REMOVED], 0, &proxy_data->server, proxy_data->type, TRUE);
		proxy_data->indicators = NULL;
	}

	if (proxy_data->dbus_listener_sub != 0) {
		g_dbus_connection_signal_unsubscribe(proxy_data->connection, proxy_data->dbus_listener_sub);
		proxy_data->dbus_listener_sub = 0;
	}

	if (proxy_data->proxy != NULL) {
		g_object_unref(G_OBJECT(proxy_data->proxy));
	}

	if (proxy_data->name != NULL) {
		g_free(proxy_data->name);
	}

	if (proxy_data->path != NULL) {
		g_free(proxy_data->path);
	}

	if (proxy_data->type != NULL) {
		g_free(proxy_data->type);
	}
	g_free(proxy_data);

	return;
}

/* Creates a todo list item for the particular server and
   path as they've signaled that they're on DBus and like
   talking about indicators. */
static void
todo_list_add (const gchar * name, const gchar * path, IndicateListener * listener)
{
	if (name == NULL || name[0] != ':') {
		return;
	}

	IndicateListenerPrivate * priv = INDICATE_LISTENER_GET_PRIVATE(listener);

	proxy_todo_t todo;
	todo.name = g_strdup(name);
	todo.path = g_strdup(path);
	todo.bus  = priv->session_bus;

	g_array_append_val(priv->proxy_todo, todo);

	if (priv->todo_idle == 0) {
		priv->todo_idle = g_idle_add(todo_idle, listener);
	}

	return;
}

/* This is the callback for when the proxy, which is attached
   to the name owner, is destroy (they fell off the bus) we handle
   that and free up our memory too. */
void
proxy_destroyed (GObject * proxy, gpointer user_data)
{
	proxy_t * proxyt = (proxy_t *)user_data;
	proxyt->proxy = NULL; /* Clear this so we don't get a double free on this guy */
	IndicateListener * listener = proxyt->listener;
	IndicateListenerPrivate * priv = INDICATE_LISTENER_GET_PRIVATE(listener);
	priv->proxies = g_list_remove_all(priv->proxies, proxyt);
	proxy_struct_destroy(proxyt);
	return;
}

gboolean
todo_idle (gpointer data)
{
	IndicateListener * listener = INDICATE_LISTENER(data);
	if (listener == NULL) {
		g_error("Listener got lost in todo_idle");
		return FALSE;
	}

	IndicateListenerPrivate * priv = INDICATE_LISTENER_GET_PRIVATE(listener);

	if (priv->proxy_todo->len == 0) {
		/* Basically if we have no todo, we need to stop running.  This
		 * is done this way to make the function error handling simpler
		 * and results in an extra run */
		priv->todo_idle = 0;
		return FALSE;
	}

	proxy_todo_t * todo = &g_array_index(priv->proxy_todo, proxy_todo_t, priv->proxy_todo->len - 1);
	/* Remove the todo list */
	priv->proxy_todo = g_array_remove_index(priv->proxy_todo, priv->proxy_todo->len - 1);

	/* Check to see if we already have this item, if so,
	   we assume that it's signal handler will handle the
	   ServerShow signal.  We're just going to exit this 
	   function. */
	if (TRUE) {
		proxy_t searchitem;
		searchitem.name = todo->name;
		searchitem.connection = todo->bus;

		GList * proxyitem = g_list_find_custom(priv->proxies, &searchitem, proxy_t_equal);
		if (proxyitem != NULL) {
			g_free(todo->name);
			g_free(todo->path);
			return TRUE;
		}
	}

	proxy_t * proxyt = g_new0(proxy_t, 1);
	proxyt->name = todo->name;
	proxyt->path = todo->path;
	proxyt->type = NULL;
	proxyt->property_proxy = NULL;
	proxyt->proxy = NULL;
	proxyt->listener = listener;
	proxyt->indicators = NULL;
	proxyt->hidden = FALSE;
	proxyt->connection = todo->bus;
	proxyt->server.name = todo->name;
	proxyt->server.proxy = NULL;
	proxyt->server.connection = proxyt->connection;
	proxyt->server.max_indicators = priv->max_indicators;

	/* Build the indicators hash */
	proxyt->indicators = g_hash_table_new(g_direct_hash, g_direct_equal);

	/* Build the proxy and ensure that it gets created.  If
	   it gets created we're all happy. */
	g_dbus_proxy_new(priv->session_bus,
	                 G_DBUS_PROXY_FLAGS_NONE,
	                 bus_indicate_interface_info,
	                 proxyt->name,
	                 todo->path,
	                 INDICATE_DBUS_IFACE,
	                 NULL, /* cancel */
	                 prox_ready_cb,
	                 proxyt);

	return TRUE; /* do the next entry */
}

/* Async result of building a proxy */
static void
prox_ready_cb (GObject * object, GAsyncResult * res, gpointer user_data)
{
	proxy_t * proxyt = (proxy_t *)user_data;
	GError * error = NULL;
	GDBusProxy * proxy = g_dbus_proxy_new_finish(res, &error);

	if (error != NULL) {
		g_warning("Unable to create proxy for %s", proxyt->name);
		g_error_free(error);
		return;
	}

	proxyt->proxy = proxy;

	g_signal_connect(G_OBJECT(proxyt->proxy), "notify::g-name-owner", G_CALLBACK(proxy_destroyed), proxyt);
	g_signal_connect(G_OBJECT(proxyt->proxy), "g-signal", G_CALLBACK(proxy_signal_cb), proxyt);

	proxyt->dbus_listener_sub = g_dbus_connection_signal_subscribe(g_dbus_proxy_get_connection(proxy),
	                                                               "org.freedesktop.DBus", /* sender */
	                                                               "org.freedesktop.DBus",
	                                                               "NameOwnerChanged",
	                                                               "/org/freedesktop/DBus", /* path */
	                                                               g_dbus_proxy_get_name(proxy), /* arg0 */
	                                                               G_DBUS_SIGNAL_FLAGS_NONE,
	                                                               proxy_name_owner_changed,
	                                                               proxyt,
	                                                               NULL); /* destroy notify */

	/* Making sure the server has the proxy as well */
	proxyt->server.proxy = proxyt->proxy;

	IndicateListenerPrivate * priv = INDICATE_LISTENER_GET_PRIVATE(proxyt->listener);
	/* Adding into the list of proxies */
	priv->proxies = g_list_prepend(priv->proxies, proxyt);

	/* We're setting the max number of indicators from the default
	   when we detect it.  This should give a reasonable amount of
	   time for listeners to set the default if they want something
	   different from infinite.  Otherwise it'd be easy to miss the
	   first couple. */
	proxyt->server.max_indicators = priv->max_indicators;
	if (proxyt->server.max_indicators != -1) {
		g_dbus_proxy_call(proxyt->proxy,
		                  "SetMaxIndicators",
		                  g_variant_new("(i)", proxyt->server.max_indicators),
		                  G_DBUS_CALL_FLAGS_NONE,
		                  -1, /* timeout */
		                  NULL,
		                  set_max_indicators_cb,
		                  proxyt->server.name);
	}

	indicate_listener_server_get_type(proxyt->listener, &proxyt->server, get_type_initial_cb, proxyt);


	g_dbus_proxy_call(proxyt->proxy,
	                  "GetIndicatorList",
	                  NULL, /* params */
	                  G_DBUS_CALL_FLAGS_NONE,
	                  -1, /* timeout */
	                  NULL,
	                  proxy_get_indicator_list,
	                  proxyt);

	return;
}

/* We need to do the right signal now */
static void
proxy_signal_cb (GDBusProxy * proxy, gchar * sender, gchar * signal, GVariant * params, gpointer user_data)
{
	proxy_t * proxyt = (proxy_t *)user_data;

	if (g_strcmp0(signal, "IndicatorAdded") == 0) {
		guint id; gchar * type;
		g_variant_get(params, "(us)", &id, &type);
		proxy_indicator_added_legacy(id, type, proxyt);
	} else if (g_strcmp0(signal, "IndicatorNew") == 0) {
		guint id;
		g_variant_get(params, "(u)", &id);
		proxy_indicator_added(id, proxyt);
	} else if (g_strcmp0(signal, "IndicatorRemoved") == 0) {
		guint id; gchar * type;
		g_variant_get(params, "(us)", &id, &type);
		proxy_indicator_removed_legacy(id, type, proxyt);
	} else if (g_strcmp0(signal, "IndicatorDelete") == 0) {
		guint id;
		g_variant_get(params, "(u)", &id);
		proxy_indicator_removed(id, proxyt);
	} else if (g_strcmp0(signal, "IndicatorModified") == 0) {
		guint id; gchar * property;
		g_variant_get(params, "(us)", &id, &property);
		proxy_indicator_modified(id, property, proxyt);
	} else if (g_strcmp0(signal, "ServerCountChanged") == 0) {
		guint count;
		g_variant_get(params, "(u)", &count);
		proxy_server_count_changed(count, proxyt);
	} else if (g_strcmp0(signal, "ServerShow") == 0) {
		/* Unused here */
	} else if (g_strcmp0(signal, "ServerHide") == 0) {
		/* Unused here */
	} else {
		g_warning("Unknown signal from server '%s'", signal);
	}

	return;
}

/* A callback function for the getting the type when it is looked
   at initially.  So that we'll send a signal that the server has
   been added so that everyone knows. */
static void
get_type_initial_cb (IndicateListener * listener, IndicateListenerServer * server, const gchar * type, gpointer data)
{
	get_type_cb(listener, server, type, data);

	proxy_t * proxyt = (proxy_t *)data;

	if (!proxyt->hidden && proxyt->type != NULL) {
		g_signal_emit(proxyt->listener, signals[SERVER_ADDED], 0, &proxyt->server, proxyt->type, TRUE);
	}

	return;
}

/* Callback from getting the type of the server.  We're not using
   this directly now, only through get_type_initial_cb right now
   though we might use it more directly later. */
static void
get_type_cb (IndicateListener * listener, IndicateListenerServer * server, const gchar * type, gpointer data)
{
	if (type == NULL) {
		/* This is usually caused by an error getting the type,
		 * which would mean that this isn't an indicator server */
		return;
	}

	proxy_t * proxyt = (proxy_t *)data;

	if (proxyt->type != NULL) {
		g_free(proxyt->type);
		proxyt->type = NULL;
	}
	proxyt->type = g_strdup(type);

	return;
}

/* A call back from setting the max indicators.  We really can't
   do anything about it, so this function is kinda useless. */
static void
set_max_indicators_cb (GObject * object, GAsyncResult * res, gpointer user_data)
{
	GError * error = NULL;
	g_dbus_proxy_call_finish(G_DBUS_PROXY(object), res, &error);

	if (error != NULL) {
		g_warning("Unable to set the max indicators on '%s': %s", (gchar *)user_data, error->message);
		g_error_free(error);
	}

	return;
}

/* Callback from the call to get the indicator list on new
   servers that we've found through introspection.  It takes
   the list and then calls indicator_added on each one. */
static void
proxy_get_indicator_list (GObject * object, GAsyncResult * res, gpointer data)
{
	GError * error = NULL;
	GVariant * retval = g_dbus_proxy_call_finish(G_DBUS_PROXY(object), res, &error);

	if (error != NULL) {
		g_warning("Unable to get indicator list");
		g_error_free(error);
		return;
	}

	proxy_t * proxyt = (proxy_t *)data;

	GVariant * list = g_variant_get_child_value(retval, 0);
	GVariantIter iter;
	g_variant_iter_init(&iter, list);
	gint id;

	while (g_variant_iter_next(&iter, "i", &id)) {
		proxy_indicator_added(id, proxyt);
	}

	if (retval != NULL) {
		g_variant_unref(retval);
	}

	return;
}


/* A fun little wrapper so that we can support the
   signals on the indicator v1 interface.  It just drops
   the type and calls the new function. */
static void
proxy_indicator_added_legacy (guint id, gchar * type, proxy_t * proxyt)
{
	return proxy_indicator_added(id, proxyt);
}

/* Gets called when we get a signal from the server that
   there is a new indicator.  We put it into our list of
   indicators and pass the signal up.  If the server hasn't
   been known about before, we create the appropriate
   structures for it. */
static void
proxy_indicator_added (guint id, proxy_t * proxyt)
{
	g_debug("Proxy Indicator Added");
	g_return_if_fail(proxyt != NULL);

	if (!g_hash_table_lookup(proxyt->indicators, GUINT_TO_POINTER(id))) {
		g_hash_table_insert(proxyt->indicators, GUINT_TO_POINTER(id), GUINT_TO_POINTER(TRUE));
		g_signal_emit(proxyt->listener, signals[INDICATOR_ADDED], 0, &proxyt->server, GUINT_TO_POINTER(id), TRUE);
	}

	return;
}

/* A fun little wrapper so that we can support the
   signals on the indicator v1 interface.  It just drops
   the type and calls the new function. */
static void
proxy_indicator_removed_legacy (guint id, gchar * type, proxy_t * proxyt)
{
	return proxy_indicator_removed(id, proxyt);
}

/* A server removed an indicator.  This function removes all the
   local data structures and then passes the signal up the stack.
   */
static void
proxy_indicator_removed (guint id, proxy_t * proxyt)
{
	g_debug("Proxy Indicator Removed");
	g_return_if_fail(proxyt != NULL);

	if (proxyt->indicators == NULL) {
		g_warning("Oddly we had an indicator removed from an interface that we didn't think had indicators.");
		return;
	}

	if (!g_hash_table_lookup(proxyt->indicators, GUINT_TO_POINTER(id))) {
		g_warning("No indicator %d on '%s'.", id, proxyt->name);
		return;
	}

	g_hash_table_remove(proxyt->indicators, GUINT_TO_POINTER(id));
	g_signal_emit(proxyt->listener, signals[INDICATOR_REMOVED], 0, &proxyt->server, GUINT_TO_POINTER(id), TRUE);

	return;
}

/* This is a signal from the server that a property on an indicator
   has been modified.  We try and find the indicator, convert all
   the parameters to the local ones, and then pass up the signal. */
static void
proxy_indicator_modified (guint id, const gchar * property, proxy_t * proxyt)
{
	g_debug("Proxy Indicator Modified");
	g_return_if_fail(proxyt != NULL);

	if (proxyt->indicators == NULL) {
		g_warning("Oddly we had an indicator modified from an interface that we didn't think had indicators.");
		return;
	}

	if (!g_hash_table_lookup(proxyt->indicators, GUINT_TO_POINTER(id))) {
		g_warning("Can not modify indicator %d with property '%s' as there are no indicators with that id on %s.", id, property, proxyt->name);
		return;
	}

	g_signal_emit(proxyt->listener, signals[INDICATOR_MODIFIED], 0, &proxyt->server, GUINT_TO_POINTER(id), property, TRUE);

	return;
}

/* This function gets called when the dbus count
   signal comes it.  Basically we're just translating
   it into a local signal with the appropraite parameters
   and structures. */
static void
proxy_server_count_changed (guint count, proxy_t * proxyt)
{
	g_debug("Proxy Server Count Changed");
	g_return_if_fail(proxyt != NULL);
	g_signal_emit(proxyt->listener, signals[SERVER_COUNT_CHANGED], 0, &proxyt->server, count, TRUE);
	return;
}

typedef enum _get_property_type get_property_type;
enum _get_property_type {
	PROPERTY_TYPE_VARIANT,
	PROPERTY_TYPE_STRING,
	PROPERTY_TYPE_TIME,
	PROPERTY_TYPE_INT,
	PROPERTY_TYPE_BOOL
};

typedef struct _get_property_t get_property_t;
struct _get_property_t {
	GCallback cb;
	gpointer data;
	IndicateListener * listener;
	IndicateListenerServer * server;
	IndicateListenerIndicator * indicator;
	gchar * property;
	get_property_type type;
};

/* Look at the right align on this comment.  Sweeeeeeeet */
/* A callback from getting a property that takes the string 
   passed across the bus and turning it into something more
   related to what we want on this side.  If it's a time it
   gets converted to a #GTimeVal, if it's an int it goes to
   a gint and if it's a bool we check that too.  This makes
   it nice to work with properties and the listener.
*/
static void
get_property_cb (GObject * object, GAsyncResult * res, gpointer user_data)
{
	get_property_t * get_property_data = (get_property_t *)user_data;
	GError * error = NULL;
	GVariant * retvalue = g_dbus_proxy_call_finish(G_DBUS_PROXY(object), res, &error);

	if (error != NULL) {
		g_warning("Unable to get property data: %s", error->message);
		g_error_free(error);
		return;
	}

	GVariant * value = g_variant_get_child_value(retvalue, 0);
	if (g_variant_is_of_type(value, G_VARIANT_TYPE_VARIANT)) {
		value = g_variant_get_variant(value);
	}

	switch (get_property_data->type) {
	case PROPERTY_TYPE_VARIANT: {
		/* Just pass the GVariant along. */
		indicate_listener_get_property_variant_cb cb =(indicate_listener_get_property_variant_cb)get_property_data->cb;
		cb(get_property_data->listener, get_property_data->server, get_property_data->indicator, get_property_data->property, value, get_property_data->data);
		break;
	}
	case PROPERTY_TYPE_STRING: {
		/* Just pass the string along. */
		indicate_listener_get_property_cb cb = (indicate_listener_get_property_cb)get_property_data->cb;
		cb(get_property_data->listener, get_property_data->server, get_property_data->indicator, get_property_data->property, g_variant_get_string(value, NULL), get_property_data->data);
		break;
	}
	case PROPERTY_TYPE_TIME: {
		/* Convert it to a time val */
		indicate_listener_get_property_time_cb cb = (indicate_listener_get_property_time_cb)get_property_data->cb;
		GTimeVal time;
		if (g_time_val_from_iso8601(g_variant_get_string(value, NULL), &time)) {
			cb(get_property_data->listener, get_property_data->server, get_property_data->indicator, get_property_data->property, &time, get_property_data->data);
		}
		break;
	}
	case PROPERTY_TYPE_INT: {
		/* Take the string and convert it to an integer */
		indicate_listener_get_property_int_cb cb = (indicate_listener_get_property_int_cb)get_property_data->cb;
		cb(get_property_data->listener, get_property_data->server, get_property_data->indicator, get_property_data->property, g_variant_get_int32(value), get_property_data->data);
		break;
	}
	case PROPERTY_TYPE_BOOL: {
		/* Check to see if it's 'true', if not assume that
		   it's false */
		indicate_listener_get_property_bool_cb cb = (indicate_listener_get_property_bool_cb)get_property_data->cb;
		cb(get_property_data->listener, get_property_data->server, get_property_data->indicator, get_property_data->property, g_variant_get_boolean(value), get_property_data->data);
		break;
	}
	}

	if (retvalue != NULL) {
		g_variant_unref(retvalue);
	}
	g_free(get_property_data->property);
	g_free(get_property_data);

	return;
};

/* A small function to take the common list of parameters and
   build a callback structure to hold them all.  Eventually we
   get the data and unwind this structure. */
static void
get_property_helper (IndicateListener * listener, IndicateListenerServer * server, IndicateListenerIndicator * indicator, gchar * property, GCallback callback, gpointer data, get_property_type prop_type)
{
	/* g_debug("get_property_helper: %s %d", property, prop_type); */
	/* TODO: Do we need to somehow refcount the server/indicator while we're waiting on this? */
	get_property_t * get_property_data = g_new0(get_property_t, 1);
	get_property_data->cb = callback;
	get_property_data->data = data;
	get_property_data->listener = listener;
	get_property_data->server = server;
	get_property_data->indicator = indicator;
	get_property_data->property = g_strdup(property);
	get_property_data->type = prop_type;
	
	g_dbus_proxy_call(server->proxy,
	                  "GetIndicatorProperty",
	                  g_variant_new("(us)",
	                                INDICATE_LISTENER_INDICATOR_ID(indicator), property),
	                  G_DBUS_CALL_FLAGS_NONE,
	                  -1, /* timeout */
	                  NULL, /* cancel */
	                  get_property_cb,
	                  get_property_data);

	return;
}

/**
	indicate_listener_get_property_variant:
	@listener: The #IndicateListener representing the connection
	@server: The server that the indicator is on
	@indicator: Which indicator is being queried
	@property: Name of the property to get
	@callback: The callback function to call with the data
	@data: Arbitrary data to give the callback

	A function to get a property from an indicator on a server
	and bring it back locally.  This wraps all the hassle of using
	the DBus API and makes it pretty easy to get properties.

	This function gets the raw GVariant data, without any conversion.
*/
void
indicate_listener_get_property_variant (IndicateListener * listener, IndicateListenerServer * server, IndicateListenerIndicator * indicator, gchar * property, indicate_listener_get_property_variant_cb callback, gpointer data)
{
	return get_property_helper(listener, server, indicator, property, G_CALLBACK(callback), data, PROPERTY_TYPE_VARIANT);
}

/**
	indicate_listener_get_property:
	@listener: The #IndicateListener representing the connection
	@server: The server that the indicator is on
	@indicator: Which indicator is being queried
	@property: Name of the property to get
	@callback: The callback function to call with the data
	@data: Arbitrary data to give the callback

	A function to get a property from an indicator on a server
	and bring it back locally.  This wraps all the hassle of using
	the DBus API and makes it pretty easy to get properties.
*/
void
indicate_listener_get_property (IndicateListener * listener, IndicateListenerServer * server, IndicateListenerIndicator * indicator, gchar * property, indicate_listener_get_property_cb callback, gpointer data)
{
	return get_property_helper(listener, server, indicator, property, G_CALLBACK(callback), data, PROPERTY_TYPE_STRING);
}

/**
	indicate_listener_get_property_time:
	@listener: The #IndicateListener representing the connection
	@server: The server that the indicator is on
	@indicator: Which indicator is being queried
	@property: Name of the property to get
	@callback: The callback function to call with the data
	@data: Arbitrary data to give the callback

	A function to get a property from an indicator on a server
	and bring it back locally.  This wraps all the hassle of using
	the DBus API and makes it pretty easy to get properties.

	Very similar to #indicate_listener_get_property but converts
	the final value into a GTimeVal for easy (and type-safe)
	usage by listeners.
*/
void
indicate_listener_get_property_time (IndicateListener * listener, IndicateListenerServer * server, IndicateListenerIndicator * indicator, gchar * property, indicate_listener_get_property_time_cb callback, gpointer data)
{
	return get_property_helper(listener, server, indicator, property, G_CALLBACK(callback), data, PROPERTY_TYPE_TIME);
}

/**
	indicate_listener_get_property_int:
	@listener: The #IndicateListener representing the connection
	@server: The server that the indicator is on
	@indicator: Which indicator is being queried
	@property: Name of the property to get
	@callback: The callback function to call with the data
	@data: Arbitrary data to give the callback

	A function to get a property from an indicator on a server
	and bring it back locally.  This wraps all the hassle of using
	the DBus API and makes it pretty easy to get properties.

	Very similar to #indicate_listener_get_property but converts
	the final value into a gint for easy (and type-safe)
	usage by listeners.
*/
void
indicate_listener_get_property_int (IndicateListener * listener, IndicateListenerServer * server, IndicateListenerIndicator * indicator, gchar * property, indicate_listener_get_property_int_cb callback, gpointer data)
{
	return get_property_helper(listener, server, indicator, property, G_CALLBACK(callback), data, PROPERTY_TYPE_INT);
}

/**
	indicate_listener_get_property_bool:
	@listener: The #IndicateListener representing the connection
	@server: The server that the indicator is on
	@indicator: Which indicator is being queried
	@property: Name of the property to get
	@callback: The callback function to call with the data
	@data: Arbitrary data to give the callback

	A function to get a property from an indicator on a server
	and bring it back locally.  This wraps all the hassle of using
	the DBus API and makes it pretty easy to get properties.

	Very similar to #indicate_listener_get_property but converts
	the final value into a gboolean for easy (and type-safe)
	usage by listeners.
*/
void
indicate_listener_get_property_bool (IndicateListener * listener, IndicateListenerServer * server, IndicateListenerIndicator * indicator, gchar * property, indicate_listener_get_property_bool_cb callback, gpointer data)
{
	return get_property_helper(listener, server, indicator, property, G_CALLBACK(callback), data, PROPERTY_TYPE_BOOL);
}

/* A callback for asking an indicator to be displayed,
   which is unlikely to fail.  So we're throwing a warning. */
static void 
listener_display_cb (GObject * object, GAsyncResult * res, gpointer user_data)
{
	GError * error = NULL;
	g_dbus_proxy_call_finish(G_DBUS_PROXY(object), res, &error);

	if (error != NULL) {
		g_warning("Listener display caused an error: %s", error->message);
	}
	return;
}

void
indicate_listener_display (IndicateListener * listener, IndicateListenerServer * server, IndicateListenerIndicator * indicator, guint timestamp)
{
	g_dbus_proxy_call(server->proxy,
	                  "ShowIndicatorToUser",
	                  g_variant_new("(uu)", INDICATE_LISTENER_INDICATOR_ID(indicator), timestamp),
	                  G_DBUS_CALL_FLAGS_NONE,
	                  -1, /* timeout */
	                  NULL, /* cancel */
	                  listener_display_cb,
	                  NULL);

	return;
}

/* A callback for saying an indicator is displayed,
   which is unlikely to fail.  So we're throwing a warning. */
static void 
listener_displayed_cb (GObject * object, GAsyncResult * res, gpointer user_data)
{
	GError * error = NULL;
	g_dbus_proxy_call_finish(G_DBUS_PROXY(object), res, &error);

	if (error != NULL) {
		g_warning("Listener displayed caused an error: %s", error->message);
	}
	return;
}

/**
	indicate_listener_displayed:
	@listener: The #IndicateListener representing the connection
	@server: The server that the indicator is on
	@indicator: Which indicator is being displayed
	@displayed: Whether it's being displayed or hidden

	This function tells the client whether we're showing this indicator
	to the user.  This doesn't mean that it's necissarilly visible right
	now, but more that there is a way for the user to get to this item
	individually.
*/
void
indicate_listener_displayed (IndicateListener * listener, IndicateListenerServer * server, IndicateListenerIndicator * indicator, gboolean displayed)
{
	g_dbus_proxy_call(server->proxy,
	                  "IndicatorDisplayed",
	                  g_variant_new("(ub)", INDICATE_LISTENER_INDICATOR_ID(indicator), displayed),
	                  G_DBUS_CALL_FLAGS_NONE,
	                  -1, /* timeout */
	                  NULL, /* cancel */
	                  listener_displayed_cb,
	                  NULL);

	if (!server->interests[INDICATE_INTEREST_INDICATOR_DISPLAY] && displayed) {
		g_warning("It's awful odd that you said in the interest survey you weren't displaying indicators and then you displayed one.  I'm just saying, you've probably confused someone besides me.");
	}

	return;
}

/* Callback data structure */
typedef struct _get_server_prop_data_t get_server_prop_data_t;
struct _get_server_prop_data_t {
	IndicateListener * listener;
	IndicateListenerServer * server;
	indicate_listener_get_server_property_cb callback;
	indicate_listener_get_server_uint_property_cb callback_uint;
	gpointer data;
};

/* This does the actual work of calling the callbacks.  It's split out
   so that we can use it in the DBus function callback but also in the
   case where we already have the cached value. */
static void
get_server_property_work (get_server_prop_data_t * prop_t, GVariant * prop)
{
	if (prop == NULL) {
		if (prop_t->callback == NULL) {
			prop_t->callback_uint(prop_t->listener, prop_t->server, 0, prop_t->data);
		} else {
			prop_t->callback(prop_t->listener, prop_t->server, NULL, prop_t->data);
		}
		return;
	}

	if (g_variant_is_of_type(prop, G_VARIANT_TYPE_STRING) && prop_t->callback != NULL) {
		prop_t->callback(prop_t->listener, prop_t->server, g_variant_get_string(prop, NULL), prop_t->data);
	} else if (g_variant_is_of_type(prop, G_VARIANT_TYPE_OBJECT_PATH) && prop_t->callback != NULL) {
		prop_t->callback(prop_t->listener, prop_t->server, g_variant_get_string(prop, NULL), prop_t->data);
	} else if (g_variant_is_of_type(prop, G_VARIANT_TYPE_UINT32) && prop_t->callback_uint != NULL) {
		prop_t->callback_uint(prop_t->listener, prop_t->server, g_variant_get_uint32(prop), prop_t->data);
	} else {
		g_warning("Really?  This can't happen.  WTF!  %s", g_variant_get_type_string(prop));
	}

	return;
}

/* Callback from getting the property off of DBus.  Try to complete
   the call and then pass it up to the worker, clearing out the
   memory allocations caused here. */
static void
get_server_property_cb (GObject * obj, GAsyncResult * res, gpointer user_data)
{
	GError * error = NULL;
	get_server_prop_data_t * prop_t = (get_server_prop_data_t *)user_data;

	GVariant * prop = g_dbus_connection_call_finish(G_DBUS_CONNECTION(obj), res, &error);
	if (error != NULL) {
		g_warning("Error getting property! %s", error->message);
		g_error_free(error);
	}

	GVariant * unwrap = NULL;
	if (prop != NULL) {
		unwrap = g_variant_get_child_value(prop, 0);
	}

	if (unwrap != NULL && g_variant_is_of_type(unwrap, G_VARIANT_TYPE_VARIANT)) {
		get_server_property_work(prop_t, g_variant_get_variant(unwrap));
	} else {
		get_server_property_work(prop_t, NULL);
	}

	if (prop != NULL) {
		g_variant_unref(prop);
	}
	
	g_object_unref(G_OBJECT(prop_t->listener));
	g_free(prop_t);
	return;
}

/* This is a helper function for all the functions that
   get properties from the server.  They all need to have
   a callback setup with an intermediary data structure
   and this function builds and populates that, then uses
   a custom callback to call their callback */
static void
get_server_property (IndicateListener * listener, IndicateListenerServer * server, indicate_listener_get_server_property_cb callback, indicate_listener_get_server_uint_property_cb callback_uint, const gchar * property_name, gpointer data)
{
	IndicateListenerPrivate * priv = INDICATE_LISTENER_GET_PRIVATE(listener);
	get_server_prop_data_t * prop_t = g_new0(get_server_prop_data_t, 1);

	prop_t->listener = listener;
	prop_t->server = server;
	prop_t->callback = callback;
	prop_t->callback_uint = callback_uint;
	prop_t->data = data;

	g_object_ref(G_OBJECT(listener));

	/* If we don't have the property cached, let's go and find
	   it on the bus. */
	g_dbus_connection_call(priv->session_bus,
	                       g_dbus_proxy_get_name(server->proxy),
	                       g_dbus_proxy_get_object_path(server->proxy),
	                       "org.freedesktop.DBus.Properties",
	                       "Get",
	                       g_variant_new("(ss)", INDICATE_DBUS_IFACE, property_name),
	                       G_VARIANT_TYPE("(v)"),
	                       G_DBUS_CALL_FLAGS_NONE,
	                       -1,
	                       NULL,
	                       get_server_property_cb,
	                       prop_t);

	return;
}

void
indicate_listener_server_get_type (IndicateListener * listener, IndicateListenerServer * server, indicate_listener_get_server_property_cb callback, gpointer data)
{
	return get_server_property(listener, server, callback, NULL, "type", data);
}

void
indicate_listener_server_get_desktop (IndicateListener * listener, IndicateListenerServer * server, indicate_listener_get_server_property_cb callback, gpointer data)
{
	return get_server_property(listener, server, callback, NULL, "desktop", data);
}

void
indicate_listener_server_get_count (IndicateListener * listener, IndicateListenerServer * server, indicate_listener_get_server_uint_property_cb callback, gpointer data)
{
	return get_server_property(listener, server, NULL, callback, "count", data);
}

void
indicate_listener_server_get_menu (IndicateListener * listener, IndicateListenerServer * server, indicate_listener_get_server_property_cb callback, gpointer data)
{
	return get_server_property(listener, server, callback, NULL, "menu", data);
}


const gchar *
indicate_listener_server_get_dbusname (IndicateListenerServer * server)
{
	if (server == NULL) return NULL;
	return server->name;
}

guint
indicate_listener_indicator_get_id (IndicateListenerIndicator * indicator)
{
	return GPOINTER_TO_UINT(indicator);
}

static const gchar *
interest_to_string (IndicateInterests interest)
{
	switch (interest) {
	case INDICATE_INTEREST_SERVER_DISPLAY:
		return INDICATE_INTEREST_STRING_SERVER_DISPLAY;
	case INDICATE_INTEREST_SERVER_SIGNAL:
		return INDICATE_INTEREST_STRING_SERVER_SIGNAL;
	case INDICATE_INTEREST_INDICATOR_DISPLAY:
		return INDICATE_INTEREST_STRING_INDICATOR_DISPLAY;
	case INDICATE_INTEREST_INDICATOR_SIGNAL:
		return INDICATE_INTEREST_STRING_INDICATOR_SIGNAL;
	case INDICATE_INTEREST_INDICATOR_COUNT:
		return INDICATE_INTEREST_STRING_INDICATOR_COUNT;
	default:
		return "";
	}
}

static void
interest_cb (GObject * object, GAsyncResult * res, gpointer user_data)
{
	GError * error = NULL;
	g_dbus_proxy_call_finish(G_DBUS_PROXY(object), res, &error);

	if (error != NULL) {
		g_warning("Unable to configure interest.");
		g_error_free(error);
	}

	return;
}

void
indicate_listener_server_show_interest (IndicateListener * listener, IndicateListenerServer * server, IndicateInterests interest)
{
	if (!(interest > INDICATE_INTEREST_NONE && interest < INDICATE_INTEREST_LAST)) {
		return;
	}

	if (!server->interests[interest]) {
		g_dbus_proxy_call(server->proxy,
		                  "ShowInterest",
		                  g_variant_new("(s)", interest_to_string(interest)),
		                  G_DBUS_CALL_FLAGS_NONE,
		                  -1, /* timeout */
		                  NULL, /* cancel */
		                  interest_cb,
		                  server);
		server->interests[interest] = TRUE;
	}
	return;
}

void
indicate_listener_server_remove_interest (IndicateListener * listener, IndicateListenerServer * server, IndicateInterests interest)
{
	if (server->interests[interest]) {
		g_dbus_proxy_call(server->proxy,
		                  "RemoveInterest",
		                  g_variant_new("(s)", interest_to_string(interest)),
		                  G_DBUS_CALL_FLAGS_NONE,
		                  -1, /* timeout */
		                  NULL, /* cancel */
		                  interest_cb,
		                  server);
		server->interests[interest] = FALSE;
	}
	return;
}

gboolean
indicate_listener_server_check_interest (IndicateListener * listener, IndicateListenerServer * server, IndicateInterests interest)
{
	return server->interests[interest];
}

GType
indicate_listener_server_get_gtype (void)
{
  static GType our_type = 0;
  
  if (our_type == 0)
    our_type = g_pointer_type_register_static ("IndicateListenerServer");

  return our_type;
}

GType
indicate_listener_indicator_get_gtype (void)
{
  static GType our_type = 0;
  
  if (our_type == 0)
    our_type = g_pointer_type_register_static ("IndicateListenerIndicator");

  return our_type;
}

/**
	indicate_listener_set_default_max_indicators:
	@listener: Instance of #IndicateListener to set on.
	@max: The new default number of max indicators.

	This function sets the number that is given to new servers
	when they start for the max number of indicators that you want
	to see.  The client should enforce this number.

	Note: This function WILL NOT reconfigure already recognized
	      servers.  It only affects new servers.
*/
void
indicate_listener_set_default_max_indicators (IndicateListener * listener, gint max)
{
	g_return_if_fail(INDICATE_IS_LISTENER(listener));
	IndicateListenerPrivate * priv = INDICATE_LISTENER_GET_PRIVATE(listener);
	priv->max_indicators = max;
	return;
}

/**
	indicate_listener_set_server_max_indicators:
	@listener: Instance of #IndicateListener to set on.
	@server: Server that the new max should be set on.
	@max: The new number of max indicators.

	This function changes the max number of indicators that
	a server should send.  If the number is different than the
	previous number a signal will be sent to the application to
	adjust then number of indicators that they have.  This is
	sent asynchronously.
*/
void
indicate_listener_set_server_max_indicators (IndicateListener * listener, IndicateListenerServer * server, gint max)
{
	g_return_if_fail(INDICATE_IS_LISTENER(listener));
	g_return_if_fail(server != NULL); /* All we can really check :-/ */

	if (server->max_indicators != max) {
		server->max_indicators = max;
		g_dbus_proxy_call(server->proxy,
		                  "SetMaxIndicators",
		                  g_variant_new("(i)", server->max_indicators),
		                  G_DBUS_CALL_FLAGS_NONE,
		                  -1, /* timeout */
		                  NULL,
		                  set_max_indicators_cb,
		                  server->name);
	}

	return;
}
