/*
 * src/nl-monitor.c     Monitor events
 *
 *		This program is free software; you can redistribute it and/or
 *		modify it under the terms of the GNU General Public License
 *		as published by the Free Software Foundation; either version
 *		2 of the License, or (at your option) any later version.
 *
 * Copyright (c) 2003-2005 Thomas Graf <tgraf@suug.ch>
 */

#include "utils.h"
#include <netlink/route/link.h>
#include <netlink/route/addr.h>

static void obj_input(struct nl_object *obj, void *arg)
{
	struct nl_dump_params dp = {
		.dp_type = NL_DUMP_STATS,
		.dp_fd = stdout,
		.dp_dump_msgtype = 1,
	};

	nl_object_dump(obj, &dp);
}

static int event_input(struct nl_msg *msg, void *arg)
{
	if (nl_msg_parse(msg, &obj_input, NULL) < 0)
		fprintf(stderr, "<<EVENT>> Unknown message type\n");

	return 0;
}

int main(int argc, char *argv[])
{
	struct nl_handle *nlh;
	struct nl_cache *link_cache;
	int err = 1;

	if (nltool_init(argc, argv) < 0)
		return -1;

	nlh = nl_handle_alloc_nondefault(nltool_cbset);
	if (nlh == NULL)
		return -1;

	nl_disable_sequence_check(nlh);

	nl_cb_set(nl_handle_get_cb(nlh), NL_CB_VALID, NL_CB_CUSTOM,
		  event_input, NULL);

	if (argc > 1 && !strcasecmp(argv[1], "-h")) {
		printf("Usage: nl-monitor [<groups>]\n");
		return 2;
	}

	nl_join_groups(nlh, RTMGRP_LINK);

	if (nl_connect(nlh, NETLINK_ROUTE) < 0) {
		fprintf(stderr, "%s\n", nl_geterror());
		goto errout;
	}

	if (nl_join_group(nlh, RTNLGRP_FIB_MAGIC) < 0) {
		fprintf(stderr, "%s\n", nl_geterror());
		goto errout;
	}

	if (nl_join_group(nlh, RTNLGRP_IPV4_ROUTE) < 0) {
		fprintf(stderr, "%s\n", nl_geterror());
		goto errout;
	}

	if ((link_cache = rtnl_link_alloc_cache(nlh)) == NULL) {
		fprintf(stderr, "%s\n", nl_geterror());
		goto errout_close;
	}

	nl_cache_mngt_provide(link_cache);

	while (1)
		nl_recvmsgs_def(nlh);

	nl_cache_destroy_and_free(link_cache);
errout_close:
	nl_close(nlh);
errout:
	return err;
}
