# Copyright 2016-2018 Canonical Ltd.  This software is licensed under the
# GNU Affero General Public License version 3 (see the file LICENSE).

"""Tests for the Region Controller API."""

import http.client
from unittest.mock import call

from django.utils.http import urlencode
from maasserver.enum import NODE_TYPE
from maasserver.models.bmc import Pod
from maasserver.testing.api import (
    APITestCase,
    explain_unexpected_response,
)
from maasserver.testing.factory import factory
from maasserver.utils.converters import json_load_bytes
from maasserver.utils.django_urls import reverse
from maasserver.utils.orm import reload_object
from maastesting.matchers import (
    MockCallsMatch,
    MockNotCalled,
)


class TestRegionControllerAPI(APITestCase.ForUser):
    """Tests for /api/2.0/regioncontrollers/<region>/."""

    def test_handler_path(self):
        self.assertEqual(
            '/MAAS/api/2.0/regioncontrollers/region-name/',
            reverse('regioncontroller_handler', args=['region-name']))

    @staticmethod
    def get_region_uri(region):
        """Get the API URI for `region`."""
        return reverse('regioncontroller_handler', args=[region.system_id])

    def test_PUT_updates_region_controller(self):
        self.become_admin()
        region = factory.make_RegionController()
        zone = factory.make_zone()
        response = self.client.put(
            self.get_region_uri(region), {'zone': zone.name})
        self.assertEqual(http.client.OK, response.status_code)
        self.assertEqual(zone.name, reload_object(region).zone.name)

    def test_PUT_requires_admin(self):
        region = factory.make_RegionController()
        response = self.client.put(self.get_region_uri(region), {})
        self.assertEqual(http.client.FORBIDDEN, response.status_code)

    def test_DELETE_delete_with_force(self):
        self.become_admin()
        vlan = factory.make_VLAN()
        subnet = factory.make_Subnet(vlan=vlan)
        region = factory.make_Node_with_Interface_on_Subnet(
            node_type=NODE_TYPE.REGION_CONTROLLER, subnet=subnet, vlan=vlan)
        ip = factory.make_StaticIPAddress(
            interface=region.interface_set.first())
        factory.make_Pod(ip_address=ip)
        mock_async_delete = self.patch(Pod, "async_delete")
        response = self.client.delete(
            self.get_region_uri(region), QUERY_STRING=urlencode({
                'force': 'true'
            }, doseq=True))
        self.assertEqual(
            http.client.NO_CONTENT, response.status_code,
            explain_unexpected_response(http.client.NO_CONTENT, response))
        self.assertThat(mock_async_delete, MockCallsMatch(call()))

    def test_DELETE_force_not_required_for_pod_region_rack(self):
        self.become_admin()
        vlan = factory.make_VLAN()
        factory.make_Subnet(vlan=vlan)
        rack = factory.make_RegionRackController(vlan=vlan)
        ip = factory.make_StaticIPAddress(
            interface=rack.interface_set.first())
        factory.make_Pod(ip_address=ip)
        mock_async_delete = self.patch(Pod, "async_delete")
        response = self.client.delete(
            self.get_region_uri(rack), QUERY_STRING=urlencode({
                'force': 'true'
            }, doseq=True))
        self.assertEqual(
            http.client.NO_CONTENT, response.status_code,
            explain_unexpected_response(http.client.NO_CONTENT, response))
        self.assertThat(mock_async_delete, MockNotCalled())

    def test_pod_DELETE_delete_without_force(self):
        self.become_admin()
        vlan = factory.make_VLAN()
        subnet = factory.make_Subnet(vlan=vlan)
        region = factory.make_Node_with_Interface_on_Subnet(
            node_type=NODE_TYPE.REGION_CONTROLLER, subnet=subnet, vlan=vlan)
        ip = factory.make_StaticIPAddress(
            interface=region.interface_set.first())
        factory.make_Pod(ip_address=ip)
        mock_async_delete = self.patch(Pod, "async_delete")
        response = self.client.delete(self.get_region_uri(region))
        self.assertEqual(
            http.client.BAD_REQUEST, response.status_code,
            explain_unexpected_response(http.client.BAD_REQUEST, response))
        self.assertThat(mock_async_delete, MockNotCalled())


class TestRegionControllersAPI(APITestCase.ForUser):
    """Tests for /api/2.0/regioncontrollers/."""

    @staticmethod
    def get_region_uri():
        """Get the API URI for `region`."""
        return reverse('regioncontrollers_handler')

    def test_handler_path(self):
        self.assertEqual(
            '/MAAS/api/2.0/regioncontrollers/',
            reverse('regioncontrollers_handler'))

    def test_read_returns_limited_fields(self):
        self.become_admin()
        factory.make_RegionController()
        response = self.client.get(reverse('regioncontrollers_handler'))
        parsed_result = json_load_bytes(response.content)
        self.assertItemsEqual(
            [
                'system_id',
                'hostname',
                'domain',
                'fqdn',
                'architecture',
                'cpu_count',
                'cpu_speed',
                'memory',
                'swap_size',
                'osystem',
                'power_state',
                'power_type',
                'resource_uri',
                'distro_series',
                'interface_set',
                'ip_addresses',
                'zone',
                'status_action',
                'node_type',
                'node_type_name',
                'current_commissioning_result_id',
                'current_testing_result_id',
                'current_installation_result_id',
                'version',
                'commissioning_status',
                'commissioning_status_name',
                'testing_status',
                'testing_status_name',
                'cpu_test_status',
                'cpu_test_status_name',
                'memory_test_status',
                'memory_test_status_name',
                'storage_test_status',
                'storage_test_status_name',
                'other_test_status',
                'other_test_status_name',
                'hardware_info',
                'tag_names',
            ],
            list(parsed_result[0]))
