/*
 * Copyright (C) 2008 Canonical Ltd
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Authored by Neil Jagdish Patel <neil.patel@canonical.com>
 *
 */

#if HAVE_CONFIG_H
#include <config.h>
#endif

#include "nl-icon-tile.h"

#include <glib.h>
#include <glib-object.h>
#include <glib/gi18n.h>
#include <clutter/clutter.h>
#include <clutter-gtk/clutter-gtk.h>

#include "nl-config.h"
#include "nl-defines.h"
#include "nl-texture-frame.h"

G_DEFINE_TYPE (NlIconTile, nl_icon_tile, CTK_TYPE_BUTTON);

#define abs(a) (a > 0 ? a : -(a))

#define NL_ICON_TILE_GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE ((obj),\
  NL_TYPE_ICON_TILE, \
  NlIconTilePrivate))

double press_x = 0.0, press_y = 0.0;

struct _NlIconTilePrivate
{
  ClutterActor *bg;
  ClutterActor *emblem;

  NlIconTileEmblemType type;
};

enum
{
  BEGIN_DRAG_MOVE,
  EMBLEM_CLICKED,

  LAST_SIGNAL
};
static guint32 _tile_signals[LAST_SIGNAL] = { 0, };

/* Globals */
static ClutterActor *sel_texture = NULL;

/* Forwards */
static gboolean on_motion (ClutterActor *actor, ClutterMotionEvent *event);
static gboolean on_enter  (NlIconTile *tile, ClutterCrossingEvent *event);
static gboolean on_leave  (NlIconTile *tile, ClutterCrossingEvent *event);
static gboolean on_button_press (NlIconTile *tile, ClutterButtonEvent *e);
static gboolean on_button_release (NlIconTile *tile, ClutterButtonEvent *e);
static void     on_key_focus_in   (ClutterActor *actor);
static void     on_key_focus_out  (ClutterActor *actor);

/* GObject stuff */
static void
nl_icon_tile_finalize (GObject *object)
{
  NlIconTilePrivate *priv;

  priv = NL_ICON_TILE_GET_PRIVATE (object);

  if (priv->bg)
    {
      g_object_unref (sel_texture);
      priv->bg = NULL;
    }

  if (priv->emblem)
    {
      clutter_actor_unparent (priv->emblem);
      priv->emblem = NULL;
    }


  G_OBJECT_CLASS (nl_icon_tile_parent_class)->finalize (object);
}

static void
get_preferred_width (ClutterActor *actor,
                     gfloat   for_height,
                     gfloat  *min_width,
                     gfloat  *nat_width)
{
  if (min_width)
    *min_width = 160.0;

  if (nat_width)
    *nat_width = 160.0;
}

static void
on_allocate (ClutterActor          *actor,
             const ClutterActorBox *box,
             ClutterAllocationFlags flags)
{
  NlIconTilePrivate *priv = NL_ICON_TILE (actor)->priv;
  ClutterActorBox    child_box;

  CLUTTER_ACTOR_CLASS (nl_icon_tile_parent_class)->allocate (actor, box,flags);

  if (CLUTTER_IS_ACTOR (priv->emblem))
    {
      child_box.x1 = box->x2 - box->x1 - clutter_actor_get_width(priv->emblem)-10;
      child_box.x2 = child_box.x1 + clutter_actor_get_width (priv->emblem);
      child_box.y1 = 10;
      child_box.y2 = child_box.y1 + clutter_actor_get_height (priv->emblem);

      clutter_actor_allocate (priv->emblem, &child_box, flags);
    }
}

static void
on_paint (ClutterActor *actor)
{
  NlIconTilePrivate *priv = NL_ICON_TILE (actor)->priv;

  CLUTTER_ACTOR_CLASS (nl_icon_tile_parent_class)->paint (actor);

  if (CLUTTER_IS_ACTOR (priv->emblem))
    clutter_actor_paint (priv->emblem);
}

static void
on_pick (ClutterActor *actor, const ClutterColor *color)
{
  NlIconTilePrivate *priv = NL_ICON_TILE (actor)->priv;

  CLUTTER_ACTOR_CLASS (nl_icon_tile_parent_class)->pick (actor, color);

  if (CLUTTER_IS_ACTOR (priv->emblem))
    clutter_actor_paint (priv->emblem);
}

static void
on_map (ClutterActor *actor)
{
  NlIconTilePrivate *priv = NL_ICON_TILE (actor)->priv;

  CLUTTER_ACTOR_CLASS (nl_icon_tile_parent_class)->map (actor);

  if (CLUTTER_IS_ACTOR (priv->emblem))
    clutter_actor_map (priv->emblem);
}

static void
on_unmap (ClutterActor *actor)
{
  NlIconTilePrivate *priv = NL_ICON_TILE (actor)->priv;

  CLUTTER_ACTOR_CLASS (nl_icon_tile_parent_class)->unmap (actor);

  if (CLUTTER_IS_ACTOR (priv->emblem))
    clutter_actor_unmap (priv->emblem);
}

static void
nl_icon_tile_class_init (NlIconTileClass *klass)
{
  GObjectClass      *obj_class = G_OBJECT_CLASS (klass);
  ClutterActorClass *act_class = CLUTTER_ACTOR_CLASS (klass);

  obj_class->finalize     = nl_icon_tile_finalize;

  act_class->get_preferred_width = get_preferred_width;
  act_class->motion_event        = on_motion;
  act_class->allocate            = on_allocate;
  act_class->paint               = on_paint;
  act_class->pick                = on_pick;
  act_class->map                 = on_map;
  act_class->unmap               = on_unmap;
  act_class->key_focus_in        = on_key_focus_in;
  act_class->key_focus_out       = on_key_focus_out;

  _tile_signals[BEGIN_DRAG_MOVE] =
    g_signal_new ("begin-drag-move",
                  G_OBJECT_CLASS_TYPE (obj_class),
                  G_SIGNAL_RUN_LAST,
                  G_STRUCT_OFFSET (NlIconTileClass, begin_drag_move),
                  NULL, NULL,
                  g_cclosure_marshal_VOID__VOID,
                  G_TYPE_NONE, 0);

  _tile_signals[EMBLEM_CLICKED] =
    g_signal_new ("emblem-clicked",
                  G_OBJECT_CLASS_TYPE (obj_class),
                  G_SIGNAL_RUN_LAST,
                  G_STRUCT_OFFSET (NlIconTileClass, emblem_clicked),
                  NULL, NULL,
                  g_cclosure_marshal_VOID__VOID,
                  G_TYPE_NONE, 0);


  g_type_class_add_private (obj_class, sizeof (NlIconTilePrivate));
}

static void
nl_icon_tile_init (NlIconTile *tile)
{
  NlIconTilePrivate *priv;
  CtkPadding padding = { 15, 15, 15, 15 };

  priv = tile->priv = NL_ICON_TILE_GET_PRIVATE (tile);
  priv->type = NL_ICON_TILE_EMBLEM_TYPE_HOVER;

  if (!CLUTTER_IS_ACTOR (sel_texture))
    {
      sel_texture = clutter_texture_new_from_file (PKGDATADIR"/icon_selected.png", NULL);
      g_object_ref_sink (sel_texture);
    }
  else
    {
      g_object_ref (sel_texture);
    }

  ctk_actor_set_padding (CTK_ACTOR (tile), &padding);
  priv->bg = nl_texture_frame_new (CLUTTER_TEXTURE (sel_texture),
                                   25, 25, 25, 25);

  ctk_actor_set_background_for_state (CTK_ACTOR (tile), CTK_STATE_SELECTED,
                                      priv->bg);

  priv->bg = nl_texture_frame_new (CLUTTER_TEXTURE (sel_texture),
                                   25, 25, 25, 25);

  ctk_actor_set_background_for_state (CTK_ACTOR (tile), CTK_STATE_PRELIGHT,
                                      priv->bg);

  g_signal_connect (tile, "enter-event", G_CALLBACK (on_enter), NULL);
  g_signal_connect (tile, "leave-event", G_CALLBACK (on_leave), NULL);
  g_signal_connect (tile, "button-press-event",
                    G_CALLBACK (on_button_press), NULL);
  g_signal_connect (tile, "button-release-event",
                    G_CALLBACK (on_button_release), NULL);
}

/*
 * Public methods
 */
ClutterActor *
nl_icon_tile_new (const gchar *title,
                  const gchar *tooltip,
                  GdkPixbuf   *icon)
{
  ClutterActor *icon_tile = NULL;
  CtkImage *image;

  icon_tile = g_object_new (NL_TYPE_ICON_TILE,
                            "label", title,
                            "orientation", CTK_ORIENTATION_VERTICAL,
                            "tooltip-text", tooltip,
                            NULL);

  image = ctk_button_get_image (CTK_BUTTON (icon_tile));
  ctk_image_set_size (image, 64);
  ctk_image_set_from_pixbuf (image, icon);

  return icon_tile;
}

/*
 * Private methods
 */
static gboolean
on_motion (ClutterActor *actor, ClutterMotionEvent *event)
{
  g_return_val_if_fail (NL_IS_ICON_TILE (actor), FALSE);

  if (event->modifier_state & CLUTTER_BUTTON1_MASK && (abs(event->x - press_x) > 3 && abs(event->y - press_y) > 3))
    {
      g_signal_emit_by_name (actor, "begin-drag-move");
      return TRUE;
    }
  return FALSE;
}

static gboolean
on_enter (NlIconTile *tile, ClutterCrossingEvent *event)
{
  g_return_val_if_fail (NL_IS_ICON_TILE (tile), FALSE);

  if (tile->priv->type == NL_ICON_TILE_EMBLEM_TYPE_ALWAYS)
    return FALSE;

  if (CLUTTER_IS_ACTOR (tile->priv->emblem))
    clutter_actor_animate (tile->priv->emblem, CLUTTER_EASE_OUT_SINE, 200,
                           "opacity", 255, NULL);

  return FALSE;
}

static gboolean
on_leave (NlIconTile *tile, ClutterCrossingEvent *event)
{
  g_return_val_if_fail (NL_IS_ICON_TILE (tile), FALSE);

  if (tile->priv->type == NL_ICON_TILE_EMBLEM_TYPE_ALWAYS)
    return FALSE;

  if (CLUTTER_IS_ACTOR (tile->priv->emblem))
    clutter_actor_animate (tile->priv->emblem, CLUTTER_EASE_OUT_SINE, 200,
                           "opacity", 0, NULL);

  return FALSE;
}

static gboolean
on_button_press (NlIconTile *tile, ClutterButtonEvent *event)
{
  NlIconTilePrivate *priv;

  press_x = event->x;
  press_y = event->y;

  g_return_val_if_fail (NL_IS_ICON_TILE (tile), FALSE);
  priv = tile->priv;

  if (priv->emblem)
    {
      gfloat eventx=0, eventy=0;
      gfloat x=0, y=0;
      gfloat width=0, height=0;

      clutter_actor_transform_stage_point (CLUTTER_ACTOR (tile),
                                           event->x, event->y, &eventx, &eventy);

      clutter_actor_get_position (priv->emblem, &x, &y);
      clutter_actor_get_size (priv->emblem, &width, &height);

      if (eventx > x
          && eventx < (x+width)
          && eventy > y
          && eventy < (y+height))
        {
          /*FIXME: Do something to the emblem */
          return TRUE;
        }
    }

  return FALSE;
}

static gboolean
on_button_release (NlIconTile *tile, ClutterButtonEvent *event)
{
  NlIconTilePrivate *priv;

  g_return_val_if_fail (NL_IS_ICON_TILE (tile), FALSE);
  priv = tile->priv;

  if (priv->emblem)
    {
      gfloat eventx=0, eventy=0;
      gfloat x=0, y=0;
      gfloat width=0, height=0;

      clutter_actor_transform_stage_point (CLUTTER_ACTOR (tile),
                                           event->x, event->y, &eventx, &eventy);

      clutter_actor_get_position (priv->emblem, &x, &y);
      clutter_actor_get_size (priv->emblem, &width, &height);

      if (eventx > x
          && eventx < (x+width)
          && eventy > y
          && eventy < (y+height))
        {
          g_signal_emit (tile, _tile_signals[EMBLEM_CLICKED], 0);
          return TRUE;
        }
    }


  return FALSE;
}

static void
on_key_focus_in   (ClutterActor *actor)
{
  CLUTTER_ACTOR_CLASS (nl_icon_tile_parent_class)->key_focus_in (actor);

  ctk_actor_set_state (CTK_ACTOR (actor), CTK_STATE_PRELIGHT);
}

static void
on_key_focus_out  (ClutterActor *actor)
{
  CLUTTER_ACTOR_CLASS (nl_icon_tile_parent_class)->key_focus_out (actor);

  ctk_actor_set_state (CTK_ACTOR (actor), CTK_STATE_NORMAL);
}

/*
 * Public Methods
 */
void
nl_icon_tile_set_emblem (NlIconTile  *tile,
                         NlIconTileEmblemType type,
                         ClutterActor *emblem)
{
  NlIconTilePrivate *priv;

  g_return_if_fail (NL_IS_ICON_TILE (tile));
  priv = tile->priv;

  if (CLUTTER_IS_ACTOR (priv->emblem))
    {
      clutter_actor_unparent (priv->emblem);
      priv->emblem = NULL;
    }
  priv->emblem = emblem;
  priv->type = type;

  if (CLUTTER_IS_ACTOR (priv->emblem))
    {
      CtkActorState state = ctk_actor_get_state (CTK_ACTOR (tile));

      clutter_actor_set_parent (emblem, CLUTTER_ACTOR (tile));
      clutter_actor_set_opacity (emblem,
                                 state == CTK_STATE_PRELIGHT ||
                                 type == NL_ICON_TILE_EMBLEM_TYPE_ALWAYS
                                 ? 255 : 0);
      clutter_actor_show (emblem);
    }

  if (CLUTTER_ACTOR_IS_VISIBLE (tile))
    clutter_actor_queue_redraw (CLUTTER_ACTOR (tile));
}

