# Copyright (c) 2004 Divmod.
# See LICENSE for details.


"""Nevow interface definitions.
"""


from nevow import compy


class IQ(compy.Interface):
    """Interface for querying. Adapters implement this for objects which may
    appear in the stan DOM to allow introspecting the DOM and finding nodes
    with certain qualities.
    """
    def patternGenerator(self, pattern, default=None):
        """Returns a psudeo-Tag which will generate clones of matching
        pattern tags forever, looping around to the beginning when running
        out of unique matches.
        
        If no matches are found, and default is None, raise an exception,
        otherwise, generate clones of default forever.

        You can use the normal stan syntax on the return value.
        
        Useful to find repeating pattern elements. Example rendering function:
        def simpleSequence(context, data):
          pattern = context.patternCloner('item')
          return [pattern(data=element) for element in data]
        """

    def allPatterns(self, pattern):
        """Return a list of all matching pattern tags, cloned.
        
        Useful if you just want to insert them in the output in one
        place.
        
        E.g. the sequence renderer's header and footer are found with this.
        """

    def onePattern(self, pattern):
        """Return a single matching pattern, cloned.
        If there is more than one matching pattern or no matching patterns,
        raise an exception.

        Useful in the case where you want to locate one and only one
        sub-tag and do something with it.
        """

    def keyed(self, key):
        """Locate the node with the key 'key', clone it, call fillSlots(key, clone)
        and return the clone.

        This method lets you effectively locate and mutate a node in the DOM.
        It is useful for setting the data special on a specific node, and also for
        calling fillSlots on a specific node, as well as other node-mutation operations
        such as setting a handler or assigning a class or id to a node.
        """


class IResource(compy.Interface):
    def locateChild(self, ctx, segments):
        """Locate another object which can be adapted to IResource
        Return a tuple of resource, path segments
        """

    def renderHTTP(self, ctx):
        """Render a request
        """


class IRenderer(compy.Interface):
    """A renderer is any instance with a rend method with (context, data) or (data) 
    signature. IRenderer is a marker interface to say that instances should be adapted to
    be called.
    """
    def rend(self, ctx, data):
        """Turn this instance into stan suitable for displaying it in a web page.
        """


class IRendererFactory(compy.Interface):
    """A renderer factory is capable of taking a renderer directive (a string)
    and returning a callable which when called, will render a portion of DOM.
    """
    def renderer(self, context, name):
        """Given a context object and a name, return a callable which responds
        to the signature (context, data) or (data) and returns an object which
        is flattenable.
        """


class IData(compy.Interface):
    """Any python object to be used as model data to be passed
    to view functions. Used for marking the context stack only.
    
    ANY python object is said to implement IData.
    """


class IGettable(compy.Interface):
    def get(self, context):
        """Return the data
        
        Return any object
        """


class ISettable(compy.Interface):
    def set(self, context, data):
        """Set the data

        This might be removed.
        """


class IContainer(compy.Interface):
    def child(self, context, name):
        """Return a conceptual child; an attribute, or a key,
        or the result of a function call.
    
        Returns any object; the result may be adapted to IGettable
        if possible.
        
        Return None if the adaptee does not have a child with the
        given name.
        
        TODO: Maybe returning None is bad, and .child should just
        raise whatever exception is natural
        """


class IRequest(compy.Interface):
    """A web request
    """

class ISerializable(compy.Interface):
    """DEPRECATED. Use nevow.flat.registerFlattener instead of registering
    an ISerializable adapter.
    """
    def serialize(self, context):
        """Serialize the adaptee, with the given context
        stack if necessary.
        """

class IStatusMessage(compy.Interface):
    """A marker interface, which should be set on the user's web session
    to an object which can be cast to a string, which will be shown to the
    user to indicate the status of the most recent operation.
    """


class IHand(compy.Interface):
    """A marker interface which indicates what object the user is currently
    holding in their hand. This is conceptually the "result" of the last operation;
    this interface can be used to mark a status string which indicates whether
    the most recent operation completed successfully, or can be used to hold
    an object which resulted from the most recent operation.
    """


class ICanHandleException(compy.Interface):
    def renderHTTP_exception(self, request, failure):
        """Render an exception to the given request object.
        """

    def renderInlineException(self, context, reason):
        """Return stan representing the exception, to be printed in the page,
        not replacing the page."""


class ICanHandleNotFound(compy.Interface):
    def renderHTTP_notFound(self, request):
        """Render a not found message to the given request.
        """


class IEventMaster(compy.Interface):
    pass


class IDocFactory(compy.Interface):
    def load(self):
        """Return a stan document tree"""


class ISession(compy.Interface):
    """A web session

    You can locate a Session object to represent a unique web session using
    ctx.locate(ISession). This default session implementation uses cookies to
    store a session identifier in the user's browser. 
    
    TODO: Need better docs; what's a session and why and how do you use it
    """


class IRemainingSegments(compy.Interface):
    """During the URL traversal process, requesting this from the context
    will result in a tuple of the segments remaining to be processed.
    
    Equivalent to request.postpath in twisted.web
    """


class ICurrentSegments(compy.Interface):
    """Requesting this from the context will result in a tuple of path segments
    which have been consumed to reach the current Page instance during
    the URL traversal process.

    Equivalent to request.prepath in twisted.web
    """

