/* md.c
 *        Copyright (C) 2002, 2003 Timo Schulz
 *
 * This file is part of OpenCDK.
 *
 * OpenCDK is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * OpenCDK is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with OpenCDK; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#include <stdio.h>

#include "opencdk.h"
#include "main.h"
#include "md.h"

struct cdk_md_hd_s {
    int secure;
    MD_CONTEXT ctx;
    int ctxsize;
    int algo;
    int mdlen;
    byte * asn;
    int asnlen;
    void (*init)( void *c );
    void (*write)( void *c, byte *buf, size_t nbytes );
    void (*final)( void *c );
    byte *(*read)( void *c );
};  


const char *
md5_get_info( int algo, size_t *contextsize,
              byte **r_asnoid, int *r_asnlen, int *r_mdlen,
              void (**r_init)( void *c ),
              void (**r_write)( void *c, byte *buf, size_t nbytes ),
              void (**r_final)( void *c ),
              byte *(**r_read)( void *c ) );

const char *
sha1_get_info( int algo, size_t *contextsize,
               byte **r_asnoid, int *r_asnlen, int *r_mdlen,
               void (**r_init)( void *c ),
               void (**r_write)( void *c, byte *buf, size_t nbytes ),
               void (**r_final)( void *c ),
               byte *(**r_read)( void *c ) );

const char *
rmd160_get_info( int algo, size_t *contextsize,
                 byte **r_asnoid, int *r_asnlen, int *r_mdlen,
                 void (**r_init)( void *c ),
                 void (**r_write)( void *c, byte *buf, size_t nbytes ),
                 void (**r_final)( void *c ),
                 byte *(**r_read)( void *c ) );


cdk_error_t
cdk_md_open( cdk_md_hd_t *r_hd, int algo, unsigned int flags )
{
    cdk_md_hd_t hd;

    if( !r_hd )
        return CDK_Inv_Value;
    hd = flags? cdk_salloc( sizeof *hd, 1 )
        : cdk_calloc( 1, sizeof * hd );
    if( !hd )
        return CDK_Out_Of_Core;
    hd->secure = flags;
    
    switch( algo ) {
      case CDK_MD_MD5:
          md5_get_info( algo, &hd->ctxsize, &hd->asn, &hd->asnlen, &hd->mdlen,
                        &hd->init, &hd->write, &hd->final, &hd->read );
          break;
        
      case CDK_MD_SHA1:
          sha1_get_info( algo, &hd->ctxsize, &hd->asn, &hd->asnlen, &hd->mdlen,
                         &hd->init, &hd->write, &hd->final, &hd->read );
          break;
        
      case CDK_MD_RMD160:
          rmd160_get_info( algo, &hd->ctxsize, &hd->asn, &hd->asnlen,
                           &hd->mdlen, &hd->init, &hd->write, &hd->final,
                           &hd->read );
          break;
        
      default:
          cdk_free( hd );
          return CDK_Inv_Algo;
    }
    memset( &hd->ctx, 0, hd->ctxsize );
    hd->init( &hd->ctx );
    hd->algo = algo;
    *r_hd = hd;
    return 0;
}


void
cdk_md_close( cdk_md_hd_t hd )
{
    if( hd ) {
        cdk_free( hd );
    }
}


int
cdk_md_test_algo( int algo )
{
    switch( algo ) {
      case CDK_MD_MD5   :
      case CDK_MD_SHA1  :
      case CDK_MD_RMD160: return 0;
    }
    return CDK_Inv_Algo;
}


void
cdk_md_write( cdk_md_hd_t hd, const void * buf, size_t buflen )
{
    if( hd )
        hd->write( &hd->ctx, (byte *)buf, buflen );
}


int
cdk_md_final( cdk_md_hd_t hd )
{
    if( hd ) {
        hd->final( &hd->ctx );
        return 0;
    }
    return CDK_Inv_Value;
}


byte *
cdk_md_read( cdk_md_hd_t hd, int algo )
{
    return hd? hd->read( &hd->ctx ) : NULL;
}


cdk_error_t
cdk_md_copy( cdk_md_hd_t *r_new, cdk_md_hd_t hd )
{
    cdk_md_hd_t new;
    
    if( !hd || !r_new )
        return CDK_Inv_Value;
    new = hd->secure ? cdk_salloc( sizeof * hd, 1 )
                     : cdk_calloc( 1, sizeof * hd );
    if( !new )
        return CDK_Out_Of_Core;
    memcpy( new, hd, sizeof * hd );
    new->secure = hd->secure;
    memcpy( &new->ctx, &hd->ctx, hd->ctxsize );
    *r_new = new;
    return 0;
}


int
cdk_md_get_algo_dlen( int algo )
{
    switch( algo ) {
      case CDK_MD_MD5   : return 16;
      case CDK_MD_SHA1  :
      case CDK_MD_RMD160: return 20;
      default           : return 0;
    }
    return 0;
}


int
cdk_md_get_asnoid( int algo, byte * buf, size_t *r_asnlen )
{
    cdk_md_hd_t hd;
    int rc;
    
    rc = cdk_md_open( &hd, algo, 0 );
    if( rc )
        return rc;
    if( buf )
        memcpy( buf, hd->asn, hd->asnlen );
    if( r_asnlen )
        *r_asnlen = hd->asnlen;
    cdk_md_close( hd );
    return 0;
}


int
cdk_md_reset( cdk_md_hd_t hd )
{
    if( hd ) {
        hd->init( &hd->ctx );
        return 0;
    }
    return CDK_Inv_Value;
}


int
cdk_md_get_algo( cdk_md_hd_t hd )
{
    if( hd )
        return hd->algo;
    return 0;
}
