/* sym-cipher.c
 *        Copyright (C) 2003 Timo Schulz
 *
 * This file is part of OpenCDK.
 *
 * OpenCDK is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * OpenCDK is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with OpenCDK; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */


#include <stdio.h>
#include <string.h>

#include "opencdk.h"
#include "main.h"
#include "cipher.h"


struct cdk_cipher_hd_s {
    cipher_spec_t * cipher;
    unsigned int flags;
    byte iv[MAX_BLOCKSIZE]; /* (this should be ulong aligned) */
    byte lastiv[MAX_BLOCKSIZE];
    int  unused; /* in IV */
    void * ctx;
};


cdk_error_t
cdk_cipher_new( cdk_cipher_hd_t *r_hd, int algo, int pgp_sync )
{
    cdk_cipher_hd_t hd;

    if( !r_hd )
        return CDK_Inv_Value;
    if( cdk_cipher_test_algo( algo ) )
        return CDK_Inv_Algo;
    hd = cdk_calloc( 1, sizeof * hd );
    if( !hd )
        return CDK_Out_Of_Core;
    if( pgp_sync )
        hd->flags = 1;
    switch( algo ) {
      case CDK_CIPHER_BLOWFISH: hd->cipher = &cipher_spec_blowfish; break;
      case CDK_CIPHER_TWOFISH:  hd->cipher = &cipher_spec_twofish; break;
      case CDK_CIPHER_3DES:     hd->cipher = &cipher_spec_3des; break;
      case CDK_CIPHER_CAST5:    hd->cipher = &cipher_spec_cast5; break;
      case CDK_CIPHER_AES:      hd->cipher = &cipher_spec_aes; break;
      case CDK_CIPHER_AES192:   hd->cipher = &cipher_spec_aes192; break;
      case CDK_CIPHER_AES256:   hd->cipher = &cipher_spec_aes256; break;
    }
    hd->ctx = cdk_calloc( 1, hd->cipher->contextsize );
    if( !hd->ctx ) {
        cdk_free( hd );
        return CDK_Out_Of_Core;
    }
    *r_hd = hd;
    return 0;
}


cdk_error_t
cdk_cipher_open( cdk_cipher_hd_t *r_hd,
                 int algo, int pgp_sync,
                 const byte * key, size_t keylen,
                 const byte * ivbuf, size_t ivlen )
{
    cdk_cipher_hd_t hd;
    int rc;

    rc = cdk_cipher_new( &hd, algo, pgp_sync );
    if( !rc )
        rc = cdk_cipher_setkey( hd, key, keylen );
    if( !rc )
        rc = cdk_cipher_setiv( hd, ivbuf, ivlen );
    if( rc ) {
        cdk_cipher_close( hd );
        hd = NULL;
    }
    *r_hd = hd;
    return rc;
}
    

void
cdk_cipher_close( cdk_cipher_hd_t hd )
{
    if( !hd )
        return;
    cdk_free( hd->ctx );
    cdk_free( hd );
}


int
cdk_cipher_decrypt( cdk_cipher_hd_t hd, byte * outbuf, const byte *inbuf,
                    size_t nbytes )
{
    byte *ivp;
    u32 temp;
    size_t blocksize;
    cipher_spec_t * c;

    if( !hd )
        return CDK_Inv_Value;
    
    c = hd->cipher;
    blocksize = hd->cipher->blocksize;

    if( nbytes <= hd->unused ) {
        /* short enough to be encoded by the remaining XOR mask */
        /* XOR the input with the IV and store input into IV */
        for( ivp=hd->iv+blocksize - hd->unused; nbytes; nbytes--,hd->unused--){
            temp = *inbuf++;
            *outbuf++ = *ivp ^ temp;
            *ivp++ = temp;   
        }
        return 0;
    }
    
    if( hd->unused ) {
        /* XOR the input with the IV and store input into IV */
        nbytes -= hd->unused;
        for( ivp = hd->iv+blocksize - hd->unused; hd->unused; hd->unused-- ) {
            temp = *inbuf++;
            *outbuf++ = *ivp ^ temp;
            *ivp++ = temp;
        }
    }

    /* now we can process complete blocks */
    while( nbytes >= blocksize ) {
        int i;
        /* encrypt the IV (and save the current one) */
        memcpy( hd->lastiv, hd->iv, blocksize );
        c->encrypt( hd->ctx, hd->iv, hd->iv );
        /* XOR the input with the IV and store input into IV */
        for( ivp = hd->iv, i = 0; i < blocksize; i++ ) {
            temp = *inbuf++;
            *outbuf++ = *ivp ^ temp;
            *ivp++ = temp;   
        }
        nbytes -= blocksize;   
    }

    if( nbytes ) {
        /* process the remaining bytes */
        /* encrypt the IV (and save the current one) */
        memcpy( hd->lastiv, hd->iv, blocksize );
        c->encrypt( hd->ctx, hd->iv, hd->iv );
        hd->unused = blocksize;
        /* and apply the xor */
        hd->unused -= nbytes;
        for(ivp=hd->iv; nbytes; nbytes-- ) {
            temp = *inbuf++;
            *outbuf++ = *ivp ^ temp;
            *ivp++ = temp;   
        }   
    }
    return 0;
}


int
cdk_cipher_encrypt( cdk_cipher_hd_t hd, byte * outbuf, const byte *inbuf,
                    size_t nbytes )
{
    byte *ivp;
    size_t blocksize;
    cipher_spec_t * c;

    if( !hd )
        return CDK_Inv_Value;

    c = hd->cipher;
    blocksize = c->blocksize;
    if( nbytes <= hd->unused ) {
        /* short enough to be encoded by the remaining XOR mask
           XOR the input with the IV and store input into IV */
        for( ivp = hd->iv+c->blocksize - hd->unused; nbytes;
             nbytes--, hd->unused-- )
            *outbuf++ = (*ivp++ ^= *inbuf++);
        return 0;
    }
    if( hd->unused ) {
        /* XOR the input with the IV and store input into IV */
        nbytes -= hd->unused;
        for( ivp = hd->iv+blocksize - hd->unused; hd->unused; hd->unused-- )
            *outbuf++ = (*ivp++ ^= *inbuf++);   
    }
    
    /* now we can process complete blocks */
    while( nbytes >= blocksize ) {
        int i;
        /* encrypt the IV (and save the current one) */
        memcpy( hd->lastiv, hd->iv, blocksize );
        c->encrypt( hd->ctx, hd->iv, hd->iv );
        /* XOR the input with the IV and store input into IV */
        for( ivp = hd->iv,i=0; i < blocksize; i++ )
            *outbuf++ = (*ivp++ ^= *inbuf++);
        nbytes -= blocksize;   
    }
    if( nbytes ) {
        /* process the remaining bytes
           encrypt the IV (and save the current one) */
        memcpy( hd->lastiv, hd->iv, blocksize );
        c->encrypt( hd->ctx, hd->iv, hd->iv );
        hd->unused = blocksize;
        /* and apply the xor */
        hd->unused -= nbytes;
        for( ivp = hd->iv; nbytes; nbytes-- )
            *outbuf++ = (*ivp++ ^= *inbuf++);   
    }
    return 0;
}


void
cdk_cipher_sync( cdk_cipher_hd_t hd )
{
    cipher_spec_t * c;

    if( !hd )
        return;
    if( hd->flags && hd->unused ) {
        c = hd->cipher;
        memmove( hd->iv + hd->unused, hd->iv, c->blocksize - hd->unused );
        memcpy( hd->iv, hd->lastiv + c->blocksize - hd->unused, hd->unused );
        hd->unused = 0;
    }
}


int
cdk_cipher_setiv( cdk_cipher_hd_t hd, const byte *ivbuf, size_t ivlen )
{
    if( !hd )
        return CDK_Inv_Value;

    memset( hd->iv, 0, hd->cipher->blocksize );
    if( ivbuf ) {
        if( ivlen > hd->cipher->blocksize )
            ivlen = hd->cipher->blocksize;
        memcpy( hd->iv, ivbuf, ivlen );
    }
    hd->unused = 0;
    return 0;
}


int
cdk_cipher_setkey( cdk_cipher_hd_t hd, const byte *keybuf, size_t keylen )
{
    if( !hd )
        return CDK_Inv_Value;
    return hd->cipher->setkey( hd->ctx, keybuf, keylen );
}


int
cdk_cipher_get_algo_blklen( int algo )
{
    if( cdk_cipher_test_algo( algo ) )
        return 0;
    switch( algo ) {
      case CDK_CIPHER_TWOFISH:
      case CDK_CIPHER_AES:
      case CDK_CIPHER_AES192:
      case CDK_CIPHER_AES256:
          return 16;
    }
    return 8;
}


int
cdk_cipher_get_algo_keylen( int algo )
{
    if( cdk_cipher_test_algo( algo ) )
        return 0;
    switch( algo ) {
      case CDK_CIPHER_BLOWFISH: return 16;
      case CDK_CIPHER_TWOFISH:  return 32;
      case CDK_CIPHER_3DES:     return 24;
      case CDK_CIPHER_CAST5:    return 16;
      case CDK_CIPHER_AES:      return 16;
      case CDK_CIPHER_AES192:   return 24;
      case CDK_CIPHER_AES256:   return 32;
    }   
    return 0;
}


int
cdk_cipher_test_algo( int algo )
{
    switch( algo ) {
      case CDK_CIPHER_BLOWFISH:
      case CDK_CIPHER_TWOFISH:
      case CDK_CIPHER_3DES:
      case CDK_CIPHER_CAST5:
      case CDK_CIPHER_AES:
      case CDK_CIPHER_AES192:
      case CDK_CIPHER_AES256:
          return 0;
    }
    return CDK_Inv_Algo;
}
