/*
 *  Copyright (c) 2008 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * either version 2, or (at your option) any later version of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include "ParserNG.h"

#include <llvm/Constants.h>
#include <llvm/DerivedTypes.h>
#include <llvm/Type.h>

#include <GTLCore/LLVMBackend/CodeGenerator_p.h>
#include <GTLCore/CompilationMessage.h>
#include <GTLCore/LLVMBackend/ExpressionResult_p.h>
#include <GTLCore/Type.h>
#include <GTLCore/Value.h>

#include "GTLCore/AST/BinaryExpression.h"
#include "GTLCore/AST/Expression.h"
#include "GTLCore/AST/FunctionDeclaration.h"
#include "GTLCore/AST/Tree.h"
#include "GTLCore/AST/UnaryExpression.h"
#include "Compiler.h"
#include "LexerNG.h"
#include "../Debug.h"
#include "GTLCore/Token_p.h"
#include "GTLCore/Function_p.h"

#include "GTLCore/TypesManager.h"
#include "GTLCore/VariablesManager_p.h"
#include <GTLCore/CompilationMessages.h>

namespace AST = GTLCore::AST;
using namespace OpenCTL;

struct ParserNG::Private {
  Compiler* compiler;
  AST::Tree* tree;
};

ParserNG::ParserNG(Compiler* _compiler, LexerNG* _lexer) : ParserBase(_compiler, _lexer), d(new Private)
{
  d->compiler = _compiler;
  d->tree = 0;
}

ParserNG::~ParserNG()
{
  delete d;
}

AST::Tree* ParserNG::parse()
{
  OCTL_DEBUG("Parsing with ParserNG");
  OCTL_ASSERT( not d->tree );
  d->tree = new AST::Tree;
  variablesManager()->startContext();
  // parse module header
  parseModuleHeader();
  // parse module body
  // check if the module has a namespace
  if(currentToken().type == GTLCore::Token::NAMESPACE )
  {
    getNextToken();
    if( isOfType( currentToken(), GTLCore::Token::IDENTIFIER ) )
    {
      setNameSpace( currentToken().string );
      getNextToken();
      isOfType( currentToken(), GTLCore::Token::STARTBRACE );
      getNextToken();
      parseModuleBody();
      isOfType( currentToken(), GTLCore::Token::ENDBRACE );
      getNextToken();
    }
  } else {
      parseModuleBody();
  }
  // Check that we have reach the end of the file
  if( isOfType(currentToken(), GTLCore::Token::END_OF_FILE ) and ( not d->compiler or d->compiler->errorMessages().errors().size() == 0 ) )
  {
    AST::Tree* tree = d->tree;
    d->tree = 0;
    return tree;
  }
  for( std::list<GTLCore::AST::FunctionDeclaration*>::const_iterator it = d->tree->functionsDeclarations().begin();
        it != d->tree->functionsDeclarations().end(); ++it)
  {
    delete (*it)->function();
  }
  delete d->tree;
  d->tree = 0;
  return 0;
}

void ParserNG::parseModuleHeader()
{
  while(true)
  {
    getNextToken(); // Get first token
    switch( currentToken().type )
    {
      case GTLCore::Token::CTLVERSION:
      {
        getNextToken();
        if( isOfType( currentToken(), GTLCore::Token::INTEGER_CONSTANT) )
        {
          if( currentToken().i != 1 )
          {
            reportError( "Unsupported CTL version: " + GTLCore::String::number( currentToken().i ), currentToken() );
          } else {
            checkNextTokenIsSemi();
          }
        }
      }
        break;
      case GTLCore::Token::IMPORT:
      {
        getNextToken();
        if( isOfType( currentToken(), GTLCore::Token::STRING_CONSTANT ) )
        {
          if(not d->compiler->importModule( currentToken().string ) )
          {
            reportError( "Fail to import module '" + currentToken().string + "'.", currentToken() );
          }
        }
        checkNextTokenIsSemi();
      }
        break;
      default:
        return;
    }
  }
}

void ParserNG::parseModuleBody()
{
  while(true)
  {
    switch(currentToken().type)
    {
      case GTLCore::Token::CONST:
        parseConstantDeclaration();
        break;
      case GTLCore::Token::STRUCT:
        parseStructDefinition();
        break;
      case GTLCore::Token::END_OF_FILE:
      case GTLCore::Token::ENDBRACE:
        return;
      default:
      {
        if( currentToken().isFunctionType() )
        {
          parseFunction();
        } else {
            OCTL_DEBUG("unexpected " << GTLCore::Token::typeToString( currentToken().type ) );
            reportUnexpected( currentToken() );
            getNextToken();
        }
      }
    }
  }
}

AST::Statement* ParserNG::parseStatement()
{
  if( isType(currentToken()) )
  {
    return parseVariableDeclaration();
  }
  switch( currentToken().type )
  {
    case GTLCore::Token::CONST:
      return parseVariableDeclaration();
    case GTLCore::Token::STRUCT:
    {
      parseStructDefinition();
      return new AST::DummyStatement();
    }
    case GTLCore::Token::STARTBRACKET:
    case GTLCore::Token::IDENTIFIER:
      return parseExpressionStatement();
    case GTLCore::Token::IF:
      return parseIfStatement();
    case GTLCore::Token::WHILE:
      return parseWhileStatement();
    case GTLCore::Token::FOR:
      return parseForStatement();
    case GTLCore::Token::RETURN:
      return parseReturnStatement();
    case GTLCore::Token::STARTBRACE:
    {
      variablesManager()->startContext();
      AST::StatementsList* statement = appendCurrentContextGarbageCollecting(parseStatementList());
      variablesManager()->endContext();
      return statement;
    }
    case GTLCore::Token::PRINT:
    {
      return parsePrintStatement();
    }
    default:
      if( currentToken().isUnaryOperator() or currentToken().isConstant() )
      {
        return parseExpressionStatement();
      }
      OCTL_DEBUG("unexpected");
      reportUnexpected( currentToken() );
      getNextToken();
      return 0;
  }
}

GTLCore::AST::Tree* ParserNG::tree()
{
  return d->tree;
}
