/*
 *  Copyright (c) 2008 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * either version 2, or (at your option) any later version of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include "AccessorExpression.h"

#include <llvm/Constants.h>
#include <llvm/Instructions.h>

#include "../LLVMBackend/CodeGenerator_p.h"
#include "../LLVMBackend/ExpressionResult_p.h"
#include "../Function.h"
#include "../Debug.h"
#include "../VariableNG_p.h"
#include "../LLVMBackend/Visitor_p.h"
#include "../ScopedName.h"
#include "../Type.h"
#include "../Type_p.h"
#include "../Utils_p.h"

#include "../wrappers/ArrayWrap.h"
#include "../wrappers/StructWrap.h"

#include "GarbageCollectionStatement.h"
#include "../LLVMBackend/ExpressionGenerationContext_p.h"

using namespace GTLCore;
using namespace GTLCore::AST;

//------------------- AccessorExpression -------------------//

LLVMBackend::ExpressionResult AccessorExpression::generateValue( LLVMBackend::GenerationContext& _gc, LLVMBackend::ExpressionGenerationContext& _egc ) const
{
  llvm::Value* ptr_ = pointer( _gc, _egc );
  GTL_ASSERT(ptr_);
  const LLVMBackend::Visitor* visitor = LLVMBackend::Visitor::getVisitorFor( type() );
  GTL_DEBUG("Type = " << *type() << " ptr = " << *ptr_);
  LLVMBackend::ExpressionResult result = visitor->get( _gc, _egc.currentBasicBlock(), ptr_, type() );
  GTL_DEBUG("Result = " << result );
  return result;
}

ExpressionResultSP AccessorExpression::generateValue( GenerationVisitor* _generationVisitor) const
{
  GTL_ABORT("Unimplemented");
  return 0;
}

llvm::BasicBlock* AccessorExpression::affect( LLVMBackend::GenerationContext& _gc, LLVMBackend::ExpressionGenerationContext& _egc, const LLVMBackend::ExpressionResult& _value )
{
  llvm::Value* ptr_ = pointer( _gc, _egc );
  const LLVMBackend::Visitor* visitor = LLVMBackend::Visitor::getVisitorFor( type() );
  return visitor->set( _gc, _egc.currentBasicBlock(), ptr_, type(), _value.value(), _value.type(), allocatedInMemory());
}

//------------------- ArraySizeAccessorExpression -------------------//

ArraySizeAccessorExpression::ArraySizeAccessorExpression(AccessorExpression* _parent )
  : m_parent( _parent )
{
}

ArraySizeAccessorExpression::~ArraySizeAccessorExpression()
{
  delete m_parent;
}

bool ArraySizeAccessorExpression::isConstant() const
{
  return true;
}
const GTLCore::Type* ArraySizeAccessorExpression::type() const
{
  return GTLCore::Type::Integer32;
}


llvm::Value* ArraySizeAccessorExpression::pointer(LLVMBackend::GenerationContext& _gc, LLVMBackend::ExpressionGenerationContext& _egc) const
{
  GTL_DEBUG("ArraySizeAccessorExpression::pointer");
  llvm::Value* val = LLVMBackend::CodeGenerator::accessArraySizePointer(_egc.currentBasicBlock(), m_parent->pointer( _gc, _egc));
  GTL_DEBUG( *val );
  return val;
}

void ArraySizeAccessorExpression::markAsReturnExpression()
{
}

//------------------- StructAccessorExpression -------------------//

StructAccessorExpression::StructAccessorExpression(AccessorExpression* _parent, unsigned int _index) : m_parent(_parent), m_index(_index)
{
#ifdef OPENGTL_ENABLE_DEBUG_OUTPUT
  m_parent->type()->structDataMember( m_index ); // let the assert be triggered inside Type::structDataMember
#endif
}

StructAccessorExpression::~StructAccessorExpression()
{
  delete m_parent;
}

llvm::Value* StructAccessorExpression::pointer(LLVMBackend::GenerationContext& _gc, LLVMBackend::ExpressionGenerationContext& _egc) const
{
  GTL_DEBUG("StructAccessorExpression::pointer");
  std::vector<llvm::Value*> indexes;
  indexes.push_back( _gc.codeGenerator()->integerToConstant(_gc.llvmContext(), INT32_C(0)));
  indexes.push_back( _gc.codeGenerator()->integerToConstant(_gc.llvmContext(), gtl_int32(m_index + STRUCT_FIRST_ELEMENT)));
  llvm::Value* ptr = m_parent->pointer( _gc, _egc);
  llvm::Value* val = llvm::GetElementPtrInst::Create( ptr, indexes.begin(), indexes.end(), "", _egc.currentBasicBlock());
  GTL_DEBUG( val );
  return val;
}

bool StructAccessorExpression::isConstant() const
{
  return m_parent->isConstant();
}

const GTLCore::Type* StructAccessorExpression::type() const
{
  return m_parent->type()->structDataMember(m_index).type();
}

void StructAccessorExpression::markAsReturnExpression()
{
  m_parent->markAsReturnExpression();
}

//------------------- ArrayAccessorExpression -------------------//

ArrayAccessorExpression::ArrayAccessorExpression(AccessorExpression * _parent, Expression* _index) : m_parent(_parent), m_index(_index)
{
}

ArrayAccessorExpression::~ArrayAccessorExpression()
{
  delete m_parent;
  delete m_index;
}

llvm::Value* ArrayAccessorExpression::pointer(LLVMBackend::GenerationContext& _gc, LLVMBackend::ExpressionGenerationContext& _egc) const
{
  GTL_DEBUG("ArrayAccessorExpression::pointer");
  llvm::Value* ptr_ = m_parent->pointer( _gc, _egc);
  GTL_ASSERT(ptr_);
  const LLVMBackend::Visitor* visitor = LLVMBackend::Visitor::getVisitorFor( m_parent->type() );
  llvm::Value* index = m_index->generateValue(_gc, _egc).value();
  llvm::Value* val = visitor->pointerToIndex(_gc, _egc, ptr_, m_parent->type(), index );
  GTL_DEBUG( *val );
  return val;
}

bool ArrayAccessorExpression::isConstant() const
{
  return m_parent->isConstant();
}

const GTLCore::Type* ArrayAccessorExpression::type() const
{
  const LLVMBackend::Visitor* visitor = LLVMBackend::Visitor::getVisitorFor( m_parent->type() );
  const GTLCore::Type* type = visitor->pointerToIndexType( m_parent->type() );
  GTL_ASSERT( type );
  return type;
}

void ArrayAccessorExpression::markAsReturnExpression()
{
  m_parent->markAsReturnExpression();
}

//------------------- VariableAccessorExpression -------------------//

VariableAccessorExpression::VariableAccessorExpression( GTLCore::VariableNG* _variable ) : m_variable(_variable)
{
}

llvm::Value* VariableAccessorExpression::pointer(LLVMBackend::GenerationContext& _gc, LLVMBackend::ExpressionGenerationContext& _egc) const
{
  GTL_DEBUG("VariableAccessorExpression::pointer");
  llvm::Value* val = m_variable->pointer(_egc.currentBasicBlock());
  GTL_DEBUG(*val);
  return val;
}

bool VariableAccessorExpression::isConstant() const
{
  return m_variable->constant();
}

const GTLCore::Type* VariableAccessorExpression::type() const
{
  return m_variable->type();
}

void VariableAccessorExpression::markAsReturnExpression()
{
  m_variable->setAllocatedInMemory( true );
}

bool VariableAccessorExpression::allocatedInMemory() const
{
  return m_variable->allocatedInMemory();
}

llvm::BasicBlock* VariableAccessorExpression::affect( LLVMBackend::GenerationContext& _gc, LLVMBackend::ExpressionGenerationContext& _egc, const LLVMBackend::ExpressionResult& _value )
{
  if( _value.functionResult() and (type()->dataType() == Type::ARRAY or type()->dataType() == Type::STRUCTURE ) and not m_variable->constantPointer() )
  {
    return m_variable->replacePointer( _gc, _egc.currentBasicBlock(), _value.value() );
  } else {
    return AccessorExpression::affect( _gc, _egc, _value );
  }
}

//------------------- FunctionMemberAccessorExpression -------------------//

FunctionMemberAccessorExpression::FunctionMemberAccessorExpression( AccessorExpression * _parent, 
                                          const Type::StructFunctionMember* _member,
                                          const std::list<AST::Expression*>& _arguments )
  : m_parent( _parent ), m_member(_member), m_arguments(_arguments)
{
  m_arguments.push_front( _parent );
}

FunctionMemberAccessorExpression::~FunctionMemberAccessorExpression()
{
  deleteAll( m_arguments );
// don't delete m_parent since it's part of the arguments list
}

bool FunctionMemberAccessorExpression::isConstant() const
{
  return true;
}

const GTLCore::Type* FunctionMemberAccessorExpression::type() const
{
  return m_member->returnType();
}

LLVMBackend::ExpressionResult FunctionMemberAccessorExpression::generateValue( LLVMBackend::GenerationContext& _gc, LLVMBackend::ExpressionGenerationContext& _egc ) const
{
  GTL_DEBUG("Call member funcion: " << m_member->function()->name());
  LLVMBackend::ExpressionResult returnValue = _gc.codeGenerator()->callFunction( _gc, _egc, m_member->function(), m_arguments );
  
  _gc.appendDelayedStatement( new PointerGarbageCollectionStatement( returnValue.value(), returnValue.type() ) );
  
  return returnValue;
}

llvm::Value* FunctionMemberAccessorExpression::pointer(LLVMBackend::GenerationContext& _gc, LLVMBackend::ExpressionGenerationContext& _egc) const
{
  GTL_ABORT("No pointer for function member accessor.");
  return 0;
}

bool FunctionMemberAccessorExpression::allocatedInMemory() const
{
  GTL_ABORT("No memory is allocated for a function member accessor.");
  return false;
}

void FunctionMemberAccessorExpression::markAsReturnExpression()
{
  // There is no easy way to know if an Array/Structure given as argument of a function
  // will or will not be returned by the function, hence the need to mark parameters
  // as part of a return expression
  for( std::list<Expression*>::iterator it = m_arguments.begin();
       it != m_arguments.end(); ++it)
  {
    (*it)->markAsReturnExpression();
  }
}
