/*
 *  Copyright (c) 2009 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * either version 2, or (at your option) any later version of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

// C++ Headers
#include <iostream>
#include <cstdlib>

// GTLCore Headers
#include <GTLCore/String.h>
#include <GTLCore/Metadata/Debug_p.h>

// OpenShiva Headers
#include <OpenShiva/Source.h>
#include <OpenShiva/Version.h>
#include <OpenShiva/Metadata.h>

#define ARG_IS(a,b) argv[ai] == GTLCore::String(a) or argv[ai] == GTLCore::String(b)

void printVersion()
{
  std::cout << OpenShiva::LibraryShortName() << " - " << OpenShiva::LibraryName() << " - " << OpenShiva::LibraryVersionString() << std::endl;
  std::cout << OpenShiva::LibraryCopyright() << std::endl;
  std::cout << "Shiva Version : " << OpenShiva::LanguageVersion() << std::endl;
}
void printHelp()
{
  std::cout << "Usage : shivatester [option] fileName.shiva" << std::endl;
  std::cout << std::endl;
  std::cout << "Options : " << std::endl;
  std::cout << "  -r --run-test           run the function 'run-test'" << std::endl;
  std::cout << "  -c --compare            compare the result of the program with a file stored on disk" << std::endl;
//   std::cout << "  -L --module-dir   add a location where to find modules" << std::endl;
  std::cout << std::endl;
  std::cout << "  -h --help               print this message" << std::endl;
  std::cout << "  -v --version            print the version information" << std::endl;
}

GTLCore::String imageToString( OpenShiva::Source::ImageType imageType )
{
  switch( imageType )
  {
    case OpenShiva::Source::Image1:
      return "Image1";
    case OpenShiva::Source::Image2:
      return "Image2";
    case OpenShiva::Source::Image3:
      return "Image3";
    case OpenShiva::Source::Image4:
      return "Image4";
    case OpenShiva::Source::Image:
      return "Image";
    case OpenShiva::Source::InvalidImage:
      return "InvalidImage";
  }
  return "Error";
}

int main(int argc, char** argv)
{
  GTLCore::String fileName = "";

  for(int ai = 1; ai < argc; ai++)
  {
    if(ARG_IS("-h","--help"))
    {
      printHelp();
      return EXIT_SUCCESS;
    } else if(ARG_IS("-v","--version"))
    {
      printVersion();
      return EXIT_SUCCESS;
    } else {
      fileName = argv[ai];
    }
  }
  
  OpenShiva::Source source;
  source.loadFromFile(fileName);
  
  std::cout << "Name: " << source.name() << std::endl;
  std::cout << "Type: ";
  switch( source.sourceType())
  {
    case OpenShiva::Source::InvalidSource:
      std::cout << "Invalid source";
      break;
    case OpenShiva::Source::Library:
      std::cout << "Libray";
      break;
    case OpenShiva::Source::GeneratorKernel:
      std::cout << "Generator kernel";
      break;
    case OpenShiva::Source::FilterKernel:
      std::cout << "Filter kernel";
      break;
    case OpenShiva::Source::CompositionKernel:
      std::cout << "Composition kernel";
      break;
  }
  std::cout << std::endl;
  if(source.metadata())
  {
    std::cout << "<<<<<<<<<<< Metadata >>>>>>>>>>>" << std::endl;
    GTLCore::Metadata::dumpAll(source.metadata());
  }
  std::cout << "<<<<<<<<<<< evaluatePixel >>>>>>>>>>>" << std::endl;
  std::cout << "Ouptut: " << imageToString( source.outputImageType() ) << std::endl;
  std::cout << "Input: " << source.countInputImages() << std::endl;
  for(int i = 0; i < source.countInputImages(); ++i)
  {
    std::cout << "Input[" << i << "]: " << imageToString(source.inputImageType(i)) << std::endl;
  }
  return EXIT_SUCCESS;
}
