/*************************************************************************
 *
 *  OpenOffice.org - a multi-platform office productivity suite
 *
 *  $RCSfile: dx_polypolygonrasterizer.hxx,v $
 *
 *  $Revision: 1.3 $
 *
 *  last change: $Author: rt $ $Date: 2005/09/07 23:30:30 $
 *
 *  The Contents of this file are made available subject to
 *  the terms of GNU Lesser General Public License Version 2.1.
 *
 *
 *    GNU Lesser General Public License Version 2.1
 *    =============================================
 *    Copyright 2005 by Sun Microsystems, Inc.
 *    901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License version 2.1, as published by the Free Software Foundation.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *    MA  02111-1307  USA
 *
 ************************************************************************/

#ifndef _DXCANVAS_POLYPOLYGONRASTERIZER_HXX
#define _DXCANVAS_POLYPOLYGONRASTERIZER_HXX

#include <canvas/debug.hxx>
#include <canvas/verbosetrace.hxx>

#ifndef _BGFX_POLYGON_B2DPOLYPOLYGONRASTERCONVERTER_HXX
#include <basegfx/polygon/b2dpolypolygonrasterconverter.hxx>
#endif
#ifndef _BGFX_NUMERIC_FTOOLS_HXX
#include <basegfx/numeric/ftools.hxx>
#endif


/* Implementation of a poly-polygon rasterizer class */

namespace dxcanvas
{
    class PolyPolygonRasterizer : public ::basegfx::B2DPolyPolygonRasterConverter
    {
    public:
        /// Determines which spans to render
        enum SpanFilter
        {
            /// Only 'off' spans are touched
            OFF_SPANS=0, // the value 0 is chosen deliberately, to match bool's false

            /// Only 'on' spans are touched
            ON_SPANS=1, // the value 1 is chosen deliberately, to match bool's true

            /// Both span types are touched
            BOTH_SPANS=2
        };

        /** Create a poly-polygon rasterizer, which renders into the
            alpha channel.

            The rasterizer clears the alpha channel to 0 on the areas
            outside of the polygon, and set it to 255 for the inside
            areas (but see the eRenderOnSpans parameter).

        	@param rPoly
            Polygon to raster convert

            @param eSpanFilter
            When ON_SPANS, 'on' spans of the polygon are rendered,
            when OFF_SPANS, the 'off' spans are rendered with the
            given alpha value. Finally, when BOTH_SPANS, both 'on' and
            'off' spans are rendered.
            
            @param nWidth
            Width of bitmap

            @param nHeight
            Height of bitmap

            @param nScanlineStride
            Stride for scanlines (byte offset to get from start of one
            scanline to the start of the next scanline)

            @param pBitmapData
            Actual bitmap data
         */
        PolyPolygonRasterizer( const ::basegfx::B2DPolyPolygon&  rPoly,
                               SpanFilter  						 eSpanFilter,
                               sal_Int32					 	 nWidth,
                               sal_Int32						 nHeight,
                               sal_Int32 						 nScanlineStride,
                               void*							 pBitmapData ) :
            B2DPolyPolygonRasterConverter( rPoly,
                                           ::basegfx::B2DRectangle( 
                                               0, 0,
                                               nWidth, nHeight) ),
            mnWidth( nWidth ),
            mnHeight( nHeight ),
            mnScanlineStride( nScanlineStride ),
            mpBitmapData( pBitmapData ),
            meSpanFilter( eSpanFilter )
        {
            ENSURE_AND_THROW( pBitmapData,
                              "PolyPolygonRasterizer::PolyPolygonRasterizer(): Invalid bitmap data" );
        }

        virtual void span(const double& rfXLeft,
                          const double& rfXRight,
                          sal_Int32 	nY, 
                          bool			bOn)
        {
            // test whether to fill this span at all:
            //
            // We don't have to fill, if span is off bitmap limits, or
            // if meRenderOnSpans tells us to skip on or off spans.
            if( (meSpanFilter == (int)bOn || 
                 meSpanFilter == BOTH_SPANS) && 
                nY >= 0 && 
                nY < static_cast<sal_Int32>(mnHeight)-1 &&
                rfXLeft < mnWidth && rfXRight >= 0 )
            {
                // clip span to bitmap bounds
                const sal_Int32 nStartX( ::std::max( 0,
                                                     ::std::min( mnWidth-1,
                                                                 ::basegfx::fround( rfXLeft ) ) ) );
                const sal_Int32 nEndX  ( ::std::max( 0,
                                                     ::std::min( mnWidth-1,
                                                                 ::basegfx::fround( rfXRight ) ) ) );

                // calc alpha fill value: off spans are set to 0, on
                // spans to 255
                const sal_uInt8 nAlphaValue( bOn*255 );

                // TODO(F3): This is completely bmp-format
                // agnostic, we simply assume PixelFormat32ARGB here!

                // force alpha to calculated value for this span
                sal_uInt8* pCurrPix( (sal_uInt8*)mpBitmapData + 
                                     mnScanlineStride*nY +
                                     4*nStartX + 3 );
                for( sal_Int32 nCurrX( nStartX ); nCurrX<nEndX; ++nCurrX, pCurrPix+=4 )
                {
                    // TODO(F3): Assuming alpha byte at pos 3 for
                    // each pixel
                    *pCurrPix = nAlphaValue;
                }
            }
        }

    private:
        const sal_Int32		mnWidth;
        const sal_Int32		mnHeight;
        const sal_Int32		mnScanlineStride;
        void* const			mpBitmapData;
        const SpanFilter 	meSpanFilter;
    };
}

#endif /* _DXCANVAS_POLYPOLYGONRASTERIZER_HXX */
