/*************************************************************************
 *
 *  OpenOffice.org - a multi-platform office productivity suite
 *
 *  $RCSfile: trivialbufferedfile.cxx,v $
 *
 *  $Revision: 1.5 $
 *
 *  last change: $Author: rt $ $Date: 2005/09/08 04:13:42 $
 *
 *  The Contents of this file are made available subject to
 *  the terms of GNU Lesser General Public License Version 2.1.
 *
 *
 *    GNU Lesser General Public License Version 2.1
 *    =============================================
 *    Copyright 2005 by Sun Microsystems, Inc.
 *    901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License version 2.1, as published by the Free Software Foundation.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *    MA  02111-1307  USA
 *
 ************************************************************************/

#include <string.h>
#include <trivialbufferedfile.hxx>

#ifndef __SGI_STL_ALGORITHM
#include <algorithm>
#endif


namespace configmgr
{

sal_uInt64 OTrivialBufferedFile::getFileSize(rtl::OUString const& _aFileURL) SAL_THROW( (io::IOException) )
{
	osl::DirectoryItem aItem;
	osl::FileBase::RC eError = osl::DirectoryItem::get(_aFileURL, aItem);
	if (eError != osl::FileBase::E_None)
	{
		rtl::OUString aUStr = FileHelper::createOSLErrorString(eError);
		throw io::IOException(aUStr, NULL);
	}
	
	osl::FileStatus aStatus(osl_FileStatus_Mask_FileSize);
	eError = aItem.getFileStatus(aStatus);
	if (eError != osl::FileBase::E_None)
	{
		rtl::OUString aUStr = FileHelper::createOSLErrorString(eError);
		throw io::IOException(aUStr, NULL);
	}
	if (aStatus.isValid(osl_FileStatus_Mask_FileSize))
		return aStatus.getFileSize();
	
	return 0;
}
// C'tor

OTrivialBufferedFile::OTrivialBufferedFile( rtl::OUString const& _sPath )
		: m_pFile( new osl::File(_sPath) ),
		  m_nPointer(0),
		  m_nSize(0),
		  m_bMustRead(true)
{
	m_nSize = this->getFileSize(_sPath);
	m_pBuffer = new sal_Int8[sal_Int32(m_nSize)];
}

/** D'tor
 */

OTrivialBufferedFile::~OTrivialBufferedFile()
{
	m_pFile->close();
	delete m_pFile;
	delete [] m_pBuffer;
}
	
osl::FileBase::RC OTrivialBufferedFile::open( sal_uInt32 uFlags )
{
	osl::FileBase::RC eError = m_pFile->open(uFlags);
	m_nPointer = 0;
	return eError;
}

osl::FileBase::RC OTrivialBufferedFile::close()
{
	osl::FileBase::RC eError = m_pFile->close();
	return eError;
}

osl::FileBase::RC OTrivialBufferedFile::setPos( sal_uInt32 uHow, sal_uInt64 uPos )
{
	OSL_ENSURE(uHow == osl_Pos_Absolut, "OTrivialBufferedFile::setPos() Parameter-Error, can't handle values != osl_Pos_Absolut");
	
	m_nPointer = uPos;
	return osl::FileBase::E_None;
}

osl::FileBase::RC OTrivialBufferedFile::getPos( sal_uInt64& uPos )
{
	uPos = m_nPointer;
	return osl::FileBase::E_None;
}

void OTrivialBufferedFile::readFromFile() SAL_THROW( (io::IOException) )
{
	// POST: This function reads the bytes really from file
	
	sal_uInt64 nReallyRead = 0;
	osl::File::RC eError = m_pFile->read(m_pBuffer, m_nSize, nReallyRead);
	if (eError != osl::File::E_None)
	{
		// ERROR!
		throw io::IOException(FileHelper::createOSLErrorString(eError), NULL);
	}
	OSL_ENSURE(nReallyRead == m_nSize, "Error, read file can't get it's complete file data");
	m_bMustRead = false;
}

osl::FileBase::RC OTrivialBufferedFile::read( void *pBuffer, sal_uInt64 uBytesRequested, sal_uInt64& rBytesRead )
{
	if (m_bMustRead)
		readFromFile();
		
	// OSL_ENSURE(uBytesRequested + m_nPointer <= m_nSize, "Pointer failure");
	// requested size may be greater than the real file size
	rBytesRead = std::min(m_nSize - m_nPointer, uBytesRequested);
	
	memcpy(pBuffer, m_pBuffer + m_nPointer, rBytesRead);
	m_nPointer += rBytesRead;	
	return osl::FileBase::E_None;
}
 
} // namespace configmgr

