/*************************************************************************
 *
 *  OpenOffice.org - a multi-platform office productivity suite
 *
 *  $RCSfile: presentation.cxx,v $
 *
 *  $Revision: 1.9 $
 *
 *  last change: $Author: rt $ $Date: 2005/09/07 20:27:58 $
 *
 *  The Contents of this file are made available subject to
 *  the terms of GNU Lesser General Public License Version 2.1.
 *
 *
 *    GNU Lesser General Public License Version 2.1
 *    =============================================
 *    Copyright 2005 by Sun Microsystems, Inc.
 *    901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License version 2.1, as published by the Free Software Foundation.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *    MA  02111-1307  USA
 *
 ************************************************************************/

#include <canvas/debug.hxx>
#include <presentation.hxx>

#ifndef _RTL_REF_HXX_
#include <rtl/ref.hxx>
#endif
#ifndef _RTL_LOGFILE_HXX_
#include <rtl/logfile.hxx>
#endif
#ifndef _COM_SUN_STAR_DRAWING_XDRAWPAGE_HPP_
#include <com/sun/star/drawing/XDrawPage.hpp>
#endif
#ifndef _COM_SUN_STAR_BEANS_XPROPERTYSET_HPP_
#include <com/sun/star/beans/XPropertySet.hpp>
#endif
#ifndef _COM_SUN_STAR_BEANS_PROPERTYVALUE_HPP_
#include <com/sun/star/beans/PropertyValue.hpp>
#endif
#ifndef _COM_SUN_STAR_UTIL_XMODIFYLISTENER_HPP_
#include <com/sun/star/util/XModifyListener.hpp>
#endif
#ifndef _COM_SUN_STAR_AWT_XPAINTLISTENER_HPP_
#include <com/sun/star/awt/XPaintListener.hpp>
#endif
#ifndef _COM_SUN_STAR_AWT_SYSTEMPOINTER_HPP_ 
#include <com/sun/star/awt/SystemPointer.hpp>
#endif
#ifndef _COM_SUN_STAR_PRESENTATION_XSLIDESHOWLISTENER_HPP_
#include <com/sun/star/presentation/XSlideShowListener.hpp>
#endif
#ifndef _COM_SUN_STAR_PRESENTATION_XSLIDESHOWVIEW_HPP_
#include <com/sun/star/presentation/XSlideShowView.hpp>
#endif
#ifndef _COM_SUN_STAR_ANIMATIONS_TRANSITIONTYPE_HPP_
#include <com/sun/star/animations/TransitionType.hpp>
#endif
#ifndef _COM_SUN_STAR_ANIMATIONS_TRANSITIONSUBTYPE_HPP_
#include <com/sun/star/animations/TransitionSubType.hpp>
#endif

#ifndef _CPPUHELPER_COMPBASE2_HXX_
#include <cppuhelper/compbase2.hxx>
#endif
#ifndef _COMPHELPER_BROADCASTHELPER_HXX_
#include <comphelper/broadcasthelper.hxx>
#endif
#ifndef _COMPHELPER_SEQUENCE_HXX_
#include <comphelper/sequence.hxx>
#endif
#ifndef COMPHELPER_INC_COMPHELPER_LISTENERNOTIFICATION_HXX
#include <comphelper/listenernotification.hxx>
#endif

#ifndef _CPPCANVAS_SPRITECANVAS_HXX
#include <cppcanvas/spritecanvas.hxx>
#endif
#ifndef _CPPCANVAS_VCLFACTORY_HXX
#include <cppcanvas/vclfactory.hxx>
#endif
#ifndef _CPPCANVAS_BASEGFXFACTORY_HXX
#include <cppcanvas/basegfxfactory.hxx>
#endif

#ifndef _BGFX_POINT_B2IPOINT_HXX
#include <basegfx/point/b2ipoint.hxx>
#endif
#ifndef _BGFX_POINT_B2DPOINT_HXX
#include <basegfx/point/b2dpoint.hxx>
#endif
#ifndef _BGFX_POLYGON_B2DPOLYGON_HXX
#include <basegfx/polygon/b2dpolygon.hxx>
#endif
#ifndef _BGFX_MATRIX_B2DHOMMATRIX_HXX
#include <basegfx/matrix/b2dhommatrix.hxx>
#endif
#ifndef _BGFX_POLYGON_B2DPOLYGONTOOLS_HXX
#include <basegfx/polygon/b2dpolygontools.hxx>
#endif
#ifndef _BGFX_POLYGON_B2DPOLYPOLYGONTOOLS_HXX
#include <basegfx/polygon/b2dpolypolygontools.hxx>
#endif
#ifndef _BGFX_TOOLS_CANVASTOOLS_HXX
#include <basegfx/tools/canvastools.hxx>
#endif

#ifndef _DEBUG_HXX
#include <tools/debug.hxx>
#endif 
#ifndef _SV_SVAPP_HXX 
#include <vcl/svapp.hxx>
#endif

#include <unoviewcontainer.hxx>
#include <shapeimporter.hxx>
#include <transitionfactory.hxx>
#include <eventmultiplexer.hxx>
#include <usereventqueue.hxx>
#include <eventqueue.hxx>
#include <activitiesqueue.hxx>
#include <activitiesfactory.hxx>
#include <interruptabledelayevent.hxx>
#include <slide.hxx>
#include <tools.hxx>
#include <unoview.hxx>
#include <slidebitmap.hxx>
#include <rehearsetimingsactivity.hxx>
#include <waitsymbol.hxx>
#include "comphelper/scopeguard.hxx"

#include "boost/utility.hpp"
#include <boost/bind.hpp>
#include <map>
#include <memory>
#include <vector>
#include <list>
#include <iterator>
#include <algorithm>


using namespace ::com::sun::star;

// -----------------------------------------------------------------------------

namespace presentation
{
    namespace internal
    {
        namespace 
        {
            //
            // ShowListeners declaration
            // ============================================================

            struct SlideEvent : public lang::EventObject
            {
                sal_Int32   nOldSlideIndex;
                sal_Int32   nNewSlideIndex;
                bool        bShowEnded;
            };

            typedef ::comphelper::OListenerContainerBase< ::com::sun::star::presentation::XSlideShowListener,
                                                          SlideEvent >   ShowListeners_Base;

            class ShowListeners : public ShowListeners_Base,
                                  private boost::noncopyable
            {
            public:
                ShowListeners( ::osl::Mutex& rMutex ) :
                    ShowListeners_Base( rMutex )
                {
                }

            protected:
                virtual bool implNotify( const uno::Reference< ::com::sun::star::presentation::XSlideShowListener >& rListener,
                                         const SlideEvent&                                                           rEvent ) throw( uno::Exception )
                {
                    if( rEvent.bShowEnded )
                        rListener->showEnded();
                    else 
                        rListener->slideChange( rEvent.nOldSlideIndex,
                                                rEvent.nNewSlideIndex );

                    return true; // continue calling listeners
                }
            };
            typedef ::boost::shared_ptr< ShowListeners >    ShowListenersPtr; 
        }


        //
        // Presentation_Impl declaration
        // ============================================================

        class Presentation_Impl : private boost::noncopyable
        {
        public:
            Presentation_Impl( const uno::Reference< uno::XComponentContext >& xContext );
            
            void dispose();

            // ==========================================================
            // implementations of Presentation class methods
            // ==========================================================

            bool nextEffect();

            bool previousSlide();

            bool nextSlide();

            bool startShapeActivity( const uno::Reference< drawing::XShape >& xShape );

            bool stopShapeActivity( const uno::Reference< drawing::XShape >& xShape );

            bool pause( bool bPauseShow );

            bool displaySlide( sal_Int32 nSlideIndex );

            sal_Int32 getCurrentSlideIndex();

            bool prefetch( const uno::Sequence< uno::Reference< drawing::XDrawPage > >&         aSlideSequence, 
                           const uno::Sequence< uno::Reference< animations::XAnimationNode > >& aRootNodeSequence, 
                           const uno::Sequence< beans::PropertyValue >&                         aShowProperties );

            bool setProperty( const beans::PropertyValue& aShowProperty );

            bool addView( const uno::Reference< ::com::sun::star::presentation::XSlideShowView >& xView );

            bool removeView( const uno::Reference< ::com::sun::star::presentation::XSlideShowView >& xView );

            void addSlideShowListener( const uno::Reference< ::com::sun::star::presentation::XSlideShowListener >& xListener );

            void removeSlideShowListener( const uno::Reference< ::com::sun::star::presentation::XSlideShowListener >& xListener );

            void addShapeEventListener( const uno::Reference< ::com::sun::star::presentation::XShapeEventListener >&    xListener, 
                                        const uno::Reference< drawing::XShape >&                                        xShape );

            void removeShapeEventListener( const uno::Reference< ::com::sun::star::presentation::XShapeEventListener >& xListener, 
                                           const uno::Reference< drawing::XShape >&                                     xShape );

            void setShapeCursor( const uno::Reference< drawing::XShape >&   xShape, 
                                 sal_Int16                                  nPointerShape );

            bool update( double* pNextTimeout );

            // ==========================================================
            // Public methods, which are called from local helper classes
            // ==========================================================

            /** Notify that the transition phase of the current slide
                has ended.

                The life of a slide has three phases: the transition
                phase, when the previous slide vanishes, and the
                current slide becomes visible, the shape animation
                phase, when shape effects are running, and the phase
                after the last shape animation has ended, but before
                the next slide transition starts.

                This method notifies the end of the first phase.
             */
            void notifySlideTransitionEnded();

            /** Notify that the shape animation phase of the current slide
                has ended.

                The life of a slide has three phases: the transition
                phase, when the previous slide vanishes, and the
                current slide becomes visible, the shape animation
                phase, when shape effects are running, and the phase
                after the last shape animation has ended, but before
                the next slide transition starts.

                This method notifies the end of the second phase.
             */
            void notifySlideAnimationsEnded();

            /** Notify that the slide has ended.

                The life of a slide has three phases: the transition
                phase, when the previous slide vanishes, and the
                current slide becomes visible, the shape animation
                phase, when shape effects are running, and the phase
                after the last shape animation has ended, but before
                the next slide transition starts.

                This method notifies the end of the third phase.
             */
            void notifySlideEnded();

            /** Notify that the view has changed.

                Currently, this method call denotes a change in view
                size and/or scaling.
             */
            void notifyViewChange();

            // ==========================================================
            // Public types
            // ==========================================================

            typedef ::std::map< uno::Reference< drawing::XShape >,
                                ::std::vector< 
                                    uno::Reference< 
                                        ::com::sun::star::presentation::XShapeEventListener > > >   ShapeEventListenerMap;
            typedef ::std::map< uno::Reference< drawing::XShape >,
                                sal_Int16 >                                                         ShapeCursorMap;

        private:
            /// Get current slide (empty shared_ptr if none)
            const SlideSharedPtr& getCurrentSlide() const;

            /// Get previous slide (empty shared_ptr if none)
            const SlideSharedPtr& getPreviousSlide() const;

            /** Show next slide, register end event

                This method is called from nextSlide() etc., and from
                SlideAnimationsEndEvent. It also registers an end
                event at the Slide to trigger the next slide.
                Furthermore, it broadcasts the slide change event to
                any XSlide
             */
            bool showNext( sal_Int32 nPage );

            /// Display/hide wait symbol on all views
            void setWaitState( const bool bOn );
            /// sets mouse cursor related to flags.
            void updateMouseCursor();

            /** Prepare a slide transition
                
                This method registers all necessary events and
                activities for a slide transition.

                @return the slide change activity, or NULL for no
                transition effect
             */
            ActivitySharedPtr createSlideTransition( const uno::Reference< drawing::XDrawPage >&    xDrawPage,
                                                     const SlideSharedPtr&                          rLeavingSlide,
                                                     const SlideSharedPtr&                          rEnteringSlide,
                                                     const EventSharedPtr&                          rTransitionEndEvent );

            /** Create a new slide.
             */
            SlideSharedPtr createSlide( sal_Int32 nPage );

            /** Prefetch given slide into slide buffer.
             */
            bool prefetchSlide( sal_Int32 nPage );

            /** if CURRENT_SLIDE is valid, preload NEXT_SLIDE into
                slide buffer, or, if NEXT_SLIDE is already valid,
                preload PREVIOUS_SLIDE
             */
            bool prefetchNextSlides();

            /** Stop currently running show
             */
            void stopShow();

            struct SlideBufferEntry
            {
                SlideBufferEntry() : mpSlide(), mnSlideIndex(0) {} // for proper init

                SlideSharedPtr  mpSlide;
                int             mnSlideIndex;
            };
            
            // holds current, previous and next slide:
            enum {
                CURRENT_SLIDE=0, PREVIOUS_SLIDE=1, NEXT_SLIDE=2,
                SLIDE_BUFFER_SIZE=3
            };
            SlideBufferEntry maSlideBuffer[SLIDE_BUFFER_SIZE];

            // iterates over all buffered slides
            template <typename func_type>
            void for_each_buffered_slide( func_type const & func )
            {
                for( std::size_t pos = 0; pos < SLIDE_BUFFER_SIZE; ++pos ) {
                    if (maSlideBuffer[pos].mpSlide.get() != 0)
                        func( maSlideBuffer[pos].mpSlide );
                }
            }

            typedef ::std::vector< uno::Reference< drawing::XDrawPage > >           VectorOfDrawPages;
            typedef ::std::vector< uno::Reference< animations::XAnimationNode > >   VectorOfRootNodes;

            VectorOfDrawPages                       maDrawPages;
            VectorOfRootNodes                       maAnimationRootNodes;

            UnoViewContainer                        maViewContainer;

            // Mutex for mpListeners (superfluous, but the class
            // demands it)
            ::osl::Mutex                            maMutex;
            ShowListenersPtr                        mpListeners;

            /// map of vectors, containing all registered listeners for a shape
            ShapeEventListenerMap                   maShapeEventListeners;
            /// map of sal_Int16 values, specifying the mouse cursor for every shape
            ShapeCursorMap                          maShapeCursors;

            ::comphelper::OptionalValue< RGBColor > maUserPaintColor;

            boost::shared_ptr<canvas::tools::ElapsedTime> mpPresTimer;
            EventQueue                              maEventQueue;
            EventMultiplexer                        maEventMultiplexer;
            ActivitiesQueue                         maActivitiesQueue;
            UserEventQueue                          maUserEventQueue;
            
            ::comphelper::OptionalValue< double >   maAutomaticAdvancementTimeout;
            boost::shared_ptr<RehearseTimingsActivity> mrehearseTimingsActivity;
            boost::shared_ptr<WaitSymbol>           mwaitSymbol;
            
            uno::Reference<uno::XComponentContext>  mxComponentContext;

            bool                                    mbWaitState;
            bool                                    mbImageAnimationsAllowed;
            bool                                    mbNoSlideTransitions;
            bool                                    mbMouseVisible;
            bool                                    mbForceManualAdvance;
            bool                                    mbShowPaused;
        };


        namespace
        {
            class SlideViewLayer : public ViewLayer,
                                   private boost::noncopyable
            {
            public:
                SlideViewLayer( const ::cppcanvas::SpriteCanvasSharedPtr& rCanvas ) :
                    mpCanvas( rCanvas )
                {
                }

                virtual ::cppcanvas::CustomSpriteSharedPtr createSprite( const ::basegfx::B2DSize& rSpriteSizePixel ) const
                {
                    return mpCanvas->createCustomSprite( rSpriteSizePixel );
                }

                virtual double getSpritePriority( double nSpritePrio ) const
                {
                    // TODO(F2): Sprite/Layer/View priority
                    return nSpritePrio;
                }

                virtual void setPriority( double nPrio )
                {
                    // TODO(F2): Sprite/Layer/View priority
                }

                virtual ::cppcanvas::CanvasSharedPtr getCanvas() const
                {
                    return mpCanvas;
                }

            private:
                ::cppcanvas::SpriteCanvasSharedPtr  mpCanvas;
            };

            class SlideView : public UnoView,
                              private boost::noncopyable
            {
            public:
                /** Create View implementation.

                    @param xView
                    View to render to.
                */
                SlideView( const uno::Reference< ::com::sun::star::presentation::XSlideShowView >& xView,
                           EventQueue&                                                             rEventQueue,
                           Presentation_Impl&                                                      rPresentation );
                ~SlideView();

                virtual ViewLayerSharedPtr createViewLayer() const;
                virtual bool updateScreen() const;
                virtual bool isContentDestroyed() const;
                virtual void clear() const;
                virtual ::cppcanvas::CanvasSharedPtr getCanvas() const;
                virtual ::basegfx::B2DHomMatrix getTransformation() const;
                virtual void setViewSize( const ::basegfx::B2DSize& rSize );
                virtual void setClip( const ::basegfx::B2DPolyPolygon& rClip );
                virtual void setMouseCursor( sal_Int16 nPointerShape );
                virtual ::cppcanvas::CustomSpriteSharedPtr createSprite( const ::basegfx::B2DSize& rSpriteSizePixel ) const;
                virtual double getSpritePriority( double nSpritePrio ) const;
                virtual void setPriority( double nPrio );
                virtual uno::Reference< ::com::sun::star::presentation::XSlideShowView > getUnoView() const;

            private:
                typedef ::cppu::WeakComponentImplHelper2< util::XModifyListener,
                                                          awt::XPaintListener >  SlideViewListener_Base;

                /** Listener class, implementing all functionality

                    Because we have to mix shared_ptr and
                    uno::Reference at the SlideView class (shared_ptr
                    for the View interface, uno::Reference semantics
                    for the listeners), we have to use that proverbial
                    extra level of indirection here. The SlideView
                    delegates all methods to this Listener object,
                    itself holding only an ::rtl::Reference to
                    it. When the SlideShow is deleted, it disposes the
                    Listener, and releasing its reference.
                 */
                class Listener : public ::comphelper::OBaseMutex,
                                 public SlideViewListener_Base,
                                 private boost::noncopyable
                {
                public:
                    /** Create Listener implementation.

                        @param xView
                        View to render to.
                    */
                    Listener( const uno::Reference< ::com::sun::star::presentation::XSlideShowView >& xView,
                              EventQueue&                                                             rEventQueue,
                              Presentation_Impl&                                                      rPresentation ) :
                        SlideViewListener_Base( m_aMutex ),
                        mxView( xView ),
                        mpCanvas(),
                        mrEventQueue( rEventQueue ),
                        mrPresentation( rPresentation ),
                        maUnitRectPoly( ::basegfx::tools::createPolygonFromRect(
                                            ::basegfx::B2DRectangle(0.0,0.0,
                                                                    1.0,1.0 ) ) ),
                        maClip(),
                        maViewTransform(),
                        maUserSize( 1.0, 1.0 ), // default size: one-by-one rectangle
                        mbContentValid( false )
                    {
                        ENSURE_AND_THROW( mxView.is(),
                                          "SlideView::SlideView(): Invalid view" );

                        mpCanvas = ::cppcanvas::VCLFactory::getInstance().createSpriteCanvas( xView->getCanvas() );
                        ENSURE_AND_THROW( mpCanvas.get(),
                                          "SlideView::SlideView(): Could not create cppcanvas" );

                        ::basegfx::unotools::homMatrixFromAffineMatrix( maViewTransform,
                                                                        xView->getTransformation() );

                        // register listeners with XSlideShowView
                        mxView->addTransformationChangedListener( this );
                        mxView->addPaintListener( this );

                        // set new transformation
                        updateCanvas();
                    }

                    ViewLayerSharedPtr createViewLayer() const
                    {
                        ::osl::MutexGuard aGuard( m_aMutex );

                        ENSURE_AND_THROW( mpCanvas.get(),
                                          "SlideView::createViewLayer(): Disposed" );

                        return ViewLayerSharedPtr( 
                            new SlideViewLayer( mpCanvas ) );
                    }

                    bool updateScreen() const
                    {
                        ::osl::MutexGuard aGuard( m_aMutex );

                        ENSURE_AND_THROW( mpCanvas.get(),
                                          "SlideView::updateScreen(): Disposed" );

                        return mpCanvas->updateScreen( false );
                    }

                    void clear() const
                    {
                        ::osl::MutexGuard aGuard( m_aMutex );

                        ENSURE_AND_THROW( mpCanvas.get(),
                                          "SlideView::clear(): Disposed" );

                        // only clear the _complete_ view, if content
                        // became invalid. Otherwise, since the
                        // View::clear does not respect our clip,
                        // partial updates might be spoiled (in that
                        // the whole view gets cleared, but only a
                        // small rectangle updated with slide content)
                        if( !mbContentValid )
                            mxView->clear();

                        mbContentValid = true;
                    }

                    bool isContentDestroyed() const
                    {
                        return !mbContentValid;
                    }

                    ::cppcanvas::CanvasSharedPtr getCanvas() const
                    {
                        ::osl::MutexGuard aGuard( m_aMutex );

                        ENSURE_AND_THROW( mpCanvas.get(),
                                          "SlideView::getCanvas(): Disposed" );

                        return mpCanvas;
                    }

                    ::basegfx::B2DHomMatrix getTransformation() const
                    {
                        ::osl::MutexGuard aGuard( m_aMutex );

                        ::basegfx::B2DHomMatrix aMatrix;
                        
                        aMatrix.scale( 1.0/maUserSize.getX(),
                                       1.0/maUserSize.getY() );

                        return maViewTransform * aMatrix;
                    }

                    void setViewSize( const ::basegfx::B2DSize& rSize )
                    {
                        ::osl::MutexGuard aGuard( m_aMutex );

                        maUserSize = rSize;

                        updateCanvas();

                        // view resized, have to update whole area
                        mbContentValid = false;
                    }

                    ::cppcanvas::CustomSpriteSharedPtr createSprite( const ::basegfx::B2DSize& rSpriteSizePixel ) const          
                    {
                        ::osl::MutexGuard aGuard( m_aMutex );

                        ENSURE_AND_THROW( mpCanvas.get(),
                                          "SlideView::createSprite(): Disposed" );

                        return mpCanvas->createCustomSprite( rSpriteSizePixel );
                    }

                    void setClip( const ::basegfx::B2DPolyPolygon& rClip )
                    {
                        if( rClip.count() == 0 &&
                            maClip.count() == 0 )
                        {
                            // bail out early for no-ops
                            return;
                        }

                        maClip = rClip;

                        if( maClip.areControlPointsUsed() )
                            maClip = ::basegfx::tools::adaptiveSubdivideByAngle( maClip );

                        // normalize polygon, preparation for clipping
                        // in updateCanvas()
                        maClip = ::basegfx::tools::correctOrientations( maClip );
                        maClip = ::basegfx::tools::removeAllIntersections(maClip);
                        maClip = ::basegfx::tools::removeNeutralPolygons(maClip, sal_True);

                        updateCanvas();
                    }

                    void setMouseCursor( sal_Int16 nPointerShape )
                    {
                        mxView->setMouseCursor( nPointerShape );
                    }

                    double getSpritePriority( double nSpritePrio ) const
                    {
                        ::osl::MutexGuard aGuard( m_aMutex );

                        // TODO(F2): Sprite/Layer/View priority
                        return nSpritePrio;
                    }

                    void setPriority( double nPrio )
                    {
                        ::osl::MutexGuard aGuard( m_aMutex );

                        // TODO(F2): Sprite/Layer/View priority
                    }

                    uno::Reference< ::com::sun::star::presentation::XSlideShowView > getUnoView() const
                    {
                        ::osl::MutexGuard aGuard( m_aMutex );

                        return mxView;
                    }

                    virtual void SAL_CALL dispose() throw (uno::RuntimeException)
                    {
                        ::osl::MutexGuard aGuard( m_aMutex );

                        // additionally, also de-register from XSlideShowView
                        if( mxView.is() )
                        {
                            mxView->removeTransformationChangedListener( this );
                            mxView->removePaintListener( this );

                            // release references
                            disposing( lang::EventObject() );
                        }

                        // call parent
                        WeakComponentImplHelperBase::dispose();
                    }

                    // XEventListener
                    virtual void SAL_CALL disposing( const lang::EventObject& Source ) throw (uno::RuntimeException)
                    {
                        mxView.clear();
                        mpCanvas.reset();
                    }

                    // XModifyListener
                    virtual void SAL_CALL modified( const lang::EventObject& aEvent ) throw (uno::RuntimeException)
                    {
                        ::osl::MutexGuard aGuard( m_aMutex );

                        ENSURE_AND_THROW( mxView.is(),
                                          "SlideView::modified(): Disposed, but event received from XSlideShowView?!" );

                        // view transformation changed. Modify canvas matrix, then
                        ::basegfx::unotools::homMatrixFromAffineMatrix( maViewTransform,
                                                                        mxView->getTransformation() );

                        updateCanvas();

                        // view resized, have to update whole area
                        mbContentValid = false;

                        // notify view change. Don't call notifyViewChange 
                        // directly, this might not be the main thread!
                        mrEventQueue.addEvent(
                            makeEvent( ::boost::bind( &Presentation_Impl::notifyViewChange,
                                                      ::boost::ref( mrPresentation ) ) ) );
                    }

                    // XPaintListener
                    virtual void SAL_CALL windowPaint( const awt::PaintEvent& e ) throw (uno::RuntimeException)
                    {
                        ::osl::MutexGuard aGuard( m_aMutex );

                        ENSURE_AND_THROW( mpCanvas.get(),
                                          "SlideView::windowPaint(): Disposed, but event received from XSlideShowView?!" );

                        // TODO(F2): Don't always call update screen here, this might tear running 
                        // animations. Implementation should synchronize with ActivitiesQueue here, 
                        // and only update when no animations are active.

                        // TODO(Q2): Watch out here, windowPaint()
                        // might not be called from the main thread
                        // (asking for SolarMutex trouble...)

                        // repaint _whole_ screen, not only changed parts
                        mpCanvas->updateScreen( true );
                    }

                protected:
                    virtual ~Listener() {} // ref-counted UNO object, we self-destruct

                private:
                    void updateCanvas()
                    {
                        ENSURE_AND_THROW( mpCanvas.get(),
                                          "SlideView::updateCanvasTransform(): Disposed" );

                        // setup canvas transformation
                        // ===========================
                        mpCanvas->setTransformation( getTransformation() );

                        // setup canvas clipping
                        // =====================

                        // our view displays a one-by-one rectangle, modified by maUserSize 
                        // (i.e. maViewTransform alone transforms a one-by-one rect to final device 
                        // coordinates). Thus, to display maUnitRectPoly correctly under the new total 
                        // view transform, we need to scale it up by maUserSize
                        ::basegfx::B2DPolyPolygon aClipPoly( maUnitRectPoly );

                        ::basegfx::B2DHomMatrix aMatrix;
                        aMatrix.scale( maUserSize.getX(),
                                       maUserSize.getY() );

                        aClipPoly.transform( aMatrix );

                        // now clip with user-supplied clip poly (if any)
                        if( maClip.count() )
                        {
                            // no need to normalize aClipPoly, it's
                            // just created positively oriented,
                            // without any self-intersections
                            aClipPoly.append( maClip );
                            aClipPoly = ::basegfx::tools::removeAllIntersections(aClipPoly);
                            aClipPoly = ::basegfx::tools::removeNeutralPolygons(aClipPoly, sal_False);
                        }

                        ::cppcanvas::PolyPolygonSharedPtr pPolyPoly( 
                            ::cppcanvas::BaseGfxFactory::getInstance().createPolyPolygon( mpCanvas, aClipPoly ) );

                        if( pPolyPoly.get() )
                            mpCanvas->setClip( pPolyPoly );
                    }

                    uno::Reference< ::com::sun::star::presentation::XSlideShowView >    mxView;
                    ::cppcanvas::SpriteCanvasSharedPtr                                  mpCanvas;

                    EventQueue&                                                         mrEventQueue;
                    Presentation_Impl&                                                  mrPresentation;

                    const ::basegfx::B2DPolygon                                         maUnitRectPoly;
                    ::basegfx::B2DPolyPolygon                                           maClip;

                    ::basegfx::B2DHomMatrix                                             maViewTransform;
                    ::basegfx::B2DSize                                                  maUserSize;

                    mutable bool                                                        mbContentValid;
                };

                typedef ::rtl::Reference< Listener > ListenerRef;
                
                // Ref-counted Listener
                ListenerRef         mpListener;
            };

            SlideView::SlideView( const uno::Reference< ::com::sun::star::presentation::XSlideShowView >&  xView,
                                  EventQueue&                                                              rEventQueue,
                                  Presentation_Impl&                                                       rPresentation ) :
                mpListener( new Listener( xView, rEventQueue, rPresentation ) )
            {
                ENSURE_AND_THROW( mpListener.is(),
                                  "SlideView::SlideView(): Could not create Listener" );
            }

            SlideView::~SlideView()
            {
                // dispose listener, making it release all references
                mpListener->dispose();
            }

            ViewLayerSharedPtr SlideView::createViewLayer() const
            {
                return mpListener->createViewLayer();
            }

            bool SlideView::updateScreen() const
            {
                return mpListener->updateScreen();
            }

            bool SlideView::isContentDestroyed() const
            {
                return mpListener->isContentDestroyed();
            }

            void SlideView::clear() const
            {
                mpListener->clear();
            }
            
            ::cppcanvas::CanvasSharedPtr SlideView::getCanvas() const
            {
                return mpListener->getCanvas();
            }

            ::basegfx::B2DHomMatrix SlideView::getTransformation() const
            {
                return mpListener->getTransformation();
            }

            void SlideView::setViewSize( const ::basegfx::B2DSize& rSize )
            {
                mpListener->setViewSize( rSize );
            }

            void SlideView::setClip( const ::basegfx::B2DPolyPolygon& rClip )
            {
                mpListener->setClip( rClip );
            }

            void SlideView::setMouseCursor( sal_Int16 nPointerShape )
            {
                mpListener->setMouseCursor( nPointerShape );
            }

            ::cppcanvas::CustomSpriteSharedPtr SlideView::createSprite( const ::basegfx::B2DSize& rSpriteSizePixel ) const
            {
                return mpListener->createSprite( rSpriteSizePixel );
            }

            double SlideView::getSpritePriority( double nSpritePrio ) const
            {
                return mpListener->getSpritePriority( nSpritePrio );
            }

            void SlideView::setPriority( double nPrio )
            {
                mpListener->setPriority( nPrio );
            }

            uno::Reference< ::com::sun::star::presentation::XSlideShowView > SlideView::getUnoView() const
            {
                return mpListener->getUnoView();
            }

            /** Event handler for slide end events.

                This handler is registered for slide animation end
                events at the global EventMultiplexer, and internally
                queues an event that in turn calls
                Presentation_Impl::notifySlideAnimationsEnded()
             */
            class SlideAnimationsEndHandler : public EventHandler,
                                              private boost::noncopyable
            {
            public:
                SlideAnimationsEndHandler( Presentation_Impl& rPresentation,
                                           EventQueue&        rEventQueue ) :
                    mrPresentation( rPresentation ),
                    mrEventQueue( rEventQueue )
                {
                }

                virtual void dispose()
                {
                }

                virtual bool handleEvent()
                {
                    // DON't call
                    // Presentation_Impl::notifySlideAnimationsEnded()
                    // directly, but queue an event. handleEvent()
                    // might be called from
                    // e.g. Presentation_Impl::showNext(), and
                    // notifySlideAnimationsEnded() must not be called
                    // in recursion.
                    mrEventQueue.addEvent( 
                        makeEvent( ::boost::bind( &Presentation_Impl::notifySlideAnimationsEnded,
                                                  ::boost::ref( mrPresentation ) ) ) );
                    return true;
                }

            private:
                Presentation_Impl&  mrPresentation;
                EventQueue&         mrEventQueue;
            };
        }


        //
        // Presentation_Impl implementation
        // ============================================================

        Presentation_Impl::Presentation_Impl( const uno::Reference< uno::XComponentContext >& xContext ) :
            maSlideBuffer(),
            maDrawPages(),
            maAnimationRootNodes(),
            maViewContainer(),
            maMutex(),
            mpListeners( new ShowListeners( maMutex ) ),
            maShapeEventListeners(),
            maShapeCursors(),
            maUserPaintColor(),
            mpPresTimer( new canvas::tools::ElapsedTime ),
            maEventQueue( mpPresTimer ),
            maEventMultiplexer( maEventQueue ),
            maActivitiesQueue( mpPresTimer, maEventMultiplexer ),
            maUserEventQueue( maEventMultiplexer,
                              maEventQueue ),
            maAutomaticAdvancementTimeout(),
            mxComponentContext( xContext ),
            mbWaitState(false),
            mbImageAnimationsAllowed( true ),
            mbNoSlideTransitions( false ),
            mbMouseVisible( true ),
            mbForceManualAdvance( false ),
            mrehearseTimingsActivity(),
            mwaitSymbol(),
            mbShowPaused( false )
        {
            maEventMultiplexer.addSlideAnimationsEndHandler( 
                EventHandlerSharedPtr( new SlideAnimationsEndHandler( *this,
                                                                      maEventQueue ) ) );
        }

        void Presentation_Impl::dispose()
        {
            mxComponentContext.clear();

            if (mwaitSymbol.get() != 0) {
                mwaitSymbol->dispose();
                mwaitSymbol.reset();
            }
            
            if (mrehearseTimingsActivity.get() != 0) {
                mrehearseTimingsActivity->dispose();
                mrehearseTimingsActivity.reset();
            }
            
            maUserEventQueue.clear();
            maEventMultiplexer.clear();
            maActivitiesQueue.clear();
            maEventQueue.clear();

            maShapeCursors.clear();
            mpListeners.reset();

            maViewContainer.clear();
            maAnimationRootNodes.clear();
            maDrawPages.clear();
            
            for( std::size_t pos = 0; pos < SLIDE_BUFFER_SIZE; ++pos )
                maSlideBuffer[pos].mpSlide.reset();
        }

        ActivitySharedPtr Presentation_Impl::createSlideTransition( const uno::Reference< drawing::XDrawPage >& xDrawPage,
                                                                    const SlideSharedPtr&                       rLeavingSlide,
                                                                    const SlideSharedPtr&                       rEnteringSlide,
                                                                    const EventSharedPtr&                       rTransitionEndEvent )
        {
            ENSURE_AND_THROW( !maViewContainer.empty(),
                              "Presentation_Impl::createSlideTransition(): No views" );

            // return empty transition, if slide transitions
            // are disabled.
            if( mbNoSlideTransitions )
                return ActivitySharedPtr();

            // retrieve slide change parameters from XDrawPage
            uno::Reference< beans::XPropertySet > xPropSet( xDrawPage,
                                                            uno::UNO_QUERY_THROW );

            sal_Int16 nTransitionType;
            if( !extractValue( nTransitionType,
                               xPropSet->getPropertyValue( 
                                   ::rtl::OUString( 
                                       RTL_CONSTASCII_USTRINGPARAM("TransitionType") ) ),
                               ShapeSharedPtr(),
                               LayerManagerSharedPtr() ) )
            {
                OSL_ENSURE( false,
                            "Presentation_Impl::createSlideTransition(): "
                            "Could not extract slide transition type from XDrawPage" );
                return ActivitySharedPtr();
            }

            sal_Int16 nTransitionSubType;
            if( !extractValue( nTransitionSubType,
                               xPropSet->getPropertyValue( 
                                   ::rtl::OUString( 
                                       RTL_CONSTASCII_USTRINGPARAM("TransitionSubtype") ) ),
                               ShapeSharedPtr(),
                               LayerManagerSharedPtr() ) )
            {
                OSL_ENSURE( false,
                            "Presentation_Impl::createSlideTransition(): "
                            "Could not extract slide transition subtype from XDrawPage" );
                return ActivitySharedPtr();
            }

            bool bTransitionDirection;
            if( !extractValue( bTransitionDirection,
                               xPropSet->getPropertyValue( 
                                   ::rtl::OUString( 
                                       RTL_CONSTASCII_USTRINGPARAM("TransitionDirection") ) ),
                               ShapeSharedPtr(),
                               LayerManagerSharedPtr() ) )
            {
                OSL_ENSURE( false,
                            "Presentation_Impl::createSlideTransition(): "
                            "Could not extract slide transition direction from XDrawPage" );
                return ActivitySharedPtr();
            }

            RGBColor aTransitionFadeColor;
            if( !extractValue( aTransitionFadeColor,
                               xPropSet->getPropertyValue( 
                                   ::rtl::OUString( 
                                       RTL_CONSTASCII_USTRINGPARAM("TransitionFadeColor") ) ),
                               ShapeSharedPtr(),
                               LayerManagerSharedPtr() ) )
            {
                OSL_ENSURE( false,
                            "Presentation_Impl::createSlideTransition(): "
                            "Could not extract slide transition fade color from XDrawPage" );
                return ActivitySharedPtr();
            }

            ::rtl::OUString aSoundURL;
            if( !extractValue( aSoundURL,
                               xPropSet->getPropertyValue( 
                                   ::rtl::OUString( 
                                       RTL_CONSTASCII_USTRINGPARAM("Sound") ) ),
                               ShapeSharedPtr(),
                               LayerManagerSharedPtr() ) )
            {
                OSL_ENSURE( false,
                            "Presentation_Impl::createSlideTransition(): "
                            "Could not determine transition sound effect URL from XDrawPage" );
                return ActivitySharedPtr();
            }

            SoundPlayerSharedPtr pSoundPlayer;
            if( aSoundURL.getLength() )
            {
                try
                {
                    pSoundPlayer = SoundPlayer::create( maEventMultiplexer,
                                                        aSoundURL,
                                                        mxComponentContext);
                }
                catch( lang::NoSupportException& ) 
                {
                    // catch possible exceptions from SoundPlayer, since
                    // being not able to playback the sound is not a hard
                    // error here (still, the slide transition should be
                    // shown).
                }
            }

            SlideChangeAnimationSharedPtr pTransition( 
                TransitionFactory::createSlideTransition( rLeavingSlide,
                                                          rEnteringSlide,
                                                          nTransitionType,
                                                          nTransitionSubType,
                                                          bTransitionDirection,
                                                          aTransitionFadeColor,
                                                          pSoundPlayer) );

            if( !pTransition.get() )
                return ActivitySharedPtr(); // no transition effect
                                            // has been
                                            // generated. Normally,
                                            // that means that simply
                                            // no transition is set on
                                            // this slide.

            // add all current views to the transition
            ::std::for_each( maViewContainer.begin(),
                             maViewContainer.end(),
                             ::boost::bind( &SlideChangeAnimation::addView,
                                            ::boost::ref( pTransition ),
                                            _1 ) );

            double nTransitionDuration;
            if( !extractValue( nTransitionDuration,
                               xPropSet->getPropertyValue( 
                                   ::rtl::OUString( 
                                       RTL_CONSTASCII_USTRINGPARAM("TransitionDuration") ) ),
                               ShapeSharedPtr(),
                               LayerManagerSharedPtr() ) )
            {
                OSL_ENSURE( false,
                            "Presentation_Impl::createSlideTransition(): "
                            "Could not extract slide transition duration from XDrawPage" );
                return ActivitySharedPtr();
            }

            return ActivitySharedPtr( 
                ActivitiesFactory::createSimpleActivity( 
                    ActivitiesFactory::CommonParameters(
                        rTransitionEndEvent,
                        maEventQueue,
                        maActivitiesQueue,
                        nTransitionDuration,
                        5,
                        false,
                        1.0,
                        0.0,
                        0.0,
                        ShapeSharedPtr(),
                        LayerManagerSharedPtr() ),
                    pTransition,
                    true ) );
        }

        namespace
        {
            class ShapeListenerRegisterer
            {
            public:
                ShapeListenerRegisterer( SlideSharedPtr& rSlide ) :
                    mrSlide( rSlide )
                {
                }

                void operator()( const Presentation_Impl::ShapeEventListenerMap::value_type& rEntry ) const
                {
                    ::std::for_each( rEntry.second.begin(),
                                     rEntry.second.end(),
                                     ::boost::bind( &Slide::addShapeEventListener,
                                                    ::boost::ref( mrSlide ),
                                                    _1,
                                                    ::boost::cref( rEntry.first ) ) );
                }

            private:
                SlideSharedPtr& mrSlide;
            };
        }

        SlideSharedPtr Presentation_Impl::createSlide( sal_Int32 nPage )
        {
            ENSURE_AND_THROW( 0 <= nPage && static_cast<int>(maDrawPages.size()) > nPage,
                              "Presentation_Impl::createSlide(): Index out of range" );

            ENSURE_AND_THROW( 0 <= nPage && static_cast<int>(maAnimationRootNodes.size()) > nPage,
                              "Presentation_Impl::createSlide(): Index out of range" );

            const uno::Reference< drawing::XDrawPage >& xDrawPage( maDrawPages[ nPage ] );

            const uno::Reference< animations::XAnimationNode >& xRootNode( maAnimationRootNodes[nPage] );

            if( !xDrawPage.is() )
                return SlideSharedPtr(); 

            SlideSharedPtr pSlide( new Slide( xDrawPage, xRootNode,
                                              nPage,
                                              maEventQueue,
                                              maActivitiesQueue,
                                              maEventMultiplexer,
                                              maUserEventQueue,
                                              mxComponentContext ) );

            // add all registered views also to the slide
            ::std::for_each( maViewContainer.begin(),
                             maViewContainer.end(),
                             ::boost::bind(&Slide::addView,
                                           ::boost::ref( pSlide ),
                                           _1) );
            // add all registered shape listeners also to the shape
            ::std::for_each( maShapeEventListeners.begin(),
                             maShapeEventListeners.end(),
                             ShapeListenerRegisterer(pSlide) );

            // add all registered shape cursors also to the shape
            ::std::for_each( maShapeCursors.begin(),
                             maShapeCursors.end(),
                             ::boost::bind( &Slide::setShapeCursor,
                                            ::boost::ref( pSlide ),
                                            ::boost::bind( 
                                                ::std::select1st< ShapeCursorMap::value_type >(),
                                                _1 ),
                                            ::boost::bind( 
                                                ::std::select2nd< ShapeCursorMap::value_type >(),
                                                _1 ) ) );
            
            // forward applicable slideshow attributes to newly
            // generated slide
            pSlide->setUserPaintColor( maUserPaintColor );
            pSlide->setImageAnimationsAllowed( mbImageAnimationsAllowed );

            updateMouseCursor();
            
            // prefetch show content (reducing latency for slide 
            // bitmap and effect start later on)
            pSlide->prefetchShow();

            return pSlide;
        }

        bool Presentation_Impl::prefetchSlide( sal_Int32 nPage )
        {
            if( nPage < 0 || nPage >= static_cast<sal_Int32>(maDrawPages.size()) )
                return false;

            // update slide buffer
            const sal_Int32 nPreviousSlide( nPage-1 );
            const sal_Int32 nNextSlide( nPage+1 );

            // Temporary slide buffer. We extract slides from
            // maSlideBuffer into that, possibly reordering them
            // (which is the reason why we don't perform that inplace)
            SlideBufferEntry aTempBuffer[SLIDE_BUFFER_SIZE];

            // Go over maSlideBuffer, and extract slides we can reuse
            // into aTempBuffer
            bool bCurrentSlideFound( false );
            for( int i=0; i<SLIDE_BUFFER_SIZE; ++i )
            {
                const sal_Int32 nCurrIndex( maSlideBuffer[i].mnSlideIndex );

                // is the current buffer entry valid in the first
                // place? If not, no need to check if maybe the index
                // matches...
                if( maSlideBuffer[i].mpSlide.get() != NULL )
                {
                    if( nCurrIndex == nPage )
                    {
                        aTempBuffer[CURRENT_SLIDE] = maSlideBuffer[i];
                        bCurrentSlideFound = true;
                    }
                    else if( nCurrIndex == nPreviousSlide )
                    {
                        aTempBuffer[PREVIOUS_SLIDE] = maSlideBuffer[i];
                    }
                    else if( nCurrIndex == nNextSlide )
                    {
                        aTempBuffer[NEXT_SLIDE] = maSlideBuffer[i];
                    }
                }
            }

            if( !bCurrentSlideFound )
            {
                // fetch current slide
                aTempBuffer[CURRENT_SLIDE].mpSlide = createSlide( nPage );
                aTempBuffer[CURRENT_SLIDE].mnSlideIndex = nPage;
            }
            
            // now dump back the result
            ::std::copy( aTempBuffer, aTempBuffer+SLIDE_BUFFER_SIZE, maSlideBuffer );

            return true;
        }

        bool Presentation_Impl::prefetchNextSlides()
        {
            if( getCurrentSlide().get() )
            {
                const sal_Int32 nNextSlideIndex( getCurrentSlideIndex() + 1 );

                // first try to prefetch following slide.
                if( nNextSlideIndex >= 0 && 
                    nNextSlideIndex < static_cast<sal_Int32>(maDrawPages.size()) )
                {
                    if( !maSlideBuffer[NEXT_SLIDE].mpSlide.get() ||
                        maSlideBuffer[NEXT_SLIDE].mnSlideIndex != nNextSlideIndex )
                    {
                        maSlideBuffer[NEXT_SLIDE].mpSlide = createSlide( nNextSlideIndex );
                        maSlideBuffer[NEXT_SLIDE].mnSlideIndex = nNextSlideIndex;

                        return true;
                    }
                }

                // could not prefetch following slide, or is already
                // prefetched. Maybe the user is going backwards?
                const sal_Int32 nPrevSlideIndex( getCurrentSlideIndex() - 1 );

                if( nPrevSlideIndex >= 0 && 
                    nPrevSlideIndex < static_cast<sal_Int32>(maDrawPages.size()) )
                {
                    if( !maSlideBuffer[PREVIOUS_SLIDE].mpSlide.get() ||
                        maSlideBuffer[PREVIOUS_SLIDE].mnSlideIndex != nPrevSlideIndex )
                    {
                        // indeed, previous slide is empty. Prefetch now.
                        maSlideBuffer[PREVIOUS_SLIDE].mpSlide = createSlide( nPrevSlideIndex );
                        maSlideBuffer[PREVIOUS_SLIDE].mnSlideIndex = nPrevSlideIndex;
                    }

                    return true;
                }
            }

            return false;
        }

        const SlideSharedPtr& Presentation_Impl::getCurrentSlide() const
        {
            return maSlideBuffer[CURRENT_SLIDE].mpSlide;
        }

        const SlideSharedPtr& Presentation_Impl::getPreviousSlide() const
        {
            return maSlideBuffer[PREVIOUS_SLIDE].mpSlide;
        }

        void Presentation_Impl::setWaitState( const bool bOn )
        {
            mbWaitState = bOn;
            if (mwaitSymbol.get() == 0) // fallback to cursor
                updateMouseCursor();
            else if (mbWaitState)
                mwaitSymbol->show();
            else
                mwaitSymbol->hide();
        }
    
        void Presentation_Impl::updateMouseCursor()
        {
            sal_Int16 mouseCursor = awt::SystemPointer::ARROW;
            if (mbWaitState && mwaitSymbol.get() == 0) // fallback to cursor
                mouseCursor = awt::SystemPointer::WAIT;
            else if (! mbMouseVisible) // INVISIBLE overrides user paint overlay
                mouseCursor = awt::SystemPointer::INVISIBLE;
            else if (maUserPaintColor.isValid())
                mouseCursor = awt::SystemPointer::PEN;
            
            maEventMultiplexer.setMouseCursor( mouseCursor );
        }

        bool Presentation_Impl::showNext( sal_Int32 nPage )
        {
            if( maViewContainer.empty() )
                return false;

            if( !mpListeners.get() )
                return false; // we're disposed

            // check range:
            // ============
            if( nPage < 0 ||
                nPage >= static_cast<sal_Int32>(maDrawPages.size()) )
            {
                return false;
            }

            // notify slide listeners about slide change.
            SlideEvent aEvent;
            aEvent.nOldSlideIndex = getCurrentSlideIndex();
            aEvent.nNewSlideIndex = nPage;
            aEvent.bShowEnded = false;
            
            // TODO(T2): Attention, this is not thread safe. We're 
            // calling out listeners with locked mutex!
			ShowListenersPtr pListeners( mpListeners );
			if( pListeners.get() )
				pListeners->notify( aEvent );

            bool bRet( false );

            // this here might take some time
            {
                const comphelper::ScopeGuard guard_(
                    boost::bind( &Presentation_Impl::setWaitState,
                                 this, false ) );
                setWaitState(true);
                
                if( prefetchSlide( nPage ) )
                {
                    const basegfx::B2ISize slideSize_(
                        getCurrentSlide()->getSlideSize() );
                    const basegfx::B2DSize slideSize( slideSize_.getX(),
                                                      slideSize_.getY() );
                    
                    // push new transformation to all views
                    ::std::for_each(
                        maViewContainer.begin(),
                        maViewContainer.end(),
                        ::boost::bind(&View::setViewSize,
                                      _1, ::boost::cref( slideSize ) ) );
                    
                    // explicitly notify view change here,
                    // because transformation might have changed:
                    // optimization, this->notifyViewChange() would
                    // repaint slide which is not necessary.
                    if (mrehearseTimingsActivity.get() != 0)
                        mrehearseTimingsActivity->notifyViewChange();
                    if (mwaitSymbol.get() != 0)
                        mwaitSymbol->notifyViewChange();
                    
                    EventSharedPtr pTransitionEndEvent(
                        makeEvent(
                            boost::bind(
                                &Presentation_Impl::notifySlideTransitionEnded,
                                this ) ) );
                    
                    // create slide transition, and add proper end event 
                    // (which then starts the slide effects
                    // via CURRENT_SLIDE.show())
                    ActivitySharedPtr pSlideChangeActivity( 
                        createSlideTransition( maDrawPages[ nPage ],
                                               getPreviousSlide(),
                                               getCurrentSlide(),
                                               pTransitionEndEvent ) );
                    
                    if( pSlideChangeActivity.get() )
                    {
                        // factory generated a slide transition - activate it!
                        maActivitiesQueue.addActivity( pSlideChangeActivity );
                    }
                    else
                    {
                        // no transition effect on this slide - schedule slide
                        // effect start event right away.
                        maEventQueue.addEvent( pTransitionEndEvent );
                    }
                    
                    bRet = true;
                }
            } // finally
            setWaitState(false);

			updateMouseCursor();

            return bRet;
        }

        void Presentation_Impl::stopShow()
        {
            // Force-end running animation
            // ===========================
            const SlideSharedPtr& rCurrSlide( getCurrentSlide() );
            if( rCurrSlide.get() )
                rCurrSlide->end();

            // clear all queues
            maEventQueue.clear();
            maActivitiesQueue.clear();

            // Attention: we MUST clear the user event queue here,
            // this is because the current slide might have registered
            // shape events (click or enter/leave), which might
            // otherwise dangle forever in the queue (because of the
            // shared ptr nature). If someone needs to change this:
            // somehow unregister those shapes at the user event queue
            // on notifySlideEnded().
            maUserEventQueue.clear();

            // re-enable automatic effect advancement
            // (maEventQueue.clear() above might have killed
            // maEventMultiplexer's tick events)
            if( maAutomaticAdvancementTimeout.isValid() )
            {
                // toggle automatic mode (enabling just again is
                // ignored by EventMultiplexer)
                maEventMultiplexer.setAutomaticMode( false );
                maEventMultiplexer.setAutomaticMode( true );
            }
        }

        bool Presentation_Impl::nextEffect()
        {
            if( !mbShowPaused )
                return maEventMultiplexer.notifyNextEffect();
            else
                return true;
        }

        bool Presentation_Impl::startShapeActivity( const uno::Reference< drawing::XShape >& xShape )
        {
            // TODO(F3): NYI
            return false;
        }

        bool Presentation_Impl::stopShapeActivity( const uno::Reference< drawing::XShape >& xShape )
        {
            // TODO(F3): NYI
            return false;
        }

        bool Presentation_Impl::pause( bool bPauseShow )
        {
            if (bPauseShow)
                mpPresTimer->pauseTimer();
            else
                mpPresTimer->continueTimer();

            maEventMultiplexer.notifyPauseMode(bPauseShow);
            
            mbShowPaused = bPauseShow;
            return true;
        }

        bool Presentation_Impl::previousSlide()
        {
            stopShow();

            return showNext( getCurrentSlideIndex()-1 );
        }

        bool Presentation_Impl::nextSlide()
        {
            stopShow(); // MUST call stopShow() here, triggers
                        // maUserEventQueue.clear(). See stopShow
                        // inline comments. What's more, stopShow()'s
                        // currSlide->end() call is now also required,
                        // notifySlideEnded() relies on that
                        // unconditionally. Otherwise, genuine shape
                        // animations (drawing layer and GIF) will not
                        // be stopped.
            return showNext( getCurrentSlideIndex()+1 );
        }

        bool Presentation_Impl::displaySlide( sal_Int32 nSlideIndex )
        {
            stopShow();

            return showNext( nSlideIndex );
        }

        sal_Int32 Presentation_Impl::getCurrentSlideIndex()
        {
            if( !getCurrentSlide().get() )
                return -1L; // no show running, thus no current slide

            return maSlideBuffer[CURRENT_SLIDE].mnSlideIndex;
        }

        bool Presentation_Impl::prefetch( const uno::Sequence< uno::Reference< drawing::XDrawPage > >&          aSlideSequence, 
                                          const uno::Sequence< uno::Reference< animations::XAnimationNode > >&  aRootNodeSequence, 
                                          const uno::Sequence< beans::PropertyValue >&                          aShowProperties )
        {
            stopShow();
            
            // Update slides and root nodes
            // ----------------------------

            // hopefully, compiler performs return value optimization here...
            VectorOfDrawPages aTmp0( 
                ::comphelper::sequenceToContainer< VectorOfDrawPages >( aSlideSequence ) );
            maDrawPages.swap( aTmp0 );

            // hopefully, compiler performs return value optimization here...
            VectorOfRootNodes aTmp1(
                ::comphelper::sequenceToContainer< VectorOfRootNodes >( aRootNodeSequence ) );
            maAnimationRootNodes.swap( aTmp1 );

            // process properties
            // ------------------

            // clear set-only properties (the property can only
            // be _set_ via the API, not cleared)
            mbNoSlideTransitions = false;

            for( sal_Int32 i=0, nLen=aShowProperties.getLength(); i<nLen; ++i )
            {
                // delegate to standard property setting
                if( !setProperty( aShowProperties[i] ) )
                {
                    // maybe a prefetch special property?
                    if( aShowProperties[i].Name.equalsAscii( "NoSlideTransitions" ) )
                    {
                        mbNoSlideTransitions = true;
                    }
                }
            }

            return true;
        }
        
        bool Presentation_Impl::addView( const uno::Reference< ::com::sun::star::presentation::XSlideShowView >& xView )
        {
            // first of all, check if view has a valid canvas
            ENSURE_AND_RETURN( xView.is(),
                               "Presentation_Impl::addView(): Invalid view" );
            ENSURE_AND_RETURN( xView->getCanvas().is(),
                               "Presentation_Impl::addView(): View does not provide a valid canvas" );

            UnoViewSharedPtr pView( new SlideView( xView,
                                                   maEventQueue,
                                                   *this ) );

            if( !maViewContainer.addView( pView ) )
                return false;

            // forward to subordinates
            // =======================

            // forward view to all slides:
            for_each_buffered_slide(
                boost::bind( &Slide::addView, _1, boost::cref(pView) ) );

            if (mrehearseTimingsActivity.get() != 0) {
                // new view for timer:
                mrehearseTimingsActivity->addView( pView );
            }

            if (mwaitSymbol.get() != 0) {
                // new view for wait symbol:
                mwaitSymbol->addView( pView );
            }
            
            // forward new view to EventMultiplexer (mouse events etc.)
            maEventMultiplexer.addView( pView );

            
            // initialize view content
            // =======================

            if (getCurrentSlide().get() != 0)
            {
                // set view transformation
                const basegfx::B2ISize slideSize =
                    getCurrentSlide()->getSlideSize();
                pView->setViewSize( basegfx::B2DSize( slideSize.getX(),
                                                      slideSize.getY() ) );
            }

            // clear view area (since its newly added, 
            // we need a clean slate)
            pView->clear();

            return true;
        }

        bool Presentation_Impl::removeView( const uno::Reference< ::com::sun::star::presentation::XSlideShowView >& xView )
        {
            ENSURE_AND_RETURN( xView.is(),
                               "Presentation_Impl::removeView(): Invalid view" );

            UnoViewSharedPtr pView( maViewContainer.removeView( xView ) );
            if( !pView.get() )
                return false;

            if (mrehearseTimingsActivity.get() != 0) {
                // disconnect timer from view:
                mrehearseTimingsActivity->removeView( pView );
            }

            if (mwaitSymbol.get() != 0) {
                // disconnect wait symbol from view:
                mwaitSymbol->removeView( pView );
            }
            
            // remove view from all slides:
            for_each_buffered_slide(
                boost::bind( &Slide::removeView, _1, boost::cref(pView) ) );

            // remove view from EventMultiplexer (mouse events etc.)
            maEventMultiplexer.removeView( pView );

            return true;
        }

        bool Presentation_Impl::setProperty( const beans::PropertyValue& aShowProperty )
        {
            if( aShowProperty.Name.equalsAscii( "AutomaticAdvancement" ) )
            {
                double nTimeout;

                if( (aShowProperty.Value >>= nTimeout) )
                {
                    // enable automatic mode
                    maEventMultiplexer.setAutomaticTimeout( nTimeout );
                    maEventMultiplexer.setAutomaticMode( true );

                    maAutomaticAdvancementTimeout.setValue( nTimeout );
                }
                else
                {
                    // disable automatic mode
                    maEventMultiplexer.setAutomaticMode( false );
                    maAutomaticAdvancementTimeout.clearValue();
                }

                return true;
            }

            if( aShowProperty.Name.equalsAscii( "UserPaintColor" ) )
            {
                sal_Int32 nColor;

                if( (aShowProperty.Value >>= nColor) )
                {
                    OSL_ENSURE( mbMouseVisible,
                                "Presentation_Impl::setProperty(): User paint overrides invisible mouse" );

                    // enable user paint
                    maUserPaintColor.setValue( 
                        unoColor2RGBColor( nColor ) );
                }
                else
                {
                    // disable user paint
                    maUserPaintColor.clearValue();
                }
                
                for_each_buffered_slide(
                    boost::bind( &Slide::setUserPaintColor, _1,
                                 boost::cref(maUserPaintColor) ) );
                updateMouseCursor();
                return true;
            }            

            if( aShowProperty.Name.equalsAscii( "AdvanceOnClick" ) )
            {
                sal_Bool bAdvanceOnClick;

                if( !(aShowProperty.Value >>= bAdvanceOnClick) )
                    return false;

                maUserEventQueue.setAdvanceOnClick( bAdvanceOnClick );

                return true;
            }            

            if( aShowProperty.Name.equalsAscii( "ImageAnimationsAllowed" ) )
            {
                if( !(aShowProperty.Value >>= mbImageAnimationsAllowed) )
                    return false;

                for_each_buffered_slide(
                    boost::bind( &Slide::setImageAnimationsAllowed,
                                 _1, mbImageAnimationsAllowed ) );
                return true;
            }            

            if( aShowProperty.Name.equalsAscii( "MouseVisible" ) )
            {
                if( !(aShowProperty.Value >>= mbMouseVisible) )
                    return false;
                updateMouseCursor();
                return true;
            }

            if( aShowProperty.Name.equalsAscii( "ForceManualAdvance" ) )
            {
                if( !(aShowProperty.Value >>= mbForceManualAdvance) )
                    return false;
                return true;
            }            

            if (aShowProperty.Name.equalsAsciiL(
                    RTL_CONSTASCII_STRINGPARAM("RehearseTimings") ))
            {
                bool bRehearseTimings = false;
                if (! (aShowProperty.Value >>= bRehearseTimings))
                    return false;
                
                if (bRehearseTimings)
                {
                    mrehearseTimingsActivity = RehearseTimingsActivity::create(
                        maEventQueue, maEventMultiplexer, maActivitiesQueue );
                    // add all previously registered views:
                    std::for_each(
                        maViewContainer.begin(),
                        maViewContainer.end(),
                        boost::bind( &RehearseTimingsActivity::addView,
                                     mrehearseTimingsActivity, _1 ) );
                }
                else if (mrehearseTimingsActivity.get() != 0) {
                    // removes timer from all views:
                    mrehearseTimingsActivity->dispose();
                    mrehearseTimingsActivity.reset();
                }
                
                return true;
            }
            
            if (aShowProperty.Name.equalsAsciiL(
                    RTL_CONSTASCII_STRINGPARAM("WaitSymbolBitmap") ))
            {
                uno::Reference<rendering::XBitmap> xBitmap;
                if (! (aShowProperty.Value >>= xBitmap))
                    return false;
                
                if (mwaitSymbol.get() != 0)
                    mwaitSymbol->dispose();
                mwaitSymbol.reset( new WaitSymbol( xBitmap,
                                                   maEventMultiplexer ) );

                // add all previously registered views:
                std::for_each( maViewContainer.begin(),
                               maViewContainer.end(),
                               boost::bind( &WaitSymbol::addView,
                                            mwaitSymbol, _1 ) );
            }
            
            return false;
        }

        void Presentation_Impl::addSlideShowListener( const uno::Reference< ::com::sun::star::presentation::XSlideShowListener >& xListener )
        {
            if( mpListeners.get() )
                mpListeners->addListener( xListener );
        }

        void Presentation_Impl::removeSlideShowListener( const uno::Reference< ::com::sun::star::presentation::XSlideShowListener >& xListener )
        {
            if( mpListeners.get() )
                mpListeners->removeListener( xListener );
        }

        void Presentation_Impl::addShapeEventListener( const uno::Reference< ::com::sun::star::presentation::XShapeEventListener >& xListener, 
                                                       const uno::Reference< drawing::XShape >&                                     xShape )
        {
            ShapeEventListenerMap::iterator aIter;
            if( (aIter=maShapeEventListeners.find( xShape )) == maShapeEventListeners.end() )
            {
                // no entry for this shape -> create one
                maShapeEventListeners.insert( 
                    ShapeEventListenerMap::value_type(xShape,
                                                      ShapeEventListenerMap::value_type::second_type(
                                                          1, xListener) ) );
            }
            else
            {
                // aIter references to an existing entry. Just add the
                // listener there
                aIter->second.push_back( xListener );
            }

            // add to each already buffered slide:
            for_each_buffered_slide(
                boost::bind(
                    &Slide::addShapeEventListener,
                    _1, boost::cref(xListener), boost::cref(xShape) ) );
        }

        void Presentation_Impl::removeShapeEventListener( const uno::Reference< ::com::sun::star::presentation::XShapeEventListener >&  xListener, 
                                                          const uno::Reference< drawing::XShape >&                                      xShape )
        {
            ShapeEventListenerMap::iterator aIter;
            if( (aIter=maShapeEventListeners.find( xShape )) != maShapeEventListeners.end() )
            {
                // entry for this shape found -> remove listener from the list
                ShapeEventListenerMap::value_type::second_type::iterator       aListener( aIter->second.end() );
                const ShapeEventListenerMap::value_type::second_type::iterator aEnd( aIter->second.end() );
                if( (aListener=::std::remove( aIter->second.begin(), 
                                          aEnd, 
                                          xListener )) != aEnd )
                {
                    // actually erase from container
                    aIter->second.erase( aListener, aEnd );
                }
            }

            // remove from each already buffered slide:
            for_each_buffered_slide(
                boost::bind(
                    &Slide::removeShapeEventListener,
                    _1, boost::cref(xListener), boost::cref(xShape) ) );
        }

        void Presentation_Impl::setShapeCursor( const uno::Reference< drawing::XShape >&    xShape, 
                                                sal_Int16                                   nPointerShape )
        {            
            ShapeCursorMap::iterator aIter;
            if( (aIter=maShapeCursors.find( xShape )) == maShapeCursors.end() )
            {
                // no entry for this shape -> create one
                if( nPointerShape != awt::SystemPointer::ARROW )
                {
                    // add new entry, unless shape shall display
                    // normal pointer arrow -> no need to handle that
                    // case
                    maShapeCursors.insert( 
                        ShapeCursorMap::value_type(xShape, nPointerShape) );
				}
            }
            else if( nPointerShape == awt::SystemPointer::ARROW )
            {
                // shape shall display normal cursor -> can disable
                // the cursor and clear the entry
                maShapeCursors.erase( xShape );
            }
            else
            {
                // existing entry found, update with new cursor ID
                aIter->second = nPointerShape;
            }

			// forward to slides
			for_each_buffered_slide( 
				::boost::bind( 
					&Slide::setShapeCursor,
                    _1,
                    ::boost::cref( xShape ),
					nPointerShape ) );
        }

        bool Presentation_Impl::update( double* pTimeoutForNextCall )
        {
            // precondition: update() must only be called from the
            // main thread!
            DBG_TESTSOLARMUTEX();

            if( mbShowPaused )
            {
                return false;
            }
            else
            {
                double nNextEventTime;

                // hold timer, while processing the queues (ensures
                // same time for all activities and events)
                {
                    const comphelper::ScopeGuard guard(
                        boost::bind( &canvas::tools::ElapsedTime::releaseTimer,
                                     boost::cref(mpPresTimer) ) );
                    mpPresTimer->holdTimer();
                    
                    // process queues:
                    nNextEventTime = maEventQueue.process();
                    maActivitiesQueue.process();
                }
                // Time held until here
                
                const bool bActivitiesLeft = (! maActivitiesQueue.isEmpty());
                const bool bTimerEventsLeft = (! maEventQueue.isEmpty());
                const bool bRet = (bActivitiesLeft || bTimerEventsLeft);
                
                if (bRet && pTimeoutForNextCall != 0)
                {
                    // calc pTimeoutForNextCall value:
                    if (bActivitiesLeft)
                    {
                        // activities left: requires immediate updates
                        *pTimeoutForNextCall = 0.0; // come back ASAP
                    }
                    else
                    {
                        // timer events left:
                        // difference from current time (nota bene:
                        // time no longer held here!) to the next event in
                        // the event queue.
                        
                        // ensure positive value:
                        *pTimeoutForNextCall = 
                            std::max( 0.0, nNextEventTime -
                                           mpPresTimer->getElapsedTime() );
                    }
                }
                
                return bRet;
            }
        }

        void Presentation_Impl::notifySlideTransitionEnded()
        {
            const SlideSharedPtr& rCurrSlide( getCurrentSlide() );

            OSL_ENSURE( rCurrSlide.get(),
                        "Presentation_Impl::notifySlideTransitionEnded(): Invalid current slide" );

            if( rCurrSlide.get() )
            {
                // first init show, to give the animations
                // the chance to register SlideStartEvents
                rCurrSlide->show();
                maEventMultiplexer.notifySlideStartEvent();
            }
        }

        void Presentation_Impl::notifySlideAnimationsEnded()
        {
            // This struct will receive the (interruptable) event,
            // that triggers the notifySlideEnded() method.
            InterruptableEventPair aNotificationEvents;

            if( maEventMultiplexer.getAutomaticMode() )
            {
                OSL_ENSURE( mrehearseTimingsActivity.get() == 0,
                            "unexpected: RehearseTimings mode!" );
                
                // schedule a slide end event, with automatic mode's
                // delay
                aNotificationEvents = makeInterruptableDelay(
                    ::boost::bind( &Presentation_Impl::notifySlideEnded,
                                   ::boost::ref( *this ) ),
                    maEventMultiplexer.getAutomaticTimeout() );
            }
            else
            {
                const SlideSharedPtr& rCurrSlide( getCurrentSlide() );
                
                OSL_ENSURE( rCurrSlide.get(),
                            "Presentation_Impl::notifySlideAnimationsEnded(): Invalid current slide" );
                
                // check whether slide transition should happen
                // 'automatically'. If yes, simply schedule the
                // specified timeout.
                // NOTE: mbForceManualAdvance and mrehearseTimings
                // override any individual slide setting, to always
                // step slides manually.
                if( !mbForceManualAdvance &&
                    mrehearseTimingsActivity.get() == 0 &&
                    rCurrSlide.get() &&
                    rCurrSlide->hasAutomaticNextSlide() )
                {
                    aNotificationEvents = makeInterruptableDelay(
                        ::boost::bind( &Presentation_Impl::notifySlideEnded,
                                       ::boost::ref( *this ) ),
                        rCurrSlide->getAutomaticNextSlideTimeout() );

                    // TODO(F2): Provide a mechanism to let the user override 
                    // this automatic timeout via next()
                }
                else
                {
                    if (mrehearseTimingsActivity.get() != 0)
                        mrehearseTimingsActivity->start();
                    
                    // generate click event. Thus, the user must
                    // trigger the actual end of a slide. No need to
                    // generate interruptable event here, there's no
                    // timeout involved.
                    aNotificationEvents.mpImmediateEvent = 
                        makeEvent( ::boost::bind( &Presentation_Impl::notifySlideEnded,
                                                  ::boost::ref( *this ) ) );
                }
            }

            // register events on the queues. To make automatic slide
            // changes interruptable, register the interruption event
            // as a nextEffectEvent target. Note that the timeout
            // event is optional (e.g. manual slide changes don't
            // generate a timeout)            
            maUserEventQueue.registerNextEffectEvent( 
                aNotificationEvents.mpImmediateEvent );

            if( aNotificationEvents.mpTimeoutEvent.get() != NULL )
                maEventQueue.addEvent( aNotificationEvents.mpTimeoutEvent );


            // current slide's main sequence is over. Now should be
            // the time to prefetch the next slide (if any), and
            // prepare the initial slide bitmap (speeds up slide
            // change setup time a lot). Show the wait cursor, this
            // indeed might take some seconds.
            {
                const comphelper::ScopeGuard guard_(
                    boost::bind( &Presentation_Impl::setWaitState, this,
                                 false ) );
                setWaitState(true);
                
                prefetchNextSlides();
                if( !maViewContainer.empty() &&
                    maSlideBuffer[NEXT_SLIDE].mpSlide.get() )
                {
                    // ignore return value, this is just to populate
                    // Slide's internal bitmap buffer, such that the time
                    // needed to generate the slide bitmap is not spent
                    // when the slide change is requested.
                    maSlideBuffer[NEXT_SLIDE].mpSlide->getCurrentSlideBitmap(
                        *maViewContainer.begin() );
                }
            } // finally
        }

        void Presentation_Impl::notifySlideEnded()
        {
            if (mrehearseTimingsActivity.get() != 0)
            {
                const double time = mrehearseTimingsActivity->stop();
                if (mrehearseTimingsActivity->hasBeenClicked())
                {
                    // save time at drawpage:
                    const sal_Int32 nCurrentSlideIndex = getCurrentSlideIndex();
                    if (nCurrentSlideIndex >= 0 &&
                        nCurrentSlideIndex < static_cast<sal_Int32>(
                            maDrawPages.size()))
                    {
                        uno::Reference<beans::XPropertySet> xPropSet(
                            maDrawPages[nCurrentSlideIndex], uno::UNO_QUERY );
                        OSL_ASSERT( xPropSet.is() );
                        if (xPropSet.is())
                        {
                            xPropSet->setPropertyValue(
                                rtl::OUString(
                                    RTL_CONSTASCII_USTRINGPARAM("Change") ),
                                uno::Any( static_cast<sal_Int32>(1) ) );
                            xPropSet->setPropertyValue(
                                rtl::OUString(
                                    RTL_CONSTASCII_USTRINGPARAM("Duration") ),
                                uno::Any( static_cast<sal_Int32>(time) ) );
                        }
                    }
                    else
                        OSL_ENSURE(false, "index out of bounds!");
                }
            }
            
            maEventMultiplexer.notifySlideEndEvent();

            if( getCurrentSlideIndex()+1 < static_cast<sal_Int32>(maDrawPages.size()) )
            {
                nextSlide(); // MUST call that: results in
                             // maUserEventQueue.clear(). See
                             // nextSlide inline comments. What's
                             // more, stopShow()'s currSlide->end()
                             // call is now also required,
                             // notifySlideEnded() relies on that
                             // unconditionally. Otherwise, genuine
                             // shape // animations (drawing layer and
                             // GIF) will not be stopped.
            }
            else
            {
                stopShow();  // MUST call that: results in
                             // maUserEventQueue.clear(). What's more,
                             // stopShow()'s currSlide->end() call is
                             // now also required, notifySlideEnded()
                             // relies on that
                             // unconditionally. Otherwise, genuine
                             // shape animations (drawing layer and
                             // GIF) will not be stopped.

                // we've reached the last slide: broadcast a show
                // ended event
                SlideEvent aEvent;
                aEvent.bShowEnded = true;

                // TODO(T2): Attention, this is not thread safe. We're 
                // calling out listeners with locked mutex!
				ShowListenersPtr pListeners( mpListeners );
				if( pListeners.get() )
		            pListeners->notify( aEvent );
	
                // force default cursor here:
                maEventMultiplexer.setMouseCursor( awt::SystemPointer::ARROW );
            }
        }

        void Presentation_Impl::notifyViewChange()
        {
            // TODO(P1): This should be restricted to the view which 
            // actually _change_ (and not to all views)

            // TODO(Q3): handle this via the new EventMultiplexer scheme!

            // forward view change to active slide
            const SlideSharedPtr& rCurrSlide( getCurrentSlide() );
            if( rCurrSlide.get() )
                rCurrSlide->paint();

            if (mwaitSymbol.get() != 0)
                mwaitSymbol->notifyViewChange();
            
            if (mrehearseTimingsActivity.get() != 0)
                mrehearseTimingsActivity->notifyViewChange();
        }

    }

    Presentation::Presentation( const uno::Reference< uno::XComponentContext >& xContext ) :
        mpImpl( new internal::Presentation_Impl( xContext ) )
    {
    }

    Presentation::~Presentation()
    {
    }

    void Presentation::dispose()
    {
        ENSURE_AND_THROW( mpImpl.get(),
                          "Presentation: pImpl is NULL" );

        mpImpl->dispose();
    }

    bool Presentation::nextEffect()
    {
        ENSURE_AND_THROW( mpImpl.get(),
                          "Presentation: pImpl is NULL" );

        return mpImpl->nextEffect();
    }

    bool Presentation::startShapeActivity( const uno::Reference< drawing::XShape >& xShape )
    {
        ENSURE_AND_THROW( mpImpl.get(),
                          "Presentation: pImpl is NULL" );
        
        return mpImpl->startShapeActivity( xShape );
    }

    bool Presentation::stopShapeActivity( const uno::Reference< drawing::XShape >& xShape )
    {
        ENSURE_AND_THROW( mpImpl.get(),
                          "Presentation: pImpl is NULL" );
        
        return mpImpl->stopShapeActivity( xShape );
    }

    bool Presentation::pause( bool bPauseShow )
    {
        ENSURE_AND_THROW( mpImpl.get(),
                          "Presentation: pImpl is NULL" );
        
        return mpImpl->pause( bPauseShow );
    }

    bool Presentation::previousSlide()
    {
        ENSURE_AND_THROW( mpImpl.get(),
                          "Presentation: pImpl is NULL" );

        return mpImpl->previousSlide();
    }

    bool Presentation::nextSlide()
    {
        ENSURE_AND_THROW( mpImpl.get(),
                          "Presentation: pImpl is NULL" );

        return mpImpl->nextSlide();
    }

    bool Presentation::displaySlide( sal_Int32 nSlideIndex )
    {
        ENSURE_AND_THROW( mpImpl.get(),
                          "Presentation: pImpl is NULL" );

        return mpImpl->displaySlide( nSlideIndex );
    }

    sal_Int32 Presentation::getCurrentSlideIndex()
    {
        ENSURE_AND_THROW( mpImpl.get(),
                          "Presentation: pImpl is NULL" );

        return mpImpl->getCurrentSlideIndex();
    }

    bool Presentation::prefetch( const uno::Sequence< uno::Reference< drawing::XDrawPage > >&           aSlideSequence, 
                                 const uno::Sequence< uno::Reference< animations::XAnimationNode > >&   aRootNodeSequence, 
                                 const uno::Sequence< beans::PropertyValue >&                           aShowProperties )
    {
        ENSURE_AND_THROW( mpImpl.get(),
                          "Presentation: pImpl is NULL" );

        return mpImpl->prefetch( aSlideSequence,
                                 aRootNodeSequence,
                                 aShowProperties );
    }

    bool Presentation::setProperty( const beans::PropertyValue& aShowProperty )
    {
        ENSURE_AND_THROW( mpImpl.get(),
                          "Presentation: pImpl is NULL" );

        return mpImpl->setProperty( aShowProperty );
    }

    bool Presentation::addView( const uno::Reference< ::com::sun::star::presentation::XSlideShowView >& xView )
    {
        ENSURE_AND_THROW( mpImpl.get(),
                          "Presentation: pImpl is NULL" );

        return mpImpl->addView( xView );
    }

    bool Presentation::removeView( const uno::Reference< ::com::sun::star::presentation::XSlideShowView >& xView )
    {
        ENSURE_AND_THROW( mpImpl.get(),
                          "Presentation: pImpl is NULL" );

        return mpImpl->removeView( xView );
    }

    void Presentation::addSlideShowListener( const uno::Reference< ::com::sun::star::presentation::XSlideShowListener >& xListener )
    {
        ENSURE_AND_THROW( mpImpl.get(),
                          "Presentation: pImpl is NULL" );

        mpImpl->addSlideShowListener( xListener );
    }

    void Presentation::removeSlideShowListener( const uno::Reference< ::com::sun::star::presentation::XSlideShowListener >& xListener )
    {
        ENSURE_AND_THROW( mpImpl.get(),
                          "Presentation: pImpl is NULL" );

        mpImpl->removeSlideShowListener( xListener );
    }

    void Presentation::addShapeEventListener( const uno::Reference< ::com::sun::star::presentation::XShapeEventListener >&  xListener,
                                              const uno::Reference< drawing::XShape >&                                      xShape )
    {
        ENSURE_AND_THROW( mpImpl.get(),
                          "Presentation: pImpl is NULL" );

        mpImpl->addShapeEventListener( xListener,
                                       xShape );
    }

    void Presentation::removeShapeEventListener( const uno::Reference< ::com::sun::star::presentation::XShapeEventListener >&   xListener, 
                                                 const uno::Reference< drawing::XShape >&                                       xShape )
    {
        ENSURE_AND_THROW( mpImpl.get(),
                          "Presentation: pImpl is NULL" );

        mpImpl->removeShapeEventListener( xListener,
                                          xShape );
    }

    void Presentation::setShapeCursor( const uno::Reference< drawing::XShape >& xShape, 
                                       sal_Int16                                nPointerShape )
    {
        ENSURE_AND_THROW( mpImpl.get(),
                          "Presentation: pImpl is NULL" );

        mpImpl->setShapeCursor( xShape, nPointerShape );
    }

    bool Presentation::update( double* pNextTimeout )
    {
        ENSURE_AND_THROW( mpImpl.get(),
                          "Presentation: pImpl is NULL" );

        return mpImpl->update( pNextTimeout );
    }

}
