// Copyright 2015 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#ifndef COMPONENTS_EXO_SUB_SURFACE_H_
#define COMPONENTS_EXO_SUB_SURFACE_H_

#include "base/macros.h"
#include "base/memory/scoped_ptr.h"
#include "components/exo/surface_delegate.h"
#include "components/exo/surface_observer.h"
#include "ui/gfx/geometry/point.h"

namespace base {
namespace trace_event {
class TracedValue;
}
}

namespace exo {
class Surface;

// This class provides functions for treating a surface as a sub-surface. A
// sub-surface has one parent surface.
class SubSurface : public SurfaceDelegate, public SurfaceObserver {
 public:
  SubSurface(Surface* surface, Surface* parent);
  ~SubSurface() override;

  // This schedules a sub-surface position change. The sub-surface will be
  // moved so, that its origin (top-left corner pixel) will be at the |position|
  // of the parent surface coordinate system.
  void SetPosition(const gfx::Point& position);

  // This removes sub-surface from the stack, and puts it back just above the
  // reference surface, changing the z-order of the sub-surfaces. The reference
  // surface must be one of the sibling surfaces, or the parent surface.
  void PlaceAbove(Surface* reference);

  // This removes sub-surface from the stack, and puts it back just below the
  // sibling surface.
  void PlaceBelow(Surface* sibiling);

  // Change the commit behaviour of the sub-surface.
  void SetCommitBehavior(bool synchronized);

  // Returns a trace value representing the state of the surface.
  scoped_ptr<base::trace_event::TracedValue> AsTracedValue() const;

  // Overridden from SurfaceDelegate:
  void OnSurfaceCommit() override;
  bool IsSurfaceSynchronized() const override;

  // Overridden from SurfaceObserver:
  void OnSurfaceDestroying(Surface* surface) override;

 private:
  Surface* surface_;
  Surface* parent_;
  bool is_synchronized_;

  DISALLOW_COPY_AND_ASSIGN(SubSurface);
};

}  // namespace exo

#endif  // COMPONENTS_EXO_SUB_SURFACE_H_
