/*
 * Copyright (C) 2008 Instituto Nokia de Tecnologia. All rights reserved.
 *
 * This file is part of QZion.
 *
 * QZion is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * QZion is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with QZion.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "qziontextblock.h"
#include "qziontextblock_p.h"

#include <QDebug>
#include <QRegion>
#include <QPainter>
#include <QApplication>
#include <QPaintEvent>

#include "qzionobject_p.h"

QZionTextBlock::QZionTextBlock(QZionAbstractCanvas *canvas)
    : QZionObject(canvas)
{
    QZINIT(QZionTextBlock, canvas);

    QZD(QZionTextBlock);
    QObject::connect(d->page(), SIGNAL(loadFinished(bool)),
                     this, SIGNAL(contentsChanged()));
    QObject::connect(d->page(), SIGNAL(loadFinished(bool)),
                     this, SIGNAL(contentsLoadFinished()));
    QObject::connect(d->page(), SIGNAL(loadProgress(int)),
                     this, SIGNAL(contentsChanged()));
}

QZionTextBlock::QZionTextBlock(QZionAbstractCanvas *canvas, const QString &text)
    : QZionObject(canvas)
{
    QZINIT(QZionTextBlock, canvas);

    QZD(QZionTextBlock);
    d->setText(text);

    QObject::connect(d->page(), SIGNAL(loadFinished(bool)),
                     this, SIGNAL(contentsChanged()));
    QObject::connect(d->page(), SIGNAL(loadFinished(bool)),
                     this, SIGNAL(contentsLoadFinished()));
    QObject::connect(d->page(), SIGNAL(loadProgress(int)),
                     this, SIGNAL(contentsChanged()));
}

QZionTextBlock::~QZionTextBlock()
{
    QZDELETE(QZionTextBlock);
}

QSize QZionTextBlock::size() const
{
    QZD(QZionTextBlock);
    return d->size();
}

void QZionTextBlock::setSize(const QSize &size)
{
    QZD(QZionTextBlock);
    d->setSize(size);
    QZionObject::setSize(size);
}

int QZionTextBlock::preferredWidth()
{
    QZD(QZionTextBlock);
    return d->preferredWidth;
}

void QZionTextBlock::setPreferredWidth(const int width)
{
    QZD(QZionTextBlock);
    d->preferredWidth = width;
    // need to do this to get the correct height (webkit way)
    d->page()->setViewportSize(QSize(width, 1));
    d->page()->setViewportSize(d->page()->mainFrame()->contentsSize());
}


QSize QZionTextBlock::contentsSize() const
{
    QZD(QZionTextBlock);
    return d->page()->mainFrame()->contentsSize() * d->scale();
}

double QZionTextBlock::scale()
{
    QZD(QZionTextBlock);
    return d->scale();
}

void QZionTextBlock::setScale(double factor)
{
    QZD(QZionTextBlock);
    d->setScale(factor);
}

QPixmap QZionTextBlock::toPixmap()
{
    QZD(QZionTextBlock);
    QPixmap pixmap(d->page()->viewportSize());
    pixmap.fill();
    QPainter painter(&pixmap);

    d->page()->mainFrame()->render(&painter);

    return pixmap;
}

QImage QZionTextBlock::toImage()
{
    QZD(QZionTextBlock);
    QImage image(d->page()->viewportSize(), QImage::Format_RGB32);
    // Fills white
    image.fill(0xFFFFFF);
    QPainter painter(&image);

    d->page()->mainFrame()->render(&painter);

    return image;
}

QString QZionTextBlock::text() const
{
    QZD(QZionTextBlock);
    return d->text();
}

QColor QZionTextBlock::color() const
{
    return QColor(255, 255, 255, 0);
}

void QZionTextBlock::setColor(const QColor &color)
{
    Q_UNUSED(color);
}

void QZionTextBlock::setText(const QString &text)
{
    QZD(QZionTextBlock);
    d->setText(text);
}

QRect QZionTextBlock::rect() const
{
    QZD(QZionTextBlock);
    return d->sceneBoundingRect().toRect().translated(pos());
}

void QZionTextBlock::paint(QPainter* painter)
{
    Q_UNUSED(painter);
}
