// Copyright (c) 2015-2019 The Khronos Group Inc.
// 
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
// 
//     http://www.apache.org/licenses/LICENSE-2.0
// 
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
// 
// ---- Exceptions to the Apache 2.0 License: ----
// 
// As an exception, if you use this Software to generate code and portions of
// this Software are embedded into the generated code as a result, you may
// redistribute such product without providing attribution as would otherwise
// be required by Sections 4(a), 4(b) and 4(d) of the License.
// 
// In addition, if you combine or link code generated by this Software with
// software that is licensed under the GPLv2 or the LGPL v2.0 or 2.1
// ("`Combined Software`") and if a court of competent jurisdiction determines
// that the patent provision (Section 3), the indemnity provision (Section 9)
// or other Section of the License conflicts with the conditions of the
// applicable GPL or LGPL license, you may retroactively and prospectively
// choose to deem waived or otherwise exclude such Section(s) of the License,
// but only in their entirety and only with respect to the Combined Software.
//     

// This header is generated from the Khronos Vulkan XML API Registry.

#ifndef VULKAN_HPP
#define VULKAN_HPP

#include <algorithm>
#include <array>
#include <cstddef>
#include <cstdint>
#include <cstring>
#include <initializer_list>
#include <string>
#include <system_error>
#include <tuple>
#include <type_traits>
#include <vulkan/vulkan.h>

#if !defined(VULKAN_HPP_DISABLE_ENHANCED_MODE)
# include <memory>
# include <vector>
#endif

#if !defined(VULKAN_HPP_ASSERT)
# include <cassert>
# define VULKAN_HPP_ASSERT   assert
#endif

static_assert( VK_HEADER_VERSION ==  101 , "Wrong VK_HEADER_VERSION!" );

// 32-bit vulkan is not typesafe for handles, so don't allow copy constructors on this platform by default.
// To enable this feature on 32-bit platforms please define VULKAN_HPP_TYPESAFE_CONVERSION
#if defined(__LP64__) || defined(_WIN64) || (defined(__x86_64__) && !defined(__ILP32__) ) || defined(_M_X64) || defined(__ia64) || defined (_M_IA64) || defined(__aarch64__) || defined(__powerpc64__)
# if !defined( VULKAN_HPP_TYPESAFE_CONVERSION )
#  define VULKAN_HPP_TYPESAFE_CONVERSION
# endif
#endif

// <tuple> includes <sys/sysmacros.h> through some other header
// this results in major(x) being resolved to gnu_dev_major(x)
// which is an expression in a constructor initializer list.
#if defined(major)
  #undef major
#endif
#if defined(minor)
  #undef minor
#endif

// Windows defines MemoryBarrier which is deprecated and collides
// with the vk::MemoryBarrier struct.
#if defined(MemoryBarrier)
  #undef MemoryBarrier
#endif

#if !defined(VULKAN_HPP_HAS_UNRESTRICTED_UNIONS)
# if defined(__clang__)
#  if __has_feature(cxx_unrestricted_unions)
#   define VULKAN_HPP_HAS_UNRESTRICTED_UNIONS
#  endif
# elif defined(__GNUC__)
#  define GCC_VERSION (__GNUC__ * 10000 + __GNUC_MINOR__ * 100 + __GNUC_PATCHLEVEL__)
#  if 40600 <= GCC_VERSION
#   define VULKAN_HPP_HAS_UNRESTRICTED_UNIONS
#  endif
# elif defined(_MSC_VER)
#  if 1900 <= _MSC_VER
#   define VULKAN_HPP_HAS_UNRESTRICTED_UNIONS
#  endif
# endif
#endif

#if !defined(VULKAN_HPP_INLINE)
# if defined(__clang___)
#  if __has_attribute(always_inline)
#   define VULKAN_HPP_INLINE __attribute__((always_inline)) __inline__
#  else
#    define VULKAN_HPP_INLINE inline
#  endif
# elif defined(__GNUC__)
#  define VULKAN_HPP_INLINE __attribute__((always_inline)) __inline__
# elif defined(_MSC_VER)
#  define VULKAN_HPP_INLINE inline
# else
#  define VULKAN_HPP_INLINE inline
# endif
#endif

#if defined(VULKAN_HPP_TYPESAFE_CONVERSION)
# define VULKAN_HPP_TYPESAFE_EXPLICIT
#else
# define VULKAN_HPP_TYPESAFE_EXPLICIT explicit
#endif

#if defined(_MSC_VER) && (_MSC_VER <= 1800)
# define VULKAN_HPP_CONSTEXPR
#else
# define VULKAN_HPP_CONSTEXPR constexpr
#endif

#if !defined(VULKAN_HPP_NAMESPACE)
#define VULKAN_HPP_NAMESPACE vk
#endif

#define VULKAN_HPP_STRINGIFY2(text) #text
#define VULKAN_HPP_STRINGIFY(text) VULKAN_HPP_STRINGIFY2(text)
#define VULKAN_HPP_NAMESPACE_STRING VULKAN_HPP_STRINGIFY(VULKAN_HPP_NAMESPACE)

namespace VULKAN_HPP_NAMESPACE
{
#if !defined(VULKAN_HPP_DISABLE_ENHANCED_MODE)
  template <typename T>
  class ArrayProxy
  {
  public:
    VULKAN_HPP_CONSTEXPR ArrayProxy(std::nullptr_t)
      : m_count(0)
      , m_ptr(nullptr)
    {}

    ArrayProxy(T & ptr)
      : m_count(1)
      , m_ptr(&ptr)
    {}

    ArrayProxy(uint32_t count, T * ptr)
      : m_count(count)
      , m_ptr(ptr)
    {}

    template <size_t N>
    ArrayProxy(std::array<typename std::remove_const<T>::type, N> & data)
      : m_count(N)
      , m_ptr(data.data())
    {}

    template <size_t N>
    ArrayProxy(std::array<typename std::remove_const<T>::type, N> const& data)
      : m_count(N)
      , m_ptr(data.data())
    {}

    template <class Allocator = std::allocator<typename std::remove_const<T>::type>>
    ArrayProxy(std::vector<typename std::remove_const<T>::type, Allocator> & data)
      : m_count(static_cast<uint32_t>(data.size()))
      , m_ptr(data.data())
    {}

    template <class Allocator = std::allocator<typename std::remove_const<T>::type>>
    ArrayProxy(std::vector<typename std::remove_const<T>::type, Allocator> const& data)
      : m_count(static_cast<uint32_t>(data.size()))
      , m_ptr(data.data())
    {}

    ArrayProxy(std::initializer_list<T> const& data)
      : m_count(static_cast<uint32_t>(data.end() - data.begin()))
      , m_ptr(data.begin())
    {}

    const T * begin() const
    {
      return m_ptr;
    }

    const T * end() const
    {
      return m_ptr + m_count;
    }

    const T & front() const
    {
      VULKAN_HPP_ASSERT(m_count && m_ptr);
      return *m_ptr;
    }

    const T & back() const
    {
      VULKAN_HPP_ASSERT(m_count && m_ptr);
      return *(m_ptr + m_count - 1);
    }

    bool empty() const
    {
      return (m_count == 0);
    }

    uint32_t size() const
    {
      return m_count;
    }

    T * data() const
    {
      return m_ptr;
    }

  private:
    uint32_t  m_count;
    T *       m_ptr;
  };
#endif

  template <typename FlagBitsType> struct FlagTraits
  {
    enum { allFlags = 0 };
  };

  template <typename BitType, typename MaskType = VkFlags>
  class Flags
  {
  public:
    VULKAN_HPP_CONSTEXPR Flags()
      : m_mask(0)
    {
    }

    Flags(BitType bit)
      : m_mask(static_cast<MaskType>(bit))
    {
    }

    Flags(Flags<BitType> const& rhs)
      : m_mask(rhs.m_mask)
    {
    }

    explicit Flags(MaskType flags)
      : m_mask(flags)
    {
    }

    Flags<BitType> & operator=(Flags<BitType> const& rhs)
    {
      m_mask = rhs.m_mask;
      return *this;
    }

    Flags<BitType> & operator|=(Flags<BitType> const& rhs)
    {
      m_mask |= rhs.m_mask;
      return *this;
    }

    Flags<BitType> & operator&=(Flags<BitType> const& rhs)
    {
      m_mask &= rhs.m_mask;
      return *this;
    }

    Flags<BitType> & operator^=(Flags<BitType> const& rhs)
    {
      m_mask ^= rhs.m_mask;
      return *this;
    }

    Flags<BitType> operator|(Flags<BitType> const& rhs) const
    {
      Flags<BitType> result(*this);
      result |= rhs;
      return result;
    }

    Flags<BitType> operator&(Flags<BitType> const& rhs) const
    {
      Flags<BitType> result(*this);
      result &= rhs;
      return result;
    }

    Flags<BitType> operator^(Flags<BitType> const& rhs) const
    {
      Flags<BitType> result(*this);
      result ^= rhs;
      return result;
    }

    bool operator!() const
    {
      return !m_mask;
    }

    Flags<BitType> operator~() const
    {
      Flags<BitType> result(*this);
      result.m_mask ^= FlagTraits<BitType>::allFlags;
      return result;
    }

    bool operator==(Flags<BitType> const& rhs) const
    {
      return m_mask == rhs.m_mask;
    }

    bool operator!=(Flags<BitType> const& rhs) const
    {
      return m_mask != rhs.m_mask;
    }

    explicit operator bool() const
    {
      return !!m_mask;
    }

    explicit operator MaskType() const
    {
        return m_mask;
    }

  private:
    MaskType  m_mask;
  };

  template <typename BitType>
  Flags<BitType> operator|(BitType bit, Flags<BitType> const& flags)
  {
    return flags | bit;
  }

  template <typename BitType>
  Flags<BitType> operator&(BitType bit, Flags<BitType> const& flags)
  {
    return flags & bit;
  }

  template <typename BitType>
  Flags<BitType> operator^(BitType bit, Flags<BitType> const& flags)
  {
    return flags ^ bit;
  }

  template <typename RefType>
  class Optional
  {
  public:
    Optional(RefType & reference) { m_ptr = &reference; }
    Optional(RefType * ptr) { m_ptr = ptr; }
    Optional(std::nullptr_t) { m_ptr = nullptr; }

    operator RefType*() const { return m_ptr; }
    RefType const* operator->() const { return m_ptr; }
    explicit operator bool() const { return !!m_ptr; }

  private:
    RefType *m_ptr;
  };

  template <typename X, typename Y> struct isStructureChainValid { enum { value = false }; };

  template <typename P, typename T>
  struct TypeList
  {
    using list = P;
    using last = T;
  };

  template <typename List, typename X>
  struct extendCheck
  {
    static const bool valid = isStructureChainValid<typename List::last, X>::value || extendCheck<typename List::list,X>::valid;
  };

  template <typename T, typename X>
  struct extendCheck<TypeList<void,T>,X>
  {
    static const bool valid = isStructureChainValid<T, X>::value;
  };

  template <typename X>
  struct extendCheck<void,X>
  {
    static const bool valid = true;
  };

  template <class Element>
  class StructureChainElement
  {
  public:
    explicit operator Element&() { return value; }
    explicit operator const Element&() const { return value; }
  private:
    Element value;
  };

  template<typename ...StructureElements>
  class StructureChain : private StructureChainElement<StructureElements>...
  {
  public:
    StructureChain()
    {
      link<void, StructureElements...>();  
    }

    StructureChain(StructureChain const &rhs)
    {
      linkAndCopy<void, StructureElements...>(rhs);
    }

    StructureChain(StructureElements const &... elems)
    {
      linkAndCopyElements<void, StructureElements...>(elems...);
    }

    StructureChain& operator=(StructureChain const &rhs)
    {
      linkAndCopy<void, StructureElements...>(rhs);
      return *this;
    }

    template<typename ClassType> ClassType& get() { return static_cast<ClassType&>(*this);}

  private:
    template<typename List, typename X>
    void link()
    {
      static_assert(extendCheck<List, X>::valid, "The structure chain is not valid!");
    }

    template<typename List, typename X, typename Y, typename ...Z>
    void link()
    {
      static_assert(extendCheck<List,X>::valid, "The structure chain is not valid!");
      X& x = static_cast<X&>(*this);
      Y& y = static_cast<Y&>(*this);
      x.pNext = &y;
      link<TypeList<List, X>, Y, Z...>();
    }

    template<typename List, typename X>
    void linkAndCopy(StructureChain const &rhs)
    {
      static_assert(extendCheck<List, X>::valid, "The structure chain is not valid!");
      static_cast<X&>(*this) = static_cast<X const &>(rhs);
    }

    template<typename List, typename X, typename Y, typename ...Z>
    void linkAndCopy(StructureChain const &rhs)
    {
      static_assert(extendCheck<List, X>::valid, "The structure chain is not valid!");
      X& x = static_cast<X&>(*this);
      Y& y = static_cast<Y&>(*this);
      x = static_cast<X const &>(rhs);
      x.pNext = &y;
      linkAndCopy<TypeList<List, X>, Y, Z...>(rhs);
    }

    template<typename List, typename X>
    void linkAndCopyElements(X const &xelem)
    {
      static_assert(extendCheck<List, X>::valid, "The structure chain is not valid!");
      static_cast<X&>(*this) = xelem;
    }

    template<typename List, typename X, typename Y, typename ...Z>
    void linkAndCopyElements(X const &xelem, Y const &yelem, Z const &... zelem)
    {
      static_assert(extendCheck<List, X>::valid, "The structure chain is not valid!");
      X& x = static_cast<X&>(*this);
      Y& y = static_cast<Y&>(*this);
      x = xelem;
      x.pNext = &y;
      linkAndCopyElements<TypeList<List, X>, Y, Z...>(yelem, zelem...);
    }
  };

#if !defined(VULKAN_HPP_NO_SMART_HANDLE)
  template <typename Type, typename Dispatch> class UniqueHandleTraits;

  template <typename Type, typename Dispatch>
  class UniqueHandle : public UniqueHandleTraits<Type,Dispatch>::deleter
  {
  private:
    using Deleter = typename UniqueHandleTraits<Type,Dispatch>::deleter;
  public:
    explicit UniqueHandle( Type const& value = Type(), Deleter const& deleter = Deleter() )
      : Deleter( deleter)
      , m_value( value )
    {}

    UniqueHandle( UniqueHandle const& ) = delete;

    UniqueHandle( UniqueHandle && other )
      : Deleter( std::move( static_cast<Deleter&>( other ) ) )
      , m_value( other.release() )
    {}

    ~UniqueHandle()
    {
      if ( m_value ) this->destroy( m_value );
    }

    UniqueHandle & operator=( UniqueHandle const& ) = delete;

    UniqueHandle & operator=( UniqueHandle && other )
    {
      reset( other.release() );
      *static_cast<Deleter*>(this) = std::move( static_cast<Deleter&>(other) );
      return *this;
    }

    explicit operator bool() const
    {
      return m_value.operator bool();
    }

    Type const* operator->() const
    {
      return &m_value;
    }

    Type * operator->()
    {
      return &m_value;
    }

    Type const& operator*() const
    {
      return m_value;
    }

    Type & operator*()
    {
      return m_value;
    }

    const Type & get() const
    {
      return m_value;
    }
    
    Type & get()
    {
      return m_value;
    }

    void reset( Type const& value = Type() )
    {
      if ( m_value != value )
      {
        if ( m_value ) this->destroy( m_value );
        m_value = value;
      }
    }

    Type release()
    {
      Type value = m_value;
      m_value = nullptr;
      return value;
    }

    void swap( UniqueHandle<Type,Dispatch> & rhs )
    {
      std::swap(m_value, rhs.m_value);
      std::swap(static_cast<Deleter&>(*this), static_cast<Deleter&>(rhs));
    }

  private:
    Type    m_value;
  };

  template <typename Type, typename Dispatch>
  VULKAN_HPP_INLINE void swap( UniqueHandle<Type,Dispatch> & lhs, UniqueHandle<Type,Dispatch> & rhs )
  {
    lhs.swap( rhs );
  }
#endif

  class DispatchLoaderStatic
  {
  public:
    VkResult vkCreateInstance( const VkInstanceCreateInfo* pCreateInfo, const VkAllocationCallbacks* pAllocator, VkInstance* pInstance ) const
    {
      return ::vkCreateInstance( pCreateInfo, pAllocator, pInstance );
    }

    VkResult vkEnumerateInstanceExtensionProperties( const char* pLayerName, uint32_t* pPropertyCount, VkExtensionProperties* pProperties ) const
    {
      return ::vkEnumerateInstanceExtensionProperties( pLayerName, pPropertyCount, pProperties );
    }

    VkResult vkEnumerateInstanceLayerProperties( uint32_t* pPropertyCount, VkLayerProperties* pProperties ) const
    {
      return ::vkEnumerateInstanceLayerProperties( pPropertyCount, pProperties );
    }

    VkResult vkEnumerateInstanceVersion( uint32_t* pApiVersion ) const
    {
      return ::vkEnumerateInstanceVersion( pApiVersion );
    }

    VkResult vkBeginCommandBuffer( VkCommandBuffer commandBuffer, const VkCommandBufferBeginInfo* pBeginInfo ) const
    {
      return ::vkBeginCommandBuffer( commandBuffer, pBeginInfo );
    }

    void vkCmdBeginConditionalRenderingEXT( VkCommandBuffer commandBuffer, const VkConditionalRenderingBeginInfoEXT* pConditionalRenderingBegin ) const
    {
      return ::vkCmdBeginConditionalRenderingEXT( commandBuffer, pConditionalRenderingBegin );
    }

    void vkCmdBeginDebugUtilsLabelEXT( VkCommandBuffer commandBuffer, const VkDebugUtilsLabelEXT* pLabelInfo ) const
    {
      return ::vkCmdBeginDebugUtilsLabelEXT( commandBuffer, pLabelInfo );
    }

    void vkCmdBeginQuery( VkCommandBuffer commandBuffer, VkQueryPool queryPool, uint32_t query, VkQueryControlFlags flags ) const
    {
      return ::vkCmdBeginQuery( commandBuffer, queryPool, query, flags );
    }

    void vkCmdBeginQueryIndexedEXT( VkCommandBuffer commandBuffer, VkQueryPool queryPool, uint32_t query, VkQueryControlFlags flags, uint32_t index ) const
    {
      return ::vkCmdBeginQueryIndexedEXT( commandBuffer, queryPool, query, flags, index );
    }

    void vkCmdBeginRenderPass( VkCommandBuffer commandBuffer, const VkRenderPassBeginInfo* pRenderPassBegin, VkSubpassContents contents ) const
    {
      return ::vkCmdBeginRenderPass( commandBuffer, pRenderPassBegin, contents );
    }

    void vkCmdBeginRenderPass2KHR( VkCommandBuffer commandBuffer, const VkRenderPassBeginInfo* pRenderPassBegin, const VkSubpassBeginInfoKHR* pSubpassBeginInfo ) const
    {
      return ::vkCmdBeginRenderPass2KHR( commandBuffer, pRenderPassBegin, pSubpassBeginInfo );
    }

    void vkCmdBeginTransformFeedbackEXT( VkCommandBuffer commandBuffer, uint32_t firstCounterBuffer, uint32_t counterBufferCount, const VkBuffer* pCounterBuffers, const VkDeviceSize* pCounterBufferOffsets ) const
    {
      return ::vkCmdBeginTransformFeedbackEXT( commandBuffer, firstCounterBuffer, counterBufferCount, pCounterBuffers, pCounterBufferOffsets );
    }

    void vkCmdBindDescriptorSets( VkCommandBuffer commandBuffer, VkPipelineBindPoint pipelineBindPoint, VkPipelineLayout layout, uint32_t firstSet, uint32_t descriptorSetCount, const VkDescriptorSet* pDescriptorSets, uint32_t dynamicOffsetCount, const uint32_t* pDynamicOffsets ) const
    {
      return ::vkCmdBindDescriptorSets( commandBuffer, pipelineBindPoint, layout, firstSet, descriptorSetCount, pDescriptorSets, dynamicOffsetCount, pDynamicOffsets );
    }

    void vkCmdBindIndexBuffer( VkCommandBuffer commandBuffer, VkBuffer buffer, VkDeviceSize offset, VkIndexType indexType ) const
    {
      return ::vkCmdBindIndexBuffer( commandBuffer, buffer, offset, indexType );
    }

    void vkCmdBindPipeline( VkCommandBuffer commandBuffer, VkPipelineBindPoint pipelineBindPoint, VkPipeline pipeline ) const
    {
      return ::vkCmdBindPipeline( commandBuffer, pipelineBindPoint, pipeline );
    }

    void vkCmdBindShadingRateImageNV( VkCommandBuffer commandBuffer, VkImageView imageView, VkImageLayout imageLayout ) const
    {
      return ::vkCmdBindShadingRateImageNV( commandBuffer, imageView, imageLayout );
    }

    void vkCmdBindTransformFeedbackBuffersEXT( VkCommandBuffer commandBuffer, uint32_t firstBinding, uint32_t bindingCount, const VkBuffer* pBuffers, const VkDeviceSize* pOffsets, const VkDeviceSize* pSizes ) const
    {
      return ::vkCmdBindTransformFeedbackBuffersEXT( commandBuffer, firstBinding, bindingCount, pBuffers, pOffsets, pSizes );
    }

    void vkCmdBindVertexBuffers( VkCommandBuffer commandBuffer, uint32_t firstBinding, uint32_t bindingCount, const VkBuffer* pBuffers, const VkDeviceSize* pOffsets ) const
    {
      return ::vkCmdBindVertexBuffers( commandBuffer, firstBinding, bindingCount, pBuffers, pOffsets );
    }

    void vkCmdBlitImage( VkCommandBuffer commandBuffer, VkImage srcImage, VkImageLayout srcImageLayout, VkImage dstImage, VkImageLayout dstImageLayout, uint32_t regionCount, const VkImageBlit* pRegions, VkFilter filter ) const
    {
      return ::vkCmdBlitImage( commandBuffer, srcImage, srcImageLayout, dstImage, dstImageLayout, regionCount, pRegions, filter );
    }

    void vkCmdBuildAccelerationStructureNV( VkCommandBuffer commandBuffer, const VkAccelerationStructureInfoNV* pInfo, VkBuffer instanceData, VkDeviceSize instanceOffset, VkBool32 update, VkAccelerationStructureNV dst, VkAccelerationStructureNV src, VkBuffer scratch, VkDeviceSize scratchOffset ) const
    {
      return ::vkCmdBuildAccelerationStructureNV( commandBuffer, pInfo, instanceData, instanceOffset, update, dst, src, scratch, scratchOffset );
    }

    void vkCmdClearAttachments( VkCommandBuffer commandBuffer, uint32_t attachmentCount, const VkClearAttachment* pAttachments, uint32_t rectCount, const VkClearRect* pRects ) const
    {
      return ::vkCmdClearAttachments( commandBuffer, attachmentCount, pAttachments, rectCount, pRects );
    }

    void vkCmdClearColorImage( VkCommandBuffer commandBuffer, VkImage image, VkImageLayout imageLayout, const VkClearColorValue* pColor, uint32_t rangeCount, const VkImageSubresourceRange* pRanges ) const
    {
      return ::vkCmdClearColorImage( commandBuffer, image, imageLayout, pColor, rangeCount, pRanges );
    }

    void vkCmdClearDepthStencilImage( VkCommandBuffer commandBuffer, VkImage image, VkImageLayout imageLayout, const VkClearDepthStencilValue* pDepthStencil, uint32_t rangeCount, const VkImageSubresourceRange* pRanges ) const
    {
      return ::vkCmdClearDepthStencilImage( commandBuffer, image, imageLayout, pDepthStencil, rangeCount, pRanges );
    }

    void vkCmdCopyAccelerationStructureNV( VkCommandBuffer commandBuffer, VkAccelerationStructureNV dst, VkAccelerationStructureNV src, VkCopyAccelerationStructureModeNV mode ) const
    {
      return ::vkCmdCopyAccelerationStructureNV( commandBuffer, dst, src, mode );
    }

    void vkCmdCopyBuffer( VkCommandBuffer commandBuffer, VkBuffer srcBuffer, VkBuffer dstBuffer, uint32_t regionCount, const VkBufferCopy* pRegions ) const
    {
      return ::vkCmdCopyBuffer( commandBuffer, srcBuffer, dstBuffer, regionCount, pRegions );
    }

    void vkCmdCopyBufferToImage( VkCommandBuffer commandBuffer, VkBuffer srcBuffer, VkImage dstImage, VkImageLayout dstImageLayout, uint32_t regionCount, const VkBufferImageCopy* pRegions ) const
    {
      return ::vkCmdCopyBufferToImage( commandBuffer, srcBuffer, dstImage, dstImageLayout, regionCount, pRegions );
    }

    void vkCmdCopyImage( VkCommandBuffer commandBuffer, VkImage srcImage, VkImageLayout srcImageLayout, VkImage dstImage, VkImageLayout dstImageLayout, uint32_t regionCount, const VkImageCopy* pRegions ) const
    {
      return ::vkCmdCopyImage( commandBuffer, srcImage, srcImageLayout, dstImage, dstImageLayout, regionCount, pRegions );
    }

    void vkCmdCopyImageToBuffer( VkCommandBuffer commandBuffer, VkImage srcImage, VkImageLayout srcImageLayout, VkBuffer dstBuffer, uint32_t regionCount, const VkBufferImageCopy* pRegions ) const
    {
      return ::vkCmdCopyImageToBuffer( commandBuffer, srcImage, srcImageLayout, dstBuffer, regionCount, pRegions );
    }

    void vkCmdCopyQueryPoolResults( VkCommandBuffer commandBuffer, VkQueryPool queryPool, uint32_t firstQuery, uint32_t queryCount, VkBuffer dstBuffer, VkDeviceSize dstOffset, VkDeviceSize stride, VkQueryResultFlags flags ) const
    {
      return ::vkCmdCopyQueryPoolResults( commandBuffer, queryPool, firstQuery, queryCount, dstBuffer, dstOffset, stride, flags );
    }

    void vkCmdDebugMarkerBeginEXT( VkCommandBuffer commandBuffer, const VkDebugMarkerMarkerInfoEXT* pMarkerInfo ) const
    {
      return ::vkCmdDebugMarkerBeginEXT( commandBuffer, pMarkerInfo );
    }

    void vkCmdDebugMarkerEndEXT( VkCommandBuffer commandBuffer ) const
    {
      return ::vkCmdDebugMarkerEndEXT( commandBuffer );
    }

    void vkCmdDebugMarkerInsertEXT( VkCommandBuffer commandBuffer, const VkDebugMarkerMarkerInfoEXT* pMarkerInfo ) const
    {
      return ::vkCmdDebugMarkerInsertEXT( commandBuffer, pMarkerInfo );
    }

    void vkCmdDispatch( VkCommandBuffer commandBuffer, uint32_t groupCountX, uint32_t groupCountY, uint32_t groupCountZ ) const
    {
      return ::vkCmdDispatch( commandBuffer, groupCountX, groupCountY, groupCountZ );
    }

    void vkCmdDispatchBase( VkCommandBuffer commandBuffer, uint32_t baseGroupX, uint32_t baseGroupY, uint32_t baseGroupZ, uint32_t groupCountX, uint32_t groupCountY, uint32_t groupCountZ ) const
    {
      return ::vkCmdDispatchBase( commandBuffer, baseGroupX, baseGroupY, baseGroupZ, groupCountX, groupCountY, groupCountZ );
    }

    void vkCmdDispatchBaseKHR( VkCommandBuffer commandBuffer, uint32_t baseGroupX, uint32_t baseGroupY, uint32_t baseGroupZ, uint32_t groupCountX, uint32_t groupCountY, uint32_t groupCountZ ) const
    {
      return ::vkCmdDispatchBaseKHR( commandBuffer, baseGroupX, baseGroupY, baseGroupZ, groupCountX, groupCountY, groupCountZ );
    }

    void vkCmdDispatchIndirect( VkCommandBuffer commandBuffer, VkBuffer buffer, VkDeviceSize offset ) const
    {
      return ::vkCmdDispatchIndirect( commandBuffer, buffer, offset );
    }

    void vkCmdDraw( VkCommandBuffer commandBuffer, uint32_t vertexCount, uint32_t instanceCount, uint32_t firstVertex, uint32_t firstInstance ) const
    {
      return ::vkCmdDraw( commandBuffer, vertexCount, instanceCount, firstVertex, firstInstance );
    }

    void vkCmdDrawIndexed( VkCommandBuffer commandBuffer, uint32_t indexCount, uint32_t instanceCount, uint32_t firstIndex, int32_t vertexOffset, uint32_t firstInstance ) const
    {
      return ::vkCmdDrawIndexed( commandBuffer, indexCount, instanceCount, firstIndex, vertexOffset, firstInstance );
    }

    void vkCmdDrawIndexedIndirect( VkCommandBuffer commandBuffer, VkBuffer buffer, VkDeviceSize offset, uint32_t drawCount, uint32_t stride ) const
    {
      return ::vkCmdDrawIndexedIndirect( commandBuffer, buffer, offset, drawCount, stride );
    }

    void vkCmdDrawIndexedIndirectCountAMD( VkCommandBuffer commandBuffer, VkBuffer buffer, VkDeviceSize offset, VkBuffer countBuffer, VkDeviceSize countBufferOffset, uint32_t maxDrawCount, uint32_t stride ) const
    {
      return ::vkCmdDrawIndexedIndirectCountAMD( commandBuffer, buffer, offset, countBuffer, countBufferOffset, maxDrawCount, stride );
    }

    void vkCmdDrawIndexedIndirectCountKHR( VkCommandBuffer commandBuffer, VkBuffer buffer, VkDeviceSize offset, VkBuffer countBuffer, VkDeviceSize countBufferOffset, uint32_t maxDrawCount, uint32_t stride ) const
    {
      return ::vkCmdDrawIndexedIndirectCountKHR( commandBuffer, buffer, offset, countBuffer, countBufferOffset, maxDrawCount, stride );
    }

    void vkCmdDrawIndirect( VkCommandBuffer commandBuffer, VkBuffer buffer, VkDeviceSize offset, uint32_t drawCount, uint32_t stride ) const
    {
      return ::vkCmdDrawIndirect( commandBuffer, buffer, offset, drawCount, stride );
    }

    void vkCmdDrawIndirectByteCountEXT( VkCommandBuffer commandBuffer, uint32_t instanceCount, uint32_t firstInstance, VkBuffer counterBuffer, VkDeviceSize counterBufferOffset, uint32_t counterOffset, uint32_t vertexStride ) const
    {
      return ::vkCmdDrawIndirectByteCountEXT( commandBuffer, instanceCount, firstInstance, counterBuffer, counterBufferOffset, counterOffset, vertexStride );
    }

    void vkCmdDrawIndirectCountAMD( VkCommandBuffer commandBuffer, VkBuffer buffer, VkDeviceSize offset, VkBuffer countBuffer, VkDeviceSize countBufferOffset, uint32_t maxDrawCount, uint32_t stride ) const
    {
      return ::vkCmdDrawIndirectCountAMD( commandBuffer, buffer, offset, countBuffer, countBufferOffset, maxDrawCount, stride );
    }

    void vkCmdDrawIndirectCountKHR( VkCommandBuffer commandBuffer, VkBuffer buffer, VkDeviceSize offset, VkBuffer countBuffer, VkDeviceSize countBufferOffset, uint32_t maxDrawCount, uint32_t stride ) const
    {
      return ::vkCmdDrawIndirectCountKHR( commandBuffer, buffer, offset, countBuffer, countBufferOffset, maxDrawCount, stride );
    }

    void vkCmdDrawMeshTasksIndirectCountNV( VkCommandBuffer commandBuffer, VkBuffer buffer, VkDeviceSize offset, VkBuffer countBuffer, VkDeviceSize countBufferOffset, uint32_t maxDrawCount, uint32_t stride ) const
    {
      return ::vkCmdDrawMeshTasksIndirectCountNV( commandBuffer, buffer, offset, countBuffer, countBufferOffset, maxDrawCount, stride );
    }

    void vkCmdDrawMeshTasksIndirectNV( VkCommandBuffer commandBuffer, VkBuffer buffer, VkDeviceSize offset, uint32_t drawCount, uint32_t stride ) const
    {
      return ::vkCmdDrawMeshTasksIndirectNV( commandBuffer, buffer, offset, drawCount, stride );
    }

    void vkCmdDrawMeshTasksNV( VkCommandBuffer commandBuffer, uint32_t taskCount, uint32_t firstTask ) const
    {
      return ::vkCmdDrawMeshTasksNV( commandBuffer, taskCount, firstTask );
    }

    void vkCmdEndConditionalRenderingEXT( VkCommandBuffer commandBuffer ) const
    {
      return ::vkCmdEndConditionalRenderingEXT( commandBuffer );
    }

    void vkCmdEndDebugUtilsLabelEXT( VkCommandBuffer commandBuffer ) const
    {
      return ::vkCmdEndDebugUtilsLabelEXT( commandBuffer );
    }

    void vkCmdEndQuery( VkCommandBuffer commandBuffer, VkQueryPool queryPool, uint32_t query ) const
    {
      return ::vkCmdEndQuery( commandBuffer, queryPool, query );
    }

    void vkCmdEndQueryIndexedEXT( VkCommandBuffer commandBuffer, VkQueryPool queryPool, uint32_t query, uint32_t index ) const
    {
      return ::vkCmdEndQueryIndexedEXT( commandBuffer, queryPool, query, index );
    }

    void vkCmdEndRenderPass( VkCommandBuffer commandBuffer ) const
    {
      return ::vkCmdEndRenderPass( commandBuffer );
    }

    void vkCmdEndRenderPass2KHR( VkCommandBuffer commandBuffer, const VkSubpassEndInfoKHR* pSubpassEndInfo ) const
    {
      return ::vkCmdEndRenderPass2KHR( commandBuffer, pSubpassEndInfo );
    }

    void vkCmdEndTransformFeedbackEXT( VkCommandBuffer commandBuffer, uint32_t firstCounterBuffer, uint32_t counterBufferCount, const VkBuffer* pCounterBuffers, const VkDeviceSize* pCounterBufferOffsets ) const
    {
      return ::vkCmdEndTransformFeedbackEXT( commandBuffer, firstCounterBuffer, counterBufferCount, pCounterBuffers, pCounterBufferOffsets );
    }

    void vkCmdExecuteCommands( VkCommandBuffer commandBuffer, uint32_t commandBufferCount, const VkCommandBuffer* pCommandBuffers ) const
    {
      return ::vkCmdExecuteCommands( commandBuffer, commandBufferCount, pCommandBuffers );
    }

    void vkCmdFillBuffer( VkCommandBuffer commandBuffer, VkBuffer dstBuffer, VkDeviceSize dstOffset, VkDeviceSize size, uint32_t data ) const
    {
      return ::vkCmdFillBuffer( commandBuffer, dstBuffer, dstOffset, size, data );
    }

    void vkCmdInsertDebugUtilsLabelEXT( VkCommandBuffer commandBuffer, const VkDebugUtilsLabelEXT* pLabelInfo ) const
    {
      return ::vkCmdInsertDebugUtilsLabelEXT( commandBuffer, pLabelInfo );
    }

    void vkCmdNextSubpass( VkCommandBuffer commandBuffer, VkSubpassContents contents ) const
    {
      return ::vkCmdNextSubpass( commandBuffer, contents );
    }

    void vkCmdNextSubpass2KHR( VkCommandBuffer commandBuffer, const VkSubpassBeginInfoKHR* pSubpassBeginInfo, const VkSubpassEndInfoKHR* pSubpassEndInfo ) const
    {
      return ::vkCmdNextSubpass2KHR( commandBuffer, pSubpassBeginInfo, pSubpassEndInfo );
    }

    void vkCmdPipelineBarrier( VkCommandBuffer commandBuffer, VkPipelineStageFlags srcStageMask, VkPipelineStageFlags dstStageMask, VkDependencyFlags dependencyFlags, uint32_t memoryBarrierCount, const VkMemoryBarrier* pMemoryBarriers, uint32_t bufferMemoryBarrierCount, const VkBufferMemoryBarrier* pBufferMemoryBarriers, uint32_t imageMemoryBarrierCount, const VkImageMemoryBarrier* pImageMemoryBarriers ) const
    {
      return ::vkCmdPipelineBarrier( commandBuffer, srcStageMask, dstStageMask, dependencyFlags, memoryBarrierCount, pMemoryBarriers, bufferMemoryBarrierCount, pBufferMemoryBarriers, imageMemoryBarrierCount, pImageMemoryBarriers );
    }

    void vkCmdProcessCommandsNVX( VkCommandBuffer commandBuffer, const VkCmdProcessCommandsInfoNVX* pProcessCommandsInfo ) const
    {
      return ::vkCmdProcessCommandsNVX( commandBuffer, pProcessCommandsInfo );
    }

    void vkCmdPushConstants( VkCommandBuffer commandBuffer, VkPipelineLayout layout, VkShaderStageFlags stageFlags, uint32_t offset, uint32_t size, const void* pValues ) const
    {
      return ::vkCmdPushConstants( commandBuffer, layout, stageFlags, offset, size, pValues );
    }

    void vkCmdPushDescriptorSetKHR( VkCommandBuffer commandBuffer, VkPipelineBindPoint pipelineBindPoint, VkPipelineLayout layout, uint32_t set, uint32_t descriptorWriteCount, const VkWriteDescriptorSet* pDescriptorWrites ) const
    {
      return ::vkCmdPushDescriptorSetKHR( commandBuffer, pipelineBindPoint, layout, set, descriptorWriteCount, pDescriptorWrites );
    }

    void vkCmdPushDescriptorSetWithTemplateKHR( VkCommandBuffer commandBuffer, VkDescriptorUpdateTemplate descriptorUpdateTemplate, VkPipelineLayout layout, uint32_t set, const void* pData ) const
    {
      return ::vkCmdPushDescriptorSetWithTemplateKHR( commandBuffer, descriptorUpdateTemplate, layout, set, pData );
    }

    void vkCmdReserveSpaceForCommandsNVX( VkCommandBuffer commandBuffer, const VkCmdReserveSpaceForCommandsInfoNVX* pReserveSpaceInfo ) const
    {
      return ::vkCmdReserveSpaceForCommandsNVX( commandBuffer, pReserveSpaceInfo );
    }

    void vkCmdResetEvent( VkCommandBuffer commandBuffer, VkEvent event, VkPipelineStageFlags stageMask ) const
    {
      return ::vkCmdResetEvent( commandBuffer, event, stageMask );
    }

    void vkCmdResetQueryPool( VkCommandBuffer commandBuffer, VkQueryPool queryPool, uint32_t firstQuery, uint32_t queryCount ) const
    {
      return ::vkCmdResetQueryPool( commandBuffer, queryPool, firstQuery, queryCount );
    }

    void vkCmdResolveImage( VkCommandBuffer commandBuffer, VkImage srcImage, VkImageLayout srcImageLayout, VkImage dstImage, VkImageLayout dstImageLayout, uint32_t regionCount, const VkImageResolve* pRegions ) const
    {
      return ::vkCmdResolveImage( commandBuffer, srcImage, srcImageLayout, dstImage, dstImageLayout, regionCount, pRegions );
    }

    void vkCmdSetBlendConstants( VkCommandBuffer commandBuffer, const float blendConstants[4] ) const
    {
      return ::vkCmdSetBlendConstants( commandBuffer, blendConstants );
    }

    void vkCmdSetCheckpointNV( VkCommandBuffer commandBuffer, const void* pCheckpointMarker ) const
    {
      return ::vkCmdSetCheckpointNV( commandBuffer, pCheckpointMarker );
    }

    void vkCmdSetCoarseSampleOrderNV( VkCommandBuffer commandBuffer, VkCoarseSampleOrderTypeNV sampleOrderType, uint32_t customSampleOrderCount, const VkCoarseSampleOrderCustomNV* pCustomSampleOrders ) const
    {
      return ::vkCmdSetCoarseSampleOrderNV( commandBuffer, sampleOrderType, customSampleOrderCount, pCustomSampleOrders );
    }

    void vkCmdSetDepthBias( VkCommandBuffer commandBuffer, float depthBiasConstantFactor, float depthBiasClamp, float depthBiasSlopeFactor ) const
    {
      return ::vkCmdSetDepthBias( commandBuffer, depthBiasConstantFactor, depthBiasClamp, depthBiasSlopeFactor );
    }

    void vkCmdSetDepthBounds( VkCommandBuffer commandBuffer, float minDepthBounds, float maxDepthBounds ) const
    {
      return ::vkCmdSetDepthBounds( commandBuffer, minDepthBounds, maxDepthBounds );
    }

    void vkCmdSetDeviceMask( VkCommandBuffer commandBuffer, uint32_t deviceMask ) const
    {
      return ::vkCmdSetDeviceMask( commandBuffer, deviceMask );
    }

    void vkCmdSetDeviceMaskKHR( VkCommandBuffer commandBuffer, uint32_t deviceMask ) const
    {
      return ::vkCmdSetDeviceMaskKHR( commandBuffer, deviceMask );
    }

    void vkCmdSetDiscardRectangleEXT( VkCommandBuffer commandBuffer, uint32_t firstDiscardRectangle, uint32_t discardRectangleCount, const VkRect2D* pDiscardRectangles ) const
    {
      return ::vkCmdSetDiscardRectangleEXT( commandBuffer, firstDiscardRectangle, discardRectangleCount, pDiscardRectangles );
    }

    void vkCmdSetEvent( VkCommandBuffer commandBuffer, VkEvent event, VkPipelineStageFlags stageMask ) const
    {
      return ::vkCmdSetEvent( commandBuffer, event, stageMask );
    }

    void vkCmdSetExclusiveScissorNV( VkCommandBuffer commandBuffer, uint32_t firstExclusiveScissor, uint32_t exclusiveScissorCount, const VkRect2D* pExclusiveScissors ) const
    {
      return ::vkCmdSetExclusiveScissorNV( commandBuffer, firstExclusiveScissor, exclusiveScissorCount, pExclusiveScissors );
    }

    void vkCmdSetLineWidth( VkCommandBuffer commandBuffer, float lineWidth ) const
    {
      return ::vkCmdSetLineWidth( commandBuffer, lineWidth );
    }

    void vkCmdSetSampleLocationsEXT( VkCommandBuffer commandBuffer, const VkSampleLocationsInfoEXT* pSampleLocationsInfo ) const
    {
      return ::vkCmdSetSampleLocationsEXT( commandBuffer, pSampleLocationsInfo );
    }

    void vkCmdSetScissor( VkCommandBuffer commandBuffer, uint32_t firstScissor, uint32_t scissorCount, const VkRect2D* pScissors ) const
    {
      return ::vkCmdSetScissor( commandBuffer, firstScissor, scissorCount, pScissors );
    }

    void vkCmdSetStencilCompareMask( VkCommandBuffer commandBuffer, VkStencilFaceFlags faceMask, uint32_t compareMask ) const
    {
      return ::vkCmdSetStencilCompareMask( commandBuffer, faceMask, compareMask );
    }

    void vkCmdSetStencilReference( VkCommandBuffer commandBuffer, VkStencilFaceFlags faceMask, uint32_t reference ) const
    {
      return ::vkCmdSetStencilReference( commandBuffer, faceMask, reference );
    }

    void vkCmdSetStencilWriteMask( VkCommandBuffer commandBuffer, VkStencilFaceFlags faceMask, uint32_t writeMask ) const
    {
      return ::vkCmdSetStencilWriteMask( commandBuffer, faceMask, writeMask );
    }

    void vkCmdSetViewport( VkCommandBuffer commandBuffer, uint32_t firstViewport, uint32_t viewportCount, const VkViewport* pViewports ) const
    {
      return ::vkCmdSetViewport( commandBuffer, firstViewport, viewportCount, pViewports );
    }

    void vkCmdSetViewportShadingRatePaletteNV( VkCommandBuffer commandBuffer, uint32_t firstViewport, uint32_t viewportCount, const VkShadingRatePaletteNV* pShadingRatePalettes ) const
    {
      return ::vkCmdSetViewportShadingRatePaletteNV( commandBuffer, firstViewport, viewportCount, pShadingRatePalettes );
    }

    void vkCmdSetViewportWScalingNV( VkCommandBuffer commandBuffer, uint32_t firstViewport, uint32_t viewportCount, const VkViewportWScalingNV* pViewportWScalings ) const
    {
      return ::vkCmdSetViewportWScalingNV( commandBuffer, firstViewport, viewportCount, pViewportWScalings );
    }

    void vkCmdTraceRaysNV( VkCommandBuffer commandBuffer, VkBuffer raygenShaderBindingTableBuffer, VkDeviceSize raygenShaderBindingOffset, VkBuffer missShaderBindingTableBuffer, VkDeviceSize missShaderBindingOffset, VkDeviceSize missShaderBindingStride, VkBuffer hitShaderBindingTableBuffer, VkDeviceSize hitShaderBindingOffset, VkDeviceSize hitShaderBindingStride, VkBuffer callableShaderBindingTableBuffer, VkDeviceSize callableShaderBindingOffset, VkDeviceSize callableShaderBindingStride, uint32_t width, uint32_t height, uint32_t depth ) const
    {
      return ::vkCmdTraceRaysNV( commandBuffer, raygenShaderBindingTableBuffer, raygenShaderBindingOffset, missShaderBindingTableBuffer, missShaderBindingOffset, missShaderBindingStride, hitShaderBindingTableBuffer, hitShaderBindingOffset, hitShaderBindingStride, callableShaderBindingTableBuffer, callableShaderBindingOffset, callableShaderBindingStride, width, height, depth );
    }

    void vkCmdUpdateBuffer( VkCommandBuffer commandBuffer, VkBuffer dstBuffer, VkDeviceSize dstOffset, VkDeviceSize dataSize, const void* pData ) const
    {
      return ::vkCmdUpdateBuffer( commandBuffer, dstBuffer, dstOffset, dataSize, pData );
    }

    void vkCmdWaitEvents( VkCommandBuffer commandBuffer, uint32_t eventCount, const VkEvent* pEvents, VkPipelineStageFlags srcStageMask, VkPipelineStageFlags dstStageMask, uint32_t memoryBarrierCount, const VkMemoryBarrier* pMemoryBarriers, uint32_t bufferMemoryBarrierCount, const VkBufferMemoryBarrier* pBufferMemoryBarriers, uint32_t imageMemoryBarrierCount, const VkImageMemoryBarrier* pImageMemoryBarriers ) const
    {
      return ::vkCmdWaitEvents( commandBuffer, eventCount, pEvents, srcStageMask, dstStageMask, memoryBarrierCount, pMemoryBarriers, bufferMemoryBarrierCount, pBufferMemoryBarriers, imageMemoryBarrierCount, pImageMemoryBarriers );
    }

    void vkCmdWriteAccelerationStructuresPropertiesNV( VkCommandBuffer commandBuffer, uint32_t accelerationStructureCount, const VkAccelerationStructureNV* pAccelerationStructures, VkQueryType queryType, VkQueryPool queryPool, uint32_t firstQuery ) const
    {
      return ::vkCmdWriteAccelerationStructuresPropertiesNV( commandBuffer, accelerationStructureCount, pAccelerationStructures, queryType, queryPool, firstQuery );
    }

    void vkCmdWriteBufferMarkerAMD( VkCommandBuffer commandBuffer, VkPipelineStageFlagBits pipelineStage, VkBuffer dstBuffer, VkDeviceSize dstOffset, uint32_t marker ) const
    {
      return ::vkCmdWriteBufferMarkerAMD( commandBuffer, pipelineStage, dstBuffer, dstOffset, marker );
    }

    void vkCmdWriteTimestamp( VkCommandBuffer commandBuffer, VkPipelineStageFlagBits pipelineStage, VkQueryPool queryPool, uint32_t query ) const
    {
      return ::vkCmdWriteTimestamp( commandBuffer, pipelineStage, queryPool, query );
    }

    VkResult vkEndCommandBuffer( VkCommandBuffer commandBuffer ) const
    {
      return ::vkEndCommandBuffer( commandBuffer );
    }

    VkResult vkResetCommandBuffer( VkCommandBuffer commandBuffer, VkCommandBufferResetFlags flags ) const
    {
      return ::vkResetCommandBuffer( commandBuffer, flags );
    }

    VkResult vkAcquireNextImage2KHR( VkDevice device, const VkAcquireNextImageInfoKHR* pAcquireInfo, uint32_t* pImageIndex ) const
    {
      return ::vkAcquireNextImage2KHR( device, pAcquireInfo, pImageIndex );
    }

    VkResult vkAcquireNextImageKHR( VkDevice device, VkSwapchainKHR swapchain, uint64_t timeout, VkSemaphore semaphore, VkFence fence, uint32_t* pImageIndex ) const
    {
      return ::vkAcquireNextImageKHR( device, swapchain, timeout, semaphore, fence, pImageIndex );
    }

    VkResult vkAllocateCommandBuffers( VkDevice device, const VkCommandBufferAllocateInfo* pAllocateInfo, VkCommandBuffer* pCommandBuffers ) const
    {
      return ::vkAllocateCommandBuffers( device, pAllocateInfo, pCommandBuffers );
    }

    VkResult vkAllocateDescriptorSets( VkDevice device, const VkDescriptorSetAllocateInfo* pAllocateInfo, VkDescriptorSet* pDescriptorSets ) const
    {
      return ::vkAllocateDescriptorSets( device, pAllocateInfo, pDescriptorSets );
    }

    VkResult vkAllocateMemory( VkDevice device, const VkMemoryAllocateInfo* pAllocateInfo, const VkAllocationCallbacks* pAllocator, VkDeviceMemory* pMemory ) const
    {
      return ::vkAllocateMemory( device, pAllocateInfo, pAllocator, pMemory );
    }

    VkResult vkBindAccelerationStructureMemoryNV( VkDevice device, uint32_t bindInfoCount, const VkBindAccelerationStructureMemoryInfoNV* pBindInfos ) const
    {
      return ::vkBindAccelerationStructureMemoryNV( device, bindInfoCount, pBindInfos );
    }

    VkResult vkBindBufferMemory( VkDevice device, VkBuffer buffer, VkDeviceMemory memory, VkDeviceSize memoryOffset ) const
    {
      return ::vkBindBufferMemory( device, buffer, memory, memoryOffset );
    }

    VkResult vkBindBufferMemory2( VkDevice device, uint32_t bindInfoCount, const VkBindBufferMemoryInfo* pBindInfos ) const
    {
      return ::vkBindBufferMemory2( device, bindInfoCount, pBindInfos );
    }

    VkResult vkBindBufferMemory2KHR( VkDevice device, uint32_t bindInfoCount, const VkBindBufferMemoryInfo* pBindInfos ) const
    {
      return ::vkBindBufferMemory2KHR( device, bindInfoCount, pBindInfos );
    }

    VkResult vkBindImageMemory( VkDevice device, VkImage image, VkDeviceMemory memory, VkDeviceSize memoryOffset ) const
    {
      return ::vkBindImageMemory( device, image, memory, memoryOffset );
    }

    VkResult vkBindImageMemory2( VkDevice device, uint32_t bindInfoCount, const VkBindImageMemoryInfo* pBindInfos ) const
    {
      return ::vkBindImageMemory2( device, bindInfoCount, pBindInfos );
    }

    VkResult vkBindImageMemory2KHR( VkDevice device, uint32_t bindInfoCount, const VkBindImageMemoryInfo* pBindInfos ) const
    {
      return ::vkBindImageMemory2KHR( device, bindInfoCount, pBindInfos );
    }

    VkResult vkCompileDeferredNV( VkDevice device, VkPipeline pipeline, uint32_t shader ) const
    {
      return ::vkCompileDeferredNV( device, pipeline, shader );
    }

    VkResult vkCreateAccelerationStructureNV( VkDevice device, const VkAccelerationStructureCreateInfoNV* pCreateInfo, const VkAllocationCallbacks* pAllocator, VkAccelerationStructureNV* pAccelerationStructure ) const
    {
      return ::vkCreateAccelerationStructureNV( device, pCreateInfo, pAllocator, pAccelerationStructure );
    }

    VkResult vkCreateBuffer( VkDevice device, const VkBufferCreateInfo* pCreateInfo, const VkAllocationCallbacks* pAllocator, VkBuffer* pBuffer ) const
    {
      return ::vkCreateBuffer( device, pCreateInfo, pAllocator, pBuffer );
    }

    VkResult vkCreateBufferView( VkDevice device, const VkBufferViewCreateInfo* pCreateInfo, const VkAllocationCallbacks* pAllocator, VkBufferView* pView ) const
    {
      return ::vkCreateBufferView( device, pCreateInfo, pAllocator, pView );
    }

    VkResult vkCreateCommandPool( VkDevice device, const VkCommandPoolCreateInfo* pCreateInfo, const VkAllocationCallbacks* pAllocator, VkCommandPool* pCommandPool ) const
    {
      return ::vkCreateCommandPool( device, pCreateInfo, pAllocator, pCommandPool );
    }

    VkResult vkCreateComputePipelines( VkDevice device, VkPipelineCache pipelineCache, uint32_t createInfoCount, const VkComputePipelineCreateInfo* pCreateInfos, const VkAllocationCallbacks* pAllocator, VkPipeline* pPipelines ) const
    {
      return ::vkCreateComputePipelines( device, pipelineCache, createInfoCount, pCreateInfos, pAllocator, pPipelines );
    }

    VkResult vkCreateDescriptorPool( VkDevice device, const VkDescriptorPoolCreateInfo* pCreateInfo, const VkAllocationCallbacks* pAllocator, VkDescriptorPool* pDescriptorPool ) const
    {
      return ::vkCreateDescriptorPool( device, pCreateInfo, pAllocator, pDescriptorPool );
    }

    VkResult vkCreateDescriptorSetLayout( VkDevice device, const VkDescriptorSetLayoutCreateInfo* pCreateInfo, const VkAllocationCallbacks* pAllocator, VkDescriptorSetLayout* pSetLayout ) const
    {
      return ::vkCreateDescriptorSetLayout( device, pCreateInfo, pAllocator, pSetLayout );
    }

    VkResult vkCreateDescriptorUpdateTemplate( VkDevice device, const VkDescriptorUpdateTemplateCreateInfo* pCreateInfo, const VkAllocationCallbacks* pAllocator, VkDescriptorUpdateTemplate* pDescriptorUpdateTemplate ) const
    {
      return ::vkCreateDescriptorUpdateTemplate( device, pCreateInfo, pAllocator, pDescriptorUpdateTemplate );
    }

    VkResult vkCreateDescriptorUpdateTemplateKHR( VkDevice device, const VkDescriptorUpdateTemplateCreateInfo* pCreateInfo, const VkAllocationCallbacks* pAllocator, VkDescriptorUpdateTemplate* pDescriptorUpdateTemplate ) const
    {
      return ::vkCreateDescriptorUpdateTemplateKHR( device, pCreateInfo, pAllocator, pDescriptorUpdateTemplate );
    }

    VkResult vkCreateEvent( VkDevice device, const VkEventCreateInfo* pCreateInfo, const VkAllocationCallbacks* pAllocator, VkEvent* pEvent ) const
    {
      return ::vkCreateEvent( device, pCreateInfo, pAllocator, pEvent );
    }

    VkResult vkCreateFence( VkDevice device, const VkFenceCreateInfo* pCreateInfo, const VkAllocationCallbacks* pAllocator, VkFence* pFence ) const
    {
      return ::vkCreateFence( device, pCreateInfo, pAllocator, pFence );
    }

    VkResult vkCreateFramebuffer( VkDevice device, const VkFramebufferCreateInfo* pCreateInfo, const VkAllocationCallbacks* pAllocator, VkFramebuffer* pFramebuffer ) const
    {
      return ::vkCreateFramebuffer( device, pCreateInfo, pAllocator, pFramebuffer );
    }

    VkResult vkCreateGraphicsPipelines( VkDevice device, VkPipelineCache pipelineCache, uint32_t createInfoCount, const VkGraphicsPipelineCreateInfo* pCreateInfos, const VkAllocationCallbacks* pAllocator, VkPipeline* pPipelines ) const
    {
      return ::vkCreateGraphicsPipelines( device, pipelineCache, createInfoCount, pCreateInfos, pAllocator, pPipelines );
    }

    VkResult vkCreateImage( VkDevice device, const VkImageCreateInfo* pCreateInfo, const VkAllocationCallbacks* pAllocator, VkImage* pImage ) const
    {
      return ::vkCreateImage( device, pCreateInfo, pAllocator, pImage );
    }

    VkResult vkCreateImageView( VkDevice device, const VkImageViewCreateInfo* pCreateInfo, const VkAllocationCallbacks* pAllocator, VkImageView* pView ) const
    {
      return ::vkCreateImageView( device, pCreateInfo, pAllocator, pView );
    }

    VkResult vkCreateIndirectCommandsLayoutNVX( VkDevice device, const VkIndirectCommandsLayoutCreateInfoNVX* pCreateInfo, const VkAllocationCallbacks* pAllocator, VkIndirectCommandsLayoutNVX* pIndirectCommandsLayout ) const
    {
      return ::vkCreateIndirectCommandsLayoutNVX( device, pCreateInfo, pAllocator, pIndirectCommandsLayout );
    }

    VkResult vkCreateObjectTableNVX( VkDevice device, const VkObjectTableCreateInfoNVX* pCreateInfo, const VkAllocationCallbacks* pAllocator, VkObjectTableNVX* pObjectTable ) const
    {
      return ::vkCreateObjectTableNVX( device, pCreateInfo, pAllocator, pObjectTable );
    }

    VkResult vkCreatePipelineCache( VkDevice device, const VkPipelineCacheCreateInfo* pCreateInfo, const VkAllocationCallbacks* pAllocator, VkPipelineCache* pPipelineCache ) const
    {
      return ::vkCreatePipelineCache( device, pCreateInfo, pAllocator, pPipelineCache );
    }

    VkResult vkCreatePipelineLayout( VkDevice device, const VkPipelineLayoutCreateInfo* pCreateInfo, const VkAllocationCallbacks* pAllocator, VkPipelineLayout* pPipelineLayout ) const
    {
      return ::vkCreatePipelineLayout( device, pCreateInfo, pAllocator, pPipelineLayout );
    }

    VkResult vkCreateQueryPool( VkDevice device, const VkQueryPoolCreateInfo* pCreateInfo, const VkAllocationCallbacks* pAllocator, VkQueryPool* pQueryPool ) const
    {
      return ::vkCreateQueryPool( device, pCreateInfo, pAllocator, pQueryPool );
    }

    VkResult vkCreateRayTracingPipelinesNV( VkDevice device, VkPipelineCache pipelineCache, uint32_t createInfoCount, const VkRayTracingPipelineCreateInfoNV* pCreateInfos, const VkAllocationCallbacks* pAllocator, VkPipeline* pPipelines ) const
    {
      return ::vkCreateRayTracingPipelinesNV( device, pipelineCache, createInfoCount, pCreateInfos, pAllocator, pPipelines );
    }

    VkResult vkCreateRenderPass( VkDevice device, const VkRenderPassCreateInfo* pCreateInfo, const VkAllocationCallbacks* pAllocator, VkRenderPass* pRenderPass ) const
    {
      return ::vkCreateRenderPass( device, pCreateInfo, pAllocator, pRenderPass );
    }

    VkResult vkCreateRenderPass2KHR( VkDevice device, const VkRenderPassCreateInfo2KHR* pCreateInfo, const VkAllocationCallbacks* pAllocator, VkRenderPass* pRenderPass ) const
    {
      return ::vkCreateRenderPass2KHR( device, pCreateInfo, pAllocator, pRenderPass );
    }

    VkResult vkCreateSampler( VkDevice device, const VkSamplerCreateInfo* pCreateInfo, const VkAllocationCallbacks* pAllocator, VkSampler* pSampler ) const
    {
      return ::vkCreateSampler( device, pCreateInfo, pAllocator, pSampler );
    }

    VkResult vkCreateSamplerYcbcrConversion( VkDevice device, const VkSamplerYcbcrConversionCreateInfo* pCreateInfo, const VkAllocationCallbacks* pAllocator, VkSamplerYcbcrConversion* pYcbcrConversion ) const
    {
      return ::vkCreateSamplerYcbcrConversion( device, pCreateInfo, pAllocator, pYcbcrConversion );
    }

    VkResult vkCreateSamplerYcbcrConversionKHR( VkDevice device, const VkSamplerYcbcrConversionCreateInfo* pCreateInfo, const VkAllocationCallbacks* pAllocator, VkSamplerYcbcrConversion* pYcbcrConversion ) const
    {
      return ::vkCreateSamplerYcbcrConversionKHR( device, pCreateInfo, pAllocator, pYcbcrConversion );
    }

    VkResult vkCreateSemaphore( VkDevice device, const VkSemaphoreCreateInfo* pCreateInfo, const VkAllocationCallbacks* pAllocator, VkSemaphore* pSemaphore ) const
    {
      return ::vkCreateSemaphore( device, pCreateInfo, pAllocator, pSemaphore );
    }

    VkResult vkCreateShaderModule( VkDevice device, const VkShaderModuleCreateInfo* pCreateInfo, const VkAllocationCallbacks* pAllocator, VkShaderModule* pShaderModule ) const
    {
      return ::vkCreateShaderModule( device, pCreateInfo, pAllocator, pShaderModule );
    }

    VkResult vkCreateSharedSwapchainsKHR( VkDevice device, uint32_t swapchainCount, const VkSwapchainCreateInfoKHR* pCreateInfos, const VkAllocationCallbacks* pAllocator, VkSwapchainKHR* pSwapchains ) const
    {
      return ::vkCreateSharedSwapchainsKHR( device, swapchainCount, pCreateInfos, pAllocator, pSwapchains );
    }

    VkResult vkCreateSwapchainKHR( VkDevice device, const VkSwapchainCreateInfoKHR* pCreateInfo, const VkAllocationCallbacks* pAllocator, VkSwapchainKHR* pSwapchain ) const
    {
      return ::vkCreateSwapchainKHR( device, pCreateInfo, pAllocator, pSwapchain );
    }

    VkResult vkCreateValidationCacheEXT( VkDevice device, const VkValidationCacheCreateInfoEXT* pCreateInfo, const VkAllocationCallbacks* pAllocator, VkValidationCacheEXT* pValidationCache ) const
    {
      return ::vkCreateValidationCacheEXT( device, pCreateInfo, pAllocator, pValidationCache );
    }

    VkResult vkDebugMarkerSetObjectNameEXT( VkDevice device, const VkDebugMarkerObjectNameInfoEXT* pNameInfo ) const
    {
      return ::vkDebugMarkerSetObjectNameEXT( device, pNameInfo );
    }

    VkResult vkDebugMarkerSetObjectTagEXT( VkDevice device, const VkDebugMarkerObjectTagInfoEXT* pTagInfo ) const
    {
      return ::vkDebugMarkerSetObjectTagEXT( device, pTagInfo );
    }

    void vkDestroyAccelerationStructureNV( VkDevice device, VkAccelerationStructureNV accelerationStructure, const VkAllocationCallbacks* pAllocator ) const
    {
      return ::vkDestroyAccelerationStructureNV( device, accelerationStructure, pAllocator );
    }

    void vkDestroyBuffer( VkDevice device, VkBuffer buffer, const VkAllocationCallbacks* pAllocator ) const
    {
      return ::vkDestroyBuffer( device, buffer, pAllocator );
    }

    void vkDestroyBufferView( VkDevice device, VkBufferView bufferView, const VkAllocationCallbacks* pAllocator ) const
    {
      return ::vkDestroyBufferView( device, bufferView, pAllocator );
    }

    void vkDestroyCommandPool( VkDevice device, VkCommandPool commandPool, const VkAllocationCallbacks* pAllocator ) const
    {
      return ::vkDestroyCommandPool( device, commandPool, pAllocator );
    }

    void vkDestroyDescriptorPool( VkDevice device, VkDescriptorPool descriptorPool, const VkAllocationCallbacks* pAllocator ) const
    {
      return ::vkDestroyDescriptorPool( device, descriptorPool, pAllocator );
    }

    void vkDestroyDescriptorSetLayout( VkDevice device, VkDescriptorSetLayout descriptorSetLayout, const VkAllocationCallbacks* pAllocator ) const
    {
      return ::vkDestroyDescriptorSetLayout( device, descriptorSetLayout, pAllocator );
    }

    void vkDestroyDescriptorUpdateTemplate( VkDevice device, VkDescriptorUpdateTemplate descriptorUpdateTemplate, const VkAllocationCallbacks* pAllocator ) const
    {
      return ::vkDestroyDescriptorUpdateTemplate( device, descriptorUpdateTemplate, pAllocator );
    }

    void vkDestroyDescriptorUpdateTemplateKHR( VkDevice device, VkDescriptorUpdateTemplate descriptorUpdateTemplate, const VkAllocationCallbacks* pAllocator ) const
    {
      return ::vkDestroyDescriptorUpdateTemplateKHR( device, descriptorUpdateTemplate, pAllocator );
    }

    void vkDestroyDevice( VkDevice device, const VkAllocationCallbacks* pAllocator ) const
    {
      return ::vkDestroyDevice( device, pAllocator );
    }

    void vkDestroyEvent( VkDevice device, VkEvent event, const VkAllocationCallbacks* pAllocator ) const
    {
      return ::vkDestroyEvent( device, event, pAllocator );
    }

    void vkDestroyFence( VkDevice device, VkFence fence, const VkAllocationCallbacks* pAllocator ) const
    {
      return ::vkDestroyFence( device, fence, pAllocator );
    }

    void vkDestroyFramebuffer( VkDevice device, VkFramebuffer framebuffer, const VkAllocationCallbacks* pAllocator ) const
    {
      return ::vkDestroyFramebuffer( device, framebuffer, pAllocator );
    }

    void vkDestroyImage( VkDevice device, VkImage image, const VkAllocationCallbacks* pAllocator ) const
    {
      return ::vkDestroyImage( device, image, pAllocator );
    }

    void vkDestroyImageView( VkDevice device, VkImageView imageView, const VkAllocationCallbacks* pAllocator ) const
    {
      return ::vkDestroyImageView( device, imageView, pAllocator );
    }

    void vkDestroyIndirectCommandsLayoutNVX( VkDevice device, VkIndirectCommandsLayoutNVX indirectCommandsLayout, const VkAllocationCallbacks* pAllocator ) const
    {
      return ::vkDestroyIndirectCommandsLayoutNVX( device, indirectCommandsLayout, pAllocator );
    }

    void vkDestroyObjectTableNVX( VkDevice device, VkObjectTableNVX objectTable, const VkAllocationCallbacks* pAllocator ) const
    {
      return ::vkDestroyObjectTableNVX( device, objectTable, pAllocator );
    }

    void vkDestroyPipeline( VkDevice device, VkPipeline pipeline, const VkAllocationCallbacks* pAllocator ) const
    {
      return ::vkDestroyPipeline( device, pipeline, pAllocator );
    }

    void vkDestroyPipelineCache( VkDevice device, VkPipelineCache pipelineCache, const VkAllocationCallbacks* pAllocator ) const
    {
      return ::vkDestroyPipelineCache( device, pipelineCache, pAllocator );
    }

    void vkDestroyPipelineLayout( VkDevice device, VkPipelineLayout pipelineLayout, const VkAllocationCallbacks* pAllocator ) const
    {
      return ::vkDestroyPipelineLayout( device, pipelineLayout, pAllocator );
    }

    void vkDestroyQueryPool( VkDevice device, VkQueryPool queryPool, const VkAllocationCallbacks* pAllocator ) const
    {
      return ::vkDestroyQueryPool( device, queryPool, pAllocator );
    }

    void vkDestroyRenderPass( VkDevice device, VkRenderPass renderPass, const VkAllocationCallbacks* pAllocator ) const
    {
      return ::vkDestroyRenderPass( device, renderPass, pAllocator );
    }

    void vkDestroySampler( VkDevice device, VkSampler sampler, const VkAllocationCallbacks* pAllocator ) const
    {
      return ::vkDestroySampler( device, sampler, pAllocator );
    }

    void vkDestroySamplerYcbcrConversion( VkDevice device, VkSamplerYcbcrConversion ycbcrConversion, const VkAllocationCallbacks* pAllocator ) const
    {
      return ::vkDestroySamplerYcbcrConversion( device, ycbcrConversion, pAllocator );
    }

    void vkDestroySamplerYcbcrConversionKHR( VkDevice device, VkSamplerYcbcrConversion ycbcrConversion, const VkAllocationCallbacks* pAllocator ) const
    {
      return ::vkDestroySamplerYcbcrConversionKHR( device, ycbcrConversion, pAllocator );
    }

    void vkDestroySemaphore( VkDevice device, VkSemaphore semaphore, const VkAllocationCallbacks* pAllocator ) const
    {
      return ::vkDestroySemaphore( device, semaphore, pAllocator );
    }

    void vkDestroyShaderModule( VkDevice device, VkShaderModule shaderModule, const VkAllocationCallbacks* pAllocator ) const
    {
      return ::vkDestroyShaderModule( device, shaderModule, pAllocator );
    }

    void vkDestroySwapchainKHR( VkDevice device, VkSwapchainKHR swapchain, const VkAllocationCallbacks* pAllocator ) const
    {
      return ::vkDestroySwapchainKHR( device, swapchain, pAllocator );
    }

    void vkDestroyValidationCacheEXT( VkDevice device, VkValidationCacheEXT validationCache, const VkAllocationCallbacks* pAllocator ) const
    {
      return ::vkDestroyValidationCacheEXT( device, validationCache, pAllocator );
    }

    VkResult vkDeviceWaitIdle( VkDevice device ) const
    {
      return ::vkDeviceWaitIdle( device );
    }

    VkResult vkDisplayPowerControlEXT( VkDevice device, VkDisplayKHR display, const VkDisplayPowerInfoEXT* pDisplayPowerInfo ) const
    {
      return ::vkDisplayPowerControlEXT( device, display, pDisplayPowerInfo );
    }

    VkResult vkFlushMappedMemoryRanges( VkDevice device, uint32_t memoryRangeCount, const VkMappedMemoryRange* pMemoryRanges ) const
    {
      return ::vkFlushMappedMemoryRanges( device, memoryRangeCount, pMemoryRanges );
    }

    void vkFreeCommandBuffers( VkDevice device, VkCommandPool commandPool, uint32_t commandBufferCount, const VkCommandBuffer* pCommandBuffers ) const
    {
      return ::vkFreeCommandBuffers( device, commandPool, commandBufferCount, pCommandBuffers );
    }

    VkResult vkFreeDescriptorSets( VkDevice device, VkDescriptorPool descriptorPool, uint32_t descriptorSetCount, const VkDescriptorSet* pDescriptorSets ) const
    {
      return ::vkFreeDescriptorSets( device, descriptorPool, descriptorSetCount, pDescriptorSets );
    }

    void vkFreeMemory( VkDevice device, VkDeviceMemory memory, const VkAllocationCallbacks* pAllocator ) const
    {
      return ::vkFreeMemory( device, memory, pAllocator );
    }

    VkResult vkGetAccelerationStructureHandleNV( VkDevice device, VkAccelerationStructureNV accelerationStructure, size_t dataSize, void* pData ) const
    {
      return ::vkGetAccelerationStructureHandleNV( device, accelerationStructure, dataSize, pData );
    }

    void vkGetAccelerationStructureMemoryRequirementsNV( VkDevice device, const VkAccelerationStructureMemoryRequirementsInfoNV* pInfo, VkMemoryRequirements2KHR* pMemoryRequirements ) const
    {
      return ::vkGetAccelerationStructureMemoryRequirementsNV( device, pInfo, pMemoryRequirements );
    }

#ifdef VK_USE_PLATFORM_ANDROID_KHR
    VkResult vkGetAndroidHardwareBufferPropertiesANDROID( VkDevice device, const struct AHardwareBuffer* buffer, VkAndroidHardwareBufferPropertiesANDROID* pProperties ) const
    {
      return ::vkGetAndroidHardwareBufferPropertiesANDROID( device, buffer, pProperties );
    }
#endif /*VK_USE_PLATFORM_ANDROID_KHR*/

    VkDeviceAddress vkGetBufferDeviceAddressEXT( VkDevice device, const VkBufferDeviceAddressInfoEXT* pInfo ) const
    {
      return ::vkGetBufferDeviceAddressEXT( device, pInfo );
    }

    void vkGetBufferMemoryRequirements( VkDevice device, VkBuffer buffer, VkMemoryRequirements* pMemoryRequirements ) const
    {
      return ::vkGetBufferMemoryRequirements( device, buffer, pMemoryRequirements );
    }

    void vkGetBufferMemoryRequirements2( VkDevice device, const VkBufferMemoryRequirementsInfo2* pInfo, VkMemoryRequirements2* pMemoryRequirements ) const
    {
      return ::vkGetBufferMemoryRequirements2( device, pInfo, pMemoryRequirements );
    }

    void vkGetBufferMemoryRequirements2KHR( VkDevice device, const VkBufferMemoryRequirementsInfo2* pInfo, VkMemoryRequirements2* pMemoryRequirements ) const
    {
      return ::vkGetBufferMemoryRequirements2KHR( device, pInfo, pMemoryRequirements );
    }

    VkResult vkGetCalibratedTimestampsEXT( VkDevice device, uint32_t timestampCount, const VkCalibratedTimestampInfoEXT* pTimestampInfos, uint64_t* pTimestamps, uint64_t* pMaxDeviation ) const
    {
      return ::vkGetCalibratedTimestampsEXT( device, timestampCount, pTimestampInfos, pTimestamps, pMaxDeviation );
    }

    void vkGetDescriptorSetLayoutSupport( VkDevice device, const VkDescriptorSetLayoutCreateInfo* pCreateInfo, VkDescriptorSetLayoutSupport* pSupport ) const
    {
      return ::vkGetDescriptorSetLayoutSupport( device, pCreateInfo, pSupport );
    }

    void vkGetDescriptorSetLayoutSupportKHR( VkDevice device, const VkDescriptorSetLayoutCreateInfo* pCreateInfo, VkDescriptorSetLayoutSupport* pSupport ) const
    {
      return ::vkGetDescriptorSetLayoutSupportKHR( device, pCreateInfo, pSupport );
    }

    void vkGetDeviceGroupPeerMemoryFeatures( VkDevice device, uint32_t heapIndex, uint32_t localDeviceIndex, uint32_t remoteDeviceIndex, VkPeerMemoryFeatureFlags* pPeerMemoryFeatures ) const
    {
      return ::vkGetDeviceGroupPeerMemoryFeatures( device, heapIndex, localDeviceIndex, remoteDeviceIndex, pPeerMemoryFeatures );
    }

    void vkGetDeviceGroupPeerMemoryFeaturesKHR( VkDevice device, uint32_t heapIndex, uint32_t localDeviceIndex, uint32_t remoteDeviceIndex, VkPeerMemoryFeatureFlags* pPeerMemoryFeatures ) const
    {
      return ::vkGetDeviceGroupPeerMemoryFeaturesKHR( device, heapIndex, localDeviceIndex, remoteDeviceIndex, pPeerMemoryFeatures );
    }

    VkResult vkGetDeviceGroupPresentCapabilitiesKHR( VkDevice device, VkDeviceGroupPresentCapabilitiesKHR* pDeviceGroupPresentCapabilities ) const
    {
      return ::vkGetDeviceGroupPresentCapabilitiesKHR( device, pDeviceGroupPresentCapabilities );
    }

    VkResult vkGetDeviceGroupSurfacePresentModesKHR( VkDevice device, VkSurfaceKHR surface, VkDeviceGroupPresentModeFlagsKHR* pModes ) const
    {
      return ::vkGetDeviceGroupSurfacePresentModesKHR( device, surface, pModes );
    }

    void vkGetDeviceMemoryCommitment( VkDevice device, VkDeviceMemory memory, VkDeviceSize* pCommittedMemoryInBytes ) const
    {
      return ::vkGetDeviceMemoryCommitment( device, memory, pCommittedMemoryInBytes );
    }

    PFN_vkVoidFunction vkGetDeviceProcAddr( VkDevice device, const char* pName ) const
    {
      return ::vkGetDeviceProcAddr( device, pName );
    }

    void vkGetDeviceQueue( VkDevice device, uint32_t queueFamilyIndex, uint32_t queueIndex, VkQueue* pQueue ) const
    {
      return ::vkGetDeviceQueue( device, queueFamilyIndex, queueIndex, pQueue );
    }

    void vkGetDeviceQueue2( VkDevice device, const VkDeviceQueueInfo2* pQueueInfo, VkQueue* pQueue ) const
    {
      return ::vkGetDeviceQueue2( device, pQueueInfo, pQueue );
    }

    VkResult vkGetEventStatus( VkDevice device, VkEvent event ) const
    {
      return ::vkGetEventStatus( device, event );
    }

    VkResult vkGetFenceFdKHR( VkDevice device, const VkFenceGetFdInfoKHR* pGetFdInfo, int* pFd ) const
    {
      return ::vkGetFenceFdKHR( device, pGetFdInfo, pFd );
    }

    VkResult vkGetFenceStatus( VkDevice device, VkFence fence ) const
    {
      return ::vkGetFenceStatus( device, fence );
    }

#ifdef VK_USE_PLATFORM_WIN32_KHR
    VkResult vkGetFenceWin32HandleKHR( VkDevice device, const VkFenceGetWin32HandleInfoKHR* pGetWin32HandleInfo, HANDLE* pHandle ) const
    {
      return ::vkGetFenceWin32HandleKHR( device, pGetWin32HandleInfo, pHandle );
    }
#endif /*VK_USE_PLATFORM_WIN32_KHR*/

    VkResult vkGetImageDrmFormatModifierPropertiesEXT( VkDevice device, VkImage image, VkImageDrmFormatModifierPropertiesEXT* pProperties ) const
    {
      return ::vkGetImageDrmFormatModifierPropertiesEXT( device, image, pProperties );
    }

    void vkGetImageMemoryRequirements( VkDevice device, VkImage image, VkMemoryRequirements* pMemoryRequirements ) const
    {
      return ::vkGetImageMemoryRequirements( device, image, pMemoryRequirements );
    }

    void vkGetImageMemoryRequirements2( VkDevice device, const VkImageMemoryRequirementsInfo2* pInfo, VkMemoryRequirements2* pMemoryRequirements ) const
    {
      return ::vkGetImageMemoryRequirements2( device, pInfo, pMemoryRequirements );
    }

    void vkGetImageMemoryRequirements2KHR( VkDevice device, const VkImageMemoryRequirementsInfo2* pInfo, VkMemoryRequirements2* pMemoryRequirements ) const
    {
      return ::vkGetImageMemoryRequirements2KHR( device, pInfo, pMemoryRequirements );
    }

    void vkGetImageSparseMemoryRequirements( VkDevice device, VkImage image, uint32_t* pSparseMemoryRequirementCount, VkSparseImageMemoryRequirements* pSparseMemoryRequirements ) const
    {
      return ::vkGetImageSparseMemoryRequirements( device, image, pSparseMemoryRequirementCount, pSparseMemoryRequirements );
    }

    void vkGetImageSparseMemoryRequirements2( VkDevice device, const VkImageSparseMemoryRequirementsInfo2* pInfo, uint32_t* pSparseMemoryRequirementCount, VkSparseImageMemoryRequirements2* pSparseMemoryRequirements ) const
    {
      return ::vkGetImageSparseMemoryRequirements2( device, pInfo, pSparseMemoryRequirementCount, pSparseMemoryRequirements );
    }

    void vkGetImageSparseMemoryRequirements2KHR( VkDevice device, const VkImageSparseMemoryRequirementsInfo2* pInfo, uint32_t* pSparseMemoryRequirementCount, VkSparseImageMemoryRequirements2* pSparseMemoryRequirements ) const
    {
      return ::vkGetImageSparseMemoryRequirements2KHR( device, pInfo, pSparseMemoryRequirementCount, pSparseMemoryRequirements );
    }

    void vkGetImageSubresourceLayout( VkDevice device, VkImage image, const VkImageSubresource* pSubresource, VkSubresourceLayout* pLayout ) const
    {
      return ::vkGetImageSubresourceLayout( device, image, pSubresource, pLayout );
    }

#ifdef VK_USE_PLATFORM_ANDROID_KHR
    VkResult vkGetMemoryAndroidHardwareBufferANDROID( VkDevice device, const VkMemoryGetAndroidHardwareBufferInfoANDROID* pInfo, struct AHardwareBuffer** pBuffer ) const
    {
      return ::vkGetMemoryAndroidHardwareBufferANDROID( device, pInfo, pBuffer );
    }
#endif /*VK_USE_PLATFORM_ANDROID_KHR*/

    VkResult vkGetMemoryFdKHR( VkDevice device, const VkMemoryGetFdInfoKHR* pGetFdInfo, int* pFd ) const
    {
      return ::vkGetMemoryFdKHR( device, pGetFdInfo, pFd );
    }

    VkResult vkGetMemoryFdPropertiesKHR( VkDevice device, VkExternalMemoryHandleTypeFlagBits handleType, int fd, VkMemoryFdPropertiesKHR* pMemoryFdProperties ) const
    {
      return ::vkGetMemoryFdPropertiesKHR( device, handleType, fd, pMemoryFdProperties );
    }

    VkResult vkGetMemoryHostPointerPropertiesEXT( VkDevice device, VkExternalMemoryHandleTypeFlagBits handleType, const void* pHostPointer, VkMemoryHostPointerPropertiesEXT* pMemoryHostPointerProperties ) const
    {
      return ::vkGetMemoryHostPointerPropertiesEXT( device, handleType, pHostPointer, pMemoryHostPointerProperties );
    }

#ifdef VK_USE_PLATFORM_WIN32_KHR
    VkResult vkGetMemoryWin32HandleKHR( VkDevice device, const VkMemoryGetWin32HandleInfoKHR* pGetWin32HandleInfo, HANDLE* pHandle ) const
    {
      return ::vkGetMemoryWin32HandleKHR( device, pGetWin32HandleInfo, pHandle );
    }
#endif /*VK_USE_PLATFORM_WIN32_KHR*/

#ifdef VK_USE_PLATFORM_WIN32_KHR
    VkResult vkGetMemoryWin32HandleNV( VkDevice device, VkDeviceMemory memory, VkExternalMemoryHandleTypeFlagsNV handleType, HANDLE* pHandle ) const
    {
      return ::vkGetMemoryWin32HandleNV( device, memory, handleType, pHandle );
    }
#endif /*VK_USE_PLATFORM_WIN32_KHR*/

#ifdef VK_USE_PLATFORM_WIN32_KHR
    VkResult vkGetMemoryWin32HandlePropertiesKHR( VkDevice device, VkExternalMemoryHandleTypeFlagBits handleType, HANDLE handle, VkMemoryWin32HandlePropertiesKHR* pMemoryWin32HandleProperties ) const
    {
      return ::vkGetMemoryWin32HandlePropertiesKHR( device, handleType, handle, pMemoryWin32HandleProperties );
    }
#endif /*VK_USE_PLATFORM_WIN32_KHR*/

    VkResult vkGetPastPresentationTimingGOOGLE( VkDevice device, VkSwapchainKHR swapchain, uint32_t* pPresentationTimingCount, VkPastPresentationTimingGOOGLE* pPresentationTimings ) const
    {
      return ::vkGetPastPresentationTimingGOOGLE( device, swapchain, pPresentationTimingCount, pPresentationTimings );
    }

    VkResult vkGetPipelineCacheData( VkDevice device, VkPipelineCache pipelineCache, size_t* pDataSize, void* pData ) const
    {
      return ::vkGetPipelineCacheData( device, pipelineCache, pDataSize, pData );
    }

    VkResult vkGetQueryPoolResults( VkDevice device, VkQueryPool queryPool, uint32_t firstQuery, uint32_t queryCount, size_t dataSize, void* pData, VkDeviceSize stride, VkQueryResultFlags flags ) const
    {
      return ::vkGetQueryPoolResults( device, queryPool, firstQuery, queryCount, dataSize, pData, stride, flags );
    }

    VkResult vkGetRayTracingShaderGroupHandlesNV( VkDevice device, VkPipeline pipeline, uint32_t firstGroup, uint32_t groupCount, size_t dataSize, void* pData ) const
    {
      return ::vkGetRayTracingShaderGroupHandlesNV( device, pipeline, firstGroup, groupCount, dataSize, pData );
    }

    VkResult vkGetRefreshCycleDurationGOOGLE( VkDevice device, VkSwapchainKHR swapchain, VkRefreshCycleDurationGOOGLE* pDisplayTimingProperties ) const
    {
      return ::vkGetRefreshCycleDurationGOOGLE( device, swapchain, pDisplayTimingProperties );
    }

    void vkGetRenderAreaGranularity( VkDevice device, VkRenderPass renderPass, VkExtent2D* pGranularity ) const
    {
      return ::vkGetRenderAreaGranularity( device, renderPass, pGranularity );
    }

    VkResult vkGetSemaphoreFdKHR( VkDevice device, const VkSemaphoreGetFdInfoKHR* pGetFdInfo, int* pFd ) const
    {
      return ::vkGetSemaphoreFdKHR( device, pGetFdInfo, pFd );
    }

#ifdef VK_USE_PLATFORM_WIN32_KHR
    VkResult vkGetSemaphoreWin32HandleKHR( VkDevice device, const VkSemaphoreGetWin32HandleInfoKHR* pGetWin32HandleInfo, HANDLE* pHandle ) const
    {
      return ::vkGetSemaphoreWin32HandleKHR( device, pGetWin32HandleInfo, pHandle );
    }
#endif /*VK_USE_PLATFORM_WIN32_KHR*/

    VkResult vkGetShaderInfoAMD( VkDevice device, VkPipeline pipeline, VkShaderStageFlagBits shaderStage, VkShaderInfoTypeAMD infoType, size_t* pInfoSize, void* pInfo ) const
    {
      return ::vkGetShaderInfoAMD( device, pipeline, shaderStage, infoType, pInfoSize, pInfo );
    }

    VkResult vkGetSwapchainCounterEXT( VkDevice device, VkSwapchainKHR swapchain, VkSurfaceCounterFlagBitsEXT counter, uint64_t* pCounterValue ) const
    {
      return ::vkGetSwapchainCounterEXT( device, swapchain, counter, pCounterValue );
    }

    VkResult vkGetSwapchainImagesKHR( VkDevice device, VkSwapchainKHR swapchain, uint32_t* pSwapchainImageCount, VkImage* pSwapchainImages ) const
    {
      return ::vkGetSwapchainImagesKHR( device, swapchain, pSwapchainImageCount, pSwapchainImages );
    }

    VkResult vkGetSwapchainStatusKHR( VkDevice device, VkSwapchainKHR swapchain ) const
    {
      return ::vkGetSwapchainStatusKHR( device, swapchain );
    }

    VkResult vkGetValidationCacheDataEXT( VkDevice device, VkValidationCacheEXT validationCache, size_t* pDataSize, void* pData ) const
    {
      return ::vkGetValidationCacheDataEXT( device, validationCache, pDataSize, pData );
    }

    VkResult vkImportFenceFdKHR( VkDevice device, const VkImportFenceFdInfoKHR* pImportFenceFdInfo ) const
    {
      return ::vkImportFenceFdKHR( device, pImportFenceFdInfo );
    }

#ifdef VK_USE_PLATFORM_WIN32_KHR
    VkResult vkImportFenceWin32HandleKHR( VkDevice device, const VkImportFenceWin32HandleInfoKHR* pImportFenceWin32HandleInfo ) const
    {
      return ::vkImportFenceWin32HandleKHR( device, pImportFenceWin32HandleInfo );
    }
#endif /*VK_USE_PLATFORM_WIN32_KHR*/

    VkResult vkImportSemaphoreFdKHR( VkDevice device, const VkImportSemaphoreFdInfoKHR* pImportSemaphoreFdInfo ) const
    {
      return ::vkImportSemaphoreFdKHR( device, pImportSemaphoreFdInfo );
    }

#ifdef VK_USE_PLATFORM_WIN32_KHR
    VkResult vkImportSemaphoreWin32HandleKHR( VkDevice device, const VkImportSemaphoreWin32HandleInfoKHR* pImportSemaphoreWin32HandleInfo ) const
    {
      return ::vkImportSemaphoreWin32HandleKHR( device, pImportSemaphoreWin32HandleInfo );
    }
#endif /*VK_USE_PLATFORM_WIN32_KHR*/

    VkResult vkInvalidateMappedMemoryRanges( VkDevice device, uint32_t memoryRangeCount, const VkMappedMemoryRange* pMemoryRanges ) const
    {
      return ::vkInvalidateMappedMemoryRanges( device, memoryRangeCount, pMemoryRanges );
    }

    VkResult vkMapMemory( VkDevice device, VkDeviceMemory memory, VkDeviceSize offset, VkDeviceSize size, VkMemoryMapFlags flags, void** ppData ) const
    {
      return ::vkMapMemory( device, memory, offset, size, flags, ppData );
    }

    VkResult vkMergePipelineCaches( VkDevice device, VkPipelineCache dstCache, uint32_t srcCacheCount, const VkPipelineCache* pSrcCaches ) const
    {
      return ::vkMergePipelineCaches( device, dstCache, srcCacheCount, pSrcCaches );
    }

    VkResult vkMergeValidationCachesEXT( VkDevice device, VkValidationCacheEXT dstCache, uint32_t srcCacheCount, const VkValidationCacheEXT* pSrcCaches ) const
    {
      return ::vkMergeValidationCachesEXT( device, dstCache, srcCacheCount, pSrcCaches );
    }

    VkResult vkRegisterDeviceEventEXT( VkDevice device, const VkDeviceEventInfoEXT* pDeviceEventInfo, const VkAllocationCallbacks* pAllocator, VkFence* pFence ) const
    {
      return ::vkRegisterDeviceEventEXT( device, pDeviceEventInfo, pAllocator, pFence );
    }

    VkResult vkRegisterDisplayEventEXT( VkDevice device, VkDisplayKHR display, const VkDisplayEventInfoEXT* pDisplayEventInfo, const VkAllocationCallbacks* pAllocator, VkFence* pFence ) const
    {
      return ::vkRegisterDisplayEventEXT( device, display, pDisplayEventInfo, pAllocator, pFence );
    }

    VkResult vkRegisterObjectsNVX( VkDevice device, VkObjectTableNVX objectTable, uint32_t objectCount, const VkObjectTableEntryNVX* const* ppObjectTableEntries, const uint32_t* pObjectIndices ) const
    {
      return ::vkRegisterObjectsNVX( device, objectTable, objectCount, ppObjectTableEntries, pObjectIndices );
    }

    VkResult vkResetCommandPool( VkDevice device, VkCommandPool commandPool, VkCommandPoolResetFlags flags ) const
    {
      return ::vkResetCommandPool( device, commandPool, flags );
    }

    VkResult vkResetDescriptorPool( VkDevice device, VkDescriptorPool descriptorPool, VkDescriptorPoolResetFlags flags ) const
    {
      return ::vkResetDescriptorPool( device, descriptorPool, flags );
    }

    VkResult vkResetEvent( VkDevice device, VkEvent event ) const
    {
      return ::vkResetEvent( device, event );
    }

    VkResult vkResetFences( VkDevice device, uint32_t fenceCount, const VkFence* pFences ) const
    {
      return ::vkResetFences( device, fenceCount, pFences );
    }

    VkResult vkSetDebugUtilsObjectNameEXT( VkDevice device, const VkDebugUtilsObjectNameInfoEXT* pNameInfo ) const
    {
      return ::vkSetDebugUtilsObjectNameEXT( device, pNameInfo );
    }

    VkResult vkSetDebugUtilsObjectTagEXT( VkDevice device, const VkDebugUtilsObjectTagInfoEXT* pTagInfo ) const
    {
      return ::vkSetDebugUtilsObjectTagEXT( device, pTagInfo );
    }

    VkResult vkSetEvent( VkDevice device, VkEvent event ) const
    {
      return ::vkSetEvent( device, event );
    }

    void vkSetHdrMetadataEXT( VkDevice device, uint32_t swapchainCount, const VkSwapchainKHR* pSwapchains, const VkHdrMetadataEXT* pMetadata ) const
    {
      return ::vkSetHdrMetadataEXT( device, swapchainCount, pSwapchains, pMetadata );
    }

    void vkTrimCommandPool( VkDevice device, VkCommandPool commandPool, VkCommandPoolTrimFlags flags ) const
    {
      return ::vkTrimCommandPool( device, commandPool, flags );
    }

    void vkTrimCommandPoolKHR( VkDevice device, VkCommandPool commandPool, VkCommandPoolTrimFlags flags ) const
    {
      return ::vkTrimCommandPoolKHR( device, commandPool, flags );
    }

    void vkUnmapMemory( VkDevice device, VkDeviceMemory memory ) const
    {
      return ::vkUnmapMemory( device, memory );
    }

    VkResult vkUnregisterObjectsNVX( VkDevice device, VkObjectTableNVX objectTable, uint32_t objectCount, const VkObjectEntryTypeNVX* pObjectEntryTypes, const uint32_t* pObjectIndices ) const
    {
      return ::vkUnregisterObjectsNVX( device, objectTable, objectCount, pObjectEntryTypes, pObjectIndices );
    }

    void vkUpdateDescriptorSetWithTemplate( VkDevice device, VkDescriptorSet descriptorSet, VkDescriptorUpdateTemplate descriptorUpdateTemplate, const void* pData ) const
    {
      return ::vkUpdateDescriptorSetWithTemplate( device, descriptorSet, descriptorUpdateTemplate, pData );
    }

    void vkUpdateDescriptorSetWithTemplateKHR( VkDevice device, VkDescriptorSet descriptorSet, VkDescriptorUpdateTemplate descriptorUpdateTemplate, const void* pData ) const
    {
      return ::vkUpdateDescriptorSetWithTemplateKHR( device, descriptorSet, descriptorUpdateTemplate, pData );
    }

    void vkUpdateDescriptorSets( VkDevice device, uint32_t descriptorWriteCount, const VkWriteDescriptorSet* pDescriptorWrites, uint32_t descriptorCopyCount, const VkCopyDescriptorSet* pDescriptorCopies ) const
    {
      return ::vkUpdateDescriptorSets( device, descriptorWriteCount, pDescriptorWrites, descriptorCopyCount, pDescriptorCopies );
    }

    VkResult vkWaitForFences( VkDevice device, uint32_t fenceCount, const VkFence* pFences, VkBool32 waitAll, uint64_t timeout ) const
    {
      return ::vkWaitForFences( device, fenceCount, pFences, waitAll, timeout );
    }

#ifdef VK_USE_PLATFORM_ANDROID_KHR
    VkResult vkCreateAndroidSurfaceKHR( VkInstance instance, const VkAndroidSurfaceCreateInfoKHR* pCreateInfo, const VkAllocationCallbacks* pAllocator, VkSurfaceKHR* pSurface ) const
    {
      return ::vkCreateAndroidSurfaceKHR( instance, pCreateInfo, pAllocator, pSurface );
    }
#endif /*VK_USE_PLATFORM_ANDROID_KHR*/

    VkResult vkCreateDebugReportCallbackEXT( VkInstance instance, const VkDebugReportCallbackCreateInfoEXT* pCreateInfo, const VkAllocationCallbacks* pAllocator, VkDebugReportCallbackEXT* pCallback ) const
    {
      return ::vkCreateDebugReportCallbackEXT( instance, pCreateInfo, pAllocator, pCallback );
    }

    VkResult vkCreateDebugUtilsMessengerEXT( VkInstance instance, const VkDebugUtilsMessengerCreateInfoEXT* pCreateInfo, const VkAllocationCallbacks* pAllocator, VkDebugUtilsMessengerEXT* pMessenger ) const
    {
      return ::vkCreateDebugUtilsMessengerEXT( instance, pCreateInfo, pAllocator, pMessenger );
    }

    VkResult vkCreateDisplayPlaneSurfaceKHR( VkInstance instance, const VkDisplaySurfaceCreateInfoKHR* pCreateInfo, const VkAllocationCallbacks* pAllocator, VkSurfaceKHR* pSurface ) const
    {
      return ::vkCreateDisplayPlaneSurfaceKHR( instance, pCreateInfo, pAllocator, pSurface );
    }

#ifdef VK_USE_PLATFORM_IOS_MVK
    VkResult vkCreateIOSSurfaceMVK( VkInstance instance, const VkIOSSurfaceCreateInfoMVK* pCreateInfo, const VkAllocationCallbacks* pAllocator, VkSurfaceKHR* pSurface ) const
    {
      return ::vkCreateIOSSurfaceMVK( instance, pCreateInfo, pAllocator, pSurface );
    }
#endif /*VK_USE_PLATFORM_IOS_MVK*/

#ifdef VK_USE_PLATFORM_FUCHSIA
    VkResult vkCreateImagePipeSurfaceFUCHSIA( VkInstance instance, const VkImagePipeSurfaceCreateInfoFUCHSIA* pCreateInfo, const VkAllocationCallbacks* pAllocator, VkSurfaceKHR* pSurface ) const
    {
      return ::vkCreateImagePipeSurfaceFUCHSIA( instance, pCreateInfo, pAllocator, pSurface );
    }
#endif /*VK_USE_PLATFORM_FUCHSIA*/

#ifdef VK_USE_PLATFORM_MACOS_MVK
    VkResult vkCreateMacOSSurfaceMVK( VkInstance instance, const VkMacOSSurfaceCreateInfoMVK* pCreateInfo, const VkAllocationCallbacks* pAllocator, VkSurfaceKHR* pSurface ) const
    {
      return ::vkCreateMacOSSurfaceMVK( instance, pCreateInfo, pAllocator, pSurface );
    }
#endif /*VK_USE_PLATFORM_MACOS_MVK*/

#ifdef VK_USE_PLATFORM_VI_NN
    VkResult vkCreateViSurfaceNN( VkInstance instance, const VkViSurfaceCreateInfoNN* pCreateInfo, const VkAllocationCallbacks* pAllocator, VkSurfaceKHR* pSurface ) const
    {
      return ::vkCreateViSurfaceNN( instance, pCreateInfo, pAllocator, pSurface );
    }
#endif /*VK_USE_PLATFORM_VI_NN*/

#ifdef VK_USE_PLATFORM_WAYLAND_KHR
    VkResult vkCreateWaylandSurfaceKHR( VkInstance instance, const VkWaylandSurfaceCreateInfoKHR* pCreateInfo, const VkAllocationCallbacks* pAllocator, VkSurfaceKHR* pSurface ) const
    {
      return ::vkCreateWaylandSurfaceKHR( instance, pCreateInfo, pAllocator, pSurface );
    }
#endif /*VK_USE_PLATFORM_WAYLAND_KHR*/

#ifdef VK_USE_PLATFORM_WIN32_KHR
    VkResult vkCreateWin32SurfaceKHR( VkInstance instance, const VkWin32SurfaceCreateInfoKHR* pCreateInfo, const VkAllocationCallbacks* pAllocator, VkSurfaceKHR* pSurface ) const
    {
      return ::vkCreateWin32SurfaceKHR( instance, pCreateInfo, pAllocator, pSurface );
    }
#endif /*VK_USE_PLATFORM_WIN32_KHR*/

#ifdef VK_USE_PLATFORM_XCB_KHR
    VkResult vkCreateXcbSurfaceKHR( VkInstance instance, const VkXcbSurfaceCreateInfoKHR* pCreateInfo, const VkAllocationCallbacks* pAllocator, VkSurfaceKHR* pSurface ) const
    {
      return ::vkCreateXcbSurfaceKHR( instance, pCreateInfo, pAllocator, pSurface );
    }
#endif /*VK_USE_PLATFORM_XCB_KHR*/

#ifdef VK_USE_PLATFORM_XLIB_KHR
    VkResult vkCreateXlibSurfaceKHR( VkInstance instance, const VkXlibSurfaceCreateInfoKHR* pCreateInfo, const VkAllocationCallbacks* pAllocator, VkSurfaceKHR* pSurface ) const
    {
      return ::vkCreateXlibSurfaceKHR( instance, pCreateInfo, pAllocator, pSurface );
    }
#endif /*VK_USE_PLATFORM_XLIB_KHR*/

    void vkDebugReportMessageEXT( VkInstance instance, VkDebugReportFlagsEXT flags, VkDebugReportObjectTypeEXT objectType, uint64_t object, size_t location, int32_t messageCode, const char* pLayerPrefix, const char* pMessage ) const
    {
      return ::vkDebugReportMessageEXT( instance, flags, objectType, object, location, messageCode, pLayerPrefix, pMessage );
    }

    void vkDestroyDebugReportCallbackEXT( VkInstance instance, VkDebugReportCallbackEXT callback, const VkAllocationCallbacks* pAllocator ) const
    {
      return ::vkDestroyDebugReportCallbackEXT( instance, callback, pAllocator );
    }

    void vkDestroyDebugUtilsMessengerEXT( VkInstance instance, VkDebugUtilsMessengerEXT messenger, const VkAllocationCallbacks* pAllocator ) const
    {
      return ::vkDestroyDebugUtilsMessengerEXT( instance, messenger, pAllocator );
    }

    void vkDestroyInstance( VkInstance instance, const VkAllocationCallbacks* pAllocator ) const
    {
      return ::vkDestroyInstance( instance, pAllocator );
    }

    void vkDestroySurfaceKHR( VkInstance instance, VkSurfaceKHR surface, const VkAllocationCallbacks* pAllocator ) const
    {
      return ::vkDestroySurfaceKHR( instance, surface, pAllocator );
    }

    VkResult vkEnumeratePhysicalDeviceGroups( VkInstance instance, uint32_t* pPhysicalDeviceGroupCount, VkPhysicalDeviceGroupProperties* pPhysicalDeviceGroupProperties ) const
    {
      return ::vkEnumeratePhysicalDeviceGroups( instance, pPhysicalDeviceGroupCount, pPhysicalDeviceGroupProperties );
    }

    VkResult vkEnumeratePhysicalDeviceGroupsKHR( VkInstance instance, uint32_t* pPhysicalDeviceGroupCount, VkPhysicalDeviceGroupProperties* pPhysicalDeviceGroupProperties ) const
    {
      return ::vkEnumeratePhysicalDeviceGroupsKHR( instance, pPhysicalDeviceGroupCount, pPhysicalDeviceGroupProperties );
    }

    VkResult vkEnumeratePhysicalDevices( VkInstance instance, uint32_t* pPhysicalDeviceCount, VkPhysicalDevice* pPhysicalDevices ) const
    {
      return ::vkEnumeratePhysicalDevices( instance, pPhysicalDeviceCount, pPhysicalDevices );
    }

    PFN_vkVoidFunction vkGetInstanceProcAddr( VkInstance instance, const char* pName ) const
    {
      return ::vkGetInstanceProcAddr( instance, pName );
    }

    void vkSubmitDebugUtilsMessageEXT( VkInstance instance, VkDebugUtilsMessageSeverityFlagBitsEXT messageSeverity, VkDebugUtilsMessageTypeFlagsEXT messageTypes, const VkDebugUtilsMessengerCallbackDataEXT* pCallbackData ) const
    {
      return ::vkSubmitDebugUtilsMessageEXT( instance, messageSeverity, messageTypes, pCallbackData );
    }

#ifdef VK_USE_PLATFORM_XLIB_XRANDR_EXT
    VkResult vkAcquireXlibDisplayEXT( VkPhysicalDevice physicalDevice, Display* dpy, VkDisplayKHR display ) const
    {
      return ::vkAcquireXlibDisplayEXT( physicalDevice, dpy, display );
    }
#endif /*VK_USE_PLATFORM_XLIB_XRANDR_EXT*/

    VkResult vkCreateDevice( VkPhysicalDevice physicalDevice, const VkDeviceCreateInfo* pCreateInfo, const VkAllocationCallbacks* pAllocator, VkDevice* pDevice ) const
    {
      return ::vkCreateDevice( physicalDevice, pCreateInfo, pAllocator, pDevice );
    }

    VkResult vkCreateDisplayModeKHR( VkPhysicalDevice physicalDevice, VkDisplayKHR display, const VkDisplayModeCreateInfoKHR* pCreateInfo, const VkAllocationCallbacks* pAllocator, VkDisplayModeKHR* pMode ) const
    {
      return ::vkCreateDisplayModeKHR( physicalDevice, display, pCreateInfo, pAllocator, pMode );
    }

    VkResult vkEnumerateDeviceExtensionProperties( VkPhysicalDevice physicalDevice, const char* pLayerName, uint32_t* pPropertyCount, VkExtensionProperties* pProperties ) const
    {
      return ::vkEnumerateDeviceExtensionProperties( physicalDevice, pLayerName, pPropertyCount, pProperties );
    }

    VkResult vkEnumerateDeviceLayerProperties( VkPhysicalDevice physicalDevice, uint32_t* pPropertyCount, VkLayerProperties* pProperties ) const
    {
      return ::vkEnumerateDeviceLayerProperties( physicalDevice, pPropertyCount, pProperties );
    }

    VkResult vkGetDisplayModeProperties2KHR( VkPhysicalDevice physicalDevice, VkDisplayKHR display, uint32_t* pPropertyCount, VkDisplayModeProperties2KHR* pProperties ) const
    {
      return ::vkGetDisplayModeProperties2KHR( physicalDevice, display, pPropertyCount, pProperties );
    }

    VkResult vkGetDisplayModePropertiesKHR( VkPhysicalDevice physicalDevice, VkDisplayKHR display, uint32_t* pPropertyCount, VkDisplayModePropertiesKHR* pProperties ) const
    {
      return ::vkGetDisplayModePropertiesKHR( physicalDevice, display, pPropertyCount, pProperties );
    }

    VkResult vkGetDisplayPlaneCapabilities2KHR( VkPhysicalDevice physicalDevice, const VkDisplayPlaneInfo2KHR* pDisplayPlaneInfo, VkDisplayPlaneCapabilities2KHR* pCapabilities ) const
    {
      return ::vkGetDisplayPlaneCapabilities2KHR( physicalDevice, pDisplayPlaneInfo, pCapabilities );
    }

    VkResult vkGetDisplayPlaneCapabilitiesKHR( VkPhysicalDevice physicalDevice, VkDisplayModeKHR mode, uint32_t planeIndex, VkDisplayPlaneCapabilitiesKHR* pCapabilities ) const
    {
      return ::vkGetDisplayPlaneCapabilitiesKHR( physicalDevice, mode, planeIndex, pCapabilities );
    }

    VkResult vkGetDisplayPlaneSupportedDisplaysKHR( VkPhysicalDevice physicalDevice, uint32_t planeIndex, uint32_t* pDisplayCount, VkDisplayKHR* pDisplays ) const
    {
      return ::vkGetDisplayPlaneSupportedDisplaysKHR( physicalDevice, planeIndex, pDisplayCount, pDisplays );
    }

    VkResult vkGetPhysicalDeviceCalibrateableTimeDomainsEXT( VkPhysicalDevice physicalDevice, uint32_t* pTimeDomainCount, VkTimeDomainEXT* pTimeDomains ) const
    {
      return ::vkGetPhysicalDeviceCalibrateableTimeDomainsEXT( physicalDevice, pTimeDomainCount, pTimeDomains );
    }

    VkResult vkGetPhysicalDeviceCooperativeMatrixPropertiesNV( VkPhysicalDevice physicalDevice, uint32_t* pPropertyCount, VkCooperativeMatrixPropertiesNV* pProperties ) const
    {
      return ::vkGetPhysicalDeviceCooperativeMatrixPropertiesNV( physicalDevice, pPropertyCount, pProperties );
    }

    VkResult vkGetPhysicalDeviceDisplayPlaneProperties2KHR( VkPhysicalDevice physicalDevice, uint32_t* pPropertyCount, VkDisplayPlaneProperties2KHR* pProperties ) const
    {
      return ::vkGetPhysicalDeviceDisplayPlaneProperties2KHR( physicalDevice, pPropertyCount, pProperties );
    }

    VkResult vkGetPhysicalDeviceDisplayPlanePropertiesKHR( VkPhysicalDevice physicalDevice, uint32_t* pPropertyCount, VkDisplayPlanePropertiesKHR* pProperties ) const
    {
      return ::vkGetPhysicalDeviceDisplayPlanePropertiesKHR( physicalDevice, pPropertyCount, pProperties );
    }

    VkResult vkGetPhysicalDeviceDisplayProperties2KHR( VkPhysicalDevice physicalDevice, uint32_t* pPropertyCount, VkDisplayProperties2KHR* pProperties ) const
    {
      return ::vkGetPhysicalDeviceDisplayProperties2KHR( physicalDevice, pPropertyCount, pProperties );
    }

    VkResult vkGetPhysicalDeviceDisplayPropertiesKHR( VkPhysicalDevice physicalDevice, uint32_t* pPropertyCount, VkDisplayPropertiesKHR* pProperties ) const
    {
      return ::vkGetPhysicalDeviceDisplayPropertiesKHR( physicalDevice, pPropertyCount, pProperties );
    }

    void vkGetPhysicalDeviceExternalBufferProperties( VkPhysicalDevice physicalDevice, const VkPhysicalDeviceExternalBufferInfo* pExternalBufferInfo, VkExternalBufferProperties* pExternalBufferProperties ) const
    {
      return ::vkGetPhysicalDeviceExternalBufferProperties( physicalDevice, pExternalBufferInfo, pExternalBufferProperties );
    }

    void vkGetPhysicalDeviceExternalBufferPropertiesKHR( VkPhysicalDevice physicalDevice, const VkPhysicalDeviceExternalBufferInfo* pExternalBufferInfo, VkExternalBufferProperties* pExternalBufferProperties ) const
    {
      return ::vkGetPhysicalDeviceExternalBufferPropertiesKHR( physicalDevice, pExternalBufferInfo, pExternalBufferProperties );
    }

    void vkGetPhysicalDeviceExternalFenceProperties( VkPhysicalDevice physicalDevice, const VkPhysicalDeviceExternalFenceInfo* pExternalFenceInfo, VkExternalFenceProperties* pExternalFenceProperties ) const
    {
      return ::vkGetPhysicalDeviceExternalFenceProperties( physicalDevice, pExternalFenceInfo, pExternalFenceProperties );
    }

    void vkGetPhysicalDeviceExternalFencePropertiesKHR( VkPhysicalDevice physicalDevice, const VkPhysicalDeviceExternalFenceInfo* pExternalFenceInfo, VkExternalFenceProperties* pExternalFenceProperties ) const
    {
      return ::vkGetPhysicalDeviceExternalFencePropertiesKHR( physicalDevice, pExternalFenceInfo, pExternalFenceProperties );
    }

    VkResult vkGetPhysicalDeviceExternalImageFormatPropertiesNV( VkPhysicalDevice physicalDevice, VkFormat format, VkImageType type, VkImageTiling tiling, VkImageUsageFlags usage, VkImageCreateFlags flags, VkExternalMemoryHandleTypeFlagsNV externalHandleType, VkExternalImageFormatPropertiesNV* pExternalImageFormatProperties ) const
    {
      return ::vkGetPhysicalDeviceExternalImageFormatPropertiesNV( physicalDevice, format, type, tiling, usage, flags, externalHandleType, pExternalImageFormatProperties );
    }

    void vkGetPhysicalDeviceExternalSemaphoreProperties( VkPhysicalDevice physicalDevice, const VkPhysicalDeviceExternalSemaphoreInfo* pExternalSemaphoreInfo, VkExternalSemaphoreProperties* pExternalSemaphoreProperties ) const
    {
      return ::vkGetPhysicalDeviceExternalSemaphoreProperties( physicalDevice, pExternalSemaphoreInfo, pExternalSemaphoreProperties );
    }

    void vkGetPhysicalDeviceExternalSemaphorePropertiesKHR( VkPhysicalDevice physicalDevice, const VkPhysicalDeviceExternalSemaphoreInfo* pExternalSemaphoreInfo, VkExternalSemaphoreProperties* pExternalSemaphoreProperties ) const
    {
      return ::vkGetPhysicalDeviceExternalSemaphorePropertiesKHR( physicalDevice, pExternalSemaphoreInfo, pExternalSemaphoreProperties );
    }

    void vkGetPhysicalDeviceFeatures( VkPhysicalDevice physicalDevice, VkPhysicalDeviceFeatures* pFeatures ) const
    {
      return ::vkGetPhysicalDeviceFeatures( physicalDevice, pFeatures );
    }

    void vkGetPhysicalDeviceFeatures2( VkPhysicalDevice physicalDevice, VkPhysicalDeviceFeatures2* pFeatures ) const
    {
      return ::vkGetPhysicalDeviceFeatures2( physicalDevice, pFeatures );
    }

    void vkGetPhysicalDeviceFeatures2KHR( VkPhysicalDevice physicalDevice, VkPhysicalDeviceFeatures2* pFeatures ) const
    {
      return ::vkGetPhysicalDeviceFeatures2KHR( physicalDevice, pFeatures );
    }

    void vkGetPhysicalDeviceFormatProperties( VkPhysicalDevice physicalDevice, VkFormat format, VkFormatProperties* pFormatProperties ) const
    {
      return ::vkGetPhysicalDeviceFormatProperties( physicalDevice, format, pFormatProperties );
    }

    void vkGetPhysicalDeviceFormatProperties2( VkPhysicalDevice physicalDevice, VkFormat format, VkFormatProperties2* pFormatProperties ) const
    {
      return ::vkGetPhysicalDeviceFormatProperties2( physicalDevice, format, pFormatProperties );
    }

    void vkGetPhysicalDeviceFormatProperties2KHR( VkPhysicalDevice physicalDevice, VkFormat format, VkFormatProperties2* pFormatProperties ) const
    {
      return ::vkGetPhysicalDeviceFormatProperties2KHR( physicalDevice, format, pFormatProperties );
    }

    void vkGetPhysicalDeviceGeneratedCommandsPropertiesNVX( VkPhysicalDevice physicalDevice, VkDeviceGeneratedCommandsFeaturesNVX* pFeatures, VkDeviceGeneratedCommandsLimitsNVX* pLimits ) const
    {
      return ::vkGetPhysicalDeviceGeneratedCommandsPropertiesNVX( physicalDevice, pFeatures, pLimits );
    }

    VkResult vkGetPhysicalDeviceImageFormatProperties( VkPhysicalDevice physicalDevice, VkFormat format, VkImageType type, VkImageTiling tiling, VkImageUsageFlags usage, VkImageCreateFlags flags, VkImageFormatProperties* pImageFormatProperties ) const
    {
      return ::vkGetPhysicalDeviceImageFormatProperties( physicalDevice, format, type, tiling, usage, flags, pImageFormatProperties );
    }

    VkResult vkGetPhysicalDeviceImageFormatProperties2( VkPhysicalDevice physicalDevice, const VkPhysicalDeviceImageFormatInfo2* pImageFormatInfo, VkImageFormatProperties2* pImageFormatProperties ) const
    {
      return ::vkGetPhysicalDeviceImageFormatProperties2( physicalDevice, pImageFormatInfo, pImageFormatProperties );
    }

    VkResult vkGetPhysicalDeviceImageFormatProperties2KHR( VkPhysicalDevice physicalDevice, const VkPhysicalDeviceImageFormatInfo2* pImageFormatInfo, VkImageFormatProperties2* pImageFormatProperties ) const
    {
      return ::vkGetPhysicalDeviceImageFormatProperties2KHR( physicalDevice, pImageFormatInfo, pImageFormatProperties );
    }

    void vkGetPhysicalDeviceMemoryProperties( VkPhysicalDevice physicalDevice, VkPhysicalDeviceMemoryProperties* pMemoryProperties ) const
    {
      return ::vkGetPhysicalDeviceMemoryProperties( physicalDevice, pMemoryProperties );
    }

    void vkGetPhysicalDeviceMemoryProperties2( VkPhysicalDevice physicalDevice, VkPhysicalDeviceMemoryProperties2* pMemoryProperties ) const
    {
      return ::vkGetPhysicalDeviceMemoryProperties2( physicalDevice, pMemoryProperties );
    }

    void vkGetPhysicalDeviceMemoryProperties2KHR( VkPhysicalDevice physicalDevice, VkPhysicalDeviceMemoryProperties2* pMemoryProperties ) const
    {
      return ::vkGetPhysicalDeviceMemoryProperties2KHR( physicalDevice, pMemoryProperties );
    }

    void vkGetPhysicalDeviceMultisamplePropertiesEXT( VkPhysicalDevice physicalDevice, VkSampleCountFlagBits samples, VkMultisamplePropertiesEXT* pMultisampleProperties ) const
    {
      return ::vkGetPhysicalDeviceMultisamplePropertiesEXT( physicalDevice, samples, pMultisampleProperties );
    }

    VkResult vkGetPhysicalDevicePresentRectanglesKHR( VkPhysicalDevice physicalDevice, VkSurfaceKHR surface, uint32_t* pRectCount, VkRect2D* pRects ) const
    {
      return ::vkGetPhysicalDevicePresentRectanglesKHR( physicalDevice, surface, pRectCount, pRects );
    }

    void vkGetPhysicalDeviceProperties( VkPhysicalDevice physicalDevice, VkPhysicalDeviceProperties* pProperties ) const
    {
      return ::vkGetPhysicalDeviceProperties( physicalDevice, pProperties );
    }

    void vkGetPhysicalDeviceProperties2( VkPhysicalDevice physicalDevice, VkPhysicalDeviceProperties2* pProperties ) const
    {
      return ::vkGetPhysicalDeviceProperties2( physicalDevice, pProperties );
    }

    void vkGetPhysicalDeviceProperties2KHR( VkPhysicalDevice physicalDevice, VkPhysicalDeviceProperties2* pProperties ) const
    {
      return ::vkGetPhysicalDeviceProperties2KHR( physicalDevice, pProperties );
    }

    void vkGetPhysicalDeviceQueueFamilyProperties( VkPhysicalDevice physicalDevice, uint32_t* pQueueFamilyPropertyCount, VkQueueFamilyProperties* pQueueFamilyProperties ) const
    {
      return ::vkGetPhysicalDeviceQueueFamilyProperties( physicalDevice, pQueueFamilyPropertyCount, pQueueFamilyProperties );
    }

    void vkGetPhysicalDeviceQueueFamilyProperties2( VkPhysicalDevice physicalDevice, uint32_t* pQueueFamilyPropertyCount, VkQueueFamilyProperties2* pQueueFamilyProperties ) const
    {
      return ::vkGetPhysicalDeviceQueueFamilyProperties2( physicalDevice, pQueueFamilyPropertyCount, pQueueFamilyProperties );
    }

    void vkGetPhysicalDeviceQueueFamilyProperties2KHR( VkPhysicalDevice physicalDevice, uint32_t* pQueueFamilyPropertyCount, VkQueueFamilyProperties2* pQueueFamilyProperties ) const
    {
      return ::vkGetPhysicalDeviceQueueFamilyProperties2KHR( physicalDevice, pQueueFamilyPropertyCount, pQueueFamilyProperties );
    }

    void vkGetPhysicalDeviceSparseImageFormatProperties( VkPhysicalDevice physicalDevice, VkFormat format, VkImageType type, VkSampleCountFlagBits samples, VkImageUsageFlags usage, VkImageTiling tiling, uint32_t* pPropertyCount, VkSparseImageFormatProperties* pProperties ) const
    {
      return ::vkGetPhysicalDeviceSparseImageFormatProperties( physicalDevice, format, type, samples, usage, tiling, pPropertyCount, pProperties );
    }

    void vkGetPhysicalDeviceSparseImageFormatProperties2( VkPhysicalDevice physicalDevice, const VkPhysicalDeviceSparseImageFormatInfo2* pFormatInfo, uint32_t* pPropertyCount, VkSparseImageFormatProperties2* pProperties ) const
    {
      return ::vkGetPhysicalDeviceSparseImageFormatProperties2( physicalDevice, pFormatInfo, pPropertyCount, pProperties );
    }

    void vkGetPhysicalDeviceSparseImageFormatProperties2KHR( VkPhysicalDevice physicalDevice, const VkPhysicalDeviceSparseImageFormatInfo2* pFormatInfo, uint32_t* pPropertyCount, VkSparseImageFormatProperties2* pProperties ) const
    {
      return ::vkGetPhysicalDeviceSparseImageFormatProperties2KHR( physicalDevice, pFormatInfo, pPropertyCount, pProperties );
    }

    VkResult vkGetPhysicalDeviceSurfaceCapabilities2EXT( VkPhysicalDevice physicalDevice, VkSurfaceKHR surface, VkSurfaceCapabilities2EXT* pSurfaceCapabilities ) const
    {
      return ::vkGetPhysicalDeviceSurfaceCapabilities2EXT( physicalDevice, surface, pSurfaceCapabilities );
    }

    VkResult vkGetPhysicalDeviceSurfaceCapabilities2KHR( VkPhysicalDevice physicalDevice, const VkPhysicalDeviceSurfaceInfo2KHR* pSurfaceInfo, VkSurfaceCapabilities2KHR* pSurfaceCapabilities ) const
    {
      return ::vkGetPhysicalDeviceSurfaceCapabilities2KHR( physicalDevice, pSurfaceInfo, pSurfaceCapabilities );
    }

    VkResult vkGetPhysicalDeviceSurfaceCapabilitiesKHR( VkPhysicalDevice physicalDevice, VkSurfaceKHR surface, VkSurfaceCapabilitiesKHR* pSurfaceCapabilities ) const
    {
      return ::vkGetPhysicalDeviceSurfaceCapabilitiesKHR( physicalDevice, surface, pSurfaceCapabilities );
    }

    VkResult vkGetPhysicalDeviceSurfaceFormats2KHR( VkPhysicalDevice physicalDevice, const VkPhysicalDeviceSurfaceInfo2KHR* pSurfaceInfo, uint32_t* pSurfaceFormatCount, VkSurfaceFormat2KHR* pSurfaceFormats ) const
    {
      return ::vkGetPhysicalDeviceSurfaceFormats2KHR( physicalDevice, pSurfaceInfo, pSurfaceFormatCount, pSurfaceFormats );
    }

    VkResult vkGetPhysicalDeviceSurfaceFormatsKHR( VkPhysicalDevice physicalDevice, VkSurfaceKHR surface, uint32_t* pSurfaceFormatCount, VkSurfaceFormatKHR* pSurfaceFormats ) const
    {
      return ::vkGetPhysicalDeviceSurfaceFormatsKHR( physicalDevice, surface, pSurfaceFormatCount, pSurfaceFormats );
    }

    VkResult vkGetPhysicalDeviceSurfacePresentModesKHR( VkPhysicalDevice physicalDevice, VkSurfaceKHR surface, uint32_t* pPresentModeCount, VkPresentModeKHR* pPresentModes ) const
    {
      return ::vkGetPhysicalDeviceSurfacePresentModesKHR( physicalDevice, surface, pPresentModeCount, pPresentModes );
    }

    VkResult vkGetPhysicalDeviceSurfaceSupportKHR( VkPhysicalDevice physicalDevice, uint32_t queueFamilyIndex, VkSurfaceKHR surface, VkBool32* pSupported ) const
    {
      return ::vkGetPhysicalDeviceSurfaceSupportKHR( physicalDevice, queueFamilyIndex, surface, pSupported );
    }

#ifdef VK_USE_PLATFORM_WAYLAND_KHR
    VkBool32 vkGetPhysicalDeviceWaylandPresentationSupportKHR( VkPhysicalDevice physicalDevice, uint32_t queueFamilyIndex, struct wl_display* display ) const
    {
      return ::vkGetPhysicalDeviceWaylandPresentationSupportKHR( physicalDevice, queueFamilyIndex, display );
    }
#endif /*VK_USE_PLATFORM_WAYLAND_KHR*/

#ifdef VK_USE_PLATFORM_WIN32_KHR
    VkBool32 vkGetPhysicalDeviceWin32PresentationSupportKHR( VkPhysicalDevice physicalDevice, uint32_t queueFamilyIndex ) const
    {
      return ::vkGetPhysicalDeviceWin32PresentationSupportKHR( physicalDevice, queueFamilyIndex );
    }
#endif /*VK_USE_PLATFORM_WIN32_KHR*/

#ifdef VK_USE_PLATFORM_XCB_KHR
    VkBool32 vkGetPhysicalDeviceXcbPresentationSupportKHR( VkPhysicalDevice physicalDevice, uint32_t queueFamilyIndex, xcb_connection_t* connection, xcb_visualid_t visual_id ) const
    {
      return ::vkGetPhysicalDeviceXcbPresentationSupportKHR( physicalDevice, queueFamilyIndex, connection, visual_id );
    }
#endif /*VK_USE_PLATFORM_XCB_KHR*/

#ifdef VK_USE_PLATFORM_XLIB_KHR
    VkBool32 vkGetPhysicalDeviceXlibPresentationSupportKHR( VkPhysicalDevice physicalDevice, uint32_t queueFamilyIndex, Display* dpy, VisualID visualID ) const
    {
      return ::vkGetPhysicalDeviceXlibPresentationSupportKHR( physicalDevice, queueFamilyIndex, dpy, visualID );
    }
#endif /*VK_USE_PLATFORM_XLIB_KHR*/

#ifdef VK_USE_PLATFORM_XLIB_XRANDR_EXT
    VkResult vkGetRandROutputDisplayEXT( VkPhysicalDevice physicalDevice, Display* dpy, RROutput rrOutput, VkDisplayKHR* pDisplay ) const
    {
      return ::vkGetRandROutputDisplayEXT( physicalDevice, dpy, rrOutput, pDisplay );
    }
#endif /*VK_USE_PLATFORM_XLIB_XRANDR_EXT*/

    VkResult vkReleaseDisplayEXT( VkPhysicalDevice physicalDevice, VkDisplayKHR display ) const
    {
      return ::vkReleaseDisplayEXT( physicalDevice, display );
    }

    void vkGetQueueCheckpointDataNV( VkQueue queue, uint32_t* pCheckpointDataCount, VkCheckpointDataNV* pCheckpointData ) const
    {
      return ::vkGetQueueCheckpointDataNV( queue, pCheckpointDataCount, pCheckpointData );
    }

    void vkQueueBeginDebugUtilsLabelEXT( VkQueue queue, const VkDebugUtilsLabelEXT* pLabelInfo ) const
    {
      return ::vkQueueBeginDebugUtilsLabelEXT( queue, pLabelInfo );
    }

    VkResult vkQueueBindSparse( VkQueue queue, uint32_t bindInfoCount, const VkBindSparseInfo* pBindInfo, VkFence fence ) const
    {
      return ::vkQueueBindSparse( queue, bindInfoCount, pBindInfo, fence );
    }

    void vkQueueEndDebugUtilsLabelEXT( VkQueue queue ) const
    {
      return ::vkQueueEndDebugUtilsLabelEXT( queue );
    }

    void vkQueueInsertDebugUtilsLabelEXT( VkQueue queue, const VkDebugUtilsLabelEXT* pLabelInfo ) const
    {
      return ::vkQueueInsertDebugUtilsLabelEXT( queue, pLabelInfo );
    }

    VkResult vkQueuePresentKHR( VkQueue queue, const VkPresentInfoKHR* pPresentInfo ) const
    {
      return ::vkQueuePresentKHR( queue, pPresentInfo );
    }

    VkResult vkQueueSubmit( VkQueue queue, uint32_t submitCount, const VkSubmitInfo* pSubmits, VkFence fence ) const
    {
      return ::vkQueueSubmit( queue, submitCount, pSubmits, fence );
    }

    VkResult vkQueueWaitIdle( VkQueue queue ) const
    {
      return ::vkQueueWaitIdle( queue );
    }
  };

  struct AllocationCallbacks;

  template <typename OwnerType, typename Dispatch>
  class ObjectDestroy
  {
    public:
      ObjectDestroy( OwnerType owner = OwnerType(), Optional<const AllocationCallbacks> allocationCallbacks = nullptr, Dispatch const &dispatch = Dispatch() )
        : m_owner( owner )
        , m_allocationCallbacks( allocationCallbacks )
        , m_dispatch( &dispatch )
      {}

      OwnerType getOwner() const { return m_owner; }
      Optional<const AllocationCallbacks> getAllocator() const { return m_allocationCallbacks; }

    protected:
      template <typename T>
      void destroy(T t)
      {
        m_owner.destroy( t, m_allocationCallbacks, *m_dispatch );
      }

    private:
      OwnerType m_owner;
      Optional<const AllocationCallbacks> m_allocationCallbacks;
      Dispatch const* m_dispatch;
  };

  class NoParent;

  template <typename Dispatch>
  class ObjectDestroy<NoParent,Dispatch>
  {
    public:
      ObjectDestroy( Optional<const AllocationCallbacks> allocationCallbacks = nullptr, Dispatch const &dispatch = Dispatch() )
        : m_allocationCallbacks( allocationCallbacks )
        , m_dispatch( &dispatch )
      {}

      Optional<const AllocationCallbacks> getAllocator() const { return m_allocationCallbacks; }

    protected:
      template <typename T>
      void destroy(T t)
      {
        t.destroy( m_allocationCallbacks, *m_dispatch );
      }

    private:
      Optional<const AllocationCallbacks> m_allocationCallbacks;
      Dispatch const* m_dispatch;
  };

  template <typename OwnerType, typename Dispatch>
  class ObjectFree
  {
    public:
      ObjectFree( OwnerType owner = OwnerType(), Optional<const AllocationCallbacks> allocationCallbacks = nullptr, Dispatch const &dispatch = Dispatch() )
        : m_owner( owner )
        , m_allocationCallbacks( allocationCallbacks )
        , m_dispatch( &dispatch )
      {}

      OwnerType getOwner() const { return m_owner; }
      Optional<const AllocationCallbacks> getAllocator() const { return m_allocationCallbacks; }

    protected:
      template <typename T>
      void destroy(T t)
      {
        m_owner.free( t, m_allocationCallbacks, *m_dispatch );
      }

    private:
      OwnerType m_owner;
      Optional<const AllocationCallbacks> m_allocationCallbacks;
      Dispatch const* m_dispatch;
  };

  template <typename OwnerType, typename PoolType, typename Dispatch>
  class PoolFree
  {
    public:
      PoolFree( OwnerType owner = OwnerType(), PoolType pool = PoolType(), Dispatch const &dispatch = Dispatch() )
        : m_owner( owner )
        , m_pool( pool )
        , m_dispatch( &dispatch )
      {}

      OwnerType getOwner() const { return m_owner; }
      PoolType getPool() const { return m_pool; }

    protected:
      template <typename T>
      void destroy(T t)
      {
        m_owner.free( m_pool, t, *m_dispatch );
      }

    private:
      OwnerType m_owner;
      PoolType m_pool;
      Dispatch const* m_dispatch;
  };

  using Bool32 = uint32_t;
  using DeviceAddress = uint64_t;
  using DeviceSize = uint64_t;
  using SampleMask = uint32_t;

  enum class AccelerationStructureMemoryRequirementsTypeNV
  {
    eObject = VK_ACCELERATION_STRUCTURE_MEMORY_REQUIREMENTS_TYPE_OBJECT_NV,
    eBuildScratch = VK_ACCELERATION_STRUCTURE_MEMORY_REQUIREMENTS_TYPE_BUILD_SCRATCH_NV,
    eUpdateScratch = VK_ACCELERATION_STRUCTURE_MEMORY_REQUIREMENTS_TYPE_UPDATE_SCRATCH_NV
  };

  VULKAN_HPP_INLINE std::string to_string( AccelerationStructureMemoryRequirementsTypeNV value )
  {
    switch ( value )
    {
      case AccelerationStructureMemoryRequirementsTypeNV::eObject : return "Object";
      case AccelerationStructureMemoryRequirementsTypeNV::eBuildScratch : return "BuildScratch";
      case AccelerationStructureMemoryRequirementsTypeNV::eUpdateScratch : return "UpdateScratch";
      default: return "invalid";
    }
  }

  enum class AccelerationStructureTypeNV
  {
    eTopLevel = VK_ACCELERATION_STRUCTURE_TYPE_TOP_LEVEL_NV,
    eBottomLevel = VK_ACCELERATION_STRUCTURE_TYPE_BOTTOM_LEVEL_NV
  };

  VULKAN_HPP_INLINE std::string to_string( AccelerationStructureTypeNV value )
  {
    switch ( value )
    {
      case AccelerationStructureTypeNV::eTopLevel : return "TopLevel";
      case AccelerationStructureTypeNV::eBottomLevel : return "BottomLevel";
      default: return "invalid";
    }
  }

  enum class AttachmentLoadOp
  {
    eLoad = VK_ATTACHMENT_LOAD_OP_LOAD,
    eClear = VK_ATTACHMENT_LOAD_OP_CLEAR,
    eDontCare = VK_ATTACHMENT_LOAD_OP_DONT_CARE
  };

  VULKAN_HPP_INLINE std::string to_string( AttachmentLoadOp value )
  {
    switch ( value )
    {
      case AttachmentLoadOp::eLoad : return "Load";
      case AttachmentLoadOp::eClear : return "Clear";
      case AttachmentLoadOp::eDontCare : return "DontCare";
      default: return "invalid";
    }
  }

  enum class AttachmentStoreOp
  {
    eStore = VK_ATTACHMENT_STORE_OP_STORE,
    eDontCare = VK_ATTACHMENT_STORE_OP_DONT_CARE
  };

  VULKAN_HPP_INLINE std::string to_string( AttachmentStoreOp value )
  {
    switch ( value )
    {
      case AttachmentStoreOp::eStore : return "Store";
      case AttachmentStoreOp::eDontCare : return "DontCare";
      default: return "invalid";
    }
  }

  enum class BlendFactor
  {
    eZero = VK_BLEND_FACTOR_ZERO,
    eOne = VK_BLEND_FACTOR_ONE,
    eSrcColor = VK_BLEND_FACTOR_SRC_COLOR,
    eOneMinusSrcColor = VK_BLEND_FACTOR_ONE_MINUS_SRC_COLOR,
    eDstColor = VK_BLEND_FACTOR_DST_COLOR,
    eOneMinusDstColor = VK_BLEND_FACTOR_ONE_MINUS_DST_COLOR,
    eSrcAlpha = VK_BLEND_FACTOR_SRC_ALPHA,
    eOneMinusSrcAlpha = VK_BLEND_FACTOR_ONE_MINUS_SRC_ALPHA,
    eDstAlpha = VK_BLEND_FACTOR_DST_ALPHA,
    eOneMinusDstAlpha = VK_BLEND_FACTOR_ONE_MINUS_DST_ALPHA,
    eConstantColor = VK_BLEND_FACTOR_CONSTANT_COLOR,
    eOneMinusConstantColor = VK_BLEND_FACTOR_ONE_MINUS_CONSTANT_COLOR,
    eConstantAlpha = VK_BLEND_FACTOR_CONSTANT_ALPHA,
    eOneMinusConstantAlpha = VK_BLEND_FACTOR_ONE_MINUS_CONSTANT_ALPHA,
    eSrcAlphaSaturate = VK_BLEND_FACTOR_SRC_ALPHA_SATURATE,
    eSrc1Color = VK_BLEND_FACTOR_SRC1_COLOR,
    eOneMinusSrc1Color = VK_BLEND_FACTOR_ONE_MINUS_SRC1_COLOR,
    eSrc1Alpha = VK_BLEND_FACTOR_SRC1_ALPHA,
    eOneMinusSrc1Alpha = VK_BLEND_FACTOR_ONE_MINUS_SRC1_ALPHA
  };

  VULKAN_HPP_INLINE std::string to_string( BlendFactor value )
  {
    switch ( value )
    {
      case BlendFactor::eZero : return "Zero";
      case BlendFactor::eOne : return "One";
      case BlendFactor::eSrcColor : return "SrcColor";
      case BlendFactor::eOneMinusSrcColor : return "OneMinusSrcColor";
      case BlendFactor::eDstColor : return "DstColor";
      case BlendFactor::eOneMinusDstColor : return "OneMinusDstColor";
      case BlendFactor::eSrcAlpha : return "SrcAlpha";
      case BlendFactor::eOneMinusSrcAlpha : return "OneMinusSrcAlpha";
      case BlendFactor::eDstAlpha : return "DstAlpha";
      case BlendFactor::eOneMinusDstAlpha : return "OneMinusDstAlpha";
      case BlendFactor::eConstantColor : return "ConstantColor";
      case BlendFactor::eOneMinusConstantColor : return "OneMinusConstantColor";
      case BlendFactor::eConstantAlpha : return "ConstantAlpha";
      case BlendFactor::eOneMinusConstantAlpha : return "OneMinusConstantAlpha";
      case BlendFactor::eSrcAlphaSaturate : return "SrcAlphaSaturate";
      case BlendFactor::eSrc1Color : return "Src1Color";
      case BlendFactor::eOneMinusSrc1Color : return "OneMinusSrc1Color";
      case BlendFactor::eSrc1Alpha : return "Src1Alpha";
      case BlendFactor::eOneMinusSrc1Alpha : return "OneMinusSrc1Alpha";
      default: return "invalid";
    }
  }

  enum class BlendOp
  {
    eAdd = VK_BLEND_OP_ADD,
    eSubtract = VK_BLEND_OP_SUBTRACT,
    eReverseSubtract = VK_BLEND_OP_REVERSE_SUBTRACT,
    eMin = VK_BLEND_OP_MIN,
    eMax = VK_BLEND_OP_MAX,
    eZeroEXT = VK_BLEND_OP_ZERO_EXT,
    eSrcEXT = VK_BLEND_OP_SRC_EXT,
    eDstEXT = VK_BLEND_OP_DST_EXT,
    eSrcOverEXT = VK_BLEND_OP_SRC_OVER_EXT,
    eDstOverEXT = VK_BLEND_OP_DST_OVER_EXT,
    eSrcInEXT = VK_BLEND_OP_SRC_IN_EXT,
    eDstInEXT = VK_BLEND_OP_DST_IN_EXT,
    eSrcOutEXT = VK_BLEND_OP_SRC_OUT_EXT,
    eDstOutEXT = VK_BLEND_OP_DST_OUT_EXT,
    eSrcAtopEXT = VK_BLEND_OP_SRC_ATOP_EXT,
    eDstAtopEXT = VK_BLEND_OP_DST_ATOP_EXT,
    eXorEXT = VK_BLEND_OP_XOR_EXT,
    eMultiplyEXT = VK_BLEND_OP_MULTIPLY_EXT,
    eScreenEXT = VK_BLEND_OP_SCREEN_EXT,
    eOverlayEXT = VK_BLEND_OP_OVERLAY_EXT,
    eDarkenEXT = VK_BLEND_OP_DARKEN_EXT,
    eLightenEXT = VK_BLEND_OP_LIGHTEN_EXT,
    eColordodgeEXT = VK_BLEND_OP_COLORDODGE_EXT,
    eColorburnEXT = VK_BLEND_OP_COLORBURN_EXT,
    eHardlightEXT = VK_BLEND_OP_HARDLIGHT_EXT,
    eSoftlightEXT = VK_BLEND_OP_SOFTLIGHT_EXT,
    eDifferenceEXT = VK_BLEND_OP_DIFFERENCE_EXT,
    eExclusionEXT = VK_BLEND_OP_EXCLUSION_EXT,
    eInvertEXT = VK_BLEND_OP_INVERT_EXT,
    eInvertRgbEXT = VK_BLEND_OP_INVERT_RGB_EXT,
    eLineardodgeEXT = VK_BLEND_OP_LINEARDODGE_EXT,
    eLinearburnEXT = VK_BLEND_OP_LINEARBURN_EXT,
    eVividlightEXT = VK_BLEND_OP_VIVIDLIGHT_EXT,
    eLinearlightEXT = VK_BLEND_OP_LINEARLIGHT_EXT,
    ePinlightEXT = VK_BLEND_OP_PINLIGHT_EXT,
    eHardmixEXT = VK_BLEND_OP_HARDMIX_EXT,
    eHslHueEXT = VK_BLEND_OP_HSL_HUE_EXT,
    eHslSaturationEXT = VK_BLEND_OP_HSL_SATURATION_EXT,
    eHslColorEXT = VK_BLEND_OP_HSL_COLOR_EXT,
    eHslLuminosityEXT = VK_BLEND_OP_HSL_LUMINOSITY_EXT,
    ePlusEXT = VK_BLEND_OP_PLUS_EXT,
    ePlusClampedEXT = VK_BLEND_OP_PLUS_CLAMPED_EXT,
    ePlusClampedAlphaEXT = VK_BLEND_OP_PLUS_CLAMPED_ALPHA_EXT,
    ePlusDarkerEXT = VK_BLEND_OP_PLUS_DARKER_EXT,
    eMinusEXT = VK_BLEND_OP_MINUS_EXT,
    eMinusClampedEXT = VK_BLEND_OP_MINUS_CLAMPED_EXT,
    eContrastEXT = VK_BLEND_OP_CONTRAST_EXT,
    eInvertOvgEXT = VK_BLEND_OP_INVERT_OVG_EXT,
    eRedEXT = VK_BLEND_OP_RED_EXT,
    eGreenEXT = VK_BLEND_OP_GREEN_EXT,
    eBlueEXT = VK_BLEND_OP_BLUE_EXT
  };

  VULKAN_HPP_INLINE std::string to_string( BlendOp value )
  {
    switch ( value )
    {
      case BlendOp::eAdd : return "Add";
      case BlendOp::eSubtract : return "Subtract";
      case BlendOp::eReverseSubtract : return "ReverseSubtract";
      case BlendOp::eMin : return "Min";
      case BlendOp::eMax : return "Max";
      case BlendOp::eZeroEXT : return "ZeroEXT";
      case BlendOp::eSrcEXT : return "SrcEXT";
      case BlendOp::eDstEXT : return "DstEXT";
      case BlendOp::eSrcOverEXT : return "SrcOverEXT";
      case BlendOp::eDstOverEXT : return "DstOverEXT";
      case BlendOp::eSrcInEXT : return "SrcInEXT";
      case BlendOp::eDstInEXT : return "DstInEXT";
      case BlendOp::eSrcOutEXT : return "SrcOutEXT";
      case BlendOp::eDstOutEXT : return "DstOutEXT";
      case BlendOp::eSrcAtopEXT : return "SrcAtopEXT";
      case BlendOp::eDstAtopEXT : return "DstAtopEXT";
      case BlendOp::eXorEXT : return "XorEXT";
      case BlendOp::eMultiplyEXT : return "MultiplyEXT";
      case BlendOp::eScreenEXT : return "ScreenEXT";
      case BlendOp::eOverlayEXT : return "OverlayEXT";
      case BlendOp::eDarkenEXT : return "DarkenEXT";
      case BlendOp::eLightenEXT : return "LightenEXT";
      case BlendOp::eColordodgeEXT : return "ColordodgeEXT";
      case BlendOp::eColorburnEXT : return "ColorburnEXT";
      case BlendOp::eHardlightEXT : return "HardlightEXT";
      case BlendOp::eSoftlightEXT : return "SoftlightEXT";
      case BlendOp::eDifferenceEXT : return "DifferenceEXT";
      case BlendOp::eExclusionEXT : return "ExclusionEXT";
      case BlendOp::eInvertEXT : return "InvertEXT";
      case BlendOp::eInvertRgbEXT : return "InvertRgbEXT";
      case BlendOp::eLineardodgeEXT : return "LineardodgeEXT";
      case BlendOp::eLinearburnEXT : return "LinearburnEXT";
      case BlendOp::eVividlightEXT : return "VividlightEXT";
      case BlendOp::eLinearlightEXT : return "LinearlightEXT";
      case BlendOp::ePinlightEXT : return "PinlightEXT";
      case BlendOp::eHardmixEXT : return "HardmixEXT";
      case BlendOp::eHslHueEXT : return "HslHueEXT";
      case BlendOp::eHslSaturationEXT : return "HslSaturationEXT";
      case BlendOp::eHslColorEXT : return "HslColorEXT";
      case BlendOp::eHslLuminosityEXT : return "HslLuminosityEXT";
      case BlendOp::ePlusEXT : return "PlusEXT";
      case BlendOp::ePlusClampedEXT : return "PlusClampedEXT";
      case BlendOp::ePlusClampedAlphaEXT : return "PlusClampedAlphaEXT";
      case BlendOp::ePlusDarkerEXT : return "PlusDarkerEXT";
      case BlendOp::eMinusEXT : return "MinusEXT";
      case BlendOp::eMinusClampedEXT : return "MinusClampedEXT";
      case BlendOp::eContrastEXT : return "ContrastEXT";
      case BlendOp::eInvertOvgEXT : return "InvertOvgEXT";
      case BlendOp::eRedEXT : return "RedEXT";
      case BlendOp::eGreenEXT : return "GreenEXT";
      case BlendOp::eBlueEXT : return "BlueEXT";
      default: return "invalid";
    }
  }

  enum class BlendOverlapEXT
  {
    eUncorrelated = VK_BLEND_OVERLAP_UNCORRELATED_EXT,
    eDisjoint = VK_BLEND_OVERLAP_DISJOINT_EXT,
    eConjoint = VK_BLEND_OVERLAP_CONJOINT_EXT
  };

  VULKAN_HPP_INLINE std::string to_string( BlendOverlapEXT value )
  {
    switch ( value )
    {
      case BlendOverlapEXT::eUncorrelated : return "Uncorrelated";
      case BlendOverlapEXT::eDisjoint : return "Disjoint";
      case BlendOverlapEXT::eConjoint : return "Conjoint";
      default: return "invalid";
    }
  }

  enum class BorderColor
  {
    eFloatTransparentBlack = VK_BORDER_COLOR_FLOAT_TRANSPARENT_BLACK,
    eIntTransparentBlack = VK_BORDER_COLOR_INT_TRANSPARENT_BLACK,
    eFloatOpaqueBlack = VK_BORDER_COLOR_FLOAT_OPAQUE_BLACK,
    eIntOpaqueBlack = VK_BORDER_COLOR_INT_OPAQUE_BLACK,
    eFloatOpaqueWhite = VK_BORDER_COLOR_FLOAT_OPAQUE_WHITE,
    eIntOpaqueWhite = VK_BORDER_COLOR_INT_OPAQUE_WHITE
  };

  VULKAN_HPP_INLINE std::string to_string( BorderColor value )
  {
    switch ( value )
    {
      case BorderColor::eFloatTransparentBlack : return "FloatTransparentBlack";
      case BorderColor::eIntTransparentBlack : return "IntTransparentBlack";
      case BorderColor::eFloatOpaqueBlack : return "FloatOpaqueBlack";
      case BorderColor::eIntOpaqueBlack : return "IntOpaqueBlack";
      case BorderColor::eFloatOpaqueWhite : return "FloatOpaqueWhite";
      case BorderColor::eIntOpaqueWhite : return "IntOpaqueWhite";
      default: return "invalid";
    }
  }

  enum class ChromaLocation
  {
    eCositedEven = VK_CHROMA_LOCATION_COSITED_EVEN,
    eMidpoint = VK_CHROMA_LOCATION_MIDPOINT,
    eCositedEvenKHR = VK_CHROMA_LOCATION_COSITED_EVEN_KHR,
    eMidpointKHR = VK_CHROMA_LOCATION_MIDPOINT_KHR
  };

  VULKAN_HPP_INLINE std::string to_string( ChromaLocation value )
  {
    switch ( value )
    {
      case ChromaLocation::eCositedEven : return "CositedEven";
      case ChromaLocation::eMidpoint : return "Midpoint";
      default: return "invalid";
    }
  }

  enum class CoarseSampleOrderTypeNV
  {
    eDefault = VK_COARSE_SAMPLE_ORDER_TYPE_DEFAULT_NV,
    eCustom = VK_COARSE_SAMPLE_ORDER_TYPE_CUSTOM_NV,
    ePixelMajor = VK_COARSE_SAMPLE_ORDER_TYPE_PIXEL_MAJOR_NV,
    eSampleMajor = VK_COARSE_SAMPLE_ORDER_TYPE_SAMPLE_MAJOR_NV
  };

  VULKAN_HPP_INLINE std::string to_string( CoarseSampleOrderTypeNV value )
  {
    switch ( value )
    {
      case CoarseSampleOrderTypeNV::eDefault : return "Default";
      case CoarseSampleOrderTypeNV::eCustom : return "Custom";
      case CoarseSampleOrderTypeNV::ePixelMajor : return "PixelMajor";
      case CoarseSampleOrderTypeNV::eSampleMajor : return "SampleMajor";
      default: return "invalid";
    }
  }

  enum class ColorSpaceKHR
  {
    eSrgbNonlinear = VK_COLOR_SPACE_SRGB_NONLINEAR_KHR,
    eDisplayP3NonlinearEXT = VK_COLOR_SPACE_DISPLAY_P3_NONLINEAR_EXT,
    eExtendedSrgbLinearEXT = VK_COLOR_SPACE_EXTENDED_SRGB_LINEAR_EXT,
    eDciP3LinearEXT = VK_COLOR_SPACE_DCI_P3_LINEAR_EXT,
    eDciP3NonlinearEXT = VK_COLOR_SPACE_DCI_P3_NONLINEAR_EXT,
    eBt709LinearEXT = VK_COLOR_SPACE_BT709_LINEAR_EXT,
    eBt709NonlinearEXT = VK_COLOR_SPACE_BT709_NONLINEAR_EXT,
    eBt2020LinearEXT = VK_COLOR_SPACE_BT2020_LINEAR_EXT,
    eHdr10St2084EXT = VK_COLOR_SPACE_HDR10_ST2084_EXT,
    eDolbyvisionEXT = VK_COLOR_SPACE_DOLBYVISION_EXT,
    eHdr10HlgEXT = VK_COLOR_SPACE_HDR10_HLG_EXT,
    eAdobergbLinearEXT = VK_COLOR_SPACE_ADOBERGB_LINEAR_EXT,
    eAdobergbNonlinearEXT = VK_COLOR_SPACE_ADOBERGB_NONLINEAR_EXT,
    ePassThroughEXT = VK_COLOR_SPACE_PASS_THROUGH_EXT,
    eExtendedSrgbNonlinearEXT = VK_COLOR_SPACE_EXTENDED_SRGB_NONLINEAR_EXT,
    eVkColorspaceSrgbNonlinear = VK_COLORSPACE_SRGB_NONLINEAR_KHR
  };

  VULKAN_HPP_INLINE std::string to_string( ColorSpaceKHR value )
  {
    switch ( value )
    {
      case ColorSpaceKHR::eSrgbNonlinear : return "SrgbNonlinear";
      case ColorSpaceKHR::eDisplayP3NonlinearEXT : return "DisplayP3NonlinearEXT";
      case ColorSpaceKHR::eExtendedSrgbLinearEXT : return "ExtendedSrgbLinearEXT";
      case ColorSpaceKHR::eDciP3LinearEXT : return "DciP3LinearEXT";
      case ColorSpaceKHR::eDciP3NonlinearEXT : return "DciP3NonlinearEXT";
      case ColorSpaceKHR::eBt709LinearEXT : return "Bt709LinearEXT";
      case ColorSpaceKHR::eBt709NonlinearEXT : return "Bt709NonlinearEXT";
      case ColorSpaceKHR::eBt2020LinearEXT : return "Bt2020LinearEXT";
      case ColorSpaceKHR::eHdr10St2084EXT : return "Hdr10St2084EXT";
      case ColorSpaceKHR::eDolbyvisionEXT : return "DolbyvisionEXT";
      case ColorSpaceKHR::eHdr10HlgEXT : return "Hdr10HlgEXT";
      case ColorSpaceKHR::eAdobergbLinearEXT : return "AdobergbLinearEXT";
      case ColorSpaceKHR::eAdobergbNonlinearEXT : return "AdobergbNonlinearEXT";
      case ColorSpaceKHR::ePassThroughEXT : return "PassThroughEXT";
      case ColorSpaceKHR::eExtendedSrgbNonlinearEXT : return "ExtendedSrgbNonlinearEXT";
      default: return "invalid";
    }
  }

  enum class CommandBufferLevel
  {
    ePrimary = VK_COMMAND_BUFFER_LEVEL_PRIMARY,
    eSecondary = VK_COMMAND_BUFFER_LEVEL_SECONDARY
  };

  VULKAN_HPP_INLINE std::string to_string( CommandBufferLevel value )
  {
    switch ( value )
    {
      case CommandBufferLevel::ePrimary : return "Primary";
      case CommandBufferLevel::eSecondary : return "Secondary";
      default: return "invalid";
    }
  }

  enum class CompareOp
  {
    eNever = VK_COMPARE_OP_NEVER,
    eLess = VK_COMPARE_OP_LESS,
    eEqual = VK_COMPARE_OP_EQUAL,
    eLessOrEqual = VK_COMPARE_OP_LESS_OR_EQUAL,
    eGreater = VK_COMPARE_OP_GREATER,
    eNotEqual = VK_COMPARE_OP_NOT_EQUAL,
    eGreaterOrEqual = VK_COMPARE_OP_GREATER_OR_EQUAL,
    eAlways = VK_COMPARE_OP_ALWAYS
  };

  VULKAN_HPP_INLINE std::string to_string( CompareOp value )
  {
    switch ( value )
    {
      case CompareOp::eNever : return "Never";
      case CompareOp::eLess : return "Less";
      case CompareOp::eEqual : return "Equal";
      case CompareOp::eLessOrEqual : return "LessOrEqual";
      case CompareOp::eGreater : return "Greater";
      case CompareOp::eNotEqual : return "NotEqual";
      case CompareOp::eGreaterOrEqual : return "GreaterOrEqual";
      case CompareOp::eAlways : return "Always";
      default: return "invalid";
    }
  }

  enum class ComponentSwizzle
  {
    eIdentity = VK_COMPONENT_SWIZZLE_IDENTITY,
    eZero = VK_COMPONENT_SWIZZLE_ZERO,
    eOne = VK_COMPONENT_SWIZZLE_ONE,
    eR = VK_COMPONENT_SWIZZLE_R,
    eG = VK_COMPONENT_SWIZZLE_G,
    eB = VK_COMPONENT_SWIZZLE_B,
    eA = VK_COMPONENT_SWIZZLE_A
  };

  VULKAN_HPP_INLINE std::string to_string( ComponentSwizzle value )
  {
    switch ( value )
    {
      case ComponentSwizzle::eIdentity : return "Identity";
      case ComponentSwizzle::eZero : return "Zero";
      case ComponentSwizzle::eOne : return "One";
      case ComponentSwizzle::eR : return "R";
      case ComponentSwizzle::eG : return "G";
      case ComponentSwizzle::eB : return "B";
      case ComponentSwizzle::eA : return "A";
      default: return "invalid";
    }
  }

  enum class ComponentTypeNV
  {
    eFloat16 = VK_COMPONENT_TYPE_FLOAT16_NV,
    eFloat32 = VK_COMPONENT_TYPE_FLOAT32_NV,
    eFloat64 = VK_COMPONENT_TYPE_FLOAT64_NV,
    eSint8 = VK_COMPONENT_TYPE_SINT8_NV,
    eSint16 = VK_COMPONENT_TYPE_SINT16_NV,
    eSint32 = VK_COMPONENT_TYPE_SINT32_NV,
    eSint64 = VK_COMPONENT_TYPE_SINT64_NV,
    eUint8 = VK_COMPONENT_TYPE_UINT8_NV,
    eUint16 = VK_COMPONENT_TYPE_UINT16_NV,
    eUint32 = VK_COMPONENT_TYPE_UINT32_NV,
    eUint64 = VK_COMPONENT_TYPE_UINT64_NV
  };

  VULKAN_HPP_INLINE std::string to_string( ComponentTypeNV value )
  {
    switch ( value )
    {
      case ComponentTypeNV::eFloat16 : return "Float16";
      case ComponentTypeNV::eFloat32 : return "Float32";
      case ComponentTypeNV::eFloat64 : return "Float64";
      case ComponentTypeNV::eSint8 : return "Sint8";
      case ComponentTypeNV::eSint16 : return "Sint16";
      case ComponentTypeNV::eSint32 : return "Sint32";
      case ComponentTypeNV::eSint64 : return "Sint64";
      case ComponentTypeNV::eUint8 : return "Uint8";
      case ComponentTypeNV::eUint16 : return "Uint16";
      case ComponentTypeNV::eUint32 : return "Uint32";
      case ComponentTypeNV::eUint64 : return "Uint64";
      default: return "invalid";
    }
  }

  enum class ConservativeRasterizationModeEXT
  {
    eDisabled = VK_CONSERVATIVE_RASTERIZATION_MODE_DISABLED_EXT,
    eOverestimate = VK_CONSERVATIVE_RASTERIZATION_MODE_OVERESTIMATE_EXT,
    eUnderestimate = VK_CONSERVATIVE_RASTERIZATION_MODE_UNDERESTIMATE_EXT
  };

  VULKAN_HPP_INLINE std::string to_string( ConservativeRasterizationModeEXT value )
  {
    switch ( value )
    {
      case ConservativeRasterizationModeEXT::eDisabled : return "Disabled";
      case ConservativeRasterizationModeEXT::eOverestimate : return "Overestimate";
      case ConservativeRasterizationModeEXT::eUnderestimate : return "Underestimate";
      default: return "invalid";
    }
  }

  enum class CopyAccelerationStructureModeNV
  {
    eClone = VK_COPY_ACCELERATION_STRUCTURE_MODE_CLONE_NV,
    eCompact = VK_COPY_ACCELERATION_STRUCTURE_MODE_COMPACT_NV
  };

  VULKAN_HPP_INLINE std::string to_string( CopyAccelerationStructureModeNV value )
  {
    switch ( value )
    {
      case CopyAccelerationStructureModeNV::eClone : return "Clone";
      case CopyAccelerationStructureModeNV::eCompact : return "Compact";
      default: return "invalid";
    }
  }

  enum class CoverageModulationModeNV
  {
    eNone = VK_COVERAGE_MODULATION_MODE_NONE_NV,
    eRgb = VK_COVERAGE_MODULATION_MODE_RGB_NV,
    eAlpha = VK_COVERAGE_MODULATION_MODE_ALPHA_NV,
    eRgba = VK_COVERAGE_MODULATION_MODE_RGBA_NV
  };

  VULKAN_HPP_INLINE std::string to_string( CoverageModulationModeNV value )
  {
    switch ( value )
    {
      case CoverageModulationModeNV::eNone : return "None";
      case CoverageModulationModeNV::eRgb : return "Rgb";
      case CoverageModulationModeNV::eAlpha : return "Alpha";
      case CoverageModulationModeNV::eRgba : return "Rgba";
      default: return "invalid";
    }
  }

  enum class DebugReportObjectTypeEXT
  {
    eUnknown = VK_DEBUG_REPORT_OBJECT_TYPE_UNKNOWN_EXT,
    eInstance = VK_DEBUG_REPORT_OBJECT_TYPE_INSTANCE_EXT,
    ePhysicalDevice = VK_DEBUG_REPORT_OBJECT_TYPE_PHYSICAL_DEVICE_EXT,
    eDevice = VK_DEBUG_REPORT_OBJECT_TYPE_DEVICE_EXT,
    eQueue = VK_DEBUG_REPORT_OBJECT_TYPE_QUEUE_EXT,
    eSemaphore = VK_DEBUG_REPORT_OBJECT_TYPE_SEMAPHORE_EXT,
    eCommandBuffer = VK_DEBUG_REPORT_OBJECT_TYPE_COMMAND_BUFFER_EXT,
    eFence = VK_DEBUG_REPORT_OBJECT_TYPE_FENCE_EXT,
    eDeviceMemory = VK_DEBUG_REPORT_OBJECT_TYPE_DEVICE_MEMORY_EXT,
    eBuffer = VK_DEBUG_REPORT_OBJECT_TYPE_BUFFER_EXT,
    eImage = VK_DEBUG_REPORT_OBJECT_TYPE_IMAGE_EXT,
    eEvent = VK_DEBUG_REPORT_OBJECT_TYPE_EVENT_EXT,
    eQueryPool = VK_DEBUG_REPORT_OBJECT_TYPE_QUERY_POOL_EXT,
    eBufferView = VK_DEBUG_REPORT_OBJECT_TYPE_BUFFER_VIEW_EXT,
    eImageView = VK_DEBUG_REPORT_OBJECT_TYPE_IMAGE_VIEW_EXT,
    eShaderModule = VK_DEBUG_REPORT_OBJECT_TYPE_SHADER_MODULE_EXT,
    ePipelineCache = VK_DEBUG_REPORT_OBJECT_TYPE_PIPELINE_CACHE_EXT,
    ePipelineLayout = VK_DEBUG_REPORT_OBJECT_TYPE_PIPELINE_LAYOUT_EXT,
    eRenderPass = VK_DEBUG_REPORT_OBJECT_TYPE_RENDER_PASS_EXT,
    ePipeline = VK_DEBUG_REPORT_OBJECT_TYPE_PIPELINE_EXT,
    eDescriptorSetLayout = VK_DEBUG_REPORT_OBJECT_TYPE_DESCRIPTOR_SET_LAYOUT_EXT,
    eSampler = VK_DEBUG_REPORT_OBJECT_TYPE_SAMPLER_EXT,
    eDescriptorPool = VK_DEBUG_REPORT_OBJECT_TYPE_DESCRIPTOR_POOL_EXT,
    eDescriptorSet = VK_DEBUG_REPORT_OBJECT_TYPE_DESCRIPTOR_SET_EXT,
    eFramebuffer = VK_DEBUG_REPORT_OBJECT_TYPE_FRAMEBUFFER_EXT,
    eCommandPool = VK_DEBUG_REPORT_OBJECT_TYPE_COMMAND_POOL_EXT,
    eSurfaceKHR = VK_DEBUG_REPORT_OBJECT_TYPE_SURFACE_KHR_EXT,
    eSwapchainKHR = VK_DEBUG_REPORT_OBJECT_TYPE_SWAPCHAIN_KHR_EXT,
    eDebugReportCallbackEXT = VK_DEBUG_REPORT_OBJECT_TYPE_DEBUG_REPORT_CALLBACK_EXT_EXT,
    eDisplayKHR = VK_DEBUG_REPORT_OBJECT_TYPE_DISPLAY_KHR_EXT,
    eDisplayModeKHR = VK_DEBUG_REPORT_OBJECT_TYPE_DISPLAY_MODE_KHR_EXT,
    eObjectTableNVX = VK_DEBUG_REPORT_OBJECT_TYPE_OBJECT_TABLE_NVX_EXT,
    eIndirectCommandsLayoutNVX = VK_DEBUG_REPORT_OBJECT_TYPE_INDIRECT_COMMANDS_LAYOUT_NVX_EXT,
    eValidationCacheEXT = VK_DEBUG_REPORT_OBJECT_TYPE_VALIDATION_CACHE_EXT_EXT,
    eSamplerYcbcrConversion = VK_DEBUG_REPORT_OBJECT_TYPE_SAMPLER_YCBCR_CONVERSION_EXT,
    eDescriptorUpdateTemplate = VK_DEBUG_REPORT_OBJECT_TYPE_DESCRIPTOR_UPDATE_TEMPLATE_EXT,
    eAccelerationStructureNV = VK_DEBUG_REPORT_OBJECT_TYPE_ACCELERATION_STRUCTURE_NV_EXT,
    eDebugReport = VK_DEBUG_REPORT_OBJECT_TYPE_DEBUG_REPORT_EXT,
    eValidationCache = VK_DEBUG_REPORT_OBJECT_TYPE_VALIDATION_CACHE_EXT,
    eDescriptorUpdateTemplateKHR = VK_DEBUG_REPORT_OBJECT_TYPE_DESCRIPTOR_UPDATE_TEMPLATE_KHR_EXT,
    eSamplerYcbcrConversionKHR = VK_DEBUG_REPORT_OBJECT_TYPE_SAMPLER_YCBCR_CONVERSION_KHR_EXT
  };

  VULKAN_HPP_INLINE std::string to_string( DebugReportObjectTypeEXT value )
  {
    switch ( value )
    {
      case DebugReportObjectTypeEXT::eUnknown : return "Unknown";
      case DebugReportObjectTypeEXT::eInstance : return "Instance";
      case DebugReportObjectTypeEXT::ePhysicalDevice : return "PhysicalDevice";
      case DebugReportObjectTypeEXT::eDevice : return "Device";
      case DebugReportObjectTypeEXT::eQueue : return "Queue";
      case DebugReportObjectTypeEXT::eSemaphore : return "Semaphore";
      case DebugReportObjectTypeEXT::eCommandBuffer : return "CommandBuffer";
      case DebugReportObjectTypeEXT::eFence : return "Fence";
      case DebugReportObjectTypeEXT::eDeviceMemory : return "DeviceMemory";
      case DebugReportObjectTypeEXT::eBuffer : return "Buffer";
      case DebugReportObjectTypeEXT::eImage : return "Image";
      case DebugReportObjectTypeEXT::eEvent : return "Event";
      case DebugReportObjectTypeEXT::eQueryPool : return "QueryPool";
      case DebugReportObjectTypeEXT::eBufferView : return "BufferView";
      case DebugReportObjectTypeEXT::eImageView : return "ImageView";
      case DebugReportObjectTypeEXT::eShaderModule : return "ShaderModule";
      case DebugReportObjectTypeEXT::ePipelineCache : return "PipelineCache";
      case DebugReportObjectTypeEXT::ePipelineLayout : return "PipelineLayout";
      case DebugReportObjectTypeEXT::eRenderPass : return "RenderPass";
      case DebugReportObjectTypeEXT::ePipeline : return "Pipeline";
      case DebugReportObjectTypeEXT::eDescriptorSetLayout : return "DescriptorSetLayout";
      case DebugReportObjectTypeEXT::eSampler : return "Sampler";
      case DebugReportObjectTypeEXT::eDescriptorPool : return "DescriptorPool";
      case DebugReportObjectTypeEXT::eDescriptorSet : return "DescriptorSet";
      case DebugReportObjectTypeEXT::eFramebuffer : return "Framebuffer";
      case DebugReportObjectTypeEXT::eCommandPool : return "CommandPool";
      case DebugReportObjectTypeEXT::eSurfaceKHR : return "SurfaceKHR";
      case DebugReportObjectTypeEXT::eSwapchainKHR : return "SwapchainKHR";
      case DebugReportObjectTypeEXT::eDebugReportCallbackEXT : return "DebugReportCallbackEXT";
      case DebugReportObjectTypeEXT::eDisplayKHR : return "DisplayKHR";
      case DebugReportObjectTypeEXT::eDisplayModeKHR : return "DisplayModeKHR";
      case DebugReportObjectTypeEXT::eObjectTableNVX : return "ObjectTableNVX";
      case DebugReportObjectTypeEXT::eIndirectCommandsLayoutNVX : return "IndirectCommandsLayoutNVX";
      case DebugReportObjectTypeEXT::eValidationCacheEXT : return "ValidationCacheEXT";
      case DebugReportObjectTypeEXT::eSamplerYcbcrConversion : return "SamplerYcbcrConversion";
      case DebugReportObjectTypeEXT::eDescriptorUpdateTemplate : return "DescriptorUpdateTemplate";
      case DebugReportObjectTypeEXT::eAccelerationStructureNV : return "AccelerationStructureNV";
      default: return "invalid";
    }
  }

  enum class DescriptorType
  {
    eSampler = VK_DESCRIPTOR_TYPE_SAMPLER,
    eCombinedImageSampler = VK_DESCRIPTOR_TYPE_COMBINED_IMAGE_SAMPLER,
    eSampledImage = VK_DESCRIPTOR_TYPE_SAMPLED_IMAGE,
    eStorageImage = VK_DESCRIPTOR_TYPE_STORAGE_IMAGE,
    eUniformTexelBuffer = VK_DESCRIPTOR_TYPE_UNIFORM_TEXEL_BUFFER,
    eStorageTexelBuffer = VK_DESCRIPTOR_TYPE_STORAGE_TEXEL_BUFFER,
    eUniformBuffer = VK_DESCRIPTOR_TYPE_UNIFORM_BUFFER,
    eStorageBuffer = VK_DESCRIPTOR_TYPE_STORAGE_BUFFER,
    eUniformBufferDynamic = VK_DESCRIPTOR_TYPE_UNIFORM_BUFFER_DYNAMIC,
    eStorageBufferDynamic = VK_DESCRIPTOR_TYPE_STORAGE_BUFFER_DYNAMIC,
    eInputAttachment = VK_DESCRIPTOR_TYPE_INPUT_ATTACHMENT,
    eInlineUniformBlockEXT = VK_DESCRIPTOR_TYPE_INLINE_UNIFORM_BLOCK_EXT,
    eAccelerationStructureNV = VK_DESCRIPTOR_TYPE_ACCELERATION_STRUCTURE_NV
  };

  VULKAN_HPP_INLINE std::string to_string( DescriptorType value )
  {
    switch ( value )
    {
      case DescriptorType::eSampler : return "Sampler";
      case DescriptorType::eCombinedImageSampler : return "CombinedImageSampler";
      case DescriptorType::eSampledImage : return "SampledImage";
      case DescriptorType::eStorageImage : return "StorageImage";
      case DescriptorType::eUniformTexelBuffer : return "UniformTexelBuffer";
      case DescriptorType::eStorageTexelBuffer : return "StorageTexelBuffer";
      case DescriptorType::eUniformBuffer : return "UniformBuffer";
      case DescriptorType::eStorageBuffer : return "StorageBuffer";
      case DescriptorType::eUniformBufferDynamic : return "UniformBufferDynamic";
      case DescriptorType::eStorageBufferDynamic : return "StorageBufferDynamic";
      case DescriptorType::eInputAttachment : return "InputAttachment";
      case DescriptorType::eInlineUniformBlockEXT : return "InlineUniformBlockEXT";
      case DescriptorType::eAccelerationStructureNV : return "AccelerationStructureNV";
      default: return "invalid";
    }
  }

  enum class DescriptorUpdateTemplateType
  {
    eDescriptorSet = VK_DESCRIPTOR_UPDATE_TEMPLATE_TYPE_DESCRIPTOR_SET,
    ePushDescriptorsKHR = VK_DESCRIPTOR_UPDATE_TEMPLATE_TYPE_PUSH_DESCRIPTORS_KHR,
    eDescriptorSetKHR = VK_DESCRIPTOR_UPDATE_TEMPLATE_TYPE_DESCRIPTOR_SET_KHR
  };

  VULKAN_HPP_INLINE std::string to_string( DescriptorUpdateTemplateType value )
  {
    switch ( value )
    {
      case DescriptorUpdateTemplateType::eDescriptorSet : return "DescriptorSet";
      case DescriptorUpdateTemplateType::ePushDescriptorsKHR : return "PushDescriptorsKHR";
      default: return "invalid";
    }
  }

  enum class DeviceEventTypeEXT
  {
    eDisplayHotplug = VK_DEVICE_EVENT_TYPE_DISPLAY_HOTPLUG_EXT
  };

  VULKAN_HPP_INLINE std::string to_string( DeviceEventTypeEXT value )
  {
    switch ( value )
    {
      case DeviceEventTypeEXT::eDisplayHotplug : return "DisplayHotplug";
      default: return "invalid";
    }
  }

  enum class DiscardRectangleModeEXT
  {
    eInclusive = VK_DISCARD_RECTANGLE_MODE_INCLUSIVE_EXT,
    eExclusive = VK_DISCARD_RECTANGLE_MODE_EXCLUSIVE_EXT
  };

  VULKAN_HPP_INLINE std::string to_string( DiscardRectangleModeEXT value )
  {
    switch ( value )
    {
      case DiscardRectangleModeEXT::eInclusive : return "Inclusive";
      case DiscardRectangleModeEXT::eExclusive : return "Exclusive";
      default: return "invalid";
    }
  }

  enum class DisplayEventTypeEXT
  {
    eFirstPixelOut = VK_DISPLAY_EVENT_TYPE_FIRST_PIXEL_OUT_EXT
  };

  VULKAN_HPP_INLINE std::string to_string( DisplayEventTypeEXT value )
  {
    switch ( value )
    {
      case DisplayEventTypeEXT::eFirstPixelOut : return "FirstPixelOut";
      default: return "invalid";
    }
  }

  enum class DisplayPowerStateEXT
  {
    eOff = VK_DISPLAY_POWER_STATE_OFF_EXT,
    eSuspend = VK_DISPLAY_POWER_STATE_SUSPEND_EXT,
    eOn = VK_DISPLAY_POWER_STATE_ON_EXT
  };

  VULKAN_HPP_INLINE std::string to_string( DisplayPowerStateEXT value )
  {
    switch ( value )
    {
      case DisplayPowerStateEXT::eOff : return "Off";
      case DisplayPowerStateEXT::eSuspend : return "Suspend";
      case DisplayPowerStateEXT::eOn : return "On";
      default: return "invalid";
    }
  }

  enum class DriverIdKHR
  {
    eAmdProprietary = VK_DRIVER_ID_AMD_PROPRIETARY_KHR,
    eAmdOpenSource = VK_DRIVER_ID_AMD_OPEN_SOURCE_KHR,
    eMesaRadv = VK_DRIVER_ID_MESA_RADV_KHR,
    eNvidiaProprietary = VK_DRIVER_ID_NVIDIA_PROPRIETARY_KHR,
    eIntelProprietaryWindows = VK_DRIVER_ID_INTEL_PROPRIETARY_WINDOWS_KHR,
    eIntelOpenSourceMESA = VK_DRIVER_ID_INTEL_OPEN_SOURCE_MESA_KHR,
    eImaginationProprietary = VK_DRIVER_ID_IMAGINATION_PROPRIETARY_KHR,
    eQualcommProprietary = VK_DRIVER_ID_QUALCOMM_PROPRIETARY_KHR,
    eArmProprietary = VK_DRIVER_ID_ARM_PROPRIETARY_KHR,
    eGooglePastel = VK_DRIVER_ID_GOOGLE_PASTEL_KHR
  };

  VULKAN_HPP_INLINE std::string to_string( DriverIdKHR value )
  {
    switch ( value )
    {
      case DriverIdKHR::eAmdProprietary : return "AmdProprietary";
      case DriverIdKHR::eAmdOpenSource : return "AmdOpenSource";
      case DriverIdKHR::eMesaRadv : return "MesaRadv";
      case DriverIdKHR::eNvidiaProprietary : return "NvidiaProprietary";
      case DriverIdKHR::eIntelProprietaryWindows : return "IntelProprietaryWindows";
      case DriverIdKHR::eIntelOpenSourceMESA : return "IntelOpenSourceMESA";
      case DriverIdKHR::eImaginationProprietary : return "ImaginationProprietary";
      case DriverIdKHR::eQualcommProprietary : return "QualcommProprietary";
      case DriverIdKHR::eArmProprietary : return "ArmProprietary";
      case DriverIdKHR::eGooglePastel : return "GooglePastel";
      default: return "invalid";
    }
  }

  enum class DynamicState
  {
    eViewport = VK_DYNAMIC_STATE_VIEWPORT,
    eScissor = VK_DYNAMIC_STATE_SCISSOR,
    eLineWidth = VK_DYNAMIC_STATE_LINE_WIDTH,
    eDepthBias = VK_DYNAMIC_STATE_DEPTH_BIAS,
    eBlendConstants = VK_DYNAMIC_STATE_BLEND_CONSTANTS,
    eDepthBounds = VK_DYNAMIC_STATE_DEPTH_BOUNDS,
    eStencilCompareMask = VK_DYNAMIC_STATE_STENCIL_COMPARE_MASK,
    eStencilWriteMask = VK_DYNAMIC_STATE_STENCIL_WRITE_MASK,
    eStencilReference = VK_DYNAMIC_STATE_STENCIL_REFERENCE,
    eViewportWScalingNV = VK_DYNAMIC_STATE_VIEWPORT_W_SCALING_NV,
    eDiscardRectangleEXT = VK_DYNAMIC_STATE_DISCARD_RECTANGLE_EXT,
    eSampleLocationsEXT = VK_DYNAMIC_STATE_SAMPLE_LOCATIONS_EXT,
    eViewportShadingRatePaletteNV = VK_DYNAMIC_STATE_VIEWPORT_SHADING_RATE_PALETTE_NV,
    eViewportCoarseSampleOrderNV = VK_DYNAMIC_STATE_VIEWPORT_COARSE_SAMPLE_ORDER_NV,
    eExclusiveScissorNV = VK_DYNAMIC_STATE_EXCLUSIVE_SCISSOR_NV
  };

  VULKAN_HPP_INLINE std::string to_string( DynamicState value )
  {
    switch ( value )
    {
      case DynamicState::eViewport : return "Viewport";
      case DynamicState::eScissor : return "Scissor";
      case DynamicState::eLineWidth : return "LineWidth";
      case DynamicState::eDepthBias : return "DepthBias";
      case DynamicState::eBlendConstants : return "BlendConstants";
      case DynamicState::eDepthBounds : return "DepthBounds";
      case DynamicState::eStencilCompareMask : return "StencilCompareMask";
      case DynamicState::eStencilWriteMask : return "StencilWriteMask";
      case DynamicState::eStencilReference : return "StencilReference";
      case DynamicState::eViewportWScalingNV : return "ViewportWScalingNV";
      case DynamicState::eDiscardRectangleEXT : return "DiscardRectangleEXT";
      case DynamicState::eSampleLocationsEXT : return "SampleLocationsEXT";
      case DynamicState::eViewportShadingRatePaletteNV : return "ViewportShadingRatePaletteNV";
      case DynamicState::eViewportCoarseSampleOrderNV : return "ViewportCoarseSampleOrderNV";
      case DynamicState::eExclusiveScissorNV : return "ExclusiveScissorNV";
      default: return "invalid";
    }
  }

  enum class Filter
  {
    eNearest = VK_FILTER_NEAREST,
    eLinear = VK_FILTER_LINEAR,
    eCubicIMG = VK_FILTER_CUBIC_IMG,
    eCubicEXT = VK_FILTER_CUBIC_EXT
  };

  VULKAN_HPP_INLINE std::string to_string( Filter value )
  {
    switch ( value )
    {
      case Filter::eNearest : return "Nearest";
      case Filter::eLinear : return "Linear";
      case Filter::eCubicIMG : return "CubicIMG";
      default: return "invalid";
    }
  }

  enum class Format
  {
    eUndefined = VK_FORMAT_UNDEFINED,
    eR4G4UnormPack8 = VK_FORMAT_R4G4_UNORM_PACK8,
    eR4G4B4A4UnormPack16 = VK_FORMAT_R4G4B4A4_UNORM_PACK16,
    eB4G4R4A4UnormPack16 = VK_FORMAT_B4G4R4A4_UNORM_PACK16,
    eR5G6B5UnormPack16 = VK_FORMAT_R5G6B5_UNORM_PACK16,
    eB5G6R5UnormPack16 = VK_FORMAT_B5G6R5_UNORM_PACK16,
    eR5G5B5A1UnormPack16 = VK_FORMAT_R5G5B5A1_UNORM_PACK16,
    eB5G5R5A1UnormPack16 = VK_FORMAT_B5G5R5A1_UNORM_PACK16,
    eA1R5G5B5UnormPack16 = VK_FORMAT_A1R5G5B5_UNORM_PACK16,
    eR8Unorm = VK_FORMAT_R8_UNORM,
    eR8Snorm = VK_FORMAT_R8_SNORM,
    eR8Uscaled = VK_FORMAT_R8_USCALED,
    eR8Sscaled = VK_FORMAT_R8_SSCALED,
    eR8Uint = VK_FORMAT_R8_UINT,
    eR8Sint = VK_FORMAT_R8_SINT,
    eR8Srgb = VK_FORMAT_R8_SRGB,
    eR8G8Unorm = VK_FORMAT_R8G8_UNORM,
    eR8G8Snorm = VK_FORMAT_R8G8_SNORM,
    eR8G8Uscaled = VK_FORMAT_R8G8_USCALED,
    eR8G8Sscaled = VK_FORMAT_R8G8_SSCALED,
    eR8G8Uint = VK_FORMAT_R8G8_UINT,
    eR8G8Sint = VK_FORMAT_R8G8_SINT,
    eR8G8Srgb = VK_FORMAT_R8G8_SRGB,
    eR8G8B8Unorm = VK_FORMAT_R8G8B8_UNORM,
    eR8G8B8Snorm = VK_FORMAT_R8G8B8_SNORM,
    eR8G8B8Uscaled = VK_FORMAT_R8G8B8_USCALED,
    eR8G8B8Sscaled = VK_FORMAT_R8G8B8_SSCALED,
    eR8G8B8Uint = VK_FORMAT_R8G8B8_UINT,
    eR8G8B8Sint = VK_FORMAT_R8G8B8_SINT,
    eR8G8B8Srgb = VK_FORMAT_R8G8B8_SRGB,
    eB8G8R8Unorm = VK_FORMAT_B8G8R8_UNORM,
    eB8G8R8Snorm = VK_FORMAT_B8G8R8_SNORM,
    eB8G8R8Uscaled = VK_FORMAT_B8G8R8_USCALED,
    eB8G8R8Sscaled = VK_FORMAT_B8G8R8_SSCALED,
    eB8G8R8Uint = VK_FORMAT_B8G8R8_UINT,
    eB8G8R8Sint = VK_FORMAT_B8G8R8_SINT,
    eB8G8R8Srgb = VK_FORMAT_B8G8R8_SRGB,
    eR8G8B8A8Unorm = VK_FORMAT_R8G8B8A8_UNORM,
    eR8G8B8A8Snorm = VK_FORMAT_R8G8B8A8_SNORM,
    eR8G8B8A8Uscaled = VK_FORMAT_R8G8B8A8_USCALED,
    eR8G8B8A8Sscaled = VK_FORMAT_R8G8B8A8_SSCALED,
    eR8G8B8A8Uint = VK_FORMAT_R8G8B8A8_UINT,
    eR8G8B8A8Sint = VK_FORMAT_R8G8B8A8_SINT,
    eR8G8B8A8Srgb = VK_FORMAT_R8G8B8A8_SRGB,
    eB8G8R8A8Unorm = VK_FORMAT_B8G8R8A8_UNORM,
    eB8G8R8A8Snorm = VK_FORMAT_B8G8R8A8_SNORM,
    eB8G8R8A8Uscaled = VK_FORMAT_B8G8R8A8_USCALED,
    eB8G8R8A8Sscaled = VK_FORMAT_B8G8R8A8_SSCALED,
    eB8G8R8A8Uint = VK_FORMAT_B8G8R8A8_UINT,
    eB8G8R8A8Sint = VK_FORMAT_B8G8R8A8_SINT,
    eB8G8R8A8Srgb = VK_FORMAT_B8G8R8A8_SRGB,
    eA8B8G8R8UnormPack32 = VK_FORMAT_A8B8G8R8_UNORM_PACK32,
    eA8B8G8R8SnormPack32 = VK_FORMAT_A8B8G8R8_SNORM_PACK32,
    eA8B8G8R8UscaledPack32 = VK_FORMAT_A8B8G8R8_USCALED_PACK32,
    eA8B8G8R8SscaledPack32 = VK_FORMAT_A8B8G8R8_SSCALED_PACK32,
    eA8B8G8R8UintPack32 = VK_FORMAT_A8B8G8R8_UINT_PACK32,
    eA8B8G8R8SintPack32 = VK_FORMAT_A8B8G8R8_SINT_PACK32,
    eA8B8G8R8SrgbPack32 = VK_FORMAT_A8B8G8R8_SRGB_PACK32,
    eA2R10G10B10UnormPack32 = VK_FORMAT_A2R10G10B10_UNORM_PACK32,
    eA2R10G10B10SnormPack32 = VK_FORMAT_A2R10G10B10_SNORM_PACK32,
    eA2R10G10B10UscaledPack32 = VK_FORMAT_A2R10G10B10_USCALED_PACK32,
    eA2R10G10B10SscaledPack32 = VK_FORMAT_A2R10G10B10_SSCALED_PACK32,
    eA2R10G10B10UintPack32 = VK_FORMAT_A2R10G10B10_UINT_PACK32,
    eA2R10G10B10SintPack32 = VK_FORMAT_A2R10G10B10_SINT_PACK32,
    eA2B10G10R10UnormPack32 = VK_FORMAT_A2B10G10R10_UNORM_PACK32,
    eA2B10G10R10SnormPack32 = VK_FORMAT_A2B10G10R10_SNORM_PACK32,
    eA2B10G10R10UscaledPack32 = VK_FORMAT_A2B10G10R10_USCALED_PACK32,
    eA2B10G10R10SscaledPack32 = VK_FORMAT_A2B10G10R10_SSCALED_PACK32,
    eA2B10G10R10UintPack32 = VK_FORMAT_A2B10G10R10_UINT_PACK32,
    eA2B10G10R10SintPack32 = VK_FORMAT_A2B10G10R10_SINT_PACK32,
    eR16Unorm = VK_FORMAT_R16_UNORM,
    eR16Snorm = VK_FORMAT_R16_SNORM,
    eR16Uscaled = VK_FORMAT_R16_USCALED,
    eR16Sscaled = VK_FORMAT_R16_SSCALED,
    eR16Uint = VK_FORMAT_R16_UINT,
    eR16Sint = VK_FORMAT_R16_SINT,
    eR16Sfloat = VK_FORMAT_R16_SFLOAT,
    eR16G16Unorm = VK_FORMAT_R16G16_UNORM,
    eR16G16Snorm = VK_FORMAT_R16G16_SNORM,
    eR16G16Uscaled = VK_FORMAT_R16G16_USCALED,
    eR16G16Sscaled = VK_FORMAT_R16G16_SSCALED,
    eR16G16Uint = VK_FORMAT_R16G16_UINT,
    eR16G16Sint = VK_FORMAT_R16G16_SINT,
    eR16G16Sfloat = VK_FORMAT_R16G16_SFLOAT,
    eR16G16B16Unorm = VK_FORMAT_R16G16B16_UNORM,
    eR16G16B16Snorm = VK_FORMAT_R16G16B16_SNORM,
    eR16G16B16Uscaled = VK_FORMAT_R16G16B16_USCALED,
    eR16G16B16Sscaled = VK_FORMAT_R16G16B16_SSCALED,
    eR16G16B16Uint = VK_FORMAT_R16G16B16_UINT,
    eR16G16B16Sint = VK_FORMAT_R16G16B16_SINT,
    eR16G16B16Sfloat = VK_FORMAT_R16G16B16_SFLOAT,
    eR16G16B16A16Unorm = VK_FORMAT_R16G16B16A16_UNORM,
    eR16G16B16A16Snorm = VK_FORMAT_R16G16B16A16_SNORM,
    eR16G16B16A16Uscaled = VK_FORMAT_R16G16B16A16_USCALED,
    eR16G16B16A16Sscaled = VK_FORMAT_R16G16B16A16_SSCALED,
    eR16G16B16A16Uint = VK_FORMAT_R16G16B16A16_UINT,
    eR16G16B16A16Sint = VK_FORMAT_R16G16B16A16_SINT,
    eR16G16B16A16Sfloat = VK_FORMAT_R16G16B16A16_SFLOAT,
    eR32Uint = VK_FORMAT_R32_UINT,
    eR32Sint = VK_FORMAT_R32_SINT,
    eR32Sfloat = VK_FORMAT_R32_SFLOAT,
    eR32G32Uint = VK_FORMAT_R32G32_UINT,
    eR32G32Sint = VK_FORMAT_R32G32_SINT,
    eR32G32Sfloat = VK_FORMAT_R32G32_SFLOAT,
    eR32G32B32Uint = VK_FORMAT_R32G32B32_UINT,
    eR32G32B32Sint = VK_FORMAT_R32G32B32_SINT,
    eR32G32B32Sfloat = VK_FORMAT_R32G32B32_SFLOAT,
    eR32G32B32A32Uint = VK_FORMAT_R32G32B32A32_UINT,
    eR32G32B32A32Sint = VK_FORMAT_R32G32B32A32_SINT,
    eR32G32B32A32Sfloat = VK_FORMAT_R32G32B32A32_SFLOAT,
    eR64Uint = VK_FORMAT_R64_UINT,
    eR64Sint = VK_FORMAT_R64_SINT,
    eR64Sfloat = VK_FORMAT_R64_SFLOAT,
    eR64G64Uint = VK_FORMAT_R64G64_UINT,
    eR64G64Sint = VK_FORMAT_R64G64_SINT,
    eR64G64Sfloat = VK_FORMAT_R64G64_SFLOAT,
    eR64G64B64Uint = VK_FORMAT_R64G64B64_UINT,
    eR64G64B64Sint = VK_FORMAT_R64G64B64_SINT,
    eR64G64B64Sfloat = VK_FORMAT_R64G64B64_SFLOAT,
    eR64G64B64A64Uint = VK_FORMAT_R64G64B64A64_UINT,
    eR64G64B64A64Sint = VK_FORMAT_R64G64B64A64_SINT,
    eR64G64B64A64Sfloat = VK_FORMAT_R64G64B64A64_SFLOAT,
    eB10G11R11UfloatPack32 = VK_FORMAT_B10G11R11_UFLOAT_PACK32,
    eE5B9G9R9UfloatPack32 = VK_FORMAT_E5B9G9R9_UFLOAT_PACK32,
    eD16Unorm = VK_FORMAT_D16_UNORM,
    eX8D24UnormPack32 = VK_FORMAT_X8_D24_UNORM_PACK32,
    eD32Sfloat = VK_FORMAT_D32_SFLOAT,
    eS8Uint = VK_FORMAT_S8_UINT,
    eD16UnormS8Uint = VK_FORMAT_D16_UNORM_S8_UINT,
    eD24UnormS8Uint = VK_FORMAT_D24_UNORM_S8_UINT,
    eD32SfloatS8Uint = VK_FORMAT_D32_SFLOAT_S8_UINT,
    eBc1RgbUnormBlock = VK_FORMAT_BC1_RGB_UNORM_BLOCK,
    eBc1RgbSrgbBlock = VK_FORMAT_BC1_RGB_SRGB_BLOCK,
    eBc1RgbaUnormBlock = VK_FORMAT_BC1_RGBA_UNORM_BLOCK,
    eBc1RgbaSrgbBlock = VK_FORMAT_BC1_RGBA_SRGB_BLOCK,
    eBc2UnormBlock = VK_FORMAT_BC2_UNORM_BLOCK,
    eBc2SrgbBlock = VK_FORMAT_BC2_SRGB_BLOCK,
    eBc3UnormBlock = VK_FORMAT_BC3_UNORM_BLOCK,
    eBc3SrgbBlock = VK_FORMAT_BC3_SRGB_BLOCK,
    eBc4UnormBlock = VK_FORMAT_BC4_UNORM_BLOCK,
    eBc4SnormBlock = VK_FORMAT_BC4_SNORM_BLOCK,
    eBc5UnormBlock = VK_FORMAT_BC5_UNORM_BLOCK,
    eBc5SnormBlock = VK_FORMAT_BC5_SNORM_BLOCK,
    eBc6HUfloatBlock = VK_FORMAT_BC6H_UFLOAT_BLOCK,
    eBc6HSfloatBlock = VK_FORMAT_BC6H_SFLOAT_BLOCK,
    eBc7UnormBlock = VK_FORMAT_BC7_UNORM_BLOCK,
    eBc7SrgbBlock = VK_FORMAT_BC7_SRGB_BLOCK,
    eEtc2R8G8B8UnormBlock = VK_FORMAT_ETC2_R8G8B8_UNORM_BLOCK,
    eEtc2R8G8B8SrgbBlock = VK_FORMAT_ETC2_R8G8B8_SRGB_BLOCK,
    eEtc2R8G8B8A1UnormBlock = VK_FORMAT_ETC2_R8G8B8A1_UNORM_BLOCK,
    eEtc2R8G8B8A1SrgbBlock = VK_FORMAT_ETC2_R8G8B8A1_SRGB_BLOCK,
    eEtc2R8G8B8A8UnormBlock = VK_FORMAT_ETC2_R8G8B8A8_UNORM_BLOCK,
    eEtc2R8G8B8A8SrgbBlock = VK_FORMAT_ETC2_R8G8B8A8_SRGB_BLOCK,
    eEacR11UnormBlock = VK_FORMAT_EAC_R11_UNORM_BLOCK,
    eEacR11SnormBlock = VK_FORMAT_EAC_R11_SNORM_BLOCK,
    eEacR11G11UnormBlock = VK_FORMAT_EAC_R11G11_UNORM_BLOCK,
    eEacR11G11SnormBlock = VK_FORMAT_EAC_R11G11_SNORM_BLOCK,
    eAstc4x4UnormBlock = VK_FORMAT_ASTC_4x4_UNORM_BLOCK,
    eAstc4x4SrgbBlock = VK_FORMAT_ASTC_4x4_SRGB_BLOCK,
    eAstc5x4UnormBlock = VK_FORMAT_ASTC_5x4_UNORM_BLOCK,
    eAstc5x4SrgbBlock = VK_FORMAT_ASTC_5x4_SRGB_BLOCK,
    eAstc5x5UnormBlock = VK_FORMAT_ASTC_5x5_UNORM_BLOCK,
    eAstc5x5SrgbBlock = VK_FORMAT_ASTC_5x5_SRGB_BLOCK,
    eAstc6x5UnormBlock = VK_FORMAT_ASTC_6x5_UNORM_BLOCK,
    eAstc6x5SrgbBlock = VK_FORMAT_ASTC_6x5_SRGB_BLOCK,
    eAstc6x6UnormBlock = VK_FORMAT_ASTC_6x6_UNORM_BLOCK,
    eAstc6x6SrgbBlock = VK_FORMAT_ASTC_6x6_SRGB_BLOCK,
    eAstc8x5UnormBlock = VK_FORMAT_ASTC_8x5_UNORM_BLOCK,
    eAstc8x5SrgbBlock = VK_FORMAT_ASTC_8x5_SRGB_BLOCK,
    eAstc8x6UnormBlock = VK_FORMAT_ASTC_8x6_UNORM_BLOCK,
    eAstc8x6SrgbBlock = VK_FORMAT_ASTC_8x6_SRGB_BLOCK,
    eAstc8x8UnormBlock = VK_FORMAT_ASTC_8x8_UNORM_BLOCK,
    eAstc8x8SrgbBlock = VK_FORMAT_ASTC_8x8_SRGB_BLOCK,
    eAstc10x5UnormBlock = VK_FORMAT_ASTC_10x5_UNORM_BLOCK,
    eAstc10x5SrgbBlock = VK_FORMAT_ASTC_10x5_SRGB_BLOCK,
    eAstc10x6UnormBlock = VK_FORMAT_ASTC_10x6_UNORM_BLOCK,
    eAstc10x6SrgbBlock = VK_FORMAT_ASTC_10x6_SRGB_BLOCK,
    eAstc10x8UnormBlock = VK_FORMAT_ASTC_10x8_UNORM_BLOCK,
    eAstc10x8SrgbBlock = VK_FORMAT_ASTC_10x8_SRGB_BLOCK,
    eAstc10x10UnormBlock = VK_FORMAT_ASTC_10x10_UNORM_BLOCK,
    eAstc10x10SrgbBlock = VK_FORMAT_ASTC_10x10_SRGB_BLOCK,
    eAstc12x10UnormBlock = VK_FORMAT_ASTC_12x10_UNORM_BLOCK,
    eAstc12x10SrgbBlock = VK_FORMAT_ASTC_12x10_SRGB_BLOCK,
    eAstc12x12UnormBlock = VK_FORMAT_ASTC_12x12_UNORM_BLOCK,
    eAstc12x12SrgbBlock = VK_FORMAT_ASTC_12x12_SRGB_BLOCK,
    eG8B8G8R8422Unorm = VK_FORMAT_G8B8G8R8_422_UNORM,
    eB8G8R8G8422Unorm = VK_FORMAT_B8G8R8G8_422_UNORM,
    eG8B8R83Plane420Unorm = VK_FORMAT_G8_B8_R8_3PLANE_420_UNORM,
    eG8B8R82Plane420Unorm = VK_FORMAT_G8_B8R8_2PLANE_420_UNORM,
    eG8B8R83Plane422Unorm = VK_FORMAT_G8_B8_R8_3PLANE_422_UNORM,
    eG8B8R82Plane422Unorm = VK_FORMAT_G8_B8R8_2PLANE_422_UNORM,
    eG8B8R83Plane444Unorm = VK_FORMAT_G8_B8_R8_3PLANE_444_UNORM,
    eR10X6UnormPack16 = VK_FORMAT_R10X6_UNORM_PACK16,
    eR10X6G10X6Unorm2Pack16 = VK_FORMAT_R10X6G10X6_UNORM_2PACK16,
    eR10X6G10X6B10X6A10X6Unorm4Pack16 = VK_FORMAT_R10X6G10X6B10X6A10X6_UNORM_4PACK16,
    eG10X6B10X6G10X6R10X6422Unorm4Pack16 = VK_FORMAT_G10X6B10X6G10X6R10X6_422_UNORM_4PACK16,
    eB10X6G10X6R10X6G10X6422Unorm4Pack16 = VK_FORMAT_B10X6G10X6R10X6G10X6_422_UNORM_4PACK16,
    eG10X6B10X6R10X63Plane420Unorm3Pack16 = VK_FORMAT_G10X6_B10X6_R10X6_3PLANE_420_UNORM_3PACK16,
    eG10X6B10X6R10X62Plane420Unorm3Pack16 = VK_FORMAT_G10X6_B10X6R10X6_2PLANE_420_UNORM_3PACK16,
    eG10X6B10X6R10X63Plane422Unorm3Pack16 = VK_FORMAT_G10X6_B10X6_R10X6_3PLANE_422_UNORM_3PACK16,
    eG10X6B10X6R10X62Plane422Unorm3Pack16 = VK_FORMAT_G10X6_B10X6R10X6_2PLANE_422_UNORM_3PACK16,
    eG10X6B10X6R10X63Plane444Unorm3Pack16 = VK_FORMAT_G10X6_B10X6_R10X6_3PLANE_444_UNORM_3PACK16,
    eR12X4UnormPack16 = VK_FORMAT_R12X4_UNORM_PACK16,
    eR12X4G12X4Unorm2Pack16 = VK_FORMAT_R12X4G12X4_UNORM_2PACK16,
    eR12X4G12X4B12X4A12X4Unorm4Pack16 = VK_FORMAT_R12X4G12X4B12X4A12X4_UNORM_4PACK16,
    eG12X4B12X4G12X4R12X4422Unorm4Pack16 = VK_FORMAT_G12X4B12X4G12X4R12X4_422_UNORM_4PACK16,
    eB12X4G12X4R12X4G12X4422Unorm4Pack16 = VK_FORMAT_B12X4G12X4R12X4G12X4_422_UNORM_4PACK16,
    eG12X4B12X4R12X43Plane420Unorm3Pack16 = VK_FORMAT_G12X4_B12X4_R12X4_3PLANE_420_UNORM_3PACK16,
    eG12X4B12X4R12X42Plane420Unorm3Pack16 = VK_FORMAT_G12X4_B12X4R12X4_2PLANE_420_UNORM_3PACK16,
    eG12X4B12X4R12X43Plane422Unorm3Pack16 = VK_FORMAT_G12X4_B12X4_R12X4_3PLANE_422_UNORM_3PACK16,
    eG12X4B12X4R12X42Plane422Unorm3Pack16 = VK_FORMAT_G12X4_B12X4R12X4_2PLANE_422_UNORM_3PACK16,
    eG12X4B12X4R12X43Plane444Unorm3Pack16 = VK_FORMAT_G12X4_B12X4_R12X4_3PLANE_444_UNORM_3PACK16,
    eG16B16G16R16422Unorm = VK_FORMAT_G16B16G16R16_422_UNORM,
    eB16G16R16G16422Unorm = VK_FORMAT_B16G16R16G16_422_UNORM,
    eG16B16R163Plane420Unorm = VK_FORMAT_G16_B16_R16_3PLANE_420_UNORM,
    eG16B16R162Plane420Unorm = VK_FORMAT_G16_B16R16_2PLANE_420_UNORM,
    eG16B16R163Plane422Unorm = VK_FORMAT_G16_B16_R16_3PLANE_422_UNORM,
    eG16B16R162Plane422Unorm = VK_FORMAT_G16_B16R16_2PLANE_422_UNORM,
    eG16B16R163Plane444Unorm = VK_FORMAT_G16_B16_R16_3PLANE_444_UNORM,
    ePvrtc12BppUnormBlockIMG = VK_FORMAT_PVRTC1_2BPP_UNORM_BLOCK_IMG,
    ePvrtc14BppUnormBlockIMG = VK_FORMAT_PVRTC1_4BPP_UNORM_BLOCK_IMG,
    ePvrtc22BppUnormBlockIMG = VK_FORMAT_PVRTC2_2BPP_UNORM_BLOCK_IMG,
    ePvrtc24BppUnormBlockIMG = VK_FORMAT_PVRTC2_4BPP_UNORM_BLOCK_IMG,
    ePvrtc12BppSrgbBlockIMG = VK_FORMAT_PVRTC1_2BPP_SRGB_BLOCK_IMG,
    ePvrtc14BppSrgbBlockIMG = VK_FORMAT_PVRTC1_4BPP_SRGB_BLOCK_IMG,
    ePvrtc22BppSrgbBlockIMG = VK_FORMAT_PVRTC2_2BPP_SRGB_BLOCK_IMG,
    ePvrtc24BppSrgbBlockIMG = VK_FORMAT_PVRTC2_4BPP_SRGB_BLOCK_IMG,
    eG8B8G8R8422UnormKHR = VK_FORMAT_G8B8G8R8_422_UNORM_KHR,
    eB8G8R8G8422UnormKHR = VK_FORMAT_B8G8R8G8_422_UNORM_KHR,
    eG8B8R83Plane420UnormKHR = VK_FORMAT_G8_B8_R8_3PLANE_420_UNORM_KHR,
    eG8B8R82Plane420UnormKHR = VK_FORMAT_G8_B8R8_2PLANE_420_UNORM_KHR,
    eG8B8R83Plane422UnormKHR = VK_FORMAT_G8_B8_R8_3PLANE_422_UNORM_KHR,
    eG8B8R82Plane422UnormKHR = VK_FORMAT_G8_B8R8_2PLANE_422_UNORM_KHR,
    eG8B8R83Plane444UnormKHR = VK_FORMAT_G8_B8_R8_3PLANE_444_UNORM_KHR,
    eR10X6UnormPack16KHR = VK_FORMAT_R10X6_UNORM_PACK16_KHR,
    eR10X6G10X6Unorm2Pack16KHR = VK_FORMAT_R10X6G10X6_UNORM_2PACK16_KHR,
    eR10X6G10X6B10X6A10X6Unorm4Pack16KHR = VK_FORMAT_R10X6G10X6B10X6A10X6_UNORM_4PACK16_KHR,
    eG10X6B10X6G10X6R10X6422Unorm4Pack16KHR = VK_FORMAT_G10X6B10X6G10X6R10X6_422_UNORM_4PACK16_KHR,
    eB10X6G10X6R10X6G10X6422Unorm4Pack16KHR = VK_FORMAT_B10X6G10X6R10X6G10X6_422_UNORM_4PACK16_KHR,
    eG10X6B10X6R10X63Plane420Unorm3Pack16KHR = VK_FORMAT_G10X6_B10X6_R10X6_3PLANE_420_UNORM_3PACK16_KHR,
    eG10X6B10X6R10X62Plane420Unorm3Pack16KHR = VK_FORMAT_G10X6_B10X6R10X6_2PLANE_420_UNORM_3PACK16_KHR,
    eG10X6B10X6R10X63Plane422Unorm3Pack16KHR = VK_FORMAT_G10X6_B10X6_R10X6_3PLANE_422_UNORM_3PACK16_KHR,
    eG10X6B10X6R10X62Plane422Unorm3Pack16KHR = VK_FORMAT_G10X6_B10X6R10X6_2PLANE_422_UNORM_3PACK16_KHR,
    eG10X6B10X6R10X63Plane444Unorm3Pack16KHR = VK_FORMAT_G10X6_B10X6_R10X6_3PLANE_444_UNORM_3PACK16_KHR,
    eR12X4UnormPack16KHR = VK_FORMAT_R12X4_UNORM_PACK16_KHR,
    eR12X4G12X4Unorm2Pack16KHR = VK_FORMAT_R12X4G12X4_UNORM_2PACK16_KHR,
    eR12X4G12X4B12X4A12X4Unorm4Pack16KHR = VK_FORMAT_R12X4G12X4B12X4A12X4_UNORM_4PACK16_KHR,
    eG12X4B12X4G12X4R12X4422Unorm4Pack16KHR = VK_FORMAT_G12X4B12X4G12X4R12X4_422_UNORM_4PACK16_KHR,
    eB12X4G12X4R12X4G12X4422Unorm4Pack16KHR = VK_FORMAT_B12X4G12X4R12X4G12X4_422_UNORM_4PACK16_KHR,
    eG12X4B12X4R12X43Plane420Unorm3Pack16KHR = VK_FORMAT_G12X4_B12X4_R12X4_3PLANE_420_UNORM_3PACK16_KHR,
    eG12X4B12X4R12X42Plane420Unorm3Pack16KHR = VK_FORMAT_G12X4_B12X4R12X4_2PLANE_420_UNORM_3PACK16_KHR,
    eG12X4B12X4R12X43Plane422Unorm3Pack16KHR = VK_FORMAT_G12X4_B12X4_R12X4_3PLANE_422_UNORM_3PACK16_KHR,
    eG12X4B12X4R12X42Plane422Unorm3Pack16KHR = VK_FORMAT_G12X4_B12X4R12X4_2PLANE_422_UNORM_3PACK16_KHR,
    eG12X4B12X4R12X43Plane444Unorm3Pack16KHR = VK_FORMAT_G12X4_B12X4_R12X4_3PLANE_444_UNORM_3PACK16_KHR,
    eG16B16G16R16422UnormKHR = VK_FORMAT_G16B16G16R16_422_UNORM_KHR,
    eB16G16R16G16422UnormKHR = VK_FORMAT_B16G16R16G16_422_UNORM_KHR,
    eG16B16R163Plane420UnormKHR = VK_FORMAT_G16_B16_R16_3PLANE_420_UNORM_KHR,
    eG16B16R162Plane420UnormKHR = VK_FORMAT_G16_B16R16_2PLANE_420_UNORM_KHR,
    eG16B16R163Plane422UnormKHR = VK_FORMAT_G16_B16_R16_3PLANE_422_UNORM_KHR,
    eG16B16R162Plane422UnormKHR = VK_FORMAT_G16_B16R16_2PLANE_422_UNORM_KHR,
    eG16B16R163Plane444UnormKHR = VK_FORMAT_G16_B16_R16_3PLANE_444_UNORM_KHR
  };

  VULKAN_HPP_INLINE std::string to_string( Format value )
  {
    switch ( value )
    {
      case Format::eUndefined : return "Undefined";
      case Format::eR4G4UnormPack8 : return "R4G4UnormPack8";
      case Format::eR4G4B4A4UnormPack16 : return "R4G4B4A4UnormPack16";
      case Format::eB4G4R4A4UnormPack16 : return "B4G4R4A4UnormPack16";
      case Format::eR5G6B5UnormPack16 : return "R5G6B5UnormPack16";
      case Format::eB5G6R5UnormPack16 : return "B5G6R5UnormPack16";
      case Format::eR5G5B5A1UnormPack16 : return "R5G5B5A1UnormPack16";
      case Format::eB5G5R5A1UnormPack16 : return "B5G5R5A1UnormPack16";
      case Format::eA1R5G5B5UnormPack16 : return "A1R5G5B5UnormPack16";
      case Format::eR8Unorm : return "R8Unorm";
      case Format::eR8Snorm : return "R8Snorm";
      case Format::eR8Uscaled : return "R8Uscaled";
      case Format::eR8Sscaled : return "R8Sscaled";
      case Format::eR8Uint : return "R8Uint";
      case Format::eR8Sint : return "R8Sint";
      case Format::eR8Srgb : return "R8Srgb";
      case Format::eR8G8Unorm : return "R8G8Unorm";
      case Format::eR8G8Snorm : return "R8G8Snorm";
      case Format::eR8G8Uscaled : return "R8G8Uscaled";
      case Format::eR8G8Sscaled : return "R8G8Sscaled";
      case Format::eR8G8Uint : return "R8G8Uint";
      case Format::eR8G8Sint : return "R8G8Sint";
      case Format::eR8G8Srgb : return "R8G8Srgb";
      case Format::eR8G8B8Unorm : return "R8G8B8Unorm";
      case Format::eR8G8B8Snorm : return "R8G8B8Snorm";
      case Format::eR8G8B8Uscaled : return "R8G8B8Uscaled";
      case Format::eR8G8B8Sscaled : return "R8G8B8Sscaled";
      case Format::eR8G8B8Uint : return "R8G8B8Uint";
      case Format::eR8G8B8Sint : return "R8G8B8Sint";
      case Format::eR8G8B8Srgb : return "R8G8B8Srgb";
      case Format::eB8G8R8Unorm : return "B8G8R8Unorm";
      case Format::eB8G8R8Snorm : return "B8G8R8Snorm";
      case Format::eB8G8R8Uscaled : return "B8G8R8Uscaled";
      case Format::eB8G8R8Sscaled : return "B8G8R8Sscaled";
      case Format::eB8G8R8Uint : return "B8G8R8Uint";
      case Format::eB8G8R8Sint : return "B8G8R8Sint";
      case Format::eB8G8R8Srgb : return "B8G8R8Srgb";
      case Format::eR8G8B8A8Unorm : return "R8G8B8A8Unorm";
      case Format::eR8G8B8A8Snorm : return "R8G8B8A8Snorm";
      case Format::eR8G8B8A8Uscaled : return "R8G8B8A8Uscaled";
      case Format::eR8G8B8A8Sscaled : return "R8G8B8A8Sscaled";
      case Format::eR8G8B8A8Uint : return "R8G8B8A8Uint";
      case Format::eR8G8B8A8Sint : return "R8G8B8A8Sint";
      case Format::eR8G8B8A8Srgb : return "R8G8B8A8Srgb";
      case Format::eB8G8R8A8Unorm : return "B8G8R8A8Unorm";
      case Format::eB8G8R8A8Snorm : return "B8G8R8A8Snorm";
      case Format::eB8G8R8A8Uscaled : return "B8G8R8A8Uscaled";
      case Format::eB8G8R8A8Sscaled : return "B8G8R8A8Sscaled";
      case Format::eB8G8R8A8Uint : return "B8G8R8A8Uint";
      case Format::eB8G8R8A8Sint : return "B8G8R8A8Sint";
      case Format::eB8G8R8A8Srgb : return "B8G8R8A8Srgb";
      case Format::eA8B8G8R8UnormPack32 : return "A8B8G8R8UnormPack32";
      case Format::eA8B8G8R8SnormPack32 : return "A8B8G8R8SnormPack32";
      case Format::eA8B8G8R8UscaledPack32 : return "A8B8G8R8UscaledPack32";
      case Format::eA8B8G8R8SscaledPack32 : return "A8B8G8R8SscaledPack32";
      case Format::eA8B8G8R8UintPack32 : return "A8B8G8R8UintPack32";
      case Format::eA8B8G8R8SintPack32 : return "A8B8G8R8SintPack32";
      case Format::eA8B8G8R8SrgbPack32 : return "A8B8G8R8SrgbPack32";
      case Format::eA2R10G10B10UnormPack32 : return "A2R10G10B10UnormPack32";
      case Format::eA2R10G10B10SnormPack32 : return "A2R10G10B10SnormPack32";
      case Format::eA2R10G10B10UscaledPack32 : return "A2R10G10B10UscaledPack32";
      case Format::eA2R10G10B10SscaledPack32 : return "A2R10G10B10SscaledPack32";
      case Format::eA2R10G10B10UintPack32 : return "A2R10G10B10UintPack32";
      case Format::eA2R10G10B10SintPack32 : return "A2R10G10B10SintPack32";
      case Format::eA2B10G10R10UnormPack32 : return "A2B10G10R10UnormPack32";
      case Format::eA2B10G10R10SnormPack32 : return "A2B10G10R10SnormPack32";
      case Format::eA2B10G10R10UscaledPack32 : return "A2B10G10R10UscaledPack32";
      case Format::eA2B10G10R10SscaledPack32 : return "A2B10G10R10SscaledPack32";
      case Format::eA2B10G10R10UintPack32 : return "A2B10G10R10UintPack32";
      case Format::eA2B10G10R10SintPack32 : return "A2B10G10R10SintPack32";
      case Format::eR16Unorm : return "R16Unorm";
      case Format::eR16Snorm : return "R16Snorm";
      case Format::eR16Uscaled : return "R16Uscaled";
      case Format::eR16Sscaled : return "R16Sscaled";
      case Format::eR16Uint : return "R16Uint";
      case Format::eR16Sint : return "R16Sint";
      case Format::eR16Sfloat : return "R16Sfloat";
      case Format::eR16G16Unorm : return "R16G16Unorm";
      case Format::eR16G16Snorm : return "R16G16Snorm";
      case Format::eR16G16Uscaled : return "R16G16Uscaled";
      case Format::eR16G16Sscaled : return "R16G16Sscaled";
      case Format::eR16G16Uint : return "R16G16Uint";
      case Format::eR16G16Sint : return "R16G16Sint";
      case Format::eR16G16Sfloat : return "R16G16Sfloat";
      case Format::eR16G16B16Unorm : return "R16G16B16Unorm";
      case Format::eR16G16B16Snorm : return "R16G16B16Snorm";
      case Format::eR16G16B16Uscaled : return "R16G16B16Uscaled";
      case Format::eR16G16B16Sscaled : return "R16G16B16Sscaled";
      case Format::eR16G16B16Uint : return "R16G16B16Uint";
      case Format::eR16G16B16Sint : return "R16G16B16Sint";
      case Format::eR16G16B16Sfloat : return "R16G16B16Sfloat";
      case Format::eR16G16B16A16Unorm : return "R16G16B16A16Unorm";
      case Format::eR16G16B16A16Snorm : return "R16G16B16A16Snorm";
      case Format::eR16G16B16A16Uscaled : return "R16G16B16A16Uscaled";
      case Format::eR16G16B16A16Sscaled : return "R16G16B16A16Sscaled";
      case Format::eR16G16B16A16Uint : return "R16G16B16A16Uint";
      case Format::eR16G16B16A16Sint : return "R16G16B16A16Sint";
      case Format::eR16G16B16A16Sfloat : return "R16G16B16A16Sfloat";
      case Format::eR32Uint : return "R32Uint";
      case Format::eR32Sint : return "R32Sint";
      case Format::eR32Sfloat : return "R32Sfloat";
      case Format::eR32G32Uint : return "R32G32Uint";
      case Format::eR32G32Sint : return "R32G32Sint";
      case Format::eR32G32Sfloat : return "R32G32Sfloat";
      case Format::eR32G32B32Uint : return "R32G32B32Uint";
      case Format::eR32G32B32Sint : return "R32G32B32Sint";
      case Format::eR32G32B32Sfloat : return "R32G32B32Sfloat";
      case Format::eR32G32B32A32Uint : return "R32G32B32A32Uint";
      case Format::eR32G32B32A32Sint : return "R32G32B32A32Sint";
      case Format::eR32G32B32A32Sfloat : return "R32G32B32A32Sfloat";
      case Format::eR64Uint : return "R64Uint";
      case Format::eR64Sint : return "R64Sint";
      case Format::eR64Sfloat : return "R64Sfloat";
      case Format::eR64G64Uint : return "R64G64Uint";
      case Format::eR64G64Sint : return "R64G64Sint";
      case Format::eR64G64Sfloat : return "R64G64Sfloat";
      case Format::eR64G64B64Uint : return "R64G64B64Uint";
      case Format::eR64G64B64Sint : return "R64G64B64Sint";
      case Format::eR64G64B64Sfloat : return "R64G64B64Sfloat";
      case Format::eR64G64B64A64Uint : return "R64G64B64A64Uint";
      case Format::eR64G64B64A64Sint : return "R64G64B64A64Sint";
      case Format::eR64G64B64A64Sfloat : return "R64G64B64A64Sfloat";
      case Format::eB10G11R11UfloatPack32 : return "B10G11R11UfloatPack32";
      case Format::eE5B9G9R9UfloatPack32 : return "E5B9G9R9UfloatPack32";
      case Format::eD16Unorm : return "D16Unorm";
      case Format::eX8D24UnormPack32 : return "X8D24UnormPack32";
      case Format::eD32Sfloat : return "D32Sfloat";
      case Format::eS8Uint : return "S8Uint";
      case Format::eD16UnormS8Uint : return "D16UnormS8Uint";
      case Format::eD24UnormS8Uint : return "D24UnormS8Uint";
      case Format::eD32SfloatS8Uint : return "D32SfloatS8Uint";
      case Format::eBc1RgbUnormBlock : return "Bc1RgbUnormBlock";
      case Format::eBc1RgbSrgbBlock : return "Bc1RgbSrgbBlock";
      case Format::eBc1RgbaUnormBlock : return "Bc1RgbaUnormBlock";
      case Format::eBc1RgbaSrgbBlock : return "Bc1RgbaSrgbBlock";
      case Format::eBc2UnormBlock : return "Bc2UnormBlock";
      case Format::eBc2SrgbBlock : return "Bc2SrgbBlock";
      case Format::eBc3UnormBlock : return "Bc3UnormBlock";
      case Format::eBc3SrgbBlock : return "Bc3SrgbBlock";
      case Format::eBc4UnormBlock : return "Bc4UnormBlock";
      case Format::eBc4SnormBlock : return "Bc4SnormBlock";
      case Format::eBc5UnormBlock : return "Bc5UnormBlock";
      case Format::eBc5SnormBlock : return "Bc5SnormBlock";
      case Format::eBc6HUfloatBlock : return "Bc6HUfloatBlock";
      case Format::eBc6HSfloatBlock : return "Bc6HSfloatBlock";
      case Format::eBc7UnormBlock : return "Bc7UnormBlock";
      case Format::eBc7SrgbBlock : return "Bc7SrgbBlock";
      case Format::eEtc2R8G8B8UnormBlock : return "Etc2R8G8B8UnormBlock";
      case Format::eEtc2R8G8B8SrgbBlock : return "Etc2R8G8B8SrgbBlock";
      case Format::eEtc2R8G8B8A1UnormBlock : return "Etc2R8G8B8A1UnormBlock";
      case Format::eEtc2R8G8B8A1SrgbBlock : return "Etc2R8G8B8A1SrgbBlock";
      case Format::eEtc2R8G8B8A8UnormBlock : return "Etc2R8G8B8A8UnormBlock";
      case Format::eEtc2R8G8B8A8SrgbBlock : return "Etc2R8G8B8A8SrgbBlock";
      case Format::eEacR11UnormBlock : return "EacR11UnormBlock";
      case Format::eEacR11SnormBlock : return "EacR11SnormBlock";
      case Format::eEacR11G11UnormBlock : return "EacR11G11UnormBlock";
      case Format::eEacR11G11SnormBlock : return "EacR11G11SnormBlock";
      case Format::eAstc4x4UnormBlock : return "Astc4x4UnormBlock";
      case Format::eAstc4x4SrgbBlock : return "Astc4x4SrgbBlock";
      case Format::eAstc5x4UnormBlock : return "Astc5x4UnormBlock";
      case Format::eAstc5x4SrgbBlock : return "Astc5x4SrgbBlock";
      case Format::eAstc5x5UnormBlock : return "Astc5x5UnormBlock";
      case Format::eAstc5x5SrgbBlock : return "Astc5x5SrgbBlock";
      case Format::eAstc6x5UnormBlock : return "Astc6x5UnormBlock";
      case Format::eAstc6x5SrgbBlock : return "Astc6x5SrgbBlock";
      case Format::eAstc6x6UnormBlock : return "Astc6x6UnormBlock";
      case Format::eAstc6x6SrgbBlock : return "Astc6x6SrgbBlock";
      case Format::eAstc8x5UnormBlock : return "Astc8x5UnormBlock";
      case Format::eAstc8x5SrgbBlock : return "Astc8x5SrgbBlock";
      case Format::eAstc8x6UnormBlock : return "Astc8x6UnormBlock";
      case Format::eAstc8x6SrgbBlock : return "Astc8x6SrgbBlock";
      case Format::eAstc8x8UnormBlock : return "Astc8x8UnormBlock";
      case Format::eAstc8x8SrgbBlock : return "Astc8x8SrgbBlock";
      case Format::eAstc10x5UnormBlock : return "Astc10x5UnormBlock";
      case Format::eAstc10x5SrgbBlock : return "Astc10x5SrgbBlock";
      case Format::eAstc10x6UnormBlock : return "Astc10x6UnormBlock";
      case Format::eAstc10x6SrgbBlock : return "Astc10x6SrgbBlock";
      case Format::eAstc10x8UnormBlock : return "Astc10x8UnormBlock";
      case Format::eAstc10x8SrgbBlock : return "Astc10x8SrgbBlock";
      case Format::eAstc10x10UnormBlock : return "Astc10x10UnormBlock";
      case Format::eAstc10x10SrgbBlock : return "Astc10x10SrgbBlock";
      case Format::eAstc12x10UnormBlock : return "Astc12x10UnormBlock";
      case Format::eAstc12x10SrgbBlock : return "Astc12x10SrgbBlock";
      case Format::eAstc12x12UnormBlock : return "Astc12x12UnormBlock";
      case Format::eAstc12x12SrgbBlock : return "Astc12x12SrgbBlock";
      case Format::eG8B8G8R8422Unorm : return "G8B8G8R8422Unorm";
      case Format::eB8G8R8G8422Unorm : return "B8G8R8G8422Unorm";
      case Format::eG8B8R83Plane420Unorm : return "G8B8R83Plane420Unorm";
      case Format::eG8B8R82Plane420Unorm : return "G8B8R82Plane420Unorm";
      case Format::eG8B8R83Plane422Unorm : return "G8B8R83Plane422Unorm";
      case Format::eG8B8R82Plane422Unorm : return "G8B8R82Plane422Unorm";
      case Format::eG8B8R83Plane444Unorm : return "G8B8R83Plane444Unorm";
      case Format::eR10X6UnormPack16 : return "R10X6UnormPack16";
      case Format::eR10X6G10X6Unorm2Pack16 : return "R10X6G10X6Unorm2Pack16";
      case Format::eR10X6G10X6B10X6A10X6Unorm4Pack16 : return "R10X6G10X6B10X6A10X6Unorm4Pack16";
      case Format::eG10X6B10X6G10X6R10X6422Unorm4Pack16 : return "G10X6B10X6G10X6R10X6422Unorm4Pack16";
      case Format::eB10X6G10X6R10X6G10X6422Unorm4Pack16 : return "B10X6G10X6R10X6G10X6422Unorm4Pack16";
      case Format::eG10X6B10X6R10X63Plane420Unorm3Pack16 : return "G10X6B10X6R10X63Plane420Unorm3Pack16";
      case Format::eG10X6B10X6R10X62Plane420Unorm3Pack16 : return "G10X6B10X6R10X62Plane420Unorm3Pack16";
      case Format::eG10X6B10X6R10X63Plane422Unorm3Pack16 : return "G10X6B10X6R10X63Plane422Unorm3Pack16";
      case Format::eG10X6B10X6R10X62Plane422Unorm3Pack16 : return "G10X6B10X6R10X62Plane422Unorm3Pack16";
      case Format::eG10X6B10X6R10X63Plane444Unorm3Pack16 : return "G10X6B10X6R10X63Plane444Unorm3Pack16";
      case Format::eR12X4UnormPack16 : return "R12X4UnormPack16";
      case Format::eR12X4G12X4Unorm2Pack16 : return "R12X4G12X4Unorm2Pack16";
      case Format::eR12X4G12X4B12X4A12X4Unorm4Pack16 : return "R12X4G12X4B12X4A12X4Unorm4Pack16";
      case Format::eG12X4B12X4G12X4R12X4422Unorm4Pack16 : return "G12X4B12X4G12X4R12X4422Unorm4Pack16";
      case Format::eB12X4G12X4R12X4G12X4422Unorm4Pack16 : return "B12X4G12X4R12X4G12X4422Unorm4Pack16";
      case Format::eG12X4B12X4R12X43Plane420Unorm3Pack16 : return "G12X4B12X4R12X43Plane420Unorm3Pack16";
      case Format::eG12X4B12X4R12X42Plane420Unorm3Pack16 : return "G12X4B12X4R12X42Plane420Unorm3Pack16";
      case Format::eG12X4B12X4R12X43Plane422Unorm3Pack16 : return "G12X4B12X4R12X43Plane422Unorm3Pack16";
      case Format::eG12X4B12X4R12X42Plane422Unorm3Pack16 : return "G12X4B12X4R12X42Plane422Unorm3Pack16";
      case Format::eG12X4B12X4R12X43Plane444Unorm3Pack16 : return "G12X4B12X4R12X43Plane444Unorm3Pack16";
      case Format::eG16B16G16R16422Unorm : return "G16B16G16R16422Unorm";
      case Format::eB16G16R16G16422Unorm : return "B16G16R16G16422Unorm";
      case Format::eG16B16R163Plane420Unorm : return "G16B16R163Plane420Unorm";
      case Format::eG16B16R162Plane420Unorm : return "G16B16R162Plane420Unorm";
      case Format::eG16B16R163Plane422Unorm : return "G16B16R163Plane422Unorm";
      case Format::eG16B16R162Plane422Unorm : return "G16B16R162Plane422Unorm";
      case Format::eG16B16R163Plane444Unorm : return "G16B16R163Plane444Unorm";
      case Format::ePvrtc12BppUnormBlockIMG : return "Pvrtc12BppUnormBlockIMG";
      case Format::ePvrtc14BppUnormBlockIMG : return "Pvrtc14BppUnormBlockIMG";
      case Format::ePvrtc22BppUnormBlockIMG : return "Pvrtc22BppUnormBlockIMG";
      case Format::ePvrtc24BppUnormBlockIMG : return "Pvrtc24BppUnormBlockIMG";
      case Format::ePvrtc12BppSrgbBlockIMG : return "Pvrtc12BppSrgbBlockIMG";
      case Format::ePvrtc14BppSrgbBlockIMG : return "Pvrtc14BppSrgbBlockIMG";
      case Format::ePvrtc22BppSrgbBlockIMG : return "Pvrtc22BppSrgbBlockIMG";
      case Format::ePvrtc24BppSrgbBlockIMG : return "Pvrtc24BppSrgbBlockIMG";
      default: return "invalid";
    }
  }

  enum class FrontFace
  {
    eCounterClockwise = VK_FRONT_FACE_COUNTER_CLOCKWISE,
    eClockwise = VK_FRONT_FACE_CLOCKWISE
  };

  VULKAN_HPP_INLINE std::string to_string( FrontFace value )
  {
    switch ( value )
    {
      case FrontFace::eCounterClockwise : return "CounterClockwise";
      case FrontFace::eClockwise : return "Clockwise";
      default: return "invalid";
    }
  }

  enum class GeometryTypeNV
  {
    eTriangles = VK_GEOMETRY_TYPE_TRIANGLES_NV,
    eAabbs = VK_GEOMETRY_TYPE_AABBS_NV
  };

  VULKAN_HPP_INLINE std::string to_string( GeometryTypeNV value )
  {
    switch ( value )
    {
      case GeometryTypeNV::eTriangles : return "Triangles";
      case GeometryTypeNV::eAabbs : return "Aabbs";
      default: return "invalid";
    }
  }

  enum class ImageLayout
  {
    eUndefined = VK_IMAGE_LAYOUT_UNDEFINED,
    eGeneral = VK_IMAGE_LAYOUT_GENERAL,
    eColorAttachmentOptimal = VK_IMAGE_LAYOUT_COLOR_ATTACHMENT_OPTIMAL,
    eDepthStencilAttachmentOptimal = VK_IMAGE_LAYOUT_DEPTH_STENCIL_ATTACHMENT_OPTIMAL,
    eDepthStencilReadOnlyOptimal = VK_IMAGE_LAYOUT_DEPTH_STENCIL_READ_ONLY_OPTIMAL,
    eShaderReadOnlyOptimal = VK_IMAGE_LAYOUT_SHADER_READ_ONLY_OPTIMAL,
    eTransferSrcOptimal = VK_IMAGE_LAYOUT_TRANSFER_SRC_OPTIMAL,
    eTransferDstOptimal = VK_IMAGE_LAYOUT_TRANSFER_DST_OPTIMAL,
    ePreinitialized = VK_IMAGE_LAYOUT_PREINITIALIZED,
    eDepthReadOnlyStencilAttachmentOptimal = VK_IMAGE_LAYOUT_DEPTH_READ_ONLY_STENCIL_ATTACHMENT_OPTIMAL,
    eDepthAttachmentStencilReadOnlyOptimal = VK_IMAGE_LAYOUT_DEPTH_ATTACHMENT_STENCIL_READ_ONLY_OPTIMAL,
    ePresentSrcKHR = VK_IMAGE_LAYOUT_PRESENT_SRC_KHR,
    eSharedPresentKHR = VK_IMAGE_LAYOUT_SHARED_PRESENT_KHR,
    eShadingRateOptimalNV = VK_IMAGE_LAYOUT_SHADING_RATE_OPTIMAL_NV,
    eFragmentDensityMapOptimalEXT = VK_IMAGE_LAYOUT_FRAGMENT_DENSITY_MAP_OPTIMAL_EXT,
    eDepthReadOnlyStencilAttachmentOptimalKHR = VK_IMAGE_LAYOUT_DEPTH_READ_ONLY_STENCIL_ATTACHMENT_OPTIMAL_KHR,
    eDepthAttachmentStencilReadOnlyOptimalKHR = VK_IMAGE_LAYOUT_DEPTH_ATTACHMENT_STENCIL_READ_ONLY_OPTIMAL_KHR
  };

  VULKAN_HPP_INLINE std::string to_string( ImageLayout value )
  {
    switch ( value )
    {
      case ImageLayout::eUndefined : return "Undefined";
      case ImageLayout::eGeneral : return "General";
      case ImageLayout::eColorAttachmentOptimal : return "ColorAttachmentOptimal";
      case ImageLayout::eDepthStencilAttachmentOptimal : return "DepthStencilAttachmentOptimal";
      case ImageLayout::eDepthStencilReadOnlyOptimal : return "DepthStencilReadOnlyOptimal";
      case ImageLayout::eShaderReadOnlyOptimal : return "ShaderReadOnlyOptimal";
      case ImageLayout::eTransferSrcOptimal : return "TransferSrcOptimal";
      case ImageLayout::eTransferDstOptimal : return "TransferDstOptimal";
      case ImageLayout::ePreinitialized : return "Preinitialized";
      case ImageLayout::eDepthReadOnlyStencilAttachmentOptimal : return "DepthReadOnlyStencilAttachmentOptimal";
      case ImageLayout::eDepthAttachmentStencilReadOnlyOptimal : return "DepthAttachmentStencilReadOnlyOptimal";
      case ImageLayout::ePresentSrcKHR : return "PresentSrcKHR";
      case ImageLayout::eSharedPresentKHR : return "SharedPresentKHR";
      case ImageLayout::eShadingRateOptimalNV : return "ShadingRateOptimalNV";
      case ImageLayout::eFragmentDensityMapOptimalEXT : return "FragmentDensityMapOptimalEXT";
      default: return "invalid";
    }
  }

  enum class ImageTiling
  {
    eOptimal = VK_IMAGE_TILING_OPTIMAL,
    eLinear = VK_IMAGE_TILING_LINEAR,
    eDrmFormatModifierEXT = VK_IMAGE_TILING_DRM_FORMAT_MODIFIER_EXT
  };

  VULKAN_HPP_INLINE std::string to_string( ImageTiling value )
  {
    switch ( value )
    {
      case ImageTiling::eOptimal : return "Optimal";
      case ImageTiling::eLinear : return "Linear";
      case ImageTiling::eDrmFormatModifierEXT : return "DrmFormatModifierEXT";
      default: return "invalid";
    }
  }

  enum class ImageType
  {
    e1D = VK_IMAGE_TYPE_1D,
    e2D = VK_IMAGE_TYPE_2D,
    e3D = VK_IMAGE_TYPE_3D
  };

  VULKAN_HPP_INLINE std::string to_string( ImageType value )
  {
    switch ( value )
    {
      case ImageType::e1D : return "1D";
      case ImageType::e2D : return "2D";
      case ImageType::e3D : return "3D";
      default: return "invalid";
    }
  }

  enum class ImageViewType
  {
    e1D = VK_IMAGE_VIEW_TYPE_1D,
    e2D = VK_IMAGE_VIEW_TYPE_2D,
    e3D = VK_IMAGE_VIEW_TYPE_3D,
    eCube = VK_IMAGE_VIEW_TYPE_CUBE,
    e1DArray = VK_IMAGE_VIEW_TYPE_1D_ARRAY,
    e2DArray = VK_IMAGE_VIEW_TYPE_2D_ARRAY,
    eCubeArray = VK_IMAGE_VIEW_TYPE_CUBE_ARRAY
  };

  VULKAN_HPP_INLINE std::string to_string( ImageViewType value )
  {
    switch ( value )
    {
      case ImageViewType::e1D : return "1D";
      case ImageViewType::e2D : return "2D";
      case ImageViewType::e3D : return "3D";
      case ImageViewType::eCube : return "Cube";
      case ImageViewType::e1DArray : return "1DArray";
      case ImageViewType::e2DArray : return "2DArray";
      case ImageViewType::eCubeArray : return "CubeArray";
      default: return "invalid";
    }
  }

  enum class IndexType
  {
    eUint16 = VK_INDEX_TYPE_UINT16,
    eUint32 = VK_INDEX_TYPE_UINT32,
    eNoneNV = VK_INDEX_TYPE_NONE_NV
  };

  VULKAN_HPP_INLINE std::string to_string( IndexType value )
  {
    switch ( value )
    {
      case IndexType::eUint16 : return "Uint16";
      case IndexType::eUint32 : return "Uint32";
      case IndexType::eNoneNV : return "NoneNV";
      default: return "invalid";
    }
  }

  enum class IndirectCommandsTokenTypeNVX
  {
    ePipeline = VK_INDIRECT_COMMANDS_TOKEN_TYPE_PIPELINE_NVX,
    eDescriptorSet = VK_INDIRECT_COMMANDS_TOKEN_TYPE_DESCRIPTOR_SET_NVX,
    eIndexBuffer = VK_INDIRECT_COMMANDS_TOKEN_TYPE_INDEX_BUFFER_NVX,
    eVertexBuffer = VK_INDIRECT_COMMANDS_TOKEN_TYPE_VERTEX_BUFFER_NVX,
    ePushConstant = VK_INDIRECT_COMMANDS_TOKEN_TYPE_PUSH_CONSTANT_NVX,
    eDrawIndexed = VK_INDIRECT_COMMANDS_TOKEN_TYPE_DRAW_INDEXED_NVX,
    eDraw = VK_INDIRECT_COMMANDS_TOKEN_TYPE_DRAW_NVX,
    eDispatch = VK_INDIRECT_COMMANDS_TOKEN_TYPE_DISPATCH_NVX
  };

  VULKAN_HPP_INLINE std::string to_string( IndirectCommandsTokenTypeNVX value )
  {
    switch ( value )
    {
      case IndirectCommandsTokenTypeNVX::ePipeline : return "Pipeline";
      case IndirectCommandsTokenTypeNVX::eDescriptorSet : return "DescriptorSet";
      case IndirectCommandsTokenTypeNVX::eIndexBuffer : return "IndexBuffer";
      case IndirectCommandsTokenTypeNVX::eVertexBuffer : return "VertexBuffer";
      case IndirectCommandsTokenTypeNVX::ePushConstant : return "PushConstant";
      case IndirectCommandsTokenTypeNVX::eDrawIndexed : return "DrawIndexed";
      case IndirectCommandsTokenTypeNVX::eDraw : return "Draw";
      case IndirectCommandsTokenTypeNVX::eDispatch : return "Dispatch";
      default: return "invalid";
    }
  }

  enum class InternalAllocationType
  {
    eExecutable = VK_INTERNAL_ALLOCATION_TYPE_EXECUTABLE
  };

  VULKAN_HPP_INLINE std::string to_string( InternalAllocationType value )
  {
    switch ( value )
    {
      case InternalAllocationType::eExecutable : return "Executable";
      default: return "invalid";
    }
  }

  enum class LogicOp
  {
    eClear = VK_LOGIC_OP_CLEAR,
    eAnd = VK_LOGIC_OP_AND,
    eAndReverse = VK_LOGIC_OP_AND_REVERSE,
    eCopy = VK_LOGIC_OP_COPY,
    eAndInverted = VK_LOGIC_OP_AND_INVERTED,
    eNoOp = VK_LOGIC_OP_NO_OP,
    eXor = VK_LOGIC_OP_XOR,
    eOr = VK_LOGIC_OP_OR,
    eNor = VK_LOGIC_OP_NOR,
    eEquivalent = VK_LOGIC_OP_EQUIVALENT,
    eInvert = VK_LOGIC_OP_INVERT,
    eOrReverse = VK_LOGIC_OP_OR_REVERSE,
    eCopyInverted = VK_LOGIC_OP_COPY_INVERTED,
    eOrInverted = VK_LOGIC_OP_OR_INVERTED,
    eNand = VK_LOGIC_OP_NAND,
    eSet = VK_LOGIC_OP_SET
  };

  VULKAN_HPP_INLINE std::string to_string( LogicOp value )
  {
    switch ( value )
    {
      case LogicOp::eClear : return "Clear";
      case LogicOp::eAnd : return "And";
      case LogicOp::eAndReverse : return "AndReverse";
      case LogicOp::eCopy : return "Copy";
      case LogicOp::eAndInverted : return "AndInverted";
      case LogicOp::eNoOp : return "NoOp";
      case LogicOp::eXor : return "Xor";
      case LogicOp::eOr : return "Or";
      case LogicOp::eNor : return "Nor";
      case LogicOp::eEquivalent : return "Equivalent";
      case LogicOp::eInvert : return "Invert";
      case LogicOp::eOrReverse : return "OrReverse";
      case LogicOp::eCopyInverted : return "CopyInverted";
      case LogicOp::eOrInverted : return "OrInverted";
      case LogicOp::eNand : return "Nand";
      case LogicOp::eSet : return "Set";
      default: return "invalid";
    }
  }

  enum class MemoryOverallocationBehaviorAMD
  {
    eDefault = VK_MEMORY_OVERALLOCATION_BEHAVIOR_DEFAULT_AMD,
    eAllowed = VK_MEMORY_OVERALLOCATION_BEHAVIOR_ALLOWED_AMD,
    eDisallowed = VK_MEMORY_OVERALLOCATION_BEHAVIOR_DISALLOWED_AMD
  };

  VULKAN_HPP_INLINE std::string to_string( MemoryOverallocationBehaviorAMD value )
  {
    switch ( value )
    {
      case MemoryOverallocationBehaviorAMD::eDefault : return "Default";
      case MemoryOverallocationBehaviorAMD::eAllowed : return "Allowed";
      case MemoryOverallocationBehaviorAMD::eDisallowed : return "Disallowed";
      default: return "invalid";
    }
  }

  enum class ObjectEntryTypeNVX
  {
    eDescriptorSet = VK_OBJECT_ENTRY_TYPE_DESCRIPTOR_SET_NVX,
    ePipeline = VK_OBJECT_ENTRY_TYPE_PIPELINE_NVX,
    eIndexBuffer = VK_OBJECT_ENTRY_TYPE_INDEX_BUFFER_NVX,
    eVertexBuffer = VK_OBJECT_ENTRY_TYPE_VERTEX_BUFFER_NVX,
    ePushConstant = VK_OBJECT_ENTRY_TYPE_PUSH_CONSTANT_NVX
  };

  VULKAN_HPP_INLINE std::string to_string( ObjectEntryTypeNVX value )
  {
    switch ( value )
    {
      case ObjectEntryTypeNVX::eDescriptorSet : return "DescriptorSet";
      case ObjectEntryTypeNVX::ePipeline : return "Pipeline";
      case ObjectEntryTypeNVX::eIndexBuffer : return "IndexBuffer";
      case ObjectEntryTypeNVX::eVertexBuffer : return "VertexBuffer";
      case ObjectEntryTypeNVX::ePushConstant : return "PushConstant";
      default: return "invalid";
    }
  }

  enum class ObjectType
  {
    eUnknown = VK_OBJECT_TYPE_UNKNOWN,
    eInstance = VK_OBJECT_TYPE_INSTANCE,
    ePhysicalDevice = VK_OBJECT_TYPE_PHYSICAL_DEVICE,
    eDevice = VK_OBJECT_TYPE_DEVICE,
    eQueue = VK_OBJECT_TYPE_QUEUE,
    eSemaphore = VK_OBJECT_TYPE_SEMAPHORE,
    eCommandBuffer = VK_OBJECT_TYPE_COMMAND_BUFFER,
    eFence = VK_OBJECT_TYPE_FENCE,
    eDeviceMemory = VK_OBJECT_TYPE_DEVICE_MEMORY,
    eBuffer = VK_OBJECT_TYPE_BUFFER,
    eImage = VK_OBJECT_TYPE_IMAGE,
    eEvent = VK_OBJECT_TYPE_EVENT,
    eQueryPool = VK_OBJECT_TYPE_QUERY_POOL,
    eBufferView = VK_OBJECT_TYPE_BUFFER_VIEW,
    eImageView = VK_OBJECT_TYPE_IMAGE_VIEW,
    eShaderModule = VK_OBJECT_TYPE_SHADER_MODULE,
    ePipelineCache = VK_OBJECT_TYPE_PIPELINE_CACHE,
    ePipelineLayout = VK_OBJECT_TYPE_PIPELINE_LAYOUT,
    eRenderPass = VK_OBJECT_TYPE_RENDER_PASS,
    ePipeline = VK_OBJECT_TYPE_PIPELINE,
    eDescriptorSetLayout = VK_OBJECT_TYPE_DESCRIPTOR_SET_LAYOUT,
    eSampler = VK_OBJECT_TYPE_SAMPLER,
    eDescriptorPool = VK_OBJECT_TYPE_DESCRIPTOR_POOL,
    eDescriptorSet = VK_OBJECT_TYPE_DESCRIPTOR_SET,
    eFramebuffer = VK_OBJECT_TYPE_FRAMEBUFFER,
    eCommandPool = VK_OBJECT_TYPE_COMMAND_POOL,
    eSamplerYcbcrConversion = VK_OBJECT_TYPE_SAMPLER_YCBCR_CONVERSION,
    eDescriptorUpdateTemplate = VK_OBJECT_TYPE_DESCRIPTOR_UPDATE_TEMPLATE,
    eSurfaceKHR = VK_OBJECT_TYPE_SURFACE_KHR,
    eSwapchainKHR = VK_OBJECT_TYPE_SWAPCHAIN_KHR,
    eDisplayKHR = VK_OBJECT_TYPE_DISPLAY_KHR,
    eDisplayModeKHR = VK_OBJECT_TYPE_DISPLAY_MODE_KHR,
    eDebugReportCallbackEXT = VK_OBJECT_TYPE_DEBUG_REPORT_CALLBACK_EXT,
    eObjectTableNVX = VK_OBJECT_TYPE_OBJECT_TABLE_NVX,
    eIndirectCommandsLayoutNVX = VK_OBJECT_TYPE_INDIRECT_COMMANDS_LAYOUT_NVX,
    eDebugUtilsMessengerEXT = VK_OBJECT_TYPE_DEBUG_UTILS_MESSENGER_EXT,
    eValidationCacheEXT = VK_OBJECT_TYPE_VALIDATION_CACHE_EXT,
    eAccelerationStructureNV = VK_OBJECT_TYPE_ACCELERATION_STRUCTURE_NV,
    eDescriptorUpdateTemplateKHR = VK_OBJECT_TYPE_DESCRIPTOR_UPDATE_TEMPLATE_KHR,
    eSamplerYcbcrConversionKHR = VK_OBJECT_TYPE_SAMPLER_YCBCR_CONVERSION_KHR
  };

  VULKAN_HPP_INLINE std::string to_string( ObjectType value )
  {
    switch ( value )
    {
      case ObjectType::eUnknown : return "Unknown";
      case ObjectType::eInstance : return "Instance";
      case ObjectType::ePhysicalDevice : return "PhysicalDevice";
      case ObjectType::eDevice : return "Device";
      case ObjectType::eQueue : return "Queue";
      case ObjectType::eSemaphore : return "Semaphore";
      case ObjectType::eCommandBuffer : return "CommandBuffer";
      case ObjectType::eFence : return "Fence";
      case ObjectType::eDeviceMemory : return "DeviceMemory";
      case ObjectType::eBuffer : return "Buffer";
      case ObjectType::eImage : return "Image";
      case ObjectType::eEvent : return "Event";
      case ObjectType::eQueryPool : return "QueryPool";
      case ObjectType::eBufferView : return "BufferView";
      case ObjectType::eImageView : return "ImageView";
      case ObjectType::eShaderModule : return "ShaderModule";
      case ObjectType::ePipelineCache : return "PipelineCache";
      case ObjectType::ePipelineLayout : return "PipelineLayout";
      case ObjectType::eRenderPass : return "RenderPass";
      case ObjectType::ePipeline : return "Pipeline";
      case ObjectType::eDescriptorSetLayout : return "DescriptorSetLayout";
      case ObjectType::eSampler : return "Sampler";
      case ObjectType::eDescriptorPool : return "DescriptorPool";
      case ObjectType::eDescriptorSet : return "DescriptorSet";
      case ObjectType::eFramebuffer : return "Framebuffer";
      case ObjectType::eCommandPool : return "CommandPool";
      case ObjectType::eSamplerYcbcrConversion : return "SamplerYcbcrConversion";
      case ObjectType::eDescriptorUpdateTemplate : return "DescriptorUpdateTemplate";
      case ObjectType::eSurfaceKHR : return "SurfaceKHR";
      case ObjectType::eSwapchainKHR : return "SwapchainKHR";
      case ObjectType::eDisplayKHR : return "DisplayKHR";
      case ObjectType::eDisplayModeKHR : return "DisplayModeKHR";
      case ObjectType::eDebugReportCallbackEXT : return "DebugReportCallbackEXT";
      case ObjectType::eObjectTableNVX : return "ObjectTableNVX";
      case ObjectType::eIndirectCommandsLayoutNVX : return "IndirectCommandsLayoutNVX";
      case ObjectType::eDebugUtilsMessengerEXT : return "DebugUtilsMessengerEXT";
      case ObjectType::eValidationCacheEXT : return "ValidationCacheEXT";
      case ObjectType::eAccelerationStructureNV : return "AccelerationStructureNV";
      default: return "invalid";
    }
  }

  enum class PhysicalDeviceType
  {
    eOther = VK_PHYSICAL_DEVICE_TYPE_OTHER,
    eIntegratedGpu = VK_PHYSICAL_DEVICE_TYPE_INTEGRATED_GPU,
    eDiscreteGpu = VK_PHYSICAL_DEVICE_TYPE_DISCRETE_GPU,
    eVirtualGpu = VK_PHYSICAL_DEVICE_TYPE_VIRTUAL_GPU,
    eCpu = VK_PHYSICAL_DEVICE_TYPE_CPU
  };

  VULKAN_HPP_INLINE std::string to_string( PhysicalDeviceType value )
  {
    switch ( value )
    {
      case PhysicalDeviceType::eOther : return "Other";
      case PhysicalDeviceType::eIntegratedGpu : return "IntegratedGpu";
      case PhysicalDeviceType::eDiscreteGpu : return "DiscreteGpu";
      case PhysicalDeviceType::eVirtualGpu : return "VirtualGpu";
      case PhysicalDeviceType::eCpu : return "Cpu";
      default: return "invalid";
    }
  }

  enum class PipelineBindPoint
  {
    eGraphics = VK_PIPELINE_BIND_POINT_GRAPHICS,
    eCompute = VK_PIPELINE_BIND_POINT_COMPUTE,
    eRayTracingNV = VK_PIPELINE_BIND_POINT_RAY_TRACING_NV
  };

  VULKAN_HPP_INLINE std::string to_string( PipelineBindPoint value )
  {
    switch ( value )
    {
      case PipelineBindPoint::eGraphics : return "Graphics";
      case PipelineBindPoint::eCompute : return "Compute";
      case PipelineBindPoint::eRayTracingNV : return "RayTracingNV";
      default: return "invalid";
    }
  }

  enum class PipelineCacheHeaderVersion
  {
    eOne = VK_PIPELINE_CACHE_HEADER_VERSION_ONE
  };

  VULKAN_HPP_INLINE std::string to_string( PipelineCacheHeaderVersion value )
  {
    switch ( value )
    {
      case PipelineCacheHeaderVersion::eOne : return "One";
      default: return "invalid";
    }
  }

  enum class PointClippingBehavior
  {
    eAllClipPlanes = VK_POINT_CLIPPING_BEHAVIOR_ALL_CLIP_PLANES,
    eUserClipPlanesOnly = VK_POINT_CLIPPING_BEHAVIOR_USER_CLIP_PLANES_ONLY,
    eAllClipPlanesKHR = VK_POINT_CLIPPING_BEHAVIOR_ALL_CLIP_PLANES_KHR,
    eUserClipPlanesOnlyKHR = VK_POINT_CLIPPING_BEHAVIOR_USER_CLIP_PLANES_ONLY_KHR
  };

  VULKAN_HPP_INLINE std::string to_string( PointClippingBehavior value )
  {
    switch ( value )
    {
      case PointClippingBehavior::eAllClipPlanes : return "AllClipPlanes";
      case PointClippingBehavior::eUserClipPlanesOnly : return "UserClipPlanesOnly";
      default: return "invalid";
    }
  }

  enum class PolygonMode
  {
    eFill = VK_POLYGON_MODE_FILL,
    eLine = VK_POLYGON_MODE_LINE,
    ePoint = VK_POLYGON_MODE_POINT,
    eFillRectangleNV = VK_POLYGON_MODE_FILL_RECTANGLE_NV
  };

  VULKAN_HPP_INLINE std::string to_string( PolygonMode value )
  {
    switch ( value )
    {
      case PolygonMode::eFill : return "Fill";
      case PolygonMode::eLine : return "Line";
      case PolygonMode::ePoint : return "Point";
      case PolygonMode::eFillRectangleNV : return "FillRectangleNV";
      default: return "invalid";
    }
  }

  enum class PresentModeKHR
  {
    eImmediate = VK_PRESENT_MODE_IMMEDIATE_KHR,
    eMailbox = VK_PRESENT_MODE_MAILBOX_KHR,
    eFifo = VK_PRESENT_MODE_FIFO_KHR,
    eFifoRelaxed = VK_PRESENT_MODE_FIFO_RELAXED_KHR,
    eSharedDemandRefresh = VK_PRESENT_MODE_SHARED_DEMAND_REFRESH_KHR,
    eSharedContinuousRefresh = VK_PRESENT_MODE_SHARED_CONTINUOUS_REFRESH_KHR
  };

  VULKAN_HPP_INLINE std::string to_string( PresentModeKHR value )
  {
    switch ( value )
    {
      case PresentModeKHR::eImmediate : return "Immediate";
      case PresentModeKHR::eMailbox : return "Mailbox";
      case PresentModeKHR::eFifo : return "Fifo";
      case PresentModeKHR::eFifoRelaxed : return "FifoRelaxed";
      case PresentModeKHR::eSharedDemandRefresh : return "SharedDemandRefresh";
      case PresentModeKHR::eSharedContinuousRefresh : return "SharedContinuousRefresh";
      default: return "invalid";
    }
  }

  enum class PrimitiveTopology
  {
    ePointList = VK_PRIMITIVE_TOPOLOGY_POINT_LIST,
    eLineList = VK_PRIMITIVE_TOPOLOGY_LINE_LIST,
    eLineStrip = VK_PRIMITIVE_TOPOLOGY_LINE_STRIP,
    eTriangleList = VK_PRIMITIVE_TOPOLOGY_TRIANGLE_LIST,
    eTriangleStrip = VK_PRIMITIVE_TOPOLOGY_TRIANGLE_STRIP,
    eTriangleFan = VK_PRIMITIVE_TOPOLOGY_TRIANGLE_FAN,
    eLineListWithAdjacency = VK_PRIMITIVE_TOPOLOGY_LINE_LIST_WITH_ADJACENCY,
    eLineStripWithAdjacency = VK_PRIMITIVE_TOPOLOGY_LINE_STRIP_WITH_ADJACENCY,
    eTriangleListWithAdjacency = VK_PRIMITIVE_TOPOLOGY_TRIANGLE_LIST_WITH_ADJACENCY,
    eTriangleStripWithAdjacency = VK_PRIMITIVE_TOPOLOGY_TRIANGLE_STRIP_WITH_ADJACENCY,
    ePatchList = VK_PRIMITIVE_TOPOLOGY_PATCH_LIST
  };

  VULKAN_HPP_INLINE std::string to_string( PrimitiveTopology value )
  {
    switch ( value )
    {
      case PrimitiveTopology::ePointList : return "PointList";
      case PrimitiveTopology::eLineList : return "LineList";
      case PrimitiveTopology::eLineStrip : return "LineStrip";
      case PrimitiveTopology::eTriangleList : return "TriangleList";
      case PrimitiveTopology::eTriangleStrip : return "TriangleStrip";
      case PrimitiveTopology::eTriangleFan : return "TriangleFan";
      case PrimitiveTopology::eLineListWithAdjacency : return "LineListWithAdjacency";
      case PrimitiveTopology::eLineStripWithAdjacency : return "LineStripWithAdjacency";
      case PrimitiveTopology::eTriangleListWithAdjacency : return "TriangleListWithAdjacency";
      case PrimitiveTopology::eTriangleStripWithAdjacency : return "TriangleStripWithAdjacency";
      case PrimitiveTopology::ePatchList : return "PatchList";
      default: return "invalid";
    }
  }

  enum class QueryType
  {
    eOcclusion = VK_QUERY_TYPE_OCCLUSION,
    ePipelineStatistics = VK_QUERY_TYPE_PIPELINE_STATISTICS,
    eTimestamp = VK_QUERY_TYPE_TIMESTAMP,
    eTransformFeedbackStreamEXT = VK_QUERY_TYPE_TRANSFORM_FEEDBACK_STREAM_EXT,
    eAccelerationStructureCompactedSizeNV = VK_QUERY_TYPE_ACCELERATION_STRUCTURE_COMPACTED_SIZE_NV
  };

  VULKAN_HPP_INLINE std::string to_string( QueryType value )
  {
    switch ( value )
    {
      case QueryType::eOcclusion : return "Occlusion";
      case QueryType::ePipelineStatistics : return "PipelineStatistics";
      case QueryType::eTimestamp : return "Timestamp";
      case QueryType::eTransformFeedbackStreamEXT : return "TransformFeedbackStreamEXT";
      case QueryType::eAccelerationStructureCompactedSizeNV : return "AccelerationStructureCompactedSizeNV";
      default: return "invalid";
    }
  }

  enum class QueueGlobalPriorityEXT
  {
    eLow = VK_QUEUE_GLOBAL_PRIORITY_LOW_EXT,
    eMedium = VK_QUEUE_GLOBAL_PRIORITY_MEDIUM_EXT,
    eHigh = VK_QUEUE_GLOBAL_PRIORITY_HIGH_EXT,
    eRealtime = VK_QUEUE_GLOBAL_PRIORITY_REALTIME_EXT
  };

  VULKAN_HPP_INLINE std::string to_string( QueueGlobalPriorityEXT value )
  {
    switch ( value )
    {
      case QueueGlobalPriorityEXT::eLow : return "Low";
      case QueueGlobalPriorityEXT::eMedium : return "Medium";
      case QueueGlobalPriorityEXT::eHigh : return "High";
      case QueueGlobalPriorityEXT::eRealtime : return "Realtime";
      default: return "invalid";
    }
  }

  enum class RasterizationOrderAMD
  {
    eStrict = VK_RASTERIZATION_ORDER_STRICT_AMD,
    eRelaxed = VK_RASTERIZATION_ORDER_RELAXED_AMD
  };

  VULKAN_HPP_INLINE std::string to_string( RasterizationOrderAMD value )
  {
    switch ( value )
    {
      case RasterizationOrderAMD::eStrict : return "Strict";
      case RasterizationOrderAMD::eRelaxed : return "Relaxed";
      default: return "invalid";
    }
  }

  enum class RayTracingShaderGroupTypeNV
  {
    eGeneral = VK_RAY_TRACING_SHADER_GROUP_TYPE_GENERAL_NV,
    eTrianglesHitGroup = VK_RAY_TRACING_SHADER_GROUP_TYPE_TRIANGLES_HIT_GROUP_NV,
    eProceduralHitGroup = VK_RAY_TRACING_SHADER_GROUP_TYPE_PROCEDURAL_HIT_GROUP_NV
  };

  VULKAN_HPP_INLINE std::string to_string( RayTracingShaderGroupTypeNV value )
  {
    switch ( value )
    {
      case RayTracingShaderGroupTypeNV::eGeneral : return "General";
      case RayTracingShaderGroupTypeNV::eTrianglesHitGroup : return "TrianglesHitGroup";
      case RayTracingShaderGroupTypeNV::eProceduralHitGroup : return "ProceduralHitGroup";
      default: return "invalid";
    }
  }

  enum class Result
  {
    eSuccess = VK_SUCCESS,
    eNotReady = VK_NOT_READY,
    eTimeout = VK_TIMEOUT,
    eEventSet = VK_EVENT_SET,
    eEventReset = VK_EVENT_RESET,
    eIncomplete = VK_INCOMPLETE,
    eErrorOutOfHostMemory = VK_ERROR_OUT_OF_HOST_MEMORY,
    eErrorOutOfDeviceMemory = VK_ERROR_OUT_OF_DEVICE_MEMORY,
    eErrorInitializationFailed = VK_ERROR_INITIALIZATION_FAILED,
    eErrorDeviceLost = VK_ERROR_DEVICE_LOST,
    eErrorMemoryMapFailed = VK_ERROR_MEMORY_MAP_FAILED,
    eErrorLayerNotPresent = VK_ERROR_LAYER_NOT_PRESENT,
    eErrorExtensionNotPresent = VK_ERROR_EXTENSION_NOT_PRESENT,
    eErrorFeatureNotPresent = VK_ERROR_FEATURE_NOT_PRESENT,
    eErrorIncompatibleDriver = VK_ERROR_INCOMPATIBLE_DRIVER,
    eErrorTooManyObjects = VK_ERROR_TOO_MANY_OBJECTS,
    eErrorFormatNotSupported = VK_ERROR_FORMAT_NOT_SUPPORTED,
    eErrorFragmentedPool = VK_ERROR_FRAGMENTED_POOL,
    eErrorOutOfPoolMemory = VK_ERROR_OUT_OF_POOL_MEMORY,
    eErrorInvalidExternalHandle = VK_ERROR_INVALID_EXTERNAL_HANDLE,
    eErrorSurfaceLostKHR = VK_ERROR_SURFACE_LOST_KHR,
    eErrorNativeWindowInUseKHR = VK_ERROR_NATIVE_WINDOW_IN_USE_KHR,
    eSuboptimalKHR = VK_SUBOPTIMAL_KHR,
    eErrorOutOfDateKHR = VK_ERROR_OUT_OF_DATE_KHR,
    eErrorIncompatibleDisplayKHR = VK_ERROR_INCOMPATIBLE_DISPLAY_KHR,
    eErrorValidationFailedEXT = VK_ERROR_VALIDATION_FAILED_EXT,
    eErrorInvalidShaderNV = VK_ERROR_INVALID_SHADER_NV,
    eErrorInvalidDrmFormatModifierPlaneLayoutEXT = VK_ERROR_INVALID_DRM_FORMAT_MODIFIER_PLANE_LAYOUT_EXT,
    eErrorFragmentationEXT = VK_ERROR_FRAGMENTATION_EXT,
    eErrorNotPermittedEXT = VK_ERROR_NOT_PERMITTED_EXT,
    eErrorInvalidDeviceAddressEXT = VK_ERROR_INVALID_DEVICE_ADDRESS_EXT,
    eErrorOutOfPoolMemoryKHR = VK_ERROR_OUT_OF_POOL_MEMORY_KHR,
    eErrorInvalidExternalHandleKHR = VK_ERROR_INVALID_EXTERNAL_HANDLE_KHR
  };

  VULKAN_HPP_INLINE std::string to_string( Result value )
  {
    switch ( value )
    {
      case Result::eSuccess : return "Success";
      case Result::eNotReady : return "NotReady";
      case Result::eTimeout : return "Timeout";
      case Result::eEventSet : return "EventSet";
      case Result::eEventReset : return "EventReset";
      case Result::eIncomplete : return "Incomplete";
      case Result::eErrorOutOfHostMemory : return "ErrorOutOfHostMemory";
      case Result::eErrorOutOfDeviceMemory : return "ErrorOutOfDeviceMemory";
      case Result::eErrorInitializationFailed : return "ErrorInitializationFailed";
      case Result::eErrorDeviceLost : return "ErrorDeviceLost";
      case Result::eErrorMemoryMapFailed : return "ErrorMemoryMapFailed";
      case Result::eErrorLayerNotPresent : return "ErrorLayerNotPresent";
      case Result::eErrorExtensionNotPresent : return "ErrorExtensionNotPresent";
      case Result::eErrorFeatureNotPresent : return "ErrorFeatureNotPresent";
      case Result::eErrorIncompatibleDriver : return "ErrorIncompatibleDriver";
      case Result::eErrorTooManyObjects : return "ErrorTooManyObjects";
      case Result::eErrorFormatNotSupported : return "ErrorFormatNotSupported";
      case Result::eErrorFragmentedPool : return "ErrorFragmentedPool";
      case Result::eErrorOutOfPoolMemory : return "ErrorOutOfPoolMemory";
      case Result::eErrorInvalidExternalHandle : return "ErrorInvalidExternalHandle";
      case Result::eErrorSurfaceLostKHR : return "ErrorSurfaceLostKHR";
      case Result::eErrorNativeWindowInUseKHR : return "ErrorNativeWindowInUseKHR";
      case Result::eSuboptimalKHR : return "SuboptimalKHR";
      case Result::eErrorOutOfDateKHR : return "ErrorOutOfDateKHR";
      case Result::eErrorIncompatibleDisplayKHR : return "ErrorIncompatibleDisplayKHR";
      case Result::eErrorValidationFailedEXT : return "ErrorValidationFailedEXT";
      case Result::eErrorInvalidShaderNV : return "ErrorInvalidShaderNV";
      case Result::eErrorInvalidDrmFormatModifierPlaneLayoutEXT : return "ErrorInvalidDrmFormatModifierPlaneLayoutEXT";
      case Result::eErrorFragmentationEXT : return "ErrorFragmentationEXT";
      case Result::eErrorNotPermittedEXT : return "ErrorNotPermittedEXT";
      case Result::eErrorInvalidDeviceAddressEXT : return "ErrorInvalidDeviceAddressEXT";
      default: return "invalid";
    }
  }

  enum class SamplerAddressMode
  {
    eRepeat = VK_SAMPLER_ADDRESS_MODE_REPEAT,
    eMirroredRepeat = VK_SAMPLER_ADDRESS_MODE_MIRRORED_REPEAT,
    eClampToEdge = VK_SAMPLER_ADDRESS_MODE_CLAMP_TO_EDGE,
    eClampToBorder = VK_SAMPLER_ADDRESS_MODE_CLAMP_TO_BORDER,
    eMirrorClampToEdge = VK_SAMPLER_ADDRESS_MODE_MIRROR_CLAMP_TO_EDGE
  };

  VULKAN_HPP_INLINE std::string to_string( SamplerAddressMode value )
  {
    switch ( value )
    {
      case SamplerAddressMode::eRepeat : return "Repeat";
      case SamplerAddressMode::eMirroredRepeat : return "MirroredRepeat";
      case SamplerAddressMode::eClampToEdge : return "ClampToEdge";
      case SamplerAddressMode::eClampToBorder : return "ClampToBorder";
      case SamplerAddressMode::eMirrorClampToEdge : return "MirrorClampToEdge";
      default: return "invalid";
    }
  }

  enum class SamplerMipmapMode
  {
    eNearest = VK_SAMPLER_MIPMAP_MODE_NEAREST,
    eLinear = VK_SAMPLER_MIPMAP_MODE_LINEAR
  };

  VULKAN_HPP_INLINE std::string to_string( SamplerMipmapMode value )
  {
    switch ( value )
    {
      case SamplerMipmapMode::eNearest : return "Nearest";
      case SamplerMipmapMode::eLinear : return "Linear";
      default: return "invalid";
    }
  }

  enum class SamplerReductionModeEXT
  {
    eWeightedAverage = VK_SAMPLER_REDUCTION_MODE_WEIGHTED_AVERAGE_EXT,
    eMin = VK_SAMPLER_REDUCTION_MODE_MIN_EXT,
    eMax = VK_SAMPLER_REDUCTION_MODE_MAX_EXT
  };

  VULKAN_HPP_INLINE std::string to_string( SamplerReductionModeEXT value )
  {
    switch ( value )
    {
      case SamplerReductionModeEXT::eWeightedAverage : return "WeightedAverage";
      case SamplerReductionModeEXT::eMin : return "Min";
      case SamplerReductionModeEXT::eMax : return "Max";
      default: return "invalid";
    }
  }

  enum class SamplerYcbcrModelConversion
  {
    eRgbIdentity = VK_SAMPLER_YCBCR_MODEL_CONVERSION_RGB_IDENTITY,
    eYcbcrIdentity = VK_SAMPLER_YCBCR_MODEL_CONVERSION_YCBCR_IDENTITY,
    eYcbcr709 = VK_SAMPLER_YCBCR_MODEL_CONVERSION_YCBCR_709,
    eYcbcr601 = VK_SAMPLER_YCBCR_MODEL_CONVERSION_YCBCR_601,
    eYcbcr2020 = VK_SAMPLER_YCBCR_MODEL_CONVERSION_YCBCR_2020,
    eRgbIdentityKHR = VK_SAMPLER_YCBCR_MODEL_CONVERSION_RGB_IDENTITY_KHR,
    eYcbcrIdentityKHR = VK_SAMPLER_YCBCR_MODEL_CONVERSION_YCBCR_IDENTITY_KHR,
    eYcbcr709KHR = VK_SAMPLER_YCBCR_MODEL_CONVERSION_YCBCR_709_KHR,
    eYcbcr601KHR = VK_SAMPLER_YCBCR_MODEL_CONVERSION_YCBCR_601_KHR,
    eYcbcr2020KHR = VK_SAMPLER_YCBCR_MODEL_CONVERSION_YCBCR_2020_KHR
  };

  VULKAN_HPP_INLINE std::string to_string( SamplerYcbcrModelConversion value )
  {
    switch ( value )
    {
      case SamplerYcbcrModelConversion::eRgbIdentity : return "RgbIdentity";
      case SamplerYcbcrModelConversion::eYcbcrIdentity : return "YcbcrIdentity";
      case SamplerYcbcrModelConversion::eYcbcr709 : return "Ycbcr709";
      case SamplerYcbcrModelConversion::eYcbcr601 : return "Ycbcr601";
      case SamplerYcbcrModelConversion::eYcbcr2020 : return "Ycbcr2020";
      default: return "invalid";
    }
  }

  enum class SamplerYcbcrRange
  {
    eItuFull = VK_SAMPLER_YCBCR_RANGE_ITU_FULL,
    eItuNarrow = VK_SAMPLER_YCBCR_RANGE_ITU_NARROW,
    eItuFullKHR = VK_SAMPLER_YCBCR_RANGE_ITU_FULL_KHR,
    eItuNarrowKHR = VK_SAMPLER_YCBCR_RANGE_ITU_NARROW_KHR
  };

  VULKAN_HPP_INLINE std::string to_string( SamplerYcbcrRange value )
  {
    switch ( value )
    {
      case SamplerYcbcrRange::eItuFull : return "ItuFull";
      case SamplerYcbcrRange::eItuNarrow : return "ItuNarrow";
      default: return "invalid";
    }
  }

  enum class ScopeNV
  {
    eDevice = VK_SCOPE_DEVICE_NV,
    eWorkgroup = VK_SCOPE_WORKGROUP_NV,
    eSubgroup = VK_SCOPE_SUBGROUP_NV,
    eQueueFamily = VK_SCOPE_QUEUE_FAMILY_NV
  };

  VULKAN_HPP_INLINE std::string to_string( ScopeNV value )
  {
    switch ( value )
    {
      case ScopeNV::eDevice : return "Device";
      case ScopeNV::eWorkgroup : return "Workgroup";
      case ScopeNV::eSubgroup : return "Subgroup";
      case ScopeNV::eQueueFamily : return "QueueFamily";
      default: return "invalid";
    }
  }

  enum class ShaderInfoTypeAMD
  {
    eStatistics = VK_SHADER_INFO_TYPE_STATISTICS_AMD,
    eBinary = VK_SHADER_INFO_TYPE_BINARY_AMD,
    eDisassembly = VK_SHADER_INFO_TYPE_DISASSEMBLY_AMD
  };

  VULKAN_HPP_INLINE std::string to_string( ShaderInfoTypeAMD value )
  {
    switch ( value )
    {
      case ShaderInfoTypeAMD::eStatistics : return "Statistics";
      case ShaderInfoTypeAMD::eBinary : return "Binary";
      case ShaderInfoTypeAMD::eDisassembly : return "Disassembly";
      default: return "invalid";
    }
  }

  enum class ShadingRatePaletteEntryNV
  {
    eNoInvocations = VK_SHADING_RATE_PALETTE_ENTRY_NO_INVOCATIONS_NV,
    e16InvocationsPerPixel = VK_SHADING_RATE_PALETTE_ENTRY_16_INVOCATIONS_PER_PIXEL_NV,
    e8InvocationsPerPixel = VK_SHADING_RATE_PALETTE_ENTRY_8_INVOCATIONS_PER_PIXEL_NV,
    e4InvocationsPerPixel = VK_SHADING_RATE_PALETTE_ENTRY_4_INVOCATIONS_PER_PIXEL_NV,
    e2InvocationsPerPixel = VK_SHADING_RATE_PALETTE_ENTRY_2_INVOCATIONS_PER_PIXEL_NV,
    e1InvocationPerPixel = VK_SHADING_RATE_PALETTE_ENTRY_1_INVOCATION_PER_PIXEL_NV,
    e1InvocationPer2X1Pixels = VK_SHADING_RATE_PALETTE_ENTRY_1_INVOCATION_PER_2X1_PIXELS_NV,
    e1InvocationPer1X2Pixels = VK_SHADING_RATE_PALETTE_ENTRY_1_INVOCATION_PER_1X2_PIXELS_NV,
    e1InvocationPer2X2Pixels = VK_SHADING_RATE_PALETTE_ENTRY_1_INVOCATION_PER_2X2_PIXELS_NV,
    e1InvocationPer4X2Pixels = VK_SHADING_RATE_PALETTE_ENTRY_1_INVOCATION_PER_4X2_PIXELS_NV,
    e1InvocationPer2X4Pixels = VK_SHADING_RATE_PALETTE_ENTRY_1_INVOCATION_PER_2X4_PIXELS_NV,
    e1InvocationPer4X4Pixels = VK_SHADING_RATE_PALETTE_ENTRY_1_INVOCATION_PER_4X4_PIXELS_NV
  };

  VULKAN_HPP_INLINE std::string to_string( ShadingRatePaletteEntryNV value )
  {
    switch ( value )
    {
      case ShadingRatePaletteEntryNV::eNoInvocations : return "NoInvocations";
      case ShadingRatePaletteEntryNV::e16InvocationsPerPixel : return "16InvocationsPerPixel";
      case ShadingRatePaletteEntryNV::e8InvocationsPerPixel : return "8InvocationsPerPixel";
      case ShadingRatePaletteEntryNV::e4InvocationsPerPixel : return "4InvocationsPerPixel";
      case ShadingRatePaletteEntryNV::e2InvocationsPerPixel : return "2InvocationsPerPixel";
      case ShadingRatePaletteEntryNV::e1InvocationPerPixel : return "1InvocationPerPixel";
      case ShadingRatePaletteEntryNV::e1InvocationPer2X1Pixels : return "1InvocationPer2X1Pixels";
      case ShadingRatePaletteEntryNV::e1InvocationPer1X2Pixels : return "1InvocationPer1X2Pixels";
      case ShadingRatePaletteEntryNV::e1InvocationPer2X2Pixels : return "1InvocationPer2X2Pixels";
      case ShadingRatePaletteEntryNV::e1InvocationPer4X2Pixels : return "1InvocationPer4X2Pixels";
      case ShadingRatePaletteEntryNV::e1InvocationPer2X4Pixels : return "1InvocationPer2X4Pixels";
      case ShadingRatePaletteEntryNV::e1InvocationPer4X4Pixels : return "1InvocationPer4X4Pixels";
      default: return "invalid";
    }
  }

  enum class SharingMode
  {
    eExclusive = VK_SHARING_MODE_EXCLUSIVE,
    eConcurrent = VK_SHARING_MODE_CONCURRENT
  };

  VULKAN_HPP_INLINE std::string to_string( SharingMode value )
  {
    switch ( value )
    {
      case SharingMode::eExclusive : return "Exclusive";
      case SharingMode::eConcurrent : return "Concurrent";
      default: return "invalid";
    }
  }

  enum class StencilOp
  {
    eKeep = VK_STENCIL_OP_KEEP,
    eZero = VK_STENCIL_OP_ZERO,
    eReplace = VK_STENCIL_OP_REPLACE,
    eIncrementAndClamp = VK_STENCIL_OP_INCREMENT_AND_CLAMP,
    eDecrementAndClamp = VK_STENCIL_OP_DECREMENT_AND_CLAMP,
    eInvert = VK_STENCIL_OP_INVERT,
    eIncrementAndWrap = VK_STENCIL_OP_INCREMENT_AND_WRAP,
    eDecrementAndWrap = VK_STENCIL_OP_DECREMENT_AND_WRAP
  };

  VULKAN_HPP_INLINE std::string to_string( StencilOp value )
  {
    switch ( value )
    {
      case StencilOp::eKeep : return "Keep";
      case StencilOp::eZero : return "Zero";
      case StencilOp::eReplace : return "Replace";
      case StencilOp::eIncrementAndClamp : return "IncrementAndClamp";
      case StencilOp::eDecrementAndClamp : return "DecrementAndClamp";
      case StencilOp::eInvert : return "Invert";
      case StencilOp::eIncrementAndWrap : return "IncrementAndWrap";
      case StencilOp::eDecrementAndWrap : return "DecrementAndWrap";
      default: return "invalid";
    }
  }

  enum class StructureType
  {
    eApplicationInfo = VK_STRUCTURE_TYPE_APPLICATION_INFO,
    eInstanceCreateInfo = VK_STRUCTURE_TYPE_INSTANCE_CREATE_INFO,
    eDeviceQueueCreateInfo = VK_STRUCTURE_TYPE_DEVICE_QUEUE_CREATE_INFO,
    eDeviceCreateInfo = VK_STRUCTURE_TYPE_DEVICE_CREATE_INFO,
    eSubmitInfo = VK_STRUCTURE_TYPE_SUBMIT_INFO,
    eMemoryAllocateInfo = VK_STRUCTURE_TYPE_MEMORY_ALLOCATE_INFO,
    eMappedMemoryRange = VK_STRUCTURE_TYPE_MAPPED_MEMORY_RANGE,
    eBindSparseInfo = VK_STRUCTURE_TYPE_BIND_SPARSE_INFO,
    eFenceCreateInfo = VK_STRUCTURE_TYPE_FENCE_CREATE_INFO,
    eSemaphoreCreateInfo = VK_STRUCTURE_TYPE_SEMAPHORE_CREATE_INFO,
    eEventCreateInfo = VK_STRUCTURE_TYPE_EVENT_CREATE_INFO,
    eQueryPoolCreateInfo = VK_STRUCTURE_TYPE_QUERY_POOL_CREATE_INFO,
    eBufferCreateInfo = VK_STRUCTURE_TYPE_BUFFER_CREATE_INFO,
    eBufferViewCreateInfo = VK_STRUCTURE_TYPE_BUFFER_VIEW_CREATE_INFO,
    eImageCreateInfo = VK_STRUCTURE_TYPE_IMAGE_CREATE_INFO,
    eImageViewCreateInfo = VK_STRUCTURE_TYPE_IMAGE_VIEW_CREATE_INFO,
    eShaderModuleCreateInfo = VK_STRUCTURE_TYPE_SHADER_MODULE_CREATE_INFO,
    ePipelineCacheCreateInfo = VK_STRUCTURE_TYPE_PIPELINE_CACHE_CREATE_INFO,
    ePipelineShaderStageCreateInfo = VK_STRUCTURE_TYPE_PIPELINE_SHADER_STAGE_CREATE_INFO,
    ePipelineVertexInputStateCreateInfo = VK_STRUCTURE_TYPE_PIPELINE_VERTEX_INPUT_STATE_CREATE_INFO,
    ePipelineInputAssemblyStateCreateInfo = VK_STRUCTURE_TYPE_PIPELINE_INPUT_ASSEMBLY_STATE_CREATE_INFO,
    ePipelineTessellationStateCreateInfo = VK_STRUCTURE_TYPE_PIPELINE_TESSELLATION_STATE_CREATE_INFO,
    ePipelineViewportStateCreateInfo = VK_STRUCTURE_TYPE_PIPELINE_VIEWPORT_STATE_CREATE_INFO,
    ePipelineRasterizationStateCreateInfo = VK_STRUCTURE_TYPE_PIPELINE_RASTERIZATION_STATE_CREATE_INFO,
    ePipelineMultisampleStateCreateInfo = VK_STRUCTURE_TYPE_PIPELINE_MULTISAMPLE_STATE_CREATE_INFO,
    ePipelineDepthStencilStateCreateInfo = VK_STRUCTURE_TYPE_PIPELINE_DEPTH_STENCIL_STATE_CREATE_INFO,
    ePipelineColorBlendStateCreateInfo = VK_STRUCTURE_TYPE_PIPELINE_COLOR_BLEND_STATE_CREATE_INFO,
    ePipelineDynamicStateCreateInfo = VK_STRUCTURE_TYPE_PIPELINE_DYNAMIC_STATE_CREATE_INFO,
    eGraphicsPipelineCreateInfo = VK_STRUCTURE_TYPE_GRAPHICS_PIPELINE_CREATE_INFO,
    eComputePipelineCreateInfo = VK_STRUCTURE_TYPE_COMPUTE_PIPELINE_CREATE_INFO,
    ePipelineLayoutCreateInfo = VK_STRUCTURE_TYPE_PIPELINE_LAYOUT_CREATE_INFO,
    eSamplerCreateInfo = VK_STRUCTURE_TYPE_SAMPLER_CREATE_INFO,
    eDescriptorSetLayoutCreateInfo = VK_STRUCTURE_TYPE_DESCRIPTOR_SET_LAYOUT_CREATE_INFO,
    eDescriptorPoolCreateInfo = VK_STRUCTURE_TYPE_DESCRIPTOR_POOL_CREATE_INFO,
    eDescriptorSetAllocateInfo = VK_STRUCTURE_TYPE_DESCRIPTOR_SET_ALLOCATE_INFO,
    eWriteDescriptorSet = VK_STRUCTURE_TYPE_WRITE_DESCRIPTOR_SET,
    eCopyDescriptorSet = VK_STRUCTURE_TYPE_COPY_DESCRIPTOR_SET,
    eFramebufferCreateInfo = VK_STRUCTURE_TYPE_FRAMEBUFFER_CREATE_INFO,
    eRenderPassCreateInfo = VK_STRUCTURE_TYPE_RENDER_PASS_CREATE_INFO,
    eCommandPoolCreateInfo = VK_STRUCTURE_TYPE_COMMAND_POOL_CREATE_INFO,
    eCommandBufferAllocateInfo = VK_STRUCTURE_TYPE_COMMAND_BUFFER_ALLOCATE_INFO,
    eCommandBufferInheritanceInfo = VK_STRUCTURE_TYPE_COMMAND_BUFFER_INHERITANCE_INFO,
    eCommandBufferBeginInfo = VK_STRUCTURE_TYPE_COMMAND_BUFFER_BEGIN_INFO,
    eRenderPassBeginInfo = VK_STRUCTURE_TYPE_RENDER_PASS_BEGIN_INFO,
    eBufferMemoryBarrier = VK_STRUCTURE_TYPE_BUFFER_MEMORY_BARRIER,
    eImageMemoryBarrier = VK_STRUCTURE_TYPE_IMAGE_MEMORY_BARRIER,
    eMemoryBarrier = VK_STRUCTURE_TYPE_MEMORY_BARRIER,
    eLoaderInstanceCreateInfo = VK_STRUCTURE_TYPE_LOADER_INSTANCE_CREATE_INFO,
    eLoaderDeviceCreateInfo = VK_STRUCTURE_TYPE_LOADER_DEVICE_CREATE_INFO,
    ePhysicalDeviceSubgroupProperties = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SUBGROUP_PROPERTIES,
    eBindBufferMemoryInfo = VK_STRUCTURE_TYPE_BIND_BUFFER_MEMORY_INFO,
    eBindImageMemoryInfo = VK_STRUCTURE_TYPE_BIND_IMAGE_MEMORY_INFO,
    ePhysicalDevice16BitStorageFeatures = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_16BIT_STORAGE_FEATURES,
    eMemoryDedicatedRequirements = VK_STRUCTURE_TYPE_MEMORY_DEDICATED_REQUIREMENTS,
    eMemoryDedicatedAllocateInfo = VK_STRUCTURE_TYPE_MEMORY_DEDICATED_ALLOCATE_INFO,
    eMemoryAllocateFlagsInfo = VK_STRUCTURE_TYPE_MEMORY_ALLOCATE_FLAGS_INFO,
    eDeviceGroupRenderPassBeginInfo = VK_STRUCTURE_TYPE_DEVICE_GROUP_RENDER_PASS_BEGIN_INFO,
    eDeviceGroupCommandBufferBeginInfo = VK_STRUCTURE_TYPE_DEVICE_GROUP_COMMAND_BUFFER_BEGIN_INFO,
    eDeviceGroupSubmitInfo = VK_STRUCTURE_TYPE_DEVICE_GROUP_SUBMIT_INFO,
    eDeviceGroupBindSparseInfo = VK_STRUCTURE_TYPE_DEVICE_GROUP_BIND_SPARSE_INFO,
    eBindBufferMemoryDeviceGroupInfo = VK_STRUCTURE_TYPE_BIND_BUFFER_MEMORY_DEVICE_GROUP_INFO,
    eBindImageMemoryDeviceGroupInfo = VK_STRUCTURE_TYPE_BIND_IMAGE_MEMORY_DEVICE_GROUP_INFO,
    ePhysicalDeviceGroupProperties = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_GROUP_PROPERTIES,
    eDeviceGroupDeviceCreateInfo = VK_STRUCTURE_TYPE_DEVICE_GROUP_DEVICE_CREATE_INFO,
    eBufferMemoryRequirementsInfo2 = VK_STRUCTURE_TYPE_BUFFER_MEMORY_REQUIREMENTS_INFO_2,
    eImageMemoryRequirementsInfo2 = VK_STRUCTURE_TYPE_IMAGE_MEMORY_REQUIREMENTS_INFO_2,
    eImageSparseMemoryRequirementsInfo2 = VK_STRUCTURE_TYPE_IMAGE_SPARSE_MEMORY_REQUIREMENTS_INFO_2,
    eMemoryRequirements2 = VK_STRUCTURE_TYPE_MEMORY_REQUIREMENTS_2,
    eSparseImageMemoryRequirements2 = VK_STRUCTURE_TYPE_SPARSE_IMAGE_MEMORY_REQUIREMENTS_2,
    ePhysicalDeviceFeatures2 = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_FEATURES_2,
    ePhysicalDeviceProperties2 = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_PROPERTIES_2,
    eFormatProperties2 = VK_STRUCTURE_TYPE_FORMAT_PROPERTIES_2,
    eImageFormatProperties2 = VK_STRUCTURE_TYPE_IMAGE_FORMAT_PROPERTIES_2,
    ePhysicalDeviceImageFormatInfo2 = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_IMAGE_FORMAT_INFO_2,
    eQueueFamilyProperties2 = VK_STRUCTURE_TYPE_QUEUE_FAMILY_PROPERTIES_2,
    ePhysicalDeviceMemoryProperties2 = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_MEMORY_PROPERTIES_2,
    eSparseImageFormatProperties2 = VK_STRUCTURE_TYPE_SPARSE_IMAGE_FORMAT_PROPERTIES_2,
    ePhysicalDeviceSparseImageFormatInfo2 = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SPARSE_IMAGE_FORMAT_INFO_2,
    ePhysicalDevicePointClippingProperties = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_POINT_CLIPPING_PROPERTIES,
    eRenderPassInputAttachmentAspectCreateInfo = VK_STRUCTURE_TYPE_RENDER_PASS_INPUT_ATTACHMENT_ASPECT_CREATE_INFO,
    eImageViewUsageCreateInfo = VK_STRUCTURE_TYPE_IMAGE_VIEW_USAGE_CREATE_INFO,
    ePipelineTessellationDomainOriginStateCreateInfo = VK_STRUCTURE_TYPE_PIPELINE_TESSELLATION_DOMAIN_ORIGIN_STATE_CREATE_INFO,
    eRenderPassMultiviewCreateInfo = VK_STRUCTURE_TYPE_RENDER_PASS_MULTIVIEW_CREATE_INFO,
    ePhysicalDeviceMultiviewFeatures = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_MULTIVIEW_FEATURES,
    ePhysicalDeviceMultiviewProperties = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_MULTIVIEW_PROPERTIES,
    ePhysicalDeviceVariablePointerFeatures = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_VARIABLE_POINTER_FEATURES,
    eProtectedSubmitInfo = VK_STRUCTURE_TYPE_PROTECTED_SUBMIT_INFO,
    ePhysicalDeviceProtectedMemoryFeatures = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_PROTECTED_MEMORY_FEATURES,
    ePhysicalDeviceProtectedMemoryProperties = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_PROTECTED_MEMORY_PROPERTIES,
    eDeviceQueueInfo2 = VK_STRUCTURE_TYPE_DEVICE_QUEUE_INFO_2,
    eSamplerYcbcrConversionCreateInfo = VK_STRUCTURE_TYPE_SAMPLER_YCBCR_CONVERSION_CREATE_INFO,
    eSamplerYcbcrConversionInfo = VK_STRUCTURE_TYPE_SAMPLER_YCBCR_CONVERSION_INFO,
    eBindImagePlaneMemoryInfo = VK_STRUCTURE_TYPE_BIND_IMAGE_PLANE_MEMORY_INFO,
    eImagePlaneMemoryRequirementsInfo = VK_STRUCTURE_TYPE_IMAGE_PLANE_MEMORY_REQUIREMENTS_INFO,
    ePhysicalDeviceSamplerYcbcrConversionFeatures = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SAMPLER_YCBCR_CONVERSION_FEATURES,
    eSamplerYcbcrConversionImageFormatProperties = VK_STRUCTURE_TYPE_SAMPLER_YCBCR_CONVERSION_IMAGE_FORMAT_PROPERTIES,
    eDescriptorUpdateTemplateCreateInfo = VK_STRUCTURE_TYPE_DESCRIPTOR_UPDATE_TEMPLATE_CREATE_INFO,
    ePhysicalDeviceExternalImageFormatInfo = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_EXTERNAL_IMAGE_FORMAT_INFO,
    eExternalImageFormatProperties = VK_STRUCTURE_TYPE_EXTERNAL_IMAGE_FORMAT_PROPERTIES,
    ePhysicalDeviceExternalBufferInfo = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_EXTERNAL_BUFFER_INFO,
    eExternalBufferProperties = VK_STRUCTURE_TYPE_EXTERNAL_BUFFER_PROPERTIES,
    ePhysicalDeviceIdProperties = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_ID_PROPERTIES,
    eExternalMemoryBufferCreateInfo = VK_STRUCTURE_TYPE_EXTERNAL_MEMORY_BUFFER_CREATE_INFO,
    eExternalMemoryImageCreateInfo = VK_STRUCTURE_TYPE_EXTERNAL_MEMORY_IMAGE_CREATE_INFO,
    eExportMemoryAllocateInfo = VK_STRUCTURE_TYPE_EXPORT_MEMORY_ALLOCATE_INFO,
    ePhysicalDeviceExternalFenceInfo = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_EXTERNAL_FENCE_INFO,
    eExternalFenceProperties = VK_STRUCTURE_TYPE_EXTERNAL_FENCE_PROPERTIES,
    eExportFenceCreateInfo = VK_STRUCTURE_TYPE_EXPORT_FENCE_CREATE_INFO,
    eExportSemaphoreCreateInfo = VK_STRUCTURE_TYPE_EXPORT_SEMAPHORE_CREATE_INFO,
    ePhysicalDeviceExternalSemaphoreInfo = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_EXTERNAL_SEMAPHORE_INFO,
    eExternalSemaphoreProperties = VK_STRUCTURE_TYPE_EXTERNAL_SEMAPHORE_PROPERTIES,
    ePhysicalDeviceMaintenance3Properties = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_MAINTENANCE_3_PROPERTIES,
    eDescriptorSetLayoutSupport = VK_STRUCTURE_TYPE_DESCRIPTOR_SET_LAYOUT_SUPPORT,
    ePhysicalDeviceShaderDrawParameterFeatures = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SHADER_DRAW_PARAMETER_FEATURES,
    eSwapchainCreateInfoKHR = VK_STRUCTURE_TYPE_SWAPCHAIN_CREATE_INFO_KHR,
    ePresentInfoKHR = VK_STRUCTURE_TYPE_PRESENT_INFO_KHR,
    eDeviceGroupPresentCapabilitiesKHR = VK_STRUCTURE_TYPE_DEVICE_GROUP_PRESENT_CAPABILITIES_KHR,
    eImageSwapchainCreateInfoKHR = VK_STRUCTURE_TYPE_IMAGE_SWAPCHAIN_CREATE_INFO_KHR,
    eBindImageMemorySwapchainInfoKHR = VK_STRUCTURE_TYPE_BIND_IMAGE_MEMORY_SWAPCHAIN_INFO_KHR,
    eAcquireNextImageInfoKHR = VK_STRUCTURE_TYPE_ACQUIRE_NEXT_IMAGE_INFO_KHR,
    eDeviceGroupPresentInfoKHR = VK_STRUCTURE_TYPE_DEVICE_GROUP_PRESENT_INFO_KHR,
    eDeviceGroupSwapchainCreateInfoKHR = VK_STRUCTURE_TYPE_DEVICE_GROUP_SWAPCHAIN_CREATE_INFO_KHR,
    eDisplayModeCreateInfoKHR = VK_STRUCTURE_TYPE_DISPLAY_MODE_CREATE_INFO_KHR,
    eDisplaySurfaceCreateInfoKHR = VK_STRUCTURE_TYPE_DISPLAY_SURFACE_CREATE_INFO_KHR,
    eDisplayPresentInfoKHR = VK_STRUCTURE_TYPE_DISPLAY_PRESENT_INFO_KHR,
    eXlibSurfaceCreateInfoKHR = VK_STRUCTURE_TYPE_XLIB_SURFACE_CREATE_INFO_KHR,
    eXcbSurfaceCreateInfoKHR = VK_STRUCTURE_TYPE_XCB_SURFACE_CREATE_INFO_KHR,
    eWaylandSurfaceCreateInfoKHR = VK_STRUCTURE_TYPE_WAYLAND_SURFACE_CREATE_INFO_KHR,
    eAndroidSurfaceCreateInfoKHR = VK_STRUCTURE_TYPE_ANDROID_SURFACE_CREATE_INFO_KHR,
    eWin32SurfaceCreateInfoKHR = VK_STRUCTURE_TYPE_WIN32_SURFACE_CREATE_INFO_KHR,
    eDebugReportCallbackCreateInfoEXT = VK_STRUCTURE_TYPE_DEBUG_REPORT_CALLBACK_CREATE_INFO_EXT,
    ePipelineRasterizationStateRasterizationOrderAMD = VK_STRUCTURE_TYPE_PIPELINE_RASTERIZATION_STATE_RASTERIZATION_ORDER_AMD,
    eDebugMarkerObjectNameInfoEXT = VK_STRUCTURE_TYPE_DEBUG_MARKER_OBJECT_NAME_INFO_EXT,
    eDebugMarkerObjectTagInfoEXT = VK_STRUCTURE_TYPE_DEBUG_MARKER_OBJECT_TAG_INFO_EXT,
    eDebugMarkerMarkerInfoEXT = VK_STRUCTURE_TYPE_DEBUG_MARKER_MARKER_INFO_EXT,
    eDedicatedAllocationImageCreateInfoNV = VK_STRUCTURE_TYPE_DEDICATED_ALLOCATION_IMAGE_CREATE_INFO_NV,
    eDedicatedAllocationBufferCreateInfoNV = VK_STRUCTURE_TYPE_DEDICATED_ALLOCATION_BUFFER_CREATE_INFO_NV,
    eDedicatedAllocationMemoryAllocateInfoNV = VK_STRUCTURE_TYPE_DEDICATED_ALLOCATION_MEMORY_ALLOCATE_INFO_NV,
    ePhysicalDeviceTransformFeedbackFeaturesEXT = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_TRANSFORM_FEEDBACK_FEATURES_EXT,
    ePhysicalDeviceTransformFeedbackPropertiesEXT = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_TRANSFORM_FEEDBACK_PROPERTIES_EXT,
    ePipelineRasterizationStateStreamCreateInfoEXT = VK_STRUCTURE_TYPE_PIPELINE_RASTERIZATION_STATE_STREAM_CREATE_INFO_EXT,
    eTextureLodGatherFormatPropertiesAMD = VK_STRUCTURE_TYPE_TEXTURE_LOD_GATHER_FORMAT_PROPERTIES_AMD,
    ePhysicalDeviceCornerSampledImageFeaturesNV = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_CORNER_SAMPLED_IMAGE_FEATURES_NV,
    eExternalMemoryImageCreateInfoNV = VK_STRUCTURE_TYPE_EXTERNAL_MEMORY_IMAGE_CREATE_INFO_NV,
    eExportMemoryAllocateInfoNV = VK_STRUCTURE_TYPE_EXPORT_MEMORY_ALLOCATE_INFO_NV,
    eImportMemoryWin32HandleInfoNV = VK_STRUCTURE_TYPE_IMPORT_MEMORY_WIN32_HANDLE_INFO_NV,
    eExportMemoryWin32HandleInfoNV = VK_STRUCTURE_TYPE_EXPORT_MEMORY_WIN32_HANDLE_INFO_NV,
    eWin32KeyedMutexAcquireReleaseInfoNV = VK_STRUCTURE_TYPE_WIN32_KEYED_MUTEX_ACQUIRE_RELEASE_INFO_NV,
    eValidationFlagsEXT = VK_STRUCTURE_TYPE_VALIDATION_FLAGS_EXT,
    eViSurfaceCreateInfoNN = VK_STRUCTURE_TYPE_VI_SURFACE_CREATE_INFO_NN,
    eImageViewAstcDecodeModeEXT = VK_STRUCTURE_TYPE_IMAGE_VIEW_ASTC_DECODE_MODE_EXT,
    ePhysicalDeviceAstcDecodeFeaturesEXT = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_ASTC_DECODE_FEATURES_EXT,
    eImportMemoryWin32HandleInfoKHR = VK_STRUCTURE_TYPE_IMPORT_MEMORY_WIN32_HANDLE_INFO_KHR,
    eExportMemoryWin32HandleInfoKHR = VK_STRUCTURE_TYPE_EXPORT_MEMORY_WIN32_HANDLE_INFO_KHR,
    eMemoryWin32HandlePropertiesKHR = VK_STRUCTURE_TYPE_MEMORY_WIN32_HANDLE_PROPERTIES_KHR,
    eMemoryGetWin32HandleInfoKHR = VK_STRUCTURE_TYPE_MEMORY_GET_WIN32_HANDLE_INFO_KHR,
    eImportMemoryFdInfoKHR = VK_STRUCTURE_TYPE_IMPORT_MEMORY_FD_INFO_KHR,
    eMemoryFdPropertiesKHR = VK_STRUCTURE_TYPE_MEMORY_FD_PROPERTIES_KHR,
    eMemoryGetFdInfoKHR = VK_STRUCTURE_TYPE_MEMORY_GET_FD_INFO_KHR,
    eWin32KeyedMutexAcquireReleaseInfoKHR = VK_STRUCTURE_TYPE_WIN32_KEYED_MUTEX_ACQUIRE_RELEASE_INFO_KHR,
    eImportSemaphoreWin32HandleInfoKHR = VK_STRUCTURE_TYPE_IMPORT_SEMAPHORE_WIN32_HANDLE_INFO_KHR,
    eExportSemaphoreWin32HandleInfoKHR = VK_STRUCTURE_TYPE_EXPORT_SEMAPHORE_WIN32_HANDLE_INFO_KHR,
    eD3D12FenceSubmitInfoKHR = VK_STRUCTURE_TYPE_D3D12_FENCE_SUBMIT_INFO_KHR,
    eSemaphoreGetWin32HandleInfoKHR = VK_STRUCTURE_TYPE_SEMAPHORE_GET_WIN32_HANDLE_INFO_KHR,
    eImportSemaphoreFdInfoKHR = VK_STRUCTURE_TYPE_IMPORT_SEMAPHORE_FD_INFO_KHR,
    eSemaphoreGetFdInfoKHR = VK_STRUCTURE_TYPE_SEMAPHORE_GET_FD_INFO_KHR,
    ePhysicalDevicePushDescriptorPropertiesKHR = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_PUSH_DESCRIPTOR_PROPERTIES_KHR,
    eCommandBufferInheritanceConditionalRenderingInfoEXT = VK_STRUCTURE_TYPE_COMMAND_BUFFER_INHERITANCE_CONDITIONAL_RENDERING_INFO_EXT,
    ePhysicalDeviceConditionalRenderingFeaturesEXT = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_CONDITIONAL_RENDERING_FEATURES_EXT,
    eConditionalRenderingBeginInfoEXT = VK_STRUCTURE_TYPE_CONDITIONAL_RENDERING_BEGIN_INFO_EXT,
    ePhysicalDeviceFloat16Int8FeaturesKHR = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_FLOAT16_INT8_FEATURES_KHR,
    ePresentRegionsKHR = VK_STRUCTURE_TYPE_PRESENT_REGIONS_KHR,
    eObjectTableCreateInfoNVX = VK_STRUCTURE_TYPE_OBJECT_TABLE_CREATE_INFO_NVX,
    eIndirectCommandsLayoutCreateInfoNVX = VK_STRUCTURE_TYPE_INDIRECT_COMMANDS_LAYOUT_CREATE_INFO_NVX,
    eCmdProcessCommandsInfoNVX = VK_STRUCTURE_TYPE_CMD_PROCESS_COMMANDS_INFO_NVX,
    eCmdReserveSpaceForCommandsInfoNVX = VK_STRUCTURE_TYPE_CMD_RESERVE_SPACE_FOR_COMMANDS_INFO_NVX,
    eDeviceGeneratedCommandsLimitsNVX = VK_STRUCTURE_TYPE_DEVICE_GENERATED_COMMANDS_LIMITS_NVX,
    eDeviceGeneratedCommandsFeaturesNVX = VK_STRUCTURE_TYPE_DEVICE_GENERATED_COMMANDS_FEATURES_NVX,
    ePipelineViewportWScalingStateCreateInfoNV = VK_STRUCTURE_TYPE_PIPELINE_VIEWPORT_W_SCALING_STATE_CREATE_INFO_NV,
    eSurfaceCapabilities2EXT = VK_STRUCTURE_TYPE_SURFACE_CAPABILITIES_2_EXT,
    eDisplayPowerInfoEXT = VK_STRUCTURE_TYPE_DISPLAY_POWER_INFO_EXT,
    eDeviceEventInfoEXT = VK_STRUCTURE_TYPE_DEVICE_EVENT_INFO_EXT,
    eDisplayEventInfoEXT = VK_STRUCTURE_TYPE_DISPLAY_EVENT_INFO_EXT,
    eSwapchainCounterCreateInfoEXT = VK_STRUCTURE_TYPE_SWAPCHAIN_COUNTER_CREATE_INFO_EXT,
    ePresentTimesInfoGOOGLE = VK_STRUCTURE_TYPE_PRESENT_TIMES_INFO_GOOGLE,
    ePhysicalDeviceMultiviewPerViewAttributesPropertiesNVX = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_MULTIVIEW_PER_VIEW_ATTRIBUTES_PROPERTIES_NVX,
    ePipelineViewportSwizzleStateCreateInfoNV = VK_STRUCTURE_TYPE_PIPELINE_VIEWPORT_SWIZZLE_STATE_CREATE_INFO_NV,
    ePhysicalDeviceDiscardRectanglePropertiesEXT = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_DISCARD_RECTANGLE_PROPERTIES_EXT,
    ePipelineDiscardRectangleStateCreateInfoEXT = VK_STRUCTURE_TYPE_PIPELINE_DISCARD_RECTANGLE_STATE_CREATE_INFO_EXT,
    ePhysicalDeviceConservativeRasterizationPropertiesEXT = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_CONSERVATIVE_RASTERIZATION_PROPERTIES_EXT,
    ePipelineRasterizationConservativeStateCreateInfoEXT = VK_STRUCTURE_TYPE_PIPELINE_RASTERIZATION_CONSERVATIVE_STATE_CREATE_INFO_EXT,
    ePhysicalDeviceDepthClipEnableFeaturesEXT = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_DEPTH_CLIP_ENABLE_FEATURES_EXT,
    ePipelineRasterizationDepthClipStateCreateInfoEXT = VK_STRUCTURE_TYPE_PIPELINE_RASTERIZATION_DEPTH_CLIP_STATE_CREATE_INFO_EXT,
    eHdrMetadataEXT = VK_STRUCTURE_TYPE_HDR_METADATA_EXT,
    eAttachmentDescription2KHR = VK_STRUCTURE_TYPE_ATTACHMENT_DESCRIPTION_2_KHR,
    eAttachmentReference2KHR = VK_STRUCTURE_TYPE_ATTACHMENT_REFERENCE_2_KHR,
    eSubpassDescription2KHR = VK_STRUCTURE_TYPE_SUBPASS_DESCRIPTION_2_KHR,
    eSubpassDependency2KHR = VK_STRUCTURE_TYPE_SUBPASS_DEPENDENCY_2_KHR,
    eRenderPassCreateInfo2KHR = VK_STRUCTURE_TYPE_RENDER_PASS_CREATE_INFO_2_KHR,
    eSubpassBeginInfoKHR = VK_STRUCTURE_TYPE_SUBPASS_BEGIN_INFO_KHR,
    eSubpassEndInfoKHR = VK_STRUCTURE_TYPE_SUBPASS_END_INFO_KHR,
    eSharedPresentSurfaceCapabilitiesKHR = VK_STRUCTURE_TYPE_SHARED_PRESENT_SURFACE_CAPABILITIES_KHR,
    eImportFenceWin32HandleInfoKHR = VK_STRUCTURE_TYPE_IMPORT_FENCE_WIN32_HANDLE_INFO_KHR,
    eExportFenceWin32HandleInfoKHR = VK_STRUCTURE_TYPE_EXPORT_FENCE_WIN32_HANDLE_INFO_KHR,
    eFenceGetWin32HandleInfoKHR = VK_STRUCTURE_TYPE_FENCE_GET_WIN32_HANDLE_INFO_KHR,
    eImportFenceFdInfoKHR = VK_STRUCTURE_TYPE_IMPORT_FENCE_FD_INFO_KHR,
    eFenceGetFdInfoKHR = VK_STRUCTURE_TYPE_FENCE_GET_FD_INFO_KHR,
    ePhysicalDeviceSurfaceInfo2KHR = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SURFACE_INFO_2_KHR,
    eSurfaceCapabilities2KHR = VK_STRUCTURE_TYPE_SURFACE_CAPABILITIES_2_KHR,
    eSurfaceFormat2KHR = VK_STRUCTURE_TYPE_SURFACE_FORMAT_2_KHR,
    eDisplayProperties2KHR = VK_STRUCTURE_TYPE_DISPLAY_PROPERTIES_2_KHR,
    eDisplayPlaneProperties2KHR = VK_STRUCTURE_TYPE_DISPLAY_PLANE_PROPERTIES_2_KHR,
    eDisplayModeProperties2KHR = VK_STRUCTURE_TYPE_DISPLAY_MODE_PROPERTIES_2_KHR,
    eDisplayPlaneInfo2KHR = VK_STRUCTURE_TYPE_DISPLAY_PLANE_INFO_2_KHR,
    eDisplayPlaneCapabilities2KHR = VK_STRUCTURE_TYPE_DISPLAY_PLANE_CAPABILITIES_2_KHR,
    eIosSurfaceCreateInfoMVK = VK_STRUCTURE_TYPE_IOS_SURFACE_CREATE_INFO_MVK,
    eMacosSurfaceCreateInfoMVK = VK_STRUCTURE_TYPE_MACOS_SURFACE_CREATE_INFO_MVK,
    eDebugUtilsObjectNameInfoEXT = VK_STRUCTURE_TYPE_DEBUG_UTILS_OBJECT_NAME_INFO_EXT,
    eDebugUtilsObjectTagInfoEXT = VK_STRUCTURE_TYPE_DEBUG_UTILS_OBJECT_TAG_INFO_EXT,
    eDebugUtilsLabelEXT = VK_STRUCTURE_TYPE_DEBUG_UTILS_LABEL_EXT,
    eDebugUtilsMessengerCallbackDataEXT = VK_STRUCTURE_TYPE_DEBUG_UTILS_MESSENGER_CALLBACK_DATA_EXT,
    eDebugUtilsMessengerCreateInfoEXT = VK_STRUCTURE_TYPE_DEBUG_UTILS_MESSENGER_CREATE_INFO_EXT,
    eAndroidHardwareBufferUsageANDROID = VK_STRUCTURE_TYPE_ANDROID_HARDWARE_BUFFER_USAGE_ANDROID,
    eAndroidHardwareBufferPropertiesANDROID = VK_STRUCTURE_TYPE_ANDROID_HARDWARE_BUFFER_PROPERTIES_ANDROID,
    eAndroidHardwareBufferFormatPropertiesANDROID = VK_STRUCTURE_TYPE_ANDROID_HARDWARE_BUFFER_FORMAT_PROPERTIES_ANDROID,
    eImportAndroidHardwareBufferInfoANDROID = VK_STRUCTURE_TYPE_IMPORT_ANDROID_HARDWARE_BUFFER_INFO_ANDROID,
    eMemoryGetAndroidHardwareBufferInfoANDROID = VK_STRUCTURE_TYPE_MEMORY_GET_ANDROID_HARDWARE_BUFFER_INFO_ANDROID,
    eExternalFormatANDROID = VK_STRUCTURE_TYPE_EXTERNAL_FORMAT_ANDROID,
    ePhysicalDeviceSamplerFilterMinmaxPropertiesEXT = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SAMPLER_FILTER_MINMAX_PROPERTIES_EXT,
    eSamplerReductionModeCreateInfoEXT = VK_STRUCTURE_TYPE_SAMPLER_REDUCTION_MODE_CREATE_INFO_EXT,
    ePhysicalDeviceInlineUniformBlockFeaturesEXT = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_INLINE_UNIFORM_BLOCK_FEATURES_EXT,
    ePhysicalDeviceInlineUniformBlockPropertiesEXT = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_INLINE_UNIFORM_BLOCK_PROPERTIES_EXT,
    eWriteDescriptorSetInlineUniformBlockEXT = VK_STRUCTURE_TYPE_WRITE_DESCRIPTOR_SET_INLINE_UNIFORM_BLOCK_EXT,
    eDescriptorPoolInlineUniformBlockCreateInfoEXT = VK_STRUCTURE_TYPE_DESCRIPTOR_POOL_INLINE_UNIFORM_BLOCK_CREATE_INFO_EXT,
    eSampleLocationsInfoEXT = VK_STRUCTURE_TYPE_SAMPLE_LOCATIONS_INFO_EXT,
    eRenderPassSampleLocationsBeginInfoEXT = VK_STRUCTURE_TYPE_RENDER_PASS_SAMPLE_LOCATIONS_BEGIN_INFO_EXT,
    ePipelineSampleLocationsStateCreateInfoEXT = VK_STRUCTURE_TYPE_PIPELINE_SAMPLE_LOCATIONS_STATE_CREATE_INFO_EXT,
    ePhysicalDeviceSampleLocationsPropertiesEXT = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SAMPLE_LOCATIONS_PROPERTIES_EXT,
    eMultisamplePropertiesEXT = VK_STRUCTURE_TYPE_MULTISAMPLE_PROPERTIES_EXT,
    eImageFormatListCreateInfoKHR = VK_STRUCTURE_TYPE_IMAGE_FORMAT_LIST_CREATE_INFO_KHR,
    ePhysicalDeviceBlendOperationAdvancedFeaturesEXT = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_BLEND_OPERATION_ADVANCED_FEATURES_EXT,
    ePhysicalDeviceBlendOperationAdvancedPropertiesEXT = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_BLEND_OPERATION_ADVANCED_PROPERTIES_EXT,
    ePipelineColorBlendAdvancedStateCreateInfoEXT = VK_STRUCTURE_TYPE_PIPELINE_COLOR_BLEND_ADVANCED_STATE_CREATE_INFO_EXT,
    ePipelineCoverageToColorStateCreateInfoNV = VK_STRUCTURE_TYPE_PIPELINE_COVERAGE_TO_COLOR_STATE_CREATE_INFO_NV,
    ePipelineCoverageModulationStateCreateInfoNV = VK_STRUCTURE_TYPE_PIPELINE_COVERAGE_MODULATION_STATE_CREATE_INFO_NV,
    eDrmFormatModifierPropertiesListEXT = VK_STRUCTURE_TYPE_DRM_FORMAT_MODIFIER_PROPERTIES_LIST_EXT,
    eDrmFormatModifierPropertiesEXT = VK_STRUCTURE_TYPE_DRM_FORMAT_MODIFIER_PROPERTIES_EXT,
    ePhysicalDeviceImageDrmFormatModifierInfoEXT = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_IMAGE_DRM_FORMAT_MODIFIER_INFO_EXT,
    eImageDrmFormatModifierListCreateInfoEXT = VK_STRUCTURE_TYPE_IMAGE_DRM_FORMAT_MODIFIER_LIST_CREATE_INFO_EXT,
    eImageDrmFormatModifierExplicitCreateInfoEXT = VK_STRUCTURE_TYPE_IMAGE_DRM_FORMAT_MODIFIER_EXPLICIT_CREATE_INFO_EXT,
    eImageDrmFormatModifierPropertiesEXT = VK_STRUCTURE_TYPE_IMAGE_DRM_FORMAT_MODIFIER_PROPERTIES_EXT,
    eValidationCacheCreateInfoEXT = VK_STRUCTURE_TYPE_VALIDATION_CACHE_CREATE_INFO_EXT,
    eShaderModuleValidationCacheCreateInfoEXT = VK_STRUCTURE_TYPE_SHADER_MODULE_VALIDATION_CACHE_CREATE_INFO_EXT,
    eDescriptorSetLayoutBindingFlagsCreateInfoEXT = VK_STRUCTURE_TYPE_DESCRIPTOR_SET_LAYOUT_BINDING_FLAGS_CREATE_INFO_EXT,
    ePhysicalDeviceDescriptorIndexingFeaturesEXT = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_DESCRIPTOR_INDEXING_FEATURES_EXT,
    ePhysicalDeviceDescriptorIndexingPropertiesEXT = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_DESCRIPTOR_INDEXING_PROPERTIES_EXT,
    eDescriptorSetVariableDescriptorCountAllocateInfoEXT = VK_STRUCTURE_TYPE_DESCRIPTOR_SET_VARIABLE_DESCRIPTOR_COUNT_ALLOCATE_INFO_EXT,
    eDescriptorSetVariableDescriptorCountLayoutSupportEXT = VK_STRUCTURE_TYPE_DESCRIPTOR_SET_VARIABLE_DESCRIPTOR_COUNT_LAYOUT_SUPPORT_EXT,
    ePipelineViewportShadingRateImageStateCreateInfoNV = VK_STRUCTURE_TYPE_PIPELINE_VIEWPORT_SHADING_RATE_IMAGE_STATE_CREATE_INFO_NV,
    ePhysicalDeviceShadingRateImageFeaturesNV = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SHADING_RATE_IMAGE_FEATURES_NV,
    ePhysicalDeviceShadingRateImagePropertiesNV = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SHADING_RATE_IMAGE_PROPERTIES_NV,
    ePipelineViewportCoarseSampleOrderStateCreateInfoNV = VK_STRUCTURE_TYPE_PIPELINE_VIEWPORT_COARSE_SAMPLE_ORDER_STATE_CREATE_INFO_NV,
    eRayTracingPipelineCreateInfoNV = VK_STRUCTURE_TYPE_RAY_TRACING_PIPELINE_CREATE_INFO_NV,
    eAccelerationStructureCreateInfoNV = VK_STRUCTURE_TYPE_ACCELERATION_STRUCTURE_CREATE_INFO_NV,
    eGeometryNV = VK_STRUCTURE_TYPE_GEOMETRY_NV,
    eGeometryTrianglesNV = VK_STRUCTURE_TYPE_GEOMETRY_TRIANGLES_NV,
    eGeometryAabbNV = VK_STRUCTURE_TYPE_GEOMETRY_AABB_NV,
    eBindAccelerationStructureMemoryInfoNV = VK_STRUCTURE_TYPE_BIND_ACCELERATION_STRUCTURE_MEMORY_INFO_NV,
    eWriteDescriptorSetAccelerationStructureNV = VK_STRUCTURE_TYPE_WRITE_DESCRIPTOR_SET_ACCELERATION_STRUCTURE_NV,
    eAccelerationStructureMemoryRequirementsInfoNV = VK_STRUCTURE_TYPE_ACCELERATION_STRUCTURE_MEMORY_REQUIREMENTS_INFO_NV,
    ePhysicalDeviceRayTracingPropertiesNV = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_RAY_TRACING_PROPERTIES_NV,
    eRayTracingShaderGroupCreateInfoNV = VK_STRUCTURE_TYPE_RAY_TRACING_SHADER_GROUP_CREATE_INFO_NV,
    eAccelerationStructureInfoNV = VK_STRUCTURE_TYPE_ACCELERATION_STRUCTURE_INFO_NV,
    ePhysicalDeviceRepresentativeFragmentTestFeaturesNV = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_REPRESENTATIVE_FRAGMENT_TEST_FEATURES_NV,
    ePipelineRepresentativeFragmentTestStateCreateInfoNV = VK_STRUCTURE_TYPE_PIPELINE_REPRESENTATIVE_FRAGMENT_TEST_STATE_CREATE_INFO_NV,
    ePhysicalDeviceImageViewImageFormatInfoEXT = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_IMAGE_VIEW_IMAGE_FORMAT_INFO_EXT,
    eFilterCubicImageViewImageFormatPropertiesEXT = VK_STRUCTURE_TYPE_FILTER_CUBIC_IMAGE_VIEW_IMAGE_FORMAT_PROPERTIES_EXT,
    eDeviceQueueGlobalPriorityCreateInfoEXT = VK_STRUCTURE_TYPE_DEVICE_QUEUE_GLOBAL_PRIORITY_CREATE_INFO_EXT,
    ePhysicalDevice8BitStorageFeaturesKHR = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_8BIT_STORAGE_FEATURES_KHR,
    eImportMemoryHostPointerInfoEXT = VK_STRUCTURE_TYPE_IMPORT_MEMORY_HOST_POINTER_INFO_EXT,
    eMemoryHostPointerPropertiesEXT = VK_STRUCTURE_TYPE_MEMORY_HOST_POINTER_PROPERTIES_EXT,
    ePhysicalDeviceExternalMemoryHostPropertiesEXT = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_EXTERNAL_MEMORY_HOST_PROPERTIES_EXT,
    ePhysicalDeviceShaderAtomicInt64FeaturesKHR = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SHADER_ATOMIC_INT64_FEATURES_KHR,
    eCalibratedTimestampInfoEXT = VK_STRUCTURE_TYPE_CALIBRATED_TIMESTAMP_INFO_EXT,
    ePhysicalDeviceShaderCorePropertiesAMD = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SHADER_CORE_PROPERTIES_AMD,
    eDeviceMemoryOverallocationCreateInfoAMD = VK_STRUCTURE_TYPE_DEVICE_MEMORY_OVERALLOCATION_CREATE_INFO_AMD,
    ePhysicalDeviceVertexAttributeDivisorPropertiesEXT = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_VERTEX_ATTRIBUTE_DIVISOR_PROPERTIES_EXT,
    ePipelineVertexInputDivisorStateCreateInfoEXT = VK_STRUCTURE_TYPE_PIPELINE_VERTEX_INPUT_DIVISOR_STATE_CREATE_INFO_EXT,
    ePhysicalDeviceVertexAttributeDivisorFeaturesEXT = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_VERTEX_ATTRIBUTE_DIVISOR_FEATURES_EXT,
    ePhysicalDeviceDriverPropertiesKHR = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_DRIVER_PROPERTIES_KHR,
    ePhysicalDeviceFloatControlsPropertiesKHR = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_FLOAT_CONTROLS_PROPERTIES_KHR,
    ePhysicalDeviceDepthStencilResolvePropertiesKHR = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_DEPTH_STENCIL_RESOLVE_PROPERTIES_KHR,
    eSubpassDescriptionDepthStencilResolveKHR = VK_STRUCTURE_TYPE_SUBPASS_DESCRIPTION_DEPTH_STENCIL_RESOLVE_KHR,
    ePhysicalDeviceComputeShaderDerivativesFeaturesNV = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_COMPUTE_SHADER_DERIVATIVES_FEATURES_NV,
    ePhysicalDeviceMeshShaderFeaturesNV = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_MESH_SHADER_FEATURES_NV,
    ePhysicalDeviceMeshShaderPropertiesNV = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_MESH_SHADER_PROPERTIES_NV,
    ePhysicalDeviceFragmentShaderBarycentricFeaturesNV = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_FRAGMENT_SHADER_BARYCENTRIC_FEATURES_NV,
    ePhysicalDeviceShaderImageFootprintFeaturesNV = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SHADER_IMAGE_FOOTPRINT_FEATURES_NV,
    ePipelineViewportExclusiveScissorStateCreateInfoNV = VK_STRUCTURE_TYPE_PIPELINE_VIEWPORT_EXCLUSIVE_SCISSOR_STATE_CREATE_INFO_NV,
    ePhysicalDeviceExclusiveScissorFeaturesNV = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_EXCLUSIVE_SCISSOR_FEATURES_NV,
    eCheckpointDataNV = VK_STRUCTURE_TYPE_CHECKPOINT_DATA_NV,
    eQueueFamilyCheckpointPropertiesNV = VK_STRUCTURE_TYPE_QUEUE_FAMILY_CHECKPOINT_PROPERTIES_NV,
    ePhysicalDeviceVulkanMemoryModelFeaturesKHR = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_VULKAN_MEMORY_MODEL_FEATURES_KHR,
    ePhysicalDevicePciBusInfoPropertiesEXT = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_PCI_BUS_INFO_PROPERTIES_EXT,
    eImagepipeSurfaceCreateInfoFUCHSIA = VK_STRUCTURE_TYPE_IMAGEPIPE_SURFACE_CREATE_INFO_FUCHSIA,
    ePhysicalDeviceFragmentDensityMapFeaturesEXT = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_FRAGMENT_DENSITY_MAP_FEATURES_EXT,
    ePhysicalDeviceFragmentDensityMapPropertiesEXT = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_FRAGMENT_DENSITY_MAP_PROPERTIES_EXT,
    eRenderPassFragmentDensityMapCreateInfoEXT = VK_STRUCTURE_TYPE_RENDER_PASS_FRAGMENT_DENSITY_MAP_CREATE_INFO_EXT,
    ePhysicalDeviceScalarBlockLayoutFeaturesEXT = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SCALAR_BLOCK_LAYOUT_FEATURES_EXT,
    ePhysicalDeviceMemoryBudgetPropertiesEXT = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_MEMORY_BUDGET_PROPERTIES_EXT,
    ePhysicalDeviceMemoryPriorityFeaturesEXT = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_MEMORY_PRIORITY_FEATURES_EXT,
    eMemoryPriorityAllocateInfoEXT = VK_STRUCTURE_TYPE_MEMORY_PRIORITY_ALLOCATE_INFO_EXT,
    ePhysicalDeviceDedicatedAllocationImageAliasingFeaturesNV = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_DEDICATED_ALLOCATION_IMAGE_ALIASING_FEATURES_NV,
    ePhysicalDeviceBufferAddressFeaturesEXT = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_BUFFER_ADDRESS_FEATURES_EXT,
    eBufferDeviceAddressInfoEXT = VK_STRUCTURE_TYPE_BUFFER_DEVICE_ADDRESS_INFO_EXT,
    eBufferDeviceAddressCreateInfoEXT = VK_STRUCTURE_TYPE_BUFFER_DEVICE_ADDRESS_CREATE_INFO_EXT,
    eImageStencilUsageCreateInfoEXT = VK_STRUCTURE_TYPE_IMAGE_STENCIL_USAGE_CREATE_INFO_EXT,
    eValidationFeaturesEXT = VK_STRUCTURE_TYPE_VALIDATION_FEATURES_EXT,
    ePhysicalDeviceCooperativeMatrixFeaturesNV = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_COOPERATIVE_MATRIX_FEATURES_NV,
    eCooperativeMatrixPropertiesNV = VK_STRUCTURE_TYPE_COOPERATIVE_MATRIX_PROPERTIES_NV,
    ePhysicalDeviceCooperativeMatrixPropertiesNV = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_COOPERATIVE_MATRIX_PROPERTIES_NV,
    eDebugReportCreateInfoEXT = VK_STRUCTURE_TYPE_DEBUG_REPORT_CREATE_INFO_EXT,
    eRenderPassMultiviewCreateInfoKHR = VK_STRUCTURE_TYPE_RENDER_PASS_MULTIVIEW_CREATE_INFO_KHR,
    ePhysicalDeviceMultiviewFeaturesKHR = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_MULTIVIEW_FEATURES_KHR,
    ePhysicalDeviceMultiviewPropertiesKHR = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_MULTIVIEW_PROPERTIES_KHR,
    ePhysicalDeviceFeatures2KHR = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_FEATURES_2_KHR,
    ePhysicalDeviceProperties2KHR = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_PROPERTIES_2_KHR,
    eFormatProperties2KHR = VK_STRUCTURE_TYPE_FORMAT_PROPERTIES_2_KHR,
    eImageFormatProperties2KHR = VK_STRUCTURE_TYPE_IMAGE_FORMAT_PROPERTIES_2_KHR,
    ePhysicalDeviceImageFormatInfo2KHR = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_IMAGE_FORMAT_INFO_2_KHR,
    eQueueFamilyProperties2KHR = VK_STRUCTURE_TYPE_QUEUE_FAMILY_PROPERTIES_2_KHR,
    ePhysicalDeviceMemoryProperties2KHR = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_MEMORY_PROPERTIES_2_KHR,
    eSparseImageFormatProperties2KHR = VK_STRUCTURE_TYPE_SPARSE_IMAGE_FORMAT_PROPERTIES_2_KHR,
    ePhysicalDeviceSparseImageFormatInfo2KHR = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SPARSE_IMAGE_FORMAT_INFO_2_KHR,
    eMemoryAllocateFlagsInfoKHR = VK_STRUCTURE_TYPE_MEMORY_ALLOCATE_FLAGS_INFO_KHR,
    eDeviceGroupRenderPassBeginInfoKHR = VK_STRUCTURE_TYPE_DEVICE_GROUP_RENDER_PASS_BEGIN_INFO_KHR,
    eDeviceGroupCommandBufferBeginInfoKHR = VK_STRUCTURE_TYPE_DEVICE_GROUP_COMMAND_BUFFER_BEGIN_INFO_KHR,
    eDeviceGroupSubmitInfoKHR = VK_STRUCTURE_TYPE_DEVICE_GROUP_SUBMIT_INFO_KHR,
    eDeviceGroupBindSparseInfoKHR = VK_STRUCTURE_TYPE_DEVICE_GROUP_BIND_SPARSE_INFO_KHR,
    eBindBufferMemoryDeviceGroupInfoKHR = VK_STRUCTURE_TYPE_BIND_BUFFER_MEMORY_DEVICE_GROUP_INFO_KHR,
    eBindImageMemoryDeviceGroupInfoKHR = VK_STRUCTURE_TYPE_BIND_IMAGE_MEMORY_DEVICE_GROUP_INFO_KHR,
    ePhysicalDeviceGroupPropertiesKHR = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_GROUP_PROPERTIES_KHR,
    eDeviceGroupDeviceCreateInfoKHR = VK_STRUCTURE_TYPE_DEVICE_GROUP_DEVICE_CREATE_INFO_KHR,
    ePhysicalDeviceExternalImageFormatInfoKHR = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_EXTERNAL_IMAGE_FORMAT_INFO_KHR,
    eExternalImageFormatPropertiesKHR = VK_STRUCTURE_TYPE_EXTERNAL_IMAGE_FORMAT_PROPERTIES_KHR,
    ePhysicalDeviceExternalBufferInfoKHR = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_EXTERNAL_BUFFER_INFO_KHR,
    eExternalBufferPropertiesKHR = VK_STRUCTURE_TYPE_EXTERNAL_BUFFER_PROPERTIES_KHR,
    ePhysicalDeviceIdPropertiesKHR = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_ID_PROPERTIES_KHR,
    eExternalMemoryBufferCreateInfoKHR = VK_STRUCTURE_TYPE_EXTERNAL_MEMORY_BUFFER_CREATE_INFO_KHR,
    eExternalMemoryImageCreateInfoKHR = VK_STRUCTURE_TYPE_EXTERNAL_MEMORY_IMAGE_CREATE_INFO_KHR,
    eExportMemoryAllocateInfoKHR = VK_STRUCTURE_TYPE_EXPORT_MEMORY_ALLOCATE_INFO_KHR,
    ePhysicalDeviceExternalSemaphoreInfoKHR = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_EXTERNAL_SEMAPHORE_INFO_KHR,
    eExternalSemaphorePropertiesKHR = VK_STRUCTURE_TYPE_EXTERNAL_SEMAPHORE_PROPERTIES_KHR,
    eExportSemaphoreCreateInfoKHR = VK_STRUCTURE_TYPE_EXPORT_SEMAPHORE_CREATE_INFO_KHR,
    ePhysicalDevice16BitStorageFeaturesKHR = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_16BIT_STORAGE_FEATURES_KHR,
    eDescriptorUpdateTemplateCreateInfoKHR = VK_STRUCTURE_TYPE_DESCRIPTOR_UPDATE_TEMPLATE_CREATE_INFO_KHR,
    ePhysicalDeviceExternalFenceInfoKHR = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_EXTERNAL_FENCE_INFO_KHR,
    eExternalFencePropertiesKHR = VK_STRUCTURE_TYPE_EXTERNAL_FENCE_PROPERTIES_KHR,
    eExportFenceCreateInfoKHR = VK_STRUCTURE_TYPE_EXPORT_FENCE_CREATE_INFO_KHR,
    ePhysicalDevicePointClippingPropertiesKHR = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_POINT_CLIPPING_PROPERTIES_KHR,
    eRenderPassInputAttachmentAspectCreateInfoKHR = VK_STRUCTURE_TYPE_RENDER_PASS_INPUT_ATTACHMENT_ASPECT_CREATE_INFO_KHR,
    eImageViewUsageCreateInfoKHR = VK_STRUCTURE_TYPE_IMAGE_VIEW_USAGE_CREATE_INFO_KHR,
    ePipelineTessellationDomainOriginStateCreateInfoKHR = VK_STRUCTURE_TYPE_PIPELINE_TESSELLATION_DOMAIN_ORIGIN_STATE_CREATE_INFO_KHR,
    ePhysicalDeviceVariablePointerFeaturesKHR = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_VARIABLE_POINTER_FEATURES_KHR,
    eMemoryDedicatedRequirementsKHR = VK_STRUCTURE_TYPE_MEMORY_DEDICATED_REQUIREMENTS_KHR,
    eMemoryDedicatedAllocateInfoKHR = VK_STRUCTURE_TYPE_MEMORY_DEDICATED_ALLOCATE_INFO_KHR,
    eBufferMemoryRequirementsInfo2KHR = VK_STRUCTURE_TYPE_BUFFER_MEMORY_REQUIREMENTS_INFO_2_KHR,
    eImageMemoryRequirementsInfo2KHR = VK_STRUCTURE_TYPE_IMAGE_MEMORY_REQUIREMENTS_INFO_2_KHR,
    eImageSparseMemoryRequirementsInfo2KHR = VK_STRUCTURE_TYPE_IMAGE_SPARSE_MEMORY_REQUIREMENTS_INFO_2_KHR,
    eMemoryRequirements2KHR = VK_STRUCTURE_TYPE_MEMORY_REQUIREMENTS_2_KHR,
    eSparseImageMemoryRequirements2KHR = VK_STRUCTURE_TYPE_SPARSE_IMAGE_MEMORY_REQUIREMENTS_2_KHR,
    eSamplerYcbcrConversionCreateInfoKHR = VK_STRUCTURE_TYPE_SAMPLER_YCBCR_CONVERSION_CREATE_INFO_KHR,
    eSamplerYcbcrConversionInfoKHR = VK_STRUCTURE_TYPE_SAMPLER_YCBCR_CONVERSION_INFO_KHR,
    eBindImagePlaneMemoryInfoKHR = VK_STRUCTURE_TYPE_BIND_IMAGE_PLANE_MEMORY_INFO_KHR,
    eImagePlaneMemoryRequirementsInfoKHR = VK_STRUCTURE_TYPE_IMAGE_PLANE_MEMORY_REQUIREMENTS_INFO_KHR,
    ePhysicalDeviceSamplerYcbcrConversionFeaturesKHR = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_SAMPLER_YCBCR_CONVERSION_FEATURES_KHR,
    eSamplerYcbcrConversionImageFormatPropertiesKHR = VK_STRUCTURE_TYPE_SAMPLER_YCBCR_CONVERSION_IMAGE_FORMAT_PROPERTIES_KHR,
    eBindBufferMemoryInfoKHR = VK_STRUCTURE_TYPE_BIND_BUFFER_MEMORY_INFO_KHR,
    eBindImageMemoryInfoKHR = VK_STRUCTURE_TYPE_BIND_IMAGE_MEMORY_INFO_KHR,
    ePhysicalDeviceMaintenance3PropertiesKHR = VK_STRUCTURE_TYPE_PHYSICAL_DEVICE_MAINTENANCE_3_PROPERTIES_KHR,
    eDescriptorSetLayoutSupportKHR = VK_STRUCTURE_TYPE_DESCRIPTOR_SET_LAYOUT_SUPPORT_KHR
  };

  VULKAN_HPP_INLINE std::string to_string( StructureType value )
  {
    switch ( value )
    {
      case StructureType::eApplicationInfo : return "ApplicationInfo";
      case StructureType::eInstanceCreateInfo : return "InstanceCreateInfo";
      case StructureType::eDeviceQueueCreateInfo : return "DeviceQueueCreateInfo";
      case StructureType::eDeviceCreateInfo : return "DeviceCreateInfo";
      case StructureType::eSubmitInfo : return "SubmitInfo";
      case StructureType::eMemoryAllocateInfo : return "MemoryAllocateInfo";
      case StructureType::eMappedMemoryRange : return "MappedMemoryRange";
      case StructureType::eBindSparseInfo : return "BindSparseInfo";
      case StructureType::eFenceCreateInfo : return "FenceCreateInfo";
      case StructureType::eSemaphoreCreateInfo : return "SemaphoreCreateInfo";
      case StructureType::eEventCreateInfo : return "EventCreateInfo";
      case StructureType::eQueryPoolCreateInfo : return "QueryPoolCreateInfo";
      case StructureType::eBufferCreateInfo : return "BufferCreateInfo";
      case StructureType::eBufferViewCreateInfo : return "BufferViewCreateInfo";
      case StructureType::eImageCreateInfo : return "ImageCreateInfo";
      case StructureType::eImageViewCreateInfo : return "ImageViewCreateInfo";
      case StructureType::eShaderModuleCreateInfo : return "ShaderModuleCreateInfo";
      case StructureType::ePipelineCacheCreateInfo : return "PipelineCacheCreateInfo";
      case StructureType::ePipelineShaderStageCreateInfo : return "PipelineShaderStageCreateInfo";
      case StructureType::ePipelineVertexInputStateCreateInfo : return "PipelineVertexInputStateCreateInfo";
      case StructureType::ePipelineInputAssemblyStateCreateInfo : return "PipelineInputAssemblyStateCreateInfo";
      case StructureType::ePipelineTessellationStateCreateInfo : return "PipelineTessellationStateCreateInfo";
      case StructureType::ePipelineViewportStateCreateInfo : return "PipelineViewportStateCreateInfo";
      case StructureType::ePipelineRasterizationStateCreateInfo : return "PipelineRasterizationStateCreateInfo";
      case StructureType::ePipelineMultisampleStateCreateInfo : return "PipelineMultisampleStateCreateInfo";
      case StructureType::ePipelineDepthStencilStateCreateInfo : return "PipelineDepthStencilStateCreateInfo";
      case StructureType::ePipelineColorBlendStateCreateInfo : return "PipelineColorBlendStateCreateInfo";
      case StructureType::ePipelineDynamicStateCreateInfo : return "PipelineDynamicStateCreateInfo";
      case StructureType::eGraphicsPipelineCreateInfo : return "GraphicsPipelineCreateInfo";
      case StructureType::eComputePipelineCreateInfo : return "ComputePipelineCreateInfo";
      case StructureType::ePipelineLayoutCreateInfo : return "PipelineLayoutCreateInfo";
      case StructureType::eSamplerCreateInfo : return "SamplerCreateInfo";
      case StructureType::eDescriptorSetLayoutCreateInfo : return "DescriptorSetLayoutCreateInfo";
      case StructureType::eDescriptorPoolCreateInfo : return "DescriptorPoolCreateInfo";
      case StructureType::eDescriptorSetAllocateInfo : return "DescriptorSetAllocateInfo";
      case StructureType::eWriteDescriptorSet : return "WriteDescriptorSet";
      case StructureType::eCopyDescriptorSet : return "CopyDescriptorSet";
      case StructureType::eFramebufferCreateInfo : return "FramebufferCreateInfo";
      case StructureType::eRenderPassCreateInfo : return "RenderPassCreateInfo";
      case StructureType::eCommandPoolCreateInfo : return "CommandPoolCreateInfo";
      case StructureType::eCommandBufferAllocateInfo : return "CommandBufferAllocateInfo";
      case StructureType::eCommandBufferInheritanceInfo : return "CommandBufferInheritanceInfo";
      case StructureType::eCommandBufferBeginInfo : return "CommandBufferBeginInfo";
      case StructureType::eRenderPassBeginInfo : return "RenderPassBeginInfo";
      case StructureType::eBufferMemoryBarrier : return "BufferMemoryBarrier";
      case StructureType::eImageMemoryBarrier : return "ImageMemoryBarrier";
      case StructureType::eMemoryBarrier : return "MemoryBarrier";
      case StructureType::eLoaderInstanceCreateInfo : return "LoaderInstanceCreateInfo";
      case StructureType::eLoaderDeviceCreateInfo : return "LoaderDeviceCreateInfo";
      case StructureType::ePhysicalDeviceSubgroupProperties : return "PhysicalDeviceSubgroupProperties";
      case StructureType::eBindBufferMemoryInfo : return "BindBufferMemoryInfo";
      case StructureType::eBindImageMemoryInfo : return "BindImageMemoryInfo";
      case StructureType::ePhysicalDevice16BitStorageFeatures : return "PhysicalDevice16BitStorageFeatures";
      case StructureType::eMemoryDedicatedRequirements : return "MemoryDedicatedRequirements";
      case StructureType::eMemoryDedicatedAllocateInfo : return "MemoryDedicatedAllocateInfo";
      case StructureType::eMemoryAllocateFlagsInfo : return "MemoryAllocateFlagsInfo";
      case StructureType::eDeviceGroupRenderPassBeginInfo : return "DeviceGroupRenderPassBeginInfo";
      case StructureType::eDeviceGroupCommandBufferBeginInfo : return "DeviceGroupCommandBufferBeginInfo";
      case StructureType::eDeviceGroupSubmitInfo : return "DeviceGroupSubmitInfo";
      case StructureType::eDeviceGroupBindSparseInfo : return "DeviceGroupBindSparseInfo";
      case StructureType::eBindBufferMemoryDeviceGroupInfo : return "BindBufferMemoryDeviceGroupInfo";
      case StructureType::eBindImageMemoryDeviceGroupInfo : return "BindImageMemoryDeviceGroupInfo";
      case StructureType::ePhysicalDeviceGroupProperties : return "PhysicalDeviceGroupProperties";
      case StructureType::eDeviceGroupDeviceCreateInfo : return "DeviceGroupDeviceCreateInfo";
      case StructureType::eBufferMemoryRequirementsInfo2 : return "BufferMemoryRequirementsInfo2";
      case StructureType::eImageMemoryRequirementsInfo2 : return "ImageMemoryRequirementsInfo2";
      case StructureType::eImageSparseMemoryRequirementsInfo2 : return "ImageSparseMemoryRequirementsInfo2";
      case StructureType::eMemoryRequirements2 : return "MemoryRequirements2";
      case StructureType::eSparseImageMemoryRequirements2 : return "SparseImageMemoryRequirements2";
      case StructureType::ePhysicalDeviceFeatures2 : return "PhysicalDeviceFeatures2";
      case StructureType::ePhysicalDeviceProperties2 : return "PhysicalDeviceProperties2";
      case StructureType::eFormatProperties2 : return "FormatProperties2";
      case StructureType::eImageFormatProperties2 : return "ImageFormatProperties2";
      case StructureType::ePhysicalDeviceImageFormatInfo2 : return "PhysicalDeviceImageFormatInfo2";
      case StructureType::eQueueFamilyProperties2 : return "QueueFamilyProperties2";
      case StructureType::ePhysicalDeviceMemoryProperties2 : return "PhysicalDeviceMemoryProperties2";
      case StructureType::eSparseImageFormatProperties2 : return "SparseImageFormatProperties2";
      case StructureType::ePhysicalDeviceSparseImageFormatInfo2 : return "PhysicalDeviceSparseImageFormatInfo2";
      case StructureType::ePhysicalDevicePointClippingProperties : return "PhysicalDevicePointClippingProperties";
      case StructureType::eRenderPassInputAttachmentAspectCreateInfo : return "RenderPassInputAttachmentAspectCreateInfo";
      case StructureType::eImageViewUsageCreateInfo : return "ImageViewUsageCreateInfo";
      case StructureType::ePipelineTessellationDomainOriginStateCreateInfo : return "PipelineTessellationDomainOriginStateCreateInfo";
      case StructureType::eRenderPassMultiviewCreateInfo : return "RenderPassMultiviewCreateInfo";
      case StructureType::ePhysicalDeviceMultiviewFeatures : return "PhysicalDeviceMultiviewFeatures";
      case StructureType::ePhysicalDeviceMultiviewProperties : return "PhysicalDeviceMultiviewProperties";
      case StructureType::ePhysicalDeviceVariablePointerFeatures : return "PhysicalDeviceVariablePointerFeatures";
      case StructureType::eProtectedSubmitInfo : return "ProtectedSubmitInfo";
      case StructureType::ePhysicalDeviceProtectedMemoryFeatures : return "PhysicalDeviceProtectedMemoryFeatures";
      case StructureType::ePhysicalDeviceProtectedMemoryProperties : return "PhysicalDeviceProtectedMemoryProperties";
      case StructureType::eDeviceQueueInfo2 : return "DeviceQueueInfo2";
      case StructureType::eSamplerYcbcrConversionCreateInfo : return "SamplerYcbcrConversionCreateInfo";
      case StructureType::eSamplerYcbcrConversionInfo : return "SamplerYcbcrConversionInfo";
      case StructureType::eBindImagePlaneMemoryInfo : return "BindImagePlaneMemoryInfo";
      case StructureType::eImagePlaneMemoryRequirementsInfo : return "ImagePlaneMemoryRequirementsInfo";
      case StructureType::ePhysicalDeviceSamplerYcbcrConversionFeatures : return "PhysicalDeviceSamplerYcbcrConversionFeatures";
      case StructureType::eSamplerYcbcrConversionImageFormatProperties : return "SamplerYcbcrConversionImageFormatProperties";
      case StructureType::eDescriptorUpdateTemplateCreateInfo : return "DescriptorUpdateTemplateCreateInfo";
      case StructureType::ePhysicalDeviceExternalImageFormatInfo : return "PhysicalDeviceExternalImageFormatInfo";
      case StructureType::eExternalImageFormatProperties : return "ExternalImageFormatProperties";
      case StructureType::ePhysicalDeviceExternalBufferInfo : return "PhysicalDeviceExternalBufferInfo";
      case StructureType::eExternalBufferProperties : return "ExternalBufferProperties";
      case StructureType::ePhysicalDeviceIdProperties : return "PhysicalDeviceIdProperties";
      case StructureType::eExternalMemoryBufferCreateInfo : return "ExternalMemoryBufferCreateInfo";
      case StructureType::eExternalMemoryImageCreateInfo : return "ExternalMemoryImageCreateInfo";
      case StructureType::eExportMemoryAllocateInfo : return "ExportMemoryAllocateInfo";
      case StructureType::ePhysicalDeviceExternalFenceInfo : return "PhysicalDeviceExternalFenceInfo";
      case StructureType::eExternalFenceProperties : return "ExternalFenceProperties";
      case StructureType::eExportFenceCreateInfo : return "ExportFenceCreateInfo";
      case StructureType::eExportSemaphoreCreateInfo : return "ExportSemaphoreCreateInfo";
      case StructureType::ePhysicalDeviceExternalSemaphoreInfo : return "PhysicalDeviceExternalSemaphoreInfo";
      case StructureType::eExternalSemaphoreProperties : return "ExternalSemaphoreProperties";
      case StructureType::ePhysicalDeviceMaintenance3Properties : return "PhysicalDeviceMaintenance3Properties";
      case StructureType::eDescriptorSetLayoutSupport : return "DescriptorSetLayoutSupport";
      case StructureType::ePhysicalDeviceShaderDrawParameterFeatures : return "PhysicalDeviceShaderDrawParameterFeatures";
      case StructureType::eSwapchainCreateInfoKHR : return "SwapchainCreateInfoKHR";
      case StructureType::ePresentInfoKHR : return "PresentInfoKHR";
      case StructureType::eDeviceGroupPresentCapabilitiesKHR : return "DeviceGroupPresentCapabilitiesKHR";
      case StructureType::eImageSwapchainCreateInfoKHR : return "ImageSwapchainCreateInfoKHR";
      case StructureType::eBindImageMemorySwapchainInfoKHR : return "BindImageMemorySwapchainInfoKHR";
      case StructureType::eAcquireNextImageInfoKHR : return "AcquireNextImageInfoKHR";
      case StructureType::eDeviceGroupPresentInfoKHR : return "DeviceGroupPresentInfoKHR";
      case StructureType::eDeviceGroupSwapchainCreateInfoKHR : return "DeviceGroupSwapchainCreateInfoKHR";
      case StructureType::eDisplayModeCreateInfoKHR : return "DisplayModeCreateInfoKHR";
      case StructureType::eDisplaySurfaceCreateInfoKHR : return "DisplaySurfaceCreateInfoKHR";
      case StructureType::eDisplayPresentInfoKHR : return "DisplayPresentInfoKHR";
      case StructureType::eXlibSurfaceCreateInfoKHR : return "XlibSurfaceCreateInfoKHR";
      case StructureType::eXcbSurfaceCreateInfoKHR : return "XcbSurfaceCreateInfoKHR";
      case StructureType::eWaylandSurfaceCreateInfoKHR : return "WaylandSurfaceCreateInfoKHR";
      case StructureType::eAndroidSurfaceCreateInfoKHR : return "AndroidSurfaceCreateInfoKHR";
      case StructureType::eWin32SurfaceCreateInfoKHR : return "Win32SurfaceCreateInfoKHR";
      case StructureType::eDebugReportCallbackCreateInfoEXT : return "DebugReportCallbackCreateInfoEXT";
      case StructureType::ePipelineRasterizationStateRasterizationOrderAMD : return "PipelineRasterizationStateRasterizationOrderAMD";
      case StructureType::eDebugMarkerObjectNameInfoEXT : return "DebugMarkerObjectNameInfoEXT";
      case StructureType::eDebugMarkerObjectTagInfoEXT : return "DebugMarkerObjectTagInfoEXT";
      case StructureType::eDebugMarkerMarkerInfoEXT : return "DebugMarkerMarkerInfoEXT";
      case StructureType::eDedicatedAllocationImageCreateInfoNV : return "DedicatedAllocationImageCreateInfoNV";
      case StructureType::eDedicatedAllocationBufferCreateInfoNV : return "DedicatedAllocationBufferCreateInfoNV";
      case StructureType::eDedicatedAllocationMemoryAllocateInfoNV : return "DedicatedAllocationMemoryAllocateInfoNV";
      case StructureType::ePhysicalDeviceTransformFeedbackFeaturesEXT : return "PhysicalDeviceTransformFeedbackFeaturesEXT";
      case StructureType::ePhysicalDeviceTransformFeedbackPropertiesEXT : return "PhysicalDeviceTransformFeedbackPropertiesEXT";
      case StructureType::ePipelineRasterizationStateStreamCreateInfoEXT : return "PipelineRasterizationStateStreamCreateInfoEXT";
      case StructureType::eTextureLodGatherFormatPropertiesAMD : return "TextureLodGatherFormatPropertiesAMD";
      case StructureType::ePhysicalDeviceCornerSampledImageFeaturesNV : return "PhysicalDeviceCornerSampledImageFeaturesNV";
      case StructureType::eExternalMemoryImageCreateInfoNV : return "ExternalMemoryImageCreateInfoNV";
      case StructureType::eExportMemoryAllocateInfoNV : return "ExportMemoryAllocateInfoNV";
      case StructureType::eImportMemoryWin32HandleInfoNV : return "ImportMemoryWin32HandleInfoNV";
      case StructureType::eExportMemoryWin32HandleInfoNV : return "ExportMemoryWin32HandleInfoNV";
      case StructureType::eWin32KeyedMutexAcquireReleaseInfoNV : return "Win32KeyedMutexAcquireReleaseInfoNV";
      case StructureType::eValidationFlagsEXT : return "ValidationFlagsEXT";
      case StructureType::eViSurfaceCreateInfoNN : return "ViSurfaceCreateInfoNN";
      case StructureType::eImageViewAstcDecodeModeEXT : return "ImageViewAstcDecodeModeEXT";
      case StructureType::ePhysicalDeviceAstcDecodeFeaturesEXT : return "PhysicalDeviceAstcDecodeFeaturesEXT";
      case StructureType::eImportMemoryWin32HandleInfoKHR : return "ImportMemoryWin32HandleInfoKHR";
      case StructureType::eExportMemoryWin32HandleInfoKHR : return "ExportMemoryWin32HandleInfoKHR";
      case StructureType::eMemoryWin32HandlePropertiesKHR : return "MemoryWin32HandlePropertiesKHR";
      case StructureType::eMemoryGetWin32HandleInfoKHR : return "MemoryGetWin32HandleInfoKHR";
      case StructureType::eImportMemoryFdInfoKHR : return "ImportMemoryFdInfoKHR";
      case StructureType::eMemoryFdPropertiesKHR : return "MemoryFdPropertiesKHR";
      case StructureType::eMemoryGetFdInfoKHR : return "MemoryGetFdInfoKHR";
      case StructureType::eWin32KeyedMutexAcquireReleaseInfoKHR : return "Win32KeyedMutexAcquireReleaseInfoKHR";
      case StructureType::eImportSemaphoreWin32HandleInfoKHR : return "ImportSemaphoreWin32HandleInfoKHR";
      case StructureType::eExportSemaphoreWin32HandleInfoKHR : return "ExportSemaphoreWin32HandleInfoKHR";
      case StructureType::eD3D12FenceSubmitInfoKHR : return "D3D12FenceSubmitInfoKHR";
      case StructureType::eSemaphoreGetWin32HandleInfoKHR : return "SemaphoreGetWin32HandleInfoKHR";
      case StructureType::eImportSemaphoreFdInfoKHR : return "ImportSemaphoreFdInfoKHR";
      case StructureType::eSemaphoreGetFdInfoKHR : return "SemaphoreGetFdInfoKHR";
      case StructureType::ePhysicalDevicePushDescriptorPropertiesKHR : return "PhysicalDevicePushDescriptorPropertiesKHR";
      case StructureType::eCommandBufferInheritanceConditionalRenderingInfoEXT : return "CommandBufferInheritanceConditionalRenderingInfoEXT";
      case StructureType::ePhysicalDeviceConditionalRenderingFeaturesEXT : return "PhysicalDeviceConditionalRenderingFeaturesEXT";
      case StructureType::eConditionalRenderingBeginInfoEXT : return "ConditionalRenderingBeginInfoEXT";
      case StructureType::ePhysicalDeviceFloat16Int8FeaturesKHR : return "PhysicalDeviceFloat16Int8FeaturesKHR";
      case StructureType::ePresentRegionsKHR : return "PresentRegionsKHR";
      case StructureType::eObjectTableCreateInfoNVX : return "ObjectTableCreateInfoNVX";
      case StructureType::eIndirectCommandsLayoutCreateInfoNVX : return "IndirectCommandsLayoutCreateInfoNVX";
      case StructureType::eCmdProcessCommandsInfoNVX : return "CmdProcessCommandsInfoNVX";
      case StructureType::eCmdReserveSpaceForCommandsInfoNVX : return "CmdReserveSpaceForCommandsInfoNVX";
      case StructureType::eDeviceGeneratedCommandsLimitsNVX : return "DeviceGeneratedCommandsLimitsNVX";
      case StructureType::eDeviceGeneratedCommandsFeaturesNVX : return "DeviceGeneratedCommandsFeaturesNVX";
      case StructureType::ePipelineViewportWScalingStateCreateInfoNV : return "PipelineViewportWScalingStateCreateInfoNV";
      case StructureType::eSurfaceCapabilities2EXT : return "SurfaceCapabilities2EXT";
      case StructureType::eDisplayPowerInfoEXT : return "DisplayPowerInfoEXT";
      case StructureType::eDeviceEventInfoEXT : return "DeviceEventInfoEXT";
      case StructureType::eDisplayEventInfoEXT : return "DisplayEventInfoEXT";
      case StructureType::eSwapchainCounterCreateInfoEXT : return "SwapchainCounterCreateInfoEXT";
      case StructureType::ePresentTimesInfoGOOGLE : return "PresentTimesInfoGOOGLE";
      case StructureType::ePhysicalDeviceMultiviewPerViewAttributesPropertiesNVX : return "PhysicalDeviceMultiviewPerViewAttributesPropertiesNVX";
      case StructureType::ePipelineViewportSwizzleStateCreateInfoNV : return "PipelineViewportSwizzleStateCreateInfoNV";
      case StructureType::ePhysicalDeviceDiscardRectanglePropertiesEXT : return "PhysicalDeviceDiscardRectanglePropertiesEXT";
      case StructureType::ePipelineDiscardRectangleStateCreateInfoEXT : return "PipelineDiscardRectangleStateCreateInfoEXT";
      case StructureType::ePhysicalDeviceConservativeRasterizationPropertiesEXT : return "PhysicalDeviceConservativeRasterizationPropertiesEXT";
      case StructureType::ePipelineRasterizationConservativeStateCreateInfoEXT : return "PipelineRasterizationConservativeStateCreateInfoEXT";
      case StructureType::ePhysicalDeviceDepthClipEnableFeaturesEXT : return "PhysicalDeviceDepthClipEnableFeaturesEXT";
      case StructureType::ePipelineRasterizationDepthClipStateCreateInfoEXT : return "PipelineRasterizationDepthClipStateCreateInfoEXT";
      case StructureType::eHdrMetadataEXT : return "HdrMetadataEXT";
      case StructureType::eAttachmentDescription2KHR : return "AttachmentDescription2KHR";
      case StructureType::eAttachmentReference2KHR : return "AttachmentReference2KHR";
      case StructureType::eSubpassDescription2KHR : return "SubpassDescription2KHR";
      case StructureType::eSubpassDependency2KHR : return "SubpassDependency2KHR";
      case StructureType::eRenderPassCreateInfo2KHR : return "RenderPassCreateInfo2KHR";
      case StructureType::eSubpassBeginInfoKHR : return "SubpassBeginInfoKHR";
      case StructureType::eSubpassEndInfoKHR : return "SubpassEndInfoKHR";
      case StructureType::eSharedPresentSurfaceCapabilitiesKHR : return "SharedPresentSurfaceCapabilitiesKHR";
      case StructureType::eImportFenceWin32HandleInfoKHR : return "ImportFenceWin32HandleInfoKHR";
      case StructureType::eExportFenceWin32HandleInfoKHR : return "ExportFenceWin32HandleInfoKHR";
      case StructureType::eFenceGetWin32HandleInfoKHR : return "FenceGetWin32HandleInfoKHR";
      case StructureType::eImportFenceFdInfoKHR : return "ImportFenceFdInfoKHR";
      case StructureType::eFenceGetFdInfoKHR : return "FenceGetFdInfoKHR";
      case StructureType::ePhysicalDeviceSurfaceInfo2KHR : return "PhysicalDeviceSurfaceInfo2KHR";
      case StructureType::eSurfaceCapabilities2KHR : return "SurfaceCapabilities2KHR";
      case StructureType::eSurfaceFormat2KHR : return "SurfaceFormat2KHR";
      case StructureType::eDisplayProperties2KHR : return "DisplayProperties2KHR";
      case StructureType::eDisplayPlaneProperties2KHR : return "DisplayPlaneProperties2KHR";
      case StructureType::eDisplayModeProperties2KHR : return "DisplayModeProperties2KHR";
      case StructureType::eDisplayPlaneInfo2KHR : return "DisplayPlaneInfo2KHR";
      case StructureType::eDisplayPlaneCapabilities2KHR : return "DisplayPlaneCapabilities2KHR";
      case StructureType::eIosSurfaceCreateInfoMVK : return "IosSurfaceCreateInfoMVK";
      case StructureType::eMacosSurfaceCreateInfoMVK : return "MacosSurfaceCreateInfoMVK";
      case StructureType::eDebugUtilsObjectNameInfoEXT : return "DebugUtilsObjectNameInfoEXT";
      case StructureType::eDebugUtilsObjectTagInfoEXT : return "DebugUtilsObjectTagInfoEXT";
      case StructureType::eDebugUtilsLabelEXT : return "DebugUtilsLabelEXT";
      case StructureType::eDebugUtilsMessengerCallbackDataEXT : return "DebugUtilsMessengerCallbackDataEXT";
      case StructureType::eDebugUtilsMessengerCreateInfoEXT : return "DebugUtilsMessengerCreateInfoEXT";
      case StructureType::eAndroidHardwareBufferUsageANDROID : return "AndroidHardwareBufferUsageANDROID";
      case StructureType::eAndroidHardwareBufferPropertiesANDROID : return "AndroidHardwareBufferPropertiesANDROID";
      case StructureType::eAndroidHardwareBufferFormatPropertiesANDROID : return "AndroidHardwareBufferFormatPropertiesANDROID";
      case StructureType::eImportAndroidHardwareBufferInfoANDROID : return "ImportAndroidHardwareBufferInfoANDROID";
      case StructureType::eMemoryGetAndroidHardwareBufferInfoANDROID : return "MemoryGetAndroidHardwareBufferInfoANDROID";
      case StructureType::eExternalFormatANDROID : return "ExternalFormatANDROID";
      case StructureType::ePhysicalDeviceSamplerFilterMinmaxPropertiesEXT : return "PhysicalDeviceSamplerFilterMinmaxPropertiesEXT";
      case StructureType::eSamplerReductionModeCreateInfoEXT : return "SamplerReductionModeCreateInfoEXT";
      case StructureType::ePhysicalDeviceInlineUniformBlockFeaturesEXT : return "PhysicalDeviceInlineUniformBlockFeaturesEXT";
      case StructureType::ePhysicalDeviceInlineUniformBlockPropertiesEXT : return "PhysicalDeviceInlineUniformBlockPropertiesEXT";
      case StructureType::eWriteDescriptorSetInlineUniformBlockEXT : return "WriteDescriptorSetInlineUniformBlockEXT";
      case StructureType::eDescriptorPoolInlineUniformBlockCreateInfoEXT : return "DescriptorPoolInlineUniformBlockCreateInfoEXT";
      case StructureType::eSampleLocationsInfoEXT : return "SampleLocationsInfoEXT";
      case StructureType::eRenderPassSampleLocationsBeginInfoEXT : return "RenderPassSampleLocationsBeginInfoEXT";
      case StructureType::ePipelineSampleLocationsStateCreateInfoEXT : return "PipelineSampleLocationsStateCreateInfoEXT";
      case StructureType::ePhysicalDeviceSampleLocationsPropertiesEXT : return "PhysicalDeviceSampleLocationsPropertiesEXT";
      case StructureType::eMultisamplePropertiesEXT : return "MultisamplePropertiesEXT";
      case StructureType::eImageFormatListCreateInfoKHR : return "ImageFormatListCreateInfoKHR";
      case StructureType::ePhysicalDeviceBlendOperationAdvancedFeaturesEXT : return "PhysicalDeviceBlendOperationAdvancedFeaturesEXT";
      case StructureType::ePhysicalDeviceBlendOperationAdvancedPropertiesEXT : return "PhysicalDeviceBlendOperationAdvancedPropertiesEXT";
      case StructureType::ePipelineColorBlendAdvancedStateCreateInfoEXT : return "PipelineColorBlendAdvancedStateCreateInfoEXT";
      case StructureType::ePipelineCoverageToColorStateCreateInfoNV : return "PipelineCoverageToColorStateCreateInfoNV";
      case StructureType::ePipelineCoverageModulationStateCreateInfoNV : return "PipelineCoverageModulationStateCreateInfoNV";
      case StructureType::eDrmFormatModifierPropertiesListEXT : return "DrmFormatModifierPropertiesListEXT";
      case StructureType::eDrmFormatModifierPropertiesEXT : return "DrmFormatModifierPropertiesEXT";
      case StructureType::ePhysicalDeviceImageDrmFormatModifierInfoEXT : return "PhysicalDeviceImageDrmFormatModifierInfoEXT";
      case StructureType::eImageDrmFormatModifierListCreateInfoEXT : return "ImageDrmFormatModifierListCreateInfoEXT";
      case StructureType::eImageDrmFormatModifierExplicitCreateInfoEXT : return "ImageDrmFormatModifierExplicitCreateInfoEXT";
      case StructureType::eImageDrmFormatModifierPropertiesEXT : return "ImageDrmFormatModifierPropertiesEXT";
      case StructureType::eValidationCacheCreateInfoEXT : return "ValidationCacheCreateInfoEXT";
      case StructureType::eShaderModuleValidationCacheCreateInfoEXT : return "ShaderModuleValidationCacheCreateInfoEXT";
      case StructureType::eDescriptorSetLayoutBindingFlagsCreateInfoEXT : return "DescriptorSetLayoutBindingFlagsCreateInfoEXT";
      case StructureType::ePhysicalDeviceDescriptorIndexingFeaturesEXT : return "PhysicalDeviceDescriptorIndexingFeaturesEXT";
      case StructureType::ePhysicalDeviceDescriptorIndexingPropertiesEXT : return "PhysicalDeviceDescriptorIndexingPropertiesEXT";
      case StructureType::eDescriptorSetVariableDescriptorCountAllocateInfoEXT : return "DescriptorSetVariableDescriptorCountAllocateInfoEXT";
      case StructureType::eDescriptorSetVariableDescriptorCountLayoutSupportEXT : return "DescriptorSetVariableDescriptorCountLayoutSupportEXT";
      case StructureType::ePipelineViewportShadingRateImageStateCreateInfoNV : return "PipelineViewportShadingRateImageStateCreateInfoNV";
      case StructureType::ePhysicalDeviceShadingRateImageFeaturesNV : return "PhysicalDeviceShadingRateImageFeaturesNV";
      case StructureType::ePhysicalDeviceShadingRateImagePropertiesNV : return "PhysicalDeviceShadingRateImagePropertiesNV";
      case StructureType::ePipelineViewportCoarseSampleOrderStateCreateInfoNV : return "PipelineViewportCoarseSampleOrderStateCreateInfoNV";
      case StructureType::eRayTracingPipelineCreateInfoNV : return "RayTracingPipelineCreateInfoNV";
      case StructureType::eAccelerationStructureCreateInfoNV : return "AccelerationStructureCreateInfoNV";
      case StructureType::eGeometryNV : return "GeometryNV";
      case StructureType::eGeometryTrianglesNV : return "GeometryTrianglesNV";
      case StructureType::eGeometryAabbNV : return "GeometryAabbNV";
      case StructureType::eBindAccelerationStructureMemoryInfoNV : return "BindAccelerationStructureMemoryInfoNV";
      case StructureType::eWriteDescriptorSetAccelerationStructureNV : return "WriteDescriptorSetAccelerationStructureNV";
      case StructureType::eAccelerationStructureMemoryRequirementsInfoNV : return "AccelerationStructureMemoryRequirementsInfoNV";
      case StructureType::ePhysicalDeviceRayTracingPropertiesNV : return "PhysicalDeviceRayTracingPropertiesNV";
      case StructureType::eRayTracingShaderGroupCreateInfoNV : return "RayTracingShaderGroupCreateInfoNV";
      case StructureType::eAccelerationStructureInfoNV : return "AccelerationStructureInfoNV";
      case StructureType::ePhysicalDeviceRepresentativeFragmentTestFeaturesNV : return "PhysicalDeviceRepresentativeFragmentTestFeaturesNV";
      case StructureType::ePipelineRepresentativeFragmentTestStateCreateInfoNV : return "PipelineRepresentativeFragmentTestStateCreateInfoNV";
      case StructureType::ePhysicalDeviceImageViewImageFormatInfoEXT : return "PhysicalDeviceImageViewImageFormatInfoEXT";
      case StructureType::eFilterCubicImageViewImageFormatPropertiesEXT : return "FilterCubicImageViewImageFormatPropertiesEXT";
      case StructureType::eDeviceQueueGlobalPriorityCreateInfoEXT : return "DeviceQueueGlobalPriorityCreateInfoEXT";
      case StructureType::ePhysicalDevice8BitStorageFeaturesKHR : return "PhysicalDevice8BitStorageFeaturesKHR";
      case StructureType::eImportMemoryHostPointerInfoEXT : return "ImportMemoryHostPointerInfoEXT";
      case StructureType::eMemoryHostPointerPropertiesEXT : return "MemoryHostPointerPropertiesEXT";
      case StructureType::ePhysicalDeviceExternalMemoryHostPropertiesEXT : return "PhysicalDeviceExternalMemoryHostPropertiesEXT";
      case StructureType::ePhysicalDeviceShaderAtomicInt64FeaturesKHR : return "PhysicalDeviceShaderAtomicInt64FeaturesKHR";
      case StructureType::eCalibratedTimestampInfoEXT : return "CalibratedTimestampInfoEXT";
      case StructureType::ePhysicalDeviceShaderCorePropertiesAMD : return "PhysicalDeviceShaderCorePropertiesAMD";
      case StructureType::eDeviceMemoryOverallocationCreateInfoAMD : return "DeviceMemoryOverallocationCreateInfoAMD";
      case StructureType::ePhysicalDeviceVertexAttributeDivisorPropertiesEXT : return "PhysicalDeviceVertexAttributeDivisorPropertiesEXT";
      case StructureType::ePipelineVertexInputDivisorStateCreateInfoEXT : return "PipelineVertexInputDivisorStateCreateInfoEXT";
      case StructureType::ePhysicalDeviceVertexAttributeDivisorFeaturesEXT : return "PhysicalDeviceVertexAttributeDivisorFeaturesEXT";
      case StructureType::ePhysicalDeviceDriverPropertiesKHR : return "PhysicalDeviceDriverPropertiesKHR";
      case StructureType::ePhysicalDeviceFloatControlsPropertiesKHR : return "PhysicalDeviceFloatControlsPropertiesKHR";
      case StructureType::ePhysicalDeviceDepthStencilResolvePropertiesKHR : return "PhysicalDeviceDepthStencilResolvePropertiesKHR";
      case StructureType::eSubpassDescriptionDepthStencilResolveKHR : return "SubpassDescriptionDepthStencilResolveKHR";
      case StructureType::ePhysicalDeviceComputeShaderDerivativesFeaturesNV : return "PhysicalDeviceComputeShaderDerivativesFeaturesNV";
      case StructureType::ePhysicalDeviceMeshShaderFeaturesNV : return "PhysicalDeviceMeshShaderFeaturesNV";
      case StructureType::ePhysicalDeviceMeshShaderPropertiesNV : return "PhysicalDeviceMeshShaderPropertiesNV";
      case StructureType::ePhysicalDeviceFragmentShaderBarycentricFeaturesNV : return "PhysicalDeviceFragmentShaderBarycentricFeaturesNV";
      case StructureType::ePhysicalDeviceShaderImageFootprintFeaturesNV : return "PhysicalDeviceShaderImageFootprintFeaturesNV";
      case StructureType::ePipelineViewportExclusiveScissorStateCreateInfoNV : return "PipelineViewportExclusiveScissorStateCreateInfoNV";
      case StructureType::ePhysicalDeviceExclusiveScissorFeaturesNV : return "PhysicalDeviceExclusiveScissorFeaturesNV";
      case StructureType::eCheckpointDataNV : return "CheckpointDataNV";
      case StructureType::eQueueFamilyCheckpointPropertiesNV : return "QueueFamilyCheckpointPropertiesNV";
      case StructureType::ePhysicalDeviceVulkanMemoryModelFeaturesKHR : return "PhysicalDeviceVulkanMemoryModelFeaturesKHR";
      case StructureType::ePhysicalDevicePciBusInfoPropertiesEXT : return "PhysicalDevicePciBusInfoPropertiesEXT";
      case StructureType::eImagepipeSurfaceCreateInfoFUCHSIA : return "ImagepipeSurfaceCreateInfoFUCHSIA";
      case StructureType::ePhysicalDeviceFragmentDensityMapFeaturesEXT : return "PhysicalDeviceFragmentDensityMapFeaturesEXT";
      case StructureType::ePhysicalDeviceFragmentDensityMapPropertiesEXT : return "PhysicalDeviceFragmentDensityMapPropertiesEXT";
      case StructureType::eRenderPassFragmentDensityMapCreateInfoEXT : return "RenderPassFragmentDensityMapCreateInfoEXT";
      case StructureType::ePhysicalDeviceScalarBlockLayoutFeaturesEXT : return "PhysicalDeviceScalarBlockLayoutFeaturesEXT";
      case StructureType::ePhysicalDeviceMemoryBudgetPropertiesEXT : return "PhysicalDeviceMemoryBudgetPropertiesEXT";
      case StructureType::ePhysicalDeviceMemoryPriorityFeaturesEXT : return "PhysicalDeviceMemoryPriorityFeaturesEXT";
      case StructureType::eMemoryPriorityAllocateInfoEXT : return "MemoryPriorityAllocateInfoEXT";
      case StructureType::ePhysicalDeviceDedicatedAllocationImageAliasingFeaturesNV : return "PhysicalDeviceDedicatedAllocationImageAliasingFeaturesNV";
      case StructureType::ePhysicalDeviceBufferAddressFeaturesEXT : return "PhysicalDeviceBufferAddressFeaturesEXT";
      case StructureType::eBufferDeviceAddressInfoEXT : return "BufferDeviceAddressInfoEXT";
      case StructureType::eBufferDeviceAddressCreateInfoEXT : return "BufferDeviceAddressCreateInfoEXT";
      case StructureType::eImageStencilUsageCreateInfoEXT : return "ImageStencilUsageCreateInfoEXT";
      case StructureType::eValidationFeaturesEXT : return "ValidationFeaturesEXT";
      case StructureType::ePhysicalDeviceCooperativeMatrixFeaturesNV : return "PhysicalDeviceCooperativeMatrixFeaturesNV";
      case StructureType::eCooperativeMatrixPropertiesNV : return "CooperativeMatrixPropertiesNV";
      case StructureType::ePhysicalDeviceCooperativeMatrixPropertiesNV : return "PhysicalDeviceCooperativeMatrixPropertiesNV";
      default: return "invalid";
    }
  }

  enum class SubpassContents
  {
    eInline = VK_SUBPASS_CONTENTS_INLINE,
    eSecondaryCommandBuffers = VK_SUBPASS_CONTENTS_SECONDARY_COMMAND_BUFFERS
  };

  VULKAN_HPP_INLINE std::string to_string( SubpassContents value )
  {
    switch ( value )
    {
      case SubpassContents::eInline : return "Inline";
      case SubpassContents::eSecondaryCommandBuffers : return "SecondaryCommandBuffers";
      default: return "invalid";
    }
  }

  enum class SystemAllocationScope
  {
    eCommand = VK_SYSTEM_ALLOCATION_SCOPE_COMMAND,
    eObject = VK_SYSTEM_ALLOCATION_SCOPE_OBJECT,
    eCache = VK_SYSTEM_ALLOCATION_SCOPE_CACHE,
    eDevice = VK_SYSTEM_ALLOCATION_SCOPE_DEVICE,
    eInstance = VK_SYSTEM_ALLOCATION_SCOPE_INSTANCE
  };

  VULKAN_HPP_INLINE std::string to_string( SystemAllocationScope value )
  {
    switch ( value )
    {
      case SystemAllocationScope::eCommand : return "Command";
      case SystemAllocationScope::eObject : return "Object";
      case SystemAllocationScope::eCache : return "Cache";
      case SystemAllocationScope::eDevice : return "Device";
      case SystemAllocationScope::eInstance : return "Instance";
      default: return "invalid";
    }
  }

  enum class TessellationDomainOrigin
  {
    eUpperLeft = VK_TESSELLATION_DOMAIN_ORIGIN_UPPER_LEFT,
    eLowerLeft = VK_TESSELLATION_DOMAIN_ORIGIN_LOWER_LEFT,
    eUpperLeftKHR = VK_TESSELLATION_DOMAIN_ORIGIN_UPPER_LEFT_KHR,
    eLowerLeftKHR = VK_TESSELLATION_DOMAIN_ORIGIN_LOWER_LEFT_KHR
  };

  VULKAN_HPP_INLINE std::string to_string( TessellationDomainOrigin value )
  {
    switch ( value )
    {
      case TessellationDomainOrigin::eUpperLeft : return "UpperLeft";
      case TessellationDomainOrigin::eLowerLeft : return "LowerLeft";
      default: return "invalid";
    }
  }

  enum class TimeDomainEXT
  {
    eDevice = VK_TIME_DOMAIN_DEVICE_EXT,
    eClockMonotonic = VK_TIME_DOMAIN_CLOCK_MONOTONIC_EXT,
    eClockMonotonicRaw = VK_TIME_DOMAIN_CLOCK_MONOTONIC_RAW_EXT,
    eQueryPerformanceCounter = VK_TIME_DOMAIN_QUERY_PERFORMANCE_COUNTER_EXT
  };

  VULKAN_HPP_INLINE std::string to_string( TimeDomainEXT value )
  {
    switch ( value )
    {
      case TimeDomainEXT::eDevice : return "Device";
      case TimeDomainEXT::eClockMonotonic : return "ClockMonotonic";
      case TimeDomainEXT::eClockMonotonicRaw : return "ClockMonotonicRaw";
      case TimeDomainEXT::eQueryPerformanceCounter : return "QueryPerformanceCounter";
      default: return "invalid";
    }
  }

  enum class ValidationCacheHeaderVersionEXT
  {
    eOne = VK_VALIDATION_CACHE_HEADER_VERSION_ONE_EXT
  };

  VULKAN_HPP_INLINE std::string to_string( ValidationCacheHeaderVersionEXT value )
  {
    switch ( value )
    {
      case ValidationCacheHeaderVersionEXT::eOne : return "One";
      default: return "invalid";
    }
  }

  enum class ValidationCheckEXT
  {
    eAll = VK_VALIDATION_CHECK_ALL_EXT,
    eShaders = VK_VALIDATION_CHECK_SHADERS_EXT
  };

  VULKAN_HPP_INLINE std::string to_string( ValidationCheckEXT value )
  {
    switch ( value )
    {
      case ValidationCheckEXT::eAll : return "All";
      case ValidationCheckEXT::eShaders : return "Shaders";
      default: return "invalid";
    }
  }

  enum class ValidationFeatureDisableEXT
  {
    eAll = VK_VALIDATION_FEATURE_DISABLE_ALL_EXT,
    eShaders = VK_VALIDATION_FEATURE_DISABLE_SHADERS_EXT,
    eThreadSafety = VK_VALIDATION_FEATURE_DISABLE_THREAD_SAFETY_EXT,
    eApiParameters = VK_VALIDATION_FEATURE_DISABLE_API_PARAMETERS_EXT,
    eObjectLifetimes = VK_VALIDATION_FEATURE_DISABLE_OBJECT_LIFETIMES_EXT,
    eCoreChecks = VK_VALIDATION_FEATURE_DISABLE_CORE_CHECKS_EXT,
    eUniqueHandles = VK_VALIDATION_FEATURE_DISABLE_UNIQUE_HANDLES_EXT
  };

  VULKAN_HPP_INLINE std::string to_string( ValidationFeatureDisableEXT value )
  {
    switch ( value )
    {
      case ValidationFeatureDisableEXT::eAll : return "All";
      case ValidationFeatureDisableEXT::eShaders : return "Shaders";
      case ValidationFeatureDisableEXT::eThreadSafety : return "ThreadSafety";
      case ValidationFeatureDisableEXT::eApiParameters : return "ApiParameters";
      case ValidationFeatureDisableEXT::eObjectLifetimes : return "ObjectLifetimes";
      case ValidationFeatureDisableEXT::eCoreChecks : return "CoreChecks";
      case ValidationFeatureDisableEXT::eUniqueHandles : return "UniqueHandles";
      default: return "invalid";
    }
  }

  enum class ValidationFeatureEnableEXT
  {
    eGpuAssisted = VK_VALIDATION_FEATURE_ENABLE_GPU_ASSISTED_EXT,
    eGpuAssistedReserveBindingSlot = VK_VALIDATION_FEATURE_ENABLE_GPU_ASSISTED_RESERVE_BINDING_SLOT_EXT
  };

  VULKAN_HPP_INLINE std::string to_string( ValidationFeatureEnableEXT value )
  {
    switch ( value )
    {
      case ValidationFeatureEnableEXT::eGpuAssisted : return "GpuAssisted";
      case ValidationFeatureEnableEXT::eGpuAssistedReserveBindingSlot : return "GpuAssistedReserveBindingSlot";
      default: return "invalid";
    }
  }

  enum class VendorId
  {
    eVIV = VK_VENDOR_ID_VIV,
    eVSI = VK_VENDOR_ID_VSI,
    eKazan = VK_VENDOR_ID_KAZAN
  };

  VULKAN_HPP_INLINE std::string to_string( VendorId value )
  {
    switch ( value )
    {
      case VendorId::eVIV : return "VIV";
      case VendorId::eVSI : return "VSI";
      case VendorId::eKazan : return "Kazan";
      default: return "invalid";
    }
  }

  enum class VertexInputRate
  {
    eVertex = VK_VERTEX_INPUT_RATE_VERTEX,
    eInstance = VK_VERTEX_INPUT_RATE_INSTANCE
  };

  VULKAN_HPP_INLINE std::string to_string( VertexInputRate value )
  {
    switch ( value )
    {
      case VertexInputRate::eVertex : return "Vertex";
      case VertexInputRate::eInstance : return "Instance";
      default: return "invalid";
    }
  }

  enum class ViewportCoordinateSwizzleNV
  {
    ePositiveX = VK_VIEWPORT_COORDINATE_SWIZZLE_POSITIVE_X_NV,
    eNegativeX = VK_VIEWPORT_COORDINATE_SWIZZLE_NEGATIVE_X_NV,
    ePositiveY = VK_VIEWPORT_COORDINATE_SWIZZLE_POSITIVE_Y_NV,
    eNegativeY = VK_VIEWPORT_COORDINATE_SWIZZLE_NEGATIVE_Y_NV,
    ePositiveZ = VK_VIEWPORT_COORDINATE_SWIZZLE_POSITIVE_Z_NV,
    eNegativeZ = VK_VIEWPORT_COORDINATE_SWIZZLE_NEGATIVE_Z_NV,
    ePositiveW = VK_VIEWPORT_COORDINATE_SWIZZLE_POSITIVE_W_NV,
    eNegativeW = VK_VIEWPORT_COORDINATE_SWIZZLE_NEGATIVE_W_NV
  };

  VULKAN_HPP_INLINE std::string to_string( ViewportCoordinateSwizzleNV value )
  {
    switch ( value )
    {
      case ViewportCoordinateSwizzleNV::ePositiveX : return "PositiveX";
      case ViewportCoordinateSwizzleNV::eNegativeX : return "NegativeX";
      case ViewportCoordinateSwizzleNV::ePositiveY : return "PositiveY";
      case ViewportCoordinateSwizzleNV::eNegativeY : return "NegativeY";
      case ViewportCoordinateSwizzleNV::ePositiveZ : return "PositiveZ";
      case ViewportCoordinateSwizzleNV::eNegativeZ : return "NegativeZ";
      case ViewportCoordinateSwizzleNV::ePositiveW : return "PositiveW";
      case ViewportCoordinateSwizzleNV::eNegativeW : return "NegativeW";
      default: return "invalid";
    }
  }

  enum class AccessFlagBits
  {
    eIndirectCommandRead = VK_ACCESS_INDIRECT_COMMAND_READ_BIT,
    eIndexRead = VK_ACCESS_INDEX_READ_BIT,
    eVertexAttributeRead = VK_ACCESS_VERTEX_ATTRIBUTE_READ_BIT,
    eUniformRead = VK_ACCESS_UNIFORM_READ_BIT,
    eInputAttachmentRead = VK_ACCESS_INPUT_ATTACHMENT_READ_BIT,
    eShaderRead = VK_ACCESS_SHADER_READ_BIT,
    eShaderWrite = VK_ACCESS_SHADER_WRITE_BIT,
    eColorAttachmentRead = VK_ACCESS_COLOR_ATTACHMENT_READ_BIT,
    eColorAttachmentWrite = VK_ACCESS_COLOR_ATTACHMENT_WRITE_BIT,
    eDepthStencilAttachmentRead = VK_ACCESS_DEPTH_STENCIL_ATTACHMENT_READ_BIT,
    eDepthStencilAttachmentWrite = VK_ACCESS_DEPTH_STENCIL_ATTACHMENT_WRITE_BIT,
    eTransferRead = VK_ACCESS_TRANSFER_READ_BIT,
    eTransferWrite = VK_ACCESS_TRANSFER_WRITE_BIT,
    eHostRead = VK_ACCESS_HOST_READ_BIT,
    eHostWrite = VK_ACCESS_HOST_WRITE_BIT,
    eMemoryRead = VK_ACCESS_MEMORY_READ_BIT,
    eMemoryWrite = VK_ACCESS_MEMORY_WRITE_BIT,
    eTransformFeedbackWriteEXT = VK_ACCESS_TRANSFORM_FEEDBACK_WRITE_BIT_EXT,
    eTransformFeedbackCounterReadEXT = VK_ACCESS_TRANSFORM_FEEDBACK_COUNTER_READ_BIT_EXT,
    eTransformFeedbackCounterWriteEXT = VK_ACCESS_TRANSFORM_FEEDBACK_COUNTER_WRITE_BIT_EXT,
    eConditionalRenderingReadEXT = VK_ACCESS_CONDITIONAL_RENDERING_READ_BIT_EXT,
    eCommandProcessReadNVX = VK_ACCESS_COMMAND_PROCESS_READ_BIT_NVX,
    eCommandProcessWriteNVX = VK_ACCESS_COMMAND_PROCESS_WRITE_BIT_NVX,
    eColorAttachmentReadNoncoherentEXT = VK_ACCESS_COLOR_ATTACHMENT_READ_NONCOHERENT_BIT_EXT,
    eShadingRateImageReadNV = VK_ACCESS_SHADING_RATE_IMAGE_READ_BIT_NV,
    eAccelerationStructureReadNV = VK_ACCESS_ACCELERATION_STRUCTURE_READ_BIT_NV,
    eAccelerationStructureWriteNV = VK_ACCESS_ACCELERATION_STRUCTURE_WRITE_BIT_NV,
    eFragmentDensityMapReadEXT = VK_ACCESS_FRAGMENT_DENSITY_MAP_READ_BIT_EXT
  };

  VULKAN_HPP_INLINE std::string to_string( AccessFlagBits value )
  {
    switch ( value )
    {
      case AccessFlagBits::eIndirectCommandRead : return "IndirectCommandRead";
      case AccessFlagBits::eIndexRead : return "IndexRead";
      case AccessFlagBits::eVertexAttributeRead : return "VertexAttributeRead";
      case AccessFlagBits::eUniformRead : return "UniformRead";
      case AccessFlagBits::eInputAttachmentRead : return "InputAttachmentRead";
      case AccessFlagBits::eShaderRead : return "ShaderRead";
      case AccessFlagBits::eShaderWrite : return "ShaderWrite";
      case AccessFlagBits::eColorAttachmentRead : return "ColorAttachmentRead";
      case AccessFlagBits::eColorAttachmentWrite : return "ColorAttachmentWrite";
      case AccessFlagBits::eDepthStencilAttachmentRead : return "DepthStencilAttachmentRead";
      case AccessFlagBits::eDepthStencilAttachmentWrite : return "DepthStencilAttachmentWrite";
      case AccessFlagBits::eTransferRead : return "TransferRead";
      case AccessFlagBits::eTransferWrite : return "TransferWrite";
      case AccessFlagBits::eHostRead : return "HostRead";
      case AccessFlagBits::eHostWrite : return "HostWrite";
      case AccessFlagBits::eMemoryRead : return "MemoryRead";
      case AccessFlagBits::eMemoryWrite : return "MemoryWrite";
      case AccessFlagBits::eTransformFeedbackWriteEXT : return "TransformFeedbackWriteEXT";
      case AccessFlagBits::eTransformFeedbackCounterReadEXT : return "TransformFeedbackCounterReadEXT";
      case AccessFlagBits::eTransformFeedbackCounterWriteEXT : return "TransformFeedbackCounterWriteEXT";
      case AccessFlagBits::eConditionalRenderingReadEXT : return "ConditionalRenderingReadEXT";
      case AccessFlagBits::eCommandProcessReadNVX : return "CommandProcessReadNVX";
      case AccessFlagBits::eCommandProcessWriteNVX : return "CommandProcessWriteNVX";
      case AccessFlagBits::eColorAttachmentReadNoncoherentEXT : return "ColorAttachmentReadNoncoherentEXT";
      case AccessFlagBits::eShadingRateImageReadNV : return "ShadingRateImageReadNV";
      case AccessFlagBits::eAccelerationStructureReadNV : return "AccelerationStructureReadNV";
      case AccessFlagBits::eAccelerationStructureWriteNV : return "AccelerationStructureWriteNV";
      case AccessFlagBits::eFragmentDensityMapReadEXT : return "FragmentDensityMapReadEXT";
      default: return "invalid";
    }
  }

  using AccessFlags = Flags<AccessFlagBits, VkAccessFlags>;

  VULKAN_HPP_INLINE AccessFlags operator|( AccessFlagBits bit0, AccessFlagBits bit1 )
  {
    return AccessFlags( bit0 ) | bit1;
  }

  VULKAN_HPP_INLINE AccessFlags operator~( AccessFlagBits bits )
  {
    return ~( AccessFlags( bits ) );
  }

  template <> struct FlagTraits<AccessFlagBits>
  {
    enum
    {
      allFlags = VkFlags(AccessFlagBits::eIndirectCommandRead) | VkFlags(AccessFlagBits::eIndexRead) | VkFlags(AccessFlagBits::eVertexAttributeRead) | VkFlags(AccessFlagBits::eUniformRead) | VkFlags(AccessFlagBits::eInputAttachmentRead) | VkFlags(AccessFlagBits::eShaderRead) | VkFlags(AccessFlagBits::eShaderWrite) | VkFlags(AccessFlagBits::eColorAttachmentRead) | VkFlags(AccessFlagBits::eColorAttachmentWrite) | VkFlags(AccessFlagBits::eDepthStencilAttachmentRead) | VkFlags(AccessFlagBits::eDepthStencilAttachmentWrite) | VkFlags(AccessFlagBits::eTransferRead) | VkFlags(AccessFlagBits::eTransferWrite) | VkFlags(AccessFlagBits::eHostRead) | VkFlags(AccessFlagBits::eHostWrite) | VkFlags(AccessFlagBits::eMemoryRead) | VkFlags(AccessFlagBits::eMemoryWrite) | VkFlags(AccessFlagBits::eTransformFeedbackWriteEXT) | VkFlags(AccessFlagBits::eTransformFeedbackCounterReadEXT) | VkFlags(AccessFlagBits::eTransformFeedbackCounterWriteEXT) | VkFlags(AccessFlagBits::eConditionalRenderingReadEXT) | VkFlags(AccessFlagBits::eCommandProcessReadNVX) | VkFlags(AccessFlagBits::eCommandProcessWriteNVX) | VkFlags(AccessFlagBits::eColorAttachmentReadNoncoherentEXT) | VkFlags(AccessFlagBits::eShadingRateImageReadNV) | VkFlags(AccessFlagBits::eAccelerationStructureReadNV) | VkFlags(AccessFlagBits::eAccelerationStructureWriteNV) | VkFlags(AccessFlagBits::eFragmentDensityMapReadEXT)
    };
  };

  VULKAN_HPP_INLINE std::string to_string( AccessFlags value  )
  {
    if ( !value ) return "{}";
    std::string result;

    if ( value & AccessFlagBits::eIndirectCommandRead ) result += "IndirectCommandRead | ";
    if ( value & AccessFlagBits::eIndexRead ) result += "IndexRead | ";
    if ( value & AccessFlagBits::eVertexAttributeRead ) result += "VertexAttributeRead | ";
    if ( value & AccessFlagBits::eUniformRead ) result += "UniformRead | ";
    if ( value & AccessFlagBits::eInputAttachmentRead ) result += "InputAttachmentRead | ";
    if ( value & AccessFlagBits::eShaderRead ) result += "ShaderRead | ";
    if ( value & AccessFlagBits::eShaderWrite ) result += "ShaderWrite | ";
    if ( value & AccessFlagBits::eColorAttachmentRead ) result += "ColorAttachmentRead | ";
    if ( value & AccessFlagBits::eColorAttachmentWrite ) result += "ColorAttachmentWrite | ";
    if ( value & AccessFlagBits::eDepthStencilAttachmentRead ) result += "DepthStencilAttachmentRead | ";
    if ( value & AccessFlagBits::eDepthStencilAttachmentWrite ) result += "DepthStencilAttachmentWrite | ";
    if ( value & AccessFlagBits::eTransferRead ) result += "TransferRead | ";
    if ( value & AccessFlagBits::eTransferWrite ) result += "TransferWrite | ";
    if ( value & AccessFlagBits::eHostRead ) result += "HostRead | ";
    if ( value & AccessFlagBits::eHostWrite ) result += "HostWrite | ";
    if ( value & AccessFlagBits::eMemoryRead ) result += "MemoryRead | ";
    if ( value & AccessFlagBits::eMemoryWrite ) result += "MemoryWrite | ";
    if ( value & AccessFlagBits::eTransformFeedbackWriteEXT ) result += "TransformFeedbackWriteEXT | ";
    if ( value & AccessFlagBits::eTransformFeedbackCounterReadEXT ) result += "TransformFeedbackCounterReadEXT | ";
    if ( value & AccessFlagBits::eTransformFeedbackCounterWriteEXT ) result += "TransformFeedbackCounterWriteEXT | ";
    if ( value & AccessFlagBits::eConditionalRenderingReadEXT ) result += "ConditionalRenderingReadEXT | ";
    if ( value & AccessFlagBits::eCommandProcessReadNVX ) result += "CommandProcessReadNVX | ";
    if ( value & AccessFlagBits::eCommandProcessWriteNVX ) result += "CommandProcessWriteNVX | ";
    if ( value & AccessFlagBits::eColorAttachmentReadNoncoherentEXT ) result += "ColorAttachmentReadNoncoherentEXT | ";
    if ( value & AccessFlagBits::eShadingRateImageReadNV ) result += "ShadingRateImageReadNV | ";
    if ( value & AccessFlagBits::eAccelerationStructureReadNV ) result += "AccelerationStructureReadNV | ";
    if ( value & AccessFlagBits::eAccelerationStructureWriteNV ) result += "AccelerationStructureWriteNV | ";
    if ( value & AccessFlagBits::eFragmentDensityMapReadEXT ) result += "FragmentDensityMapReadEXT";
    return result;
  }

#ifdef VK_USE_PLATFORM_ANDROID_KHR
  enum class AndroidSurfaceCreateFlagBitsKHR
  {};

  VULKAN_HPP_INLINE std::string to_string( AndroidSurfaceCreateFlagBitsKHR )
  {
    return "(void)";
  }

  using AndroidSurfaceCreateFlagsKHR = Flags<AndroidSurfaceCreateFlagBitsKHR, VkAndroidSurfaceCreateFlagsKHR>;

  VULKAN_HPP_INLINE std::string to_string( AndroidSurfaceCreateFlagsKHR  )
  {
    return "{}";
  }
#endif /*VK_USE_PLATFORM_ANDROID_KHR*/

  enum class AttachmentDescriptionFlagBits
  {
    eMayAlias = VK_ATTACHMENT_DESCRIPTION_MAY_ALIAS_BIT
  };

  VULKAN_HPP_INLINE std::string to_string( AttachmentDescriptionFlagBits value )
  {
    switch ( value )
    {
      case AttachmentDescriptionFlagBits::eMayAlias : return "MayAlias";
      default: return "invalid";
    }
  }

  using AttachmentDescriptionFlags = Flags<AttachmentDescriptionFlagBits, VkAttachmentDescriptionFlags>;

  VULKAN_HPP_INLINE AttachmentDescriptionFlags operator|( AttachmentDescriptionFlagBits bit0, AttachmentDescriptionFlagBits bit1 )
  {
    return AttachmentDescriptionFlags( bit0 ) | bit1;
  }

  VULKAN_HPP_INLINE AttachmentDescriptionFlags operator~( AttachmentDescriptionFlagBits bits )
  {
    return ~( AttachmentDescriptionFlags( bits ) );
  }

  template <> struct FlagTraits<AttachmentDescriptionFlagBits>
  {
    enum
    {
      allFlags = VkFlags(AttachmentDescriptionFlagBits::eMayAlias)
    };
  };

  VULKAN_HPP_INLINE std::string to_string( AttachmentDescriptionFlags value  )
  {
    if ( !value ) return "{}";
    std::string result;

    if ( value & AttachmentDescriptionFlagBits::eMayAlias ) result += "MayAlias";
    return result;
  }

  enum class BufferCreateFlagBits
  {
    eSparseBinding = VK_BUFFER_CREATE_SPARSE_BINDING_BIT,
    eSparseResidency = VK_BUFFER_CREATE_SPARSE_RESIDENCY_BIT,
    eSparseAliased = VK_BUFFER_CREATE_SPARSE_ALIASED_BIT,
    eProtected = VK_BUFFER_CREATE_PROTECTED_BIT,
    eDeviceAddressCaptureReplayEXT = VK_BUFFER_CREATE_DEVICE_ADDRESS_CAPTURE_REPLAY_BIT_EXT
  };

  VULKAN_HPP_INLINE std::string to_string( BufferCreateFlagBits value )
  {
    switch ( value )
    {
      case BufferCreateFlagBits::eSparseBinding : return "SparseBinding";
      case BufferCreateFlagBits::eSparseResidency : return "SparseResidency";
      case BufferCreateFlagBits::eSparseAliased : return "SparseAliased";
      case BufferCreateFlagBits::eProtected : return "Protected";
      case BufferCreateFlagBits::eDeviceAddressCaptureReplayEXT : return "DeviceAddressCaptureReplayEXT";
      default: return "invalid";
    }
  }

  using BufferCreateFlags = Flags<BufferCreateFlagBits, VkBufferCreateFlags>;

  VULKAN_HPP_INLINE BufferCreateFlags operator|( BufferCreateFlagBits bit0, BufferCreateFlagBits bit1 )
  {
    return BufferCreateFlags( bit0 ) | bit1;
  }

  VULKAN_HPP_INLINE BufferCreateFlags operator~( BufferCreateFlagBits bits )
  {
    return ~( BufferCreateFlags( bits ) );
  }

  template <> struct FlagTraits<BufferCreateFlagBits>
  {
    enum
    {
      allFlags = VkFlags(BufferCreateFlagBits::eSparseBinding) | VkFlags(BufferCreateFlagBits::eSparseResidency) | VkFlags(BufferCreateFlagBits::eSparseAliased) | VkFlags(BufferCreateFlagBits::eProtected) | VkFlags(BufferCreateFlagBits::eDeviceAddressCaptureReplayEXT)
    };
  };

  VULKAN_HPP_INLINE std::string to_string( BufferCreateFlags value  )
  {
    if ( !value ) return "{}";
    std::string result;

    if ( value & BufferCreateFlagBits::eSparseBinding ) result += "SparseBinding | ";
    if ( value & BufferCreateFlagBits::eSparseResidency ) result += "SparseResidency | ";
    if ( value & BufferCreateFlagBits::eSparseAliased ) result += "SparseAliased | ";
    if ( value & BufferCreateFlagBits::eProtected ) result += "Protected | ";
    if ( value & BufferCreateFlagBits::eDeviceAddressCaptureReplayEXT ) result += "DeviceAddressCaptureReplayEXT";
    return result;
  }

  enum class BufferUsageFlagBits
  {
    eTransferSrc = VK_BUFFER_USAGE_TRANSFER_SRC_BIT,
    eTransferDst = VK_BUFFER_USAGE_TRANSFER_DST_BIT,
    eUniformTexelBuffer = VK_BUFFER_USAGE_UNIFORM_TEXEL_BUFFER_BIT,
    eStorageTexelBuffer = VK_BUFFER_USAGE_STORAGE_TEXEL_BUFFER_BIT,
    eUniformBuffer = VK_BUFFER_USAGE_UNIFORM_BUFFER_BIT,
    eStorageBuffer = VK_BUFFER_USAGE_STORAGE_BUFFER_BIT,
    eIndexBuffer = VK_BUFFER_USAGE_INDEX_BUFFER_BIT,
    eVertexBuffer = VK_BUFFER_USAGE_VERTEX_BUFFER_BIT,
    eIndirectBuffer = VK_BUFFER_USAGE_INDIRECT_BUFFER_BIT,
    eTransformFeedbackBufferEXT = VK_BUFFER_USAGE_TRANSFORM_FEEDBACK_BUFFER_BIT_EXT,
    eTransformFeedbackCounterBufferEXT = VK_BUFFER_USAGE_TRANSFORM_FEEDBACK_COUNTER_BUFFER_BIT_EXT,
    eConditionalRenderingEXT = VK_BUFFER_USAGE_CONDITIONAL_RENDERING_BIT_EXT,
    eRayTracingNV = VK_BUFFER_USAGE_RAY_TRACING_BIT_NV,
    eShaderDeviceAddressEXT = VK_BUFFER_USAGE_SHADER_DEVICE_ADDRESS_BIT_EXT
  };

  VULKAN_HPP_INLINE std::string to_string( BufferUsageFlagBits value )
  {
    switch ( value )
    {
      case BufferUsageFlagBits::eTransferSrc : return "TransferSrc";
      case BufferUsageFlagBits::eTransferDst : return "TransferDst";
      case BufferUsageFlagBits::eUniformTexelBuffer : return "UniformTexelBuffer";
      case BufferUsageFlagBits::eStorageTexelBuffer : return "StorageTexelBuffer";
      case BufferUsageFlagBits::eUniformBuffer : return "UniformBuffer";
      case BufferUsageFlagBits::eStorageBuffer : return "StorageBuffer";
      case BufferUsageFlagBits::eIndexBuffer : return "IndexBuffer";
      case BufferUsageFlagBits::eVertexBuffer : return "VertexBuffer";
      case BufferUsageFlagBits::eIndirectBuffer : return "IndirectBuffer";
      case BufferUsageFlagBits::eTransformFeedbackBufferEXT : return "TransformFeedbackBufferEXT";
      case BufferUsageFlagBits::eTransformFeedbackCounterBufferEXT : return "TransformFeedbackCounterBufferEXT";
      case BufferUsageFlagBits::eConditionalRenderingEXT : return "ConditionalRenderingEXT";
      case BufferUsageFlagBits::eRayTracingNV : return "RayTracingNV";
      case BufferUsageFlagBits::eShaderDeviceAddressEXT : return "ShaderDeviceAddressEXT";
      default: return "invalid";
    }
  }

  using BufferUsageFlags = Flags<BufferUsageFlagBits, VkBufferUsageFlags>;

  VULKAN_HPP_INLINE BufferUsageFlags operator|( BufferUsageFlagBits bit0, BufferUsageFlagBits bit1 )
  {
    return BufferUsageFlags( bit0 ) | bit1;
  }

  VULKAN_HPP_INLINE BufferUsageFlags operator~( BufferUsageFlagBits bits )
  {
    return ~( BufferUsageFlags( bits ) );
  }

  template <> struct FlagTraits<BufferUsageFlagBits>
  {
    enum
    {
      allFlags = VkFlags(BufferUsageFlagBits::eTransferSrc) | VkFlags(BufferUsageFlagBits::eTransferDst) | VkFlags(BufferUsageFlagBits::eUniformTexelBuffer) | VkFlags(BufferUsageFlagBits::eStorageTexelBuffer) | VkFlags(BufferUsageFlagBits::eUniformBuffer) | VkFlags(BufferUsageFlagBits::eStorageBuffer) | VkFlags(BufferUsageFlagBits::eIndexBuffer) | VkFlags(BufferUsageFlagBits::eVertexBuffer) | VkFlags(BufferUsageFlagBits::eIndirectBuffer) | VkFlags(BufferUsageFlagBits::eTransformFeedbackBufferEXT) | VkFlags(BufferUsageFlagBits::eTransformFeedbackCounterBufferEXT) | VkFlags(BufferUsageFlagBits::eConditionalRenderingEXT) | VkFlags(BufferUsageFlagBits::eRayTracingNV) | VkFlags(BufferUsageFlagBits::eShaderDeviceAddressEXT)
    };
  };

  VULKAN_HPP_INLINE std::string to_string( BufferUsageFlags value  )
  {
    if ( !value ) return "{}";
    std::string result;

    if ( value & BufferUsageFlagBits::eTransferSrc ) result += "TransferSrc | ";
    if ( value & BufferUsageFlagBits::eTransferDst ) result += "TransferDst | ";
    if ( value & BufferUsageFlagBits::eUniformTexelBuffer ) result += "UniformTexelBuffer | ";
    if ( value & BufferUsageFlagBits::eStorageTexelBuffer ) result += "StorageTexelBuffer | ";
    if ( value & BufferUsageFlagBits::eUniformBuffer ) result += "UniformBuffer | ";
    if ( value & BufferUsageFlagBits::eStorageBuffer ) result += "StorageBuffer | ";
    if ( value & BufferUsageFlagBits::eIndexBuffer ) result += "IndexBuffer | ";
    if ( value & BufferUsageFlagBits::eVertexBuffer ) result += "VertexBuffer | ";
    if ( value & BufferUsageFlagBits::eIndirectBuffer ) result += "IndirectBuffer | ";
    if ( value & BufferUsageFlagBits::eTransformFeedbackBufferEXT ) result += "TransformFeedbackBufferEXT | ";
    if ( value & BufferUsageFlagBits::eTransformFeedbackCounterBufferEXT ) result += "TransformFeedbackCounterBufferEXT | ";
    if ( value & BufferUsageFlagBits::eConditionalRenderingEXT ) result += "ConditionalRenderingEXT | ";
    if ( value & BufferUsageFlagBits::eRayTracingNV ) result += "RayTracingNV | ";
    if ( value & BufferUsageFlagBits::eShaderDeviceAddressEXT ) result += "ShaderDeviceAddressEXT";
    return result;
  }

  enum class BufferViewCreateFlagBits
  {};

  VULKAN_HPP_INLINE std::string to_string( BufferViewCreateFlagBits )
  {
    return "(void)";
  }

  using BufferViewCreateFlags = Flags<BufferViewCreateFlagBits, VkBufferViewCreateFlags>;

  VULKAN_HPP_INLINE std::string to_string( BufferViewCreateFlags  )
  {
    return "{}";
  }

  enum class BuildAccelerationStructureFlagBitsNV
  {
    eAllowUpdate = VK_BUILD_ACCELERATION_STRUCTURE_ALLOW_UPDATE_BIT_NV,
    eAllowCompaction = VK_BUILD_ACCELERATION_STRUCTURE_ALLOW_COMPACTION_BIT_NV,
    ePreferFastTrace = VK_BUILD_ACCELERATION_STRUCTURE_PREFER_FAST_TRACE_BIT_NV,
    ePreferFastBuild = VK_BUILD_ACCELERATION_STRUCTURE_PREFER_FAST_BUILD_BIT_NV,
    eLowMemory = VK_BUILD_ACCELERATION_STRUCTURE_LOW_MEMORY_BIT_NV
  };

  VULKAN_HPP_INLINE std::string to_string( BuildAccelerationStructureFlagBitsNV value )
  {
    switch ( value )
    {
      case BuildAccelerationStructureFlagBitsNV::eAllowUpdate : return "AllowUpdate";
      case BuildAccelerationStructureFlagBitsNV::eAllowCompaction : return "AllowCompaction";
      case BuildAccelerationStructureFlagBitsNV::ePreferFastTrace : return "PreferFastTrace";
      case BuildAccelerationStructureFlagBitsNV::ePreferFastBuild : return "PreferFastBuild";
      case BuildAccelerationStructureFlagBitsNV::eLowMemory : return "LowMemory";
      default: return "invalid";
    }
  }

  using BuildAccelerationStructureFlagsNV = Flags<BuildAccelerationStructureFlagBitsNV, VkBuildAccelerationStructureFlagsNV>;

  VULKAN_HPP_INLINE BuildAccelerationStructureFlagsNV operator|( BuildAccelerationStructureFlagBitsNV bit0, BuildAccelerationStructureFlagBitsNV bit1 )
  {
    return BuildAccelerationStructureFlagsNV( bit0 ) | bit1;
  }

  VULKAN_HPP_INLINE BuildAccelerationStructureFlagsNV operator~( BuildAccelerationStructureFlagBitsNV bits )
  {
    return ~( BuildAccelerationStructureFlagsNV( bits ) );
  }

  template <> struct FlagTraits<BuildAccelerationStructureFlagBitsNV>
  {
    enum
    {
      allFlags = VkFlags(BuildAccelerationStructureFlagBitsNV::eAllowUpdate) | VkFlags(BuildAccelerationStructureFlagBitsNV::eAllowCompaction) | VkFlags(BuildAccelerationStructureFlagBitsNV::ePreferFastTrace) | VkFlags(BuildAccelerationStructureFlagBitsNV::ePreferFastBuild) | VkFlags(BuildAccelerationStructureFlagBitsNV::eLowMemory)
    };
  };

  VULKAN_HPP_INLINE std::string to_string( BuildAccelerationStructureFlagsNV value  )
  {
    if ( !value ) return "{}";
    std::string result;

    if ( value & BuildAccelerationStructureFlagBitsNV::eAllowUpdate ) result += "AllowUpdate | ";
    if ( value & BuildAccelerationStructureFlagBitsNV::eAllowCompaction ) result += "AllowCompaction | ";
    if ( value & BuildAccelerationStructureFlagBitsNV::ePreferFastTrace ) result += "PreferFastTrace | ";
    if ( value & BuildAccelerationStructureFlagBitsNV::ePreferFastBuild ) result += "PreferFastBuild | ";
    if ( value & BuildAccelerationStructureFlagBitsNV::eLowMemory ) result += "LowMemory";
    return result;
  }

  enum class ColorComponentFlagBits
  {
    eR = VK_COLOR_COMPONENT_R_BIT,
    eG = VK_COLOR_COMPONENT_G_BIT,
    eB = VK_COLOR_COMPONENT_B_BIT,
    eA = VK_COLOR_COMPONENT_A_BIT
  };

  VULKAN_HPP_INLINE std::string to_string( ColorComponentFlagBits value )
  {
    switch ( value )
    {
      case ColorComponentFlagBits::eR : return "R";
      case ColorComponentFlagBits::eG : return "G";
      case ColorComponentFlagBits::eB : return "B";
      case ColorComponentFlagBits::eA : return "A";
      default: return "invalid";
    }
  }

  using ColorComponentFlags = Flags<ColorComponentFlagBits, VkColorComponentFlags>;

  VULKAN_HPP_INLINE ColorComponentFlags operator|( ColorComponentFlagBits bit0, ColorComponentFlagBits bit1 )
  {
    return ColorComponentFlags( bit0 ) | bit1;
  }

  VULKAN_HPP_INLINE ColorComponentFlags operator~( ColorComponentFlagBits bits )
  {
    return ~( ColorComponentFlags( bits ) );
  }

  template <> struct FlagTraits<ColorComponentFlagBits>
  {
    enum
    {
      allFlags = VkFlags(ColorComponentFlagBits::eR) | VkFlags(ColorComponentFlagBits::eG) | VkFlags(ColorComponentFlagBits::eB) | VkFlags(ColorComponentFlagBits::eA)
    };
  };

  VULKAN_HPP_INLINE std::string to_string( ColorComponentFlags value  )
  {
    if ( !value ) return "{}";
    std::string result;

    if ( value & ColorComponentFlagBits::eR ) result += "R | ";
    if ( value & ColorComponentFlagBits::eG ) result += "G | ";
    if ( value & ColorComponentFlagBits::eB ) result += "B | ";
    if ( value & ColorComponentFlagBits::eA ) result += "A";
    return result;
  }

  enum class CommandBufferResetFlagBits
  {
    eReleaseResources = VK_COMMAND_BUFFER_RESET_RELEASE_RESOURCES_BIT
  };

  VULKAN_HPP_INLINE std::string to_string( CommandBufferResetFlagBits value )
  {
    switch ( value )
    {
      case CommandBufferResetFlagBits::eReleaseResources : return "ReleaseResources";
      default: return "invalid";
    }
  }

  using CommandBufferResetFlags = Flags<CommandBufferResetFlagBits, VkCommandBufferResetFlags>;

  VULKAN_HPP_INLINE CommandBufferResetFlags operator|( CommandBufferResetFlagBits bit0, CommandBufferResetFlagBits bit1 )
  {
    return CommandBufferResetFlags( bit0 ) | bit1;
  }

  VULKAN_HPP_INLINE CommandBufferResetFlags operator~( CommandBufferResetFlagBits bits )
  {
    return ~( CommandBufferResetFlags( bits ) );
  }

  template <> struct FlagTraits<CommandBufferResetFlagBits>
  {
    enum
    {
      allFlags = VkFlags(CommandBufferResetFlagBits::eReleaseResources)
    };
  };

  VULKAN_HPP_INLINE std::string to_string( CommandBufferResetFlags value  )
  {
    if ( !value ) return "{}";
    std::string result;

    if ( value & CommandBufferResetFlagBits::eReleaseResources ) result += "ReleaseResources";
    return result;
  }

  enum class CommandBufferUsageFlagBits
  {
    eOneTimeSubmit = VK_COMMAND_BUFFER_USAGE_ONE_TIME_SUBMIT_BIT,
    eRenderPassContinue = VK_COMMAND_BUFFER_USAGE_RENDER_PASS_CONTINUE_BIT,
    eSimultaneousUse = VK_COMMAND_BUFFER_USAGE_SIMULTANEOUS_USE_BIT
  };

  VULKAN_HPP_INLINE std::string to_string( CommandBufferUsageFlagBits value )
  {
    switch ( value )
    {
      case CommandBufferUsageFlagBits::eOneTimeSubmit : return "OneTimeSubmit";
      case CommandBufferUsageFlagBits::eRenderPassContinue : return "RenderPassContinue";
      case CommandBufferUsageFlagBits::eSimultaneousUse : return "SimultaneousUse";
      default: return "invalid";
    }
  }

  using CommandBufferUsageFlags = Flags<CommandBufferUsageFlagBits, VkCommandBufferUsageFlags>;

  VULKAN_HPP_INLINE CommandBufferUsageFlags operator|( CommandBufferUsageFlagBits bit0, CommandBufferUsageFlagBits bit1 )
  {
    return CommandBufferUsageFlags( bit0 ) | bit1;
  }

  VULKAN_HPP_INLINE CommandBufferUsageFlags operator~( CommandBufferUsageFlagBits bits )
  {
    return ~( CommandBufferUsageFlags( bits ) );
  }

  template <> struct FlagTraits<CommandBufferUsageFlagBits>
  {
    enum
    {
      allFlags = VkFlags(CommandBufferUsageFlagBits::eOneTimeSubmit) | VkFlags(CommandBufferUsageFlagBits::eRenderPassContinue) | VkFlags(CommandBufferUsageFlagBits::eSimultaneousUse)
    };
  };

  VULKAN_HPP_INLINE std::string to_string( CommandBufferUsageFlags value  )
  {
    if ( !value ) return "{}";
    std::string result;

    if ( value & CommandBufferUsageFlagBits::eOneTimeSubmit ) result += "OneTimeSubmit | ";
    if ( value & CommandBufferUsageFlagBits::eRenderPassContinue ) result += "RenderPassContinue | ";
    if ( value & CommandBufferUsageFlagBits::eSimultaneousUse ) result += "SimultaneousUse";
    return result;
  }

  enum class CommandPoolCreateFlagBits
  {
    eTransient = VK_COMMAND_POOL_CREATE_TRANSIENT_BIT,
    eResetCommandBuffer = VK_COMMAND_POOL_CREATE_RESET_COMMAND_BUFFER_BIT,
    eProtected = VK_COMMAND_POOL_CREATE_PROTECTED_BIT
  };

  VULKAN_HPP_INLINE std::string to_string( CommandPoolCreateFlagBits value )
  {
    switch ( value )
    {
      case CommandPoolCreateFlagBits::eTransient : return "Transient";
      case CommandPoolCreateFlagBits::eResetCommandBuffer : return "ResetCommandBuffer";
      case CommandPoolCreateFlagBits::eProtected : return "Protected";
      default: return "invalid";
    }
  }

  using CommandPoolCreateFlags = Flags<CommandPoolCreateFlagBits, VkCommandPoolCreateFlags>;

  VULKAN_HPP_INLINE CommandPoolCreateFlags operator|( CommandPoolCreateFlagBits bit0, CommandPoolCreateFlagBits bit1 )
  {
    return CommandPoolCreateFlags( bit0 ) | bit1;
  }

  VULKAN_HPP_INLINE CommandPoolCreateFlags operator~( CommandPoolCreateFlagBits bits )
  {
    return ~( CommandPoolCreateFlags( bits ) );
  }

  template <> struct FlagTraits<CommandPoolCreateFlagBits>
  {
    enum
    {
      allFlags = VkFlags(CommandPoolCreateFlagBits::eTransient) | VkFlags(CommandPoolCreateFlagBits::eResetCommandBuffer) | VkFlags(CommandPoolCreateFlagBits::eProtected)
    };
  };

  VULKAN_HPP_INLINE std::string to_string( CommandPoolCreateFlags value  )
  {
    if ( !value ) return "{}";
    std::string result;

    if ( value & CommandPoolCreateFlagBits::eTransient ) result += "Transient | ";
    if ( value & CommandPoolCreateFlagBits::eResetCommandBuffer ) result += "ResetCommandBuffer | ";
    if ( value & CommandPoolCreateFlagBits::eProtected ) result += "Protected";
    return result;
  }

  enum class CommandPoolResetFlagBits
  {
    eReleaseResources = VK_COMMAND_POOL_RESET_RELEASE_RESOURCES_BIT
  };

  VULKAN_HPP_INLINE std::string to_string( CommandPoolResetFlagBits value )
  {
    switch ( value )
    {
      case CommandPoolResetFlagBits::eReleaseResources : return "ReleaseResources";
      default: return "invalid";
    }
  }

  using CommandPoolResetFlags = Flags<CommandPoolResetFlagBits, VkCommandPoolResetFlags>;

  VULKAN_HPP_INLINE CommandPoolResetFlags operator|( CommandPoolResetFlagBits bit0, CommandPoolResetFlagBits bit1 )
  {
    return CommandPoolResetFlags( bit0 ) | bit1;
  }

  VULKAN_HPP_INLINE CommandPoolResetFlags operator~( CommandPoolResetFlagBits bits )
  {
    return ~( CommandPoolResetFlags( bits ) );
  }

  template <> struct FlagTraits<CommandPoolResetFlagBits>
  {
    enum
    {
      allFlags = VkFlags(CommandPoolResetFlagBits::eReleaseResources)
    };
  };

  VULKAN_HPP_INLINE std::string to_string( CommandPoolResetFlags value  )
  {
    if ( !value ) return "{}";
    std::string result;

    if ( value & CommandPoolResetFlagBits::eReleaseResources ) result += "ReleaseResources";
    return result;
  }

  enum class CommandPoolTrimFlagBits
  {};

  VULKAN_HPP_INLINE std::string to_string( CommandPoolTrimFlagBits )
  {
    return "(void)";
  }

  using CommandPoolTrimFlags = Flags<CommandPoolTrimFlagBits, VkCommandPoolTrimFlags>;

  using CommandPoolTrimFlagsKHR = CommandPoolTrimFlags;

  VULKAN_HPP_INLINE std::string to_string( CommandPoolTrimFlags  )
  {
    return "{}";
  }

  enum class CompositeAlphaFlagBitsKHR
  {
    eOpaque = VK_COMPOSITE_ALPHA_OPAQUE_BIT_KHR,
    ePreMultiplied = VK_COMPOSITE_ALPHA_PRE_MULTIPLIED_BIT_KHR,
    ePostMultiplied = VK_COMPOSITE_ALPHA_POST_MULTIPLIED_BIT_KHR,
    eInherit = VK_COMPOSITE_ALPHA_INHERIT_BIT_KHR
  };

  VULKAN_HPP_INLINE std::string to_string( CompositeAlphaFlagBitsKHR value )
  {
    switch ( value )
    {
      case CompositeAlphaFlagBitsKHR::eOpaque : return "Opaque";
      case CompositeAlphaFlagBitsKHR::ePreMultiplied : return "PreMultiplied";
      case CompositeAlphaFlagBitsKHR::ePostMultiplied : return "PostMultiplied";
      case CompositeAlphaFlagBitsKHR::eInherit : return "Inherit";
      default: return "invalid";
    }
  }

  using CompositeAlphaFlagsKHR = Flags<CompositeAlphaFlagBitsKHR, VkCompositeAlphaFlagsKHR>;

  VULKAN_HPP_INLINE CompositeAlphaFlagsKHR operator|( CompositeAlphaFlagBitsKHR bit0, CompositeAlphaFlagBitsKHR bit1 )
  {
    return CompositeAlphaFlagsKHR( bit0 ) | bit1;
  }

  VULKAN_HPP_INLINE CompositeAlphaFlagsKHR operator~( CompositeAlphaFlagBitsKHR bits )
  {
    return ~( CompositeAlphaFlagsKHR( bits ) );
  }

  template <> struct FlagTraits<CompositeAlphaFlagBitsKHR>
  {
    enum
    {
      allFlags = VkFlags(CompositeAlphaFlagBitsKHR::eOpaque) | VkFlags(CompositeAlphaFlagBitsKHR::ePreMultiplied) | VkFlags(CompositeAlphaFlagBitsKHR::ePostMultiplied) | VkFlags(CompositeAlphaFlagBitsKHR::eInherit)
    };
  };

  VULKAN_HPP_INLINE std::string to_string( CompositeAlphaFlagsKHR value  )
  {
    if ( !value ) return "{}";
    std::string result;

    if ( value & CompositeAlphaFlagBitsKHR::eOpaque ) result += "Opaque | ";
    if ( value & CompositeAlphaFlagBitsKHR::ePreMultiplied ) result += "PreMultiplied | ";
    if ( value & CompositeAlphaFlagBitsKHR::ePostMultiplied ) result += "PostMultiplied | ";
    if ( value & CompositeAlphaFlagBitsKHR::eInherit ) result += "Inherit";
    return result;
  }

  enum class ConditionalRenderingFlagBitsEXT
  {
    eInverted = VK_CONDITIONAL_RENDERING_INVERTED_BIT_EXT
  };

  VULKAN_HPP_INLINE std::string to_string( ConditionalRenderingFlagBitsEXT value )
  {
    switch ( value )
    {
      case ConditionalRenderingFlagBitsEXT::eInverted : return "Inverted";
      default: return "invalid";
    }
  }

  using ConditionalRenderingFlagsEXT = Flags<ConditionalRenderingFlagBitsEXT, VkConditionalRenderingFlagsEXT>;

  VULKAN_HPP_INLINE ConditionalRenderingFlagsEXT operator|( ConditionalRenderingFlagBitsEXT bit0, ConditionalRenderingFlagBitsEXT bit1 )
  {
    return ConditionalRenderingFlagsEXT( bit0 ) | bit1;
  }

  VULKAN_HPP_INLINE ConditionalRenderingFlagsEXT operator~( ConditionalRenderingFlagBitsEXT bits )
  {
    return ~( ConditionalRenderingFlagsEXT( bits ) );
  }

  template <> struct FlagTraits<ConditionalRenderingFlagBitsEXT>
  {
    enum
    {
      allFlags = VkFlags(ConditionalRenderingFlagBitsEXT::eInverted)
    };
  };

  VULKAN_HPP_INLINE std::string to_string( ConditionalRenderingFlagsEXT value  )
  {
    if ( !value ) return "{}";
    std::string result;

    if ( value & ConditionalRenderingFlagBitsEXT::eInverted ) result += "Inverted";
    return result;
  }

  enum class CullModeFlagBits
  {
    eNone = VK_CULL_MODE_NONE,
    eFront = VK_CULL_MODE_FRONT_BIT,
    eBack = VK_CULL_MODE_BACK_BIT,
    eFrontAndBack = VK_CULL_MODE_FRONT_AND_BACK
  };

  VULKAN_HPP_INLINE std::string to_string( CullModeFlagBits value )
  {
    switch ( value )
    {
      case CullModeFlagBits::eNone : return "None";
      case CullModeFlagBits::eFront : return "Front";
      case CullModeFlagBits::eBack : return "Back";
      case CullModeFlagBits::eFrontAndBack : return "FrontAndBack";
      default: return "invalid";
    }
  }

  using CullModeFlags = Flags<CullModeFlagBits, VkCullModeFlags>;

  VULKAN_HPP_INLINE CullModeFlags operator|( CullModeFlagBits bit0, CullModeFlagBits bit1 )
  {
    return CullModeFlags( bit0 ) | bit1;
  }

  VULKAN_HPP_INLINE CullModeFlags operator~( CullModeFlagBits bits )
  {
    return ~( CullModeFlags( bits ) );
  }

  template <> struct FlagTraits<CullModeFlagBits>
  {
    enum
    {
      allFlags = VkFlags(CullModeFlagBits::eNone) | VkFlags(CullModeFlagBits::eFront) | VkFlags(CullModeFlagBits::eBack) | VkFlags(CullModeFlagBits::eFrontAndBack)
    };
  };

  VULKAN_HPP_INLINE std::string to_string( CullModeFlags value  )
  {
    if ( !value ) return "{}";
    std::string result;

    if ( value & CullModeFlagBits::eNone ) result += "None | ";
    if ( value & CullModeFlagBits::eFront ) result += "Front | ";
    if ( value & CullModeFlagBits::eBack ) result += "Back | ";
    if ( value & CullModeFlagBits::eFrontAndBack ) result += "FrontAndBack";
    return result;
  }

  enum class DebugReportFlagBitsEXT
  {
    eInformation = VK_DEBUG_REPORT_INFORMATION_BIT_EXT,
    eWarning = VK_DEBUG_REPORT_WARNING_BIT_EXT,
    ePerformanceWarning = VK_DEBUG_REPORT_PERFORMANCE_WARNING_BIT_EXT,
    eError = VK_DEBUG_REPORT_ERROR_BIT_EXT,
    eDebug = VK_DEBUG_REPORT_DEBUG_BIT_EXT
  };

  VULKAN_HPP_INLINE std::string to_string( DebugReportFlagBitsEXT value )
  {
    switch ( value )
    {
      case DebugReportFlagBitsEXT::eInformation : return "Information";
      case DebugReportFlagBitsEXT::eWarning : return "Warning";
      case DebugReportFlagBitsEXT::ePerformanceWarning : return "PerformanceWarning";
      case DebugReportFlagBitsEXT::eError : return "Error";
      case DebugReportFlagBitsEXT::eDebug : return "Debug";
      default: return "invalid";
    }
  }

  using DebugReportFlagsEXT = Flags<DebugReportFlagBitsEXT, VkDebugReportFlagsEXT>;

  VULKAN_HPP_INLINE DebugReportFlagsEXT operator|( DebugReportFlagBitsEXT bit0, DebugReportFlagBitsEXT bit1 )
  {
    return DebugReportFlagsEXT( bit0 ) | bit1;
  }

  VULKAN_HPP_INLINE DebugReportFlagsEXT operator~( DebugReportFlagBitsEXT bits )
  {
    return ~( DebugReportFlagsEXT( bits ) );
  }

  template <> struct FlagTraits<DebugReportFlagBitsEXT>
  {
    enum
    {
      allFlags = VkFlags(DebugReportFlagBitsEXT::eInformation) | VkFlags(DebugReportFlagBitsEXT::eWarning) | VkFlags(DebugReportFlagBitsEXT::ePerformanceWarning) | VkFlags(DebugReportFlagBitsEXT::eError) | VkFlags(DebugReportFlagBitsEXT::eDebug)
    };
  };

  VULKAN_HPP_INLINE std::string to_string( DebugReportFlagsEXT value  )
  {
    if ( !value ) return "{}";
    std::string result;

    if ( value & DebugReportFlagBitsEXT::eInformation ) result += "Information | ";
    if ( value & DebugReportFlagBitsEXT::eWarning ) result += "Warning | ";
    if ( value & DebugReportFlagBitsEXT::ePerformanceWarning ) result += "PerformanceWarning | ";
    if ( value & DebugReportFlagBitsEXT::eError ) result += "Error | ";
    if ( value & DebugReportFlagBitsEXT::eDebug ) result += "Debug";
    return result;
  }

  enum class DebugUtilsMessageSeverityFlagBitsEXT
  {
    eVerbose = VK_DEBUG_UTILS_MESSAGE_SEVERITY_VERBOSE_BIT_EXT,
    eInfo = VK_DEBUG_UTILS_MESSAGE_SEVERITY_INFO_BIT_EXT,
    eWarning = VK_DEBUG_UTILS_MESSAGE_SEVERITY_WARNING_BIT_EXT,
    eError = VK_DEBUG_UTILS_MESSAGE_SEVERITY_ERROR_BIT_EXT
  };

  VULKAN_HPP_INLINE std::string to_string( DebugUtilsMessageSeverityFlagBitsEXT value )
  {
    switch ( value )
    {
      case DebugUtilsMessageSeverityFlagBitsEXT::eVerbose : return "Verbose";
      case DebugUtilsMessageSeverityFlagBitsEXT::eInfo : return "Info";
      case DebugUtilsMessageSeverityFlagBitsEXT::eWarning : return "Warning";
      case DebugUtilsMessageSeverityFlagBitsEXT::eError : return "Error";
      default: return "invalid";
    }
  }

  using DebugUtilsMessageSeverityFlagsEXT = Flags<DebugUtilsMessageSeverityFlagBitsEXT, VkDebugUtilsMessageSeverityFlagsEXT>;

  VULKAN_HPP_INLINE DebugUtilsMessageSeverityFlagsEXT operator|( DebugUtilsMessageSeverityFlagBitsEXT bit0, DebugUtilsMessageSeverityFlagBitsEXT bit1 )
  {
    return DebugUtilsMessageSeverityFlagsEXT( bit0 ) | bit1;
  }

  VULKAN_HPP_INLINE DebugUtilsMessageSeverityFlagsEXT operator~( DebugUtilsMessageSeverityFlagBitsEXT bits )
  {
    return ~( DebugUtilsMessageSeverityFlagsEXT( bits ) );
  }

  template <> struct FlagTraits<DebugUtilsMessageSeverityFlagBitsEXT>
  {
    enum
    {
      allFlags = VkFlags(DebugUtilsMessageSeverityFlagBitsEXT::eVerbose) | VkFlags(DebugUtilsMessageSeverityFlagBitsEXT::eInfo) | VkFlags(DebugUtilsMessageSeverityFlagBitsEXT::eWarning) | VkFlags(DebugUtilsMessageSeverityFlagBitsEXT::eError)
    };
  };

  VULKAN_HPP_INLINE std::string to_string( DebugUtilsMessageSeverityFlagsEXT value  )
  {
    if ( !value ) return "{}";
    std::string result;

    if ( value & DebugUtilsMessageSeverityFlagBitsEXT::eVerbose ) result += "Verbose | ";
    if ( value & DebugUtilsMessageSeverityFlagBitsEXT::eInfo ) result += "Info | ";
    if ( value & DebugUtilsMessageSeverityFlagBitsEXT::eWarning ) result += "Warning | ";
    if ( value & DebugUtilsMessageSeverityFlagBitsEXT::eError ) result += "Error";
    return result;
  }

  enum class DebugUtilsMessageTypeFlagBitsEXT
  {
    eGeneral = VK_DEBUG_UTILS_MESSAGE_TYPE_GENERAL_BIT_EXT,
    eValidation = VK_DEBUG_UTILS_MESSAGE_TYPE_VALIDATION_BIT_EXT,
    ePerformance = VK_DEBUG_UTILS_MESSAGE_TYPE_PERFORMANCE_BIT_EXT
  };

  VULKAN_HPP_INLINE std::string to_string( DebugUtilsMessageTypeFlagBitsEXT value )
  {
    switch ( value )
    {
      case DebugUtilsMessageTypeFlagBitsEXT::eGeneral : return "General";
      case DebugUtilsMessageTypeFlagBitsEXT::eValidation : return "Validation";
      case DebugUtilsMessageTypeFlagBitsEXT::ePerformance : return "Performance";
      default: return "invalid";
    }
  }

  using DebugUtilsMessageTypeFlagsEXT = Flags<DebugUtilsMessageTypeFlagBitsEXT, VkDebugUtilsMessageTypeFlagsEXT>;

  VULKAN_HPP_INLINE DebugUtilsMessageTypeFlagsEXT operator|( DebugUtilsMessageTypeFlagBitsEXT bit0, DebugUtilsMessageTypeFlagBitsEXT bit1 )
  {
    return DebugUtilsMessageTypeFlagsEXT( bit0 ) | bit1;
  }

  VULKAN_HPP_INLINE DebugUtilsMessageTypeFlagsEXT operator~( DebugUtilsMessageTypeFlagBitsEXT bits )
  {
    return ~( DebugUtilsMessageTypeFlagsEXT( bits ) );
  }

  template <> struct FlagTraits<DebugUtilsMessageTypeFlagBitsEXT>
  {
    enum
    {
      allFlags = VkFlags(DebugUtilsMessageTypeFlagBitsEXT::eGeneral) | VkFlags(DebugUtilsMessageTypeFlagBitsEXT::eValidation) | VkFlags(DebugUtilsMessageTypeFlagBitsEXT::ePerformance)
    };
  };

  VULKAN_HPP_INLINE std::string to_string( DebugUtilsMessageTypeFlagsEXT value  )
  {
    if ( !value ) return "{}";
    std::string result;

    if ( value & DebugUtilsMessageTypeFlagBitsEXT::eGeneral ) result += "General | ";
    if ( value & DebugUtilsMessageTypeFlagBitsEXT::eValidation ) result += "Validation | ";
    if ( value & DebugUtilsMessageTypeFlagBitsEXT::ePerformance ) result += "Performance";
    return result;
  }

  enum class DebugUtilsMessengerCallbackDataFlagBitsEXT
  {};

  VULKAN_HPP_INLINE std::string to_string( DebugUtilsMessengerCallbackDataFlagBitsEXT )
  {
    return "(void)";
  }

  using DebugUtilsMessengerCallbackDataFlagsEXT = Flags<DebugUtilsMessengerCallbackDataFlagBitsEXT, VkDebugUtilsMessengerCallbackDataFlagsEXT>;

  VULKAN_HPP_INLINE std::string to_string( DebugUtilsMessengerCallbackDataFlagsEXT  )
  {
    return "{}";
  }

  enum class DebugUtilsMessengerCreateFlagBitsEXT
  {};

  VULKAN_HPP_INLINE std::string to_string( DebugUtilsMessengerCreateFlagBitsEXT )
  {
    return "(void)";
  }

  using DebugUtilsMessengerCreateFlagsEXT = Flags<DebugUtilsMessengerCreateFlagBitsEXT, VkDebugUtilsMessengerCreateFlagsEXT>;

  VULKAN_HPP_INLINE std::string to_string( DebugUtilsMessengerCreateFlagsEXT  )
  {
    return "{}";
  }

  enum class DependencyFlagBits
  {
    eByRegion = VK_DEPENDENCY_BY_REGION_BIT,
    eDeviceGroup = VK_DEPENDENCY_DEVICE_GROUP_BIT,
    eViewLocal = VK_DEPENDENCY_VIEW_LOCAL_BIT,
    eViewLocalKHR = VK_DEPENDENCY_VIEW_LOCAL_BIT_KHR,
    eDeviceGroupKHR = VK_DEPENDENCY_DEVICE_GROUP_BIT_KHR
  };

  VULKAN_HPP_INLINE std::string to_string( DependencyFlagBits value )
  {
    switch ( value )
    {
      case DependencyFlagBits::eByRegion : return "ByRegion";
      case DependencyFlagBits::eDeviceGroup : return "DeviceGroup";
      case DependencyFlagBits::eViewLocal : return "ViewLocal";
      default: return "invalid";
    }
  }

  using DependencyFlags = Flags<DependencyFlagBits, VkDependencyFlags>;

  VULKAN_HPP_INLINE DependencyFlags operator|( DependencyFlagBits bit0, DependencyFlagBits bit1 )
  {
    return DependencyFlags( bit0 ) | bit1;
  }

  VULKAN_HPP_INLINE DependencyFlags operator~( DependencyFlagBits bits )
  {
    return ~( DependencyFlags( bits ) );
  }

  template <> struct FlagTraits<DependencyFlagBits>
  {
    enum
    {
      allFlags = VkFlags(DependencyFlagBits::eByRegion) | VkFlags(DependencyFlagBits::eDeviceGroup) | VkFlags(DependencyFlagBits::eViewLocal)
    };
  };

  VULKAN_HPP_INLINE std::string to_string( DependencyFlags value  )
  {
    if ( !value ) return "{}";
    std::string result;

    if ( value & DependencyFlagBits::eByRegion ) result += "ByRegion | ";
    if ( value & DependencyFlagBits::eDeviceGroup ) result += "DeviceGroup | ";
    if ( value & DependencyFlagBits::eViewLocal ) result += "ViewLocal";
    return result;
  }

  enum class DescriptorBindingFlagBitsEXT
  {
    eUpdateAfterBind = VK_DESCRIPTOR_BINDING_UPDATE_AFTER_BIND_BIT_EXT,
    eUpdateUnusedWhilePending = VK_DESCRIPTOR_BINDING_UPDATE_UNUSED_WHILE_PENDING_BIT_EXT,
    ePartiallyBound = VK_DESCRIPTOR_BINDING_PARTIALLY_BOUND_BIT_EXT,
    eVariableDescriptorCount = VK_DESCRIPTOR_BINDING_VARIABLE_DESCRIPTOR_COUNT_BIT_EXT
  };

  VULKAN_HPP_INLINE std::string to_string( DescriptorBindingFlagBitsEXT value )
  {
    switch ( value )
    {
      case DescriptorBindingFlagBitsEXT::eUpdateAfterBind : return "UpdateAfterBind";
      case DescriptorBindingFlagBitsEXT::eUpdateUnusedWhilePending : return "UpdateUnusedWhilePending";
      case DescriptorBindingFlagBitsEXT::ePartiallyBound : return "PartiallyBound";
      case DescriptorBindingFlagBitsEXT::eVariableDescriptorCount : return "VariableDescriptorCount";
      default: return "invalid";
    }
  }

  using DescriptorBindingFlagsEXT = Flags<DescriptorBindingFlagBitsEXT, VkDescriptorBindingFlagsEXT>;

  VULKAN_HPP_INLINE DescriptorBindingFlagsEXT operator|( DescriptorBindingFlagBitsEXT bit0, DescriptorBindingFlagBitsEXT bit1 )
  {
    return DescriptorBindingFlagsEXT( bit0 ) | bit1;
  }

  VULKAN_HPP_INLINE DescriptorBindingFlagsEXT operator~( DescriptorBindingFlagBitsEXT bits )
  {
    return ~( DescriptorBindingFlagsEXT( bits ) );
  }

  template <> struct FlagTraits<DescriptorBindingFlagBitsEXT>
  {
    enum
    {
      allFlags = VkFlags(DescriptorBindingFlagBitsEXT::eUpdateAfterBind) | VkFlags(DescriptorBindingFlagBitsEXT::eUpdateUnusedWhilePending) | VkFlags(DescriptorBindingFlagBitsEXT::ePartiallyBound) | VkFlags(DescriptorBindingFlagBitsEXT::eVariableDescriptorCount)
    };
  };

  VULKAN_HPP_INLINE std::string to_string( DescriptorBindingFlagsEXT value  )
  {
    if ( !value ) return "{}";
    std::string result;

    if ( value & DescriptorBindingFlagBitsEXT::eUpdateAfterBind ) result += "UpdateAfterBind | ";
    if ( value & DescriptorBindingFlagBitsEXT::eUpdateUnusedWhilePending ) result += "UpdateUnusedWhilePending | ";
    if ( value & DescriptorBindingFlagBitsEXT::ePartiallyBound ) result += "PartiallyBound | ";
    if ( value & DescriptorBindingFlagBitsEXT::eVariableDescriptorCount ) result += "VariableDescriptorCount";
    return result;
  }

  enum class DescriptorPoolCreateFlagBits
  {
    eFreeDescriptorSet = VK_DESCRIPTOR_POOL_CREATE_FREE_DESCRIPTOR_SET_BIT,
    eUpdateAfterBindEXT = VK_DESCRIPTOR_POOL_CREATE_UPDATE_AFTER_BIND_BIT_EXT
  };

  VULKAN_HPP_INLINE std::string to_string( DescriptorPoolCreateFlagBits value )
  {
    switch ( value )
    {
      case DescriptorPoolCreateFlagBits::eFreeDescriptorSet : return "FreeDescriptorSet";
      case DescriptorPoolCreateFlagBits::eUpdateAfterBindEXT : return "UpdateAfterBindEXT";
      default: return "invalid";
    }
  }

  using DescriptorPoolCreateFlags = Flags<DescriptorPoolCreateFlagBits, VkDescriptorPoolCreateFlags>;

  VULKAN_HPP_INLINE DescriptorPoolCreateFlags operator|( DescriptorPoolCreateFlagBits bit0, DescriptorPoolCreateFlagBits bit1 )
  {
    return DescriptorPoolCreateFlags( bit0 ) | bit1;
  }

  VULKAN_HPP_INLINE DescriptorPoolCreateFlags operator~( DescriptorPoolCreateFlagBits bits )
  {
    return ~( DescriptorPoolCreateFlags( bits ) );
  }

  template <> struct FlagTraits<DescriptorPoolCreateFlagBits>
  {
    enum
    {
      allFlags = VkFlags(DescriptorPoolCreateFlagBits::eFreeDescriptorSet) | VkFlags(DescriptorPoolCreateFlagBits::eUpdateAfterBindEXT)
    };
  };

  VULKAN_HPP_INLINE std::string to_string( DescriptorPoolCreateFlags value  )
  {
    if ( !value ) return "{}";
    std::string result;

    if ( value & DescriptorPoolCreateFlagBits::eFreeDescriptorSet ) result += "FreeDescriptorSet | ";
    if ( value & DescriptorPoolCreateFlagBits::eUpdateAfterBindEXT ) result += "UpdateAfterBindEXT";
    return result;
  }

  enum class DescriptorPoolResetFlagBits
  {};

  VULKAN_HPP_INLINE std::string to_string( DescriptorPoolResetFlagBits )
  {
    return "(void)";
  }

  using DescriptorPoolResetFlags = Flags<DescriptorPoolResetFlagBits, VkDescriptorPoolResetFlags>;

  VULKAN_HPP_INLINE std::string to_string( DescriptorPoolResetFlags  )
  {
    return "{}";
  }

  enum class DescriptorSetLayoutCreateFlagBits
  {
    ePushDescriptorKHR = VK_DESCRIPTOR_SET_LAYOUT_CREATE_PUSH_DESCRIPTOR_BIT_KHR,
    eUpdateAfterBindPoolEXT = VK_DESCRIPTOR_SET_LAYOUT_CREATE_UPDATE_AFTER_BIND_POOL_BIT_EXT
  };

  VULKAN_HPP_INLINE std::string to_string( DescriptorSetLayoutCreateFlagBits value )
  {
    switch ( value )
    {
      case DescriptorSetLayoutCreateFlagBits::ePushDescriptorKHR : return "PushDescriptorKHR";
      case DescriptorSetLayoutCreateFlagBits::eUpdateAfterBindPoolEXT : return "UpdateAfterBindPoolEXT";
      default: return "invalid";
    }
  }

  using DescriptorSetLayoutCreateFlags = Flags<DescriptorSetLayoutCreateFlagBits, VkDescriptorSetLayoutCreateFlags>;

  VULKAN_HPP_INLINE DescriptorSetLayoutCreateFlags operator|( DescriptorSetLayoutCreateFlagBits bit0, DescriptorSetLayoutCreateFlagBits bit1 )
  {
    return DescriptorSetLayoutCreateFlags( bit0 ) | bit1;
  }

  VULKAN_HPP_INLINE DescriptorSetLayoutCreateFlags operator~( DescriptorSetLayoutCreateFlagBits bits )
  {
    return ~( DescriptorSetLayoutCreateFlags( bits ) );
  }

  template <> struct FlagTraits<DescriptorSetLayoutCreateFlagBits>
  {
    enum
    {
      allFlags = VkFlags(DescriptorSetLayoutCreateFlagBits::ePushDescriptorKHR) | VkFlags(DescriptorSetLayoutCreateFlagBits::eUpdateAfterBindPoolEXT)
    };
  };

  VULKAN_HPP_INLINE std::string to_string( DescriptorSetLayoutCreateFlags value  )
  {
    if ( !value ) return "{}";
    std::string result;

    if ( value & DescriptorSetLayoutCreateFlagBits::ePushDescriptorKHR ) result += "PushDescriptorKHR | ";
    if ( value & DescriptorSetLayoutCreateFlagBits::eUpdateAfterBindPoolEXT ) result += "UpdateAfterBindPoolEXT";
    return result;
  }

  enum class DescriptorUpdateTemplateCreateFlagBits
  {};

  VULKAN_HPP_INLINE std::string to_string( DescriptorUpdateTemplateCreateFlagBits )
  {
    return "(void)";
  }

  using DescriptorUpdateTemplateCreateFlags = Flags<DescriptorUpdateTemplateCreateFlagBits, VkDescriptorUpdateTemplateCreateFlags>;

  using DescriptorUpdateTemplateCreateFlagsKHR = DescriptorUpdateTemplateCreateFlags;

  VULKAN_HPP_INLINE std::string to_string( DescriptorUpdateTemplateCreateFlags  )
  {
    return "{}";
  }

  enum class DeviceCreateFlagBits
  {};

  VULKAN_HPP_INLINE std::string to_string( DeviceCreateFlagBits )
  {
    return "(void)";
  }

  using DeviceCreateFlags = Flags<DeviceCreateFlagBits, VkDeviceCreateFlags>;

  VULKAN_HPP_INLINE std::string to_string( DeviceCreateFlags  )
  {
    return "{}";
  }

  enum class DeviceGroupPresentModeFlagBitsKHR
  {
    eLocal = VK_DEVICE_GROUP_PRESENT_MODE_LOCAL_BIT_KHR,
    eRemote = VK_DEVICE_GROUP_PRESENT_MODE_REMOTE_BIT_KHR,
    eSum = VK_DEVICE_GROUP_PRESENT_MODE_SUM_BIT_KHR,
    eLocalMultiDevice = VK_DEVICE_GROUP_PRESENT_MODE_LOCAL_MULTI_DEVICE_BIT_KHR
  };

  VULKAN_HPP_INLINE std::string to_string( DeviceGroupPresentModeFlagBitsKHR value )
  {
    switch ( value )
    {
      case DeviceGroupPresentModeFlagBitsKHR::eLocal : return "Local";
      case DeviceGroupPresentModeFlagBitsKHR::eRemote : return "Remote";
      case DeviceGroupPresentModeFlagBitsKHR::eSum : return "Sum";
      case DeviceGroupPresentModeFlagBitsKHR::eLocalMultiDevice : return "LocalMultiDevice";
      default: return "invalid";
    }
  }

  using DeviceGroupPresentModeFlagsKHR = Flags<DeviceGroupPresentModeFlagBitsKHR, VkDeviceGroupPresentModeFlagsKHR>;

  VULKAN_HPP_INLINE DeviceGroupPresentModeFlagsKHR operator|( DeviceGroupPresentModeFlagBitsKHR bit0, DeviceGroupPresentModeFlagBitsKHR bit1 )
  {
    return DeviceGroupPresentModeFlagsKHR( bit0 ) | bit1;
  }

  VULKAN_HPP_INLINE DeviceGroupPresentModeFlagsKHR operator~( DeviceGroupPresentModeFlagBitsKHR bits )
  {
    return ~( DeviceGroupPresentModeFlagsKHR( bits ) );
  }

  template <> struct FlagTraits<DeviceGroupPresentModeFlagBitsKHR>
  {
    enum
    {
      allFlags = VkFlags(DeviceGroupPresentModeFlagBitsKHR::eLocal) | VkFlags(DeviceGroupPresentModeFlagBitsKHR::eRemote) | VkFlags(DeviceGroupPresentModeFlagBitsKHR::eSum) | VkFlags(DeviceGroupPresentModeFlagBitsKHR::eLocalMultiDevice)
    };
  };

  VULKAN_HPP_INLINE std::string to_string( DeviceGroupPresentModeFlagsKHR value  )
  {
    if ( !value ) return "{}";
    std::string result;

    if ( value & DeviceGroupPresentModeFlagBitsKHR::eLocal ) result += "Local | ";
    if ( value & DeviceGroupPresentModeFlagBitsKHR::eRemote ) result += "Remote | ";
    if ( value & DeviceGroupPresentModeFlagBitsKHR::eSum ) result += "Sum | ";
    if ( value & DeviceGroupPresentModeFlagBitsKHR::eLocalMultiDevice ) result += "LocalMultiDevice";
    return result;
  }

  enum class DeviceQueueCreateFlagBits
  {
    eProtected = VK_DEVICE_QUEUE_CREATE_PROTECTED_BIT
  };

  VULKAN_HPP_INLINE std::string to_string( DeviceQueueCreateFlagBits value )
  {
    switch ( value )
    {
      case DeviceQueueCreateFlagBits::eProtected : return "Protected";
      default: return "invalid";
    }
  }

  using DeviceQueueCreateFlags = Flags<DeviceQueueCreateFlagBits, VkDeviceQueueCreateFlags>;

  VULKAN_HPP_INLINE DeviceQueueCreateFlags operator|( DeviceQueueCreateFlagBits bit0, DeviceQueueCreateFlagBits bit1 )
  {
    return DeviceQueueCreateFlags( bit0 ) | bit1;
  }

  VULKAN_HPP_INLINE DeviceQueueCreateFlags operator~( DeviceQueueCreateFlagBits bits )
  {
    return ~( DeviceQueueCreateFlags( bits ) );
  }

  template <> struct FlagTraits<DeviceQueueCreateFlagBits>
  {
    enum
    {
      allFlags = VkFlags(DeviceQueueCreateFlagBits::eProtected)
    };
  };

  VULKAN_HPP_INLINE std::string to_string( DeviceQueueCreateFlags value  )
  {
    if ( !value ) return "{}";
    std::string result;

    if ( value & DeviceQueueCreateFlagBits::eProtected ) result += "Protected";
    return result;
  }

  enum class DisplayModeCreateFlagBitsKHR
  {};

  VULKAN_HPP_INLINE std::string to_string( DisplayModeCreateFlagBitsKHR )
  {
    return "(void)";
  }

  using DisplayModeCreateFlagsKHR = Flags<DisplayModeCreateFlagBitsKHR, VkDisplayModeCreateFlagsKHR>;

  VULKAN_HPP_INLINE std::string to_string( DisplayModeCreateFlagsKHR  )
  {
    return "{}";
  }

  enum class DisplayPlaneAlphaFlagBitsKHR
  {
    eOpaque = VK_DISPLAY_PLANE_ALPHA_OPAQUE_BIT_KHR,
    eGlobal = VK_DISPLAY_PLANE_ALPHA_GLOBAL_BIT_KHR,
    ePerPixel = VK_DISPLAY_PLANE_ALPHA_PER_PIXEL_BIT_KHR,
    ePerPixelPremultiplied = VK_DISPLAY_PLANE_ALPHA_PER_PIXEL_PREMULTIPLIED_BIT_KHR
  };

  VULKAN_HPP_INLINE std::string to_string( DisplayPlaneAlphaFlagBitsKHR value )
  {
    switch ( value )
    {
      case DisplayPlaneAlphaFlagBitsKHR::eOpaque : return "Opaque";
      case DisplayPlaneAlphaFlagBitsKHR::eGlobal : return "Global";
      case DisplayPlaneAlphaFlagBitsKHR::ePerPixel : return "PerPixel";
      case DisplayPlaneAlphaFlagBitsKHR::ePerPixelPremultiplied : return "PerPixelPremultiplied";
      default: return "invalid";
    }
  }

  using DisplayPlaneAlphaFlagsKHR = Flags<DisplayPlaneAlphaFlagBitsKHR, VkDisplayPlaneAlphaFlagsKHR>;

  VULKAN_HPP_INLINE DisplayPlaneAlphaFlagsKHR operator|( DisplayPlaneAlphaFlagBitsKHR bit0, DisplayPlaneAlphaFlagBitsKHR bit1 )
  {
    return DisplayPlaneAlphaFlagsKHR( bit0 ) | bit1;
  }

  VULKAN_HPP_INLINE DisplayPlaneAlphaFlagsKHR operator~( DisplayPlaneAlphaFlagBitsKHR bits )
  {
    return ~( DisplayPlaneAlphaFlagsKHR( bits ) );
  }

  template <> struct FlagTraits<DisplayPlaneAlphaFlagBitsKHR>
  {
    enum
    {
      allFlags = VkFlags(DisplayPlaneAlphaFlagBitsKHR::eOpaque) | VkFlags(DisplayPlaneAlphaFlagBitsKHR::eGlobal) | VkFlags(DisplayPlaneAlphaFlagBitsKHR::ePerPixel) | VkFlags(DisplayPlaneAlphaFlagBitsKHR::ePerPixelPremultiplied)
    };
  };

  VULKAN_HPP_INLINE std::string to_string( DisplayPlaneAlphaFlagsKHR value  )
  {
    if ( !value ) return "{}";
    std::string result;

    if ( value & DisplayPlaneAlphaFlagBitsKHR::eOpaque ) result += "Opaque | ";
    if ( value & DisplayPlaneAlphaFlagBitsKHR::eGlobal ) result += "Global | ";
    if ( value & DisplayPlaneAlphaFlagBitsKHR::ePerPixel ) result += "PerPixel | ";
    if ( value & DisplayPlaneAlphaFlagBitsKHR::ePerPixelPremultiplied ) result += "PerPixelPremultiplied";
    return result;
  }

  enum class DisplaySurfaceCreateFlagBitsKHR
  {};

  VULKAN_HPP_INLINE std::string to_string( DisplaySurfaceCreateFlagBitsKHR )
  {
    return "(void)";
  }

  using DisplaySurfaceCreateFlagsKHR = Flags<DisplaySurfaceCreateFlagBitsKHR, VkDisplaySurfaceCreateFlagsKHR>;

  VULKAN_HPP_INLINE std::string to_string( DisplaySurfaceCreateFlagsKHR  )
  {
    return "{}";
  }

  enum class EventCreateFlagBits
  {};

  VULKAN_HPP_INLINE std::string to_string( EventCreateFlagBits )
  {
    return "(void)";
  }

  using EventCreateFlags = Flags<EventCreateFlagBits, VkEventCreateFlags>;

  VULKAN_HPP_INLINE std::string to_string( EventCreateFlags  )
  {
    return "{}";
  }

  enum class ExternalFenceFeatureFlagBits
  {
    eExportable = VK_EXTERNAL_FENCE_FEATURE_EXPORTABLE_BIT,
    eImportable = VK_EXTERNAL_FENCE_FEATURE_IMPORTABLE_BIT,
    eExportableKHR = VK_EXTERNAL_FENCE_FEATURE_EXPORTABLE_BIT_KHR,
    eImportableKHR = VK_EXTERNAL_FENCE_FEATURE_IMPORTABLE_BIT_KHR
  };

  VULKAN_HPP_INLINE std::string to_string( ExternalFenceFeatureFlagBits value )
  {
    switch ( value )
    {
      case ExternalFenceFeatureFlagBits::eExportable : return "Exportable";
      case ExternalFenceFeatureFlagBits::eImportable : return "Importable";
      default: return "invalid";
    }
  }

  using ExternalFenceFeatureFlags = Flags<ExternalFenceFeatureFlagBits, VkExternalFenceFeatureFlags>;

  VULKAN_HPP_INLINE ExternalFenceFeatureFlags operator|( ExternalFenceFeatureFlagBits bit0, ExternalFenceFeatureFlagBits bit1 )
  {
    return ExternalFenceFeatureFlags( bit0 ) | bit1;
  }

  VULKAN_HPP_INLINE ExternalFenceFeatureFlags operator~( ExternalFenceFeatureFlagBits bits )
  {
    return ~( ExternalFenceFeatureFlags( bits ) );
  }

  template <> struct FlagTraits<ExternalFenceFeatureFlagBits>
  {
    enum
    {
      allFlags = VkFlags(ExternalFenceFeatureFlagBits::eExportable) | VkFlags(ExternalFenceFeatureFlagBits::eImportable)
    };
  };

  using ExternalFenceFeatureFlagsKHR = ExternalFenceFeatureFlags;

  VULKAN_HPP_INLINE std::string to_string( ExternalFenceFeatureFlags value  )
  {
    if ( !value ) return "{}";
    std::string result;

    if ( value & ExternalFenceFeatureFlagBits::eExportable ) result += "Exportable | ";
    if ( value & ExternalFenceFeatureFlagBits::eImportable ) result += "Importable";
    return result;
  }

  enum class ExternalFenceHandleTypeFlagBits
  {
    eOpaqueFd = VK_EXTERNAL_FENCE_HANDLE_TYPE_OPAQUE_FD_BIT,
    eOpaqueWin32 = VK_EXTERNAL_FENCE_HANDLE_TYPE_OPAQUE_WIN32_BIT,
    eOpaqueWin32Kmt = VK_EXTERNAL_FENCE_HANDLE_TYPE_OPAQUE_WIN32_KMT_BIT,
    eSyncFd = VK_EXTERNAL_FENCE_HANDLE_TYPE_SYNC_FD_BIT,
    eOpaqueFdKHR = VK_EXTERNAL_FENCE_HANDLE_TYPE_OPAQUE_FD_BIT_KHR,
    eOpaqueWin32KHR = VK_EXTERNAL_FENCE_HANDLE_TYPE_OPAQUE_WIN32_BIT_KHR,
    eOpaqueWin32KmtKHR = VK_EXTERNAL_FENCE_HANDLE_TYPE_OPAQUE_WIN32_KMT_BIT_KHR,
    eSyncFdKHR = VK_EXTERNAL_FENCE_HANDLE_TYPE_SYNC_FD_BIT_KHR
  };

  VULKAN_HPP_INLINE std::string to_string( ExternalFenceHandleTypeFlagBits value )
  {
    switch ( value )
    {
      case ExternalFenceHandleTypeFlagBits::eOpaqueFd : return "OpaqueFd";
      case ExternalFenceHandleTypeFlagBits::eOpaqueWin32 : return "OpaqueWin32";
      case ExternalFenceHandleTypeFlagBits::eOpaqueWin32Kmt : return "OpaqueWin32Kmt";
      case ExternalFenceHandleTypeFlagBits::eSyncFd : return "SyncFd";
      default: return "invalid";
    }
  }

  using ExternalFenceHandleTypeFlags = Flags<ExternalFenceHandleTypeFlagBits, VkExternalFenceHandleTypeFlags>;

  VULKAN_HPP_INLINE ExternalFenceHandleTypeFlags operator|( ExternalFenceHandleTypeFlagBits bit0, ExternalFenceHandleTypeFlagBits bit1 )
  {
    return ExternalFenceHandleTypeFlags( bit0 ) | bit1;
  }

  VULKAN_HPP_INLINE ExternalFenceHandleTypeFlags operator~( ExternalFenceHandleTypeFlagBits bits )
  {
    return ~( ExternalFenceHandleTypeFlags( bits ) );
  }

  template <> struct FlagTraits<ExternalFenceHandleTypeFlagBits>
  {
    enum
    {
      allFlags = VkFlags(ExternalFenceHandleTypeFlagBits::eOpaqueFd) | VkFlags(ExternalFenceHandleTypeFlagBits::eOpaqueWin32) | VkFlags(ExternalFenceHandleTypeFlagBits::eOpaqueWin32Kmt) | VkFlags(ExternalFenceHandleTypeFlagBits::eSyncFd)
    };
  };

  using ExternalFenceHandleTypeFlagsKHR = ExternalFenceHandleTypeFlags;

  VULKAN_HPP_INLINE std::string to_string( ExternalFenceHandleTypeFlags value  )
  {
    if ( !value ) return "{}";
    std::string result;

    if ( value & ExternalFenceHandleTypeFlagBits::eOpaqueFd ) result += "OpaqueFd | ";
    if ( value & ExternalFenceHandleTypeFlagBits::eOpaqueWin32 ) result += "OpaqueWin32 | ";
    if ( value & ExternalFenceHandleTypeFlagBits::eOpaqueWin32Kmt ) result += "OpaqueWin32Kmt | ";
    if ( value & ExternalFenceHandleTypeFlagBits::eSyncFd ) result += "SyncFd";
    return result;
  }

  enum class ExternalMemoryFeatureFlagBits
  {
    eDedicatedOnly = VK_EXTERNAL_MEMORY_FEATURE_DEDICATED_ONLY_BIT,
    eExportable = VK_EXTERNAL_MEMORY_FEATURE_EXPORTABLE_BIT,
    eImportable = VK_EXTERNAL_MEMORY_FEATURE_IMPORTABLE_BIT,
    eDedicatedOnlyKHR = VK_EXTERNAL_MEMORY_FEATURE_DEDICATED_ONLY_BIT_KHR,
    eExportableKHR = VK_EXTERNAL_MEMORY_FEATURE_EXPORTABLE_BIT_KHR,
    eImportableKHR = VK_EXTERNAL_MEMORY_FEATURE_IMPORTABLE_BIT_KHR
  };

  VULKAN_HPP_INLINE std::string to_string( ExternalMemoryFeatureFlagBits value )
  {
    switch ( value )
    {
      case ExternalMemoryFeatureFlagBits::eDedicatedOnly : return "DedicatedOnly";
      case ExternalMemoryFeatureFlagBits::eExportable : return "Exportable";
      case ExternalMemoryFeatureFlagBits::eImportable : return "Importable";
      default: return "invalid";
    }
  }

  using ExternalMemoryFeatureFlags = Flags<ExternalMemoryFeatureFlagBits, VkExternalMemoryFeatureFlags>;

  VULKAN_HPP_INLINE ExternalMemoryFeatureFlags operator|( ExternalMemoryFeatureFlagBits bit0, ExternalMemoryFeatureFlagBits bit1 )
  {
    return ExternalMemoryFeatureFlags( bit0 ) | bit1;
  }

  VULKAN_HPP_INLINE ExternalMemoryFeatureFlags operator~( ExternalMemoryFeatureFlagBits bits )
  {
    return ~( ExternalMemoryFeatureFlags( bits ) );
  }

  template <> struct FlagTraits<ExternalMemoryFeatureFlagBits>
  {
    enum
    {
      allFlags = VkFlags(ExternalMemoryFeatureFlagBits::eDedicatedOnly) | VkFlags(ExternalMemoryFeatureFlagBits::eExportable) | VkFlags(ExternalMemoryFeatureFlagBits::eImportable)
    };
  };

  using ExternalMemoryFeatureFlagsKHR = ExternalMemoryFeatureFlags;

  VULKAN_HPP_INLINE std::string to_string( ExternalMemoryFeatureFlags value  )
  {
    if ( !value ) return "{}";
    std::string result;

    if ( value & ExternalMemoryFeatureFlagBits::eDedicatedOnly ) result += "DedicatedOnly | ";
    if ( value & ExternalMemoryFeatureFlagBits::eExportable ) result += "Exportable | ";
    if ( value & ExternalMemoryFeatureFlagBits::eImportable ) result += "Importable";
    return result;
  }

  enum class ExternalMemoryFeatureFlagBitsNV
  {
    eDedicatedOnly = VK_EXTERNAL_MEMORY_FEATURE_DEDICATED_ONLY_BIT_NV,
    eExportable = VK_EXTERNAL_MEMORY_FEATURE_EXPORTABLE_BIT_NV,
    eImportable = VK_EXTERNAL_MEMORY_FEATURE_IMPORTABLE_BIT_NV
  };

  VULKAN_HPP_INLINE std::string to_string( ExternalMemoryFeatureFlagBitsNV value )
  {
    switch ( value )
    {
      case ExternalMemoryFeatureFlagBitsNV::eDedicatedOnly : return "DedicatedOnly";
      case ExternalMemoryFeatureFlagBitsNV::eExportable : return "Exportable";
      case ExternalMemoryFeatureFlagBitsNV::eImportable : return "Importable";
      default: return "invalid";
    }
  }

  using ExternalMemoryFeatureFlagsNV = Flags<ExternalMemoryFeatureFlagBitsNV, VkExternalMemoryFeatureFlagsNV>;

  VULKAN_HPP_INLINE ExternalMemoryFeatureFlagsNV operator|( ExternalMemoryFeatureFlagBitsNV bit0, ExternalMemoryFeatureFlagBitsNV bit1 )
  {
    return ExternalMemoryFeatureFlagsNV( bit0 ) | bit1;
  }

  VULKAN_HPP_INLINE ExternalMemoryFeatureFlagsNV operator~( ExternalMemoryFeatureFlagBitsNV bits )
  {
    return ~( ExternalMemoryFeatureFlagsNV( bits ) );
  }

  template <> struct FlagTraits<ExternalMemoryFeatureFlagBitsNV>
  {
    enum
    {
      allFlags = VkFlags(ExternalMemoryFeatureFlagBitsNV::eDedicatedOnly) | VkFlags(ExternalMemoryFeatureFlagBitsNV::eExportable) | VkFlags(ExternalMemoryFeatureFlagBitsNV::eImportable)
    };
  };

  VULKAN_HPP_INLINE std::string to_string( ExternalMemoryFeatureFlagsNV value  )
  {
    if ( !value ) return "{}";
    std::string result;

    if ( value & ExternalMemoryFeatureFlagBitsNV::eDedicatedOnly ) result += "DedicatedOnly | ";
    if ( value & ExternalMemoryFeatureFlagBitsNV::eExportable ) result += "Exportable | ";
    if ( value & ExternalMemoryFeatureFlagBitsNV::eImportable ) result += "Importable";
    return result;
  }

  enum class ExternalMemoryHandleTypeFlagBits
  {
    eOpaqueFd = VK_EXTERNAL_MEMORY_HANDLE_TYPE_OPAQUE_FD_BIT,
    eOpaqueWin32 = VK_EXTERNAL_MEMORY_HANDLE_TYPE_OPAQUE_WIN32_BIT,
    eOpaqueWin32Kmt = VK_EXTERNAL_MEMORY_HANDLE_TYPE_OPAQUE_WIN32_KMT_BIT,
    eD3D11Texture = VK_EXTERNAL_MEMORY_HANDLE_TYPE_D3D11_TEXTURE_BIT,
    eD3D11TextureKmt = VK_EXTERNAL_MEMORY_HANDLE_TYPE_D3D11_TEXTURE_KMT_BIT,
    eD3D12Heap = VK_EXTERNAL_MEMORY_HANDLE_TYPE_D3D12_HEAP_BIT,
    eD3D12Resource = VK_EXTERNAL_MEMORY_HANDLE_TYPE_D3D12_RESOURCE_BIT,
    eDmaBufEXT = VK_EXTERNAL_MEMORY_HANDLE_TYPE_DMA_BUF_BIT_EXT,
    eAndroidHardwareBufferANDROID = VK_EXTERNAL_MEMORY_HANDLE_TYPE_ANDROID_HARDWARE_BUFFER_BIT_ANDROID,
    eHostAllocationEXT = VK_EXTERNAL_MEMORY_HANDLE_TYPE_HOST_ALLOCATION_BIT_EXT,
    eHostMappedForeignMemoryEXT = VK_EXTERNAL_MEMORY_HANDLE_TYPE_HOST_MAPPED_FOREIGN_MEMORY_BIT_EXT,
    eOpaqueFdKHR = VK_EXTERNAL_MEMORY_HANDLE_TYPE_OPAQUE_FD_BIT_KHR,
    eOpaqueWin32KHR = VK_EXTERNAL_MEMORY_HANDLE_TYPE_OPAQUE_WIN32_BIT_KHR,
    eOpaqueWin32KmtKHR = VK_EXTERNAL_MEMORY_HANDLE_TYPE_OPAQUE_WIN32_KMT_BIT_KHR,
    eD3D11TextureKHR = VK_EXTERNAL_MEMORY_HANDLE_TYPE_D3D11_TEXTURE_BIT_KHR,
    eD3D11TextureKmtKHR = VK_EXTERNAL_MEMORY_HANDLE_TYPE_D3D11_TEXTURE_KMT_BIT_KHR,
    eD3D12HeapKHR = VK_EXTERNAL_MEMORY_HANDLE_TYPE_D3D12_HEAP_BIT_KHR,
    eD3D12ResourceKHR = VK_EXTERNAL_MEMORY_HANDLE_TYPE_D3D12_RESOURCE_BIT_KHR
  };

  VULKAN_HPP_INLINE std::string to_string( ExternalMemoryHandleTypeFlagBits value )
  {
    switch ( value )
    {
      case ExternalMemoryHandleTypeFlagBits::eOpaqueFd : return "OpaqueFd";
      case ExternalMemoryHandleTypeFlagBits::eOpaqueWin32 : return "OpaqueWin32";
      case ExternalMemoryHandleTypeFlagBits::eOpaqueWin32Kmt : return "OpaqueWin32Kmt";
      case ExternalMemoryHandleTypeFlagBits::eD3D11Texture : return "D3D11Texture";
      case ExternalMemoryHandleTypeFlagBits::eD3D11TextureKmt : return "D3D11TextureKmt";
      case ExternalMemoryHandleTypeFlagBits::eD3D12Heap : return "D3D12Heap";
      case ExternalMemoryHandleTypeFlagBits::eD3D12Resource : return "D3D12Resource";
      case ExternalMemoryHandleTypeFlagBits::eDmaBufEXT : return "DmaBufEXT";
      case ExternalMemoryHandleTypeFlagBits::eAndroidHardwareBufferANDROID : return "AndroidHardwareBufferANDROID";
      case ExternalMemoryHandleTypeFlagBits::eHostAllocationEXT : return "HostAllocationEXT";
      case ExternalMemoryHandleTypeFlagBits::eHostMappedForeignMemoryEXT : return "HostMappedForeignMemoryEXT";
      default: return "invalid";
    }
  }

  using ExternalMemoryHandleTypeFlags = Flags<ExternalMemoryHandleTypeFlagBits, VkExternalMemoryHandleTypeFlags>;

  VULKAN_HPP_INLINE ExternalMemoryHandleTypeFlags operator|( ExternalMemoryHandleTypeFlagBits bit0, ExternalMemoryHandleTypeFlagBits bit1 )
  {
    return ExternalMemoryHandleTypeFlags( bit0 ) | bit1;
  }

  VULKAN_HPP_INLINE ExternalMemoryHandleTypeFlags operator~( ExternalMemoryHandleTypeFlagBits bits )
  {
    return ~( ExternalMemoryHandleTypeFlags( bits ) );
  }

  template <> struct FlagTraits<ExternalMemoryHandleTypeFlagBits>
  {
    enum
    {
      allFlags = VkFlags(ExternalMemoryHandleTypeFlagBits::eOpaqueFd) | VkFlags(ExternalMemoryHandleTypeFlagBits::eOpaqueWin32) | VkFlags(ExternalMemoryHandleTypeFlagBits::eOpaqueWin32Kmt) | VkFlags(ExternalMemoryHandleTypeFlagBits::eD3D11Texture) | VkFlags(ExternalMemoryHandleTypeFlagBits::eD3D11TextureKmt) | VkFlags(ExternalMemoryHandleTypeFlagBits::eD3D12Heap) | VkFlags(ExternalMemoryHandleTypeFlagBits::eD3D12Resource) | VkFlags(ExternalMemoryHandleTypeFlagBits::eDmaBufEXT) | VkFlags(ExternalMemoryHandleTypeFlagBits::eAndroidHardwareBufferANDROID) | VkFlags(ExternalMemoryHandleTypeFlagBits::eHostAllocationEXT) | VkFlags(ExternalMemoryHandleTypeFlagBits::eHostMappedForeignMemoryEXT)
    };
  };

  using ExternalMemoryHandleTypeFlagsKHR = ExternalMemoryHandleTypeFlags;

  VULKAN_HPP_INLINE std::string to_string( ExternalMemoryHandleTypeFlags value  )
  {
    if ( !value ) return "{}";
    std::string result;

    if ( value & ExternalMemoryHandleTypeFlagBits::eOpaqueFd ) result += "OpaqueFd | ";
    if ( value & ExternalMemoryHandleTypeFlagBits::eOpaqueWin32 ) result += "OpaqueWin32 | ";
    if ( value & ExternalMemoryHandleTypeFlagBits::eOpaqueWin32Kmt ) result += "OpaqueWin32Kmt | ";
    if ( value & ExternalMemoryHandleTypeFlagBits::eD3D11Texture ) result += "D3D11Texture | ";
    if ( value & ExternalMemoryHandleTypeFlagBits::eD3D11TextureKmt ) result += "D3D11TextureKmt | ";
    if ( value & ExternalMemoryHandleTypeFlagBits::eD3D12Heap ) result += "D3D12Heap | ";
    if ( value & ExternalMemoryHandleTypeFlagBits::eD3D12Resource ) result += "D3D12Resource | ";
    if ( value & ExternalMemoryHandleTypeFlagBits::eDmaBufEXT ) result += "DmaBufEXT | ";
    if ( value & ExternalMemoryHandleTypeFlagBits::eAndroidHardwareBufferANDROID ) result += "AndroidHardwareBufferANDROID | ";
    if ( value & ExternalMemoryHandleTypeFlagBits::eHostAllocationEXT ) result += "HostAllocationEXT | ";
    if ( value & ExternalMemoryHandleTypeFlagBits::eHostMappedForeignMemoryEXT ) result += "HostMappedForeignMemoryEXT";
    return result;
  }

  enum class ExternalMemoryHandleTypeFlagBitsNV
  {
    eOpaqueWin32 = VK_EXTERNAL_MEMORY_HANDLE_TYPE_OPAQUE_WIN32_BIT_NV,
    eOpaqueWin32Kmt = VK_EXTERNAL_MEMORY_HANDLE_TYPE_OPAQUE_WIN32_KMT_BIT_NV,
    eD3D11Image = VK_EXTERNAL_MEMORY_HANDLE_TYPE_D3D11_IMAGE_BIT_NV,
    eD3D11ImageKmt = VK_EXTERNAL_MEMORY_HANDLE_TYPE_D3D11_IMAGE_KMT_BIT_NV
  };

  VULKAN_HPP_INLINE std::string to_string( ExternalMemoryHandleTypeFlagBitsNV value )
  {
    switch ( value )
    {
      case ExternalMemoryHandleTypeFlagBitsNV::eOpaqueWin32 : return "OpaqueWin32";
      case ExternalMemoryHandleTypeFlagBitsNV::eOpaqueWin32Kmt : return "OpaqueWin32Kmt";
      case ExternalMemoryHandleTypeFlagBitsNV::eD3D11Image : return "D3D11Image";
      case ExternalMemoryHandleTypeFlagBitsNV::eD3D11ImageKmt : return "D3D11ImageKmt";
      default: return "invalid";
    }
  }

  using ExternalMemoryHandleTypeFlagsNV = Flags<ExternalMemoryHandleTypeFlagBitsNV, VkExternalMemoryHandleTypeFlagsNV>;

  VULKAN_HPP_INLINE ExternalMemoryHandleTypeFlagsNV operator|( ExternalMemoryHandleTypeFlagBitsNV bit0, ExternalMemoryHandleTypeFlagBitsNV bit1 )
  {
    return ExternalMemoryHandleTypeFlagsNV( bit0 ) | bit1;
  }

  VULKAN_HPP_INLINE ExternalMemoryHandleTypeFlagsNV operator~( ExternalMemoryHandleTypeFlagBitsNV bits )
  {
    return ~( ExternalMemoryHandleTypeFlagsNV( bits ) );
  }

  template <> struct FlagTraits<ExternalMemoryHandleTypeFlagBitsNV>
  {
    enum
    {
      allFlags = VkFlags(ExternalMemoryHandleTypeFlagBitsNV::eOpaqueWin32) | VkFlags(ExternalMemoryHandleTypeFlagBitsNV::eOpaqueWin32Kmt) | VkFlags(ExternalMemoryHandleTypeFlagBitsNV::eD3D11Image) | VkFlags(ExternalMemoryHandleTypeFlagBitsNV::eD3D11ImageKmt)
    };
  };

  VULKAN_HPP_INLINE std::string to_string( ExternalMemoryHandleTypeFlagsNV value  )
  {
    if ( !value ) return "{}";
    std::string result;

    if ( value & ExternalMemoryHandleTypeFlagBitsNV::eOpaqueWin32 ) result += "OpaqueWin32 | ";
    if ( value & ExternalMemoryHandleTypeFlagBitsNV::eOpaqueWin32Kmt ) result += "OpaqueWin32Kmt | ";
    if ( value & ExternalMemoryHandleTypeFlagBitsNV::eD3D11Image ) result += "D3D11Image | ";
    if ( value & ExternalMemoryHandleTypeFlagBitsNV::eD3D11ImageKmt ) result += "D3D11ImageKmt";
    return result;
  }

  enum class ExternalSemaphoreFeatureFlagBits
  {
    eExportable = VK_EXTERNAL_SEMAPHORE_FEATURE_EXPORTABLE_BIT,
    eImportable = VK_EXTERNAL_SEMAPHORE_FEATURE_IMPORTABLE_BIT,
    eExportableKHR = VK_EXTERNAL_SEMAPHORE_FEATURE_EXPORTABLE_BIT_KHR,
    eImportableKHR = VK_EXTERNAL_SEMAPHORE_FEATURE_IMPORTABLE_BIT_KHR
  };

  VULKAN_HPP_INLINE std::string to_string( ExternalSemaphoreFeatureFlagBits value )
  {
    switch ( value )
    {
      case ExternalSemaphoreFeatureFlagBits::eExportable : return "Exportable";
      case ExternalSemaphoreFeatureFlagBits::eImportable : return "Importable";
      default: return "invalid";
    }
  }

  using ExternalSemaphoreFeatureFlags = Flags<ExternalSemaphoreFeatureFlagBits, VkExternalSemaphoreFeatureFlags>;

  VULKAN_HPP_INLINE ExternalSemaphoreFeatureFlags operator|( ExternalSemaphoreFeatureFlagBits bit0, ExternalSemaphoreFeatureFlagBits bit1 )
  {
    return ExternalSemaphoreFeatureFlags( bit0 ) | bit1;
  }

  VULKAN_HPP_INLINE ExternalSemaphoreFeatureFlags operator~( ExternalSemaphoreFeatureFlagBits bits )
  {
    return ~( ExternalSemaphoreFeatureFlags( bits ) );
  }

  template <> struct FlagTraits<ExternalSemaphoreFeatureFlagBits>
  {
    enum
    {
      allFlags = VkFlags(ExternalSemaphoreFeatureFlagBits::eExportable) | VkFlags(ExternalSemaphoreFeatureFlagBits::eImportable)
    };
  };

  using ExternalSemaphoreFeatureFlagsKHR = ExternalSemaphoreFeatureFlags;

  VULKAN_HPP_INLINE std::string to_string( ExternalSemaphoreFeatureFlags value  )
  {
    if ( !value ) return "{}";
    std::string result;

    if ( value & ExternalSemaphoreFeatureFlagBits::eExportable ) result += "Exportable | ";
    if ( value & ExternalSemaphoreFeatureFlagBits::eImportable ) result += "Importable";
    return result;
  }

  enum class ExternalSemaphoreHandleTypeFlagBits
  {
    eOpaqueFd = VK_EXTERNAL_SEMAPHORE_HANDLE_TYPE_OPAQUE_FD_BIT,
    eOpaqueWin32 = VK_EXTERNAL_SEMAPHORE_HANDLE_TYPE_OPAQUE_WIN32_BIT,
    eOpaqueWin32Kmt = VK_EXTERNAL_SEMAPHORE_HANDLE_TYPE_OPAQUE_WIN32_KMT_BIT,
    eD3D12Fence = VK_EXTERNAL_SEMAPHORE_HANDLE_TYPE_D3D12_FENCE_BIT,
    eSyncFd = VK_EXTERNAL_SEMAPHORE_HANDLE_TYPE_SYNC_FD_BIT,
    eOpaqueFdKHR = VK_EXTERNAL_SEMAPHORE_HANDLE_TYPE_OPAQUE_FD_BIT_KHR,
    eOpaqueWin32KHR = VK_EXTERNAL_SEMAPHORE_HANDLE_TYPE_OPAQUE_WIN32_BIT_KHR,
    eOpaqueWin32KmtKHR = VK_EXTERNAL_SEMAPHORE_HANDLE_TYPE_OPAQUE_WIN32_KMT_BIT_KHR,
    eD3D12FenceKHR = VK_EXTERNAL_SEMAPHORE_HANDLE_TYPE_D3D12_FENCE_BIT_KHR,
    eSyncFdKHR = VK_EXTERNAL_SEMAPHORE_HANDLE_TYPE_SYNC_FD_BIT_KHR
  };

  VULKAN_HPP_INLINE std::string to_string( ExternalSemaphoreHandleTypeFlagBits value )
  {
    switch ( value )
    {
      case ExternalSemaphoreHandleTypeFlagBits::eOpaqueFd : return "OpaqueFd";
      case ExternalSemaphoreHandleTypeFlagBits::eOpaqueWin32 : return "OpaqueWin32";
      case ExternalSemaphoreHandleTypeFlagBits::eOpaqueWin32Kmt : return "OpaqueWin32Kmt";
      case ExternalSemaphoreHandleTypeFlagBits::eD3D12Fence : return "D3D12Fence";
      case ExternalSemaphoreHandleTypeFlagBits::eSyncFd : return "SyncFd";
      default: return "invalid";
    }
  }

  using ExternalSemaphoreHandleTypeFlags = Flags<ExternalSemaphoreHandleTypeFlagBits, VkExternalSemaphoreHandleTypeFlags>;

  VULKAN_HPP_INLINE ExternalSemaphoreHandleTypeFlags operator|( ExternalSemaphoreHandleTypeFlagBits bit0, ExternalSemaphoreHandleTypeFlagBits bit1 )
  {
    return ExternalSemaphoreHandleTypeFlags( bit0 ) | bit1;
  }

  VULKAN_HPP_INLINE ExternalSemaphoreHandleTypeFlags operator~( ExternalSemaphoreHandleTypeFlagBits bits )
  {
    return ~( ExternalSemaphoreHandleTypeFlags( bits ) );
  }

  template <> struct FlagTraits<ExternalSemaphoreHandleTypeFlagBits>
  {
    enum
    {
      allFlags = VkFlags(ExternalSemaphoreHandleTypeFlagBits::eOpaqueFd) | VkFlags(ExternalSemaphoreHandleTypeFlagBits::eOpaqueWin32) | VkFlags(ExternalSemaphoreHandleTypeFlagBits::eOpaqueWin32Kmt) | VkFlags(ExternalSemaphoreHandleTypeFlagBits::eD3D12Fence) | VkFlags(ExternalSemaphoreHandleTypeFlagBits::eSyncFd)
    };
  };

  using ExternalSemaphoreHandleTypeFlagsKHR = ExternalSemaphoreHandleTypeFlags;

  VULKAN_HPP_INLINE std::string to_string( ExternalSemaphoreHandleTypeFlags value  )
  {
    if ( !value ) return "{}";
    std::string result;

    if ( value & ExternalSemaphoreHandleTypeFlagBits::eOpaqueFd ) result += "OpaqueFd | ";
    if ( value & ExternalSemaphoreHandleTypeFlagBits::eOpaqueWin32 ) result += "OpaqueWin32 | ";
    if ( value & ExternalSemaphoreHandleTypeFlagBits::eOpaqueWin32Kmt ) result += "OpaqueWin32Kmt | ";
    if ( value & ExternalSemaphoreHandleTypeFlagBits::eD3D12Fence ) result += "D3D12Fence | ";
    if ( value & ExternalSemaphoreHandleTypeFlagBits::eSyncFd ) result += "SyncFd";
    return result;
  }

  enum class FenceCreateFlagBits
  {
    eSignaled = VK_FENCE_CREATE_SIGNALED_BIT
  };

  VULKAN_HPP_INLINE std::string to_string( FenceCreateFlagBits value )
  {
    switch ( value )
    {
      case FenceCreateFlagBits::eSignaled : return "Signaled";
      default: return "invalid";
    }
  }

  using FenceCreateFlags = Flags<FenceCreateFlagBits, VkFenceCreateFlags>;

  VULKAN_HPP_INLINE FenceCreateFlags operator|( FenceCreateFlagBits bit0, FenceCreateFlagBits bit1 )
  {
    return FenceCreateFlags( bit0 ) | bit1;
  }

  VULKAN_HPP_INLINE FenceCreateFlags operator~( FenceCreateFlagBits bits )
  {
    return ~( FenceCreateFlags( bits ) );
  }

  template <> struct FlagTraits<FenceCreateFlagBits>
  {
    enum
    {
      allFlags = VkFlags(FenceCreateFlagBits::eSignaled)
    };
  };

  VULKAN_HPP_INLINE std::string to_string( FenceCreateFlags value  )
  {
    if ( !value ) return "{}";
    std::string result;

    if ( value & FenceCreateFlagBits::eSignaled ) result += "Signaled";
    return result;
  }

  enum class FenceImportFlagBits
  {
    eTemporary = VK_FENCE_IMPORT_TEMPORARY_BIT,
    eTemporaryKHR = VK_FENCE_IMPORT_TEMPORARY_BIT_KHR
  };

  VULKAN_HPP_INLINE std::string to_string( FenceImportFlagBits value )
  {
    switch ( value )
    {
      case FenceImportFlagBits::eTemporary : return "Temporary";
      default: return "invalid";
    }
  }

  using FenceImportFlags = Flags<FenceImportFlagBits, VkFenceImportFlags>;

  VULKAN_HPP_INLINE FenceImportFlags operator|( FenceImportFlagBits bit0, FenceImportFlagBits bit1 )
  {
    return FenceImportFlags( bit0 ) | bit1;
  }

  VULKAN_HPP_INLINE FenceImportFlags operator~( FenceImportFlagBits bits )
  {
    return ~( FenceImportFlags( bits ) );
  }

  template <> struct FlagTraits<FenceImportFlagBits>
  {
    enum
    {
      allFlags = VkFlags(FenceImportFlagBits::eTemporary)
    };
  };

  using FenceImportFlagsKHR = FenceImportFlags;

  VULKAN_HPP_INLINE std::string to_string( FenceImportFlags value  )
  {
    if ( !value ) return "{}";
    std::string result;

    if ( value & FenceImportFlagBits::eTemporary ) result += "Temporary";
    return result;
  }

  enum class FormatFeatureFlagBits
  {
    eSampledImage = VK_FORMAT_FEATURE_SAMPLED_IMAGE_BIT,
    eStorageImage = VK_FORMAT_FEATURE_STORAGE_IMAGE_BIT,
    eStorageImageAtomic = VK_FORMAT_FEATURE_STORAGE_IMAGE_ATOMIC_BIT,
    eUniformTexelBuffer = VK_FORMAT_FEATURE_UNIFORM_TEXEL_BUFFER_BIT,
    eStorageTexelBuffer = VK_FORMAT_FEATURE_STORAGE_TEXEL_BUFFER_BIT,
    eStorageTexelBufferAtomic = VK_FORMAT_FEATURE_STORAGE_TEXEL_BUFFER_ATOMIC_BIT,
    eVertexBuffer = VK_FORMAT_FEATURE_VERTEX_BUFFER_BIT,
    eColorAttachment = VK_FORMAT_FEATURE_COLOR_ATTACHMENT_BIT,
    eColorAttachmentBlend = VK_FORMAT_FEATURE_COLOR_ATTACHMENT_BLEND_BIT,
    eDepthStencilAttachment = VK_FORMAT_FEATURE_DEPTH_STENCIL_ATTACHMENT_BIT,
    eBlitSrc = VK_FORMAT_FEATURE_BLIT_SRC_BIT,
    eBlitDst = VK_FORMAT_FEATURE_BLIT_DST_BIT,
    eSampledImageFilterLinear = VK_FORMAT_FEATURE_SAMPLED_IMAGE_FILTER_LINEAR_BIT,
    eTransferSrc = VK_FORMAT_FEATURE_TRANSFER_SRC_BIT,
    eTransferDst = VK_FORMAT_FEATURE_TRANSFER_DST_BIT,
    eMidpointChromaSamples = VK_FORMAT_FEATURE_MIDPOINT_CHROMA_SAMPLES_BIT,
    eSampledImageYcbcrConversionLinearFilter = VK_FORMAT_FEATURE_SAMPLED_IMAGE_YCBCR_CONVERSION_LINEAR_FILTER_BIT,
    eSampledImageYcbcrConversionSeparateReconstructionFilter = VK_FORMAT_FEATURE_SAMPLED_IMAGE_YCBCR_CONVERSION_SEPARATE_RECONSTRUCTION_FILTER_BIT,
    eSampledImageYcbcrConversionChromaReconstructionExplicit = VK_FORMAT_FEATURE_SAMPLED_IMAGE_YCBCR_CONVERSION_CHROMA_RECONSTRUCTION_EXPLICIT_BIT,
    eSampledImageYcbcrConversionChromaReconstructionExplicitForceable = VK_FORMAT_FEATURE_SAMPLED_IMAGE_YCBCR_CONVERSION_CHROMA_RECONSTRUCTION_EXPLICIT_FORCEABLE_BIT,
    eDisjoint = VK_FORMAT_FEATURE_DISJOINT_BIT,
    eCositedChromaSamples = VK_FORMAT_FEATURE_COSITED_CHROMA_SAMPLES_BIT,
    eSampledImageFilterCubicIMG = VK_FORMAT_FEATURE_SAMPLED_IMAGE_FILTER_CUBIC_BIT_IMG,
    eSampledImageFilterMinmaxEXT = VK_FORMAT_FEATURE_SAMPLED_IMAGE_FILTER_MINMAX_BIT_EXT,
    eFragmentDensityMapEXT = VK_FORMAT_FEATURE_FRAGMENT_DENSITY_MAP_BIT_EXT,
    eTransferSrcKHR = VK_FORMAT_FEATURE_TRANSFER_SRC_BIT_KHR,
    eTransferDstKHR = VK_FORMAT_FEATURE_TRANSFER_DST_BIT_KHR,
    eMidpointChromaSamplesKHR = VK_FORMAT_FEATURE_MIDPOINT_CHROMA_SAMPLES_BIT_KHR,
    eSampledImageYcbcrConversionLinearFilterKHR = VK_FORMAT_FEATURE_SAMPLED_IMAGE_YCBCR_CONVERSION_LINEAR_FILTER_BIT_KHR,
    eSampledImageYcbcrConversionSeparateReconstructionFilterKHR = VK_FORMAT_FEATURE_SAMPLED_IMAGE_YCBCR_CONVERSION_SEPARATE_RECONSTRUCTION_FILTER_BIT_KHR,
    eSampledImageYcbcrConversionChromaReconstructionExplicitKHR = VK_FORMAT_FEATURE_SAMPLED_IMAGE_YCBCR_CONVERSION_CHROMA_RECONSTRUCTION_EXPLICIT_BIT_KHR,
    eSampledImageYcbcrConversionChromaReconstructionExplicitForceableKHR = VK_FORMAT_FEATURE_SAMPLED_IMAGE_YCBCR_CONVERSION_CHROMA_RECONSTRUCTION_EXPLICIT_FORCEABLE_BIT_KHR,
    eDisjointKHR = VK_FORMAT_FEATURE_DISJOINT_BIT_KHR,
    eCositedChromaSamplesKHR = VK_FORMAT_FEATURE_COSITED_CHROMA_SAMPLES_BIT_KHR,
    eSampledImageFilterCubicEXT = VK_FORMAT_FEATURE_SAMPLED_IMAGE_FILTER_CUBIC_BIT_EXT
  };

  VULKAN_HPP_INLINE std::string to_string( FormatFeatureFlagBits value )
  {
    switch ( value )
    {
      case FormatFeatureFlagBits::eSampledImage : return "SampledImage";
      case FormatFeatureFlagBits::eStorageImage : return "StorageImage";
      case FormatFeatureFlagBits::eStorageImageAtomic : return "StorageImageAtomic";
      case FormatFeatureFlagBits::eUniformTexelBuffer : return "UniformTexelBuffer";
      case FormatFeatureFlagBits::eStorageTexelBuffer : return "StorageTexelBuffer";
      case FormatFeatureFlagBits::eStorageTexelBufferAtomic : return "StorageTexelBufferAtomic";
      case FormatFeatureFlagBits::eVertexBuffer : return "VertexBuffer";
      case FormatFeatureFlagBits::eColorAttachment : return "ColorAttachment";
      case FormatFeatureFlagBits::eColorAttachmentBlend : return "ColorAttachmentBlend";
      case FormatFeatureFlagBits::eDepthStencilAttachment : return "DepthStencilAttachment";
      case FormatFeatureFlagBits::eBlitSrc : return "BlitSrc";
      case FormatFeatureFlagBits::eBlitDst : return "BlitDst";
      case FormatFeatureFlagBits::eSampledImageFilterLinear : return "SampledImageFilterLinear";
      case FormatFeatureFlagBits::eTransferSrc : return "TransferSrc";
      case FormatFeatureFlagBits::eTransferDst : return "TransferDst";
      case FormatFeatureFlagBits::eMidpointChromaSamples : return "MidpointChromaSamples";
      case FormatFeatureFlagBits::eSampledImageYcbcrConversionLinearFilter : return "SampledImageYcbcrConversionLinearFilter";
      case FormatFeatureFlagBits::eSampledImageYcbcrConversionSeparateReconstructionFilter : return "SampledImageYcbcrConversionSeparateReconstructionFilter";
      case FormatFeatureFlagBits::eSampledImageYcbcrConversionChromaReconstructionExplicit : return "SampledImageYcbcrConversionChromaReconstructionExplicit";
      case FormatFeatureFlagBits::eSampledImageYcbcrConversionChromaReconstructionExplicitForceable : return "SampledImageYcbcrConversionChromaReconstructionExplicitForceable";
      case FormatFeatureFlagBits::eDisjoint : return "Disjoint";
      case FormatFeatureFlagBits::eCositedChromaSamples : return "CositedChromaSamples";
      case FormatFeatureFlagBits::eSampledImageFilterCubicIMG : return "SampledImageFilterCubicIMG";
      case FormatFeatureFlagBits::eSampledImageFilterMinmaxEXT : return "SampledImageFilterMinmaxEXT";
      case FormatFeatureFlagBits::eFragmentDensityMapEXT : return "FragmentDensityMapEXT";
      default: return "invalid";
    }
  }

  using FormatFeatureFlags = Flags<FormatFeatureFlagBits, VkFormatFeatureFlags>;

  VULKAN_HPP_INLINE FormatFeatureFlags operator|( FormatFeatureFlagBits bit0, FormatFeatureFlagBits bit1 )
  {
    return FormatFeatureFlags( bit0 ) | bit1;
  }

  VULKAN_HPP_INLINE FormatFeatureFlags operator~( FormatFeatureFlagBits bits )
  {
    return ~( FormatFeatureFlags( bits ) );
  }

  template <> struct FlagTraits<FormatFeatureFlagBits>
  {
    enum
    {
      allFlags = VkFlags(FormatFeatureFlagBits::eSampledImage) | VkFlags(FormatFeatureFlagBits::eStorageImage) | VkFlags(FormatFeatureFlagBits::eStorageImageAtomic) | VkFlags(FormatFeatureFlagBits::eUniformTexelBuffer) | VkFlags(FormatFeatureFlagBits::eStorageTexelBuffer) | VkFlags(FormatFeatureFlagBits::eStorageTexelBufferAtomic) | VkFlags(FormatFeatureFlagBits::eVertexBuffer) | VkFlags(FormatFeatureFlagBits::eColorAttachment) | VkFlags(FormatFeatureFlagBits::eColorAttachmentBlend) | VkFlags(FormatFeatureFlagBits::eDepthStencilAttachment) | VkFlags(FormatFeatureFlagBits::eBlitSrc) | VkFlags(FormatFeatureFlagBits::eBlitDst) | VkFlags(FormatFeatureFlagBits::eSampledImageFilterLinear) | VkFlags(FormatFeatureFlagBits::eTransferSrc) | VkFlags(FormatFeatureFlagBits::eTransferDst) | VkFlags(FormatFeatureFlagBits::eMidpointChromaSamples) | VkFlags(FormatFeatureFlagBits::eSampledImageYcbcrConversionLinearFilter) | VkFlags(FormatFeatureFlagBits::eSampledImageYcbcrConversionSeparateReconstructionFilter) | VkFlags(FormatFeatureFlagBits::eSampledImageYcbcrConversionChromaReconstructionExplicit) | VkFlags(FormatFeatureFlagBits::eSampledImageYcbcrConversionChromaReconstructionExplicitForceable) | VkFlags(FormatFeatureFlagBits::eDisjoint) | VkFlags(FormatFeatureFlagBits::eCositedChromaSamples) | VkFlags(FormatFeatureFlagBits::eSampledImageFilterCubicIMG) | VkFlags(FormatFeatureFlagBits::eSampledImageFilterMinmaxEXT) | VkFlags(FormatFeatureFlagBits::eFragmentDensityMapEXT)
    };
  };

  VULKAN_HPP_INLINE std::string to_string( FormatFeatureFlags value  )
  {
    if ( !value ) return "{}";
    std::string result;

    if ( value & FormatFeatureFlagBits::eSampledImage ) result += "SampledImage | ";
    if ( value & FormatFeatureFlagBits::eStorageImage ) result += "StorageImage | ";
    if ( value & FormatFeatureFlagBits::eStorageImageAtomic ) result += "StorageImageAtomic | ";
    if ( value & FormatFeatureFlagBits::eUniformTexelBuffer ) result += "UniformTexelBuffer | ";
    if ( value & FormatFeatureFlagBits::eStorageTexelBuffer ) result += "StorageTexelBuffer | ";
    if ( value & FormatFeatureFlagBits::eStorageTexelBufferAtomic ) result += "StorageTexelBufferAtomic | ";
    if ( value & FormatFeatureFlagBits::eVertexBuffer ) result += "VertexBuffer | ";
    if ( value & FormatFeatureFlagBits::eColorAttachment ) result += "ColorAttachment | ";
    if ( value & FormatFeatureFlagBits::eColorAttachmentBlend ) result += "ColorAttachmentBlend | ";
    if ( value & FormatFeatureFlagBits::eDepthStencilAttachment ) result += "DepthStencilAttachment | ";
    if ( value & FormatFeatureFlagBits::eBlitSrc ) result += "BlitSrc | ";
    if ( value & FormatFeatureFlagBits::eBlitDst ) result += "BlitDst | ";
    if ( value & FormatFeatureFlagBits::eSampledImageFilterLinear ) result += "SampledImageFilterLinear | ";
    if ( value & FormatFeatureFlagBits::eTransferSrc ) result += "TransferSrc | ";
    if ( value & FormatFeatureFlagBits::eTransferDst ) result += "TransferDst | ";
    if ( value & FormatFeatureFlagBits::eMidpointChromaSamples ) result += "MidpointChromaSamples | ";
    if ( value & FormatFeatureFlagBits::eSampledImageYcbcrConversionLinearFilter ) result += "SampledImageYcbcrConversionLinearFilter | ";
    if ( value & FormatFeatureFlagBits::eSampledImageYcbcrConversionSeparateReconstructionFilter ) result += "SampledImageYcbcrConversionSeparateReconstructionFilter | ";
    if ( value & FormatFeatureFlagBits::eSampledImageYcbcrConversionChromaReconstructionExplicit ) result += "SampledImageYcbcrConversionChromaReconstructionExplicit | ";
    if ( value & FormatFeatureFlagBits::eSampledImageYcbcrConversionChromaReconstructionExplicitForceable ) result += "SampledImageYcbcrConversionChromaReconstructionExplicitForceable | ";
    if ( value & FormatFeatureFlagBits::eDisjoint ) result += "Disjoint | ";
    if ( value & FormatFeatureFlagBits::eCositedChromaSamples ) result += "CositedChromaSamples | ";
    if ( value & FormatFeatureFlagBits::eSampledImageFilterCubicIMG ) result += "SampledImageFilterCubicIMG | ";
    if ( value & FormatFeatureFlagBits::eSampledImageFilterMinmaxEXT ) result += "SampledImageFilterMinmaxEXT | ";
    if ( value & FormatFeatureFlagBits::eFragmentDensityMapEXT ) result += "FragmentDensityMapEXT";
    return result;
  }

  enum class FramebufferCreateFlagBits
  {};

  VULKAN_HPP_INLINE std::string to_string( FramebufferCreateFlagBits )
  {
    return "(void)";
  }

  using FramebufferCreateFlags = Flags<FramebufferCreateFlagBits, VkFramebufferCreateFlags>;

  VULKAN_HPP_INLINE std::string to_string( FramebufferCreateFlags  )
  {
    return "{}";
  }

  enum class GeometryFlagBitsNV
  {
    eOpaque = VK_GEOMETRY_OPAQUE_BIT_NV,
    eNoDuplicateAnyHitInvocation = VK_GEOMETRY_NO_DUPLICATE_ANY_HIT_INVOCATION_BIT_NV
  };

  VULKAN_HPP_INLINE std::string to_string( GeometryFlagBitsNV value )
  {
    switch ( value )
    {
      case GeometryFlagBitsNV::eOpaque : return "Opaque";
      case GeometryFlagBitsNV::eNoDuplicateAnyHitInvocation : return "NoDuplicateAnyHitInvocation";
      default: return "invalid";
    }
  }

  using GeometryFlagsNV = Flags<GeometryFlagBitsNV, VkGeometryFlagsNV>;

  VULKAN_HPP_INLINE GeometryFlagsNV operator|( GeometryFlagBitsNV bit0, GeometryFlagBitsNV bit1 )
  {
    return GeometryFlagsNV( bit0 ) | bit1;
  }

  VULKAN_HPP_INLINE GeometryFlagsNV operator~( GeometryFlagBitsNV bits )
  {
    return ~( GeometryFlagsNV( bits ) );
  }

  template <> struct FlagTraits<GeometryFlagBitsNV>
  {
    enum
    {
      allFlags = VkFlags(GeometryFlagBitsNV::eOpaque) | VkFlags(GeometryFlagBitsNV::eNoDuplicateAnyHitInvocation)
    };
  };

  VULKAN_HPP_INLINE std::string to_string( GeometryFlagsNV value  )
  {
    if ( !value ) return "{}";
    std::string result;

    if ( value & GeometryFlagBitsNV::eOpaque ) result += "Opaque | ";
    if ( value & GeometryFlagBitsNV::eNoDuplicateAnyHitInvocation ) result += "NoDuplicateAnyHitInvocation";
    return result;
  }

  enum class GeometryInstanceFlagBitsNV
  {
    eTriangleCullDisable = VK_GEOMETRY_INSTANCE_TRIANGLE_CULL_DISABLE_BIT_NV,
    eTriangleFrontCounterclockwise = VK_GEOMETRY_INSTANCE_TRIANGLE_FRONT_COUNTERCLOCKWISE_BIT_NV,
    eForceOpaque = VK_GEOMETRY_INSTANCE_FORCE_OPAQUE_BIT_NV,
    eForceNoOpaque = VK_GEOMETRY_INSTANCE_FORCE_NO_OPAQUE_BIT_NV
  };

  VULKAN_HPP_INLINE std::string to_string( GeometryInstanceFlagBitsNV value )
  {
    switch ( value )
    {
      case GeometryInstanceFlagBitsNV::eTriangleCullDisable : return "TriangleCullDisable";
      case GeometryInstanceFlagBitsNV::eTriangleFrontCounterclockwise : return "TriangleFrontCounterclockwise";
      case GeometryInstanceFlagBitsNV::eForceOpaque : return "ForceOpaque";
      case GeometryInstanceFlagBitsNV::eForceNoOpaque : return "ForceNoOpaque";
      default: return "invalid";
    }
  }

  using GeometryInstanceFlagsNV = Flags<GeometryInstanceFlagBitsNV, VkGeometryInstanceFlagsNV>;

  VULKAN_HPP_INLINE GeometryInstanceFlagsNV operator|( GeometryInstanceFlagBitsNV bit0, GeometryInstanceFlagBitsNV bit1 )
  {
    return GeometryInstanceFlagsNV( bit0 ) | bit1;
  }

  VULKAN_HPP_INLINE GeometryInstanceFlagsNV operator~( GeometryInstanceFlagBitsNV bits )
  {
    return ~( GeometryInstanceFlagsNV( bits ) );
  }

  template <> struct FlagTraits<GeometryInstanceFlagBitsNV>
  {
    enum
    {
      allFlags = VkFlags(GeometryInstanceFlagBitsNV::eTriangleCullDisable) | VkFlags(GeometryInstanceFlagBitsNV::eTriangleFrontCounterclockwise) | VkFlags(GeometryInstanceFlagBitsNV::eForceOpaque) | VkFlags(GeometryInstanceFlagBitsNV::eForceNoOpaque)
    };
  };

  VULKAN_HPP_INLINE std::string to_string( GeometryInstanceFlagsNV value  )
  {
    if ( !value ) return "{}";
    std::string result;

    if ( value & GeometryInstanceFlagBitsNV::eTriangleCullDisable ) result += "TriangleCullDisable | ";
    if ( value & GeometryInstanceFlagBitsNV::eTriangleFrontCounterclockwise ) result += "TriangleFrontCounterclockwise | ";
    if ( value & GeometryInstanceFlagBitsNV::eForceOpaque ) result += "ForceOpaque | ";
    if ( value & GeometryInstanceFlagBitsNV::eForceNoOpaque ) result += "ForceNoOpaque";
    return result;
  }

#ifdef VK_USE_PLATFORM_IOS_MVK
  enum class IOSSurfaceCreateFlagBitsMVK
  {};

  VULKAN_HPP_INLINE std::string to_string( IOSSurfaceCreateFlagBitsMVK )
  {
    return "(void)";
  }

  using IOSSurfaceCreateFlagsMVK = Flags<IOSSurfaceCreateFlagBitsMVK, VkIOSSurfaceCreateFlagsMVK>;

  VULKAN_HPP_INLINE std::string to_string( IOSSurfaceCreateFlagsMVK  )
  {
    return "{}";
  }
#endif /*VK_USE_PLATFORM_IOS_MVK*/

  enum class ImageAspectFlagBits
  {
    eColor = VK_IMAGE_ASPECT_COLOR_BIT,
    eDepth = VK_IMAGE_ASPECT_DEPTH_BIT,
    eStencil = VK_IMAGE_ASPECT_STENCIL_BIT,
    eMetadata = VK_IMAGE_ASPECT_METADATA_BIT,
    ePlane0 = VK_IMAGE_ASPECT_PLANE_0_BIT,
    ePlane1 = VK_IMAGE_ASPECT_PLANE_1_BIT,
    ePlane2 = VK_IMAGE_ASPECT_PLANE_2_BIT,
    eMemoryPlane0EXT = VK_IMAGE_ASPECT_MEMORY_PLANE_0_BIT_EXT,
    eMemoryPlane1EXT = VK_IMAGE_ASPECT_MEMORY_PLANE_1_BIT_EXT,
    eMemoryPlane2EXT = VK_IMAGE_ASPECT_MEMORY_PLANE_2_BIT_EXT,
    eMemoryPlane3EXT = VK_IMAGE_ASPECT_MEMORY_PLANE_3_BIT_EXT,
    ePlane0KHR = VK_IMAGE_ASPECT_PLANE_0_BIT_KHR,
    ePlane1KHR = VK_IMAGE_ASPECT_PLANE_1_BIT_KHR,
    ePlane2KHR = VK_IMAGE_ASPECT_PLANE_2_BIT_KHR
  };

  VULKAN_HPP_INLINE std::string to_string( ImageAspectFlagBits value )
  {
    switch ( value )
    {
      case ImageAspectFlagBits::eColor : return "Color";
      case ImageAspectFlagBits::eDepth : return "Depth";
      case ImageAspectFlagBits::eStencil : return "Stencil";
      case ImageAspectFlagBits::eMetadata : return "Metadata";
      case ImageAspectFlagBits::ePlane0 : return "Plane0";
      case ImageAspectFlagBits::ePlane1 : return "Plane1";
      case ImageAspectFlagBits::ePlane2 : return "Plane2";
      case ImageAspectFlagBits::eMemoryPlane0EXT : return "MemoryPlane0EXT";
      case ImageAspectFlagBits::eMemoryPlane1EXT : return "MemoryPlane1EXT";
      case ImageAspectFlagBits::eMemoryPlane2EXT : return "MemoryPlane2EXT";
      case ImageAspectFlagBits::eMemoryPlane3EXT : return "MemoryPlane3EXT";
      default: return "invalid";
    }
  }

  using ImageAspectFlags = Flags<ImageAspectFlagBits, VkImageAspectFlags>;

  VULKAN_HPP_INLINE ImageAspectFlags operator|( ImageAspectFlagBits bit0, ImageAspectFlagBits bit1 )
  {
    return ImageAspectFlags( bit0 ) | bit1;
  }

  VULKAN_HPP_INLINE ImageAspectFlags operator~( ImageAspectFlagBits bits )
  {
    return ~( ImageAspectFlags( bits ) );
  }

  template <> struct FlagTraits<ImageAspectFlagBits>
  {
    enum
    {
      allFlags = VkFlags(ImageAspectFlagBits::eColor) | VkFlags(ImageAspectFlagBits::eDepth) | VkFlags(ImageAspectFlagBits::eStencil) | VkFlags(ImageAspectFlagBits::eMetadata) | VkFlags(ImageAspectFlagBits::ePlane0) | VkFlags(ImageAspectFlagBits::ePlane1) | VkFlags(ImageAspectFlagBits::ePlane2) | VkFlags(ImageAspectFlagBits::eMemoryPlane0EXT) | VkFlags(ImageAspectFlagBits::eMemoryPlane1EXT) | VkFlags(ImageAspectFlagBits::eMemoryPlane2EXT) | VkFlags(ImageAspectFlagBits::eMemoryPlane3EXT)
    };
  };

  VULKAN_HPP_INLINE std::string to_string( ImageAspectFlags value  )
  {
    if ( !value ) return "{}";
    std::string result;

    if ( value & ImageAspectFlagBits::eColor ) result += "Color | ";
    if ( value & ImageAspectFlagBits::eDepth ) result += "Depth | ";
    if ( value & ImageAspectFlagBits::eStencil ) result += "Stencil | ";
    if ( value & ImageAspectFlagBits::eMetadata ) result += "Metadata | ";
    if ( value & ImageAspectFlagBits::ePlane0 ) result += "Plane0 | ";
    if ( value & ImageAspectFlagBits::ePlane1 ) result += "Plane1 | ";
    if ( value & ImageAspectFlagBits::ePlane2 ) result += "Plane2 | ";
    if ( value & ImageAspectFlagBits::eMemoryPlane0EXT ) result += "MemoryPlane0EXT | ";
    if ( value & ImageAspectFlagBits::eMemoryPlane1EXT ) result += "MemoryPlane1EXT | ";
    if ( value & ImageAspectFlagBits::eMemoryPlane2EXT ) result += "MemoryPlane2EXT | ";
    if ( value & ImageAspectFlagBits::eMemoryPlane3EXT ) result += "MemoryPlane3EXT";
    return result;
  }

  enum class ImageCreateFlagBits
  {
    eSparseBinding = VK_IMAGE_CREATE_SPARSE_BINDING_BIT,
    eSparseResidency = VK_IMAGE_CREATE_SPARSE_RESIDENCY_BIT,
    eSparseAliased = VK_IMAGE_CREATE_SPARSE_ALIASED_BIT,
    eMutableFormat = VK_IMAGE_CREATE_MUTABLE_FORMAT_BIT,
    eCubeCompatible = VK_IMAGE_CREATE_CUBE_COMPATIBLE_BIT,
    eAlias = VK_IMAGE_CREATE_ALIAS_BIT,
    eSplitInstanceBindRegions = VK_IMAGE_CREATE_SPLIT_INSTANCE_BIND_REGIONS_BIT,
    e2DArrayCompatible = VK_IMAGE_CREATE_2D_ARRAY_COMPATIBLE_BIT,
    eBlockTexelViewCompatible = VK_IMAGE_CREATE_BLOCK_TEXEL_VIEW_COMPATIBLE_BIT,
    eExtendedUsage = VK_IMAGE_CREATE_EXTENDED_USAGE_BIT,
    eProtected = VK_IMAGE_CREATE_PROTECTED_BIT,
    eDisjoint = VK_IMAGE_CREATE_DISJOINT_BIT,
    eCornerSampledNV = VK_IMAGE_CREATE_CORNER_SAMPLED_BIT_NV,
    eSampleLocationsCompatibleDepthEXT = VK_IMAGE_CREATE_SAMPLE_LOCATIONS_COMPATIBLE_DEPTH_BIT_EXT,
    eSubsampledEXT = VK_IMAGE_CREATE_SUBSAMPLED_BIT_EXT,
    eSplitInstanceBindRegionsKHR = VK_IMAGE_CREATE_SPLIT_INSTANCE_BIND_REGIONS_BIT_KHR,
    e2DArrayCompatibleKHR = VK_IMAGE_CREATE_2D_ARRAY_COMPATIBLE_BIT_KHR,
    eBlockTexelViewCompatibleKHR = VK_IMAGE_CREATE_BLOCK_TEXEL_VIEW_COMPATIBLE_BIT_KHR,
    eExtendedUsageKHR = VK_IMAGE_CREATE_EXTENDED_USAGE_BIT_KHR,
    eDisjointKHR = VK_IMAGE_CREATE_DISJOINT_BIT_KHR,
    eAliasKHR = VK_IMAGE_CREATE_ALIAS_BIT_KHR
  };

  VULKAN_HPP_INLINE std::string to_string( ImageCreateFlagBits value )
  {
    switch ( value )
    {
      case ImageCreateFlagBits::eSparseBinding : return "SparseBinding";
      case ImageCreateFlagBits::eSparseResidency : return "SparseResidency";
      case ImageCreateFlagBits::eSparseAliased : return "SparseAliased";
      case ImageCreateFlagBits::eMutableFormat : return "MutableFormat";
      case ImageCreateFlagBits::eCubeCompatible : return "CubeCompatible";
      case ImageCreateFlagBits::eAlias : return "Alias";
      case ImageCreateFlagBits::eSplitInstanceBindRegions : return "SplitInstanceBindRegions";
      case ImageCreateFlagBits::e2DArrayCompatible : return "2DArrayCompatible";
      case ImageCreateFlagBits::eBlockTexelViewCompatible : return "BlockTexelViewCompatible";
      case ImageCreateFlagBits::eExtendedUsage : return "ExtendedUsage";
      case ImageCreateFlagBits::eProtected : return "Protected";
      case ImageCreateFlagBits::eDisjoint : return "Disjoint";
      case ImageCreateFlagBits::eCornerSampledNV : return "CornerSampledNV";
      case ImageCreateFlagBits::eSampleLocationsCompatibleDepthEXT : return "SampleLocationsCompatibleDepthEXT";
      case ImageCreateFlagBits::eSubsampledEXT : return "SubsampledEXT";
      default: return "invalid";
    }
  }

  using ImageCreateFlags = Flags<ImageCreateFlagBits, VkImageCreateFlags>;

  VULKAN_HPP_INLINE ImageCreateFlags operator|( ImageCreateFlagBits bit0, ImageCreateFlagBits bit1 )
  {
    return ImageCreateFlags( bit0 ) | bit1;
  }

  VULKAN_HPP_INLINE ImageCreateFlags operator~( ImageCreateFlagBits bits )
  {
    return ~( ImageCreateFlags( bits ) );
  }

  template <> struct FlagTraits<ImageCreateFlagBits>
  {
    enum
    {
      allFlags = VkFlags(ImageCreateFlagBits::eSparseBinding) | VkFlags(ImageCreateFlagBits::eSparseResidency) | VkFlags(ImageCreateFlagBits::eSparseAliased) | VkFlags(ImageCreateFlagBits::eMutableFormat) | VkFlags(ImageCreateFlagBits::eCubeCompatible) | VkFlags(ImageCreateFlagBits::eAlias) | VkFlags(ImageCreateFlagBits::eSplitInstanceBindRegions) | VkFlags(ImageCreateFlagBits::e2DArrayCompatible) | VkFlags(ImageCreateFlagBits::eBlockTexelViewCompatible) | VkFlags(ImageCreateFlagBits::eExtendedUsage) | VkFlags(ImageCreateFlagBits::eProtected) | VkFlags(ImageCreateFlagBits::eDisjoint) | VkFlags(ImageCreateFlagBits::eCornerSampledNV) | VkFlags(ImageCreateFlagBits::eSampleLocationsCompatibleDepthEXT) | VkFlags(ImageCreateFlagBits::eSubsampledEXT)
    };
  };

  VULKAN_HPP_INLINE std::string to_string( ImageCreateFlags value  )
  {
    if ( !value ) return "{}";
    std::string result;

    if ( value & ImageCreateFlagBits::eSparseBinding ) result += "SparseBinding | ";
    if ( value & ImageCreateFlagBits::eSparseResidency ) result += "SparseResidency | ";
    if ( value & ImageCreateFlagBits::eSparseAliased ) result += "SparseAliased | ";
    if ( value & ImageCreateFlagBits::eMutableFormat ) result += "MutableFormat | ";
    if ( value & ImageCreateFlagBits::eCubeCompatible ) result += "CubeCompatible | ";
    if ( value & ImageCreateFlagBits::eAlias ) result += "Alias | ";
    if ( value & ImageCreateFlagBits::eSplitInstanceBindRegions ) result += "SplitInstanceBindRegions | ";
    if ( value & ImageCreateFlagBits::e2DArrayCompatible ) result += "2DArrayCompatible | ";
    if ( value & ImageCreateFlagBits::eBlockTexelViewCompatible ) result += "BlockTexelViewCompatible | ";
    if ( value & ImageCreateFlagBits::eExtendedUsage ) result += "ExtendedUsage | ";
    if ( value & ImageCreateFlagBits::eProtected ) result += "Protected | ";
    if ( value & ImageCreateFlagBits::eDisjoint ) result += "Disjoint | ";
    if ( value & ImageCreateFlagBits::eCornerSampledNV ) result += "CornerSampledNV | ";
    if ( value & ImageCreateFlagBits::eSampleLocationsCompatibleDepthEXT ) result += "SampleLocationsCompatibleDepthEXT | ";
    if ( value & ImageCreateFlagBits::eSubsampledEXT ) result += "SubsampledEXT";
    return result;
  }

#ifdef VK_USE_PLATFORM_FUCHSIA
  enum class ImagePipeSurfaceCreateFlagBitsFUCHSIA
  {};

  VULKAN_HPP_INLINE std::string to_string( ImagePipeSurfaceCreateFlagBitsFUCHSIA )
  {
    return "(void)";
  }

  using ImagePipeSurfaceCreateFlagsFUCHSIA = Flags<ImagePipeSurfaceCreateFlagBitsFUCHSIA, VkImagePipeSurfaceCreateFlagsFUCHSIA>;

  VULKAN_HPP_INLINE std::string to_string( ImagePipeSurfaceCreateFlagsFUCHSIA  )
  {
    return "{}";
  }
#endif /*VK_USE_PLATFORM_FUCHSIA*/

  enum class ImageUsageFlagBits
  {
    eTransferSrc = VK_IMAGE_USAGE_TRANSFER_SRC_BIT,
    eTransferDst = VK_IMAGE_USAGE_TRANSFER_DST_BIT,
    eSampled = VK_IMAGE_USAGE_SAMPLED_BIT,
    eStorage = VK_IMAGE_USAGE_STORAGE_BIT,
    eColorAttachment = VK_IMAGE_USAGE_COLOR_ATTACHMENT_BIT,
    eDepthStencilAttachment = VK_IMAGE_USAGE_DEPTH_STENCIL_ATTACHMENT_BIT,
    eTransientAttachment = VK_IMAGE_USAGE_TRANSIENT_ATTACHMENT_BIT,
    eInputAttachment = VK_IMAGE_USAGE_INPUT_ATTACHMENT_BIT,
    eShadingRateImageNV = VK_IMAGE_USAGE_SHADING_RATE_IMAGE_BIT_NV,
    eFragmentDensityMapEXT = VK_IMAGE_USAGE_FRAGMENT_DENSITY_MAP_BIT_EXT
  };

  VULKAN_HPP_INLINE std::string to_string( ImageUsageFlagBits value )
  {
    switch ( value )
    {
      case ImageUsageFlagBits::eTransferSrc : return "TransferSrc";
      case ImageUsageFlagBits::eTransferDst : return "TransferDst";
      case ImageUsageFlagBits::eSampled : return "Sampled";
      case ImageUsageFlagBits::eStorage : return "Storage";
      case ImageUsageFlagBits::eColorAttachment : return "ColorAttachment";
      case ImageUsageFlagBits::eDepthStencilAttachment : return "DepthStencilAttachment";
      case ImageUsageFlagBits::eTransientAttachment : return "TransientAttachment";
      case ImageUsageFlagBits::eInputAttachment : return "InputAttachment";
      case ImageUsageFlagBits::eShadingRateImageNV : return "ShadingRateImageNV";
      case ImageUsageFlagBits::eFragmentDensityMapEXT : return "FragmentDensityMapEXT";
      default: return "invalid";
    }
  }

  using ImageUsageFlags = Flags<ImageUsageFlagBits, VkImageUsageFlags>;

  VULKAN_HPP_INLINE ImageUsageFlags operator|( ImageUsageFlagBits bit0, ImageUsageFlagBits bit1 )
  {
    return ImageUsageFlags( bit0 ) | bit1;
  }

  VULKAN_HPP_INLINE ImageUsageFlags operator~( ImageUsageFlagBits bits )
  {
    return ~( ImageUsageFlags( bits ) );
  }

  template <> struct FlagTraits<ImageUsageFlagBits>
  {
    enum
    {
      allFlags = VkFlags(ImageUsageFlagBits::eTransferSrc) | VkFlags(ImageUsageFlagBits::eTransferDst) | VkFlags(ImageUsageFlagBits::eSampled) | VkFlags(ImageUsageFlagBits::eStorage) | VkFlags(ImageUsageFlagBits::eColorAttachment) | VkFlags(ImageUsageFlagBits::eDepthStencilAttachment) | VkFlags(ImageUsageFlagBits::eTransientAttachment) | VkFlags(ImageUsageFlagBits::eInputAttachment) | VkFlags(ImageUsageFlagBits::eShadingRateImageNV) | VkFlags(ImageUsageFlagBits::eFragmentDensityMapEXT)
    };
  };

  VULKAN_HPP_INLINE std::string to_string( ImageUsageFlags value  )
  {
    if ( !value ) return "{}";
    std::string result;

    if ( value & ImageUsageFlagBits::eTransferSrc ) result += "TransferSrc | ";
    if ( value & ImageUsageFlagBits::eTransferDst ) result += "TransferDst | ";
    if ( value & ImageUsageFlagBits::eSampled ) result += "Sampled | ";
    if ( value & ImageUsageFlagBits::eStorage ) result += "Storage | ";
    if ( value & ImageUsageFlagBits::eColorAttachment ) result += "ColorAttachment | ";
    if ( value & ImageUsageFlagBits::eDepthStencilAttachment ) result += "DepthStencilAttachment | ";
    if ( value & ImageUsageFlagBits::eTransientAttachment ) result += "TransientAttachment | ";
    if ( value & ImageUsageFlagBits::eInputAttachment ) result += "InputAttachment | ";
    if ( value & ImageUsageFlagBits::eShadingRateImageNV ) result += "ShadingRateImageNV | ";
    if ( value & ImageUsageFlagBits::eFragmentDensityMapEXT ) result += "FragmentDensityMapEXT";
    return result;
  }

  enum class ImageViewCreateFlagBits
  {
    eFragmentDensityMapDynamicEXT = VK_IMAGE_VIEW_CREATE_FRAGMENT_DENSITY_MAP_DYNAMIC_BIT_EXT
  };

  VULKAN_HPP_INLINE std::string to_string( ImageViewCreateFlagBits value )
  {
    switch ( value )
    {
      case ImageViewCreateFlagBits::eFragmentDensityMapDynamicEXT : return "FragmentDensityMapDynamicEXT";
      default: return "invalid";
    }
  }

  using ImageViewCreateFlags = Flags<ImageViewCreateFlagBits, VkImageViewCreateFlags>;

  VULKAN_HPP_INLINE ImageViewCreateFlags operator|( ImageViewCreateFlagBits bit0, ImageViewCreateFlagBits bit1 )
  {
    return ImageViewCreateFlags( bit0 ) | bit1;
  }

  VULKAN_HPP_INLINE ImageViewCreateFlags operator~( ImageViewCreateFlagBits bits )
  {
    return ~( ImageViewCreateFlags( bits ) );
  }

  template <> struct FlagTraits<ImageViewCreateFlagBits>
  {
    enum
    {
      allFlags = VkFlags(ImageViewCreateFlagBits::eFragmentDensityMapDynamicEXT)
    };
  };

  VULKAN_HPP_INLINE std::string to_string( ImageViewCreateFlags value  )
  {
    if ( !value ) return "{}";
    std::string result;

    if ( value & ImageViewCreateFlagBits::eFragmentDensityMapDynamicEXT ) result += "FragmentDensityMapDynamicEXT";
    return result;
  }

  enum class IndirectCommandsLayoutUsageFlagBitsNVX
  {
    eUnorderedSequences = VK_INDIRECT_COMMANDS_LAYOUT_USAGE_UNORDERED_SEQUENCES_BIT_NVX,
    eSparseSequences = VK_INDIRECT_COMMANDS_LAYOUT_USAGE_SPARSE_SEQUENCES_BIT_NVX,
    eEmptyExecutions = VK_INDIRECT_COMMANDS_LAYOUT_USAGE_EMPTY_EXECUTIONS_BIT_NVX,
    eIndexedSequences = VK_INDIRECT_COMMANDS_LAYOUT_USAGE_INDEXED_SEQUENCES_BIT_NVX
  };

  VULKAN_HPP_INLINE std::string to_string( IndirectCommandsLayoutUsageFlagBitsNVX value )
  {
    switch ( value )
    {
      case IndirectCommandsLayoutUsageFlagBitsNVX::eUnorderedSequences : return "UnorderedSequences";
      case IndirectCommandsLayoutUsageFlagBitsNVX::eSparseSequences : return "SparseSequences";
      case IndirectCommandsLayoutUsageFlagBitsNVX::eEmptyExecutions : return "EmptyExecutions";
      case IndirectCommandsLayoutUsageFlagBitsNVX::eIndexedSequences : return "IndexedSequences";
      default: return "invalid";
    }
  }

  using IndirectCommandsLayoutUsageFlagsNVX = Flags<IndirectCommandsLayoutUsageFlagBitsNVX, VkIndirectCommandsLayoutUsageFlagsNVX>;

  VULKAN_HPP_INLINE IndirectCommandsLayoutUsageFlagsNVX operator|( IndirectCommandsLayoutUsageFlagBitsNVX bit0, IndirectCommandsLayoutUsageFlagBitsNVX bit1 )
  {
    return IndirectCommandsLayoutUsageFlagsNVX( bit0 ) | bit1;
  }

  VULKAN_HPP_INLINE IndirectCommandsLayoutUsageFlagsNVX operator~( IndirectCommandsLayoutUsageFlagBitsNVX bits )
  {
    return ~( IndirectCommandsLayoutUsageFlagsNVX( bits ) );
  }

  template <> struct FlagTraits<IndirectCommandsLayoutUsageFlagBitsNVX>
  {
    enum
    {
      allFlags = VkFlags(IndirectCommandsLayoutUsageFlagBitsNVX::eUnorderedSequences) | VkFlags(IndirectCommandsLayoutUsageFlagBitsNVX::eSparseSequences) | VkFlags(IndirectCommandsLayoutUsageFlagBitsNVX::eEmptyExecutions) | VkFlags(IndirectCommandsLayoutUsageFlagBitsNVX::eIndexedSequences)
    };
  };

  VULKAN_HPP_INLINE std::string to_string( IndirectCommandsLayoutUsageFlagsNVX value  )
  {
    if ( !value ) return "{}";
    std::string result;

    if ( value & IndirectCommandsLayoutUsageFlagBitsNVX::eUnorderedSequences ) result += "UnorderedSequences | ";
    if ( value & IndirectCommandsLayoutUsageFlagBitsNVX::eSparseSequences ) result += "SparseSequences | ";
    if ( value & IndirectCommandsLayoutUsageFlagBitsNVX::eEmptyExecutions ) result += "EmptyExecutions | ";
    if ( value & IndirectCommandsLayoutUsageFlagBitsNVX::eIndexedSequences ) result += "IndexedSequences";
    return result;
  }

  enum class InstanceCreateFlagBits
  {};

  VULKAN_HPP_INLINE std::string to_string( InstanceCreateFlagBits )
  {
    return "(void)";
  }

  using InstanceCreateFlags = Flags<InstanceCreateFlagBits, VkInstanceCreateFlags>;

  VULKAN_HPP_INLINE std::string to_string( InstanceCreateFlags  )
  {
    return "{}";
  }

#ifdef VK_USE_PLATFORM_MACOS_MVK
  enum class MacOSSurfaceCreateFlagBitsMVK
  {};

  VULKAN_HPP_INLINE std::string to_string( MacOSSurfaceCreateFlagBitsMVK )
  {
    return "(void)";
  }

  using MacOSSurfaceCreateFlagsMVK = Flags<MacOSSurfaceCreateFlagBitsMVK, VkMacOSSurfaceCreateFlagsMVK>;

  VULKAN_HPP_INLINE std::string to_string( MacOSSurfaceCreateFlagsMVK  )
  {
    return "{}";
  }
#endif /*VK_USE_PLATFORM_MACOS_MVK*/

  enum class MemoryAllocateFlagBits
  {
    eDeviceMask = VK_MEMORY_ALLOCATE_DEVICE_MASK_BIT,
    eDeviceMaskKHR = VK_MEMORY_ALLOCATE_DEVICE_MASK_BIT_KHR
  };

  VULKAN_HPP_INLINE std::string to_string( MemoryAllocateFlagBits value )
  {
    switch ( value )
    {
      case MemoryAllocateFlagBits::eDeviceMask : return "DeviceMask";
      default: return "invalid";
    }
  }

  using MemoryAllocateFlags = Flags<MemoryAllocateFlagBits, VkMemoryAllocateFlags>;

  VULKAN_HPP_INLINE MemoryAllocateFlags operator|( MemoryAllocateFlagBits bit0, MemoryAllocateFlagBits bit1 )
  {
    return MemoryAllocateFlags( bit0 ) | bit1;
  }

  VULKAN_HPP_INLINE MemoryAllocateFlags operator~( MemoryAllocateFlagBits bits )
  {
    return ~( MemoryAllocateFlags( bits ) );
  }

  template <> struct FlagTraits<MemoryAllocateFlagBits>
  {
    enum
    {
      allFlags = VkFlags(MemoryAllocateFlagBits::eDeviceMask)
    };
  };

  using MemoryAllocateFlagsKHR = MemoryAllocateFlags;

  VULKAN_HPP_INLINE std::string to_string( MemoryAllocateFlags value  )
  {
    if ( !value ) return "{}";
    std::string result;

    if ( value & MemoryAllocateFlagBits::eDeviceMask ) result += "DeviceMask";
    return result;
  }

  enum class MemoryHeapFlagBits
  {
    eDeviceLocal = VK_MEMORY_HEAP_DEVICE_LOCAL_BIT,
    eMultiInstance = VK_MEMORY_HEAP_MULTI_INSTANCE_BIT,
    eMultiInstanceKHR = VK_MEMORY_HEAP_MULTI_INSTANCE_BIT_KHR
  };

  VULKAN_HPP_INLINE std::string to_string( MemoryHeapFlagBits value )
  {
    switch ( value )
    {
      case MemoryHeapFlagBits::eDeviceLocal : return "DeviceLocal";
      case MemoryHeapFlagBits::eMultiInstance : return "MultiInstance";
      default: return "invalid";
    }
  }

  using MemoryHeapFlags = Flags<MemoryHeapFlagBits, VkMemoryHeapFlags>;

  VULKAN_HPP_INLINE MemoryHeapFlags operator|( MemoryHeapFlagBits bit0, MemoryHeapFlagBits bit1 )
  {
    return MemoryHeapFlags( bit0 ) | bit1;
  }

  VULKAN_HPP_INLINE MemoryHeapFlags operator~( MemoryHeapFlagBits bits )
  {
    return ~( MemoryHeapFlags( bits ) );
  }

  template <> struct FlagTraits<MemoryHeapFlagBits>
  {
    enum
    {
      allFlags = VkFlags(MemoryHeapFlagBits::eDeviceLocal) | VkFlags(MemoryHeapFlagBits::eMultiInstance)
    };
  };

  VULKAN_HPP_INLINE std::string to_string( MemoryHeapFlags value  )
  {
    if ( !value ) return "{}";
    std::string result;

    if ( value & MemoryHeapFlagBits::eDeviceLocal ) result += "DeviceLocal | ";
    if ( value & MemoryHeapFlagBits::eMultiInstance ) result += "MultiInstance";
    return result;
  }

  enum class MemoryMapFlagBits
  {};

  VULKAN_HPP_INLINE std::string to_string( MemoryMapFlagBits )
  {
    return "(void)";
  }

  using MemoryMapFlags = Flags<MemoryMapFlagBits, VkMemoryMapFlags>;

  VULKAN_HPP_INLINE std::string to_string( MemoryMapFlags  )
  {
    return "{}";
  }

  enum class MemoryPropertyFlagBits
  {
    eDeviceLocal = VK_MEMORY_PROPERTY_DEVICE_LOCAL_BIT,
    eHostVisible = VK_MEMORY_PROPERTY_HOST_VISIBLE_BIT,
    eHostCoherent = VK_MEMORY_PROPERTY_HOST_COHERENT_BIT,
    eHostCached = VK_MEMORY_PROPERTY_HOST_CACHED_BIT,
    eLazilyAllocated = VK_MEMORY_PROPERTY_LAZILY_ALLOCATED_BIT,
    eProtected = VK_MEMORY_PROPERTY_PROTECTED_BIT
  };

  VULKAN_HPP_INLINE std::string to_string( MemoryPropertyFlagBits value )
  {
    switch ( value )
    {
      case MemoryPropertyFlagBits::eDeviceLocal : return "DeviceLocal";
      case MemoryPropertyFlagBits::eHostVisible : return "HostVisible";
      case MemoryPropertyFlagBits::eHostCoherent : return "HostCoherent";
      case MemoryPropertyFlagBits::eHostCached : return "HostCached";
      case MemoryPropertyFlagBits::eLazilyAllocated : return "LazilyAllocated";
      case MemoryPropertyFlagBits::eProtected : return "Protected";
      default: return "invalid";
    }
  }

  using MemoryPropertyFlags = Flags<MemoryPropertyFlagBits, VkMemoryPropertyFlags>;

  VULKAN_HPP_INLINE MemoryPropertyFlags operator|( MemoryPropertyFlagBits bit0, MemoryPropertyFlagBits bit1 )
  {
    return MemoryPropertyFlags( bit0 ) | bit1;
  }

  VULKAN_HPP_INLINE MemoryPropertyFlags operator~( MemoryPropertyFlagBits bits )
  {
    return ~( MemoryPropertyFlags( bits ) );
  }

  template <> struct FlagTraits<MemoryPropertyFlagBits>
  {
    enum
    {
      allFlags = VkFlags(MemoryPropertyFlagBits::eDeviceLocal) | VkFlags(MemoryPropertyFlagBits::eHostVisible) | VkFlags(MemoryPropertyFlagBits::eHostCoherent) | VkFlags(MemoryPropertyFlagBits::eHostCached) | VkFlags(MemoryPropertyFlagBits::eLazilyAllocated) | VkFlags(MemoryPropertyFlagBits::eProtected)
    };
  };

  VULKAN_HPP_INLINE std::string to_string( MemoryPropertyFlags value  )
  {
    if ( !value ) return "{}";
    std::string result;

    if ( value & MemoryPropertyFlagBits::eDeviceLocal ) result += "DeviceLocal | ";
    if ( value & MemoryPropertyFlagBits::eHostVisible ) result += "HostVisible | ";
    if ( value & MemoryPropertyFlagBits::eHostCoherent ) result += "HostCoherent | ";
    if ( value & MemoryPropertyFlagBits::eHostCached ) result += "HostCached | ";
    if ( value & MemoryPropertyFlagBits::eLazilyAllocated ) result += "LazilyAllocated | ";
    if ( value & MemoryPropertyFlagBits::eProtected ) result += "Protected";
    return result;
  }

  enum class ObjectEntryUsageFlagBitsNVX
  {
    eGraphics = VK_OBJECT_ENTRY_USAGE_GRAPHICS_BIT_NVX,
    eCompute = VK_OBJECT_ENTRY_USAGE_COMPUTE_BIT_NVX
  };

  VULKAN_HPP_INLINE std::string to_string( ObjectEntryUsageFlagBitsNVX value )
  {
    switch ( value )
    {
      case ObjectEntryUsageFlagBitsNVX::eGraphics : return "Graphics";
      case ObjectEntryUsageFlagBitsNVX::eCompute : return "Compute";
      default: return "invalid";
    }
  }

  using ObjectEntryUsageFlagsNVX = Flags<ObjectEntryUsageFlagBitsNVX, VkObjectEntryUsageFlagsNVX>;

  VULKAN_HPP_INLINE ObjectEntryUsageFlagsNVX operator|( ObjectEntryUsageFlagBitsNVX bit0, ObjectEntryUsageFlagBitsNVX bit1 )
  {
    return ObjectEntryUsageFlagsNVX( bit0 ) | bit1;
  }

  VULKAN_HPP_INLINE ObjectEntryUsageFlagsNVX operator~( ObjectEntryUsageFlagBitsNVX bits )
  {
    return ~( ObjectEntryUsageFlagsNVX( bits ) );
  }

  template <> struct FlagTraits<ObjectEntryUsageFlagBitsNVX>
  {
    enum
    {
      allFlags = VkFlags(ObjectEntryUsageFlagBitsNVX::eGraphics) | VkFlags(ObjectEntryUsageFlagBitsNVX::eCompute)
    };
  };

  VULKAN_HPP_INLINE std::string to_string( ObjectEntryUsageFlagsNVX value  )
  {
    if ( !value ) return "{}";
    std::string result;

    if ( value & ObjectEntryUsageFlagBitsNVX::eGraphics ) result += "Graphics | ";
    if ( value & ObjectEntryUsageFlagBitsNVX::eCompute ) result += "Compute";
    return result;
  }

  enum class PeerMemoryFeatureFlagBits
  {
    eCopySrc = VK_PEER_MEMORY_FEATURE_COPY_SRC_BIT,
    eCopyDst = VK_PEER_MEMORY_FEATURE_COPY_DST_BIT,
    eGenericSrc = VK_PEER_MEMORY_FEATURE_GENERIC_SRC_BIT,
    eGenericDst = VK_PEER_MEMORY_FEATURE_GENERIC_DST_BIT,
    eCopySrcKHR = VK_PEER_MEMORY_FEATURE_COPY_SRC_BIT_KHR,
    eCopyDstKHR = VK_PEER_MEMORY_FEATURE_COPY_DST_BIT_KHR,
    eGenericSrcKHR = VK_PEER_MEMORY_FEATURE_GENERIC_SRC_BIT_KHR,
    eGenericDstKHR = VK_PEER_MEMORY_FEATURE_GENERIC_DST_BIT_KHR
  };

  VULKAN_HPP_INLINE std::string to_string( PeerMemoryFeatureFlagBits value )
  {
    switch ( value )
    {
      case PeerMemoryFeatureFlagBits::eCopySrc : return "CopySrc";
      case PeerMemoryFeatureFlagBits::eCopyDst : return "CopyDst";
      case PeerMemoryFeatureFlagBits::eGenericSrc : return "GenericSrc";
      case PeerMemoryFeatureFlagBits::eGenericDst : return "GenericDst";
      default: return "invalid";
    }
  }

  using PeerMemoryFeatureFlags = Flags<PeerMemoryFeatureFlagBits, VkPeerMemoryFeatureFlags>;

  VULKAN_HPP_INLINE PeerMemoryFeatureFlags operator|( PeerMemoryFeatureFlagBits bit0, PeerMemoryFeatureFlagBits bit1 )
  {
    return PeerMemoryFeatureFlags( bit0 ) | bit1;
  }

  VULKAN_HPP_INLINE PeerMemoryFeatureFlags operator~( PeerMemoryFeatureFlagBits bits )
  {
    return ~( PeerMemoryFeatureFlags( bits ) );
  }

  template <> struct FlagTraits<PeerMemoryFeatureFlagBits>
  {
    enum
    {
      allFlags = VkFlags(PeerMemoryFeatureFlagBits::eCopySrc) | VkFlags(PeerMemoryFeatureFlagBits::eCopyDst) | VkFlags(PeerMemoryFeatureFlagBits::eGenericSrc) | VkFlags(PeerMemoryFeatureFlagBits::eGenericDst)
    };
  };

  using PeerMemoryFeatureFlagsKHR = PeerMemoryFeatureFlags;

  VULKAN_HPP_INLINE std::string to_string( PeerMemoryFeatureFlags value  )
  {
    if ( !value ) return "{}";
    std::string result;

    if ( value & PeerMemoryFeatureFlagBits::eCopySrc ) result += "CopySrc | ";
    if ( value & PeerMemoryFeatureFlagBits::eCopyDst ) result += "CopyDst | ";
    if ( value & PeerMemoryFeatureFlagBits::eGenericSrc ) result += "GenericSrc | ";
    if ( value & PeerMemoryFeatureFlagBits::eGenericDst ) result += "GenericDst";
    return result;
  }

  enum class PipelineCacheCreateFlagBits
  {};

  VULKAN_HPP_INLINE std::string to_string( PipelineCacheCreateFlagBits )
  {
    return "(void)";
  }

  using PipelineCacheCreateFlags = Flags<PipelineCacheCreateFlagBits, VkPipelineCacheCreateFlags>;

  VULKAN_HPP_INLINE std::string to_string( PipelineCacheCreateFlags  )
  {
    return "{}";
  }

  enum class PipelineColorBlendStateCreateFlagBits
  {};

  VULKAN_HPP_INLINE std::string to_string( PipelineColorBlendStateCreateFlagBits )
  {
    return "(void)";
  }

  using PipelineColorBlendStateCreateFlags = Flags<PipelineColorBlendStateCreateFlagBits, VkPipelineColorBlendStateCreateFlags>;

  VULKAN_HPP_INLINE std::string to_string( PipelineColorBlendStateCreateFlags  )
  {
    return "{}";
  }

  enum class PipelineCoverageModulationStateCreateFlagBitsNV
  {};

  VULKAN_HPP_INLINE std::string to_string( PipelineCoverageModulationStateCreateFlagBitsNV )
  {
    return "(void)";
  }

  using PipelineCoverageModulationStateCreateFlagsNV = Flags<PipelineCoverageModulationStateCreateFlagBitsNV, VkPipelineCoverageModulationStateCreateFlagsNV>;

  VULKAN_HPP_INLINE std::string to_string( PipelineCoverageModulationStateCreateFlagsNV  )
  {
    return "{}";
  }

  enum class PipelineCoverageToColorStateCreateFlagBitsNV
  {};

  VULKAN_HPP_INLINE std::string to_string( PipelineCoverageToColorStateCreateFlagBitsNV )
  {
    return "(void)";
  }

  using PipelineCoverageToColorStateCreateFlagsNV = Flags<PipelineCoverageToColorStateCreateFlagBitsNV, VkPipelineCoverageToColorStateCreateFlagsNV>;

  VULKAN_HPP_INLINE std::string to_string( PipelineCoverageToColorStateCreateFlagsNV  )
  {
    return "{}";
  }

  enum class PipelineCreateFlagBits
  {
    eDisableOptimization = VK_PIPELINE_CREATE_DISABLE_OPTIMIZATION_BIT,
    eAllowDerivatives = VK_PIPELINE_CREATE_ALLOW_DERIVATIVES_BIT,
    eDerivative = VK_PIPELINE_CREATE_DERIVATIVE_BIT,
    eViewIndexFromDeviceIndex = VK_PIPELINE_CREATE_VIEW_INDEX_FROM_DEVICE_INDEX_BIT,
    eDispatchBase = VK_PIPELINE_CREATE_DISPATCH_BASE,
    eDeferCompileNV = VK_PIPELINE_CREATE_DEFER_COMPILE_BIT_NV,
    eViewIndexFromDeviceIndexKHR = VK_PIPELINE_CREATE_VIEW_INDEX_FROM_DEVICE_INDEX_BIT_KHR,
    eDispatchBaseKHR = VK_PIPELINE_CREATE_DISPATCH_BASE_KHR
  };

  VULKAN_HPP_INLINE std::string to_string( PipelineCreateFlagBits value )
  {
    switch ( value )
    {
      case PipelineCreateFlagBits::eDisableOptimization : return "DisableOptimization";
      case PipelineCreateFlagBits::eAllowDerivatives : return "AllowDerivatives";
      case PipelineCreateFlagBits::eDerivative : return "Derivative";
      case PipelineCreateFlagBits::eViewIndexFromDeviceIndex : return "ViewIndexFromDeviceIndex";
      case PipelineCreateFlagBits::eDispatchBase : return "DispatchBase";
      case PipelineCreateFlagBits::eDeferCompileNV : return "DeferCompileNV";
      default: return "invalid";
    }
  }

  using PipelineCreateFlags = Flags<PipelineCreateFlagBits, VkPipelineCreateFlags>;

  VULKAN_HPP_INLINE PipelineCreateFlags operator|( PipelineCreateFlagBits bit0, PipelineCreateFlagBits bit1 )
  {
    return PipelineCreateFlags( bit0 ) | bit1;
  }

  VULKAN_HPP_INLINE PipelineCreateFlags operator~( PipelineCreateFlagBits bits )
  {
    return ~( PipelineCreateFlags( bits ) );
  }

  template <> struct FlagTraits<PipelineCreateFlagBits>
  {
    enum
    {
      allFlags = VkFlags(PipelineCreateFlagBits::eDisableOptimization) | VkFlags(PipelineCreateFlagBits::eAllowDerivatives) | VkFlags(PipelineCreateFlagBits::eDerivative) | VkFlags(PipelineCreateFlagBits::eViewIndexFromDeviceIndex) | VkFlags(PipelineCreateFlagBits::eDispatchBase) | VkFlags(PipelineCreateFlagBits::eDeferCompileNV)
    };
  };

  VULKAN_HPP_INLINE std::string to_string( PipelineCreateFlags value  )
  {
    if ( !value ) return "{}";
    std::string result;

    if ( value & PipelineCreateFlagBits::eDisableOptimization ) result += "DisableOptimization | ";
    if ( value & PipelineCreateFlagBits::eAllowDerivatives ) result += "AllowDerivatives | ";
    if ( value & PipelineCreateFlagBits::eDerivative ) result += "Derivative | ";
    if ( value & PipelineCreateFlagBits::eViewIndexFromDeviceIndex ) result += "ViewIndexFromDeviceIndex | ";
    if ( value & PipelineCreateFlagBits::eDispatchBase ) result += "DispatchBase | ";
    if ( value & PipelineCreateFlagBits::eDeferCompileNV ) result += "DeferCompileNV";
    return result;
  }

  enum class PipelineDepthStencilStateCreateFlagBits
  {};

  VULKAN_HPP_INLINE std::string to_string( PipelineDepthStencilStateCreateFlagBits )
  {
    return "(void)";
  }

  using PipelineDepthStencilStateCreateFlags = Flags<PipelineDepthStencilStateCreateFlagBits, VkPipelineDepthStencilStateCreateFlags>;

  VULKAN_HPP_INLINE std::string to_string( PipelineDepthStencilStateCreateFlags  )
  {
    return "{}";
  }

  enum class PipelineDiscardRectangleStateCreateFlagBitsEXT
  {};

  VULKAN_HPP_INLINE std::string to_string( PipelineDiscardRectangleStateCreateFlagBitsEXT )
  {
    return "(void)";
  }

  using PipelineDiscardRectangleStateCreateFlagsEXT = Flags<PipelineDiscardRectangleStateCreateFlagBitsEXT, VkPipelineDiscardRectangleStateCreateFlagsEXT>;

  VULKAN_HPP_INLINE std::string to_string( PipelineDiscardRectangleStateCreateFlagsEXT  )
  {
    return "{}";
  }

  enum class PipelineDynamicStateCreateFlagBits
  {};

  VULKAN_HPP_INLINE std::string to_string( PipelineDynamicStateCreateFlagBits )
  {
    return "(void)";
  }

  using PipelineDynamicStateCreateFlags = Flags<PipelineDynamicStateCreateFlagBits, VkPipelineDynamicStateCreateFlags>;

  VULKAN_HPP_INLINE std::string to_string( PipelineDynamicStateCreateFlags  )
  {
    return "{}";
  }

  enum class PipelineInputAssemblyStateCreateFlagBits
  {};

  VULKAN_HPP_INLINE std::string to_string( PipelineInputAssemblyStateCreateFlagBits )
  {
    return "(void)";
  }

  using PipelineInputAssemblyStateCreateFlags = Flags<PipelineInputAssemblyStateCreateFlagBits, VkPipelineInputAssemblyStateCreateFlags>;

  VULKAN_HPP_INLINE std::string to_string( PipelineInputAssemblyStateCreateFlags  )
  {
    return "{}";
  }

  enum class PipelineLayoutCreateFlagBits
  {};

  VULKAN_HPP_INLINE std::string to_string( PipelineLayoutCreateFlagBits )
  {
    return "(void)";
  }

  using PipelineLayoutCreateFlags = Flags<PipelineLayoutCreateFlagBits, VkPipelineLayoutCreateFlags>;

  VULKAN_HPP_INLINE std::string to_string( PipelineLayoutCreateFlags  )
  {
    return "{}";
  }

  enum class PipelineMultisampleStateCreateFlagBits
  {};

  VULKAN_HPP_INLINE std::string to_string( PipelineMultisampleStateCreateFlagBits )
  {
    return "(void)";
  }

  using PipelineMultisampleStateCreateFlags = Flags<PipelineMultisampleStateCreateFlagBits, VkPipelineMultisampleStateCreateFlags>;

  VULKAN_HPP_INLINE std::string to_string( PipelineMultisampleStateCreateFlags  )
  {
    return "{}";
  }

  enum class PipelineRasterizationConservativeStateCreateFlagBitsEXT
  {};

  VULKAN_HPP_INLINE std::string to_string( PipelineRasterizationConservativeStateCreateFlagBitsEXT )
  {
    return "(void)";
  }

  using PipelineRasterizationConservativeStateCreateFlagsEXT = Flags<PipelineRasterizationConservativeStateCreateFlagBitsEXT, VkPipelineRasterizationConservativeStateCreateFlagsEXT>;

  VULKAN_HPP_INLINE std::string to_string( PipelineRasterizationConservativeStateCreateFlagsEXT  )
  {
    return "{}";
  }

  enum class PipelineRasterizationDepthClipStateCreateFlagBitsEXT
  {};

  VULKAN_HPP_INLINE std::string to_string( PipelineRasterizationDepthClipStateCreateFlagBitsEXT )
  {
    return "(void)";
  }

  using PipelineRasterizationDepthClipStateCreateFlagsEXT = Flags<PipelineRasterizationDepthClipStateCreateFlagBitsEXT, VkPipelineRasterizationDepthClipStateCreateFlagsEXT>;

  VULKAN_HPP_INLINE std::string to_string( PipelineRasterizationDepthClipStateCreateFlagsEXT  )
  {
    return "{}";
  }

  enum class PipelineRasterizationStateCreateFlagBits
  {};

  VULKAN_HPP_INLINE std::string to_string( PipelineRasterizationStateCreateFlagBits )
  {
    return "(void)";
  }

  using PipelineRasterizationStateCreateFlags = Flags<PipelineRasterizationStateCreateFlagBits, VkPipelineRasterizationStateCreateFlags>;

  VULKAN_HPP_INLINE std::string to_string( PipelineRasterizationStateCreateFlags  )
  {
    return "{}";
  }

  enum class PipelineRasterizationStateStreamCreateFlagBitsEXT
  {};

  VULKAN_HPP_INLINE std::string to_string( PipelineRasterizationStateStreamCreateFlagBitsEXT )
  {
    return "(void)";
  }

  using PipelineRasterizationStateStreamCreateFlagsEXT = Flags<PipelineRasterizationStateStreamCreateFlagBitsEXT, VkPipelineRasterizationStateStreamCreateFlagsEXT>;

  VULKAN_HPP_INLINE std::string to_string( PipelineRasterizationStateStreamCreateFlagsEXT  )
  {
    return "{}";
  }

  enum class PipelineShaderStageCreateFlagBits
  {};

  VULKAN_HPP_INLINE std::string to_string( PipelineShaderStageCreateFlagBits )
  {
    return "(void)";
  }

  using PipelineShaderStageCreateFlags = Flags<PipelineShaderStageCreateFlagBits, VkPipelineShaderStageCreateFlags>;

  VULKAN_HPP_INLINE std::string to_string( PipelineShaderStageCreateFlags  )
  {
    return "{}";
  }

  enum class PipelineStageFlagBits
  {
    eTopOfPipe = VK_PIPELINE_STAGE_TOP_OF_PIPE_BIT,
    eDrawIndirect = VK_PIPELINE_STAGE_DRAW_INDIRECT_BIT,
    eVertexInput = VK_PIPELINE_STAGE_VERTEX_INPUT_BIT,
    eVertexShader = VK_PIPELINE_STAGE_VERTEX_SHADER_BIT,
    eTessellationControlShader = VK_PIPELINE_STAGE_TESSELLATION_CONTROL_SHADER_BIT,
    eTessellationEvaluationShader = VK_PIPELINE_STAGE_TESSELLATION_EVALUATION_SHADER_BIT,
    eGeometryShader = VK_PIPELINE_STAGE_GEOMETRY_SHADER_BIT,
    eFragmentShader = VK_PIPELINE_STAGE_FRAGMENT_SHADER_BIT,
    eEarlyFragmentTests = VK_PIPELINE_STAGE_EARLY_FRAGMENT_TESTS_BIT,
    eLateFragmentTests = VK_PIPELINE_STAGE_LATE_FRAGMENT_TESTS_BIT,
    eColorAttachmentOutput = VK_PIPELINE_STAGE_COLOR_ATTACHMENT_OUTPUT_BIT,
    eComputeShader = VK_PIPELINE_STAGE_COMPUTE_SHADER_BIT,
    eTransfer = VK_PIPELINE_STAGE_TRANSFER_BIT,
    eBottomOfPipe = VK_PIPELINE_STAGE_BOTTOM_OF_PIPE_BIT,
    eHost = VK_PIPELINE_STAGE_HOST_BIT,
    eAllGraphics = VK_PIPELINE_STAGE_ALL_GRAPHICS_BIT,
    eAllCommands = VK_PIPELINE_STAGE_ALL_COMMANDS_BIT,
    eTransformFeedbackEXT = VK_PIPELINE_STAGE_TRANSFORM_FEEDBACK_BIT_EXT,
    eConditionalRenderingEXT = VK_PIPELINE_STAGE_CONDITIONAL_RENDERING_BIT_EXT,
    eCommandProcessNVX = VK_PIPELINE_STAGE_COMMAND_PROCESS_BIT_NVX,
    eShadingRateImageNV = VK_PIPELINE_STAGE_SHADING_RATE_IMAGE_BIT_NV,
    eRayTracingShaderNV = VK_PIPELINE_STAGE_RAY_TRACING_SHADER_BIT_NV,
    eAccelerationStructureBuildNV = VK_PIPELINE_STAGE_ACCELERATION_STRUCTURE_BUILD_BIT_NV,
    eTaskShaderNV = VK_PIPELINE_STAGE_TASK_SHADER_BIT_NV,
    eMeshShaderNV = VK_PIPELINE_STAGE_MESH_SHADER_BIT_NV,
    eFragmentDensityProcessEXT = VK_PIPELINE_STAGE_FRAGMENT_DENSITY_PROCESS_BIT_EXT
  };

  VULKAN_HPP_INLINE std::string to_string( PipelineStageFlagBits value )
  {
    switch ( value )
    {
      case PipelineStageFlagBits::eTopOfPipe : return "TopOfPipe";
      case PipelineStageFlagBits::eDrawIndirect : return "DrawIndirect";
      case PipelineStageFlagBits::eVertexInput : return "VertexInput";
      case PipelineStageFlagBits::eVertexShader : return "VertexShader";
      case PipelineStageFlagBits::eTessellationControlShader : return "TessellationControlShader";
      case PipelineStageFlagBits::eTessellationEvaluationShader : return "TessellationEvaluationShader";
      case PipelineStageFlagBits::eGeometryShader : return "GeometryShader";
      case PipelineStageFlagBits::eFragmentShader : return "FragmentShader";
      case PipelineStageFlagBits::eEarlyFragmentTests : return "EarlyFragmentTests";
      case PipelineStageFlagBits::eLateFragmentTests : return "LateFragmentTests";
      case PipelineStageFlagBits::eColorAttachmentOutput : return "ColorAttachmentOutput";
      case PipelineStageFlagBits::eComputeShader : return "ComputeShader";
      case PipelineStageFlagBits::eTransfer : return "Transfer";
      case PipelineStageFlagBits::eBottomOfPipe : return "BottomOfPipe";
      case PipelineStageFlagBits::eHost : return "Host";
      case PipelineStageFlagBits::eAllGraphics : return "AllGraphics";
      case PipelineStageFlagBits::eAllCommands : return "AllCommands";
      case PipelineStageFlagBits::eTransformFeedbackEXT : return "TransformFeedbackEXT";
      case PipelineStageFlagBits::eConditionalRenderingEXT : return "ConditionalRenderingEXT";
      case PipelineStageFlagBits::eCommandProcessNVX : return "CommandProcessNVX";
      case PipelineStageFlagBits::eShadingRateImageNV : return "ShadingRateImageNV";
      case PipelineStageFlagBits::eRayTracingShaderNV : return "RayTracingShaderNV";
      case PipelineStageFlagBits::eAccelerationStructureBuildNV : return "AccelerationStructureBuildNV";
      case PipelineStageFlagBits::eTaskShaderNV : return "TaskShaderNV";
      case PipelineStageFlagBits::eMeshShaderNV : return "MeshShaderNV";
      case PipelineStageFlagBits::eFragmentDensityProcessEXT : return "FragmentDensityProcessEXT";
      default: return "invalid";
    }
  }

  using PipelineStageFlags = Flags<PipelineStageFlagBits, VkPipelineStageFlags>;

  VULKAN_HPP_INLINE PipelineStageFlags operator|( PipelineStageFlagBits bit0, PipelineStageFlagBits bit1 )
  {
    return PipelineStageFlags( bit0 ) | bit1;
  }

  VULKAN_HPP_INLINE PipelineStageFlags operator~( PipelineStageFlagBits bits )
  {
    return ~( PipelineStageFlags( bits ) );
  }

  template <> struct FlagTraits<PipelineStageFlagBits>
  {
    enum
    {
      allFlags = VkFlags(PipelineStageFlagBits::eTopOfPipe) | VkFlags(PipelineStageFlagBits::eDrawIndirect) | VkFlags(PipelineStageFlagBits::eVertexInput) | VkFlags(PipelineStageFlagBits::eVertexShader) | VkFlags(PipelineStageFlagBits::eTessellationControlShader) | VkFlags(PipelineStageFlagBits::eTessellationEvaluationShader) | VkFlags(PipelineStageFlagBits::eGeometryShader) | VkFlags(PipelineStageFlagBits::eFragmentShader) | VkFlags(PipelineStageFlagBits::eEarlyFragmentTests) | VkFlags(PipelineStageFlagBits::eLateFragmentTests) | VkFlags(PipelineStageFlagBits::eColorAttachmentOutput) | VkFlags(PipelineStageFlagBits::eComputeShader) | VkFlags(PipelineStageFlagBits::eTransfer) | VkFlags(PipelineStageFlagBits::eBottomOfPipe) | VkFlags(PipelineStageFlagBits::eHost) | VkFlags(PipelineStageFlagBits::eAllGraphics) | VkFlags(PipelineStageFlagBits::eAllCommands) | VkFlags(PipelineStageFlagBits::eTransformFeedbackEXT) | VkFlags(PipelineStageFlagBits::eConditionalRenderingEXT) | VkFlags(PipelineStageFlagBits::eCommandProcessNVX) | VkFlags(PipelineStageFlagBits::eShadingRateImageNV) | VkFlags(PipelineStageFlagBits::eRayTracingShaderNV) | VkFlags(PipelineStageFlagBits::eAccelerationStructureBuildNV) | VkFlags(PipelineStageFlagBits::eTaskShaderNV) | VkFlags(PipelineStageFlagBits::eMeshShaderNV) | VkFlags(PipelineStageFlagBits::eFragmentDensityProcessEXT)
    };
  };

  VULKAN_HPP_INLINE std::string to_string( PipelineStageFlags value  )
  {
    if ( !value ) return "{}";
    std::string result;

    if ( value & PipelineStageFlagBits::eTopOfPipe ) result += "TopOfPipe | ";
    if ( value & PipelineStageFlagBits::eDrawIndirect ) result += "DrawIndirect | ";
    if ( value & PipelineStageFlagBits::eVertexInput ) result += "VertexInput | ";
    if ( value & PipelineStageFlagBits::eVertexShader ) result += "VertexShader | ";
    if ( value & PipelineStageFlagBits::eTessellationControlShader ) result += "TessellationControlShader | ";
    if ( value & PipelineStageFlagBits::eTessellationEvaluationShader ) result += "TessellationEvaluationShader | ";
    if ( value & PipelineStageFlagBits::eGeometryShader ) result += "GeometryShader | ";
    if ( value & PipelineStageFlagBits::eFragmentShader ) result += "FragmentShader | ";
    if ( value & PipelineStageFlagBits::eEarlyFragmentTests ) result += "EarlyFragmentTests | ";
    if ( value & PipelineStageFlagBits::eLateFragmentTests ) result += "LateFragmentTests | ";
    if ( value & PipelineStageFlagBits::eColorAttachmentOutput ) result += "ColorAttachmentOutput | ";
    if ( value & PipelineStageFlagBits::eComputeShader ) result += "ComputeShader | ";
    if ( value & PipelineStageFlagBits::eTransfer ) result += "Transfer | ";
    if ( value & PipelineStageFlagBits::eBottomOfPipe ) result += "BottomOfPipe | ";
    if ( value & PipelineStageFlagBits::eHost ) result += "Host | ";
    if ( value & PipelineStageFlagBits::eAllGraphics ) result += "AllGraphics | ";
    if ( value & PipelineStageFlagBits::eAllCommands ) result += "AllCommands | ";
    if ( value & PipelineStageFlagBits::eTransformFeedbackEXT ) result += "TransformFeedbackEXT | ";
    if ( value & PipelineStageFlagBits::eConditionalRenderingEXT ) result += "ConditionalRenderingEXT | ";
    if ( value & PipelineStageFlagBits::eCommandProcessNVX ) result += "CommandProcessNVX | ";
    if ( value & PipelineStageFlagBits::eShadingRateImageNV ) result += "ShadingRateImageNV | ";
    if ( value & PipelineStageFlagBits::eRayTracingShaderNV ) result += "RayTracingShaderNV | ";
    if ( value & PipelineStageFlagBits::eAccelerationStructureBuildNV ) result += "AccelerationStructureBuildNV | ";
    if ( value & PipelineStageFlagBits::eTaskShaderNV ) result += "TaskShaderNV | ";
    if ( value & PipelineStageFlagBits::eMeshShaderNV ) result += "MeshShaderNV | ";
    if ( value & PipelineStageFlagBits::eFragmentDensityProcessEXT ) result += "FragmentDensityProcessEXT";
    return result;
  }

  enum class PipelineTessellationStateCreateFlagBits
  {};

  VULKAN_HPP_INLINE std::string to_string( PipelineTessellationStateCreateFlagBits )
  {
    return "(void)";
  }

  using PipelineTessellationStateCreateFlags = Flags<PipelineTessellationStateCreateFlagBits, VkPipelineTessellationStateCreateFlags>;

  VULKAN_HPP_INLINE std::string to_string( PipelineTessellationStateCreateFlags  )
  {
    return "{}";
  }

  enum class PipelineVertexInputStateCreateFlagBits
  {};

  VULKAN_HPP_INLINE std::string to_string( PipelineVertexInputStateCreateFlagBits )
  {
    return "(void)";
  }

  using PipelineVertexInputStateCreateFlags = Flags<PipelineVertexInputStateCreateFlagBits, VkPipelineVertexInputStateCreateFlags>;

  VULKAN_HPP_INLINE std::string to_string( PipelineVertexInputStateCreateFlags  )
  {
    return "{}";
  }

  enum class PipelineViewportStateCreateFlagBits
  {};

  VULKAN_HPP_INLINE std::string to_string( PipelineViewportStateCreateFlagBits )
  {
    return "(void)";
  }

  using PipelineViewportStateCreateFlags = Flags<PipelineViewportStateCreateFlagBits, VkPipelineViewportStateCreateFlags>;

  VULKAN_HPP_INLINE std::string to_string( PipelineViewportStateCreateFlags  )
  {
    return "{}";
  }

  enum class PipelineViewportSwizzleStateCreateFlagBitsNV
  {};

  VULKAN_HPP_INLINE std::string to_string( PipelineViewportSwizzleStateCreateFlagBitsNV )
  {
    return "(void)";
  }

  using PipelineViewportSwizzleStateCreateFlagsNV = Flags<PipelineViewportSwizzleStateCreateFlagBitsNV, VkPipelineViewportSwizzleStateCreateFlagsNV>;

  VULKAN_HPP_INLINE std::string to_string( PipelineViewportSwizzleStateCreateFlagsNV  )
  {
    return "{}";
  }

  enum class QueryControlFlagBits
  {
    ePrecise = VK_QUERY_CONTROL_PRECISE_BIT
  };

  VULKAN_HPP_INLINE std::string to_string( QueryControlFlagBits value )
  {
    switch ( value )
    {
      case QueryControlFlagBits::ePrecise : return "Precise";
      default: return "invalid";
    }
  }

  using QueryControlFlags = Flags<QueryControlFlagBits, VkQueryControlFlags>;

  VULKAN_HPP_INLINE QueryControlFlags operator|( QueryControlFlagBits bit0, QueryControlFlagBits bit1 )
  {
    return QueryControlFlags( bit0 ) | bit1;
  }

  VULKAN_HPP_INLINE QueryControlFlags operator~( QueryControlFlagBits bits )
  {
    return ~( QueryControlFlags( bits ) );
  }

  template <> struct FlagTraits<QueryControlFlagBits>
  {
    enum
    {
      allFlags = VkFlags(QueryControlFlagBits::ePrecise)
    };
  };

  VULKAN_HPP_INLINE std::string to_string( QueryControlFlags value  )
  {
    if ( !value ) return "{}";
    std::string result;

    if ( value & QueryControlFlagBits::ePrecise ) result += "Precise";
    return result;
  }

  enum class QueryPipelineStatisticFlagBits
  {
    eInputAssemblyVertices = VK_QUERY_PIPELINE_STATISTIC_INPUT_ASSEMBLY_VERTICES_BIT,
    eInputAssemblyPrimitives = VK_QUERY_PIPELINE_STATISTIC_INPUT_ASSEMBLY_PRIMITIVES_BIT,
    eVertexShaderInvocations = VK_QUERY_PIPELINE_STATISTIC_VERTEX_SHADER_INVOCATIONS_BIT,
    eGeometryShaderInvocations = VK_QUERY_PIPELINE_STATISTIC_GEOMETRY_SHADER_INVOCATIONS_BIT,
    eGeometryShaderPrimitives = VK_QUERY_PIPELINE_STATISTIC_GEOMETRY_SHADER_PRIMITIVES_BIT,
    eClippingInvocations = VK_QUERY_PIPELINE_STATISTIC_CLIPPING_INVOCATIONS_BIT,
    eClippingPrimitives = VK_QUERY_PIPELINE_STATISTIC_CLIPPING_PRIMITIVES_BIT,
    eFragmentShaderInvocations = VK_QUERY_PIPELINE_STATISTIC_FRAGMENT_SHADER_INVOCATIONS_BIT,
    eTessellationControlShaderPatches = VK_QUERY_PIPELINE_STATISTIC_TESSELLATION_CONTROL_SHADER_PATCHES_BIT,
    eTessellationEvaluationShaderInvocations = VK_QUERY_PIPELINE_STATISTIC_TESSELLATION_EVALUATION_SHADER_INVOCATIONS_BIT,
    eComputeShaderInvocations = VK_QUERY_PIPELINE_STATISTIC_COMPUTE_SHADER_INVOCATIONS_BIT
  };

  VULKAN_HPP_INLINE std::string to_string( QueryPipelineStatisticFlagBits value )
  {
    switch ( value )
    {
      case QueryPipelineStatisticFlagBits::eInputAssemblyVertices : return "InputAssemblyVertices";
      case QueryPipelineStatisticFlagBits::eInputAssemblyPrimitives : return "InputAssemblyPrimitives";
      case QueryPipelineStatisticFlagBits::eVertexShaderInvocations : return "VertexShaderInvocations";
      case QueryPipelineStatisticFlagBits::eGeometryShaderInvocations : return "GeometryShaderInvocations";
      case QueryPipelineStatisticFlagBits::eGeometryShaderPrimitives : return "GeometryShaderPrimitives";
      case QueryPipelineStatisticFlagBits::eClippingInvocations : return "ClippingInvocations";
      case QueryPipelineStatisticFlagBits::eClippingPrimitives : return "ClippingPrimitives";
      case QueryPipelineStatisticFlagBits::eFragmentShaderInvocations : return "FragmentShaderInvocations";
      case QueryPipelineStatisticFlagBits::eTessellationControlShaderPatches : return "TessellationControlShaderPatches";
      case QueryPipelineStatisticFlagBits::eTessellationEvaluationShaderInvocations : return "TessellationEvaluationShaderInvocations";
      case QueryPipelineStatisticFlagBits::eComputeShaderInvocations : return "ComputeShaderInvocations";
      default: return "invalid";
    }
  }

  using QueryPipelineStatisticFlags = Flags<QueryPipelineStatisticFlagBits, VkQueryPipelineStatisticFlags>;

  VULKAN_HPP_INLINE QueryPipelineStatisticFlags operator|( QueryPipelineStatisticFlagBits bit0, QueryPipelineStatisticFlagBits bit1 )
  {
    return QueryPipelineStatisticFlags( bit0 ) | bit1;
  }

  VULKAN_HPP_INLINE QueryPipelineStatisticFlags operator~( QueryPipelineStatisticFlagBits bits )
  {
    return ~( QueryPipelineStatisticFlags( bits ) );
  }

  template <> struct FlagTraits<QueryPipelineStatisticFlagBits>
  {
    enum
    {
      allFlags = VkFlags(QueryPipelineStatisticFlagBits::eInputAssemblyVertices) | VkFlags(QueryPipelineStatisticFlagBits::eInputAssemblyPrimitives) | VkFlags(QueryPipelineStatisticFlagBits::eVertexShaderInvocations) | VkFlags(QueryPipelineStatisticFlagBits::eGeometryShaderInvocations) | VkFlags(QueryPipelineStatisticFlagBits::eGeometryShaderPrimitives) | VkFlags(QueryPipelineStatisticFlagBits::eClippingInvocations) | VkFlags(QueryPipelineStatisticFlagBits::eClippingPrimitives) | VkFlags(QueryPipelineStatisticFlagBits::eFragmentShaderInvocations) | VkFlags(QueryPipelineStatisticFlagBits::eTessellationControlShaderPatches) | VkFlags(QueryPipelineStatisticFlagBits::eTessellationEvaluationShaderInvocations) | VkFlags(QueryPipelineStatisticFlagBits::eComputeShaderInvocations)
    };
  };

  VULKAN_HPP_INLINE std::string to_string( QueryPipelineStatisticFlags value  )
  {
    if ( !value ) return "{}";
    std::string result;

    if ( value & QueryPipelineStatisticFlagBits::eInputAssemblyVertices ) result += "InputAssemblyVertices | ";
    if ( value & QueryPipelineStatisticFlagBits::eInputAssemblyPrimitives ) result += "InputAssemblyPrimitives | ";
    if ( value & QueryPipelineStatisticFlagBits::eVertexShaderInvocations ) result += "VertexShaderInvocations | ";
    if ( value & QueryPipelineStatisticFlagBits::eGeometryShaderInvocations ) result += "GeometryShaderInvocations | ";
    if ( value & QueryPipelineStatisticFlagBits::eGeometryShaderPrimitives ) result += "GeometryShaderPrimitives | ";
    if ( value & QueryPipelineStatisticFlagBits::eClippingInvocations ) result += "ClippingInvocations | ";
    if ( value & QueryPipelineStatisticFlagBits::eClippingPrimitives ) result += "ClippingPrimitives | ";
    if ( value & QueryPipelineStatisticFlagBits::eFragmentShaderInvocations ) result += "FragmentShaderInvocations | ";
    if ( value & QueryPipelineStatisticFlagBits::eTessellationControlShaderPatches ) result += "TessellationControlShaderPatches | ";
    if ( value & QueryPipelineStatisticFlagBits::eTessellationEvaluationShaderInvocations ) result += "TessellationEvaluationShaderInvocations | ";
    if ( value & QueryPipelineStatisticFlagBits::eComputeShaderInvocations ) result += "ComputeShaderInvocations";
    return result;
  }

  enum class QueryPoolCreateFlagBits
  {};

  VULKAN_HPP_INLINE std::string to_string( QueryPoolCreateFlagBits )
  {
    return "(void)";
  }

  using QueryPoolCreateFlags = Flags<QueryPoolCreateFlagBits, VkQueryPoolCreateFlags>;

  VULKAN_HPP_INLINE std::string to_string( QueryPoolCreateFlags  )
  {
    return "{}";
  }

  enum class QueryResultFlagBits
  {
    e64 = VK_QUERY_RESULT_64_BIT,
    eWait = VK_QUERY_RESULT_WAIT_BIT,
    eWithAvailability = VK_QUERY_RESULT_WITH_AVAILABILITY_BIT,
    ePartial = VK_QUERY_RESULT_PARTIAL_BIT
  };

  VULKAN_HPP_INLINE std::string to_string( QueryResultFlagBits value )
  {
    switch ( value )
    {
      case QueryResultFlagBits::e64 : return "64";
      case QueryResultFlagBits::eWait : return "Wait";
      case QueryResultFlagBits::eWithAvailability : return "WithAvailability";
      case QueryResultFlagBits::ePartial : return "Partial";
      default: return "invalid";
    }
  }

  using QueryResultFlags = Flags<QueryResultFlagBits, VkQueryResultFlags>;

  VULKAN_HPP_INLINE QueryResultFlags operator|( QueryResultFlagBits bit0, QueryResultFlagBits bit1 )
  {
    return QueryResultFlags( bit0 ) | bit1;
  }

  VULKAN_HPP_INLINE QueryResultFlags operator~( QueryResultFlagBits bits )
  {
    return ~( QueryResultFlags( bits ) );
  }

  template <> struct FlagTraits<QueryResultFlagBits>
  {
    enum
    {
      allFlags = VkFlags(QueryResultFlagBits::e64) | VkFlags(QueryResultFlagBits::eWait) | VkFlags(QueryResultFlagBits::eWithAvailability) | VkFlags(QueryResultFlagBits::ePartial)
    };
  };

  VULKAN_HPP_INLINE std::string to_string( QueryResultFlags value  )
  {
    if ( !value ) return "{}";
    std::string result;

    if ( value & QueryResultFlagBits::e64 ) result += "64 | ";
    if ( value & QueryResultFlagBits::eWait ) result += "Wait | ";
    if ( value & QueryResultFlagBits::eWithAvailability ) result += "WithAvailability | ";
    if ( value & QueryResultFlagBits::ePartial ) result += "Partial";
    return result;
  }

  enum class QueueFlagBits
  {
    eGraphics = VK_QUEUE_GRAPHICS_BIT,
    eCompute = VK_QUEUE_COMPUTE_BIT,
    eTransfer = VK_QUEUE_TRANSFER_BIT,
    eSparseBinding = VK_QUEUE_SPARSE_BINDING_BIT,
    eProtected = VK_QUEUE_PROTECTED_BIT
  };

  VULKAN_HPP_INLINE std::string to_string( QueueFlagBits value )
  {
    switch ( value )
    {
      case QueueFlagBits::eGraphics : return "Graphics";
      case QueueFlagBits::eCompute : return "Compute";
      case QueueFlagBits::eTransfer : return "Transfer";
      case QueueFlagBits::eSparseBinding : return "SparseBinding";
      case QueueFlagBits::eProtected : return "Protected";
      default: return "invalid";
    }
  }

  using QueueFlags = Flags<QueueFlagBits, VkQueueFlags>;

  VULKAN_HPP_INLINE QueueFlags operator|( QueueFlagBits bit0, QueueFlagBits bit1 )
  {
    return QueueFlags( bit0 ) | bit1;
  }

  VULKAN_HPP_INLINE QueueFlags operator~( QueueFlagBits bits )
  {
    return ~( QueueFlags( bits ) );
  }

  template <> struct FlagTraits<QueueFlagBits>
  {
    enum
    {
      allFlags = VkFlags(QueueFlagBits::eGraphics) | VkFlags(QueueFlagBits::eCompute) | VkFlags(QueueFlagBits::eTransfer) | VkFlags(QueueFlagBits::eSparseBinding) | VkFlags(QueueFlagBits::eProtected)
    };
  };

  VULKAN_HPP_INLINE std::string to_string( QueueFlags value  )
  {
    if ( !value ) return "{}";
    std::string result;

    if ( value & QueueFlagBits::eGraphics ) result += "Graphics | ";
    if ( value & QueueFlagBits::eCompute ) result += "Compute | ";
    if ( value & QueueFlagBits::eTransfer ) result += "Transfer | ";
    if ( value & QueueFlagBits::eSparseBinding ) result += "SparseBinding | ";
    if ( value & QueueFlagBits::eProtected ) result += "Protected";
    return result;
  }

  enum class RenderPassCreateFlagBits
  {};

  VULKAN_HPP_INLINE std::string to_string( RenderPassCreateFlagBits )
  {
    return "(void)";
  }

  using RenderPassCreateFlags = Flags<RenderPassCreateFlagBits, VkRenderPassCreateFlags>;

  VULKAN_HPP_INLINE std::string to_string( RenderPassCreateFlags  )
  {
    return "{}";
  }

  enum class ResolveModeFlagBitsKHR
  {
    eNone = VK_RESOLVE_MODE_NONE_KHR,
    eSampleZero = VK_RESOLVE_MODE_SAMPLE_ZERO_BIT_KHR,
    eAverage = VK_RESOLVE_MODE_AVERAGE_BIT_KHR,
    eMin = VK_RESOLVE_MODE_MIN_BIT_KHR,
    eMax = VK_RESOLVE_MODE_MAX_BIT_KHR
  };

  VULKAN_HPP_INLINE std::string to_string( ResolveModeFlagBitsKHR value )
  {
    switch ( value )
    {
      case ResolveModeFlagBitsKHR::eNone : return "None";
      case ResolveModeFlagBitsKHR::eSampleZero : return "SampleZero";
      case ResolveModeFlagBitsKHR::eAverage : return "Average";
      case ResolveModeFlagBitsKHR::eMin : return "Min";
      case ResolveModeFlagBitsKHR::eMax : return "Max";
      default: return "invalid";
    }
  }

  using ResolveModeFlagsKHR = Flags<ResolveModeFlagBitsKHR, VkResolveModeFlagsKHR>;

  VULKAN_HPP_INLINE ResolveModeFlagsKHR operator|( ResolveModeFlagBitsKHR bit0, ResolveModeFlagBitsKHR bit1 )
  {
    return ResolveModeFlagsKHR( bit0 ) | bit1;
  }

  VULKAN_HPP_INLINE ResolveModeFlagsKHR operator~( ResolveModeFlagBitsKHR bits )
  {
    return ~( ResolveModeFlagsKHR( bits ) );
  }

  template <> struct FlagTraits<ResolveModeFlagBitsKHR>
  {
    enum
    {
      allFlags = VkFlags(ResolveModeFlagBitsKHR::eNone) | VkFlags(ResolveModeFlagBitsKHR::eSampleZero) | VkFlags(ResolveModeFlagBitsKHR::eAverage) | VkFlags(ResolveModeFlagBitsKHR::eMin) | VkFlags(ResolveModeFlagBitsKHR::eMax)
    };
  };

  VULKAN_HPP_INLINE std::string to_string( ResolveModeFlagsKHR value  )
  {
    if ( !value ) return "{}";
    std::string result;

    if ( value & ResolveModeFlagBitsKHR::eNone ) result += "None | ";
    if ( value & ResolveModeFlagBitsKHR::eSampleZero ) result += "SampleZero | ";
    if ( value & ResolveModeFlagBitsKHR::eAverage ) result += "Average | ";
    if ( value & ResolveModeFlagBitsKHR::eMin ) result += "Min | ";
    if ( value & ResolveModeFlagBitsKHR::eMax ) result += "Max";
    return result;
  }

  enum class SampleCountFlagBits
  {
    e1 = VK_SAMPLE_COUNT_1_BIT,
    e2 = VK_SAMPLE_COUNT_2_BIT,
    e4 = VK_SAMPLE_COUNT_4_BIT,
    e8 = VK_SAMPLE_COUNT_8_BIT,
    e16 = VK_SAMPLE_COUNT_16_BIT,
    e32 = VK_SAMPLE_COUNT_32_BIT,
    e64 = VK_SAMPLE_COUNT_64_BIT
  };

  VULKAN_HPP_INLINE std::string to_string( SampleCountFlagBits value )
  {
    switch ( value )
    {
      case SampleCountFlagBits::e1 : return "1";
      case SampleCountFlagBits::e2 : return "2";
      case SampleCountFlagBits::e4 : return "4";
      case SampleCountFlagBits::e8 : return "8";
      case SampleCountFlagBits::e16 : return "16";
      case SampleCountFlagBits::e32 : return "32";
      case SampleCountFlagBits::e64 : return "64";
      default: return "invalid";
    }
  }

  using SampleCountFlags = Flags<SampleCountFlagBits, VkSampleCountFlags>;

  VULKAN_HPP_INLINE SampleCountFlags operator|( SampleCountFlagBits bit0, SampleCountFlagBits bit1 )
  {
    return SampleCountFlags( bit0 ) | bit1;
  }

  VULKAN_HPP_INLINE SampleCountFlags operator~( SampleCountFlagBits bits )
  {
    return ~( SampleCountFlags( bits ) );
  }

  template <> struct FlagTraits<SampleCountFlagBits>
  {
    enum
    {
      allFlags = VkFlags(SampleCountFlagBits::e1) | VkFlags(SampleCountFlagBits::e2) | VkFlags(SampleCountFlagBits::e4) | VkFlags(SampleCountFlagBits::e8) | VkFlags(SampleCountFlagBits::e16) | VkFlags(SampleCountFlagBits::e32) | VkFlags(SampleCountFlagBits::e64)
    };
  };

  VULKAN_HPP_INLINE std::string to_string( SampleCountFlags value  )
  {
    if ( !value ) return "{}";
    std::string result;

    if ( value & SampleCountFlagBits::e1 ) result += "1 | ";
    if ( value & SampleCountFlagBits::e2 ) result += "2 | ";
    if ( value & SampleCountFlagBits::e4 ) result += "4 | ";
    if ( value & SampleCountFlagBits::e8 ) result += "8 | ";
    if ( value & SampleCountFlagBits::e16 ) result += "16 | ";
    if ( value & SampleCountFlagBits::e32 ) result += "32 | ";
    if ( value & SampleCountFlagBits::e64 ) result += "64";
    return result;
  }

  enum class SamplerCreateFlagBits
  {
    eSubsampledEXT = VK_SAMPLER_CREATE_SUBSAMPLED_BIT_EXT,
    eSubsampledCoarseReconstructionEXT = VK_SAMPLER_CREATE_SUBSAMPLED_COARSE_RECONSTRUCTION_BIT_EXT
  };

  VULKAN_HPP_INLINE std::string to_string( SamplerCreateFlagBits value )
  {
    switch ( value )
    {
      case SamplerCreateFlagBits::eSubsampledEXT : return "SubsampledEXT";
      case SamplerCreateFlagBits::eSubsampledCoarseReconstructionEXT : return "SubsampledCoarseReconstructionEXT";
      default: return "invalid";
    }
  }

  using SamplerCreateFlags = Flags<SamplerCreateFlagBits, VkSamplerCreateFlags>;

  VULKAN_HPP_INLINE SamplerCreateFlags operator|( SamplerCreateFlagBits bit0, SamplerCreateFlagBits bit1 )
  {
    return SamplerCreateFlags( bit0 ) | bit1;
  }

  VULKAN_HPP_INLINE SamplerCreateFlags operator~( SamplerCreateFlagBits bits )
  {
    return ~( SamplerCreateFlags( bits ) );
  }

  template <> struct FlagTraits<SamplerCreateFlagBits>
  {
    enum
    {
      allFlags = VkFlags(SamplerCreateFlagBits::eSubsampledEXT) | VkFlags(SamplerCreateFlagBits::eSubsampledCoarseReconstructionEXT)
    };
  };

  VULKAN_HPP_INLINE std::string to_string( SamplerCreateFlags value  )
  {
    if ( !value ) return "{}";
    std::string result;

    if ( value & SamplerCreateFlagBits::eSubsampledEXT ) result += "SubsampledEXT | ";
    if ( value & SamplerCreateFlagBits::eSubsampledCoarseReconstructionEXT ) result += "SubsampledCoarseReconstructionEXT";
    return result;
  }

  enum class SemaphoreCreateFlagBits
  {};

  VULKAN_HPP_INLINE std::string to_string( SemaphoreCreateFlagBits )
  {
    return "(void)";
  }

  using SemaphoreCreateFlags = Flags<SemaphoreCreateFlagBits, VkSemaphoreCreateFlags>;

  VULKAN_HPP_INLINE std::string to_string( SemaphoreCreateFlags  )
  {
    return "{}";
  }

  enum class SemaphoreImportFlagBits
  {
    eTemporary = VK_SEMAPHORE_IMPORT_TEMPORARY_BIT,
    eTemporaryKHR = VK_SEMAPHORE_IMPORT_TEMPORARY_BIT_KHR
  };

  VULKAN_HPP_INLINE std::string to_string( SemaphoreImportFlagBits value )
  {
    switch ( value )
    {
      case SemaphoreImportFlagBits::eTemporary : return "Temporary";
      default: return "invalid";
    }
  }

  using SemaphoreImportFlags = Flags<SemaphoreImportFlagBits, VkSemaphoreImportFlags>;

  VULKAN_HPP_INLINE SemaphoreImportFlags operator|( SemaphoreImportFlagBits bit0, SemaphoreImportFlagBits bit1 )
  {
    return SemaphoreImportFlags( bit0 ) | bit1;
  }

  VULKAN_HPP_INLINE SemaphoreImportFlags operator~( SemaphoreImportFlagBits bits )
  {
    return ~( SemaphoreImportFlags( bits ) );
  }

  template <> struct FlagTraits<SemaphoreImportFlagBits>
  {
    enum
    {
      allFlags = VkFlags(SemaphoreImportFlagBits::eTemporary)
    };
  };

  using SemaphoreImportFlagsKHR = SemaphoreImportFlags;

  VULKAN_HPP_INLINE std::string to_string( SemaphoreImportFlags value  )
  {
    if ( !value ) return "{}";
    std::string result;

    if ( value & SemaphoreImportFlagBits::eTemporary ) result += "Temporary";
    return result;
  }

  enum class ShaderModuleCreateFlagBits
  {};

  VULKAN_HPP_INLINE std::string to_string( ShaderModuleCreateFlagBits )
  {
    return "(void)";
  }

  using ShaderModuleCreateFlags = Flags<ShaderModuleCreateFlagBits, VkShaderModuleCreateFlags>;

  VULKAN_HPP_INLINE std::string to_string( ShaderModuleCreateFlags  )
  {
    return "{}";
  }

  enum class ShaderStageFlagBits
  {
    eVertex = VK_SHADER_STAGE_VERTEX_BIT,
    eTessellationControl = VK_SHADER_STAGE_TESSELLATION_CONTROL_BIT,
    eTessellationEvaluation = VK_SHADER_STAGE_TESSELLATION_EVALUATION_BIT,
    eGeometry = VK_SHADER_STAGE_GEOMETRY_BIT,
    eFragment = VK_SHADER_STAGE_FRAGMENT_BIT,
    eCompute = VK_SHADER_STAGE_COMPUTE_BIT,
    eAllGraphics = VK_SHADER_STAGE_ALL_GRAPHICS,
    eAll = VK_SHADER_STAGE_ALL,
    eRaygenNV = VK_SHADER_STAGE_RAYGEN_BIT_NV,
    eAnyHitNV = VK_SHADER_STAGE_ANY_HIT_BIT_NV,
    eClosestHitNV = VK_SHADER_STAGE_CLOSEST_HIT_BIT_NV,
    eMissNV = VK_SHADER_STAGE_MISS_BIT_NV,
    eIntersectionNV = VK_SHADER_STAGE_INTERSECTION_BIT_NV,
    eCallableNV = VK_SHADER_STAGE_CALLABLE_BIT_NV,
    eTaskNV = VK_SHADER_STAGE_TASK_BIT_NV,
    eMeshNV = VK_SHADER_STAGE_MESH_BIT_NV
  };

  VULKAN_HPP_INLINE std::string to_string( ShaderStageFlagBits value )
  {
    switch ( value )
    {
      case ShaderStageFlagBits::eVertex : return "Vertex";
      case ShaderStageFlagBits::eTessellationControl : return "TessellationControl";
      case ShaderStageFlagBits::eTessellationEvaluation : return "TessellationEvaluation";
      case ShaderStageFlagBits::eGeometry : return "Geometry";
      case ShaderStageFlagBits::eFragment : return "Fragment";
      case ShaderStageFlagBits::eCompute : return "Compute";
      case ShaderStageFlagBits::eAllGraphics : return "AllGraphics";
      case ShaderStageFlagBits::eAll : return "All";
      case ShaderStageFlagBits::eRaygenNV : return "RaygenNV";
      case ShaderStageFlagBits::eAnyHitNV : return "AnyHitNV";
      case ShaderStageFlagBits::eClosestHitNV : return "ClosestHitNV";
      case ShaderStageFlagBits::eMissNV : return "MissNV";
      case ShaderStageFlagBits::eIntersectionNV : return "IntersectionNV";
      case ShaderStageFlagBits::eCallableNV : return "CallableNV";
      case ShaderStageFlagBits::eTaskNV : return "TaskNV";
      case ShaderStageFlagBits::eMeshNV : return "MeshNV";
      default: return "invalid";
    }
  }

  using ShaderStageFlags = Flags<ShaderStageFlagBits, VkShaderStageFlags>;

  VULKAN_HPP_INLINE ShaderStageFlags operator|( ShaderStageFlagBits bit0, ShaderStageFlagBits bit1 )
  {
    return ShaderStageFlags( bit0 ) | bit1;
  }

  VULKAN_HPP_INLINE ShaderStageFlags operator~( ShaderStageFlagBits bits )
  {
    return ~( ShaderStageFlags( bits ) );
  }

  template <> struct FlagTraits<ShaderStageFlagBits>
  {
    enum
    {
      allFlags = VkFlags(ShaderStageFlagBits::eVertex) | VkFlags(ShaderStageFlagBits::eTessellationControl) | VkFlags(ShaderStageFlagBits::eTessellationEvaluation) | VkFlags(ShaderStageFlagBits::eGeometry) | VkFlags(ShaderStageFlagBits::eFragment) | VkFlags(ShaderStageFlagBits::eCompute) | VkFlags(ShaderStageFlagBits::eAllGraphics) | VkFlags(ShaderStageFlagBits::eAll) | VkFlags(ShaderStageFlagBits::eRaygenNV) | VkFlags(ShaderStageFlagBits::eAnyHitNV) | VkFlags(ShaderStageFlagBits::eClosestHitNV) | VkFlags(ShaderStageFlagBits::eMissNV) | VkFlags(ShaderStageFlagBits::eIntersectionNV) | VkFlags(ShaderStageFlagBits::eCallableNV) | VkFlags(ShaderStageFlagBits::eTaskNV) | VkFlags(ShaderStageFlagBits::eMeshNV)
    };
  };

  VULKAN_HPP_INLINE std::string to_string( ShaderStageFlags value  )
  {
    if ( !value ) return "{}";
    std::string result;

    if ( value & ShaderStageFlagBits::eVertex ) result += "Vertex | ";
    if ( value & ShaderStageFlagBits::eTessellationControl ) result += "TessellationControl | ";
    if ( value & ShaderStageFlagBits::eTessellationEvaluation ) result += "TessellationEvaluation | ";
    if ( value & ShaderStageFlagBits::eGeometry ) result += "Geometry | ";
    if ( value & ShaderStageFlagBits::eFragment ) result += "Fragment | ";
    if ( value & ShaderStageFlagBits::eCompute ) result += "Compute | ";
    if ( value & ShaderStageFlagBits::eAllGraphics ) result += "AllGraphics | ";
    if ( value & ShaderStageFlagBits::eAll ) result += "All | ";
    if ( value & ShaderStageFlagBits::eRaygenNV ) result += "RaygenNV | ";
    if ( value & ShaderStageFlagBits::eAnyHitNV ) result += "AnyHitNV | ";
    if ( value & ShaderStageFlagBits::eClosestHitNV ) result += "ClosestHitNV | ";
    if ( value & ShaderStageFlagBits::eMissNV ) result += "MissNV | ";
    if ( value & ShaderStageFlagBits::eIntersectionNV ) result += "IntersectionNV | ";
    if ( value & ShaderStageFlagBits::eCallableNV ) result += "CallableNV | ";
    if ( value & ShaderStageFlagBits::eTaskNV ) result += "TaskNV | ";
    if ( value & ShaderStageFlagBits::eMeshNV ) result += "MeshNV";
    return result;
  }

  enum class SparseImageFormatFlagBits
  {
    eSingleMiptail = VK_SPARSE_IMAGE_FORMAT_SINGLE_MIPTAIL_BIT,
    eAlignedMipSize = VK_SPARSE_IMAGE_FORMAT_ALIGNED_MIP_SIZE_BIT,
    eNonstandardBlockSize = VK_SPARSE_IMAGE_FORMAT_NONSTANDARD_BLOCK_SIZE_BIT
  };

  VULKAN_HPP_INLINE std::string to_string( SparseImageFormatFlagBits value )
  {
    switch ( value )
    {
      case SparseImageFormatFlagBits::eSingleMiptail : return "SingleMiptail";
      case SparseImageFormatFlagBits::eAlignedMipSize : return "AlignedMipSize";
      case SparseImageFormatFlagBits::eNonstandardBlockSize : return "NonstandardBlockSize";
      default: return "invalid";
    }
  }

  using SparseImageFormatFlags = Flags<SparseImageFormatFlagBits, VkSparseImageFormatFlags>;

  VULKAN_HPP_INLINE SparseImageFormatFlags operator|( SparseImageFormatFlagBits bit0, SparseImageFormatFlagBits bit1 )
  {
    return SparseImageFormatFlags( bit0 ) | bit1;
  }

  VULKAN_HPP_INLINE SparseImageFormatFlags operator~( SparseImageFormatFlagBits bits )
  {
    return ~( SparseImageFormatFlags( bits ) );
  }

  template <> struct FlagTraits<SparseImageFormatFlagBits>
  {
    enum
    {
      allFlags = VkFlags(SparseImageFormatFlagBits::eSingleMiptail) | VkFlags(SparseImageFormatFlagBits::eAlignedMipSize) | VkFlags(SparseImageFormatFlagBits::eNonstandardBlockSize)
    };
  };

  VULKAN_HPP_INLINE std::string to_string( SparseImageFormatFlags value  )
  {
    if ( !value ) return "{}";
    std::string result;

    if ( value & SparseImageFormatFlagBits::eSingleMiptail ) result += "SingleMiptail | ";
    if ( value & SparseImageFormatFlagBits::eAlignedMipSize ) result += "AlignedMipSize | ";
    if ( value & SparseImageFormatFlagBits::eNonstandardBlockSize ) result += "NonstandardBlockSize";
    return result;
  }

  enum class SparseMemoryBindFlagBits
  {
    eMetadata = VK_SPARSE_MEMORY_BIND_METADATA_BIT
  };

  VULKAN_HPP_INLINE std::string to_string( SparseMemoryBindFlagBits value )
  {
    switch ( value )
    {
      case SparseMemoryBindFlagBits::eMetadata : return "Metadata";
      default: return "invalid";
    }
  }

  using SparseMemoryBindFlags = Flags<SparseMemoryBindFlagBits, VkSparseMemoryBindFlags>;

  VULKAN_HPP_INLINE SparseMemoryBindFlags operator|( SparseMemoryBindFlagBits bit0, SparseMemoryBindFlagBits bit1 )
  {
    return SparseMemoryBindFlags( bit0 ) | bit1;
  }

  VULKAN_HPP_INLINE SparseMemoryBindFlags operator~( SparseMemoryBindFlagBits bits )
  {
    return ~( SparseMemoryBindFlags( bits ) );
  }

  template <> struct FlagTraits<SparseMemoryBindFlagBits>
  {
    enum
    {
      allFlags = VkFlags(SparseMemoryBindFlagBits::eMetadata)
    };
  };

  VULKAN_HPP_INLINE std::string to_string( SparseMemoryBindFlags value  )
  {
    if ( !value ) return "{}";
    std::string result;

    if ( value & SparseMemoryBindFlagBits::eMetadata ) result += "Metadata";
    return result;
  }

  enum class StencilFaceFlagBits
  {
    eFront = VK_STENCIL_FACE_FRONT_BIT,
    eBack = VK_STENCIL_FACE_BACK_BIT,
    eVkStencilFrontAndBack = VK_STENCIL_FRONT_AND_BACK
  };

  VULKAN_HPP_INLINE std::string to_string( StencilFaceFlagBits value )
  {
    switch ( value )
    {
      case StencilFaceFlagBits::eFront : return "Front";
      case StencilFaceFlagBits::eBack : return "Back";
      case StencilFaceFlagBits::eVkStencilFrontAndBack : return "VkStencilFrontAndBack";
      default: return "invalid";
    }
  }

  using StencilFaceFlags = Flags<StencilFaceFlagBits, VkStencilFaceFlags>;

  VULKAN_HPP_INLINE StencilFaceFlags operator|( StencilFaceFlagBits bit0, StencilFaceFlagBits bit1 )
  {
    return StencilFaceFlags( bit0 ) | bit1;
  }

  VULKAN_HPP_INLINE StencilFaceFlags operator~( StencilFaceFlagBits bits )
  {
    return ~( StencilFaceFlags( bits ) );
  }

  template <> struct FlagTraits<StencilFaceFlagBits>
  {
    enum
    {
      allFlags = VkFlags(StencilFaceFlagBits::eFront) | VkFlags(StencilFaceFlagBits::eBack) | VkFlags(StencilFaceFlagBits::eVkStencilFrontAndBack)
    };
  };

  VULKAN_HPP_INLINE std::string to_string( StencilFaceFlags value  )
  {
    if ( !value ) return "{}";
    std::string result;

    if ( value & StencilFaceFlagBits::eFront ) result += "Front | ";
    if ( value & StencilFaceFlagBits::eBack ) result += "Back | ";
    if ( value & StencilFaceFlagBits::eVkStencilFrontAndBack ) result += "VkStencilFrontAndBack";
    return result;
  }

  enum class SubgroupFeatureFlagBits
  {
    eBasic = VK_SUBGROUP_FEATURE_BASIC_BIT,
    eVote = VK_SUBGROUP_FEATURE_VOTE_BIT,
    eArithmetic = VK_SUBGROUP_FEATURE_ARITHMETIC_BIT,
    eBallot = VK_SUBGROUP_FEATURE_BALLOT_BIT,
    eShuffle = VK_SUBGROUP_FEATURE_SHUFFLE_BIT,
    eShuffleRelative = VK_SUBGROUP_FEATURE_SHUFFLE_RELATIVE_BIT,
    eClustered = VK_SUBGROUP_FEATURE_CLUSTERED_BIT,
    eQuad = VK_SUBGROUP_FEATURE_QUAD_BIT,
    ePartitionedNV = VK_SUBGROUP_FEATURE_PARTITIONED_BIT_NV
  };

  VULKAN_HPP_INLINE std::string to_string( SubgroupFeatureFlagBits value )
  {
    switch ( value )
    {
      case SubgroupFeatureFlagBits::eBasic : return "Basic";
      case SubgroupFeatureFlagBits::eVote : return "Vote";
      case SubgroupFeatureFlagBits::eArithmetic : return "Arithmetic";
      case SubgroupFeatureFlagBits::eBallot : return "Ballot";
      case SubgroupFeatureFlagBits::eShuffle : return "Shuffle";
      case SubgroupFeatureFlagBits::eShuffleRelative : return "ShuffleRelative";
      case SubgroupFeatureFlagBits::eClustered : return "Clustered";
      case SubgroupFeatureFlagBits::eQuad : return "Quad";
      case SubgroupFeatureFlagBits::ePartitionedNV : return "PartitionedNV";
      default: return "invalid";
    }
  }

  using SubgroupFeatureFlags = Flags<SubgroupFeatureFlagBits, VkSubgroupFeatureFlags>;

  VULKAN_HPP_INLINE SubgroupFeatureFlags operator|( SubgroupFeatureFlagBits bit0, SubgroupFeatureFlagBits bit1 )
  {
    return SubgroupFeatureFlags( bit0 ) | bit1;
  }

  VULKAN_HPP_INLINE SubgroupFeatureFlags operator~( SubgroupFeatureFlagBits bits )
  {
    return ~( SubgroupFeatureFlags( bits ) );
  }

  template <> struct FlagTraits<SubgroupFeatureFlagBits>
  {
    enum
    {
      allFlags = VkFlags(SubgroupFeatureFlagBits::eBasic) | VkFlags(SubgroupFeatureFlagBits::eVote) | VkFlags(SubgroupFeatureFlagBits::eArithmetic) | VkFlags(SubgroupFeatureFlagBits::eBallot) | VkFlags(SubgroupFeatureFlagBits::eShuffle) | VkFlags(SubgroupFeatureFlagBits::eShuffleRelative) | VkFlags(SubgroupFeatureFlagBits::eClustered) | VkFlags(SubgroupFeatureFlagBits::eQuad) | VkFlags(SubgroupFeatureFlagBits::ePartitionedNV)
    };
  };

  VULKAN_HPP_INLINE std::string to_string( SubgroupFeatureFlags value  )
  {
    if ( !value ) return "{}";
    std::string result;

    if ( value & SubgroupFeatureFlagBits::eBasic ) result += "Basic | ";
    if ( value & SubgroupFeatureFlagBits::eVote ) result += "Vote | ";
    if ( value & SubgroupFeatureFlagBits::eArithmetic ) result += "Arithmetic | ";
    if ( value & SubgroupFeatureFlagBits::eBallot ) result += "Ballot | ";
    if ( value & SubgroupFeatureFlagBits::eShuffle ) result += "Shuffle | ";
    if ( value & SubgroupFeatureFlagBits::eShuffleRelative ) result += "ShuffleRelative | ";
    if ( value & SubgroupFeatureFlagBits::eClustered ) result += "Clustered | ";
    if ( value & SubgroupFeatureFlagBits::eQuad ) result += "Quad | ";
    if ( value & SubgroupFeatureFlagBits::ePartitionedNV ) result += "PartitionedNV";
    return result;
  }

  enum class SubpassDescriptionFlagBits
  {
    ePerViewAttributesNVX = VK_SUBPASS_DESCRIPTION_PER_VIEW_ATTRIBUTES_BIT_NVX,
    ePerViewPositionXOnlyNVX = VK_SUBPASS_DESCRIPTION_PER_VIEW_POSITION_X_ONLY_BIT_NVX
  };

  VULKAN_HPP_INLINE std::string to_string( SubpassDescriptionFlagBits value )
  {
    switch ( value )
    {
      case SubpassDescriptionFlagBits::ePerViewAttributesNVX : return "PerViewAttributesNVX";
      case SubpassDescriptionFlagBits::ePerViewPositionXOnlyNVX : return "PerViewPositionXOnlyNVX";
      default: return "invalid";
    }
  }

  using SubpassDescriptionFlags = Flags<SubpassDescriptionFlagBits, VkSubpassDescriptionFlags>;

  VULKAN_HPP_INLINE SubpassDescriptionFlags operator|( SubpassDescriptionFlagBits bit0, SubpassDescriptionFlagBits bit1 )
  {
    return SubpassDescriptionFlags( bit0 ) | bit1;
  }

  VULKAN_HPP_INLINE SubpassDescriptionFlags operator~( SubpassDescriptionFlagBits bits )
  {
    return ~( SubpassDescriptionFlags( bits ) );
  }

  template <> struct FlagTraits<SubpassDescriptionFlagBits>
  {
    enum
    {
      allFlags = VkFlags(SubpassDescriptionFlagBits::ePerViewAttributesNVX) | VkFlags(SubpassDescriptionFlagBits::ePerViewPositionXOnlyNVX)
    };
  };

  VULKAN_HPP_INLINE std::string to_string( SubpassDescriptionFlags value  )
  {
    if ( !value ) return "{}";
    std::string result;

    if ( value & SubpassDescriptionFlagBits::ePerViewAttributesNVX ) result += "PerViewAttributesNVX | ";
    if ( value & SubpassDescriptionFlagBits::ePerViewPositionXOnlyNVX ) result += "PerViewPositionXOnlyNVX";
    return result;
  }

  enum class SurfaceCounterFlagBitsEXT
  {
    eVblank = VK_SURFACE_COUNTER_VBLANK_EXT
  };

  VULKAN_HPP_INLINE std::string to_string( SurfaceCounterFlagBitsEXT value )
  {
    switch ( value )
    {
      case SurfaceCounterFlagBitsEXT::eVblank : return "Vblank";
      default: return "invalid";
    }
  }

  using SurfaceCounterFlagsEXT = Flags<SurfaceCounterFlagBitsEXT, VkSurfaceCounterFlagsEXT>;

  VULKAN_HPP_INLINE SurfaceCounterFlagsEXT operator|( SurfaceCounterFlagBitsEXT bit0, SurfaceCounterFlagBitsEXT bit1 )
  {
    return SurfaceCounterFlagsEXT( bit0 ) | bit1;
  }

  VULKAN_HPP_INLINE SurfaceCounterFlagsEXT operator~( SurfaceCounterFlagBitsEXT bits )
  {
    return ~( SurfaceCounterFlagsEXT( bits ) );
  }

  template <> struct FlagTraits<SurfaceCounterFlagBitsEXT>
  {
    enum
    {
      allFlags = VkFlags(SurfaceCounterFlagBitsEXT::eVblank)
    };
  };

  VULKAN_HPP_INLINE std::string to_string( SurfaceCounterFlagsEXT value  )
  {
    if ( !value ) return "{}";
    std::string result;

    if ( value & SurfaceCounterFlagBitsEXT::eVblank ) result += "Vblank";
    return result;
  }

  enum class SurfaceTransformFlagBitsKHR
  {
    eIdentity = VK_SURFACE_TRANSFORM_IDENTITY_BIT_KHR,
    eRotate90 = VK_SURFACE_TRANSFORM_ROTATE_90_BIT_KHR,
    eRotate180 = VK_SURFACE_TRANSFORM_ROTATE_180_BIT_KHR,
    eRotate270 = VK_SURFACE_TRANSFORM_ROTATE_270_BIT_KHR,
    eHorizontalMirror = VK_SURFACE_TRANSFORM_HORIZONTAL_MIRROR_BIT_KHR,
    eHorizontalMirrorRotate90 = VK_SURFACE_TRANSFORM_HORIZONTAL_MIRROR_ROTATE_90_BIT_KHR,
    eHorizontalMirrorRotate180 = VK_SURFACE_TRANSFORM_HORIZONTAL_MIRROR_ROTATE_180_BIT_KHR,
    eHorizontalMirrorRotate270 = VK_SURFACE_TRANSFORM_HORIZONTAL_MIRROR_ROTATE_270_BIT_KHR,
    eInherit = VK_SURFACE_TRANSFORM_INHERIT_BIT_KHR
  };

  VULKAN_HPP_INLINE std::string to_string( SurfaceTransformFlagBitsKHR value )
  {
    switch ( value )
    {
      case SurfaceTransformFlagBitsKHR::eIdentity : return "Identity";
      case SurfaceTransformFlagBitsKHR::eRotate90 : return "Rotate90";
      case SurfaceTransformFlagBitsKHR::eRotate180 : return "Rotate180";
      case SurfaceTransformFlagBitsKHR::eRotate270 : return "Rotate270";
      case SurfaceTransformFlagBitsKHR::eHorizontalMirror : return "HorizontalMirror";
      case SurfaceTransformFlagBitsKHR::eHorizontalMirrorRotate90 : return "HorizontalMirrorRotate90";
      case SurfaceTransformFlagBitsKHR::eHorizontalMirrorRotate180 : return "HorizontalMirrorRotate180";
      case SurfaceTransformFlagBitsKHR::eHorizontalMirrorRotate270 : return "HorizontalMirrorRotate270";
      case SurfaceTransformFlagBitsKHR::eInherit : return "Inherit";
      default: return "invalid";
    }
  }

  using SurfaceTransformFlagsKHR = Flags<SurfaceTransformFlagBitsKHR, VkSurfaceTransformFlagsKHR>;

  VULKAN_HPP_INLINE SurfaceTransformFlagsKHR operator|( SurfaceTransformFlagBitsKHR bit0, SurfaceTransformFlagBitsKHR bit1 )
  {
    return SurfaceTransformFlagsKHR( bit0 ) | bit1;
  }

  VULKAN_HPP_INLINE SurfaceTransformFlagsKHR operator~( SurfaceTransformFlagBitsKHR bits )
  {
    return ~( SurfaceTransformFlagsKHR( bits ) );
  }

  template <> struct FlagTraits<SurfaceTransformFlagBitsKHR>
  {
    enum
    {
      allFlags = VkFlags(SurfaceTransformFlagBitsKHR::eIdentity) | VkFlags(SurfaceTransformFlagBitsKHR::eRotate90) | VkFlags(SurfaceTransformFlagBitsKHR::eRotate180) | VkFlags(SurfaceTransformFlagBitsKHR::eRotate270) | VkFlags(SurfaceTransformFlagBitsKHR::eHorizontalMirror) | VkFlags(SurfaceTransformFlagBitsKHR::eHorizontalMirrorRotate90) | VkFlags(SurfaceTransformFlagBitsKHR::eHorizontalMirrorRotate180) | VkFlags(SurfaceTransformFlagBitsKHR::eHorizontalMirrorRotate270) | VkFlags(SurfaceTransformFlagBitsKHR::eInherit)
    };
  };

  VULKAN_HPP_INLINE std::string to_string( SurfaceTransformFlagsKHR value  )
  {
    if ( !value ) return "{}";
    std::string result;

    if ( value & SurfaceTransformFlagBitsKHR::eIdentity ) result += "Identity | ";
    if ( value & SurfaceTransformFlagBitsKHR::eRotate90 ) result += "Rotate90 | ";
    if ( value & SurfaceTransformFlagBitsKHR::eRotate180 ) result += "Rotate180 | ";
    if ( value & SurfaceTransformFlagBitsKHR::eRotate270 ) result += "Rotate270 | ";
    if ( value & SurfaceTransformFlagBitsKHR::eHorizontalMirror ) result += "HorizontalMirror | ";
    if ( value & SurfaceTransformFlagBitsKHR::eHorizontalMirrorRotate90 ) result += "HorizontalMirrorRotate90 | ";
    if ( value & SurfaceTransformFlagBitsKHR::eHorizontalMirrorRotate180 ) result += "HorizontalMirrorRotate180 | ";
    if ( value & SurfaceTransformFlagBitsKHR::eHorizontalMirrorRotate270 ) result += "HorizontalMirrorRotate270 | ";
    if ( value & SurfaceTransformFlagBitsKHR::eInherit ) result += "Inherit";
    return result;
  }

  enum class SwapchainCreateFlagBitsKHR
  {
    eSplitInstanceBindRegions = VK_SWAPCHAIN_CREATE_SPLIT_INSTANCE_BIND_REGIONS_BIT_KHR,
    eProtected = VK_SWAPCHAIN_CREATE_PROTECTED_BIT_KHR,
    eMutableFormat = VK_SWAPCHAIN_CREATE_MUTABLE_FORMAT_BIT_KHR
  };

  VULKAN_HPP_INLINE std::string to_string( SwapchainCreateFlagBitsKHR value )
  {
    switch ( value )
    {
      case SwapchainCreateFlagBitsKHR::eSplitInstanceBindRegions : return "SplitInstanceBindRegions";
      case SwapchainCreateFlagBitsKHR::eProtected : return "Protected";
      case SwapchainCreateFlagBitsKHR::eMutableFormat : return "MutableFormat";
      default: return "invalid";
    }
  }

  using SwapchainCreateFlagsKHR = Flags<SwapchainCreateFlagBitsKHR, VkSwapchainCreateFlagsKHR>;

  VULKAN_HPP_INLINE SwapchainCreateFlagsKHR operator|( SwapchainCreateFlagBitsKHR bit0, SwapchainCreateFlagBitsKHR bit1 )
  {
    return SwapchainCreateFlagsKHR( bit0 ) | bit1;
  }

  VULKAN_HPP_INLINE SwapchainCreateFlagsKHR operator~( SwapchainCreateFlagBitsKHR bits )
  {
    return ~( SwapchainCreateFlagsKHR( bits ) );
  }

  template <> struct FlagTraits<SwapchainCreateFlagBitsKHR>
  {
    enum
    {
      allFlags = VkFlags(SwapchainCreateFlagBitsKHR::eSplitInstanceBindRegions) | VkFlags(SwapchainCreateFlagBitsKHR::eProtected) | VkFlags(SwapchainCreateFlagBitsKHR::eMutableFormat)
    };
  };

  VULKAN_HPP_INLINE std::string to_string( SwapchainCreateFlagsKHR value  )
  {
    if ( !value ) return "{}";
    std::string result;

    if ( value & SwapchainCreateFlagBitsKHR::eSplitInstanceBindRegions ) result += "SplitInstanceBindRegions | ";
    if ( value & SwapchainCreateFlagBitsKHR::eProtected ) result += "Protected | ";
    if ( value & SwapchainCreateFlagBitsKHR::eMutableFormat ) result += "MutableFormat";
    return result;
  }

  enum class ValidationCacheCreateFlagBitsEXT
  {};

  VULKAN_HPP_INLINE std::string to_string( ValidationCacheCreateFlagBitsEXT )
  {
    return "(void)";
  }

  using ValidationCacheCreateFlagsEXT = Flags<ValidationCacheCreateFlagBitsEXT, VkValidationCacheCreateFlagsEXT>;

  VULKAN_HPP_INLINE std::string to_string( ValidationCacheCreateFlagsEXT  )
  {
    return "{}";
  }

#ifdef VK_USE_PLATFORM_VI_NN
  enum class ViSurfaceCreateFlagBitsNN
  {};

  VULKAN_HPP_INLINE std::string to_string( ViSurfaceCreateFlagBitsNN )
  {
    return "(void)";
  }

  using ViSurfaceCreateFlagsNN = Flags<ViSurfaceCreateFlagBitsNN, VkViSurfaceCreateFlagsNN>;

  VULKAN_HPP_INLINE std::string to_string( ViSurfaceCreateFlagsNN  )
  {
    return "{}";
  }
#endif /*VK_USE_PLATFORM_VI_NN*/

#ifdef VK_USE_PLATFORM_WAYLAND_KHR
  enum class WaylandSurfaceCreateFlagBitsKHR
  {};

  VULKAN_HPP_INLINE std::string to_string( WaylandSurfaceCreateFlagBitsKHR )
  {
    return "(void)";
  }

  using WaylandSurfaceCreateFlagsKHR = Flags<WaylandSurfaceCreateFlagBitsKHR, VkWaylandSurfaceCreateFlagsKHR>;

  VULKAN_HPP_INLINE std::string to_string( WaylandSurfaceCreateFlagsKHR  )
  {
    return "{}";
  }
#endif /*VK_USE_PLATFORM_WAYLAND_KHR*/

#ifdef VK_USE_PLATFORM_WIN32_KHR
  enum class Win32SurfaceCreateFlagBitsKHR
  {};

  VULKAN_HPP_INLINE std::string to_string( Win32SurfaceCreateFlagBitsKHR )
  {
    return "(void)";
  }

  using Win32SurfaceCreateFlagsKHR = Flags<Win32SurfaceCreateFlagBitsKHR, VkWin32SurfaceCreateFlagsKHR>;

  VULKAN_HPP_INLINE std::string to_string( Win32SurfaceCreateFlagsKHR  )
  {
    return "{}";
  }
#endif /*VK_USE_PLATFORM_WIN32_KHR*/

#ifdef VK_USE_PLATFORM_XCB_KHR
  enum class XcbSurfaceCreateFlagBitsKHR
  {};

  VULKAN_HPP_INLINE std::string to_string( XcbSurfaceCreateFlagBitsKHR )
  {
    return "(void)";
  }

  using XcbSurfaceCreateFlagsKHR = Flags<XcbSurfaceCreateFlagBitsKHR, VkXcbSurfaceCreateFlagsKHR>;

  VULKAN_HPP_INLINE std::string to_string( XcbSurfaceCreateFlagsKHR  )
  {
    return "{}";
  }
#endif /*VK_USE_PLATFORM_XCB_KHR*/

#ifdef VK_USE_PLATFORM_XLIB_KHR
  enum class XlibSurfaceCreateFlagBitsKHR
  {};

  VULKAN_HPP_INLINE std::string to_string( XlibSurfaceCreateFlagBitsKHR )
  {
    return "(void)";
  }

  using XlibSurfaceCreateFlagsKHR = Flags<XlibSurfaceCreateFlagBitsKHR, VkXlibSurfaceCreateFlagsKHR>;

  VULKAN_HPP_INLINE std::string to_string( XlibSurfaceCreateFlagsKHR  )
  {
    return "{}";
  }
#endif /*VK_USE_PLATFORM_XLIB_KHR*/
} // namespace VULKAN_HPP_NAMESPACE

namespace std
{
  template <>
  struct is_error_code_enum<VULKAN_HPP_NAMESPACE::Result> : public true_type
  {};
}

namespace VULKAN_HPP_NAMESPACE
{
#ifndef VULKAN_HPP_NO_EXCEPTIONS
#if defined(_MSC_VER) && (_MSC_VER == 1800)
# define noexcept _NOEXCEPT
#endif

  class ErrorCategoryImpl : public std::error_category
  {
    public:
    virtual const char* name() const noexcept override { return VULKAN_HPP_NAMESPACE_STRING"::Result"; }
    virtual std::string message(int ev) const override { return to_string(static_cast<Result>(ev)); }
  };

  class Error
  {
    public:
    virtual ~Error() = default;

    virtual const char* what() const noexcept = 0;
  };

  class LogicError : public Error, public std::logic_error
  {
    public:
    explicit LogicError( const std::string& what )
      : Error(), std::logic_error(what) {}
    explicit LogicError( char const * what )
      : Error(), std::logic_error(what) {}
    virtual ~LogicError() = default;

    virtual const char* what() const noexcept { return std::logic_error::what(); }
  };

  class SystemError : public Error, public std::system_error
  {
    public:
    SystemError( std::error_code ec )
      : Error(), std::system_error(ec) {}
    SystemError( std::error_code ec, std::string const& what )
      : Error(), std::system_error(ec, what) {}
    SystemError( std::error_code ec, char const * what )
      : Error(), std::system_error(ec, what) {}
    SystemError( int ev, std::error_category const& ecat )
      : Error(), std::system_error(ev, ecat) {}
    SystemError( int ev, std::error_category const& ecat, std::string const& what)
      : Error(), std::system_error(ev, ecat, what) {}
    SystemError( int ev, std::error_category const& ecat, char const * what)
      : Error(), std::system_error(ev, ecat, what) {}
    virtual ~SystemError() = default;

    virtual const char* what() const noexcept { return std::system_error::what(); }
  };

#if defined(_MSC_VER) && (_MSC_VER == 1800)
# undef noexcept
#endif

  VULKAN_HPP_INLINE const std::error_category& errorCategory()
  {
    static ErrorCategoryImpl instance;
    return instance;
  }

  VULKAN_HPP_INLINE std::error_code make_error_code(Result e)
  {
    return std::error_code(static_cast<int>(e), errorCategory());
  }

  VULKAN_HPP_INLINE std::error_condition make_error_condition(Result e)
  {
    return std::error_condition(static_cast<int>(e), errorCategory());
  }

  class OutOfHostMemoryError : public SystemError
  {
  public:
    OutOfHostMemoryError( std::string const& message )
      : SystemError( make_error_code( Result::eErrorOutOfHostMemory ), message ) {}
    OutOfHostMemoryError( char const * message )
      : SystemError( make_error_code( Result::eErrorOutOfHostMemory ), message ) {}
  };

  class OutOfDeviceMemoryError : public SystemError
  {
  public:
    OutOfDeviceMemoryError( std::string const& message )
      : SystemError( make_error_code( Result::eErrorOutOfDeviceMemory ), message ) {}
    OutOfDeviceMemoryError( char const * message )
      : SystemError( make_error_code( Result::eErrorOutOfDeviceMemory ), message ) {}
  };

  class InitializationFailedError : public SystemError
  {
  public:
    InitializationFailedError( std::string const& message )
      : SystemError( make_error_code( Result::eErrorInitializationFailed ), message ) {}
    InitializationFailedError( char const * message )
      : SystemError( make_error_code( Result::eErrorInitializationFailed ), message ) {}
  };

  class DeviceLostError : public SystemError
  {
  public:
    DeviceLostError( std::string const& message )
      : SystemError( make_error_code( Result::eErrorDeviceLost ), message ) {}
    DeviceLostError( char const * message )
      : SystemError( make_error_code( Result::eErrorDeviceLost ), message ) {}
  };

  class MemoryMapFailedError : public SystemError
  {
  public:
    MemoryMapFailedError( std::string const& message )
      : SystemError( make_error_code( Result::eErrorMemoryMapFailed ), message ) {}
    MemoryMapFailedError( char const * message )
      : SystemError( make_error_code( Result::eErrorMemoryMapFailed ), message ) {}
  };

  class LayerNotPresentError : public SystemError
  {
  public:
    LayerNotPresentError( std::string const& message )
      : SystemError( make_error_code( Result::eErrorLayerNotPresent ), message ) {}
    LayerNotPresentError( char const * message )
      : SystemError( make_error_code( Result::eErrorLayerNotPresent ), message ) {}
  };

  class ExtensionNotPresentError : public SystemError
  {
  public:
    ExtensionNotPresentError( std::string const& message )
      : SystemError( make_error_code( Result::eErrorExtensionNotPresent ), message ) {}
    ExtensionNotPresentError( char const * message )
      : SystemError( make_error_code( Result::eErrorExtensionNotPresent ), message ) {}
  };

  class FeatureNotPresentError : public SystemError
  {
  public:
    FeatureNotPresentError( std::string const& message )
      : SystemError( make_error_code( Result::eErrorFeatureNotPresent ), message ) {}
    FeatureNotPresentError( char const * message )
      : SystemError( make_error_code( Result::eErrorFeatureNotPresent ), message ) {}
  };

  class IncompatibleDriverError : public SystemError
  {
  public:
    IncompatibleDriverError( std::string const& message )
      : SystemError( make_error_code( Result::eErrorIncompatibleDriver ), message ) {}
    IncompatibleDriverError( char const * message )
      : SystemError( make_error_code( Result::eErrorIncompatibleDriver ), message ) {}
  };

  class TooManyObjectsError : public SystemError
  {
  public:
    TooManyObjectsError( std::string const& message )
      : SystemError( make_error_code( Result::eErrorTooManyObjects ), message ) {}
    TooManyObjectsError( char const * message )
      : SystemError( make_error_code( Result::eErrorTooManyObjects ), message ) {}
  };

  class FormatNotSupportedError : public SystemError
  {
  public:
    FormatNotSupportedError( std::string const& message )
      : SystemError( make_error_code( Result::eErrorFormatNotSupported ), message ) {}
    FormatNotSupportedError( char const * message )
      : SystemError( make_error_code( Result::eErrorFormatNotSupported ), message ) {}
  };

  class FragmentedPoolError : public SystemError
  {
  public:
    FragmentedPoolError( std::string const& message )
      : SystemError( make_error_code( Result::eErrorFragmentedPool ), message ) {}
    FragmentedPoolError( char const * message )
      : SystemError( make_error_code( Result::eErrorFragmentedPool ), message ) {}
  };

  class OutOfPoolMemoryError : public SystemError
  {
  public:
    OutOfPoolMemoryError( std::string const& message )
      : SystemError( make_error_code( Result::eErrorOutOfPoolMemory ), message ) {}
    OutOfPoolMemoryError( char const * message )
      : SystemError( make_error_code( Result::eErrorOutOfPoolMemory ), message ) {}
  };

  class InvalidExternalHandleError : public SystemError
  {
  public:
    InvalidExternalHandleError( std::string const& message )
      : SystemError( make_error_code( Result::eErrorInvalidExternalHandle ), message ) {}
    InvalidExternalHandleError( char const * message )
      : SystemError( make_error_code( Result::eErrorInvalidExternalHandle ), message ) {}
  };

  class SurfaceLostKHRError : public SystemError
  {
  public:
    SurfaceLostKHRError( std::string const& message )
      : SystemError( make_error_code( Result::eErrorSurfaceLostKHR ), message ) {}
    SurfaceLostKHRError( char const * message )
      : SystemError( make_error_code( Result::eErrorSurfaceLostKHR ), message ) {}
  };

  class NativeWindowInUseKHRError : public SystemError
  {
  public:
    NativeWindowInUseKHRError( std::string const& message )
      : SystemError( make_error_code( Result::eErrorNativeWindowInUseKHR ), message ) {}
    NativeWindowInUseKHRError( char const * message )
      : SystemError( make_error_code( Result::eErrorNativeWindowInUseKHR ), message ) {}
  };

  class OutOfDateKHRError : public SystemError
  {
  public:
    OutOfDateKHRError( std::string const& message )
      : SystemError( make_error_code( Result::eErrorOutOfDateKHR ), message ) {}
    OutOfDateKHRError( char const * message )
      : SystemError( make_error_code( Result::eErrorOutOfDateKHR ), message ) {}
  };

  class IncompatibleDisplayKHRError : public SystemError
  {
  public:
    IncompatibleDisplayKHRError( std::string const& message )
      : SystemError( make_error_code( Result::eErrorIncompatibleDisplayKHR ), message ) {}
    IncompatibleDisplayKHRError( char const * message )
      : SystemError( make_error_code( Result::eErrorIncompatibleDisplayKHR ), message ) {}
  };

  class ValidationFailedEXTError : public SystemError
  {
  public:
    ValidationFailedEXTError( std::string const& message )
      : SystemError( make_error_code( Result::eErrorValidationFailedEXT ), message ) {}
    ValidationFailedEXTError( char const * message )
      : SystemError( make_error_code( Result::eErrorValidationFailedEXT ), message ) {}
  };

  class InvalidShaderNVError : public SystemError
  {
  public:
    InvalidShaderNVError( std::string const& message )
      : SystemError( make_error_code( Result::eErrorInvalidShaderNV ), message ) {}
    InvalidShaderNVError( char const * message )
      : SystemError( make_error_code( Result::eErrorInvalidShaderNV ), message ) {}
  };

  class InvalidDrmFormatModifierPlaneLayoutEXTError : public SystemError
  {
  public:
    InvalidDrmFormatModifierPlaneLayoutEXTError( std::string const& message )
      : SystemError( make_error_code( Result::eErrorInvalidDrmFormatModifierPlaneLayoutEXT ), message ) {}
    InvalidDrmFormatModifierPlaneLayoutEXTError( char const * message )
      : SystemError( make_error_code( Result::eErrorInvalidDrmFormatModifierPlaneLayoutEXT ), message ) {}
  };

  class FragmentationEXTError : public SystemError
  {
  public:
    FragmentationEXTError( std::string const& message )
      : SystemError( make_error_code( Result::eErrorFragmentationEXT ), message ) {}
    FragmentationEXTError( char const * message )
      : SystemError( make_error_code( Result::eErrorFragmentationEXT ), message ) {}
  };

  class NotPermittedEXTError : public SystemError
  {
  public:
    NotPermittedEXTError( std::string const& message )
      : SystemError( make_error_code( Result::eErrorNotPermittedEXT ), message ) {}
    NotPermittedEXTError( char const * message )
      : SystemError( make_error_code( Result::eErrorNotPermittedEXT ), message ) {}
  };

  class InvalidDeviceAddressEXTError : public SystemError
  {
  public:
    InvalidDeviceAddressEXTError( std::string const& message )
      : SystemError( make_error_code( Result::eErrorInvalidDeviceAddressEXT ), message ) {}
    InvalidDeviceAddressEXTError( char const * message )
      : SystemError( make_error_code( Result::eErrorInvalidDeviceAddressEXT ), message ) {}
  };


  VULKAN_HPP_INLINE void throwResultException( Result result, char const * message )
  {
    switch ( result )
    {
      case Result::eErrorOutOfHostMemory: throw OutOfHostMemoryError( message );
      case Result::eErrorOutOfDeviceMemory: throw OutOfDeviceMemoryError( message );
      case Result::eErrorInitializationFailed: throw InitializationFailedError( message );
      case Result::eErrorDeviceLost: throw DeviceLostError( message );
      case Result::eErrorMemoryMapFailed: throw MemoryMapFailedError( message );
      case Result::eErrorLayerNotPresent: throw LayerNotPresentError( message );
      case Result::eErrorExtensionNotPresent: throw ExtensionNotPresentError( message );
      case Result::eErrorFeatureNotPresent: throw FeatureNotPresentError( message );
      case Result::eErrorIncompatibleDriver: throw IncompatibleDriverError( message );
      case Result::eErrorTooManyObjects: throw TooManyObjectsError( message );
      case Result::eErrorFormatNotSupported: throw FormatNotSupportedError( message );
      case Result::eErrorFragmentedPool: throw FragmentedPoolError( message );
      case Result::eErrorOutOfPoolMemory: throw OutOfPoolMemoryError( message );
      case Result::eErrorInvalidExternalHandle: throw InvalidExternalHandleError( message );
      case Result::eErrorSurfaceLostKHR: throw SurfaceLostKHRError( message );
      case Result::eErrorNativeWindowInUseKHR: throw NativeWindowInUseKHRError( message );
      case Result::eErrorOutOfDateKHR: throw OutOfDateKHRError( message );
      case Result::eErrorIncompatibleDisplayKHR: throw IncompatibleDisplayKHRError( message );
      case Result::eErrorValidationFailedEXT: throw ValidationFailedEXTError( message );
      case Result::eErrorInvalidShaderNV: throw InvalidShaderNVError( message );
      case Result::eErrorInvalidDrmFormatModifierPlaneLayoutEXT: throw InvalidDrmFormatModifierPlaneLayoutEXTError( message );
      case Result::eErrorFragmentationEXT: throw FragmentationEXTError( message );
      case Result::eErrorNotPermittedEXT: throw NotPermittedEXTError( message );
      case Result::eErrorInvalidDeviceAddressEXT: throw InvalidDeviceAddressEXTError( message );
      default: throw SystemError( make_error_code( result ) );
    }
  }
#endif

  template <typename T> void ignore(T const&) {}

  template <typename T>
  struct ResultValue
  {
    ResultValue( Result r, T & v )
      : result( r )
      , value( v )
    {}

    ResultValue( Result r, T && v )
      : result( r )
      , value( std::move( v ) )
    {}

    Result  result;
    T       value;

    operator std::tuple<Result&, T&>() { return std::tuple<Result&, T&>(result, value); }
  };

  template <typename T>
  struct ResultValueType
  {
#ifdef VULKAN_HPP_NO_EXCEPTIONS
    typedef ResultValue<T>  type;
#else
    typedef T               type;
#endif
  };

  template <>
  struct ResultValueType<void>
  {
#ifdef VULKAN_HPP_NO_EXCEPTIONS
    typedef Result type;
#else
    typedef void   type;
#endif
  };

  VULKAN_HPP_INLINE ResultValueType<void>::type createResultValue( Result result, char const * message )
  {
#ifdef VULKAN_HPP_NO_EXCEPTIONS
    ignore(message);
    VULKAN_HPP_ASSERT( result == Result::eSuccess );
    return result;
#else
    if ( result != Result::eSuccess )
    {
      throwResultException( result, message );
    }
#endif
  }

  template <typename T>
  VULKAN_HPP_INLINE typename ResultValueType<T>::type createResultValue( Result result, T & data, char const * message )
  {
#ifdef VULKAN_HPP_NO_EXCEPTIONS
    ignore(message);
    VULKAN_HPP_ASSERT( result == Result::eSuccess );
    return ResultValue<T>( result, std::move( data ) );
#else
    if ( result != Result::eSuccess )
    {
      throwResultException( result, message );
    }
    return std::move( data );
#endif
  }

  VULKAN_HPP_INLINE Result createResultValue( Result result, char const * message, std::initializer_list<Result> successCodes )
  {
#ifdef VULKAN_HPP_NO_EXCEPTIONS
    ignore(message);
    VULKAN_HPP_ASSERT( std::find( successCodes.begin(), successCodes.end(), result ) != successCodes.end() );
#else
    if ( std::find( successCodes.begin(), successCodes.end(), result ) == successCodes.end() )
    {
      throwResultException( result, message );
    }
#endif
    return result;
  }

  template <typename T>
  VULKAN_HPP_INLINE ResultValue<T> createResultValue( Result result, T & data, char const * message, std::initializer_list<Result> successCodes )
  {
#ifdef VULKAN_HPP_NO_EXCEPTIONS
    ignore(message);
    VULKAN_HPP_ASSERT( std::find( successCodes.begin(), successCodes.end(), result ) != successCodes.end() );
#else
    if ( std::find( successCodes.begin(), successCodes.end(), result ) == successCodes.end() )
    {
      throwResultException( result, message );
    }
#endif
    return ResultValue<T>( result, data );
  }

#ifndef VULKAN_HPP_NO_SMART_HANDLE
  template <typename T, typename D>
  VULKAN_HPP_INLINE typename ResultValueType<UniqueHandle<T,D>>::type createResultValue( Result result, T & data, char const * message, typename UniqueHandleTraits<T,D>::deleter const& deleter )
  {
#ifdef VULKAN_HPP_NO_EXCEPTIONS
    ignore(message);
    VULKAN_HPP_ASSERT( result == Result::eSuccess );
    return ResultValue<UniqueHandle<T,D>>( result, UniqueHandle<T,D>(data, deleter) );
#else
    if ( result != Result::eSuccess )
    {
      throwResultException( result, message );
    }
    return UniqueHandle<T,D>(data, deleter);
#endif
  }
#endif

  struct AccelerationStructureCreateInfoNV;
  struct AccelerationStructureInfoNV;
  struct AccelerationStructureMemoryRequirementsInfoNV;
  struct AcquireNextImageInfoKHR;
  struct AllocationCallbacks;
#ifdef VK_USE_PLATFORM_ANDROID_KHR
  struct AndroidHardwareBufferPropertiesANDROID;
#endif /*VK_USE_PLATFORM_ANDROID_KHR*/
#ifdef VK_USE_PLATFORM_ANDROID_KHR
  struct AndroidSurfaceCreateInfoKHR;
#endif /*VK_USE_PLATFORM_ANDROID_KHR*/
  struct BindAccelerationStructureMemoryInfoNV;
  struct BindBufferMemoryInfo;
  using BindBufferMemoryInfoKHR = BindBufferMemoryInfo;
  struct BindImageMemoryInfo;
  using BindImageMemoryInfoKHR = BindImageMemoryInfo;
  struct BindSparseInfo;
  struct BufferCopy;
  struct BufferCreateInfo;
  struct BufferDeviceAddressInfoEXT;
  struct BufferImageCopy;
  struct BufferMemoryBarrier;
  struct BufferMemoryRequirementsInfo2;
  using BufferMemoryRequirementsInfo2KHR = BufferMemoryRequirementsInfo2;
  struct BufferViewCreateInfo;
  struct CalibratedTimestampInfoEXT;
  struct CheckpointDataNV;
  struct ClearAttachment;
  union ClearColorValue;
  struct ClearDepthStencilValue;
  struct ClearRect;
  struct CmdProcessCommandsInfoNVX;
  struct CmdReserveSpaceForCommandsInfoNVX;
  struct CoarseSampleOrderCustomNV;
  struct CommandBufferAllocateInfo;
  struct CommandBufferBeginInfo;
  struct CommandPoolCreateInfo;
  struct ComputePipelineCreateInfo;
  struct ConditionalRenderingBeginInfoEXT;
  struct CooperativeMatrixPropertiesNV;
  struct CopyDescriptorSet;
  struct DebugMarkerMarkerInfoEXT;
  struct DebugMarkerObjectNameInfoEXT;
  struct DebugMarkerObjectTagInfoEXT;
  struct DebugReportCallbackCreateInfoEXT;
  struct DebugUtilsLabelEXT;
  struct DebugUtilsMessengerCallbackDataEXT;
  struct DebugUtilsMessengerCreateInfoEXT;
  struct DebugUtilsObjectNameInfoEXT;
  struct DebugUtilsObjectTagInfoEXT;
  struct DescriptorPoolCreateInfo;
  struct DescriptorSetAllocateInfo;
  struct DescriptorSetLayoutCreateInfo;
  struct DescriptorSetLayoutSupport;
  using DescriptorSetLayoutSupportKHR = DescriptorSetLayoutSupport;
  struct DescriptorUpdateTemplateCreateInfo;
  using DescriptorUpdateTemplateCreateInfoKHR = DescriptorUpdateTemplateCreateInfo;
  struct DeviceCreateInfo;
  struct DeviceEventInfoEXT;
  struct DeviceGeneratedCommandsFeaturesNVX;
  struct DeviceGeneratedCommandsLimitsNVX;
  struct DeviceGroupPresentCapabilitiesKHR;
  struct DeviceQueueInfo2;
  struct DisplayEventInfoEXT;
  struct DisplayModeCreateInfoKHR;
  struct DisplayModeProperties2KHR;
  struct DisplayModePropertiesKHR;
  struct DisplayPlaneCapabilities2KHR;
  struct DisplayPlaneCapabilitiesKHR;
  struct DisplayPlaneInfo2KHR;
  struct DisplayPlaneProperties2KHR;
  struct DisplayPlanePropertiesKHR;
  struct DisplayPowerInfoEXT;
  struct DisplayProperties2KHR;
  struct DisplayPropertiesKHR;
  struct DisplaySurfaceCreateInfoKHR;
  struct EventCreateInfo;
  struct ExtensionProperties;
  struct Extent2D;
  struct ExternalBufferProperties;
  using ExternalBufferPropertiesKHR = ExternalBufferProperties;
  struct ExternalFenceProperties;
  using ExternalFencePropertiesKHR = ExternalFenceProperties;
  struct ExternalImageFormatPropertiesNV;
  struct ExternalSemaphoreProperties;
  using ExternalSemaphorePropertiesKHR = ExternalSemaphoreProperties;
  struct FenceCreateInfo;
  struct FenceGetFdInfoKHR;
#ifdef VK_USE_PLATFORM_WIN32_KHR
  struct FenceGetWin32HandleInfoKHR;
#endif /*VK_USE_PLATFORM_WIN32_KHR*/
  struct FormatProperties;
  struct FormatProperties2;
  using FormatProperties2KHR = FormatProperties2;
  struct FramebufferCreateInfo;
  struct GraphicsPipelineCreateInfo;
  struct HdrMetadataEXT;
#ifdef VK_USE_PLATFORM_IOS_MVK
  struct IOSSurfaceCreateInfoMVK;
#endif /*VK_USE_PLATFORM_IOS_MVK*/
  struct ImageBlit;
  struct ImageCopy;
  struct ImageCreateInfo;
  struct ImageDrmFormatModifierPropertiesEXT;
  struct ImageFormatProperties;
  struct ImageFormatProperties2;
  using ImageFormatProperties2KHR = ImageFormatProperties2;
  struct ImageMemoryBarrier;
  struct ImageMemoryRequirementsInfo2;
  using ImageMemoryRequirementsInfo2KHR = ImageMemoryRequirementsInfo2;
#ifdef VK_USE_PLATFORM_FUCHSIA
  struct ImagePipeSurfaceCreateInfoFUCHSIA;
#endif /*VK_USE_PLATFORM_FUCHSIA*/
  struct ImageResolve;
  struct ImageSparseMemoryRequirementsInfo2;
  using ImageSparseMemoryRequirementsInfo2KHR = ImageSparseMemoryRequirementsInfo2;
  struct ImageSubresource;
  struct ImageSubresourceRange;
  struct ImageViewCreateInfo;
  struct ImportFenceFdInfoKHR;
#ifdef VK_USE_PLATFORM_WIN32_KHR
  struct ImportFenceWin32HandleInfoKHR;
#endif /*VK_USE_PLATFORM_WIN32_KHR*/
  struct ImportSemaphoreFdInfoKHR;
#ifdef VK_USE_PLATFORM_WIN32_KHR
  struct ImportSemaphoreWin32HandleInfoKHR;
#endif /*VK_USE_PLATFORM_WIN32_KHR*/
  struct IndirectCommandsLayoutCreateInfoNVX;
  struct InstanceCreateInfo;
  struct LayerProperties;
#ifdef VK_USE_PLATFORM_MACOS_MVK
  struct MacOSSurfaceCreateInfoMVK;
#endif /*VK_USE_PLATFORM_MACOS_MVK*/
  struct MappedMemoryRange;
  struct MemoryAllocateInfo;
  struct MemoryBarrier;
  struct MemoryFdPropertiesKHR;
#ifdef VK_USE_PLATFORM_ANDROID_KHR
  struct MemoryGetAndroidHardwareBufferInfoANDROID;
#endif /*VK_USE_PLATFORM_ANDROID_KHR*/
  struct MemoryGetFdInfoKHR;
#ifdef VK_USE_PLATFORM_WIN32_KHR
  struct MemoryGetWin32HandleInfoKHR;
#endif /*VK_USE_PLATFORM_WIN32_KHR*/
  struct MemoryHostPointerPropertiesEXT;
  struct MemoryRequirements;
  struct MemoryRequirements2;
  using MemoryRequirements2KHR = MemoryRequirements2;
#ifdef VK_USE_PLATFORM_WIN32_KHR
  struct MemoryWin32HandlePropertiesKHR;
#endif /*VK_USE_PLATFORM_WIN32_KHR*/
  struct MultisamplePropertiesEXT;
  struct ObjectTableCreateInfoNVX;
  struct ObjectTableEntryNVX;
  struct PastPresentationTimingGOOGLE;
  struct PhysicalDeviceExternalBufferInfo;
  using PhysicalDeviceExternalBufferInfoKHR = PhysicalDeviceExternalBufferInfo;
  struct PhysicalDeviceExternalFenceInfo;
  using PhysicalDeviceExternalFenceInfoKHR = PhysicalDeviceExternalFenceInfo;
  struct PhysicalDeviceExternalSemaphoreInfo;
  using PhysicalDeviceExternalSemaphoreInfoKHR = PhysicalDeviceExternalSemaphoreInfo;
  struct PhysicalDeviceFeatures;
  struct PhysicalDeviceFeatures2;
  using PhysicalDeviceFeatures2KHR = PhysicalDeviceFeatures2;
  struct PhysicalDeviceGroupProperties;
  using PhysicalDeviceGroupPropertiesKHR = PhysicalDeviceGroupProperties;
  struct PhysicalDeviceImageFormatInfo2;
  using PhysicalDeviceImageFormatInfo2KHR = PhysicalDeviceImageFormatInfo2;
  struct PhysicalDeviceMemoryProperties;
  struct PhysicalDeviceMemoryProperties2;
  using PhysicalDeviceMemoryProperties2KHR = PhysicalDeviceMemoryProperties2;
  struct PhysicalDeviceProperties;
  struct PhysicalDeviceProperties2;
  using PhysicalDeviceProperties2KHR = PhysicalDeviceProperties2;
  struct PhysicalDeviceSparseImageFormatInfo2;
  using PhysicalDeviceSparseImageFormatInfo2KHR = PhysicalDeviceSparseImageFormatInfo2;
  struct PhysicalDeviceSurfaceInfo2KHR;
  struct PipelineCacheCreateInfo;
  struct PipelineLayoutCreateInfo;
  struct PresentInfoKHR;
  struct QueryPoolCreateInfo;
  struct QueueFamilyProperties;
  struct QueueFamilyProperties2;
  using QueueFamilyProperties2KHR = QueueFamilyProperties2;
  struct RayTracingPipelineCreateInfoNV;
  struct Rect2D;
  struct RefreshCycleDurationGOOGLE;
  struct RenderPassBeginInfo;
  struct RenderPassCreateInfo;
  struct RenderPassCreateInfo2KHR;
  struct SampleLocationsInfoEXT;
  struct SamplerCreateInfo;
  struct SamplerYcbcrConversionCreateInfo;
  using SamplerYcbcrConversionCreateInfoKHR = SamplerYcbcrConversionCreateInfo;
  struct SemaphoreCreateInfo;
  struct SemaphoreGetFdInfoKHR;
#ifdef VK_USE_PLATFORM_WIN32_KHR
  struct SemaphoreGetWin32HandleInfoKHR;
#endif /*VK_USE_PLATFORM_WIN32_KHR*/
  struct ShaderModuleCreateInfo;
  struct ShadingRatePaletteNV;
  struct SparseImageFormatProperties;
  struct SparseImageFormatProperties2;
  using SparseImageFormatProperties2KHR = SparseImageFormatProperties2;
  struct SparseImageMemoryRequirements;
  struct SparseImageMemoryRequirements2;
  using SparseImageMemoryRequirements2KHR = SparseImageMemoryRequirements2;
  struct SubmitInfo;
  struct SubpassBeginInfoKHR;
  struct SubpassEndInfoKHR;
  struct SubresourceLayout;
  struct SurfaceCapabilities2EXT;
  struct SurfaceCapabilities2KHR;
  struct SurfaceCapabilitiesKHR;
  struct SurfaceFormat2KHR;
  struct SurfaceFormatKHR;
  struct SwapchainCreateInfoKHR;
  struct ValidationCacheCreateInfoEXT;
#ifdef VK_USE_PLATFORM_VI_NN
  struct ViSurfaceCreateInfoNN;
#endif /*VK_USE_PLATFORM_VI_NN*/
  struct Viewport;
  struct ViewportWScalingNV;
#ifdef VK_USE_PLATFORM_WAYLAND_KHR
  struct WaylandSurfaceCreateInfoKHR;
#endif /*VK_USE_PLATFORM_WAYLAND_KHR*/
#ifdef VK_USE_PLATFORM_WIN32_KHR
  struct Win32SurfaceCreateInfoKHR;
#endif /*VK_USE_PLATFORM_WIN32_KHR*/
  struct WriteDescriptorSet;
#ifdef VK_USE_PLATFORM_XCB_KHR
  struct XcbSurfaceCreateInfoKHR;
#endif /*VK_USE_PLATFORM_XCB_KHR*/
#ifdef VK_USE_PLATFORM_XLIB_KHR
  struct XlibSurfaceCreateInfoKHR;
#endif /*VK_USE_PLATFORM_XLIB_KHR*/

  class SurfaceKHR
  {
  public:
    VULKAN_HPP_CONSTEXPR SurfaceKHR()
      : m_surfaceKHR(VK_NULL_HANDLE)
    {}

    VULKAN_HPP_CONSTEXPR SurfaceKHR( std::nullptr_t )
      : m_surfaceKHR(VK_NULL_HANDLE)
    {}

    VULKAN_HPP_TYPESAFE_EXPLICIT SurfaceKHR( VkSurfaceKHR surfaceKHR )
      : m_surfaceKHR( surfaceKHR )
    {}

#if defined(VULKAN_HPP_TYPESAFE_CONVERSION)
    SurfaceKHR & operator=(VkSurfaceKHR surfaceKHR)
    {
      m_surfaceKHR = surfaceKHR;
      return *this; 
    }
#endif

    SurfaceKHR & operator=( std::nullptr_t )
    {
      m_surfaceKHR = VK_NULL_HANDLE;
      return *this;
    }

    bool operator==( SurfaceKHR const & rhs ) const
    {
      return m_surfaceKHR == rhs.m_surfaceKHR;
    }

    bool operator!=(SurfaceKHR const & rhs ) const
    {
      return m_surfaceKHR != rhs.m_surfaceKHR;
    }

    bool operator<(SurfaceKHR const & rhs ) const
    {
      return m_surfaceKHR < rhs.m_surfaceKHR;
    }

    VULKAN_HPP_TYPESAFE_EXPLICIT operator VkSurfaceKHR() const
    {
      return m_surfaceKHR;
    }

    explicit operator bool() const
    {
      return m_surfaceKHR != VK_NULL_HANDLE;
    }

    bool operator!() const
    {
      return m_surfaceKHR == VK_NULL_HANDLE;
    }

  private:
    VkSurfaceKHR m_surfaceKHR;
  };
  static_assert( sizeof( SurfaceKHR ) == sizeof( VkSurfaceKHR ), "handle and wrapper have different size!" );

  class DebugReportCallbackEXT
  {
  public:
    VULKAN_HPP_CONSTEXPR DebugReportCallbackEXT()
      : m_debugReportCallbackEXT(VK_NULL_HANDLE)
    {}

    VULKAN_HPP_CONSTEXPR DebugReportCallbackEXT( std::nullptr_t )
      : m_debugReportCallbackEXT(VK_NULL_HANDLE)
    {}

    VULKAN_HPP_TYPESAFE_EXPLICIT DebugReportCallbackEXT( VkDebugReportCallbackEXT debugReportCallbackEXT )
      : m_debugReportCallbackEXT( debugReportCallbackEXT )
    {}

#if defined(VULKAN_HPP_TYPESAFE_CONVERSION)
    DebugReportCallbackEXT & operator=(VkDebugReportCallbackEXT debugReportCallbackEXT)
    {
      m_debugReportCallbackEXT = debugReportCallbackEXT;
      return *this; 
    }
#endif

    DebugReportCallbackEXT & operator=( std::nullptr_t )
    {
      m_debugReportCallbackEXT = VK_NULL_HANDLE;
      return *this;
    }

    bool operator==( DebugReportCallbackEXT const & rhs ) const
    {
      return m_debugReportCallbackEXT == rhs.m_debugReportCallbackEXT;
    }

    bool operator!=(DebugReportCallbackEXT const & rhs ) const
    {
      return m_debugReportCallbackEXT != rhs.m_debugReportCallbackEXT;
    }

    bool operator<(DebugReportCallbackEXT const & rhs ) const
    {
      return m_debugReportCallbackEXT < rhs.m_debugReportCallbackEXT;
    }

    VULKAN_HPP_TYPESAFE_EXPLICIT operator VkDebugReportCallbackEXT() const
    {
      return m_debugReportCallbackEXT;
    }

    explicit operator bool() const
    {
      return m_debugReportCallbackEXT != VK_NULL_HANDLE;
    }

    bool operator!() const
    {
      return m_debugReportCallbackEXT == VK_NULL_HANDLE;
    }

  private:
    VkDebugReportCallbackEXT m_debugReportCallbackEXT;
  };
  static_assert( sizeof( DebugReportCallbackEXT ) == sizeof( VkDebugReportCallbackEXT ), "handle and wrapper have different size!" );

  class DebugUtilsMessengerEXT
  {
  public:
    VULKAN_HPP_CONSTEXPR DebugUtilsMessengerEXT()
      : m_debugUtilsMessengerEXT(VK_NULL_HANDLE)
    {}

    VULKAN_HPP_CONSTEXPR DebugUtilsMessengerEXT( std::nullptr_t )
      : m_debugUtilsMessengerEXT(VK_NULL_HANDLE)
    {}

    VULKAN_HPP_TYPESAFE_EXPLICIT DebugUtilsMessengerEXT( VkDebugUtilsMessengerEXT debugUtilsMessengerEXT )
      : m_debugUtilsMessengerEXT( debugUtilsMessengerEXT )
    {}

#if defined(VULKAN_HPP_TYPESAFE_CONVERSION)
    DebugUtilsMessengerEXT & operator=(VkDebugUtilsMessengerEXT debugUtilsMessengerEXT)
    {
      m_debugUtilsMessengerEXT = debugUtilsMessengerEXT;
      return *this; 
    }
#endif

    DebugUtilsMessengerEXT & operator=( std::nullptr_t )
    {
      m_debugUtilsMessengerEXT = VK_NULL_HANDLE;
      return *this;
    }

    bool operator==( DebugUtilsMessengerEXT const & rhs ) const
    {
      return m_debugUtilsMessengerEXT == rhs.m_debugUtilsMessengerEXT;
    }

    bool operator!=(DebugUtilsMessengerEXT const & rhs ) const
    {
      return m_debugUtilsMessengerEXT != rhs.m_debugUtilsMessengerEXT;
    }

    bool operator<(DebugUtilsMessengerEXT const & rhs ) const
    {
      return m_debugUtilsMessengerEXT < rhs.m_debugUtilsMessengerEXT;
    }

    VULKAN_HPP_TYPESAFE_EXPLICIT operator VkDebugUtilsMessengerEXT() const
    {
      return m_debugUtilsMessengerEXT;
    }

    explicit operator bool() const
    {
      return m_debugUtilsMessengerEXT != VK_NULL_HANDLE;
    }

    bool operator!() const
    {
      return m_debugUtilsMessengerEXT == VK_NULL_HANDLE;
    }

  private:
    VkDebugUtilsMessengerEXT m_debugUtilsMessengerEXT;
  };
  static_assert( sizeof( DebugUtilsMessengerEXT ) == sizeof( VkDebugUtilsMessengerEXT ), "handle and wrapper have different size!" );

  class DisplayKHR
  {
  public:
    VULKAN_HPP_CONSTEXPR DisplayKHR()
      : m_displayKHR(VK_NULL_HANDLE)
    {}

    VULKAN_HPP_CONSTEXPR DisplayKHR( std::nullptr_t )
      : m_displayKHR(VK_NULL_HANDLE)
    {}

    VULKAN_HPP_TYPESAFE_EXPLICIT DisplayKHR( VkDisplayKHR displayKHR )
      : m_displayKHR( displayKHR )
    {}

#if defined(VULKAN_HPP_TYPESAFE_CONVERSION)
    DisplayKHR & operator=(VkDisplayKHR displayKHR)
    {
      m_displayKHR = displayKHR;
      return *this; 
    }
#endif

    DisplayKHR & operator=( std::nullptr_t )
    {
      m_displayKHR = VK_NULL_HANDLE;
      return *this;
    }

    bool operator==( DisplayKHR const & rhs ) const
    {
      return m_displayKHR == rhs.m_displayKHR;
    }

    bool operator!=(DisplayKHR const & rhs ) const
    {
      return m_displayKHR != rhs.m_displayKHR;
    }

    bool operator<(DisplayKHR const & rhs ) const
    {
      return m_displayKHR < rhs.m_displayKHR;
    }

    VULKAN_HPP_TYPESAFE_EXPLICIT operator VkDisplayKHR() const
    {
      return m_displayKHR;
    }

    explicit operator bool() const
    {
      return m_displayKHR != VK_NULL_HANDLE;
    }

    bool operator!() const
    {
      return m_displayKHR == VK_NULL_HANDLE;
    }

  private:
    VkDisplayKHR m_displayKHR;
  };
  static_assert( sizeof( DisplayKHR ) == sizeof( VkDisplayKHR ), "handle and wrapper have different size!" );

  class SwapchainKHR
  {
  public:
    VULKAN_HPP_CONSTEXPR SwapchainKHR()
      : m_swapchainKHR(VK_NULL_HANDLE)
    {}

    VULKAN_HPP_CONSTEXPR SwapchainKHR( std::nullptr_t )
      : m_swapchainKHR(VK_NULL_HANDLE)
    {}

    VULKAN_HPP_TYPESAFE_EXPLICIT SwapchainKHR( VkSwapchainKHR swapchainKHR )
      : m_swapchainKHR( swapchainKHR )
    {}

#if defined(VULKAN_HPP_TYPESAFE_CONVERSION)
    SwapchainKHR & operator=(VkSwapchainKHR swapchainKHR)
    {
      m_swapchainKHR = swapchainKHR;
      return *this; 
    }
#endif

    SwapchainKHR & operator=( std::nullptr_t )
    {
      m_swapchainKHR = VK_NULL_HANDLE;
      return *this;
    }

    bool operator==( SwapchainKHR const & rhs ) const
    {
      return m_swapchainKHR == rhs.m_swapchainKHR;
    }

    bool operator!=(SwapchainKHR const & rhs ) const
    {
      return m_swapchainKHR != rhs.m_swapchainKHR;
    }

    bool operator<(SwapchainKHR const & rhs ) const
    {
      return m_swapchainKHR < rhs.m_swapchainKHR;
    }

    VULKAN_HPP_TYPESAFE_EXPLICIT operator VkSwapchainKHR() const
    {
      return m_swapchainKHR;
    }

    explicit operator bool() const
    {
      return m_swapchainKHR != VK_NULL_HANDLE;
    }

    bool operator!() const
    {
      return m_swapchainKHR == VK_NULL_HANDLE;
    }

  private:
    VkSwapchainKHR m_swapchainKHR;
  };
  static_assert( sizeof( SwapchainKHR ) == sizeof( VkSwapchainKHR ), "handle and wrapper have different size!" );

  class Semaphore
  {
  public:
    VULKAN_HPP_CONSTEXPR Semaphore()
      : m_semaphore(VK_NULL_HANDLE)
    {}

    VULKAN_HPP_CONSTEXPR Semaphore( std::nullptr_t )
      : m_semaphore(VK_NULL_HANDLE)
    {}

    VULKAN_HPP_TYPESAFE_EXPLICIT Semaphore( VkSemaphore semaphore )
      : m_semaphore( semaphore )
    {}

#if defined(VULKAN_HPP_TYPESAFE_CONVERSION)
    Semaphore & operator=(VkSemaphore semaphore)
    {
      m_semaphore = semaphore;
      return *this; 
    }
#endif

    Semaphore & operator=( std::nullptr_t )
    {
      m_semaphore = VK_NULL_HANDLE;
      return *this;
    }

    bool operator==( Semaphore const & rhs ) const
    {
      return m_semaphore == rhs.m_semaphore;
    }

    bool operator!=(Semaphore const & rhs ) const
    {
      return m_semaphore != rhs.m_semaphore;
    }

    bool operator<(Semaphore const & rhs ) const
    {
      return m_semaphore < rhs.m_semaphore;
    }

    VULKAN_HPP_TYPESAFE_EXPLICIT operator VkSemaphore() const
    {
      return m_semaphore;
    }

    explicit operator bool() const
    {
      return m_semaphore != VK_NULL_HANDLE;
    }

    bool operator!() const
    {
      return m_semaphore == VK_NULL_HANDLE;
    }

  private:
    VkSemaphore m_semaphore;
  };
  static_assert( sizeof( Semaphore ) == sizeof( VkSemaphore ), "handle and wrapper have different size!" );

  class Fence
  {
  public:
    VULKAN_HPP_CONSTEXPR Fence()
      : m_fence(VK_NULL_HANDLE)
    {}

    VULKAN_HPP_CONSTEXPR Fence( std::nullptr_t )
      : m_fence(VK_NULL_HANDLE)
    {}

    VULKAN_HPP_TYPESAFE_EXPLICIT Fence( VkFence fence )
      : m_fence( fence )
    {}

#if defined(VULKAN_HPP_TYPESAFE_CONVERSION)
    Fence & operator=(VkFence fence)
    {
      m_fence = fence;
      return *this; 
    }
#endif

    Fence & operator=( std::nullptr_t )
    {
      m_fence = VK_NULL_HANDLE;
      return *this;
    }

    bool operator==( Fence const & rhs ) const
    {
      return m_fence == rhs.m_fence;
    }

    bool operator!=(Fence const & rhs ) const
    {
      return m_fence != rhs.m_fence;
    }

    bool operator<(Fence const & rhs ) const
    {
      return m_fence < rhs.m_fence;
    }

    VULKAN_HPP_TYPESAFE_EXPLICIT operator VkFence() const
    {
      return m_fence;
    }

    explicit operator bool() const
    {
      return m_fence != VK_NULL_HANDLE;
    }

    bool operator!() const
    {
      return m_fence == VK_NULL_HANDLE;
    }

  private:
    VkFence m_fence;
  };
  static_assert( sizeof( Fence ) == sizeof( VkFence ), "handle and wrapper have different size!" );

  class QueryPool
  {
  public:
    VULKAN_HPP_CONSTEXPR QueryPool()
      : m_queryPool(VK_NULL_HANDLE)
    {}

    VULKAN_HPP_CONSTEXPR QueryPool( std::nullptr_t )
      : m_queryPool(VK_NULL_HANDLE)
    {}

    VULKAN_HPP_TYPESAFE_EXPLICIT QueryPool( VkQueryPool queryPool )
      : m_queryPool( queryPool )
    {}

#if defined(VULKAN_HPP_TYPESAFE_CONVERSION)
    QueryPool & operator=(VkQueryPool queryPool)
    {
      m_queryPool = queryPool;
      return *this; 
    }
#endif

    QueryPool & operator=( std::nullptr_t )
    {
      m_queryPool = VK_NULL_HANDLE;
      return *this;
    }

    bool operator==( QueryPool const & rhs ) const
    {
      return m_queryPool == rhs.m_queryPool;
    }

    bool operator!=(QueryPool const & rhs ) const
    {
      return m_queryPool != rhs.m_queryPool;
    }

    bool operator<(QueryPool const & rhs ) const
    {
      return m_queryPool < rhs.m_queryPool;
    }

    VULKAN_HPP_TYPESAFE_EXPLICIT operator VkQueryPool() const
    {
      return m_queryPool;
    }

    explicit operator bool() const
    {
      return m_queryPool != VK_NULL_HANDLE;
    }

    bool operator!() const
    {
      return m_queryPool == VK_NULL_HANDLE;
    }

  private:
    VkQueryPool m_queryPool;
  };
  static_assert( sizeof( QueryPool ) == sizeof( VkQueryPool ), "handle and wrapper have different size!" );

  class Buffer
  {
  public:
    VULKAN_HPP_CONSTEXPR Buffer()
      : m_buffer(VK_NULL_HANDLE)
    {}

    VULKAN_HPP_CONSTEXPR Buffer( std::nullptr_t )
      : m_buffer(VK_NULL_HANDLE)
    {}

    VULKAN_HPP_TYPESAFE_EXPLICIT Buffer( VkBuffer buffer )
      : m_buffer( buffer )
    {}

#if defined(VULKAN_HPP_TYPESAFE_CONVERSION)
    Buffer & operator=(VkBuffer buffer)
    {
      m_buffer = buffer;
      return *this; 
    }
#endif

    Buffer & operator=( std::nullptr_t )
    {
      m_buffer = VK_NULL_HANDLE;
      return *this;
    }

    bool operator==( Buffer const & rhs ) const
    {
      return m_buffer == rhs.m_buffer;
    }

    bool operator!=(Buffer const & rhs ) const
    {
      return m_buffer != rhs.m_buffer;
    }

    bool operator<(Buffer const & rhs ) const
    {
      return m_buffer < rhs.m_buffer;
    }

    VULKAN_HPP_TYPESAFE_EXPLICIT operator VkBuffer() const
    {
      return m_buffer;
    }

    explicit operator bool() const
    {
      return m_buffer != VK_NULL_HANDLE;
    }

    bool operator!() const
    {
      return m_buffer == VK_NULL_HANDLE;
    }

  private:
    VkBuffer m_buffer;
  };
  static_assert( sizeof( Buffer ) == sizeof( VkBuffer ), "handle and wrapper have different size!" );

  class PipelineLayout
  {
  public:
    VULKAN_HPP_CONSTEXPR PipelineLayout()
      : m_pipelineLayout(VK_NULL_HANDLE)
    {}

    VULKAN_HPP_CONSTEXPR PipelineLayout( std::nullptr_t )
      : m_pipelineLayout(VK_NULL_HANDLE)
    {}

    VULKAN_HPP_TYPESAFE_EXPLICIT PipelineLayout( VkPipelineLayout pipelineLayout )
      : m_pipelineLayout( pipelineLayout )
    {}

#if defined(VULKAN_HPP_TYPESAFE_CONVERSION)
    PipelineLayout & operator=(VkPipelineLayout pipelineLayout)
    {
      m_pipelineLayout = pipelineLayout;
      return *this; 
    }
#endif

    PipelineLayout & operator=( std::nullptr_t )
    {
      m_pipelineLayout = VK_NULL_HANDLE;
      return *this;
    }

    bool operator==( PipelineLayout const & rhs ) const
    {
      return m_pipelineLayout == rhs.m_pipelineLayout;
    }

    bool operator!=(PipelineLayout const & rhs ) const
    {
      return m_pipelineLayout != rhs.m_pipelineLayout;
    }

    bool operator<(PipelineLayout const & rhs ) const
    {
      return m_pipelineLayout < rhs.m_pipelineLayout;
    }

    VULKAN_HPP_TYPESAFE_EXPLICIT operator VkPipelineLayout() const
    {
      return m_pipelineLayout;
    }

    explicit operator bool() const
    {
      return m_pipelineLayout != VK_NULL_HANDLE;
    }

    bool operator!() const
    {
      return m_pipelineLayout == VK_NULL_HANDLE;
    }

  private:
    VkPipelineLayout m_pipelineLayout;
  };
  static_assert( sizeof( PipelineLayout ) == sizeof( VkPipelineLayout ), "handle and wrapper have different size!" );

  class DescriptorSet
  {
  public:
    VULKAN_HPP_CONSTEXPR DescriptorSet()
      : m_descriptorSet(VK_NULL_HANDLE)
    {}

    VULKAN_HPP_CONSTEXPR DescriptorSet( std::nullptr_t )
      : m_descriptorSet(VK_NULL_HANDLE)
    {}

    VULKAN_HPP_TYPESAFE_EXPLICIT DescriptorSet( VkDescriptorSet descriptorSet )
      : m_descriptorSet( descriptorSet )
    {}

#if defined(VULKAN_HPP_TYPESAFE_CONVERSION)
    DescriptorSet & operator=(VkDescriptorSet descriptorSet)
    {
      m_descriptorSet = descriptorSet;
      return *this; 
    }
#endif

    DescriptorSet & operator=( std::nullptr_t )
    {
      m_descriptorSet = VK_NULL_HANDLE;
      return *this;
    }

    bool operator==( DescriptorSet const & rhs ) const
    {
      return m_descriptorSet == rhs.m_descriptorSet;
    }

    bool operator!=(DescriptorSet const & rhs ) const
    {
      return m_descriptorSet != rhs.m_descriptorSet;
    }

    bool operator<(DescriptorSet const & rhs ) const
    {
      return m_descriptorSet < rhs.m_descriptorSet;
    }

    VULKAN_HPP_TYPESAFE_EXPLICIT operator VkDescriptorSet() const
    {
      return m_descriptorSet;
    }

    explicit operator bool() const
    {
      return m_descriptorSet != VK_NULL_HANDLE;
    }

    bool operator!() const
    {
      return m_descriptorSet == VK_NULL_HANDLE;
    }

  private:
    VkDescriptorSet m_descriptorSet;
  };
  static_assert( sizeof( DescriptorSet ) == sizeof( VkDescriptorSet ), "handle and wrapper have different size!" );

  class Pipeline
  {
  public:
    VULKAN_HPP_CONSTEXPR Pipeline()
      : m_pipeline(VK_NULL_HANDLE)
    {}

    VULKAN_HPP_CONSTEXPR Pipeline( std::nullptr_t )
      : m_pipeline(VK_NULL_HANDLE)
    {}

    VULKAN_HPP_TYPESAFE_EXPLICIT Pipeline( VkPipeline pipeline )
      : m_pipeline( pipeline )
    {}

#if defined(VULKAN_HPP_TYPESAFE_CONVERSION)
    Pipeline & operator=(VkPipeline pipeline)
    {
      m_pipeline = pipeline;
      return *this; 
    }
#endif

    Pipeline & operator=( std::nullptr_t )
    {
      m_pipeline = VK_NULL_HANDLE;
      return *this;
    }

    bool operator==( Pipeline const & rhs ) const
    {
      return m_pipeline == rhs.m_pipeline;
    }

    bool operator!=(Pipeline const & rhs ) const
    {
      return m_pipeline != rhs.m_pipeline;
    }

    bool operator<(Pipeline const & rhs ) const
    {
      return m_pipeline < rhs.m_pipeline;
    }

    VULKAN_HPP_TYPESAFE_EXPLICIT operator VkPipeline() const
    {
      return m_pipeline;
    }

    explicit operator bool() const
    {
      return m_pipeline != VK_NULL_HANDLE;
    }

    bool operator!() const
    {
      return m_pipeline == VK_NULL_HANDLE;
    }

  private:
    VkPipeline m_pipeline;
  };
  static_assert( sizeof( Pipeline ) == sizeof( VkPipeline ), "handle and wrapper have different size!" );

  class ImageView
  {
  public:
    VULKAN_HPP_CONSTEXPR ImageView()
      : m_imageView(VK_NULL_HANDLE)
    {}

    VULKAN_HPP_CONSTEXPR ImageView( std::nullptr_t )
      : m_imageView(VK_NULL_HANDLE)
    {}

    VULKAN_HPP_TYPESAFE_EXPLICIT ImageView( VkImageView imageView )
      : m_imageView( imageView )
    {}

#if defined(VULKAN_HPP_TYPESAFE_CONVERSION)
    ImageView & operator=(VkImageView imageView)
    {
      m_imageView = imageView;
      return *this; 
    }
#endif

    ImageView & operator=( std::nullptr_t )
    {
      m_imageView = VK_NULL_HANDLE;
      return *this;
    }

    bool operator==( ImageView const & rhs ) const
    {
      return m_imageView == rhs.m_imageView;
    }

    bool operator!=(ImageView const & rhs ) const
    {
      return m_imageView != rhs.m_imageView;
    }

    bool operator<(ImageView const & rhs ) const
    {
      return m_imageView < rhs.m_imageView;
    }

    VULKAN_HPP_TYPESAFE_EXPLICIT operator VkImageView() const
    {
      return m_imageView;
    }

    explicit operator bool() const
    {
      return m_imageView != VK_NULL_HANDLE;
    }

    bool operator!() const
    {
      return m_imageView == VK_NULL_HANDLE;
    }

  private:
    VkImageView m_imageView;
  };
  static_assert( sizeof( ImageView ) == sizeof( VkImageView ), "handle and wrapper have different size!" );

  class Image
  {
  public:
    VULKAN_HPP_CONSTEXPR Image()
      : m_image(VK_NULL_HANDLE)
    {}

    VULKAN_HPP_CONSTEXPR Image( std::nullptr_t )
      : m_image(VK_NULL_HANDLE)
    {}

    VULKAN_HPP_TYPESAFE_EXPLICIT Image( VkImage image )
      : m_image( image )
    {}

#if defined(VULKAN_HPP_TYPESAFE_CONVERSION)
    Image & operator=(VkImage image)
    {
      m_image = image;
      return *this; 
    }
#endif

    Image & operator=( std::nullptr_t )
    {
      m_image = VK_NULL_HANDLE;
      return *this;
    }

    bool operator==( Image const & rhs ) const
    {
      return m_image == rhs.m_image;
    }

    bool operator!=(Image const & rhs ) const
    {
      return m_image != rhs.m_image;
    }

    bool operator<(Image const & rhs ) const
    {
      return m_image < rhs.m_image;
    }

    VULKAN_HPP_TYPESAFE_EXPLICIT operator VkImage() const
    {
      return m_image;
    }

    explicit operator bool() const
    {
      return m_image != VK_NULL_HANDLE;
    }

    bool operator!() const
    {
      return m_image == VK_NULL_HANDLE;
    }

  private:
    VkImage m_image;
  };
  static_assert( sizeof( Image ) == sizeof( VkImage ), "handle and wrapper have different size!" );

  class AccelerationStructureNV
  {
  public:
    VULKAN_HPP_CONSTEXPR AccelerationStructureNV()
      : m_accelerationStructureNV(VK_NULL_HANDLE)
    {}

    VULKAN_HPP_CONSTEXPR AccelerationStructureNV( std::nullptr_t )
      : m_accelerationStructureNV(VK_NULL_HANDLE)
    {}

    VULKAN_HPP_TYPESAFE_EXPLICIT AccelerationStructureNV( VkAccelerationStructureNV accelerationStructureNV )
      : m_accelerationStructureNV( accelerationStructureNV )
    {}

#if defined(VULKAN_HPP_TYPESAFE_CONVERSION)
    AccelerationStructureNV & operator=(VkAccelerationStructureNV accelerationStructureNV)
    {
      m_accelerationStructureNV = accelerationStructureNV;
      return *this; 
    }
#endif

    AccelerationStructureNV & operator=( std::nullptr_t )
    {
      m_accelerationStructureNV = VK_NULL_HANDLE;
      return *this;
    }

    bool operator==( AccelerationStructureNV const & rhs ) const
    {
      return m_accelerationStructureNV == rhs.m_accelerationStructureNV;
    }

    bool operator!=(AccelerationStructureNV const & rhs ) const
    {
      return m_accelerationStructureNV != rhs.m_accelerationStructureNV;
    }

    bool operator<(AccelerationStructureNV const & rhs ) const
    {
      return m_accelerationStructureNV < rhs.m_accelerationStructureNV;
    }

    VULKAN_HPP_TYPESAFE_EXPLICIT operator VkAccelerationStructureNV() const
    {
      return m_accelerationStructureNV;
    }

    explicit operator bool() const
    {
      return m_accelerationStructureNV != VK_NULL_HANDLE;
    }

    bool operator!() const
    {
      return m_accelerationStructureNV == VK_NULL_HANDLE;
    }

  private:
    VkAccelerationStructureNV m_accelerationStructureNV;
  };
  static_assert( sizeof( AccelerationStructureNV ) == sizeof( VkAccelerationStructureNV ), "handle and wrapper have different size!" );

  class DescriptorUpdateTemplate
  {
  public:
    VULKAN_HPP_CONSTEXPR DescriptorUpdateTemplate()
      : m_descriptorUpdateTemplate(VK_NULL_HANDLE)
    {}

    VULKAN_HPP_CONSTEXPR DescriptorUpdateTemplate( std::nullptr_t )
      : m_descriptorUpdateTemplate(VK_NULL_HANDLE)
    {}

    VULKAN_HPP_TYPESAFE_EXPLICIT DescriptorUpdateTemplate( VkDescriptorUpdateTemplate descriptorUpdateTemplate )
      : m_descriptorUpdateTemplate( descriptorUpdateTemplate )
    {}

#if defined(VULKAN_HPP_TYPESAFE_CONVERSION)
    DescriptorUpdateTemplate & operator=(VkDescriptorUpdateTemplate descriptorUpdateTemplate)
    {
      m_descriptorUpdateTemplate = descriptorUpdateTemplate;
      return *this; 
    }
#endif

    DescriptorUpdateTemplate & operator=( std::nullptr_t )
    {
      m_descriptorUpdateTemplate = VK_NULL_HANDLE;
      return *this;
    }

    bool operator==( DescriptorUpdateTemplate const & rhs ) const
    {
      return m_descriptorUpdateTemplate == rhs.m_descriptorUpdateTemplate;
    }

    bool operator!=(DescriptorUpdateTemplate const & rhs ) const
    {
      return m_descriptorUpdateTemplate != rhs.m_descriptorUpdateTemplate;
    }

    bool operator<(DescriptorUpdateTemplate const & rhs ) const
    {
      return m_descriptorUpdateTemplate < rhs.m_descriptorUpdateTemplate;
    }

    VULKAN_HPP_TYPESAFE_EXPLICIT operator VkDescriptorUpdateTemplate() const
    {
      return m_descriptorUpdateTemplate;
    }

    explicit operator bool() const
    {
      return m_descriptorUpdateTemplate != VK_NULL_HANDLE;
    }

    bool operator!() const
    {
      return m_descriptorUpdateTemplate == VK_NULL_HANDLE;
    }

  private:
    VkDescriptorUpdateTemplate m_descriptorUpdateTemplate;
  };
  static_assert( sizeof( DescriptorUpdateTemplate ) == sizeof( VkDescriptorUpdateTemplate ), "handle and wrapper have different size!" );
  using DescriptorUpdateTemplateKHR = DescriptorUpdateTemplate;

  class Event
  {
  public:
    VULKAN_HPP_CONSTEXPR Event()
      : m_event(VK_NULL_HANDLE)
    {}

    VULKAN_HPP_CONSTEXPR Event( std::nullptr_t )
      : m_event(VK_NULL_HANDLE)
    {}

    VULKAN_HPP_TYPESAFE_EXPLICIT Event( VkEvent event )
      : m_event( event )
    {}

#if defined(VULKAN_HPP_TYPESAFE_CONVERSION)
    Event & operator=(VkEvent event)
    {
      m_event = event;
      return *this; 
    }
#endif

    Event & operator=( std::nullptr_t )
    {
      m_event = VK_NULL_HANDLE;
      return *this;
    }

    bool operator==( Event const & rhs ) const
    {
      return m_event == rhs.m_event;
    }

    bool operator!=(Event const & rhs ) const
    {
      return m_event != rhs.m_event;
    }

    bool operator<(Event const & rhs ) const
    {
      return m_event < rhs.m_event;
    }

    VULKAN_HPP_TYPESAFE_EXPLICIT operator VkEvent() const
    {
      return m_event;
    }

    explicit operator bool() const
    {
      return m_event != VK_NULL_HANDLE;
    }

    bool operator!() const
    {
      return m_event == VK_NULL_HANDLE;
    }

  private:
    VkEvent m_event;
  };
  static_assert( sizeof( Event ) == sizeof( VkEvent ), "handle and wrapper have different size!" );

  class CommandBuffer
  {
  public:
    VULKAN_HPP_CONSTEXPR CommandBuffer()
      : m_commandBuffer(VK_NULL_HANDLE)
    {}

    VULKAN_HPP_CONSTEXPR CommandBuffer( std::nullptr_t )
      : m_commandBuffer(VK_NULL_HANDLE)
    {}

    VULKAN_HPP_TYPESAFE_EXPLICIT CommandBuffer( VkCommandBuffer commandBuffer )
      : m_commandBuffer( commandBuffer )
    {}

#if defined(VULKAN_HPP_TYPESAFE_CONVERSION)
    CommandBuffer & operator=(VkCommandBuffer commandBuffer)
    {
      m_commandBuffer = commandBuffer;
      return *this; 
    }
#endif

    CommandBuffer & operator=( std::nullptr_t )
    {
      m_commandBuffer = VK_NULL_HANDLE;
      return *this;
    }

    bool operator==( CommandBuffer const & rhs ) const
    {
      return m_commandBuffer == rhs.m_commandBuffer;
    }

    bool operator!=(CommandBuffer const & rhs ) const
    {
      return m_commandBuffer != rhs.m_commandBuffer;
    }

    bool operator<(CommandBuffer const & rhs ) const
    {
      return m_commandBuffer < rhs.m_commandBuffer;
    }

    template<typename Dispatch = DispatchLoaderStatic>
    Result begin( const CommandBufferBeginInfo* pBeginInfo, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    ResultValueType<void>::type begin( const CommandBufferBeginInfo & beginInfo, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    void beginConditionalRenderingEXT( const ConditionalRenderingBeginInfoEXT* pConditionalRenderingBegin, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    void beginConditionalRenderingEXT( const ConditionalRenderingBeginInfoEXT & conditionalRenderingBegin, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    void beginDebugUtilsLabelEXT( const DebugUtilsLabelEXT* pLabelInfo, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    void beginDebugUtilsLabelEXT( const DebugUtilsLabelEXT & labelInfo, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    void beginQuery( QueryPool queryPool, uint32_t query, QueryControlFlags flags, Dispatch const &d = Dispatch() ) const;

    template<typename Dispatch = DispatchLoaderStatic>
    void beginQueryIndexedEXT( QueryPool queryPool, uint32_t query, QueryControlFlags flags, uint32_t index, Dispatch const &d = Dispatch() ) const;

    template<typename Dispatch = DispatchLoaderStatic>
    void beginRenderPass( const RenderPassBeginInfo* pRenderPassBegin, SubpassContents contents, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    void beginRenderPass( const RenderPassBeginInfo & renderPassBegin, SubpassContents contents, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    void beginRenderPass2KHR( const RenderPassBeginInfo* pRenderPassBegin, const SubpassBeginInfoKHR* pSubpassBeginInfo, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    void beginRenderPass2KHR( const RenderPassBeginInfo & renderPassBegin, const SubpassBeginInfoKHR & subpassBeginInfo, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    void beginTransformFeedbackEXT( uint32_t firstCounterBuffer, uint32_t counterBufferCount, const Buffer* pCounterBuffers, const DeviceSize* pCounterBufferOffsets, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    void beginTransformFeedbackEXT( uint32_t firstCounterBuffer, ArrayProxy<const Buffer> counterBuffers, ArrayProxy<const DeviceSize> counterBufferOffsets, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    void bindDescriptorSets( PipelineBindPoint pipelineBindPoint, PipelineLayout layout, uint32_t firstSet, uint32_t descriptorSetCount, const DescriptorSet* pDescriptorSets, uint32_t dynamicOffsetCount, const uint32_t* pDynamicOffsets, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    void bindDescriptorSets( PipelineBindPoint pipelineBindPoint, PipelineLayout layout, uint32_t firstSet, ArrayProxy<const DescriptorSet> descriptorSets, ArrayProxy<const uint32_t> dynamicOffsets, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    void bindIndexBuffer( Buffer buffer, DeviceSize offset, IndexType indexType, Dispatch const &d = Dispatch() ) const;

    template<typename Dispatch = DispatchLoaderStatic>
    void bindPipeline( PipelineBindPoint pipelineBindPoint, Pipeline pipeline, Dispatch const &d = Dispatch() ) const;

    template<typename Dispatch = DispatchLoaderStatic>
    void bindShadingRateImageNV( ImageView imageView, ImageLayout imageLayout, Dispatch const &d = Dispatch() ) const;

    template<typename Dispatch = DispatchLoaderStatic>
    void bindTransformFeedbackBuffersEXT( uint32_t firstBinding, uint32_t bindingCount, const Buffer* pBuffers, const DeviceSize* pOffsets, const DeviceSize* pSizes, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    void bindTransformFeedbackBuffersEXT( uint32_t firstBinding, ArrayProxy<const Buffer> buffers, ArrayProxy<const DeviceSize> offsets, ArrayProxy<const DeviceSize> sizes, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    void bindVertexBuffers( uint32_t firstBinding, uint32_t bindingCount, const Buffer* pBuffers, const DeviceSize* pOffsets, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    void bindVertexBuffers( uint32_t firstBinding, ArrayProxy<const Buffer> buffers, ArrayProxy<const DeviceSize> offsets, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    void blitImage( Image srcImage, ImageLayout srcImageLayout, Image dstImage, ImageLayout dstImageLayout, uint32_t regionCount, const ImageBlit* pRegions, Filter filter, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    void blitImage( Image srcImage, ImageLayout srcImageLayout, Image dstImage, ImageLayout dstImageLayout, ArrayProxy<const ImageBlit> regions, Filter filter, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    void buildAccelerationStructureNV( const AccelerationStructureInfoNV* pInfo, Buffer instanceData, DeviceSize instanceOffset, Bool32 update, AccelerationStructureNV dst, AccelerationStructureNV src, Buffer scratch, DeviceSize scratchOffset, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    void buildAccelerationStructureNV( const AccelerationStructureInfoNV & info, Buffer instanceData, DeviceSize instanceOffset, Bool32 update, AccelerationStructureNV dst, AccelerationStructureNV src, Buffer scratch, DeviceSize scratchOffset, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    void clearAttachments( uint32_t attachmentCount, const ClearAttachment* pAttachments, uint32_t rectCount, const ClearRect* pRects, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    void clearAttachments( ArrayProxy<const ClearAttachment> attachments, ArrayProxy<const ClearRect> rects, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    void clearColorImage( Image image, ImageLayout imageLayout, const ClearColorValue* pColor, uint32_t rangeCount, const ImageSubresourceRange* pRanges, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    void clearColorImage( Image image, ImageLayout imageLayout, const ClearColorValue & color, ArrayProxy<const ImageSubresourceRange> ranges, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    void clearDepthStencilImage( Image image, ImageLayout imageLayout, const ClearDepthStencilValue* pDepthStencil, uint32_t rangeCount, const ImageSubresourceRange* pRanges, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    void clearDepthStencilImage( Image image, ImageLayout imageLayout, const ClearDepthStencilValue & depthStencil, ArrayProxy<const ImageSubresourceRange> ranges, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    void copyAccelerationStructureNV( AccelerationStructureNV dst, AccelerationStructureNV src, CopyAccelerationStructureModeNV mode, Dispatch const &d = Dispatch() ) const;

    template<typename Dispatch = DispatchLoaderStatic>
    void copyBuffer( Buffer srcBuffer, Buffer dstBuffer, uint32_t regionCount, const BufferCopy* pRegions, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    void copyBuffer( Buffer srcBuffer, Buffer dstBuffer, ArrayProxy<const BufferCopy> regions, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    void copyBufferToImage( Buffer srcBuffer, Image dstImage, ImageLayout dstImageLayout, uint32_t regionCount, const BufferImageCopy* pRegions, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    void copyBufferToImage( Buffer srcBuffer, Image dstImage, ImageLayout dstImageLayout, ArrayProxy<const BufferImageCopy> regions, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    void copyImage( Image srcImage, ImageLayout srcImageLayout, Image dstImage, ImageLayout dstImageLayout, uint32_t regionCount, const ImageCopy* pRegions, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    void copyImage( Image srcImage, ImageLayout srcImageLayout, Image dstImage, ImageLayout dstImageLayout, ArrayProxy<const ImageCopy> regions, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    void copyImageToBuffer( Image srcImage, ImageLayout srcImageLayout, Buffer dstBuffer, uint32_t regionCount, const BufferImageCopy* pRegions, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    void copyImageToBuffer( Image srcImage, ImageLayout srcImageLayout, Buffer dstBuffer, ArrayProxy<const BufferImageCopy> regions, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    void copyQueryPoolResults( QueryPool queryPool, uint32_t firstQuery, uint32_t queryCount, Buffer dstBuffer, DeviceSize dstOffset, DeviceSize stride, QueryResultFlags flags, Dispatch const &d = Dispatch() ) const;

    template<typename Dispatch = DispatchLoaderStatic>
    void debugMarkerBeginEXT( const DebugMarkerMarkerInfoEXT* pMarkerInfo, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    void debugMarkerBeginEXT( const DebugMarkerMarkerInfoEXT & markerInfo, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    void debugMarkerEndEXT(Dispatch const &d = Dispatch() ) const;

    template<typename Dispatch = DispatchLoaderStatic>
    void debugMarkerInsertEXT( const DebugMarkerMarkerInfoEXT* pMarkerInfo, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    void debugMarkerInsertEXT( const DebugMarkerMarkerInfoEXT & markerInfo, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    void dispatch( uint32_t groupCountX, uint32_t groupCountY, uint32_t groupCountZ, Dispatch const &d = Dispatch() ) const;

    template<typename Dispatch = DispatchLoaderStatic>
    void dispatchBase( uint32_t baseGroupX, uint32_t baseGroupY, uint32_t baseGroupZ, uint32_t groupCountX, uint32_t groupCountY, uint32_t groupCountZ, Dispatch const &d = Dispatch() ) const;

    template<typename Dispatch = DispatchLoaderStatic>
    void dispatchBaseKHR( uint32_t baseGroupX, uint32_t baseGroupY, uint32_t baseGroupZ, uint32_t groupCountX, uint32_t groupCountY, uint32_t groupCountZ, Dispatch const &d = Dispatch() ) const;

    template<typename Dispatch = DispatchLoaderStatic>
    void dispatchIndirect( Buffer buffer, DeviceSize offset, Dispatch const &d = Dispatch() ) const;

    template<typename Dispatch = DispatchLoaderStatic>
    void draw( uint32_t vertexCount, uint32_t instanceCount, uint32_t firstVertex, uint32_t firstInstance, Dispatch const &d = Dispatch() ) const;

    template<typename Dispatch = DispatchLoaderStatic>
    void drawIndexed( uint32_t indexCount, uint32_t instanceCount, uint32_t firstIndex, int32_t vertexOffset, uint32_t firstInstance, Dispatch const &d = Dispatch() ) const;

    template<typename Dispatch = DispatchLoaderStatic>
    void drawIndexedIndirect( Buffer buffer, DeviceSize offset, uint32_t drawCount, uint32_t stride, Dispatch const &d = Dispatch() ) const;

    template<typename Dispatch = DispatchLoaderStatic>
    void drawIndexedIndirectCountAMD( Buffer buffer, DeviceSize offset, Buffer countBuffer, DeviceSize countBufferOffset, uint32_t maxDrawCount, uint32_t stride, Dispatch const &d = Dispatch() ) const;

    template<typename Dispatch = DispatchLoaderStatic>
    void drawIndexedIndirectCountKHR( Buffer buffer, DeviceSize offset, Buffer countBuffer, DeviceSize countBufferOffset, uint32_t maxDrawCount, uint32_t stride, Dispatch const &d = Dispatch() ) const;

    template<typename Dispatch = DispatchLoaderStatic>
    void drawIndirect( Buffer buffer, DeviceSize offset, uint32_t drawCount, uint32_t stride, Dispatch const &d = Dispatch() ) const;

    template<typename Dispatch = DispatchLoaderStatic>
    void drawIndirectByteCountEXT( uint32_t instanceCount, uint32_t firstInstance, Buffer counterBuffer, DeviceSize counterBufferOffset, uint32_t counterOffset, uint32_t vertexStride, Dispatch const &d = Dispatch() ) const;

    template<typename Dispatch = DispatchLoaderStatic>
    void drawIndirectCountAMD( Buffer buffer, DeviceSize offset, Buffer countBuffer, DeviceSize countBufferOffset, uint32_t maxDrawCount, uint32_t stride, Dispatch const &d = Dispatch() ) const;

    template<typename Dispatch = DispatchLoaderStatic>
    void drawIndirectCountKHR( Buffer buffer, DeviceSize offset, Buffer countBuffer, DeviceSize countBufferOffset, uint32_t maxDrawCount, uint32_t stride, Dispatch const &d = Dispatch() ) const;

    template<typename Dispatch = DispatchLoaderStatic>
    void drawMeshTasksIndirectCountNV( Buffer buffer, DeviceSize offset, Buffer countBuffer, DeviceSize countBufferOffset, uint32_t maxDrawCount, uint32_t stride, Dispatch const &d = Dispatch() ) const;

    template<typename Dispatch = DispatchLoaderStatic>
    void drawMeshTasksIndirectNV( Buffer buffer, DeviceSize offset, uint32_t drawCount, uint32_t stride, Dispatch const &d = Dispatch() ) const;

    template<typename Dispatch = DispatchLoaderStatic>
    void drawMeshTasksNV( uint32_t taskCount, uint32_t firstTask, Dispatch const &d = Dispatch() ) const;

    template<typename Dispatch = DispatchLoaderStatic>
    void endConditionalRenderingEXT(Dispatch const &d = Dispatch() ) const;

    template<typename Dispatch = DispatchLoaderStatic>
    void endDebugUtilsLabelEXT(Dispatch const &d = Dispatch() ) const;

    template<typename Dispatch = DispatchLoaderStatic>
    void endQuery( QueryPool queryPool, uint32_t query, Dispatch const &d = Dispatch() ) const;

    template<typename Dispatch = DispatchLoaderStatic>
    void endQueryIndexedEXT( QueryPool queryPool, uint32_t query, uint32_t index, Dispatch const &d = Dispatch() ) const;

    template<typename Dispatch = DispatchLoaderStatic>
    void endRenderPass(Dispatch const &d = Dispatch() ) const;

    template<typename Dispatch = DispatchLoaderStatic>
    void endRenderPass2KHR( const SubpassEndInfoKHR* pSubpassEndInfo, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    void endRenderPass2KHR( const SubpassEndInfoKHR & subpassEndInfo, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    void endTransformFeedbackEXT( uint32_t firstCounterBuffer, uint32_t counterBufferCount, const Buffer* pCounterBuffers, const DeviceSize* pCounterBufferOffsets, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    void endTransformFeedbackEXT( uint32_t firstCounterBuffer, ArrayProxy<const Buffer> counterBuffers, ArrayProxy<const DeviceSize> counterBufferOffsets, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    void executeCommands( uint32_t commandBufferCount, const CommandBuffer* pCommandBuffers, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    void executeCommands( ArrayProxy<const CommandBuffer> commandBuffers, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    void fillBuffer( Buffer dstBuffer, DeviceSize dstOffset, DeviceSize size, uint32_t data, Dispatch const &d = Dispatch() ) const;

    template<typename Dispatch = DispatchLoaderStatic>
    void insertDebugUtilsLabelEXT( const DebugUtilsLabelEXT* pLabelInfo, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    void insertDebugUtilsLabelEXT( const DebugUtilsLabelEXT & labelInfo, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    void nextSubpass( SubpassContents contents, Dispatch const &d = Dispatch() ) const;

    template<typename Dispatch = DispatchLoaderStatic>
    void nextSubpass2KHR( const SubpassBeginInfoKHR* pSubpassBeginInfo, const SubpassEndInfoKHR* pSubpassEndInfo, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    void nextSubpass2KHR( const SubpassBeginInfoKHR & subpassBeginInfo, const SubpassEndInfoKHR & subpassEndInfo, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    void pipelineBarrier( PipelineStageFlags srcStageMask, PipelineStageFlags dstStageMask, DependencyFlags dependencyFlags, uint32_t memoryBarrierCount, const MemoryBarrier* pMemoryBarriers, uint32_t bufferMemoryBarrierCount, const BufferMemoryBarrier* pBufferMemoryBarriers, uint32_t imageMemoryBarrierCount, const ImageMemoryBarrier* pImageMemoryBarriers, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    void pipelineBarrier( PipelineStageFlags srcStageMask, PipelineStageFlags dstStageMask, DependencyFlags dependencyFlags, ArrayProxy<const MemoryBarrier> memoryBarriers, ArrayProxy<const BufferMemoryBarrier> bufferMemoryBarriers, ArrayProxy<const ImageMemoryBarrier> imageMemoryBarriers, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    void processCommandsNVX( const CmdProcessCommandsInfoNVX* pProcessCommandsInfo, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    void processCommandsNVX( const CmdProcessCommandsInfoNVX & processCommandsInfo, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    void pushConstants( PipelineLayout layout, ShaderStageFlags stageFlags, uint32_t offset, uint32_t size, const void* pValues, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename T, typename Dispatch = DispatchLoaderStatic>
    void pushConstants( PipelineLayout layout, ShaderStageFlags stageFlags, uint32_t offset, ArrayProxy<const T> values, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    void pushDescriptorSetKHR( PipelineBindPoint pipelineBindPoint, PipelineLayout layout, uint32_t set, uint32_t descriptorWriteCount, const WriteDescriptorSet* pDescriptorWrites, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    void pushDescriptorSetKHR( PipelineBindPoint pipelineBindPoint, PipelineLayout layout, uint32_t set, ArrayProxy<const WriteDescriptorSet> descriptorWrites, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    void pushDescriptorSetWithTemplateKHR( DescriptorUpdateTemplate descriptorUpdateTemplate, PipelineLayout layout, uint32_t set, const void* pData, Dispatch const &d = Dispatch() ) const;

    template<typename Dispatch = DispatchLoaderStatic>
    void reserveSpaceForCommandsNVX( const CmdReserveSpaceForCommandsInfoNVX* pReserveSpaceInfo, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    void reserveSpaceForCommandsNVX( const CmdReserveSpaceForCommandsInfoNVX & reserveSpaceInfo, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    void resetEvent( Event event, PipelineStageFlags stageMask, Dispatch const &d = Dispatch() ) const;

    template<typename Dispatch = DispatchLoaderStatic>
    void resetQueryPool( QueryPool queryPool, uint32_t firstQuery, uint32_t queryCount, Dispatch const &d = Dispatch() ) const;

    template<typename Dispatch = DispatchLoaderStatic>
    void resolveImage( Image srcImage, ImageLayout srcImageLayout, Image dstImage, ImageLayout dstImageLayout, uint32_t regionCount, const ImageResolve* pRegions, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    void resolveImage( Image srcImage, ImageLayout srcImageLayout, Image dstImage, ImageLayout dstImageLayout, ArrayProxy<const ImageResolve> regions, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    void setBlendConstants( const float blendConstants[4], Dispatch const &d = Dispatch() ) const;

    template<typename Dispatch = DispatchLoaderStatic>
    void setCheckpointNV( const void* pCheckpointMarker, Dispatch const &d = Dispatch() ) const;

    template<typename Dispatch = DispatchLoaderStatic>
    void setCoarseSampleOrderNV( CoarseSampleOrderTypeNV sampleOrderType, uint32_t customSampleOrderCount, const CoarseSampleOrderCustomNV* pCustomSampleOrders, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    void setCoarseSampleOrderNV( CoarseSampleOrderTypeNV sampleOrderType, ArrayProxy<const CoarseSampleOrderCustomNV> customSampleOrders, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    void setDepthBias( float depthBiasConstantFactor, float depthBiasClamp, float depthBiasSlopeFactor, Dispatch const &d = Dispatch() ) const;

    template<typename Dispatch = DispatchLoaderStatic>
    void setDepthBounds( float minDepthBounds, float maxDepthBounds, Dispatch const &d = Dispatch() ) const;

    template<typename Dispatch = DispatchLoaderStatic>
    void setDeviceMask( uint32_t deviceMask, Dispatch const &d = Dispatch() ) const;

    template<typename Dispatch = DispatchLoaderStatic>
    void setDeviceMaskKHR( uint32_t deviceMask, Dispatch const &d = Dispatch() ) const;

    template<typename Dispatch = DispatchLoaderStatic>
    void setDiscardRectangleEXT( uint32_t firstDiscardRectangle, uint32_t discardRectangleCount, const Rect2D* pDiscardRectangles, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    void setDiscardRectangleEXT( uint32_t firstDiscardRectangle, ArrayProxy<const Rect2D> discardRectangles, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    void setEvent( Event event, PipelineStageFlags stageMask, Dispatch const &d = Dispatch() ) const;

    template<typename Dispatch = DispatchLoaderStatic>
    void setExclusiveScissorNV( uint32_t firstExclusiveScissor, uint32_t exclusiveScissorCount, const Rect2D* pExclusiveScissors, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    void setExclusiveScissorNV( uint32_t firstExclusiveScissor, ArrayProxy<const Rect2D> exclusiveScissors, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    void setLineWidth( float lineWidth, Dispatch const &d = Dispatch() ) const;

    template<typename Dispatch = DispatchLoaderStatic>
    void setSampleLocationsEXT( const SampleLocationsInfoEXT* pSampleLocationsInfo, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    void setSampleLocationsEXT( const SampleLocationsInfoEXT & sampleLocationsInfo, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    void setScissor( uint32_t firstScissor, uint32_t scissorCount, const Rect2D* pScissors, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    void setScissor( uint32_t firstScissor, ArrayProxy<const Rect2D> scissors, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    void setStencilCompareMask( StencilFaceFlags faceMask, uint32_t compareMask, Dispatch const &d = Dispatch() ) const;

    template<typename Dispatch = DispatchLoaderStatic>
    void setStencilReference( StencilFaceFlags faceMask, uint32_t reference, Dispatch const &d = Dispatch() ) const;

    template<typename Dispatch = DispatchLoaderStatic>
    void setStencilWriteMask( StencilFaceFlags faceMask, uint32_t writeMask, Dispatch const &d = Dispatch() ) const;

    template<typename Dispatch = DispatchLoaderStatic>
    void setViewport( uint32_t firstViewport, uint32_t viewportCount, const Viewport* pViewports, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    void setViewport( uint32_t firstViewport, ArrayProxy<const Viewport> viewports, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    void setViewportShadingRatePaletteNV( uint32_t firstViewport, uint32_t viewportCount, const ShadingRatePaletteNV* pShadingRatePalettes, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    void setViewportShadingRatePaletteNV( uint32_t firstViewport, ArrayProxy<const ShadingRatePaletteNV> shadingRatePalettes, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    void setViewportWScalingNV( uint32_t firstViewport, uint32_t viewportCount, const ViewportWScalingNV* pViewportWScalings, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    void setViewportWScalingNV( uint32_t firstViewport, ArrayProxy<const ViewportWScalingNV> viewportWScalings, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    void traceRaysNV( Buffer raygenShaderBindingTableBuffer, DeviceSize raygenShaderBindingOffset, Buffer missShaderBindingTableBuffer, DeviceSize missShaderBindingOffset, DeviceSize missShaderBindingStride, Buffer hitShaderBindingTableBuffer, DeviceSize hitShaderBindingOffset, DeviceSize hitShaderBindingStride, Buffer callableShaderBindingTableBuffer, DeviceSize callableShaderBindingOffset, DeviceSize callableShaderBindingStride, uint32_t width, uint32_t height, uint32_t depth, Dispatch const &d = Dispatch() ) const;

    template<typename Dispatch = DispatchLoaderStatic>
    void updateBuffer( Buffer dstBuffer, DeviceSize dstOffset, DeviceSize dataSize, const void* pData, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename T, typename Dispatch = DispatchLoaderStatic>
    void updateBuffer( Buffer dstBuffer, DeviceSize dstOffset, ArrayProxy<const T> data, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    void waitEvents( uint32_t eventCount, const Event* pEvents, PipelineStageFlags srcStageMask, PipelineStageFlags dstStageMask, uint32_t memoryBarrierCount, const MemoryBarrier* pMemoryBarriers, uint32_t bufferMemoryBarrierCount, const BufferMemoryBarrier* pBufferMemoryBarriers, uint32_t imageMemoryBarrierCount, const ImageMemoryBarrier* pImageMemoryBarriers, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    void waitEvents( ArrayProxy<const Event> events, PipelineStageFlags srcStageMask, PipelineStageFlags dstStageMask, ArrayProxy<const MemoryBarrier> memoryBarriers, ArrayProxy<const BufferMemoryBarrier> bufferMemoryBarriers, ArrayProxy<const ImageMemoryBarrier> imageMemoryBarriers, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    void writeAccelerationStructuresPropertiesNV( uint32_t accelerationStructureCount, const AccelerationStructureNV* pAccelerationStructures, QueryType queryType, QueryPool queryPool, uint32_t firstQuery, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    void writeAccelerationStructuresPropertiesNV( ArrayProxy<const AccelerationStructureNV> accelerationStructures, QueryType queryType, QueryPool queryPool, uint32_t firstQuery, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    void writeBufferMarkerAMD( PipelineStageFlagBits pipelineStage, Buffer dstBuffer, DeviceSize dstOffset, uint32_t marker, Dispatch const &d = Dispatch() ) const;

    template<typename Dispatch = DispatchLoaderStatic>
    void writeTimestamp( PipelineStageFlagBits pipelineStage, QueryPool queryPool, uint32_t query, Dispatch const &d = Dispatch() ) const;

#ifdef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    Result end(Dispatch const &d = Dispatch() ) const;
#else
    template<typename Dispatch = DispatchLoaderStatic>
    ResultValueType<void>::type end(Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

#ifdef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    Result reset( CommandBufferResetFlags flags, Dispatch const &d = Dispatch() ) const;
#else
    template<typename Dispatch = DispatchLoaderStatic>
    ResultValueType<void>::type reset( CommandBufferResetFlags flags, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    VULKAN_HPP_TYPESAFE_EXPLICIT operator VkCommandBuffer() const
    {
      return m_commandBuffer;
    }

    explicit operator bool() const
    {
      return m_commandBuffer != VK_NULL_HANDLE;
    }

    bool operator!() const
    {
      return m_commandBuffer == VK_NULL_HANDLE;
    }

  private:
    VkCommandBuffer m_commandBuffer;
  };
  static_assert( sizeof( CommandBuffer ) == sizeof( VkCommandBuffer ), "handle and wrapper have different size!" );

  class DeviceMemory
  {
  public:
    VULKAN_HPP_CONSTEXPR DeviceMemory()
      : m_deviceMemory(VK_NULL_HANDLE)
    {}

    VULKAN_HPP_CONSTEXPR DeviceMemory( std::nullptr_t )
      : m_deviceMemory(VK_NULL_HANDLE)
    {}

    VULKAN_HPP_TYPESAFE_EXPLICIT DeviceMemory( VkDeviceMemory deviceMemory )
      : m_deviceMemory( deviceMemory )
    {}

#if defined(VULKAN_HPP_TYPESAFE_CONVERSION)
    DeviceMemory & operator=(VkDeviceMemory deviceMemory)
    {
      m_deviceMemory = deviceMemory;
      return *this; 
    }
#endif

    DeviceMemory & operator=( std::nullptr_t )
    {
      m_deviceMemory = VK_NULL_HANDLE;
      return *this;
    }

    bool operator==( DeviceMemory const & rhs ) const
    {
      return m_deviceMemory == rhs.m_deviceMemory;
    }

    bool operator!=(DeviceMemory const & rhs ) const
    {
      return m_deviceMemory != rhs.m_deviceMemory;
    }

    bool operator<(DeviceMemory const & rhs ) const
    {
      return m_deviceMemory < rhs.m_deviceMemory;
    }

    VULKAN_HPP_TYPESAFE_EXPLICIT operator VkDeviceMemory() const
    {
      return m_deviceMemory;
    }

    explicit operator bool() const
    {
      return m_deviceMemory != VK_NULL_HANDLE;
    }

    bool operator!() const
    {
      return m_deviceMemory == VK_NULL_HANDLE;
    }

  private:
    VkDeviceMemory m_deviceMemory;
  };
  static_assert( sizeof( DeviceMemory ) == sizeof( VkDeviceMemory ), "handle and wrapper have different size!" );

  class BufferView
  {
  public:
    VULKAN_HPP_CONSTEXPR BufferView()
      : m_bufferView(VK_NULL_HANDLE)
    {}

    VULKAN_HPP_CONSTEXPR BufferView( std::nullptr_t )
      : m_bufferView(VK_NULL_HANDLE)
    {}

    VULKAN_HPP_TYPESAFE_EXPLICIT BufferView( VkBufferView bufferView )
      : m_bufferView( bufferView )
    {}

#if defined(VULKAN_HPP_TYPESAFE_CONVERSION)
    BufferView & operator=(VkBufferView bufferView)
    {
      m_bufferView = bufferView;
      return *this; 
    }
#endif

    BufferView & operator=( std::nullptr_t )
    {
      m_bufferView = VK_NULL_HANDLE;
      return *this;
    }

    bool operator==( BufferView const & rhs ) const
    {
      return m_bufferView == rhs.m_bufferView;
    }

    bool operator!=(BufferView const & rhs ) const
    {
      return m_bufferView != rhs.m_bufferView;
    }

    bool operator<(BufferView const & rhs ) const
    {
      return m_bufferView < rhs.m_bufferView;
    }

    VULKAN_HPP_TYPESAFE_EXPLICIT operator VkBufferView() const
    {
      return m_bufferView;
    }

    explicit operator bool() const
    {
      return m_bufferView != VK_NULL_HANDLE;
    }

    bool operator!() const
    {
      return m_bufferView == VK_NULL_HANDLE;
    }

  private:
    VkBufferView m_bufferView;
  };
  static_assert( sizeof( BufferView ) == sizeof( VkBufferView ), "handle and wrapper have different size!" );

  class CommandPool
  {
  public:
    VULKAN_HPP_CONSTEXPR CommandPool()
      : m_commandPool(VK_NULL_HANDLE)
    {}

    VULKAN_HPP_CONSTEXPR CommandPool( std::nullptr_t )
      : m_commandPool(VK_NULL_HANDLE)
    {}

    VULKAN_HPP_TYPESAFE_EXPLICIT CommandPool( VkCommandPool commandPool )
      : m_commandPool( commandPool )
    {}

#if defined(VULKAN_HPP_TYPESAFE_CONVERSION)
    CommandPool & operator=(VkCommandPool commandPool)
    {
      m_commandPool = commandPool;
      return *this; 
    }
#endif

    CommandPool & operator=( std::nullptr_t )
    {
      m_commandPool = VK_NULL_HANDLE;
      return *this;
    }

    bool operator==( CommandPool const & rhs ) const
    {
      return m_commandPool == rhs.m_commandPool;
    }

    bool operator!=(CommandPool const & rhs ) const
    {
      return m_commandPool != rhs.m_commandPool;
    }

    bool operator<(CommandPool const & rhs ) const
    {
      return m_commandPool < rhs.m_commandPool;
    }

    VULKAN_HPP_TYPESAFE_EXPLICIT operator VkCommandPool() const
    {
      return m_commandPool;
    }

    explicit operator bool() const
    {
      return m_commandPool != VK_NULL_HANDLE;
    }

    bool operator!() const
    {
      return m_commandPool == VK_NULL_HANDLE;
    }

  private:
    VkCommandPool m_commandPool;
  };
  static_assert( sizeof( CommandPool ) == sizeof( VkCommandPool ), "handle and wrapper have different size!" );

  class PipelineCache
  {
  public:
    VULKAN_HPP_CONSTEXPR PipelineCache()
      : m_pipelineCache(VK_NULL_HANDLE)
    {}

    VULKAN_HPP_CONSTEXPR PipelineCache( std::nullptr_t )
      : m_pipelineCache(VK_NULL_HANDLE)
    {}

    VULKAN_HPP_TYPESAFE_EXPLICIT PipelineCache( VkPipelineCache pipelineCache )
      : m_pipelineCache( pipelineCache )
    {}

#if defined(VULKAN_HPP_TYPESAFE_CONVERSION)
    PipelineCache & operator=(VkPipelineCache pipelineCache)
    {
      m_pipelineCache = pipelineCache;
      return *this; 
    }
#endif

    PipelineCache & operator=( std::nullptr_t )
    {
      m_pipelineCache = VK_NULL_HANDLE;
      return *this;
    }

    bool operator==( PipelineCache const & rhs ) const
    {
      return m_pipelineCache == rhs.m_pipelineCache;
    }

    bool operator!=(PipelineCache const & rhs ) const
    {
      return m_pipelineCache != rhs.m_pipelineCache;
    }

    bool operator<(PipelineCache const & rhs ) const
    {
      return m_pipelineCache < rhs.m_pipelineCache;
    }

    VULKAN_HPP_TYPESAFE_EXPLICIT operator VkPipelineCache() const
    {
      return m_pipelineCache;
    }

    explicit operator bool() const
    {
      return m_pipelineCache != VK_NULL_HANDLE;
    }

    bool operator!() const
    {
      return m_pipelineCache == VK_NULL_HANDLE;
    }

  private:
    VkPipelineCache m_pipelineCache;
  };
  static_assert( sizeof( PipelineCache ) == sizeof( VkPipelineCache ), "handle and wrapper have different size!" );

  class DescriptorPool
  {
  public:
    VULKAN_HPP_CONSTEXPR DescriptorPool()
      : m_descriptorPool(VK_NULL_HANDLE)
    {}

    VULKAN_HPP_CONSTEXPR DescriptorPool( std::nullptr_t )
      : m_descriptorPool(VK_NULL_HANDLE)
    {}

    VULKAN_HPP_TYPESAFE_EXPLICIT DescriptorPool( VkDescriptorPool descriptorPool )
      : m_descriptorPool( descriptorPool )
    {}

#if defined(VULKAN_HPP_TYPESAFE_CONVERSION)
    DescriptorPool & operator=(VkDescriptorPool descriptorPool)
    {
      m_descriptorPool = descriptorPool;
      return *this; 
    }
#endif

    DescriptorPool & operator=( std::nullptr_t )
    {
      m_descriptorPool = VK_NULL_HANDLE;
      return *this;
    }

    bool operator==( DescriptorPool const & rhs ) const
    {
      return m_descriptorPool == rhs.m_descriptorPool;
    }

    bool operator!=(DescriptorPool const & rhs ) const
    {
      return m_descriptorPool != rhs.m_descriptorPool;
    }

    bool operator<(DescriptorPool const & rhs ) const
    {
      return m_descriptorPool < rhs.m_descriptorPool;
    }

    VULKAN_HPP_TYPESAFE_EXPLICIT operator VkDescriptorPool() const
    {
      return m_descriptorPool;
    }

    explicit operator bool() const
    {
      return m_descriptorPool != VK_NULL_HANDLE;
    }

    bool operator!() const
    {
      return m_descriptorPool == VK_NULL_HANDLE;
    }

  private:
    VkDescriptorPool m_descriptorPool;
  };
  static_assert( sizeof( DescriptorPool ) == sizeof( VkDescriptorPool ), "handle and wrapper have different size!" );

  class DescriptorSetLayout
  {
  public:
    VULKAN_HPP_CONSTEXPR DescriptorSetLayout()
      : m_descriptorSetLayout(VK_NULL_HANDLE)
    {}

    VULKAN_HPP_CONSTEXPR DescriptorSetLayout( std::nullptr_t )
      : m_descriptorSetLayout(VK_NULL_HANDLE)
    {}

    VULKAN_HPP_TYPESAFE_EXPLICIT DescriptorSetLayout( VkDescriptorSetLayout descriptorSetLayout )
      : m_descriptorSetLayout( descriptorSetLayout )
    {}

#if defined(VULKAN_HPP_TYPESAFE_CONVERSION)
    DescriptorSetLayout & operator=(VkDescriptorSetLayout descriptorSetLayout)
    {
      m_descriptorSetLayout = descriptorSetLayout;
      return *this; 
    }
#endif

    DescriptorSetLayout & operator=( std::nullptr_t )
    {
      m_descriptorSetLayout = VK_NULL_HANDLE;
      return *this;
    }

    bool operator==( DescriptorSetLayout const & rhs ) const
    {
      return m_descriptorSetLayout == rhs.m_descriptorSetLayout;
    }

    bool operator!=(DescriptorSetLayout const & rhs ) const
    {
      return m_descriptorSetLayout != rhs.m_descriptorSetLayout;
    }

    bool operator<(DescriptorSetLayout const & rhs ) const
    {
      return m_descriptorSetLayout < rhs.m_descriptorSetLayout;
    }

    VULKAN_HPP_TYPESAFE_EXPLICIT operator VkDescriptorSetLayout() const
    {
      return m_descriptorSetLayout;
    }

    explicit operator bool() const
    {
      return m_descriptorSetLayout != VK_NULL_HANDLE;
    }

    bool operator!() const
    {
      return m_descriptorSetLayout == VK_NULL_HANDLE;
    }

  private:
    VkDescriptorSetLayout m_descriptorSetLayout;
  };
  static_assert( sizeof( DescriptorSetLayout ) == sizeof( VkDescriptorSetLayout ), "handle and wrapper have different size!" );

  class Framebuffer
  {
  public:
    VULKAN_HPP_CONSTEXPR Framebuffer()
      : m_framebuffer(VK_NULL_HANDLE)
    {}

    VULKAN_HPP_CONSTEXPR Framebuffer( std::nullptr_t )
      : m_framebuffer(VK_NULL_HANDLE)
    {}

    VULKAN_HPP_TYPESAFE_EXPLICIT Framebuffer( VkFramebuffer framebuffer )
      : m_framebuffer( framebuffer )
    {}

#if defined(VULKAN_HPP_TYPESAFE_CONVERSION)
    Framebuffer & operator=(VkFramebuffer framebuffer)
    {
      m_framebuffer = framebuffer;
      return *this; 
    }
#endif

    Framebuffer & operator=( std::nullptr_t )
    {
      m_framebuffer = VK_NULL_HANDLE;
      return *this;
    }

    bool operator==( Framebuffer const & rhs ) const
    {
      return m_framebuffer == rhs.m_framebuffer;
    }

    bool operator!=(Framebuffer const & rhs ) const
    {
      return m_framebuffer != rhs.m_framebuffer;
    }

    bool operator<(Framebuffer const & rhs ) const
    {
      return m_framebuffer < rhs.m_framebuffer;
    }

    VULKAN_HPP_TYPESAFE_EXPLICIT operator VkFramebuffer() const
    {
      return m_framebuffer;
    }

    explicit operator bool() const
    {
      return m_framebuffer != VK_NULL_HANDLE;
    }

    bool operator!() const
    {
      return m_framebuffer == VK_NULL_HANDLE;
    }

  private:
    VkFramebuffer m_framebuffer;
  };
  static_assert( sizeof( Framebuffer ) == sizeof( VkFramebuffer ), "handle and wrapper have different size!" );

  class IndirectCommandsLayoutNVX
  {
  public:
    VULKAN_HPP_CONSTEXPR IndirectCommandsLayoutNVX()
      : m_indirectCommandsLayoutNVX(VK_NULL_HANDLE)
    {}

    VULKAN_HPP_CONSTEXPR IndirectCommandsLayoutNVX( std::nullptr_t )
      : m_indirectCommandsLayoutNVX(VK_NULL_HANDLE)
    {}

    VULKAN_HPP_TYPESAFE_EXPLICIT IndirectCommandsLayoutNVX( VkIndirectCommandsLayoutNVX indirectCommandsLayoutNVX )
      : m_indirectCommandsLayoutNVX( indirectCommandsLayoutNVX )
    {}

#if defined(VULKAN_HPP_TYPESAFE_CONVERSION)
    IndirectCommandsLayoutNVX & operator=(VkIndirectCommandsLayoutNVX indirectCommandsLayoutNVX)
    {
      m_indirectCommandsLayoutNVX = indirectCommandsLayoutNVX;
      return *this; 
    }
#endif

    IndirectCommandsLayoutNVX & operator=( std::nullptr_t )
    {
      m_indirectCommandsLayoutNVX = VK_NULL_HANDLE;
      return *this;
    }

    bool operator==( IndirectCommandsLayoutNVX const & rhs ) const
    {
      return m_indirectCommandsLayoutNVX == rhs.m_indirectCommandsLayoutNVX;
    }

    bool operator!=(IndirectCommandsLayoutNVX const & rhs ) const
    {
      return m_indirectCommandsLayoutNVX != rhs.m_indirectCommandsLayoutNVX;
    }

    bool operator<(IndirectCommandsLayoutNVX const & rhs ) const
    {
      return m_indirectCommandsLayoutNVX < rhs.m_indirectCommandsLayoutNVX;
    }

    VULKAN_HPP_TYPESAFE_EXPLICIT operator VkIndirectCommandsLayoutNVX() const
    {
      return m_indirectCommandsLayoutNVX;
    }

    explicit operator bool() const
    {
      return m_indirectCommandsLayoutNVX != VK_NULL_HANDLE;
    }

    bool operator!() const
    {
      return m_indirectCommandsLayoutNVX == VK_NULL_HANDLE;
    }

  private:
    VkIndirectCommandsLayoutNVX m_indirectCommandsLayoutNVX;
  };
  static_assert( sizeof( IndirectCommandsLayoutNVX ) == sizeof( VkIndirectCommandsLayoutNVX ), "handle and wrapper have different size!" );

  class ObjectTableNVX
  {
  public:
    VULKAN_HPP_CONSTEXPR ObjectTableNVX()
      : m_objectTableNVX(VK_NULL_HANDLE)
    {}

    VULKAN_HPP_CONSTEXPR ObjectTableNVX( std::nullptr_t )
      : m_objectTableNVX(VK_NULL_HANDLE)
    {}

    VULKAN_HPP_TYPESAFE_EXPLICIT ObjectTableNVX( VkObjectTableNVX objectTableNVX )
      : m_objectTableNVX( objectTableNVX )
    {}

#if defined(VULKAN_HPP_TYPESAFE_CONVERSION)
    ObjectTableNVX & operator=(VkObjectTableNVX objectTableNVX)
    {
      m_objectTableNVX = objectTableNVX;
      return *this; 
    }
#endif

    ObjectTableNVX & operator=( std::nullptr_t )
    {
      m_objectTableNVX = VK_NULL_HANDLE;
      return *this;
    }

    bool operator==( ObjectTableNVX const & rhs ) const
    {
      return m_objectTableNVX == rhs.m_objectTableNVX;
    }

    bool operator!=(ObjectTableNVX const & rhs ) const
    {
      return m_objectTableNVX != rhs.m_objectTableNVX;
    }

    bool operator<(ObjectTableNVX const & rhs ) const
    {
      return m_objectTableNVX < rhs.m_objectTableNVX;
    }

    VULKAN_HPP_TYPESAFE_EXPLICIT operator VkObjectTableNVX() const
    {
      return m_objectTableNVX;
    }

    explicit operator bool() const
    {
      return m_objectTableNVX != VK_NULL_HANDLE;
    }

    bool operator!() const
    {
      return m_objectTableNVX == VK_NULL_HANDLE;
    }

  private:
    VkObjectTableNVX m_objectTableNVX;
  };
  static_assert( sizeof( ObjectTableNVX ) == sizeof( VkObjectTableNVX ), "handle and wrapper have different size!" );

  class RenderPass
  {
  public:
    VULKAN_HPP_CONSTEXPR RenderPass()
      : m_renderPass(VK_NULL_HANDLE)
    {}

    VULKAN_HPP_CONSTEXPR RenderPass( std::nullptr_t )
      : m_renderPass(VK_NULL_HANDLE)
    {}

    VULKAN_HPP_TYPESAFE_EXPLICIT RenderPass( VkRenderPass renderPass )
      : m_renderPass( renderPass )
    {}

#if defined(VULKAN_HPP_TYPESAFE_CONVERSION)
    RenderPass & operator=(VkRenderPass renderPass)
    {
      m_renderPass = renderPass;
      return *this; 
    }
#endif

    RenderPass & operator=( std::nullptr_t )
    {
      m_renderPass = VK_NULL_HANDLE;
      return *this;
    }

    bool operator==( RenderPass const & rhs ) const
    {
      return m_renderPass == rhs.m_renderPass;
    }

    bool operator!=(RenderPass const & rhs ) const
    {
      return m_renderPass != rhs.m_renderPass;
    }

    bool operator<(RenderPass const & rhs ) const
    {
      return m_renderPass < rhs.m_renderPass;
    }

    VULKAN_HPP_TYPESAFE_EXPLICIT operator VkRenderPass() const
    {
      return m_renderPass;
    }

    explicit operator bool() const
    {
      return m_renderPass != VK_NULL_HANDLE;
    }

    bool operator!() const
    {
      return m_renderPass == VK_NULL_HANDLE;
    }

  private:
    VkRenderPass m_renderPass;
  };
  static_assert( sizeof( RenderPass ) == sizeof( VkRenderPass ), "handle and wrapper have different size!" );

  class Sampler
  {
  public:
    VULKAN_HPP_CONSTEXPR Sampler()
      : m_sampler(VK_NULL_HANDLE)
    {}

    VULKAN_HPP_CONSTEXPR Sampler( std::nullptr_t )
      : m_sampler(VK_NULL_HANDLE)
    {}

    VULKAN_HPP_TYPESAFE_EXPLICIT Sampler( VkSampler sampler )
      : m_sampler( sampler )
    {}

#if defined(VULKAN_HPP_TYPESAFE_CONVERSION)
    Sampler & operator=(VkSampler sampler)
    {
      m_sampler = sampler;
      return *this; 
    }
#endif

    Sampler & operator=( std::nullptr_t )
    {
      m_sampler = VK_NULL_HANDLE;
      return *this;
    }

    bool operator==( Sampler const & rhs ) const
    {
      return m_sampler == rhs.m_sampler;
    }

    bool operator!=(Sampler const & rhs ) const
    {
      return m_sampler != rhs.m_sampler;
    }

    bool operator<(Sampler const & rhs ) const
    {
      return m_sampler < rhs.m_sampler;
    }

    VULKAN_HPP_TYPESAFE_EXPLICIT operator VkSampler() const
    {
      return m_sampler;
    }

    explicit operator bool() const
    {
      return m_sampler != VK_NULL_HANDLE;
    }

    bool operator!() const
    {
      return m_sampler == VK_NULL_HANDLE;
    }

  private:
    VkSampler m_sampler;
  };
  static_assert( sizeof( Sampler ) == sizeof( VkSampler ), "handle and wrapper have different size!" );

  class SamplerYcbcrConversion
  {
  public:
    VULKAN_HPP_CONSTEXPR SamplerYcbcrConversion()
      : m_samplerYcbcrConversion(VK_NULL_HANDLE)
    {}

    VULKAN_HPP_CONSTEXPR SamplerYcbcrConversion( std::nullptr_t )
      : m_samplerYcbcrConversion(VK_NULL_HANDLE)
    {}

    VULKAN_HPP_TYPESAFE_EXPLICIT SamplerYcbcrConversion( VkSamplerYcbcrConversion samplerYcbcrConversion )
      : m_samplerYcbcrConversion( samplerYcbcrConversion )
    {}

#if defined(VULKAN_HPP_TYPESAFE_CONVERSION)
    SamplerYcbcrConversion & operator=(VkSamplerYcbcrConversion samplerYcbcrConversion)
    {
      m_samplerYcbcrConversion = samplerYcbcrConversion;
      return *this; 
    }
#endif

    SamplerYcbcrConversion & operator=( std::nullptr_t )
    {
      m_samplerYcbcrConversion = VK_NULL_HANDLE;
      return *this;
    }

    bool operator==( SamplerYcbcrConversion const & rhs ) const
    {
      return m_samplerYcbcrConversion == rhs.m_samplerYcbcrConversion;
    }

    bool operator!=(SamplerYcbcrConversion const & rhs ) const
    {
      return m_samplerYcbcrConversion != rhs.m_samplerYcbcrConversion;
    }

    bool operator<(SamplerYcbcrConversion const & rhs ) const
    {
      return m_samplerYcbcrConversion < rhs.m_samplerYcbcrConversion;
    }

    VULKAN_HPP_TYPESAFE_EXPLICIT operator VkSamplerYcbcrConversion() const
    {
      return m_samplerYcbcrConversion;
    }

    explicit operator bool() const
    {
      return m_samplerYcbcrConversion != VK_NULL_HANDLE;
    }

    bool operator!() const
    {
      return m_samplerYcbcrConversion == VK_NULL_HANDLE;
    }

  private:
    VkSamplerYcbcrConversion m_samplerYcbcrConversion;
  };
  static_assert( sizeof( SamplerYcbcrConversion ) == sizeof( VkSamplerYcbcrConversion ), "handle and wrapper have different size!" );
  using SamplerYcbcrConversionKHR = SamplerYcbcrConversion;

  class ShaderModule
  {
  public:
    VULKAN_HPP_CONSTEXPR ShaderModule()
      : m_shaderModule(VK_NULL_HANDLE)
    {}

    VULKAN_HPP_CONSTEXPR ShaderModule( std::nullptr_t )
      : m_shaderModule(VK_NULL_HANDLE)
    {}

    VULKAN_HPP_TYPESAFE_EXPLICIT ShaderModule( VkShaderModule shaderModule )
      : m_shaderModule( shaderModule )
    {}

#if defined(VULKAN_HPP_TYPESAFE_CONVERSION)
    ShaderModule & operator=(VkShaderModule shaderModule)
    {
      m_shaderModule = shaderModule;
      return *this; 
    }
#endif

    ShaderModule & operator=( std::nullptr_t )
    {
      m_shaderModule = VK_NULL_HANDLE;
      return *this;
    }

    bool operator==( ShaderModule const & rhs ) const
    {
      return m_shaderModule == rhs.m_shaderModule;
    }

    bool operator!=(ShaderModule const & rhs ) const
    {
      return m_shaderModule != rhs.m_shaderModule;
    }

    bool operator<(ShaderModule const & rhs ) const
    {
      return m_shaderModule < rhs.m_shaderModule;
    }

    VULKAN_HPP_TYPESAFE_EXPLICIT operator VkShaderModule() const
    {
      return m_shaderModule;
    }

    explicit operator bool() const
    {
      return m_shaderModule != VK_NULL_HANDLE;
    }

    bool operator!() const
    {
      return m_shaderModule == VK_NULL_HANDLE;
    }

  private:
    VkShaderModule m_shaderModule;
  };
  static_assert( sizeof( ShaderModule ) == sizeof( VkShaderModule ), "handle and wrapper have different size!" );

  class ValidationCacheEXT
  {
  public:
    VULKAN_HPP_CONSTEXPR ValidationCacheEXT()
      : m_validationCacheEXT(VK_NULL_HANDLE)
    {}

    VULKAN_HPP_CONSTEXPR ValidationCacheEXT( std::nullptr_t )
      : m_validationCacheEXT(VK_NULL_HANDLE)
    {}

    VULKAN_HPP_TYPESAFE_EXPLICIT ValidationCacheEXT( VkValidationCacheEXT validationCacheEXT )
      : m_validationCacheEXT( validationCacheEXT )
    {}

#if defined(VULKAN_HPP_TYPESAFE_CONVERSION)
    ValidationCacheEXT & operator=(VkValidationCacheEXT validationCacheEXT)
    {
      m_validationCacheEXT = validationCacheEXT;
      return *this; 
    }
#endif

    ValidationCacheEXT & operator=( std::nullptr_t )
    {
      m_validationCacheEXT = VK_NULL_HANDLE;
      return *this;
    }

    bool operator==( ValidationCacheEXT const & rhs ) const
    {
      return m_validationCacheEXT == rhs.m_validationCacheEXT;
    }

    bool operator!=(ValidationCacheEXT const & rhs ) const
    {
      return m_validationCacheEXT != rhs.m_validationCacheEXT;
    }

    bool operator<(ValidationCacheEXT const & rhs ) const
    {
      return m_validationCacheEXT < rhs.m_validationCacheEXT;
    }

    VULKAN_HPP_TYPESAFE_EXPLICIT operator VkValidationCacheEXT() const
    {
      return m_validationCacheEXT;
    }

    explicit operator bool() const
    {
      return m_validationCacheEXT != VK_NULL_HANDLE;
    }

    bool operator!() const
    {
      return m_validationCacheEXT == VK_NULL_HANDLE;
    }

  private:
    VkValidationCacheEXT m_validationCacheEXT;
  };
  static_assert( sizeof( ValidationCacheEXT ) == sizeof( VkValidationCacheEXT ), "handle and wrapper have different size!" );

  class Queue
  {
  public:
    VULKAN_HPP_CONSTEXPR Queue()
      : m_queue(VK_NULL_HANDLE)
    {}

    VULKAN_HPP_CONSTEXPR Queue( std::nullptr_t )
      : m_queue(VK_NULL_HANDLE)
    {}

    VULKAN_HPP_TYPESAFE_EXPLICIT Queue( VkQueue queue )
      : m_queue( queue )
    {}

#if defined(VULKAN_HPP_TYPESAFE_CONVERSION)
    Queue & operator=(VkQueue queue)
    {
      m_queue = queue;
      return *this; 
    }
#endif

    Queue & operator=( std::nullptr_t )
    {
      m_queue = VK_NULL_HANDLE;
      return *this;
    }

    bool operator==( Queue const & rhs ) const
    {
      return m_queue == rhs.m_queue;
    }

    bool operator!=(Queue const & rhs ) const
    {
      return m_queue != rhs.m_queue;
    }

    bool operator<(Queue const & rhs ) const
    {
      return m_queue < rhs.m_queue;
    }

    template<typename Dispatch = DispatchLoaderStatic>
    void getCheckpointDataNV( uint32_t* pCheckpointDataCount, CheckpointDataNV* pCheckpointData, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Allocator = std::allocator<CheckpointDataNV>, typename Dispatch = DispatchLoaderStatic>
    std::vector<CheckpointDataNV,Allocator> getCheckpointDataNV(Dispatch const &d = Dispatch() ) const;
    template<typename Allocator = std::allocator<CheckpointDataNV>, typename Dispatch = DispatchLoaderStatic>
    std::vector<CheckpointDataNV,Allocator> getCheckpointDataNV(Allocator const& vectorAllocator, Dispatch const &d ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    void beginDebugUtilsLabelEXT( const DebugUtilsLabelEXT* pLabelInfo, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    void beginDebugUtilsLabelEXT( const DebugUtilsLabelEXT & labelInfo, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    Result bindSparse( uint32_t bindInfoCount, const BindSparseInfo* pBindInfo, Fence fence, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    ResultValueType<void>::type bindSparse( ArrayProxy<const BindSparseInfo> bindInfo, Fence fence, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    void endDebugUtilsLabelEXT(Dispatch const &d = Dispatch() ) const;

    template<typename Dispatch = DispatchLoaderStatic>
    void insertDebugUtilsLabelEXT( const DebugUtilsLabelEXT* pLabelInfo, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    void insertDebugUtilsLabelEXT( const DebugUtilsLabelEXT & labelInfo, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    Result presentKHR( const PresentInfoKHR* pPresentInfo, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    Result presentKHR( const PresentInfoKHR & presentInfo, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    Result submit( uint32_t submitCount, const SubmitInfo* pSubmits, Fence fence, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    ResultValueType<void>::type submit( ArrayProxy<const SubmitInfo> submits, Fence fence, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

#ifdef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    Result waitIdle(Dispatch const &d = Dispatch() ) const;
#else
    template<typename Dispatch = DispatchLoaderStatic>
    ResultValueType<void>::type waitIdle(Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    VULKAN_HPP_TYPESAFE_EXPLICIT operator VkQueue() const
    {
      return m_queue;
    }

    explicit operator bool() const
    {
      return m_queue != VK_NULL_HANDLE;
    }

    bool operator!() const
    {
      return m_queue == VK_NULL_HANDLE;
    }

  private:
    VkQueue m_queue;
  };
  static_assert( sizeof( Queue ) == sizeof( VkQueue ), "handle and wrapper have different size!" );

#ifndef VULKAN_HPP_NO_SMART_HANDLE
  class Device;
  template <typename Dispatch> class UniqueHandleTraits<AccelerationStructureNV, Dispatch> { public: using deleter = ObjectDestroy<Device, Dispatch>; };
  using UniqueAccelerationStructureNV = UniqueHandle<AccelerationStructureNV, DispatchLoaderStatic>;
  template <typename Dispatch> class UniqueHandleTraits<Buffer, Dispatch> { public: using deleter = ObjectDestroy<Device, Dispatch>; };
  using UniqueBuffer = UniqueHandle<Buffer, DispatchLoaderStatic>;
  template <typename Dispatch> class UniqueHandleTraits<BufferView, Dispatch> { public: using deleter = ObjectDestroy<Device, Dispatch>; };
  using UniqueBufferView = UniqueHandle<BufferView, DispatchLoaderStatic>;
  template <typename Dispatch> class UniqueHandleTraits<CommandBuffer, Dispatch> { public: using deleter = PoolFree<Device, CommandPool, Dispatch>; };
  using UniqueCommandBuffer = UniqueHandle<CommandBuffer, DispatchLoaderStatic>;
  template <typename Dispatch> class UniqueHandleTraits<CommandPool, Dispatch> { public: using deleter = ObjectDestroy<Device, Dispatch>; };
  using UniqueCommandPool = UniqueHandle<CommandPool, DispatchLoaderStatic>;
  template <typename Dispatch> class UniqueHandleTraits<DescriptorPool, Dispatch> { public: using deleter = ObjectDestroy<Device, Dispatch>; };
  using UniqueDescriptorPool = UniqueHandle<DescriptorPool, DispatchLoaderStatic>;
  template <typename Dispatch> class UniqueHandleTraits<DescriptorSet, Dispatch> { public: using deleter = PoolFree<Device, DescriptorPool, Dispatch>; };
  using UniqueDescriptorSet = UniqueHandle<DescriptorSet, DispatchLoaderStatic>;
  template <typename Dispatch> class UniqueHandleTraits<DescriptorSetLayout, Dispatch> { public: using deleter = ObjectDestroy<Device, Dispatch>; };
  using UniqueDescriptorSetLayout = UniqueHandle<DescriptorSetLayout, DispatchLoaderStatic>;
  template <typename Dispatch> class UniqueHandleTraits<DescriptorUpdateTemplate, Dispatch> { public: using deleter = ObjectDestroy<Device, Dispatch>; };
  using UniqueDescriptorUpdateTemplate = UniqueHandle<DescriptorUpdateTemplate, DispatchLoaderStatic>;
  template <typename Dispatch> class UniqueHandleTraits<DeviceMemory, Dispatch> { public: using deleter = ObjectFree<Device, Dispatch>; };
  using UniqueDeviceMemory = UniqueHandle<DeviceMemory, DispatchLoaderStatic>;
  template <typename Dispatch> class UniqueHandleTraits<Event, Dispatch> { public: using deleter = ObjectDestroy<Device, Dispatch>; };
  using UniqueEvent = UniqueHandle<Event, DispatchLoaderStatic>;
  template <typename Dispatch> class UniqueHandleTraits<Fence, Dispatch> { public: using deleter = ObjectDestroy<Device, Dispatch>; };
  using UniqueFence = UniqueHandle<Fence, DispatchLoaderStatic>;
  template <typename Dispatch> class UniqueHandleTraits<Framebuffer, Dispatch> { public: using deleter = ObjectDestroy<Device, Dispatch>; };
  using UniqueFramebuffer = UniqueHandle<Framebuffer, DispatchLoaderStatic>;
  template <typename Dispatch> class UniqueHandleTraits<Image, Dispatch> { public: using deleter = ObjectDestroy<Device, Dispatch>; };
  using UniqueImage = UniqueHandle<Image, DispatchLoaderStatic>;
  template <typename Dispatch> class UniqueHandleTraits<ImageView, Dispatch> { public: using deleter = ObjectDestroy<Device, Dispatch>; };
  using UniqueImageView = UniqueHandle<ImageView, DispatchLoaderStatic>;
  template <typename Dispatch> class UniqueHandleTraits<IndirectCommandsLayoutNVX, Dispatch> { public: using deleter = ObjectDestroy<Device, Dispatch>; };
  using UniqueIndirectCommandsLayoutNVX = UniqueHandle<IndirectCommandsLayoutNVX, DispatchLoaderStatic>;
  template <typename Dispatch> class UniqueHandleTraits<ObjectTableNVX, Dispatch> { public: using deleter = ObjectDestroy<Device, Dispatch>; };
  using UniqueObjectTableNVX = UniqueHandle<ObjectTableNVX, DispatchLoaderStatic>;
  template <typename Dispatch> class UniqueHandleTraits<Pipeline, Dispatch> { public: using deleter = ObjectDestroy<Device, Dispatch>; };
  using UniquePipeline = UniqueHandle<Pipeline, DispatchLoaderStatic>;
  template <typename Dispatch> class UniqueHandleTraits<PipelineCache, Dispatch> { public: using deleter = ObjectDestroy<Device, Dispatch>; };
  using UniquePipelineCache = UniqueHandle<PipelineCache, DispatchLoaderStatic>;
  template <typename Dispatch> class UniqueHandleTraits<PipelineLayout, Dispatch> { public: using deleter = ObjectDestroy<Device, Dispatch>; };
  using UniquePipelineLayout = UniqueHandle<PipelineLayout, DispatchLoaderStatic>;
  template <typename Dispatch> class UniqueHandleTraits<QueryPool, Dispatch> { public: using deleter = ObjectDestroy<Device, Dispatch>; };
  using UniqueQueryPool = UniqueHandle<QueryPool, DispatchLoaderStatic>;
  template <typename Dispatch> class UniqueHandleTraits<RenderPass, Dispatch> { public: using deleter = ObjectDestroy<Device, Dispatch>; };
  using UniqueRenderPass = UniqueHandle<RenderPass, DispatchLoaderStatic>;
  template <typename Dispatch> class UniqueHandleTraits<Sampler, Dispatch> { public: using deleter = ObjectDestroy<Device, Dispatch>; };
  using UniqueSampler = UniqueHandle<Sampler, DispatchLoaderStatic>;
  template <typename Dispatch> class UniqueHandleTraits<SamplerYcbcrConversion, Dispatch> { public: using deleter = ObjectDestroy<Device, Dispatch>; };
  using UniqueSamplerYcbcrConversion = UniqueHandle<SamplerYcbcrConversion, DispatchLoaderStatic>;
  template <typename Dispatch> class UniqueHandleTraits<Semaphore, Dispatch> { public: using deleter = ObjectDestroy<Device, Dispatch>; };
  using UniqueSemaphore = UniqueHandle<Semaphore, DispatchLoaderStatic>;
  template <typename Dispatch> class UniqueHandleTraits<ShaderModule, Dispatch> { public: using deleter = ObjectDestroy<Device, Dispatch>; };
  using UniqueShaderModule = UniqueHandle<ShaderModule, DispatchLoaderStatic>;
  template <typename Dispatch> class UniqueHandleTraits<SwapchainKHR, Dispatch> { public: using deleter = ObjectDestroy<Device, Dispatch>; };
  using UniqueSwapchainKHR = UniqueHandle<SwapchainKHR, DispatchLoaderStatic>;
  template <typename Dispatch> class UniqueHandleTraits<ValidationCacheEXT, Dispatch> { public: using deleter = ObjectDestroy<Device, Dispatch>; };
  using UniqueValidationCacheEXT = UniqueHandle<ValidationCacheEXT, DispatchLoaderStatic>;
#endif /*VULKAN_HPP_NO_SMART_HANDLE*/

  class Device
  {
  public:
    VULKAN_HPP_CONSTEXPR Device()
      : m_device(VK_NULL_HANDLE)
    {}

    VULKAN_HPP_CONSTEXPR Device( std::nullptr_t )
      : m_device(VK_NULL_HANDLE)
    {}

    VULKAN_HPP_TYPESAFE_EXPLICIT Device( VkDevice device )
      : m_device( device )
    {}

#if defined(VULKAN_HPP_TYPESAFE_CONVERSION)
    Device & operator=(VkDevice device)
    {
      m_device = device;
      return *this; 
    }
#endif

    Device & operator=( std::nullptr_t )
    {
      m_device = VK_NULL_HANDLE;
      return *this;
    }

    bool operator==( Device const & rhs ) const
    {
      return m_device == rhs.m_device;
    }

    bool operator!=(Device const & rhs ) const
    {
      return m_device != rhs.m_device;
    }

    bool operator<(Device const & rhs ) const
    {
      return m_device < rhs.m_device;
    }

    template<typename Dispatch = DispatchLoaderStatic>
    Result acquireNextImage2KHR( const AcquireNextImageInfoKHR* pAcquireInfo, uint32_t* pImageIndex, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    ResultValue<uint32_t> acquireNextImage2KHR( const AcquireNextImageInfoKHR & acquireInfo, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    Result acquireNextImageKHR( SwapchainKHR swapchain, uint64_t timeout, Semaphore semaphore, Fence fence, uint32_t* pImageIndex, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    ResultValue<uint32_t> acquireNextImageKHR( SwapchainKHR swapchain, uint64_t timeout, Semaphore semaphore, Fence fence, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    Result allocateCommandBuffers( const CommandBufferAllocateInfo* pAllocateInfo, CommandBuffer* pCommandBuffers, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Allocator = std::allocator<CommandBuffer>, typename Dispatch = DispatchLoaderStatic>
    typename ResultValueType<std::vector<CommandBuffer,Allocator>>::type allocateCommandBuffers( const CommandBufferAllocateInfo & allocateInfo, Dispatch const &d = Dispatch() ) const;
    template<typename Allocator = std::allocator<CommandBuffer>, typename Dispatch = DispatchLoaderStatic>
    typename ResultValueType<std::vector<CommandBuffer,Allocator>>::type allocateCommandBuffers( const CommandBufferAllocateInfo & allocateInfo, Allocator const& vectorAllocator, Dispatch const &d ) const;
#ifndef VULKAN_HPP_NO_SMART_HANDLE
    template<typename Allocator = std::allocator<UniqueCommandBuffer>, typename Dispatch = DispatchLoaderStatic>
    typename ResultValueType<std::vector<UniqueHandle<CommandBuffer,Dispatch>,Allocator>>::type allocateCommandBuffersUnique( const CommandBufferAllocateInfo & allocateInfo, Dispatch const &d = Dispatch() ) const;
    template<typename Allocator = std::allocator<UniqueCommandBuffer>, typename Dispatch = DispatchLoaderStatic>
    typename ResultValueType<std::vector<UniqueHandle<CommandBuffer,Dispatch>,Allocator>>::type allocateCommandBuffersUnique( const CommandBufferAllocateInfo & allocateInfo, Allocator const& vectorAllocator, Dispatch const &d ) const;
#endif /*VULKAN_HPP_NO_SMART_HANDLE*/
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    Result allocateDescriptorSets( const DescriptorSetAllocateInfo* pAllocateInfo, DescriptorSet* pDescriptorSets, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Allocator = std::allocator<DescriptorSet>, typename Dispatch = DispatchLoaderStatic>
    typename ResultValueType<std::vector<DescriptorSet,Allocator>>::type allocateDescriptorSets( const DescriptorSetAllocateInfo & allocateInfo, Dispatch const &d = Dispatch() ) const;
    template<typename Allocator = std::allocator<DescriptorSet>, typename Dispatch = DispatchLoaderStatic>
    typename ResultValueType<std::vector<DescriptorSet,Allocator>>::type allocateDescriptorSets( const DescriptorSetAllocateInfo & allocateInfo, Allocator const& vectorAllocator, Dispatch const &d ) const;
#ifndef VULKAN_HPP_NO_SMART_HANDLE
    template<typename Allocator = std::allocator<UniqueDescriptorSet>, typename Dispatch = DispatchLoaderStatic>
    typename ResultValueType<std::vector<UniqueHandle<DescriptorSet,Dispatch>,Allocator>>::type allocateDescriptorSetsUnique( const DescriptorSetAllocateInfo & allocateInfo, Dispatch const &d = Dispatch() ) const;
    template<typename Allocator = std::allocator<UniqueDescriptorSet>, typename Dispatch = DispatchLoaderStatic>
    typename ResultValueType<std::vector<UniqueHandle<DescriptorSet,Dispatch>,Allocator>>::type allocateDescriptorSetsUnique( const DescriptorSetAllocateInfo & allocateInfo, Allocator const& vectorAllocator, Dispatch const &d ) const;
#endif /*VULKAN_HPP_NO_SMART_HANDLE*/
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    Result allocateMemory( const MemoryAllocateInfo* pAllocateInfo, const AllocationCallbacks* pAllocator, DeviceMemory* pMemory, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    ResultValueType<DeviceMemory>::type allocateMemory( const MemoryAllocateInfo & allocateInfo, Optional<const AllocationCallbacks> allocator = nullptr, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_NO_SMART_HANDLE
    template<typename Dispatch = DispatchLoaderStatic>
    typename ResultValueType<UniqueHandle<DeviceMemory,Dispatch>>::type allocateMemoryUnique( const MemoryAllocateInfo & allocateInfo, Optional<const AllocationCallbacks> allocator = nullptr, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_NO_SMART_HANDLE*/
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    Result bindAccelerationStructureMemoryNV( uint32_t bindInfoCount, const BindAccelerationStructureMemoryInfoNV* pBindInfos, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    ResultValueType<void>::type bindAccelerationStructureMemoryNV( ArrayProxy<const BindAccelerationStructureMemoryInfoNV> bindInfos, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

#ifdef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    Result bindBufferMemory( Buffer buffer, DeviceMemory memory, DeviceSize memoryOffset, Dispatch const &d = Dispatch() ) const;
#else
    template<typename Dispatch = DispatchLoaderStatic>
    ResultValueType<void>::type bindBufferMemory( Buffer buffer, DeviceMemory memory, DeviceSize memoryOffset, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    Result bindBufferMemory2( uint32_t bindInfoCount, const BindBufferMemoryInfo* pBindInfos, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    ResultValueType<void>::type bindBufferMemory2( ArrayProxy<const BindBufferMemoryInfo> bindInfos, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    Result bindBufferMemory2KHR( uint32_t bindInfoCount, const BindBufferMemoryInfo* pBindInfos, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    ResultValueType<void>::type bindBufferMemory2KHR( ArrayProxy<const BindBufferMemoryInfo> bindInfos, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

#ifdef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    Result bindImageMemory( Image image, DeviceMemory memory, DeviceSize memoryOffset, Dispatch const &d = Dispatch() ) const;
#else
    template<typename Dispatch = DispatchLoaderStatic>
    ResultValueType<void>::type bindImageMemory( Image image, DeviceMemory memory, DeviceSize memoryOffset, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    Result bindImageMemory2( uint32_t bindInfoCount, const BindImageMemoryInfo* pBindInfos, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    ResultValueType<void>::type bindImageMemory2( ArrayProxy<const BindImageMemoryInfo> bindInfos, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    Result bindImageMemory2KHR( uint32_t bindInfoCount, const BindImageMemoryInfo* pBindInfos, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    ResultValueType<void>::type bindImageMemory2KHR( ArrayProxy<const BindImageMemoryInfo> bindInfos, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

#ifdef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    Result compileDeferredNV( Pipeline pipeline, uint32_t shader, Dispatch const &d = Dispatch() ) const;
#else
    template<typename Dispatch = DispatchLoaderStatic>
    ResultValueType<void>::type compileDeferredNV( Pipeline pipeline, uint32_t shader, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    Result createAccelerationStructureNV( const AccelerationStructureCreateInfoNV* pCreateInfo, const AllocationCallbacks* pAllocator, AccelerationStructureNV* pAccelerationStructure, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    ResultValueType<AccelerationStructureNV>::type createAccelerationStructureNV( const AccelerationStructureCreateInfoNV & createInfo, Optional<const AllocationCallbacks> allocator = nullptr, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_NO_SMART_HANDLE
    template<typename Dispatch = DispatchLoaderStatic>
    typename ResultValueType<UniqueHandle<AccelerationStructureNV,Dispatch>>::type createAccelerationStructureNVUnique( const AccelerationStructureCreateInfoNV & createInfo, Optional<const AllocationCallbacks> allocator = nullptr, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_NO_SMART_HANDLE*/
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    Result createBuffer( const BufferCreateInfo* pCreateInfo, const AllocationCallbacks* pAllocator, Buffer* pBuffer, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    ResultValueType<Buffer>::type createBuffer( const BufferCreateInfo & createInfo, Optional<const AllocationCallbacks> allocator = nullptr, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_NO_SMART_HANDLE
    template<typename Dispatch = DispatchLoaderStatic>
    typename ResultValueType<UniqueHandle<Buffer,Dispatch>>::type createBufferUnique( const BufferCreateInfo & createInfo, Optional<const AllocationCallbacks> allocator = nullptr, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_NO_SMART_HANDLE*/
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    Result createBufferView( const BufferViewCreateInfo* pCreateInfo, const AllocationCallbacks* pAllocator, BufferView* pView, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    ResultValueType<BufferView>::type createBufferView( const BufferViewCreateInfo & createInfo, Optional<const AllocationCallbacks> allocator = nullptr, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_NO_SMART_HANDLE
    template<typename Dispatch = DispatchLoaderStatic>
    typename ResultValueType<UniqueHandle<BufferView,Dispatch>>::type createBufferViewUnique( const BufferViewCreateInfo & createInfo, Optional<const AllocationCallbacks> allocator = nullptr, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_NO_SMART_HANDLE*/
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    Result createCommandPool( const CommandPoolCreateInfo* pCreateInfo, const AllocationCallbacks* pAllocator, CommandPool* pCommandPool, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    ResultValueType<CommandPool>::type createCommandPool( const CommandPoolCreateInfo & createInfo, Optional<const AllocationCallbacks> allocator = nullptr, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_NO_SMART_HANDLE
    template<typename Dispatch = DispatchLoaderStatic>
    typename ResultValueType<UniqueHandle<CommandPool,Dispatch>>::type createCommandPoolUnique( const CommandPoolCreateInfo & createInfo, Optional<const AllocationCallbacks> allocator = nullptr, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_NO_SMART_HANDLE*/
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    Result createComputePipelines( PipelineCache pipelineCache, uint32_t createInfoCount, const ComputePipelineCreateInfo* pCreateInfos, const AllocationCallbacks* pAllocator, Pipeline* pPipelines, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Allocator = std::allocator<Pipeline>, typename Dispatch = DispatchLoaderStatic>
    typename ResultValueType<std::vector<Pipeline,Allocator>>::type createComputePipelines( PipelineCache pipelineCache, ArrayProxy<const ComputePipelineCreateInfo> createInfos, Optional<const AllocationCallbacks> allocator = nullptr, Dispatch const &d = Dispatch() ) const;
    template<typename Allocator = std::allocator<Pipeline>, typename Dispatch = DispatchLoaderStatic>
    typename ResultValueType<std::vector<Pipeline,Allocator>>::type createComputePipelines( PipelineCache pipelineCache, ArrayProxy<const ComputePipelineCreateInfo> createInfos, Optional<const AllocationCallbacks> allocator, Allocator const& vectorAllocator, Dispatch const &d ) const;
    template<typename Dispatch = DispatchLoaderStatic>
    ResultValueType<Pipeline>::type createComputePipeline( PipelineCache pipelineCache, const ComputePipelineCreateInfo & createInfo, Optional<const AllocationCallbacks> allocator = nullptr, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_NO_SMART_HANDLE
    template<typename Allocator = std::allocator<UniquePipeline>, typename Dispatch = DispatchLoaderStatic>
    typename ResultValueType<std::vector<UniqueHandle<Pipeline,Dispatch>,Allocator>>::type createComputePipelinesUnique( PipelineCache pipelineCache, ArrayProxy<const ComputePipelineCreateInfo> createInfos, Optional<const AllocationCallbacks> allocator = nullptr, Dispatch const &d = Dispatch() ) const;
    template<typename Allocator = std::allocator<UniquePipeline>, typename Dispatch = DispatchLoaderStatic>
    typename ResultValueType<std::vector<UniqueHandle<Pipeline,Dispatch>,Allocator>>::type createComputePipelinesUnique( PipelineCache pipelineCache, ArrayProxy<const ComputePipelineCreateInfo> createInfos, Optional<const AllocationCallbacks> allocator, Allocator const& vectorAllocator, Dispatch const &d ) const;
    template<typename Dispatch = DispatchLoaderStatic>
    typename ResultValueType<UniqueHandle<Pipeline,Dispatch>>::type createComputePipelineUnique( PipelineCache pipelineCache, const ComputePipelineCreateInfo & createInfo, Optional<const AllocationCallbacks> allocator = nullptr, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_NO_SMART_HANDLE*/
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    Result createDescriptorPool( const DescriptorPoolCreateInfo* pCreateInfo, const AllocationCallbacks* pAllocator, DescriptorPool* pDescriptorPool, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    ResultValueType<DescriptorPool>::type createDescriptorPool( const DescriptorPoolCreateInfo & createInfo, Optional<const AllocationCallbacks> allocator = nullptr, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_NO_SMART_HANDLE
    template<typename Dispatch = DispatchLoaderStatic>
    typename ResultValueType<UniqueHandle<DescriptorPool,Dispatch>>::type createDescriptorPoolUnique( const DescriptorPoolCreateInfo & createInfo, Optional<const AllocationCallbacks> allocator = nullptr, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_NO_SMART_HANDLE*/
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    Result createDescriptorSetLayout( const DescriptorSetLayoutCreateInfo* pCreateInfo, const AllocationCallbacks* pAllocator, DescriptorSetLayout* pSetLayout, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    ResultValueType<DescriptorSetLayout>::type createDescriptorSetLayout( const DescriptorSetLayoutCreateInfo & createInfo, Optional<const AllocationCallbacks> allocator = nullptr, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_NO_SMART_HANDLE
    template<typename Dispatch = DispatchLoaderStatic>
    typename ResultValueType<UniqueHandle<DescriptorSetLayout,Dispatch>>::type createDescriptorSetLayoutUnique( const DescriptorSetLayoutCreateInfo & createInfo, Optional<const AllocationCallbacks> allocator = nullptr, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_NO_SMART_HANDLE*/
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    Result createDescriptorUpdateTemplate( const DescriptorUpdateTemplateCreateInfo* pCreateInfo, const AllocationCallbacks* pAllocator, DescriptorUpdateTemplate* pDescriptorUpdateTemplate, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    ResultValueType<DescriptorUpdateTemplate>::type createDescriptorUpdateTemplate( const DescriptorUpdateTemplateCreateInfo & createInfo, Optional<const AllocationCallbacks> allocator = nullptr, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_NO_SMART_HANDLE
    template<typename Dispatch = DispatchLoaderStatic>
    typename ResultValueType<UniqueHandle<DescriptorUpdateTemplate,Dispatch>>::type createDescriptorUpdateTemplateUnique( const DescriptorUpdateTemplateCreateInfo & createInfo, Optional<const AllocationCallbacks> allocator = nullptr, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_NO_SMART_HANDLE*/
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    Result createDescriptorUpdateTemplateKHR( const DescriptorUpdateTemplateCreateInfo* pCreateInfo, const AllocationCallbacks* pAllocator, DescriptorUpdateTemplate* pDescriptorUpdateTemplate, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    ResultValueType<DescriptorUpdateTemplate>::type createDescriptorUpdateTemplateKHR( const DescriptorUpdateTemplateCreateInfo & createInfo, Optional<const AllocationCallbacks> allocator = nullptr, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_NO_SMART_HANDLE
    template<typename Dispatch = DispatchLoaderStatic>
    typename ResultValueType<UniqueHandle<DescriptorUpdateTemplate,Dispatch>>::type createDescriptorUpdateTemplateKHRUnique( const DescriptorUpdateTemplateCreateInfo & createInfo, Optional<const AllocationCallbacks> allocator = nullptr, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_NO_SMART_HANDLE*/
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    Result createEvent( const EventCreateInfo* pCreateInfo, const AllocationCallbacks* pAllocator, Event* pEvent, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    ResultValueType<Event>::type createEvent( const EventCreateInfo & createInfo, Optional<const AllocationCallbacks> allocator = nullptr, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_NO_SMART_HANDLE
    template<typename Dispatch = DispatchLoaderStatic>
    typename ResultValueType<UniqueHandle<Event,Dispatch>>::type createEventUnique( const EventCreateInfo & createInfo, Optional<const AllocationCallbacks> allocator = nullptr, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_NO_SMART_HANDLE*/
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    Result createFence( const FenceCreateInfo* pCreateInfo, const AllocationCallbacks* pAllocator, Fence* pFence, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    ResultValueType<Fence>::type createFence( const FenceCreateInfo & createInfo, Optional<const AllocationCallbacks> allocator = nullptr, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_NO_SMART_HANDLE
    template<typename Dispatch = DispatchLoaderStatic>
    typename ResultValueType<UniqueHandle<Fence,Dispatch>>::type createFenceUnique( const FenceCreateInfo & createInfo, Optional<const AllocationCallbacks> allocator = nullptr, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_NO_SMART_HANDLE*/
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    Result createFramebuffer( const FramebufferCreateInfo* pCreateInfo, const AllocationCallbacks* pAllocator, Framebuffer* pFramebuffer, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    ResultValueType<Framebuffer>::type createFramebuffer( const FramebufferCreateInfo & createInfo, Optional<const AllocationCallbacks> allocator = nullptr, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_NO_SMART_HANDLE
    template<typename Dispatch = DispatchLoaderStatic>
    typename ResultValueType<UniqueHandle<Framebuffer,Dispatch>>::type createFramebufferUnique( const FramebufferCreateInfo & createInfo, Optional<const AllocationCallbacks> allocator = nullptr, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_NO_SMART_HANDLE*/
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    Result createGraphicsPipelines( PipelineCache pipelineCache, uint32_t createInfoCount, const GraphicsPipelineCreateInfo* pCreateInfos, const AllocationCallbacks* pAllocator, Pipeline* pPipelines, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Allocator = std::allocator<Pipeline>, typename Dispatch = DispatchLoaderStatic>
    typename ResultValueType<std::vector<Pipeline,Allocator>>::type createGraphicsPipelines( PipelineCache pipelineCache, ArrayProxy<const GraphicsPipelineCreateInfo> createInfos, Optional<const AllocationCallbacks> allocator = nullptr, Dispatch const &d = Dispatch() ) const;
    template<typename Allocator = std::allocator<Pipeline>, typename Dispatch = DispatchLoaderStatic>
    typename ResultValueType<std::vector<Pipeline,Allocator>>::type createGraphicsPipelines( PipelineCache pipelineCache, ArrayProxy<const GraphicsPipelineCreateInfo> createInfos, Optional<const AllocationCallbacks> allocator, Allocator const& vectorAllocator, Dispatch const &d ) const;
    template<typename Dispatch = DispatchLoaderStatic>
    ResultValueType<Pipeline>::type createGraphicsPipeline( PipelineCache pipelineCache, const GraphicsPipelineCreateInfo & createInfo, Optional<const AllocationCallbacks> allocator = nullptr, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_NO_SMART_HANDLE
    template<typename Allocator = std::allocator<UniquePipeline>, typename Dispatch = DispatchLoaderStatic>
    typename ResultValueType<std::vector<UniqueHandle<Pipeline,Dispatch>,Allocator>>::type createGraphicsPipelinesUnique( PipelineCache pipelineCache, ArrayProxy<const GraphicsPipelineCreateInfo> createInfos, Optional<const AllocationCallbacks> allocator = nullptr, Dispatch const &d = Dispatch() ) const;
    template<typename Allocator = std::allocator<UniquePipeline>, typename Dispatch = DispatchLoaderStatic>
    typename ResultValueType<std::vector<UniqueHandle<Pipeline,Dispatch>,Allocator>>::type createGraphicsPipelinesUnique( PipelineCache pipelineCache, ArrayProxy<const GraphicsPipelineCreateInfo> createInfos, Optional<const AllocationCallbacks> allocator, Allocator const& vectorAllocator, Dispatch const &d ) const;
    template<typename Dispatch = DispatchLoaderStatic>
    typename ResultValueType<UniqueHandle<Pipeline,Dispatch>>::type createGraphicsPipelineUnique( PipelineCache pipelineCache, const GraphicsPipelineCreateInfo & createInfo, Optional<const AllocationCallbacks> allocator = nullptr, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_NO_SMART_HANDLE*/
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    Result createImage( const ImageCreateInfo* pCreateInfo, const AllocationCallbacks* pAllocator, Image* pImage, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    ResultValueType<Image>::type createImage( const ImageCreateInfo & createInfo, Optional<const AllocationCallbacks> allocator = nullptr, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_NO_SMART_HANDLE
    template<typename Dispatch = DispatchLoaderStatic>
    typename ResultValueType<UniqueHandle<Image,Dispatch>>::type createImageUnique( const ImageCreateInfo & createInfo, Optional<const AllocationCallbacks> allocator = nullptr, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_NO_SMART_HANDLE*/
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    Result createImageView( const ImageViewCreateInfo* pCreateInfo, const AllocationCallbacks* pAllocator, ImageView* pView, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    ResultValueType<ImageView>::type createImageView( const ImageViewCreateInfo & createInfo, Optional<const AllocationCallbacks> allocator = nullptr, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_NO_SMART_HANDLE
    template<typename Dispatch = DispatchLoaderStatic>
    typename ResultValueType<UniqueHandle<ImageView,Dispatch>>::type createImageViewUnique( const ImageViewCreateInfo & createInfo, Optional<const AllocationCallbacks> allocator = nullptr, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_NO_SMART_HANDLE*/
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    Result createIndirectCommandsLayoutNVX( const IndirectCommandsLayoutCreateInfoNVX* pCreateInfo, const AllocationCallbacks* pAllocator, IndirectCommandsLayoutNVX* pIndirectCommandsLayout, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    ResultValueType<IndirectCommandsLayoutNVX>::type createIndirectCommandsLayoutNVX( const IndirectCommandsLayoutCreateInfoNVX & createInfo, Optional<const AllocationCallbacks> allocator = nullptr, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_NO_SMART_HANDLE
    template<typename Dispatch = DispatchLoaderStatic>
    typename ResultValueType<UniqueHandle<IndirectCommandsLayoutNVX,Dispatch>>::type createIndirectCommandsLayoutNVXUnique( const IndirectCommandsLayoutCreateInfoNVX & createInfo, Optional<const AllocationCallbacks> allocator = nullptr, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_NO_SMART_HANDLE*/
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    Result createObjectTableNVX( const ObjectTableCreateInfoNVX* pCreateInfo, const AllocationCallbacks* pAllocator, ObjectTableNVX* pObjectTable, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    ResultValueType<ObjectTableNVX>::type createObjectTableNVX( const ObjectTableCreateInfoNVX & createInfo, Optional<const AllocationCallbacks> allocator = nullptr, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_NO_SMART_HANDLE
    template<typename Dispatch = DispatchLoaderStatic>
    typename ResultValueType<UniqueHandle<ObjectTableNVX,Dispatch>>::type createObjectTableNVXUnique( const ObjectTableCreateInfoNVX & createInfo, Optional<const AllocationCallbacks> allocator = nullptr, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_NO_SMART_HANDLE*/
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    Result createPipelineCache( const PipelineCacheCreateInfo* pCreateInfo, const AllocationCallbacks* pAllocator, PipelineCache* pPipelineCache, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    ResultValueType<PipelineCache>::type createPipelineCache( const PipelineCacheCreateInfo & createInfo, Optional<const AllocationCallbacks> allocator = nullptr, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_NO_SMART_HANDLE
    template<typename Dispatch = DispatchLoaderStatic>
    typename ResultValueType<UniqueHandle<PipelineCache,Dispatch>>::type createPipelineCacheUnique( const PipelineCacheCreateInfo & createInfo, Optional<const AllocationCallbacks> allocator = nullptr, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_NO_SMART_HANDLE*/
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    Result createPipelineLayout( const PipelineLayoutCreateInfo* pCreateInfo, const AllocationCallbacks* pAllocator, PipelineLayout* pPipelineLayout, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    ResultValueType<PipelineLayout>::type createPipelineLayout( const PipelineLayoutCreateInfo & createInfo, Optional<const AllocationCallbacks> allocator = nullptr, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_NO_SMART_HANDLE
    template<typename Dispatch = DispatchLoaderStatic>
    typename ResultValueType<UniqueHandle<PipelineLayout,Dispatch>>::type createPipelineLayoutUnique( const PipelineLayoutCreateInfo & createInfo, Optional<const AllocationCallbacks> allocator = nullptr, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_NO_SMART_HANDLE*/
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    Result createQueryPool( const QueryPoolCreateInfo* pCreateInfo, const AllocationCallbacks* pAllocator, QueryPool* pQueryPool, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    ResultValueType<QueryPool>::type createQueryPool( const QueryPoolCreateInfo & createInfo, Optional<const AllocationCallbacks> allocator = nullptr, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_NO_SMART_HANDLE
    template<typename Dispatch = DispatchLoaderStatic>
    typename ResultValueType<UniqueHandle<QueryPool,Dispatch>>::type createQueryPoolUnique( const QueryPoolCreateInfo & createInfo, Optional<const AllocationCallbacks> allocator = nullptr, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_NO_SMART_HANDLE*/
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    Result createRayTracingPipelinesNV( PipelineCache pipelineCache, uint32_t createInfoCount, const RayTracingPipelineCreateInfoNV* pCreateInfos, const AllocationCallbacks* pAllocator, Pipeline* pPipelines, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Allocator = std::allocator<Pipeline>, typename Dispatch = DispatchLoaderStatic>
    typename ResultValueType<std::vector<Pipeline,Allocator>>::type createRayTracingPipelinesNV( PipelineCache pipelineCache, ArrayProxy<const RayTracingPipelineCreateInfoNV> createInfos, Optional<const AllocationCallbacks> allocator = nullptr, Dispatch const &d = Dispatch() ) const;
    template<typename Allocator = std::allocator<Pipeline>, typename Dispatch = DispatchLoaderStatic>
    typename ResultValueType<std::vector<Pipeline,Allocator>>::type createRayTracingPipelinesNV( PipelineCache pipelineCache, ArrayProxy<const RayTracingPipelineCreateInfoNV> createInfos, Optional<const AllocationCallbacks> allocator, Allocator const& vectorAllocator, Dispatch const &d ) const;
    template<typename Dispatch = DispatchLoaderStatic>
    ResultValueType<Pipeline>::type createRayTracingPipelineNV( PipelineCache pipelineCache, const RayTracingPipelineCreateInfoNV & createInfo, Optional<const AllocationCallbacks> allocator = nullptr, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_NO_SMART_HANDLE
    template<typename Allocator = std::allocator<UniquePipeline>, typename Dispatch = DispatchLoaderStatic>
    typename ResultValueType<std::vector<UniqueHandle<Pipeline,Dispatch>,Allocator>>::type createRayTracingPipelinesNVUnique( PipelineCache pipelineCache, ArrayProxy<const RayTracingPipelineCreateInfoNV> createInfos, Optional<const AllocationCallbacks> allocator = nullptr, Dispatch const &d = Dispatch() ) const;
    template<typename Allocator = std::allocator<UniquePipeline>, typename Dispatch = DispatchLoaderStatic>
    typename ResultValueType<std::vector<UniqueHandle<Pipeline,Dispatch>,Allocator>>::type createRayTracingPipelinesNVUnique( PipelineCache pipelineCache, ArrayProxy<const RayTracingPipelineCreateInfoNV> createInfos, Optional<const AllocationCallbacks> allocator, Allocator const& vectorAllocator, Dispatch const &d ) const;
    template<typename Dispatch = DispatchLoaderStatic>
    typename ResultValueType<UniqueHandle<Pipeline,Dispatch>>::type createRayTracingPipelineNVUnique( PipelineCache pipelineCache, const RayTracingPipelineCreateInfoNV & createInfo, Optional<const AllocationCallbacks> allocator = nullptr, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_NO_SMART_HANDLE*/
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    Result createRenderPass( const RenderPassCreateInfo* pCreateInfo, const AllocationCallbacks* pAllocator, RenderPass* pRenderPass, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    ResultValueType<RenderPass>::type createRenderPass( const RenderPassCreateInfo & createInfo, Optional<const AllocationCallbacks> allocator = nullptr, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_NO_SMART_HANDLE
    template<typename Dispatch = DispatchLoaderStatic>
    typename ResultValueType<UniqueHandle<RenderPass,Dispatch>>::type createRenderPassUnique( const RenderPassCreateInfo & createInfo, Optional<const AllocationCallbacks> allocator = nullptr, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_NO_SMART_HANDLE*/
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    Result createRenderPass2KHR( const RenderPassCreateInfo2KHR* pCreateInfo, const AllocationCallbacks* pAllocator, RenderPass* pRenderPass, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    ResultValueType<RenderPass>::type createRenderPass2KHR( const RenderPassCreateInfo2KHR & createInfo, Optional<const AllocationCallbacks> allocator = nullptr, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_NO_SMART_HANDLE
    template<typename Dispatch = DispatchLoaderStatic>
    typename ResultValueType<UniqueHandle<RenderPass,Dispatch>>::type createRenderPass2KHRUnique( const RenderPassCreateInfo2KHR & createInfo, Optional<const AllocationCallbacks> allocator = nullptr, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_NO_SMART_HANDLE*/
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    Result createSampler( const SamplerCreateInfo* pCreateInfo, const AllocationCallbacks* pAllocator, Sampler* pSampler, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    ResultValueType<Sampler>::type createSampler( const SamplerCreateInfo & createInfo, Optional<const AllocationCallbacks> allocator = nullptr, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_NO_SMART_HANDLE
    template<typename Dispatch = DispatchLoaderStatic>
    typename ResultValueType<UniqueHandle<Sampler,Dispatch>>::type createSamplerUnique( const SamplerCreateInfo & createInfo, Optional<const AllocationCallbacks> allocator = nullptr, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_NO_SMART_HANDLE*/
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    Result createSamplerYcbcrConversion( const SamplerYcbcrConversionCreateInfo* pCreateInfo, const AllocationCallbacks* pAllocator, SamplerYcbcrConversion* pYcbcrConversion, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    ResultValueType<SamplerYcbcrConversion>::type createSamplerYcbcrConversion( const SamplerYcbcrConversionCreateInfo & createInfo, Optional<const AllocationCallbacks> allocator = nullptr, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_NO_SMART_HANDLE
    template<typename Dispatch = DispatchLoaderStatic>
    typename ResultValueType<UniqueHandle<SamplerYcbcrConversion,Dispatch>>::type createSamplerYcbcrConversionUnique( const SamplerYcbcrConversionCreateInfo & createInfo, Optional<const AllocationCallbacks> allocator = nullptr, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_NO_SMART_HANDLE*/
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    Result createSamplerYcbcrConversionKHR( const SamplerYcbcrConversionCreateInfo* pCreateInfo, const AllocationCallbacks* pAllocator, SamplerYcbcrConversion* pYcbcrConversion, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    ResultValueType<SamplerYcbcrConversion>::type createSamplerYcbcrConversionKHR( const SamplerYcbcrConversionCreateInfo & createInfo, Optional<const AllocationCallbacks> allocator = nullptr, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_NO_SMART_HANDLE
    template<typename Dispatch = DispatchLoaderStatic>
    typename ResultValueType<UniqueHandle<SamplerYcbcrConversion,Dispatch>>::type createSamplerYcbcrConversionKHRUnique( const SamplerYcbcrConversionCreateInfo & createInfo, Optional<const AllocationCallbacks> allocator = nullptr, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_NO_SMART_HANDLE*/
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    Result createSemaphore( const SemaphoreCreateInfo* pCreateInfo, const AllocationCallbacks* pAllocator, Semaphore* pSemaphore, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    ResultValueType<Semaphore>::type createSemaphore( const SemaphoreCreateInfo & createInfo, Optional<const AllocationCallbacks> allocator = nullptr, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_NO_SMART_HANDLE
    template<typename Dispatch = DispatchLoaderStatic>
    typename ResultValueType<UniqueHandle<Semaphore,Dispatch>>::type createSemaphoreUnique( const SemaphoreCreateInfo & createInfo, Optional<const AllocationCallbacks> allocator = nullptr, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_NO_SMART_HANDLE*/
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    Result createShaderModule( const ShaderModuleCreateInfo* pCreateInfo, const AllocationCallbacks* pAllocator, ShaderModule* pShaderModule, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    ResultValueType<ShaderModule>::type createShaderModule( const ShaderModuleCreateInfo & createInfo, Optional<const AllocationCallbacks> allocator = nullptr, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_NO_SMART_HANDLE
    template<typename Dispatch = DispatchLoaderStatic>
    typename ResultValueType<UniqueHandle<ShaderModule,Dispatch>>::type createShaderModuleUnique( const ShaderModuleCreateInfo & createInfo, Optional<const AllocationCallbacks> allocator = nullptr, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_NO_SMART_HANDLE*/
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    Result createSharedSwapchainsKHR( uint32_t swapchainCount, const SwapchainCreateInfoKHR* pCreateInfos, const AllocationCallbacks* pAllocator, SwapchainKHR* pSwapchains, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Allocator = std::allocator<SwapchainKHR>, typename Dispatch = DispatchLoaderStatic>
    typename ResultValueType<std::vector<SwapchainKHR,Allocator>>::type createSharedSwapchainsKHR( ArrayProxy<const SwapchainCreateInfoKHR> createInfos, Optional<const AllocationCallbacks> allocator = nullptr, Dispatch const &d = Dispatch() ) const;
    template<typename Allocator = std::allocator<SwapchainKHR>, typename Dispatch = DispatchLoaderStatic>
    typename ResultValueType<std::vector<SwapchainKHR,Allocator>>::type createSharedSwapchainsKHR( ArrayProxy<const SwapchainCreateInfoKHR> createInfos, Optional<const AllocationCallbacks> allocator, Allocator const& vectorAllocator, Dispatch const &d ) const;
    template<typename Dispatch = DispatchLoaderStatic>
    ResultValueType<SwapchainKHR>::type createSharedSwapchainKHR( const SwapchainCreateInfoKHR & createInfo, Optional<const AllocationCallbacks> allocator = nullptr, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_NO_SMART_HANDLE
    template<typename Allocator = std::allocator<UniqueSwapchainKHR>, typename Dispatch = DispatchLoaderStatic>
    typename ResultValueType<std::vector<UniqueHandle<SwapchainKHR,Dispatch>,Allocator>>::type createSharedSwapchainsKHRUnique( ArrayProxy<const SwapchainCreateInfoKHR> createInfos, Optional<const AllocationCallbacks> allocator = nullptr, Dispatch const &d = Dispatch() ) const;
    template<typename Allocator = std::allocator<UniqueSwapchainKHR>, typename Dispatch = DispatchLoaderStatic>
    typename ResultValueType<std::vector<UniqueHandle<SwapchainKHR,Dispatch>,Allocator>>::type createSharedSwapchainsKHRUnique( ArrayProxy<const SwapchainCreateInfoKHR> createInfos, Optional<const AllocationCallbacks> allocator, Allocator const& vectorAllocator, Dispatch const &d ) const;
    template<typename Dispatch = DispatchLoaderStatic>
    typename ResultValueType<UniqueHandle<SwapchainKHR,Dispatch>>::type createSharedSwapchainKHRUnique( const SwapchainCreateInfoKHR & createInfo, Optional<const AllocationCallbacks> allocator = nullptr, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_NO_SMART_HANDLE*/
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    Result createSwapchainKHR( const SwapchainCreateInfoKHR* pCreateInfo, const AllocationCallbacks* pAllocator, SwapchainKHR* pSwapchain, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    ResultValueType<SwapchainKHR>::type createSwapchainKHR( const SwapchainCreateInfoKHR & createInfo, Optional<const AllocationCallbacks> allocator = nullptr, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_NO_SMART_HANDLE
    template<typename Dispatch = DispatchLoaderStatic>
    typename ResultValueType<UniqueHandle<SwapchainKHR,Dispatch>>::type createSwapchainKHRUnique( const SwapchainCreateInfoKHR & createInfo, Optional<const AllocationCallbacks> allocator = nullptr, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_NO_SMART_HANDLE*/
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    Result createValidationCacheEXT( const ValidationCacheCreateInfoEXT* pCreateInfo, const AllocationCallbacks* pAllocator, ValidationCacheEXT* pValidationCache, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    ResultValueType<ValidationCacheEXT>::type createValidationCacheEXT( const ValidationCacheCreateInfoEXT & createInfo, Optional<const AllocationCallbacks> allocator = nullptr, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_NO_SMART_HANDLE
    template<typename Dispatch = DispatchLoaderStatic>
    typename ResultValueType<UniqueHandle<ValidationCacheEXT,Dispatch>>::type createValidationCacheEXTUnique( const ValidationCacheCreateInfoEXT & createInfo, Optional<const AllocationCallbacks> allocator = nullptr, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_NO_SMART_HANDLE*/
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    Result debugMarkerSetObjectNameEXT( const DebugMarkerObjectNameInfoEXT* pNameInfo, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    ResultValueType<void>::type debugMarkerSetObjectNameEXT( const DebugMarkerObjectNameInfoEXT & nameInfo, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    Result debugMarkerSetObjectTagEXT( const DebugMarkerObjectTagInfoEXT* pTagInfo, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    ResultValueType<void>::type debugMarkerSetObjectTagEXT( const DebugMarkerObjectTagInfoEXT & tagInfo, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    void destroyAccelerationStructureNV( AccelerationStructureNV accelerationStructure, const AllocationCallbacks* pAllocator, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    void destroyAccelerationStructureNV( AccelerationStructureNV accelerationStructure, Optional<const AllocationCallbacks> allocator = nullptr, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    void destroy( AccelerationStructureNV accelerationStructure, const AllocationCallbacks* pAllocator, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    void destroy( AccelerationStructureNV accelerationStructure, Optional<const AllocationCallbacks> allocator = nullptr, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    void destroyBuffer( Buffer buffer, const AllocationCallbacks* pAllocator, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    void destroyBuffer( Buffer buffer, Optional<const AllocationCallbacks> allocator = nullptr, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    void destroy( Buffer buffer, const AllocationCallbacks* pAllocator, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    void destroy( Buffer buffer, Optional<const AllocationCallbacks> allocator = nullptr, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    void destroyBufferView( BufferView bufferView, const AllocationCallbacks* pAllocator, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    void destroyBufferView( BufferView bufferView, Optional<const AllocationCallbacks> allocator = nullptr, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    void destroy( BufferView bufferView, const AllocationCallbacks* pAllocator, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    void destroy( BufferView bufferView, Optional<const AllocationCallbacks> allocator = nullptr, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    void destroyCommandPool( CommandPool commandPool, const AllocationCallbacks* pAllocator, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    void destroyCommandPool( CommandPool commandPool, Optional<const AllocationCallbacks> allocator = nullptr, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    void destroy( CommandPool commandPool, const AllocationCallbacks* pAllocator, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    void destroy( CommandPool commandPool, Optional<const AllocationCallbacks> allocator = nullptr, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    void destroyDescriptorPool( DescriptorPool descriptorPool, const AllocationCallbacks* pAllocator, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    void destroyDescriptorPool( DescriptorPool descriptorPool, Optional<const AllocationCallbacks> allocator = nullptr, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    void destroy( DescriptorPool descriptorPool, const AllocationCallbacks* pAllocator, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    void destroy( DescriptorPool descriptorPool, Optional<const AllocationCallbacks> allocator = nullptr, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    void destroyDescriptorSetLayout( DescriptorSetLayout descriptorSetLayout, const AllocationCallbacks* pAllocator, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    void destroyDescriptorSetLayout( DescriptorSetLayout descriptorSetLayout, Optional<const AllocationCallbacks> allocator = nullptr, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    void destroy( DescriptorSetLayout descriptorSetLayout, const AllocationCallbacks* pAllocator, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    void destroy( DescriptorSetLayout descriptorSetLayout, Optional<const AllocationCallbacks> allocator = nullptr, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    void destroyDescriptorUpdateTemplate( DescriptorUpdateTemplate descriptorUpdateTemplate, const AllocationCallbacks* pAllocator, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    void destroyDescriptorUpdateTemplate( DescriptorUpdateTemplate descriptorUpdateTemplate, Optional<const AllocationCallbacks> allocator = nullptr, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    void destroy( DescriptorUpdateTemplate descriptorUpdateTemplate, const AllocationCallbacks* pAllocator, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    void destroy( DescriptorUpdateTemplate descriptorUpdateTemplate, Optional<const AllocationCallbacks> allocator = nullptr, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    void destroyDescriptorUpdateTemplateKHR( DescriptorUpdateTemplate descriptorUpdateTemplate, const AllocationCallbacks* pAllocator, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    void destroyDescriptorUpdateTemplateKHR( DescriptorUpdateTemplate descriptorUpdateTemplate, Optional<const AllocationCallbacks> allocator = nullptr, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    void destroy( const AllocationCallbacks* pAllocator, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    void destroy( Optional<const AllocationCallbacks> allocator = nullptr, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    void destroyEvent( Event event, const AllocationCallbacks* pAllocator, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    void destroyEvent( Event event, Optional<const AllocationCallbacks> allocator = nullptr, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    void destroy( Event event, const AllocationCallbacks* pAllocator, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    void destroy( Event event, Optional<const AllocationCallbacks> allocator = nullptr, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    void destroyFence( Fence fence, const AllocationCallbacks* pAllocator, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    void destroyFence( Fence fence, Optional<const AllocationCallbacks> allocator = nullptr, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    void destroy( Fence fence, const AllocationCallbacks* pAllocator, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    void destroy( Fence fence, Optional<const AllocationCallbacks> allocator = nullptr, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    void destroyFramebuffer( Framebuffer framebuffer, const AllocationCallbacks* pAllocator, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    void destroyFramebuffer( Framebuffer framebuffer, Optional<const AllocationCallbacks> allocator = nullptr, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    void destroy( Framebuffer framebuffer, const AllocationCallbacks* pAllocator, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    void destroy( Framebuffer framebuffer, Optional<const AllocationCallbacks> allocator = nullptr, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    void destroyImage( Image image, const AllocationCallbacks* pAllocator, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    void destroyImage( Image image, Optional<const AllocationCallbacks> allocator = nullptr, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    void destroy( Image image, const AllocationCallbacks* pAllocator, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    void destroy( Image image, Optional<const AllocationCallbacks> allocator = nullptr, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    void destroyImageView( ImageView imageView, const AllocationCallbacks* pAllocator, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    void destroyImageView( ImageView imageView, Optional<const AllocationCallbacks> allocator = nullptr, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    void destroy( ImageView imageView, const AllocationCallbacks* pAllocator, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    void destroy( ImageView imageView, Optional<const AllocationCallbacks> allocator = nullptr, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    void destroyIndirectCommandsLayoutNVX( IndirectCommandsLayoutNVX indirectCommandsLayout, const AllocationCallbacks* pAllocator, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    void destroyIndirectCommandsLayoutNVX( IndirectCommandsLayoutNVX indirectCommandsLayout, Optional<const AllocationCallbacks> allocator = nullptr, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    void destroy( IndirectCommandsLayoutNVX indirectCommandsLayout, const AllocationCallbacks* pAllocator, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    void destroy( IndirectCommandsLayoutNVX indirectCommandsLayout, Optional<const AllocationCallbacks> allocator = nullptr, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    void destroyObjectTableNVX( ObjectTableNVX objectTable, const AllocationCallbacks* pAllocator, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    void destroyObjectTableNVX( ObjectTableNVX objectTable, Optional<const AllocationCallbacks> allocator = nullptr, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    void destroy( ObjectTableNVX objectTable, const AllocationCallbacks* pAllocator, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    void destroy( ObjectTableNVX objectTable, Optional<const AllocationCallbacks> allocator = nullptr, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    void destroyPipeline( Pipeline pipeline, const AllocationCallbacks* pAllocator, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    void destroyPipeline( Pipeline pipeline, Optional<const AllocationCallbacks> allocator = nullptr, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    void destroy( Pipeline pipeline, const AllocationCallbacks* pAllocator, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    void destroy( Pipeline pipeline, Optional<const AllocationCallbacks> allocator = nullptr, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    void destroyPipelineCache( PipelineCache pipelineCache, const AllocationCallbacks* pAllocator, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    void destroyPipelineCache( PipelineCache pipelineCache, Optional<const AllocationCallbacks> allocator = nullptr, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    void destroy( PipelineCache pipelineCache, const AllocationCallbacks* pAllocator, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    void destroy( PipelineCache pipelineCache, Optional<const AllocationCallbacks> allocator = nullptr, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    void destroyPipelineLayout( PipelineLayout pipelineLayout, const AllocationCallbacks* pAllocator, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    void destroyPipelineLayout( PipelineLayout pipelineLayout, Optional<const AllocationCallbacks> allocator = nullptr, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    void destroy( PipelineLayout pipelineLayout, const AllocationCallbacks* pAllocator, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    void destroy( PipelineLayout pipelineLayout, Optional<const AllocationCallbacks> allocator = nullptr, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    void destroyQueryPool( QueryPool queryPool, const AllocationCallbacks* pAllocator, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    void destroyQueryPool( QueryPool queryPool, Optional<const AllocationCallbacks> allocator = nullptr, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    void destroy( QueryPool queryPool, const AllocationCallbacks* pAllocator, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    void destroy( QueryPool queryPool, Optional<const AllocationCallbacks> allocator = nullptr, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    void destroyRenderPass( RenderPass renderPass, const AllocationCallbacks* pAllocator, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    void destroyRenderPass( RenderPass renderPass, Optional<const AllocationCallbacks> allocator = nullptr, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    void destroy( RenderPass renderPass, const AllocationCallbacks* pAllocator, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    void destroy( RenderPass renderPass, Optional<const AllocationCallbacks> allocator = nullptr, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    void destroySampler( Sampler sampler, const AllocationCallbacks* pAllocator, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    void destroySampler( Sampler sampler, Optional<const AllocationCallbacks> allocator = nullptr, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    void destroy( Sampler sampler, const AllocationCallbacks* pAllocator, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    void destroy( Sampler sampler, Optional<const AllocationCallbacks> allocator = nullptr, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    void destroySamplerYcbcrConversion( SamplerYcbcrConversion ycbcrConversion, const AllocationCallbacks* pAllocator, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    void destroySamplerYcbcrConversion( SamplerYcbcrConversion ycbcrConversion, Optional<const AllocationCallbacks> allocator = nullptr, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    void destroy( SamplerYcbcrConversion ycbcrConversion, const AllocationCallbacks* pAllocator, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    void destroy( SamplerYcbcrConversion ycbcrConversion, Optional<const AllocationCallbacks> allocator = nullptr, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    void destroySamplerYcbcrConversionKHR( SamplerYcbcrConversion ycbcrConversion, const AllocationCallbacks* pAllocator, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    void destroySamplerYcbcrConversionKHR( SamplerYcbcrConversion ycbcrConversion, Optional<const AllocationCallbacks> allocator = nullptr, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    void destroySemaphore( Semaphore semaphore, const AllocationCallbacks* pAllocator, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    void destroySemaphore( Semaphore semaphore, Optional<const AllocationCallbacks> allocator = nullptr, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    void destroy( Semaphore semaphore, const AllocationCallbacks* pAllocator, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    void destroy( Semaphore semaphore, Optional<const AllocationCallbacks> allocator = nullptr, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    void destroyShaderModule( ShaderModule shaderModule, const AllocationCallbacks* pAllocator, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    void destroyShaderModule( ShaderModule shaderModule, Optional<const AllocationCallbacks> allocator = nullptr, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    void destroy( ShaderModule shaderModule, const AllocationCallbacks* pAllocator, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    void destroy( ShaderModule shaderModule, Optional<const AllocationCallbacks> allocator = nullptr, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    void destroySwapchainKHR( SwapchainKHR swapchain, const AllocationCallbacks* pAllocator, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    void destroySwapchainKHR( SwapchainKHR swapchain, Optional<const AllocationCallbacks> allocator = nullptr, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    void destroy( SwapchainKHR swapchain, const AllocationCallbacks* pAllocator, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    void destroy( SwapchainKHR swapchain, Optional<const AllocationCallbacks> allocator = nullptr, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    void destroyValidationCacheEXT( ValidationCacheEXT validationCache, const AllocationCallbacks* pAllocator, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    void destroyValidationCacheEXT( ValidationCacheEXT validationCache, Optional<const AllocationCallbacks> allocator = nullptr, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    void destroy( ValidationCacheEXT validationCache, const AllocationCallbacks* pAllocator, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    void destroy( ValidationCacheEXT validationCache, Optional<const AllocationCallbacks> allocator = nullptr, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

#ifdef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    Result waitIdle(Dispatch const &d = Dispatch() ) const;
#else
    template<typename Dispatch = DispatchLoaderStatic>
    ResultValueType<void>::type waitIdle(Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    Result displayPowerControlEXT( DisplayKHR display, const DisplayPowerInfoEXT* pDisplayPowerInfo, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    ResultValueType<void>::type displayPowerControlEXT( DisplayKHR display, const DisplayPowerInfoEXT & displayPowerInfo, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    Result flushMappedMemoryRanges( uint32_t memoryRangeCount, const MappedMemoryRange* pMemoryRanges, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    ResultValueType<void>::type flushMappedMemoryRanges( ArrayProxy<const MappedMemoryRange> memoryRanges, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    void freeCommandBuffers( CommandPool commandPool, uint32_t commandBufferCount, const CommandBuffer* pCommandBuffers, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    void freeCommandBuffers( CommandPool commandPool, ArrayProxy<const CommandBuffer> commandBuffers, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    void free( CommandPool commandPool, uint32_t commandBufferCount, const CommandBuffer* pCommandBuffers, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    void free( CommandPool commandPool, ArrayProxy<const CommandBuffer> commandBuffers, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    Result freeDescriptorSets( DescriptorPool descriptorPool, uint32_t descriptorSetCount, const DescriptorSet* pDescriptorSets, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    ResultValueType<void>::type freeDescriptorSets( DescriptorPool descriptorPool, ArrayProxy<const DescriptorSet> descriptorSets, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    Result free( DescriptorPool descriptorPool, uint32_t descriptorSetCount, const DescriptorSet* pDescriptorSets, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    ResultValueType<void>::type free( DescriptorPool descriptorPool, ArrayProxy<const DescriptorSet> descriptorSets, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    void freeMemory( DeviceMemory memory, const AllocationCallbacks* pAllocator, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    void freeMemory( DeviceMemory memory, Optional<const AllocationCallbacks> allocator = nullptr, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    void free( DeviceMemory memory, const AllocationCallbacks* pAllocator, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    void free( DeviceMemory memory, Optional<const AllocationCallbacks> allocator = nullptr, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    Result getAccelerationStructureHandleNV( AccelerationStructureNV accelerationStructure, size_t dataSize, void* pData, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename T, typename Dispatch = DispatchLoaderStatic>
    ResultValueType<void>::type getAccelerationStructureHandleNV( AccelerationStructureNV accelerationStructure, ArrayProxy<T> data, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    void getAccelerationStructureMemoryRequirementsNV( const AccelerationStructureMemoryRequirementsInfoNV* pInfo, MemoryRequirements2KHR* pMemoryRequirements, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    MemoryRequirements2KHR getAccelerationStructureMemoryRequirementsNV( const AccelerationStructureMemoryRequirementsInfoNV & info, Dispatch const &d = Dispatch() ) const;
    template<typename X, typename Y, typename ...Z, typename Dispatch = DispatchLoaderStatic>
    StructureChain<X, Y, Z...> getAccelerationStructureMemoryRequirementsNV( const AccelerationStructureMemoryRequirementsInfoNV & info, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

#ifdef VK_USE_PLATFORM_ANDROID_KHR
    template<typename Dispatch = DispatchLoaderStatic>
    Result getAndroidHardwareBufferPropertiesANDROID( const struct AHardwareBuffer* buffer, AndroidHardwareBufferPropertiesANDROID* pProperties, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    ResultValueType<AndroidHardwareBufferPropertiesANDROID>::type getAndroidHardwareBufferPropertiesANDROID( const struct AHardwareBuffer & buffer, Dispatch const &d = Dispatch() ) const;
    template<typename X, typename Y, typename ...Z, typename Dispatch = DispatchLoaderStatic>
    typename ResultValueType<StructureChain<X, Y, Z...>>::type getAndroidHardwareBufferPropertiesANDROID( const struct AHardwareBuffer & buffer, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/
#endif /*VK_USE_PLATFORM_ANDROID_KHR*/

    template<typename Dispatch = DispatchLoaderStatic>
    DeviceAddress getBufferAddressEXT( const BufferDeviceAddressInfoEXT* pInfo, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    DeviceAddress getBufferAddressEXT( const BufferDeviceAddressInfoEXT & info, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    void getBufferMemoryRequirements( Buffer buffer, MemoryRequirements* pMemoryRequirements, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    MemoryRequirements getBufferMemoryRequirements( Buffer buffer, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    void getBufferMemoryRequirements2( const BufferMemoryRequirementsInfo2* pInfo, MemoryRequirements2* pMemoryRequirements, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    MemoryRequirements2 getBufferMemoryRequirements2( const BufferMemoryRequirementsInfo2 & info, Dispatch const &d = Dispatch() ) const;
    template<typename X, typename Y, typename ...Z, typename Dispatch = DispatchLoaderStatic>
    StructureChain<X, Y, Z...> getBufferMemoryRequirements2( const BufferMemoryRequirementsInfo2 & info, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    void getBufferMemoryRequirements2KHR( const BufferMemoryRequirementsInfo2* pInfo, MemoryRequirements2* pMemoryRequirements, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    MemoryRequirements2 getBufferMemoryRequirements2KHR( const BufferMemoryRequirementsInfo2 & info, Dispatch const &d = Dispatch() ) const;
    template<typename X, typename Y, typename ...Z, typename Dispatch = DispatchLoaderStatic>
    StructureChain<X, Y, Z...> getBufferMemoryRequirements2KHR( const BufferMemoryRequirementsInfo2 & info, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    Result getCalibratedTimestampsEXT( uint32_t timestampCount, const CalibratedTimestampInfoEXT* pTimestampInfos, uint64_t* pTimestamps, uint64_t* pMaxDeviation, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    ResultValueType<uint64_t>::type getCalibratedTimestampsEXT( ArrayProxy<const CalibratedTimestampInfoEXT> timestampInfos, ArrayProxy<uint64_t> timestamps, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    void getDescriptorSetLayoutSupport( const DescriptorSetLayoutCreateInfo* pCreateInfo, DescriptorSetLayoutSupport* pSupport, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    DescriptorSetLayoutSupport getDescriptorSetLayoutSupport( const DescriptorSetLayoutCreateInfo & createInfo, Dispatch const &d = Dispatch() ) const;
    template<typename X, typename Y, typename ...Z, typename Dispatch = DispatchLoaderStatic>
    StructureChain<X, Y, Z...> getDescriptorSetLayoutSupport( const DescriptorSetLayoutCreateInfo & createInfo, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    void getDescriptorSetLayoutSupportKHR( const DescriptorSetLayoutCreateInfo* pCreateInfo, DescriptorSetLayoutSupport* pSupport, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    DescriptorSetLayoutSupport getDescriptorSetLayoutSupportKHR( const DescriptorSetLayoutCreateInfo & createInfo, Dispatch const &d = Dispatch() ) const;
    template<typename X, typename Y, typename ...Z, typename Dispatch = DispatchLoaderStatic>
    StructureChain<X, Y, Z...> getDescriptorSetLayoutSupportKHR( const DescriptorSetLayoutCreateInfo & createInfo, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    void getGroupPeerMemoryFeatures( uint32_t heapIndex, uint32_t localDeviceIndex, uint32_t remoteDeviceIndex, PeerMemoryFeatureFlags* pPeerMemoryFeatures, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    PeerMemoryFeatureFlags getGroupPeerMemoryFeatures( uint32_t heapIndex, uint32_t localDeviceIndex, uint32_t remoteDeviceIndex, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    void getGroupPeerMemoryFeaturesKHR( uint32_t heapIndex, uint32_t localDeviceIndex, uint32_t remoteDeviceIndex, PeerMemoryFeatureFlags* pPeerMemoryFeatures, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    PeerMemoryFeatureFlags getGroupPeerMemoryFeaturesKHR( uint32_t heapIndex, uint32_t localDeviceIndex, uint32_t remoteDeviceIndex, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    Result getGroupPresentCapabilitiesKHR( DeviceGroupPresentCapabilitiesKHR* pDeviceGroupPresentCapabilities, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    ResultValueType<DeviceGroupPresentCapabilitiesKHR>::type getGroupPresentCapabilitiesKHR(Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    Result getGroupSurfacePresentModesKHR( SurfaceKHR surface, DeviceGroupPresentModeFlagsKHR* pModes, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    ResultValueType<DeviceGroupPresentModeFlagsKHR>::type getGroupSurfacePresentModesKHR( SurfaceKHR surface, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    void getMemoryCommitment( DeviceMemory memory, DeviceSize* pCommittedMemoryInBytes, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    DeviceSize getMemoryCommitment( DeviceMemory memory, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    PFN_vkVoidFunction getProcAddr( const char* pName, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    PFN_vkVoidFunction getProcAddr( const std::string & name, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    void getQueue( uint32_t queueFamilyIndex, uint32_t queueIndex, Queue* pQueue, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    Queue getQueue( uint32_t queueFamilyIndex, uint32_t queueIndex, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    void getQueue2( const DeviceQueueInfo2* pQueueInfo, Queue* pQueue, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    Queue getQueue2( const DeviceQueueInfo2 & queueInfo, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    Result getEventStatus( Event event, Dispatch const &d = Dispatch() ) const;

    template<typename Dispatch = DispatchLoaderStatic>
    Result getFenceFdKHR( const FenceGetFdInfoKHR* pGetFdInfo, int* pFd, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    ResultValueType<int>::type getFenceFdKHR( const FenceGetFdInfoKHR & getFdInfo, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    Result getFenceStatus( Fence fence, Dispatch const &d = Dispatch() ) const;

#ifdef VK_USE_PLATFORM_WIN32_KHR
    template<typename Dispatch = DispatchLoaderStatic>
    Result getFenceWin32HandleKHR( const FenceGetWin32HandleInfoKHR* pGetWin32HandleInfo, HANDLE* pHandle, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    ResultValueType<HANDLE>::type getFenceWin32HandleKHR( const FenceGetWin32HandleInfoKHR & getWin32HandleInfo, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/
#endif /*VK_USE_PLATFORM_WIN32_KHR*/

    template<typename Dispatch = DispatchLoaderStatic>
    Result getImageDrmFormatModifierPropertiesEXT( Image image, ImageDrmFormatModifierPropertiesEXT* pProperties, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    ResultValueType<ImageDrmFormatModifierPropertiesEXT>::type getImageDrmFormatModifierPropertiesEXT( Image image, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    void getImageMemoryRequirements( Image image, MemoryRequirements* pMemoryRequirements, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    MemoryRequirements getImageMemoryRequirements( Image image, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    void getImageMemoryRequirements2( const ImageMemoryRequirementsInfo2* pInfo, MemoryRequirements2* pMemoryRequirements, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    MemoryRequirements2 getImageMemoryRequirements2( const ImageMemoryRequirementsInfo2 & info, Dispatch const &d = Dispatch() ) const;
    template<typename X, typename Y, typename ...Z, typename Dispatch = DispatchLoaderStatic>
    StructureChain<X, Y, Z...> getImageMemoryRequirements2( const ImageMemoryRequirementsInfo2 & info, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    void getImageMemoryRequirements2KHR( const ImageMemoryRequirementsInfo2* pInfo, MemoryRequirements2* pMemoryRequirements, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    MemoryRequirements2 getImageMemoryRequirements2KHR( const ImageMemoryRequirementsInfo2 & info, Dispatch const &d = Dispatch() ) const;
    template<typename X, typename Y, typename ...Z, typename Dispatch = DispatchLoaderStatic>
    StructureChain<X, Y, Z...> getImageMemoryRequirements2KHR( const ImageMemoryRequirementsInfo2 & info, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    void getImageSparseMemoryRequirements( Image image, uint32_t* pSparseMemoryRequirementCount, SparseImageMemoryRequirements* pSparseMemoryRequirements, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Allocator = std::allocator<SparseImageMemoryRequirements>, typename Dispatch = DispatchLoaderStatic>
    std::vector<SparseImageMemoryRequirements,Allocator> getImageSparseMemoryRequirements( Image image, Dispatch const &d = Dispatch() ) const;
    template<typename Allocator = std::allocator<SparseImageMemoryRequirements>, typename Dispatch = DispatchLoaderStatic>
    std::vector<SparseImageMemoryRequirements,Allocator> getImageSparseMemoryRequirements( Image image, Allocator const& vectorAllocator, Dispatch const &d ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    void getImageSparseMemoryRequirements2( const ImageSparseMemoryRequirementsInfo2* pInfo, uint32_t* pSparseMemoryRequirementCount, SparseImageMemoryRequirements2* pSparseMemoryRequirements, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Allocator = std::allocator<SparseImageMemoryRequirements2>, typename Dispatch = DispatchLoaderStatic>
    std::vector<SparseImageMemoryRequirements2,Allocator> getImageSparseMemoryRequirements2( const ImageSparseMemoryRequirementsInfo2 & info, Dispatch const &d = Dispatch() ) const;
    template<typename Allocator = std::allocator<SparseImageMemoryRequirements2>, typename Dispatch = DispatchLoaderStatic>
    std::vector<SparseImageMemoryRequirements2,Allocator> getImageSparseMemoryRequirements2( const ImageSparseMemoryRequirementsInfo2 & info, Allocator const& vectorAllocator, Dispatch const &d ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    void getImageSparseMemoryRequirements2KHR( const ImageSparseMemoryRequirementsInfo2* pInfo, uint32_t* pSparseMemoryRequirementCount, SparseImageMemoryRequirements2* pSparseMemoryRequirements, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Allocator = std::allocator<SparseImageMemoryRequirements2>, typename Dispatch = DispatchLoaderStatic>
    std::vector<SparseImageMemoryRequirements2,Allocator> getImageSparseMemoryRequirements2KHR( const ImageSparseMemoryRequirementsInfo2 & info, Dispatch const &d = Dispatch() ) const;
    template<typename Allocator = std::allocator<SparseImageMemoryRequirements2>, typename Dispatch = DispatchLoaderStatic>
    std::vector<SparseImageMemoryRequirements2,Allocator> getImageSparseMemoryRequirements2KHR( const ImageSparseMemoryRequirementsInfo2 & info, Allocator const& vectorAllocator, Dispatch const &d ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    void getImageSubresourceLayout( Image image, const ImageSubresource* pSubresource, SubresourceLayout* pLayout, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    SubresourceLayout getImageSubresourceLayout( Image image, const ImageSubresource & subresource, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

#ifdef VK_USE_PLATFORM_ANDROID_KHR
    template<typename Dispatch = DispatchLoaderStatic>
    Result getMemoryAndroidHardwareBufferANDROID( const MemoryGetAndroidHardwareBufferInfoANDROID* pInfo, struct AHardwareBuffer** pBuffer, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    ResultValueType<struct AHardwareBuffer*>::type getMemoryAndroidHardwareBufferANDROID( const MemoryGetAndroidHardwareBufferInfoANDROID & info, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/
#endif /*VK_USE_PLATFORM_ANDROID_KHR*/

    template<typename Dispatch = DispatchLoaderStatic>
    Result getMemoryFdKHR( const MemoryGetFdInfoKHR* pGetFdInfo, int* pFd, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    ResultValueType<int>::type getMemoryFdKHR( const MemoryGetFdInfoKHR & getFdInfo, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    Result getMemoryFdPropertiesKHR( ExternalMemoryHandleTypeFlagBits handleType, int fd, MemoryFdPropertiesKHR* pMemoryFdProperties, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    ResultValueType<MemoryFdPropertiesKHR>::type getMemoryFdPropertiesKHR( ExternalMemoryHandleTypeFlagBits handleType, int fd, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    Result getMemoryHostPointerPropertiesEXT( ExternalMemoryHandleTypeFlagBits handleType, const void* pHostPointer, MemoryHostPointerPropertiesEXT* pMemoryHostPointerProperties, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    ResultValueType<MemoryHostPointerPropertiesEXT>::type getMemoryHostPointerPropertiesEXT( ExternalMemoryHandleTypeFlagBits handleType, const void* pHostPointer, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

#ifdef VK_USE_PLATFORM_WIN32_KHR
    template<typename Dispatch = DispatchLoaderStatic>
    Result getMemoryWin32HandleKHR( const MemoryGetWin32HandleInfoKHR* pGetWin32HandleInfo, HANDLE* pHandle, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    ResultValueType<HANDLE>::type getMemoryWin32HandleKHR( const MemoryGetWin32HandleInfoKHR & getWin32HandleInfo, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/
#endif /*VK_USE_PLATFORM_WIN32_KHR*/

#ifdef VK_USE_PLATFORM_WIN32_KHR
    template<typename Dispatch = DispatchLoaderStatic>
    Result getMemoryWin32HandleNV( DeviceMemory memory, ExternalMemoryHandleTypeFlagsNV handleType, HANDLE* pHandle, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    ResultValueType<HANDLE>::type getMemoryWin32HandleNV( DeviceMemory memory, ExternalMemoryHandleTypeFlagsNV handleType, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/
#endif /*VK_USE_PLATFORM_WIN32_KHR*/

#ifdef VK_USE_PLATFORM_WIN32_KHR
    template<typename Dispatch = DispatchLoaderStatic>
    Result getMemoryWin32HandlePropertiesKHR( ExternalMemoryHandleTypeFlagBits handleType, HANDLE handle, MemoryWin32HandlePropertiesKHR* pMemoryWin32HandleProperties, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    ResultValueType<MemoryWin32HandlePropertiesKHR>::type getMemoryWin32HandlePropertiesKHR( ExternalMemoryHandleTypeFlagBits handleType, HANDLE handle, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/
#endif /*VK_USE_PLATFORM_WIN32_KHR*/

    template<typename Dispatch = DispatchLoaderStatic>
    Result getPastPresentationTimingGOOGLE( SwapchainKHR swapchain, uint32_t* pPresentationTimingCount, PastPresentationTimingGOOGLE* pPresentationTimings, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Allocator = std::allocator<PastPresentationTimingGOOGLE>, typename Dispatch = DispatchLoaderStatic>
    typename ResultValueType<std::vector<PastPresentationTimingGOOGLE,Allocator>>::type getPastPresentationTimingGOOGLE( SwapchainKHR swapchain, Dispatch const &d = Dispatch() ) const;
    template<typename Allocator = std::allocator<PastPresentationTimingGOOGLE>, typename Dispatch = DispatchLoaderStatic>
    typename ResultValueType<std::vector<PastPresentationTimingGOOGLE,Allocator>>::type getPastPresentationTimingGOOGLE( SwapchainKHR swapchain, Allocator const& vectorAllocator, Dispatch const &d ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    Result getPipelineCacheData( PipelineCache pipelineCache, size_t* pDataSize, void* pData, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Allocator = std::allocator<uint8_t>, typename Dispatch = DispatchLoaderStatic>
    typename ResultValueType<std::vector<uint8_t,Allocator>>::type getPipelineCacheData( PipelineCache pipelineCache, Dispatch const &d = Dispatch() ) const;
    template<typename Allocator = std::allocator<uint8_t>, typename Dispatch = DispatchLoaderStatic>
    typename ResultValueType<std::vector<uint8_t,Allocator>>::type getPipelineCacheData( PipelineCache pipelineCache, Allocator const& vectorAllocator, Dispatch const &d ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    Result getQueryPoolResults( QueryPool queryPool, uint32_t firstQuery, uint32_t queryCount, size_t dataSize, void* pData, DeviceSize stride, QueryResultFlags flags, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename T, typename Dispatch = DispatchLoaderStatic>
    Result getQueryPoolResults( QueryPool queryPool, uint32_t firstQuery, uint32_t queryCount, ArrayProxy<T> data, DeviceSize stride, QueryResultFlags flags, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    Result getRayTracingShaderGroupHandlesNV( Pipeline pipeline, uint32_t firstGroup, uint32_t groupCount, size_t dataSize, void* pData, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename T, typename Dispatch = DispatchLoaderStatic>
    ResultValueType<void>::type getRayTracingShaderGroupHandlesNV( Pipeline pipeline, uint32_t firstGroup, uint32_t groupCount, ArrayProxy<T> data, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    Result getRefreshCycleDurationGOOGLE( SwapchainKHR swapchain, RefreshCycleDurationGOOGLE* pDisplayTimingProperties, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    ResultValueType<RefreshCycleDurationGOOGLE>::type getRefreshCycleDurationGOOGLE( SwapchainKHR swapchain, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    void getRenderAreaGranularity( RenderPass renderPass, Extent2D* pGranularity, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    Extent2D getRenderAreaGranularity( RenderPass renderPass, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    Result getSemaphoreFdKHR( const SemaphoreGetFdInfoKHR* pGetFdInfo, int* pFd, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    ResultValueType<int>::type getSemaphoreFdKHR( const SemaphoreGetFdInfoKHR & getFdInfo, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

#ifdef VK_USE_PLATFORM_WIN32_KHR
    template<typename Dispatch = DispatchLoaderStatic>
    Result getSemaphoreWin32HandleKHR( const SemaphoreGetWin32HandleInfoKHR* pGetWin32HandleInfo, HANDLE* pHandle, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    ResultValueType<HANDLE>::type getSemaphoreWin32HandleKHR( const SemaphoreGetWin32HandleInfoKHR & getWin32HandleInfo, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/
#endif /*VK_USE_PLATFORM_WIN32_KHR*/

    template<typename Dispatch = DispatchLoaderStatic>
    Result getShaderInfoAMD( Pipeline pipeline, ShaderStageFlagBits shaderStage, ShaderInfoTypeAMD infoType, size_t* pInfoSize, void* pInfo, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Allocator = std::allocator<uint8_t>, typename Dispatch = DispatchLoaderStatic>
    typename ResultValueType<std::vector<uint8_t,Allocator>>::type getShaderInfoAMD( Pipeline pipeline, ShaderStageFlagBits shaderStage, ShaderInfoTypeAMD infoType, Dispatch const &d = Dispatch() ) const;
    template<typename Allocator = std::allocator<uint8_t>, typename Dispatch = DispatchLoaderStatic>
    typename ResultValueType<std::vector<uint8_t,Allocator>>::type getShaderInfoAMD( Pipeline pipeline, ShaderStageFlagBits shaderStage, ShaderInfoTypeAMD infoType, Allocator const& vectorAllocator, Dispatch const &d ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    Result getSwapchainCounterEXT( SwapchainKHR swapchain, SurfaceCounterFlagBitsEXT counter, uint64_t* pCounterValue, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    ResultValueType<uint64_t>::type getSwapchainCounterEXT( SwapchainKHR swapchain, SurfaceCounterFlagBitsEXT counter, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    Result getSwapchainImagesKHR( SwapchainKHR swapchain, uint32_t* pSwapchainImageCount, Image* pSwapchainImages, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Allocator = std::allocator<Image>, typename Dispatch = DispatchLoaderStatic>
    typename ResultValueType<std::vector<Image,Allocator>>::type getSwapchainImagesKHR( SwapchainKHR swapchain, Dispatch const &d = Dispatch() ) const;
    template<typename Allocator = std::allocator<Image>, typename Dispatch = DispatchLoaderStatic>
    typename ResultValueType<std::vector<Image,Allocator>>::type getSwapchainImagesKHR( SwapchainKHR swapchain, Allocator const& vectorAllocator, Dispatch const &d ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    Result getSwapchainStatusKHR( SwapchainKHR swapchain, Dispatch const &d = Dispatch() ) const;

    template<typename Dispatch = DispatchLoaderStatic>
    Result getValidationCacheDataEXT( ValidationCacheEXT validationCache, size_t* pDataSize, void* pData, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Allocator = std::allocator<uint8_t>, typename Dispatch = DispatchLoaderStatic>
    typename ResultValueType<std::vector<uint8_t,Allocator>>::type getValidationCacheDataEXT( ValidationCacheEXT validationCache, Dispatch const &d = Dispatch() ) const;
    template<typename Allocator = std::allocator<uint8_t>, typename Dispatch = DispatchLoaderStatic>
    typename ResultValueType<std::vector<uint8_t,Allocator>>::type getValidationCacheDataEXT( ValidationCacheEXT validationCache, Allocator const& vectorAllocator, Dispatch const &d ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    Result importFenceFdKHR( const ImportFenceFdInfoKHR* pImportFenceFdInfo, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    ResultValueType<void>::type importFenceFdKHR( const ImportFenceFdInfoKHR & importFenceFdInfo, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

#ifdef VK_USE_PLATFORM_WIN32_KHR
    template<typename Dispatch = DispatchLoaderStatic>
    Result importFenceWin32HandleKHR( const ImportFenceWin32HandleInfoKHR* pImportFenceWin32HandleInfo, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    ResultValueType<void>::type importFenceWin32HandleKHR( const ImportFenceWin32HandleInfoKHR & importFenceWin32HandleInfo, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/
#endif /*VK_USE_PLATFORM_WIN32_KHR*/

    template<typename Dispatch = DispatchLoaderStatic>
    Result importSemaphoreFdKHR( const ImportSemaphoreFdInfoKHR* pImportSemaphoreFdInfo, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    ResultValueType<void>::type importSemaphoreFdKHR( const ImportSemaphoreFdInfoKHR & importSemaphoreFdInfo, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

#ifdef VK_USE_PLATFORM_WIN32_KHR
    template<typename Dispatch = DispatchLoaderStatic>
    Result importSemaphoreWin32HandleKHR( const ImportSemaphoreWin32HandleInfoKHR* pImportSemaphoreWin32HandleInfo, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    ResultValueType<void>::type importSemaphoreWin32HandleKHR( const ImportSemaphoreWin32HandleInfoKHR & importSemaphoreWin32HandleInfo, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/
#endif /*VK_USE_PLATFORM_WIN32_KHR*/

    template<typename Dispatch = DispatchLoaderStatic>
    Result invalidateMappedMemoryRanges( uint32_t memoryRangeCount, const MappedMemoryRange* pMemoryRanges, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    ResultValueType<void>::type invalidateMappedMemoryRanges( ArrayProxy<const MappedMemoryRange> memoryRanges, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    Result mapMemory( DeviceMemory memory, DeviceSize offset, DeviceSize size, MemoryMapFlags flags, void** ppData, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    ResultValueType<void*>::type mapMemory( DeviceMemory memory, DeviceSize offset, DeviceSize size, MemoryMapFlags flags = MemoryMapFlags(), Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    Result mergePipelineCaches( PipelineCache dstCache, uint32_t srcCacheCount, const PipelineCache* pSrcCaches, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    ResultValueType<void>::type mergePipelineCaches( PipelineCache dstCache, ArrayProxy<const PipelineCache> srcCaches, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    Result mergeValidationCachesEXT( ValidationCacheEXT dstCache, uint32_t srcCacheCount, const ValidationCacheEXT* pSrcCaches, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    ResultValueType<void>::type mergeValidationCachesEXT( ValidationCacheEXT dstCache, ArrayProxy<const ValidationCacheEXT> srcCaches, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    Result registerEventEXT( const DeviceEventInfoEXT* pDeviceEventInfo, const AllocationCallbacks* pAllocator, Fence* pFence, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    ResultValueType<Fence>::type registerEventEXT( const DeviceEventInfoEXT & deviceEventInfo, Optional<const AllocationCallbacks> allocator = nullptr, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    Result registerDisplayEventEXT( DisplayKHR display, const DisplayEventInfoEXT* pDisplayEventInfo, const AllocationCallbacks* pAllocator, Fence* pFence, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    ResultValueType<Fence>::type registerDisplayEventEXT( DisplayKHR display, const DisplayEventInfoEXT & displayEventInfo, Optional<const AllocationCallbacks> allocator = nullptr, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    Result registerObjectsNVX( ObjectTableNVX objectTable, uint32_t objectCount, const ObjectTableEntryNVX* const* ppObjectTableEntries, const uint32_t* pObjectIndices, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    ResultValueType<void>::type registerObjectsNVX( ObjectTableNVX objectTable, ArrayProxy<const ObjectTableEntryNVX* const> pObjectTableEntries, ArrayProxy<const uint32_t> objectIndices, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

#ifdef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    Result resetCommandPool( CommandPool commandPool, CommandPoolResetFlags flags, Dispatch const &d = Dispatch() ) const;
#else
    template<typename Dispatch = DispatchLoaderStatic>
    ResultValueType<void>::type resetCommandPool( CommandPool commandPool, CommandPoolResetFlags flags, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

#ifdef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    Result resetDescriptorPool( DescriptorPool descriptorPool, DescriptorPoolResetFlags flags = DescriptorPoolResetFlags(), Dispatch const &d = Dispatch() ) const;
#else
    template<typename Dispatch = DispatchLoaderStatic>
    ResultValueType<void>::type resetDescriptorPool( DescriptorPool descriptorPool, DescriptorPoolResetFlags flags = DescriptorPoolResetFlags(), Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

#ifdef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    Result resetEvent( Event event, Dispatch const &d = Dispatch() ) const;
#else
    template<typename Dispatch = DispatchLoaderStatic>
    ResultValueType<void>::type resetEvent( Event event, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    Result resetFences( uint32_t fenceCount, const Fence* pFences, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    ResultValueType<void>::type resetFences( ArrayProxy<const Fence> fences, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    Result setDebugUtilsObjectNameEXT( const DebugUtilsObjectNameInfoEXT* pNameInfo, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    ResultValueType<void>::type setDebugUtilsObjectNameEXT( const DebugUtilsObjectNameInfoEXT & nameInfo, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    Result setDebugUtilsObjectTagEXT( const DebugUtilsObjectTagInfoEXT* pTagInfo, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    ResultValueType<void>::type setDebugUtilsObjectTagEXT( const DebugUtilsObjectTagInfoEXT & tagInfo, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

#ifdef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    Result setEvent( Event event, Dispatch const &d = Dispatch() ) const;
#else
    template<typename Dispatch = DispatchLoaderStatic>
    ResultValueType<void>::type setEvent( Event event, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    void setHdrMetadataEXT( uint32_t swapchainCount, const SwapchainKHR* pSwapchains, const HdrMetadataEXT* pMetadata, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    void setHdrMetadataEXT( ArrayProxy<const SwapchainKHR> swapchains, ArrayProxy<const HdrMetadataEXT> metadata, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    void trimCommandPool( CommandPool commandPool, CommandPoolTrimFlags flags = CommandPoolTrimFlags(), Dispatch const &d = Dispatch() ) const;

    template<typename Dispatch = DispatchLoaderStatic>
    void trimCommandPoolKHR( CommandPool commandPool, CommandPoolTrimFlags flags = CommandPoolTrimFlags(), Dispatch const &d = Dispatch() ) const;

    template<typename Dispatch = DispatchLoaderStatic>
    void unmapMemory( DeviceMemory memory, Dispatch const &d = Dispatch() ) const;

    template<typename Dispatch = DispatchLoaderStatic>
    Result unregisterObjectsNVX( ObjectTableNVX objectTable, uint32_t objectCount, const ObjectEntryTypeNVX* pObjectEntryTypes, const uint32_t* pObjectIndices, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    ResultValueType<void>::type unregisterObjectsNVX( ObjectTableNVX objectTable, ArrayProxy<const ObjectEntryTypeNVX> objectEntryTypes, ArrayProxy<const uint32_t> objectIndices, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    void updateDescriptorSetWithTemplate( DescriptorSet descriptorSet, DescriptorUpdateTemplate descriptorUpdateTemplate, const void* pData, Dispatch const &d = Dispatch() ) const;

    template<typename Dispatch = DispatchLoaderStatic>
    void updateDescriptorSetWithTemplateKHR( DescriptorSet descriptorSet, DescriptorUpdateTemplate descriptorUpdateTemplate, const void* pData, Dispatch const &d = Dispatch() ) const;

    template<typename Dispatch = DispatchLoaderStatic>
    void updateDescriptorSets( uint32_t descriptorWriteCount, const WriteDescriptorSet* pDescriptorWrites, uint32_t descriptorCopyCount, const CopyDescriptorSet* pDescriptorCopies, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    void updateDescriptorSets( ArrayProxy<const WriteDescriptorSet> descriptorWrites, ArrayProxy<const CopyDescriptorSet> descriptorCopies, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    Result waitForFences( uint32_t fenceCount, const Fence* pFences, Bool32 waitAll, uint64_t timeout, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    Result waitForFences( ArrayProxy<const Fence> fences, Bool32 waitAll, uint64_t timeout, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    VULKAN_HPP_TYPESAFE_EXPLICIT operator VkDevice() const
    {
      return m_device;
    }

    explicit operator bool() const
    {
      return m_device != VK_NULL_HANDLE;
    }

    bool operator!() const
    {
      return m_device == VK_NULL_HANDLE;
    }

  private:
    VkDevice m_device;
  };
  static_assert( sizeof( Device ) == sizeof( VkDevice ), "handle and wrapper have different size!" );

  class DisplayModeKHR
  {
  public:
    VULKAN_HPP_CONSTEXPR DisplayModeKHR()
      : m_displayModeKHR(VK_NULL_HANDLE)
    {}

    VULKAN_HPP_CONSTEXPR DisplayModeKHR( std::nullptr_t )
      : m_displayModeKHR(VK_NULL_HANDLE)
    {}

    VULKAN_HPP_TYPESAFE_EXPLICIT DisplayModeKHR( VkDisplayModeKHR displayModeKHR )
      : m_displayModeKHR( displayModeKHR )
    {}

#if defined(VULKAN_HPP_TYPESAFE_CONVERSION)
    DisplayModeKHR & operator=(VkDisplayModeKHR displayModeKHR)
    {
      m_displayModeKHR = displayModeKHR;
      return *this; 
    }
#endif

    DisplayModeKHR & operator=( std::nullptr_t )
    {
      m_displayModeKHR = VK_NULL_HANDLE;
      return *this;
    }

    bool operator==( DisplayModeKHR const & rhs ) const
    {
      return m_displayModeKHR == rhs.m_displayModeKHR;
    }

    bool operator!=(DisplayModeKHR const & rhs ) const
    {
      return m_displayModeKHR != rhs.m_displayModeKHR;
    }

    bool operator<(DisplayModeKHR const & rhs ) const
    {
      return m_displayModeKHR < rhs.m_displayModeKHR;
    }

    VULKAN_HPP_TYPESAFE_EXPLICIT operator VkDisplayModeKHR() const
    {
      return m_displayModeKHR;
    }

    explicit operator bool() const
    {
      return m_displayModeKHR != VK_NULL_HANDLE;
    }

    bool operator!() const
    {
      return m_displayModeKHR == VK_NULL_HANDLE;
    }

  private:
    VkDisplayModeKHR m_displayModeKHR;
  };
  static_assert( sizeof( DisplayModeKHR ) == sizeof( VkDisplayModeKHR ), "handle and wrapper have different size!" );

#ifndef VULKAN_HPP_NO_SMART_HANDLE
  template <typename Dispatch> class UniqueHandleTraits<Device, Dispatch> { public: using deleter = ObjectDestroy<NoParent, Dispatch>; };
  using UniqueDevice = UniqueHandle<Device, DispatchLoaderStatic>;
#endif /*VULKAN_HPP_NO_SMART_HANDLE*/

  class PhysicalDevice
  {
  public:
    VULKAN_HPP_CONSTEXPR PhysicalDevice()
      : m_physicalDevice(VK_NULL_HANDLE)
    {}

    VULKAN_HPP_CONSTEXPR PhysicalDevice( std::nullptr_t )
      : m_physicalDevice(VK_NULL_HANDLE)
    {}

    VULKAN_HPP_TYPESAFE_EXPLICIT PhysicalDevice( VkPhysicalDevice physicalDevice )
      : m_physicalDevice( physicalDevice )
    {}

#if defined(VULKAN_HPP_TYPESAFE_CONVERSION)
    PhysicalDevice & operator=(VkPhysicalDevice physicalDevice)
    {
      m_physicalDevice = physicalDevice;
      return *this; 
    }
#endif

    PhysicalDevice & operator=( std::nullptr_t )
    {
      m_physicalDevice = VK_NULL_HANDLE;
      return *this;
    }

    bool operator==( PhysicalDevice const & rhs ) const
    {
      return m_physicalDevice == rhs.m_physicalDevice;
    }

    bool operator!=(PhysicalDevice const & rhs ) const
    {
      return m_physicalDevice != rhs.m_physicalDevice;
    }

    bool operator<(PhysicalDevice const & rhs ) const
    {
      return m_physicalDevice < rhs.m_physicalDevice;
    }

#ifdef VK_USE_PLATFORM_XLIB_XRANDR_EXT
    template<typename Dispatch = DispatchLoaderStatic>
    Result acquireXlibDisplayEXT( Display* dpy, DisplayKHR display, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    ResultValueType<Display>::type acquireXlibDisplayEXT( DisplayKHR display, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/
#endif /*VK_USE_PLATFORM_XLIB_XRANDR_EXT*/

    template<typename Dispatch = DispatchLoaderStatic>
    Result createDevice( const DeviceCreateInfo* pCreateInfo, const AllocationCallbacks* pAllocator, Device* pDevice, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    ResultValueType<Device>::type createDevice( const DeviceCreateInfo & createInfo, Optional<const AllocationCallbacks> allocator = nullptr, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_NO_SMART_HANDLE
    template<typename Dispatch = DispatchLoaderStatic>
    typename ResultValueType<UniqueHandle<Device,Dispatch>>::type createDeviceUnique( const DeviceCreateInfo & createInfo, Optional<const AllocationCallbacks> allocator = nullptr, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_NO_SMART_HANDLE*/
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    Result createDisplayModeKHR( DisplayKHR display, const DisplayModeCreateInfoKHR* pCreateInfo, const AllocationCallbacks* pAllocator, DisplayModeKHR* pMode, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    ResultValueType<DisplayModeKHR>::type createDisplayModeKHR( DisplayKHR display, const DisplayModeCreateInfoKHR & createInfo, Optional<const AllocationCallbacks> allocator = nullptr, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    Result enumerateDeviceExtensionProperties( const char* pLayerName, uint32_t* pPropertyCount, ExtensionProperties* pProperties, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Allocator = std::allocator<ExtensionProperties>, typename Dispatch = DispatchLoaderStatic>
    typename ResultValueType<std::vector<ExtensionProperties,Allocator>>::type enumerateDeviceExtensionProperties( Optional<const std::string> layerName = nullptr, Dispatch const &d = Dispatch() ) const;
    template<typename Allocator = std::allocator<ExtensionProperties>, typename Dispatch = DispatchLoaderStatic>
    typename ResultValueType<std::vector<ExtensionProperties,Allocator>>::type enumerateDeviceExtensionProperties( Optional<const std::string> layerName, Allocator const& vectorAllocator, Dispatch const &d ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    Result enumerateDeviceLayerProperties( uint32_t* pPropertyCount, LayerProperties* pProperties, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Allocator = std::allocator<LayerProperties>, typename Dispatch = DispatchLoaderStatic>
    typename ResultValueType<std::vector<LayerProperties,Allocator>>::type enumerateDeviceLayerProperties(Dispatch const &d = Dispatch() ) const;
    template<typename Allocator = std::allocator<LayerProperties>, typename Dispatch = DispatchLoaderStatic>
    typename ResultValueType<std::vector<LayerProperties,Allocator>>::type enumerateDeviceLayerProperties(Allocator const& vectorAllocator, Dispatch const &d ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    Result getDisplayModeProperties2KHR( DisplayKHR display, uint32_t* pPropertyCount, DisplayModeProperties2KHR* pProperties, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Allocator = std::allocator<DisplayModeProperties2KHR>, typename Dispatch = DispatchLoaderStatic>
    typename ResultValueType<std::vector<DisplayModeProperties2KHR,Allocator>>::type getDisplayModeProperties2KHR( DisplayKHR display, Dispatch const &d = Dispatch() ) const;
    template<typename Allocator = std::allocator<DisplayModeProperties2KHR>, typename Dispatch = DispatchLoaderStatic>
    typename ResultValueType<std::vector<DisplayModeProperties2KHR,Allocator>>::type getDisplayModeProperties2KHR( DisplayKHR display, Allocator const& vectorAllocator, Dispatch const &d ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    Result getDisplayModePropertiesKHR( DisplayKHR display, uint32_t* pPropertyCount, DisplayModePropertiesKHR* pProperties, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Allocator = std::allocator<DisplayModePropertiesKHR>, typename Dispatch = DispatchLoaderStatic>
    typename ResultValueType<std::vector<DisplayModePropertiesKHR,Allocator>>::type getDisplayModePropertiesKHR( DisplayKHR display, Dispatch const &d = Dispatch() ) const;
    template<typename Allocator = std::allocator<DisplayModePropertiesKHR>, typename Dispatch = DispatchLoaderStatic>
    typename ResultValueType<std::vector<DisplayModePropertiesKHR,Allocator>>::type getDisplayModePropertiesKHR( DisplayKHR display, Allocator const& vectorAllocator, Dispatch const &d ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    Result getDisplayPlaneCapabilities2KHR( const DisplayPlaneInfo2KHR* pDisplayPlaneInfo, DisplayPlaneCapabilities2KHR* pCapabilities, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    ResultValueType<DisplayPlaneCapabilities2KHR>::type getDisplayPlaneCapabilities2KHR( const DisplayPlaneInfo2KHR & displayPlaneInfo, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    Result getDisplayPlaneCapabilitiesKHR( DisplayModeKHR mode, uint32_t planeIndex, DisplayPlaneCapabilitiesKHR* pCapabilities, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    ResultValueType<DisplayPlaneCapabilitiesKHR>::type getDisplayPlaneCapabilitiesKHR( DisplayModeKHR mode, uint32_t planeIndex, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    Result getDisplayPlaneSupportedDisplaysKHR( uint32_t planeIndex, uint32_t* pDisplayCount, DisplayKHR* pDisplays, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Allocator = std::allocator<DisplayKHR>, typename Dispatch = DispatchLoaderStatic>
    typename ResultValueType<std::vector<DisplayKHR,Allocator>>::type getDisplayPlaneSupportedDisplaysKHR( uint32_t planeIndex, Dispatch const &d = Dispatch() ) const;
    template<typename Allocator = std::allocator<DisplayKHR>, typename Dispatch = DispatchLoaderStatic>
    typename ResultValueType<std::vector<DisplayKHR,Allocator>>::type getDisplayPlaneSupportedDisplaysKHR( uint32_t planeIndex, Allocator const& vectorAllocator, Dispatch const &d ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    Result getCalibrateableTimeDomainsEXT( uint32_t* pTimeDomainCount, TimeDomainEXT* pTimeDomains, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Allocator = std::allocator<TimeDomainEXT>, typename Dispatch = DispatchLoaderStatic>
    typename ResultValueType<std::vector<TimeDomainEXT,Allocator>>::type getCalibrateableTimeDomainsEXT(Dispatch const &d = Dispatch() ) const;
    template<typename Allocator = std::allocator<TimeDomainEXT>, typename Dispatch = DispatchLoaderStatic>
    typename ResultValueType<std::vector<TimeDomainEXT,Allocator>>::type getCalibrateableTimeDomainsEXT(Allocator const& vectorAllocator, Dispatch const &d ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    Result getCooperativeMatrixPropertiesNV( uint32_t* pPropertyCount, CooperativeMatrixPropertiesNV* pProperties, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Allocator = std::allocator<CooperativeMatrixPropertiesNV>, typename Dispatch = DispatchLoaderStatic>
    typename ResultValueType<std::vector<CooperativeMatrixPropertiesNV,Allocator>>::type getCooperativeMatrixPropertiesNV(Dispatch const &d = Dispatch() ) const;
    template<typename Allocator = std::allocator<CooperativeMatrixPropertiesNV>, typename Dispatch = DispatchLoaderStatic>
    typename ResultValueType<std::vector<CooperativeMatrixPropertiesNV,Allocator>>::type getCooperativeMatrixPropertiesNV(Allocator const& vectorAllocator, Dispatch const &d ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    Result getDisplayPlaneProperties2KHR( uint32_t* pPropertyCount, DisplayPlaneProperties2KHR* pProperties, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Allocator = std::allocator<DisplayPlaneProperties2KHR>, typename Dispatch = DispatchLoaderStatic>
    typename ResultValueType<std::vector<DisplayPlaneProperties2KHR,Allocator>>::type getDisplayPlaneProperties2KHR(Dispatch const &d = Dispatch() ) const;
    template<typename Allocator = std::allocator<DisplayPlaneProperties2KHR>, typename Dispatch = DispatchLoaderStatic>
    typename ResultValueType<std::vector<DisplayPlaneProperties2KHR,Allocator>>::type getDisplayPlaneProperties2KHR(Allocator const& vectorAllocator, Dispatch const &d ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    Result getDisplayPlanePropertiesKHR( uint32_t* pPropertyCount, DisplayPlanePropertiesKHR* pProperties, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Allocator = std::allocator<DisplayPlanePropertiesKHR>, typename Dispatch = DispatchLoaderStatic>
    typename ResultValueType<std::vector<DisplayPlanePropertiesKHR,Allocator>>::type getDisplayPlanePropertiesKHR(Dispatch const &d = Dispatch() ) const;
    template<typename Allocator = std::allocator<DisplayPlanePropertiesKHR>, typename Dispatch = DispatchLoaderStatic>
    typename ResultValueType<std::vector<DisplayPlanePropertiesKHR,Allocator>>::type getDisplayPlanePropertiesKHR(Allocator const& vectorAllocator, Dispatch const &d ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    Result getDisplayProperties2KHR( uint32_t* pPropertyCount, DisplayProperties2KHR* pProperties, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Allocator = std::allocator<DisplayProperties2KHR>, typename Dispatch = DispatchLoaderStatic>
    typename ResultValueType<std::vector<DisplayProperties2KHR,Allocator>>::type getDisplayProperties2KHR(Dispatch const &d = Dispatch() ) const;
    template<typename Allocator = std::allocator<DisplayProperties2KHR>, typename Dispatch = DispatchLoaderStatic>
    typename ResultValueType<std::vector<DisplayProperties2KHR,Allocator>>::type getDisplayProperties2KHR(Allocator const& vectorAllocator, Dispatch const &d ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    Result getDisplayPropertiesKHR( uint32_t* pPropertyCount, DisplayPropertiesKHR* pProperties, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Allocator = std::allocator<DisplayPropertiesKHR>, typename Dispatch = DispatchLoaderStatic>
    typename ResultValueType<std::vector<DisplayPropertiesKHR,Allocator>>::type getDisplayPropertiesKHR(Dispatch const &d = Dispatch() ) const;
    template<typename Allocator = std::allocator<DisplayPropertiesKHR>, typename Dispatch = DispatchLoaderStatic>
    typename ResultValueType<std::vector<DisplayPropertiesKHR,Allocator>>::type getDisplayPropertiesKHR(Allocator const& vectorAllocator, Dispatch const &d ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    void getExternalBufferProperties( const PhysicalDeviceExternalBufferInfo* pExternalBufferInfo, ExternalBufferProperties* pExternalBufferProperties, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    ExternalBufferProperties getExternalBufferProperties( const PhysicalDeviceExternalBufferInfo & externalBufferInfo, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    void getExternalBufferPropertiesKHR( const PhysicalDeviceExternalBufferInfo* pExternalBufferInfo, ExternalBufferProperties* pExternalBufferProperties, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    ExternalBufferProperties getExternalBufferPropertiesKHR( const PhysicalDeviceExternalBufferInfo & externalBufferInfo, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    void getExternalFenceProperties( const PhysicalDeviceExternalFenceInfo* pExternalFenceInfo, ExternalFenceProperties* pExternalFenceProperties, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    ExternalFenceProperties getExternalFenceProperties( const PhysicalDeviceExternalFenceInfo & externalFenceInfo, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    void getExternalFencePropertiesKHR( const PhysicalDeviceExternalFenceInfo* pExternalFenceInfo, ExternalFenceProperties* pExternalFenceProperties, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    ExternalFenceProperties getExternalFencePropertiesKHR( const PhysicalDeviceExternalFenceInfo & externalFenceInfo, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    Result getExternalImageFormatPropertiesNV( Format format, ImageType type, ImageTiling tiling, ImageUsageFlags usage, ImageCreateFlags flags, ExternalMemoryHandleTypeFlagsNV externalHandleType, ExternalImageFormatPropertiesNV* pExternalImageFormatProperties, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    ResultValueType<ExternalImageFormatPropertiesNV>::type getExternalImageFormatPropertiesNV( Format format, ImageType type, ImageTiling tiling, ImageUsageFlags usage, ImageCreateFlags flags, ExternalMemoryHandleTypeFlagsNV externalHandleType, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    void getExternalSemaphoreProperties( const PhysicalDeviceExternalSemaphoreInfo* pExternalSemaphoreInfo, ExternalSemaphoreProperties* pExternalSemaphoreProperties, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    ExternalSemaphoreProperties getExternalSemaphoreProperties( const PhysicalDeviceExternalSemaphoreInfo & externalSemaphoreInfo, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    void getExternalSemaphorePropertiesKHR( const PhysicalDeviceExternalSemaphoreInfo* pExternalSemaphoreInfo, ExternalSemaphoreProperties* pExternalSemaphoreProperties, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    ExternalSemaphoreProperties getExternalSemaphorePropertiesKHR( const PhysicalDeviceExternalSemaphoreInfo & externalSemaphoreInfo, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    void getFeatures( PhysicalDeviceFeatures* pFeatures, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    PhysicalDeviceFeatures getFeatures(Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    void getFeatures2( PhysicalDeviceFeatures2* pFeatures, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    PhysicalDeviceFeatures2 getFeatures2(Dispatch const &d = Dispatch() ) const;
    template<typename X, typename Y, typename ...Z, typename Dispatch = DispatchLoaderStatic>
    StructureChain<X, Y, Z...> getFeatures2(Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    void getFeatures2KHR( PhysicalDeviceFeatures2* pFeatures, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    PhysicalDeviceFeatures2 getFeatures2KHR(Dispatch const &d = Dispatch() ) const;
    template<typename X, typename Y, typename ...Z, typename Dispatch = DispatchLoaderStatic>
    StructureChain<X, Y, Z...> getFeatures2KHR(Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    void getFormatProperties( Format format, FormatProperties* pFormatProperties, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    FormatProperties getFormatProperties( Format format, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    void getFormatProperties2( Format format, FormatProperties2* pFormatProperties, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    FormatProperties2 getFormatProperties2( Format format, Dispatch const &d = Dispatch() ) const;
    template<typename X, typename Y, typename ...Z, typename Dispatch = DispatchLoaderStatic>
    StructureChain<X, Y, Z...> getFormatProperties2( Format format, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    void getFormatProperties2KHR( Format format, FormatProperties2* pFormatProperties, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    FormatProperties2 getFormatProperties2KHR( Format format, Dispatch const &d = Dispatch() ) const;
    template<typename X, typename Y, typename ...Z, typename Dispatch = DispatchLoaderStatic>
    StructureChain<X, Y, Z...> getFormatProperties2KHR( Format format, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    void getGeneratedCommandsPropertiesNVX( DeviceGeneratedCommandsFeaturesNVX* pFeatures, DeviceGeneratedCommandsLimitsNVX* pLimits, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    DeviceGeneratedCommandsLimitsNVX getGeneratedCommandsPropertiesNVX( DeviceGeneratedCommandsFeaturesNVX & features, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    Result getImageFormatProperties( Format format, ImageType type, ImageTiling tiling, ImageUsageFlags usage, ImageCreateFlags flags, ImageFormatProperties* pImageFormatProperties, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    ResultValueType<ImageFormatProperties>::type getImageFormatProperties( Format format, ImageType type, ImageTiling tiling, ImageUsageFlags usage, ImageCreateFlags flags, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    Result getImageFormatProperties2( const PhysicalDeviceImageFormatInfo2* pImageFormatInfo, ImageFormatProperties2* pImageFormatProperties, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    ResultValueType<ImageFormatProperties2>::type getImageFormatProperties2( const PhysicalDeviceImageFormatInfo2 & imageFormatInfo, Dispatch const &d = Dispatch() ) const;
    template<typename X, typename Y, typename ...Z, typename Dispatch = DispatchLoaderStatic>
    typename ResultValueType<StructureChain<X, Y, Z...>>::type getImageFormatProperties2( const PhysicalDeviceImageFormatInfo2 & imageFormatInfo, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    Result getImageFormatProperties2KHR( const PhysicalDeviceImageFormatInfo2* pImageFormatInfo, ImageFormatProperties2* pImageFormatProperties, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    ResultValueType<ImageFormatProperties2>::type getImageFormatProperties2KHR( const PhysicalDeviceImageFormatInfo2 & imageFormatInfo, Dispatch const &d = Dispatch() ) const;
    template<typename X, typename Y, typename ...Z, typename Dispatch = DispatchLoaderStatic>
    typename ResultValueType<StructureChain<X, Y, Z...>>::type getImageFormatProperties2KHR( const PhysicalDeviceImageFormatInfo2 & imageFormatInfo, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    void getMemoryProperties( PhysicalDeviceMemoryProperties* pMemoryProperties, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    PhysicalDeviceMemoryProperties getMemoryProperties(Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    void getMemoryProperties2( PhysicalDeviceMemoryProperties2* pMemoryProperties, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    PhysicalDeviceMemoryProperties2 getMemoryProperties2(Dispatch const &d = Dispatch() ) const;
    template<typename X, typename Y, typename ...Z, typename Dispatch = DispatchLoaderStatic>
    StructureChain<X, Y, Z...> getMemoryProperties2(Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    void getMemoryProperties2KHR( PhysicalDeviceMemoryProperties2* pMemoryProperties, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    PhysicalDeviceMemoryProperties2 getMemoryProperties2KHR(Dispatch const &d = Dispatch() ) const;
    template<typename X, typename Y, typename ...Z, typename Dispatch = DispatchLoaderStatic>
    StructureChain<X, Y, Z...> getMemoryProperties2KHR(Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    void getMultisamplePropertiesEXT( SampleCountFlagBits samples, MultisamplePropertiesEXT* pMultisampleProperties, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    MultisamplePropertiesEXT getMultisamplePropertiesEXT( SampleCountFlagBits samples, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    Result getPresentRectanglesKHR( SurfaceKHR surface, uint32_t* pRectCount, Rect2D* pRects, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Allocator = std::allocator<Rect2D>, typename Dispatch = DispatchLoaderStatic>
    typename ResultValueType<std::vector<Rect2D,Allocator>>::type getPresentRectanglesKHR( SurfaceKHR surface, Dispatch const &d = Dispatch() ) const;
    template<typename Allocator = std::allocator<Rect2D>, typename Dispatch = DispatchLoaderStatic>
    typename ResultValueType<std::vector<Rect2D,Allocator>>::type getPresentRectanglesKHR( SurfaceKHR surface, Allocator const& vectorAllocator, Dispatch const &d ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    void getProperties( PhysicalDeviceProperties* pProperties, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    PhysicalDeviceProperties getProperties(Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    void getProperties2( PhysicalDeviceProperties2* pProperties, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    PhysicalDeviceProperties2 getProperties2(Dispatch const &d = Dispatch() ) const;
    template<typename X, typename Y, typename ...Z, typename Dispatch = DispatchLoaderStatic>
    StructureChain<X, Y, Z...> getProperties2(Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    void getProperties2KHR( PhysicalDeviceProperties2* pProperties, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    PhysicalDeviceProperties2 getProperties2KHR(Dispatch const &d = Dispatch() ) const;
    template<typename X, typename Y, typename ...Z, typename Dispatch = DispatchLoaderStatic>
    StructureChain<X, Y, Z...> getProperties2KHR(Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    void getQueueFamilyProperties( uint32_t* pQueueFamilyPropertyCount, QueueFamilyProperties* pQueueFamilyProperties, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Allocator = std::allocator<QueueFamilyProperties>, typename Dispatch = DispatchLoaderStatic>
    std::vector<QueueFamilyProperties,Allocator> getQueueFamilyProperties(Dispatch const &d = Dispatch() ) const;
    template<typename Allocator = std::allocator<QueueFamilyProperties>, typename Dispatch = DispatchLoaderStatic>
    std::vector<QueueFamilyProperties,Allocator> getQueueFamilyProperties(Allocator const& vectorAllocator, Dispatch const &d ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    void getQueueFamilyProperties2( uint32_t* pQueueFamilyPropertyCount, QueueFamilyProperties2* pQueueFamilyProperties, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Allocator = std::allocator<QueueFamilyProperties2>, typename Dispatch = DispatchLoaderStatic>
    std::vector<QueueFamilyProperties2,Allocator> getQueueFamilyProperties2(Dispatch const &d = Dispatch() ) const;
    template<typename Allocator = std::allocator<QueueFamilyProperties2>, typename Dispatch = DispatchLoaderStatic>
    std::vector<QueueFamilyProperties2,Allocator> getQueueFamilyProperties2(Allocator const& vectorAllocator, Dispatch const &d ) const;
    template<typename StructureChain, typename Allocator = std::allocator<StructureChain>, typename Dispatch = DispatchLoaderStatic>
    std::vector<StructureChain,Allocator> getQueueFamilyProperties2(Dispatch const &d = Dispatch() ) const;
    template<typename StructureChain, typename Allocator = std::allocator<StructureChain>, typename Dispatch = DispatchLoaderStatic>
    std::vector<StructureChain,Allocator> getQueueFamilyProperties2(Allocator const& vectorAllocator, Dispatch const &d ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    void getQueueFamilyProperties2KHR( uint32_t* pQueueFamilyPropertyCount, QueueFamilyProperties2* pQueueFamilyProperties, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Allocator = std::allocator<QueueFamilyProperties2>, typename Dispatch = DispatchLoaderStatic>
    std::vector<QueueFamilyProperties2,Allocator> getQueueFamilyProperties2KHR(Dispatch const &d = Dispatch() ) const;
    template<typename Allocator = std::allocator<QueueFamilyProperties2>, typename Dispatch = DispatchLoaderStatic>
    std::vector<QueueFamilyProperties2,Allocator> getQueueFamilyProperties2KHR(Allocator const& vectorAllocator, Dispatch const &d ) const;
    template<typename StructureChain, typename Allocator = std::allocator<StructureChain>, typename Dispatch = DispatchLoaderStatic>
    std::vector<StructureChain,Allocator> getQueueFamilyProperties2KHR(Dispatch const &d = Dispatch() ) const;
    template<typename StructureChain, typename Allocator = std::allocator<StructureChain>, typename Dispatch = DispatchLoaderStatic>
    std::vector<StructureChain,Allocator> getQueueFamilyProperties2KHR(Allocator const& vectorAllocator, Dispatch const &d ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    void getSparseImageFormatProperties( Format format, ImageType type, SampleCountFlagBits samples, ImageUsageFlags usage, ImageTiling tiling, uint32_t* pPropertyCount, SparseImageFormatProperties* pProperties, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Allocator = std::allocator<SparseImageFormatProperties>, typename Dispatch = DispatchLoaderStatic>
    std::vector<SparseImageFormatProperties,Allocator> getSparseImageFormatProperties( Format format, ImageType type, SampleCountFlagBits samples, ImageUsageFlags usage, ImageTiling tiling, Dispatch const &d = Dispatch() ) const;
    template<typename Allocator = std::allocator<SparseImageFormatProperties>, typename Dispatch = DispatchLoaderStatic>
    std::vector<SparseImageFormatProperties,Allocator> getSparseImageFormatProperties( Format format, ImageType type, SampleCountFlagBits samples, ImageUsageFlags usage, ImageTiling tiling, Allocator const& vectorAllocator, Dispatch const &d ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    void getSparseImageFormatProperties2( const PhysicalDeviceSparseImageFormatInfo2* pFormatInfo, uint32_t* pPropertyCount, SparseImageFormatProperties2* pProperties, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Allocator = std::allocator<SparseImageFormatProperties2>, typename Dispatch = DispatchLoaderStatic>
    std::vector<SparseImageFormatProperties2,Allocator> getSparseImageFormatProperties2( const PhysicalDeviceSparseImageFormatInfo2 & formatInfo, Dispatch const &d = Dispatch() ) const;
    template<typename Allocator = std::allocator<SparseImageFormatProperties2>, typename Dispatch = DispatchLoaderStatic>
    std::vector<SparseImageFormatProperties2,Allocator> getSparseImageFormatProperties2( const PhysicalDeviceSparseImageFormatInfo2 & formatInfo, Allocator const& vectorAllocator, Dispatch const &d ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    void getSparseImageFormatProperties2KHR( const PhysicalDeviceSparseImageFormatInfo2* pFormatInfo, uint32_t* pPropertyCount, SparseImageFormatProperties2* pProperties, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Allocator = std::allocator<SparseImageFormatProperties2>, typename Dispatch = DispatchLoaderStatic>
    std::vector<SparseImageFormatProperties2,Allocator> getSparseImageFormatProperties2KHR( const PhysicalDeviceSparseImageFormatInfo2 & formatInfo, Dispatch const &d = Dispatch() ) const;
    template<typename Allocator = std::allocator<SparseImageFormatProperties2>, typename Dispatch = DispatchLoaderStatic>
    std::vector<SparseImageFormatProperties2,Allocator> getSparseImageFormatProperties2KHR( const PhysicalDeviceSparseImageFormatInfo2 & formatInfo, Allocator const& vectorAllocator, Dispatch const &d ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    Result getSurfaceCapabilities2EXT( SurfaceKHR surface, SurfaceCapabilities2EXT* pSurfaceCapabilities, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    ResultValueType<SurfaceCapabilities2EXT>::type getSurfaceCapabilities2EXT( SurfaceKHR surface, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    Result getSurfaceCapabilities2KHR( const PhysicalDeviceSurfaceInfo2KHR* pSurfaceInfo, SurfaceCapabilities2KHR* pSurfaceCapabilities, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    ResultValueType<SurfaceCapabilities2KHR>::type getSurfaceCapabilities2KHR( const PhysicalDeviceSurfaceInfo2KHR & surfaceInfo, Dispatch const &d = Dispatch() ) const;
    template<typename X, typename Y, typename ...Z, typename Dispatch = DispatchLoaderStatic>
    typename ResultValueType<StructureChain<X, Y, Z...>>::type getSurfaceCapabilities2KHR( const PhysicalDeviceSurfaceInfo2KHR & surfaceInfo, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    Result getSurfaceCapabilitiesKHR( SurfaceKHR surface, SurfaceCapabilitiesKHR* pSurfaceCapabilities, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    ResultValueType<SurfaceCapabilitiesKHR>::type getSurfaceCapabilitiesKHR( SurfaceKHR surface, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    Result getSurfaceFormats2KHR( const PhysicalDeviceSurfaceInfo2KHR* pSurfaceInfo, uint32_t* pSurfaceFormatCount, SurfaceFormat2KHR* pSurfaceFormats, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Allocator = std::allocator<SurfaceFormat2KHR>, typename Dispatch = DispatchLoaderStatic>
    typename ResultValueType<std::vector<SurfaceFormat2KHR,Allocator>>::type getSurfaceFormats2KHR( const PhysicalDeviceSurfaceInfo2KHR & surfaceInfo, Dispatch const &d = Dispatch() ) const;
    template<typename Allocator = std::allocator<SurfaceFormat2KHR>, typename Dispatch = DispatchLoaderStatic>
    typename ResultValueType<std::vector<SurfaceFormat2KHR,Allocator>>::type getSurfaceFormats2KHR( const PhysicalDeviceSurfaceInfo2KHR & surfaceInfo, Allocator const& vectorAllocator, Dispatch const &d ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    Result getSurfaceFormatsKHR( SurfaceKHR surface, uint32_t* pSurfaceFormatCount, SurfaceFormatKHR* pSurfaceFormats, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Allocator = std::allocator<SurfaceFormatKHR>, typename Dispatch = DispatchLoaderStatic>
    typename ResultValueType<std::vector<SurfaceFormatKHR,Allocator>>::type getSurfaceFormatsKHR( SurfaceKHR surface, Dispatch const &d = Dispatch() ) const;
    template<typename Allocator = std::allocator<SurfaceFormatKHR>, typename Dispatch = DispatchLoaderStatic>
    typename ResultValueType<std::vector<SurfaceFormatKHR,Allocator>>::type getSurfaceFormatsKHR( SurfaceKHR surface, Allocator const& vectorAllocator, Dispatch const &d ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    Result getSurfacePresentModesKHR( SurfaceKHR surface, uint32_t* pPresentModeCount, PresentModeKHR* pPresentModes, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Allocator = std::allocator<PresentModeKHR>, typename Dispatch = DispatchLoaderStatic>
    typename ResultValueType<std::vector<PresentModeKHR,Allocator>>::type getSurfacePresentModesKHR( SurfaceKHR surface, Dispatch const &d = Dispatch() ) const;
    template<typename Allocator = std::allocator<PresentModeKHR>, typename Dispatch = DispatchLoaderStatic>
    typename ResultValueType<std::vector<PresentModeKHR,Allocator>>::type getSurfacePresentModesKHR( SurfaceKHR surface, Allocator const& vectorAllocator, Dispatch const &d ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    Result getSurfaceSupportKHR( uint32_t queueFamilyIndex, SurfaceKHR surface, Bool32* pSupported, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    ResultValueType<Bool32>::type getSurfaceSupportKHR( uint32_t queueFamilyIndex, SurfaceKHR surface, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

#ifdef VK_USE_PLATFORM_WAYLAND_KHR
    template<typename Dispatch = DispatchLoaderStatic>
    Bool32 getWaylandPresentationSupportKHR( uint32_t queueFamilyIndex, struct wl_display* display, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    Bool32 getWaylandPresentationSupportKHR( uint32_t queueFamilyIndex, struct wl_display & display, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/
#endif /*VK_USE_PLATFORM_WAYLAND_KHR*/

#ifdef VK_USE_PLATFORM_WIN32_KHR
    template<typename Dispatch = DispatchLoaderStatic>
    Bool32 getWin32PresentationSupportKHR( uint32_t queueFamilyIndex, Dispatch const &d = Dispatch() ) const;
#endif /*VK_USE_PLATFORM_WIN32_KHR*/

#ifdef VK_USE_PLATFORM_XCB_KHR
    template<typename Dispatch = DispatchLoaderStatic>
    Bool32 getXcbPresentationSupportKHR( uint32_t queueFamilyIndex, xcb_connection_t* connection, xcb_visualid_t visual_id, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    Bool32 getXcbPresentationSupportKHR( uint32_t queueFamilyIndex, xcb_connection_t & connection, xcb_visualid_t visual_id, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/
#endif /*VK_USE_PLATFORM_XCB_KHR*/

#ifdef VK_USE_PLATFORM_XLIB_KHR
    template<typename Dispatch = DispatchLoaderStatic>
    Bool32 getXlibPresentationSupportKHR( uint32_t queueFamilyIndex, Display* dpy, VisualID visualID, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    Bool32 getXlibPresentationSupportKHR( uint32_t queueFamilyIndex, Display & dpy, VisualID visualID, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/
#endif /*VK_USE_PLATFORM_XLIB_KHR*/

#ifdef VK_USE_PLATFORM_XLIB_XRANDR_EXT
    template<typename Dispatch = DispatchLoaderStatic>
    Result getRandROutputDisplayEXT( Display* dpy, RROutput rrOutput, DisplayKHR* pDisplay, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    ResultValueType<DisplayKHR>::type getRandROutputDisplayEXT( Display & dpy, RROutput rrOutput, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/
#endif /*VK_USE_PLATFORM_XLIB_XRANDR_EXT*/

#ifdef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    Result releaseDisplayEXT( DisplayKHR display, Dispatch const &d = Dispatch() ) const;
#else
    template<typename Dispatch = DispatchLoaderStatic>
    ResultValueType<void>::type releaseDisplayEXT( DisplayKHR display, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    VULKAN_HPP_TYPESAFE_EXPLICIT operator VkPhysicalDevice() const
    {
      return m_physicalDevice;
    }

    explicit operator bool() const
    {
      return m_physicalDevice != VK_NULL_HANDLE;
    }

    bool operator!() const
    {
      return m_physicalDevice == VK_NULL_HANDLE;
    }

  private:
    VkPhysicalDevice m_physicalDevice;
  };
  static_assert( sizeof( PhysicalDevice ) == sizeof( VkPhysicalDevice ), "handle and wrapper have different size!" );

#ifndef VULKAN_HPP_NO_SMART_HANDLE
  class Instance;
  template <typename Dispatch> class UniqueHandleTraits<DebugReportCallbackEXT, Dispatch> { public: using deleter = ObjectDestroy<Instance, Dispatch>; };
  using UniqueDebugReportCallbackEXT = UniqueHandle<DebugReportCallbackEXT, DispatchLoaderStatic>;
  template <typename Dispatch> class UniqueHandleTraits<DebugUtilsMessengerEXT, Dispatch> { public: using deleter = ObjectDestroy<Instance, Dispatch>; };
  using UniqueDebugUtilsMessengerEXT = UniqueHandle<DebugUtilsMessengerEXT, DispatchLoaderStatic>;
  template <typename Dispatch> class UniqueHandleTraits<SurfaceKHR, Dispatch> { public: using deleter = ObjectDestroy<Instance, Dispatch>; };
  using UniqueSurfaceKHR = UniqueHandle<SurfaceKHR, DispatchLoaderStatic>;
#endif /*VULKAN_HPP_NO_SMART_HANDLE*/

  class Instance
  {
  public:
    VULKAN_HPP_CONSTEXPR Instance()
      : m_instance(VK_NULL_HANDLE)
    {}

    VULKAN_HPP_CONSTEXPR Instance( std::nullptr_t )
      : m_instance(VK_NULL_HANDLE)
    {}

    VULKAN_HPP_TYPESAFE_EXPLICIT Instance( VkInstance instance )
      : m_instance( instance )
    {}

#if defined(VULKAN_HPP_TYPESAFE_CONVERSION)
    Instance & operator=(VkInstance instance)
    {
      m_instance = instance;
      return *this; 
    }
#endif

    Instance & operator=( std::nullptr_t )
    {
      m_instance = VK_NULL_HANDLE;
      return *this;
    }

    bool operator==( Instance const & rhs ) const
    {
      return m_instance == rhs.m_instance;
    }

    bool operator!=(Instance const & rhs ) const
    {
      return m_instance != rhs.m_instance;
    }

    bool operator<(Instance const & rhs ) const
    {
      return m_instance < rhs.m_instance;
    }

#ifdef VK_USE_PLATFORM_ANDROID_KHR
    template<typename Dispatch = DispatchLoaderStatic>
    Result createAndroidSurfaceKHR( const AndroidSurfaceCreateInfoKHR* pCreateInfo, const AllocationCallbacks* pAllocator, SurfaceKHR* pSurface, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    ResultValueType<SurfaceKHR>::type createAndroidSurfaceKHR( const AndroidSurfaceCreateInfoKHR & createInfo, Optional<const AllocationCallbacks> allocator = nullptr, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_NO_SMART_HANDLE
    template<typename Dispatch = DispatchLoaderStatic>
    typename ResultValueType<UniqueHandle<SurfaceKHR,Dispatch>>::type createAndroidSurfaceKHRUnique( const AndroidSurfaceCreateInfoKHR & createInfo, Optional<const AllocationCallbacks> allocator = nullptr, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_NO_SMART_HANDLE*/
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/
#endif /*VK_USE_PLATFORM_ANDROID_KHR*/

    template<typename Dispatch = DispatchLoaderStatic>
    Result createDebugReportCallbackEXT( const DebugReportCallbackCreateInfoEXT* pCreateInfo, const AllocationCallbacks* pAllocator, DebugReportCallbackEXT* pCallback, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    ResultValueType<DebugReportCallbackEXT>::type createDebugReportCallbackEXT( const DebugReportCallbackCreateInfoEXT & createInfo, Optional<const AllocationCallbacks> allocator = nullptr, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_NO_SMART_HANDLE
    template<typename Dispatch = DispatchLoaderStatic>
    typename ResultValueType<UniqueHandle<DebugReportCallbackEXT,Dispatch>>::type createDebugReportCallbackEXTUnique( const DebugReportCallbackCreateInfoEXT & createInfo, Optional<const AllocationCallbacks> allocator = nullptr, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_NO_SMART_HANDLE*/
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    Result createDebugUtilsMessengerEXT( const DebugUtilsMessengerCreateInfoEXT* pCreateInfo, const AllocationCallbacks* pAllocator, DebugUtilsMessengerEXT* pMessenger, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    ResultValueType<DebugUtilsMessengerEXT>::type createDebugUtilsMessengerEXT( const DebugUtilsMessengerCreateInfoEXT & createInfo, Optional<const AllocationCallbacks> allocator = nullptr, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_NO_SMART_HANDLE
    template<typename Dispatch = DispatchLoaderStatic>
    typename ResultValueType<UniqueHandle<DebugUtilsMessengerEXT,Dispatch>>::type createDebugUtilsMessengerEXTUnique( const DebugUtilsMessengerCreateInfoEXT & createInfo, Optional<const AllocationCallbacks> allocator = nullptr, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_NO_SMART_HANDLE*/
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    Result createDisplayPlaneSurfaceKHR( const DisplaySurfaceCreateInfoKHR* pCreateInfo, const AllocationCallbacks* pAllocator, SurfaceKHR* pSurface, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    ResultValueType<SurfaceKHR>::type createDisplayPlaneSurfaceKHR( const DisplaySurfaceCreateInfoKHR & createInfo, Optional<const AllocationCallbacks> allocator = nullptr, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_NO_SMART_HANDLE
    template<typename Dispatch = DispatchLoaderStatic>
    typename ResultValueType<UniqueHandle<SurfaceKHR,Dispatch>>::type createDisplayPlaneSurfaceKHRUnique( const DisplaySurfaceCreateInfoKHR & createInfo, Optional<const AllocationCallbacks> allocator = nullptr, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_NO_SMART_HANDLE*/
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

#ifdef VK_USE_PLATFORM_IOS_MVK
    template<typename Dispatch = DispatchLoaderStatic>
    Result createIOSSurfaceMVK( const IOSSurfaceCreateInfoMVK* pCreateInfo, const AllocationCallbacks* pAllocator, SurfaceKHR* pSurface, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    ResultValueType<SurfaceKHR>::type createIOSSurfaceMVK( const IOSSurfaceCreateInfoMVK & createInfo, Optional<const AllocationCallbacks> allocator = nullptr, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_NO_SMART_HANDLE
    template<typename Dispatch = DispatchLoaderStatic>
    typename ResultValueType<UniqueHandle<SurfaceKHR,Dispatch>>::type createIOSSurfaceMVKUnique( const IOSSurfaceCreateInfoMVK & createInfo, Optional<const AllocationCallbacks> allocator = nullptr, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_NO_SMART_HANDLE*/
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/
#endif /*VK_USE_PLATFORM_IOS_MVK*/

#ifdef VK_USE_PLATFORM_FUCHSIA
    template<typename Dispatch = DispatchLoaderStatic>
    Result createImagePipeSurfaceFUCHSIA( const ImagePipeSurfaceCreateInfoFUCHSIA* pCreateInfo, const AllocationCallbacks* pAllocator, SurfaceKHR* pSurface, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    ResultValueType<SurfaceKHR>::type createImagePipeSurfaceFUCHSIA( const ImagePipeSurfaceCreateInfoFUCHSIA & createInfo, Optional<const AllocationCallbacks> allocator = nullptr, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_NO_SMART_HANDLE
    template<typename Dispatch = DispatchLoaderStatic>
    typename ResultValueType<UniqueHandle<SurfaceKHR,Dispatch>>::type createImagePipeSurfaceFUCHSIAUnique( const ImagePipeSurfaceCreateInfoFUCHSIA & createInfo, Optional<const AllocationCallbacks> allocator = nullptr, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_NO_SMART_HANDLE*/
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/
#endif /*VK_USE_PLATFORM_FUCHSIA*/

#ifdef VK_USE_PLATFORM_MACOS_MVK
    template<typename Dispatch = DispatchLoaderStatic>
    Result createMacOSSurfaceMVK( const MacOSSurfaceCreateInfoMVK* pCreateInfo, const AllocationCallbacks* pAllocator, SurfaceKHR* pSurface, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    ResultValueType<SurfaceKHR>::type createMacOSSurfaceMVK( const MacOSSurfaceCreateInfoMVK & createInfo, Optional<const AllocationCallbacks> allocator = nullptr, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_NO_SMART_HANDLE
    template<typename Dispatch = DispatchLoaderStatic>
    typename ResultValueType<UniqueHandle<SurfaceKHR,Dispatch>>::type createMacOSSurfaceMVKUnique( const MacOSSurfaceCreateInfoMVK & createInfo, Optional<const AllocationCallbacks> allocator = nullptr, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_NO_SMART_HANDLE*/
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/
#endif /*VK_USE_PLATFORM_MACOS_MVK*/

#ifdef VK_USE_PLATFORM_VI_NN
    template<typename Dispatch = DispatchLoaderStatic>
    Result createViSurfaceNN( const ViSurfaceCreateInfoNN* pCreateInfo, const AllocationCallbacks* pAllocator, SurfaceKHR* pSurface, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    ResultValueType<SurfaceKHR>::type createViSurfaceNN( const ViSurfaceCreateInfoNN & createInfo, Optional<const AllocationCallbacks> allocator = nullptr, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_NO_SMART_HANDLE
    template<typename Dispatch = DispatchLoaderStatic>
    typename ResultValueType<UniqueHandle<SurfaceKHR,Dispatch>>::type createViSurfaceNNUnique( const ViSurfaceCreateInfoNN & createInfo, Optional<const AllocationCallbacks> allocator = nullptr, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_NO_SMART_HANDLE*/
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/
#endif /*VK_USE_PLATFORM_VI_NN*/

#ifdef VK_USE_PLATFORM_WAYLAND_KHR
    template<typename Dispatch = DispatchLoaderStatic>
    Result createWaylandSurfaceKHR( const WaylandSurfaceCreateInfoKHR* pCreateInfo, const AllocationCallbacks* pAllocator, SurfaceKHR* pSurface, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    ResultValueType<SurfaceKHR>::type createWaylandSurfaceKHR( const WaylandSurfaceCreateInfoKHR & createInfo, Optional<const AllocationCallbacks> allocator = nullptr, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_NO_SMART_HANDLE
    template<typename Dispatch = DispatchLoaderStatic>
    typename ResultValueType<UniqueHandle<SurfaceKHR,Dispatch>>::type createWaylandSurfaceKHRUnique( const WaylandSurfaceCreateInfoKHR & createInfo, Optional<const AllocationCallbacks> allocator = nullptr, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_NO_SMART_HANDLE*/
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/
#endif /*VK_USE_PLATFORM_WAYLAND_KHR*/

#ifdef VK_USE_PLATFORM_WIN32_KHR
    template<typename Dispatch = DispatchLoaderStatic>
    Result createWin32SurfaceKHR( const Win32SurfaceCreateInfoKHR* pCreateInfo, const AllocationCallbacks* pAllocator, SurfaceKHR* pSurface, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    ResultValueType<SurfaceKHR>::type createWin32SurfaceKHR( const Win32SurfaceCreateInfoKHR & createInfo, Optional<const AllocationCallbacks> allocator = nullptr, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_NO_SMART_HANDLE
    template<typename Dispatch = DispatchLoaderStatic>
    typename ResultValueType<UniqueHandle<SurfaceKHR,Dispatch>>::type createWin32SurfaceKHRUnique( const Win32SurfaceCreateInfoKHR & createInfo, Optional<const AllocationCallbacks> allocator = nullptr, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_NO_SMART_HANDLE*/
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/
#endif /*VK_USE_PLATFORM_WIN32_KHR*/

#ifdef VK_USE_PLATFORM_XCB_KHR
    template<typename Dispatch = DispatchLoaderStatic>
    Result createXcbSurfaceKHR( const XcbSurfaceCreateInfoKHR* pCreateInfo, const AllocationCallbacks* pAllocator, SurfaceKHR* pSurface, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    ResultValueType<SurfaceKHR>::type createXcbSurfaceKHR( const XcbSurfaceCreateInfoKHR & createInfo, Optional<const AllocationCallbacks> allocator = nullptr, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_NO_SMART_HANDLE
    template<typename Dispatch = DispatchLoaderStatic>
    typename ResultValueType<UniqueHandle<SurfaceKHR,Dispatch>>::type createXcbSurfaceKHRUnique( const XcbSurfaceCreateInfoKHR & createInfo, Optional<const AllocationCallbacks> allocator = nullptr, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_NO_SMART_HANDLE*/
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/
#endif /*VK_USE_PLATFORM_XCB_KHR*/

#ifdef VK_USE_PLATFORM_XLIB_KHR
    template<typename Dispatch = DispatchLoaderStatic>
    Result createXlibSurfaceKHR( const XlibSurfaceCreateInfoKHR* pCreateInfo, const AllocationCallbacks* pAllocator, SurfaceKHR* pSurface, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    ResultValueType<SurfaceKHR>::type createXlibSurfaceKHR( const XlibSurfaceCreateInfoKHR & createInfo, Optional<const AllocationCallbacks> allocator = nullptr, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_NO_SMART_HANDLE
    template<typename Dispatch = DispatchLoaderStatic>
    typename ResultValueType<UniqueHandle<SurfaceKHR,Dispatch>>::type createXlibSurfaceKHRUnique( const XlibSurfaceCreateInfoKHR & createInfo, Optional<const AllocationCallbacks> allocator = nullptr, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_NO_SMART_HANDLE*/
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/
#endif /*VK_USE_PLATFORM_XLIB_KHR*/

    template<typename Dispatch = DispatchLoaderStatic>
    void debugReportMessageEXT( DebugReportFlagsEXT flags, DebugReportObjectTypeEXT objectType, uint64_t object, size_t location, int32_t messageCode, const char* pLayerPrefix, const char* pMessage, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    void debugReportMessageEXT( DebugReportFlagsEXT flags, DebugReportObjectTypeEXT objectType, uint64_t object, size_t location, int32_t messageCode, const std::string & layerPrefix, const std::string & message, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    void destroyDebugReportCallbackEXT( DebugReportCallbackEXT callback, const AllocationCallbacks* pAllocator, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    void destroyDebugReportCallbackEXT( DebugReportCallbackEXT callback, Optional<const AllocationCallbacks> allocator = nullptr, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    void destroy( DebugReportCallbackEXT callback, const AllocationCallbacks* pAllocator, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    void destroy( DebugReportCallbackEXT callback, Optional<const AllocationCallbacks> allocator = nullptr, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    void destroyDebugUtilsMessengerEXT( DebugUtilsMessengerEXT messenger, const AllocationCallbacks* pAllocator, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    void destroyDebugUtilsMessengerEXT( DebugUtilsMessengerEXT messenger, Optional<const AllocationCallbacks> allocator = nullptr, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    void destroy( DebugUtilsMessengerEXT messenger, const AllocationCallbacks* pAllocator, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    void destroy( DebugUtilsMessengerEXT messenger, Optional<const AllocationCallbacks> allocator = nullptr, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    void destroy( const AllocationCallbacks* pAllocator, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    void destroy( Optional<const AllocationCallbacks> allocator = nullptr, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    void destroySurfaceKHR( SurfaceKHR surface, const AllocationCallbacks* pAllocator, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    void destroySurfaceKHR( SurfaceKHR surface, Optional<const AllocationCallbacks> allocator = nullptr, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    void destroy( SurfaceKHR surface, const AllocationCallbacks* pAllocator, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    void destroy( SurfaceKHR surface, Optional<const AllocationCallbacks> allocator = nullptr, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    Result enumeratePhysicalDeviceGroups( uint32_t* pPhysicalDeviceGroupCount, PhysicalDeviceGroupProperties* pPhysicalDeviceGroupProperties, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Allocator = std::allocator<PhysicalDeviceGroupProperties>, typename Dispatch = DispatchLoaderStatic>
    typename ResultValueType<std::vector<PhysicalDeviceGroupProperties,Allocator>>::type enumeratePhysicalDeviceGroups(Dispatch const &d = Dispatch() ) const;
    template<typename Allocator = std::allocator<PhysicalDeviceGroupProperties>, typename Dispatch = DispatchLoaderStatic>
    typename ResultValueType<std::vector<PhysicalDeviceGroupProperties,Allocator>>::type enumeratePhysicalDeviceGroups(Allocator const& vectorAllocator, Dispatch const &d ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    Result enumeratePhysicalDeviceGroupsKHR( uint32_t* pPhysicalDeviceGroupCount, PhysicalDeviceGroupProperties* pPhysicalDeviceGroupProperties, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Allocator = std::allocator<PhysicalDeviceGroupProperties>, typename Dispatch = DispatchLoaderStatic>
    typename ResultValueType<std::vector<PhysicalDeviceGroupProperties,Allocator>>::type enumeratePhysicalDeviceGroupsKHR(Dispatch const &d = Dispatch() ) const;
    template<typename Allocator = std::allocator<PhysicalDeviceGroupProperties>, typename Dispatch = DispatchLoaderStatic>
    typename ResultValueType<std::vector<PhysicalDeviceGroupProperties,Allocator>>::type enumeratePhysicalDeviceGroupsKHR(Allocator const& vectorAllocator, Dispatch const &d ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    Result enumeratePhysicalDevices( uint32_t* pPhysicalDeviceCount, PhysicalDevice* pPhysicalDevices, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Allocator = std::allocator<PhysicalDevice>, typename Dispatch = DispatchLoaderStatic>
    typename ResultValueType<std::vector<PhysicalDevice,Allocator>>::type enumeratePhysicalDevices(Dispatch const &d = Dispatch() ) const;
    template<typename Allocator = std::allocator<PhysicalDevice>, typename Dispatch = DispatchLoaderStatic>
    typename ResultValueType<std::vector<PhysicalDevice,Allocator>>::type enumeratePhysicalDevices(Allocator const& vectorAllocator, Dispatch const &d ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    PFN_vkVoidFunction getProcAddr( const char* pName, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    PFN_vkVoidFunction getProcAddr( const std::string & name, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    template<typename Dispatch = DispatchLoaderStatic>
    void submitDebugUtilsMessageEXT( DebugUtilsMessageSeverityFlagBitsEXT messageSeverity, DebugUtilsMessageTypeFlagsEXT messageTypes, const DebugUtilsMessengerCallbackDataEXT* pCallbackData, Dispatch const &d = Dispatch() ) const;
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
    template<typename Dispatch = DispatchLoaderStatic>
    void submitDebugUtilsMessageEXT( DebugUtilsMessageSeverityFlagBitsEXT messageSeverity, DebugUtilsMessageTypeFlagsEXT messageTypes, const DebugUtilsMessengerCallbackDataEXT & callbackData, Dispatch const &d = Dispatch() ) const;
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

    VULKAN_HPP_TYPESAFE_EXPLICIT operator VkInstance() const
    {
      return m_instance;
    }

    explicit operator bool() const
    {
      return m_instance != VK_NULL_HANDLE;
    }

    bool operator!() const
    {
      return m_instance == VK_NULL_HANDLE;
    }

  private:
    VkInstance m_instance;
  };
  static_assert( sizeof( Instance ) == sizeof( VkInstance ), "handle and wrapper have different size!" );

#ifndef VULKAN_HPP_NO_SMART_HANDLE
  template <typename Dispatch> class UniqueHandleTraits<Instance, Dispatch> { public: using deleter = ObjectDestroy<NoParent, Dispatch>; };
  using UniqueInstance = UniqueHandle<Instance, DispatchLoaderStatic>;
#endif /*VULKAN_HPP_NO_SMART_HANDLE*/

  template<typename Dispatch = DispatchLoaderStatic>
  Result createInstance( const InstanceCreateInfo* pCreateInfo, const AllocationCallbacks* pAllocator, Instance* pInstance, Dispatch const &d = Dispatch() );
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
  template<typename Dispatch = DispatchLoaderStatic>
  ResultValueType<Instance>::type createInstance( const InstanceCreateInfo & createInfo, Optional<const AllocationCallbacks> allocator = nullptr, Dispatch const &d = Dispatch() );
#ifndef VULKAN_HPP_NO_SMART_HANDLE
  template<typename Dispatch = DispatchLoaderStatic>
  typename ResultValueType<UniqueHandle<Instance,Dispatch>>::type createInstanceUnique( const InstanceCreateInfo & createInfo, Optional<const AllocationCallbacks> allocator = nullptr, Dispatch const &d = Dispatch() );
#endif /*VULKAN_HPP_NO_SMART_HANDLE*/
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

  template<typename Dispatch = DispatchLoaderStatic>
  Result enumerateInstanceExtensionProperties( const char* pLayerName, uint32_t* pPropertyCount, ExtensionProperties* pProperties, Dispatch const &d = Dispatch() );
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
  template<typename Allocator = std::allocator<ExtensionProperties>, typename Dispatch = DispatchLoaderStatic>
  typename ResultValueType<std::vector<ExtensionProperties,Allocator>>::type enumerateInstanceExtensionProperties( Optional<const std::string> layerName = nullptr, Dispatch const &d = Dispatch() );
  template<typename Allocator = std::allocator<ExtensionProperties>, typename Dispatch = DispatchLoaderStatic>
  typename ResultValueType<std::vector<ExtensionProperties,Allocator>>::type enumerateInstanceExtensionProperties( Optional<const std::string> layerName, Allocator const& vectorAllocator, Dispatch const &d );
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

  template<typename Dispatch = DispatchLoaderStatic>
  Result enumerateInstanceLayerProperties( uint32_t* pPropertyCount, LayerProperties* pProperties, Dispatch const &d = Dispatch() );
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
  template<typename Allocator = std::allocator<LayerProperties>, typename Dispatch = DispatchLoaderStatic>
  typename ResultValueType<std::vector<LayerProperties,Allocator>>::type enumerateInstanceLayerProperties(Dispatch const &d = Dispatch() );
  template<typename Allocator = std::allocator<LayerProperties>, typename Dispatch = DispatchLoaderStatic>
  typename ResultValueType<std::vector<LayerProperties,Allocator>>::type enumerateInstanceLayerProperties(Allocator const& vectorAllocator, Dispatch const &d );
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

  template<typename Dispatch = DispatchLoaderStatic>
  Result enumerateInstanceVersion( uint32_t* pApiVersion, Dispatch const &d = Dispatch() );
#ifndef VULKAN_HPP_DISABLE_ENHANCED_MODE
  template<typename Dispatch = DispatchLoaderStatic>
  ResultValueType<uint32_t>::type enumerateInstanceVersion(Dispatch const &d = Dispatch() );
#endif /*VULKAN_HPP_DISABLE_ENHANCED_MODE*/

  struct GeometryTrianglesNV
  {
    GeometryTrianglesNV( Buffer vertexData_ = Buffer(),
                         DeviceSize vertexOffset_ = 0,
                         uint32_t vertexCount_ = 0,
                         DeviceSize vertexStride_ = 0,
                         Format vertexFormat_ = Format::eUndefined,
                         Buffer indexData_ = Buffer(),
                         DeviceSize indexOffset_ = 0,
                         uint32_t indexCount_ = 0,
                         IndexType indexType_ = IndexType::eUint16,
                         Buffer transformData_ = Buffer(),
                         DeviceSize transformOffset_ = 0 )
      : vertexData( vertexData_ )
      , vertexOffset( vertexOffset_ )
      , vertexCount( vertexCount_ )
      , vertexStride( vertexStride_ )
      , vertexFormat( vertexFormat_ )
      , indexData( indexData_ )
      , indexOffset( indexOffset_ )
      , indexCount( indexCount_ )
      , indexType( indexType_ )
      , transformData( transformData_ )
      , transformOffset( transformOffset_ )
    {}

    GeometryTrianglesNV( VkGeometryTrianglesNV const & rhs )
    {
      memcpy( this, &rhs, sizeof( GeometryTrianglesNV ) );
    }

    GeometryTrianglesNV& operator=( VkGeometryTrianglesNV const & rhs )
    {
      memcpy( this, &rhs, sizeof( GeometryTrianglesNV ) );
      return *this;
    }

    GeometryTrianglesNV & setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    GeometryTrianglesNV & setVertexData( Buffer vertexData_ )
    {
      vertexData = vertexData_;
      return *this;
    }

    GeometryTrianglesNV & setVertexOffset( DeviceSize vertexOffset_ )
    {
      vertexOffset = vertexOffset_;
      return *this;
    }

    GeometryTrianglesNV & setVertexCount( uint32_t vertexCount_ )
    {
      vertexCount = vertexCount_;
      return *this;
    }

    GeometryTrianglesNV & setVertexStride( DeviceSize vertexStride_ )
    {
      vertexStride = vertexStride_;
      return *this;
    }

    GeometryTrianglesNV & setVertexFormat( Format vertexFormat_ )
    {
      vertexFormat = vertexFormat_;
      return *this;
    }

    GeometryTrianglesNV & setIndexData( Buffer indexData_ )
    {
      indexData = indexData_;
      return *this;
    }

    GeometryTrianglesNV & setIndexOffset( DeviceSize indexOffset_ )
    {
      indexOffset = indexOffset_;
      return *this;
    }

    GeometryTrianglesNV & setIndexCount( uint32_t indexCount_ )
    {
      indexCount = indexCount_;
      return *this;
    }

    GeometryTrianglesNV & setIndexType( IndexType indexType_ )
    {
      indexType = indexType_;
      return *this;
    }

    GeometryTrianglesNV & setTransformData( Buffer transformData_ )
    {
      transformData = transformData_;
      return *this;
    }

    GeometryTrianglesNV & setTransformOffset( DeviceSize transformOffset_ )
    {
      transformOffset = transformOffset_;
      return *this;
    }

    operator VkGeometryTrianglesNV const&() const
    {
      return *reinterpret_cast<const VkGeometryTrianglesNV*>( this );
    }

    operator VkGeometryTrianglesNV &()
    {
      return *reinterpret_cast<VkGeometryTrianglesNV*>( this );
    }

    bool operator==( GeometryTrianglesNV const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( vertexData == rhs.vertexData )
          && ( vertexOffset == rhs.vertexOffset )
          && ( vertexCount == rhs.vertexCount )
          && ( vertexStride == rhs.vertexStride )
          && ( vertexFormat == rhs.vertexFormat )
          && ( indexData == rhs.indexData )
          && ( indexOffset == rhs.indexOffset )
          && ( indexCount == rhs.indexCount )
          && ( indexType == rhs.indexType )
          && ( transformData == rhs.transformData )
          && ( transformOffset == rhs.transformOffset );
    }

    bool operator!=( GeometryTrianglesNV const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eGeometryTrianglesNV;

  public:
    const void* pNext = nullptr;
    Buffer vertexData;
    DeviceSize vertexOffset;
    uint32_t vertexCount;
    DeviceSize vertexStride;
    Format vertexFormat;
    Buffer indexData;
    DeviceSize indexOffset;
    uint32_t indexCount;
    IndexType indexType;
    Buffer transformData;
    DeviceSize transformOffset;
  };
  static_assert( sizeof( GeometryTrianglesNV ) == sizeof( VkGeometryTrianglesNV ), "struct and wrapper have different size!" );

  struct GeometryAABBNV
  {
    GeometryAABBNV( Buffer aabbData_ = Buffer(),
                    uint32_t numAABBs_ = 0,
                    uint32_t stride_ = 0,
                    DeviceSize offset_ = 0 )
      : aabbData( aabbData_ )
      , numAABBs( numAABBs_ )
      , stride( stride_ )
      , offset( offset_ )
    {}

    GeometryAABBNV( VkGeometryAABBNV const & rhs )
    {
      memcpy( this, &rhs, sizeof( GeometryAABBNV ) );
    }

    GeometryAABBNV& operator=( VkGeometryAABBNV const & rhs )
    {
      memcpy( this, &rhs, sizeof( GeometryAABBNV ) );
      return *this;
    }

    GeometryAABBNV & setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    GeometryAABBNV & setAabbData( Buffer aabbData_ )
    {
      aabbData = aabbData_;
      return *this;
    }

    GeometryAABBNV & setNumAABBs( uint32_t numAABBs_ )
    {
      numAABBs = numAABBs_;
      return *this;
    }

    GeometryAABBNV & setStride( uint32_t stride_ )
    {
      stride = stride_;
      return *this;
    }

    GeometryAABBNV & setOffset( DeviceSize offset_ )
    {
      offset = offset_;
      return *this;
    }

    operator VkGeometryAABBNV const&() const
    {
      return *reinterpret_cast<const VkGeometryAABBNV*>( this );
    }

    operator VkGeometryAABBNV &()
    {
      return *reinterpret_cast<VkGeometryAABBNV*>( this );
    }

    bool operator==( GeometryAABBNV const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( aabbData == rhs.aabbData )
          && ( numAABBs == rhs.numAABBs )
          && ( stride == rhs.stride )
          && ( offset == rhs.offset );
    }

    bool operator!=( GeometryAABBNV const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eGeometryAabbNV;

  public:
    const void* pNext = nullptr;
    Buffer aabbData;
    uint32_t numAABBs;
    uint32_t stride;
    DeviceSize offset;
  };
  static_assert( sizeof( GeometryAABBNV ) == sizeof( VkGeometryAABBNV ), "struct and wrapper have different size!" );

  struct GeometryDataNV
  {
    GeometryDataNV( GeometryTrianglesNV triangles_ = GeometryTrianglesNV(),
                    GeometryAABBNV aabbs_ = GeometryAABBNV() )
      : triangles( triangles_ )
      , aabbs( aabbs_ )
    {}

    GeometryDataNV( VkGeometryDataNV const & rhs )
    {
      memcpy( this, &rhs, sizeof( GeometryDataNV ) );
    }

    GeometryDataNV& operator=( VkGeometryDataNV const & rhs )
    {
      memcpy( this, &rhs, sizeof( GeometryDataNV ) );
      return *this;
    }

    GeometryDataNV & setTriangles( GeometryTrianglesNV triangles_ )
    {
      triangles = triangles_;
      return *this;
    }

    GeometryDataNV & setAabbs( GeometryAABBNV aabbs_ )
    {
      aabbs = aabbs_;
      return *this;
    }

    operator VkGeometryDataNV const&() const
    {
      return *reinterpret_cast<const VkGeometryDataNV*>( this );
    }

    operator VkGeometryDataNV &()
    {
      return *reinterpret_cast<VkGeometryDataNV*>( this );
    }

    bool operator==( GeometryDataNV const& rhs ) const
    {
      return ( triangles == rhs.triangles )
          && ( aabbs == rhs.aabbs );
    }

    bool operator!=( GeometryDataNV const& rhs ) const
    {
      return !operator==( rhs );
    }

    GeometryTrianglesNV triangles;
    GeometryAABBNV aabbs;
  };
  static_assert( sizeof( GeometryDataNV ) == sizeof( VkGeometryDataNV ), "struct and wrapper have different size!" );

  struct GeometryNV
  {
    GeometryNV( GeometryTypeNV geometryType_ = GeometryTypeNV::eTriangles,
                GeometryDataNV geometry_ = GeometryDataNV(),
                GeometryFlagsNV flags_ = GeometryFlagsNV() )
      : geometryType( geometryType_ )
      , geometry( geometry_ )
      , flags( flags_ )
    {}

    GeometryNV( VkGeometryNV const & rhs )
    {
      memcpy( this, &rhs, sizeof( GeometryNV ) );
    }

    GeometryNV& operator=( VkGeometryNV const & rhs )
    {
      memcpy( this, &rhs, sizeof( GeometryNV ) );
      return *this;
    }

    GeometryNV & setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    GeometryNV & setGeometryType( GeometryTypeNV geometryType_ )
    {
      geometryType = geometryType_;
      return *this;
    }

    GeometryNV & setGeometry( GeometryDataNV geometry_ )
    {
      geometry = geometry_;
      return *this;
    }

    GeometryNV & setFlags( GeometryFlagsNV flags_ )
    {
      flags = flags_;
      return *this;
    }

    operator VkGeometryNV const&() const
    {
      return *reinterpret_cast<const VkGeometryNV*>( this );
    }

    operator VkGeometryNV &()
    {
      return *reinterpret_cast<VkGeometryNV*>( this );
    }

    bool operator==( GeometryNV const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( geometryType == rhs.geometryType )
          && ( geometry == rhs.geometry )
          && ( flags == rhs.flags );
    }

    bool operator!=( GeometryNV const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eGeometryNV;

  public:
    const void* pNext = nullptr;
    GeometryTypeNV geometryType;
    GeometryDataNV geometry;
    GeometryFlagsNV flags;
  };
  static_assert( sizeof( GeometryNV ) == sizeof( VkGeometryNV ), "struct and wrapper have different size!" );

  struct AccelerationStructureInfoNV
  {
    AccelerationStructureInfoNV( AccelerationStructureTypeNV type_ = AccelerationStructureTypeNV::eTopLevel,
                                 BuildAccelerationStructureFlagsNV flags_ = BuildAccelerationStructureFlagsNV(),
                                 uint32_t instanceCount_ = 0,
                                 uint32_t geometryCount_ = 0,
                                 const GeometryNV* pGeometries_ = nullptr )
      : type( type_ )
      , flags( flags_ )
      , instanceCount( instanceCount_ )
      , geometryCount( geometryCount_ )
      , pGeometries( pGeometries_ )
    {}

    AccelerationStructureInfoNV( VkAccelerationStructureInfoNV const & rhs )
    {
      memcpy( this, &rhs, sizeof( AccelerationStructureInfoNV ) );
    }

    AccelerationStructureInfoNV& operator=( VkAccelerationStructureInfoNV const & rhs )
    {
      memcpy( this, &rhs, sizeof( AccelerationStructureInfoNV ) );
      return *this;
    }

    AccelerationStructureInfoNV & setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    AccelerationStructureInfoNV & setType( AccelerationStructureTypeNV type_ )
    {
      type = type_;
      return *this;
    }

    AccelerationStructureInfoNV & setFlags( BuildAccelerationStructureFlagsNV flags_ )
    {
      flags = flags_;
      return *this;
    }

    AccelerationStructureInfoNV & setInstanceCount( uint32_t instanceCount_ )
    {
      instanceCount = instanceCount_;
      return *this;
    }

    AccelerationStructureInfoNV & setGeometryCount( uint32_t geometryCount_ )
    {
      geometryCount = geometryCount_;
      return *this;
    }

    AccelerationStructureInfoNV & setPGeometries( const GeometryNV* pGeometries_ )
    {
      pGeometries = pGeometries_;
      return *this;
    }

    operator VkAccelerationStructureInfoNV const&() const
    {
      return *reinterpret_cast<const VkAccelerationStructureInfoNV*>( this );
    }

    operator VkAccelerationStructureInfoNV &()
    {
      return *reinterpret_cast<VkAccelerationStructureInfoNV*>( this );
    }

    bool operator==( AccelerationStructureInfoNV const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( type == rhs.type )
          && ( flags == rhs.flags )
          && ( instanceCount == rhs.instanceCount )
          && ( geometryCount == rhs.geometryCount )
          && ( pGeometries == rhs.pGeometries );
    }

    bool operator!=( AccelerationStructureInfoNV const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eAccelerationStructureInfoNV;

  public:
    const void* pNext = nullptr;
    AccelerationStructureTypeNV type;
    BuildAccelerationStructureFlagsNV flags;
    uint32_t instanceCount;
    uint32_t geometryCount;
    const GeometryNV* pGeometries;
  };
  static_assert( sizeof( AccelerationStructureInfoNV ) == sizeof( VkAccelerationStructureInfoNV ), "struct and wrapper have different size!" );

  struct AccelerationStructureCreateInfoNV
  {
    AccelerationStructureCreateInfoNV( DeviceSize compactedSize_ = 0,
                                       AccelerationStructureInfoNV info_ = AccelerationStructureInfoNV() )
      : compactedSize( compactedSize_ )
      , info( info_ )
    {}

    AccelerationStructureCreateInfoNV( VkAccelerationStructureCreateInfoNV const & rhs )
    {
      memcpy( this, &rhs, sizeof( AccelerationStructureCreateInfoNV ) );
    }

    AccelerationStructureCreateInfoNV& operator=( VkAccelerationStructureCreateInfoNV const & rhs )
    {
      memcpy( this, &rhs, sizeof( AccelerationStructureCreateInfoNV ) );
      return *this;
    }

    AccelerationStructureCreateInfoNV & setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    AccelerationStructureCreateInfoNV & setCompactedSize( DeviceSize compactedSize_ )
    {
      compactedSize = compactedSize_;
      return *this;
    }

    AccelerationStructureCreateInfoNV & setInfo( AccelerationStructureInfoNV info_ )
    {
      info = info_;
      return *this;
    }

    operator VkAccelerationStructureCreateInfoNV const&() const
    {
      return *reinterpret_cast<const VkAccelerationStructureCreateInfoNV*>( this );
    }

    operator VkAccelerationStructureCreateInfoNV &()
    {
      return *reinterpret_cast<VkAccelerationStructureCreateInfoNV*>( this );
    }

    bool operator==( AccelerationStructureCreateInfoNV const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( compactedSize == rhs.compactedSize )
          && ( info == rhs.info );
    }

    bool operator!=( AccelerationStructureCreateInfoNV const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eAccelerationStructureCreateInfoNV;

  public:
    const void* pNext = nullptr;
    DeviceSize compactedSize;
    AccelerationStructureInfoNV info;
  };
  static_assert( sizeof( AccelerationStructureCreateInfoNV ) == sizeof( VkAccelerationStructureCreateInfoNV ), "struct and wrapper have different size!" );

  struct AccelerationStructureMemoryRequirementsInfoNV
  {
    AccelerationStructureMemoryRequirementsInfoNV( AccelerationStructureMemoryRequirementsTypeNV type_ = AccelerationStructureMemoryRequirementsTypeNV::eObject,
                                                   AccelerationStructureNV accelerationStructure_ = AccelerationStructureNV() )
      : type( type_ )
      , accelerationStructure( accelerationStructure_ )
    {}

    AccelerationStructureMemoryRequirementsInfoNV( VkAccelerationStructureMemoryRequirementsInfoNV const & rhs )
    {
      memcpy( this, &rhs, sizeof( AccelerationStructureMemoryRequirementsInfoNV ) );
    }

    AccelerationStructureMemoryRequirementsInfoNV& operator=( VkAccelerationStructureMemoryRequirementsInfoNV const & rhs )
    {
      memcpy( this, &rhs, sizeof( AccelerationStructureMemoryRequirementsInfoNV ) );
      return *this;
    }

    AccelerationStructureMemoryRequirementsInfoNV & setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    AccelerationStructureMemoryRequirementsInfoNV & setType( AccelerationStructureMemoryRequirementsTypeNV type_ )
    {
      type = type_;
      return *this;
    }

    AccelerationStructureMemoryRequirementsInfoNV & setAccelerationStructure( AccelerationStructureNV accelerationStructure_ )
    {
      accelerationStructure = accelerationStructure_;
      return *this;
    }

    operator VkAccelerationStructureMemoryRequirementsInfoNV const&() const
    {
      return *reinterpret_cast<const VkAccelerationStructureMemoryRequirementsInfoNV*>( this );
    }

    operator VkAccelerationStructureMemoryRequirementsInfoNV &()
    {
      return *reinterpret_cast<VkAccelerationStructureMemoryRequirementsInfoNV*>( this );
    }

    bool operator==( AccelerationStructureMemoryRequirementsInfoNV const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( type == rhs.type )
          && ( accelerationStructure == rhs.accelerationStructure );
    }

    bool operator!=( AccelerationStructureMemoryRequirementsInfoNV const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eAccelerationStructureMemoryRequirementsInfoNV;

  public:
    const void* pNext = nullptr;
    AccelerationStructureMemoryRequirementsTypeNV type;
    AccelerationStructureNV accelerationStructure;
  };
  static_assert( sizeof( AccelerationStructureMemoryRequirementsInfoNV ) == sizeof( VkAccelerationStructureMemoryRequirementsInfoNV ), "struct and wrapper have different size!" );

  struct AcquireNextImageInfoKHR
  {
    AcquireNextImageInfoKHR( SwapchainKHR swapchain_ = SwapchainKHR(),
                             uint64_t timeout_ = 0,
                             Semaphore semaphore_ = Semaphore(),
                             Fence fence_ = Fence(),
                             uint32_t deviceMask_ = 0 )
      : swapchain( swapchain_ )
      , timeout( timeout_ )
      , semaphore( semaphore_ )
      , fence( fence_ )
      , deviceMask( deviceMask_ )
    {}

    AcquireNextImageInfoKHR( VkAcquireNextImageInfoKHR const & rhs )
    {
      memcpy( this, &rhs, sizeof( AcquireNextImageInfoKHR ) );
    }

    AcquireNextImageInfoKHR& operator=( VkAcquireNextImageInfoKHR const & rhs )
    {
      memcpy( this, &rhs, sizeof( AcquireNextImageInfoKHR ) );
      return *this;
    }

    AcquireNextImageInfoKHR & setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    AcquireNextImageInfoKHR & setSwapchain( SwapchainKHR swapchain_ )
    {
      swapchain = swapchain_;
      return *this;
    }

    AcquireNextImageInfoKHR & setTimeout( uint64_t timeout_ )
    {
      timeout = timeout_;
      return *this;
    }

    AcquireNextImageInfoKHR & setSemaphore( Semaphore semaphore_ )
    {
      semaphore = semaphore_;
      return *this;
    }

    AcquireNextImageInfoKHR & setFence( Fence fence_ )
    {
      fence = fence_;
      return *this;
    }

    AcquireNextImageInfoKHR & setDeviceMask( uint32_t deviceMask_ )
    {
      deviceMask = deviceMask_;
      return *this;
    }

    operator VkAcquireNextImageInfoKHR const&() const
    {
      return *reinterpret_cast<const VkAcquireNextImageInfoKHR*>( this );
    }

    operator VkAcquireNextImageInfoKHR &()
    {
      return *reinterpret_cast<VkAcquireNextImageInfoKHR*>( this );
    }

    bool operator==( AcquireNextImageInfoKHR const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( swapchain == rhs.swapchain )
          && ( timeout == rhs.timeout )
          && ( semaphore == rhs.semaphore )
          && ( fence == rhs.fence )
          && ( deviceMask == rhs.deviceMask );
    }

    bool operator!=( AcquireNextImageInfoKHR const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eAcquireNextImageInfoKHR;

  public:
    const void* pNext = nullptr;
    SwapchainKHR swapchain;
    uint64_t timeout;
    Semaphore semaphore;
    Fence fence;
    uint32_t deviceMask;
  };
  static_assert( sizeof( AcquireNextImageInfoKHR ) == sizeof( VkAcquireNextImageInfoKHR ), "struct and wrapper have different size!" );

  struct AllocationCallbacks
  {
    AllocationCallbacks( void* pUserData_ = nullptr,
                         PFN_vkAllocationFunction pfnAllocation_ = nullptr,
                         PFN_vkReallocationFunction pfnReallocation_ = nullptr,
                         PFN_vkFreeFunction pfnFree_ = nullptr,
                         PFN_vkInternalAllocationNotification pfnInternalAllocation_ = nullptr,
                         PFN_vkInternalFreeNotification pfnInternalFree_ = nullptr )
      : pUserData( pUserData_ )
      , pfnAllocation( pfnAllocation_ )
      , pfnReallocation( pfnReallocation_ )
      , pfnFree( pfnFree_ )
      , pfnInternalAllocation( pfnInternalAllocation_ )
      , pfnInternalFree( pfnInternalFree_ )
    {}

    AllocationCallbacks( VkAllocationCallbacks const & rhs )
    {
      memcpy( this, &rhs, sizeof( AllocationCallbacks ) );
    }

    AllocationCallbacks& operator=( VkAllocationCallbacks const & rhs )
    {
      memcpy( this, &rhs, sizeof( AllocationCallbacks ) );
      return *this;
    }

    AllocationCallbacks & setPUserData( void* pUserData_ )
    {
      pUserData = pUserData_;
      return *this;
    }

    AllocationCallbacks & setPfnAllocation( PFN_vkAllocationFunction pfnAllocation_ )
    {
      pfnAllocation = pfnAllocation_;
      return *this;
    }

    AllocationCallbacks & setPfnReallocation( PFN_vkReallocationFunction pfnReallocation_ )
    {
      pfnReallocation = pfnReallocation_;
      return *this;
    }

    AllocationCallbacks & setPfnFree( PFN_vkFreeFunction pfnFree_ )
    {
      pfnFree = pfnFree_;
      return *this;
    }

    AllocationCallbacks & setPfnInternalAllocation( PFN_vkInternalAllocationNotification pfnInternalAllocation_ )
    {
      pfnInternalAllocation = pfnInternalAllocation_;
      return *this;
    }

    AllocationCallbacks & setPfnInternalFree( PFN_vkInternalFreeNotification pfnInternalFree_ )
    {
      pfnInternalFree = pfnInternalFree_;
      return *this;
    }

    operator VkAllocationCallbacks const&() const
    {
      return *reinterpret_cast<const VkAllocationCallbacks*>( this );
    }

    operator VkAllocationCallbacks &()
    {
      return *reinterpret_cast<VkAllocationCallbacks*>( this );
    }

    bool operator==( AllocationCallbacks const& rhs ) const
    {
      return ( pUserData == rhs.pUserData )
          && ( pfnAllocation == rhs.pfnAllocation )
          && ( pfnReallocation == rhs.pfnReallocation )
          && ( pfnFree == rhs.pfnFree )
          && ( pfnInternalAllocation == rhs.pfnInternalAllocation )
          && ( pfnInternalFree == rhs.pfnInternalFree );
    }

    bool operator!=( AllocationCallbacks const& rhs ) const
    {
      return !operator==( rhs );
    }

    void* pUserData;
    PFN_vkAllocationFunction pfnAllocation;
    PFN_vkReallocationFunction pfnReallocation;
    PFN_vkFreeFunction pfnFree;
    PFN_vkInternalAllocationNotification pfnInternalAllocation;
    PFN_vkInternalFreeNotification pfnInternalFree;
  };
  static_assert( sizeof( AllocationCallbacks ) == sizeof( VkAllocationCallbacks ), "struct and wrapper have different size!" );

  struct ComponentMapping
  {
    ComponentMapping( ComponentSwizzle r_ = ComponentSwizzle::eIdentity,
                      ComponentSwizzle g_ = ComponentSwizzle::eIdentity,
                      ComponentSwizzle b_ = ComponentSwizzle::eIdentity,
                      ComponentSwizzle a_ = ComponentSwizzle::eIdentity )
      : r( r_ )
      , g( g_ )
      , b( b_ )
      , a( a_ )
    {}

    ComponentMapping( VkComponentMapping const & rhs )
    {
      memcpy( this, &rhs, sizeof( ComponentMapping ) );
    }

    ComponentMapping& operator=( VkComponentMapping const & rhs )
    {
      memcpy( this, &rhs, sizeof( ComponentMapping ) );
      return *this;
    }

    ComponentMapping & setR( ComponentSwizzle r_ )
    {
      r = r_;
      return *this;
    }

    ComponentMapping & setG( ComponentSwizzle g_ )
    {
      g = g_;
      return *this;
    }

    ComponentMapping & setB( ComponentSwizzle b_ )
    {
      b = b_;
      return *this;
    }

    ComponentMapping & setA( ComponentSwizzle a_ )
    {
      a = a_;
      return *this;
    }

    operator VkComponentMapping const&() const
    {
      return *reinterpret_cast<const VkComponentMapping*>( this );
    }

    operator VkComponentMapping &()
    {
      return *reinterpret_cast<VkComponentMapping*>( this );
    }

    bool operator==( ComponentMapping const& rhs ) const
    {
      return ( r == rhs.r )
          && ( g == rhs.g )
          && ( b == rhs.b )
          && ( a == rhs.a );
    }

    bool operator!=( ComponentMapping const& rhs ) const
    {
      return !operator==( rhs );
    }

    ComponentSwizzle r;
    ComponentSwizzle g;
    ComponentSwizzle b;
    ComponentSwizzle a;
  };
  static_assert( sizeof( ComponentMapping ) == sizeof( VkComponentMapping ), "struct and wrapper have different size!" );

#ifdef VK_USE_PLATFORM_ANDROID_KHR
  struct AndroidHardwareBufferFormatPropertiesANDROID
  {
    operator VkAndroidHardwareBufferFormatPropertiesANDROID const&() const
    {
      return *reinterpret_cast<const VkAndroidHardwareBufferFormatPropertiesANDROID*>( this );
    }

    operator VkAndroidHardwareBufferFormatPropertiesANDROID &()
    {
      return *reinterpret_cast<VkAndroidHardwareBufferFormatPropertiesANDROID*>( this );
    }

    bool operator==( AndroidHardwareBufferFormatPropertiesANDROID const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( format == rhs.format )
          && ( externalFormat == rhs.externalFormat )
          && ( formatFeatures == rhs.formatFeatures )
          && ( samplerYcbcrConversionComponents == rhs.samplerYcbcrConversionComponents )
          && ( suggestedYcbcrModel == rhs.suggestedYcbcrModel )
          && ( suggestedYcbcrRange == rhs.suggestedYcbcrRange )
          && ( suggestedXChromaOffset == rhs.suggestedXChromaOffset )
          && ( suggestedYChromaOffset == rhs.suggestedYChromaOffset );
    }

    bool operator!=( AndroidHardwareBufferFormatPropertiesANDROID const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eAndroidHardwareBufferFormatPropertiesANDROID;

  public:
    void* pNext = nullptr;
    Format format;
    uint64_t externalFormat;
    FormatFeatureFlags formatFeatures;
    ComponentMapping samplerYcbcrConversionComponents;
    SamplerYcbcrModelConversion suggestedYcbcrModel;
    SamplerYcbcrRange suggestedYcbcrRange;
    ChromaLocation suggestedXChromaOffset;
    ChromaLocation suggestedYChromaOffset;
  };
  static_assert( sizeof( AndroidHardwareBufferFormatPropertiesANDROID ) == sizeof( VkAndroidHardwareBufferFormatPropertiesANDROID ), "struct and wrapper have different size!" );
#endif /*VK_USE_PLATFORM_ANDROID_KHR*/

#ifdef VK_USE_PLATFORM_ANDROID_KHR
  struct AndroidHardwareBufferPropertiesANDROID
  {
    operator VkAndroidHardwareBufferPropertiesANDROID const&() const
    {
      return *reinterpret_cast<const VkAndroidHardwareBufferPropertiesANDROID*>( this );
    }

    operator VkAndroidHardwareBufferPropertiesANDROID &()
    {
      return *reinterpret_cast<VkAndroidHardwareBufferPropertiesANDROID*>( this );
    }

    bool operator==( AndroidHardwareBufferPropertiesANDROID const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( allocationSize == rhs.allocationSize )
          && ( memoryTypeBits == rhs.memoryTypeBits );
    }

    bool operator!=( AndroidHardwareBufferPropertiesANDROID const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eAndroidHardwareBufferPropertiesANDROID;

  public:
    void* pNext = nullptr;
    DeviceSize allocationSize;
    uint32_t memoryTypeBits;
  };
  static_assert( sizeof( AndroidHardwareBufferPropertiesANDROID ) == sizeof( VkAndroidHardwareBufferPropertiesANDROID ), "struct and wrapper have different size!" );
#endif /*VK_USE_PLATFORM_ANDROID_KHR*/

#ifdef VK_USE_PLATFORM_ANDROID_KHR
  struct AndroidHardwareBufferUsageANDROID
  {
    operator VkAndroidHardwareBufferUsageANDROID const&() const
    {
      return *reinterpret_cast<const VkAndroidHardwareBufferUsageANDROID*>( this );
    }

    operator VkAndroidHardwareBufferUsageANDROID &()
    {
      return *reinterpret_cast<VkAndroidHardwareBufferUsageANDROID*>( this );
    }

    bool operator==( AndroidHardwareBufferUsageANDROID const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( androidHardwareBufferUsage == rhs.androidHardwareBufferUsage );
    }

    bool operator!=( AndroidHardwareBufferUsageANDROID const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eAndroidHardwareBufferUsageANDROID;

  public:
    void* pNext = nullptr;
    uint64_t androidHardwareBufferUsage;
  };
  static_assert( sizeof( AndroidHardwareBufferUsageANDROID ) == sizeof( VkAndroidHardwareBufferUsageANDROID ), "struct and wrapper have different size!" );
#endif /*VK_USE_PLATFORM_ANDROID_KHR*/

#ifdef VK_USE_PLATFORM_ANDROID_KHR
  struct AndroidSurfaceCreateInfoKHR
  {
    AndroidSurfaceCreateInfoKHR( AndroidSurfaceCreateFlagsKHR flags_ = AndroidSurfaceCreateFlagsKHR(),
                                 struct ANativeWindow* window_ = nullptr )
      : flags( flags_ )
      , window( window_ )
    {}

    AndroidSurfaceCreateInfoKHR( VkAndroidSurfaceCreateInfoKHR const & rhs )
    {
      memcpy( this, &rhs, sizeof( AndroidSurfaceCreateInfoKHR ) );
    }

    AndroidSurfaceCreateInfoKHR& operator=( VkAndroidSurfaceCreateInfoKHR const & rhs )
    {
      memcpy( this, &rhs, sizeof( AndroidSurfaceCreateInfoKHR ) );
      return *this;
    }

    AndroidSurfaceCreateInfoKHR & setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    AndroidSurfaceCreateInfoKHR & setFlags( AndroidSurfaceCreateFlagsKHR flags_ )
    {
      flags = flags_;
      return *this;
    }

    AndroidSurfaceCreateInfoKHR & setWindow( struct ANativeWindow* window_ )
    {
      window = window_;
      return *this;
    }

    operator VkAndroidSurfaceCreateInfoKHR const&() const
    {
      return *reinterpret_cast<const VkAndroidSurfaceCreateInfoKHR*>( this );
    }

    operator VkAndroidSurfaceCreateInfoKHR &()
    {
      return *reinterpret_cast<VkAndroidSurfaceCreateInfoKHR*>( this );
    }

    bool operator==( AndroidSurfaceCreateInfoKHR const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( flags == rhs.flags )
          && ( window == rhs.window );
    }

    bool operator!=( AndroidSurfaceCreateInfoKHR const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eAndroidSurfaceCreateInfoKHR;

  public:
    const void* pNext = nullptr;
    AndroidSurfaceCreateFlagsKHR flags;
    struct ANativeWindow* window;
  };
  static_assert( sizeof( AndroidSurfaceCreateInfoKHR ) == sizeof( VkAndroidSurfaceCreateInfoKHR ), "struct and wrapper have different size!" );
#endif /*VK_USE_PLATFORM_ANDROID_KHR*/

  struct ApplicationInfo
  {
    ApplicationInfo( const char* pApplicationName_ = nullptr,
                     uint32_t applicationVersion_ = 0,
                     const char* pEngineName_ = nullptr,
                     uint32_t engineVersion_ = 0,
                     uint32_t apiVersion_ = 0 )
      : pApplicationName( pApplicationName_ )
      , applicationVersion( applicationVersion_ )
      , pEngineName( pEngineName_ )
      , engineVersion( engineVersion_ )
      , apiVersion( apiVersion_ )
    {}

    ApplicationInfo( VkApplicationInfo const & rhs )
    {
      memcpy( this, &rhs, sizeof( ApplicationInfo ) );
    }

    ApplicationInfo& operator=( VkApplicationInfo const & rhs )
    {
      memcpy( this, &rhs, sizeof( ApplicationInfo ) );
      return *this;
    }

    ApplicationInfo & setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    ApplicationInfo & setPApplicationName( const char* pApplicationName_ )
    {
      pApplicationName = pApplicationName_;
      return *this;
    }

    ApplicationInfo & setApplicationVersion( uint32_t applicationVersion_ )
    {
      applicationVersion = applicationVersion_;
      return *this;
    }

    ApplicationInfo & setPEngineName( const char* pEngineName_ )
    {
      pEngineName = pEngineName_;
      return *this;
    }

    ApplicationInfo & setEngineVersion( uint32_t engineVersion_ )
    {
      engineVersion = engineVersion_;
      return *this;
    }

    ApplicationInfo & setApiVersion( uint32_t apiVersion_ )
    {
      apiVersion = apiVersion_;
      return *this;
    }

    operator VkApplicationInfo const&() const
    {
      return *reinterpret_cast<const VkApplicationInfo*>( this );
    }

    operator VkApplicationInfo &()
    {
      return *reinterpret_cast<VkApplicationInfo*>( this );
    }

    bool operator==( ApplicationInfo const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( pApplicationName == rhs.pApplicationName )
          && ( applicationVersion == rhs.applicationVersion )
          && ( pEngineName == rhs.pEngineName )
          && ( engineVersion == rhs.engineVersion )
          && ( apiVersion == rhs.apiVersion );
    }

    bool operator!=( ApplicationInfo const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eApplicationInfo;

  public:
    const void* pNext = nullptr;
    const char* pApplicationName;
    uint32_t applicationVersion;
    const char* pEngineName;
    uint32_t engineVersion;
    uint32_t apiVersion;
  };
  static_assert( sizeof( ApplicationInfo ) == sizeof( VkApplicationInfo ), "struct and wrapper have different size!" );

  struct AttachmentDescription
  {
    AttachmentDescription( AttachmentDescriptionFlags flags_ = AttachmentDescriptionFlags(),
                           Format format_ = Format::eUndefined,
                           SampleCountFlagBits samples_ = SampleCountFlagBits::e1,
                           AttachmentLoadOp loadOp_ = AttachmentLoadOp::eLoad,
                           AttachmentStoreOp storeOp_ = AttachmentStoreOp::eStore,
                           AttachmentLoadOp stencilLoadOp_ = AttachmentLoadOp::eLoad,
                           AttachmentStoreOp stencilStoreOp_ = AttachmentStoreOp::eStore,
                           ImageLayout initialLayout_ = ImageLayout::eUndefined,
                           ImageLayout finalLayout_ = ImageLayout::eUndefined )
      : flags( flags_ )
      , format( format_ )
      , samples( samples_ )
      , loadOp( loadOp_ )
      , storeOp( storeOp_ )
      , stencilLoadOp( stencilLoadOp_ )
      , stencilStoreOp( stencilStoreOp_ )
      , initialLayout( initialLayout_ )
      , finalLayout( finalLayout_ )
    {}

    AttachmentDescription( VkAttachmentDescription const & rhs )
    {
      memcpy( this, &rhs, sizeof( AttachmentDescription ) );
    }

    AttachmentDescription& operator=( VkAttachmentDescription const & rhs )
    {
      memcpy( this, &rhs, sizeof( AttachmentDescription ) );
      return *this;
    }

    AttachmentDescription & setFlags( AttachmentDescriptionFlags flags_ )
    {
      flags = flags_;
      return *this;
    }

    AttachmentDescription & setFormat( Format format_ )
    {
      format = format_;
      return *this;
    }

    AttachmentDescription & setSamples( SampleCountFlagBits samples_ )
    {
      samples = samples_;
      return *this;
    }

    AttachmentDescription & setLoadOp( AttachmentLoadOp loadOp_ )
    {
      loadOp = loadOp_;
      return *this;
    }

    AttachmentDescription & setStoreOp( AttachmentStoreOp storeOp_ )
    {
      storeOp = storeOp_;
      return *this;
    }

    AttachmentDescription & setStencilLoadOp( AttachmentLoadOp stencilLoadOp_ )
    {
      stencilLoadOp = stencilLoadOp_;
      return *this;
    }

    AttachmentDescription & setStencilStoreOp( AttachmentStoreOp stencilStoreOp_ )
    {
      stencilStoreOp = stencilStoreOp_;
      return *this;
    }

    AttachmentDescription & setInitialLayout( ImageLayout initialLayout_ )
    {
      initialLayout = initialLayout_;
      return *this;
    }

    AttachmentDescription & setFinalLayout( ImageLayout finalLayout_ )
    {
      finalLayout = finalLayout_;
      return *this;
    }

    operator VkAttachmentDescription const&() const
    {
      return *reinterpret_cast<const VkAttachmentDescription*>( this );
    }

    operator VkAttachmentDescription &()
    {
      return *reinterpret_cast<VkAttachmentDescription*>( this );
    }

    bool operator==( AttachmentDescription const& rhs ) const
    {
      return ( flags == rhs.flags )
          && ( format == rhs.format )
          && ( samples == rhs.samples )
          && ( loadOp == rhs.loadOp )
          && ( storeOp == rhs.storeOp )
          && ( stencilLoadOp == rhs.stencilLoadOp )
          && ( stencilStoreOp == rhs.stencilStoreOp )
          && ( initialLayout == rhs.initialLayout )
          && ( finalLayout == rhs.finalLayout );
    }

    bool operator!=( AttachmentDescription const& rhs ) const
    {
      return !operator==( rhs );
    }

    AttachmentDescriptionFlags flags;
    Format format;
    SampleCountFlagBits samples;
    AttachmentLoadOp loadOp;
    AttachmentStoreOp storeOp;
    AttachmentLoadOp stencilLoadOp;
    AttachmentStoreOp stencilStoreOp;
    ImageLayout initialLayout;
    ImageLayout finalLayout;
  };
  static_assert( sizeof( AttachmentDescription ) == sizeof( VkAttachmentDescription ), "struct and wrapper have different size!" );

  struct AttachmentDescription2KHR
  {
    AttachmentDescription2KHR( AttachmentDescriptionFlags flags_ = AttachmentDescriptionFlags(),
                               Format format_ = Format::eUndefined,
                               SampleCountFlagBits samples_ = SampleCountFlagBits::e1,
                               AttachmentLoadOp loadOp_ = AttachmentLoadOp::eLoad,
                               AttachmentStoreOp storeOp_ = AttachmentStoreOp::eStore,
                               AttachmentLoadOp stencilLoadOp_ = AttachmentLoadOp::eLoad,
                               AttachmentStoreOp stencilStoreOp_ = AttachmentStoreOp::eStore,
                               ImageLayout initialLayout_ = ImageLayout::eUndefined,
                               ImageLayout finalLayout_ = ImageLayout::eUndefined )
      : flags( flags_ )
      , format( format_ )
      , samples( samples_ )
      , loadOp( loadOp_ )
      , storeOp( storeOp_ )
      , stencilLoadOp( stencilLoadOp_ )
      , stencilStoreOp( stencilStoreOp_ )
      , initialLayout( initialLayout_ )
      , finalLayout( finalLayout_ )
    {}

    AttachmentDescription2KHR( VkAttachmentDescription2KHR const & rhs )
    {
      memcpy( this, &rhs, sizeof( AttachmentDescription2KHR ) );
    }

    AttachmentDescription2KHR& operator=( VkAttachmentDescription2KHR const & rhs )
    {
      memcpy( this, &rhs, sizeof( AttachmentDescription2KHR ) );
      return *this;
    }

    AttachmentDescription2KHR & setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    AttachmentDescription2KHR & setFlags( AttachmentDescriptionFlags flags_ )
    {
      flags = flags_;
      return *this;
    }

    AttachmentDescription2KHR & setFormat( Format format_ )
    {
      format = format_;
      return *this;
    }

    AttachmentDescription2KHR & setSamples( SampleCountFlagBits samples_ )
    {
      samples = samples_;
      return *this;
    }

    AttachmentDescription2KHR & setLoadOp( AttachmentLoadOp loadOp_ )
    {
      loadOp = loadOp_;
      return *this;
    }

    AttachmentDescription2KHR & setStoreOp( AttachmentStoreOp storeOp_ )
    {
      storeOp = storeOp_;
      return *this;
    }

    AttachmentDescription2KHR & setStencilLoadOp( AttachmentLoadOp stencilLoadOp_ )
    {
      stencilLoadOp = stencilLoadOp_;
      return *this;
    }

    AttachmentDescription2KHR & setStencilStoreOp( AttachmentStoreOp stencilStoreOp_ )
    {
      stencilStoreOp = stencilStoreOp_;
      return *this;
    }

    AttachmentDescription2KHR & setInitialLayout( ImageLayout initialLayout_ )
    {
      initialLayout = initialLayout_;
      return *this;
    }

    AttachmentDescription2KHR & setFinalLayout( ImageLayout finalLayout_ )
    {
      finalLayout = finalLayout_;
      return *this;
    }

    operator VkAttachmentDescription2KHR const&() const
    {
      return *reinterpret_cast<const VkAttachmentDescription2KHR*>( this );
    }

    operator VkAttachmentDescription2KHR &()
    {
      return *reinterpret_cast<VkAttachmentDescription2KHR*>( this );
    }

    bool operator==( AttachmentDescription2KHR const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( flags == rhs.flags )
          && ( format == rhs.format )
          && ( samples == rhs.samples )
          && ( loadOp == rhs.loadOp )
          && ( storeOp == rhs.storeOp )
          && ( stencilLoadOp == rhs.stencilLoadOp )
          && ( stencilStoreOp == rhs.stencilStoreOp )
          && ( initialLayout == rhs.initialLayout )
          && ( finalLayout == rhs.finalLayout );
    }

    bool operator!=( AttachmentDescription2KHR const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eAttachmentDescription2KHR;

  public:
    const void* pNext = nullptr;
    AttachmentDescriptionFlags flags;
    Format format;
    SampleCountFlagBits samples;
    AttachmentLoadOp loadOp;
    AttachmentStoreOp storeOp;
    AttachmentLoadOp stencilLoadOp;
    AttachmentStoreOp stencilStoreOp;
    ImageLayout initialLayout;
    ImageLayout finalLayout;
  };
  static_assert( sizeof( AttachmentDescription2KHR ) == sizeof( VkAttachmentDescription2KHR ), "struct and wrapper have different size!" );

  struct AttachmentReference
  {
    AttachmentReference( uint32_t attachment_ = 0,
                         ImageLayout layout_ = ImageLayout::eUndefined )
      : attachment( attachment_ )
      , layout( layout_ )
    {}

    AttachmentReference( VkAttachmentReference const & rhs )
    {
      memcpy( this, &rhs, sizeof( AttachmentReference ) );
    }

    AttachmentReference& operator=( VkAttachmentReference const & rhs )
    {
      memcpy( this, &rhs, sizeof( AttachmentReference ) );
      return *this;
    }

    AttachmentReference & setAttachment( uint32_t attachment_ )
    {
      attachment = attachment_;
      return *this;
    }

    AttachmentReference & setLayout( ImageLayout layout_ )
    {
      layout = layout_;
      return *this;
    }

    operator VkAttachmentReference const&() const
    {
      return *reinterpret_cast<const VkAttachmentReference*>( this );
    }

    operator VkAttachmentReference &()
    {
      return *reinterpret_cast<VkAttachmentReference*>( this );
    }

    bool operator==( AttachmentReference const& rhs ) const
    {
      return ( attachment == rhs.attachment )
          && ( layout == rhs.layout );
    }

    bool operator!=( AttachmentReference const& rhs ) const
    {
      return !operator==( rhs );
    }

    uint32_t attachment;
    ImageLayout layout;
  };
  static_assert( sizeof( AttachmentReference ) == sizeof( VkAttachmentReference ), "struct and wrapper have different size!" );

  struct AttachmentReference2KHR
  {
    AttachmentReference2KHR( uint32_t attachment_ = 0,
                             ImageLayout layout_ = ImageLayout::eUndefined,
                             ImageAspectFlags aspectMask_ = ImageAspectFlags() )
      : attachment( attachment_ )
      , layout( layout_ )
      , aspectMask( aspectMask_ )
    {}

    AttachmentReference2KHR( VkAttachmentReference2KHR const & rhs )
    {
      memcpy( this, &rhs, sizeof( AttachmentReference2KHR ) );
    }

    AttachmentReference2KHR& operator=( VkAttachmentReference2KHR const & rhs )
    {
      memcpy( this, &rhs, sizeof( AttachmentReference2KHR ) );
      return *this;
    }

    AttachmentReference2KHR & setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    AttachmentReference2KHR & setAttachment( uint32_t attachment_ )
    {
      attachment = attachment_;
      return *this;
    }

    AttachmentReference2KHR & setLayout( ImageLayout layout_ )
    {
      layout = layout_;
      return *this;
    }

    AttachmentReference2KHR & setAspectMask( ImageAspectFlags aspectMask_ )
    {
      aspectMask = aspectMask_;
      return *this;
    }

    operator VkAttachmentReference2KHR const&() const
    {
      return *reinterpret_cast<const VkAttachmentReference2KHR*>( this );
    }

    operator VkAttachmentReference2KHR &()
    {
      return *reinterpret_cast<VkAttachmentReference2KHR*>( this );
    }

    bool operator==( AttachmentReference2KHR const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( attachment == rhs.attachment )
          && ( layout == rhs.layout )
          && ( aspectMask == rhs.aspectMask );
    }

    bool operator!=( AttachmentReference2KHR const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eAttachmentReference2KHR;

  public:
    const void* pNext = nullptr;
    uint32_t attachment;
    ImageLayout layout;
    ImageAspectFlags aspectMask;
  };
  static_assert( sizeof( AttachmentReference2KHR ) == sizeof( VkAttachmentReference2KHR ), "struct and wrapper have different size!" );

  struct Extent2D
  {
    Extent2D( uint32_t width_ = 0,
              uint32_t height_ = 0 )
      : width( width_ )
      , height( height_ )
    {}

    Extent2D( VkExtent2D const & rhs )
    {
      memcpy( this, &rhs, sizeof( Extent2D ) );
    }

    Extent2D& operator=( VkExtent2D const & rhs )
    {
      memcpy( this, &rhs, sizeof( Extent2D ) );
      return *this;
    }

    Extent2D & setWidth( uint32_t width_ )
    {
      width = width_;
      return *this;
    }

    Extent2D & setHeight( uint32_t height_ )
    {
      height = height_;
      return *this;
    }

    operator VkExtent2D const&() const
    {
      return *reinterpret_cast<const VkExtent2D*>( this );
    }

    operator VkExtent2D &()
    {
      return *reinterpret_cast<VkExtent2D*>( this );
    }

    bool operator==( Extent2D const& rhs ) const
    {
      return ( width == rhs.width )
          && ( height == rhs.height );
    }

    bool operator!=( Extent2D const& rhs ) const
    {
      return !operator==( rhs );
    }

    uint32_t width;
    uint32_t height;
  };
  static_assert( sizeof( Extent2D ) == sizeof( VkExtent2D ), "struct and wrapper have different size!" );

  struct SampleLocationEXT
  {
    SampleLocationEXT( float x_ = 0,
                       float y_ = 0 )
      : x( x_ )
      , y( y_ )
    {}

    SampleLocationEXT( VkSampleLocationEXT const & rhs )
    {
      memcpy( this, &rhs, sizeof( SampleLocationEXT ) );
    }

    SampleLocationEXT& operator=( VkSampleLocationEXT const & rhs )
    {
      memcpy( this, &rhs, sizeof( SampleLocationEXT ) );
      return *this;
    }

    SampleLocationEXT & setX( float x_ )
    {
      x = x_;
      return *this;
    }

    SampleLocationEXT & setY( float y_ )
    {
      y = y_;
      return *this;
    }

    operator VkSampleLocationEXT const&() const
    {
      return *reinterpret_cast<const VkSampleLocationEXT*>( this );
    }

    operator VkSampleLocationEXT &()
    {
      return *reinterpret_cast<VkSampleLocationEXT*>( this );
    }

    bool operator==( SampleLocationEXT const& rhs ) const
    {
      return ( x == rhs.x )
          && ( y == rhs.y );
    }

    bool operator!=( SampleLocationEXT const& rhs ) const
    {
      return !operator==( rhs );
    }

    float x;
    float y;
  };
  static_assert( sizeof( SampleLocationEXT ) == sizeof( VkSampleLocationEXT ), "struct and wrapper have different size!" );

  struct SampleLocationsInfoEXT
  {
    SampleLocationsInfoEXT( SampleCountFlagBits sampleLocationsPerPixel_ = SampleCountFlagBits::e1,
                            Extent2D sampleLocationGridSize_ = Extent2D(),
                            uint32_t sampleLocationsCount_ = 0,
                            const SampleLocationEXT* pSampleLocations_ = nullptr )
      : sampleLocationsPerPixel( sampleLocationsPerPixel_ )
      , sampleLocationGridSize( sampleLocationGridSize_ )
      , sampleLocationsCount( sampleLocationsCount_ )
      , pSampleLocations( pSampleLocations_ )
    {}

    SampleLocationsInfoEXT( VkSampleLocationsInfoEXT const & rhs )
    {
      memcpy( this, &rhs, sizeof( SampleLocationsInfoEXT ) );
    }

    SampleLocationsInfoEXT& operator=( VkSampleLocationsInfoEXT const & rhs )
    {
      memcpy( this, &rhs, sizeof( SampleLocationsInfoEXT ) );
      return *this;
    }

    SampleLocationsInfoEXT & setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    SampleLocationsInfoEXT & setSampleLocationsPerPixel( SampleCountFlagBits sampleLocationsPerPixel_ )
    {
      sampleLocationsPerPixel = sampleLocationsPerPixel_;
      return *this;
    }

    SampleLocationsInfoEXT & setSampleLocationGridSize( Extent2D sampleLocationGridSize_ )
    {
      sampleLocationGridSize = sampleLocationGridSize_;
      return *this;
    }

    SampleLocationsInfoEXT & setSampleLocationsCount( uint32_t sampleLocationsCount_ )
    {
      sampleLocationsCount = sampleLocationsCount_;
      return *this;
    }

    SampleLocationsInfoEXT & setPSampleLocations( const SampleLocationEXT* pSampleLocations_ )
    {
      pSampleLocations = pSampleLocations_;
      return *this;
    }

    operator VkSampleLocationsInfoEXT const&() const
    {
      return *reinterpret_cast<const VkSampleLocationsInfoEXT*>( this );
    }

    operator VkSampleLocationsInfoEXT &()
    {
      return *reinterpret_cast<VkSampleLocationsInfoEXT*>( this );
    }

    bool operator==( SampleLocationsInfoEXT const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( sampleLocationsPerPixel == rhs.sampleLocationsPerPixel )
          && ( sampleLocationGridSize == rhs.sampleLocationGridSize )
          && ( sampleLocationsCount == rhs.sampleLocationsCount )
          && ( pSampleLocations == rhs.pSampleLocations );
    }

    bool operator!=( SampleLocationsInfoEXT const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eSampleLocationsInfoEXT;

  public:
    const void* pNext = nullptr;
    SampleCountFlagBits sampleLocationsPerPixel;
    Extent2D sampleLocationGridSize;
    uint32_t sampleLocationsCount;
    const SampleLocationEXT* pSampleLocations;
  };
  static_assert( sizeof( SampleLocationsInfoEXT ) == sizeof( VkSampleLocationsInfoEXT ), "struct and wrapper have different size!" );

  struct AttachmentSampleLocationsEXT
  {
    AttachmentSampleLocationsEXT( uint32_t attachmentIndex_ = 0,
                                  SampleLocationsInfoEXT sampleLocationsInfo_ = SampleLocationsInfoEXT() )
      : attachmentIndex( attachmentIndex_ )
      , sampleLocationsInfo( sampleLocationsInfo_ )
    {}

    AttachmentSampleLocationsEXT( VkAttachmentSampleLocationsEXT const & rhs )
    {
      memcpy( this, &rhs, sizeof( AttachmentSampleLocationsEXT ) );
    }

    AttachmentSampleLocationsEXT& operator=( VkAttachmentSampleLocationsEXT const & rhs )
    {
      memcpy( this, &rhs, sizeof( AttachmentSampleLocationsEXT ) );
      return *this;
    }

    AttachmentSampleLocationsEXT & setAttachmentIndex( uint32_t attachmentIndex_ )
    {
      attachmentIndex = attachmentIndex_;
      return *this;
    }

    AttachmentSampleLocationsEXT & setSampleLocationsInfo( SampleLocationsInfoEXT sampleLocationsInfo_ )
    {
      sampleLocationsInfo = sampleLocationsInfo_;
      return *this;
    }

    operator VkAttachmentSampleLocationsEXT const&() const
    {
      return *reinterpret_cast<const VkAttachmentSampleLocationsEXT*>( this );
    }

    operator VkAttachmentSampleLocationsEXT &()
    {
      return *reinterpret_cast<VkAttachmentSampleLocationsEXT*>( this );
    }

    bool operator==( AttachmentSampleLocationsEXT const& rhs ) const
    {
      return ( attachmentIndex == rhs.attachmentIndex )
          && ( sampleLocationsInfo == rhs.sampleLocationsInfo );
    }

    bool operator!=( AttachmentSampleLocationsEXT const& rhs ) const
    {
      return !operator==( rhs );
    }

    uint32_t attachmentIndex;
    SampleLocationsInfoEXT sampleLocationsInfo;
  };
  static_assert( sizeof( AttachmentSampleLocationsEXT ) == sizeof( VkAttachmentSampleLocationsEXT ), "struct and wrapper have different size!" );

  struct BaseInStructure
  {
    BaseInStructure(  )
    {}

    BaseInStructure( VkBaseInStructure const & rhs )
    {
      memcpy( this, &rhs, sizeof( BaseInStructure ) );
    }

    BaseInStructure& operator=( VkBaseInStructure const & rhs )
    {
      memcpy( this, &rhs, sizeof( BaseInStructure ) );
      return *this;
    }

    BaseInStructure & setPNext( const struct BaseInStructure* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    operator VkBaseInStructure const&() const
    {
      return *reinterpret_cast<const VkBaseInStructure*>( this );
    }

    operator VkBaseInStructure &()
    {
      return *reinterpret_cast<VkBaseInStructure*>( this );
    }

    bool operator==( BaseInStructure const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext );
    }

    bool operator!=( BaseInStructure const& rhs ) const
    {
      return !operator==( rhs );
    }

    StructureType sType;
    const struct BaseInStructure* pNext = nullptr;
  };
  static_assert( sizeof( BaseInStructure ) == sizeof( VkBaseInStructure ), "struct and wrapper have different size!" );

  struct BaseOutStructure
  {
    BaseOutStructure(  )
    {}

    BaseOutStructure( VkBaseOutStructure const & rhs )
    {
      memcpy( this, &rhs, sizeof( BaseOutStructure ) );
    }

    BaseOutStructure& operator=( VkBaseOutStructure const & rhs )
    {
      memcpy( this, &rhs, sizeof( BaseOutStructure ) );
      return *this;
    }

    BaseOutStructure & setPNext( struct BaseOutStructure* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    operator VkBaseOutStructure const&() const
    {
      return *reinterpret_cast<const VkBaseOutStructure*>( this );
    }

    operator VkBaseOutStructure &()
    {
      return *reinterpret_cast<VkBaseOutStructure*>( this );
    }

    bool operator==( BaseOutStructure const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext );
    }

    bool operator!=( BaseOutStructure const& rhs ) const
    {
      return !operator==( rhs );
    }

    StructureType sType;
    struct BaseOutStructure* pNext = nullptr;
  };
  static_assert( sizeof( BaseOutStructure ) == sizeof( VkBaseOutStructure ), "struct and wrapper have different size!" );

  struct BindAccelerationStructureMemoryInfoNV
  {
    BindAccelerationStructureMemoryInfoNV( AccelerationStructureNV accelerationStructure_ = AccelerationStructureNV(),
                                           DeviceMemory memory_ = DeviceMemory(),
                                           DeviceSize memoryOffset_ = 0,
                                           uint32_t deviceIndexCount_ = 0,
                                           const uint32_t* pDeviceIndices_ = nullptr )
      : accelerationStructure( accelerationStructure_ )
      , memory( memory_ )
      , memoryOffset( memoryOffset_ )
      , deviceIndexCount( deviceIndexCount_ )
      , pDeviceIndices( pDeviceIndices_ )
    {}

    BindAccelerationStructureMemoryInfoNV( VkBindAccelerationStructureMemoryInfoNV const & rhs )
    {
      memcpy( this, &rhs, sizeof( BindAccelerationStructureMemoryInfoNV ) );
    }

    BindAccelerationStructureMemoryInfoNV& operator=( VkBindAccelerationStructureMemoryInfoNV const & rhs )
    {
      memcpy( this, &rhs, sizeof( BindAccelerationStructureMemoryInfoNV ) );
      return *this;
    }

    BindAccelerationStructureMemoryInfoNV & setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    BindAccelerationStructureMemoryInfoNV & setAccelerationStructure( AccelerationStructureNV accelerationStructure_ )
    {
      accelerationStructure = accelerationStructure_;
      return *this;
    }

    BindAccelerationStructureMemoryInfoNV & setMemory( DeviceMemory memory_ )
    {
      memory = memory_;
      return *this;
    }

    BindAccelerationStructureMemoryInfoNV & setMemoryOffset( DeviceSize memoryOffset_ )
    {
      memoryOffset = memoryOffset_;
      return *this;
    }

    BindAccelerationStructureMemoryInfoNV & setDeviceIndexCount( uint32_t deviceIndexCount_ )
    {
      deviceIndexCount = deviceIndexCount_;
      return *this;
    }

    BindAccelerationStructureMemoryInfoNV & setPDeviceIndices( const uint32_t* pDeviceIndices_ )
    {
      pDeviceIndices = pDeviceIndices_;
      return *this;
    }

    operator VkBindAccelerationStructureMemoryInfoNV const&() const
    {
      return *reinterpret_cast<const VkBindAccelerationStructureMemoryInfoNV*>( this );
    }

    operator VkBindAccelerationStructureMemoryInfoNV &()
    {
      return *reinterpret_cast<VkBindAccelerationStructureMemoryInfoNV*>( this );
    }

    bool operator==( BindAccelerationStructureMemoryInfoNV const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( accelerationStructure == rhs.accelerationStructure )
          && ( memory == rhs.memory )
          && ( memoryOffset == rhs.memoryOffset )
          && ( deviceIndexCount == rhs.deviceIndexCount )
          && ( pDeviceIndices == rhs.pDeviceIndices );
    }

    bool operator!=( BindAccelerationStructureMemoryInfoNV const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eBindAccelerationStructureMemoryInfoNV;

  public:
    const void* pNext = nullptr;
    AccelerationStructureNV accelerationStructure;
    DeviceMemory memory;
    DeviceSize memoryOffset;
    uint32_t deviceIndexCount;
    const uint32_t* pDeviceIndices;
  };
  static_assert( sizeof( BindAccelerationStructureMemoryInfoNV ) == sizeof( VkBindAccelerationStructureMemoryInfoNV ), "struct and wrapper have different size!" );

  struct BindBufferMemoryDeviceGroupInfo
  {
    BindBufferMemoryDeviceGroupInfo( uint32_t deviceIndexCount_ = 0,
                                     const uint32_t* pDeviceIndices_ = nullptr )
      : deviceIndexCount( deviceIndexCount_ )
      , pDeviceIndices( pDeviceIndices_ )
    {}

    BindBufferMemoryDeviceGroupInfo( VkBindBufferMemoryDeviceGroupInfo const & rhs )
    {
      memcpy( this, &rhs, sizeof( BindBufferMemoryDeviceGroupInfo ) );
    }

    BindBufferMemoryDeviceGroupInfo& operator=( VkBindBufferMemoryDeviceGroupInfo const & rhs )
    {
      memcpy( this, &rhs, sizeof( BindBufferMemoryDeviceGroupInfo ) );
      return *this;
    }

    BindBufferMemoryDeviceGroupInfo & setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    BindBufferMemoryDeviceGroupInfo & setDeviceIndexCount( uint32_t deviceIndexCount_ )
    {
      deviceIndexCount = deviceIndexCount_;
      return *this;
    }

    BindBufferMemoryDeviceGroupInfo & setPDeviceIndices( const uint32_t* pDeviceIndices_ )
    {
      pDeviceIndices = pDeviceIndices_;
      return *this;
    }

    operator VkBindBufferMemoryDeviceGroupInfo const&() const
    {
      return *reinterpret_cast<const VkBindBufferMemoryDeviceGroupInfo*>( this );
    }

    operator VkBindBufferMemoryDeviceGroupInfo &()
    {
      return *reinterpret_cast<VkBindBufferMemoryDeviceGroupInfo*>( this );
    }

    bool operator==( BindBufferMemoryDeviceGroupInfo const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( deviceIndexCount == rhs.deviceIndexCount )
          && ( pDeviceIndices == rhs.pDeviceIndices );
    }

    bool operator!=( BindBufferMemoryDeviceGroupInfo const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eBindBufferMemoryDeviceGroupInfo;

  public:
    const void* pNext = nullptr;
    uint32_t deviceIndexCount;
    const uint32_t* pDeviceIndices;
  };
  static_assert( sizeof( BindBufferMemoryDeviceGroupInfo ) == sizeof( VkBindBufferMemoryDeviceGroupInfo ), "struct and wrapper have different size!" );

  struct BindBufferMemoryInfo
  {
    BindBufferMemoryInfo( Buffer buffer_ = Buffer(),
                          DeviceMemory memory_ = DeviceMemory(),
                          DeviceSize memoryOffset_ = 0 )
      : buffer( buffer_ )
      , memory( memory_ )
      , memoryOffset( memoryOffset_ )
    {}

    BindBufferMemoryInfo( VkBindBufferMemoryInfo const & rhs )
    {
      memcpy( this, &rhs, sizeof( BindBufferMemoryInfo ) );
    }

    BindBufferMemoryInfo& operator=( VkBindBufferMemoryInfo const & rhs )
    {
      memcpy( this, &rhs, sizeof( BindBufferMemoryInfo ) );
      return *this;
    }

    BindBufferMemoryInfo & setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    BindBufferMemoryInfo & setBuffer( Buffer buffer_ )
    {
      buffer = buffer_;
      return *this;
    }

    BindBufferMemoryInfo & setMemory( DeviceMemory memory_ )
    {
      memory = memory_;
      return *this;
    }

    BindBufferMemoryInfo & setMemoryOffset( DeviceSize memoryOffset_ )
    {
      memoryOffset = memoryOffset_;
      return *this;
    }

    operator VkBindBufferMemoryInfo const&() const
    {
      return *reinterpret_cast<const VkBindBufferMemoryInfo*>( this );
    }

    operator VkBindBufferMemoryInfo &()
    {
      return *reinterpret_cast<VkBindBufferMemoryInfo*>( this );
    }

    bool operator==( BindBufferMemoryInfo const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( buffer == rhs.buffer )
          && ( memory == rhs.memory )
          && ( memoryOffset == rhs.memoryOffset );
    }

    bool operator!=( BindBufferMemoryInfo const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eBindBufferMemoryInfo;

  public:
    const void* pNext = nullptr;
    Buffer buffer;
    DeviceMemory memory;
    DeviceSize memoryOffset;
  };
  static_assert( sizeof( BindBufferMemoryInfo ) == sizeof( VkBindBufferMemoryInfo ), "struct and wrapper have different size!" );

  struct Offset2D
  {
    Offset2D( int32_t x_ = 0,
              int32_t y_ = 0 )
      : x( x_ )
      , y( y_ )
    {}

    Offset2D( VkOffset2D const & rhs )
    {
      memcpy( this, &rhs, sizeof( Offset2D ) );
    }

    Offset2D& operator=( VkOffset2D const & rhs )
    {
      memcpy( this, &rhs, sizeof( Offset2D ) );
      return *this;
    }

    Offset2D & setX( int32_t x_ )
    {
      x = x_;
      return *this;
    }

    Offset2D & setY( int32_t y_ )
    {
      y = y_;
      return *this;
    }

    operator VkOffset2D const&() const
    {
      return *reinterpret_cast<const VkOffset2D*>( this );
    }

    operator VkOffset2D &()
    {
      return *reinterpret_cast<VkOffset2D*>( this );
    }

    bool operator==( Offset2D const& rhs ) const
    {
      return ( x == rhs.x )
          && ( y == rhs.y );
    }

    bool operator!=( Offset2D const& rhs ) const
    {
      return !operator==( rhs );
    }

    int32_t x;
    int32_t y;
  };
  static_assert( sizeof( Offset2D ) == sizeof( VkOffset2D ), "struct and wrapper have different size!" );

  struct Rect2D
  {
    Rect2D( Offset2D offset_ = Offset2D(),
            Extent2D extent_ = Extent2D() )
      : offset( offset_ )
      , extent( extent_ )
    {}

    Rect2D( VkRect2D const & rhs )
    {
      memcpy( this, &rhs, sizeof( Rect2D ) );
    }

    Rect2D& operator=( VkRect2D const & rhs )
    {
      memcpy( this, &rhs, sizeof( Rect2D ) );
      return *this;
    }

    Rect2D & setOffset( Offset2D offset_ )
    {
      offset = offset_;
      return *this;
    }

    Rect2D & setExtent( Extent2D extent_ )
    {
      extent = extent_;
      return *this;
    }

    operator VkRect2D const&() const
    {
      return *reinterpret_cast<const VkRect2D*>( this );
    }

    operator VkRect2D &()
    {
      return *reinterpret_cast<VkRect2D*>( this );
    }

    bool operator==( Rect2D const& rhs ) const
    {
      return ( offset == rhs.offset )
          && ( extent == rhs.extent );
    }

    bool operator!=( Rect2D const& rhs ) const
    {
      return !operator==( rhs );
    }

    Offset2D offset;
    Extent2D extent;
  };
  static_assert( sizeof( Rect2D ) == sizeof( VkRect2D ), "struct and wrapper have different size!" );

  struct BindImageMemoryDeviceGroupInfo
  {
    BindImageMemoryDeviceGroupInfo( uint32_t deviceIndexCount_ = 0,
                                    const uint32_t* pDeviceIndices_ = nullptr,
                                    uint32_t splitInstanceBindRegionCount_ = 0,
                                    const Rect2D* pSplitInstanceBindRegions_ = nullptr )
      : deviceIndexCount( deviceIndexCount_ )
      , pDeviceIndices( pDeviceIndices_ )
      , splitInstanceBindRegionCount( splitInstanceBindRegionCount_ )
      , pSplitInstanceBindRegions( pSplitInstanceBindRegions_ )
    {}

    BindImageMemoryDeviceGroupInfo( VkBindImageMemoryDeviceGroupInfo const & rhs )
    {
      memcpy( this, &rhs, sizeof( BindImageMemoryDeviceGroupInfo ) );
    }

    BindImageMemoryDeviceGroupInfo& operator=( VkBindImageMemoryDeviceGroupInfo const & rhs )
    {
      memcpy( this, &rhs, sizeof( BindImageMemoryDeviceGroupInfo ) );
      return *this;
    }

    BindImageMemoryDeviceGroupInfo & setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    BindImageMemoryDeviceGroupInfo & setDeviceIndexCount( uint32_t deviceIndexCount_ )
    {
      deviceIndexCount = deviceIndexCount_;
      return *this;
    }

    BindImageMemoryDeviceGroupInfo & setPDeviceIndices( const uint32_t* pDeviceIndices_ )
    {
      pDeviceIndices = pDeviceIndices_;
      return *this;
    }

    BindImageMemoryDeviceGroupInfo & setSplitInstanceBindRegionCount( uint32_t splitInstanceBindRegionCount_ )
    {
      splitInstanceBindRegionCount = splitInstanceBindRegionCount_;
      return *this;
    }

    BindImageMemoryDeviceGroupInfo & setPSplitInstanceBindRegions( const Rect2D* pSplitInstanceBindRegions_ )
    {
      pSplitInstanceBindRegions = pSplitInstanceBindRegions_;
      return *this;
    }

    operator VkBindImageMemoryDeviceGroupInfo const&() const
    {
      return *reinterpret_cast<const VkBindImageMemoryDeviceGroupInfo*>( this );
    }

    operator VkBindImageMemoryDeviceGroupInfo &()
    {
      return *reinterpret_cast<VkBindImageMemoryDeviceGroupInfo*>( this );
    }

    bool operator==( BindImageMemoryDeviceGroupInfo const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( deviceIndexCount == rhs.deviceIndexCount )
          && ( pDeviceIndices == rhs.pDeviceIndices )
          && ( splitInstanceBindRegionCount == rhs.splitInstanceBindRegionCount )
          && ( pSplitInstanceBindRegions == rhs.pSplitInstanceBindRegions );
    }

    bool operator!=( BindImageMemoryDeviceGroupInfo const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eBindImageMemoryDeviceGroupInfo;

  public:
    const void* pNext = nullptr;
    uint32_t deviceIndexCount;
    const uint32_t* pDeviceIndices;
    uint32_t splitInstanceBindRegionCount;
    const Rect2D* pSplitInstanceBindRegions;
  };
  static_assert( sizeof( BindImageMemoryDeviceGroupInfo ) == sizeof( VkBindImageMemoryDeviceGroupInfo ), "struct and wrapper have different size!" );

  struct BindImageMemoryInfo
  {
    BindImageMemoryInfo( Image image_ = Image(),
                         DeviceMemory memory_ = DeviceMemory(),
                         DeviceSize memoryOffset_ = 0 )
      : image( image_ )
      , memory( memory_ )
      , memoryOffset( memoryOffset_ )
    {}

    BindImageMemoryInfo( VkBindImageMemoryInfo const & rhs )
    {
      memcpy( this, &rhs, sizeof( BindImageMemoryInfo ) );
    }

    BindImageMemoryInfo& operator=( VkBindImageMemoryInfo const & rhs )
    {
      memcpy( this, &rhs, sizeof( BindImageMemoryInfo ) );
      return *this;
    }

    BindImageMemoryInfo & setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    BindImageMemoryInfo & setImage( Image image_ )
    {
      image = image_;
      return *this;
    }

    BindImageMemoryInfo & setMemory( DeviceMemory memory_ )
    {
      memory = memory_;
      return *this;
    }

    BindImageMemoryInfo & setMemoryOffset( DeviceSize memoryOffset_ )
    {
      memoryOffset = memoryOffset_;
      return *this;
    }

    operator VkBindImageMemoryInfo const&() const
    {
      return *reinterpret_cast<const VkBindImageMemoryInfo*>( this );
    }

    operator VkBindImageMemoryInfo &()
    {
      return *reinterpret_cast<VkBindImageMemoryInfo*>( this );
    }

    bool operator==( BindImageMemoryInfo const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( image == rhs.image )
          && ( memory == rhs.memory )
          && ( memoryOffset == rhs.memoryOffset );
    }

    bool operator!=( BindImageMemoryInfo const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eBindImageMemoryInfo;

  public:
    const void* pNext = nullptr;
    Image image;
    DeviceMemory memory;
    DeviceSize memoryOffset;
  };
  static_assert( sizeof( BindImageMemoryInfo ) == sizeof( VkBindImageMemoryInfo ), "struct and wrapper have different size!" );

  struct BindImageMemorySwapchainInfoKHR
  {
    BindImageMemorySwapchainInfoKHR( SwapchainKHR swapchain_ = SwapchainKHR(),
                                     uint32_t imageIndex_ = 0 )
      : swapchain( swapchain_ )
      , imageIndex( imageIndex_ )
    {}

    BindImageMemorySwapchainInfoKHR( VkBindImageMemorySwapchainInfoKHR const & rhs )
    {
      memcpy( this, &rhs, sizeof( BindImageMemorySwapchainInfoKHR ) );
    }

    BindImageMemorySwapchainInfoKHR& operator=( VkBindImageMemorySwapchainInfoKHR const & rhs )
    {
      memcpy( this, &rhs, sizeof( BindImageMemorySwapchainInfoKHR ) );
      return *this;
    }

    BindImageMemorySwapchainInfoKHR & setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    BindImageMemorySwapchainInfoKHR & setSwapchain( SwapchainKHR swapchain_ )
    {
      swapchain = swapchain_;
      return *this;
    }

    BindImageMemorySwapchainInfoKHR & setImageIndex( uint32_t imageIndex_ )
    {
      imageIndex = imageIndex_;
      return *this;
    }

    operator VkBindImageMemorySwapchainInfoKHR const&() const
    {
      return *reinterpret_cast<const VkBindImageMemorySwapchainInfoKHR*>( this );
    }

    operator VkBindImageMemorySwapchainInfoKHR &()
    {
      return *reinterpret_cast<VkBindImageMemorySwapchainInfoKHR*>( this );
    }

    bool operator==( BindImageMemorySwapchainInfoKHR const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( swapchain == rhs.swapchain )
          && ( imageIndex == rhs.imageIndex );
    }

    bool operator!=( BindImageMemorySwapchainInfoKHR const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eBindImageMemorySwapchainInfoKHR;

  public:
    const void* pNext = nullptr;
    SwapchainKHR swapchain;
    uint32_t imageIndex;
  };
  static_assert( sizeof( BindImageMemorySwapchainInfoKHR ) == sizeof( VkBindImageMemorySwapchainInfoKHR ), "struct and wrapper have different size!" );

  struct BindImagePlaneMemoryInfo
  {
    BindImagePlaneMemoryInfo( ImageAspectFlagBits planeAspect_ = ImageAspectFlagBits::eColor )
      : planeAspect( planeAspect_ )
    {}

    BindImagePlaneMemoryInfo( VkBindImagePlaneMemoryInfo const & rhs )
    {
      memcpy( this, &rhs, sizeof( BindImagePlaneMemoryInfo ) );
    }

    BindImagePlaneMemoryInfo& operator=( VkBindImagePlaneMemoryInfo const & rhs )
    {
      memcpy( this, &rhs, sizeof( BindImagePlaneMemoryInfo ) );
      return *this;
    }

    BindImagePlaneMemoryInfo & setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    BindImagePlaneMemoryInfo & setPlaneAspect( ImageAspectFlagBits planeAspect_ )
    {
      planeAspect = planeAspect_;
      return *this;
    }

    operator VkBindImagePlaneMemoryInfo const&() const
    {
      return *reinterpret_cast<const VkBindImagePlaneMemoryInfo*>( this );
    }

    operator VkBindImagePlaneMemoryInfo &()
    {
      return *reinterpret_cast<VkBindImagePlaneMemoryInfo*>( this );
    }

    bool operator==( BindImagePlaneMemoryInfo const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( planeAspect == rhs.planeAspect );
    }

    bool operator!=( BindImagePlaneMemoryInfo const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eBindImagePlaneMemoryInfo;

  public:
    const void* pNext = nullptr;
    ImageAspectFlagBits planeAspect;
  };
  static_assert( sizeof( BindImagePlaneMemoryInfo ) == sizeof( VkBindImagePlaneMemoryInfo ), "struct and wrapper have different size!" );

  struct SparseMemoryBind
  {
    SparseMemoryBind( DeviceSize resourceOffset_ = 0,
                      DeviceSize size_ = 0,
                      DeviceMemory memory_ = DeviceMemory(),
                      DeviceSize memoryOffset_ = 0,
                      SparseMemoryBindFlags flags_ = SparseMemoryBindFlags() )
      : resourceOffset( resourceOffset_ )
      , size( size_ )
      , memory( memory_ )
      , memoryOffset( memoryOffset_ )
      , flags( flags_ )
    {}

    SparseMemoryBind( VkSparseMemoryBind const & rhs )
    {
      memcpy( this, &rhs, sizeof( SparseMemoryBind ) );
    }

    SparseMemoryBind& operator=( VkSparseMemoryBind const & rhs )
    {
      memcpy( this, &rhs, sizeof( SparseMemoryBind ) );
      return *this;
    }

    SparseMemoryBind & setResourceOffset( DeviceSize resourceOffset_ )
    {
      resourceOffset = resourceOffset_;
      return *this;
    }

    SparseMemoryBind & setSize( DeviceSize size_ )
    {
      size = size_;
      return *this;
    }

    SparseMemoryBind & setMemory( DeviceMemory memory_ )
    {
      memory = memory_;
      return *this;
    }

    SparseMemoryBind & setMemoryOffset( DeviceSize memoryOffset_ )
    {
      memoryOffset = memoryOffset_;
      return *this;
    }

    SparseMemoryBind & setFlags( SparseMemoryBindFlags flags_ )
    {
      flags = flags_;
      return *this;
    }

    operator VkSparseMemoryBind const&() const
    {
      return *reinterpret_cast<const VkSparseMemoryBind*>( this );
    }

    operator VkSparseMemoryBind &()
    {
      return *reinterpret_cast<VkSparseMemoryBind*>( this );
    }

    bool operator==( SparseMemoryBind const& rhs ) const
    {
      return ( resourceOffset == rhs.resourceOffset )
          && ( size == rhs.size )
          && ( memory == rhs.memory )
          && ( memoryOffset == rhs.memoryOffset )
          && ( flags == rhs.flags );
    }

    bool operator!=( SparseMemoryBind const& rhs ) const
    {
      return !operator==( rhs );
    }

    DeviceSize resourceOffset;
    DeviceSize size;
    DeviceMemory memory;
    DeviceSize memoryOffset;
    SparseMemoryBindFlags flags;
  };
  static_assert( sizeof( SparseMemoryBind ) == sizeof( VkSparseMemoryBind ), "struct and wrapper have different size!" );

  struct SparseBufferMemoryBindInfo
  {
    SparseBufferMemoryBindInfo( Buffer buffer_ = Buffer(),
                                uint32_t bindCount_ = 0,
                                const SparseMemoryBind* pBinds_ = nullptr )
      : buffer( buffer_ )
      , bindCount( bindCount_ )
      , pBinds( pBinds_ )
    {}

    SparseBufferMemoryBindInfo( VkSparseBufferMemoryBindInfo const & rhs )
    {
      memcpy( this, &rhs, sizeof( SparseBufferMemoryBindInfo ) );
    }

    SparseBufferMemoryBindInfo& operator=( VkSparseBufferMemoryBindInfo const & rhs )
    {
      memcpy( this, &rhs, sizeof( SparseBufferMemoryBindInfo ) );
      return *this;
    }

    SparseBufferMemoryBindInfo & setBuffer( Buffer buffer_ )
    {
      buffer = buffer_;
      return *this;
    }

    SparseBufferMemoryBindInfo & setBindCount( uint32_t bindCount_ )
    {
      bindCount = bindCount_;
      return *this;
    }

    SparseBufferMemoryBindInfo & setPBinds( const SparseMemoryBind* pBinds_ )
    {
      pBinds = pBinds_;
      return *this;
    }

    operator VkSparseBufferMemoryBindInfo const&() const
    {
      return *reinterpret_cast<const VkSparseBufferMemoryBindInfo*>( this );
    }

    operator VkSparseBufferMemoryBindInfo &()
    {
      return *reinterpret_cast<VkSparseBufferMemoryBindInfo*>( this );
    }

    bool operator==( SparseBufferMemoryBindInfo const& rhs ) const
    {
      return ( buffer == rhs.buffer )
          && ( bindCount == rhs.bindCount )
          && ( pBinds == rhs.pBinds );
    }

    bool operator!=( SparseBufferMemoryBindInfo const& rhs ) const
    {
      return !operator==( rhs );
    }

    Buffer buffer;
    uint32_t bindCount;
    const SparseMemoryBind* pBinds;
  };
  static_assert( sizeof( SparseBufferMemoryBindInfo ) == sizeof( VkSparseBufferMemoryBindInfo ), "struct and wrapper have different size!" );

  struct SparseImageOpaqueMemoryBindInfo
  {
    SparseImageOpaqueMemoryBindInfo( Image image_ = Image(),
                                     uint32_t bindCount_ = 0,
                                     const SparseMemoryBind* pBinds_ = nullptr )
      : image( image_ )
      , bindCount( bindCount_ )
      , pBinds( pBinds_ )
    {}

    SparseImageOpaqueMemoryBindInfo( VkSparseImageOpaqueMemoryBindInfo const & rhs )
    {
      memcpy( this, &rhs, sizeof( SparseImageOpaqueMemoryBindInfo ) );
    }

    SparseImageOpaqueMemoryBindInfo& operator=( VkSparseImageOpaqueMemoryBindInfo const & rhs )
    {
      memcpy( this, &rhs, sizeof( SparseImageOpaqueMemoryBindInfo ) );
      return *this;
    }

    SparseImageOpaqueMemoryBindInfo & setImage( Image image_ )
    {
      image = image_;
      return *this;
    }

    SparseImageOpaqueMemoryBindInfo & setBindCount( uint32_t bindCount_ )
    {
      bindCount = bindCount_;
      return *this;
    }

    SparseImageOpaqueMemoryBindInfo & setPBinds( const SparseMemoryBind* pBinds_ )
    {
      pBinds = pBinds_;
      return *this;
    }

    operator VkSparseImageOpaqueMemoryBindInfo const&() const
    {
      return *reinterpret_cast<const VkSparseImageOpaqueMemoryBindInfo*>( this );
    }

    operator VkSparseImageOpaqueMemoryBindInfo &()
    {
      return *reinterpret_cast<VkSparseImageOpaqueMemoryBindInfo*>( this );
    }

    bool operator==( SparseImageOpaqueMemoryBindInfo const& rhs ) const
    {
      return ( image == rhs.image )
          && ( bindCount == rhs.bindCount )
          && ( pBinds == rhs.pBinds );
    }

    bool operator!=( SparseImageOpaqueMemoryBindInfo const& rhs ) const
    {
      return !operator==( rhs );
    }

    Image image;
    uint32_t bindCount;
    const SparseMemoryBind* pBinds;
  };
  static_assert( sizeof( SparseImageOpaqueMemoryBindInfo ) == sizeof( VkSparseImageOpaqueMemoryBindInfo ), "struct and wrapper have different size!" );

  struct ImageSubresource
  {
    ImageSubresource( ImageAspectFlags aspectMask_ = ImageAspectFlags(),
                      uint32_t mipLevel_ = 0,
                      uint32_t arrayLayer_ = 0 )
      : aspectMask( aspectMask_ )
      , mipLevel( mipLevel_ )
      , arrayLayer( arrayLayer_ )
    {}

    ImageSubresource( VkImageSubresource const & rhs )
    {
      memcpy( this, &rhs, sizeof( ImageSubresource ) );
    }

    ImageSubresource& operator=( VkImageSubresource const & rhs )
    {
      memcpy( this, &rhs, sizeof( ImageSubresource ) );
      return *this;
    }

    ImageSubresource & setAspectMask( ImageAspectFlags aspectMask_ )
    {
      aspectMask = aspectMask_;
      return *this;
    }

    ImageSubresource & setMipLevel( uint32_t mipLevel_ )
    {
      mipLevel = mipLevel_;
      return *this;
    }

    ImageSubresource & setArrayLayer( uint32_t arrayLayer_ )
    {
      arrayLayer = arrayLayer_;
      return *this;
    }

    operator VkImageSubresource const&() const
    {
      return *reinterpret_cast<const VkImageSubresource*>( this );
    }

    operator VkImageSubresource &()
    {
      return *reinterpret_cast<VkImageSubresource*>( this );
    }

    bool operator==( ImageSubresource const& rhs ) const
    {
      return ( aspectMask == rhs.aspectMask )
          && ( mipLevel == rhs.mipLevel )
          && ( arrayLayer == rhs.arrayLayer );
    }

    bool operator!=( ImageSubresource const& rhs ) const
    {
      return !operator==( rhs );
    }

    ImageAspectFlags aspectMask;
    uint32_t mipLevel;
    uint32_t arrayLayer;
  };
  static_assert( sizeof( ImageSubresource ) == sizeof( VkImageSubresource ), "struct and wrapper have different size!" );

  struct Offset3D
  {
    Offset3D( int32_t x_ = 0,
              int32_t y_ = 0,
              int32_t z_ = 0 )
      : x( x_ )
      , y( y_ )
      , z( z_ )
    {}

    explicit Offset3D( Offset2D const& offset2D,
                       int32_t z_ = 0 )
      : x( offset2D.x )
      , y( offset2D.y )
      , z( z_ )
    {}

    Offset3D( VkOffset3D const & rhs )
    {
      memcpy( this, &rhs, sizeof( Offset3D ) );
    }

    Offset3D& operator=( VkOffset3D const & rhs )
    {
      memcpy( this, &rhs, sizeof( Offset3D ) );
      return *this;
    }

    Offset3D & setX( int32_t x_ )
    {
      x = x_;
      return *this;
    }

    Offset3D & setY( int32_t y_ )
    {
      y = y_;
      return *this;
    }

    Offset3D & setZ( int32_t z_ )
    {
      z = z_;
      return *this;
    }

    operator VkOffset3D const&() const
    {
      return *reinterpret_cast<const VkOffset3D*>( this );
    }

    operator VkOffset3D &()
    {
      return *reinterpret_cast<VkOffset3D*>( this );
    }

    bool operator==( Offset3D const& rhs ) const
    {
      return ( x == rhs.x )
          && ( y == rhs.y )
          && ( z == rhs.z );
    }

    bool operator!=( Offset3D const& rhs ) const
    {
      return !operator==( rhs );
    }

    int32_t x;
    int32_t y;
    int32_t z;
  };
  static_assert( sizeof( Offset3D ) == sizeof( VkOffset3D ), "struct and wrapper have different size!" );

  struct Extent3D
  {
    Extent3D( uint32_t width_ = 0,
              uint32_t height_ = 0,
              uint32_t depth_ = 0 )
      : width( width_ )
      , height( height_ )
      , depth( depth_ )
    {}

    explicit Extent3D( Extent2D const& extent2D,
                       uint32_t depth_ = 0 )
      : width( extent2D.width )
      , height( extent2D.height )
      , depth( depth_ )
    {}

    Extent3D( VkExtent3D const & rhs )
    {
      memcpy( this, &rhs, sizeof( Extent3D ) );
    }

    Extent3D& operator=( VkExtent3D const & rhs )
    {
      memcpy( this, &rhs, sizeof( Extent3D ) );
      return *this;
    }

    Extent3D & setWidth( uint32_t width_ )
    {
      width = width_;
      return *this;
    }

    Extent3D & setHeight( uint32_t height_ )
    {
      height = height_;
      return *this;
    }

    Extent3D & setDepth( uint32_t depth_ )
    {
      depth = depth_;
      return *this;
    }

    operator VkExtent3D const&() const
    {
      return *reinterpret_cast<const VkExtent3D*>( this );
    }

    operator VkExtent3D &()
    {
      return *reinterpret_cast<VkExtent3D*>( this );
    }

    bool operator==( Extent3D const& rhs ) const
    {
      return ( width == rhs.width )
          && ( height == rhs.height )
          && ( depth == rhs.depth );
    }

    bool operator!=( Extent3D const& rhs ) const
    {
      return !operator==( rhs );
    }

    uint32_t width;
    uint32_t height;
    uint32_t depth;
  };
  static_assert( sizeof( Extent3D ) == sizeof( VkExtent3D ), "struct and wrapper have different size!" );

  struct SparseImageMemoryBind
  {
    SparseImageMemoryBind( ImageSubresource subresource_ = ImageSubresource(),
                           Offset3D offset_ = Offset3D(),
                           Extent3D extent_ = Extent3D(),
                           DeviceMemory memory_ = DeviceMemory(),
                           DeviceSize memoryOffset_ = 0,
                           SparseMemoryBindFlags flags_ = SparseMemoryBindFlags() )
      : subresource( subresource_ )
      , offset( offset_ )
      , extent( extent_ )
      , memory( memory_ )
      , memoryOffset( memoryOffset_ )
      , flags( flags_ )
    {}

    SparseImageMemoryBind( VkSparseImageMemoryBind const & rhs )
    {
      memcpy( this, &rhs, sizeof( SparseImageMemoryBind ) );
    }

    SparseImageMemoryBind& operator=( VkSparseImageMemoryBind const & rhs )
    {
      memcpy( this, &rhs, sizeof( SparseImageMemoryBind ) );
      return *this;
    }

    SparseImageMemoryBind & setSubresource( ImageSubresource subresource_ )
    {
      subresource = subresource_;
      return *this;
    }

    SparseImageMemoryBind & setOffset( Offset3D offset_ )
    {
      offset = offset_;
      return *this;
    }

    SparseImageMemoryBind & setExtent( Extent3D extent_ )
    {
      extent = extent_;
      return *this;
    }

    SparseImageMemoryBind & setMemory( DeviceMemory memory_ )
    {
      memory = memory_;
      return *this;
    }

    SparseImageMemoryBind & setMemoryOffset( DeviceSize memoryOffset_ )
    {
      memoryOffset = memoryOffset_;
      return *this;
    }

    SparseImageMemoryBind & setFlags( SparseMemoryBindFlags flags_ )
    {
      flags = flags_;
      return *this;
    }

    operator VkSparseImageMemoryBind const&() const
    {
      return *reinterpret_cast<const VkSparseImageMemoryBind*>( this );
    }

    operator VkSparseImageMemoryBind &()
    {
      return *reinterpret_cast<VkSparseImageMemoryBind*>( this );
    }

    bool operator==( SparseImageMemoryBind const& rhs ) const
    {
      return ( subresource == rhs.subresource )
          && ( offset == rhs.offset )
          && ( extent == rhs.extent )
          && ( memory == rhs.memory )
          && ( memoryOffset == rhs.memoryOffset )
          && ( flags == rhs.flags );
    }

    bool operator!=( SparseImageMemoryBind const& rhs ) const
    {
      return !operator==( rhs );
    }

    ImageSubresource subresource;
    Offset3D offset;
    Extent3D extent;
    DeviceMemory memory;
    DeviceSize memoryOffset;
    SparseMemoryBindFlags flags;
  };
  static_assert( sizeof( SparseImageMemoryBind ) == sizeof( VkSparseImageMemoryBind ), "struct and wrapper have different size!" );

  struct SparseImageMemoryBindInfo
  {
    SparseImageMemoryBindInfo( Image image_ = Image(),
                               uint32_t bindCount_ = 0,
                               const SparseImageMemoryBind* pBinds_ = nullptr )
      : image( image_ )
      , bindCount( bindCount_ )
      , pBinds( pBinds_ )
    {}

    SparseImageMemoryBindInfo( VkSparseImageMemoryBindInfo const & rhs )
    {
      memcpy( this, &rhs, sizeof( SparseImageMemoryBindInfo ) );
    }

    SparseImageMemoryBindInfo& operator=( VkSparseImageMemoryBindInfo const & rhs )
    {
      memcpy( this, &rhs, sizeof( SparseImageMemoryBindInfo ) );
      return *this;
    }

    SparseImageMemoryBindInfo & setImage( Image image_ )
    {
      image = image_;
      return *this;
    }

    SparseImageMemoryBindInfo & setBindCount( uint32_t bindCount_ )
    {
      bindCount = bindCount_;
      return *this;
    }

    SparseImageMemoryBindInfo & setPBinds( const SparseImageMemoryBind* pBinds_ )
    {
      pBinds = pBinds_;
      return *this;
    }

    operator VkSparseImageMemoryBindInfo const&() const
    {
      return *reinterpret_cast<const VkSparseImageMemoryBindInfo*>( this );
    }

    operator VkSparseImageMemoryBindInfo &()
    {
      return *reinterpret_cast<VkSparseImageMemoryBindInfo*>( this );
    }

    bool operator==( SparseImageMemoryBindInfo const& rhs ) const
    {
      return ( image == rhs.image )
          && ( bindCount == rhs.bindCount )
          && ( pBinds == rhs.pBinds );
    }

    bool operator!=( SparseImageMemoryBindInfo const& rhs ) const
    {
      return !operator==( rhs );
    }

    Image image;
    uint32_t bindCount;
    const SparseImageMemoryBind* pBinds;
  };
  static_assert( sizeof( SparseImageMemoryBindInfo ) == sizeof( VkSparseImageMemoryBindInfo ), "struct and wrapper have different size!" );

  struct BindSparseInfo
  {
    BindSparseInfo( uint32_t waitSemaphoreCount_ = 0,
                    const Semaphore* pWaitSemaphores_ = nullptr,
                    uint32_t bufferBindCount_ = 0,
                    const SparseBufferMemoryBindInfo* pBufferBinds_ = nullptr,
                    uint32_t imageOpaqueBindCount_ = 0,
                    const SparseImageOpaqueMemoryBindInfo* pImageOpaqueBinds_ = nullptr,
                    uint32_t imageBindCount_ = 0,
                    const SparseImageMemoryBindInfo* pImageBinds_ = nullptr,
                    uint32_t signalSemaphoreCount_ = 0,
                    const Semaphore* pSignalSemaphores_ = nullptr )
      : waitSemaphoreCount( waitSemaphoreCount_ )
      , pWaitSemaphores( pWaitSemaphores_ )
      , bufferBindCount( bufferBindCount_ )
      , pBufferBinds( pBufferBinds_ )
      , imageOpaqueBindCount( imageOpaqueBindCount_ )
      , pImageOpaqueBinds( pImageOpaqueBinds_ )
      , imageBindCount( imageBindCount_ )
      , pImageBinds( pImageBinds_ )
      , signalSemaphoreCount( signalSemaphoreCount_ )
      , pSignalSemaphores( pSignalSemaphores_ )
    {}

    BindSparseInfo( VkBindSparseInfo const & rhs )
    {
      memcpy( this, &rhs, sizeof( BindSparseInfo ) );
    }

    BindSparseInfo& operator=( VkBindSparseInfo const & rhs )
    {
      memcpy( this, &rhs, sizeof( BindSparseInfo ) );
      return *this;
    }

    BindSparseInfo & setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    BindSparseInfo & setWaitSemaphoreCount( uint32_t waitSemaphoreCount_ )
    {
      waitSemaphoreCount = waitSemaphoreCount_;
      return *this;
    }

    BindSparseInfo & setPWaitSemaphores( const Semaphore* pWaitSemaphores_ )
    {
      pWaitSemaphores = pWaitSemaphores_;
      return *this;
    }

    BindSparseInfo & setBufferBindCount( uint32_t bufferBindCount_ )
    {
      bufferBindCount = bufferBindCount_;
      return *this;
    }

    BindSparseInfo & setPBufferBinds( const SparseBufferMemoryBindInfo* pBufferBinds_ )
    {
      pBufferBinds = pBufferBinds_;
      return *this;
    }

    BindSparseInfo & setImageOpaqueBindCount( uint32_t imageOpaqueBindCount_ )
    {
      imageOpaqueBindCount = imageOpaqueBindCount_;
      return *this;
    }

    BindSparseInfo & setPImageOpaqueBinds( const SparseImageOpaqueMemoryBindInfo* pImageOpaqueBinds_ )
    {
      pImageOpaqueBinds = pImageOpaqueBinds_;
      return *this;
    }

    BindSparseInfo & setImageBindCount( uint32_t imageBindCount_ )
    {
      imageBindCount = imageBindCount_;
      return *this;
    }

    BindSparseInfo & setPImageBinds( const SparseImageMemoryBindInfo* pImageBinds_ )
    {
      pImageBinds = pImageBinds_;
      return *this;
    }

    BindSparseInfo & setSignalSemaphoreCount( uint32_t signalSemaphoreCount_ )
    {
      signalSemaphoreCount = signalSemaphoreCount_;
      return *this;
    }

    BindSparseInfo & setPSignalSemaphores( const Semaphore* pSignalSemaphores_ )
    {
      pSignalSemaphores = pSignalSemaphores_;
      return *this;
    }

    operator VkBindSparseInfo const&() const
    {
      return *reinterpret_cast<const VkBindSparseInfo*>( this );
    }

    operator VkBindSparseInfo &()
    {
      return *reinterpret_cast<VkBindSparseInfo*>( this );
    }

    bool operator==( BindSparseInfo const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( waitSemaphoreCount == rhs.waitSemaphoreCount )
          && ( pWaitSemaphores == rhs.pWaitSemaphores )
          && ( bufferBindCount == rhs.bufferBindCount )
          && ( pBufferBinds == rhs.pBufferBinds )
          && ( imageOpaqueBindCount == rhs.imageOpaqueBindCount )
          && ( pImageOpaqueBinds == rhs.pImageOpaqueBinds )
          && ( imageBindCount == rhs.imageBindCount )
          && ( pImageBinds == rhs.pImageBinds )
          && ( signalSemaphoreCount == rhs.signalSemaphoreCount )
          && ( pSignalSemaphores == rhs.pSignalSemaphores );
    }

    bool operator!=( BindSparseInfo const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eBindSparseInfo;

  public:
    const void* pNext = nullptr;
    uint32_t waitSemaphoreCount;
    const Semaphore* pWaitSemaphores;
    uint32_t bufferBindCount;
    const SparseBufferMemoryBindInfo* pBufferBinds;
    uint32_t imageOpaqueBindCount;
    const SparseImageOpaqueMemoryBindInfo* pImageOpaqueBinds;
    uint32_t imageBindCount;
    const SparseImageMemoryBindInfo* pImageBinds;
    uint32_t signalSemaphoreCount;
    const Semaphore* pSignalSemaphores;
  };
  static_assert( sizeof( BindSparseInfo ) == sizeof( VkBindSparseInfo ), "struct and wrapper have different size!" );

  struct BufferCopy
  {
    BufferCopy( DeviceSize srcOffset_ = 0,
                DeviceSize dstOffset_ = 0,
                DeviceSize size_ = 0 )
      : srcOffset( srcOffset_ )
      , dstOffset( dstOffset_ )
      , size( size_ )
    {}

    BufferCopy( VkBufferCopy const & rhs )
    {
      memcpy( this, &rhs, sizeof( BufferCopy ) );
    }

    BufferCopy& operator=( VkBufferCopy const & rhs )
    {
      memcpy( this, &rhs, sizeof( BufferCopy ) );
      return *this;
    }

    BufferCopy & setSrcOffset( DeviceSize srcOffset_ )
    {
      srcOffset = srcOffset_;
      return *this;
    }

    BufferCopy & setDstOffset( DeviceSize dstOffset_ )
    {
      dstOffset = dstOffset_;
      return *this;
    }

    BufferCopy & setSize( DeviceSize size_ )
    {
      size = size_;
      return *this;
    }

    operator VkBufferCopy const&() const
    {
      return *reinterpret_cast<const VkBufferCopy*>( this );
    }

    operator VkBufferCopy &()
    {
      return *reinterpret_cast<VkBufferCopy*>( this );
    }

    bool operator==( BufferCopy const& rhs ) const
    {
      return ( srcOffset == rhs.srcOffset )
          && ( dstOffset == rhs.dstOffset )
          && ( size == rhs.size );
    }

    bool operator!=( BufferCopy const& rhs ) const
    {
      return !operator==( rhs );
    }

    DeviceSize srcOffset;
    DeviceSize dstOffset;
    DeviceSize size;
  };
  static_assert( sizeof( BufferCopy ) == sizeof( VkBufferCopy ), "struct and wrapper have different size!" );

  struct BufferCreateInfo
  {
    BufferCreateInfo( BufferCreateFlags flags_ = BufferCreateFlags(),
                      DeviceSize size_ = 0,
                      BufferUsageFlags usage_ = BufferUsageFlags(),
                      SharingMode sharingMode_ = SharingMode::eExclusive,
                      uint32_t queueFamilyIndexCount_ = 0,
                      const uint32_t* pQueueFamilyIndices_ = nullptr )
      : flags( flags_ )
      , size( size_ )
      , usage( usage_ )
      , sharingMode( sharingMode_ )
      , queueFamilyIndexCount( queueFamilyIndexCount_ )
      , pQueueFamilyIndices( pQueueFamilyIndices_ )
    {}

    BufferCreateInfo( VkBufferCreateInfo const & rhs )
    {
      memcpy( this, &rhs, sizeof( BufferCreateInfo ) );
    }

    BufferCreateInfo& operator=( VkBufferCreateInfo const & rhs )
    {
      memcpy( this, &rhs, sizeof( BufferCreateInfo ) );
      return *this;
    }

    BufferCreateInfo & setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    BufferCreateInfo & setFlags( BufferCreateFlags flags_ )
    {
      flags = flags_;
      return *this;
    }

    BufferCreateInfo & setSize( DeviceSize size_ )
    {
      size = size_;
      return *this;
    }

    BufferCreateInfo & setUsage( BufferUsageFlags usage_ )
    {
      usage = usage_;
      return *this;
    }

    BufferCreateInfo & setSharingMode( SharingMode sharingMode_ )
    {
      sharingMode = sharingMode_;
      return *this;
    }

    BufferCreateInfo & setQueueFamilyIndexCount( uint32_t queueFamilyIndexCount_ )
    {
      queueFamilyIndexCount = queueFamilyIndexCount_;
      return *this;
    }

    BufferCreateInfo & setPQueueFamilyIndices( const uint32_t* pQueueFamilyIndices_ )
    {
      pQueueFamilyIndices = pQueueFamilyIndices_;
      return *this;
    }

    operator VkBufferCreateInfo const&() const
    {
      return *reinterpret_cast<const VkBufferCreateInfo*>( this );
    }

    operator VkBufferCreateInfo &()
    {
      return *reinterpret_cast<VkBufferCreateInfo*>( this );
    }

    bool operator==( BufferCreateInfo const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( flags == rhs.flags )
          && ( size == rhs.size )
          && ( usage == rhs.usage )
          && ( sharingMode == rhs.sharingMode )
          && ( queueFamilyIndexCount == rhs.queueFamilyIndexCount )
          && ( pQueueFamilyIndices == rhs.pQueueFamilyIndices );
    }

    bool operator!=( BufferCreateInfo const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eBufferCreateInfo;

  public:
    const void* pNext = nullptr;
    BufferCreateFlags flags;
    DeviceSize size;
    BufferUsageFlags usage;
    SharingMode sharingMode;
    uint32_t queueFamilyIndexCount;
    const uint32_t* pQueueFamilyIndices;
  };
  static_assert( sizeof( BufferCreateInfo ) == sizeof( VkBufferCreateInfo ), "struct and wrapper have different size!" );

  struct BufferDeviceAddressCreateInfoEXT
  {
    BufferDeviceAddressCreateInfoEXT( DeviceSize deviceAddress_ = 0 )
      : deviceAddress( deviceAddress_ )
    {}

    BufferDeviceAddressCreateInfoEXT( VkBufferDeviceAddressCreateInfoEXT const & rhs )
    {
      memcpy( this, &rhs, sizeof( BufferDeviceAddressCreateInfoEXT ) );
    }

    BufferDeviceAddressCreateInfoEXT& operator=( VkBufferDeviceAddressCreateInfoEXT const & rhs )
    {
      memcpy( this, &rhs, sizeof( BufferDeviceAddressCreateInfoEXT ) );
      return *this;
    }

    BufferDeviceAddressCreateInfoEXT & setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    BufferDeviceAddressCreateInfoEXT & setDeviceAddress( DeviceSize deviceAddress_ )
    {
      deviceAddress = deviceAddress_;
      return *this;
    }

    operator VkBufferDeviceAddressCreateInfoEXT const&() const
    {
      return *reinterpret_cast<const VkBufferDeviceAddressCreateInfoEXT*>( this );
    }

    operator VkBufferDeviceAddressCreateInfoEXT &()
    {
      return *reinterpret_cast<VkBufferDeviceAddressCreateInfoEXT*>( this );
    }

    bool operator==( BufferDeviceAddressCreateInfoEXT const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( deviceAddress == rhs.deviceAddress );
    }

    bool operator!=( BufferDeviceAddressCreateInfoEXT const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eBufferDeviceAddressCreateInfoEXT;

  public:
    const void* pNext = nullptr;
    DeviceSize deviceAddress;
  };
  static_assert( sizeof( BufferDeviceAddressCreateInfoEXT ) == sizeof( VkBufferDeviceAddressCreateInfoEXT ), "struct and wrapper have different size!" );

  struct BufferDeviceAddressInfoEXT
  {
    BufferDeviceAddressInfoEXT( Buffer buffer_ = Buffer() )
      : buffer( buffer_ )
    {}

    BufferDeviceAddressInfoEXT( VkBufferDeviceAddressInfoEXT const & rhs )
    {
      memcpy( this, &rhs, sizeof( BufferDeviceAddressInfoEXT ) );
    }

    BufferDeviceAddressInfoEXT& operator=( VkBufferDeviceAddressInfoEXT const & rhs )
    {
      memcpy( this, &rhs, sizeof( BufferDeviceAddressInfoEXT ) );
      return *this;
    }

    BufferDeviceAddressInfoEXT & setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    BufferDeviceAddressInfoEXT & setBuffer( Buffer buffer_ )
    {
      buffer = buffer_;
      return *this;
    }

    operator VkBufferDeviceAddressInfoEXT const&() const
    {
      return *reinterpret_cast<const VkBufferDeviceAddressInfoEXT*>( this );
    }

    operator VkBufferDeviceAddressInfoEXT &()
    {
      return *reinterpret_cast<VkBufferDeviceAddressInfoEXT*>( this );
    }

    bool operator==( BufferDeviceAddressInfoEXT const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( buffer == rhs.buffer );
    }

    bool operator!=( BufferDeviceAddressInfoEXT const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eBufferDeviceAddressInfoEXT;

  public:
    const void* pNext = nullptr;
    Buffer buffer;
  };
  static_assert( sizeof( BufferDeviceAddressInfoEXT ) == sizeof( VkBufferDeviceAddressInfoEXT ), "struct and wrapper have different size!" );

  struct ImageSubresourceLayers
  {
    ImageSubresourceLayers( ImageAspectFlags aspectMask_ = ImageAspectFlags(),
                            uint32_t mipLevel_ = 0,
                            uint32_t baseArrayLayer_ = 0,
                            uint32_t layerCount_ = 0 )
      : aspectMask( aspectMask_ )
      , mipLevel( mipLevel_ )
      , baseArrayLayer( baseArrayLayer_ )
      , layerCount( layerCount_ )
    {}

    ImageSubresourceLayers( VkImageSubresourceLayers const & rhs )
    {
      memcpy( this, &rhs, sizeof( ImageSubresourceLayers ) );
    }

    ImageSubresourceLayers& operator=( VkImageSubresourceLayers const & rhs )
    {
      memcpy( this, &rhs, sizeof( ImageSubresourceLayers ) );
      return *this;
    }

    ImageSubresourceLayers & setAspectMask( ImageAspectFlags aspectMask_ )
    {
      aspectMask = aspectMask_;
      return *this;
    }

    ImageSubresourceLayers & setMipLevel( uint32_t mipLevel_ )
    {
      mipLevel = mipLevel_;
      return *this;
    }

    ImageSubresourceLayers & setBaseArrayLayer( uint32_t baseArrayLayer_ )
    {
      baseArrayLayer = baseArrayLayer_;
      return *this;
    }

    ImageSubresourceLayers & setLayerCount( uint32_t layerCount_ )
    {
      layerCount = layerCount_;
      return *this;
    }

    operator VkImageSubresourceLayers const&() const
    {
      return *reinterpret_cast<const VkImageSubresourceLayers*>( this );
    }

    operator VkImageSubresourceLayers &()
    {
      return *reinterpret_cast<VkImageSubresourceLayers*>( this );
    }

    bool operator==( ImageSubresourceLayers const& rhs ) const
    {
      return ( aspectMask == rhs.aspectMask )
          && ( mipLevel == rhs.mipLevel )
          && ( baseArrayLayer == rhs.baseArrayLayer )
          && ( layerCount == rhs.layerCount );
    }

    bool operator!=( ImageSubresourceLayers const& rhs ) const
    {
      return !operator==( rhs );
    }

    ImageAspectFlags aspectMask;
    uint32_t mipLevel;
    uint32_t baseArrayLayer;
    uint32_t layerCount;
  };
  static_assert( sizeof( ImageSubresourceLayers ) == sizeof( VkImageSubresourceLayers ), "struct and wrapper have different size!" );

  struct BufferImageCopy
  {
    BufferImageCopy( DeviceSize bufferOffset_ = 0,
                     uint32_t bufferRowLength_ = 0,
                     uint32_t bufferImageHeight_ = 0,
                     ImageSubresourceLayers imageSubresource_ = ImageSubresourceLayers(),
                     Offset3D imageOffset_ = Offset3D(),
                     Extent3D imageExtent_ = Extent3D() )
      : bufferOffset( bufferOffset_ )
      , bufferRowLength( bufferRowLength_ )
      , bufferImageHeight( bufferImageHeight_ )
      , imageSubresource( imageSubresource_ )
      , imageOffset( imageOffset_ )
      , imageExtent( imageExtent_ )
    {}

    BufferImageCopy( VkBufferImageCopy const & rhs )
    {
      memcpy( this, &rhs, sizeof( BufferImageCopy ) );
    }

    BufferImageCopy& operator=( VkBufferImageCopy const & rhs )
    {
      memcpy( this, &rhs, sizeof( BufferImageCopy ) );
      return *this;
    }

    BufferImageCopy & setBufferOffset( DeviceSize bufferOffset_ )
    {
      bufferOffset = bufferOffset_;
      return *this;
    }

    BufferImageCopy & setBufferRowLength( uint32_t bufferRowLength_ )
    {
      bufferRowLength = bufferRowLength_;
      return *this;
    }

    BufferImageCopy & setBufferImageHeight( uint32_t bufferImageHeight_ )
    {
      bufferImageHeight = bufferImageHeight_;
      return *this;
    }

    BufferImageCopy & setImageSubresource( ImageSubresourceLayers imageSubresource_ )
    {
      imageSubresource = imageSubresource_;
      return *this;
    }

    BufferImageCopy & setImageOffset( Offset3D imageOffset_ )
    {
      imageOffset = imageOffset_;
      return *this;
    }

    BufferImageCopy & setImageExtent( Extent3D imageExtent_ )
    {
      imageExtent = imageExtent_;
      return *this;
    }

    operator VkBufferImageCopy const&() const
    {
      return *reinterpret_cast<const VkBufferImageCopy*>( this );
    }

    operator VkBufferImageCopy &()
    {
      return *reinterpret_cast<VkBufferImageCopy*>( this );
    }

    bool operator==( BufferImageCopy const& rhs ) const
    {
      return ( bufferOffset == rhs.bufferOffset )
          && ( bufferRowLength == rhs.bufferRowLength )
          && ( bufferImageHeight == rhs.bufferImageHeight )
          && ( imageSubresource == rhs.imageSubresource )
          && ( imageOffset == rhs.imageOffset )
          && ( imageExtent == rhs.imageExtent );
    }

    bool operator!=( BufferImageCopy const& rhs ) const
    {
      return !operator==( rhs );
    }

    DeviceSize bufferOffset;
    uint32_t bufferRowLength;
    uint32_t bufferImageHeight;
    ImageSubresourceLayers imageSubresource;
    Offset3D imageOffset;
    Extent3D imageExtent;
  };
  static_assert( sizeof( BufferImageCopy ) == sizeof( VkBufferImageCopy ), "struct and wrapper have different size!" );

  struct BufferMemoryBarrier
  {
    BufferMemoryBarrier( AccessFlags srcAccessMask_ = AccessFlags(),
                         AccessFlags dstAccessMask_ = AccessFlags(),
                         uint32_t srcQueueFamilyIndex_ = 0,
                         uint32_t dstQueueFamilyIndex_ = 0,
                         Buffer buffer_ = Buffer(),
                         DeviceSize offset_ = 0,
                         DeviceSize size_ = 0 )
      : srcAccessMask( srcAccessMask_ )
      , dstAccessMask( dstAccessMask_ )
      , srcQueueFamilyIndex( srcQueueFamilyIndex_ )
      , dstQueueFamilyIndex( dstQueueFamilyIndex_ )
      , buffer( buffer_ )
      , offset( offset_ )
      , size( size_ )
    {}

    BufferMemoryBarrier( VkBufferMemoryBarrier const & rhs )
    {
      memcpy( this, &rhs, sizeof( BufferMemoryBarrier ) );
    }

    BufferMemoryBarrier& operator=( VkBufferMemoryBarrier const & rhs )
    {
      memcpy( this, &rhs, sizeof( BufferMemoryBarrier ) );
      return *this;
    }

    BufferMemoryBarrier & setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    BufferMemoryBarrier & setSrcAccessMask( AccessFlags srcAccessMask_ )
    {
      srcAccessMask = srcAccessMask_;
      return *this;
    }

    BufferMemoryBarrier & setDstAccessMask( AccessFlags dstAccessMask_ )
    {
      dstAccessMask = dstAccessMask_;
      return *this;
    }

    BufferMemoryBarrier & setSrcQueueFamilyIndex( uint32_t srcQueueFamilyIndex_ )
    {
      srcQueueFamilyIndex = srcQueueFamilyIndex_;
      return *this;
    }

    BufferMemoryBarrier & setDstQueueFamilyIndex( uint32_t dstQueueFamilyIndex_ )
    {
      dstQueueFamilyIndex = dstQueueFamilyIndex_;
      return *this;
    }

    BufferMemoryBarrier & setBuffer( Buffer buffer_ )
    {
      buffer = buffer_;
      return *this;
    }

    BufferMemoryBarrier & setOffset( DeviceSize offset_ )
    {
      offset = offset_;
      return *this;
    }

    BufferMemoryBarrier & setSize( DeviceSize size_ )
    {
      size = size_;
      return *this;
    }

    operator VkBufferMemoryBarrier const&() const
    {
      return *reinterpret_cast<const VkBufferMemoryBarrier*>( this );
    }

    operator VkBufferMemoryBarrier &()
    {
      return *reinterpret_cast<VkBufferMemoryBarrier*>( this );
    }

    bool operator==( BufferMemoryBarrier const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( srcAccessMask == rhs.srcAccessMask )
          && ( dstAccessMask == rhs.dstAccessMask )
          && ( srcQueueFamilyIndex == rhs.srcQueueFamilyIndex )
          && ( dstQueueFamilyIndex == rhs.dstQueueFamilyIndex )
          && ( buffer == rhs.buffer )
          && ( offset == rhs.offset )
          && ( size == rhs.size );
    }

    bool operator!=( BufferMemoryBarrier const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eBufferMemoryBarrier;

  public:
    const void* pNext = nullptr;
    AccessFlags srcAccessMask;
    AccessFlags dstAccessMask;
    uint32_t srcQueueFamilyIndex;
    uint32_t dstQueueFamilyIndex;
    Buffer buffer;
    DeviceSize offset;
    DeviceSize size;
  };
  static_assert( sizeof( BufferMemoryBarrier ) == sizeof( VkBufferMemoryBarrier ), "struct and wrapper have different size!" );

  struct BufferMemoryRequirementsInfo2
  {
    BufferMemoryRequirementsInfo2( Buffer buffer_ = Buffer() )
      : buffer( buffer_ )
    {}

    BufferMemoryRequirementsInfo2( VkBufferMemoryRequirementsInfo2 const & rhs )
    {
      memcpy( this, &rhs, sizeof( BufferMemoryRequirementsInfo2 ) );
    }

    BufferMemoryRequirementsInfo2& operator=( VkBufferMemoryRequirementsInfo2 const & rhs )
    {
      memcpy( this, &rhs, sizeof( BufferMemoryRequirementsInfo2 ) );
      return *this;
    }

    BufferMemoryRequirementsInfo2 & setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    BufferMemoryRequirementsInfo2 & setBuffer( Buffer buffer_ )
    {
      buffer = buffer_;
      return *this;
    }

    operator VkBufferMemoryRequirementsInfo2 const&() const
    {
      return *reinterpret_cast<const VkBufferMemoryRequirementsInfo2*>( this );
    }

    operator VkBufferMemoryRequirementsInfo2 &()
    {
      return *reinterpret_cast<VkBufferMemoryRequirementsInfo2*>( this );
    }

    bool operator==( BufferMemoryRequirementsInfo2 const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( buffer == rhs.buffer );
    }

    bool operator!=( BufferMemoryRequirementsInfo2 const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eBufferMemoryRequirementsInfo2;

  public:
    const void* pNext = nullptr;
    Buffer buffer;
  };
  static_assert( sizeof( BufferMemoryRequirementsInfo2 ) == sizeof( VkBufferMemoryRequirementsInfo2 ), "struct and wrapper have different size!" );

  struct BufferViewCreateInfo
  {
    BufferViewCreateInfo( BufferViewCreateFlags flags_ = BufferViewCreateFlags(),
                          Buffer buffer_ = Buffer(),
                          Format format_ = Format::eUndefined,
                          DeviceSize offset_ = 0,
                          DeviceSize range_ = 0 )
      : flags( flags_ )
      , buffer( buffer_ )
      , format( format_ )
      , offset( offset_ )
      , range( range_ )
    {}

    BufferViewCreateInfo( VkBufferViewCreateInfo const & rhs )
    {
      memcpy( this, &rhs, sizeof( BufferViewCreateInfo ) );
    }

    BufferViewCreateInfo& operator=( VkBufferViewCreateInfo const & rhs )
    {
      memcpy( this, &rhs, sizeof( BufferViewCreateInfo ) );
      return *this;
    }

    BufferViewCreateInfo & setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    BufferViewCreateInfo & setFlags( BufferViewCreateFlags flags_ )
    {
      flags = flags_;
      return *this;
    }

    BufferViewCreateInfo & setBuffer( Buffer buffer_ )
    {
      buffer = buffer_;
      return *this;
    }

    BufferViewCreateInfo & setFormat( Format format_ )
    {
      format = format_;
      return *this;
    }

    BufferViewCreateInfo & setOffset( DeviceSize offset_ )
    {
      offset = offset_;
      return *this;
    }

    BufferViewCreateInfo & setRange( DeviceSize range_ )
    {
      range = range_;
      return *this;
    }

    operator VkBufferViewCreateInfo const&() const
    {
      return *reinterpret_cast<const VkBufferViewCreateInfo*>( this );
    }

    operator VkBufferViewCreateInfo &()
    {
      return *reinterpret_cast<VkBufferViewCreateInfo*>( this );
    }

    bool operator==( BufferViewCreateInfo const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( flags == rhs.flags )
          && ( buffer == rhs.buffer )
          && ( format == rhs.format )
          && ( offset == rhs.offset )
          && ( range == rhs.range );
    }

    bool operator!=( BufferViewCreateInfo const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eBufferViewCreateInfo;

  public:
    const void* pNext = nullptr;
    BufferViewCreateFlags flags;
    Buffer buffer;
    Format format;
    DeviceSize offset;
    DeviceSize range;
  };
  static_assert( sizeof( BufferViewCreateInfo ) == sizeof( VkBufferViewCreateInfo ), "struct and wrapper have different size!" );

  struct CalibratedTimestampInfoEXT
  {
    CalibratedTimestampInfoEXT( TimeDomainEXT timeDomain_ = TimeDomainEXT::eDevice )
      : timeDomain( timeDomain_ )
    {}

    CalibratedTimestampInfoEXT( VkCalibratedTimestampInfoEXT const & rhs )
    {
      memcpy( this, &rhs, sizeof( CalibratedTimestampInfoEXT ) );
    }

    CalibratedTimestampInfoEXT& operator=( VkCalibratedTimestampInfoEXT const & rhs )
    {
      memcpy( this, &rhs, sizeof( CalibratedTimestampInfoEXT ) );
      return *this;
    }

    CalibratedTimestampInfoEXT & setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    CalibratedTimestampInfoEXT & setTimeDomain( TimeDomainEXT timeDomain_ )
    {
      timeDomain = timeDomain_;
      return *this;
    }

    operator VkCalibratedTimestampInfoEXT const&() const
    {
      return *reinterpret_cast<const VkCalibratedTimestampInfoEXT*>( this );
    }

    operator VkCalibratedTimestampInfoEXT &()
    {
      return *reinterpret_cast<VkCalibratedTimestampInfoEXT*>( this );
    }

    bool operator==( CalibratedTimestampInfoEXT const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( timeDomain == rhs.timeDomain );
    }

    bool operator!=( CalibratedTimestampInfoEXT const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eCalibratedTimestampInfoEXT;

  public:
    const void* pNext = nullptr;
    TimeDomainEXT timeDomain;
  };
  static_assert( sizeof( CalibratedTimestampInfoEXT ) == sizeof( VkCalibratedTimestampInfoEXT ), "struct and wrapper have different size!" );

  struct CheckpointDataNV
  {
    operator VkCheckpointDataNV const&() const
    {
      return *reinterpret_cast<const VkCheckpointDataNV*>( this );
    }

    operator VkCheckpointDataNV &()
    {
      return *reinterpret_cast<VkCheckpointDataNV*>( this );
    }

    bool operator==( CheckpointDataNV const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( stage == rhs.stage )
          && ( pCheckpointMarker == rhs.pCheckpointMarker );
    }

    bool operator!=( CheckpointDataNV const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eCheckpointDataNV;

  public:
    void* pNext = nullptr;
    PipelineStageFlagBits stage;
    void* pCheckpointMarker;
  };
  static_assert( sizeof( CheckpointDataNV ) == sizeof( VkCheckpointDataNV ), "struct and wrapper have different size!" );

  union ClearColorValue
  {
    ClearColorValue( const std::array<float,4>& float32_ = { { 0 } } )
    {
      memcpy( float32, float32_.data(), 4 * sizeof( float ) );
    }

    ClearColorValue( const std::array<int32_t,4>& int32_ )
    {
      memcpy( int32, int32_.data(), 4 * sizeof( int32_t ) );
    }

    ClearColorValue( const std::array<uint32_t,4>& uint32_ )
    {
      memcpy( uint32, uint32_.data(), 4 * sizeof( uint32_t ) );
    }


    ClearColorValue & setFloat32( std::array<float,4> float32_ )
    {
      memcpy( float32, float32_.data(), 4 * sizeof( float ) );
      return *this;
    }

    ClearColorValue & setInt32( std::array<int32_t,4> int32_ )
    {
      memcpy( int32, int32_.data(), 4 * sizeof( int32_t ) );
      return *this;
    }

    ClearColorValue & setUint32( std::array<uint32_t,4> uint32_ )
    {
      memcpy( uint32, uint32_.data(), 4 * sizeof( uint32_t ) );
      return *this;
    }
    operator VkClearColorValue const&() const
    {
      return *reinterpret_cast<const VkClearColorValue*>(this);
    }

    operator VkClearColorValue &()
    {
      return *reinterpret_cast<VkClearColorValue*>(this);
    }

    float float32[4];
    int32_t int32[4];
    uint32_t uint32[4];
  };

  struct ClearDepthStencilValue
  {
    ClearDepthStencilValue( float depth_ = 0,
                            uint32_t stencil_ = 0 )
      : depth( depth_ )
      , stencil( stencil_ )
    {}

    ClearDepthStencilValue( VkClearDepthStencilValue const & rhs )
    {
      memcpy( this, &rhs, sizeof( ClearDepthStencilValue ) );
    }

    ClearDepthStencilValue& operator=( VkClearDepthStencilValue const & rhs )
    {
      memcpy( this, &rhs, sizeof( ClearDepthStencilValue ) );
      return *this;
    }

    ClearDepthStencilValue & setDepth( float depth_ )
    {
      depth = depth_;
      return *this;
    }

    ClearDepthStencilValue & setStencil( uint32_t stencil_ )
    {
      stencil = stencil_;
      return *this;
    }

    operator VkClearDepthStencilValue const&() const
    {
      return *reinterpret_cast<const VkClearDepthStencilValue*>( this );
    }

    operator VkClearDepthStencilValue &()
    {
      return *reinterpret_cast<VkClearDepthStencilValue*>( this );
    }

    bool operator==( ClearDepthStencilValue const& rhs ) const
    {
      return ( depth == rhs.depth )
          && ( stencil == rhs.stencil );
    }

    bool operator!=( ClearDepthStencilValue const& rhs ) const
    {
      return !operator==( rhs );
    }

    float depth;
    uint32_t stencil;
  };
  static_assert( sizeof( ClearDepthStencilValue ) == sizeof( VkClearDepthStencilValue ), "struct and wrapper have different size!" );

  union ClearValue
  {
    ClearValue( ClearColorValue color_ = ClearColorValue() )
    {
      color = color_;
    }

    ClearValue( ClearDepthStencilValue depthStencil_ )
    {
      depthStencil = depthStencil_;
    }


    ClearValue & setColor( ClearColorValue color_ )
    {
      color = color_;
      return *this;
    }

    ClearValue & setDepthStencil( ClearDepthStencilValue depthStencil_ )
    {
      depthStencil = depthStencil_;
      return *this;
    }
    operator VkClearValue const&() const
    {
      return *reinterpret_cast<const VkClearValue*>(this);
    }

    operator VkClearValue &()
    {
      return *reinterpret_cast<VkClearValue*>(this);
    }

#ifdef VULKAN_HPP_HAS_UNRESTRICTED_UNIONS
    ClearColorValue color;
    ClearDepthStencilValue depthStencil;
#else
    VkClearColorValue color;
    VkClearDepthStencilValue depthStencil;
#endif  /*VULKAN_HPP_HAS_UNRESTRICTED_UNIONS*/
  };

  struct ClearAttachment
  {
    ClearAttachment( ImageAspectFlags aspectMask_ = ImageAspectFlags(),
                     uint32_t colorAttachment_ = 0,
                     ClearValue clearValue_ = ClearValue() )
      : aspectMask( aspectMask_ )
      , colorAttachment( colorAttachment_ )
      , clearValue( clearValue_ )
    {}

    ClearAttachment( VkClearAttachment const & rhs )
    {
      memcpy( this, &rhs, sizeof( ClearAttachment ) );
    }

    ClearAttachment& operator=( VkClearAttachment const & rhs )
    {
      memcpy( this, &rhs, sizeof( ClearAttachment ) );
      return *this;
    }

    ClearAttachment & setAspectMask( ImageAspectFlags aspectMask_ )
    {
      aspectMask = aspectMask_;
      return *this;
    }

    ClearAttachment & setColorAttachment( uint32_t colorAttachment_ )
    {
      colorAttachment = colorAttachment_;
      return *this;
    }

    ClearAttachment & setClearValue( ClearValue clearValue_ )
    {
      clearValue = clearValue_;
      return *this;
    }

    operator VkClearAttachment const&() const
    {
      return *reinterpret_cast<const VkClearAttachment*>( this );
    }

    operator VkClearAttachment &()
    {
      return *reinterpret_cast<VkClearAttachment*>( this );
    }

    ImageAspectFlags aspectMask;
    uint32_t colorAttachment;
    ClearValue clearValue;
  };
  static_assert( sizeof( ClearAttachment ) == sizeof( VkClearAttachment ), "struct and wrapper have different size!" );

  struct ClearRect
  {
    ClearRect( Rect2D rect_ = Rect2D(),
               uint32_t baseArrayLayer_ = 0,
               uint32_t layerCount_ = 0 )
      : rect( rect_ )
      , baseArrayLayer( baseArrayLayer_ )
      , layerCount( layerCount_ )
    {}

    ClearRect( VkClearRect const & rhs )
    {
      memcpy( this, &rhs, sizeof( ClearRect ) );
    }

    ClearRect& operator=( VkClearRect const & rhs )
    {
      memcpy( this, &rhs, sizeof( ClearRect ) );
      return *this;
    }

    ClearRect & setRect( Rect2D rect_ )
    {
      rect = rect_;
      return *this;
    }

    ClearRect & setBaseArrayLayer( uint32_t baseArrayLayer_ )
    {
      baseArrayLayer = baseArrayLayer_;
      return *this;
    }

    ClearRect & setLayerCount( uint32_t layerCount_ )
    {
      layerCount = layerCount_;
      return *this;
    }

    operator VkClearRect const&() const
    {
      return *reinterpret_cast<const VkClearRect*>( this );
    }

    operator VkClearRect &()
    {
      return *reinterpret_cast<VkClearRect*>( this );
    }

    bool operator==( ClearRect const& rhs ) const
    {
      return ( rect == rhs.rect )
          && ( baseArrayLayer == rhs.baseArrayLayer )
          && ( layerCount == rhs.layerCount );
    }

    bool operator!=( ClearRect const& rhs ) const
    {
      return !operator==( rhs );
    }

    Rect2D rect;
    uint32_t baseArrayLayer;
    uint32_t layerCount;
  };
  static_assert( sizeof( ClearRect ) == sizeof( VkClearRect ), "struct and wrapper have different size!" );

  struct IndirectCommandsTokenNVX
  {
    IndirectCommandsTokenNVX( IndirectCommandsTokenTypeNVX tokenType_ = IndirectCommandsTokenTypeNVX::ePipeline,
                              Buffer buffer_ = Buffer(),
                              DeviceSize offset_ = 0 )
      : tokenType( tokenType_ )
      , buffer( buffer_ )
      , offset( offset_ )
    {}

    IndirectCommandsTokenNVX( VkIndirectCommandsTokenNVX const & rhs )
    {
      memcpy( this, &rhs, sizeof( IndirectCommandsTokenNVX ) );
    }

    IndirectCommandsTokenNVX& operator=( VkIndirectCommandsTokenNVX const & rhs )
    {
      memcpy( this, &rhs, sizeof( IndirectCommandsTokenNVX ) );
      return *this;
    }

    IndirectCommandsTokenNVX & setTokenType( IndirectCommandsTokenTypeNVX tokenType_ )
    {
      tokenType = tokenType_;
      return *this;
    }

    IndirectCommandsTokenNVX & setBuffer( Buffer buffer_ )
    {
      buffer = buffer_;
      return *this;
    }

    IndirectCommandsTokenNVX & setOffset( DeviceSize offset_ )
    {
      offset = offset_;
      return *this;
    }

    operator VkIndirectCommandsTokenNVX const&() const
    {
      return *reinterpret_cast<const VkIndirectCommandsTokenNVX*>( this );
    }

    operator VkIndirectCommandsTokenNVX &()
    {
      return *reinterpret_cast<VkIndirectCommandsTokenNVX*>( this );
    }

    bool operator==( IndirectCommandsTokenNVX const& rhs ) const
    {
      return ( tokenType == rhs.tokenType )
          && ( buffer == rhs.buffer )
          && ( offset == rhs.offset );
    }

    bool operator!=( IndirectCommandsTokenNVX const& rhs ) const
    {
      return !operator==( rhs );
    }

    IndirectCommandsTokenTypeNVX tokenType;
    Buffer buffer;
    DeviceSize offset;
  };
  static_assert( sizeof( IndirectCommandsTokenNVX ) == sizeof( VkIndirectCommandsTokenNVX ), "struct and wrapper have different size!" );

  struct CmdProcessCommandsInfoNVX
  {
    CmdProcessCommandsInfoNVX( ObjectTableNVX objectTable_ = ObjectTableNVX(),
                               IndirectCommandsLayoutNVX indirectCommandsLayout_ = IndirectCommandsLayoutNVX(),
                               uint32_t indirectCommandsTokenCount_ = 0,
                               const IndirectCommandsTokenNVX* pIndirectCommandsTokens_ = nullptr,
                               uint32_t maxSequencesCount_ = 0,
                               CommandBuffer targetCommandBuffer_ = CommandBuffer(),
                               Buffer sequencesCountBuffer_ = Buffer(),
                               DeviceSize sequencesCountOffset_ = 0,
                               Buffer sequencesIndexBuffer_ = Buffer(),
                               DeviceSize sequencesIndexOffset_ = 0 )
      : objectTable( objectTable_ )
      , indirectCommandsLayout( indirectCommandsLayout_ )
      , indirectCommandsTokenCount( indirectCommandsTokenCount_ )
      , pIndirectCommandsTokens( pIndirectCommandsTokens_ )
      , maxSequencesCount( maxSequencesCount_ )
      , targetCommandBuffer( targetCommandBuffer_ )
      , sequencesCountBuffer( sequencesCountBuffer_ )
      , sequencesCountOffset( sequencesCountOffset_ )
      , sequencesIndexBuffer( sequencesIndexBuffer_ )
      , sequencesIndexOffset( sequencesIndexOffset_ )
    {}

    CmdProcessCommandsInfoNVX( VkCmdProcessCommandsInfoNVX const & rhs )
    {
      memcpy( this, &rhs, sizeof( CmdProcessCommandsInfoNVX ) );
    }

    CmdProcessCommandsInfoNVX& operator=( VkCmdProcessCommandsInfoNVX const & rhs )
    {
      memcpy( this, &rhs, sizeof( CmdProcessCommandsInfoNVX ) );
      return *this;
    }

    CmdProcessCommandsInfoNVX & setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    CmdProcessCommandsInfoNVX & setObjectTable( ObjectTableNVX objectTable_ )
    {
      objectTable = objectTable_;
      return *this;
    }

    CmdProcessCommandsInfoNVX & setIndirectCommandsLayout( IndirectCommandsLayoutNVX indirectCommandsLayout_ )
    {
      indirectCommandsLayout = indirectCommandsLayout_;
      return *this;
    }

    CmdProcessCommandsInfoNVX & setIndirectCommandsTokenCount( uint32_t indirectCommandsTokenCount_ )
    {
      indirectCommandsTokenCount = indirectCommandsTokenCount_;
      return *this;
    }

    CmdProcessCommandsInfoNVX & setPIndirectCommandsTokens( const IndirectCommandsTokenNVX* pIndirectCommandsTokens_ )
    {
      pIndirectCommandsTokens = pIndirectCommandsTokens_;
      return *this;
    }

    CmdProcessCommandsInfoNVX & setMaxSequencesCount( uint32_t maxSequencesCount_ )
    {
      maxSequencesCount = maxSequencesCount_;
      return *this;
    }

    CmdProcessCommandsInfoNVX & setTargetCommandBuffer( CommandBuffer targetCommandBuffer_ )
    {
      targetCommandBuffer = targetCommandBuffer_;
      return *this;
    }

    CmdProcessCommandsInfoNVX & setSequencesCountBuffer( Buffer sequencesCountBuffer_ )
    {
      sequencesCountBuffer = sequencesCountBuffer_;
      return *this;
    }

    CmdProcessCommandsInfoNVX & setSequencesCountOffset( DeviceSize sequencesCountOffset_ )
    {
      sequencesCountOffset = sequencesCountOffset_;
      return *this;
    }

    CmdProcessCommandsInfoNVX & setSequencesIndexBuffer( Buffer sequencesIndexBuffer_ )
    {
      sequencesIndexBuffer = sequencesIndexBuffer_;
      return *this;
    }

    CmdProcessCommandsInfoNVX & setSequencesIndexOffset( DeviceSize sequencesIndexOffset_ )
    {
      sequencesIndexOffset = sequencesIndexOffset_;
      return *this;
    }

    operator VkCmdProcessCommandsInfoNVX const&() const
    {
      return *reinterpret_cast<const VkCmdProcessCommandsInfoNVX*>( this );
    }

    operator VkCmdProcessCommandsInfoNVX &()
    {
      return *reinterpret_cast<VkCmdProcessCommandsInfoNVX*>( this );
    }

    bool operator==( CmdProcessCommandsInfoNVX const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( objectTable == rhs.objectTable )
          && ( indirectCommandsLayout == rhs.indirectCommandsLayout )
          && ( indirectCommandsTokenCount == rhs.indirectCommandsTokenCount )
          && ( pIndirectCommandsTokens == rhs.pIndirectCommandsTokens )
          && ( maxSequencesCount == rhs.maxSequencesCount )
          && ( targetCommandBuffer == rhs.targetCommandBuffer )
          && ( sequencesCountBuffer == rhs.sequencesCountBuffer )
          && ( sequencesCountOffset == rhs.sequencesCountOffset )
          && ( sequencesIndexBuffer == rhs.sequencesIndexBuffer )
          && ( sequencesIndexOffset == rhs.sequencesIndexOffset );
    }

    bool operator!=( CmdProcessCommandsInfoNVX const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eCmdProcessCommandsInfoNVX;

  public:
    const void* pNext = nullptr;
    ObjectTableNVX objectTable;
    IndirectCommandsLayoutNVX indirectCommandsLayout;
    uint32_t indirectCommandsTokenCount;
    const IndirectCommandsTokenNVX* pIndirectCommandsTokens;
    uint32_t maxSequencesCount;
    CommandBuffer targetCommandBuffer;
    Buffer sequencesCountBuffer;
    DeviceSize sequencesCountOffset;
    Buffer sequencesIndexBuffer;
    DeviceSize sequencesIndexOffset;
  };
  static_assert( sizeof( CmdProcessCommandsInfoNVX ) == sizeof( VkCmdProcessCommandsInfoNVX ), "struct and wrapper have different size!" );

  struct CmdReserveSpaceForCommandsInfoNVX
  {
    CmdReserveSpaceForCommandsInfoNVX( ObjectTableNVX objectTable_ = ObjectTableNVX(),
                                       IndirectCommandsLayoutNVX indirectCommandsLayout_ = IndirectCommandsLayoutNVX(),
                                       uint32_t maxSequencesCount_ = 0 )
      : objectTable( objectTable_ )
      , indirectCommandsLayout( indirectCommandsLayout_ )
      , maxSequencesCount( maxSequencesCount_ )
    {}

    CmdReserveSpaceForCommandsInfoNVX( VkCmdReserveSpaceForCommandsInfoNVX const & rhs )
    {
      memcpy( this, &rhs, sizeof( CmdReserveSpaceForCommandsInfoNVX ) );
    }

    CmdReserveSpaceForCommandsInfoNVX& operator=( VkCmdReserveSpaceForCommandsInfoNVX const & rhs )
    {
      memcpy( this, &rhs, sizeof( CmdReserveSpaceForCommandsInfoNVX ) );
      return *this;
    }

    CmdReserveSpaceForCommandsInfoNVX & setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    CmdReserveSpaceForCommandsInfoNVX & setObjectTable( ObjectTableNVX objectTable_ )
    {
      objectTable = objectTable_;
      return *this;
    }

    CmdReserveSpaceForCommandsInfoNVX & setIndirectCommandsLayout( IndirectCommandsLayoutNVX indirectCommandsLayout_ )
    {
      indirectCommandsLayout = indirectCommandsLayout_;
      return *this;
    }

    CmdReserveSpaceForCommandsInfoNVX & setMaxSequencesCount( uint32_t maxSequencesCount_ )
    {
      maxSequencesCount = maxSequencesCount_;
      return *this;
    }

    operator VkCmdReserveSpaceForCommandsInfoNVX const&() const
    {
      return *reinterpret_cast<const VkCmdReserveSpaceForCommandsInfoNVX*>( this );
    }

    operator VkCmdReserveSpaceForCommandsInfoNVX &()
    {
      return *reinterpret_cast<VkCmdReserveSpaceForCommandsInfoNVX*>( this );
    }

    bool operator==( CmdReserveSpaceForCommandsInfoNVX const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( objectTable == rhs.objectTable )
          && ( indirectCommandsLayout == rhs.indirectCommandsLayout )
          && ( maxSequencesCount == rhs.maxSequencesCount );
    }

    bool operator!=( CmdReserveSpaceForCommandsInfoNVX const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eCmdReserveSpaceForCommandsInfoNVX;

  public:
    const void* pNext = nullptr;
    ObjectTableNVX objectTable;
    IndirectCommandsLayoutNVX indirectCommandsLayout;
    uint32_t maxSequencesCount;
  };
  static_assert( sizeof( CmdReserveSpaceForCommandsInfoNVX ) == sizeof( VkCmdReserveSpaceForCommandsInfoNVX ), "struct and wrapper have different size!" );

  struct CoarseSampleLocationNV
  {
    CoarseSampleLocationNV( uint32_t pixelX_ = 0,
                            uint32_t pixelY_ = 0,
                            uint32_t sample_ = 0 )
      : pixelX( pixelX_ )
      , pixelY( pixelY_ )
      , sample( sample_ )
    {}

    CoarseSampleLocationNV( VkCoarseSampleLocationNV const & rhs )
    {
      memcpy( this, &rhs, sizeof( CoarseSampleLocationNV ) );
    }

    CoarseSampleLocationNV& operator=( VkCoarseSampleLocationNV const & rhs )
    {
      memcpy( this, &rhs, sizeof( CoarseSampleLocationNV ) );
      return *this;
    }

    CoarseSampleLocationNV & setPixelX( uint32_t pixelX_ )
    {
      pixelX = pixelX_;
      return *this;
    }

    CoarseSampleLocationNV & setPixelY( uint32_t pixelY_ )
    {
      pixelY = pixelY_;
      return *this;
    }

    CoarseSampleLocationNV & setSample( uint32_t sample_ )
    {
      sample = sample_;
      return *this;
    }

    operator VkCoarseSampleLocationNV const&() const
    {
      return *reinterpret_cast<const VkCoarseSampleLocationNV*>( this );
    }

    operator VkCoarseSampleLocationNV &()
    {
      return *reinterpret_cast<VkCoarseSampleLocationNV*>( this );
    }

    bool operator==( CoarseSampleLocationNV const& rhs ) const
    {
      return ( pixelX == rhs.pixelX )
          && ( pixelY == rhs.pixelY )
          && ( sample == rhs.sample );
    }

    bool operator!=( CoarseSampleLocationNV const& rhs ) const
    {
      return !operator==( rhs );
    }

    uint32_t pixelX;
    uint32_t pixelY;
    uint32_t sample;
  };
  static_assert( sizeof( CoarseSampleLocationNV ) == sizeof( VkCoarseSampleLocationNV ), "struct and wrapper have different size!" );

  struct CoarseSampleOrderCustomNV
  {
    CoarseSampleOrderCustomNV( ShadingRatePaletteEntryNV shadingRate_ = ShadingRatePaletteEntryNV::eNoInvocations,
                               uint32_t sampleCount_ = 0,
                               uint32_t sampleLocationCount_ = 0,
                               const CoarseSampleLocationNV* pSampleLocations_ = nullptr )
      : shadingRate( shadingRate_ )
      , sampleCount( sampleCount_ )
      , sampleLocationCount( sampleLocationCount_ )
      , pSampleLocations( pSampleLocations_ )
    {}

    CoarseSampleOrderCustomNV( VkCoarseSampleOrderCustomNV const & rhs )
    {
      memcpy( this, &rhs, sizeof( CoarseSampleOrderCustomNV ) );
    }

    CoarseSampleOrderCustomNV& operator=( VkCoarseSampleOrderCustomNV const & rhs )
    {
      memcpy( this, &rhs, sizeof( CoarseSampleOrderCustomNV ) );
      return *this;
    }

    CoarseSampleOrderCustomNV & setShadingRate( ShadingRatePaletteEntryNV shadingRate_ )
    {
      shadingRate = shadingRate_;
      return *this;
    }

    CoarseSampleOrderCustomNV & setSampleCount( uint32_t sampleCount_ )
    {
      sampleCount = sampleCount_;
      return *this;
    }

    CoarseSampleOrderCustomNV & setSampleLocationCount( uint32_t sampleLocationCount_ )
    {
      sampleLocationCount = sampleLocationCount_;
      return *this;
    }

    CoarseSampleOrderCustomNV & setPSampleLocations( const CoarseSampleLocationNV* pSampleLocations_ )
    {
      pSampleLocations = pSampleLocations_;
      return *this;
    }

    operator VkCoarseSampleOrderCustomNV const&() const
    {
      return *reinterpret_cast<const VkCoarseSampleOrderCustomNV*>( this );
    }

    operator VkCoarseSampleOrderCustomNV &()
    {
      return *reinterpret_cast<VkCoarseSampleOrderCustomNV*>( this );
    }

    bool operator==( CoarseSampleOrderCustomNV const& rhs ) const
    {
      return ( shadingRate == rhs.shadingRate )
          && ( sampleCount == rhs.sampleCount )
          && ( sampleLocationCount == rhs.sampleLocationCount )
          && ( pSampleLocations == rhs.pSampleLocations );
    }

    bool operator!=( CoarseSampleOrderCustomNV const& rhs ) const
    {
      return !operator==( rhs );
    }

    ShadingRatePaletteEntryNV shadingRate;
    uint32_t sampleCount;
    uint32_t sampleLocationCount;
    const CoarseSampleLocationNV* pSampleLocations;
  };
  static_assert( sizeof( CoarseSampleOrderCustomNV ) == sizeof( VkCoarseSampleOrderCustomNV ), "struct and wrapper have different size!" );

  struct CommandBufferAllocateInfo
  {
    CommandBufferAllocateInfo( CommandPool commandPool_ = CommandPool(),
                               CommandBufferLevel level_ = CommandBufferLevel::ePrimary,
                               uint32_t commandBufferCount_ = 0 )
      : commandPool( commandPool_ )
      , level( level_ )
      , commandBufferCount( commandBufferCount_ )
    {}

    CommandBufferAllocateInfo( VkCommandBufferAllocateInfo const & rhs )
    {
      memcpy( this, &rhs, sizeof( CommandBufferAllocateInfo ) );
    }

    CommandBufferAllocateInfo& operator=( VkCommandBufferAllocateInfo const & rhs )
    {
      memcpy( this, &rhs, sizeof( CommandBufferAllocateInfo ) );
      return *this;
    }

    CommandBufferAllocateInfo & setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    CommandBufferAllocateInfo & setCommandPool( CommandPool commandPool_ )
    {
      commandPool = commandPool_;
      return *this;
    }

    CommandBufferAllocateInfo & setLevel( CommandBufferLevel level_ )
    {
      level = level_;
      return *this;
    }

    CommandBufferAllocateInfo & setCommandBufferCount( uint32_t commandBufferCount_ )
    {
      commandBufferCount = commandBufferCount_;
      return *this;
    }

    operator VkCommandBufferAllocateInfo const&() const
    {
      return *reinterpret_cast<const VkCommandBufferAllocateInfo*>( this );
    }

    operator VkCommandBufferAllocateInfo &()
    {
      return *reinterpret_cast<VkCommandBufferAllocateInfo*>( this );
    }

    bool operator==( CommandBufferAllocateInfo const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( commandPool == rhs.commandPool )
          && ( level == rhs.level )
          && ( commandBufferCount == rhs.commandBufferCount );
    }

    bool operator!=( CommandBufferAllocateInfo const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eCommandBufferAllocateInfo;

  public:
    const void* pNext = nullptr;
    CommandPool commandPool;
    CommandBufferLevel level;
    uint32_t commandBufferCount;
  };
  static_assert( sizeof( CommandBufferAllocateInfo ) == sizeof( VkCommandBufferAllocateInfo ), "struct and wrapper have different size!" );

  struct CommandBufferInheritanceInfo
  {
    CommandBufferInheritanceInfo( RenderPass renderPass_ = RenderPass(),
                                  uint32_t subpass_ = 0,
                                  Framebuffer framebuffer_ = Framebuffer(),
                                  Bool32 occlusionQueryEnable_ = 0,
                                  QueryControlFlags queryFlags_ = QueryControlFlags(),
                                  QueryPipelineStatisticFlags pipelineStatistics_ = QueryPipelineStatisticFlags() )
      : renderPass( renderPass_ )
      , subpass( subpass_ )
      , framebuffer( framebuffer_ )
      , occlusionQueryEnable( occlusionQueryEnable_ )
      , queryFlags( queryFlags_ )
      , pipelineStatistics( pipelineStatistics_ )
    {}

    CommandBufferInheritanceInfo( VkCommandBufferInheritanceInfo const & rhs )
    {
      memcpy( this, &rhs, sizeof( CommandBufferInheritanceInfo ) );
    }

    CommandBufferInheritanceInfo& operator=( VkCommandBufferInheritanceInfo const & rhs )
    {
      memcpy( this, &rhs, sizeof( CommandBufferInheritanceInfo ) );
      return *this;
    }

    CommandBufferInheritanceInfo & setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    CommandBufferInheritanceInfo & setRenderPass( RenderPass renderPass_ )
    {
      renderPass = renderPass_;
      return *this;
    }

    CommandBufferInheritanceInfo & setSubpass( uint32_t subpass_ )
    {
      subpass = subpass_;
      return *this;
    }

    CommandBufferInheritanceInfo & setFramebuffer( Framebuffer framebuffer_ )
    {
      framebuffer = framebuffer_;
      return *this;
    }

    CommandBufferInheritanceInfo & setOcclusionQueryEnable( Bool32 occlusionQueryEnable_ )
    {
      occlusionQueryEnable = occlusionQueryEnable_;
      return *this;
    }

    CommandBufferInheritanceInfo & setQueryFlags( QueryControlFlags queryFlags_ )
    {
      queryFlags = queryFlags_;
      return *this;
    }

    CommandBufferInheritanceInfo & setPipelineStatistics( QueryPipelineStatisticFlags pipelineStatistics_ )
    {
      pipelineStatistics = pipelineStatistics_;
      return *this;
    }

    operator VkCommandBufferInheritanceInfo const&() const
    {
      return *reinterpret_cast<const VkCommandBufferInheritanceInfo*>( this );
    }

    operator VkCommandBufferInheritanceInfo &()
    {
      return *reinterpret_cast<VkCommandBufferInheritanceInfo*>( this );
    }

    bool operator==( CommandBufferInheritanceInfo const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( renderPass == rhs.renderPass )
          && ( subpass == rhs.subpass )
          && ( framebuffer == rhs.framebuffer )
          && ( occlusionQueryEnable == rhs.occlusionQueryEnable )
          && ( queryFlags == rhs.queryFlags )
          && ( pipelineStatistics == rhs.pipelineStatistics );
    }

    bool operator!=( CommandBufferInheritanceInfo const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eCommandBufferInheritanceInfo;

  public:
    const void* pNext = nullptr;
    RenderPass renderPass;
    uint32_t subpass;
    Framebuffer framebuffer;
    Bool32 occlusionQueryEnable;
    QueryControlFlags queryFlags;
    QueryPipelineStatisticFlags pipelineStatistics;
  };
  static_assert( sizeof( CommandBufferInheritanceInfo ) == sizeof( VkCommandBufferInheritanceInfo ), "struct and wrapper have different size!" );

  struct CommandBufferBeginInfo
  {
    CommandBufferBeginInfo( CommandBufferUsageFlags flags_ = CommandBufferUsageFlags(),
                            const CommandBufferInheritanceInfo* pInheritanceInfo_ = nullptr )
      : flags( flags_ )
      , pInheritanceInfo( pInheritanceInfo_ )
    {}

    CommandBufferBeginInfo( VkCommandBufferBeginInfo const & rhs )
    {
      memcpy( this, &rhs, sizeof( CommandBufferBeginInfo ) );
    }

    CommandBufferBeginInfo& operator=( VkCommandBufferBeginInfo const & rhs )
    {
      memcpy( this, &rhs, sizeof( CommandBufferBeginInfo ) );
      return *this;
    }

    CommandBufferBeginInfo & setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    CommandBufferBeginInfo & setFlags( CommandBufferUsageFlags flags_ )
    {
      flags = flags_;
      return *this;
    }

    CommandBufferBeginInfo & setPInheritanceInfo( const CommandBufferInheritanceInfo* pInheritanceInfo_ )
    {
      pInheritanceInfo = pInheritanceInfo_;
      return *this;
    }

    operator VkCommandBufferBeginInfo const&() const
    {
      return *reinterpret_cast<const VkCommandBufferBeginInfo*>( this );
    }

    operator VkCommandBufferBeginInfo &()
    {
      return *reinterpret_cast<VkCommandBufferBeginInfo*>( this );
    }

    bool operator==( CommandBufferBeginInfo const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( flags == rhs.flags )
          && ( pInheritanceInfo == rhs.pInheritanceInfo );
    }

    bool operator!=( CommandBufferBeginInfo const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eCommandBufferBeginInfo;

  public:
    const void* pNext = nullptr;
    CommandBufferUsageFlags flags;
    const CommandBufferInheritanceInfo* pInheritanceInfo;
  };
  static_assert( sizeof( CommandBufferBeginInfo ) == sizeof( VkCommandBufferBeginInfo ), "struct and wrapper have different size!" );

  struct CommandBufferInheritanceConditionalRenderingInfoEXT
  {
    CommandBufferInheritanceConditionalRenderingInfoEXT( Bool32 conditionalRenderingEnable_ = 0 )
      : conditionalRenderingEnable( conditionalRenderingEnable_ )
    {}

    CommandBufferInheritanceConditionalRenderingInfoEXT( VkCommandBufferInheritanceConditionalRenderingInfoEXT const & rhs )
    {
      memcpy( this, &rhs, sizeof( CommandBufferInheritanceConditionalRenderingInfoEXT ) );
    }

    CommandBufferInheritanceConditionalRenderingInfoEXT& operator=( VkCommandBufferInheritanceConditionalRenderingInfoEXT const & rhs )
    {
      memcpy( this, &rhs, sizeof( CommandBufferInheritanceConditionalRenderingInfoEXT ) );
      return *this;
    }

    CommandBufferInheritanceConditionalRenderingInfoEXT & setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    CommandBufferInheritanceConditionalRenderingInfoEXT & setConditionalRenderingEnable( Bool32 conditionalRenderingEnable_ )
    {
      conditionalRenderingEnable = conditionalRenderingEnable_;
      return *this;
    }

    operator VkCommandBufferInheritanceConditionalRenderingInfoEXT const&() const
    {
      return *reinterpret_cast<const VkCommandBufferInheritanceConditionalRenderingInfoEXT*>( this );
    }

    operator VkCommandBufferInheritanceConditionalRenderingInfoEXT &()
    {
      return *reinterpret_cast<VkCommandBufferInheritanceConditionalRenderingInfoEXT*>( this );
    }

    bool operator==( CommandBufferInheritanceConditionalRenderingInfoEXT const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( conditionalRenderingEnable == rhs.conditionalRenderingEnable );
    }

    bool operator!=( CommandBufferInheritanceConditionalRenderingInfoEXT const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eCommandBufferInheritanceConditionalRenderingInfoEXT;

  public:
    const void* pNext = nullptr;
    Bool32 conditionalRenderingEnable;
  };
  static_assert( sizeof( CommandBufferInheritanceConditionalRenderingInfoEXT ) == sizeof( VkCommandBufferInheritanceConditionalRenderingInfoEXT ), "struct and wrapper have different size!" );

  struct CommandPoolCreateInfo
  {
    CommandPoolCreateInfo( CommandPoolCreateFlags flags_ = CommandPoolCreateFlags(),
                           uint32_t queueFamilyIndex_ = 0 )
      : flags( flags_ )
      , queueFamilyIndex( queueFamilyIndex_ )
    {}

    CommandPoolCreateInfo( VkCommandPoolCreateInfo const & rhs )
    {
      memcpy( this, &rhs, sizeof( CommandPoolCreateInfo ) );
    }

    CommandPoolCreateInfo& operator=( VkCommandPoolCreateInfo const & rhs )
    {
      memcpy( this, &rhs, sizeof( CommandPoolCreateInfo ) );
      return *this;
    }

    CommandPoolCreateInfo & setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    CommandPoolCreateInfo & setFlags( CommandPoolCreateFlags flags_ )
    {
      flags = flags_;
      return *this;
    }

    CommandPoolCreateInfo & setQueueFamilyIndex( uint32_t queueFamilyIndex_ )
    {
      queueFamilyIndex = queueFamilyIndex_;
      return *this;
    }

    operator VkCommandPoolCreateInfo const&() const
    {
      return *reinterpret_cast<const VkCommandPoolCreateInfo*>( this );
    }

    operator VkCommandPoolCreateInfo &()
    {
      return *reinterpret_cast<VkCommandPoolCreateInfo*>( this );
    }

    bool operator==( CommandPoolCreateInfo const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( flags == rhs.flags )
          && ( queueFamilyIndex == rhs.queueFamilyIndex );
    }

    bool operator!=( CommandPoolCreateInfo const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eCommandPoolCreateInfo;

  public:
    const void* pNext = nullptr;
    CommandPoolCreateFlags flags;
    uint32_t queueFamilyIndex;
  };
  static_assert( sizeof( CommandPoolCreateInfo ) == sizeof( VkCommandPoolCreateInfo ), "struct and wrapper have different size!" );

  struct SpecializationMapEntry
  {
    SpecializationMapEntry( uint32_t constantID_ = 0,
                            uint32_t offset_ = 0,
                            size_t size_ = 0 )
      : constantID( constantID_ )
      , offset( offset_ )
      , size( size_ )
    {}

    SpecializationMapEntry( VkSpecializationMapEntry const & rhs )
    {
      memcpy( this, &rhs, sizeof( SpecializationMapEntry ) );
    }

    SpecializationMapEntry& operator=( VkSpecializationMapEntry const & rhs )
    {
      memcpy( this, &rhs, sizeof( SpecializationMapEntry ) );
      return *this;
    }

    SpecializationMapEntry & setConstantID( uint32_t constantID_ )
    {
      constantID = constantID_;
      return *this;
    }

    SpecializationMapEntry & setOffset( uint32_t offset_ )
    {
      offset = offset_;
      return *this;
    }

    SpecializationMapEntry & setSize( size_t size_ )
    {
      size = size_;
      return *this;
    }

    operator VkSpecializationMapEntry const&() const
    {
      return *reinterpret_cast<const VkSpecializationMapEntry*>( this );
    }

    operator VkSpecializationMapEntry &()
    {
      return *reinterpret_cast<VkSpecializationMapEntry*>( this );
    }

    bool operator==( SpecializationMapEntry const& rhs ) const
    {
      return ( constantID == rhs.constantID )
          && ( offset == rhs.offset )
          && ( size == rhs.size );
    }

    bool operator!=( SpecializationMapEntry const& rhs ) const
    {
      return !operator==( rhs );
    }

    uint32_t constantID;
    uint32_t offset;
    size_t size;
  };
  static_assert( sizeof( SpecializationMapEntry ) == sizeof( VkSpecializationMapEntry ), "struct and wrapper have different size!" );

  struct SpecializationInfo
  {
    SpecializationInfo( uint32_t mapEntryCount_ = 0,
                        const SpecializationMapEntry* pMapEntries_ = nullptr,
                        size_t dataSize_ = 0,
                        const void* pData_ = nullptr )
      : mapEntryCount( mapEntryCount_ )
      , pMapEntries( pMapEntries_ )
      , dataSize( dataSize_ )
      , pData( pData_ )
    {}

    SpecializationInfo( VkSpecializationInfo const & rhs )
    {
      memcpy( this, &rhs, sizeof( SpecializationInfo ) );
    }

    SpecializationInfo& operator=( VkSpecializationInfo const & rhs )
    {
      memcpy( this, &rhs, sizeof( SpecializationInfo ) );
      return *this;
    }

    SpecializationInfo & setMapEntryCount( uint32_t mapEntryCount_ )
    {
      mapEntryCount = mapEntryCount_;
      return *this;
    }

    SpecializationInfo & setPMapEntries( const SpecializationMapEntry* pMapEntries_ )
    {
      pMapEntries = pMapEntries_;
      return *this;
    }

    SpecializationInfo & setDataSize( size_t dataSize_ )
    {
      dataSize = dataSize_;
      return *this;
    }

    SpecializationInfo & setPData( const void* pData_ )
    {
      pData = pData_;
      return *this;
    }

    operator VkSpecializationInfo const&() const
    {
      return *reinterpret_cast<const VkSpecializationInfo*>( this );
    }

    operator VkSpecializationInfo &()
    {
      return *reinterpret_cast<VkSpecializationInfo*>( this );
    }

    bool operator==( SpecializationInfo const& rhs ) const
    {
      return ( mapEntryCount == rhs.mapEntryCount )
          && ( pMapEntries == rhs.pMapEntries )
          && ( dataSize == rhs.dataSize )
          && ( pData == rhs.pData );
    }

    bool operator!=( SpecializationInfo const& rhs ) const
    {
      return !operator==( rhs );
    }

    uint32_t mapEntryCount;
    const SpecializationMapEntry* pMapEntries;
    size_t dataSize;
    const void* pData;
  };
  static_assert( sizeof( SpecializationInfo ) == sizeof( VkSpecializationInfo ), "struct and wrapper have different size!" );

  struct PipelineShaderStageCreateInfo
  {
    PipelineShaderStageCreateInfo( PipelineShaderStageCreateFlags flags_ = PipelineShaderStageCreateFlags(),
                                   ShaderStageFlagBits stage_ = ShaderStageFlagBits::eVertex,
                                   ShaderModule module_ = ShaderModule(),
                                   const char* pName_ = nullptr,
                                   const SpecializationInfo* pSpecializationInfo_ = nullptr )
      : flags( flags_ )
      , stage( stage_ )
      , module( module_ )
      , pName( pName_ )
      , pSpecializationInfo( pSpecializationInfo_ )
    {}

    PipelineShaderStageCreateInfo( VkPipelineShaderStageCreateInfo const & rhs )
    {
      memcpy( this, &rhs, sizeof( PipelineShaderStageCreateInfo ) );
    }

    PipelineShaderStageCreateInfo& operator=( VkPipelineShaderStageCreateInfo const & rhs )
    {
      memcpy( this, &rhs, sizeof( PipelineShaderStageCreateInfo ) );
      return *this;
    }

    PipelineShaderStageCreateInfo & setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    PipelineShaderStageCreateInfo & setFlags( PipelineShaderStageCreateFlags flags_ )
    {
      flags = flags_;
      return *this;
    }

    PipelineShaderStageCreateInfo & setStage( ShaderStageFlagBits stage_ )
    {
      stage = stage_;
      return *this;
    }

    PipelineShaderStageCreateInfo & setModule( ShaderModule module_ )
    {
      module = module_;
      return *this;
    }

    PipelineShaderStageCreateInfo & setPName( const char* pName_ )
    {
      pName = pName_;
      return *this;
    }

    PipelineShaderStageCreateInfo & setPSpecializationInfo( const SpecializationInfo* pSpecializationInfo_ )
    {
      pSpecializationInfo = pSpecializationInfo_;
      return *this;
    }

    operator VkPipelineShaderStageCreateInfo const&() const
    {
      return *reinterpret_cast<const VkPipelineShaderStageCreateInfo*>( this );
    }

    operator VkPipelineShaderStageCreateInfo &()
    {
      return *reinterpret_cast<VkPipelineShaderStageCreateInfo*>( this );
    }

    bool operator==( PipelineShaderStageCreateInfo const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( flags == rhs.flags )
          && ( stage == rhs.stage )
          && ( module == rhs.module )
          && ( pName == rhs.pName )
          && ( pSpecializationInfo == rhs.pSpecializationInfo );
    }

    bool operator!=( PipelineShaderStageCreateInfo const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::ePipelineShaderStageCreateInfo;

  public:
    const void* pNext = nullptr;
    PipelineShaderStageCreateFlags flags;
    ShaderStageFlagBits stage;
    ShaderModule module;
    const char* pName;
    const SpecializationInfo* pSpecializationInfo;
  };
  static_assert( sizeof( PipelineShaderStageCreateInfo ) == sizeof( VkPipelineShaderStageCreateInfo ), "struct and wrapper have different size!" );

  struct ComputePipelineCreateInfo
  {
    ComputePipelineCreateInfo( PipelineCreateFlags flags_ = PipelineCreateFlags(),
                               PipelineShaderStageCreateInfo stage_ = PipelineShaderStageCreateInfo(),
                               PipelineLayout layout_ = PipelineLayout(),
                               Pipeline basePipelineHandle_ = Pipeline(),
                               int32_t basePipelineIndex_ = 0 )
      : flags( flags_ )
      , stage( stage_ )
      , layout( layout_ )
      , basePipelineHandle( basePipelineHandle_ )
      , basePipelineIndex( basePipelineIndex_ )
    {}

    ComputePipelineCreateInfo( VkComputePipelineCreateInfo const & rhs )
    {
      memcpy( this, &rhs, sizeof( ComputePipelineCreateInfo ) );
    }

    ComputePipelineCreateInfo& operator=( VkComputePipelineCreateInfo const & rhs )
    {
      memcpy( this, &rhs, sizeof( ComputePipelineCreateInfo ) );
      return *this;
    }

    ComputePipelineCreateInfo & setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    ComputePipelineCreateInfo & setFlags( PipelineCreateFlags flags_ )
    {
      flags = flags_;
      return *this;
    }

    ComputePipelineCreateInfo & setStage( PipelineShaderStageCreateInfo stage_ )
    {
      stage = stage_;
      return *this;
    }

    ComputePipelineCreateInfo & setLayout( PipelineLayout layout_ )
    {
      layout = layout_;
      return *this;
    }

    ComputePipelineCreateInfo & setBasePipelineHandle( Pipeline basePipelineHandle_ )
    {
      basePipelineHandle = basePipelineHandle_;
      return *this;
    }

    ComputePipelineCreateInfo & setBasePipelineIndex( int32_t basePipelineIndex_ )
    {
      basePipelineIndex = basePipelineIndex_;
      return *this;
    }

    operator VkComputePipelineCreateInfo const&() const
    {
      return *reinterpret_cast<const VkComputePipelineCreateInfo*>( this );
    }

    operator VkComputePipelineCreateInfo &()
    {
      return *reinterpret_cast<VkComputePipelineCreateInfo*>( this );
    }

    bool operator==( ComputePipelineCreateInfo const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( flags == rhs.flags )
          && ( stage == rhs.stage )
          && ( layout == rhs.layout )
          && ( basePipelineHandle == rhs.basePipelineHandle )
          && ( basePipelineIndex == rhs.basePipelineIndex );
    }

    bool operator!=( ComputePipelineCreateInfo const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eComputePipelineCreateInfo;

  public:
    const void* pNext = nullptr;
    PipelineCreateFlags flags;
    PipelineShaderStageCreateInfo stage;
    PipelineLayout layout;
    Pipeline basePipelineHandle;
    int32_t basePipelineIndex;
  };
  static_assert( sizeof( ComputePipelineCreateInfo ) == sizeof( VkComputePipelineCreateInfo ), "struct and wrapper have different size!" );

  struct ConditionalRenderingBeginInfoEXT
  {
    ConditionalRenderingBeginInfoEXT( Buffer buffer_ = Buffer(),
                                      DeviceSize offset_ = 0,
                                      ConditionalRenderingFlagsEXT flags_ = ConditionalRenderingFlagsEXT() )
      : buffer( buffer_ )
      , offset( offset_ )
      , flags( flags_ )
    {}

    ConditionalRenderingBeginInfoEXT( VkConditionalRenderingBeginInfoEXT const & rhs )
    {
      memcpy( this, &rhs, sizeof( ConditionalRenderingBeginInfoEXT ) );
    }

    ConditionalRenderingBeginInfoEXT& operator=( VkConditionalRenderingBeginInfoEXT const & rhs )
    {
      memcpy( this, &rhs, sizeof( ConditionalRenderingBeginInfoEXT ) );
      return *this;
    }

    ConditionalRenderingBeginInfoEXT & setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    ConditionalRenderingBeginInfoEXT & setBuffer( Buffer buffer_ )
    {
      buffer = buffer_;
      return *this;
    }

    ConditionalRenderingBeginInfoEXT & setOffset( DeviceSize offset_ )
    {
      offset = offset_;
      return *this;
    }

    ConditionalRenderingBeginInfoEXT & setFlags( ConditionalRenderingFlagsEXT flags_ )
    {
      flags = flags_;
      return *this;
    }

    operator VkConditionalRenderingBeginInfoEXT const&() const
    {
      return *reinterpret_cast<const VkConditionalRenderingBeginInfoEXT*>( this );
    }

    operator VkConditionalRenderingBeginInfoEXT &()
    {
      return *reinterpret_cast<VkConditionalRenderingBeginInfoEXT*>( this );
    }

    bool operator==( ConditionalRenderingBeginInfoEXT const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( buffer == rhs.buffer )
          && ( offset == rhs.offset )
          && ( flags == rhs.flags );
    }

    bool operator!=( ConditionalRenderingBeginInfoEXT const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eConditionalRenderingBeginInfoEXT;

  public:
    const void* pNext = nullptr;
    Buffer buffer;
    DeviceSize offset;
    ConditionalRenderingFlagsEXT flags;
  };
  static_assert( sizeof( ConditionalRenderingBeginInfoEXT ) == sizeof( VkConditionalRenderingBeginInfoEXT ), "struct and wrapper have different size!" );

  struct ConformanceVersionKHR
  {
    ConformanceVersionKHR( uint8_t major_ = 0,
                           uint8_t minor_ = 0,
                           uint8_t subminor_ = 0,
                           uint8_t patch_ = 0 )
      : major( major_ )
      , minor( minor_ )
      , subminor( subminor_ )
      , patch( patch_ )
    {}

    ConformanceVersionKHR( VkConformanceVersionKHR const & rhs )
    {
      memcpy( this, &rhs, sizeof( ConformanceVersionKHR ) );
    }

    ConformanceVersionKHR& operator=( VkConformanceVersionKHR const & rhs )
    {
      memcpy( this, &rhs, sizeof( ConformanceVersionKHR ) );
      return *this;
    }

    ConformanceVersionKHR & setMajor( uint8_t major_ )
    {
      major = major_;
      return *this;
    }

    ConformanceVersionKHR & setMinor( uint8_t minor_ )
    {
      minor = minor_;
      return *this;
    }

    ConformanceVersionKHR & setSubminor( uint8_t subminor_ )
    {
      subminor = subminor_;
      return *this;
    }

    ConformanceVersionKHR & setPatch( uint8_t patch_ )
    {
      patch = patch_;
      return *this;
    }

    operator VkConformanceVersionKHR const&() const
    {
      return *reinterpret_cast<const VkConformanceVersionKHR*>( this );
    }

    operator VkConformanceVersionKHR &()
    {
      return *reinterpret_cast<VkConformanceVersionKHR*>( this );
    }

    bool operator==( ConformanceVersionKHR const& rhs ) const
    {
      return ( major == rhs.major )
          && ( minor == rhs.minor )
          && ( subminor == rhs.subminor )
          && ( patch == rhs.patch );
    }

    bool operator!=( ConformanceVersionKHR const& rhs ) const
    {
      return !operator==( rhs );
    }

    uint8_t major;
    uint8_t minor;
    uint8_t subminor;
    uint8_t patch;
  };
  static_assert( sizeof( ConformanceVersionKHR ) == sizeof( VkConformanceVersionKHR ), "struct and wrapper have different size!" );

  struct CooperativeMatrixPropertiesNV
  {
    CooperativeMatrixPropertiesNV( uint32_t MSize_ = 0,
                                   uint32_t NSize_ = 0,
                                   uint32_t KSize_ = 0,
                                   ComponentTypeNV AType_ = ComponentTypeNV::eFloat16,
                                   ComponentTypeNV BType_ = ComponentTypeNV::eFloat16,
                                   ComponentTypeNV CType_ = ComponentTypeNV::eFloat16,
                                   ComponentTypeNV DType_ = ComponentTypeNV::eFloat16,
                                   ScopeNV scope_ = ScopeNV::eDevice )
      : MSize( MSize_ )
      , NSize( NSize_ )
      , KSize( KSize_ )
      , AType( AType_ )
      , BType( BType_ )
      , CType( CType_ )
      , DType( DType_ )
      , scope( scope_ )
    {}

    CooperativeMatrixPropertiesNV( VkCooperativeMatrixPropertiesNV const & rhs )
    {
      memcpy( this, &rhs, sizeof( CooperativeMatrixPropertiesNV ) );
    }

    CooperativeMatrixPropertiesNV& operator=( VkCooperativeMatrixPropertiesNV const & rhs )
    {
      memcpy( this, &rhs, sizeof( CooperativeMatrixPropertiesNV ) );
      return *this;
    }

    CooperativeMatrixPropertiesNV & setPNext( void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    CooperativeMatrixPropertiesNV & setMSize( uint32_t MSize_ )
    {
      MSize = MSize_;
      return *this;
    }

    CooperativeMatrixPropertiesNV & setNSize( uint32_t NSize_ )
    {
      NSize = NSize_;
      return *this;
    }

    CooperativeMatrixPropertiesNV & setKSize( uint32_t KSize_ )
    {
      KSize = KSize_;
      return *this;
    }

    CooperativeMatrixPropertiesNV & setAType( ComponentTypeNV AType_ )
    {
      AType = AType_;
      return *this;
    }

    CooperativeMatrixPropertiesNV & setBType( ComponentTypeNV BType_ )
    {
      BType = BType_;
      return *this;
    }

    CooperativeMatrixPropertiesNV & setCType( ComponentTypeNV CType_ )
    {
      CType = CType_;
      return *this;
    }

    CooperativeMatrixPropertiesNV & setDType( ComponentTypeNV DType_ )
    {
      DType = DType_;
      return *this;
    }

    CooperativeMatrixPropertiesNV & setScope( ScopeNV scope_ )
    {
      scope = scope_;
      return *this;
    }

    operator VkCooperativeMatrixPropertiesNV const&() const
    {
      return *reinterpret_cast<const VkCooperativeMatrixPropertiesNV*>( this );
    }

    operator VkCooperativeMatrixPropertiesNV &()
    {
      return *reinterpret_cast<VkCooperativeMatrixPropertiesNV*>( this );
    }

    bool operator==( CooperativeMatrixPropertiesNV const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( MSize == rhs.MSize )
          && ( NSize == rhs.NSize )
          && ( KSize == rhs.KSize )
          && ( AType == rhs.AType )
          && ( BType == rhs.BType )
          && ( CType == rhs.CType )
          && ( DType == rhs.DType )
          && ( scope == rhs.scope );
    }

    bool operator!=( CooperativeMatrixPropertiesNV const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eCooperativeMatrixPropertiesNV;

  public:
    void* pNext = nullptr;
    uint32_t MSize;
    uint32_t NSize;
    uint32_t KSize;
    ComponentTypeNV AType;
    ComponentTypeNV BType;
    ComponentTypeNV CType;
    ComponentTypeNV DType;
    ScopeNV scope;
  };
  static_assert( sizeof( CooperativeMatrixPropertiesNV ) == sizeof( VkCooperativeMatrixPropertiesNV ), "struct and wrapper have different size!" );

  struct CopyDescriptorSet
  {
    CopyDescriptorSet( DescriptorSet srcSet_ = DescriptorSet(),
                       uint32_t srcBinding_ = 0,
                       uint32_t srcArrayElement_ = 0,
                       DescriptorSet dstSet_ = DescriptorSet(),
                       uint32_t dstBinding_ = 0,
                       uint32_t dstArrayElement_ = 0,
                       uint32_t descriptorCount_ = 0 )
      : srcSet( srcSet_ )
      , srcBinding( srcBinding_ )
      , srcArrayElement( srcArrayElement_ )
      , dstSet( dstSet_ )
      , dstBinding( dstBinding_ )
      , dstArrayElement( dstArrayElement_ )
      , descriptorCount( descriptorCount_ )
    {}

    CopyDescriptorSet( VkCopyDescriptorSet const & rhs )
    {
      memcpy( this, &rhs, sizeof( CopyDescriptorSet ) );
    }

    CopyDescriptorSet& operator=( VkCopyDescriptorSet const & rhs )
    {
      memcpy( this, &rhs, sizeof( CopyDescriptorSet ) );
      return *this;
    }

    CopyDescriptorSet & setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    CopyDescriptorSet & setSrcSet( DescriptorSet srcSet_ )
    {
      srcSet = srcSet_;
      return *this;
    }

    CopyDescriptorSet & setSrcBinding( uint32_t srcBinding_ )
    {
      srcBinding = srcBinding_;
      return *this;
    }

    CopyDescriptorSet & setSrcArrayElement( uint32_t srcArrayElement_ )
    {
      srcArrayElement = srcArrayElement_;
      return *this;
    }

    CopyDescriptorSet & setDstSet( DescriptorSet dstSet_ )
    {
      dstSet = dstSet_;
      return *this;
    }

    CopyDescriptorSet & setDstBinding( uint32_t dstBinding_ )
    {
      dstBinding = dstBinding_;
      return *this;
    }

    CopyDescriptorSet & setDstArrayElement( uint32_t dstArrayElement_ )
    {
      dstArrayElement = dstArrayElement_;
      return *this;
    }

    CopyDescriptorSet & setDescriptorCount( uint32_t descriptorCount_ )
    {
      descriptorCount = descriptorCount_;
      return *this;
    }

    operator VkCopyDescriptorSet const&() const
    {
      return *reinterpret_cast<const VkCopyDescriptorSet*>( this );
    }

    operator VkCopyDescriptorSet &()
    {
      return *reinterpret_cast<VkCopyDescriptorSet*>( this );
    }

    bool operator==( CopyDescriptorSet const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( srcSet == rhs.srcSet )
          && ( srcBinding == rhs.srcBinding )
          && ( srcArrayElement == rhs.srcArrayElement )
          && ( dstSet == rhs.dstSet )
          && ( dstBinding == rhs.dstBinding )
          && ( dstArrayElement == rhs.dstArrayElement )
          && ( descriptorCount == rhs.descriptorCount );
    }

    bool operator!=( CopyDescriptorSet const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eCopyDescriptorSet;

  public:
    const void* pNext = nullptr;
    DescriptorSet srcSet;
    uint32_t srcBinding;
    uint32_t srcArrayElement;
    DescriptorSet dstSet;
    uint32_t dstBinding;
    uint32_t dstArrayElement;
    uint32_t descriptorCount;
  };
  static_assert( sizeof( CopyDescriptorSet ) == sizeof( VkCopyDescriptorSet ), "struct and wrapper have different size!" );

#ifdef VK_USE_PLATFORM_WIN32_KHR
  struct D3D12FenceSubmitInfoKHR
  {
    D3D12FenceSubmitInfoKHR( uint32_t waitSemaphoreValuesCount_ = 0,
                             const uint64_t* pWaitSemaphoreValues_ = nullptr,
                             uint32_t signalSemaphoreValuesCount_ = 0,
                             const uint64_t* pSignalSemaphoreValues_ = nullptr )
      : waitSemaphoreValuesCount( waitSemaphoreValuesCount_ )
      , pWaitSemaphoreValues( pWaitSemaphoreValues_ )
      , signalSemaphoreValuesCount( signalSemaphoreValuesCount_ )
      , pSignalSemaphoreValues( pSignalSemaphoreValues_ )
    {}

    D3D12FenceSubmitInfoKHR( VkD3D12FenceSubmitInfoKHR const & rhs )
    {
      memcpy( this, &rhs, sizeof( D3D12FenceSubmitInfoKHR ) );
    }

    D3D12FenceSubmitInfoKHR& operator=( VkD3D12FenceSubmitInfoKHR const & rhs )
    {
      memcpy( this, &rhs, sizeof( D3D12FenceSubmitInfoKHR ) );
      return *this;
    }

    D3D12FenceSubmitInfoKHR & setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    D3D12FenceSubmitInfoKHR & setWaitSemaphoreValuesCount( uint32_t waitSemaphoreValuesCount_ )
    {
      waitSemaphoreValuesCount = waitSemaphoreValuesCount_;
      return *this;
    }

    D3D12FenceSubmitInfoKHR & setPWaitSemaphoreValues( const uint64_t* pWaitSemaphoreValues_ )
    {
      pWaitSemaphoreValues = pWaitSemaphoreValues_;
      return *this;
    }

    D3D12FenceSubmitInfoKHR & setSignalSemaphoreValuesCount( uint32_t signalSemaphoreValuesCount_ )
    {
      signalSemaphoreValuesCount = signalSemaphoreValuesCount_;
      return *this;
    }

    D3D12FenceSubmitInfoKHR & setPSignalSemaphoreValues( const uint64_t* pSignalSemaphoreValues_ )
    {
      pSignalSemaphoreValues = pSignalSemaphoreValues_;
      return *this;
    }

    operator VkD3D12FenceSubmitInfoKHR const&() const
    {
      return *reinterpret_cast<const VkD3D12FenceSubmitInfoKHR*>( this );
    }

    operator VkD3D12FenceSubmitInfoKHR &()
    {
      return *reinterpret_cast<VkD3D12FenceSubmitInfoKHR*>( this );
    }

    bool operator==( D3D12FenceSubmitInfoKHR const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( waitSemaphoreValuesCount == rhs.waitSemaphoreValuesCount )
          && ( pWaitSemaphoreValues == rhs.pWaitSemaphoreValues )
          && ( signalSemaphoreValuesCount == rhs.signalSemaphoreValuesCount )
          && ( pSignalSemaphoreValues == rhs.pSignalSemaphoreValues );
    }

    bool operator!=( D3D12FenceSubmitInfoKHR const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eD3D12FenceSubmitInfoKHR;

  public:
    const void* pNext = nullptr;
    uint32_t waitSemaphoreValuesCount;
    const uint64_t* pWaitSemaphoreValues;
    uint32_t signalSemaphoreValuesCount;
    const uint64_t* pSignalSemaphoreValues;
  };
  static_assert( sizeof( D3D12FenceSubmitInfoKHR ) == sizeof( VkD3D12FenceSubmitInfoKHR ), "struct and wrapper have different size!" );
#endif /*VK_USE_PLATFORM_WIN32_KHR*/

  struct DebugMarkerMarkerInfoEXT
  {
    DebugMarkerMarkerInfoEXT( const char* pMarkerName_ = nullptr,
                              std::array<float,4> const& color_ = { { 0 } } )
      : pMarkerName( pMarkerName_ )
    {
      memcpy( &color, color_.data(), 4 * sizeof( float ) );
    }

    DebugMarkerMarkerInfoEXT( VkDebugMarkerMarkerInfoEXT const & rhs )
    {
      memcpy( this, &rhs, sizeof( DebugMarkerMarkerInfoEXT ) );
    }

    DebugMarkerMarkerInfoEXT& operator=( VkDebugMarkerMarkerInfoEXT const & rhs )
    {
      memcpy( this, &rhs, sizeof( DebugMarkerMarkerInfoEXT ) );
      return *this;
    }

    DebugMarkerMarkerInfoEXT & setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    DebugMarkerMarkerInfoEXT & setPMarkerName( const char* pMarkerName_ )
    {
      pMarkerName = pMarkerName_;
      return *this;
    }

    DebugMarkerMarkerInfoEXT & setColor( std::array<float,4> color_ )
    {
      memcpy( color, color_.data(), 4 * sizeof( float ) );
      return *this;
    }

    operator VkDebugMarkerMarkerInfoEXT const&() const
    {
      return *reinterpret_cast<const VkDebugMarkerMarkerInfoEXT*>( this );
    }

    operator VkDebugMarkerMarkerInfoEXT &()
    {
      return *reinterpret_cast<VkDebugMarkerMarkerInfoEXT*>( this );
    }

    bool operator==( DebugMarkerMarkerInfoEXT const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( pMarkerName == rhs.pMarkerName )
          && ( memcmp( color, rhs.color, 4 * sizeof( float ) ) == 0 );
    }

    bool operator!=( DebugMarkerMarkerInfoEXT const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eDebugMarkerMarkerInfoEXT;

  public:
    const void* pNext = nullptr;
    const char* pMarkerName;
    float color[4];
  };
  static_assert( sizeof( DebugMarkerMarkerInfoEXT ) == sizeof( VkDebugMarkerMarkerInfoEXT ), "struct and wrapper have different size!" );

  struct DebugMarkerObjectNameInfoEXT
  {
    DebugMarkerObjectNameInfoEXT( DebugReportObjectTypeEXT objectType_ = DebugReportObjectTypeEXT::eUnknown,
                                  uint64_t object_ = 0,
                                  const char* pObjectName_ = nullptr )
      : objectType( objectType_ )
      , object( object_ )
      , pObjectName( pObjectName_ )
    {}

    DebugMarkerObjectNameInfoEXT( VkDebugMarkerObjectNameInfoEXT const & rhs )
    {
      memcpy( this, &rhs, sizeof( DebugMarkerObjectNameInfoEXT ) );
    }

    DebugMarkerObjectNameInfoEXT& operator=( VkDebugMarkerObjectNameInfoEXT const & rhs )
    {
      memcpy( this, &rhs, sizeof( DebugMarkerObjectNameInfoEXT ) );
      return *this;
    }

    DebugMarkerObjectNameInfoEXT & setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    DebugMarkerObjectNameInfoEXT & setObjectType( DebugReportObjectTypeEXT objectType_ )
    {
      objectType = objectType_;
      return *this;
    }

    DebugMarkerObjectNameInfoEXT & setObject( uint64_t object_ )
    {
      object = object_;
      return *this;
    }

    DebugMarkerObjectNameInfoEXT & setPObjectName( const char* pObjectName_ )
    {
      pObjectName = pObjectName_;
      return *this;
    }

    operator VkDebugMarkerObjectNameInfoEXT const&() const
    {
      return *reinterpret_cast<const VkDebugMarkerObjectNameInfoEXT*>( this );
    }

    operator VkDebugMarkerObjectNameInfoEXT &()
    {
      return *reinterpret_cast<VkDebugMarkerObjectNameInfoEXT*>( this );
    }

    bool operator==( DebugMarkerObjectNameInfoEXT const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( objectType == rhs.objectType )
          && ( object == rhs.object )
          && ( pObjectName == rhs.pObjectName );
    }

    bool operator!=( DebugMarkerObjectNameInfoEXT const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eDebugMarkerObjectNameInfoEXT;

  public:
    const void* pNext = nullptr;
    DebugReportObjectTypeEXT objectType;
    uint64_t object;
    const char* pObjectName;
  };
  static_assert( sizeof( DebugMarkerObjectNameInfoEXT ) == sizeof( VkDebugMarkerObjectNameInfoEXT ), "struct and wrapper have different size!" );

  struct DebugMarkerObjectTagInfoEXT
  {
    DebugMarkerObjectTagInfoEXT( DebugReportObjectTypeEXT objectType_ = DebugReportObjectTypeEXT::eUnknown,
                                 uint64_t object_ = 0,
                                 uint64_t tagName_ = 0,
                                 size_t tagSize_ = 0,
                                 const void* pTag_ = nullptr )
      : objectType( objectType_ )
      , object( object_ )
      , tagName( tagName_ )
      , tagSize( tagSize_ )
      , pTag( pTag_ )
    {}

    DebugMarkerObjectTagInfoEXT( VkDebugMarkerObjectTagInfoEXT const & rhs )
    {
      memcpy( this, &rhs, sizeof( DebugMarkerObjectTagInfoEXT ) );
    }

    DebugMarkerObjectTagInfoEXT& operator=( VkDebugMarkerObjectTagInfoEXT const & rhs )
    {
      memcpy( this, &rhs, sizeof( DebugMarkerObjectTagInfoEXT ) );
      return *this;
    }

    DebugMarkerObjectTagInfoEXT & setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    DebugMarkerObjectTagInfoEXT & setObjectType( DebugReportObjectTypeEXT objectType_ )
    {
      objectType = objectType_;
      return *this;
    }

    DebugMarkerObjectTagInfoEXT & setObject( uint64_t object_ )
    {
      object = object_;
      return *this;
    }

    DebugMarkerObjectTagInfoEXT & setTagName( uint64_t tagName_ )
    {
      tagName = tagName_;
      return *this;
    }

    DebugMarkerObjectTagInfoEXT & setTagSize( size_t tagSize_ )
    {
      tagSize = tagSize_;
      return *this;
    }

    DebugMarkerObjectTagInfoEXT & setPTag( const void* pTag_ )
    {
      pTag = pTag_;
      return *this;
    }

    operator VkDebugMarkerObjectTagInfoEXT const&() const
    {
      return *reinterpret_cast<const VkDebugMarkerObjectTagInfoEXT*>( this );
    }

    operator VkDebugMarkerObjectTagInfoEXT &()
    {
      return *reinterpret_cast<VkDebugMarkerObjectTagInfoEXT*>( this );
    }

    bool operator==( DebugMarkerObjectTagInfoEXT const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( objectType == rhs.objectType )
          && ( object == rhs.object )
          && ( tagName == rhs.tagName )
          && ( tagSize == rhs.tagSize )
          && ( pTag == rhs.pTag );
    }

    bool operator!=( DebugMarkerObjectTagInfoEXT const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eDebugMarkerObjectTagInfoEXT;

  public:
    const void* pNext = nullptr;
    DebugReportObjectTypeEXT objectType;
    uint64_t object;
    uint64_t tagName;
    size_t tagSize;
    const void* pTag;
  };
  static_assert( sizeof( DebugMarkerObjectTagInfoEXT ) == sizeof( VkDebugMarkerObjectTagInfoEXT ), "struct and wrapper have different size!" );

  struct DebugReportCallbackCreateInfoEXT
  {
    DebugReportCallbackCreateInfoEXT( DebugReportFlagsEXT flags_ = DebugReportFlagsEXT(),
                                      PFN_vkDebugReportCallbackEXT pfnCallback_ = nullptr,
                                      void* pUserData_ = nullptr )
      : flags( flags_ )
      , pfnCallback( pfnCallback_ )
      , pUserData( pUserData_ )
    {}

    DebugReportCallbackCreateInfoEXT( VkDebugReportCallbackCreateInfoEXT const & rhs )
    {
      memcpy( this, &rhs, sizeof( DebugReportCallbackCreateInfoEXT ) );
    }

    DebugReportCallbackCreateInfoEXT& operator=( VkDebugReportCallbackCreateInfoEXT const & rhs )
    {
      memcpy( this, &rhs, sizeof( DebugReportCallbackCreateInfoEXT ) );
      return *this;
    }

    DebugReportCallbackCreateInfoEXT & setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    DebugReportCallbackCreateInfoEXT & setFlags( DebugReportFlagsEXT flags_ )
    {
      flags = flags_;
      return *this;
    }

    DebugReportCallbackCreateInfoEXT & setPfnCallback( PFN_vkDebugReportCallbackEXT pfnCallback_ )
    {
      pfnCallback = pfnCallback_;
      return *this;
    }

    DebugReportCallbackCreateInfoEXT & setPUserData( void* pUserData_ )
    {
      pUserData = pUserData_;
      return *this;
    }

    operator VkDebugReportCallbackCreateInfoEXT const&() const
    {
      return *reinterpret_cast<const VkDebugReportCallbackCreateInfoEXT*>( this );
    }

    operator VkDebugReportCallbackCreateInfoEXT &()
    {
      return *reinterpret_cast<VkDebugReportCallbackCreateInfoEXT*>( this );
    }

    bool operator==( DebugReportCallbackCreateInfoEXT const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( flags == rhs.flags )
          && ( pfnCallback == rhs.pfnCallback )
          && ( pUserData == rhs.pUserData );
    }

    bool operator!=( DebugReportCallbackCreateInfoEXT const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eDebugReportCallbackCreateInfoEXT;

  public:
    const void* pNext = nullptr;
    DebugReportFlagsEXT flags;
    PFN_vkDebugReportCallbackEXT pfnCallback;
    void* pUserData;
  };
  static_assert( sizeof( DebugReportCallbackCreateInfoEXT ) == sizeof( VkDebugReportCallbackCreateInfoEXT ), "struct and wrapper have different size!" );

  struct DebugUtilsLabelEXT
  {
    DebugUtilsLabelEXT( const char* pLabelName_ = nullptr,
                        std::array<float,4> const& color_ = { { 0 } } )
      : pLabelName( pLabelName_ )
    {
      memcpy( &color, color_.data(), 4 * sizeof( float ) );
    }

    DebugUtilsLabelEXT( VkDebugUtilsLabelEXT const & rhs )
    {
      memcpy( this, &rhs, sizeof( DebugUtilsLabelEXT ) );
    }

    DebugUtilsLabelEXT& operator=( VkDebugUtilsLabelEXT const & rhs )
    {
      memcpy( this, &rhs, sizeof( DebugUtilsLabelEXT ) );
      return *this;
    }

    DebugUtilsLabelEXT & setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    DebugUtilsLabelEXT & setPLabelName( const char* pLabelName_ )
    {
      pLabelName = pLabelName_;
      return *this;
    }

    DebugUtilsLabelEXT & setColor( std::array<float,4> color_ )
    {
      memcpy( color, color_.data(), 4 * sizeof( float ) );
      return *this;
    }

    operator VkDebugUtilsLabelEXT const&() const
    {
      return *reinterpret_cast<const VkDebugUtilsLabelEXT*>( this );
    }

    operator VkDebugUtilsLabelEXT &()
    {
      return *reinterpret_cast<VkDebugUtilsLabelEXT*>( this );
    }

    bool operator==( DebugUtilsLabelEXT const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( pLabelName == rhs.pLabelName )
          && ( memcmp( color, rhs.color, 4 * sizeof( float ) ) == 0 );
    }

    bool operator!=( DebugUtilsLabelEXT const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eDebugUtilsLabelEXT;

  public:
    const void* pNext = nullptr;
    const char* pLabelName;
    float color[4];
  };
  static_assert( sizeof( DebugUtilsLabelEXT ) == sizeof( VkDebugUtilsLabelEXT ), "struct and wrapper have different size!" );

  struct DebugUtilsObjectNameInfoEXT
  {
    DebugUtilsObjectNameInfoEXT( ObjectType objectType_ = ObjectType::eUnknown,
                                 uint64_t objectHandle_ = 0,
                                 const char* pObjectName_ = nullptr )
      : objectType( objectType_ )
      , objectHandle( objectHandle_ )
      , pObjectName( pObjectName_ )
    {}

    DebugUtilsObjectNameInfoEXT( VkDebugUtilsObjectNameInfoEXT const & rhs )
    {
      memcpy( this, &rhs, sizeof( DebugUtilsObjectNameInfoEXT ) );
    }

    DebugUtilsObjectNameInfoEXT& operator=( VkDebugUtilsObjectNameInfoEXT const & rhs )
    {
      memcpy( this, &rhs, sizeof( DebugUtilsObjectNameInfoEXT ) );
      return *this;
    }

    DebugUtilsObjectNameInfoEXT & setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    DebugUtilsObjectNameInfoEXT & setObjectType( ObjectType objectType_ )
    {
      objectType = objectType_;
      return *this;
    }

    DebugUtilsObjectNameInfoEXT & setObjectHandle( uint64_t objectHandle_ )
    {
      objectHandle = objectHandle_;
      return *this;
    }

    DebugUtilsObjectNameInfoEXT & setPObjectName( const char* pObjectName_ )
    {
      pObjectName = pObjectName_;
      return *this;
    }

    operator VkDebugUtilsObjectNameInfoEXT const&() const
    {
      return *reinterpret_cast<const VkDebugUtilsObjectNameInfoEXT*>( this );
    }

    operator VkDebugUtilsObjectNameInfoEXT &()
    {
      return *reinterpret_cast<VkDebugUtilsObjectNameInfoEXT*>( this );
    }

    bool operator==( DebugUtilsObjectNameInfoEXT const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( objectType == rhs.objectType )
          && ( objectHandle == rhs.objectHandle )
          && ( pObjectName == rhs.pObjectName );
    }

    bool operator!=( DebugUtilsObjectNameInfoEXT const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eDebugUtilsObjectNameInfoEXT;

  public:
    const void* pNext = nullptr;
    ObjectType objectType;
    uint64_t objectHandle;
    const char* pObjectName;
  };
  static_assert( sizeof( DebugUtilsObjectNameInfoEXT ) == sizeof( VkDebugUtilsObjectNameInfoEXT ), "struct and wrapper have different size!" );

  struct DebugUtilsMessengerCallbackDataEXT
  {
    DebugUtilsMessengerCallbackDataEXT( DebugUtilsMessengerCallbackDataFlagsEXT flags_ = DebugUtilsMessengerCallbackDataFlagsEXT(),
                                        const char* pMessageIdName_ = nullptr,
                                        int32_t messageIdNumber_ = 0,
                                        const char* pMessage_ = nullptr,
                                        uint32_t queueLabelCount_ = 0,
                                        const DebugUtilsLabelEXT* pQueueLabels_ = nullptr,
                                        uint32_t cmdBufLabelCount_ = 0,
                                        const DebugUtilsLabelEXT* pCmdBufLabels_ = nullptr,
                                        uint32_t objectCount_ = 0,
                                        const DebugUtilsObjectNameInfoEXT* pObjects_ = nullptr )
      : flags( flags_ )
      , pMessageIdName( pMessageIdName_ )
      , messageIdNumber( messageIdNumber_ )
      , pMessage( pMessage_ )
      , queueLabelCount( queueLabelCount_ )
      , pQueueLabels( pQueueLabels_ )
      , cmdBufLabelCount( cmdBufLabelCount_ )
      , pCmdBufLabels( pCmdBufLabels_ )
      , objectCount( objectCount_ )
      , pObjects( pObjects_ )
    {}

    DebugUtilsMessengerCallbackDataEXT( VkDebugUtilsMessengerCallbackDataEXT const & rhs )
    {
      memcpy( this, &rhs, sizeof( DebugUtilsMessengerCallbackDataEXT ) );
    }

    DebugUtilsMessengerCallbackDataEXT& operator=( VkDebugUtilsMessengerCallbackDataEXT const & rhs )
    {
      memcpy( this, &rhs, sizeof( DebugUtilsMessengerCallbackDataEXT ) );
      return *this;
    }

    DebugUtilsMessengerCallbackDataEXT & setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    DebugUtilsMessengerCallbackDataEXT & setFlags( DebugUtilsMessengerCallbackDataFlagsEXT flags_ )
    {
      flags = flags_;
      return *this;
    }

    DebugUtilsMessengerCallbackDataEXT & setPMessageIdName( const char* pMessageIdName_ )
    {
      pMessageIdName = pMessageIdName_;
      return *this;
    }

    DebugUtilsMessengerCallbackDataEXT & setMessageIdNumber( int32_t messageIdNumber_ )
    {
      messageIdNumber = messageIdNumber_;
      return *this;
    }

    DebugUtilsMessengerCallbackDataEXT & setPMessage( const char* pMessage_ )
    {
      pMessage = pMessage_;
      return *this;
    }

    DebugUtilsMessengerCallbackDataEXT & setQueueLabelCount( uint32_t queueLabelCount_ )
    {
      queueLabelCount = queueLabelCount_;
      return *this;
    }

    DebugUtilsMessengerCallbackDataEXT & setPQueueLabels( const DebugUtilsLabelEXT* pQueueLabels_ )
    {
      pQueueLabels = pQueueLabels_;
      return *this;
    }

    DebugUtilsMessengerCallbackDataEXT & setCmdBufLabelCount( uint32_t cmdBufLabelCount_ )
    {
      cmdBufLabelCount = cmdBufLabelCount_;
      return *this;
    }

    DebugUtilsMessengerCallbackDataEXT & setPCmdBufLabels( const DebugUtilsLabelEXT* pCmdBufLabels_ )
    {
      pCmdBufLabels = pCmdBufLabels_;
      return *this;
    }

    DebugUtilsMessengerCallbackDataEXT & setObjectCount( uint32_t objectCount_ )
    {
      objectCount = objectCount_;
      return *this;
    }

    DebugUtilsMessengerCallbackDataEXT & setPObjects( const DebugUtilsObjectNameInfoEXT* pObjects_ )
    {
      pObjects = pObjects_;
      return *this;
    }

    operator VkDebugUtilsMessengerCallbackDataEXT const&() const
    {
      return *reinterpret_cast<const VkDebugUtilsMessengerCallbackDataEXT*>( this );
    }

    operator VkDebugUtilsMessengerCallbackDataEXT &()
    {
      return *reinterpret_cast<VkDebugUtilsMessengerCallbackDataEXT*>( this );
    }

    bool operator==( DebugUtilsMessengerCallbackDataEXT const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( flags == rhs.flags )
          && ( pMessageIdName == rhs.pMessageIdName )
          && ( messageIdNumber == rhs.messageIdNumber )
          && ( pMessage == rhs.pMessage )
          && ( queueLabelCount == rhs.queueLabelCount )
          && ( pQueueLabels == rhs.pQueueLabels )
          && ( cmdBufLabelCount == rhs.cmdBufLabelCount )
          && ( pCmdBufLabels == rhs.pCmdBufLabels )
          && ( objectCount == rhs.objectCount )
          && ( pObjects == rhs.pObjects );
    }

    bool operator!=( DebugUtilsMessengerCallbackDataEXT const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eDebugUtilsMessengerCallbackDataEXT;

  public:
    const void* pNext = nullptr;
    DebugUtilsMessengerCallbackDataFlagsEXT flags;
    const char* pMessageIdName;
    int32_t messageIdNumber;
    const char* pMessage;
    uint32_t queueLabelCount;
    const DebugUtilsLabelEXT* pQueueLabels;
    uint32_t cmdBufLabelCount;
    const DebugUtilsLabelEXT* pCmdBufLabels;
    uint32_t objectCount;
    const DebugUtilsObjectNameInfoEXT* pObjects;
  };
  static_assert( sizeof( DebugUtilsMessengerCallbackDataEXT ) == sizeof( VkDebugUtilsMessengerCallbackDataEXT ), "struct and wrapper have different size!" );

  struct DebugUtilsMessengerCreateInfoEXT
  {
    DebugUtilsMessengerCreateInfoEXT( DebugUtilsMessengerCreateFlagsEXT flags_ = DebugUtilsMessengerCreateFlagsEXT(),
                                      DebugUtilsMessageSeverityFlagsEXT messageSeverity_ = DebugUtilsMessageSeverityFlagsEXT(),
                                      DebugUtilsMessageTypeFlagsEXT messageType_ = DebugUtilsMessageTypeFlagsEXT(),
                                      PFN_vkDebugUtilsMessengerCallbackEXT pfnUserCallback_ = nullptr,
                                      void* pUserData_ = nullptr )
      : flags( flags_ )
      , messageSeverity( messageSeverity_ )
      , messageType( messageType_ )
      , pfnUserCallback( pfnUserCallback_ )
      , pUserData( pUserData_ )
    {}

    DebugUtilsMessengerCreateInfoEXT( VkDebugUtilsMessengerCreateInfoEXT const & rhs )
    {
      memcpy( this, &rhs, sizeof( DebugUtilsMessengerCreateInfoEXT ) );
    }

    DebugUtilsMessengerCreateInfoEXT& operator=( VkDebugUtilsMessengerCreateInfoEXT const & rhs )
    {
      memcpy( this, &rhs, sizeof( DebugUtilsMessengerCreateInfoEXT ) );
      return *this;
    }

    DebugUtilsMessengerCreateInfoEXT & setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    DebugUtilsMessengerCreateInfoEXT & setFlags( DebugUtilsMessengerCreateFlagsEXT flags_ )
    {
      flags = flags_;
      return *this;
    }

    DebugUtilsMessengerCreateInfoEXT & setMessageSeverity( DebugUtilsMessageSeverityFlagsEXT messageSeverity_ )
    {
      messageSeverity = messageSeverity_;
      return *this;
    }

    DebugUtilsMessengerCreateInfoEXT & setMessageType( DebugUtilsMessageTypeFlagsEXT messageType_ )
    {
      messageType = messageType_;
      return *this;
    }

    DebugUtilsMessengerCreateInfoEXT & setPfnUserCallback( PFN_vkDebugUtilsMessengerCallbackEXT pfnUserCallback_ )
    {
      pfnUserCallback = pfnUserCallback_;
      return *this;
    }

    DebugUtilsMessengerCreateInfoEXT & setPUserData( void* pUserData_ )
    {
      pUserData = pUserData_;
      return *this;
    }

    operator VkDebugUtilsMessengerCreateInfoEXT const&() const
    {
      return *reinterpret_cast<const VkDebugUtilsMessengerCreateInfoEXT*>( this );
    }

    operator VkDebugUtilsMessengerCreateInfoEXT &()
    {
      return *reinterpret_cast<VkDebugUtilsMessengerCreateInfoEXT*>( this );
    }

    bool operator==( DebugUtilsMessengerCreateInfoEXT const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( flags == rhs.flags )
          && ( messageSeverity == rhs.messageSeverity )
          && ( messageType == rhs.messageType )
          && ( pfnUserCallback == rhs.pfnUserCallback )
          && ( pUserData == rhs.pUserData );
    }

    bool operator!=( DebugUtilsMessengerCreateInfoEXT const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eDebugUtilsMessengerCreateInfoEXT;

  public:
    const void* pNext = nullptr;
    DebugUtilsMessengerCreateFlagsEXT flags;
    DebugUtilsMessageSeverityFlagsEXT messageSeverity;
    DebugUtilsMessageTypeFlagsEXT messageType;
    PFN_vkDebugUtilsMessengerCallbackEXT pfnUserCallback;
    void* pUserData;
  };
  static_assert( sizeof( DebugUtilsMessengerCreateInfoEXT ) == sizeof( VkDebugUtilsMessengerCreateInfoEXT ), "struct and wrapper have different size!" );

  struct DebugUtilsObjectTagInfoEXT
  {
    DebugUtilsObjectTagInfoEXT( ObjectType objectType_ = ObjectType::eUnknown,
                                uint64_t objectHandle_ = 0,
                                uint64_t tagName_ = 0,
                                size_t tagSize_ = 0,
                                const void* pTag_ = nullptr )
      : objectType( objectType_ )
      , objectHandle( objectHandle_ )
      , tagName( tagName_ )
      , tagSize( tagSize_ )
      , pTag( pTag_ )
    {}

    DebugUtilsObjectTagInfoEXT( VkDebugUtilsObjectTagInfoEXT const & rhs )
    {
      memcpy( this, &rhs, sizeof( DebugUtilsObjectTagInfoEXT ) );
    }

    DebugUtilsObjectTagInfoEXT& operator=( VkDebugUtilsObjectTagInfoEXT const & rhs )
    {
      memcpy( this, &rhs, sizeof( DebugUtilsObjectTagInfoEXT ) );
      return *this;
    }

    DebugUtilsObjectTagInfoEXT & setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    DebugUtilsObjectTagInfoEXT & setObjectType( ObjectType objectType_ )
    {
      objectType = objectType_;
      return *this;
    }

    DebugUtilsObjectTagInfoEXT & setObjectHandle( uint64_t objectHandle_ )
    {
      objectHandle = objectHandle_;
      return *this;
    }

    DebugUtilsObjectTagInfoEXT & setTagName( uint64_t tagName_ )
    {
      tagName = tagName_;
      return *this;
    }

    DebugUtilsObjectTagInfoEXT & setTagSize( size_t tagSize_ )
    {
      tagSize = tagSize_;
      return *this;
    }

    DebugUtilsObjectTagInfoEXT & setPTag( const void* pTag_ )
    {
      pTag = pTag_;
      return *this;
    }

    operator VkDebugUtilsObjectTagInfoEXT const&() const
    {
      return *reinterpret_cast<const VkDebugUtilsObjectTagInfoEXT*>( this );
    }

    operator VkDebugUtilsObjectTagInfoEXT &()
    {
      return *reinterpret_cast<VkDebugUtilsObjectTagInfoEXT*>( this );
    }

    bool operator==( DebugUtilsObjectTagInfoEXT const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( objectType == rhs.objectType )
          && ( objectHandle == rhs.objectHandle )
          && ( tagName == rhs.tagName )
          && ( tagSize == rhs.tagSize )
          && ( pTag == rhs.pTag );
    }

    bool operator!=( DebugUtilsObjectTagInfoEXT const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eDebugUtilsObjectTagInfoEXT;

  public:
    const void* pNext = nullptr;
    ObjectType objectType;
    uint64_t objectHandle;
    uint64_t tagName;
    size_t tagSize;
    const void* pTag;
  };
  static_assert( sizeof( DebugUtilsObjectTagInfoEXT ) == sizeof( VkDebugUtilsObjectTagInfoEXT ), "struct and wrapper have different size!" );

  struct DedicatedAllocationBufferCreateInfoNV
  {
    DedicatedAllocationBufferCreateInfoNV( Bool32 dedicatedAllocation_ = 0 )
      : dedicatedAllocation( dedicatedAllocation_ )
    {}

    DedicatedAllocationBufferCreateInfoNV( VkDedicatedAllocationBufferCreateInfoNV const & rhs )
    {
      memcpy( this, &rhs, sizeof( DedicatedAllocationBufferCreateInfoNV ) );
    }

    DedicatedAllocationBufferCreateInfoNV& operator=( VkDedicatedAllocationBufferCreateInfoNV const & rhs )
    {
      memcpy( this, &rhs, sizeof( DedicatedAllocationBufferCreateInfoNV ) );
      return *this;
    }

    DedicatedAllocationBufferCreateInfoNV & setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    DedicatedAllocationBufferCreateInfoNV & setDedicatedAllocation( Bool32 dedicatedAllocation_ )
    {
      dedicatedAllocation = dedicatedAllocation_;
      return *this;
    }

    operator VkDedicatedAllocationBufferCreateInfoNV const&() const
    {
      return *reinterpret_cast<const VkDedicatedAllocationBufferCreateInfoNV*>( this );
    }

    operator VkDedicatedAllocationBufferCreateInfoNV &()
    {
      return *reinterpret_cast<VkDedicatedAllocationBufferCreateInfoNV*>( this );
    }

    bool operator==( DedicatedAllocationBufferCreateInfoNV const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( dedicatedAllocation == rhs.dedicatedAllocation );
    }

    bool operator!=( DedicatedAllocationBufferCreateInfoNV const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eDedicatedAllocationBufferCreateInfoNV;

  public:
    const void* pNext = nullptr;
    Bool32 dedicatedAllocation;
  };
  static_assert( sizeof( DedicatedAllocationBufferCreateInfoNV ) == sizeof( VkDedicatedAllocationBufferCreateInfoNV ), "struct and wrapper have different size!" );

  struct DedicatedAllocationImageCreateInfoNV
  {
    DedicatedAllocationImageCreateInfoNV( Bool32 dedicatedAllocation_ = 0 )
      : dedicatedAllocation( dedicatedAllocation_ )
    {}

    DedicatedAllocationImageCreateInfoNV( VkDedicatedAllocationImageCreateInfoNV const & rhs )
    {
      memcpy( this, &rhs, sizeof( DedicatedAllocationImageCreateInfoNV ) );
    }

    DedicatedAllocationImageCreateInfoNV& operator=( VkDedicatedAllocationImageCreateInfoNV const & rhs )
    {
      memcpy( this, &rhs, sizeof( DedicatedAllocationImageCreateInfoNV ) );
      return *this;
    }

    DedicatedAllocationImageCreateInfoNV & setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    DedicatedAllocationImageCreateInfoNV & setDedicatedAllocation( Bool32 dedicatedAllocation_ )
    {
      dedicatedAllocation = dedicatedAllocation_;
      return *this;
    }

    operator VkDedicatedAllocationImageCreateInfoNV const&() const
    {
      return *reinterpret_cast<const VkDedicatedAllocationImageCreateInfoNV*>( this );
    }

    operator VkDedicatedAllocationImageCreateInfoNV &()
    {
      return *reinterpret_cast<VkDedicatedAllocationImageCreateInfoNV*>( this );
    }

    bool operator==( DedicatedAllocationImageCreateInfoNV const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( dedicatedAllocation == rhs.dedicatedAllocation );
    }

    bool operator!=( DedicatedAllocationImageCreateInfoNV const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eDedicatedAllocationImageCreateInfoNV;

  public:
    const void* pNext = nullptr;
    Bool32 dedicatedAllocation;
  };
  static_assert( sizeof( DedicatedAllocationImageCreateInfoNV ) == sizeof( VkDedicatedAllocationImageCreateInfoNV ), "struct and wrapper have different size!" );

  struct DedicatedAllocationMemoryAllocateInfoNV
  {
    DedicatedAllocationMemoryAllocateInfoNV( Image image_ = Image(),
                                             Buffer buffer_ = Buffer() )
      : image( image_ )
      , buffer( buffer_ )
    {}

    DedicatedAllocationMemoryAllocateInfoNV( VkDedicatedAllocationMemoryAllocateInfoNV const & rhs )
    {
      memcpy( this, &rhs, sizeof( DedicatedAllocationMemoryAllocateInfoNV ) );
    }

    DedicatedAllocationMemoryAllocateInfoNV& operator=( VkDedicatedAllocationMemoryAllocateInfoNV const & rhs )
    {
      memcpy( this, &rhs, sizeof( DedicatedAllocationMemoryAllocateInfoNV ) );
      return *this;
    }

    DedicatedAllocationMemoryAllocateInfoNV & setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    DedicatedAllocationMemoryAllocateInfoNV & setImage( Image image_ )
    {
      image = image_;
      return *this;
    }

    DedicatedAllocationMemoryAllocateInfoNV & setBuffer( Buffer buffer_ )
    {
      buffer = buffer_;
      return *this;
    }

    operator VkDedicatedAllocationMemoryAllocateInfoNV const&() const
    {
      return *reinterpret_cast<const VkDedicatedAllocationMemoryAllocateInfoNV*>( this );
    }

    operator VkDedicatedAllocationMemoryAllocateInfoNV &()
    {
      return *reinterpret_cast<VkDedicatedAllocationMemoryAllocateInfoNV*>( this );
    }

    bool operator==( DedicatedAllocationMemoryAllocateInfoNV const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( image == rhs.image )
          && ( buffer == rhs.buffer );
    }

    bool operator!=( DedicatedAllocationMemoryAllocateInfoNV const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eDedicatedAllocationMemoryAllocateInfoNV;

  public:
    const void* pNext = nullptr;
    Image image;
    Buffer buffer;
  };
  static_assert( sizeof( DedicatedAllocationMemoryAllocateInfoNV ) == sizeof( VkDedicatedAllocationMemoryAllocateInfoNV ), "struct and wrapper have different size!" );

  struct DescriptorBufferInfo
  {
    DescriptorBufferInfo( Buffer buffer_ = Buffer(),
                          DeviceSize offset_ = 0,
                          DeviceSize range_ = 0 )
      : buffer( buffer_ )
      , offset( offset_ )
      , range( range_ )
    {}

    DescriptorBufferInfo( VkDescriptorBufferInfo const & rhs )
    {
      memcpy( this, &rhs, sizeof( DescriptorBufferInfo ) );
    }

    DescriptorBufferInfo& operator=( VkDescriptorBufferInfo const & rhs )
    {
      memcpy( this, &rhs, sizeof( DescriptorBufferInfo ) );
      return *this;
    }

    DescriptorBufferInfo & setBuffer( Buffer buffer_ )
    {
      buffer = buffer_;
      return *this;
    }

    DescriptorBufferInfo & setOffset( DeviceSize offset_ )
    {
      offset = offset_;
      return *this;
    }

    DescriptorBufferInfo & setRange( DeviceSize range_ )
    {
      range = range_;
      return *this;
    }

    operator VkDescriptorBufferInfo const&() const
    {
      return *reinterpret_cast<const VkDescriptorBufferInfo*>( this );
    }

    operator VkDescriptorBufferInfo &()
    {
      return *reinterpret_cast<VkDescriptorBufferInfo*>( this );
    }

    bool operator==( DescriptorBufferInfo const& rhs ) const
    {
      return ( buffer == rhs.buffer )
          && ( offset == rhs.offset )
          && ( range == rhs.range );
    }

    bool operator!=( DescriptorBufferInfo const& rhs ) const
    {
      return !operator==( rhs );
    }

    Buffer buffer;
    DeviceSize offset;
    DeviceSize range;
  };
  static_assert( sizeof( DescriptorBufferInfo ) == sizeof( VkDescriptorBufferInfo ), "struct and wrapper have different size!" );

  struct DescriptorImageInfo
  {
    DescriptorImageInfo( Sampler sampler_ = Sampler(),
                         ImageView imageView_ = ImageView(),
                         ImageLayout imageLayout_ = ImageLayout::eUndefined )
      : sampler( sampler_ )
      , imageView( imageView_ )
      , imageLayout( imageLayout_ )
    {}

    DescriptorImageInfo( VkDescriptorImageInfo const & rhs )
    {
      memcpy( this, &rhs, sizeof( DescriptorImageInfo ) );
    }

    DescriptorImageInfo& operator=( VkDescriptorImageInfo const & rhs )
    {
      memcpy( this, &rhs, sizeof( DescriptorImageInfo ) );
      return *this;
    }

    DescriptorImageInfo & setSampler( Sampler sampler_ )
    {
      sampler = sampler_;
      return *this;
    }

    DescriptorImageInfo & setImageView( ImageView imageView_ )
    {
      imageView = imageView_;
      return *this;
    }

    DescriptorImageInfo & setImageLayout( ImageLayout imageLayout_ )
    {
      imageLayout = imageLayout_;
      return *this;
    }

    operator VkDescriptorImageInfo const&() const
    {
      return *reinterpret_cast<const VkDescriptorImageInfo*>( this );
    }

    operator VkDescriptorImageInfo &()
    {
      return *reinterpret_cast<VkDescriptorImageInfo*>( this );
    }

    bool operator==( DescriptorImageInfo const& rhs ) const
    {
      return ( sampler == rhs.sampler )
          && ( imageView == rhs.imageView )
          && ( imageLayout == rhs.imageLayout );
    }

    bool operator!=( DescriptorImageInfo const& rhs ) const
    {
      return !operator==( rhs );
    }

    Sampler sampler;
    ImageView imageView;
    ImageLayout imageLayout;
  };
  static_assert( sizeof( DescriptorImageInfo ) == sizeof( VkDescriptorImageInfo ), "struct and wrapper have different size!" );

  struct DescriptorPoolSize
  {
    DescriptorPoolSize( DescriptorType type_ = DescriptorType::eSampler,
                        uint32_t descriptorCount_ = 0 )
      : type( type_ )
      , descriptorCount( descriptorCount_ )
    {}

    DescriptorPoolSize( VkDescriptorPoolSize const & rhs )
    {
      memcpy( this, &rhs, sizeof( DescriptorPoolSize ) );
    }

    DescriptorPoolSize& operator=( VkDescriptorPoolSize const & rhs )
    {
      memcpy( this, &rhs, sizeof( DescriptorPoolSize ) );
      return *this;
    }

    DescriptorPoolSize & setType( DescriptorType type_ )
    {
      type = type_;
      return *this;
    }

    DescriptorPoolSize & setDescriptorCount( uint32_t descriptorCount_ )
    {
      descriptorCount = descriptorCount_;
      return *this;
    }

    operator VkDescriptorPoolSize const&() const
    {
      return *reinterpret_cast<const VkDescriptorPoolSize*>( this );
    }

    operator VkDescriptorPoolSize &()
    {
      return *reinterpret_cast<VkDescriptorPoolSize*>( this );
    }

    bool operator==( DescriptorPoolSize const& rhs ) const
    {
      return ( type == rhs.type )
          && ( descriptorCount == rhs.descriptorCount );
    }

    bool operator!=( DescriptorPoolSize const& rhs ) const
    {
      return !operator==( rhs );
    }

    DescriptorType type;
    uint32_t descriptorCount;
  };
  static_assert( sizeof( DescriptorPoolSize ) == sizeof( VkDescriptorPoolSize ), "struct and wrapper have different size!" );

  struct DescriptorPoolCreateInfo
  {
    DescriptorPoolCreateInfo( DescriptorPoolCreateFlags flags_ = DescriptorPoolCreateFlags(),
                              uint32_t maxSets_ = 0,
                              uint32_t poolSizeCount_ = 0,
                              const DescriptorPoolSize* pPoolSizes_ = nullptr )
      : flags( flags_ )
      , maxSets( maxSets_ )
      , poolSizeCount( poolSizeCount_ )
      , pPoolSizes( pPoolSizes_ )
    {}

    DescriptorPoolCreateInfo( VkDescriptorPoolCreateInfo const & rhs )
    {
      memcpy( this, &rhs, sizeof( DescriptorPoolCreateInfo ) );
    }

    DescriptorPoolCreateInfo& operator=( VkDescriptorPoolCreateInfo const & rhs )
    {
      memcpy( this, &rhs, sizeof( DescriptorPoolCreateInfo ) );
      return *this;
    }

    DescriptorPoolCreateInfo & setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    DescriptorPoolCreateInfo & setFlags( DescriptorPoolCreateFlags flags_ )
    {
      flags = flags_;
      return *this;
    }

    DescriptorPoolCreateInfo & setMaxSets( uint32_t maxSets_ )
    {
      maxSets = maxSets_;
      return *this;
    }

    DescriptorPoolCreateInfo & setPoolSizeCount( uint32_t poolSizeCount_ )
    {
      poolSizeCount = poolSizeCount_;
      return *this;
    }

    DescriptorPoolCreateInfo & setPPoolSizes( const DescriptorPoolSize* pPoolSizes_ )
    {
      pPoolSizes = pPoolSizes_;
      return *this;
    }

    operator VkDescriptorPoolCreateInfo const&() const
    {
      return *reinterpret_cast<const VkDescriptorPoolCreateInfo*>( this );
    }

    operator VkDescriptorPoolCreateInfo &()
    {
      return *reinterpret_cast<VkDescriptorPoolCreateInfo*>( this );
    }

    bool operator==( DescriptorPoolCreateInfo const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( flags == rhs.flags )
          && ( maxSets == rhs.maxSets )
          && ( poolSizeCount == rhs.poolSizeCount )
          && ( pPoolSizes == rhs.pPoolSizes );
    }

    bool operator!=( DescriptorPoolCreateInfo const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eDescriptorPoolCreateInfo;

  public:
    const void* pNext = nullptr;
    DescriptorPoolCreateFlags flags;
    uint32_t maxSets;
    uint32_t poolSizeCount;
    const DescriptorPoolSize* pPoolSizes;
  };
  static_assert( sizeof( DescriptorPoolCreateInfo ) == sizeof( VkDescriptorPoolCreateInfo ), "struct and wrapper have different size!" );

  struct DescriptorPoolInlineUniformBlockCreateInfoEXT
  {
    DescriptorPoolInlineUniformBlockCreateInfoEXT( uint32_t maxInlineUniformBlockBindings_ = 0 )
      : maxInlineUniformBlockBindings( maxInlineUniformBlockBindings_ )
    {}

    DescriptorPoolInlineUniformBlockCreateInfoEXT( VkDescriptorPoolInlineUniformBlockCreateInfoEXT const & rhs )
    {
      memcpy( this, &rhs, sizeof( DescriptorPoolInlineUniformBlockCreateInfoEXT ) );
    }

    DescriptorPoolInlineUniformBlockCreateInfoEXT& operator=( VkDescriptorPoolInlineUniformBlockCreateInfoEXT const & rhs )
    {
      memcpy( this, &rhs, sizeof( DescriptorPoolInlineUniformBlockCreateInfoEXT ) );
      return *this;
    }

    DescriptorPoolInlineUniformBlockCreateInfoEXT & setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    DescriptorPoolInlineUniformBlockCreateInfoEXT & setMaxInlineUniformBlockBindings( uint32_t maxInlineUniformBlockBindings_ )
    {
      maxInlineUniformBlockBindings = maxInlineUniformBlockBindings_;
      return *this;
    }

    operator VkDescriptorPoolInlineUniformBlockCreateInfoEXT const&() const
    {
      return *reinterpret_cast<const VkDescriptorPoolInlineUniformBlockCreateInfoEXT*>( this );
    }

    operator VkDescriptorPoolInlineUniformBlockCreateInfoEXT &()
    {
      return *reinterpret_cast<VkDescriptorPoolInlineUniformBlockCreateInfoEXT*>( this );
    }

    bool operator==( DescriptorPoolInlineUniformBlockCreateInfoEXT const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( maxInlineUniformBlockBindings == rhs.maxInlineUniformBlockBindings );
    }

    bool operator!=( DescriptorPoolInlineUniformBlockCreateInfoEXT const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eDescriptorPoolInlineUniformBlockCreateInfoEXT;

  public:
    const void* pNext = nullptr;
    uint32_t maxInlineUniformBlockBindings;
  };
  static_assert( sizeof( DescriptorPoolInlineUniformBlockCreateInfoEXT ) == sizeof( VkDescriptorPoolInlineUniformBlockCreateInfoEXT ), "struct and wrapper have different size!" );

  struct DescriptorSetAllocateInfo
  {
    DescriptorSetAllocateInfo( DescriptorPool descriptorPool_ = DescriptorPool(),
                               uint32_t descriptorSetCount_ = 0,
                               const DescriptorSetLayout* pSetLayouts_ = nullptr )
      : descriptorPool( descriptorPool_ )
      , descriptorSetCount( descriptorSetCount_ )
      , pSetLayouts( pSetLayouts_ )
    {}

    DescriptorSetAllocateInfo( VkDescriptorSetAllocateInfo const & rhs )
    {
      memcpy( this, &rhs, sizeof( DescriptorSetAllocateInfo ) );
    }

    DescriptorSetAllocateInfo& operator=( VkDescriptorSetAllocateInfo const & rhs )
    {
      memcpy( this, &rhs, sizeof( DescriptorSetAllocateInfo ) );
      return *this;
    }

    DescriptorSetAllocateInfo & setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    DescriptorSetAllocateInfo & setDescriptorPool( DescriptorPool descriptorPool_ )
    {
      descriptorPool = descriptorPool_;
      return *this;
    }

    DescriptorSetAllocateInfo & setDescriptorSetCount( uint32_t descriptorSetCount_ )
    {
      descriptorSetCount = descriptorSetCount_;
      return *this;
    }

    DescriptorSetAllocateInfo & setPSetLayouts( const DescriptorSetLayout* pSetLayouts_ )
    {
      pSetLayouts = pSetLayouts_;
      return *this;
    }

    operator VkDescriptorSetAllocateInfo const&() const
    {
      return *reinterpret_cast<const VkDescriptorSetAllocateInfo*>( this );
    }

    operator VkDescriptorSetAllocateInfo &()
    {
      return *reinterpret_cast<VkDescriptorSetAllocateInfo*>( this );
    }

    bool operator==( DescriptorSetAllocateInfo const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( descriptorPool == rhs.descriptorPool )
          && ( descriptorSetCount == rhs.descriptorSetCount )
          && ( pSetLayouts == rhs.pSetLayouts );
    }

    bool operator!=( DescriptorSetAllocateInfo const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eDescriptorSetAllocateInfo;

  public:
    const void* pNext = nullptr;
    DescriptorPool descriptorPool;
    uint32_t descriptorSetCount;
    const DescriptorSetLayout* pSetLayouts;
  };
  static_assert( sizeof( DescriptorSetAllocateInfo ) == sizeof( VkDescriptorSetAllocateInfo ), "struct and wrapper have different size!" );

  struct DescriptorSetLayoutBinding
  {
    DescriptorSetLayoutBinding( uint32_t binding_ = 0,
                                DescriptorType descriptorType_ = DescriptorType::eSampler,
                                uint32_t descriptorCount_ = 0,
                                ShaderStageFlags stageFlags_ = ShaderStageFlags(),
                                const Sampler* pImmutableSamplers_ = nullptr )
      : binding( binding_ )
      , descriptorType( descriptorType_ )
      , descriptorCount( descriptorCount_ )
      , stageFlags( stageFlags_ )
      , pImmutableSamplers( pImmutableSamplers_ )
    {}

    DescriptorSetLayoutBinding( VkDescriptorSetLayoutBinding const & rhs )
    {
      memcpy( this, &rhs, sizeof( DescriptorSetLayoutBinding ) );
    }

    DescriptorSetLayoutBinding& operator=( VkDescriptorSetLayoutBinding const & rhs )
    {
      memcpy( this, &rhs, sizeof( DescriptorSetLayoutBinding ) );
      return *this;
    }

    DescriptorSetLayoutBinding & setBinding( uint32_t binding_ )
    {
      binding = binding_;
      return *this;
    }

    DescriptorSetLayoutBinding & setDescriptorType( DescriptorType descriptorType_ )
    {
      descriptorType = descriptorType_;
      return *this;
    }

    DescriptorSetLayoutBinding & setDescriptorCount( uint32_t descriptorCount_ )
    {
      descriptorCount = descriptorCount_;
      return *this;
    }

    DescriptorSetLayoutBinding & setStageFlags( ShaderStageFlags stageFlags_ )
    {
      stageFlags = stageFlags_;
      return *this;
    }

    DescriptorSetLayoutBinding & setPImmutableSamplers( const Sampler* pImmutableSamplers_ )
    {
      pImmutableSamplers = pImmutableSamplers_;
      return *this;
    }

    operator VkDescriptorSetLayoutBinding const&() const
    {
      return *reinterpret_cast<const VkDescriptorSetLayoutBinding*>( this );
    }

    operator VkDescriptorSetLayoutBinding &()
    {
      return *reinterpret_cast<VkDescriptorSetLayoutBinding*>( this );
    }

    bool operator==( DescriptorSetLayoutBinding const& rhs ) const
    {
      return ( binding == rhs.binding )
          && ( descriptorType == rhs.descriptorType )
          && ( descriptorCount == rhs.descriptorCount )
          && ( stageFlags == rhs.stageFlags )
          && ( pImmutableSamplers == rhs.pImmutableSamplers );
    }

    bool operator!=( DescriptorSetLayoutBinding const& rhs ) const
    {
      return !operator==( rhs );
    }

    uint32_t binding;
    DescriptorType descriptorType;
    uint32_t descriptorCount;
    ShaderStageFlags stageFlags;
    const Sampler* pImmutableSamplers;
  };
  static_assert( sizeof( DescriptorSetLayoutBinding ) == sizeof( VkDescriptorSetLayoutBinding ), "struct and wrapper have different size!" );

  struct DescriptorSetLayoutBindingFlagsCreateInfoEXT
  {
    DescriptorSetLayoutBindingFlagsCreateInfoEXT( uint32_t bindingCount_ = 0,
                                                  const DescriptorBindingFlagsEXT* pBindingFlags_ = nullptr )
      : bindingCount( bindingCount_ )
      , pBindingFlags( pBindingFlags_ )
    {}

    DescriptorSetLayoutBindingFlagsCreateInfoEXT( VkDescriptorSetLayoutBindingFlagsCreateInfoEXT const & rhs )
    {
      memcpy( this, &rhs, sizeof( DescriptorSetLayoutBindingFlagsCreateInfoEXT ) );
    }

    DescriptorSetLayoutBindingFlagsCreateInfoEXT& operator=( VkDescriptorSetLayoutBindingFlagsCreateInfoEXT const & rhs )
    {
      memcpy( this, &rhs, sizeof( DescriptorSetLayoutBindingFlagsCreateInfoEXT ) );
      return *this;
    }

    DescriptorSetLayoutBindingFlagsCreateInfoEXT & setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    DescriptorSetLayoutBindingFlagsCreateInfoEXT & setBindingCount( uint32_t bindingCount_ )
    {
      bindingCount = bindingCount_;
      return *this;
    }

    DescriptorSetLayoutBindingFlagsCreateInfoEXT & setPBindingFlags( const DescriptorBindingFlagsEXT* pBindingFlags_ )
    {
      pBindingFlags = pBindingFlags_;
      return *this;
    }

    operator VkDescriptorSetLayoutBindingFlagsCreateInfoEXT const&() const
    {
      return *reinterpret_cast<const VkDescriptorSetLayoutBindingFlagsCreateInfoEXT*>( this );
    }

    operator VkDescriptorSetLayoutBindingFlagsCreateInfoEXT &()
    {
      return *reinterpret_cast<VkDescriptorSetLayoutBindingFlagsCreateInfoEXT*>( this );
    }

    bool operator==( DescriptorSetLayoutBindingFlagsCreateInfoEXT const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( bindingCount == rhs.bindingCount )
          && ( pBindingFlags == rhs.pBindingFlags );
    }

    bool operator!=( DescriptorSetLayoutBindingFlagsCreateInfoEXT const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eDescriptorSetLayoutBindingFlagsCreateInfoEXT;

  public:
    const void* pNext = nullptr;
    uint32_t bindingCount;
    const DescriptorBindingFlagsEXT* pBindingFlags;
  };
  static_assert( sizeof( DescriptorSetLayoutBindingFlagsCreateInfoEXT ) == sizeof( VkDescriptorSetLayoutBindingFlagsCreateInfoEXT ), "struct and wrapper have different size!" );

  struct DescriptorSetLayoutCreateInfo
  {
    DescriptorSetLayoutCreateInfo( DescriptorSetLayoutCreateFlags flags_ = DescriptorSetLayoutCreateFlags(),
                                   uint32_t bindingCount_ = 0,
                                   const DescriptorSetLayoutBinding* pBindings_ = nullptr )
      : flags( flags_ )
      , bindingCount( bindingCount_ )
      , pBindings( pBindings_ )
    {}

    DescriptorSetLayoutCreateInfo( VkDescriptorSetLayoutCreateInfo const & rhs )
    {
      memcpy( this, &rhs, sizeof( DescriptorSetLayoutCreateInfo ) );
    }

    DescriptorSetLayoutCreateInfo& operator=( VkDescriptorSetLayoutCreateInfo const & rhs )
    {
      memcpy( this, &rhs, sizeof( DescriptorSetLayoutCreateInfo ) );
      return *this;
    }

    DescriptorSetLayoutCreateInfo & setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    DescriptorSetLayoutCreateInfo & setFlags( DescriptorSetLayoutCreateFlags flags_ )
    {
      flags = flags_;
      return *this;
    }

    DescriptorSetLayoutCreateInfo & setBindingCount( uint32_t bindingCount_ )
    {
      bindingCount = bindingCount_;
      return *this;
    }

    DescriptorSetLayoutCreateInfo & setPBindings( const DescriptorSetLayoutBinding* pBindings_ )
    {
      pBindings = pBindings_;
      return *this;
    }

    operator VkDescriptorSetLayoutCreateInfo const&() const
    {
      return *reinterpret_cast<const VkDescriptorSetLayoutCreateInfo*>( this );
    }

    operator VkDescriptorSetLayoutCreateInfo &()
    {
      return *reinterpret_cast<VkDescriptorSetLayoutCreateInfo*>( this );
    }

    bool operator==( DescriptorSetLayoutCreateInfo const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( flags == rhs.flags )
          && ( bindingCount == rhs.bindingCount )
          && ( pBindings == rhs.pBindings );
    }

    bool operator!=( DescriptorSetLayoutCreateInfo const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eDescriptorSetLayoutCreateInfo;

  public:
    const void* pNext = nullptr;
    DescriptorSetLayoutCreateFlags flags;
    uint32_t bindingCount;
    const DescriptorSetLayoutBinding* pBindings;
  };
  static_assert( sizeof( DescriptorSetLayoutCreateInfo ) == sizeof( VkDescriptorSetLayoutCreateInfo ), "struct and wrapper have different size!" );

  struct DescriptorSetLayoutSupport
  {
    operator VkDescriptorSetLayoutSupport const&() const
    {
      return *reinterpret_cast<const VkDescriptorSetLayoutSupport*>( this );
    }

    operator VkDescriptorSetLayoutSupport &()
    {
      return *reinterpret_cast<VkDescriptorSetLayoutSupport*>( this );
    }

    bool operator==( DescriptorSetLayoutSupport const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( supported == rhs.supported );
    }

    bool operator!=( DescriptorSetLayoutSupport const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eDescriptorSetLayoutSupport;

  public:
    void* pNext = nullptr;
    Bool32 supported;
  };
  static_assert( sizeof( DescriptorSetLayoutSupport ) == sizeof( VkDescriptorSetLayoutSupport ), "struct and wrapper have different size!" );

  struct DescriptorSetVariableDescriptorCountAllocateInfoEXT
  {
    DescriptorSetVariableDescriptorCountAllocateInfoEXT( uint32_t descriptorSetCount_ = 0,
                                                         const uint32_t* pDescriptorCounts_ = nullptr )
      : descriptorSetCount( descriptorSetCount_ )
      , pDescriptorCounts( pDescriptorCounts_ )
    {}

    DescriptorSetVariableDescriptorCountAllocateInfoEXT( VkDescriptorSetVariableDescriptorCountAllocateInfoEXT const & rhs )
    {
      memcpy( this, &rhs, sizeof( DescriptorSetVariableDescriptorCountAllocateInfoEXT ) );
    }

    DescriptorSetVariableDescriptorCountAllocateInfoEXT& operator=( VkDescriptorSetVariableDescriptorCountAllocateInfoEXT const & rhs )
    {
      memcpy( this, &rhs, sizeof( DescriptorSetVariableDescriptorCountAllocateInfoEXT ) );
      return *this;
    }

    DescriptorSetVariableDescriptorCountAllocateInfoEXT & setPNext( const void* pNext_ )
    {
      pNext = pNext_;
      return *this;
    }

    DescriptorSetVariableDescriptorCountAllocateInfoEXT & setDescriptorSetCount( uint32_t descriptorSetCount_ )
    {
      descriptorSetCount = descriptorSetCount_;
      return *this;
    }

    DescriptorSetVariableDescriptorCountAllocateInfoEXT & setPDescriptorCounts( const uint32_t* pDescriptorCounts_ )
    {
      pDescriptorCounts = pDescriptorCounts_;
      return *this;
    }

    operator VkDescriptorSetVariableDescriptorCountAllocateInfoEXT const&() const
    {
      return *reinterpret_cast<const VkDescriptorSetVariableDescriptorCountAllocateInfoEXT*>( this );
    }

    operator VkDescriptorSetVariableDescriptorCountAllocateInfoEXT &()
    {
      return *reinterpret_cast<VkDescriptorSetVariableDescriptorCountAllocateInfoEXT*>( this );
    }

    bool operator==( DescriptorSetVariableDescriptorCountAllocateInfoEXT const& rhs ) const
    {
      return ( sType == rhs.sType )
          && ( pNext == rhs.pNext )
          && ( descriptorSetCount == rhs.descriptorSetCount )
          && ( pDescriptorCounts == rhs.pDescriptorCounts );
    }

    bool operator!=( DescriptorSetVariableDescriptorCountAllocateInfoEXT const& rhs ) const
    {
      return !operator==( rhs );
    }

  private:
    StructureType sType = StructureType::eDescriptorSetVariableDescriptorCountAllocateInfoEXT;

  public:
    const void* pNext = nullptr;
    uint32_t descriptorSetCount;
    const uint32_t* pDescriptorCounts;
  };
  static_assert( sizeof( DescriptorSetVariableDescriptorCountAllocateInfoEXT ) == sizeof( VkDescriptorSetVariableDescriptorCountAllocateInfoEXT ), "struct and wrapper have different size!" );

  struct DescriptorSetVariableDescriptorCountLayoutSupportEXT
  {
    operator VkDescriptorSetVariableDescriptorCountLayoutSupportEXT const&() const
    {
      return *reinterpret_cast<const VkDescriptorSetVariableDescriptorCountLayoutSupportEXT*>( this );
  