/*******************************************************************************
 * Copyright (c) 2000, 2003 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Common Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/cpl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.jdt.core.util;

/**
 * Description of a .class file. This class reifies the internal structure of a .class
 * file following the JVM specifications.
 *  
 * This interface may be implemented by clients. 
 * 
 * @since 2.0
 */
public interface IClassFileReader {
	/**
	 * This value should be used to read completely each part of a .class file.
	 */
	int ALL 					= 0xFFFF;
	
	/**
	 * This value should be used to read only the constant pool entries of a .class file.
	 */
	int CONSTANT_POOL 			= 0x0001;

	/**
	 * This value should be used to read the constant pool entries and 
	 * the method infos of a .class file.
	 */
	int METHOD_INFOS 			= 0x0002 + CONSTANT_POOL;

	/**
	 * This value should be used to read the constant pool entries and 
	 * the field infos of a .class file.
	 */
	int FIELD_INFOS 			= 0x0004 + CONSTANT_POOL;

	/**
	 * This value should be used to read the constant pool entries and 
	 * the super interface names of a .class file.
	 */
	int SUPER_INTERFACES 		= 0x0008 + CONSTANT_POOL;

	/**
	 * This value should be used to read the constant pool entries and 
	 * the attributes of a .class file.
	 */
	int CLASSFILE_ATTRIBUTES 	= 0x0010 + CONSTANT_POOL;

	/**
	 * This value should be used to read the method bodies.
	 * It has to be used with METHOD_INFOS.
	 */
	int METHOD_BODIES 			= 0x0020;

	/**
	 * This value should be used to read the whole contents of the .class file except the
	 * method bodies.
	 */
	int ALL_BUT_METHOD_BODIES   = ALL & ~METHOD_BODIES;
	
	/**
	 * Answer back the access flag of the .class file.
	 * 
	 * @return the access flag of the .class file
	 */
	int getAccessFlags();
	
	/**
	 * Answer back the array of field infos of the .class file, 
	 * an empty array if none.
	 * 
	 * @return the array of field infos of the .class file, an empty array if none
	 */
	IFieldInfo[] getFieldInfos();

	/**
	 * Answer back the names of interfaces implemented by this .class file,
	 * an empty array if none. The names are returned as described in the 
	 * JVM specifications.
	 * 
	 * @return the names of interfaces implemented by this .class file, an empty array if none
	 */
	char[][] getInterfaceNames();

	/**
	 * Answer back the indexes in the constant pool of interfaces implemented 
	 * by this .class file, an empty array if none.
	 * 
	 * @return the indexes in the constant pool of interfaces implemented 
	 * by this .class file, an empty array if none
	 */
	int[] getInterfaceIndexes();

	/**
	 * Answer back the inner classes attribute of this .class file, null if none.
	 * 
	 * @return the inner classes attribute of this .class file, null if none
	 */
	IInnerClassesAttribute getInnerClassesAttribute();

	/**
	 * Answer back the array of method infos of this .class file,
	 * an empty array if none.
	 * 
	 * @return the array of method infos of this .class file,
	 * an empty array if none
	 */
	IMethodInfo[] getMethodInfos();

	/**
	 * Answer back the qualified name of the .class file.
	 * The name is returned as described in the JVM specifications.
	 *  
	 * @return the qualified name of the .class file
	 */
	char[] getClassName();

	/**
	 * Answer back the index of the class name in the constant pool 
	 * of the .class file.
	 *  
	 * @return the index of the class name in the constant pool 
	 */
	int getClassIndex();
		
	/**
	 * Answer back the qualified name of the superclass of this .class file.
	 * The name is returned as described in the JVM specifications. Answer null if 
	 * getSuperclassIndex() is zero.
	 * 
	 * @return the qualified name of the superclass of this .class file, null if getSuperclassIndex() is zero
	 */
	char[] getSuperclassName();

	/**
	 * Answer back the index of the superclass name in the constant pool 
	 * of the .class file. Answer 0 if this .class file represents java.lang.Object.
	 *  
	 * @return the index of the superclass name in the constant pool 
	 * of the .class file, 0 if this .class file represents java.lang.Object.
	 */
	int getSuperclassIndex();

	/**
	 * Answer true if this .class file represents an class, false otherwise.
	 * 
	 * @return true if this .class file represents an class, false otherwise
	 */
	boolean isClass();

	/**
	 * Answer true if this .class file represents an interface, false otherwise.
	 * 
	 * @return true if this .class file represents an interface, false otherwise
	 */
	boolean isInterface();

	/**
	 * Answer the source file attribute, if it exists, null otherwise.
	 * 
	 * @return the source file attribute, if it exists, null otherwise
	 */
	ISourceAttribute getSourceFileAttribute();

	/**
	 * Answer the constant pool of this .class file.
	 * 
	 * @return the constant pool of this .class file
	 */
	IConstantPool getConstantPool();
	
	/**
	 * Answer the minor version of this .class file.
	 * 
	 * @return the minor version of this .class file
	 */
	int getMinorVersion();

	/**
	 * Answer the major version of this .class file.
	 * 
	 * @return the major version of this .class file
	 */
	int getMajorVersion();

	/**
	 * Answer back the attribute number of the .class file.
	 * 
	 * @return the attribute number of the .class file
	 */
	int getAttributeCount();
	
	/**
	 * Answer back the collection of all attributes of the field info. It 
	 * includes SyntheticAttribute, ConstantValueAttributes, etc. Answers an empty
	 * array if none.
	 * 
	 * @return the collection of all attributes of the field info. It 
	 * includes SyntheticAttribute, ConstantValueAttributes, etc. Answers an empty
	 * array if none
	 */
	IClassFileAttribute[] getAttributes();
	
	/**
	 * Answer back the magic number.
	 * 
	 * @return the magic number
	 */
	int getMagic();
	
	/**
	 * Answer back the number of field infos.
	 * 
	 * @return the number of field infos
	 */
	int getFieldsCount();

	/**
	 * Answer back the number of method infos.
	 * 
	 * @return the number of method infos
	 */
	int getMethodsCount();
}
