//
// File:        UserOptions.java
// Package:     gov.llnl.babel
// Release:     $Name: release-0-8-8 $
// Revision:     @(#) $Id: UserOptions.java,v 1.30 2003/10/05 17:49:25 norris Exp $
// Description: singleton Map for configuring general, user-related options
// 
// Copyright (c) 2000-2002, The Regents of the University of Calfornia.
// Produced at the Lawrence Livermore National Laboratory.
// Written by the Components Team <components@llnl.gov>
// UCRL-CODE-2002-054
// All rights reserved.
// 
// This file is part of Babel. For more information, see
// http://www.llnl.gov/CASC/components/. Please read the COPYRIGHT file
// for Our Notice and the LICENSE file for the GNU Lesser General Public
// License.
// 
// This program is free software; you can redistribute it and/or modify it
// under the terms of the GNU Lesser General Public License (as published by
// the Free Software Foundation) version 2.1 dated February 1999.
// 
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the IMPLIED WARRANTY OF
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the terms and
// conditions of the GNU Lesser General Public License for more details.
// 
// You should have recieved a copy of the GNU Lesser General Public License
// along with this program; if not, write to the Free Software Foundation,
// Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

package gov.llnl.babel;

import java.lang.InstantiationException;
import java.lang.IllegalAccessException;
import gnu.getopt.Getopt;
import gnu.getopt.LongOpt;
import gov.llnl.babel.symbols.RegexMatch;
import gov.llnl.babel.symbols.RegexUnsupportedException;
import gov.llnl.babel.symbols.BadRegexException;
import gov.llnl.babel.backend.CodeGenerationFactory;


/**
 * <code>UserOptions</code> is a singleton class that manages the
 * wide variety of settings available to a user.  The configuration 
 * options are read in through an array of strings (args).  Newly
 * specified arguments always replace the values of the previously 
 * specified ones with the exception of the repository path where
 * new arguments are appended to the existing list.
 */

/*
 * DIRECTIONS FOR ADDING NEW OPTIONS:
 * 1. Choose a unique long string and/or, if appropriate, a unique
 *    single character) for the command.  
 * 2. For long strings, do the following to d_long_opts (below):
 *     a. add a new LongOpt object to the d_long_opts array static initializer.
 *     b. Add new character command (or some other unique value) to third 
 *        argument of Getopt constructor in method parseCommandlineOptions.
 *        If option requires an argument, postfix a single colon ':'
 *        If option has optional argument, postfix a double colon '::'
 * 3. Update the printUsage method.
 * 4. TEST
 *    Note one way to use this class's main to test is to temporarily 
 *    unpackage this, Version, and BabelConfiguration classes and set 
 *    classpath to this and the directory that contains the java-getopt 
 *    jar file.
 * 5. cvs commit
 *
 * DIRECTIONS FOR ADDING SUPPORT FOR NEW LANGUAGE(S):
 * 1. Update printUsage method.
 * 2. Update the appropriate validClientLanguage (for client), 
 *    validRemoteLanguage (for remote client), and validServerLanguage
 *    (for server).
 * 3. TEST
 * 4. cvs commit
 */

public class UserOptions
{ 
  /**
   * The valid long options.
   */
  private static final LongOpt[] d_long_opts = 
  {
    new LongOpt("client",                LongOpt.REQUIRED_ARGUMENT, null, 'c'),
    new LongOpt("generate-subdirs",      LongOpt.NO_ARGUMENT,       null, 'g'),
    new LongOpt("help",                  LongOpt.NO_ARGUMENT,       null, 'h'),
    new LongOpt("output-directory",      LongOpt.REQUIRED_ARGUMENT, null, 'o'), 
    new LongOpt("vpath",                 LongOpt.REQUIRED_ARGUMENT, null, 'V'), 
    new LongOpt("parse-check",           LongOpt.NO_ARGUMENT,       null, 'p'),
    new LongOpt("remote",                LongOpt.REQUIRED_ARGUMENT, null, 'r'), 
    new LongOpt("repository-path",       LongOpt.REQUIRED_ARGUMENT, null, 'R'), 
    new LongOpt("server",                LongOpt.REQUIRED_ARGUMENT, null, 's'), 
    new LongOpt("text",                  LongOpt.REQUIRED_ARGUMENT, null, 't'), 
    new LongOpt("xml",                   LongOpt.NO_ARGUMENT,       null, 'x'),
    new LongOpt("version",               LongOpt.NO_ARGUMENT,       null, 'v'),
    new LongOpt("exclude",               LongOpt.REQUIRED_ARGUMENT, null, 'e'),
    new LongOpt("generate-sidl-stdlib",  LongOpt.NO_ARGUMENT,       null, 5), 
    new LongOpt("no-default-repository", LongOpt.NO_ARGUMENT,       null, 6), 
    new LongOpt("suppress-timestamp",    LongOpt.NO_ARGUMENT,       null, 7), 
    new LongOpt("comment-local-only",    LongOpt.NO_ARGUMENT,       null, 8),
    new LongOpt("hide-glue",             LongOpt.NO_ARGUMENT,       null, 'u'),
    new LongOpt("exclude-external",      LongOpt.NO_ARGUMENT,       null, 'E'),    
    new LongOpt("language-subdir",       LongOpt.NO_ARGUMENT,       null, 'l'),
  };

  /**
   * The indicator as to whether or not the user specified options that
   * require further processing.
   */
  private boolean d_can_proceed = false;

  /**
   * The handle to singleton instance of this class.
   */
  private static UserOptions s_my_instance = null;
  

  /**
   * Since this is a singleton class, the constructor is protected.
   */
  protected UserOptions() {
  }


  /**
   * Return the singleton instance of the user options.  
   * If the user options instance has not yet been created, 
   * then it will be created by this call.
   */
  public static UserOptions getInstance() {
    if (s_my_instance == null) {
      s_my_instance = new UserOptions();
    }
    return s_my_instance;
  }


  /** 
   * Extract options from the command line.
   *
   * @param args String array of command line arguments.
   *
   * @return index of the first unused argument.
   *         can iterate from this value to 'args.length'.
   */
  public int parseCommandlineOptions( BabelConfiguration theConfig,
                                      String             args[] ) 
  {
    Getopt commandlineOptions = 
      new Getopt("babel", args, "Eghpvxluo:c:r:R:s:t:e:", d_long_opts);

    int numRequired    = 0;
    boolean helpSet    = false;
    boolean versionSet = false;

    int c; 
    while ((c = commandlineOptions.getopt()) != -1) 
    {
      switch (c) {
      case 5 :
        theConfig.setGenerateStdlib(true);
        numRequired = numRequired + 1;
        break;
      case 6:
        /*
         * This option is parsed by the babel.in driver script so isn't
         * really appropriate in this class EXCEPT that we need to make
         * sure it appears in the help file.  Processing should continue
         * even if it does get passed in as an argument.
         */
        System.err.println("Babel: Warning: no-default-repository is a babel driver"
                           + " script option only.");
        break;
      case 7:
        theConfig.setSuppressTimestamps(true);
        break;
      case 8:
        theConfig.setCommentLocalOnly(true);
        break;
      case 'p' :
        theConfig.setParseCheckOnly(true);
        numRequired = numRequired + 1;
        break;
      case 'c' :
        String clang = commandlineOptions.getOptarg().toLowerCase();
        if ( validClientLanguage(clang) ) {
          theConfig.setGenerateClient(true);
          theConfig.setTargetLanguage(clang);
          numRequired = numRequired + 1;
        } else {
          System.err.println("Babel: Error: Unsupported language (" + clang 
                             + ") specified for client option.");
        }
        break;
      case 'r' :
        String rlang = commandlineOptions.getOptarg().toLowerCase();
        if ( validRemoteLanguage(rlang) ) {
          theConfig.setGenerateRemote(true);
          theConfig.setTargetLanguage(rlang);
          //ToDo...Uncomment the following once supported.
          //numRequired = numRequired + 1;
        } else {
          System.err.println("Babel: Error: Unsupported language (" + rlang 
                             + ") specified for remote option.");
        }
        //ToDo...Remove the following once supported.
        System.err.println("\nBabel: *** Sorry. Remote invocation is not yet supported"
                           + ". ***\n");
        break;
      case 's' :
        String slang = commandlineOptions.getOptarg().toLowerCase();
        if ( validServerLanguage(slang) ) {
          theConfig.setGenerateServer(true);
          theConfig.setTargetLanguage(slang);
          numRequired = numRequired + 1;
        } else {
          System.err.println("Babel: Error: Unsupported language (" + slang 
                             + ") specified for server option.");
        }
        break;
      case 't' :
        String tlang = commandlineOptions.getOptarg().toLowerCase();
        if ( validTextRepresentation(tlang) ) {
          theConfig.setGenerateText(true);
          theConfig.setTargetLanguage(tlang);
          numRequired = numRequired + 1;
        } else {
          System.err.println("Babel: Error: Unsupported language representation"
                             + " (" + tlang + ") specified for text option.");
        }
        break;
      case 'x' :		// Deprecated!
        theConfig.setGenerateXML(true);
        numRequired = numRequired + 1;
        break;
      case 'h':
        helpSet     = true;
        numRequired = numRequired + 1;
        break;
      case 'v':
        versionSet  = true;
        numRequired = numRequired + 1;
        break;
      case 'R':
        theConfig.addToRepositoryPath(commandlineOptions.getOptarg());
        break;
      case 'o' :
        theConfig.setOutputDirectory(commandlineOptions.getOptarg());
        break;
      case 'V' :
        theConfig.setVPathDirectory(commandlineOptions.getOptarg());
        break;
      case 'g':
        theConfig.setMakePackageSubdirs(true);
        break;
      case 'e':
        try {
          theConfig.addExcluded
            (new RegexMatch(commandlineOptions.getOptarg()));
        }
        catch(RegexUnsupportedException rue) {
          System.err.println("Babel: The --exclude=<regex> | -e <regex> feature requires a Java runtime environment with the");
          System.err.println("1.4 library or higher for regular expression support.");
          System.err.println("You must use a more recent version of Java to use this feature.");
        }
        catch(BadRegexException bre) {
          System.err.println("Babel: " + bre.getMessage());
        }
        break;
      case 'l': 
        theConfig.setMakeLanguageSubdir(true);
        break;
      case 'u':
        theConfig.setMakeGlueSubdirs(true);
        break;
      case 'E':
        theConfig.setExcludeExternal(true);
				break;
      case ':':
        // the option requires an argument
        break;
      case '?':
        // the option is not valid
        break;      
      default :
        System.err.println("Babel: Error: Should never have reached default in gov."
                           + "llnl.babel.UserOptions.");
        break;	
      }
    }

    /*
     * If no options were encountered and there's nothing else on the line, 
     * then force displaying the help.
     */
    int returnIndex = commandlineOptions.getOptind();
    if ( (returnIndex == 0) && (returnIndex == args.length) ) {
      helpSet     = true;
      numRequired = 1;
    }


    /*
     * If building the Python client, then allow generation of the SIDL
     * Standard library to ensure it happens for only one Python client.
     * Note that this is a deviation from the requirement that no more
     * than one of the specified fundamental options be allowed on the
     * command line at a time, but it is the "best" thing to do in this
     * case.
     */
    if (  theConfig.generateClient() 
       && theConfig.getTargetLanguage().equals("python")
       && theConfig.generateStdlib()  ) {
      System.err.println("Babel: Warning:  Allowing concurrent request for python "
                         + "client and SIDL standard library generation.");
      numRequired = numRequired - 1;
    }

    /*
     * Check to make sure the user specified no more than one of the 
     * fundamental, required options.
     */
    if (numRequired > 1) 
    {
      System.err.println("Babel: Error: Only one of the following options or its "
                         + "equivalent is allowed:");
      System.err.println("         --help, --version, --client, --server, "
//ToDo...Replace current list with current commented once remote supported.
//                         + "--remote, --text,");
                         + "--text,");
      System.err.println("         --parse-check, or --generate-sidl-stdlib.");
    }
    else if (numRequired < 1) {
      System.err.println("Babel: Error: You must enter one of the following options "
                         + "or its equivalent along with");
      System.err.println("       any associated argument:");
      System.err.println("         --help, --version, --client, --server, "
//ToDo...Replace current list with current commented once remote supported.
//                         + "--remote, --text,");
                         + "--text,");
      System.err.println("         --parse-check, or --generate-sidl-stdlib.");
    } 
    else
    {
      /*
       * Finalize any temporary options or ensure option consistency as
       * needed.
       */

      if ( versionSet || helpSet ) {
          if (versionSet) { 
              System.out.println( "Babel version " + Version.VERSION );
          }
          if ( helpSet ) {
              System.out.println();
              printUsage();
          }
      } else {
        d_can_proceed = true;

        if ( theConfig.generateStdlib() ) {
          /*
           * Make sure, regardless of user ordering, SIDL stdlib generation 
           * always creates full server (in C).  One exception is when
           * the Python client is being generated since the standard lib
           * symbols need to be built at the same time.
           */
          theConfig.setGenerateServer(true);
          if (  !theConfig.generateClient()
             || !"python".equals(theConfig.getTargetLanguage()) ) {
            theConfig.setTargetLanguage("c");
          }
        }
      }
    }

    /*
     * Return the index to the first unused argument for further processing.
     */
    return returnIndex;
  }


  /**
   * Print the command line options and contact information.
   */
  public void printUsage()
  { 
    System.out.println("Usage:  babel [ -h | --help ]");
    System.out.println("   or   babel [ -v | --version ]");
    System.out.println("   or   babel option(s) sidlfilename1 ... "
                       + "sidlfilenameN");
    System.out.println();
    System.out.println("where help, version, and option(s) are:");
    System.out.println(" -h       | --help          Display usage information "
                       + "and exit.");
    System.out.println(" -v       | --version       Display version and exit.");
    System.out.println(" -p       | --parse-check   Parse the sidl file but do "
                       + "not generate code.");
    System.out.println(" -x       | --xml           Generate only SIDL XML ("
                       + "deprecated; use -tXML).");
    System.out.println(" -c<lang> | --client=<lang> Generate only client code "
                       + "in specified language");
    System.out.println("                              (C | C++ | F77 | F90 |"
                       + " Java | Python).");
//ToDo...Uncomment the following once remote is supported.
//    System.out.println(" -r<lang> | --remote=<lang> Generate only remote "
//                       + "client code in specified");
//    System.out.println("                              language ().");
    System.out.println(" -s<lang> | --server=<lang> Generate server (and "
                       + "client) code in specified");
    System.out.println("                              language (C | C++ | F77 "
                       + "| F90 | Java | Python).");
    System.out.println(" -t<form> | --text=<form>   Generate text in specified"
                       + " form (XML | SIDL), where");
    System.out.println("                              XML updates the "
                       + "repository.");
    System.out.println(" -o<dir>  | --output-directory=<dir>");
    System.out.println("                            Set Babel output directory "
                       + "(\'.\' default).");
    System.out.println(" -R<path> | --repository-path=<path>");
    System.out.println("                            Set semicolon-separated URL"
                       + " list used to resolve");
    System.out.println("                              symbols.");
    System.out.println(" -e<regex> | --exclude=<regex>");
    System.out.println("                            " +
                       "Symbols matching the regular expression");
    System.out.println("                            " +
                       "are excluded from code generation.");
    System.out.println("                            " +
                       "Requires JRE 1.4 virtual machine or higher.");
    System.out.println("                            matching package hierarchy.");
    System.out.println(" -E        | --exclude-external");
    System.out.println("                            " +
                       "Code is generated only for the symbol hierarchies rooted");
    System.out.println("                            " +
                       "at the symbols specified on the command line.");
    System.out.println(" -V<path>  | --vpath=<path> Prepend alternative search path for reading Impl");
    System.out.println("                            splicer blocks.  Does not affect where Impls are");
    System.out.println("                            generated.");
    System.out.println("                            NOTE: --vpath=. is a no-op for autoconf/automake");
    System.out.println("                            If you really want current directory, use"); 
    System.out.println("                            another argument like `pwd` or even ./.");
    System.out.println(" -g       | --generate-subdirs");
    System.out.println("                            Postpend Java-style package subdirs");
    System.out.println("                            to paths specified by -o and -V.");
    System.out.println(" -l       | --language-subdir");
    System.out.println("                            Generate code in a language-specific");
    System.out.println("                            subdirectory.");
    System.out.println(" -u       | --hide-glue");
    System.out.println("                            Put \"glue\" (not modifiable by user) generated");
    System.out.println("                            code in a glue/ subdirectory.");
    System.out.println(" --no-default-repository    Prohibit use of default to "
                       + "resolve symbols.");
    System.out.println(" --suppress-timestamp       Suppress timestamps in "
                       + "generated files.");
    System.out.println(" --generate-sidl-stdlib     Regenerate only the SIDL "
                       + "standard library.");
    System.out.println
      (" --comment-local-only       Only comment locally defined methods in stub.");
    System.out.println();
    System.out.println("Note:  One of the following options or its "
                       + "equivalent must be specified:");
    System.out.println("         --help, --version, --client, --server, "
                       + "--xml, --text, --parse-check, or");
    System.out.println("         --generate-sidl-stdlib.");
    System.out.println();
    System.out.println("If you have any suggestions or questions not answered "
                       + "by the documenation,");
    System.out.println("please send email to components@llnl.gov.");
  }
  
  
  /**
   * Check if the specified client-side language is supported.
   */
  private boolean validClientLanguage( String lang ) 
  { 
    CodeGenerationFactory factory = CodeGenerationFactory.getInstance();
    return (factory.getCodeGenerator(lang, "stub") != null);
  }


  /**
   * Check if the specified remote client-side language is supported.
   */
  private static boolean validRemoteLanguage( String lang ) 
  { 
    boolean isValid = false;
    return isValid; 
  }


  /**
   * Check if the specified server-side language is supported.
   */
  private static boolean validServerLanguage( String lang ) 
  { 
    CodeGenerationFactory factory = CodeGenerationFactory.getInstance();
    return (factory.getCodeGenerator(lang, "skel") != null);
  }


  /**
   * Check if the specified textual representation is supported.
   */
  private boolean validTextRepresentation( String lang ) 
  { 
    boolean isValid = false;
    if ( lang.equals("xml") ) {
      isValid = true;
    } else {
      CodeGenerationFactory factory = CodeGenerationFactory.getInstance();
      isValid = factory.getCodeGenerator(lang, "text") != null;
    }
    return isValid;
  }


  /**
   * Check if the user options indicate processing can continue.
   */
  public boolean canProceed() { 
    return d_can_proceed; 
  }


  /*********************************************************************
   * Used for testing purposes only!
   */
  public static void main( String args[] ) 
  { 
    UserOptions opts          = getInstance();
    BabelConfiguration config = new BabelConfiguration();

    System.out.println("\n...Processing the command line...");
    opts.parseCommandlineOptions(config, args);

    if ( opts.canProceed() ) {
      System.out.println("\n*** Able to proceed with compilation request ***");
    } else {
      System.out.println("\n*** CANNOT proceed with compilation ***");
    }

    System.out.println("\n\n...Printing resulting settings...");
    System.out.println("Generate Client      = " + config.generateClient());
//ToDo...Uncomment the following once remote supported.
//    System.out.println("Generate Remote      = " + config.generateRemote());
    System.out.println("Generate Server      = " + config.generateServer());
    System.out.println("Generate Text        = " + config.generateText());
    System.out.println("Generate SIDL Stdlib = " + config.generateStdlib());
    System.out.println("Parse Only           = " + config.parseCheckOnly());
    System.out.println("Target Language      = " + config.getTargetLanguage());
    System.out.println("Output Directory     = " + config.getOutputDirectory());
    System.out.println("VPATH Directory      = " + config.getVPathDirectory());
    System.out.println("Make Pkg Subdirs     = " + config.makePackageSubdirs());
    System.out.println("Repository Path      = " + config.getRepositoryPath());
    System.out.println("Suppress Timestamps  = " + config.suppressTimestamps());
    System.out.println("Make Language Subdir = " + config.makeLanguageSubdir());
    System.out.println("Make Glue Subdir     = " + config.makeGlueSubdirs());
    System.out.println("Exclude External Pkgs= " + config.excludeExternal());

    System.out.println("\n\n...Done.");
  }
}
