//
// File:        IOR.java
// Package:     gov.llnl.babel.backend
// Release:     $Name: release-0-8-8 $
// Revision:    @(#) $Id: IOR.java,v 1.29 2003/02/04 23:49:08 epperly Exp $
// Description: common SIDL to IOR routines shared by code generators
//
// Copyright (c) 2000-2001, The Regents of the University of Calfornia.
// Produced at the Lawrence Livermore National Laboratory.
// Written by the Components Team <components@llnl.gov>
// UCRL-CODE-2002-054
// All rights reserved.
// 
// This file is part of Babel. For more information, see
// http://www.llnl.gov/CASC/components/. Please read the COPYRIGHT file
// for Our Notice and the LICENSE file for the GNU Lesser General Public
// License.
// 
// This program is free software; you can redistribute it and/or modify it
// under the terms of the GNU Lesser General Public License (as published by
// the Free Software Foundation) version 2.1 dated February 1999.
// 
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the IMPLIED WARRANTY OF
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the terms and
// conditions of the GNU Lesser General Public License for more details.
// 
// You should have recieved a copy of the GNU Lesser General Public License
// along with this program; if not, write to the Free Software Foundation,
// Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

package gov.llnl.babel.backend;

import gov.llnl.babel.backend.CodeGenerationException;
import gov.llnl.babel.backend.Utilities;
import gov.llnl.babel.symbols.Argument;
import gov.llnl.babel.symbols.Comment;
import gov.llnl.babel.symbols.Method;
import gov.llnl.babel.symbols.Symbol;
import gov.llnl.babel.symbols.SymbolID;
import gov.llnl.babel.symbols.Type;
import gov.llnl.babel.symbols.Version;
import gov.llnl.babel.BabelConfiguration;
import java.util.Iterator;
import java.util.List;

/**
 * Class <code>IOR</code> contains common SIDL to IOR translation
 * routines shared by the backend code generators.  This class simply
 * collects many common IOR language bindings into one place.
 */
public class IOR {
  public final static int MAJOR_VERSION = 0;
  public final static int MINOR_VERSION = 8;

   private final static String s_types[] = {
      "void",
      "SIDL_bool",
      "char",
      "struct SIDL_dcomplex",
      "double",
      "struct SIDL_fcomplex",
      "float",
      "int32_t",
      "int64_t",
      "void*",
      "char*"
   };

   private final static String s_array_types[] = {
      null,
      "struct SIDL_bool__array*",
      "struct SIDL_char__array*",
      "struct SIDL_dcomplex__array*",
      "struct SIDL_double__array*",
      "struct SIDL_fcomplex__array*",
      "struct SIDL_float__array*",
      "struct SIDL_int__array*",
      "struct SIDL_long__array*",
      "struct SIDL_opaque__array*",
      "struct SIDL_string__array*"
   };

  private final static String [] s_builtinMethods = {
    "_cast",                    // the CAST method
    "_delete",                  // the DELETE method
    "_ctor",                    // the CONSTRUCTOR method
    "_dtor"                     // the DESTRUCTOR method
  };

  private final static String[] s_builtin_comments = {
    "Cast method for interface and class type conversions.",
    "Delete method called automatically by IOR to destroy object.",
    "Class constructor called when the class is created.",
    "Class destructor called when the class is deleted."
  };

  /**
   * The number of builtin methods that an interface has. Builtin methods
   * are implicitly defined methods that are required for the inner
   * workings of the IOR or to support the language bindings. The names
   * of the builtins are numbers
   * <code>0...INTERFACE_BUILTIN_METHODS-1</code> and are available from
   * the method <code>getBuiltinMethod</code>.
   *
   * @see #getBuiltinMethod
   */
  public static final int INTERFACE_BUILTIN_METHODS = 2;

  /**
   * The number of builtin methods that a class has. Builtin methods are
   * implicitly defined methods that are required for the inner
   * workings of the IOR or to support the language bindings. The names
   * of the builtins are numbers
   * <code>0...CLASS_BUILTIN_METHODS-1</code> and are available from
   * the method <code>getBuiltinMethod</code>.
   */
  public static final int CLASS_BUILTIN_METHODS = 4;

  /**
   * The index of the builtin method for casting.
   */
  public static final int CAST = 0;

  /**
   * The index of the builtin method for deleting an object.
   */
  public static final int DELETE = 1;

  /**
   * The index of the builtin method for constructing a
   * class.
   */
  public static final int CONSTRUCTOR = 2;

  /**
   * The index of the builtin method for destructing a
   * class.
   */
  public static final int DESTRUCTOR = 3;

  private static String s_exceptionType = null;
  private static String s_objectType    = null;
  private static String s_interfaceType = null;

  static {
    SymbolID id = new SymbolID(BabelConfiguration.getBaseException(),
                               new Version());
    s_exceptionType = getObjectName(id) + " *";
    id = new SymbolID(BabelConfiguration.getBaseClass(),
                      new Version());
    s_objectType = getObjectName(id) + " *";
    id = new SymbolID(BabelConfiguration.getBaseInterface(),
                      new Version());
    s_interfaceType = getObjectName(id) + " *";
  }

  /**
   * Return the method description of a particular builtin method. This will
   * raise an <code>ArrayIndexOutOfBoundsException</code> if
   * <code>index</code> is less than zero or greater than or equal to the
   * number of builtin methods.
   *
   * @param index       the index of the builtin method that is
   *                    desired. Generally, one of <code>CAST</code>,
   *                    <code>DELETE</code>, <code>CONSTRUCTOR</code>,
   *                    or <code>DESTRUCTOR</code>.
   * @param id          the name of the symbol
   * @return a description the method.
   * @exception java.lang.ArrayIndexOutOfBoundsException
   *    this runtime exception is thrown if <code>index</code> is out of
   *    bounds.
   * @see #CAST
   * @see #DELETE
   * @see #CONSTRUCTOR
   * @see #DESTRUCTOR
   */
  public static Method getBuiltinMethod(int index,
                                        SymbolID id)
  {
    Method m = new Method();
    m.setMethodName(getBuiltinName(index));
    m.setDefinitionModifier(Method.NORMAL);

    String[] s = new String[1];
    s[0] = s_builtin_comments[index];
    m.setComment(new Comment(s));

    if (CAST == index) {
      Argument a = new
        Argument(false, Argument.IN, new Type(Type.STRING), "name");
      m.setReturnType(new Type(Type.OPAQUE));
      m.addArgument(a);
    }
    else {
      m.setReturnType(new Type(Type.VOID));
    }
    return m;
  }

  /**
   * Return the name of a builtin method.
   *
   * @param index       the index of the builtin method that is
   *                    desired. Generally, one of <code>CAST</code>,
   *                    <code>DELETE</code>, <code>CONSTRUCTOR</code>,
   *                    or <code>DESTRUCTOR</code>.
   * @exception java.lang.ArrayIndexOutOfBoundsException
   *    this runtime exception is thrown if <code>index</code> is out of
   *    bounds.
   * @see #CAST
   * @see #DELETE
   * @see #CONSTRUCTOR
   * @see #DESTRUCTOR
   */
  public static String getBuiltinName(int index) {
    return s_builtinMethods[index];
  }

  /**
   * Generate the name of an entry in the entry point vector or the
   * static entry point vector.
   *
   * @param methodName  the name of the method that is an element
   *                    in the entry point vector.
   */
  public static String getVectorEntry(String methodName) {
    StringBuffer buf = new StringBuffer(2 + methodName.length());
    buf.append("f_").append(methodName);
    return buf.toString();
  }

   /**
    * Generate the header filename associated with a symbol identifier.
    * Replace the "." scope separators in the symbol by underscores and
    * append the suffix "_IOR.h".
    */
   public static String getHeaderFile(SymbolID id) {
      return id.getFullName().replace('.', '_') + "_IOR.h";
   }

   /**
    * Generate the source filename associated with a symbol identifier.
    * Replace the "." scope separators in the symbol by underscores and
    * append the suffix "_IOR.c".
    */
   public static String getSourceFile(SymbolID id) {
      return id.getFullName().replace('.', '_') + "_IOR.c";
   }

   /**
    * Convert a symbol name into an IOR identifier.  This method replaces
    * the "." scope separators in the symbol by underscores.
    */
   public static String getSymbolName(SymbolID id) {
      return id.getFullName().replace('.', '_');
   }

   /**
    * Convert a SIDL enumerated type into its symbol name, which is
    * "enum " followed by the symbol name followed by "__enum".
    */
   public static String getEnumName(SymbolID id) {
      return "enum " + getSymbolName(id) + "__enum";
   }

  /**
   * Get struct name for extern entry point structure.
   */
  public static String getExternalName(SymbolID id)
  {
    return "struct " + getSymbolName(id) + "__external";
  }

  /**
   * Get the name of the function that returns the structure of
   * external entry points.
   */
  public static String getExternalFunc(SymbolID id)
  {
    return getSymbolName(id) + "__externals";
  }

   /**
    * Convert a SIDL interface or class into its symbol name, which is
    * "struct " followed by the symbol name followed by "__object".
    */
   public static String getObjectName(SymbolID id) {
      return "struct " + getSymbolName(id) + "__object";
   }

   /**
    * Convert a SIDL symbol into its array representation, which is
    * "struct " followed by the symbol name followed by "__array".
    */
   public static String getArrayName(SymbolID id) {
      return "struct " + getSymbolName(id) + "__array";
   }

   /**
    * Convert a SIDL symbol into its method entry point vector (EPV)
    * name, which is "struct " followed by the symbol name followed by
    * "__epv".
    */
   public static String getEPVName(SymbolID id) {
      return "struct " + getSymbolName(id) + "__epv";
   }

   /**
    * Convert a SIDL symbol into its static entry point vector (SEPV)
    * name, which is "struct " followed by the symbol name followed by
    * "__sepv".
    */
   public static String getSEPVName(SymbolID id) {
      return "struct " + getSymbolName(id) + "__sepv";
   }

   /**
    * Convert a SIDL symbol into the name of its associated constructor,
    * which is the symbol name appended with "__new".
    */
   public static String getNewName(SymbolID id) {
      return getSymbolName(id) + "__new";
   }

   /**
    * Convert a SIDL symbol into the name of its associated remote
    * constructor, which is the symbol name appended with "__remote".
    */
   public static String getRemoteName(SymbolID id) {
      return getSymbolName(id) + "__remote";
   }

   /**
    * Convert a SIDL symbol into the name of its associated set EPV
    * method, which is the symbol name appended with "__set_epv".
    */
   public static String getSetEPVName(SymbolID id) {
      return getSymbolName(id) + "__set_epv";
   }
   /**
    * Convert a SIDL symbol into the name of its associated set static
    * EPV method, which is the symbol name appended with "__set_sepv".
    */
   public static String getSetSEPVName(SymbolID id) {
      return getSymbolName(id) + "__set_sepv";
   }

   /**
    * Convert a SIDL symbol into the name of its associated statics
    * method, which is the symbol name appended with "__statics".
    */
   public static String getStaticsName(SymbolID id) {
      return getSymbolName(id) + "__statics";
   }

   /**
    * Convert a SIDL symbol into the name of its associated init
    * method, which is the symbol name appended with "__init".
    */
   public static String getInitName(SymbolID id) {
      return getSymbolName(id) + "__init";
   }
   /**
    * Convert a SIDL symbol into the name of its associated fini
    * method, which is the symbol name appended with "__fini".
    */
   public static String getFiniName(SymbolID id) {
      return getSymbolName(id) + "__fini";
   }

   /**
    * Convert a SIDL symbol into the name of its associated fini
    * method, which is the symbol name appended with "__fini".
    */
   public static String getVersionName(SymbolID id) {
      return getSymbolName(id) + "__IOR_version";
   }

  /**
   * Return the name of the type of the implicit exception argument.
   * The return value is of the form "struct X_Y_Z *" where X_Y_Z
   * depends on the name of the base exception class and its mapping
   * to a C struct name.
   */
  public static String getExceptionType() {
    return s_exceptionType;
  }

  /**
   * Return the name of the type of the implicit base class type.
   * The return value is of the form "struct X_Y_Z *" where X_Y_Z
   * depends on the name of the base class and its mapping
   * to a C struct name.
   */
  public static String getClassType() {
    return s_objectType;
  }

  /**
   * Return the name of the type of the base interface type.
   * The return value is of the form "struct X_Y_Z *" where X_Y_Z
   * depends on the name of the base interface and its mapping
   * to a C struct name.
   */
  public static String getInterfaceType() {
    return s_interfaceType;
  }

  public static String getSymbolType(Symbol sym) {
    if (sym.getSymbolType() == Symbol.ENUM) {
      return getEnumName(sym.getSymbolID());
    } else {
      return getObjectName(sym.getSymbolID()) + "*";
    }
  }

   /**
    * Generate an IOR return string for the specified SIDL type.  Most
    * of the SIDL return strings are listed in the static structures defined
    * at the start of this class.  Symbol types and array types require
    * special processing.
    */
   public static String getReturnString(Type type)
         throws CodeGenerationException {
      /*
       * If the type is one of the primitive types, then just return
       * its string value from the lookup table.
       */
      int t = type.getType();
      if (t < s_types.length) {
         return s_types[t];
      }

      /*
       * If the type is a symbol, then look up the symbol type and return
       * the associated type name.
       */
      if (t == Type.SYMBOL) {
         Symbol symbol = Utilities.lookupSymbol(type.getSymbolID());
         return getSymbolType(symbol);
      }

      /*
       * If the type is an array, then either return one of the primitive
       * array types or construct the corresponding array type.
       */
      if (t == Type.ARRAY) {
         Type atype = type.getArrayType();
         int a = atype.getType();
         if (a < s_array_types.length) {
            return s_array_types[a];
         } else {
            return getArrayName(atype.getSymbolID()) + "*";
         }
      }

      return null;
   }

   /**
    * Generate an IOR argument string for the specified SIDL argument.
    * The formal argument name is not included.
    */
   public static String getArgumentString(Argument arg)
         throws CodeGenerationException {
      String s = getReturnString(arg.getType());
      int type = arg.getType().getType();
      if (arg.getMode() == Argument.IN) {
        if (type == Type.STRING) {
          s = "const " + s;
        }
      }
      else {
        s = s + "*";
      }
      return s;
   }

   /**
    * Generate an IOR argument string with the formal argument name.
    */
   public static String getArgumentWithFormal(Argument arg)
         throws CodeGenerationException {
      return getArgumentString(arg) + " " + arg.getFormalName();
   }

   /**
    * Generate a cast string for the specified method.  The string
    * argument self represents the name of the object.  A code generation
    * exception is thrown if any of the required symbols do not exist in
    * the symbol table.
    */
   public static String getCast(Method method, String self)
         throws CodeGenerationException {

      /*
       * Begin the cast string with the return type and self object reference.
       */
      StringBuffer cast = new StringBuffer();
      cast.append("(");
      cast.append(getReturnString(method.getReturnType()));
      cast.append(" (*)(");
      cast.append(self);

      /*
       * Add the method arguments to the cast clause as well as an
       * optional exception argument.
       */
      boolean has_throws = !method.getThrows().isEmpty();
      List args = method.getArgumentList();

      if ((args.size() > 0) || has_throws) {
         cast.append(",");
      }

      for (Iterator a = args.iterator(); a.hasNext(); ) {
         Argument arg = (Argument) a.next();
         cast.append(getArgumentString(arg));
         if (a.hasNext() || has_throws) {
            cast.append(",");
         }
      }

      if (has_throws) {
        cast.append(getExceptionType());
        cast.append('*');
      }
      cast.append("))");

      return cast.toString();
   }
}
