//
// File:        StubHeader.java
// Package:     gov.llnl.babel.backend.c
// Release:     $Name: release-0-8-8 $
// Revision:    @(#) $Id: StubHeader.java,v 1.28 2003/09/18 14:46:41 epperly Exp $
// Description: write C client (i.e., stub) header to a pretty writer stream
//
// Copyright (c) 2000-2001, The Regents of the University of Calfornia.
// Produced at the Lawrence Livermore National Laboratory.
// Written by the Components Team <components@llnl.gov>
// UCRL-CODE-2002-054
// All rights reserved.
// 
// This file is part of Babel. For more information, see
// http://www.llnl.gov/CASC/components/. Please read the COPYRIGHT file
// for Our Notice and the LICENSE file for the GNU Lesser General Public
// License.
// 
// This program is free software; you can redistribute it and/or modify it
// under the terms of the GNU Lesser General Public License (as published by
// the Free Software Foundation) version 2.1 dated February 1999.
// 
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the IMPLIED WARRANTY OF
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the terms and
// conditions of the GNU Lesser General Public License for more details.
// 
// You should have recieved a copy of the GNU Lesser General Public License
// along with this program; if not, write to the Free Software Foundation,
// Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

package gov.llnl.babel.backend.c;

import gov.llnl.babel.backend.CodeConstants;
import gov.llnl.babel.backend.CodeGenerationException;
import gov.llnl.babel.backend.IOR;
import gov.llnl.babel.backend.Utilities;
import gov.llnl.babel.backend.ior.IORSource;
import gov.llnl.babel.backend.writers.LanguageWriterForC;
import gov.llnl.babel.symbols.Argument;
import gov.llnl.babel.symbols.Comment;
import gov.llnl.babel.symbols.Enumeration;
import gov.llnl.babel.symbols.Extendable;
import gov.llnl.babel.symbols.Method;
import gov.llnl.babel.symbols.Package;
import gov.llnl.babel.symbols.Symbol;
import gov.llnl.babel.symbols.SymbolID;
import gov.llnl.babel.BabelConfiguration;
import java.util.Collection;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Set;

/**
 * Class <code>StubHeader</code> writes a C client header to a language
 * writer output stream.  The constructor takes a language writer stream
 * and method <code>generateCode</code> writes the C client header code
 * for the specified symbol to the output stream.  The language writer
 * output stream is not closed by this object.
 */
public class StubHeader {
  private final static String SIDL_EXCEPTION =
    BabelConfiguration.getBaseException();

  private LanguageWriterForC d_writer;

  private boolean d_comment_all;

  /**
   * This is a convenience utility function that writes the C client
   * header information into the provided language writer output stream.
   * The output stream is not closed on exit.  A code generation
   * exception is thrown if an error is detected.
   *
   * @param symbol the symbol for which a C client header will
   *               be written.
   *
   * @param writer the output writer to which the header will
   *               be written. This will not be closed.
   *
   * @exception gov.llnl.backend.CodeGenerationException
   *    this is a catch all exception. It can be caused by I/O trouble or
   *    violations of the data type invariants.
   */
  public static void generateCode(Symbol             symbol,
                                  LanguageWriterForC writer)
    throws CodeGenerationException
  {
    StubHeader header = new StubHeader(writer);
    header.generateCode(symbol);
  }

  /**
   * Create a <code>StubHeader</code> object that will write symbol
   * information to the provided output language writer stream.
   *
   * @param writer the output writer to which the header will
   *               be written. This will not be closed.
   */
  public StubHeader(LanguageWriterForC writer) {
    d_writer = writer;
    d_comment_all = !BabelConfiguration.getInstance().getCommentLocalOnly();
  }

  /**
   * Write C client header information for the provided symbol to the
   * language writer output stream provided in the constructor.  This
   * method does not close the writer output stream and may be called
   * for more than one symbol (although the written header may not be
   * valid input for the C compiler).  A code generation exception is
   * written if an error is detected.
   *
   * @param symbol the <code>Symbol</code> whose header will be
   *               written.
   *
   * @exception gov.llnl.babel.backend.CodeGenerationException
   *    this is a catch all exception. It can be caused by I/O trouble
   *    or violations of the data type invariants.
   */
  public void generateCode(Symbol symbol)
    throws CodeGenerationException
  {
    if (symbol != null) {
      generatePrologue(symbol);

      switch (symbol.getSymbolType()) {
        case Symbol.CLASS:
          generateExtendable((Extendable) symbol);
          break;
        case Symbol.ENUM:
          generateEnumeration((Enumeration) symbol);
          break;
        case Symbol.INTERFACE:
          generateExtendable((Extendable) symbol);
          break;
        case Symbol.PACKAGE:
          generatePackage((Package) symbol);
          break;
        default:
          throw new CodeGenerationException("Unsupported symbol type.");
      }
    } else {
      throw new CodeGenerationException("Unexpected null Symbol.");
    }
  }

  /**
   * Output a banner and the main header guard at the top of the file.
   *
   * @param sym the <code>Symbol</code> whose header will be written.
   */
  private void generatePrologue(Symbol sym) {
    final SymbolID id     = sym.getSymbolID();
    String         header = C.getHeaderFile(id);

    d_writer.writeBanner(sym, header, false, 
      CodeConstants.C_DESC_STUB_PREFIX + id.getFullName());
    d_writer.openHeaderGuard(header);
  }

  /**
   * Output the closing block methods as needed at the end of the file.
   *
   * @param closeExtern the boolean used to indicate if the C++ extern block
   *     needs to be closed.
   */
  private void generateEpilogue(boolean closeExtern) {
    /*
     * Conclude the header file by closing the C++ extern block, if
     * indicated, and the header include guard.
     */
    if (closeExtern) {
      d_writer.closeCxxExtern();
    }
    d_writer.closeHeaderGuard();
  }

  /**
   * Generate the C header for a SIDL enumerated type.  This method simply
   * includes the IOR enumerated type header file and typedefs that type
   * to a C type.
   *
   * @param enm the <code>Enumeration</code> whose header is being written.
   */
  private void generateEnumeration(Enumeration enm) {
    SymbolID id = enm.getSymbolID();
    ArrayMethods ar = new ArrayMethods(id, true);

    d_writer.generateInclude(IOR.getHeaderFile(id), true);
    d_writer.println();

    d_writer.openCxxExtern();
    ar.generateHeader(d_writer);
    generateEpilogue(true);
  }

  /**
   * Generate a C client header for a SIDL class or interface description.
   * The header file consists of the typedef the defines the symbol type.
   * Note that the typedef comes before any external includes to solve
   * the problem with forward references.  After the typedef comes the
   * external includes, followed by special methods such as cast and new,
   * followed by the regular methods.  The header concludes with close
   * statements for the header guards.
   *
   * @param ext the <code>Extendable</code> whose header is being written.
   *
   * @exception gov.llnl.babel.backend.CodeGenerationException
   *    this is a catch all exception. It can be caused by I/O trouble
   *    or violations of the data type invariants.
   */
  private void generateExtendable(Extendable ext)
    throws CodeGenerationException
  {
    SymbolID id = ext.getSymbolID();

    /*
     * Output a documentation comment, forward declare the object, and
     * define the symbol typedef.
     */
    d_writer.writeComment(ext, true);
    d_writer.println(IOR.getObjectName(id) + ";");
    d_writer.println(IOR.getArrayName(id) + ";");
    d_writer.print("typedef " + IOR.getObjectName(id) + "* ");
    d_writer.println(C.getObjectName(id) + ";");
    d_writer.println();

    /*
     * Generate the includes and open the C++ extern block.
     */
    generateIncludes(ext);
    d_writer.openCxxExtern();

    /*
     * Output standard function signatures for all methods.
     */
    generateMethodPrototypes(ext);

    /*
     * Output array method signatures
     */
    ArrayMethods ar = new ArrayMethods(id, false);
    ar.generateHeader(d_writer);

    /*
     * Conclude the header file by closing block(s) as needed.
     */
    generateEpilogue(true);
  }

  /**
   * Generate the C client header for a SIDL package description.  The
   * package header file consists of the standard header information along
   * with include statements for all package symbols.
   *
   * @param p the <code>Package</code> whose header is being written.
   */
  private void generatePackage(Package p) {
    SymbolID id   = p.getSymbolID();

    /*
     * Write out the C include files for each of the symbols within
     * the package.
     */
    List entries = Utilities.sort(p.getSymbols().keySet());
    for (Iterator i = entries.iterator(); i.hasNext(); ) {
      String include = C.getHeaderFile((SymbolID) i.next());
      d_writer.generateInclude(include, true);
    }
    d_writer.println();

    generateEpilogue(false);
  }

  /**
   * Generate the list of include files required to satisfy data
   * dependencies within this header file and output the appropriate
   * include lines.  If any of the symbols do not exist in the symbol
   * table, then throw a code generation exception.
   */
  private void generateIncludes(Extendable ext) throws CodeGenerationException {

    /*
     * Create the set of include symbol identifiers.
     */
    Set includes = new HashSet();

    for (Iterator i = ext.getMethods(true).iterator(); i.hasNext(); ) {
      Method method = (Method) i.next();
      includes.addAll(method.getSymbolReferences());

      if (!method.getThrows().isEmpty()) {
        Symbol symbol = Utilities.lookupSymbol(SIDL_EXCEPTION);
        includes.add(symbol.getSymbolID());
      }
    }

    /*
     * We should always include the base header file.
     */
    d_writer.writeComment("Includes for all header dependencies.", false);
    d_writer.generateInclude("SIDL_header.h", true);

    /*
     * Remove this symbol from the dependency set and iterate over symbols if
     * there are any remaining symbols in the set.
     */
    includes.remove(ext.getSymbolID());
    if (!includes.isEmpty()) {
      List entries = Utilities.sort(includes);
      for (Iterator i = entries.iterator(); i.hasNext(); ) {
        String header = C.getHeaderFile((SymbolID) i.next());
        d_writer.generateInclude(header, true);
      }
      d_writer.println();
    }
  }

  /**
   * Generate method prototypes for the methods in the interface or class.
   */
  private void generateMethodPrototypes(Extendable ext)
    throws CodeGenerationException {
    /*
     * Generate the name for this entry point vector as well as a pointer
     * to "self" for this structure.  For classes, self will be the object
     * structure whereas for interfaces it is void*.
     */
    SymbolID id   = ext.getSymbolID();

    /*
     * Output the constructor if this class or interface is not abstract.
     */
    if (!ext.isAbstract()) {
      d_writer.writeComment("Constructor function for the class.", true);
      d_writer.println(C.getSymbolName(id));
      d_writer.println(C.getFullMethodName(id, "_create") + "(void);");
      d_writer.println();
    }

    /*
     * Output each of the method prototypes from the interface or class.
     */
    Collection methods = ext.getMethods(true);
    for (Iterator m = methods.iterator(); m.hasNext(); ) {
      Method method = (Method) m.next();
      generateMethodSignature(id, method, 
                              d_comment_all || ext.isLocal(method));
      d_writer.println();
    }

    /*
     * Output the cast methods for the class or interface.
     */
    d_writer.writeComment(
      "Cast method for interface and class type conversions.", true);
    d_writer.println(C.getSymbolName(id));
    d_writer.println(C.getFullMethodName(id, "_cast") + "(");
    d_writer.increaseTabLevel();
    d_writer.println("void* obj);");
    d_writer.decreaseTabLevel();
    d_writer.println();

    d_writer.writeComment(
      "String cast method for interface and class type conversions.", true);
    d_writer.println("void*");
    d_writer.println(C.getFullMethodName(id, "_cast2") + "(");
    d_writer.increaseTabLevel();
    d_writer.println("void* obj,");
    d_writer.println("const char* type);");
    d_writer.decreaseTabLevel();
    d_writer.println();
  }

  /**
   * Generate the method signature.
   *
   * @param id the <code>SymbolID</code> of the <code>Extendable</code> whose
   *   header is being written.
   *
   * @param method the <code>Method</code> whose signature is being output.
   * 
   * @param docComment iff <code>true</code> this will print a document
   *                   comment for the method if one is available. If
   *                   <code>false</code>, not document comment will
   *                   be printed.
   *
   * @exception gov.llnl.babel.backend.CodeGenerationException
   *    this is a catch all exception. It can be caused by I/O trouble or
   *    violations of the data type invariants.
   */
  private void generateMethodSignature(SymbolID id,
                                       Method   method,
                                       boolean  docComment)
    throws CodeGenerationException
  {
    if (docComment) {
      d_writer.writeComment(method, true);
    }
    d_writer.println(C.getReturnString(method.getReturnType()));
    d_writer.print(C.getFullMethodName(id, method));
    d_writer.println("(");
    d_writer.increaseTabLevel();
    generateArgumentList(id, method);
    d_writer.println(");");
    d_writer.decreaseTabLevel();
  }

  /**
   * Generate the argument list, excluding the enclosing parentheses.
   *
   * @param id the <code>SymbolID</code> of the <code>Extendable</code> whose
   *   header is being written.
   *
   * @param method the <code>Method</code> whose signature is being output.
   *
   * @exception gov.llnl.babel.backend.CodeGenerationException
   *    this is a catch all exception. It can be caused by I/O trouble or
   *    violations of the data type invariants.
   */
  private void generateArgumentList(SymbolID id,
                                    Method   method)
    throws CodeGenerationException
  {
    Iterator a = Utilities.extendArgs(id, method).iterator();
    if (a.hasNext()) {
      while (a.hasNext()) {
        Argument arg = (Argument) a.next();
        d_writer.print(C.getArgumentWithFormal(arg));
        if (a.hasNext()) {
          d_writer.println(",");
        }
      }
    }
    else {
      d_writer.print("void");
    }
  }
}
