//
// File:        CxxImplSource.java
// Package:     gov.llnl.babel.backend.cxx
// Release:     $Name: release-0-8-8 $
// Revision:    @(#) $Revision: 1.21 $
// Date:        $Date: 2003/03/17 23:43:26 $
// Description: Write Cxx extension header file for a BABEL extendable
// 
// This is typically directed by GenCxxClient.
// Copyright (c) 2000-2001, The Regents of the University of Calfornia.
// Produced at the Lawrence Livermore National Laboratory.
// Written by the Components Team <components@llnl.gov>
// UCRL-CODE-2002-054
// All rights reserved.
// 
// This file is part of Babel. For more information, see
// http://www.llnl.gov/CASC/components/. Please read the COPYRIGHT file
// for Our Notice and the LICENSE file for the GNU Lesser General Public
// License.
// 
// This program is free software; you can redistribute it and/or modify it
// under the terms of the GNU Lesser General Public License (as published by
// the Free Software Foundation) version 2.1 dated February 1999.
// 
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the IMPLIED WARRANTY OF
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the terms and
// conditions of the GNU Lesser General Public License for more details.
// 
// You should have recieved a copy of the GNU Lesser General Public License
// along with this program; if not, write to the Free Software Foundation,
// Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

package gov.llnl.babel.backend.cxx;

import gov.llnl.babel.BabelConfiguration;
import gov.llnl.babel.backend.CodeConstants;
import gov.llnl.babel.backend.CodeGenerationException;
import gov.llnl.babel.backend.CodeGenerator;
import gov.llnl.babel.backend.CodeSplicer;
import gov.llnl.babel.backend.cxx.Cxx;
import gov.llnl.babel.backend.FileManager;
import gov.llnl.babel.backend.IOR;
import gov.llnl.babel.backend.Utilities;
import gov.llnl.babel.backend.writers.LanguageWriter;
import gov.llnl.babel.backend.writers.LanguageWriterForCxx;
import gov.llnl.babel.backend.writers.LineRedirector;
import gov.llnl.babel.symbols.Argument;
import gov.llnl.babel.symbols.Class;
import gov.llnl.babel.symbols.Comment;
import gov.llnl.babel.symbols.Extendable;
import gov.llnl.babel.symbols.Method;
import gov.llnl.babel.symbols.Symbol;
import gov.llnl.babel.symbols.SymbolID;
import gov.llnl.babel.symbols.Version;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.Set;

/**
 * Create and write a header for a Cxx C extension class to wrap a 
 * BABEL extendable in a Cxx object. The header has to expose a 
 * function to create a wrapped IOR, a function to check if a 
 * <code>PyObject</code> is an instance of this extension type, and
 * an import macro.
 */
public class CxxImplSource {
  private Extendable d_ext = null;
  private LanguageWriterForCxx d_writer = null;
  private CodeSplicer d_splicer = null;
  
  /**
   * Create an object capable of generating the header file for a
   * BABEL extendable.
   *
   * @param ext   an interface or class symbol that needs a header
   *              file for a Cxx C extension class.
   */
  public CxxImplSource(Extendable ext) {
    d_ext = ext;
  }
  
  /**
   * Generate the header file for the extendable with which this object was
   * created.
   *
   * @exception gov.llnl.babel.backend.CodeGenerationException
   *    this is a catch all exception for problems during the code
   *    generation phase.
   */
  public synchronized void generateCode() 
    throws CodeGenerationException 
  {
    final SymbolID id = d_ext.getSymbolID();
    final int type= d_ext.getSymbolType();
    String filename = Cxx.generateFilename( id, 
                                            Cxx.FILE_ROLE_IMPL,
                                            Cxx.FILE_TYPE_CXX_SOURCE );
    //System.out.println("Create " + filename + "..." );
    
    try { 
      d_splicer = FileManager.getInstance().
        getCodeSplicer(id, type, filename);
      d_writer = Cxx.createSource( d_ext, Cxx.FILE_ROLE_IMPL, "IMPLSRCS");
      d_splicer.setLineRedirector( (LineRedirector) d_writer );
      writeIncludes();

      writeCtorDtor();

      writeSIDLDefinedMethods();

      d_splicer.splice( d_ext.getSymbolID().getFullName() + "._misc", 
                        d_writer, "Put miscellaneous code here");
      d_writer.println();

      if (d_splicer.hasUnusedSymbolEdits()) {
        d_writer.beginBlockComment(true);
        d_writer.println(CodeConstants.C_BEGIN_UNREFERENCED_METHODS);
        d_writer.println(CodeConstants.C_UNREFERENCED_COMMENT1);
        d_writer.println(CodeConstants.C_UNREFERENCED_COMMENT2);
        d_writer.println(CodeConstants.C_UNREFERENCED_COMMENT3);
        d_writer.endBlockComment(true);
        d_splicer.outputUnusedSymbolEdits( d_writer.getPrintWriter() );
        d_writer.writeCommentLine(CodeConstants.C_END_UNREFERENCED_METHODS);
      }
    } catch ( java.io.IOException ex) { 
      throw new CodeGenerationException("IOException : " + ex.getMessage() );
    } finally { 
      if (d_writer != null) {
        d_writer.close();
        d_writer = null;
      }
    }
  }

  private void writeIncludes() { 
    d_writer.generateInclude( Cxx.generateFilename( d_ext.getSymbolID(), 
                                                    Cxx.FILE_ROLE_IMPL,  
                                                    Cxx.FILE_TYPE_CXX_HEADER),
                              false );
    d_writer.println();
    d_splicer.splice( d_ext.getSymbolID().getFullName() + "._includes", 
                      d_writer, "Put additional includes or other arbitrary code here...");
    d_writer.println();
  }                   
  
  private void writeCtorDtor() {

    d_writer.writeCommentLine("user defined constructor");
    d_writer.print("void " + Utilities.replace(d_ext.getSymbolID().getFullName(),".","::"));
    d_writer.println("_impl::_ctor() {");
    d_writer.increaseTabLevel();
    d_splicer.splice( d_ext.getSymbolID().getFullName() + "._ctor", 
                      d_writer, 
                      "add construction details here");
    d_writer.decreaseTabLevel();
    d_writer.println("}");
    d_writer.println();

    d_writer.writeCommentLine("user defined destructor");
    d_writer.print("void " + Utilities.replace(d_ext.getSymbolID().getFullName(),".","::"));
    d_writer.println("_impl::_dtor() {");
    d_writer.increaseTabLevel();
    d_splicer.splice( d_ext.getSymbolID().getFullName() + "._dtor",
                      d_writer,
                      "add destruction details here");
    d_writer.decreaseTabLevel();
    d_writer.println("}");
    d_writer.println();
  }

  private void writeSIDLDefinedMethods() throws CodeGenerationException { 
    List static_methods = (List) d_ext.getStaticMethods(false);
    if ( static_methods.size() > 0 ) { 
      d_writer.writeCommentLine("user defined static methods:");
      for (Iterator m = static_methods.iterator(); m.hasNext(); ) {
        Method method = (Method) m.next();
        generateMethod(method);
      }
    } else { 
      d_writer.writeCommentLine("user defined static methods: (none)");
    }
    
    d_writer.println();
    List nonstatic_methods = (List) d_ext.getNonstaticMethods(false);
    if ( nonstatic_methods.size() > 0 ) { 
      d_writer.writeCommentLine("user defined non-static methods:");
      for (Iterator m = nonstatic_methods.iterator(); m.hasNext(); ) {
        Method method = (Method) m.next();
	if ( !method.isAbstract() ) { 	    
	    generateMethod( method );
	}
      }
    } else { 
      d_writer.writeCommentLine("user defined non-static methods: (none)");
    }
    d_writer.println();

  }

  private void generateMethod( Method method ) 
    throws CodeGenerationException 
  { 
    if ( method == null ) { return ; }

    d_writer.writeComment ( method, true );

    d_writer.println( Cxx.getReturnString( method.getReturnType() ) );
    d_writer.print(Utilities.replace(d_ext.getSymbolID().getFullName(),".","::"));
    d_writer.print("_impl::");
    d_writer.print( method.getShortMethodName() );
    
    if ( method.getArgumentList().size() > 0 ) { 
      d_writer.println(" (");
      d_writer.increaseTabLevel();
      Cxx.generateArgumentList( d_writer, method );
      d_writer.println(" ) ");
      d_writer.decreaseTabLevel();
      Cxx.generateThrowsList( d_writer, method, false );
      d_writer.println("{");
    } else { 
      d_writer.println(" () ");      
      Cxx.generateThrowsList( d_writer, method, false);
      d_writer.println();
      d_writer.println("{");
    }
    d_writer.increaseTabLevel();
    d_splicer.splice( d_ext.getSymbolID().getFullName() + "." +
                      method.getLongMethodName(), 
                      d_writer, 
                      "insert implementation here");
    d_writer.decreaseTabLevel();
    d_writer.println("}");
    d_writer.println();    
  }
}
