//
// File:        ImplSource.java
// Package:     gov.llnl.babel.backend.fortran
// Release:     $Name: release-0-8-8 $
// Revision:    @(#) $Revision: 1.13 $
// Description: Generate a skeleton implementation in Fortran
//
// Copyright (c) 2000-2002, The Regents of the University of Calfornia.
// Produced at the Lawrence Livermore National Laboratory.
// Written by the Components Team <components@llnl.gov>
// UCRL-CODE-2002-054
// All rights reserved.
//
// This file is part of Babel. For more information, see
// http://www.llnl.gov/CASC/components/. Please read the COPYRIGHT file
// for Our Notice and the LICENSE file for the GNU Lesser General Public
// License.
//
// This program is free software; you can redistribute it and/or modify it
// under the terms of the GNU Lesser General Public License (as published by
// the Free Software Foundation) version 2.1 dated February 1999.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the IMPLIED WARRANTY OF
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the terms and
// conditions of the GNU Lesser General Public License for more details.
//
// You should have recieved a copy of the GNU Lesser General Public License
// along with this program; if not, write to the Free Software Foundation,
// Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA


package gov.llnl.babel.backend.fortran;

import gov.llnl.babel.BabelConfiguration;
import gov.llnl.babel.backend.CodeConstants;
import gov.llnl.babel.backend.CodeGenerationException;
import gov.llnl.babel.backend.CodeSplicer;
import gov.llnl.babel.backend.IOR;
import gov.llnl.babel.backend.Utilities;
import gov.llnl.babel.backend.fortran.Fortran;
import gov.llnl.babel.backend.fortran.StubSource;
import gov.llnl.babel.backend.mangler.FortranMangler;
import gov.llnl.babel.backend.mangler.NameMangler;
import gov.llnl.babel.backend.mangler.NonMangler;
import gov.llnl.babel.backend.writers.LanguageWriter;
import gov.llnl.babel.backend.writers.LanguageWriterForFortran;
import gov.llnl.babel.symbols.Argument;
import gov.llnl.babel.symbols.Class;
import gov.llnl.babel.symbols.Method;
import gov.llnl.babel.symbols.Symbol;
import gov.llnl.babel.symbols.SymbolID;
import gov.llnl.babel.symbols.SymbolTable;
import gov.llnl.babel.symbols.Type;
import gov.llnl.babel.symbols.Version;
import java.security.NoSuchAlgorithmException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Set;

/**
 * This class provides the ability to write a FORTRAN file with a
 * subroutine template for each method the end user has to implement for
 * a SIDL class. The class will retain the previous user provided
 * implementation when overwriting a implementation file.
 */
public class ImplSource 
{
  private static final String s_use = ".use";

  /**
   * The code splicer stores the source code from the previous
   * implementation when overwriting an impl file.
   */
  private CodeSplicer          d_splicer;

  /**
   * This is the output device.
   */
  private LanguageWriterForFortran d_lw;

  /**
   * Hold a FORTRAN name mangler.
   */
  private NameMangler d_mang;

  /**
   * The version of Fortran being generated.
   */
  private final boolean d_isF90 = "f90".
    equals(BabelConfiguration.getInstance().getTargetLanguage());

  /**
   * Generate an instance to generate a FORTRAN implementation template.
   * 
   * @param writer    the output device to which the FORTRAN implementation
   *                  should be written.
   * @param splicer   this stores the previous implementation when one
   *                  exists.
   * @exception java.security.NoSuchAlgorithmException
   *                  thrown when the JVM has not SHA hash algorithm.
   */
  public ImplSource(LanguageWriterForFortran writer,
                    CodeSplicer              splicer)
    throws java.security.NoSuchAlgorithmException
  {
    d_lw      = writer;
    d_splicer = splicer;
    if (d_isF90) {
      d_mang =  new FortranMangler(AbbrevHeader.MAXNAME,
                                   AbbrevHeader.MAXUNMANGLED);
    }
    else {
      d_mang  =  new NonMangler();
    }
  }

  /**
   * Return the argument declaratoin, tailored to the specific flavor
   * of FORTRAN.
   *
   * @param a    the argument
   *
   * @return     the argument declaration string
   */
  public String getArgumentDeclaration(Argument a)
    throws CodeGenerationException
  {
    if (d_isF90) {
      return Fortran.getReturnString(a.getType()) + " :: " + a.getFormalName();
    } else {
      return Fortran.getReturnString(a.getType()) + " " + a.getFormalName();
    }
  }

  /**
   * Generate the list of referenced symbols (including implied ones like
   * SIDL.BaseException when the method has exceptions).
   */
  private Set extendedReferences(Method m, SymbolID id)
  {
    Set refs = m.getSymbolReferences();
    HashSet set = new HashSet(refs.size() + 2);
    set.addAll(refs);
    if (!m.getThrows().isEmpty()) {
      Symbol baseException = SymbolTable.getInstance().lookupSymbol
        (BabelConfiguration.getBaseException());
      set.add(baseException.getSymbolID());
    }
    set.add(id);
    return set;
  }

  private static void checkType(Type t, Set result)
    throws CodeGenerationException
  {
    if (Type.ARRAY == t.getDetailedType()) {
      final Type arrayType = t.getArrayType();
      switch (arrayType.getDetailedType()) {
      case Type.ENUM:
      case Type.CLASS:
      case Type.INTERFACE:
      case Type.SYMBOL:
        result.add(arrayType.getSymbolID());
        break;
      case Type.ARRAY:
      case Type.PACKAGE:
        throw new CodeGenerationException
          ("Bad argument type in method");
      case Type.VOID:
        // no action
        break;
      default:
        result.add(new SymbolID("SIDL." + arrayType.getTypeString(),
                                new Version()));
        break;
      }
    }
  }

  private static Set arrayReferences(Method m)
    throws CodeGenerationException
  {
    Collection args = m.getArgumentList();
    HashSet result = new HashSet(args.size());
    Iterator i = args.iterator();
    while (i.hasNext()) {
      final Type t = ((Argument)i.next()).getType();
      checkType(t, result);
    }
    checkType(m.getReturnType(), result);
    return result;
  }

  private void addUseForReferences(Method m,
                                   SymbolID id)
    throws CodeGenerationException
  {
    Collection refs      = extendedReferences(m, id);
    HashSet appeared = new HashSet(refs.size());
    Iterator i = refs.iterator();
    while (i.hasNext()) {
      final SymbolID refid = (SymbolID)i.next();
      if (!appeared.contains(refid)) {
        d_lw.println("use " + Fortran.getModule(refid));
        appeared.add(refid);
      }
    }
    // do array references now
    refs = arrayReferences(m);
    appeared.clear();
    i = refs.iterator();
    while (i.hasNext()) {
      final SymbolID refid = (SymbolID)i.next();
      if (!appeared.contains(refid)) {
        d_lw.println("use " + Fortran.getArrayModule(refid));
        appeared.add(refid);
      }
    }
  }

  /**
   * Write the FORTRAN server side implementation subroutine 
   * that corresponds to a SIDL method. This writes the method
   * signature, declares the type of the argument, and then includes
   * a {@link gov.llnl.babel.backend.CodeSplicer CodeSplicer} section and
   * its contents.
   * 
   * @param m     the method whose implementation template is to be
   *              written.
   * @param id    the name of the class that owns this method.
   * @exception gov.llnl.babel.backend.CodeGenerationException
   *   a catch all exception to indicate problems in the code generation
   *   phase.
   */
  private void extendAndGenerate(Method   m,
                                 SymbolID id)
    throws CodeGenerationException
  {
    List extendedArgs    = StubSource.extendArgs(id, m);
    String splicerTag    = id.getFullName() + '.' + m.getLongMethodName();
    String splicerTagUse = splicerTag + s_use;
    String methodImplName = Fortran.getMethodImplName(id, m, d_mang);
    d_lw.writeComment(m, false);
    if (d_isF90) {
      d_lw.print("recursive ");
    }
    d_lw.print("subroutine " + methodImplName + "(");
    Iterator i = extendedArgs.iterator();
    while (i.hasNext()) {
      d_lw.print(((Argument)i.next()).getFormalName());
      if (i.hasNext()) {
        d_lw.print(", ");
      }
    }
    d_lw.println(")");
    if (d_isF90) {
      d_lw.increaseTabLevel();
      addUseForReferences(m, id);
      d_lw.println("use " + id.getFullName().replace('.','_') + "_impl");
      d_splicer.splice (splicerTagUse, d_lw, "Insert use statements here...");
    }
    d_lw.println("implicit none");
    i = extendedArgs.iterator();
    while (i.hasNext()) {
      Argument a = (Argument)i.next();
      d_lw.println(getArgumentDeclaration(a));
    }
    d_lw.println();
    if (d_isF90) {
      d_lw.decreaseTabLevel();
    }
    d_splicer.splice(splicerTag, d_lw, "Insert the implementation here...");
    if (d_isF90) {
      d_lw.println("end subroutine " + methodImplName);
    } else {
      d_lw.println("end");
    }
  }

  /**
   * Add the builtin methods to the list of methods that must be
   * implemented.
   *
   * @param ext the class whose implementation is being written.
   */
  private Collection extendMethods(Class ext) {
    Collection localMethods = ext.getMethods(false);
    final SymbolID id = ext.getSymbolID();
    ArrayList  extendedMethods = new ArrayList(localMethods.size()+2);
    extendedMethods.
      add(IOR.getBuiltinMethod(IOR.CONSTRUCTOR, id));
    extendedMethods.
      add(IOR.getBuiltinMethod(IOR.DESTRUCTOR, id));
    extendedMethods.addAll(localMethods);
    return extendedMethods;
  }

  private void addMiscCodeSection(String sectionName)
  {
    if (d_splicer.hasSymbol(sectionName)) {
      d_splicer.outputSymbolEdits(sectionName, d_lw.getPrintWriter());
    }
    else {
      d_lw.writeCommentLine(CodeSplicer.getBeginString(sectionName));
      d_lw.writeCommentLine("Insert extra code here...");
      d_lw.writeCommentLine(CodeSplicer.getEndString(sectionName));
    }
  }

  /**
   * Write #include for all the abbreviation files for
   * referenced symbols.
   */
  private void writeIncludes(Class cls)
    throws CodeGenerationException
  {
    Iterator i = ModuleSource.extendedReferences(cls).iterator();
    d_lw.disableLineBreak();
    while (i.hasNext()) {
      SymbolID id = (SymbolID)i.next();
      d_lw.printlnUnformatted("#include \"" +
                              Fortran.getStubNameFile(id) + "\"");
    }
    i = cls.getBasicArrayRefs().iterator();
    while (i.hasNext()) {
      SymbolID id = (SymbolID)i.next();
      d_lw.printlnUnformatted("#include \"" +
                              Fortran.getStubNameFile(id) + "\"");
    }
    d_lw.enableLineBreak();
  }

  /**
   * Generate the implementation FORTRAN file for a SIDL class.  The
   * implementation file contains all the subroutines that need to be
   * implemented for the class, and when replacing an implementation file, 
   * the previous implementation is retained.
   * 
   * @param cls    the SIDL class whose implementation is to be written.
   * @exception gov.llnl.babel.backend.CodeGenerationException
   *   a catch all exception to indicate problems in the code generation
   *   phase.
   */
  public void generateCode(Class cls)
    throws CodeGenerationException
  {
    Collection methods = extendMethods(cls);
    final SymbolID id = cls.getSymbolID();

    d_lw.writeBanner(cls, Fortran.getImplFile(id), true, 
                     CodeConstants.C_DESC_IMPL_PREFIX + id.getFullName());
    d_lw.println();
    d_lw.writeComment(cls, false);
    d_lw.println();

    if (d_isF90) {
      writeIncludes(cls);
    }

    addMiscCodeSection("_miscellaneous_code_start");
    d_lw.println();
    d_lw.println();

    Iterator i = methods.iterator();
    while (i.hasNext()) {
      Method m = (Method)i.next();
      if (!m.isAbstract()) {
        d_lw.println();
        d_lw.println();

        extendAndGenerate(m, id);
      }
    }
    d_lw.println();
    d_lw.println();
    addMiscCodeSection("_miscellaneous_code_end");

    /*
     * Finally, output unreferenced edit blocks, if any.
     */
    if (d_splicer.hasUnusedSymbolEdits()) {
      d_lw.println();
      d_lw.beginBlockComment(true);
      d_lw.println(CodeConstants.C_BEGIN_UNREFERENCED_METHODS);
      d_lw.println(CodeConstants.C_UNREFERENCED_COMMENT1);
      d_lw.println(CodeConstants.C_UNREFERENCED_COMMENT2);
      d_lw.println(CodeConstants.C_UNREFERENCED_COMMENT3);
      d_lw.endBlockComment(true);
      d_splicer.outputUnusedSymbolEdits(d_lw.getPrintWriter()); 
      d_lw.writeCommentLine(CodeConstants.C_END_UNREFERENCED_METHODS);
    }
  }

  /**
   * Generate the implementation FORTRAN file for a SIDL class.  The
   * implementation file contains all the subroutines that need to be
   * implemented for the class, and when replacing an implementation file, 
   * the previous implementation is retained.
   *
   * @exception gov.llnl.babel.backend.CodeGenerationException
   *   a catch all exception to indicate problems in the code generation
   *   phase.
   * @exception java.security.NoSuchAlgorithmException
   *   A problem with the name mangler.
  */
  public static void generateCode(Class cls,
                                  LanguageWriterForFortran writer,
                                  CodeSplicer splicer)
    throws CodeGenerationException, NoSuchAlgorithmException
  {
    ImplSource source = new ImplSource(writer, splicer);
    source.generateCode(cls);
  }
}
