//
// File:        StubSource.java
// Package:     gov.llnl.babel.backend.fortran
// Release:     $Name: release-0-8-8 $
// Revision:    @(#) $Revision: 1.9 $
// Description: Generate code to allow FORTRAN calls to BABEL
//
// Copyright (c) 2000-2003, The Regents of the University of Calfornia.
// Produced at the Lawrence Livermore National Laboratory.
// Written by the Components Team <components@llnl.gov>
// UCRL-CODE-2002-054
// All rights reserved.
//
// This file is part of Babel. For more information, see
// http://www.llnl.gov/CASC/components/. Please read the COPYRIGHT file
// for Our Notice and the LICENSE file for the GNU Lesser General Public
// License.
//
// This program is free software; you can redistribute it and/or modify it
// under the terms of the GNU Lesser General Public License (as published by
// the Free Software Foundation) version 2.1 dated February 1999.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the IMPLIED WARRANTY OF
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the terms and
// conditions of the GNU Lesser General Public License for more details.
//
// You should have recieved a copy of the GNU Lesser General Public License
// along with this program; if not, write to the Free Software Foundation,
// Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA


package gov.llnl.babel.backend.fortran;

import gov.llnl.babel.BabelConfiguration;
import gov.llnl.babel.backend.CodeConstants;
import gov.llnl.babel.backend.CodeGenerationException;
import gov.llnl.babel.backend.IOR;
import gov.llnl.babel.backend.Utilities;
import gov.llnl.babel.backend.writers.LanguageWriter;
import gov.llnl.babel.backend.writers.LanguageWriterForC;
import gov.llnl.babel.backend.writers.LanguageWriterForFortran;
import gov.llnl.babel.symbols.Argument;
import gov.llnl.babel.symbols.Comment;
import gov.llnl.babel.symbols.Enumeration;
import gov.llnl.babel.symbols.Extendable;
import gov.llnl.babel.symbols.Method;
import gov.llnl.babel.symbols.Symbol;
import gov.llnl.babel.symbols.SymbolID;
import gov.llnl.babel.symbols.SymbolTable;
import gov.llnl.babel.symbols.Type;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Set;

/**
 * This class generates the C code that sits between a FORTRAN client and
 * the internal object representation (IOR) of a SIDL object/interface.
 *
 * For each method, this generates a C function that will be called from
 * FORTRAN.  This C function massages the arguments from Fortran, calls
 * the IOR, massages the out values from the IOR call, and returns the
 * outgoing values to the Fortran caller.
 */
public class StubSource {

  /**
   * The argument name that holds the object/interface pointer in
   * a call to an object method. It's conceptually like the this pointer in
   * C++.  In FORTRAN, the object has to be passed explicitly.
   */
  public static final String s_self = "self";

  /**
   * The argument name that holds the return value of a method. The FORTRAN
   * binding treats all return values as an additional out parameter
   * tacked on to the end of the argument list.
   */
  public static final String s_return = "retval";

  /**
   * The argument name that holds the exception pointer which a method may
   * throw an exception.
   */
  public static final String s_exception = "exception";

  /**
   * An <code>#ifdef</code> to check whether character argument should be
   * treated like strings.
   */
  public static final String s_charCheck = "#ifdef " 
                                           + Fortran.getFortranPrefix() 
                                           + "_CHAR_AS_STRING";

  /**
   * This string is prepended to a proxy variable (a variable that takes the
   * place of another variable).  A proxy variable exists when the IOR and
   * Fortran representations are not directly compatible.
   */
  private static final String s_proxy = "_proxy_";

  /**
   * This string is prepended to a proxy variable (a variable that takes the
   * place of another variable).  A proxy variable exists when the IOR and
   * Fortran representations are not directly compatible.
   */
  private static final String s_proxyTwo = "_alt_";

  /**
   * The name of the variable holding the static entry point vector for the
   * class.
   */
  private static final String s_epv = "_epv";

  /**
   * A local cache of the name of the base exception type.
   */
  private static final String s_exceptionType = 
    BabelConfiguration.getBaseException();

  /**
   * This writer controls where the generated C code goes.
   */
  private LanguageWriter d_writer;

  /**
   * Create an object to generate the stub code for a SIDL class/interface.
   * This is frequently called from {@link #generateCode(Symbol,
   * LanguageWriter) generateCode} rather than used directly.
   *
   * @param writer the stub code is generated to this output device.
   */
  public StubSource(LanguageWriter writer) {
    d_writer = writer;
  }

  /**
   * Write a comma and newline to <code>writer</code> iff
   * <code>needComma</code> is  <code>true</code>.  This always returns
   * <code>false</code>.
   *
   * @param writer    the device to which the comma should be output.
   * @param needComma If <code>true</code>, this method will write a
   *                  comma followed by a newline; otherwise, this
   *                  method takes no action.
   * @return <code>false</code> is always returned.
   */
  public static boolean comma(LanguageWriter writer,
                              boolean        needComma)
  {
    if (needComma) {
      writer.println(",");
    }
    return false;
  }

  /**
   * Write an argument declaration in C for an argument being passed in from
   * a FORTRAN caller or from C to a FORTRAN subroutine.
   *
   * @param writer    the place where the code is generated.
   * @param argName   the formal name of the argument.
   * @param argType   the type of the argument.
   * @param needComma whether a comma is needed or not.
   * @return <code>true</code> means a comma is needed before
   *         the next argument; <code>false</code> means a comma
   *         is not needed.
   * @exception gov.llnl.babel.backend.CodeGenerationException
   *    the type of the argument is unsupported.
   */
  public static boolean declareArgument(LanguageWriter writer,
                                        String         argName,
                                        Type           argType,
                                        boolean        needComma)
    throws CodeGenerationException
  {
    switch(argType.getDetailedType()) {
    case Type.VOID:
      // do nothing
      break;
    case Type.DCOMPLEX:
    case Type.DOUBLE:
    case Type.FCOMPLEX:
    case Type.FLOAT:
    case Type.BOOLEAN:
    case Type.ENUM:
    case Type.INT:
    case Type.LONG:
    case Type.CLASS:
    case Type.INTERFACE:
    case Type.ARRAY:
    case Type.OPAQUE:
      writer.print(Fortran.getFortranTypeInC(argType) + " *" + argName);
      needComma = true;
      break;
    case Type.STRING:
      writer.println(Fortran.getFortranPrefix() + "_String " + argName);
      writer.print(Fortran.getFortranPrefix() + "_STR_NEAR_LEN_DECL(" + argName 
                   + ")");
      needComma = true;
      break;
    case Type.CHAR:
      writer.printlnUnformatted(s_charCheck);
      writer.println(Fortran.getFortranPrefix() + "_String " + argName);
      writer.println(Fortran.getFortranPrefix() + "_STR_NEAR_LEN_DECL(" 
                     + argName + ")");
      writer.printlnUnformatted("#else");
      writer.println("char *" + argName);
      writer.printlnUnformatted("#endif");
      needComma = true;
      break;
    default:
      throw new CodeGenerationException
        ("Unsupported Fortran argument type: " + argType.getTypeString() +
         " " + argName);
    }
    return needComma;
  }

  /**
   * Generate the compiler independent form of the function name.
   *
   * @param writer     the place where the symbol is written
   * @param methodName  the potentially mixed case form of the function
   *                    name.
   */
  private static void generateMethodSymbol(LanguageWriter writer,
                                           String         methodName)
  {
    writer.println("void");
    writer.disableLineBreak();
    writer.println(Fortran.getFortranSymbol() + "(" + 
                   methodName.toLowerCase() + ',' +
                   methodName.toUpperCase() + ',' +
                   methodName + ")");
    writer.println("(");
  }

  /**
   * Write declarations for each of the normal arguments for the
   * FORTRAN compatible method declaration.  This writes one argument
   * for each <code>Argument</code> in the iterator. This does not
   * write the extra arguments that are required by the FORTRAN compiler.
   * The FORTRAN compiler may pass string lengths as separate extra
   * argument.
   *
   * @param writer  where the arguments are declared.
   * @param i       a iterator to the collection of {@link
   *                gov.llnl.babel.symbols.Argument Arguments}.
   * @return <code>true</code> if a comma is needed before the
   *         next parameter declaration.
   * @exception gov.llnl.babel.backend.CodeGenerationException
   *    a general catch all exception for problems generating the stub.
   */
  private static boolean declareNormalArguments(LanguageWriter writer,
                                                Iterator       i)
    throws CodeGenerationException
  {
    boolean needComma = false;
    while (i.hasNext()) {
      needComma = comma(writer, needComma);
      Argument a = (Argument)i.next();
      needComma = 
        declareArgument(writer, a.getFormalName(), a.getType(), needComma);
    }
    return needComma;
  }
  
  /**
   * Write declarations for the extra arguments that are required by some
   * FORTRAN compilers for things like string lengths.  There may or may
   * not be any extra arguments.
   *
   * @param writer    where the arguments are declared.
   * @param i         a iterator to the collection of {@link
   *                gov.llnl.babel.symbols.Argument Arguments}.
   * @param needComma whether a comma is needed before the next argument.
   * @return <code>true</code> if a comma is needed before the
   *         next parameter declaration.
   */
  private static boolean declareExtraArguments(LanguageWriter writer,
                                               Iterator       i,
                                               boolean        needComma)
  {
    while (i.hasNext()) {
      Argument a = (Argument)i.next();
      final int t = a.getType().getDetailedType();
      if ((Type.STRING == t) || (Type.CHAR == t)) {
        if (needComma) {
          writer.println();
          needComma = false;
        }
        if (Type.CHAR == t) {
          writer.printlnUnformatted(s_charCheck);
        }
        writer.println(Fortran.getFortranPrefix() + "_STR_FAR_LEN_DECL(" 
                       + a.getFormalName() + ")");
        if (Type.CHAR == t) {
          writer.printlnUnformatted("#endif");
        }
      }
    }
    return needComma;
  }

  /**
   * Generate the C signature for a FORTRAN subroutine to be called from C
   * or for a C function to be called from FORTRAN.  This uses a set of
   * preprocessor macros to handle the conventions of the FORTRAN compiler.
   *
   * @param writer     the place where the signature is written.
   * @param methodName the name of the function.
   * @param argument   a list of {@link gov.llnl.babel.symbols.Argument
   *                   Argument} objects.
   * @exception gov.llnl.babel.backend.CodeGenerationException
   *    something went wrong -- probably an unsupported type.
   */
  public static void generateSignature(LanguageWriter writer,
                                       String         methodName,
                                       List           arguments)
    throws CodeGenerationException
  {
    generateMethodSymbol(writer, methodName);
    if (arguments.isEmpty()) {
      writer.print("void");
    }
    else {
      boolean needComma;
      writer.increaseTabLevel();
      needComma = declareNormalArguments(writer, arguments.iterator());
      needComma =
	declareExtraArguments(writer, arguments.iterator(), needComma);
      
      if (needComma) {
	writer.println();
      }
      writer.decreaseTabLevel();
    }
    writer.print(")");
    writer.enableLineBreak();
  }

  /**
   * Return <code>true</code> iff the type present requires a proxy.  A
   * proxy is required when the FORTRAN types is not directly compatible
   * with the C type for a particular SIDL type.
   *
   * @param t   the SIDL type description
   * @return <code>true</code> means that <code>t</code> requires a proxy;
   *  <code>false</code> means that <code>t</code> does not require a
   * proxy.
   */
  static public boolean hasProxy(Type t) {
    switch (t.getDetailedType()) {
    case Type.BOOLEAN:
    case Type.CHAR:
    case Type.OPAQUE:
    case Type.STRING:
    case Type.CLASS:
    case Type.INTERFACE:
    case Type.ENUM:
    case Type.ARRAY:
      return true;
    default:
      return false;
    }
  }

  /**
   * Return <code>true</code> if a particular SIDL type is implemented using
   * a pointer type.
   * 
   * @param t    the SIDL type description.
   * @return <code>true</code> means the type is implemented using a pointer
   * type; <code>false</code> means the type is not implemented using a
   * pointer type.
   */
  public static boolean isPointer(Type t) {
    switch(t.getDetailedType()) {
    case Type.ARRAY:
    case Type.STRING:
    case Type.OPAQUE:
    case Type.CLASS:
    case Type.INTERFACE:
      return true;
    default:
      return false;
    }
  }

  /**
   * Declare C variables to act as proxies (stand ins) for things whose
   * FORTRAN type is not directly compatible with the IOR.
   *
   * @param arguments a list of <code>Argument</code> objects.
   * @exception gov.llnl.babel.backend.CodeGenerationException
   * a general catch all exception for something going wrong.
   */
  private void declareProxies(List arguments) 
    throws CodeGenerationException
  {
    Iterator i = arguments.iterator();
    while (i.hasNext()) {
      Argument a    = (Argument)i.next();
      Type     t    = a.getType();
      String   name = a.getFormalName();
      if (hasProxy(t)) {
        d_writer.print(IOR.getReturnString(t) + " " + s_proxy + name);
        if (isPointer(t)) {
          d_writer.println(" = NULL;");
        }
        else {
          d_writer.println(";");
        }
      }
    }
  }

  /**
   * Write assignment statements (conceptually speaking) to copy material
   * from the actually incoming parameters to the corresponding proxy
   * variables.  In some cases, copying the incoming values requires
   * something more complex than a simple assignment statement.
   *
   * @param arguments  the list of all parameters.
   * @exception gov.llnl.babel.backend.CodeGenerationException
   *   something went wrong while generating the statements.
   */
  private void copyIncomingValues(List arguments)
    throws CodeGenerationException
  {
    Iterator i = arguments.iterator();
    while (i.hasNext()) {
      Argument a    = (Argument)i.next();
      if (Argument.OUT != a.getMode()) {
        Type     t    = a.getType();
        String   name = a.getFormalName();
        switch (t.getDetailedType()) {
        case Type.BOOLEAN:
          d_writer.println(s_proxy + name + " = ((*" + name + " == " 
                           + Fortran.getFortranPrefix() 
                           + "_TRUE) ? TRUE : FALSE);");
          break;
        case Type.CHAR:
          d_writer.printlnUnformatted(s_charCheck);
          d_writer.println(s_proxy + name + " = *" + Fortran.getFortranPrefix() 
                           + "_STR(" + name + ");");
          d_writer.printlnUnformatted("#else");
          d_writer.println(s_proxy + name + " = *" + name + ";");
          d_writer.printlnUnformatted("#endif");
          break;
        case Type.STRING:
          d_writer.println(s_proxy + name + " =");
          d_writer.increaseTabLevel();
          d_writer.println("SIDL_copy_fortran_str(" + Fortran.getFortranPrefix()
                           + "_STR(" + name + "),");
          d_writer.increaseTabLevel();
          d_writer.println(Fortran.getFortranPrefix() + "_STR_LEN(" + name 
                           + "));");
          d_writer.decreaseTabLevel();
          d_writer.decreaseTabLevel();
          break;
        case Type.ENUM:
          d_writer.println(s_proxy + name + " =");
          d_writer.increaseTabLevel();
          d_writer.println("(" + IOR.getEnumName(t.getSymbolID()) + ")");
          d_writer.println(name + ";");
          d_writer.decreaseTabLevel();
          break;
        case Type.ARRAY:
          d_writer.println(s_proxy + name + " =");
          d_writer.increaseTabLevel();
          d_writer.println("(" + IOR.getReturnString(t) + ")");
          if (Fortran.isFortran90()) {
            d_writer.println("(ptrdiff_t)(" + name + "->d_ior);");
          }
          else {
            d_writer.println("(ptrdiff_t)(*" + name + ");");
          }
          d_writer.decreaseTabLevel();
          break;
        case Type.OPAQUE:
        case Type.CLASS:
        case Type.INTERFACE:
          d_writer.println(s_proxy + name + " =");
          d_writer.increaseTabLevel();
          d_writer.println("(" + IOR.getReturnString(t) + ")");
          d_writer.println("(ptrdiff_t)(*" + name + ");");
          d_writer.decreaseTabLevel();
          break;
        }
      }
    }
  }

  /**
   * Copy values from the proxy variables to the actual parameters.  It is
   * assumed that the parameters are <code>out</code> or <code>inout</code>
   * parameters.
   *
   * @param arguments the list of all arguments.
   * @exception gov.llnl.babel.backend.CodeGenerationException
   *   something went wrong while generating the statements.
   */
  private void copyOutgoingValues(List arguments)
    throws CodeGenerationException
  {
    Iterator i = arguments.iterator();
    while (i.hasNext()) {
      Argument a = (Argument)i.next();
      String name = a.getFormalName();
      if (Argument.IN != a.getMode() && !name.equals(s_exception)) {
        Type   t    = a.getType();
        switch (t.getDetailedType()) {
        case Type.BOOLEAN:
          d_writer.println("*" + name + " = ((" + s_proxy + name
                           + " == TRUE) ? " + Fortran.getFortranPrefix() 
                           + "_TRUE : " + Fortran.getFortranPrefix() 
                           + "_FALSE);");
          break;
        case Type.ARRAY:
          if (Fortran.isFortran90()) {
            if (Fortran.hasDirectAccess(t)) {
              Type dataType = t.getArrayType();
              d_writer.println("if (SIDL_" + dataType.getTypeString() +
                               "__array_convert2f90(" + s_proxy +
                               name + ", " + t.getArrayDimension() + ", " +
                               name + ")) {");
              d_writer.increaseTabLevel();
              d_writer.writeCommentLine("Copy to contiguous column-order");
              d_writer.println(IOR.getReturnString(t) + " " + 
                             s_proxyTwo + name + " =");
              d_writer.increaseTabLevel();
              d_writer.print(Fortran.getEnsureArray(dataType));
              d_writer.println(s_proxy + name + ", " +
                               t.getArrayDimension() + ",");
              d_writer.increaseTabLevel();
              d_writer.println("SIDL_column_major_order);");
              d_writer.decreaseTabLevel();
              d_writer.decreaseTabLevel();
              d_writer.println(Fortran.getDelRefArray(dataType) +
                               s_proxy + name + ");");
              d_writer.println("if (SIDL_" + dataType.getTypeString() +
                               "__array_convert2f90(" + s_proxyTwo +
                               name + ", " + t.getArrayDimension() + ", " +
                               name + ")) {");
              d_writer.increaseTabLevel();
              d_writer.writeCommentLine("We're S.O.L.");
              d_writer.println(
                 "fprintf(stderr, \"convert2f90 failed: %p %d\\n\", (void*)"
                 + s_proxyTwo + name + ", " + t.getArrayDimension() +
                 ");");
              d_writer.println("exit(1); /*NOTREACHED*/");
              d_writer.decreaseTabLevel();
              d_writer.println("}");
              d_writer.decreaseTabLevel();
              d_writer.println("}");
            }
            else {
              d_writer.println(name + "->d_ior = (ptrdiff_t)" + s_proxy +
                               name + ";");
            }
          }
          else {
            d_writer.println("*" + name + " = (ptrdiff_t)" + s_proxy +
                             name + ";");
          }
          break;
        case Type.CLASS:
        case Type.INTERFACE:
        case Type.OPAQUE:
          d_writer.println("*" + name + " = (ptrdiff_t)" + s_proxy +
                           name + ";");
          break;
        case Type.CHAR:
          d_writer.printlnUnformatted(s_charCheck);
          d_writer.println("*" + Fortran.getFortranPrefix() + "_STR(" + name 
                           + ") = " + s_proxy + name + ";");
          d_writer.printlnUnformatted("#else");
          d_writer.println("*" + name + " = " + s_proxy + name + ";");
          d_writer.printlnUnformatted("#endif");
          break;
        case Type.STRING:
          d_writer.println("SIDL_copy_c_str(");
          d_writer.increaseTabLevel();
          d_writer.println(Fortran.getFortranPrefix() + "_STR(" + name + "),");
          d_writer.println(Fortran.getFortranPrefix() + "_STR_LEN(" + name 
                           + "),");
          d_writer.println(s_proxy + name + ");");
          d_writer.decreaseTabLevel();
          break;
        case Type.ENUM:
          d_writer.println("*" + name + " = (int)");
          d_writer.increaseTabLevel();
          d_writer.println(s_proxy + name + ";");
          d_writer.decreaseTabLevel();
          break;
        case Type.SYMBOL:
          throw new CodeGenerationException("Unsupported Type: " +
                                            t.getTypeString());
        }
      }
    }
  }

  /**
   * Write the code to free any resources allocated by the stub function or
   * by the IOR that is no longer needed.
   *
   * @param arguments  the list of arguments for which resources may have
   *                   been allocated.
   */
  private void freeResources(List arguments) {
    Iterator i = arguments.iterator();
    while (i.hasNext()) {
      Argument a    = (Argument)i.next();
      if (Type.STRING == a.getType().getDetailedType()) {
        String name = a.getFormalName();
        d_writer.println("free((void *)" + s_proxy + name + ");");
      }
    }
  }

  /**
   * Write a declaration for the entry point vector pointer.
   *
   * @param isStatic <code>true</code> means the static method
   *                 entry point vector should be used, and
   *                 <code>false</code> means the object entry
   *                 point vector should be used.
   * @param id       the symbol whose EPV or SEPV is needed.
   */
  private void declareEntryPointVector(boolean isStatic, SymbolID id) {
    if (isStatic) {
      d_writer.println("const "
                     + IOR.getSEPVName(id)
                     + " *"
                     + s_epv + " = _getSEPV();");
    } else {
      d_writer.println(IOR.getEPVName(id)
                     + " *"
                     + s_epv
                     + " = NULL;");
    }
  }
  
  /**
   * Generate the _getIOR function that provides access to the IOR functions
   * either through static linking or dynamic loading.
   */
  private void generateGetIOR(SymbolID id) {
    final String ext_name = IOR.getExternalName(id);
    d_writer.writeComment("Return pointer to internal IOR functions.", false);

    d_writer.println("static const " + ext_name + "* _getIOR(void)");
    d_writer.println("{");
    d_writer.increaseTabLevel();
    d_writer.println("static const " + ext_name + " *_ior = NULL;");
    d_writer.println("if (!_ior) {");
    d_writer.increaseTabLevel();
    if (BabelConfiguration.isSIDLBaseClass(id)) {
      d_writer.println("_ior = " + IOR.getExternalFunc(id) + "();");
    } else {
      d_writer.printlnUnformatted("#ifdef SIDL_STATIC_LIBRARY");
      d_writer.println("_ior = " + IOR.getExternalFunc(id) + "();");
      d_writer.printlnUnformatted("#else");
      d_writer.println("const " + ext_name + "*(*dll_f)(void) =");
      d_writer.increaseTabLevel();
      d_writer.print("(const " + ext_name + "*(*)(void)) ");
      d_writer.println("SIDL_Loader_lookupSymbol(");
      d_writer.increaseTabLevel();
      d_writer.println("\"" + IOR.getExternalFunc(id) + "\");");
      d_writer.decreaseTabLevel();
      d_writer.decreaseTabLevel();
      d_writer.println("_ior = (dll_f ? (*dll_f)() : NULL);");
      d_writer.println("if (!_ior) {");
      d_writer.increaseTabLevel();
      d_writer.disableLineBreak();
      d_writer.println(
        "fputs(\"Unable to find the implementation for "
        + id.getFullName() + "; please set SIDL_DLL_PATH\\n\", stderr);");
      d_writer.enableLineBreak();
      d_writer.println("exit(-1);");
      d_writer.decreaseTabLevel();
      d_writer.println("}");
      d_writer.printlnUnformatted("#endif");
    }
    d_writer.decreaseTabLevel();
    d_writer.println("}");
    d_writer.println("return _ior;");
    d_writer.decreaseTabLevel();
    d_writer.println("}");
    d_writer.println();
  }

  /**
   * Return the pointer that provides access to the static EPV.
   */
  private void generateGetStaticEPV(SymbolID id) {
    String sepv_name = IOR.getSEPVName(id);
    d_writer.writeComment("Return pointer to static functions.", false);

    d_writer.println("static const " + sepv_name + "* _getSEPV(void)");
    d_writer.println("{");
    d_writer.increaseTabLevel();
    d_writer.println("static const " + sepv_name + " *_sepv = NULL;");
    d_writer.println("if (!_sepv) {");
    d_writer.increaseTabLevel();
    d_writer.println("_sepv = (*(_getIOR()->getStaticEPV))();");
    d_writer.decreaseTabLevel();
    d_writer.println("}");
    d_writer.println("return _sepv;");
    d_writer.decreaseTabLevel();
    d_writer.println("}");
    d_writer.println();
  }

  /**
   * Write the code to fetch the entry point vector needed for a method
   * call. If the case of a static EPV, it requires a function call, and
   * in the case of an object EPV, it requires dereferencing the IOR
   * pointer.
   *
   * @param isStatic <code>true</code> means it is a static method.
   * @param id       the name of the symbol who's EPV is needed.
   */
  private void getEntryPointVector(boolean isStatic, SymbolID id) {
    if (!isStatic) {
      d_writer.println(s_epv + " = " + s_proxy + s_self + "->d_epv;");
    }
  }

  /**
   * Write an argument to be passed to an IOR method call.  This is a helper
   * function for <code>makeMethodCall</code>.
   *
   * @param a           an Argument.
   * @param isInterface indicate whether then method is being called on
   *                    an object or interface.
   * @param needComma   whether a comma is needed before the next argument.
   * @return <code>true</code> means a comma is needed before the next
   *         argument; <code>false</code> means a comma is not needed.
   */
  private boolean passArgument(Argument a,
                               boolean  isInterface,
                               boolean  needComma)
  {
    String varName = a.getFormalName();
    if (!s_return.equals(varName)) {
      needComma = comma(d_writer, needComma);
      if (hasProxy(a.getType())) {
        if (Argument.IN != a.getMode()) {
          d_writer.print("&");
        }
        d_writer.print(s_proxy + varName);
        if (isInterface && s_self.equals(varName)) {
          d_writer.print("->d_object");
        }
      }
      else {
        if (Argument.IN == a.getMode()) {
          d_writer.print("*");
        }
        d_writer.print(varName);
      }
      needComma = true;
    }
    return needComma;
  }

  /**
   * Write the code to call the IOR.  This uses the original parameters
   * where possible and proxies otherwise.
   *
   * @param arguments   the list of arguments.  This list may have
   *                    implicitly defined arguments like the object, return
   *                    value and exception arguments.
   * @param m           the method to call on the IOR.
   * @param isInterface <code>true</code> means the method call of
   *                    an interface; <code>false</code> means the method
   *                    call to an object.
   * @exception gov.llnl.babel.backend.CodeGenerationException
   *    something bad happened.
   */
  private void makeMethodCall(List    arguments,
                              Method  m,
                              boolean isInterface)
    throws CodeGenerationException
  {
    Type returnType = m.getReturnType();
    Iterator i = arguments.iterator();
    boolean needComma = false;
    if (Type.VOID != returnType.getDetailedType()) {
      if (hasProxy(returnType)) {
        d_writer.print(s_proxy);
      }
      else {
        d_writer.print("*");
      }
      d_writer.println(s_return + " = ");
      d_writer.increaseTabLevel();
    }
    d_writer.println("(*(" + s_epv + "->" + 
                     IOR.getVectorEntry(m.getLongMethodName()) +
                     "))(");
    d_writer.increaseTabLevel();
    while (i.hasNext()) {
      needComma = passArgument((Argument)i.next(), isInterface, needComma);
    }
    d_writer.println();
    d_writer.decreaseTabLevel();
    d_writer.println(");");
    if (Type.VOID != returnType.getDetailedType()) {
      d_writer.decreaseTabLevel();
    }
  }

  /**
   * If a SIDL method, <code>m</code>, can thrown an exception, the stub
   * must check the exception argument. If the exception argument indicates
   * that an exception occured, it must be translated back to FORTRAN and
   * the other argument values should be ignored.
   *
   * @param m add exception handling if this SIDL method can throw an 
   *              exception.
   */
  private void checkExceptionBlock(Method m) {
    if (!m.getThrows().isEmpty()) {
      d_writer.println("if (" + s_proxy + s_exception + ") {");
      d_writer.increaseTabLevel();
      d_writer.println("*" + s_exception + " = (ptrdiff_t)" +
                       s_proxy + s_exception + ";");
      d_writer.decreaseTabLevel();
      d_writer.println("}");
      d_writer.println("else {");
      d_writer.increaseTabLevel();
      d_writer.println("*" + s_exception + " = (ptrdiff_t)NULL;");
    }
  }

  /**
   * Write the end of the exception block if the SIDL method can throw an
   * exception.
   *
   * @param m   close the exception block if this method can throw an
   *            exception.
   */
  private void endExceptionBlock(Method m) {
    if (!m.getThrows().isEmpty()) {
      d_writer.decreaseTabLevel();
      d_writer.println("}");
    }
  }

  /**
   * Generate the stub for a particular method. This generates the
   * signature, declares proxies, translates incoming values, makes the
   * call, and then does outgoing value processing.
   *
   * @param name      the name of the method. This may be different than the
   *                  original name of the method in <code>m</code>..
   * @param arguments the extended list of arguments. This has the argument
   *                  for <code>m</code> with extra arguments as needed for
   *                  the self pointer, the return value, and the exception
   *                  argument.
   * @param m         information about the argument to be written.
   * @param id        the name of the symbol who owns the method.
   * @param isInterface whether the symbol who owns the method is a class
   *                    or an interface.
   * @exception gov.llnl.babel.backend.CodeGenerationException
   *  this exception provides feedback that something went wrong.
   */
  private void generateMethod(String   name, 
                              List     arguments, 
                              Method   m, 
                              SymbolID id,
                              boolean  isInterface) 
    throws CodeGenerationException 
  {
    d_writer.writeComment(m, false);
    generateSignature(d_writer, name, arguments);
    d_writer.println();
    d_writer.println("{");
    d_writer.increaseTabLevel();
    declareEntryPointVector(m.isStatic(), id);
    declareProxies(arguments);
    copyIncomingValues(arguments);
    getEntryPointVector(m.isStatic(), id);
    makeMethodCall(arguments, m, isInterface);
    checkExceptionBlock(m);
    copyOutgoingValues(arguments);
    endExceptionBlock(m);
    freeResources(arguments);
    d_writer.decreaseTabLevel();
    d_writer.println("}");
  }

  /**
   * Add extra arguments to the original argument list of a method as needed
   * for the self pointer, the return value and the exception argument.
   * This makes these implicit arguments explicit and prevents having each
   * of these be a special case throughout the code.
   *
   * @param selfId  the name of the class/interface who owns the method.
   * @param m       the method whose argument list will be extended.
   * @exception gov.llnl.babel.backend.CodeGenerationException
   *  a catch all exception for problems in the code generation phase.
   */
  public static List extendArgs(SymbolID selfId, Method m) 
    throws CodeGenerationException
  {
    final List origArgs = m.getArgumentList();
    ArrayList result = new ArrayList(origArgs.size()+3);
    if (Method.STATIC != m.getDefinitionModifier()) {
      result.add
        (new Argument(false, Argument.IN, new Type(selfId), s_self));
    }
    result.addAll(origArgs);
    if (Type.VOID != m.getReturnType().getDetailedType()) {
      result.add
        (new Argument(false, Argument.OUT, m.getReturnType(), s_return));
    }
    if (!m.getThrows().isEmpty()) {
      Symbol ex = Utilities.lookupSymbol(s_exceptionType);
      result.add(new Argument(false, Argument.OUT, 
                              new Type(ex.getSymbolID()), s_exception));
    }
    return result;
  }

  /**
   * Generate the expanded set of referenced <code>SymbolID</code>'s. This 
   * includes <code>SIDL.BaseException</code> if any of the methods throws
   * an exception.
   *
   * @param ext       the class or interface to generate includes for.
   * @exception gov.llnl.babel.backend.CodeGenerationException
   *   a catch all exception for problems in the code generation phase.
   */
  public static Set extendedReferences(Extendable ext)
   throws CodeGenerationException
  {
    Set result = new HashSet();
    for(Iterator i = ext.getMethods(true).iterator(); i.hasNext(); ) {
      Method method = (Method)i.next();
      result.addAll(method.getSymbolReferences());
      if (!method.getThrows().isEmpty()) {
        Symbol symbol = Utilities.lookupSymbol(s_exceptionType);
        result.add(symbol.getSymbolID());
      }
    }
    return result;
  }

  /**
   * Generate a sequence of <code>#include</code. preprocessor directives
   * required by the stub.
   * 
   * @param writer    the output device where output is sent.
   * @param ext       the class or interface to generate includes for.
   * @exception gov.llnl.babel.backend.CodeGenerationException
   *   a catch all exception for problems in the code generation phase.
   */
  public static void generateIncludes(LanguageWriterForC writer,
                                      Extendable         ext) 
    throws CodeGenerationException
  {
    final SymbolID id = ext.getSymbolID();

    writer.printlnUnformatted("#include <stddef.h>");
    writer.printlnUnformatted("#include <stdlib.h>");
    writer.generateInclude("SIDLfortran.h", false);
    writer.generateInclude("SIDLf90array.h", true);
    writer.generateInclude("SIDL_header.h", false);
    writer.generateInclude("SIDL_interface_IOR.h", true);
    writer.printlnUnformatted("#include <stdio.h>");
    if (!BabelConfiguration.isSIDLBaseClass(id)) {
      writer.printlnUnformatted("#include \"babel_config.h\"");
      writer.printlnUnformatted("#ifdef SIDL_DYNAMIC_LIBRARY");
      writer.generateInclude("SIDL_Loader.h", false);
      writer.printlnUnformatted("#endif");
    }
    writer.generateInclude(IOR.getHeaderFile(id), false);
    if ("f90".equals(BabelConfiguration.getInstance().getTargetLanguage())) {
      writer.generateInclude(Fortran.getStubNameFile(id), false);
    }

    Set includes = extendedReferences(ext);
    includes.remove(id);

    for(Iterator i = includes.iterator(); i.hasNext(); ) {
      writer.generateInclude(IOR.getHeaderFile((SymbolID)i.next()), false);
    }
  }

  /**
   * Generate the extended argument list for a method and then generate the
   * stub code for the method.
   * 
   * @param m           the method to be generated.
   * @param id          the name of the class/interface who owns the method.
   * @param isInterface is the method owned by a class or interface.
   * @exception gov.llnl.babel.backend.CodeGenerationException
   *  a catch all exception for problems in the code generation phase.
   */
  private void extendAndGenerate(Method   m,
                                 SymbolID id,
                                 boolean  isInterface) 
    throws CodeGenerationException
  {
    String name = Fortran.getMethodStubName(id, m);
    List extendedArgs = extendArgs(id, m);
    generateMethod(name, extendedArgs, m, id, isInterface);
  }

  /**
   * Generate the create method for a class.  The create method makes a new
   * wrapped instance of a class.
   *
   * @param id    the name of the class to write a creation method for.
   * @exception gov.llnl.babel.backend.CodeGenerationException
   *   a catch all exception for problems during the code generation phase.
   */
  private void generateCreateMethod(SymbolID id) 
    throws CodeGenerationException
  {
    String methodName;

    if ("f90".equals(BabelConfiguration.getInstance().getTargetLanguage())) {
      methodName = id.getFullName().replace('.', '_') + "_new" 
        + Fortran.getMethodSuffix();
    }
    else {
      methodName = id.getFullName().replace('.', '_') + "__create" 
        + Fortran.getMethodSuffix();
    }
    d_writer.writeComment("Constructor for the class.", false);
    d_writer.println("void");
    d_writer.disableLineBreak();
    d_writer.println(Fortran.getFortranSymbol() + "(" + methodName.toLowerCase()
                     + ',' + methodName.toUpperCase() + ',' + methodName + ")");
    d_writer.enableLineBreak();
    d_writer.println("(");
    d_writer.increaseTabLevel();
    d_writer.println("int64_t *" + s_self);
    d_writer.decreaseTabLevel();
    d_writer.println(")");
    d_writer.println("{");
    d_writer.increaseTabLevel();
    d_writer.println("*self = (ptrdiff_t) (*(_getIOR()->createObject))();");
    d_writer.decreaseTabLevel();
    d_writer.println("}");
    d_writer.println();
  }

  private void generateCast(SymbolID id)
    throws CodeGenerationException
  {
    Method m = StubDoc.createCast(id);
    List extendedArgs = extendArgs(id, m);
    String name = Fortran.getMethodStubName(id, m);
    d_writer.writeComment(m, false);
    generateSignature(d_writer, name, extendedArgs);
    d_writer.println();
    d_writer.println("{");
    d_writer.increaseTabLevel();
    d_writer.println
      ("struct SIDL_BaseInterface__object  *_base =");
    d_writer.increaseTabLevel();
    d_writer.println("(struct SIDL_BaseInterface__object *)(ptrdiff_t)*ref;");
    d_writer.decreaseTabLevel();
    d_writer.println("if (_base) {");
    d_writer.increaseTabLevel();
    d_writer.println("*retval = (ptrdiff_t)(");
    d_writer.increaseTabLevel();
    d_writer.println("*_base->d_epv->" +
                     IOR.getVectorEntry(IOR.getBuiltinName(IOR.CAST))
                     + ")(");
    d_writer.println("_base->d_object,");
    d_writer.println("\"" + id.getFullName() + "\");");
    d_writer.decreaseTabLevel();
    d_writer.decreaseTabLevel();
    d_writer.println("}");
    d_writer.println("else {");
    d_writer.increaseTabLevel();
    d_writer.println("*retval = 0;");
    d_writer.decreaseTabLevel();
    d_writer.println("}");
    d_writer.decreaseTabLevel();
    d_writer.println("}");
    d_writer.println();
  }

  private void generateCastTwo(SymbolID id, boolean isInterface)
    throws CodeGenerationException
  {
    Method m = IOR.getBuiltinMethod(IOR.CAST, id);
    generateMethod(Fortran.getMethodStubName(id, StubDoc.createCastTwo(id)),
                   extendArgs(id, m),
                   m, id, isInterface);
  }

  /**
   * This procedure writes the whole stub file for a class/interface.
   * It writes a banner, writes the includes needed, generates a create
   * method for classes, generates the cast methods, and then creates
   * stubs for all the methods.
   *
   * @param ext   a class/interface for which a stub file will be written.
   * @exception gov.llnl.babel.backend.CodeGenerationException
   *  a catch all exception for problems during the code generation phase.
   */
  private void generateExtendable(Extendable ext)
    throws CodeGenerationException
  {
    Iterator i  = ext.getMethods(true).iterator();
    final SymbolID id = ext.getSymbolID();
    final boolean isInterface = ext.isInterface();

    d_writer.writeBanner(ext, Fortran.getStubFile(id), false, 
                         CodeConstants.C_DESC_STUB_PREFIX + id.getFullName());
    d_writer.writeComment(ext, false);

    generateIncludes((LanguageWriterForC) d_writer, ext);
    d_writer.println();

    if (!ext.isInterface()) {
      generateGetIOR(id);
    }
    if (ext.hasStaticMethod(true)) {
      generateGetStaticEPV(id);
    }

    if (!ext.isAbstract()) {
      generateCreateMethod(id);
    }

    generateCast(id);
    generateCastTwo(id, isInterface);
                         
    while (i.hasNext()) {
      d_writer.println();
      extendAndGenerate((Method)i.next(), id, isInterface);
    }

    d_writer.println();
    FortArrayMethods fam = new FortArrayMethods(id, false);
    fam.generateStub(d_writer);
  }

  /**
   * Generate a FORTRAN include file containing integer constants for the
   * members of an enumerated type.
   * 
   * @param enm an enumeration object to provide an include file for.
   * @exception gov.llnl.babel.backend.CodeGenerationException
   *  a catch all exception to indicate problems during the code generation
   *  phase of the SIDL processing.
   */
  public void generateEnum(Enumeration enm)
    throws CodeGenerationException
  {
    final SymbolID id = enm.getSymbolID();
    if (d_writer instanceof LanguageWriterForFortran) {
      Iterator i;
      d_writer.writeBanner(enm, Fortran.getEnumStubFile(id), false,
                           CodeConstants.C_DESC_STUB_PREFIX + id.getFullName());
      d_writer.println();
      d_writer.writeComment(enm, false);
      d_writer.println();
      i = enm.getEnumerators().iterator();
      while (i.hasNext()){
        String sym = (String)i.next();
        Comment cmt = enm.getEnumeratorComment(sym);
        d_writer.writeComment(cmt, true);
        d_writer.print(Fortran.getReturnString(new Type(id)));
        if ("f90".equals(BabelConfiguration.getInstance().getTargetLanguage()))
        {
          d_writer.print(" :: ");
        } else {
          d_writer.print(" ");
        }
        d_writer.println(sym);
        d_writer.println("parameter (" + sym + " = " + 
                         enm.getEnumeratorValue(sym) + ")");
        if (cmt != null) {
          d_writer.println();
        }
      }
    }
    else {
      d_writer.writeBanner(enm, Fortran.getEnumStubImpl(id), false,
                           CodeConstants.C_DESC_STUB_PREFIX + 
                           id.getFullName());
      ((LanguageWriterForC)d_writer).generateInclude("SIDL_int_IOR.h", true);
      ((LanguageWriterForC)d_writer).generateInclude("SIDLfortran.h", true);
      d_writer.printlnUnformatted("#include <stddef.h>");
      FortArrayMethods fam = new FortArrayMethods(id, true);
      fam.generateStub(d_writer);
    }
  }

  /**
   * Generate a C file to provide FORTRAN stubs for a SIDL
   * object/interface. The stubs allow FORTRAN clients to make calls on
   * objects and interfaces or static methods.  No stub code is generated
   * enumerated types and packages.  Outside clients typically use 
   * {@link #generateCode generateCode} instead of calling this method
   * directly.
   *
   * @param symbol  the symbol for which stubs will be generated.
   * @exception gov.llnl.babel.backend.CodeGenerationException
   *  a catch all exception to indicate problems during the code generation
   *  phase of the SIDL processing.
   */
  public void generateCode(Symbol symbol)
    throws CodeGenerationException
  {
    switch(symbol.getSymbolType()) {
    case Symbol.CLASS:
    case Symbol.INTERFACE:
      if (d_writer instanceof LanguageWriterForC) {
        generateExtendable((Extendable)symbol);
      }
      else {
        throw new CodeGenerationException
          ("Extendable stub requires C language writer.");
      }
      break;
    case Symbol.ENUM:
      generateEnum((Enumeration)symbol);
      break;
    case Symbol.PACKAGE:
      break;
    default:
      throw new CodeGenerationException("Unsupported symbol type.");
    }
  }

  /**
   * Generate a C file to provide FORTRAN stubs for a SIDL
   * object/interface. The stubs allow FORTRAN clients to make calls on
   * objects and interfaces or static methods.  No stub code is generated
   * enumerated types and packages.
   *
   * @param ext     the symbol for which stubs will be generated.
   * @param writer  the output device where the stub should be written.
   * @exception gov.llnl.babel.backend.CodeGenerationException
   *  a catch all exception to indicate problems during the code generation
   *  phase of the SIDL processing.
   */
  public static void generateCode(Symbol         ext,
                                  LanguageWriter writer)
    throws CodeGenerationException
  {
    StubSource source = new StubSource(writer);
    source.generateCode(ext);
  }
}
