//
// File:        ClientJava.java
// Package:     gov.llnl.babel.backend.jdk
// Release:     $Name: release-0-8-8 $
// Revision:    @(#) $Id: ClientJava.java,v 1.6 2003/09/18 14:46:45 epperly Exp $
// Description: write client (stub) code that supports Java clients
//
// Copyright (c) 2000-2003, The Regents of the University of Calfornia.
// Produced at the Lawrence Livermore National Laboratory.
// Written by the Components Team <components@llnl.gov>
// UCRL-CODE-2002-054
// All rights reserved.
// 
// This file is part of Babel. For more information, see
// http://www.llnl.gov/CASC/components/. Please read the COPYRIGHT file
// for Our Notice and the LICENSE file for the GNU Lesser General Public
// License.
// 
// This program is free software; you can redistribute it and/or modify it
// under the terms of the GNU Lesser General Public License (as published by
// the Free Software Foundation) version 2.1 dated February 1999.
// 
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the IMPLIED WARRANTY OF
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the terms and
// conditions of the GNU Lesser General Public License for more details.
// 
// You should have recieved a copy of the GNU Lesser General Public License
// along with this program; if not, write to the Free Software Foundation,
// Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

package gov.llnl.babel.backend.jdk;

import gov.llnl.babel.BabelConfiguration;
import gov.llnl.babel.backend.CodeConstants;
import gov.llnl.babel.backend.CodeGenerationException;
import gov.llnl.babel.backend.Utilities;
import gov.llnl.babel.backend.jdk.Java;
import gov.llnl.babel.backend.writers.LanguageWriterForJava;
import gov.llnl.babel.symbols.Argument;
import gov.llnl.babel.symbols.Class;
import gov.llnl.babel.symbols.Comment;
import gov.llnl.babel.symbols.Enumeration;
import gov.llnl.babel.symbols.Extendable;
import gov.llnl.babel.symbols.Interface;
import gov.llnl.babel.symbols.Method;
import gov.llnl.babel.symbols.Symbol;
import gov.llnl.babel.symbols.SymbolID;
import gov.llnl.babel.symbols.SymbolUtilities;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.Set;

/**
 * Class <code>ClientJava</code> writes the Java native code descriptions that
 * will point to the JNI C code written by <code>ClientJNI</code>.  The class
 * constructor takes a language writer and method <code>generateCode</code>
 * writes the Java client code for the specified symbol to the output stream.
 * The language writer output stream is not closed by this object.
 */
public class ClientJava {
  private final static int INTERFACE = 0;
  private final static int WRAPPER   = 1;
  private final static int CLASS     = 2;

  private LanguageWriterForJava d_writer;

  /**
   * This is a convenience utility function that writes the Java client
   * information into the provided language writer output stream.  The
   * output stream is not closed on exit.  A code generation exception
   * is thrown if an error is detected, such as I/O trouble or a violation
   * of data type invariants.
   */
  public static void generateCode(Symbol symbol, LanguageWriterForJava writer)
      throws CodeGenerationException {
    ClientJava source = new ClientJava(writer);
    source.generateCode(symbol);
  }

  /**
   * Create a <code>ClientJava</code> object that will write symbol
   * information to the provided output language writer stream.
   */
  public ClientJava(LanguageWriterForJava writer) {
    d_writer = writer;
  }

  /**
   * Write Java client information for the provided symbol to the language
   * writer output stream provided in the class constructor.  This method
   * does not close the writer output stream.  Code is currently generated
   * only for SIDL enumerations, interfaces, and classes, since packages do
   * not require JNI support.
   */
  public void generateCode(Symbol symbol) throws CodeGenerationException {
    if (symbol == null) {
      throw new CodeGenerationException("Unexpected null symbol object");
    }
    switch (symbol.getSymbolType()) {
    case Symbol.CLASS:
      generateExtendable((Extendable) symbol);
      break;
    case Symbol.ENUM:
      generateEnumeration((Enumeration) symbol);
      break;
    case Symbol.INTERFACE:
      generateExtendable((Extendable) symbol);
      break;
    case Symbol.PACKAGE:
      // do nothing for a package
      break;
    default:
      throw new CodeGenerationException("Unsupported symbol type");
    }
  }

  /**
   * Generate the Java representation for a SIDL enumerated type.  This
   * method simply outputs a Java interface with a number of final constants.
   */
  private void generateEnumeration(Enumeration enm) {
    SymbolID id = enm.getSymbolID();
    String file = Java.getClientJavaFile(id);

    /*
     * Output the banner and package statement.
     */
    d_writer.writeBanner(enm, file, CodeConstants.C_IS_NOT_IMPL,
                         CodeConstants.C_DESC_STUB_PREFIX + id.getFullName());
    writeJavaPackage(id);

    /*
     * Output the enumerators as integer types within a public interface.
     */
    d_writer.writeComment(enm, true);
    d_writer.println("public interface " + Java.getJavaSymbolName(id) + " {");
    d_writer.increaseTabLevel();
    int maxlength = Utilities.getWidth(enm.getEnumerators());
    for (Iterator e = enm.getIterator(); e.hasNext(); ) {
      String name = (String) e.next();
      Comment cmt = enm.getEnumeratorComment(name);
      d_writer.writeComment(cmt, true);
      d_writer.print("public final static int ");
      d_writer.printAligned(name, maxlength);
      d_writer.print(" = ");
      d_writer.print(String.valueOf(enm.getEnumeratorValue(name)));
      d_writer.println(";");
      if (cmt != null) {
        d_writer.println();
      }
    }
    d_writer.decreaseTabLevel();
    d_writer.println("}");
  }

  /**
   * Generate the Java client source for a SIDL class or interface type.
   * For the most part, the Java source defines the interfaces and classes
   * and the native methods.  All of the real work is done by the JNI code.
   */
  private void generateExtendable(Extendable ext)
      throws CodeGenerationException {
    SymbolID id = ext.getSymbolID();
    String file = Java.getClientJavaFile(id);

    /*
     * Output the banner, package statement, and comment.
     */
    d_writer.writeBanner(ext, file, CodeConstants.C_IS_NOT_IMPL,
                         CodeConstants.C_DESC_STUB_PREFIX + id.getFullName());
    writeJavaPackage(id);
    d_writer.writeComment(ext, true);

    /*
     * If this is an interface, then output all the methods in the interface
     * and generate the interface wrapper inner class that defines the native
     * methods that call the IOR.
     */
    if (ext.isInterface()) {
      /*
       * First output the interface definition with all parent interfaces.
       * All interface inherit from the Java base interface as well as all
       * other SIDL interfaces.
       */
      d_writer.println("public interface "
                     + Java.getJavaSymbolName(id)
                     + " extends");
      d_writer.increaseTabLevel();

      d_writer.print(Java.getJavaBaseInterface());
      Collection parents = ext.getParentInterfaces(false);
      if ((parents == null) || parents.isEmpty()) {
        d_writer.println();
      } else {
        d_writer.println(",");
        for (Iterator p = parents.iterator(); p.hasNext(); ) {
          Interface i = (Interface) p.next();
          d_writer.print(Java.getFullJavaSymbolName(i.getSymbolID()));
          if (p.hasNext()) {
            d_writer.print(",");
          }
          d_writer.println();
        }
      }

      d_writer.decreaseTabLevel();
      d_writer.println("{");
      d_writer.increaseTabLevel();

      /*
       * Output the interface methods in sorted order.  All methods are
       * abstract.
       */
      List local_methods = (List) ext.getMethods(false);
      for (Iterator m = local_methods.iterator(); m.hasNext(); ) {
        Method method = (Method) m.next();
        generateMethod(method, INTERFACE);
        d_writer.println();
      }

      /*
       * Now output the wrapper class that implements the methods via JNI.
       * Begin with a brief documentation blurb.  Make sure to extend the
       * SIDL Java base class and implmenet the outer interface.
       */
      d_writer.writeComment(
        "Inner wrapper class that implements interface methods.", true);
      d_writer.println("public class " + Java.getInterfaceWrapper());
      d_writer.increaseTabLevel();
      d_writer.println("extends " + Java.getJavaBaseClass());
      d_writer.println("implements " + Java.getFullJavaSymbolName(id));
      d_writer.decreaseTabLevel();
      d_writer.println("{");
      d_writer.increaseTabLevel();

      writeRegisterFunction(id);

      /*
       * Output the class constructor that initializes the base object.
       */
      d_writer.writeComment("Class constructor for the wrapper class.", true);
      d_writer.println("public "
                     + Java.getInterfaceWrapper()
                     + "(long ior) {");
      d_writer.increaseTabLevel();
      d_writer.println("super(ior);");
      d_writer.decreaseTabLevel();
      d_writer.println("}");
      d_writer.println();

      /*
       * Output the methods in WRAPPER mode.  We need to iterate over all the
       * methods in the interface since we will implement all of them.
       */
      List all_methods = (List) ext.getMethods(true);
      for (Iterator m = all_methods.iterator(); m.hasNext(); ) {
        Method method = (Method) m.next();
        generateMethod(method, WRAPPER);
        if (m.hasNext()) {
          d_writer.println();
        }
      }

      d_writer.decreaseTabLevel();
      d_writer.println("}");
      d_writer.println();

      /*
       * Output the holder class and finish the interface declaration.
       */
      writeHolderClass(id);
      d_writer.decreaseTabLevel();
      d_writer.println("}");

    /*
     * If this is a class, then output the methods in the class as native
     * implementations that call IOR routines.
     */
    } else {
      Class cls = (Class) ext;

      /*
       * Output the class documentation and the inheritance information.
       * Note that we must declare the class as abstract if any of the methods
       * are abstract.  If the class does not have a parent, then it must
       * inherit from the SIDL Java base class.
       */
      d_writer.print("public ");
      if (ext.isAbstract()) {
        d_writer.print("abstract ");
      }
      d_writer.println("class " + Java.getJavaSymbolName(id) + " extends");
      d_writer.increaseTabLevel();

      Class c = cls.getParentClass();
      if (c == null) {
        d_writer.println(Java.getJavaBaseClass() + " implements");
      } else {
        d_writer.println(Java.getFullJavaSymbolName(c.getSymbolID())
                       + " implements");
      }

      Collection parents = ext.getParentInterfaces(true);
      for (Iterator p = parents.iterator(); p.hasNext(); ) {
        Interface i = (Interface) p.next();
        d_writer.print(Java.getFullJavaSymbolName(i.getSymbolID()));
        if (p.hasNext()) {
          d_writer.print(",");
        }
        d_writer.println();
      }

      d_writer.decreaseTabLevel();
      d_writer.println("{");
      d_writer.increaseTabLevel();

      writeRegisterFunction(id);

      /*
       * If not abstract, then output a private native method that returns
       * an IOR reference and also a zero-argument default constructor.
       */
      if (!cls.isAbstract()) {
        d_writer.writeComment("Default constructor for the class.", true);
        d_writer.println("public " + Java.getJavaSymbolName(id) + "() {");
        d_writer.increaseTabLevel();
        d_writer.println("super(_create_ior());");
        d_writer.decreaseTabLevel();
        d_writer.println("}");
        d_writer.println();

        d_writer.writeComment("Private method to create IOR reference.", true);
        d_writer.println("private static native long _create_ior();");
        d_writer.println();
      }

      /*
       * Output the class constructor that initializes the base object.
       */
      d_writer.writeComment("Public constructor for an existing IOR.", true);
      d_writer.println("public "
                     + Java.getJavaSymbolName(id)
                     + "(long ior) {");
      d_writer.increaseTabLevel();
      d_writer.println("super(ior);");
      d_writer.decreaseTabLevel();
      d_writer.println("}");
      d_writer.println();

      /*
       * Output all of the methods in CLASS mode.
       */
      List local_methods = (List) ext.getMethods(false);
      for (Iterator m = local_methods.iterator(); m.hasNext(); ) {
        Method method = (Method) m.next();
        generateMethod(method, CLASS);
        d_writer.println();
      }

      /*
       * Write a holder class for inout and out arguments and close the class
       * scope.
       */
      writeHolderClass(id);
      d_writer.decreaseTabLevel();
      d_writer.println("}");
    }
  }

  /**
   * Write the java package information followed by an additional newline.
   */
  private void writeJavaPackage(SymbolID id) {
    String pkg = SymbolUtilities.getParentPackage(id.getFullName());
    d_writer.println("package " + pkg + ";");
    d_writer.println();
  }

  /**
   * Output the static block that will dynamically load the JNI implementation.
   * This block calls the <code>_registerNatives</code> function in the SIDL
   * run-time Java base class.
   */
  private void writeRegisterFunction(SymbolID id) {
    d_writer.writeComment("Register the JNI native routines.", false);
    d_writer.println("static {");
    d_writer.increaseTabLevel();
    d_writer.println(Java.getJavaBaseClass()
                   + "._registerNatives(\""
                   + id.getFullName()
                   + "\");");
    d_writer.decreaseTabLevel();
    d_writer.println("}");
    d_writer.println();
  }

  /**
   * Write an inner holder class used for out and inout arguments.  The holder
   * class has a single argument - the Java object - which may be get or set
   * by accessor methods.
   */
  private void writeHolderClass(SymbolID id) {
    String holder = Java.getHolderName();
    String holdee = Java.getFullJavaSymbolName(id);

    d_writer.writeComment("Holder class for inout and out arguments.", true);
    d_writer.println("public class " + holder + " {");
    d_writer.increaseTabLevel();
    d_writer.println("private " + holdee + " d_obj;");
    d_writer.println();

    d_writer.writeComment("Create a holder with a null holdee object.", true);
    d_writer.println("public " + holder + "() {");
    d_writer.increaseTabLevel();
    d_writer.println("d_obj = null;");
    d_writer.decreaseTabLevel();
    d_writer.println("}");
    d_writer.println();

    d_writer.writeComment("Create a holder with the specified object.", true);
    d_writer.println("public " + holder + "(" + holdee + " obj) {");
    d_writer.increaseTabLevel();
    d_writer.println("d_obj = obj;");
    d_writer.decreaseTabLevel();
    d_writer.println("}");
    d_writer.println();

    d_writer.writeComment("Set the value of the holdee object.", true);
    d_writer.println("public void set(" + holdee + " obj) {");
    d_writer.increaseTabLevel();
    d_writer.println("d_obj = obj;");
    d_writer.decreaseTabLevel();
    d_writer.println("}");
    d_writer.println();

    d_writer.writeComment("Get the value of the holdee object.", true);
    d_writer.println("public " + holdee + " get() {");
    d_writer.increaseTabLevel();
    d_writer.println("return d_obj;");
    d_writer.decreaseTabLevel();
    d_writer.println("}");

    d_writer.decreaseTabLevel();
    d_writer.println("}");
  }

  /**
   * Generate the method descriptions for a SIDL method.  The mode argument
   * determines how the method is to be created.  If INTERFACE, then the method
   * is from an interface description and the method is abstract.  If WRAPPER,
   * then the method is from an interface wrapper and all the methods are
   * implemented as native methods using JNI.  If CLASS, then the method is
   * from a class and non-abstract methods are implemented using native calls.
   */
  private void generateMethod(Method method, int mode) {
    d_writer.writeComment(method, true);

    /*
     * Output the method visibility, modifier (abstract, final, or static), and
     * whether it is a native method.
     */
    d_writer.print("public ");
    if ((mode != WRAPPER) && method.isAbstract()) {
      d_writer.print("abstract ");
    } else if (method.isFinal()) {
      d_writer.print("final ");
    } else if (method.isStatic()) {
      d_writer.print("static ");
    }
    if ((mode == WRAPPER) || !method.isAbstract()) {
      d_writer.print("native ");
    }

    /*
     * Output the return type and method name and begin the argument list.
     */
//ToDo...Make sure this is supposed to be the short, not long, name...
    d_writer.print(Java.getJavaReturnType(method.getReturnType())
                 + " "
                 + method.getShortMethodName()
                 + "(");

    /*
     * Output the argument list and a throws clause.  The logic is a little
     * tricky to make the output look pretty.
     */
    List args = method.getArgumentList();
    Set thrws = method.getThrows();

    if (args.isEmpty() && thrws.isEmpty()) {
      d_writer.println(");");
    } else {
      if (!args.isEmpty()) {
        d_writer.println();
        d_writer.increaseTabLevel();
        for (Iterator a = args.iterator(); a.hasNext(); ) {
          Argument arg = (Argument) a.next();
          d_writer.print(Java.getJavaFormalArgument(arg));
          if (a.hasNext()) {
            d_writer.println(",");
          }
        }
      }
      if (thrws.isEmpty()) {
        d_writer.println(");");
      } else {
        d_writer.println(") throws");
        if (args.isEmpty()) {
          d_writer.increaseTabLevel();
        }
        for (Iterator t = thrws.iterator(); t.hasNext(); ) {
          SymbolID tid = (SymbolID) t.next();
          d_writer.print(Java.getFullJavaSymbolName(tid));
          d_writer.println(t.hasNext() ? "," : ";");
        }
      }
      d_writer.decreaseTabLevel();
    }
  }
}
