//
// File:        PythonServerLaunch.java
// Package:     gov.llnl.babel.backend.python
// Release:     $Name: release-0-8-8 $
// Revision:    @(#) $Revision: 1.2 $
// Date:        $Date: 2002/08/20 16:55:31 $
// Description: Initialize Python before loading the real implementation
// 
// Copyright (c) 2000-2001, The Regents of the University of Calfornia.
// Produced at the Lawrence Livermore National Laboratory.
// Written by the Components Team <components@llnl.gov>
// UCRL-CODE-2002-054
// All rights reserved.
// 
// This file is part of Babel. For more information, see
// http://www.llnl.gov/CASC/components/. Please read the COPYRIGHT file
// for Our Notice and the LICENSE file for the GNU Lesser General Public
// License.
// 
// This program is free software; you can redistribute it and/or modify it
// under the terms of the GNU Lesser General Public License (as published by
// the Free Software Foundation) version 2.1 dated February 1999.
// 
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the IMPLIED WARRANTY OF
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the terms and
// conditions of the GNU Lesser General Public License for more details.
// 
// You should have recieved a copy of the GNU Lesser General Public License
// along with this program; if not, write to the Free Software Foundation,
// Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

package gov.llnl.babel.backend.python;

import gov.llnl.babel.backend.CodeGenerationException;
import gov.llnl.babel.backend.IOR;
import gov.llnl.babel.backend.writers.LanguageWriterForC;
import gov.llnl.babel.symbols.Class;
import gov.llnl.babel.symbols.SymbolID;

public class PythonServerLaunch {
  LanguageWriterForC d_lw = null;
  Class d_cls;
  boolean d_hasStatic;

  PythonServerLaunch(Class cls)
  {
    d_cls = cls;
    d_hasStatic = d_cls.hasStaticMethod(false);
  }

  private void writeIncludes()
  {
    d_lw.generateInclude("SIDL_Python.h", true);
    d_lw.generateInclude("SIDL_Loader.h", true);
    d_lw.println();
  }

  private void writeForwardDecls()
  {
    final SymbolID id = d_cls.getSymbolID();
    d_lw.println(IOR.getEPVName(id) + ";");
    if (d_hasStatic) {
      d_lw.println(IOR.getSEPVName(id) + ";");
    }
    d_lw.println();
  }

  private void writeStaticVars()
  {
    final SymbolID id = d_cls.getSymbolID();
    d_lw.println("static void (*s_getEPV)");
    d_lw.increaseTabLevel();
    d_lw.println("(" + IOR.getEPVName(id) + " *);");
    d_lw.decreaseTabLevel();
    if (d_hasStatic) {
      d_lw.println("static void (*s_getSEPV)");
      d_lw.increaseTabLevel();
      d_lw.println("(" + IOR.getSEPVName(id) + " *);");
      d_lw.decreaseTabLevel();
    }
    d_lw.println();
  }
  
  private void writeInit()
  {
    final SymbolID id = d_cls.getSymbolID();
    d_lw.println("static void");
    d_lw.println("initImplementation(void) {");
    d_lw.increaseTabLevel();
    d_lw.println("static int notInitialized = 1;");
    d_lw.println("if (notInitialized) {");
    d_lw.increaseTabLevel();
    d_lw.println("notInitialized = 0;");
    d_lw.println("SIDL_Python_Init();");
    d_lw.println("s_getEPV = (void (*)(" + IOR.getEPVName(id) + " *))");
    d_lw.increaseTabLevel();
    d_lw.println("SIDL_Loader_lookupSymbol(\"" +
                 Python.getSetEPVName(d_cls.getSymbolID()) + "\");");
    d_lw.decreaseTabLevel();
    if (d_hasStatic) {
      d_lw.increaseTabLevel();
      d_lw.println("s_getSEPV = (void (*)(" + IOR.getSEPVName(id) + " *))");
      d_lw.println("SIDL_Loader_lookupSymbol(\"" +
                   Python.getSetSEPVName(d_cls.getSymbolID()) + "\");");
      d_lw.decreaseTabLevel();
    }
    d_lw.decreaseTabLevel();
    d_lw.println("}");
    d_lw.decreaseTabLevel();
    d_lw.println("}");
    d_lw.println();
  }

  private void writeGetEPV()
  {
    final SymbolID id = d_cls.getSymbolID();
    d_lw.println("void");
    d_lw.println(IOR.getSetEPVName(id) + "(" +
                 IOR.getEPVName(id) + " *epv) {");
    d_lw.increaseTabLevel();
    d_lw.println("initImplementation();");
    d_lw.println("if (s_getEPV) (*s_getEPV)(epv);");
    d_lw.decreaseTabLevel();
    d_lw.println("}");
    d_lw.println();
  }

  private void writeGetSEPV()
  {
    final SymbolID id = d_cls.getSymbolID();
    d_lw.println("void");
    d_lw.println(IOR.getSetSEPVName(id) + "(" +
                 IOR.getSEPVName(id) + " *sepv) {");
    d_lw.increaseTabLevel();
    d_lw.println("initImplementation();");
    d_lw.println("if (s_getSEPV) (*s_getSEPV)(sepv);");
    d_lw.decreaseTabLevel();
    d_lw.println("}");
    d_lw.println();
  }

  public synchronized void generateCode()
    throws CodeGenerationException
  {
    try {
      d_lw = Python.createLaunch(d_cls, "launch the implementation");
      writeIncludes();
      writeForwardDecls();
      writeStaticVars();
      writeInit();
      writeGetEPV();
      if (d_hasStatic) {
        writeGetSEPV();
      }
    }
    finally {
      if (d_lw != null) {
        d_lw.close();
        d_lw = null;
      }
    }
  }
}
