//
// File:        PythonServerPySource.java
// Package:     gov.llnl.babel.backend.pythong
// Release:     $Name: release-0-8-8 $
// Revision:    @(#) $Revision: 1.12 $
// Date:        $Date: 2002/08/20 16:55:31 $
// Description: Generate Python implementation file in Python
// 
// Copyright (c) 2000-2001, The Regents of the University of Calfornia.
// Produced at the Lawrence Livermore National Laboratory.
// Written by the Components Team <components@llnl.gov>
// UCRL-CODE-2002-054
// All rights reserved.
// 
// This file is part of Babel. For more information, see
// http://www.llnl.gov/CASC/components/. Please read the COPYRIGHT file
// for Our Notice and the LICENSE file for the GNU Lesser General Public
// License.
// 
// This program is free software; you can redistribute it and/or modify it
// under the terms of the GNU Lesser General Public License (as published by
// the Free Software Foundation) version 2.1 dated February 1999.
// 
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the IMPLIED WARRANTY OF
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the terms and
// conditions of the GNU Lesser General Public License for more details.
// 
// You should have recieved a copy of the GNU Lesser General Public License
// along with this program; if not, write to the Free Software Foundation,
// Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

package gov.llnl.babel.backend.python;

import gov.llnl.babel.backend.CodeConstants;
import gov.llnl.babel.backend.CodeGenerationException;
import gov.llnl.babel.backend.CodeSplicer;
import gov.llnl.babel.backend.IOR;
import gov.llnl.babel.backend.Utilities;
import gov.llnl.babel.backend.python.Python;
import gov.llnl.babel.backend.writers.LanguageWriter;
import gov.llnl.babel.backend.writers.LanguageWriterForPython;
import gov.llnl.babel.symbols.Argument;
import gov.llnl.babel.symbols.Class;
import gov.llnl.babel.symbols.Comment;
import gov.llnl.babel.symbols.Extendable;
import gov.llnl.babel.symbols.Method;
import gov.llnl.babel.symbols.Symbol;
import gov.llnl.babel.symbols.SymbolID;
import gov.llnl.babel.symbols.Type;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Set;

/**
 * The purpose of this class is to generate Python source code to implement
 * a SIDL class. Parts of the Python source code must be filled by the
 * developer, and these parts are retained using a <code>CodeSplicer</code>.
 *
 * Each locally defined method is partially implemented in this file.
 */
public class PythonServerPySource {
  /**
   * A write for the Python source file.
   */
  private LanguageWriterForPython d_lw = null;

  /**
   * This stores code segments from the previous impl file.
   */
  private CodeSplicer d_splicer = null;

  /**
   * The class whose implementation is being written.
   */
  private Class d_cls = null;
  
  public PythonServerPySource(Class cls)
  {
    d_cls = cls;
  }

  private void addSplicerBlock(String name, boolean addPass)
  {
    if (d_splicer.hasSymbol(name)) {
      d_splicer.outputSymbolEdits(name, d_lw.getPrintWriter());
    }
    else {
      d_lw.writeCommentLine(CodeSplicer.getBeginString(name));
      d_lw.writeCommentLine("Put your code here...");
      if (addPass) {
        d_lw.println("pass");
      }
      d_lw.writeCommentLine(CodeSplicer.getEndString(name));
    }
  }

  private void importReferences()
  {
    Iterator i = d_cls.getSymbolReferences().iterator();
    while (i.hasNext()) {
      SymbolID id = (SymbolID)i.next();
      d_lw.println("import " + id.getFullName());
    }
    d_lw.println();
  }

  private void addDocComment(Comment comment)
  {
    if (comment != null){
      final String[] text = comment.getComment();
      if (text != null && (text.length > 0)) {
        d_lw.println("\"\"\"\\");
        for(int i = 0; i < text.length; ++i) {
          d_lw.printlnUnformatted(LanguageWriterForPython.
                                  toPythonString(text[i]));
        }
        d_lw.printlnUnformatted("\"\"\"");
      }
    }
  }

  private void addExpectedIncomingComment(List argList)
  {
    if (argList.size() > 0){
      Iterator args = argList.iterator();
      d_lw.beginBlockComment(false);
      d_lw.println("SIDL EXPECTED INCOMING TYPES");
      d_lw.println("============================");
      while (args.hasNext()) {
        Argument arg= (Argument)args.next();
        if (Argument.OUT != arg.getMode()) {
          d_lw.println(arg.getType().getTypeString() + " " +
                       arg.getFormalName());
        }
      }
      d_lw.endBlockComment(false);
    }
  }

  private void addExpectedReturnComment(Type returnType, List argList)
  {
    List returnTypes = TranslateArguments.extractOut(returnType, argList);
    final int size = returnTypes.size();
    d_lw.beginBlockComment(false);
    d_lw.println("SIDL EXPECTED RETURN VALUE(s)");
    d_lw.println("=============================");
    if (size > 0) {
      Iterator args;
      if (size > 1) {
        args = returnTypes.iterator();
        d_lw.print("(" + ((Argument)args.next()).getFormalName());
        while (args.hasNext()) {
          d_lw.print(", " + ((Argument)args.next()).getFormalName());
        }
        d_lw.println(")");
      }
      args = returnTypes.iterator();
      while (args.hasNext()){
        Argument arg = (Argument)args.next();
        d_lw.println(arg.getType().getTypeString() + " " +
                     arg.getFormalName());
      }
    }
    else {
      d_lw.writeCommentLine("None");
    }
    d_lw.endBlockComment(false);
  }

  private void writeMethodImpl(Method m)
  {
    List argList = m.getArgumentList();
    boolean needComma = false;
    Iterator args = argList.iterator();
    d_lw.print("def " + m.getLongMethodName() + "(");
    if (Method.STATIC != m.getDefinitionModifier()) {
      d_lw.print("self");
      needComma = true;
    }
    while (args.hasNext()) {
      Argument arg = (Argument)args.next();
      if (Argument.OUT != arg.getMode()) {
        if (needComma) {
          d_lw.print(", ");
        }
        else {
          needComma = true;
        }
        d_lw.print(arg.getFormalName());
      }
    }
    d_lw.println("):");
    d_lw.increaseTabLevel();
    addExpectedIncomingComment(argList);
    addExpectedReturnComment(m.getReturnType(), argList);
    addDocComment(m.getComment());
    addSplicerBlock(m.getLongMethodName(), true);
    d_lw.decreaseTabLevel();
    d_lw.println();
  }

  private void writeStaticMethods()
  {
    // Get a list of static methods
    Iterator i = d_cls.getStaticMethods(false).iterator();
    while (i.hasNext()) {
      Method m = (Method)i.next();
      writeMethodImpl(m);
    }
  }

  private void writeClass() {
    d_lw.println("class " + d_cls.getSymbolID().getShortName() + ":");
    d_lw.increaseTabLevel();
    addDocComment(d_cls.getComment());
    d_lw.println();
    d_lw.writeCommentLine("All calls to SIDL methods should use __IORself");
    d_lw.println();
    d_lw.println("def __init__(self, IORself):");
    d_lw.increaseTabLevel();
    d_lw.println("self.__IORself = IORself");
    addSplicerBlock("__init__", false);
    d_lw.decreaseTabLevel();
    d_lw.println();
    Iterator i = d_cls.getNonstaticMethods(false).iterator();
    while (i.hasNext()) {
      Method m = (Method)i.next();
      if (!m.isAbstract()) {
        writeMethodImpl(m);
      }
    }
    d_lw.decreaseTabLevel();
  }
  
  public synchronized void generateCode()
    throws CodeGenerationException
  {
    try {
      final String filename = Python.implFilename(d_cls);
      d_splicer = Python.getPySplicer(d_cls, filename);
      d_lw = Python.createPyWriter(d_cls, filename,
                                   "Implementation of SIDL class " + 
                                   d_cls.getFullName() + 
                                   " in Python.");
      addSplicerBlock("_initial", false);
      d_lw.println();
      importReferences();
      if (d_cls.hasStaticMethod(false)) {
        addSplicerBlock("_before_static", false);
        d_lw.println();
        writeStaticMethods();
      }
      addSplicerBlock("_before_type", false);
      d_lw.println();
      writeClass();
      addSplicerBlock("_final", false);
      if (d_splicer.hasUnusedSymbolEdits()) {
        d_lw.println();
        d_lw.println("pass");
        d_lw.beginBlockComment(true);
        d_lw.println(CodeConstants.C_BEGIN_UNREFERENCED_METHODS);
        d_lw.println(CodeConstants.C_UNREFERENCED_COMMENT1);
        d_lw.println(CodeConstants.C_UNREFERENCED_COMMENT2);
        d_lw.println(CodeConstants.C_UNREFERENCED_COMMENT3);
        d_lw.endBlockComment(true);
        d_splicer.outputUnusedSymbolEdits(d_lw.getPrintWriter()); 
        d_lw.writeCommentLine(CodeConstants.C_END_UNREFERENCED_METHODS);
      }
    }
    catch (IOException ioe) {
      throw new CodeGenerationException("Unable to get code splicer");
    }
    finally {
      if (d_lw != null) {
        d_lw.close();
        d_lw = null;
      }
    }
  }
}
