//
// File:        Extendable.java
// Package:     gov.llnl.babel.symbols
// Release:     $Name: release-0-8-8 $
// Revision:    @(#) $Id: Extendable.java,v 1.11 2003/03/18 17:46:27 epperly Exp $
// Description: SIDL base symbol for both classes and interfaces
//
// Copyright (c) 2000-2001, The Regents of the University of Calfornia.
// Produced at the Lawrence Livermore National Laboratory.
// Written by the Components Team <components@llnl.gov>
// UCRL-CODE-2002-054
// All rights reserved.
// 
// This file is part of Babel. For more information, see
// http://www.llnl.gov/CASC/components/. Please read the COPYRIGHT file
// for Our Notice and the LICENSE file for the GNU Lesser General Public
// License.
// 
// This program is free software; you can redistribute it and/or modify it
// under the terms of the GNU Lesser General Public License (as published by
// the Free Software Foundation) version 2.1 dated February 1999.
// 
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the IMPLIED WARRANTY OF
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the terms and
// conditions of the GNU Lesser General Public License for more details.
// 
// You should have recieved a copy of the GNU Lesser General Public License
// along with this program; if not, write to the Free Software Foundation,
// Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

package gov.llnl.babel.symbols;

import gov.llnl.babel.symbols.Comment;
import gov.llnl.babel.symbols.Interface;
import gov.llnl.babel.symbols.Metadata;
import gov.llnl.babel.symbols.Method;
import gov.llnl.babel.symbols.Symbol;
import gov.llnl.babel.symbols.SymbolID;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.ListIterator;
import java.util.Map;
import java.util.Set;

/**
 * Class <code>Extendable</code> is a base class for SIDL symbols of type
 * class and interface.  It brings together common methods and implementation
 * for both final classes.  Class and interfaces have a set of methods and
 * interface inheritance in common.  Classes have the additional property
 * that they can be extended by another class; that functionality is not
 * defined here.  Many of the member functions take a boolean argument that
 * selects whether the method refers to this particular extendable only or
 * to this extendable and all of its parents in the SIDL inheritance system.
 * Constraints on the validity of methods added to this object must be checked
 * by the parser.
 *
 * Key design goals include providing mechanisms to:
 * - ensure method lookups in O(1) by both long and short names;
 * - ensure fast return of abstract, static, non-static, local, and both
 *   local and parent (or all) methods;
 * - ensure original ordering of methods preserved when returning lists;
 */
public abstract class Extendable extends Symbol {
  private ArrayList d_all_abstract_methods;
  private HashMap   d_all_interfaces;
  private ArrayList d_all_methods;
  private HashMap   d_all_methods_long;
  private HashMap   d_all_methods_short;
  private HashSet   d_all_references;
  private HashSet   d_all_basicarrays;
  private ArrayList d_all_nonstatic_methods;
  private ArrayList d_all_static_methods;

  private HashMap   d_local_interfaces;
  private ArrayList d_local_methods;
  private HashMap   d_local_methods_long;
  private HashMap   d_local_methods_short;
  private ArrayList d_local_nonstatic_methods;
  private ArrayList d_local_static_methods;


  /**
   * Create an empty <code>Extendable</code> object that will be constructed
   * by calls to other member functions.
   */
  public Extendable(SymbolID id, int type, Comment comment) {
    super(id, type, comment);
    d_all_abstract_methods    = new ArrayList();
    d_all_interfaces          = new HashMap();
    d_all_methods             = new ArrayList();
    d_all_methods_long        = new HashMap();
    d_all_methods_short       = new HashMap();
    d_all_references          = new HashSet();
    d_all_basicarrays         = new HashSet();
    d_all_nonstatic_methods   = new ArrayList();
    d_all_static_methods      = new ArrayList();

    d_local_interfaces        = new HashMap();
    d_local_methods           = new ArrayList();
    d_local_methods_long      = new HashMap();
    d_local_methods_short     = new HashMap();
    d_local_nonstatic_methods = new ArrayList();
    d_local_static_methods    = new ArrayList();

    d_all_references.add(id);
  }


  /**
   * Create an empty <code>Extendable</code> object that will be constructed
   * by calls to other member functions.
   */
  public Extendable(SymbolID id, int type, Comment comment, Metadata m) {
    super(id, type, comment, m);
    d_all_abstract_methods    = new ArrayList();
    d_all_interfaces          = new HashMap();
    d_all_methods             = new ArrayList();
    d_all_methods_long        = new HashMap();
    d_all_methods_short       = new HashMap();
    d_all_references          = new HashSet();
    d_all_basicarrays         = new HashSet();
    d_all_nonstatic_methods   = new ArrayList();
    d_all_static_methods      = new ArrayList();

    d_local_interfaces        = new HashMap();
    d_local_methods           = new ArrayList();
    d_local_methods_long      = new HashMap();
    d_local_methods_short     = new HashMap();
    d_local_nonstatic_methods = new ArrayList();
    d_local_static_methods    = new ArrayList();

    d_all_references.add(id);
  }


  /**
   * Return whether this object contains any abstract methods.  A class is
   * abstract if and only if it has any abstract methods.  An interface must
   * always be abstract, even if it contains no methods.
   */
  abstract public boolean isAbstract();


  /**
   * Return whether this object represents an interface or a class.
   */
  abstract public boolean isInterface();


  /**
   * Add a new method to this object.  No checking is done whether this
   * method is valid for this particular extendable.  The new method will
   * over-write any existing method unless the new method is abstract.
   */
  public void addMethod(Method method) {
    addToMethodLists(method, false);
    d_all_references.addAll(method.getSymbolReferences());
    d_all_basicarrays.addAll(method.getBasicArrays());
  }


  /**
   * Return the methods in this interface as a <code>Collection</code>.
   * If the boolean argument is true, then all methods are returned;
   * otherwise, only locally defined methods are returned.  Each element
   * in the collection is of type <code>Method</code>.
   */
  public Collection getMethods(boolean all) {
    return all ? d_all_methods : d_local_methods;
  }


  /**
   * Return the non-static methods in this interface as a 
   * <code>Collection</code>.  If the boolean argument is true, then all 
   * non-static methods are returned; otherwise, only locally defined non-
   * static methods are returned.  Each element in the collection is of 
   * type <code>Method</code>.
   */
  public Collection getNonstaticMethods(boolean all) {
    return all ? d_all_nonstatic_methods : d_local_nonstatic_methods;
  }


  /**
   * Return the static methods in this interface as a <code>Collection</code>.
   * If the boolean argument is true, then all static methods are returned;
   * otherwise, only locally defined static methods are returned.  Each 
   * element in the collection is of type <code>Method</code>.
   */
  public Collection getStaticMethods(boolean all) {
    return all ? d_all_static_methods : d_local_static_methods;
  }


  /**
   * Return the abstract methods for this class or interface, which
   * includes all parent classes and interfaces.  Each element in the
   * collection is of type <code>Method</code>.
   */
  public Collection getAbstractMethods() {
    return d_all_abstract_methods;
  }

  /**
   * Return <code>true</code> if and only if the method given is locally
   * defined in this extendable.
   *
   * @param m   the method of interest.
   * @return <code>true</code> if <code>m</code> is locally defined
   *         in this Extendable.
   */
  public boolean isLocal(Method m)
  {
    return (m != null) &&
      m.equals(d_local_methods_long.get(m.getLongMethodName()));
  }


  /**
   * Lookup the specified method by long method name.  If the boolean 
   * argument is true, then search all locally defined methods and all 
   * parents.  If no matching method name is found, then return null.
   */
  public Method lookupMethodByLongName(String name, boolean all) {
    return (Method) (all ? d_all_methods_long.get(name) 
                     : d_local_methods_long.get(name));
  }


  /**
   * Lookup the specified method by short method name, returning the
   * associated collection of methods, if found; otherwise, return null.  
   * If the boolean argument is true, then search all locally defined 
   * methods and all parents.  
   */
  public Collection lookupMethodByShortName(String name, boolean all) {
    return (Collection) (all ? d_all_methods_short.get(name)
                         : d_local_methods_short.get(name));
  }


  /**
   * Query whether the specified method exists by long name.  If the 
   * boolean argument is true, then search all locally defined methods 
   * and all parents.
   */
  public boolean hasMethodByLongName(String name, boolean all) {
    return all ? d_all_methods_long.containsKey(name) 
      : d_local_methods_long.containsKey(name);
  }


  /**
   * Query whether the specified method exists by short name.  If the 
   * boolean argument is true, then search all locally defined methods 
   * and all parents.  Recall there may be multiple methods with the
   * same short name but here we only care if there is at least one.
   */
  public boolean hasMethodByShortName(String name, boolean all) {
    return all ? d_all_methods_short.containsKey(name) 
      : d_local_methods_short.containsKey(name);
  }


  /**
   * Return whether any of the methods are static methods.  If the
   * boolean argument is true, then search all locally defined methods
   * and parents.
   */
  abstract public boolean hasStaticMethod(boolean all);


  /**
   * Return whether any of the methods throw exceptions.  If the
   * boolean argument is true, then search all locally defined methods
   * and parents.
   */
  public boolean hasExceptionThrowingMethod(boolean all) {
    boolean has_exception_throwing_method = false;
     
    for (Iterator i = getMethods(all).iterator(); i.hasNext(); ) {
      Method method = (Method) i.next();
      if (method.getThrows().size()>0) { 
        has_exception_throwing_method = true;
        break;
      }
    }
    return has_exception_throwing_method;
  }


  /**
   * Add a new parent interface to this class.  This method will be
   * implemented by the <code>Class</code> and <code>Interface</code>
   * subclasses.
   */
  abstract public void addParentInterface(Interface parent);


  /**
   * Return the parent interfaces in a <code>Collection</code>.  Each
   * member of the collection is an <code>Interface</code>.  If the
   * boolean argument is true, then return all parents; otherwise, return
   * only direct parents.
   */
  public Collection getParentInterfaces(boolean all) {
    return all ? d_all_interfaces.values() : d_local_interfaces.values();
  }

  /**
   * Return the parent interfaces and/or class in a <code>Collection</code>.
   * Each member of the collection is an <code>Extendable</code>.
   * If the boolean argument is true, the return all ancestors; otherwise,
   * return only direct parents. There are no duplicate entries in the
   * returned collection.
   */
  public Collection getParents(boolean all)
  {
    return getParentInterfaces(all);
  }

  /**
   * Return <code>true</code> if this object implements or
   * extends <code>ext</code> directly or indirectly. If <code>true</code>
   * is returned, it means that <code>ext</code> is somewhere above this
   * object in the type hierarchy.
   */
  public boolean hasAncestor(Extendable ext)
  {
    if (ext == null) return false;
    Collection parents = getParents(false);
    Iterator i = parents.iterator();
    while (i.hasNext()) {
      Extendable parent = (Extendable)i.next();
      if (ext == parent) return true;
    }
    i = parents.iterator();
    while (i.hasNext()) {
      Extendable parent = (Extendable)i.next();
      if (parent.hasAncestor(ext)) return true;
    }
    return false;
  }


  /**
   * Return whether this class has the specified parent interface.  If
   * the boolean argument is true, then all parents are searched; otherwise,
   * only the direct parents of this interface are searched.
   */
  public boolean hasParentInterface(SymbolID id, boolean all) {
    return all ? d_all_interfaces.containsKey(id)
      : d_local_interfaces.containsKey(id);
  }


  /**
   * Return the set of symbol references for this class.  These are
   * defined as all references for this object as well as its parents.
   * The set of references includes this symbol name, as well.  Each
   * element of the set is a <code>SymbolID</code>.
   */
  public Set getSymbolReferences() {
    return d_all_references;
  }

  public Set getAllSymbolReferences() {
    return getSymbolReferences();
  }

  /**
   * Return the set of basic array types for this class.
   */
  public Set getBasicArrayRefs() {
    return d_all_basicarrays;
  }


  /**
   * Protected method called by parents to add their methods and their
   * interfaces to this extendable object.
   */
  protected void addParentData(Extendable ext) {
    /*
     * Add all parent methods to the list of all methods in this
     * class.  Do not override an existing method in the methods
     * list if the new method is abstract, since abstract methods
     * cannot override other methods.  If the method is abstract,
     * then add it to the list of abstract methods; otherwise, remove
     * it from the list.
     */
    for (Iterator m = ext.getMethods(true).iterator(); m.hasNext(); ) {
      Method method = (Method) m.next();
      addToMethodLists(method, true);
    }

    /*
     * Add the parent and its parents to the list of parents.  That is
     * a lot of parents, let me tell you.
     */
    if (ext.isInterface()) {
      d_all_interfaces.put(ext.getSymbolID(), ext);
      d_local_interfaces.put(ext.getSymbolID(), ext);
    }

    Collection parents = ext.getParentInterfaces(true);
    for (Iterator p = parents.iterator(); p.hasNext(); ) {
      Extendable e = (Extendable) p.next();
      d_all_interfaces.put(e.getSymbolID(), e);
    }

    d_all_references.addAll(ext.getSymbolReferences());
    d_all_basicarrays.addAll(ext.getBasicArrayRefs());
  }


  /**
   * Determines if there is a method in the specified list that has the
   * same long method name as the given method.  If so, the method in
   * the list will be replaced if replaceIt is true; otherwise, the
   * method entry in the list will be removed.
   *
   * Assumption:  There is only one entry in the list for each long 
   * method name.
   */
  private void findExistingEntry(ArrayList list, Method method, 
                                 boolean replaceIt)
  {
    String  name  = method.getLongMethodName();
    boolean found = false;
    Method  item  = null;

    for (ListIterator iter = list.listIterator(); iter.hasNext() && !found; )
      {
        item = (Method) iter.next();
        if (item.getLongMethodName().equals(name)) {
          found = true;
          iter.remove();
        }
      }
    if (replaceIt) {
      list.add(method);
    }
  }


  /**
   * Adds the method to the specified short-name method list.
   */
  private void addToShortMethodList(HashMap list, Method method) {
    String name = method.getShortMethodName();

    if (list.containsKey(name)) {
      findExistingEntry((ArrayList) list.get(name), method, true);
    } else {
      ArrayList sublist = new ArrayList();
      sublist.add(method);
      list.put(name, sublist);
    }
  }


  /**
   * Adds a method to all and, if not isParent, local method lists
   * provided the method does not exist or is not abstract.  Note
   * the latter case is used to overwrite existing methods.
   */
  private void addToMethodLists(Method method, boolean isParent) {
    String long_name = method.getLongMethodName();

    if (!(hasMethodByLongName(long_name, true) && method.isAbstract())) {
      boolean is_static  = method.isStatic();

      /* 
       * First process the lists of local and parent methods (i.e., the
       * "all" lists).
       */
      findExistingEntry(d_all_methods, method, true);
      d_all_methods_long.put(long_name, method);
      addToShortMethodList(d_all_methods_short, method);
      if (is_static) {
        findExistingEntry(d_all_static_methods, method, true);
      } else {
        findExistingEntry(d_all_nonstatic_methods, method, true);
      }
      if (method.isAbstract()) {
        findExistingEntry(d_all_abstract_methods, method, true);
      } else {
        findExistingEntry(d_all_abstract_methods, method, false);
      }

      /* 
       * Now process the lists of local methods, provided not adding 
       * parent data. 
       */
      if (!isParent) {
        findExistingEntry(d_local_methods, method, true);
        d_local_methods_long.put(long_name, method);
        addToShortMethodList(d_local_methods_short, method);
        if (is_static) {
          findExistingEntry(d_local_static_methods, method, true);
        } else {
          findExistingEntry(d_local_nonstatic_methods, method, true);
        }
      }
    }
  }
}
